/*
 *  $Id: cmap_asciiexport.c 28142 2025-06-24 16:44:35Z yeti-dn $
 *  Copyright (C) 2025 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libprocess/lawn.h>
#include <libgwydgets/gwydataview.h>
#include <libgwymodule/gwymodule-cmap.h>
#include <app/gwyapp.h>
#include <app/gwymoduleutils.h>

#define RUN_MODES (GWY_RUN_INTERACTIVE)

enum {
    PARAM_ALL_CURVES,
    PARAM_ALL_SEGMENTS,
    PARAM_CURVE,
    PARAM_SEGMENT,
    PARAM_ENGLISH_COMMENT,
    PARAM_DECIMAL_DOT,
    PARAM_PRECISION,
};

typedef struct {
    GwyParams *params;
    GwyLawn *lawn;
    GwyContainer *data;
    gint id;
} ModuleArgs;

typedef struct {
    ModuleArgs *args;
    GtkWidget *dialog;
    GwyParamTable *table;
} ModuleGUI;

static gboolean         module_register     (void);
static GwyParamDef*     define_module_params(void);
static void             module_main         (GwyContainer *data,
                                             GwyRunType runtype);
static GwyDialogOutcome run_gui             (ModuleArgs *args);
static void             param_changed       (ModuleGUI *gui,
                                             gint id);
static gchar*           export_lawn         (gpointer user_data,
                                             gssize *data_len);
static void             destroy_lawn        (gchar *data,
                                             gpointer user_data);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Exports curve map data to a text file."),
    "Yeti <yeti@gwyddion.net>",
    "1.0",
    "David Nečas (Yeti)",
    "2025",
};

GWY_MODULE_QUERY2(module_info, cmap_asciiexport)

static gboolean
module_register(void)
{
    gwy_curve_map_func_register("cmap_asciiexport",
                                (GwyCurveMapFunc)&module_main,
                                N_("/Export _Text..."),
                                NULL,
                                RUN_MODES,
                                GWY_MENU_FLAG_CURVE_MAP,
                                N_("Export curve map data to a text file"));


    return TRUE;
}

static GwyParamDef*
define_module_params(void)
{
    static GwyParamDef *paramdef = NULL;

    if (paramdef)
        return paramdef;

    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, gwy_curve_map_func_current());
    gwy_param_def_add_boolean(paramdef, PARAM_ALL_CURVES, "all_curves", _("_All curves"), TRUE);
    gwy_param_def_add_boolean(paramdef, PARAM_ALL_SEGMENTS, "all_segments", _("All _segments"), TRUE);
    gwy_param_def_add_lawn_curve(paramdef, PARAM_CURVE, "curve", NULL);
    gwy_param_def_add_lawn_segment(paramdef, PARAM_SEGMENT, "segment", NULL);
    gwy_param_def_add_boolean(paramdef, PARAM_ENGLISH_COMMENT, "english-comment", _("Keep comment in English"),
                              FALSE);
    gwy_param_def_add_boolean(paramdef, PARAM_DECIMAL_DOT, "decimal-dot", _("Use _dot as decimal separator"), TRUE);
    gwy_param_def_add_int(paramdef, PARAM_PRECISION, "precision", _("_Precision"), 0, 16, 5);

    return paramdef;
}

static void
module_main(GwyContainer *data, GwyRunType runtype)
{
    ModuleArgs args;
    GwyDialogOutcome outcome = GWY_DIALOG_PROCEED;

    g_return_if_fail(runtype & RUN_MODES);

    gwy_clear(&args, 1);
    gwy_app_data_browser_get_current(GWY_APP_LAWN, &args.lawn,
                                     GWY_APP_LAWN_ID, &args.id,
                                     0);
    args.params = gwy_params_new_from_settings(define_module_params());
    args.data = data;

    if (runtype == GWY_RUN_INTERACTIVE) {
        outcome = run_gui(&args);
        gwy_params_save_to_settings(args.params);
        if (outcome == GWY_DIALOG_CANCEL)
            goto end;
    }

    gwy_save_auxiliary_with_callback(_("Export to Text File"), NULL, export_lawn, destroy_lawn, &args);

end:
    g_object_unref(args.params);
}

static GwyDialogOutcome
run_gui(ModuleArgs *args)
{
    GwyParamTable *table;
    GwyDialog *dialog;
    ModuleGUI gui;
    gboolean needs_decimal_dot_option;

    gwy_clear(&gui, 1);
    gui.args = args;
    needs_decimal_dot_option = !gwy_strequal(gwy_get_decimal_separator(), ".");

    gui.dialog = gwy_dialog_new(_("Export Text"));
    dialog = GWY_DIALOG(gui.dialog);
    gwy_dialog_add_buttons(dialog, GWY_RESPONSE_RESET, GTK_RESPONSE_CANCEL, GTK_RESPONSE_OK, 0);

    table = gui.table = gwy_param_table_new(args->params);
    gwy_param_table_append_checkbox(table, PARAM_ALL_CURVES);
    gwy_param_table_append_lawn_curve(table, PARAM_CURVE, args->lawn);
    gwy_param_table_append_separator(table);
    if (gwy_lawn_get_n_segments(args->lawn)) {
        gwy_param_table_append_checkbox(table, PARAM_ALL_SEGMENTS);
        gwy_param_table_append_lawn_segment(table, PARAM_SEGMENT, args->lawn);
    }
    if (needs_decimal_dot_option)
        gwy_param_table_append_checkbox(table, PARAM_DECIMAL_DOT);
    gwy_param_table_append_checkbox(table, PARAM_ENGLISH_COMMENT);
    gwy_param_table_append_slider(table, PARAM_PRECISION);
    gwy_param_table_slider_set_mapping(table, PARAM_PRECISION, GWY_SCALE_MAPPING_LINEAR);
    gwy_dialog_add_content(dialog, gwy_param_table_widget(table), FALSE, FALSE, 0);
    gwy_dialog_add_param_table(dialog, table);

    g_signal_connect_swapped(table, "param-changed", G_CALLBACK(param_changed), &gui);

    return gwy_dialog_run(dialog);
}

static void
param_changed(ModuleGUI *gui, gint id)
{
    GwyParamTable *table = gui->table;
    GwyParams *params = gui->args->params;

    if (id < 0 || id == PARAM_ALL_CURVES) {
        gboolean all_curves = gwy_params_get_boolean(params, PARAM_ALL_CURVES);
        gwy_param_table_set_sensitive(table, PARAM_CURVE, !all_curves);
    }
    if ((id < 0 || id == PARAM_ALL_SEGMENTS) && gwy_param_table_exists(table, PARAM_SEGMENT)) {
        gboolean all_segments = gwy_params_get_boolean(params, PARAM_ALL_SEGMENTS);
        gwy_param_table_set_sensitive(gui->table, PARAM_SEGMENT, !all_segments);
    }
}

static void
append_lateral_value(GString *str, const gchar *name, gdouble value,
                     gboolean translate, gboolean decimal_dot, GwySIValueFormat *vf)
{
    gdouble v = value/vf->magnitude;
    g_string_append_printf(str, "# %s ", translate ? _(name) : name);
    gwy_append_doubles_to_gstring(str, &v, 1, 6, "", decimal_dot);
    g_string_append_printf(str, " %s\n", vf->units);
}

static gchar*
export_lawn(gpointer user_data, gssize *data_len)
{
    ModuleArgs *args = (ModuleArgs*)user_data;
    GwyLawn *lawn = args->lawn;
    GwyParams *params = args->params;
    gint precision = gwy_params_get_int(params, PARAM_PRECISION);
    gboolean decimal_dot = gwy_params_get_boolean(params, PARAM_DECIMAL_DOT);
    gboolean all_curves = gwy_params_get_boolean(params, PARAM_ALL_CURVES);
    gboolean all_segments = gwy_params_get_boolean(params, PARAM_ALL_SEGMENTS);
    gboolean english_comment = gwy_params_get_boolean(params, PARAM_ENGLISH_COMMENT);
    gint curve = gwy_params_get_int(params, PARAM_CURVE);
    gint segment = gwy_params_get_int(params, PARAM_SEGMENT);
    gint ncurves = gwy_lawn_get_n_curves(lawn);
    gint nsegments = gwy_lawn_get_n_segments(lawn);
    gint xres = gwy_lawn_get_xres(lawn), yres = gwy_lawn_get_yres(lawn);
    GString *str = g_string_new(NULL);
    GwySIValueFormat *vf = NULL;
    GwySIUnit *unit;
    const gchar *label;
    gint i, j, k, curve_from, curve_to;
    gchar *s, *t;
    gdouble v;

    /* Add a ‘magic header’. The other fields are kind of generic and do not permig telling apart exported curve maps
     * from other things easily. */
    g_string_assign(str, "# Curve map text export\n");

    s = gwy_app_get_lawn_title(args->data, args->id);
    g_string_append_printf(str, "# %s %s\n", english_comment ? "Channel:" : _("Channel:"), s);
    g_free(s);

    g_string_append_printf(str, "# %s %d\n", english_comment ? "Rows:" : _("Rows:"), gwy_lawn_get_yres(lawn));
    g_string_append_printf(str, "# %s %d\n", english_comment ? "Columns:" : _("Columns:"), gwy_lawn_get_xres(lawn));
    g_string_append_printf(str, "# %s %d\n", english_comment ? "Curves:" : _("Curves:"), all_curves ? ncurves : 1);
    if (nsegments && !all_segments) {
        label = gwy_lawn_get_segment_label(lawn, segment);
        if (label)
            g_string_append_printf(str, "# %s %s\n", english_comment ? "Segment:" : _("Segment:"), label);
        else
            g_string_append_printf(str, "# %s %d\n", english_comment ? "Segment:" : _("Segment:"), segment);
    }

    vf = gwy_lawn_get_value_format_xy(lawn, GWY_SI_UNIT_FORMAT_VFMARKUP, vf);

    append_lateral_value(str, N_("Width:"), gwy_lawn_get_xreal(lawn), !english_comment, decimal_dot, vf);
    append_lateral_value(str, N_("Height:"), gwy_lawn_get_yreal(lawn), !english_comment, decimal_dot, vf);
    if ((v = gwy_lawn_get_xoffset(lawn)))
        append_lateral_value(str, N_("XOffset:"), v, !english_comment, decimal_dot, vf);
    if ((v = gwy_lawn_get_yoffset(lawn)))
        append_lateral_value(str, N_("YOffset:"), v, !english_comment, decimal_dot, vf);

    if (all_curves) {
        curve_from = 0;
        curve_to = ncurves;
    }
    else {
        curve_from = curve;
        curve_to = curve+1;
    }

    for (k = curve_from; k < curve_to; k++) {
        label = gwy_lawn_get_curve_label(lawn, k);
        if (label) {
            s = g_strdup_printf(english_comment ? "Curve%d:" : _("Curve%d:"), k+1 - curve_from);
            g_string_append_printf(str, "# %s %s\n", s, label);
            g_free(s);
        }
        unit = gwy_lawn_get_si_unit_curve(lawn, k);
        s = g_strdup_printf(english_comment ? "CurveUnit%d:" : _("CurveUnit%d:"), k+1 - curve_from);
        t = gwy_si_unit_get_string(unit, GWY_SI_UNIT_FORMAT_PLAIN);
        g_string_append_printf(str, "# %s %s\n", s, t);
        g_free(t);
        g_free(s);
    }
    gwy_si_unit_value_format_free(vf);

    for (i = 0; i < yres; i++) {
        for (j = 0; j < xres; j++) {
            for (k = curve_from; k < curve_to; k++) {
                gint npoints;
                const gdouble *cdata = gwy_lawn_get_curve_data_const(lawn, j, i, k, &npoints);
                gint kfrom = 0, kto = npoints;
                if (nsegments && !all_segments) {
                    const gint *segments = gwy_lawn_get_segments(lawn, j, i, NULL);
                    kfrom = MAX(segments[2*k], 0);
                    kto = MIN(segments[2*k + 1], npoints);
                    kto = MAX(kto, kfrom);
                }
                gwy_append_doubles_to_gstring(str, cdata + kfrom, kto - kfrom, precision, " ", decimal_dot);
                g_string_append_c(str, '\n');
            }
        }
    }

    *data_len = str->len;
    return g_string_free(str, FALSE);
}

static void
destroy_lawn(gchar *data, G_GNUC_UNUSED gpointer user_data)
{
    g_free(data);
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
