# Handle Scope

## jerryx_handle_scope

**Summary**
It is often necessary to make the lifespan of handles shorter than the lifespan of a native method. Even though the native code could only use the most recent handle, all of the associated objects would also be kept alive since they all share the same scope.

To handle this case, JerryScript HandleScope extension provides the ability to establish a new 'scope' to which newly created handles will be associated. Once those handles are no longer required, the scope can be 'closed' and any handles associated with the scope are invalidated. The methods available to open/close scopes are `jerryx_open_handle_scope` and `jerryx_close_handle_scope`.

JerryScript only supports a single nested hierarchy of scopes. There is only one active scope at any time, and all new handles will be associated with that scope while it is active. Scopes must be closed in the reverse order from which they are opened. In addition, all scopes created within a native method must be closed before returning from that method.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"
#include "jerryscript-ext/handle-scope.h"

static jerry_value_t
create_object (void)
{
  jerry_value_t obj = jerry_object ();
  return obj;
} /* create_object */

static void
test_handle_scope_val (void)
{
  jerryx_handle_scope scope;
  jerryx_open_handle_scope (&scope);
  jerry_value_t obj = jerryx_create_handle (create_object ());

  jerryx_close_handle_scope (scope);
  // now obj has been released
} /* test_handle_scope_val */

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  test_handle_scope_val ();
  jerry_heap_gc (JERRY_GC_PRESSURE_LOW);

  jerry_cleanup ();
} /* main */
```

## jerryx_escapable_handle_scope

**Summary**

It is necessary in common cases that a handle has to be promote to outer scope and prevent from been garbage collected. To handle this case, a escapable handle scope has been proposed from which one object can be promoted to the outer scope. The method available to escape an object from been release at current scope is `jerryx_escape_handle`.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"
#include "jerryscript-ext/handle-scope.h"

static jerry_value_t
create_object (void)
{
  jerryx_escapable_handle_scope scope;
  jerryx_open_escapable_handle_scope (&scope);
  jerry_value_t obj = jerryx_create_handle (jerry_object ());

  jerry_value_t escaped_obj;
  jerryx_escape_handle(scope, obj, &escaped_obj);
  jerryx_close_handle_scope (scope);
  // escaped_obj has now been escaped to outer scope, thus not released at this point

  return escaped_obj;
} /* create_object */

static void
test_handle_scope_val (void)
{
  jerryx_handle_scope scope;
  jerryx_open_handle_scope (&scope);
  jerry_value_t obj = create_object ();

  jerryx_close_handle_scope (scope);
  // now obj has been released
} /* test_handle_scope_val */

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  test_handle_scope_val ();
  jerry_heap_gc (JERRY_GC_PRESSURE_LOW);

  jerry_cleanup ();
} /* main */
```

**See also**

- [jerry_value_t](../docs/02.API-REFERENCE.md#jerry_value_t)
- [jerry_value_copy](../docs/02.API-REFERENCE.md#jerry_value_copy)
- [jerry_value_free](../docs/02.API-REFERENCE.md#jerry_value_free)

## Pre-allocated list of handle scopes and handles

To prevent trapping into system calls frequently, a pre-allocated dedicated list mechanism has been introduced to the implementation of JerryX handle scope.

To change the size of pre-allocation list, use build definition `JERRYX_HANDLE_PRELIST_SIZE` and `JERRYX_SCOPE_PRELIST_SIZE` to alter the default value of 20.
