// bipedobject.h
//
// rag-doll code, based on ODEHuman.cpp from Tyler Streeter's 
// Medieval Playground.
// cleaned up, and ported to plib by Bram Stolk

#define DENSITY 1.0

#ifndef BIPEDOBJECT_H
#define BIPEDOBJECT_H

#include <ode/ode.h>

#include "worldobject.h"


class BipedObject : public WorldObject
{

  public:

    BipedObject
    (
      dWorldID worldid, 
      dSpaceID space,
      dReal posx, dReal posy, dReal posz, 
      dReal size,
      ssgEntity *model_torso=0,
      ssgEntity *model_head=0,
      ssgEntity *model_upperarm=0,
      ssgEntity *model_lowerarm=0,
      ssgEntity *model_upperleg=0,
      ssgEntity *model_lowerleg=0,
      ssgEntity *model_foot=0
    )  :
      WorldObject(0)
    {
      name = "biped";

      torsoDimensions[1] = 1.0*size;
      torsoDimensions[2] = 2.0*size;
      torsoDimensions[0] = 0.5*size;
  
      upperArmDimensions[1] = 0.5*size;
      upperArmDimensions[2] = 1.5*size;
      upperArmDimensions[0] = 0.4*size;

      lowerArmDimensions[1] = 0.5*size;
      lowerArmDimensions[2] = 1.5*size;
      lowerArmDimensions[0] = 0.4*size;

      upperLegDimensions[1] = 0.6*size;
      upperLegDimensions[2] = 1.7*size;
      upperLegDimensions[0] = 0.5*size;
    
      lowerLegDimensions[1] = 0.5*size;
      lowerLegDimensions[2] = 1.7*size;
      lowerLegDimensions[0] = 0.5*size;
    
      footDimensions[1] = 0.5*size;
      footDimensions[2] = 0.3*size;
      footDimensions[0] = 1.0*size;

      headRadius = 0.5*size;

      geom_torso = dCreateBox(0, torsoDimensions[0], torsoDimensions[1], torsoDimensions[2]);
      geom_head = dCreateSphere(0, headRadius);
      geom_leftupperarm = dCreateBox(0, upperArmDimensions[0], upperArmDimensions[1], upperArmDimensions[2]);
      geom_rightupperarm = dCreateBox(0, upperArmDimensions[0], upperArmDimensions[1], upperArmDimensions[2]);
      geom_leftlowerarm = dCreateBox(0, lowerArmDimensions[0], lowerArmDimensions[1], lowerArmDimensions[2]);
      geom_rightlowerarm = dCreateBox(0, lowerArmDimensions[0], lowerArmDimensions[1], lowerArmDimensions[2]);
      geom_leftupperleg = dCreateBox(0, upperLegDimensions[0], upperLegDimensions[1], upperLegDimensions[2]);
      geom_rightupperleg = dCreateBox(0, upperLegDimensions[0], upperLegDimensions[1], upperLegDimensions[2]);
      geom_leftlowerleg = dCreateBox(0, lowerLegDimensions[0], lowerLegDimensions[1], lowerLegDimensions[2]);
      geom_rightlowerleg = dCreateBox(0, lowerLegDimensions[0], lowerLegDimensions[1], lowerLegDimensions[2]);
      geom_leftfoot = dCreateBox(0, footDimensions[0], footDimensions[1], footDimensions[2]);
      geom_rightfoot = dCreateBox(0, footDimensions[0], footDimensions[1], footDimensions[2]);

      body_torso = dBodyCreate(worldid);
      body_head = dBodyCreate(worldid);
      body_leftupperarm = dBodyCreate(worldid);
      body_rightupperarm = dBodyCreate(worldid);
      body_leftlowerarm = dBodyCreate(worldid);
      body_rightlowerarm = dBodyCreate(worldid);
      body_leftupperleg = dBodyCreate(worldid);
      body_rightupperleg = dBodyCreate(worldid);
      body_leftlowerleg = dBodyCreate(worldid);
      body_rightlowerleg = dBodyCreate(worldid);
      body_leftfoot = dBodyCreate(worldid);
      body_rightfoot = dBodyCreate(worldid);

      dMass m;
      dMassSetBox (&m,DENSITY,torsoDimensions[0],torsoDimensions[1],torsoDimensions[2]); 
      dBodySetMass (body_torso, &m);
      dMassSetBox (&m,DENSITY,upperArmDimensions[0],upperArmDimensions[1],upperArmDimensions[2]); 
      dBodySetMass (body_leftupperarm, &m);
      dBodySetMass (body_rightupperarm, &m);
      dMassSetBox (&m,DENSITY,lowerArmDimensions[0],lowerArmDimensions[1],lowerArmDimensions[2]); 
      dBodySetMass (body_leftlowerarm, &m);
      dBodySetMass (body_rightlowerarm, &m);
      dMassSetBox (&m,DENSITY,upperLegDimensions[0],upperLegDimensions[1],upperLegDimensions[2]); 
      dBodySetMass (body_leftupperleg, &m);
      dBodySetMass (body_rightupperleg, &m);
      dMassSetBox (&m,DENSITY,lowerLegDimensions[0],lowerLegDimensions[1],lowerLegDimensions[2]); 
      dBodySetMass (body_leftlowerleg, &m);
      dBodySetMass (body_rightlowerleg, &m);
      dMassSetBox (&m,DENSITY,footDimensions[0],footDimensions[1],footDimensions[2]); 
      dBodySetMass (body_leftfoot, &m);
      dBodySetMass (body_rightfoot, &m);
      dMassSetSphere (&m, DENSITY, headRadius);
      dBodySetMass (body_head, &m);

      dGeomSetBody(geom_torso, body_torso);
      dGeomSetBody(geom_head, body_head);
      dGeomSetBody(geom_leftupperarm, body_leftupperarm);
      dGeomSetBody(geom_rightupperarm, body_rightupperarm);
      dGeomSetBody(geom_leftlowerarm, body_leftlowerarm);
      dGeomSetBody(geom_rightlowerarm, body_rightlowerarm);
      dGeomSetBody(geom_leftupperleg, body_leftupperleg);
      dGeomSetBody(geom_rightupperleg, body_rightupperleg);
      dGeomSetBody(geom_leftlowerleg, body_leftlowerleg);
      dGeomSetBody(geom_rightlowerleg, body_rightlowerleg);
      dGeomSetBody(geom_leftfoot, body_leftfoot);
      dGeomSetBody(geom_rightfoot, body_rightfoot);

#if 0
      dQuaternion q;
      dQFromAxisAndAngle(q, 0,0,1, M_PI/2);
      dBodySetQuaternion(body_torso, q);
#endif
      SetInitialPosition(posx, posy, posz, dBodyGetQuaternion(body_torso));

      // Create the joints

      neck = dJointCreateHinge(worldid, 0);
      leftShoulder = dJointCreateHinge(worldid, 0);
      rightShoulder = dJointCreateHinge(worldid, 0);
      leftElbow = dJointCreateHinge(worldid, 0);
      rightElbow = dJointCreateHinge(worldid, 0);
      leftHip = dJointCreateHinge(worldid, 0);
      rightHip = dJointCreateHinge(worldid, 0);
      leftKnee = dJointCreateHinge(worldid, 0);
      rightKnee = dJointCreateHinge(worldid, 0);
      leftAnkle = dJointCreateHinge(worldid, 0);
      rightAnkle = dJointCreateHinge(worldid, 0);

      leftFootLock = dJointCreateFixed(worldid, 0);
      rightFootLock = dJointCreateFixed(worldid, 0);
      torsoLock = dJointCreateFixed(worldid, 0);

      dJointAttach(neck, body_torso, body_head);
      dJointAttach(leftShoulder, body_torso, body_leftupperarm);
      dJointAttach(rightShoulder, body_torso, body_rightupperarm);
      dJointAttach(leftElbow, body_leftupperarm, body_leftlowerarm);
      dJointAttach(rightElbow, body_rightupperarm, body_rightlowerarm);
      dJointAttach(leftHip, body_torso, body_leftupperleg);
      dJointAttach(rightHip, body_torso, body_rightupperleg);
      dJointAttach(leftKnee, body_leftupperleg, body_leftlowerleg);
      dJointAttach(rightKnee, body_rightupperleg, body_rightlowerleg);
      dJointAttach(leftAnkle, body_leftlowerleg, body_leftfoot);
      dJointAttach(rightAnkle, body_rightlowerleg, body_rightfoot);

      dJointSetFeedback(neck, &neck_feedback);

        //UPDATE SOME HINGE JOINTS WITH BALL-AND-SOCKET/AMOTORS
	//neck = dJointCreateUniversal(dynamicsWorldID, 0);
	//dJointAttach(neck, torso->GetBodyID(), head->GetBodyID());
	//jointPoint = torso->GetPosition();
	//jointPoint.y += torso->GetYDimension()/2;
	//dJointSetUniversalAnchor(neck, jointPoint.x, jointPoint.y, jointPoint.z);
	//dJointSetUniversalAxis1(neck, 1, 0, 0);
	//dJointSetUniversalAxis2(neck, 0, 0, 1);
	//dJointSetUniversalParam(neck, dParamLoStop, -M_PI/3);
	//dJointSetUniversalParam(neck, dParamHiStop, M_PI/3);
	//dJointSetUniversalParam(neck, dParamLoStop2, -M_PI/3);
	//dJointSetUniversalParam(neck, dParamHiStop2, M_PI/3);

      dJointSetHingeAnchor(neck, posx, posy, posz+0.5*torsoDimensions[2]);
      dJointSetHingeAxis(neck, 0, 1, 0);
      dJointSetHingeParam(neck, dParamLoStop, -M_PI/3);
      dJointSetHingeParam(neck, dParamHiStop,  M_PI/3);

	//leftShoulder = dJointCreateUniversal(dynamicsWorldID, 0);
	//dJointAttach(leftShoulder, torso->GetBodyID(), leftUpperArm->GetBodyID());
	//jointPoint = torso->GetPosition();
	//jointPoint.y += torso->GetYDimension()/2;
	//jointPoint.x -= torso->GetXDimension()/2;
	//dJointSetUniversalAnchor(leftShoulder, jointPoint.x, jointPoint.y, jointPoint.z);
	//dJointSetUniversalAxis1(leftShoulder, 1, 0, 0);
	//dJointSetUniversalAxis2(leftShoulder, 0, 0, 1);
	//dJointSetUniversalParam(leftShoulder, dParamLoStop, -M_PI+0.001); //must be > -PI
	//dJointSetUniversalParam(leftShoulder, dParamHiStop, M_PI-0.001); //must be < PI
	//dJointSetUniversalParam(leftShoulder, dParamLoStop2, 0);
	//dJointSetUniversalParam(leftShoulder, dParamHiStop2, 3*M_PI/4);
      dJointSetHingeAnchor(leftShoulder, posx, posy-0.5*torsoDimensions[1], posz+0.4*torsoDimensions[2]);
      dJointSetHingeAxis(leftShoulder, 0, 1, 0);

	//rightShoulder = dJointCreateUniversal(dynamicsWorldID, 0);
	//dJointAttach(rightShoulder, torso->GetBodyID(), rightUpperArm->GetBodyID());
	//jointPoint = torso->GetPosition();
	//jointPoint.y += torso->GetYDimension()/2;
	//jointPoint.x += torso->GetXDimension()/2;
	//dJointSetUniversalAnchor(rightShoulder, jointPoint.x, jointPoint.y, jointPoint.z);
	//dJointSetUniversalAxis1(rightShoulder, 1, 0, 0);
	//dJointSetUniversalAxis2(rightShoulder, 0, 0, 1);
	//dJointSetUniversalParam(rightShoulder, dParamLoStop, -M_PI+0.001); //must be > -PI
	//dJointSetUniversalParam(rightShoulder, dParamHiStop, M_PI-0.001); //must be < PI
	//dJointSetUniversalParam(rightShoulder, dParamLoStop2, -3*M_PI/4);
	//dJointSetUniversalParam(rightShoulder, dParamHiStop2, 0);
      dJointSetHingeAnchor(rightShoulder, posx, posy+0.5*torsoDimensions[1], posz+0.4*torsoDimensions[2]);
      dJointSetHingeAxis(rightShoulder, 0, 1, 0);

      dJointSetHingeAnchor(leftElbow, posx, posy, posz+torsoDimensions[2]/2-upperArmDimensions[2]);
      dJointSetHingeAxis(leftElbow, 0, 1, 0);
      dJointSetHingeParam(leftElbow, dParamLoStop, 0);
      dJointSetHingeParam(leftElbow, dParamHiStop, 4*M_PI/5);

      dJointSetHingeAnchor(rightElbow, posx, posy, posz+torsoDimensions[2]/2-upperArmDimensions[2]);
      dJointSetHingeAxis(rightElbow, 0, 1, 0);
      dJointSetHingeParam(rightElbow, dParamLoStop, 0);
      dJointSetHingeParam(rightElbow, dParamHiStop, 4*M_PI/5);

	//leftHip = dJointCreateUniversal(dynamicsWorldID, 0);
	//dJointAttach(leftHip, torso->GetBodyID(), leftUpperLeg->GetBodyID());
	//jointPoint = leftUpperLeg->GetPosition();
	//jointPoint.y += leftUpperLeg->GetYDimension()/2;
	//dJointSetUniversalAnchor(leftHip, jointPoint.x, jointPoint.y, jointPoint.z);
	//dJointSetUniversalAxis1(leftHip, 1, 0, 0);
	//dJointSetUniversalAxis2(leftHip, 0, 0, 1);
	//dJointSetUniversalParam(leftHip, dParamLoStop, -M_PI/6);
	//dJointSetUniversalParam(leftHip, dParamHiStop, 3*M_PI/4);
	//dJointSetUniversalParam(leftHip, dParamLoStop2, 0);
	//dJointSetUniversalParam(leftHip, dParamHiStop2, M_PI/4);
      dJointSetHingeAnchor(leftHip, posx, posy, posz-0.5*torsoDimensions[2]);
      dJointSetHingeAxis(leftHip, 0, 1, 0);
      dJointSetHingeParam(leftHip, dParamLoStop, -M_PI/6);
      dJointSetHingeParam(leftHip, dParamHiStop, 3*M_PI/4);

	//rightHip = dJointCreateUniversal(dynamicsWorldID, 0);
	//dJointAttach(rightHip, torso->GetBodyID(), rightUpperLeg->GetBodyID());
	//jointPoint = rightUpperLeg->GetPosition();
	//jointPoint.y += rightUpperLeg->GetYDimension()/2;
	//dJointSetUniversalAnchor(rightHip, jointPoint.x, jointPoint.y, jointPoint.z);
	//dJointSetUniversalAxis1(rightHip, 1, 0, 0);
	//dJointSetUniversalAxis2(rightHip, 0, 0, 1);
	//dJointSetUniversalParam(rightHip, dParamLoStop, -M_PI/6);
	//dJointSetUniversalParam(rightHip, dParamHiStop, 3*M_PI/4);
	//dJointSetUniversalParam(rightHip, dParamLoStop2, -M_PI/4);
	//dJointSetUniversalParam(rightHip, dParamHiStop2, 0);
      dJointSetHingeAnchor(rightHip, posx, posy, posz-0.5*torsoDimensions[2]);
      dJointSetHingeAxis(rightHip, 0, 1, 0);
      dJointSetHingeParam(rightHip, dParamLoStop, -M_PI/6);
      dJointSetHingeParam(rightHip, dParamHiStop, 3*M_PI/4);

      dJointSetHingeAnchor(leftKnee, posx+0.5*upperLegDimensions[0], posy, posz-0.5*torsoDimensions[2]-upperLegDimensions[2]);
      dJointSetHingeAxis(leftKnee, 0, 1, 0);
      dJointSetHingeParam(leftKnee, dParamLoStop, -M_PI/2);
      dJointSetHingeParam(leftKnee, dParamHiStop, 0);
    
      dJointSetHingeAnchor(rightKnee, posx+0.5*upperLegDimensions[0], posy, posz-0.5*torsoDimensions[2]-upperLegDimensions[2]);
      dJointSetHingeAxis(rightKnee, 0, 1, 0);
      dJointSetHingeParam(rightKnee, dParamLoStop, -M_PI/2);
      dJointSetHingeParam(rightKnee, dParamHiStop, 0);

      dJointSetHingeAnchor(leftAnkle, posx, posy, posz-0.5*torsoDimensions[2]-upperLegDimensions[2]-lowerLegDimensions[2]);
      dJointSetHingeAxis(leftAnkle, 0, 1, 0);
      dJointSetHingeParam(leftAnkle, dParamLoStop, -M_PI/2);
      dJointSetHingeParam(leftAnkle, dParamHiStop, M_PI/6);

      dJointSetHingeAnchor(rightAnkle, posx, posy, posz-0.5*torsoDimensions[2]-upperLegDimensions[2]-lowerLegDimensions[2]);
      dJointSetHingeAxis(rightAnkle, 0, 1, 0);
      dJointSetHingeParam(rightAnkle, dParamLoStop, -M_PI/2);
      dJointSetHingeParam(rightAnkle, dParamHiStop, M_PI/6);

      // Friction for shoulders

      motor_leftshoulder = dJointCreateAMotor(worldid, 0);
      dJointAttach(motor_leftshoulder, body_torso, body_leftupperarm);
      dJointSetAMotorNumAxes(motor_leftshoulder, 1);
      dJointSetAMotorAxis(motor_leftshoulder, 0, 1, 0,1,0);
      dJointSetAMotorParam(motor_leftshoulder, dParamVel, 0.0);
      dJointSetAMotorParam(motor_leftshoulder, dParamFMax, 0.002);

      motor_rightshoulder = dJointCreateAMotor(worldid, 0);
      dJointAttach(motor_rightshoulder, body_torso, body_rightupperarm);
      dJointSetAMotorNumAxes(motor_rightshoulder, 1);
      dJointSetAMotorAxis(motor_rightshoulder, 0, 1, 0,1,0);
      dJointSetAMotorParam(motor_rightshoulder, dParamVel, 0.0);
      dJointSetAMotorParam(motor_rightshoulder, dParamFMax, 0.002);

      dGeomSetData (geom_torso, this);
      dGeomSetData (geom_head, this);
      dGeomSetData (geom_leftupperarm, this);
      dGeomSetData (geom_rightupperarm, this);
      dGeomSetData (geom_leftlowerarm, this);
      dGeomSetData (geom_rightlowerarm, this);
      dGeomSetData (geom_leftupperleg, this);
      dGeomSetData (geom_rightupperleg, this);
      dGeomSetData (geom_leftlowerleg, this);
      dGeomSetData (geom_rightlowerleg, this);
      dGeomSetData (geom_leftfoot, this);
      dGeomSetData (geom_rightfoot, this);
    
      // Add to space

      biped_space = dSimpleSpaceCreate(space);

      dSpaceAdd (biped_space, geom_torso);
      dSpaceAdd (biped_space, geom_head);
      dSpaceAdd (biped_space, geom_leftupperarm);
      dSpaceAdd (biped_space, geom_rightupperarm);
      dSpaceAdd (biped_space, geom_leftlowerarm);
      dSpaceAdd (biped_space, geom_rightlowerarm);
      dSpaceAdd (biped_space, geom_leftupperleg);
      dSpaceAdd (biped_space, geom_rightupperleg);
      dSpaceAdd (biped_space, geom_leftlowerleg);
      dSpaceAdd (biped_space, geom_rightlowerleg);
      dSpaceAdd (biped_space, geom_leftfoot);
      dSpaceAdd (biped_space, geom_rightfoot);


      // plib stuff
      ssgBranch *branch = new ssgBranch();
    
      assert(model_torso);
      assert(model_upperarm);
    
      trf_torso = new ssgTransform();
      trf_head = new ssgTransform();
      trf_leftupperarm = new ssgTransform();
      trf_rightupperarm = new ssgTransform();
      trf_leftlowerarm = new ssgTransform();
      trf_rightlowerarm = new ssgTransform();
      trf_leftupperleg = new ssgTransform();
      trf_rightupperleg = new ssgTransform();
      trf_leftlowerleg = new ssgTransform();
      trf_rightlowerleg = new ssgTransform();
      trf_leftfoot = new ssgTransform();
      trf_rightfoot = new ssgTransform();
    
      trf_torso->addKid(model_torso);
      trf_head->addKid(model_head);
      trf_leftupperarm->addKid(model_upperarm);
      trf_rightupperarm->addKid(model_upperarm);
      trf_leftlowerarm->addKid(model_lowerarm);
      trf_rightlowerarm->addKid(model_lowerarm);
      trf_leftupperleg->addKid(model_upperleg);
      trf_rightupperleg->addKid(model_upperleg);
      trf_leftlowerleg->addKid(model_lowerleg);
      trf_rightlowerleg->addKid(model_lowerleg);
      trf_leftfoot->addKid(model_foot);
      trf_rightfoot->addKid(model_foot);
    
      branch->addKid(trf_torso);
      branch->addKid(trf_head);
      branch->addKid(trf_leftupperarm);
      branch->addKid(trf_rightupperarm);
      branch->addKid(trf_leftlowerarm);
      branch->addKid(trf_rightlowerarm);
      branch->addKid(trf_leftupperleg);
      branch->addKid(trf_rightupperleg);
      branch->addKid(trf_leftlowerleg);
      branch->addKid(trf_rightlowerleg);
      branch->addKid(trf_leftfoot);
      branch->addKid(trf_rightfoot);
    
      entity = branch;
    }



    ~BipedObject()
    {
      dJointDestroy(leftElbow);
      dJointDestroy(rightElbow);
      dJointDestroy(leftShoulder);
      dJointDestroy(rightShoulder);
      dJointDestroy(neck);
      dJointDestroy(leftHip);
      dJointDestroy(rightHip);
      dJointDestroy(leftKnee);
      dJointDestroy(rightKnee);
      dJointDestroy(leftAnkle);
      dJointDestroy(rightAnkle);

      dJointDestroy(leftFootLock);
      dJointDestroy(rightFootLock);
      dJointDestroy(torsoLock);

      dGeomDestroy(geom_head);
      dGeomDestroy(geom_torso);
      dGeomDestroy(geom_leftupperarm);
      dGeomDestroy(geom_rightupperarm);
      dGeomDestroy(geom_leftlowerarm);
      dGeomDestroy(geom_rightlowerarm);
      dGeomDestroy(geom_leftupperleg);
      dGeomDestroy(geom_rightupperleg);
      dGeomDestroy(geom_leftlowerleg);
      dGeomDestroy(geom_rightlowerleg);
      dGeomDestroy(geom_leftfoot);
      dGeomDestroy(geom_rightfoot);
    }


    void SetInitialPosition(dReal x, dReal y, dReal z, const dQuaternion q)
    {
      // position torso
      dBodySetPosition(body_torso, x, y, z);

      //position head
      dReal headOffset = torsoDimensions[2]/2 + headRadius;
      dBodySetPosition(body_head, x, y, z + headOffset);

      //position upper arms
      dReal upperArmOffsetY = torsoDimensions[1]/2 + upperArmDimensions[1]/2;
      dReal upperArmOffsetZ = torsoDimensions[2]/2 - upperArmDimensions[2]/2;
      dBodySetPosition(body_leftupperarm,  x, y+upperArmOffsetY, z+upperArmOffsetZ);
      dBodySetPosition(body_rightupperarm, x, y-upperArmOffsetY, z+upperArmOffsetZ);

      //position lower arms
      dReal lowerArmOffsetY = torsoDimensions[1]/2 + lowerArmDimensions[1]/2;
      dReal lowerArmOffsetZ = torsoDimensions[2]/2 - upperArmDimensions[2] - lowerArmDimensions[2]/2;
      dBodySetPosition(body_leftlowerarm,  x, y+lowerArmOffsetY, z+lowerArmOffsetZ);
      dBodySetPosition(body_rightlowerarm, x, y-lowerArmOffsetY, z+lowerArmOffsetZ);

      //position upper legs
      dReal upperLegOffsetY =  torsoDimensions[1]/2 - upperLegDimensions[1]/2;
      dReal upperLegOffsetZ = -torsoDimensions[2]/2 - upperLegDimensions[2]/2;
      dBodySetPosition(body_leftupperleg,  x, y+upperLegOffsetY, z+upperLegOffsetZ);
      dBodySetPosition(body_rightupperleg, x, y-upperLegOffsetY, z+upperLegOffsetZ);

      //position lower legs
      dReal lowerLegOffsetY =  torsoDimensions[1]/2 - lowerLegDimensions[1]/2;
      dReal lowerLegOffsetZ = -torsoDimensions[2]/2 - upperLegDimensions[2] - lowerLegDimensions[2]/2;
      dBodySetPosition(body_leftlowerleg,  x, y+lowerLegOffsetY, z+lowerLegOffsetZ);
      dBodySetPosition(body_rightlowerleg, x, y-lowerLegOffsetY, z+lowerLegOffsetZ);

      //position feet
      dReal feetOffsetY =  torsoDimensions[1]/2 - footDimensions[1]/2;
      dReal feetOffsetZ = -torsoDimensions[2]/2 - upperLegDimensions[2] - lowerLegDimensions[2] - footDimensions[2]/2;
      dReal feetOffsetX =  lowerLegDimensions[0]/2;
      dBodySetPosition(body_leftfoot,  x+feetOffsetX, y+feetOffsetY, z+feetOffsetZ);
      dBodySetPosition(body_rightfoot, x+feetOffsetX, y-feetOffsetY, z+feetOffsetZ);

      dBodySetForce(body_torso, 0, 0, 0);
      dBodySetForce(body_head, 0, 0, 0);
      dBodySetForce(body_leftupperarm, 0, 0, 0);
      dBodySetForce(body_rightupperarm, 0, 0, 0);
      dBodySetForce(body_leftlowerarm, 0, 0, 0);
      dBodySetForce(body_rightlowerarm, 0, 0, 0);
      dBodySetForce(body_leftupperleg, 0, 0, 0);
      dBodySetForce(body_rightupperleg, 0, 0, 0);
      dBodySetForce(body_leftlowerleg, 0, 0, 0);
      dBodySetForce(body_rightlowerleg, 0, 0, 0);
      dBodySetForce(body_leftfoot, 0, 0, 0);
      dBodySetForce(body_rightfoot, 0, 0, 0);

      dBodySetLinearVel(body_torso, 0, 0, 0);
      dBodySetLinearVel(body_head, 0, 0, 0);
      dBodySetLinearVel(body_leftupperarm, 0, 0, 0);
      dBodySetLinearVel(body_rightupperarm, 0, 0, 0);
      dBodySetLinearVel(body_leftlowerarm, 0, 0, 0);
      dBodySetLinearVel(body_rightlowerarm, 0, 0, 0);
      dBodySetLinearVel(body_leftupperleg, 0, 0, 0);
      dBodySetLinearVel(body_rightupperleg, 0, 0, 0);
      dBodySetLinearVel(body_leftlowerleg, 0, 0, 0);
      dBodySetLinearVel(body_rightlowerleg, 0, 0, 0);
      dBodySetLinearVel(body_leftfoot, 0, 0, 0);
      dBodySetLinearVel(body_rightfoot, 0, 0, 0);

      dBodySetAngularVel(body_torso, 0, 0, 0);
      dBodySetAngularVel(body_head, 0, 0, 0);
      dBodySetAngularVel(body_leftupperarm, 0, 0, 0);
      dBodySetAngularVel(body_rightupperarm, 0, 0, 0);
      dBodySetAngularVel(body_leftlowerarm, 0, 0, 0);
      dBodySetAngularVel(body_rightlowerarm, 0, 0, 0);
      dBodySetAngularVel(body_leftupperleg, 0, 0, 0);
      dBodySetAngularVel(body_rightupperleg, 0, 0, 0);
      dBodySetAngularVel(body_leftlowerleg, 0, 0, 0);
      dBodySetAngularVel(body_rightlowerleg, 0, 0, 0);
      dBodySetAngularVel(body_leftfoot, 0, 0, 0);
      dBodySetAngularVel(body_rightfoot, 0, 0, 0);

      dBodySetQuaternion(body_torso, q);
      dBodySetQuaternion(body_head, q);
      dBodySetQuaternion(body_leftupperarm, q);
      dBodySetQuaternion(body_rightupperarm, q);
      dBodySetQuaternion(body_leftlowerarm, q);
      dBodySetQuaternion(body_rightlowerarm, q);
      dBodySetQuaternion(body_leftupperleg, q);
      dBodySetQuaternion(body_rightupperleg, q);
      dBodySetQuaternion(body_leftlowerleg, q);
      dBodySetQuaternion(body_rightlowerleg, q);
      dBodySetQuaternion(body_leftfoot, q);
      dBodySetQuaternion(body_rightfoot, q);
    }

    void Sustain(float dt)
    {
      SetTransformFromBody(trf_torso, body_torso);
      SetTransformFromBody(trf_head, body_head);
      SetTransformFromBody(trf_leftupperarm, body_leftupperarm);
      SetTransformFromBody(trf_rightupperarm, body_rightupperarm);
      SetTransformFromBody(trf_leftlowerarm, body_leftlowerarm);
      SetTransformFromBody(trf_rightlowerarm, body_rightlowerarm);
      SetTransformFromBody(trf_leftupperleg, body_leftupperleg);
      SetTransformFromBody(trf_rightupperleg, body_rightupperleg);
      SetTransformFromBody(trf_leftlowerleg, body_leftlowerleg);
      SetTransformFromBody(trf_rightlowerleg, body_rightlowerleg);
      SetTransformFromBody(trf_leftfoot, body_leftfoot);
      SetTransformFromBody(trf_rightfoot, body_rightfoot);
    }


    float NeckForce(void)
    {
      // See what force is exerted on the head
      dReal *f = neck_feedback.f1;
      return sqrt(f[0]*f[0]+f[1]*f[1]+f[2]*f[2]);
    }


    void Pull(float force)
    {
      dBodyAddForce(body_head, 0,0,force);
    }


    void FixFeet(dBodyID body)
    {
      dJointAttach(leftFootLock, body_leftfoot, body);
      dJointAttach(rightFootLock, body_rightfoot, body);
      dJointSetFixed(leftFootLock);
      dJointSetFixed(rightFootLock);
    }


    void ReleaseFeet(void)
    {
      dJointAttach(leftFootLock, 0, 0);
      dJointAttach(rightFootLock, 0, 0);
    }


    void FixTorso(dBodyID body)
    {
      dJointAttach(torsoLock, body_torso, body);
      dJointSetFixed(torsoLock);
    }


    void ReleaseTorso(void)
    {
      dJointAttach(torsoLock, 0, 0);
    }


    ssgTransform *GetTransform(void) { return trf_torso; }

    void GetPos(sgVec3 dst) const { sgMat4 m; trf_torso->getTransform(m); sgCopyVec3(dst, m[3]); }


  protected:

    dVector3 torsoDimensions;
    dVector3 upperArmDimensions;
    dVector3 lowerArmDimensions;
    dVector3 upperLegDimensions;
    dVector3 lowerLegDimensions;
    dVector3 footDimensions;
    dReal headRadius;

    dSpaceID biped_space;

    dGeomID geom_head;
    dGeomID geom_torso;
    dGeomID geom_leftupperarm;
    dGeomID geom_rightupperarm;
    dGeomID geom_leftupperleg;
    dGeomID geom_rightupperleg;
    dGeomID geom_leftlowerarm;
    dGeomID geom_rightlowerarm;
    dGeomID geom_leftlowerleg;
    dGeomID geom_rightlowerleg;
    dGeomID geom_leftfoot;
    dGeomID geom_rightfoot;

    dBodyID body_head;
    dBodyID body_torso;
    dBodyID body_leftupperarm;
    dBodyID body_rightupperarm;
    dBodyID body_leftupperleg;
    dBodyID body_rightupperleg;
    dBodyID body_leftlowerarm;
    dBodyID body_rightlowerarm;
    dBodyID body_leftlowerleg;
    dBodyID body_rightlowerleg;
    dBodyID body_leftfoot;
    dBodyID body_rightfoot;

    dJointID neck;
    dJointID leftElbow;
    dJointID rightElbow;
    dJointID leftHip;
    dJointID rightHip;
    dJointID leftShoulder;
    dJointID rightShoulder;
    dJointID leftAnkle;
    dJointID rightAnkle;
    dJointID leftKnee;
    dJointID rightKnee;

    dJointID leftFootLock;
    dJointID rightFootLock;
    dJointID torsoLock;

    dJointID motor_leftshoulder;
    dJointID motor_rightshoulder;

    dJointFeedback neck_feedback;

    // plib stuff
    ssgTransform *trf_torso;
    ssgTransform *trf_leftupperarm;
    ssgTransform *trf_rightupperarm;
    ssgTransform *trf_leftlowerarm;
    ssgTransform *trf_rightlowerarm;
    ssgTransform *trf_leftupperleg;
    ssgTransform *trf_rightupperleg;
    ssgTransform *trf_leftlowerleg;
    ssgTransform *trf_rightlowerleg;
    ssgTransform *trf_leftfoot;
    ssgTransform *trf_rightfoot;
    ssgTransform *trf_head;
};

#endif

