// -*- C++ -*-
/**
 * \file Package.h
 * This file is part of LyX, the document processor.
 * Licence details can be found in the file COPYING.
 *
 * \author Angus Leeming
 *
 * Full author contact details are available in file CREDITS.
 *
 * A store of the paths to the various different directories used
 * by LyX. These paths differ markedly from one OS to another,
 * following the local Windows, macOS or Posix conventions.
 */

#ifndef PACKAGE_H
#define PACKAGE_H

#include "support/FileName.h"

#include <string>

namespace lyx {
namespace support {

class Package;



/** Initialise package() with the command line data.
 *  This data is exactly as it was passed in the argv[] array.
 *
 *  @param command_line_arg0 argv[0], the name of the LyX executable
 *  as passed on the command line.
 *
 *  @param command_line_system_support_dir, the LyX system dir,
 *  as specified on the command line with "-sysdir <path to dir>".
 *
 *  @param command_line_user_support_dir, the LyX user dir,
 *  as specified on the command line with "-userdir <path to dir>".
 */
void init_package(std::string const & command_line_arg0,
		  std::string const & command_line_system_support_dir,
		  std::string const & command_line_user_support_dir);

/** Accessor to the global data.
 *  Asserts that init_package() has been called first.
 */
Package const & package();

class Package {
public:
	/// Default constructor does not lead to the paths being set.
	Package() : explicit_user_support_dir_(false), in_build_dir_(false) {}

	/** Called by init_package, above.
	 *  All paths will be initialized.
	 */
	Package(std::string const & command_line_arg0,
		std::string const & command_line_system_support_dir,
		std::string const & command_line_user_support_dir);

	/** The directory containing the main executable (LyX or tex2lyx).
	 */
	FileName const & binary_dir() const { return binary_dir_; }

	/** The absolute path to the LyX executable.
	 */
	FileName const & lyx_binary() const { return lyx_binary_; }

	/** The absolute path to the LyX package directory.
	 *  This is one level up from the binary dir.
	 */
	FileName const & lyx_dir() const { return lyx_dir_; }

	/** The top of the LyX source code tree.
	 */
	static FileName const & top_srcdir();

	/** The path to the system-level support files
	 *  we're actually going to use.
	 */
	FileName const & system_support() const { return system_support_dir_; }

	/** The path to the autogenerated support files
	 *  when running in-place.
	 */
	FileName const & build_support() const { return build_support_dir_; }

	/** The path to the user-level support files.
	 */
	FileName const & user_support() const { return user_support_dir_; }

	/** The user_support directory was set explicitly using either
	 *  the -userdir command line switch or
	 *  the LYX_USERDIR_${major}${minor}x environment variable.
	 */
	bool explicit_user_support() const { return explicit_user_support_dir_; }

	/** The path to the locale directory.
	 */
	FileName const & locale_dir() const { return locale_dir_; }

	/** The file name that should contain the message file (.mo)
	 *  for language code \param c. Does not check whether the
	 *  file exists. Handles running in place.
	 */
	FileName messages_file(std::string const & c) const;

	/** The default document directory.
	 *  Can be reset by LyXRC.
	 */
	FileName & document_dir() const { return document_dir_; }

	/** The path to the system temporary directory.
	 *  (Eg /tmp on *nix.)
	 */
	FileName const & system_temp_dir() const { return system_temp_dir_; }

	//@{
	/** The path to the temporary directory used by %LyX.
	 *  (Eg /tmp/lyx_tmpdir800nBI1z9 on *nix.)
	 *  Can be reset by LyXRC.
	 */
	FileName const & temp_dir() const { return temp_dir_; }
	void set_temp_dir(FileName const & temp_dir) const;
	//@}

	/** Used when setting the user_support directory.
	 *  Used also when expanding "~/" or contracting to "~/". (filetools.cpp)
	 *  Used in emergencyWrite (BufferList.cpp) as one possible location
	 *  for the dump.
	 *  This may be empty (e. g. when run under a CGI environment)
	 */
	static FileName const & get_home_dir();

	/// Run configure.py
	int reconfigureUserLyXDir(std::string const & option) const;

	///
	std::string getConfigureLockName() const;

private:
	FileName binary_dir_;
	FileName lyx_binary_;
	FileName lyx_dir_;
	FileName system_support_dir_;
	FileName build_support_dir_;
	FileName user_support_dir_;
	FileName locale_dir_;
	mutable FileName document_dir_;
	mutable FileName temp_dir_;
	FileName system_temp_dir_;
	/** Command to run the configure script.
	 *  Caution: This is "ready-to-run", i.e. in the locale encoding, not
	 *  utf8.
	 */
	mutable std::string configure_command_;
	bool explicit_user_support_dir_;
	bool in_build_dir_;
};

} // namespace support
} // namespace lyx

#endif // PACKAGE_H
