/**
 * Mandelbulber v2, a 3D fractal generator  _%}}i*<.         ______
 * Copyright (C) 2020 Mandelbulber Team   _>]|=||i=i<,      / ____/ __    __
 *                                        \><||i|=>>%)     / /   __/ /___/ /_
 * This file is part of Mandelbulber.     )<=i=]=|=i<>    / /__ /_  __/_  __/
 * The project is licensed under GPLv3,   -<>>=|><|||`    \____/ /_/   /_/
 * see also COPYING file in this folder.    ~+{i%+++
 *
 * Pseudo Kleinian Knighty - Theli-at's Pseudo Kleinian (Scale 1 JuliaBox + Something
 * @reference https://github.com/Syntopia/Fragmentarium/blob/master/
 * Fragmentarium-Source/Examples/Knighty%20Collection/PseudoKleinian.frag
 */

#include "all_fractal_definitions.h"

cFractalPseudoKleinian::cFractalPseudoKleinian() : cAbstractFractal()
{
	nameInComboBox = "Pseudo Kleinian";
	internalName = "pseudo_kleinian";
	internalID = fractal::pseudoKleinian;
	DEType = analyticDEType;
	DEFunctionType = pseudoKleinianDEFunction;
	cpixelAddition = cpixelDisabledByDefault;
	defaultBailout = 100.0;
	DEAnalyticFunction = analyticFunctionPseudoKleinian;
	coloringFunction = coloringFunctionDefault;
}

void cFractalPseudoKleinian::FormulaCode(CVector4 &z, const sFractal *fractal, sExtendedAux &aux)
{
	double oldZz = z.z;

	// sphere inversion slot#1 iter == 0 added v2.17
	if (fractal->transformCommon.sphereInversionEnabledFalse
		&& aux.i >= fractal->transformCommon.startIterationsX
		&& aux.i < fractal->transformCommon.stopIterations1)
	{
		double rr = 1.0;
		z += fractal->transformCommon.offset000;
		rr = z.Dot(z);
		z *= fractal->transformCommon.maxR2d1 / rr;
		z += fractal->transformCommon.additionConstantA000 - fractal->transformCommon.offset000;
		aux.DE = aux.DE * (fractal->transformCommon.maxR2d1 / rr) + fractal->analyticDE.offset0;
	}

	// prism shape
	if (fractal->transformCommon.functionEnabledPFalse
			&& aux.i >= fractal->transformCommon.startIterationsP
			&& aux.i < fractal->transformCommon.stopIterationsP1)
	{
		z.y = fabs(z.y);
		z.z = fabs(z.z);
		double dot1 = (z.x * -SQRT_3_4 + z.y * 0.5) * fractal->transformCommon.scale;
		double t = max(0.0, dot1);
		z.x -= t * -SQRT_3;
		z.y = fabs(z.y - t);
		if (z.y > z.z)
		{
			t = z.y;
			z.y = z.z;
			z.z = t;
		}
		z -= fractal->transformCommon.constantMultiplier000 * CVector4(SQRT_3_4, 1.5, 1.5, 0.0);
		// z was pos, now some points neg (ie neg shift)
		if (z.z > z.x)
		{
			t = z.z;
			z.z = z.x;
			z.x = t;
		}
		if (z.x > 0.0)
		{
			z.y = max(0.0, z.y);
			z.z = max(0.0, z.z);
		}
	}

	// box fold abs() tglad fold added v2.17
	if (fractal->transformCommon.functionEnabledByFalse
			&& aux.i >= fractal->transformCommon.startIterationsE
			&& aux.i < fractal->transformCommon.stopIterationsE)
	{
		z.x = fabs(z.x + fractal->transformCommon.additionConstant111.x)
					- fabs(z.x - fractal->transformCommon.additionConstant111.x) - z.x;
		z.y = fabs(z.y + fractal->transformCommon.additionConstant111.y)
					- fabs(z.y - fractal->transformCommon.additionConstant111.y) - z.y;
		if (fractal->transformCommon.functionEnabledBy)
		{
			z.z = fabs(z.z + fractal->transformCommon.additionConstant111.z)
						- fabs(z.z - fractal->transformCommon.additionConstant111.z) - z.z;
		}
	}

	// box fold
	if (fractal->transformCommon.functionEnabledBxFalse
			&& aux.i >= fractal->transformCommon.startIterationsA
			&& aux.i < fractal->transformCommon.stopIterationsA)
	{
		if (fabs(z.x) > fractal->mandelbox.foldingLimit)
		{
			z.x = sign(z.x) * fractal->mandelbox.foldingValue - z.x;
			aux.color += fractal->mandelbox.color.factor.x;
		}
		if (fabs(z.y) > fractal->mandelbox.foldingLimit)
		{
			z.y = sign(z.y) * fractal->mandelbox.foldingValue - z.y;
			aux.color += fractal->mandelbox.color.factor.y;
		}
		double zLimit = fractal->mandelbox.foldingLimit * fractal->transformCommon.scale1;
		double zValue = fractal->mandelbox.foldingValue * fractal->transformCommon.scale1;
		if (fabs(z.z) > zLimit)
		{
			z.z = sign(z.z) * zValue - z.z;
			aux.color += fractal->mandelbox.color.factor.z;
		}
	}

	// PseudoKleinian
	double k = 1.0;
	CVector4 cSize = fractal->transformCommon.additionConstant0777;
	if (aux.i >= fractal->transformCommon.startIterationsC
			&& aux.i < fractal->transformCommon.stopIterationsC)
	{
		CVector4 tempZ = z; //  correct c++ version. non conditional mult 2.0
		if (z.x > cSize.x) tempZ.x = cSize.x;
		if (z.x < -cSize.x) tempZ.x = -cSize.x;
		if (z.y > cSize.y) tempZ.y = cSize.y;
		if (z.y < -cSize.y) tempZ.y = -cSize.y;
		if (z.z > cSize.z) tempZ.z = cSize.z;
		if (z.z < -cSize.z) tempZ.z = -cSize.z;
		z = tempZ * 2.0 - z;
		k = max(fractal->transformCommon.minR05 / z.Dot(z), 1.0);
		z *= k;
		if (fractal->transformCommon.functionEnabledNFalse) z.z = -z.z;
		aux.DE *= k + fractal->analyticDE.tweak005;
		aux.pseudoKleinianDE = fractal->analyticDE.scale1;
	}

	// rotation
	if (fractal->transformCommon.functionEnabledRFalse
			&& aux.i >= fractal->transformCommon.startIterationsR
			&& aux.i < fractal->transformCommon.stopIterationsR)
		z = fractal->transformCommon.rotationMatrix.RotateVector(z);

	// offset
	z += fractal->transformCommon.additionConstant000;

	// color
	if (fractal->foldColor.auxColorEnabledFalse && aux.i >= fractal->foldColor.startIterationsA
		&& aux.i < fractal->foldColor.stopIterationsA)
	{
		double colorAdd = 0.0;
		colorAdd += fractal->foldColor.difs0000.x * k;
		colorAdd += fractal->foldColor.difs0000.y * fabs(z.z);
		colorAdd += fractal->foldColor.difs0000.z * fabs(z.z - oldZz);

		if (fractal->foldColor.auxColorEnabledAFalse
				&& aux.i >= fractal->transformCommon.startIterationsT
					&& aux.i < fractal->transformCommon.stopIterationsT)
		{
			double Size = 2.0 * cSize.x * fractal->transformCommon.constantMultiplier111.x;
			double bb = ((z.x + Size) / Size) + fractal->transformCommon.additionConstantP000.x;
			bb = fabs(bb - round(bb)) * fractal->transformCommon.constantMultiplierC111.x;
			double dd = ((aux.const_c.x + Size) / Size) + fractal->transformCommon.additionConstantP000.x;
			dd = fabs(dd - round(dd)) * fractal->transformCommon.constantMultiplierC111.x;

			Size = 2.0 * cSize.y * fractal->transformCommon.constantMultiplier111.y;
			double cc = ((z.y + Size) / Size) + fractal->transformCommon.additionConstantP000.y;
			cc = fabs(cc - round(cc)) * fractal->transformCommon.constantMultiplierC111.y;
			double ee = ((aux.const_c.y + Size) / Size) + fractal->transformCommon.additionConstantP000.y;
			ee = fabs(ee - round(ee)) * fractal->transformCommon.constantMultiplierC111.y;


			if (!fractal->transformCommon.functionEnabledAxFalse)
			{
				bb = bb + cc;
				dd = dd + ee;
			}
			else
			{
				bb = bb * bb + cc * cc;
				dd = dd * dd + ee * ee;
			}

			if (fractal->transformCommon.functionEnabledAFalse)
			{	Size = 2.0 * cSize.z * fractal->transformCommon.constantMultiplier111.z;
				double aa = ((z.z + Size) / Size) + fractal->transformCommon.additionConstantP000.z;
				aa = fabs(aa - round(aa)) * fractal->transformCommon.constantMultiplierC111.z;
				bb = bb + aa;
				double ff = ((aux.const_c.z + Size) / Size) + fractal->transformCommon.additionConstantP000.z;
				ff = fabs(ff - round(ff)) * fractal->transformCommon.constantMultiplierC111.z;
				dd = dd + ff;
			}
			bb = dd * (1.0 - fractal->foldColor.difs1) + bb * fractal->foldColor.difs1; // mix

			colorAdd += fractal->foldColor.difs0000.w * bb;
		}
		aux.color += colorAdd;
	}
}
