/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.routing;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.apache.lucene.util.CollectionUtil;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodes;
import org.elasticsearch.cluster.routing.IndexRoutingTable;
import org.elasticsearch.cluster.routing.IndexShardRoutingTable;
import org.elasticsearch.cluster.routing.RecoverySource;
import org.elasticsearch.cluster.routing.RelocationFailureInfo;
import org.elasticsearch.cluster.routing.RoutingChangesObserver;
import org.elasticsearch.cluster.routing.RoutingNode;
import org.elasticsearch.cluster.routing.RoutingTable;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.routing.UnassignedInfo;
import org.elasticsearch.cluster.routing.allocation.ExistingShardsAllocator;
import org.elasticsearch.common.collect.Iterators;
import org.elasticsearch.common.util.Maps;
import org.elasticsearch.core.Assertions;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.index.Index;
import org.elasticsearch.index.shard.ShardId;

public class RoutingNodes
implements Iterable<RoutingNode> {
    private final Map<String, RoutingNode> nodesToShards;
    private final UnassignedShards unassignedShards;
    private final Map<ShardId, List<ShardRouting>> assignedShards;
    private final boolean readOnly;
    private int inactivePrimaryCount = 0;
    private int inactiveShardCount = 0;
    private int relocatingShards = 0;
    private final Map<String, Set<String>> attributeValuesByAttribute;
    private final Map<String, Recoveries> recoveriesPerNode;
    private static final List<ShardRouting> EMPTY = Collections.emptyList();

    public static RoutingNodes immutable(RoutingTable routingTable, DiscoveryNodes discoveryNodes) {
        return new RoutingNodes(routingTable, discoveryNodes, true);
    }

    public static RoutingNodes mutable(RoutingTable routingTable, DiscoveryNodes discoveryNodes) {
        return new RoutingNodes(routingTable, discoveryNodes, false);
    }

    private RoutingNodes(RoutingTable routingTable, DiscoveryNodes discoveryNodes, boolean readOnly) {
        this.readOnly = readOnly;
        this.recoveriesPerNode = new HashMap<String, Recoveries>();
        int indexCount = routingTable.indicesRouting().size();
        this.assignedShards = Maps.newMapWithExpectedSize(indexCount);
        this.unassignedShards = new UnassignedShards(this);
        this.attributeValuesByAttribute = Collections.synchronizedMap(new HashMap());
        this.nodesToShards = Maps.newMapWithExpectedSize(discoveryNodes.getDataNodes().size());
        Set<String> dataNodes = discoveryNodes.getDataNodes().keySet();
        int sizeGuess = dataNodes.isEmpty() ? indexCount : 2 * indexCount / dataNodes.size();
        for (String node : discoveryNodes.getDataNodes().keySet()) {
            this.nodesToShards.put(node, new RoutingNode(node, discoveryNodes.get(node), sizeGuess));
        }
        Function<String, RoutingNode> createRoutingNode = k -> new RoutingNode((String)k, discoveryNodes.get((String)k), sizeGuess);
        for (IndexRoutingTable indexRoutingTable : routingTable.indicesRouting().values()) {
            for (int shardId = 0; shardId < indexRoutingTable.size(); ++shardId) {
                IndexShardRoutingTable indexShard = indexRoutingTable.shard(shardId);
                assert (indexShard.primary != null);
                for (int copy = 0; copy < indexShard.size(); ++copy) {
                    ShardRouting shard = indexShard.shard(copy);
                    if (shard.assignedToNode()) {
                        this.nodesToShards.computeIfAbsent(shard.currentNodeId(), createRoutingNode).addWithoutValidation(shard);
                        this.assignedShardsAdd(shard);
                        if (shard.relocating()) {
                            ++this.relocatingShards;
                            ShardRouting targetShardRouting = shard.getTargetRelocatingShard();
                            this.addInitialRecovery(targetShardRouting, indexShard.primary);
                            this.nodesToShards.computeIfAbsent(shard.relocatingNodeId(), createRoutingNode).addWithoutValidation(targetShardRouting);
                            this.assignedShardsAdd(targetShardRouting);
                            continue;
                        }
                        if (!shard.initializing()) continue;
                        if (shard.primary()) {
                            ++this.inactivePrimaryCount;
                        }
                        ++this.inactiveShardCount;
                        this.addInitialRecovery(shard, indexShard.primary);
                        continue;
                    }
                    this.unassignedShards.add(shard);
                }
            }
        }
        assert (this.invariant());
    }

    private boolean invariant() {
        this.nodesToShards.values().forEach(RoutingNode::invariant);
        return true;
    }

    private RoutingNodes(RoutingNodes routingNodes) {
        assert (routingNodes.readOnly) : "tried to create a mutable copy from a mutable instance";
        this.readOnly = false;
        this.nodesToShards = Maps.copyOf(routingNodes.nodesToShards, RoutingNode::copy);
        this.assignedShards = Maps.copyOf(routingNodes.assignedShards, ArrayList::new);
        this.unassignedShards = routingNodes.unassignedShards.copyFor(this);
        this.inactivePrimaryCount = routingNodes.inactivePrimaryCount;
        this.inactiveShardCount = routingNodes.inactiveShardCount;
        this.relocatingShards = routingNodes.relocatingShards;
        this.attributeValuesByAttribute = Collections.synchronizedMap(Maps.copyOf(routingNodes.attributeValuesByAttribute, HashSet::new));
        this.recoveriesPerNode = Maps.copyOf(routingNodes.recoveriesPerNode, Recoveries::copy);
    }

    public RoutingNodes mutableCopy() {
        return new RoutingNodes(this);
    }

    private void addRecovery(ShardRouting routing) {
        this.updateRecoveryCounts(routing, true, this.findAssignedPrimaryIfPeerRecovery(routing));
    }

    private void removeRecovery(ShardRouting routing) {
        this.updateRecoveryCounts(routing, false, this.findAssignedPrimaryIfPeerRecovery(routing));
    }

    private void addInitialRecovery(ShardRouting routing, ShardRouting initialPrimaryShard) {
        this.updateRecoveryCounts(routing, true, initialPrimaryShard);
    }

    private void updateRecoveryCounts(ShardRouting routing, boolean increment, @Nullable ShardRouting primary) {
        int howMany;
        int n = howMany = increment ? 1 : -1;
        assert (routing.initializing()) : "routing must be initializing: " + String.valueOf(routing);
        assert (primary == null || primary.assignedToNode()) : "shard is initializing but its primary is not assigned to a node";
        Recoveries.getOrAdd(this.recoveriesPerNode, routing.currentNodeId()).addIncoming(howMany);
        if (routing.recoverySource().getType() == RecoverySource.Type.PEER) {
            if (primary == null) {
                throw new IllegalStateException("shard [" + String.valueOf(routing) + "] is peer recovering but primary is unassigned");
            }
            Recoveries.getOrAdd(this.recoveriesPerNode, primary.currentNodeId()).addOutgoing(howMany);
            if (!increment && routing.primary() && routing.relocatingNodeId() != null) {
                int numRecoveringReplicas = 0;
                for (ShardRouting assigned : this.assignedShards(routing.shardId())) {
                    if (assigned.primary() || !assigned.initializing() || assigned.recoverySource().getType() != RecoverySource.Type.PEER) continue;
                    ++numRecoveringReplicas;
                }
                this.recoveriesPerNode.get(routing.relocatingNodeId()).addOutgoing(-numRecoveringReplicas);
                this.recoveriesPerNode.get(routing.currentNodeId()).addOutgoing(numRecoveringReplicas);
            }
        }
    }

    public int getIncomingRecoveries(String nodeId) {
        return this.recoveriesPerNode.getOrDefault(nodeId, Recoveries.EMPTY).getIncoming();
    }

    public int getOutgoingRecoveries(String nodeId) {
        return this.recoveriesPerNode.getOrDefault(nodeId, Recoveries.EMPTY).getOutgoing();
    }

    @Nullable
    private ShardRouting findAssignedPrimaryIfPeerRecovery(ShardRouting routing) {
        List<ShardRouting> shardRoutings;
        ShardRouting primary = null;
        if (routing.recoverySource() != null && routing.recoverySource().getType() == RecoverySource.Type.PEER && (shardRoutings = this.assignedShards.get(routing.shardId())) != null) {
            for (ShardRouting shardRouting : shardRoutings) {
                if (!shardRouting.primary()) continue;
                if (shardRouting.active()) {
                    return shardRouting;
                }
                if (primary == null) {
                    primary = shardRouting;
                    continue;
                }
                if (primary.relocatingNodeId() == null) continue;
                primary = shardRouting;
            }
        }
        return primary;
    }

    public Set<String> getAllNodeIds() {
        return Collections.unmodifiableSet(this.nodesToShards.keySet());
    }

    @Override
    public Iterator<RoutingNode> iterator() {
        return Collections.unmodifiableCollection(this.nodesToShards.values()).iterator();
    }

    public Stream<RoutingNode> stream() {
        return this.nodesToShards.values().stream();
    }

    public Iterator<RoutingNode> mutableIterator() {
        this.ensureMutable();
        return this.nodesToShards.values().iterator();
    }

    public UnassignedShards unassigned() {
        return this.unassignedShards;
    }

    public RoutingNode node(String nodeId) {
        return this.nodesToShards.get(nodeId);
    }

    public Set<String> getAttributeValues(String attributeName) {
        return this.attributeValuesByAttribute.computeIfAbsent(attributeName, ignored -> this.stream().map(r -> r.node().getAttributes().get(attributeName)).filter(Objects::nonNull).collect(Collectors.toSet()));
    }

    public boolean hasUnassignedPrimaries() {
        return this.unassignedShards.getNumPrimaries() + this.unassignedShards.getNumIgnoredPrimaries() > 0;
    }

    public boolean hasUnassignedShards() {
        return !this.unassignedShards.isEmpty() || !this.unassignedShards.isIgnoredEmpty();
    }

    public boolean hasInactivePrimaries() {
        return this.inactivePrimaryCount > 0;
    }

    public boolean hasInactiveReplicas() {
        return this.inactiveShardCount > this.inactivePrimaryCount;
    }

    public boolean hasInactiveShards() {
        return this.inactiveShardCount > 0;
    }

    public int getRelocatingShardCount() {
        return this.relocatingShards;
    }

    public List<ShardRouting> assignedShards(ShardId shardId) {
        List<ShardRouting> replicaSet = this.assignedShards.get(shardId);
        return replicaSet == null ? EMPTY : Collections.unmodifiableList(replicaSet);
    }

    @Nullable
    public ShardRouting getByAllocationId(ShardId shardId, String allocationId) {
        List<ShardRouting> replicaSet = this.assignedShards.get(shardId);
        if (replicaSet == null) {
            return null;
        }
        for (ShardRouting shardRouting : replicaSet) {
            if (!shardRouting.allocationId().getId().equals(allocationId)) continue;
            return shardRouting;
        }
        return null;
    }

    public ShardRouting activePrimary(ShardId shardId) {
        for (ShardRouting shardRouting : this.assignedShards(shardId)) {
            if (!shardRouting.primary() || !shardRouting.active()) continue;
            return shardRouting;
        }
        return null;
    }

    public ShardRouting activePromotableReplicaWithHighestVersion(ShardId shardId) {
        return this.assignedShards(shardId).stream().filter(shr -> !shr.primary() && shr.active()).filter(shr -> this.node(shr.currentNodeId()) != null).filter(ShardRouting::isPromotableToPrimary).max(Comparator.comparing(shr -> this.node(shr.currentNodeId()).node(), Comparator.nullsFirst(Comparator.comparing(DiscoveryNode::getVersion)))).orElse(null);
    }

    public boolean allShardsActive(ShardId shardId, Metadata metadata) {
        List<ShardRouting> shards = this.assignedShards(shardId);
        int shardCopies = metadata.getIndexSafe(shardId.getIndex()).getNumberOfReplicas() + 1;
        if (shards.size() < shardCopies) {
            return false;
        }
        int active = 0;
        for (ShardRouting shard : shards) {
            if (!shard.active()) continue;
            ++active;
        }
        assert (active <= shardCopies);
        return active == shardCopies;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder("routing_nodes:\n");
        for (RoutingNode routingNode : this) {
            sb.append(routingNode.prettyPrint());
        }
        sb.append("---- unassigned\n");
        for (ShardRouting shardEntry : this.unassignedShards) {
            sb.append("--------").append(shardEntry.shortSummary()).append('\n');
        }
        return sb.toString();
    }

    public ShardRouting initializeShard(ShardRouting unassignedShard, String nodeId, @Nullable String existingAllocationId, long expectedSize, RoutingChangesObserver routingChangesObserver) {
        this.ensureMutable();
        assert (unassignedShard.unassigned()) : "expected an unassigned shard " + String.valueOf(unassignedShard);
        ShardRouting initializedShard = unassignedShard.initialize(nodeId, existingAllocationId, expectedSize);
        this.node(nodeId).add(initializedShard);
        ++this.inactiveShardCount;
        if (initializedShard.primary()) {
            ++this.inactivePrimaryCount;
        }
        this.addRecovery(initializedShard);
        this.assignedShardsAdd(initializedShard);
        routingChangesObserver.shardInitialized(unassignedShard, initializedShard);
        return initializedShard;
    }

    public Tuple<ShardRouting, ShardRouting> relocateShard(ShardRouting startedShard, String nodeId, long expectedShardSize, String reason, RoutingChangesObserver changes) {
        this.ensureMutable();
        ++this.relocatingShards;
        ShardRouting source = startedShard.relocate(nodeId, expectedShardSize);
        ShardRouting target = source.getTargetRelocatingShard();
        this.updateAssigned(startedShard, source);
        this.node(target.currentNodeId()).add(target);
        this.assignedShardsAdd(target);
        this.addRecovery(target);
        changes.relocationStarted(startedShard, target, reason);
        return Tuple.tuple((Object)source, (Object)target);
    }

    public ShardRouting startShard(ShardRouting initializingShard, RoutingChangesObserver routingChangesObserver, long startedExpectedShardSize) {
        this.ensureMutable();
        ShardRouting startedShard = this.started(initializingShard, startedExpectedShardSize);
        routingChangesObserver.shardStarted(initializingShard, startedShard);
        if (initializingShard.relocatingNodeId() != null) {
            RoutingNode relocationSourceNode = this.node(initializingShard.relocatingNodeId());
            ShardRouting relocationSourceShard = relocationSourceNode.getByShardId(initializingShard.shardId());
            assert (relocationSourceShard.isRelocationSourceOf(initializingShard));
            assert (relocationSourceShard.getTargetRelocatingShard() == initializingShard) : "relocation target mismatch, expected: " + String.valueOf(initializingShard) + " but was: " + String.valueOf(relocationSourceShard.getTargetRelocatingShard());
            this.remove(relocationSourceShard);
            routingChangesObserver.relocationCompleted(relocationSourceShard);
            if (startedShard.primary()) {
                List<ShardRouting> assignedShards = this.assignedShards(startedShard.shardId());
                for (ShardRouting routing : new ArrayList<ShardRouting>(assignedShards)) {
                    if (!routing.initializing() || routing.primary()) continue;
                    if (routing.isRelocationTarget()) {
                        ShardRouting sourceShard = this.getByAllocationId(routing.shardId(), routing.allocationId().getRelocationId());
                        ShardRouting startedReplica = this.cancelRelocation(sourceShard);
                        this.remove(routing);
                        routingChangesObserver.shardFailed(routing, new UnassignedInfo(UnassignedInfo.Reason.REINITIALIZED, "primary changed"));
                        this.relocateShard(startedReplica, sourceShard.relocatingNodeId(), sourceShard.getExpectedShardSize(), "restarting relocation", routingChangesObserver);
                        continue;
                    }
                    ShardRouting reinitializedReplica = this.reinitReplica(routing);
                    routingChangesObserver.initializedReplicaReinitialized(routing, reinitializedReplica);
                }
            }
        }
        return startedShard;
    }

    public void failShard(ShardRouting failedShard, UnassignedInfo unassignedInfo, RoutingChangesObserver routingChangesObserver) {
        List<ShardRouting> assignedShards;
        this.ensureMutable();
        assert (failedShard.assignedToNode()) : "only assigned shards can be failed";
        assert (this.getByAllocationId(failedShard.shardId(), failedShard.allocationId().getId()) == failedShard) : "shard routing to fail does not exist in routing table, expected: " + String.valueOf(failedShard) + " but was: " + String.valueOf(this.getByAllocationId(failedShard.shardId(), failedShard.allocationId().getId()));
        if (failedShard.primary() && !(assignedShards = this.assignedShards(failedShard.shardId())).isEmpty()) {
            for (ShardRouting routing : new ArrayList<ShardRouting>(assignedShards)) {
                if (routing.primary() || !routing.initializing()) continue;
                ShardRouting replicaShard = this.getByAllocationId(routing.shardId(), routing.allocationId().getId());
                assert (replicaShard != null) : "failed to re-resolve " + String.valueOf(routing) + " when failing replicas";
                UnassignedInfo primaryFailedUnassignedInfo = new UnassignedInfo(UnassignedInfo.Reason.PRIMARY_FAILED, "primary failed while replica initializing", null, 0, unassignedInfo.unassignedTimeNanos(), unassignedInfo.unassignedTimeMillis(), false, UnassignedInfo.AllocationStatus.NO_ATTEMPT, Collections.emptySet(), routing.currentNodeId());
                this.failShard(replicaShard, primaryFailedUnassignedInfo, routingChangesObserver);
            }
        }
        if (failedShard.relocating()) {
            ShardRouting targetShard = this.getByAllocationId(failedShard.shardId(), failedShard.allocationId().getRelocationId());
            assert (targetShard.isRelocationTargetOf(failedShard));
            if (failedShard.primary()) {
                this.remove(targetShard);
                routingChangesObserver.shardFailed(targetShard, unassignedInfo);
            } else {
                this.removeRelocationSource(targetShard);
                routingChangesObserver.relocationSourceRemoved(targetShard);
            }
        }
        if (failedShard.initializing()) {
            if (failedShard.relocatingNodeId() == null) {
                if (failedShard.primary()) {
                    this.unassignPrimaryAndPromoteActiveReplicaIfExists(failedShard, unassignedInfo, routingChangesObserver);
                } else {
                    this.moveToUnassigned(failedShard, unassignedInfo);
                }
            } else {
                ShardRouting sourceShard = this.getByAllocationId(failedShard.shardId(), failedShard.allocationId().getRelocationId());
                assert (sourceShard.isRelocationSourceOf(failedShard));
                this.cancelRelocation(sourceShard);
                this.remove(failedShard);
            }
        } else {
            assert (failedShard.active());
            if (failedShard.primary()) {
                this.unassignPrimaryAndPromoteActiveReplicaIfExists(failedShard, unassignedInfo, routingChangesObserver);
            } else if (failedShard.relocating()) {
                this.remove(failedShard);
            } else {
                this.moveToUnassigned(failedShard, unassignedInfo);
            }
        }
        routingChangesObserver.shardFailed(failedShard, unassignedInfo);
        assert (this.node(failedShard.currentNodeId()).getByShardId(failedShard.shardId()) == null) : "failedShard " + String.valueOf(failedShard) + " was matched but wasn't removed";
    }

    private void unassignPrimaryAndPromoteActiveReplicaIfExists(ShardRouting failedPrimary, UnassignedInfo unassignedInfo, RoutingChangesObserver routingChangesObserver) {
        assert (failedPrimary.primary());
        ShardRouting replicaToPromote = this.activePromotableReplicaWithHighestVersion(failedPrimary.shardId());
        if (replicaToPromote == null) {
            this.moveToUnassigned(failedPrimary, unassignedInfo);
            for (ShardRouting unpromotableReplica : List.copyOf(this.assignedShards(failedPrimary.shardId()))) {
                assert (!unpromotableReplica.primary()) : unpromotableReplica;
                assert (!unpromotableReplica.isPromotableToPrimary()) : unpromotableReplica;
                this.moveToUnassigned(unpromotableReplica, new UnassignedInfo(UnassignedInfo.Reason.UNPROMOTABLE_REPLICA, unassignedInfo.message(), unassignedInfo.failure(), 0, unassignedInfo.unassignedTimeNanos(), unassignedInfo.unassignedTimeMillis(), false, UnassignedInfo.AllocationStatus.NO_ATTEMPT, Set.of(), unpromotableReplica.currentNodeId()));
            }
        } else {
            this.movePrimaryToUnassignedAndDemoteToReplica(failedPrimary, unassignedInfo);
            this.promoteReplicaToPrimary(replicaToPromote, routingChangesObserver);
        }
    }

    private void promoteReplicaToPrimary(ShardRouting activeReplica, RoutingChangesObserver routingChangesObserver) {
        assert (activeReplica.started()) : "replica relocation should have been cancelled: " + String.valueOf(activeReplica);
        this.promoteActiveReplicaShardToPrimary(activeReplica);
        routingChangesObserver.replicaPromoted(activeReplica);
    }

    private ShardRouting started(ShardRouting shard, long expectedShardSize) {
        assert (shard.initializing()) : "expected an initializing shard " + String.valueOf(shard);
        if (shard.relocatingNodeId() == null) {
            --this.inactiveShardCount;
            if (shard.primary()) {
                --this.inactivePrimaryCount;
            }
        }
        this.removeRecovery(shard);
        ShardRouting startedShard = shard.moveToStarted(expectedShardSize);
        this.updateAssigned(shard, startedShard);
        return startedShard;
    }

    private ShardRouting cancelRelocation(ShardRouting shard) {
        --this.relocatingShards;
        ShardRouting cancelledShard = shard.cancelRelocation();
        this.updateAssigned(shard, cancelledShard);
        return cancelledShard;
    }

    private ShardRouting promoteActiveReplicaShardToPrimary(ShardRouting replicaShard) {
        assert (replicaShard.active()) : "non-active shard cannot be promoted to primary: " + String.valueOf(replicaShard);
        assert (!replicaShard.primary()) : "primary shard cannot be promoted to primary: " + String.valueOf(replicaShard);
        ShardRouting primaryShard = replicaShard.moveActiveReplicaToPrimary();
        this.updateAssigned(replicaShard, primaryShard);
        return primaryShard;
    }

    private void remove(ShardRouting shard) {
        assert (!shard.unassigned()) : "only assigned shards can be removed here (" + String.valueOf(shard) + ")";
        this.node(shard.currentNodeId()).remove(shard);
        if (shard.initializing() && shard.relocatingNodeId() == null) {
            --this.inactiveShardCount;
            assert (this.inactiveShardCount >= 0);
            if (shard.primary()) {
                --this.inactivePrimaryCount;
            }
        } else if (shard.relocating()) {
            shard = this.cancelRelocation(shard);
        }
        this.assignedShardsRemove(shard);
        if (shard.initializing()) {
            this.removeRecovery(shard);
        }
    }

    private ShardRouting removeRelocationSource(ShardRouting shard) {
        assert (shard.isRelocationTarget()) : "only relocation target shards can have their relocation source removed (" + String.valueOf(shard) + ")";
        ShardRouting relocationMarkerRemoved = shard.removeRelocationSource();
        this.updateAssigned(shard, relocationMarkerRemoved);
        ++this.inactiveShardCount;
        return relocationMarkerRemoved;
    }

    private void assignedShardsAdd(ShardRouting shard) {
        assert (!shard.unassigned()) : "unassigned shard " + String.valueOf(shard) + " cannot be added to list of assigned shards";
        List shards = this.assignedShards.computeIfAbsent(shard.shardId(), k -> new ArrayList());
        assert (RoutingNodes.assertInstanceNotInList(shard, shards)) : "shard " + String.valueOf(shard) + " cannot appear twice in list of assigned shards";
        shards.add(shard);
    }

    private static boolean assertInstanceNotInList(ShardRouting shard, List<ShardRouting> shards) {
        for (ShardRouting s : shards) {
            assert (s != shard);
        }
        return true;
    }

    private void assignedShardsRemove(ShardRouting shard) {
        List<ShardRouting> replicaSet = this.assignedShards.get(shard.shardId());
        if (replicaSet != null) {
            Iterator<ShardRouting> iterator = replicaSet.iterator();
            while (iterator.hasNext()) {
                if (shard != iterator.next()) continue;
                iterator.remove();
                return;
            }
        }
        assert (false) : "No shard found to remove";
    }

    private ShardRouting reinitReplica(ShardRouting shard) {
        assert (!shard.primary()) : "shard must be a replica: " + String.valueOf(shard);
        assert (shard.initializing()) : "can only reinitialize an initializing replica: " + String.valueOf(shard);
        assert (!shard.isRelocationTarget()) : "replication target cannot be reinitialized: " + String.valueOf(shard);
        ShardRouting reinitializedShard = shard.reinitializeReplicaShard();
        this.updateAssigned(shard, reinitializedShard);
        return reinitializedShard;
    }

    private void updateAssigned(ShardRouting oldShard, ShardRouting newShard) {
        assert (oldShard.shardId().equals(newShard.shardId())) : "can only update " + String.valueOf(oldShard) + " by shard with same shard id but was " + String.valueOf(newShard);
        assert (!oldShard.unassigned() && !newShard.unassigned()) : "only assigned shards can be updated in list of assigned shards (prev: " + String.valueOf(oldShard) + ", new: " + String.valueOf(newShard) + ")";
        assert (oldShard.currentNodeId().equals(newShard.currentNodeId())) : "shard to update " + String.valueOf(oldShard) + " can only update " + String.valueOf(oldShard) + " by shard assigned to same node but was " + String.valueOf(newShard);
        this.node(oldShard.currentNodeId()).update(oldShard, newShard);
        List shardsWithMatchingShardId = this.assignedShards.computeIfAbsent(oldShard.shardId(), k -> new ArrayList());
        int previousShardIndex = shardsWithMatchingShardId.indexOf(oldShard);
        assert (previousShardIndex >= 0) : "shard to update " + String.valueOf(oldShard) + " does not exist in list of assigned shards";
        shardsWithMatchingShardId.set(previousShardIndex, newShard);
    }

    private ShardRouting moveToUnassigned(ShardRouting shard, UnassignedInfo unassignedInfo) {
        assert (!shard.unassigned()) : "only assigned shards can be moved to unassigned (" + String.valueOf(shard) + ")";
        this.remove(shard);
        ShardRouting unassigned = shard.moveToUnassigned(unassignedInfo);
        this.unassignedShards.add(unassigned);
        return unassigned;
    }

    private ShardRouting movePrimaryToUnassignedAndDemoteToReplica(ShardRouting shard, UnassignedInfo unassignedInfo) {
        assert (!shard.unassigned()) : "only assigned shards can be moved to unassigned (" + String.valueOf(shard) + ")";
        assert (shard.primary()) : "only primary can be demoted to replica (" + String.valueOf(shard) + ")";
        this.remove(shard);
        ShardRouting unassigned = shard.moveToUnassigned(unassignedInfo).moveUnassignedFromPrimary();
        this.unassignedShards.add(unassigned);
        return unassigned;
    }

    public int size() {
        return this.nodesToShards.size();
    }

    public Map<ShardId, List<ShardRouting>> getAssignedShards() {
        return Collections.unmodifiableMap(this.assignedShards);
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        RoutingNodes that = (RoutingNodes)o;
        return this.readOnly == that.readOnly && this.inactivePrimaryCount == that.inactivePrimaryCount && this.inactiveShardCount == that.inactiveShardCount && this.relocatingShards == that.relocatingShards && this.nodesToShards.equals(that.nodesToShards) && this.unassignedShards.equals(that.unassignedShards) && this.assignedShards.equals(that.assignedShards) && this.attributeValuesByAttribute.equals(that.attributeValuesByAttribute) && this.recoveriesPerNode.equals(that.recoveriesPerNode);
    }

    public int hashCode() {
        return Objects.hash(this.nodesToShards, this.unassignedShards, this.assignedShards, this.readOnly, this.inactivePrimaryCount, this.inactiveShardCount, this.relocatingShards, this.attributeValuesByAttribute, this.recoveriesPerNode);
    }

    public static boolean assertShardStats(RoutingNodes routingNodes) {
        if (!Assertions.ENABLED) {
            return true;
        }
        int unassignedPrimaryCount = 0;
        int unassignedIgnoredPrimaryCount = 0;
        int inactivePrimaryCount = 0;
        int inactiveShardCount = 0;
        int relocating = 0;
        HashMap<Index, Integer> indicesAndShards = new HashMap<Index, Integer>();
        for (RoutingNode node : routingNodes) {
            for (ShardRouting shardRouting : node) {
                Object i;
                if (shardRouting.initializing() && shardRouting.relocatingNodeId() == null) {
                    ++inactiveShardCount;
                    if (shardRouting.primary()) {
                        ++inactivePrimaryCount;
                    }
                }
                if (shardRouting.relocating()) {
                    ++relocating;
                }
                if ((i = (Integer)indicesAndShards.get(shardRouting.index())) == null) {
                    i = shardRouting.id();
                }
                indicesAndShards.put(shardRouting.index(), Math.max((Integer)i, shardRouting.id()));
            }
        }
        Set entries = indicesAndShards.entrySet();
        HashMap<ShardId, HashSet> shardsByShardId = new HashMap<ShardId, HashSet>();
        for (RoutingNode routingNode : routingNodes) {
            for (ShardRouting shardRouting : routingNode) {
                HashSet shards = shardsByShardId.computeIfAbsent(new ShardId(shardRouting.index(), shardRouting.id()), k -> new HashSet());
                shards.add(shardRouting);
            }
        }
        for (Map.Entry entry : entries) {
            Index index = (Index)entry.getKey();
            for (int i = 0; i <= (Integer)entry.getValue(); ++i) {
                ShardId shardId = new ShardId(index, i);
                HashSet shards = (HashSet)shardsByShardId.get(shardId);
                List<ShardRouting> mutableShardRoutings = routingNodes.assignedShards(shardId);
                assert (shards == null && mutableShardRoutings.size() == 0 || shards != null && shards.size() == mutableShardRoutings.size() && shards.containsAll(mutableShardRoutings));
            }
        }
        for (ShardRouting shardRouting : routingNodes.unassigned()) {
            if (!shardRouting.primary()) continue;
            ++unassignedPrimaryCount;
        }
        for (ShardRouting shardRouting : routingNodes.unassigned().ignored()) {
            if (!shardRouting.primary()) continue;
            ++unassignedIgnoredPrimaryCount;
        }
        for (Map.Entry entry : routingNodes.recoveriesPerNode.entrySet()) {
            String node = (String)entry.getKey();
            Recoveries value = (Recoveries)entry.getValue();
            int incoming = 0;
            int outgoing = 0;
            RoutingNode routingNode = routingNodes.nodesToShards.get(node);
            if (routingNode != null) {
                for (ShardRouting routing : routingNode) {
                    if (routing.initializing()) {
                        ++incoming;
                    }
                    if (!routing.primary() || routing.isRelocationTarget()) continue;
                    for (ShardRouting assigned : routingNodes.assignedShards.get(routing.shardId())) {
                        if (!assigned.initializing() || assigned.recoverySource().getType() != RecoverySource.Type.PEER) continue;
                        ++outgoing;
                    }
                }
            }
            assert (incoming == value.incoming) : incoming + " != " + value.incoming + " node: " + String.valueOf(routingNode);
            assert (outgoing == value.outgoing) : outgoing + " != " + value.outgoing + " node: " + String.valueOf(routingNode);
        }
        assert (unassignedPrimaryCount == routingNodes.unassignedShards.getNumPrimaries()) : "Unassigned primaries is [" + unassignedPrimaryCount + "] but RoutingNodes returned unassigned primaries [" + routingNodes.unassigned().getNumPrimaries() + "]";
        assert (unassignedIgnoredPrimaryCount == routingNodes.unassignedShards.getNumIgnoredPrimaries()) : "Unassigned ignored primaries is [" + unassignedIgnoredPrimaryCount + "] but RoutingNodes returned unassigned ignored primaries [" + routingNodes.unassigned().getNumIgnoredPrimaries() + "]";
        assert (inactivePrimaryCount == routingNodes.inactivePrimaryCount) : "Inactive Primary count [" + inactivePrimaryCount + "] but RoutingNodes returned inactive primaries [" + routingNodes.inactivePrimaryCount + "]";
        assert (inactiveShardCount == routingNodes.inactiveShardCount) : "Inactive Shard count [" + inactiveShardCount + "] but RoutingNodes returned inactive shards [" + routingNodes.inactiveShardCount + "]";
        assert (routingNodes.getRelocatingShardCount() == relocating) : "Relocating shards mismatch [" + routingNodes.getRelocatingShardCount() + "] but expected [" + relocating + "]";
        return true;
    }

    private void ensureMutable() {
        if (this.readOnly) {
            throw new IllegalStateException("can't modify RoutingNodes - readonly");
        }
    }

    public boolean hasAllocationFailures() {
        return this.unassignedShards.stream().anyMatch(shardRouting -> {
            if (shardRouting.unassignedInfo() == null) {
                return false;
            }
            return shardRouting.unassignedInfo().failedAllocations() > 0;
        });
    }

    public void resetFailedCounter(RoutingChangesObserver routingChangesObserver) {
        UnassignedShards.UnassignedIterator unassignedIterator = this.unassigned().iterator();
        while (unassignedIterator.hasNext()) {
            ShardRouting shardRouting = unassignedIterator.next();
            UnassignedInfo unassignedInfo = shardRouting.unassignedInfo();
            unassignedIterator.updateUnassigned(new UnassignedInfo(unassignedInfo.failedAllocations() > 0 ? UnassignedInfo.Reason.MANUAL_ALLOCATION : unassignedInfo.reason(), unassignedInfo.message(), unassignedInfo.failure(), 0, unassignedInfo.unassignedTimeNanos(), unassignedInfo.unassignedTimeMillis(), unassignedInfo.delayed(), unassignedInfo.lastAllocationStatus(), Collections.emptySet(), unassignedInfo.lastAllocatedNodeId()), shardRouting.recoverySource(), routingChangesObserver);
        }
        for (RoutingNode routingNode : this) {
            ArrayList<ShardRouting> shardsWithRelocationFailures = new ArrayList<ShardRouting>();
            for (ShardRouting shardRouting : routingNode) {
                if (shardRouting.relocationFailureInfo() == null || shardRouting.relocationFailureInfo().failedRelocations() <= 0) continue;
                shardsWithRelocationFailures.add(shardRouting);
            }
            for (ShardRouting original : shardsWithRelocationFailures) {
                ShardRouting updated = original.updateRelocationFailure(RelocationFailureInfo.NO_FAILURES);
                routingNode.update(original, updated);
                this.assignedShardsRemove(original);
                this.assignedShardsAdd(updated);
            }
        }
    }

    public Iterator<ShardRouting> nodeInterleavedShardIterator() {
        final ArrayDeque<Iterator<ShardRouting>> queue = new ArrayDeque<Iterator<ShardRouting>>(this.nodesToShards.size());
        for (RoutingNode routingNode : this.nodesToShards.values()) {
            ShardRouting[] shards = routingNode.copyShards();
            if (shards.length <= 0) continue;
            queue.add(Iterators.forArray(shards));
        }
        return new Iterator<ShardRouting>(){

            @Override
            public boolean hasNext() {
                return !queue.isEmpty();
            }

            @Override
            public ShardRouting next() {
                if (queue.isEmpty()) {
                    throw new NoSuchElementException();
                }
                Iterator nodeIterator = (Iterator)queue.poll();
                assert (nodeIterator.hasNext());
                ShardRouting nextShard = (ShardRouting)nodeIterator.next();
                if (nodeIterator.hasNext()) {
                    queue.offer(nodeIterator);
                }
                return nextShard;
            }
        };
    }

    public static final class UnassignedShards
    implements Iterable<ShardRouting> {
        private final RoutingNodes nodes;
        private final List<ShardRouting> unassigned;
        private final List<ShardRouting> ignored;
        private int primaries;
        private int ignoredPrimaries;

        public UnassignedShards(RoutingNodes nodes) {
            this(nodes, new ArrayList<ShardRouting>(), new ArrayList<ShardRouting>(), 0, 0);
        }

        private UnassignedShards(RoutingNodes nodes, List<ShardRouting> unassigned, List<ShardRouting> ignored, int primaries, int ignoredPrimaries) {
            this.nodes = nodes;
            this.unassigned = unassigned;
            this.ignored = ignored;
            this.primaries = primaries;
            this.ignoredPrimaries = ignoredPrimaries;
        }

        public UnassignedShards copyFor(RoutingNodes newNodes) {
            return new UnassignedShards(newNodes, new ArrayList<ShardRouting>(this.unassigned), new ArrayList<ShardRouting>(this.ignored), this.primaries, this.ignoredPrimaries);
        }

        public void add(ShardRouting shardRouting) {
            if (shardRouting.primary()) {
                ++this.primaries;
            }
            this.unassigned.add(shardRouting);
        }

        public void sort(Comparator<ShardRouting> comparator) {
            this.nodes.ensureMutable();
            CollectionUtil.timSort(this.unassigned, comparator);
        }

        public int size() {
            return this.unassigned.size();
        }

        public int getNumPrimaries() {
            return this.primaries;
        }

        public int getNumIgnoredPrimaries() {
            return this.ignoredPrimaries;
        }

        public UnassignedIterator iterator() {
            return new UnassignedIterator();
        }

        public Stream<ShardRouting> stream() {
            return StreamSupport.stream(this.spliterator(), false);
        }

        public List<ShardRouting> ignored() {
            return Collections.unmodifiableList(this.ignored);
        }

        public void ignoreShard(ShardRouting shard, UnassignedInfo.AllocationStatus allocationStatus, RoutingChangesObserver changes) {
            this.nodes.ensureMutable();
            if (shard.primary()) {
                ++this.ignoredPrimaries;
                UnassignedInfo currInfo = shard.unassignedInfo();
                assert (currInfo != null);
                if (!allocationStatus.equals(currInfo.lastAllocationStatus())) {
                    UnassignedInfo newInfo = new UnassignedInfo(currInfo.reason(), currInfo.message(), currInfo.failure(), currInfo.failedAllocations(), currInfo.unassignedTimeNanos(), currInfo.unassignedTimeMillis(), currInfo.delayed(), allocationStatus, currInfo.failedNodeIds(), currInfo.lastAllocatedNodeId());
                    ShardRouting updatedShard = shard.updateUnassigned(newInfo, shard.recoverySource());
                    changes.unassignedInfoUpdated(shard, newInfo);
                    shard = updatedShard;
                }
            }
            this.ignored.add(shard);
        }

        public void resetIgnored() {
            assert (this.unassigned.size() == 0);
            this.unassigned.addAll(this.ignored);
            this.ignored.clear();
        }

        public boolean isEmpty() {
            return this.unassigned.isEmpty();
        }

        public boolean isIgnoredEmpty() {
            return this.ignored.isEmpty();
        }

        public ShardRouting[] drain() {
            this.nodes.ensureMutable();
            ShardRouting[] mutableShardRoutings = this.unassigned.toArray(new ShardRouting[this.unassigned.size()]);
            this.unassigned.clear();
            this.primaries = 0;
            return mutableShardRoutings;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            UnassignedShards that = (UnassignedShards)o;
            return this.primaries == that.primaries && this.ignoredPrimaries == that.ignoredPrimaries && this.unassigned.equals(that.unassigned) && this.ignored.equals(that.ignored);
        }

        public int hashCode() {
            return Objects.hash(this.unassigned, this.ignored, this.primaries, this.ignoredPrimaries);
        }

        public class UnassignedIterator
        implements Iterator<ShardRouting>,
        ExistingShardsAllocator.UnassignedAllocationHandler {
            private final ListIterator<ShardRouting> iterator;
            private ShardRouting current;

            public UnassignedIterator() {
                this.iterator = UnassignedShards.this.unassigned.listIterator();
            }

            @Override
            public boolean hasNext() {
                return this.iterator.hasNext();
            }

            @Override
            public ShardRouting next() {
                this.current = this.iterator.next();
                return this.current;
            }

            @Override
            public ShardRouting initialize(String nodeId, @Nullable String existingAllocationId, long expectedShardSize, RoutingChangesObserver routingChangesObserver) {
                UnassignedShards.this.nodes.ensureMutable();
                this.innerRemove();
                return UnassignedShards.this.nodes.initializeShard(this.current, nodeId, existingAllocationId, expectedShardSize, routingChangesObserver);
            }

            @Override
            public void removeAndIgnore(UnassignedInfo.AllocationStatus attempt, RoutingChangesObserver changes) {
                UnassignedShards.this.nodes.ensureMutable();
                this.innerRemove();
                UnassignedShards.this.ignoreShard(this.current, attempt, changes);
            }

            private void updateShardRouting(ShardRouting shardRouting) {
                this.current = shardRouting;
                this.iterator.set(shardRouting);
            }

            @Override
            public ShardRouting updateUnassigned(UnassignedInfo unassignedInfo, RecoverySource recoverySource, RoutingChangesObserver changes) {
                UnassignedShards.this.nodes.ensureMutable();
                ShardRouting updatedShardRouting = this.current.updateUnassigned(unassignedInfo, recoverySource);
                changes.unassignedInfoUpdated(this.current, unassignedInfo);
                this.updateShardRouting(updatedShardRouting);
                return updatedShardRouting;
            }

            @Override
            public void remove() {
                throw new UnsupportedOperationException("remove is not supported in unassigned iterator, use removeAndIgnore or initialize");
            }

            private void innerRemove() {
                this.iterator.remove();
                if (this.current.primary()) {
                    --UnassignedShards.this.primaries;
                }
            }
        }
    }

    private static final class Recoveries {
        private static final Recoveries EMPTY = new Recoveries();
        private int incoming = 0;
        private int outgoing = 0;

        private Recoveries() {
        }

        public Recoveries copy() {
            Recoveries copy = new Recoveries();
            copy.incoming = this.incoming;
            copy.outgoing = this.outgoing;
            return copy;
        }

        void addOutgoing(int howMany) {
            assert (this.outgoing + howMany >= 0) : this.outgoing + howMany + " must be >= 0";
            this.outgoing += howMany;
        }

        void addIncoming(int howMany) {
            assert (this.incoming + howMany >= 0) : this.incoming + howMany + " must be >= 0";
            this.incoming += howMany;
        }

        int getOutgoing() {
            return this.outgoing;
        }

        int getIncoming() {
            return this.incoming;
        }

        public static Recoveries getOrAdd(Map<String, Recoveries> map, String key) {
            Recoveries recoveries = map.get(key);
            if (recoveries == null) {
                recoveries = new Recoveries();
                map.put(key, recoveries);
            }
            return recoveries;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Recoveries that = (Recoveries)o;
            return this.incoming == that.incoming && this.outgoing == that.outgoing;
        }

        public int hashCode() {
            return Objects.hash(this.incoming, this.outgoing);
        }
    }
}

