// vim:set noet cinoptions= sw=4 ts=4:
// This file is part of the eix project and distributed under the
// terms of the GNU General Public License v2.
//
// Copyright (c)
//   Wolfgang Frisch <xororand@users.sourceforge.net>
//   Emil Beinroth <emilbeinroth@gmx.net>
//   Martin Väth <martin@mvath.de>

#include <config.h>  // IWYU pragma: keep

#include <unistd.h>

#include <cstdlib>
#include <cstring>

#include <algorithm>
#include <string>
#include <vector>

#include "database/header.h"
#include "database/io.h"
#include "database/package_reader.h"
#include "eixTk/ansicolor.h"
#include "eixTk/argsreader.h"
#include "eixTk/attribute.h"
#include "eixTk/diagnostics.h"
#include "eixTk/dialect.h"
#include "eixTk/eixint.h"
#include "eixTk/filenames.h"
#include "eixTk/formated.h"
#include "eixTk/i18n.h"
#include "eixTk/likely.h"
#include "eixTk/null.h"
#include "eixTk/outputstring.h"
#include "eixTk/parseerror.h"
#include "eixTk/ptr_container.h"
#include "eixTk/stringtypes.h"
#include "eixTk/stringutils.h"
#include "eixTk/unordered_map.h"
#include "eixTk/utils.h"
#include "eixrc/eixrc.h"
#include "eixrc/global.h"
#include "main/main.h"
#include "output/formatstring-print.h"
#include "output/formatstring.h"
#include "output/print-formats.h"
#include "output/print-proto.h"
#include "output/print-xml.h"
#include "portage/basicversion.h"
#include "portage/conf/portagesettings.h"
#include "portage/extendedversion.h"
#include "portage/keywords.h"
#include "portage/mask.h"
#include "portage/package.h"
#include "portage/packagetree.h"
#include "portage/set_stability.h"
#include "portage/vardbpkg.h"
#include "search/algorithms.h"
#include "search/matchtree.h"
#include "search/packagetest.h"
#include "various/cli.h"
#include "various/drop_permissions.h"

#define VAR_DB_PKG "/var/db/pkg/"

template<typename m_Type> class MaskList;

using std::string;
using std::vector;

typedef eix::ptr_container<vector<Package *> > PackageList;

static void dump_help();
ATTRIBUTE_NONNULL_ static bool opencache(Database *db, const char *filename, const char *tooltext);
ATTRIBUTE_NONNULL((1, 2)) static bool print_overlay_table(PrintFormat *fmt, DBHeader *header, PrintFormat::OverlayUsed *overlay_used);
ATTRIBUTE_NONNULL_ static void parseFormat(const char *sourcename, const char *content);
ATTRIBUTE_NONNULL_ static void set_format(EixRc *rc);
ATTRIBUTE_NONNULL_ static void setup_defaults(EixRc *rc, bool is_tty);
ATTRIBUTE_NONNULL_ static bool is_current_dbversion(const char *filename, const char *tooltext);
static void print_wordvec(const WordVec& vec);
static void print_unused(const string& filename, const string& excludefiles, const PackageList& packagelist, bool test_empty);
static void print_removed(const string& dirname, const string& excludefiles, const PackageList& packagelist);
inline static void print_unused(const string& filename, const string& excludefiles, const PackageList& packagelist);
inline static void print_unused(const string& filename, const string& excludefiles, const PackageList& packagelist) {
	print_unused(filename, excludefiles, packagelist, false);
}

/**
Show a short help screen with options and commands
**/
static void dump_help() {
	eix::say(_("Usage: %s [options] EXPRESSION\n"
"\n"
"Search for packages in the index generated by eix-update.\n"
"EXPRESSION is true or false. Packages for which the EXPRESSION evaluates true\n"
"are included in the final report.\n"
"\n"
"EXPRESSION ::= [ --not | -! ] BRACE_OR_TEST |\n"
"               EXPRESSION [ --and | -a ] EXPRESSION |\n"
"               EXPRESSION [ --or  | -o ] EXPRESSION |\n"
"BRACE_OR_TEST ::= --open | -( EXPRESSION --close | -) |\n"
"               TEST_WITH_OPTIONS\n"
"TEST_WITH_OPTIONS ::= TEST_OPTIONS [PATTERN]\n"
"\n"
"Global:\n"
"   Exclusive options:\n"
"     -h, --help            show this screen and exit\n"
"     -V, --version         show version and exit\n"
"     --dump                dump variables to stdout\n"
"     --dump-defaults       dump default values of variables\n"
"     --print               print the expanded value of a variable\n"
"     --known-vars          print all variable names known to --print\n"
"     --print-all-eapis     print all EAPI used in some version\n"
"     --print-all-useflags  print all IUSE/REQUIRED_USE used in some version\n"
"     --print-all-keywords  print all KEYWORDS used in some version\n"
"     --print-all-slots     print all SLOT strings used in some version\n"
"     --print-all-licenses  print all LICENSE strings used in some package\n"
"     --print-all-depends   print all words occurring in some {,R,P}DEPEND\n"
"                           (needs DEP=true)\n"
"     --print-world-sets    print the world sets\n"
"     --print-profile-paths print all paths of current profile\n"
"     --256                 Print all ansi color palettes\n"
"     --256d                Print ansi color palettes for foreground (dark)\n"
"     --256d0               Print ansi color palette dark (normal)\n"
"     --256d1               Print ansi color palette dark (bright)\n"
"     --256l                Print ansi color palettes for foreground (light)\n"
"     --256l0               Print ansi color palette light (normal)\n"
"     --256l1               Print ansi color palette light (bright)\n"
"     --256b                Print ansi color palette for background\n"
"\n"
"   Special:\n"
"     -t  --test-non-matching Before other output, print non-matching entries\n"
"                           of /etc/portage/package.* and non-matching names\n"
"                           of installed packages; this option is best\n"
"                           combined with -T to clean up /etc/portage/package.*\n"
"     -Q, --quick (toggle)  don't read unguessable slots of installed packages\n"
"         --care            always read slots of installed packages\n"
"         --deps-installed  always read deps of installed packages\n"
"         --cache-file      use another cache-file instead of %s\n"
"     -R  --remote (toggle)  use remote cache-file %s\n"
"     -Z  --remote2 (toggle) use remote cache-file %s\n"
"\n"
"   Output:\n"
"     -q, --quiet (toggle)   no output. Typically combined with -0\n"
"         --nowarn (toggle)  suppress warnings about wrong portage config\n"
"         --ansi             reset the ansi 256 color palette\n"
"     -n, --nocolor          do not use ANSI color codes\n"
"     -F, --force-color      force colorful output\n"
"         --color            color always/never/auto\n"
"     -*, --pure-packages    Omit printing of overlay names and package number\n"
"     -#, --only-names       --pure-packages with format <category>/<name>\n"
"     -0  --brief (toggle)   Print at most one package then stop. See -q\n"
"                            Usually faster with COUNT_ONLY_PRINTED=false\n"
"         --brief2 (toggle)  Print at most two packages then stop\n"
"     --xml (toggle)         output results in XML format\n"
"     --proto (toggle)       output results in protobuf format\n"
"     -c, --compact          compact search results\n"
"     -v, --verbose          verbose search results\n"
"     -N, --normal           ignores -c, -v, and DEFAULT_FORMAT\n"
"         --format           format string\n"
"     -x, --versionsort  (toggle) sort output by slots/versions\n"
"     -l, --versionlines (toggle) print available versions line-by-line\n"
"                            (and print additional data for each version)\n"
"\n"
"TEST_OPTIONS:\n"
"  Miscellaneous:\n"
"    -I, --installed       Next expression only matches installed packages\n"
"    -i, --multi-installed Match packages installed in several versions\n"
"    -d, --dup-packages    Match duplicated packages\n"
"    -D, --dup-versions    Match packages with duplicated versions\n"
"    -1, --slotted         Match packages with a nontrivial slot\n"
"    -2, --slots           Match packages with two different slots\n"
"    -u, --upgrade[+-]     Match packages without best slotted version\n"
"                          +: settings from LOCAL_PORTAGE_CONFIG=true\n"
"                          -: settings from LOCAL_PORTAGE_CONFIG=false\n"
"    --stable[+-]          Match packages with a stable version\n"
"    --testing[+-]         Match packages with a testing or stable version\n"
"    --non-masked[+-]      Match packages with a non-masked version\n"
"    --system[+-]          Match @system packages\n"
"    --profile[+-]         Match @profile packages\n"
"    --installed-unstable  Match packages with a non-stable installed version\n"
"    --installed-testing   Match packages with a testing    installed version\n"
"    --installed-masked    Match packages with a masked     installed version\n"
"    --world-file          Match packages of world file or @system\n"
"    --world-set           Match packages of a world set or @system\n"
"    --world               Match packages of @world (file, set or @system)\n"
"    --selected-file       Match packages of world file\n"
"    --selected-set        Match packages of a world set\n"
"    --selected            Match packages of @selected (world file or set)\n"
"    --binary              Match packages with *.tbz2, *.gpkg.tar, or *.xpak\n"
"    --multi-binary NR     Match packages with at least NR --binary files\n"
"    --nonvirtual                     Match packages from nonvirtual overlays\n"
"    --virtual                        Match packages from virtual overlays\n"
"    -O, --overlay                    Match packages from overlays\n"
"    --in-overlay OVERLAY             Match packages from OVERLAY\n"
"    --only-in-overlay OVERLAY        Match packages only in OVERLAY\n"
"    -J, --installed-overlay Match packages installed from overlays\n"
"    --installed-from-overlay OVERLAY Packages installed from OVERLAY\n"
"    --installed-in-some-overlay      Packages with an installed version\n"
"                                     provided by some overlay\n"
"    --installed-in-overlay OVERLAY   Packages with an installed version\n"
"                                     provided from OVERLAY\n"
"    --restrict-fetch          Match packages with RESTRICT=fetch\n"
"    --restrict-mirror         Match packages with RESTRICT=mirror\n"
"    --restrict-primaryuri     Match packages with RESTRICT=primaryuri\n"
"    --restrict-binchecks      Match packages with RESTRICT=binchecks\n"
"    --restrict-strip          Match packages with RESTRICT=strip\n"
"    --restrict-test           Match packages with RESTRICT=test\n"
"    --restrict-userpriv       Match packages with RESTRICT=userpriv\n"
"    --restrict-installsources Match packages with RESTRICT=installsources\n"
"    --restrict-bindist        Match packages with RESTRICT=bindist\n"
"    --restrict-parallel       Match packages with RESTRICT=parallel\n"
"    --properties-interactive  Match packages with PROPERTIES=interactive\n"
"    --properties-live         Match packages with PROPERTIES=live\n"
"    --properties-virtual      Match packages with PROPERTIES=virtual\n"
"    --properties-set          Match packages with PROPERTIES=set\n"
"    -T, --test-obsolete   Match packages with obsolete entries in\n"
"                          /etc/portage/package.* (see man eix)\n"
"                          Use -t to check non-existing packages\n"
"    -|, --pipe            Use input from pipe of emerge -pv\n"
"    --pipe-mask           As --pipe, but input is assumed to be masks\n"
"\n"
"  Search Fields:\n"
"    -y, --any               any search field can match (same as -SACsHL...)\n"
"    -S, --description       description\n"
"    -A, --category-name     \"category/name\"\n"
"    -C, --category          category\n"
"    -s, --name              name (default)\n"
"    -H, --homepage          homepage\n"
"    -L, --license           license\n"
"    --deps                  same as --available-deps --installed-deps\n"
"    --depend                same as --available-depend --installed-depend\n"
"    --rdepend               same as --available-rdepend --installed-rdepend\n"
"    --pdepend               same as --available-pdepend --installed-pdepend\n"
"    --bdepend               same as --available-bdepend --installed-bdepend\n"
"    --idepend               same as --available-idepend --installed-idepend\n"
"    --available-deps        same as --available-depend --available-rdepend\n"
"                            --available-pdepend --available-bdepend\n"
"                            --available-idepend\n"
"    --available-depend      depend (of available version; only if DEP=true)\n"
"    --available-rdepend     rdepend (of available version; only if DEP=true)\n"
"    --available-pdepend     pdepend (of available version; only if DEP=true)\n"
"    --available-bdepend     bdepend (of available version; only if DEP=true)\n"
"    --available-idepend     idepend (of available version; only if DEP=true)\n"
"    --installed-deps        same as --installed-depend --installed-rdepend\n"
"                            --installed-pdepend --installed-bdepend\n"
"                            --installed-idepend\n"
"    --installed-depend      depend (of installed version)\n"
"    --installed-rdepend     rdepend (of installed version)\n"
"    --installed-pdepend     pdepend (of installed version)\n"
"    --installed-bdepend     bdepend (of installed version)\n"
"    --installed-idepend     idepend (of installed version)\n"
"    --set                   local package set name\n"
"    --src-uri               SRC_URI\n"
"    --eapi                  EAPI\n"
"    --installed-eapi        EAPI of installed version\n"
"    --slot                  slot\n"
"    --fullslot              slot with subslot\n"
"    --installed-slot        slot of installed version\n"
"    --installed-fullslot    slot with subslot of installed version\n"
"    -U, --use               useflag (of the ebuild)\n"
"    --installed-with-use    enabled useflag (of installed package)\n"
"    --installed-without-use disabled useflag (of installed package)\n"
"\n"
"  Type of Pattern:\n"
"    -r, --regex           Pattern is a regexp, ignoring case (default)\n"
"        --regex-case      Pattern is a regexp, taking case into account\n"
"    -e, --exact           Pattern is the exact string\n"
"    -z, --substring       Pattern is a substring\n"
"    -b, --begin           Pattern is the beginning of the string\n"
"        --end             Pattern is the end       of the string\n"
"    -p, --pattern         Pattern is a wildcards-pattern\n"
"    -f [m], --fuzzy [m]   Use fuzzy-search with a max. levenshtein-distance m\n"
"\n"
"This program is covered by the GNU General Public License. See COPYING for\n"
"further information.")) %
	program_name %
	EIX_CACHEFILE %
	EIX_REMOTECACHEFILE1 %
	EIX_REMOTECACHEFILE2;
}

static const char *formatstring;
static const char *eix_cachefile(NULLPTR);
static const char *var_to_print(NULLPTR);
static const char *color(NULLPTR);

enum OverlayMode {
	mode_list_used_renumbered  = 0,
	mode_list_used             = 1,
	mode_list_all_if_any       = 2,
	mode_list_all              = 3,
	mode_list_none             = 4
};

static OverlayMode overlay_mode;

static PrintFormat *format;

static eix::TinyUnsigned remote_default;

static const ParseError *parse_error;

/**
Local options for argument reading
**/
static struct LocalOptions {
	bool
		ansi,
		palette256,
		palette256b,
		palette256d,
		palette256d0,
		palette256d1,
		palette256l,
		palette256l0,
		palette256l1,
		be_quiet,
		no_warn,
		quick,
		care,
		deps_installed,
		verbose_output,
		compact_output,
		normal_output,
		show_help,
		show_version,
		pure_packages,
		only_names,
		brief,
		brief2,
		dump_eixrc,
		dump_defaults,
		known_vars,
		xml,
		proto,
		test_unused,
		do_debug,
		ignore_etc_portage,
		is_current,
		remote,
		remote2,
		hash_eapi,
		hash_iuse,
		hash_keywords,
		hash_slot,
		hash_license,
		hash_depend,
		print_profile_paths,
		world_sets;
} rc_options;

/**
Arguments and options
**/
class EixOptionList : public OptionList {
	public:
		EixOptionList();
};

EixOptionList::EixOptionList() {
	// Global options
	// The following might give a memory leak with -flto for unknown reasons:
	// EMPLACE_BACK(Option, ("ansi",          O_ANSI,  Option::BOOLEAN_T,     &rc_options.ansi));
	push_back(Option("ansi",          O_ANSI,  Option::BOOLEAN_T,     &rc_options.ansi));
	push_back(Option("256",           O_P256,  Option::BOOLEAN_T,     &rc_options.palette256));
	push_back(Option("256d",          O_P256D, Option::BOOLEAN_T,     &rc_options.palette256d));
	push_back(Option("256d0",         O_P256D0, Option::BOOLEAN_T,    &rc_options.palette256d0));
	push_back(Option("256d1",         O_P256D1, Option::BOOLEAN_T,    &rc_options.palette256d1));
	push_back(Option("256l",          O_P256L, Option::BOOLEAN_T,     &rc_options.palette256l));
	push_back(Option("256l0",         O_P256L0, Option::BOOLEAN_T,    &rc_options.palette256l0));
	push_back(Option("256l1",         O_P256L1, Option::BOOLEAN_T,    &rc_options.palette256l1));
	push_back(Option("256b",          O_P256B, Option::BOOLEAN_T,     &rc_options.palette256b));
	push_back(Option("quiet",         'q',     Option::BOOLEAN,       &rc_options.be_quiet));
	push_back(Option("nowarn",        O_NOWARN, Option::BOOLEAN,      &rc_options.no_warn));
	push_back(Option("quick",         'Q',     Option::BOOLEAN,       &rc_options.quick));
	push_back(Option("care",          O_CARE,  Option::BOOLEAN_T,     &rc_options.care));
	push_back(Option("deps-installed", O_DEPS_INSTALLED, Option::BOOLEAN_T, &rc_options.deps_installed));

	push_back(Option("nocolor",       'n',     Option::BOOLEAN_T,     &format->no_color));
	push_back(Option("force-color",   'F',     Option::BOOLEAN_F,     &format->no_color));
	push_back(Option("color",         O_COLOR, Option::STRING,        &color));
	push_back(Option("versionlines",  'l',     Option::BOOLEAN,       &format->style_version_lines));
	push_back(Option("versionsort",   'x',     Option::BOOLEAN,       &format->slot_sorted));
	push_back(Option("pure-packages", '*',     Option::BOOLEAN,       &rc_options.pure_packages));
	push_back(Option("only-names",    '#',     Option::BOOLEAN,       &rc_options.only_names));
	push_back(Option("brief",         '0',     Option::BOOLEAN,       &rc_options.brief));
	push_back(Option("brief2",       O_BRIEF2, Option::BOOLEAN,       &rc_options.brief2));

	push_back(Option("verbose",       'v',     Option::BOOLEAN_T,     &rc_options.verbose_output));
	push_back(Option("compact",       'c',     Option::BOOLEAN_T,     &rc_options.compact_output));
	push_back(Option("normal",        'N',     Option::BOOLEAN_T,     &rc_options.normal_output));
	push_back(Option("xml",           O_XML,   Option::BOOLEAN,       &rc_options.xml));
	push_back(Option("proto",         O_PROTO, Option::BOOLEAN,       &rc_options.proto));
	push_back(Option("help",          'h',     Option::BOOLEAN_T,     &rc_options.show_help));
	push_back(Option("version",       'V',     Option::BOOLEAN_T,     &rc_options.show_version));
	push_back(Option("dump",          O_DUMP,  Option::BOOLEAN_T,     &rc_options.dump_eixrc));
	push_back(Option("dump-defaults", O_DUMP_DEFAULTS, Option::BOOLEAN_T, &rc_options.dump_defaults));
	push_back(Option("known-vars",   O_KNOWN_VARS, Option::BOOLEAN_T, &rc_options.known_vars));
	push_back(Option("test-non-matching", 't', Option::BOOLEAN_T,     &rc_options.test_unused));
	push_back(Option("debug",         O_DEBUG, Option::BOOLEAN_T,     &rc_options.do_debug));

	push_back(Option("print-all-eapis",     O_HASH_EAPI,     Option::BOOLEAN_T, &rc_options.hash_eapi));
	push_back(Option("print-all-useflags",  O_HASH_IUSE,     Option::BOOLEAN_T, &rc_options.hash_iuse));
	push_back(Option("print-all-keywords",  O_HASH_KEYWORDS, Option::BOOLEAN_T, &rc_options.hash_keywords));
	push_back(Option("print-all-slots",     O_HASH_SLOT,     Option::BOOLEAN_T, &rc_options.hash_slot));
	push_back(Option("print-all-licenses",  O_HASH_LICENSE,  Option::BOOLEAN_T, &rc_options.hash_license));
	push_back(Option("print-all-depends",   O_HASH_DEPEND,   Option::BOOLEAN_T, &rc_options.hash_depend));
	push_back(Option("print-world-sets",    O_WORLD_SETS,    Option::BOOLEAN_T, &rc_options.world_sets));
	push_back(Option("print-profile-paths", O_PROFILE_PATHS, Option::BOOLEAN_T, &rc_options.print_profile_paths));

	push_back(Option("ignore-etc-portage",  O_IGNORE_ETC_PORTAGE, Option::BOOLEAN_T,  &rc_options.ignore_etc_portage));

	push_back(Option("print",               O_PRINT_VAR,    Option::STRING,     &var_to_print));

	push_back(Option("format",         O_FMT,         Option::STRING,   &formatstring));

	push_back(Option("cache-file",     O_EIX_CACHEFILE, Option::STRING, &eix_cachefile));
	push_back(Option("remote",         'R', Option::BOOLEAN, &rc_options.remote));
	push_back(Option("remote2",        'Z', Option::BOOLEAN, &rc_options.remote2));

	// Options for criteria
	push_back(Option("installed",     'I'));
	push_back(Option("multi-installed", 'i'));
	push_back(Option("slotted",       '1'));
	push_back(Option("slots",         '2'));
	push_back(Option("upgrade",       'u'));
	push_back(Option("upgrade+",      O_UPGRADE_LOCAL));
	push_back(Option("upgrade-",      O_UPGRADE_NONLOCAL));
	push_back(Option("stable",        O_STABLE_DEFAULT));
	push_back(Option("testing",       O_TESTING_DEFAULT));
	push_back(Option("non-masked",    O_NONMASKED_DEFAULT));
	push_back(Option("binary",        O_BINARY));
	push_back(Option("multi-binary" , O_MULTIBINARY,              Option::KEEP_STRING_OPTIONAL));
	push_back(Option("world-file",    O_WORLD_FILE));
	push_back(Option("world-set",     O_WORLD_SET));
	push_back(Option("world",         O_WORLD_ALL));
	push_back(Option("selected-file", O_SELECTED_FILE));
	push_back(Option("selected-set",  O_SELECTED_SET));
	push_back(Option("selected",      O_SELECTED_ALL));
	push_back(Option("system",        O_SYSTEM_DEFAULT));
	push_back(Option("profile",       O_PROFILE_DEFAULT));
	push_back(Option("stable+",       O_STABLE_LOCAL));
	push_back(Option("testing+",      O_TESTING_LOCAL));
	push_back(Option("non-masked+",   O_NONMASKED_LOCAL));
	push_back(Option("system+",       O_SYSTEM_LOCAL));
	push_back(Option("profile+",      O_PROFILE_LOCAL));
	push_back(Option("stable-",       O_STABLE_NONLOCAL));
	push_back(Option("testing-",      O_TESTING_NONLOCAL));
	push_back(Option("non-masked-",   O_NONMASKED_NONLOCAL));
	push_back(Option("system-",       O_SYSTEM_NONLOCAL));
	push_back(Option("profile-",      O_PROFILE_NONLOCAL));
	push_back(Option("installed-unstable", O_INSTALLED_UNSTABLE));
	push_back(Option("installed-testing",  O_INSTALLED_TESTING));
	push_back(Option("installed-masked",   O_INSTALLED_MASKED));
	push_back(Option("overlay",              'O'));
	push_back(Option("installed-overlay",    'J'));
	push_back(Option("installed-from-overlay", O_FROM_OVERLAY,    Option::KEEP_STRING_OPTIONAL));
	push_back(Option("nonvirtual",           O_NONVIRTUAL));
	push_back(Option("virtual",              O_VIRTUAL));
	push_back(Option("in-overlay",           O_OVERLAY,           Option::KEEP_STRING_OPTIONAL));
	push_back(Option("only-in-overlay",      O_ONLY_OVERLAY,      Option::KEEP_STRING_OPTIONAL));
	push_back(Option("installed-in-some-overlay", O_INSTALLED_SOME));
	push_back(Option("installed-in-overlay", O_INSTALLED_OVERLAY, Option::KEEP_STRING_OPTIONAL));
	push_back(Option("restrict-fetch",         O_RESTRICT_FETCH));
	push_back(Option("restrict-mirror",        O_RESTRICT_MIRROR));
	push_back(Option("restrict-primaryuri",    O_RESTRICT_PRIMARYURI));
	push_back(Option("restrict-binchecks",     O_RESTRICT_BINCHECKS));
	push_back(Option("restrict-strip",         O_RESTRICT_STRIP));
	push_back(Option("restrict-test",          O_RESTRICT_TEST));
	push_back(Option("restrict-userpriv",      O_RESTRICT_USERPRIV));
	push_back(Option("restrict-installsources", O_RESTRICT_INSTALLSOURCES));
	push_back(Option("restrict-bindist",       O_RESTRICT_BINDIST));
	push_back(Option("restrict-parallel",      O_RESTRICT_PARALLEL));
	push_back(Option("properties-interactive", O_PROPERTIES_INTERACTIVE));
	push_back(Option("properties-live",        O_PROPERTIES_LIVE));
	push_back(Option("properties-virtual",     O_PROPERTIES_VIRTUAL));
	push_back(Option("properties-set",         O_PROPERTIES_SET));
	push_back(Option("dup-packages",  'd'));
	push_back(Option("dup-versions",  'D'));
	push_back(Option("test-obsolete", 'T'));
	push_back(Option("pipe",          '|'));
	push_back(Option("pipe-mask",     O_PIPE_MASK));

	// Algorithms for a criterion
	push_back(Option("fuzzy",         'f'));
	push_back(Option("regex",         'r'));
	push_back(Option("regex-case",    O_REGEX_CASE));
	push_back(Option("exact",         'e'));
	push_back(Option("pattern",       'p'));
	push_back(Option("begin",         'b'));
	push_back(Option("substring",     'z'));
	push_back(Option("end",           O_END_ALGO));

	// What to match in this criterion
	push_back(Option("any",           'y'));
	push_back(Option("name",          's'));
	push_back(Option("src-uri",       O_SEARCH_SRC_URI));
	push_back(Option("eapi",          O_SEARCH_EAPI));
	push_back(Option("installed-eapi", O_SEARCH_INST_EAPI));
	push_back(Option("slot",          O_SEARCH_SLOT));
	push_back(Option("fullslot",      O_SEARCH_FULLSLOT));
	push_back(Option("installed-slot", O_SEARCH_INST_SLOT));
	push_back(Option("installed-fullslot", O_SEARCH_INST_FULLSLOT));
	push_back(Option("category",      'C'));
	push_back(Option("category-name", 'A'));
	push_back(Option("description",   'S'));
	push_back(Option("license",       'L'));
	push_back(Option("homepage",      'H'));
	push_back(Option("deps",          O_DEPS));
	push_back(Option("depend",        O_DEPEND));
	push_back(Option("rdepend",       O_RDEPEND));
	push_back(Option("pdepend",       O_PDEPEND));
	push_back(Option("bdepend",       O_BDEPEND));
	push_back(Option("idepend",       O_IDEPEND));
	push_back(Option("available-deps",    O_AVAILABLE_DEPS));
	push_back(Option("available-depend",  O_AVAILABLE_DEPEND));
	push_back(Option("available-rdepend", O_AVAILABLE_RDEPEND));
	push_back(Option("available-pdepend", O_AVAILABLE_PDEPEND));
	push_back(Option("available-bdepend", O_AVAILABLE_BDEPEND));
	push_back(Option("available-idepend", O_AVAILABLE_IDEPEND));
	push_back(Option("installed-deps",    O_INSTALLED_DEPS));
	push_back(Option("installed-depend",  O_INSTALLED_DEPEND));
	push_back(Option("installed-rdepend", O_INSTALLED_RDEPEND));
	push_back(Option("installed-pdepend", O_INSTALLED_PDEPEND));
	push_back(Option("installed-bdepend", O_INSTALLED_BDEPEND));
	push_back(Option("installed-idepend", O_INSTALLED_IDEPEND));
	push_back(Option("set",           O_SEARCH_SET));
	push_back(Option("use",           'U'));
	push_back(Option("installed-with-use",    O_INSTALLED_WITH_USE));
	push_back(Option("installed-without-use", O_INSTALLED_WITHOUT_USE));

	// What to do with the next one
	push_back(Option("not",           '!'));
	push_back(Option("or",            'o'));
	push_back(Option("and",           'a'));
	push_back(Option("open",          '('));
	push_back(Option("close",         ')'));
}

/**
Setup default values for all global variables
**/
static void setup_defaults(EixRc *rc, bool is_tty) {
	// Setup defaults
	std::memset(&rc_options, 0, sizeof(rc_options));

	Depend::use_depend           = rc->getBool("DEP");
	Version::use_required_use    = rc->getBool("REQUIRED_USE");
	ExtendedVersion::use_src_uri = rc->getBool("SRC_URI");

	rc_options.quick           = rc->getBool("QUICKMODE");
	rc_options.be_quiet        = rc->getBool("QUIETMODE");
	rc_options.care            = rc->getBool("CAREMODE");
	rc_options.deps_installed  = rc->getBool("DEPS_INSTALLED");
	switch(rc->getInteger("REMOTE_DEFAULT")) {
		case 1:
			remote_default = 1;
			rc_options.remote = true;
			break;
		case 2:
			remote_default = 2;
			rc_options.remote2 = true;
			break;
		default:
			remote_default = 0;
			break;
	}

	formatstring               = NULLPTR;
	format->setupResources(rc);
	format->no_color            = (rc->getBool("NOCOLORS") ? true :
		(rc->getBool("FORCE_COLORS") ? false : (!is_tty)));
	format->style_version_lines = rc->getBool("STYLE_VERSION_LINES");
	format->slot_sorted         = !rc->getBool("STYLE_VERSION_SORTED");
	format->recommend_mode      = rc->getLocalMode("RECOMMEND_LOCAL_MODE");

	string overlay((*rc)["OVERLAYS_LIST"]);
	if(overlay.find("if") != string::npos) {
		overlay_mode = mode_list_all_if_any;
	} else if(likely(overlay.find("number") != string::npos)) {
		overlay_mode = mode_list_used_renumbered;
	} else if(likely(overlay.find("used") != string::npos)) {
		overlay_mode = mode_list_used;
	} else if(unlikely(overlay.find("no") != string::npos) ||
		unlikely(overlay.find("false") != string::npos)) {
		overlay_mode = mode_list_none;
	} else {
		overlay_mode = mode_list_all;
	}
}

static bool print_overlay_table(PrintFormat *fmt, DBHeader *header, PrintFormat::OverlayUsed *overlay_used) {
	bool printed_overlay(false);
	for(ExtendedVersion::Overlay i((overlay_mode == mode_list_all) ? 0 : 1);
		likely(i != header->countOverlays()); ++i) {
		if((i != 0) && (overlay_used != NULLPTR)) {
			if(!((*overlay_used)[i-1])) {
				continue;
			}
		}
		OutputString s;
		fmt->overlay_keytext(&s, i);
		eix::say("%s %s%s%s")
			% s
			% fmt->color_overlayname
			% header->getOverlay(i).human_readable()
			% fmt->color_overlaynameend;
		printed_overlay = true;
	}
	return printed_overlay;
}

static void parseFormat(const char *sourcename, const char *content) {
	string error_text;
	if(likely(format->parseFormat(content, &error_text))) {
		return;
	}
	eix::say_error(_("problems while parsing %s: %s"))
			% sourcename % error_text;
	std::exit(EXIT_FAILURE);
}

static void set_format(EixRc *rc) {
	if(unlikely(formatstring != NULLPTR)) {
		parseFormat("--format", formatstring);
		return;
	}
	eix::SignedBool use_verbose(0);
	if(likely(!rc_options.normal_output)) {
		if(unlikely(rc_options.verbose_output)) {
			use_verbose = 1;
		} else if(unlikely(rc_options.compact_output)) {
			use_verbose = -1;
		} else {
			const string& s((*rc)["DEFAULT_FORMAT"]);
			if(unlikely(casecontains(s, "verb"))) {
				use_verbose = 1;
			} else if(unlikely(casecontains(s, "comp"))) {
				use_verbose = -1;
			}
		}
	}
	if(likely(use_verbose == 0)) {
		parseFormat("FORMAT", (*rc)["FORMAT"].c_str());
	} else if(likely(use_verbose > 0)) {
		parseFormat("FORMAT_VERBOSE", (*rc)["FORMAT_VERBOSE"].c_str());
	} else {
		parseFormat("FORMAT_COMPACT", (*rc)["FORMAT_COMPACT"].c_str());
	}
}

int run_eix(int argc, char** argv) {
	// Initialize static classes
	Eapi::init_static();
	Category::init_static();
	ExtendedVersion::init_static();
	PackageTest::init_static();
	PortageSettings::init_static();
	PrintFormat::init_static();
	format = new PrintFormat(get_package_property);

	EixRc& eixrc(get_eixrc(EIX_VARS_PREFIX)); {
		string errtext;
		bool success(drop_permissions(&eixrc, &errtext));
		if(!errtext.empty()) {
			eix::say_error() % errtext;
		}
		if(!success) {
			return EXIT_FAILURE;
		}
	}

	// Setup defaults for all global variables like rc_options
	bool is_tty(isatty(1) != 0);
	setup_defaults(&eixrc, is_tty);

	// Read our options from the commandline.
	ArgumentReader argreader(argc, argv, EixOptionList());

	if(unlikely(color != NULLPTR)) {
		if(unlikely(strncasecmp("au", color, 2) == 0)) {
			format->no_color = !is_tty;
		} else {
			format->no_color = !EixRc::istrue(color);
		}
	}

	if(unlikely(rc_options.ansi)) {
		AnsiColor::AnsiPalette();
	}

	if(unlikely(var_to_print != NULLPTR)) {
		if(eixrc.print_var(var_to_print)) {
			return EXIT_SUCCESS;
		}
		return EXIT_FAILURE;
	}

	if(unlikely(rc_options.known_vars)) {
		eixrc.known_vars();
		return EXIT_SUCCESS;
	}

	// Dump eixrc-stuff
	if(unlikely(rc_options.dump_eixrc || rc_options.dump_defaults)) {
		eixrc.dumpDefaults(stdout, rc_options.dump_defaults);
		return EXIT_SUCCESS;
	}

	// Show help screen
	if(unlikely(rc_options.show_help)) {
		dump_help();
		return EXIT_SUCCESS;
	}

	string cachefile;
	const char *tooltext("eix-update");
	if(unlikely(eix_cachefile != NULLPTR)) {
		cachefile = eix_cachefile;
	} else {
		if((remote_default == 1) && rc_options.remote2) {
			rc_options.remote = false;  // Command line option wins
		}
		if(rc_options.remote) {
			cachefile = eixrc["EIX_REMOTE1"];
			tooltext = "eix-remote add1/update1";
		} else if(rc_options.remote2) {
			cachefile = eixrc["EIX_REMOTE2"];
			tooltext = "eix-remote add2/update2";
		}
		if(cachefile.empty()) {
			cachefile = eixrc["EIX_CACHEFILE"];
		}
	}

	// Only check if the versions uses the current layout
	if(unlikely(rc_options.is_current)) {
		return (is_current_dbversion(cachefile.c_str(), tooltext) ? EXIT_SUCCESS : EXIT_FAILURE);
	}

	// Show version
	if(unlikely(rc_options.show_version)) {
		dump_version();
	}

	// Honour a STFU
	if(unlikely(rc_options.be_quiet)) {
		rc_options.pure_packages = true;
		if(!freopen(DEV_NULL, "w", stdout)) {
			eix::say_error(_("cannot redirect to \"%s\"")) % DEV_NULL;
			return EXIT_FAILURE;
		}
	} {  // Print color palette if requested
		AnsiColor::WhichPalette palette;
		if(unlikely(rc_options.palette256)) {
			palette = AnsiColor::PALETTE_ALL;
		} else {
			palette = (unlikely(rc_options.palette256b) ?
				AnsiColor::PALETTE_B : AnsiColor::PALETTE_NONE);
			if(unlikely(rc_options.palette256d)) {
				palette |= AnsiColor::PALETTE_D;
			} else {
				if(unlikely(rc_options.palette256d0)) {
					palette |= AnsiColor::PALETTE_D0;
				}
				if(unlikely(rc_options.palette256d1)) {
					palette |= AnsiColor::PALETTE_D1;
				}
			}
			if(unlikely(rc_options.palette256l)) {
				palette |= AnsiColor::PALETTE_L;
			} else {
				if(unlikely(rc_options.palette256l0)) {
					palette |= AnsiColor::PALETTE_L0;
				}
				if(unlikely(rc_options.palette256l1)) {
					palette |= AnsiColor::PALETTE_L1;
				}
			}
		}
		if(unlikely(palette != AnsiColor::PALETTE_NONE)) {
			AnsiColor::PrintPalette(palette);
			return EXIT_SUCCESS;
		}
	}

	bool only_printed;

	if(unlikely(rc_options.xml || rc_options.proto)) {
		rc_options.pure_packages = format->no_color = true;
		only_printed = false;
	} else {
		only_printed = eixrc.getBool("COUNT_ONLY_PRINTED");
	}

	if(unlikely(rc_options.only_names)) {
		rc_options.pure_packages = format->no_color = true;
		format->parseFormat("<category>/<name>\n", NULLPTR);
	} else {
		set_format(&eixrc);
	}

	format->setupColors();

	if(unlikely(rc_options.pure_packages)) {
		overlay_mode = mode_list_none;
	}

	parse_error = new ParseError(rc_options.no_warn);
	PortageSettings portagesettings(&eixrc, parse_error, true, false, rc_options.print_profile_paths);
	if(unlikely(rc_options.print_profile_paths)) {
		return EXIT_SUCCESS;
	}

	string var_db_pkg(eixrc["EPREFIX_INSTALLED"] + VAR_DB_PKG);
	VarDbPkg varpkg_db(var_db_pkg, !rc_options.quick, rc_options.care,
		rc_options.deps_installed,
		eixrc.getBool("RESTRICT_INSTALLED"),
		eixrc.getBool("CARE_RESTRICT_INSTALLED"),
		eixrc.getBool("USE_BUILD_TIME"));
	varpkg_db.check_installed_overlays = eixrc.getBoolText("CHECK_INSTALLED_OVERLAYS", "repository");

	MaskList<Mask> *marked_list(NULLPTR);

	/* Open database file */
	Database db;
	if(unlikely(!opencache(&db, cachefile.c_str(), tooltext))) {
		return EXIT_FAILURE;
	}
	DBHeader header;

	if(unlikely(!db.read_header(&header, NULLPTR, 0))) {
		eix::say_error(_(
			"%s was created with an incompatible eix-update:\n"
			"It uses database format %s (current is %s).\n"
			"Please run \"%s\" and try again."))
			% cachefile
			% header.version % DBHeader::current
			% tooltext;
		return EXIT_FAILURE;
	}

	if(unlikely(rc_options.hash_eapi)) {
		header.eapi_hash.output();
		return EXIT_SUCCESS;
	}
	if(unlikely(rc_options.hash_iuse)) {
		header.iuse_hash.output();
		return EXIT_SUCCESS;
	}
	if(unlikely(rc_options.hash_keywords)) {
		header.keywords_hash.output();
		return EXIT_SUCCESS;
	}
	if(unlikely(rc_options.hash_slot)) {
		header.slot_hash.output();
		return EXIT_SUCCESS;
	}
	if(unlikely(rc_options.hash_license)) {
		header.license_hash.output();
		return EXIT_SUCCESS;
	}
	if(unlikely(rc_options.hash_depend)) {
		header.depend_hash.output_depends();
		return EXIT_SUCCESS;
	}
	if(unlikely(rc_options.world_sets)) {
		const WordVec *p(portagesettings.get_world_sets());
		for(WordVec::const_iterator it(p->begin());
			likely(it != p->end()); ++it) {
			eix::say() % (*it);
		}
		return EXIT_SUCCESS;
	}

	portagesettings.store_world_sets(&(header.world_sets));

	if(header.countOverlays() != 0) {
		header.set_priorities(&portagesettings);
		format->clear_virtual(header.countOverlays());
		for(ExtendedVersion::Overlay i(1); likely(i != header.countOverlays()); ++i)
			format->set_as_virtual(i, is_virtual((eixrc["EPREFIX_VIRTUAL"] + header.getOverlay(i).path).c_str()));
	}

	LocalMode local_mode(LOCALMODE_DEFAULT);
	if(unlikely(!eixrc.getBool("LOCAL_PORTAGE_CONFIG"))) {
		rc_options.ignore_etc_portage = true;
		local_mode = LOCALMODE_NONLOCAL;
	} else if(!rc_options.ignore_etc_portage) {
		local_mode = LOCALMODE_LOCAL;
	}
	// Save lot of time: avoid redundant remasking
	if(format->recommend_mode == local_mode) {
		format->recommend_mode = LOCALMODE_DEFAULT;
	}

	SetStability stability(&portagesettings, !rc_options.ignore_etc_portage, false, eixrc.getBool("ALWAYS_ACCEPT_KEYWORDS"));

	MatchTree *matchtree = new MatchTree(eixrc.getBool("DEFAULT_IS_OR"));
	parse_cli(matchtree, &eixrc, &varpkg_db, &portagesettings, format, &stability, &header, parse_error, &marked_list, argreader);

	PackageList matches;
	PackageList all_packages; {
		PackageReader reader(&db, header, &portagesettings);
		bool add_rest(false);
		while(likely(reader.next())) {
			if(unlikely(add_rest)) {
				all_packages.PUSH_BACK(reader.release());
			} else if(unlikely(matchtree->match(&reader))) {
				Package *release(reader.release());
				if(unlikely(release == NULLPTR)) {
					break;
				}
				matches.PUSH_BACK(release);
				if(unlikely(only_printed &&
					(rc_options.brief ||
						(rc_options.brief2 && (matches.size() > 1))))) {
					if(unlikely(rc_options.test_unused)) {
						add_rest = true;
					} else {
						break;
					}
				}
				if(unlikely(rc_options.test_unused)) {
					all_packages.PUSH_BACK(release);
				}
			} else {
				if(unlikely(rc_options.test_unused)) {
					Package *release(reader.release());
					if(unlikely(release == NULLPTR)) {
						break;
					}
					all_packages.PUSH_BACK(release);
				} else if(unlikely(!reader.skip())) {
					break;
				}
			}
		}
		const char *err_cstr(reader.get_errtext());
		if(unlikely(err_cstr != NULLPTR)) {
			eix::say_error() % err_cstr;
			return EXIT_FAILURE;
		}
	}

	// Delete old matchtree
	delete matchtree;

	if(unlikely(rc_options.test_unused)) {
		bool empty(eixrc.getBool("TEST_FOR_EMPTY"));
		if(likely(eixrc.getBool("TEST_KEYWORDS"))) {
			print_unused(eixrc.m_eprefixconf + USER_KEYWORDS_FILE1,
				eixrc["KEYWORDS_NONEXISTENT"],
				all_packages);
			print_unused(eixrc.m_eprefixconf + USER_KEYWORDS_FILE2,
				eixrc["KEYWORDS_NONEXISTENT"],
				all_packages);
		}
		if(likely(eixrc.getBool("TEST_MASK"))) {
			print_unused(eixrc.m_eprefixconf + USER_MASK_FILE,
				eixrc["MASK_NONEXISTENT"],
				all_packages);
		}
		if(likely(eixrc.getBool("TEST_UNMASK"))) {
			print_unused(eixrc.m_eprefixconf + USER_UNMASK_FILE,
				eixrc["UNMASK_NONEXISTENT"],
				all_packages);
		}
		if(likely(eixrc.getBool("TEST_USE"))) {
			print_unused(eixrc.m_eprefixconf + USER_USE_FILE,
				eixrc["USE_NONEXISTENT"],
				all_packages, empty);
		}
		if(likely(eixrc.getBool("TEST_ENV"))) {
			print_unused(eixrc.m_eprefixconf + USER_ENV_FILE,
				eixrc["ENV_NONEXISTENT"],
				all_packages, empty);
		}
		if(likely(eixrc.getBool("TEST_LICENSE"))) {
			print_unused(eixrc.m_eprefixconf + USER_LICENSE_FILE,
				eixrc["LICENSE_NONEXISTENT"],
				all_packages, empty);
		}
		if(likely(eixrc.getBool("TEST_RESTRICT"))) {
			print_unused(eixrc.m_eprefixconf + USER_RESTRICT_FILE,
				eixrc["LICENSE_RESTRICT"],
				all_packages, empty);
		}
		if(likely(eixrc.getBool("TEST_CFLAGS"))) {
			print_unused(eixrc.m_eprefixconf + USER_CFLAGS_FILE,
				eixrc["CFLAGS_NONEXISTENT"],
				all_packages, empty);
		}
		if(likely(eixrc.getBool("TEST_REMOVED"))) {
			print_removed(var_db_pkg, eixrc["INSTALLED_NONEXISTENT"], all_packages);
		}
	}

	/* Sort the found matches by rating */
	if(unlikely(FuzzyAlgorithm::sort_by_levenshtein())) {
		std::sort(matches.begin(), matches.end(), FuzzyAlgorithm::compare);
	}

	format->set_marked_list(marked_list);
	if(overlay_mode != mode_list_used_renumbered) {
		format->set_overlay_translations(NULLPTR);
	}
	bool need_overlay_table(false);
	PrintFormat::OverlayUsed overlay_used(header.countOverlays(), false);
	format->set_overlay_used(&overlay_used, &need_overlay_table);
	PackageList::size_type count(0);
	PrintFormats *print_formats(NULLPTR);
	if(rc_options.xml || rc_options.proto || rc_options.be_quiet) {
		overlay_mode = mode_list_none;
		rc_options.pure_packages = true;
	}
	if (!matches.empty()) {
		if(rc_options.xml) {
			if (unlikely(rc_options.proto)) {
				eix::say_error(_("--xml and --proto must not be specified simultaneously"));
				std::exit(EXIT_FAILURE);
			}
			print_formats = new PrintXml(&header, &varpkg_db, format, &stability, &eixrc,
				portagesettings["PORTDIR"]);
		} else if (rc_options.proto) {
			print_formats = new PrintProto(&header, &varpkg_db, format, &stability);
		}
		if (print_formats != NULLPTR) {
			print_formats->start();
		}
	}
	bool have_printed(false);
	bool reached_limit(false), over_limit(false);
	string limit_var(rc_options.compact_output ? "EIX_LIMIT_COMPACT" : "EIX_LIMIT");
	eix::Treesize limit(is_tty ? eixrc.getInteger(limit_var) : 0);
	for(PackageList::iterator it(matches.begin());
		likely(it != matches.end()); ++it) {
		stability.set_stability(*it);

		if(unlikely(print_formats != NULLPTR)) {
			print_formats->package(*it);
			continue;
		}

		if(it->largest_overlay != 0) {
			need_overlay_table = true;
			if(overlay_mode <= mode_list_used) {
				for(Package::iterator ver(it->begin());
					likely(ver != it->end()); ++ver) {
					ExtendedVersion::Overlay key(ver->overlay_key);
					if(key > 0) {
						overlay_used[key - 1] = true;
					}
				}
			}
		}
		if(overlay_mode != mode_list_used_renumbered) {
			if(format->print(*it, &header, &varpkg_db, &portagesettings, &stability, reached_limit)) {
				have_printed = true;
				++count;
				if(unlikely(reached_limit)) {
					over_limit = true;
				} else if(unlikely(count == limit)) {
					reached_limit = true;
				}
				if(unlikely(rc_options.brief || (rc_options.brief2 && count > 1))) {
					break;
				}
			}
		}
	}
	switch(overlay_mode) {
		case mode_list_all:
			need_overlay_table = true;
			break;
		case mode_list_none:
			need_overlay_table = false;
			break;
		default:
			break;
	}
	PrintFormat::OverlayTranslations overlay_num(header.countOverlays(), 0);
	if(overlay_mode == mode_list_used_renumbered) {
		ExtendedVersion::Overlay i(1);
		PrintFormat::OverlayUsed::iterator uit(overlay_used.begin());
		PrintFormat::OverlayTranslations::iterator nit(overlay_num.begin());
		for(; likely(uit != overlay_used.end()); ++uit, ++nit) {
			if(*uit == true) {
				*nit = i++;
			}
		}
		format->set_overlay_translations(&overlay_num);
		for(PackageList::iterator it(matches.begin());
			likely(it != matches.end()); ++it) {
			if(format->print(*it, &header, &varpkg_db, &portagesettings, &stability, reached_limit)) {
				have_printed = true;
				++count;
				if(unlikely(reached_limit)) {
					over_limit = true;
				} else if(unlikely(count == limit)) {
					reached_limit = true;
				}
				if(unlikely(rc_options.brief || (rc_options.brief2 && count > 1))) {
					break;
				}
			}
		}
	}
	bool printed_overlay(false);
	if(need_overlay_table) {
		if(print_overlay_table(format, &header,
			(overlay_mode <= mode_list_used)? &overlay_used : NULLPTR)) {
			printed_overlay = have_printed = true;
		}
	}
	if(unlikely(print_formats != NULLPTR)) {
		print_formats->finish();
		delete print_formats;
	}

	if(!only_printed) {
		count = matches.size();
	}
	eix::SignedBool print_count_always(rc_options.pure_packages ? -1 :
		eixrc.getBoolText("PRINT_COUNT_ALWAYS", "never"));
	if(likely(print_count_always >= 0)) {
		if((print_count_always != 0) || (count > 1)) {
			have_printed = true;
			if(printed_overlay) {
				eix::say_empty();
			}
			eix::say("%s%s%s")
				% format->color_numbertext
				% (eix::format(N_("Found %s match",
					"Found %s matches", count))
					% count)
				% format->color_numbertextend;
		} else if(unlikely(count == 0)) {
			have_printed = true;
			eix::say("%s%s%s")
				% format->color_numbertext
				% _("No matches found")
				% format->color_numbertextend;
		}
	}
	if(likely(have_printed)) {
		eix::print() % format->color_end;
		if(unlikely(over_limit)) {
			eix::say(N_(
			"Only %s match displayed on terminal\n"
			"Set %s=0 to show all matches",
			"Only %s matches displayed on terminal\n"
			"Set %s=0 to show all matches", limit))
				% limit
				% limit_var;
		}
	}

	// Delete matches (or all_packages, respectively)
	if(unlikely(rc_options.test_unused)) {
		all_packages.delete_and_clear();
		matches.clear();
	} else {
		matches.delete_and_clear();
	}
	delete marked_list;

	if(unlikely(!count)) {
GCC_DIAG_OFF(sign-conversion)
		return eixrc.getInteger("NOFOUND_STATUS");
GCC_DIAG_ON(sign-conversion)
	}
	if(count > 1) {
GCC_DIAG_OFF(sign-conversion)
		return eixrc.getInteger("MOREFOUND_STATUS");
GCC_DIAG_ON(sign-conversion)
	}
	return EXIT_SUCCESS;
}  // NOLINT(readability/fn_size)

static bool opencache(Database *db, const char *filename, const char *tooltext) {
	if(likely(db->openread(filename))) {
		return true;
	}
	eix::say_error(_(
		"cannot open database file %s for reading\n"
		"Did you forget to create it with \"%s\"?"))
		% filename % tooltext;
	return false;
}

static bool is_current_dbversion(const char *filename, const char *tooltext) {
	Database db;
	if(unlikely(!opencache(&db, filename, tooltext))) {
		return false;
	}
	DBHeader header;
	return db.read_header(&header, NULLPTR, 0);
}

static void print_wordvec(const WordVec& vec) {
	for(WordVec::const_iterator it(vec.begin());
		likely(it != vec.end()); ++it) {
		eix::say() % (*it);
	}
	eix::say("--");
}

static void print_unused(const string& filename, const string& excludefiles, const PackageList& packagelist, bool test_empty) {
	WordVec unused;
	LineVec lines;
	WordSet excludes;
	bool know_excludes(false);
	pushback_lines(filename.c_str(), &lines, true);
	for(WordVec::iterator i(lines.begin());
		likely(i != lines.end()); ++i) {
		if(i->empty()) {
			continue;
		}
		if(unlikely(!know_excludes)) {
			know_excludes = true;
			WordVec excludelist;
			split_string(&excludelist, excludefiles, true);
			for(WordVec::const_iterator it(excludelist.begin());
				likely(it != excludelist.end()); ++it) {
				LineVec excl;
				pushback_lines(it->c_str(), &excl, true, false);
				join_and_split(&excludes, excl);
			}
		}

		string::size_type n(i->find_first_of("\t "));
		BasicVersion::ParseResult r;
		string errtext;
		KeywordMask m;
		if(n == string::npos) {
			if(unlikely(excludes.count(*i) != 0)) {
				continue;
			}
			if(unlikely(test_empty)) {
				unused.PUSH_BACK(MOVE(*i));
				continue;
			}
			r = m.parseMask(i->c_str(), &errtext, false);
		} else {
			string it(*i, 0, n);
			if(excludes.count(it) != 0) {
				continue;
			}
			r = m.parseMask(it.c_str(), &errtext, false);
		}
		if(r != BasicVersion::parsedOK) {
			parse_error->output(filename, lines.begin(), i, errtext);
			continue;
		}
		PackageList::const_iterator pi(packagelist.begin());
		for( ; likely(pi != packagelist.end()); ++pi) {
			if(m.ismatch(**pi)) {
				break;
			}
		}
		if(pi != packagelist.end()) {
			continue;
		}
		unused.PUSH_BACK(MOVE(*i));
	}
	if(unused.empty()) {
		eix::say(test_empty ?
			_("No non-matching or empty entries in %s") :
			_("No non-matching entries in %s"))
			% filename;
		return;
	}
	eix::say(test_empty ?
		_("Non-matching or empty entries in %s:") :
		_("Non-matching entries in %s:"))
		% filename;
	print_wordvec(unused);
}

static void print_removed(const string& dirname, const string& excludefiles, const PackageList& packagelist) {
	/* For faster testing, we build a category->name map */
	typedef UNORDERED_MAP<string, WordUnorderedSet> CatName;
	CatName cat_name;
	for(PackageList::const_iterator pit(packagelist.begin());
		likely(pit != packagelist.end()); ++pit) {
		cat_name[pit->category].INSERT(pit->name);
	}

	/* This will contain categories/packages to be printed */
	WordVec failure;

	/* Read all installed packages (not versions!) and fill failures */
	WordSet excludes;
	bool know_excludes(false);
	WordVec categories;
	pushback_files(dirname, &categories, NULLPTR, 2, true, false);
	for(WordVec::const_iterator cit(categories.begin());
		likely(cit != categories.end()); ++cit) {
		WordVec names;
		string cat_slash(*cit);
		cat_slash.append(1, '/');
		pushback_files(dirname + cat_slash, &names, NULLPTR, 2, true, false);
		CatName::const_iterator cat(cat_name.find(*cit));
		const WordIterateSet *ns((cat == cat_name.end()) ? NULLPTR : &(cat->second));
		for(WordVec::const_iterator nit(names.begin());
			likely(nit != names.end()); ++nit) {
			string curr_name;
			if(unlikely(!ExplodeAtom::split_name(&curr_name, nit->c_str()))) {
				continue;
			}
			if(unlikely((ns == NULLPTR) || (ns->count(curr_name) == 0))) {
				if(unlikely(!know_excludes)) {
					know_excludes = true;
					WordVec excludelist;
					split_string(&excludelist, excludefiles, true);
					for(WordVec::const_iterator it(excludelist.begin());
						likely(it != excludelist.end()); ++it) {
						LineVec excl;
						pushback_lines(it->c_str(), &excl, true, false);
						join_and_split(&excludes, excl);
					}
				}
				if(likely(excludes.count(curr_name) == 0)) {
					string fullname(cat_slash + curr_name);
					if(excludes.count(fullname) == 0) {
						failure.PUSH_BACK(MOVE(fullname));
					}
				}
			}
		}
	}
	if(likely(failure.empty())) {
		eix::say(_("The names of all installed packages are in the database."));
		return;
	}
	eix::say(_("The following installed packages are not in the database:"));
	print_wordvec(failure);
}
