/*
 * Copyright 2016-2017 Józef Kucia for CodeWeavers
 * Copyright 2020-2021 Philip Rebohle for Valve Corporation
 * Copyright 2020-2021 Joshua Ashton for Valve Corporation
 * Copyright 2020-2021 Hans-Kristian Arntzen for Valve Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
 */

#define VKD3D_DBG_CHANNEL VKD3D_DBG_CHANNEL_API
#include "d3d12_crosstest.h"

void test_bindless_heap_sm66_uav_counter(void)
{
    D3D12_ROOT_SIGNATURE_DESC root_signature_desc;
    D3D12_FEATURE_DATA_SHADER_MODEL shader_model;
    D3D12_UNORDERED_ACCESS_VIEW_DESC uav_desc;
    D3D12_ROOT_PARAMETER root_parameters[1];
    ID3D12DescriptorHeap *heap;
    struct test_context context;
    ID3D12Resource *resource;
    struct resource_readback rb;
    static const uint32_t counts[] = { 129, 234, 22, 31 };
    unsigned int i;
    HRESULT hr;

    static const BYTE cs_code[] =
    {
#if 0
    cbuffer Cbuf : register(b0) { uint index; };

    [numthreads(1, 1, 1)]
    void main()
    {
            RWStructuredBuffer<uint> Buf = ResourceDescriptorHeap[index];
            uint o;
            InterlockedAdd(Buf[0], 3, o);
            Buf.IncrementCounter();
    }
#endif
        0x44, 0x58, 0x42, 0x43, 0x00, 0xf3, 0xb6, 0x78, 0xf8, 0x72, 0x40, 0x1f, 0x35, 0x61, 0x3e, 0xb1, 0xe3, 0xf3, 0x72, 0x8f, 0x01, 0x00, 0x00, 0x00, 0xa0, 0x06, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0xd0, 0x00, 0x00, 0x00, 0xec, 0x00, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x60, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd7, 0xc1, 0x9e, 0x9a,
        0x01, 0xee, 0x1f, 0x4b, 0x5c, 0x6f, 0xd5, 0xb1, 0xb1, 0x69, 0xcd, 0x8d, 0x44, 0x58, 0x49, 0x4c, 0xac, 0x05, 0x00, 0x00, 0x66, 0x00, 0x05, 0x00, 0x6b, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c,
        0x06, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x94, 0x05, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x62, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02,
        0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90,
        0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07,
        0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
        0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x2b, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84,
        0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x6c, 0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0x10, 0x1a, 0xf7, 0x0c, 0x97, 0x3f, 0x61,
        0x0f, 0x21, 0xf9, 0x21, 0xd0, 0x0c, 0x0b, 0x81, 0x02, 0x32, 0x47, 0x00, 0x06, 0x73, 0x04, 0x41, 0x29, 0x8a, 0x31, 0xe7, 0x10, 0x2a, 0x88, 0x31, 0xc5, 0x18, 0x63, 0x0c, 0xa9, 0x42, 0x18, 0x53,
        0x06, 0xb1, 0xa3, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0x7c, 0x6e, 0xa3, 0x8a, 0x95, 0x98, 0x7c, 0xe4, 0xb6, 0x11, 0x61, 0x8c, 0x31, 0x85, 0x70, 0xa6, 0x18, 0x7a, 0xb7, 0x0d, 0x97, 0x3f, 0x61,
        0x0f, 0x21, 0xf9, 0x2b, 0x21, 0x39, 0x54, 0x24, 0x10, 0x69, 0xe4, 0x3c, 0x44, 0x34, 0x21, 0x84, 0x84, 0x04, 0x63, 0x0a, 0x51, 0x4c, 0x89, 0x24, 0x0f, 0x1a, 0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2,
        0x57, 0x42, 0xda, 0x90, 0x66, 0x40, 0x84, 0x31, 0x66, 0x94, 0xa2, 0x18, 0x6a, 0x0e, 0xd5, 0x81, 0x80, 0x23, 0xb8, 0x80, 0x2a, 0x50, 0x60, 0x08, 0xcf, 0x11, 0x80, 0x02, 0x00, 0x00, 0x00, 0x00,
        0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30,
        0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a,
        0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07,
        0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60,
        0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x08, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x14, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x30, 0x40, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4,
        0x79, 0x80, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x23, 0x01, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xa7, 0x02, 0x02, 0x20, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x2c, 0x10, 0x09, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x10, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x32, 0x25, 0x50, 0x0c, 0xe5,
        0x30, 0x02, 0x50, 0x08, 0xa4, 0x0b, 0x10, 0x10, 0x10, 0x10, 0x81, 0xf2, 0x0c, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44,
        0x8f, 0x0c, 0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x24, 0xc6, 0xe5, 0xc6, 0x45, 0x46, 0x26, 0x46, 0xc6, 0x85, 0x06, 0x06, 0x04, 0xa5, 0x0c, 0x86, 0x66, 0xc6, 0x8c, 0x26, 0x2c, 0x46, 0x26, 0x65, 0x43,
        0x10, 0x4c, 0x10, 0x8c, 0x63, 0x82, 0x60, 0x20, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x18, 0xc9, 0x06, 0x61, 0x30, 0x28, 0x8c, 0xcd, 0x6d, 0x18, 0x10, 0x82, 0x98, 0x20, 0x18, 0xca, 0x04, 0x21, 0x6b,
        0x08, 0x4c, 0x10, 0x8c, 0x65, 0x03, 0xa2, 0x2c, 0x8c, 0xa2, 0x0c, 0x0d, 0xb0, 0x21, 0x70, 0x36, 0x10, 0x00, 0xf0, 0x00, 0x13, 0x04, 0x01, 0x20, 0xd1, 0x16, 0x96, 0xe6, 0x36, 0x41, 0xd0, 0x98,
        0x0d, 0xc3, 0x30, 0x0c, 0x1b, 0x08, 0x65, 0x6a, 0xa8, 0x0d, 0x45, 0x24, 0x01, 0x50, 0x55, 0x85, 0x8d, 0xcd, 0xae, 0xcd, 0x25, 0x8d, 0xac, 0xcc, 0x8d, 0x6e, 0x4a, 0x10, 0x54, 0x21, 0xc3, 0x73,
        0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x9b, 0x12, 0x10, 0x4d, 0xc8, 0xf0, 0x5c, 0xec, 0xc2, 0xd8, 0xec, 0xca, 0xe4, 0xa6, 0x04, 0x46, 0x1d, 0x32, 0x3c, 0x97, 0x39, 0xb4, 0x30, 0xb2, 0x32,
        0xb9, 0xa6, 0x37, 0xb2, 0x32, 0xb6, 0x29, 0x01, 0x52, 0x86, 0x0c, 0xcf, 0x45, 0xae, 0x6c, 0xee, 0xad, 0x4e, 0x6e, 0xac, 0x6c, 0x6e, 0x4a, 0xf0, 0xd4, 0x21, 0xc3, 0x73, 0x29, 0x73, 0xa3, 0x93,
        0xcb, 0x83, 0x7a, 0x4b, 0x73, 0xa3, 0x9b, 0x9b, 0x12, 0x54, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x49, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88,
        0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce,
        0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48,
        0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e,
        0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b,
        0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78,
        0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1,
        0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39,
        0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xc8, 0x21, 0x07, 0x7c, 0x70,
        0x03, 0x72, 0x10, 0x87, 0x73, 0x70, 0x03, 0x7b, 0x08, 0x07, 0x79, 0x60, 0x87, 0x70, 0xc8, 0x87, 0x77, 0xa8, 0x07, 0x7a, 0x00, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x21, 0x00, 0x00, 0x00,
        0x56, 0x40, 0x0d, 0x97, 0xef, 0x3c, 0x3e, 0xd0, 0x34, 0xce, 0x04, 0x4c, 0x44, 0x08, 0x34, 0xc3, 0x42, 0x98, 0x40, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0xc0, 0xe4, 0x30, 0x88, 0xb0, 0x21, 0x0d, 0xfa,
        0xf8, 0xc8, 0x6d, 0xdb, 0x40, 0x36, 0x5c, 0xbe, 0xf3, 0xf8, 0x01, 0x55, 0x14, 0x44, 0xe4, 0x3e, 0x03, 0x30, 0x11, 0x9c, 0x43, 0x35, 0x13, 0x11, 0x19, 0xc1, 0x36, 0x5c, 0xbe, 0xf3, 0xf8, 0x42,
        0x40, 0x15, 0x05, 0x11, 0x95, 0x0e, 0x30, 0x94, 0x84, 0x01, 0x08, 0x98, 0x8f, 0xdc, 0xb6, 0x19, 0x74, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f, 0x44, 0x04, 0x30, 0x11, 0x21, 0xd0, 0x0c, 0x0b, 0xf1, 0x45,
        0x0e, 0xb3, 0x21, 0xcd, 0x80, 0x34, 0x86, 0x05, 0x68, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f, 0x44, 0x04, 0x30, 0x11, 0x21, 0xd0, 0x0c, 0x0b, 0xf1, 0x45, 0x0e, 0x13, 0x12, 0xc0, 0x63, 0x00, 0x04, 0x03,
        0x20, 0x0d, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x34, 0x8a, 0xa0, 0x04, 0xc8, 0x14, 0xec, 0x40,
        0xc9, 0x0e, 0x94, 0x46, 0xd9, 0x15, 0xed, 0x40, 0x61, 0x02, 0x13, 0x14, 0x67, 0xc0, 0x0c, 0x40, 0x19, 0x14, 0x63, 0x00, 0x9d, 0x11, 0x00, 0x8a, 0x73, 0x08, 0xd2, 0x32, 0x87, 0x50, 0x2d, 0x44,
        0xe7, 0x20, 0x14, 0x45, 0x71, 0x00, 0x00, 0x00, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x50, 0x61, 0x4d, 0x40, 0x11, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0x20, 0x69, 0x4f, 0x40, 0x8c, 0x18, 0x1c,
        0x00, 0x08, 0x82, 0xc1, 0xc3, 0x35, 0x81, 0x35, 0x9a, 0x10, 0x00, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x80, 0x84, 0x81, 0x13, 0x20, 0xc8, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x48, 0xde, 0x14,
        0x1c, 0x23, 0x06, 0x0b, 0x00, 0x82, 0x60, 0xa0, 0x88, 0x81, 0x13, 0x68, 0x9a, 0xd6, 0x30, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0xc0, 0x88, 0x01, 0x23, 0x58, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
    static const D3D12_SHADER_BYTECODE cs_code_dxil = SHADER_BYTECODE(cs_code);

    if (!init_compute_test_context(&context))
        return;

    shader_model.HighestShaderModel = D3D_SHADER_MODEL_6_6;
    hr = ID3D12Device_CheckFeatureSupport(context.device, D3D12_FEATURE_SHADER_MODEL,
            &shader_model, sizeof(shader_model));
    if (FAILED(hr) || shader_model.HighestShaderModel < D3D_SHADER_MODEL_6_6)
    {
        skip("Shader model 6.6 not supported by device.\n");
        destroy_test_context(&context);
        return;
    }

    memset(&root_signature_desc, 0, sizeof(root_signature_desc));
    root_signature_desc.Flags = D3D12_ROOT_SIGNATURE_FLAG_CBV_SRV_UAV_HEAP_DIRECTLY_INDEXED;
    root_signature_desc.pParameters = root_parameters;
    root_signature_desc.NumParameters = ARRAY_SIZE(root_parameters);
    memset(root_parameters, 0, sizeof(root_parameters));
    root_parameters[0].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[0].ParameterType = D3D12_ROOT_PARAMETER_TYPE_32BIT_CONSTANTS;
    root_parameters[0].Constants.ShaderRegister = 0;
    root_parameters[0].Constants.RegisterSpace = 0;
    root_parameters[0].Constants.Num32BitValues = 1;
    create_root_signature(context.device, &root_signature_desc, &context.root_signature);

    resource = create_default_buffer(context.device, D3D12_UAV_COUNTER_PLACEMENT_ALIGNMENT * 4, D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);
    context.pipeline_state = create_compute_pipeline_state(context.device, context.root_signature, cs_code_dxil);
    heap = create_gpu_descriptor_heap(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV, 4);

    memset(&uav_desc, 0, sizeof(uav_desc));
    for (i = 0; i < 4; i++)
    {
        uav_desc.ViewDimension = D3D12_UAV_DIMENSION_BUFFER;
        uav_desc.Format = DXGI_FORMAT_UNKNOWN;
        uav_desc.Buffer.CounterOffsetInBytes = D3D12_UAV_COUNTER_PLACEMENT_ALIGNMENT * i;
        uav_desc.Buffer.FirstElement = 1 + i;
        uav_desc.Buffer.NumElements = 1;
        uav_desc.Buffer.StructureByteStride = 4;
        ID3D12Device_CreateUnorderedAccessView(context.device, resource, resource, &uav_desc, get_cpu_descriptor_handle(&context, heap, i));
    }

    ID3D12GraphicsCommandList_SetDescriptorHeaps(context.list, 1, &heap);
    ID3D12GraphicsCommandList_SetComputeRootSignature(context.list, context.root_signature);
    ID3D12GraphicsCommandList_SetPipelineState(context.list, context.pipeline_state);
    for (i = 0; i < 4; i++)
    {
        ID3D12GraphicsCommandList_SetComputeRoot32BitConstant(context.list, 0, i, 0);
        ID3D12GraphicsCommandList_Dispatch(context.list, counts[i], 1, 1);
    }

    transition_resource_state(context.list, resource, D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);
    get_buffer_readback_with_command_list(resource, DXGI_FORMAT_R32_UINT, &rb, context.queue, context.list);

    for (i = 0; i < 4; i++)
    {
        uint32_t uav_value, counter_value;
        uav_value = get_readback_uint(&rb, i + 1, 0, 0);
        counter_value = get_readback_uint(&rb, i * D3D12_UAV_COUNTER_PLACEMENT_ALIGNMENT / sizeof(uint32_t), 0, 0);
        ok(uav_value == counts[i] * 3, "UAV value %u, expected %u, got %u.\n", i, counts[i] * 3, uav_value);
        ok(counter_value == counts[i], "Counter value %u, expected %u, got %u.\n", i, counts[i], counter_value);
    }

    release_resource_readback(&rb);
    ID3D12DescriptorHeap_Release(heap);
    ID3D12Resource_Release(resource);
    destroy_test_context(&context);
}

void test_bindless_heap_sm66(void)
{
    D3D12_ROOT_SIGNATURE_DESC root_signature_desc;
    D3D12_FEATURE_DATA_SHADER_MODEL shader_model;
    D3D12_UNORDERED_ACCESS_VIEW_DESC uav_desc;
    D3D12_SHADER_RESOURCE_VIEW_DESC srv_desc;
    D3D12_CONSTANT_BUFFER_VIEW_DESC cbv_desc;
    D3D12_ROOT_PARAMETER root_parameters[1];
    ID3D12DescriptorHeap *cpu_resource_heap;
    ID3D12GraphicsCommandList *command_list;
    D3D12_CPU_DESCRIPTOR_HANDLE cpu_handle;
    D3D12_GPU_DESCRIPTOR_HANDLE gpu_handle;
    unsigned int descriptor_size_sampler;
    ID3D12RootSignature *root_signature;
    ID3D12DescriptorHeap *resource_heap;
    ID3D12Resource *input_textures[256];
    ID3D12DescriptorHeap *sampler_heap;
    unsigned int i, descriptor_size;
    D3D12_SAMPLER_DESC sampler_desc;
    ID3D12DescriptorHeap *heaps[2];
    ID3D12Resource *output_buffer;
    ID3D12Resource *input_buffer;
    struct resource_readback rb;
    struct test_context context;
    ID3D12PipelineState *pso;
    ID3D12Device *device;
    FLOAT clear_values[4];
    unsigned int x, y;
    D3D12_RECT rect;
    HRESULT hr;

    static const BYTE cs_code[] =
    {
#if 0
cbuffer heap_offsets : register(b0)
{
        uint src_uav_tex_offset;
        uint src_srv_tex_offset;
        uint src_srv_buffer_offset;
        uint dst_uav_buffer_offset;
};

struct Dummy { uint v; };

[numthreads(64, 1, 1)]
void main(uint thr : SV_DispatchThreadID)
{
        uint input_value = 0;

        if (thr % 2 == 0)
        {
                Texture2D<float> src = ResourceDescriptorHeap[NonUniformResourceIndex(src_srv_tex_offset + thr)];
                SamplerState samp = SamplerDescriptorHeap[NonUniformResourceIndex(thr)];
                input_value += uint(src.SampleLevel(samp, float2(1.25, 1.25), 0.0));
        }
        else
        {
                RWTexture2D<float> src = ResourceDescriptorHeap[NonUniformResourceIndex(src_uav_tex_offset + thr)];
                input_value += uint(src[int2(0, 0)]);
        }

        // Test different descriptor types.
        if (thr % 16 == 0)
        {
                ConstantBuffer<Dummy> src = ResourceDescriptorHeap[NonUniformResourceIndex(src_srv_buffer_offset + thr)];
                input_value += src.v;
        }
        else if (thr % 4 == 0)
        {
                ByteAddressBuffer src = ResourceDescriptorHeap[NonUniformResourceIndex(src_srv_buffer_offset + thr)];
                input_value += src.Load(0);
        }
        else if (thr % 4 == 1)
        {
                Buffer<uint> src = ResourceDescriptorHeap[NonUniformResourceIndex(src_srv_buffer_offset + thr)];
                input_value += src[0];
        }
        else
        {
                StructuredBuffer<uint> dst = ResourceDescriptorHeap[NonUniformResourceIndex(src_srv_buffer_offset + thr)];
                input_value += dst[0];
        }

        // Test different descriptor types.
        if (thr % 4 == 0)
        {
                RWByteAddressBuffer dst = ResourceDescriptorHeap[NonUniformResourceIndex(dst_uav_buffer_offset + thr)];
                dst.Store(0, input_value);
        }
        else if (thr % 4 == 1)
        {
                RWBuffer<uint> dst = ResourceDescriptorHeap[NonUniformResourceIndex(dst_uav_buffer_offset + thr)];
                dst[0] = input_value;
        }
        else
        {
                RWStructuredBuffer<uint> dst = ResourceDescriptorHeap[NonUniformResourceIndex(dst_uav_buffer_offset + thr)];
                dst[0] = input_value;
        }
}
#endif
        0x44, 0x58, 0x42, 0x43, 0x3a, 0x81, 0x6c, 0xe3, 0x19, 0x63, 0x28, 0xb7, 0x92, 0x22, 0xd1, 0x43, 0xf7, 0xac, 0x2d, 0x5a, 0x01, 0x00, 0x00, 0x00, 0x4c, 0x0a, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0xd0, 0x00, 0x00, 0x00, 0xec, 0x00, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x60, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf1, 0x83, 0xbb, 0x3e,
        0xbd, 0xf9, 0x98, 0xb8, 0x80, 0xde, 0x7f, 0x96, 0x9b, 0x08, 0xdb, 0x6f, 0x44, 0x58, 0x49, 0x4c, 0x58, 0x09, 0x00, 0x00, 0x66, 0x00, 0x05, 0x00, 0x56, 0x02, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c,
        0x06, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x40, 0x09, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x4d, 0x02, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x18, 0x45, 0x02,
        0x42, 0x92, 0x0b, 0x42, 0xc4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x62, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90,
        0x11, 0x23, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x31, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07,
        0x40, 0x02, 0xa8, 0x0d, 0x86, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
        0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x46, 0x00, 0x00, 0x00, 0x32, 0x22, 0x88, 0x09, 0x20, 0x64, 0x85, 0x04, 0x13, 0x23, 0xa4, 0x84,
        0x04, 0x13, 0x23, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8c, 0x8c, 0x0b, 0x84, 0xc4, 0x4c, 0x10, 0xa4, 0xc1, 0x08, 0x40, 0x09, 0x00, 0x0a, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0xc3, 0x30, 0x10,
        0x31, 0x47, 0x80, 0x90, 0x71, 0xcf, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0x1f, 0x02, 0xcd, 0xb0, 0x10, 0x28, 0x38, 0xe6, 0x08, 0x82, 0x52, 0x20, 0xc3, 0x90, 0x24, 0xa4, 0x1c, 0x35, 0x5c, 0xfe,
        0x84, 0x3d, 0x84, 0xe4, 0x73, 0x1b, 0x55, 0xac, 0xc4, 0xe4, 0x23, 0xb7, 0x8d, 0x88, 0x61, 0x18, 0x86, 0x42, 0x30, 0x03, 0x32, 0x50, 0x33, 0x03, 0x70, 0xd3, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92,
        0xbf, 0x12, 0xd2, 0x4a, 0x4c, 0x7e, 0x71, 0xdb, 0xa8, 0x78, 0x9e, 0xe7, 0x19, 0x0a, 0x03, 0x0d, 0x08, 0xf2, 0x3c, 0xcf, 0x33, 0x0c, 0xc3, 0x43, 0x51, 0x51, 0xa0, 0x01, 0x19, 0x86, 0x61, 0x18,
        0x86, 0x81, 0xa6, 0x9b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x95, 0x90, 0x56, 0x62, 0xf2, 0x91, 0xdb, 0x46, 0xc5, 0x30, 0x0c, 0xc3, 0x50, 0x8e, 0x6a, 0x40, 0x86, 0xc1, 0x18, 0xc8, 0x2a,
        0x0b, 0x30, 0x20, 0xc3, 0x30, 0x0c, 0xc3, 0x60, 0x0c, 0x84, 0x15, 0x05, 0x18, 0x90, 0x61, 0x18, 0x86, 0x61, 0x30, 0x48, 0x2b, 0x45, 0x35, 0x20, 0xc3, 0x40, 0xdc, 0x6d, 0xc3, 0xe5, 0x4f, 0xd8,
        0x43, 0x48, 0xfe, 0x4a, 0x48, 0x0e, 0x15, 0x09, 0x44, 0x1a, 0x39, 0x0f, 0x11, 0x4d, 0x08, 0x21, 0x21, 0x61, 0x18, 0x0a, 0x81, 0x0c, 0x88, 0x47, 0xdf, 0x41, 0xc3, 0xe5, 0x4f, 0xd8, 0x43, 0x48,
        0xfe, 0x4a, 0x48, 0x1b, 0xd2, 0x0c, 0x88, 0x18, 0x86, 0xc1, 0x28, 0x05, 0x32, 0x84, 0x43, 0x42, 0xe2, 0x40, 0xc0, 0x31, 0x07, 0x01, 0x3c, 0xbf, 0x53, 0x14, 0x12, 0x31, 0x49, 0x88, 0x18, 0x86,
        0x61, 0xa0, 0x72, 0x8e, 0x00, 0x14, 0xa6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50,
        0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78,
        0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07,
        0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0,
        0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x86, 0x3c, 0x04, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x14, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x34,
        0x40, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x89, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x33, 0x01, 0x01, 0x20, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xc7, 0x02, 0x02, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x0f, 0x06, 0x04, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43,
        0x1e, 0x0d, 0x08, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x1c, 0x10, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x3e, 0x20, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x88, 0x01, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x64, 0x81, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14,
        0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x4a, 0xa0, 0x18, 0x46, 0x00, 0x0a, 0xa4, 0x10, 0x0a, 0x30, 0x80, 0xce, 0x02, 0x04, 0x04, 0x04, 0xc4, 0x20, 0x73, 0x06,
        0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x39, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b,
        0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9,
        0x10, 0x04, 0x13, 0x84, 0x81, 0x99, 0x20, 0x0c, 0xcd, 0x06, 0x61, 0x20, 0x28, 0x8c, 0xcd, 0x6d, 0x18, 0x0c, 0x82, 0x98, 0x20, 0x0c, 0xce, 0x04, 0xc1, 0x0c, 0x26, 0x02, 0x13, 0x84, 0xe1, 0xd9,
        0x80, 0x20, 0x89, 0x82, 0x20, 0xc3, 0x02, 0x6c, 0x08, 0x98, 0x0d, 0x04, 0x00, 0x34, 0xc0, 0x04, 0x41, 0x00, 0x48, 0xb4, 0x85, 0xa5, 0xb9, 0x4d, 0x10, 0xce, 0x40, 0x9a, 0x20, 0x0c, 0xd0, 0x04,
        0x61, 0x88, 0x36, 0x0c, 0xd3, 0x30, 0x6c, 0x20, 0x90, 0x48, 0xa2, 0x36, 0x14, 0x0f, 0x04, 0x38, 0x55, 0x15, 0x36, 0x36, 0xbb, 0x36, 0x97, 0x34, 0xb2, 0x32, 0x37, 0xba, 0x29, 0x41, 0x50, 0x85,
        0x0c, 0xcf, 0xc5, 0xae, 0x4c, 0x6e, 0x2e, 0xed, 0xcd, 0x6d, 0x4a, 0x40, 0x34, 0x21, 0xc3, 0x73, 0xb1, 0x0b, 0x63, 0xb3, 0x2b, 0x93, 0x9b, 0x12, 0x10, 0x75, 0xc8, 0xf0, 0x5c, 0xe6, 0xd0, 0xc2,
        0xc8, 0xca, 0xe4, 0x9a, 0xde, 0xc8, 0xca, 0xd8, 0xa6, 0x04, 0x46, 0x19, 0x32, 0x3c, 0x17, 0xb9, 0xb2, 0xb9, 0xb7, 0x3a, 0xb9, 0xb1, 0xb2, 0xb9, 0x29, 0x41, 0x53, 0x87, 0x0c, 0xcf, 0xa5, 0xcc,
        0x8d, 0x4e, 0x2e, 0x0f, 0xea, 0x2d, 0xcd, 0x8d, 0x6e, 0x6e, 0x4a, 0x50, 0x01, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x56, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66,
        0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e,
        0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b,
        0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0,
        0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83,
        0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76,
        0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30,
        0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43,
        0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4,
        0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90,
        0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x30, 0xc3, 0x81, 0xc8, 0x01, 0x1f, 0xdc, 0xc0, 0x1d, 0xde, 0xc1, 0x1d, 0xea, 0xc1, 0x1d, 0xd2, 0xc1, 0x1c, 0xde, 0x41, 0x1e, 0xda, 0x61, 0x06, 0x13, 0x91, 0x03,
        0x3e, 0xb8, 0x81, 0x38, 0xc8, 0x43, 0x39, 0x84, 0xc3, 0x3a, 0xb8, 0x81, 0x38, 0xc8, 0x03, 0x00, 0x71, 0x20, 0x00, 0x00, 0x37, 0x00, 0x00, 0x00, 0xa6, 0x40, 0x0d, 0x97, 0xef, 0x3c, 0x3e, 0xd0,
        0x34, 0xce, 0x04, 0x4c, 0x44, 0x08, 0x34, 0xc3, 0x42, 0x58, 0x02, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0x01, 0x55, 0x14, 0x44, 0x54, 0x3a, 0xc0, 0xe0, 0x23, 0xb7, 0x6d, 0x08, 0xd5, 0x70, 0xf9, 0xce,
        0xe3, 0x07, 0x54, 0x51, 0x10, 0x11, 0x3b, 0x39, 0x11, 0xe1, 0x23, 0xb7, 0x6d, 0x03, 0xdb, 0x70, 0xf9, 0xce, 0xe3, 0x0b, 0x01, 0x55, 0x14, 0x44, 0x54, 0x3a, 0xc0, 0x50, 0x12, 0x06, 0x20, 0x60,
        0x3e, 0x72, 0xdb, 0xb6, 0xd0, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44, 0x84, 0x40, 0x33, 0x2c, 0xc4, 0x17, 0x39, 0xcc, 0x86, 0x34, 0x03, 0xd2, 0x18, 0x26, 0xa0, 0x0d, 0x97, 0xef,
        0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44, 0x84, 0x40, 0x33, 0x2c, 0xc4, 0x17, 0x39, 0x4c, 0x48, 0x00, 0x8f, 0x19, 0x5c, 0xc3, 0xe5, 0x3b, 0x8f, 0x1f, 0x01, 0xd6, 0x46, 0x15, 0x05, 0x11, 0x95, 0x0e,
        0x30, 0xf8, 0xc8, 0x6d, 0xdb, 0x01, 0x36, 0x5c, 0xbe, 0xf3, 0xf8, 0x11, 0x60, 0x6d, 0x54, 0x51, 0x10, 0x11, 0x3b, 0x39, 0x11, 0xe1, 0x23, 0xb7, 0x6d, 0x04, 0xd5, 0x70, 0xf9, 0xce, 0xe3, 0x4b,
        0x00, 0xf3, 0x2c, 0x44, 0x49, 0x54, 0xc4, 0xe2, 0x17, 0xb7, 0x6d, 0x05, 0xd5, 0x70, 0xf9, 0xce, 0xe3, 0x4f, 0xc4, 0x35, 0x51, 0x11, 0x51, 0x3a, 0xc0, 0xe0, 0x17, 0xb7, 0x6d, 0x01, 0xd2, 0x70,
        0xf9, 0xce, 0xe3, 0x4f, 0x47, 0x44, 0x00, 0x83, 0x38, 0xf8, 0xc8, 0x6d, 0x1b, 0x00, 0xc1, 0x00, 0x48, 0x03, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0xbb, 0x00, 0x00, 0x00, 0x13, 0x04, 0x50, 0x2c,
        0x10, 0x00, 0x00, 0x00, 0x21, 0x00, 0x00, 0x00, 0x34, 0xcc, 0x00, 0x14, 0xec, 0x40, 0xd1, 0x0e, 0x94, 0x5d, 0x69, 0x94, 0x24, 0x41, 0x59, 0x0a, 0x94, 0x22, 0x41, 0x61, 0x0a, 0x94, 0xec, 0x40,
        0xe9, 0x06, 0x14, 0x41, 0x71, 0x14, 0x5f, 0x11, 0x02, 0x15, 0x61, 0x40, 0x79, 0x94, 0x41, 0x59, 0x14, 0x45, 0x21, 0x06, 0x14, 0x46, 0x59, 0x02, 0x15, 0x25, 0x50, 0x29, 0x06, 0x14, 0x26, 0x10,
        0x19, 0x25, 0x50, 0x04, 0xe5, 0x41, 0x49, 0x0d, 0x8c, 0x00, 0xd0, 0x33, 0x46, 0x00, 0x82, 0x20, 0xa8, 0x7f, 0x33, 0x00, 0x23, 0x00, 0xe4, 0xcd, 0x21, 0x60, 0xcf, 0x1c, 0xc2, 0x97, 0xcd, 0x21,
        0x80, 0x81, 0x33, 0x87, 0x20, 0x06, 0xd9, 0x1c, 0x02, 0x06, 0xcd, 0x21, 0x98, 0x81, 0x33, 0x87, 0x70, 0x06, 0xdb, 0x1c, 0x42, 0x1a, 0x40, 0x73, 0x08, 0x6a, 0xe0, 0xcc, 0x21, 0xac, 0xc1, 0x36,
        0x87, 0xd0, 0x06, 0x10, 0x85, 0x73, 0x10, 0x8e, 0xe3, 0xbc, 0x01, 0x00, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0x20, 0x06, 0x77, 0x50, 0x06, 0x01, 0x1d, 0x40, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60,
        0xf0, 0xe5, 0x81, 0x1b, 0x04, 0xcd, 0x88, 0x81, 0x01, 0x80, 0x20, 0x18, 0x10, 0xa3, 0x60, 0x06, 0x76, 0x50, 0x41, 0x1e, 0xe8, 0x70, 0x43, 0x80, 0x07, 0x60, 0x30, 0x62, 0x70, 0x00, 0x20, 0x08,
        0x06, 0x0d, 0x29, 0xc0, 0x01, 0x91, 0x07, 0xb3, 0x0c, 0x81, 0x20, 0x8c, 0x26, 0x04, 0x41, 0x05, 0x05, 0x8c, 0x18, 0x20, 0x00, 0x08, 0x82, 0x81, 0x82, 0x0a, 0x74, 0x10, 0x60, 0xd9, 0x88, 0xc1,
        0x01, 0x80, 0x20, 0x18, 0x7c, 0xa1, 0x60, 0x07, 0x01, 0x35, 0x62, 0x80, 0x00, 0x20, 0x08, 0x06, 0x8a, 0x2a, 0xd8, 0x01, 0xb2, 0x6d, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0xf0, 0x8d, 0x02, 0x1e,
        0x04, 0xd5, 0x88, 0xc1, 0x03, 0x80, 0x20, 0x18, 0x44, 0xaa, 0xd0, 0x06, 0x43, 0xb0, 0x6d, 0x9a, 0x06, 0x0a, 0xa0, 0xa0, 0x07, 0xd9, 0x68, 0x42, 0x00, 0xcc, 0x12, 0x0c, 0xa3, 0x09, 0x09, 0x50,
        0x41, 0x03, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0xa0, 0xc0, 0x02, 0x1f, 0x04, 0x60, 0x10, 0x06, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0xf0, 0xa5, 0x82, 0x1f, 0x04, 0xda, 0x88, 0x41, 0x03, 0x80,
        0x20, 0x18, 0x4c, 0xaf, 0x10, 0x07, 0x01, 0x28, 0x98, 0x82, 0x29, 0x80, 0x02, 0x28, 0x80, 0x02, 0x28, 0x8c, 0x26, 0x04, 0xc0, 0x2c, 0xc1, 0x30, 0x50, 0xf1, 0x38, 0x81, 0x20, 0x5c, 0x30, 0x4c,
        0x51, 0x74, 0xa0, 0xc3, 0x0d, 0xc1, 0x2a, 0x80, 0x41, 0x59, 0x75, 0xa0, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0xd0, 0xe0, 0x02, 0x29, 0x60, 0xad, 0x30, 0x9a, 0x10, 0x08, 0x15, 0x64, 0x30, 0x62,
        0x80, 0x00, 0x20, 0x08, 0x06, 0x0a, 0x2f, 0xa0, 0x42, 0xc0, 0x06, 0x6d, 0x30, 0xcb, 0x40, 0x14, 0xc6, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x7c, 0xb5, 0xa0, 0x0a, 0x41, 0x19, 0x8c, 0x18, 0x1c,
        0x00, 0x08, 0x82, 0x41, 0xd3, 0x0b, 0xa9, 0x10, 0xc8, 0xc2, 0x68, 0x42, 0x00, 0xcc, 0x12, 0x28, 0xc3, 0x0d, 0x08, 0x2d, 0x80, 0xc1, 0x2c, 0x83, 0x71, 0x04, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60,
        0xf0, 0xe5, 0x82, 0x2b, 0x14, 0x68, 0x30, 0x62, 0xa0, 0x00, 0x20, 0x08, 0x06, 0x96, 0x2f, 0xa8, 0x42, 0x60, 0x0b, 0xb0, 0x60, 0x07, 0xb4, 0x30, 0x9a, 0x10, 0x00, 0xb3, 0x04, 0xca, 0x70, 0x03,
        0xa3, 0x0b, 0x60, 0x30, 0xcb, 0x80, 0x24, 0xc1, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x7c, 0xbd, 0x20, 0x0b, 0xc9, 0x1a, 0x8c, 0x18, 0x20, 0x00, 0x08, 0x82, 0x01, 0xf7, 0x0b, 0xa0, 0x10, 0xe8,
        0x02, 0x2d, 0x8c, 0x26, 0x04, 0xc0, 0x2c, 0x81, 0x32, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0x1f, 0x38, 0xd4, 0x02, 0xd3, 0x06, 0x23, 0x06, 0x0a, 0x00, 0x82, 0x60, 0x60, 0x95, 0x43, 0x2c, 0x04,
        0xbd, 0xd0, 0x0b, 0x7d, 0xb0, 0x0b, 0xa3, 0x09, 0x01, 0x30, 0x4b, 0xa0, 0x0c, 0x94, 0x0c, 0x7e, 0x61, 0x98, 0x41, 0x22, 0x1a, 0x88, 0x19, 0x10, 0x03, 0x25, 0x83, 0x64, 0x10, 0x89, 0x82, 0x68,
        0x44, 0x05, 0x18, 0x0c, 0x37, 0x0c, 0xe2, 0x00, 0x06, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0xd0, 0xb8, 0x83, 0x2e, 0xb8, 0xc1, 0x38, 0x8c, 0x26, 0x04, 0x43, 0x05, 0x6f, 0x00, 0x23, 0x06, 0x08,
        0x00, 0x82, 0x60, 0xa0, 0xc8, 0x83, 0x2f, 0x04, 0xa2, 0x30, 0x0a, 0xb3, 0x0c, 0x0b, 0x53, 0x8c, 0x18, 0x1c, 0x00, 0x08, 0x82, 0xc1, 0xb7, 0x0e, 0xe0, 0x10, 0xdc, 0xc1, 0x88, 0x81, 0x03, 0x80,
        0x20, 0x18, 0x60, 0xef, 0xa0, 0x0b, 0x01, 0x3a, 0x88, 0xc3, 0x21, 0x0e, 0xe2, 0x20, 0x0e, 0xa8, 0x60, 0x0e, 0xb3, 0x04, 0xcf, 0x70, 0x43, 0xa2, 0x0e, 0x60, 0x30, 0xcb, 0xd0, 0x38, 0xc1, 0x88,
        0xc1, 0x01, 0x80, 0x20, 0x18, 0x7c, 0xed, 0x20, 0x0e, 0x03, 0x1e, 0x8c, 0x18, 0x34, 0x00, 0x08, 0x82, 0x81, 0x06, 0x0f, 0xac, 0x10, 0xa8, 0x03, 0x39, 0x24, 0x49, 0x92, 0xa0, 0xc2, 0x2c, 0xc1,
        0x33, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0x9f, 0x3b, 0x8c, 0x03, 0x81, 0x07, 0x23, 0x06, 0x0e, 0x00, 0x82, 0x60, 0x80, 0xc9, 0x43, 0x2f, 0x04, 0xeb, 0xb0, 0x0e, 0x4a, 0x39, 0x94, 0x43, 0x39,
        0xac, 0x42, 0x3a, 0xcc, 0x12, 0x3c, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
    static const D3D12_SHADER_BYTECODE cs_code_dxil = SHADER_BYTECODE(cs_code);
    uint32_t initial_buffer_data[ARRAY_SIZE(input_textures)];

    if (!init_compute_test_context(&context))
        return;

    device = context.device;
    command_list = context.list;
    descriptor_size = ID3D12Device_GetDescriptorHandleIncrementSize(device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV);
    descriptor_size_sampler = ID3D12Device_GetDescriptorHandleIncrementSize(device, D3D12_DESCRIPTOR_HEAP_TYPE_SAMPLER);

    shader_model.HighestShaderModel = D3D_SHADER_MODEL_6_6;
    hr = ID3D12Device_CheckFeatureSupport(device, D3D12_FEATURE_SHADER_MODEL,
            &shader_model, sizeof(shader_model));
    if (FAILED(hr) || shader_model.HighestShaderModel < D3D_SHADER_MODEL_6_6)
    {
        skip("Shader model 6.6 not supported by device.\n");
        destroy_test_context(&context);
        return;
    }

    memset(&root_signature_desc, 0, sizeof(root_signature_desc));
    root_signature_desc.Flags =
            D3D12_ROOT_SIGNATURE_FLAG_CBV_SRV_UAV_HEAP_DIRECTLY_INDEXED |
            D3D12_ROOT_SIGNATURE_FLAG_SAMPLER_HEAP_DIRECTLY_INDEXED;
    root_signature_desc.pParameters = root_parameters;
    root_signature_desc.NumParameters = ARRAY_SIZE(root_parameters);
    memset(root_parameters, 0, sizeof(root_parameters));
    root_parameters[0].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[0].ParameterType = D3D12_ROOT_PARAMETER_TYPE_32BIT_CONSTANTS;
    root_parameters[0].Constants.ShaderRegister = 0;
    root_parameters[0].Constants.RegisterSpace = 0;
    root_parameters[0].Constants.Num32BitValues = 4;

    hr = create_root_signature(device, &root_signature_desc, &root_signature);
    ok(SUCCEEDED(hr), "Failed to create root signature, hr #%x.\n", hr);

    pso = create_compute_pipeline_state(device, root_signature, cs_code_dxil);
    cpu_resource_heap = create_cpu_descriptor_heap(device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV,
            ARRAY_SIZE(input_textures) * root_parameters[0].Constants.Num32BitValues);
    resource_heap = create_gpu_descriptor_heap(device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV,
            ARRAY_SIZE(input_textures) * root_parameters[0].Constants.Num32BitValues);
    sampler_heap = create_gpu_descriptor_heap(device, D3D12_DESCRIPTOR_HEAP_TYPE_SAMPLER, ARRAY_SIZE(input_textures));

    heaps[0] = resource_heap;
    heaps[1] = sampler_heap;
    ID3D12GraphicsCommandList_SetDescriptorHeaps(command_list, ARRAY_SIZE(heaps), heaps);

    for (i = 0; i < ARRAY_SIZE(input_textures); i++)
    {
        input_textures[i] = create_default_texture2d(device, 2, 2, 1, 1,
                DXGI_FORMAT_R32_FLOAT,
                D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);

        memset(&uav_desc, 0, sizeof(uav_desc));
        uav_desc.Format = DXGI_FORMAT_R32_FLOAT;
        uav_desc.ViewDimension = D3D12_UAV_DIMENSION_TEXTURE2D;

        cpu_handle = ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(resource_heap);
        cpu_handle.ptr += i * descriptor_size;
        ID3D12Device_CreateUnorderedAccessView(device, input_textures[i], NULL, &uav_desc, cpu_handle);

        cpu_handle = ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(cpu_resource_heap);
        cpu_handle.ptr += i * descriptor_size;
        ID3D12Device_CreateUnorderedAccessView(device, input_textures[i], NULL, &uav_desc, cpu_handle);

        gpu_handle = ID3D12DescriptorHeap_GetGPUDescriptorHandleForHeapStart(resource_heap);
        gpu_handle.ptr += i * descriptor_size;
        ID3D12Device_CreateUnorderedAccessView(device, input_textures[i], NULL, &uav_desc, cpu_handle);

        memset(clear_values, 0, sizeof(clear_values));

        for (y = 0; y < 2; y++)
        {
            for (x = 0; x < 2; x++)
            {
                clear_values[0] = i + 1 + 1000 * (y * 2 + x);
                set_rect(&rect, x, y, x + 1, y + 1);
                ID3D12GraphicsCommandList_ClearUnorderedAccessViewFloat(command_list, gpu_handle, cpu_handle,
                        input_textures[i], clear_values, 1, &rect);
            }
        }

        memset(&srv_desc, 0, sizeof(srv_desc));
        srv_desc.Format = DXGI_FORMAT_R32_FLOAT;
        srv_desc.ViewDimension = D3D12_SRV_DIMENSION_TEXTURE2D;
        srv_desc.Shader4ComponentMapping = D3D12_DEFAULT_SHADER_4_COMPONENT_MAPPING;
        srv_desc.Texture2D.MipLevels = 1;

        cpu_handle = ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(resource_heap);
        cpu_handle.ptr += (i + 1 * ARRAY_SIZE(input_textures)) * descriptor_size;
        ID3D12Device_CreateShaderResourceView(device, input_textures[i], &srv_desc, cpu_handle);

        if (i % 2 == 0)
        {
            transition_resource_state(command_list, input_textures[i],
                    D3D12_RESOURCE_STATE_UNORDERED_ACCESS,
                    D3D12_RESOURCE_STATE_NON_PIXEL_SHADER_RESOURCE);

            memset(&sampler_desc, 0, sizeof(sampler_desc));
            sampler_desc.Filter = D3D12_FILTER_MIN_MAG_MIP_POINT;
            sampler_desc.AddressU = i % 4 == 0 ? D3D12_TEXTURE_ADDRESS_MODE_CLAMP : D3D12_TEXTURE_ADDRESS_MODE_WRAP;
            sampler_desc.AddressV = D3D12_TEXTURE_ADDRESS_MODE_CLAMP;
            sampler_desc.AddressW = sampler_desc.AddressU;

            cpu_handle = ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(sampler_heap);
            cpu_handle.ptr += i * descriptor_size_sampler;
            ID3D12Device_CreateSampler(device, &sampler_desc, cpu_handle);
        }
        else
            uav_barrier(command_list, input_textures[i]);
    }

    for (i = 0; i < ARRAY_SIZE(input_textures); i++)
        initial_buffer_data[i] = i + 10000;
    input_buffer = create_upload_buffer(device, sizeof(initial_buffer_data), initial_buffer_data);

    output_buffer = create_default_buffer(device, ARRAY_SIZE(input_textures) * sizeof(uint32_t),
            D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS,
            D3D12_RESOURCE_STATE_UNORDERED_ACCESS);

    for (i = 0; i < ARRAY_SIZE(input_textures); i++)
    {
        memset(&srv_desc, 0, sizeof(srv_desc));

        srv_desc.ViewDimension = D3D12_SRV_DIMENSION_BUFFER;
        srv_desc.Shader4ComponentMapping = D3D12_DEFAULT_SHADER_4_COMPONENT_MAPPING;

        cpu_handle = ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(resource_heap);
        cpu_handle.ptr += (i + 2 * ARRAY_SIZE(input_textures)) * descriptor_size;

        if (i % 16 == 0)
        {
            memset(&cbv_desc, 0, sizeof(cbv_desc));
            cbv_desc.SizeInBytes = 256;
            cbv_desc.BufferLocation = ID3D12Resource_GetGPUVirtualAddress(input_buffer) + (i % 32 == 0 ? 256 : 0);
            ID3D12Device_CreateConstantBufferView(device, &cbv_desc, cpu_handle);
        }
        else
        {
            if (i % 4 == 0)
            {
                srv_desc.Format = DXGI_FORMAT_R32_TYPELESS;
                srv_desc.Buffer.NumElements = 4;
                srv_desc.Buffer.FirstElement = i;
                srv_desc.Buffer.Flags = D3D12_BUFFER_SRV_FLAG_RAW;
            }
            else if (i % 4 == 1)
            {
                srv_desc.Format = DXGI_FORMAT_R32_UINT;
                srv_desc.Buffer.NumElements = 1;
                srv_desc.Buffer.FirstElement = i;
            }
            else
            {
                srv_desc.Format = DXGI_FORMAT_UNKNOWN;
                srv_desc.Buffer.StructureByteStride = 4;
                srv_desc.Buffer.FirstElement = i;
                srv_desc.Buffer.NumElements = 1;
            }

            ID3D12Device_CreateShaderResourceView(device, input_buffer, &srv_desc, cpu_handle);
        }
    }

    for (i = 0; i < ARRAY_SIZE(input_textures); i++)
    {
        memset(&uav_desc, 0, sizeof(uav_desc));

        uav_desc.ViewDimension = D3D12_UAV_DIMENSION_BUFFER;
        if (i % 4 == 0)
        {
            uav_desc.Format = DXGI_FORMAT_R32_TYPELESS;
            uav_desc.Buffer.NumElements = 4;
            uav_desc.Buffer.FirstElement = i;
            uav_desc.Buffer.Flags = D3D12_BUFFER_UAV_FLAG_RAW;
        }
        else if (i % 4 == 1)
        {
            uav_desc.Format = DXGI_FORMAT_R32_UINT;
            uav_desc.Buffer.NumElements = 1;
            uav_desc.Buffer.FirstElement = i;
        }
        else
        {
            uav_desc.Format = DXGI_FORMAT_UNKNOWN;
            uav_desc.Buffer.StructureByteStride = 4;
            uav_desc.Buffer.FirstElement = i;
            uav_desc.Buffer.NumElements = 1;
        }

        cpu_handle = ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(resource_heap);
        cpu_handle.ptr += (i + 3 * ARRAY_SIZE(input_textures)) * descriptor_size;
        ID3D12Device_CreateUnorderedAccessView(device, output_buffer, NULL, &uav_desc, cpu_handle);
    }

    ID3D12GraphicsCommandList_SetComputeRootSignature(command_list, root_signature);
    ID3D12GraphicsCommandList_SetPipelineState(command_list, pso);
    for (i = 0; i < root_parameters[0].Constants.Num32BitValues; i++)
        ID3D12GraphicsCommandList_SetComputeRoot32BitConstant(command_list, 0, i * ARRAY_SIZE(input_textures), i);
    ID3D12GraphicsCommandList_Dispatch(command_list, ARRAY_SIZE(input_textures) / 64, 1, 1);

    transition_resource_state(command_list, output_buffer, D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);
    get_buffer_readback_with_command_list(output_buffer, DXGI_FORMAT_R32_UINT, &rb, context.queue, command_list);

    for (i = 0; i < ARRAY_SIZE(input_textures); i++)
    {
        UINT value, expected;
        value = get_readback_uint(&rb, i, 0, 0);
        expected = 0;
        if (i % 2 == 0)
        {
            expected += i + 1; /* SRV texture reads. */
            if (i % 4 == 0)
                expected += 3000; /* CLAMP sampler used, we'll sample pixel (1, 1). */
            else
                expected += 2000; /* WRAP, CLAMP used, sample pixel (0, 1). */
        }
        else
        {
            expected += i + 1; /* UAV texture reads. */
        }

        if (i % 16 == 0)
        {
            /* CBV reads. */
            if (i % 32 == 0)
                expected += 64 + 10000;
            else
                expected += 10000;
        }
        else
            expected += i + 10000; /* Buffer reads. */
        ok(expected == value, "Value %u mismatch, expected %u, got %u.\n", i, expected, value);
    }

    release_resource_readback(&rb);

    for (i = 0; i < ARRAY_SIZE(input_textures); i++)
        ID3D12Resource_Release(input_textures[i]);
    ID3D12Resource_Release(input_buffer);
    ID3D12Resource_Release(output_buffer);
    ID3D12DescriptorHeap_Release(cpu_resource_heap);
    ID3D12DescriptorHeap_Release(resource_heap);
    ID3D12DescriptorHeap_Release(sampler_heap);
    ID3D12RootSignature_Release(root_signature);
    ID3D12PipelineState_Release(pso);
    destroy_test_context(&context);
}

static void test_bindless_srv(bool use_dxil)
{
    D3D12_ROOT_SIGNATURE_DESC root_signature_desc;
    D3D12_ROOT_PARAMETER root_parameters[3];
    D3D12_DESCRIPTOR_RANGE descriptor_ranges[4];
    ID3D12DescriptorHeap* heap;

    ID3D12Resource *input_buffers[256];
    ID3D12Resource *input_textures[256];
    ID3D12Resource *output_buffer;
    struct resource_readback rb;

    ID3D12GraphicsCommandList* command_list;
    D3D12_CPU_DESCRIPTOR_HANDLE cpu_handle;
    D3D12_GPU_DESCRIPTOR_HANDLE gpu_handle;
    unsigned int i, descriptor_size;
    struct test_context context;
    ID3D12CommandQueue* queue;
    HRESULT hr;

#if 0
    // Space 1 and 2 have an offset of 0 descriptors, so pattern of descriptors is
    // [ buf, tex, buf, tex ]
    StructuredBuffer<uint> Buffers[] : register(t4, space1);
    Texture2D<uint> Textures[] : register(t4, space2);

    // Space 3 and 4 have an effective offset of 2 descriptor,
    // so pattern of descriptors is still
    // [ buf, tex, buf, tex ]
    StructuredBuffer<uint> AliasBuffers[64] : register(t4, space3);
    Texture2D<uint> AliasTextures[64] : register(t4, space4);

    StructuredBuffer<uint> StandaloneBuffer : register(t100, space3);
    Texture2D<uint> StandaloneTexture : register(t199, space3);

    RWByteAddressBuffer OBuffer : register(u0);

    [numthreads(64, 1, 1)]
    void main(uint idx : SV_DispatchThreadID)
    {
        uint result = 0;

        if (idx & 1)
            result += Textures[NonUniformResourceIndex(idx)].Load(int3(0, 0, 0));
        else
            result += Buffers[NonUniformResourceIndex(idx)].Load(0);

        if (idx & 1)
            result += AliasTextures[NonUniformResourceIndex(idx)].Load(int3(0, 0, 0)) << 8;
        else
            result += AliasBuffers[NonUniformResourceIndex(idx)].Load(0) << 8;

        result *= StandaloneBuffer.Load(0);
        result *= StandaloneTexture.Load(int3(0, 0, 0));
        OBuffer.Store(4 * idx, result);
    }
#endif
    static const DWORD cs_code_dxbc[] =
    {
        0x43425844, 0x2b452826, 0x244a6b65, 0xdf8a8f8b, 0xf8326669, 0x00000001, 0x000003c4, 0x00000003,
        0x0000002c, 0x0000003c, 0x0000004c, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x00000008, 0x00000000, 0x00000008, 0x58454853, 0x00000370, 0x00050051, 0x000000dc, 0x0100086a,
        0x070000a2, 0x00307e46, 0x00000000, 0x00000004, 0xffffffff, 0x00000004, 0x00000001, 0x07001858,
        0x00307e46, 0x00000001, 0x00000004, 0xffffffff, 0x00004444, 0x00000002, 0x070000a2, 0x00307e46,
        0x00000002, 0x00000004, 0x00000043, 0x00000004, 0x00000003, 0x070000a2, 0x00307e46, 0x00000003,
        0x00000064, 0x00000064, 0x00000004, 0x00000003, 0x07001858, 0x00307e46, 0x00000004, 0x000000c7,
        0x000000c7, 0x00004444, 0x00000003, 0x07001858, 0x00307e46, 0x00000005, 0x00000004, 0x00000043,
        0x00004444, 0x00000004, 0x0600009d, 0x0031ee46, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
        0x0200005f, 0x00020012, 0x02000068, 0x00000001, 0x0400009b, 0x00000040, 0x00000001, 0x00000001,
        0x06000001, 0x00100012, 0x00000000, 0x0002000a, 0x00004001, 0x00000001, 0x0304001f, 0x0010000a,
        0x00000000, 0x04000036, 0x00100012, 0x00000000, 0x0002000a, 0x0e00002d, 0x00100012, 0x00000000,
        0x00004002, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x86207e46, 0x00020001, 0x00000001,
        0x00000004, 0x0010000a, 0x00000000, 0x04000036, 0x00100042, 0x00000000, 0x0002000a, 0x0e00002d,
        0x00100042, 0x00000000, 0x00004002, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x86207c96,
        0x00020001, 0x00000005, 0x00000004, 0x0010002a, 0x00000000, 0x07000029, 0x00100042, 0x00000000,
        0x0010002a, 0x00000000, 0x00004001, 0x00000008, 0x0700001e, 0x00100012, 0x00000000, 0x0010002a,
        0x00000000, 0x0010000a, 0x00000000, 0x01000012, 0x04000036, 0x00100022, 0x00000000, 0x0002000a,
        0x0d0000a7, 0x00100022, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x86207006,
        0x00020001, 0x00000000, 0x00000004, 0x0010001a, 0x00000000, 0x04000036, 0x00100042, 0x00000000,
        0x0002000a, 0x0d0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000,
        0x86207006, 0x00020001, 0x00000002, 0x00000004, 0x0010002a, 0x00000000, 0x07000029, 0x00100042,
        0x00000000, 0x0010002a, 0x00000000, 0x00004001, 0x00000008, 0x0700001e, 0x00100012, 0x00000000,
        0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x01000015, 0x0a0000a7, 0x00100022, 0x00000000,
        0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x00207006, 0x00000003, 0x00000064, 0x08000026,
        0x0000d000, 0x00100012, 0x00000000, 0x0010001a, 0x00000000, 0x0010000a, 0x00000000, 0x0b00002d,
        0x00100022, 0x00000000, 0x00004002, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00207e16,
        0x00000004, 0x000000c7, 0x08000026, 0x0000d000, 0x00100012, 0x00000000, 0x0010001a, 0x00000000,
        0x0010000a, 0x00000000, 0x06000029, 0x00100022, 0x00000000, 0x0002000a, 0x00004001, 0x00000002,
        0x080000a6, 0x0021e012, 0x00000000, 0x00000000, 0x0010001a, 0x00000000, 0x0010000a, 0x00000000,
        0x0100003e,
    };
    static const BYTE cs_code_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0x83, 0xe9, 0xc2, 0xf4, 0x11, 0x42, 0x30, 0xa0, 0xa7, 0xa8, 0x3a, 0xc8, 0x50, 0x35, 0xb7, 0x25, 0x01, 0x00, 0x00, 0x00, 0xf4, 0x08, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0x18, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0xac, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x43, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x43, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0xc7, 0x00, 0x00, 0x00, 0xc7, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0xd4, 0x07, 0x00, 0x00,
        0x60, 0x00, 0x05, 0x00, 0xf5, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0xbc, 0x07, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00,
        0xec, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c,
        0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x18, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xc4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x62, 0x88, 0x48, 0x90, 0x14, 0x20,
        0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x23, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x31, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x86, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff,
        0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x56, 0x00, 0x00, 0x00,
        0x32, 0x22, 0x88, 0x09, 0x20, 0x64, 0x85, 0x04, 0x13, 0x23, 0xa4, 0x84, 0x04, 0x13, 0x23, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8c, 0x8c, 0x0b, 0x84, 0xc4, 0x4c, 0x10, 0x94, 0xc1, 0x08,
        0x40, 0x09, 0x00, 0x0a, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0xc3, 0x30, 0x10, 0x71, 0xd3, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0xbf, 0x12, 0xd2, 0x4a, 0x4c, 0x3e, 0x72, 0xdb, 0xa8, 0x18, 0x86, 0x61,
        0x18, 0xe6, 0x08, 0x10, 0x3a, 0xee, 0x19, 0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2, 0x43, 0xa0, 0x19, 0x16, 0x02, 0x05, 0x48, 0x51, 0x8c, 0x21, 0x19, 0x86, 0x61, 0x18, 0x86, 0x81, 0x94, 0x39, 0x82,
        0xa0, 0x18, 0xc9, 0x70, 0x0c, 0x03, 0x43, 0x4d, 0x29, 0x8c, 0x21, 0x19, 0x06, 0x7a, 0x8a, 0x02, 0x0c, 0xc9, 0x30, 0x0c, 0xc3, 0x30, 0x1c, 0x14, 0x0d, 0x04, 0xcc, 0x44, 0x06, 0xe3, 0xc0, 0x0e,
        0xe1, 0x30, 0x0f, 0xf3, 0xe0, 0x06, 0xb3, 0x40, 0x0f, 0xf2, 0x50, 0x0f, 0xe3, 0x40, 0x0f, 0xf5, 0x20, 0x0f, 0xe5, 0x40, 0x0e, 0xa2, 0x50, 0x0f, 0xe6, 0x60, 0x0e, 0xe5, 0x20, 0x0f, 0x7c, 0x50,
        0x0f, 0xee, 0x30, 0x0f, 0xe9, 0x70, 0x0e, 0xee, 0x50, 0x0e, 0xe4, 0x00, 0x06, 0xe9, 0xe0, 0x0e, 0xf4, 0xe0, 0x07, 0x28, 0x18, 0x12, 0x80, 0xa2, 0x6a, 0x26, 0x34, 0x18, 0x07, 0x76, 0x08, 0x87,
        0x79, 0x98, 0x07, 0x37, 0xa0, 0x85, 0x72, 0xc0, 0x07, 0x7a, 0xa8, 0x07, 0x79, 0x28, 0x07, 0x39, 0x20, 0x05, 0x3e, 0xa8, 0x07, 0x77, 0x98, 0x87, 0x74, 0x38, 0x07, 0x77, 0x28, 0x07, 0x72, 0x00,
        0x83, 0x74, 0x70, 0x07, 0x7a, 0xf0, 0x03, 0x3d, 0xd0, 0x83, 0x76, 0x48, 0x07, 0x78, 0x98, 0x87, 0x5f, 0xa0, 0x87, 0x7c, 0x80, 0x87, 0x72, 0x40, 0xc1, 0x30, 0x53, 0x37, 0x0e, 0xec, 0x10, 0x0e,
        0xf3, 0x30, 0x0f, 0x6e, 0x40, 0x0b, 0xe5, 0x80, 0x0f, 0xf4, 0x50, 0x0f, 0xf2, 0x50, 0x0e, 0x72, 0x40, 0x0a, 0x7c, 0x50, 0x0f, 0xee, 0x30, 0x0f, 0xe9, 0x70, 0x0e, 0xee, 0x50, 0x0e, 0xe4, 0x00,
        0x06, 0xe9, 0xe0, 0x0e, 0xf4, 0xe0, 0x07, 0x48, 0x18, 0xae, 0x04, 0xc0, 0x28, 0x4b, 0x20, 0x8a, 0xb6, 0x04, 0xc2, 0xa8, 0x23, 0x8a, 0xb0, 0xd3, 0xa4, 0x29, 0xa2, 0x84, 0xc9, 0x5f, 0xe1, 0x0d,
        0x9b, 0x08, 0x6d, 0x18, 0x22, 0x42, 0x92, 0x36, 0xaa, 0x28, 0x88, 0x08, 0x05, 0x03, 0x85, 0x73, 0x04, 0xa0, 0x30, 0x05, 0x00, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87,
        0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d,
        0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07,
        0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20,
        0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76,
        0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x04, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x14, 0x20,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x34, 0x40, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x79, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x60, 0xc8, 0x13, 0x01, 0x01, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x16, 0x08, 0x00, 0x12, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90,
        0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x4a, 0x60, 0x04, 0xa0, 0x10, 0x8a, 0xa1, 0x06, 0x0a, 0xa3, 0x08, 0x4a, 0xa1, 0x0c, 0x0a, 0x30, 0xa0, 0x90, 0x03, 0xca, 0x71, 0xa0, 0x2c, 0x88,
        0x19, 0x01, 0xa0, 0xb1, 0x40, 0x48, 0x9b, 0x01, 0x20, 0x6e, 0x06, 0x80, 0xbc, 0x19, 0x00, 0xfa, 0x66, 0x00, 0x08, 0x9c, 0x01, 0x20, 0x71, 0x06, 0x80, 0xac, 0x19, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x79, 0x18, 0x00, 0x00, 0x5c, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x88, 0x0c, 0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26, 0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05,
        0x06, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x06, 0x63, 0x82, 0x30, 0x1c, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x0c, 0xc8, 0x06, 0x61, 0x30, 0x28, 0x8c, 0xcd, 0x4d, 0x10, 0x86, 0x64, 0xc3, 0x80, 0x24, 0xc4,
        0x04, 0xc1, 0xda, 0x08, 0x4c, 0x10, 0x06, 0x65, 0x82, 0x30, 0x2c, 0x1b, 0x12, 0x62, 0x61, 0x06, 0xa3, 0x71, 0x88, 0x63, 0x82, 0xa0, 0x55, 0x13, 0x84, 0x81, 0x99, 0x20, 0x0c, 0xcd, 0x06, 0x81,
        0x90, 0x36, 0x24, 0x03, 0xc4, 0x44, 0x46, 0x13, 0x11, 0xd3, 0x04, 0x81, 0xb3, 0x26, 0x08, 0x83, 0x33, 0x41, 0x18, 0x9e, 0x0d, 0x49, 0x54, 0x31, 0x96, 0x71, 0x39, 0xc4, 0x31, 0x41, 0xf0, 0xae,
        0x0d, 0x89, 0x95, 0x31, 0x86, 0x71, 0x45, 0xc4, 0x34, 0x41, 0xf8, 0xb0, 0x09, 0xc2, 0x00, 0x6d, 0x48, 0x8c, 0x8d, 0xb1, 0xb8, 0xc1, 0x21, 0x8e, 0x09, 0x02, 0x18, 0x64, 0x13, 0x84, 0x21, 0xda,
        0x90, 0x48, 0x1e, 0x63, 0x7d, 0x43, 0x44, 0x4c, 0x1b, 0x8c, 0x87, 0xc2, 0xb4, 0x0e, 0x0c, 0x26, 0x08, 0x62, 0xa0, 0x4d, 0x10, 0x06, 0x69, 0x82, 0xc0, 0x4c, 0x1b, 0x16, 0x42, 0x0c, 0x18, 0x82,
        0x18, 0xc6, 0x80, 0x0c, 0xc8, 0x80, 0x0c, 0x80, 0x0d, 0x41, 0x19, 0x6c, 0x20, 0xc2, 0xc0, 0x0c, 0x00, 0x60, 0x82, 0x20, 0x00, 0x1b, 0x80, 0x0d, 0x03, 0x91, 0x06, 0x69, 0xb0, 0x21, 0x50, 0x83,
        0x0d, 0xc3, 0x80, 0x06, 0x6b, 0x40, 0xa2, 0x2d, 0x2c, 0xcd, 0x6d, 0x82, 0x30, 0x06, 0xd4, 0x86, 0xe1, 0x1a, 0x86, 0x0d, 0x04, 0xe1, 0x06, 0xc6, 0x1b, 0x6c, 0x28, 0xd0, 0xa0, 0x0d, 0x80, 0x33,
        0x80, 0x83, 0x2a, 0x6c, 0x6c, 0x76, 0x6d, 0x2e, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x53, 0x82, 0xa0, 0x0a, 0x19, 0x9e, 0x8b, 0x5d, 0x99, 0xdc, 0x5c, 0xda, 0x9b, 0xdb, 0x94, 0x80, 0x68, 0x42, 0x86,
        0xe7, 0x62, 0x17, 0xc6, 0x66, 0x57, 0x26, 0x37, 0x25, 0x30, 0xea, 0x90, 0xe1, 0xb9, 0xcc, 0xa1, 0x85, 0x91, 0x95, 0xc9, 0x35, 0xbd, 0x91, 0x95, 0xb1, 0x4d, 0x09, 0x92, 0x32, 0x64, 0x78, 0x2e,
        0x72, 0x65, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x73, 0x53, 0x02, 0x33, 0xa8, 0x44, 0x86, 0xe7, 0x42, 0x97, 0x07, 0x57, 0x16, 0xe4, 0xe6, 0xf6, 0x46, 0x17, 0x46, 0x97, 0xf6, 0xe6, 0x36, 0x37,
        0x25, 0x58, 0x83, 0x3a, 0x64, 0x78, 0x2e, 0x65, 0x6e, 0x74, 0x72, 0x79, 0x50, 0x6f, 0x69, 0x6e, 0x74, 0x73, 0x53, 0x02, 0x38, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x51, 0x00, 0x00, 0x00,
        0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10,
        0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03,
        0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e,
        0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b,
        0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90,
        0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e,
        0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca,
        0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82,
        0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xcc, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x74, 0x60, 0x07, 0x37, 0x90, 0x87, 0x72, 0x98, 0x87, 0x77, 0xa8, 0x07, 0x79, 0x18, 0x87, 0x72, 0x70, 0x83, 0x70,
        0xa0, 0x07, 0x7a, 0x90, 0x87, 0x74, 0x10, 0x87, 0x7a, 0xa0, 0x87, 0x72, 0x98, 0xe1, 0x3c, 0xe4, 0x80, 0x0f, 0x6e, 0xe0, 0x0e, 0xef, 0xe0, 0x0e, 0xf5, 0xe0, 0x0e, 0xe9, 0x60, 0x0e, 0xef, 0x20,
        0x0f, 0xed, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x17, 0x00, 0x00, 0x00, 0x46, 0x40, 0x0d, 0x97, 0xef, 0x3c, 0x7e, 0x40, 0x15, 0x05, 0x11, 0x95, 0x0e, 0x30, 0xf8, 0xc8, 0x6d, 0x5b, 0x41, 0x35,
        0x5c, 0xbe, 0xf3, 0xf8, 0x01, 0x55, 0x14, 0x44, 0xc4, 0x4e, 0x4e, 0x44, 0xf8, 0xc8, 0x6d, 0xdb, 0x80, 0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x44, 0x00, 0x13, 0x11, 0x02, 0xcd, 0xb0, 0x10, 0x26,
        0x50, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0x44, 0x5c, 0x13, 0x15, 0x11, 0xa5, 0x03, 0x0c, 0x3e, 0x72, 0xdb, 0x16, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0x74, 0x44, 0x04, 0x30, 0x88, 0x83, 0x8f, 0xdc,
        0xb6, 0x01, 0x10, 0x0c, 0x80, 0x34, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x13, 0x04, 0x47, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x34, 0xcc, 0x00, 0x94,
        0x5c, 0x11, 0x06, 0x14, 0x62, 0x40, 0xe9, 0x06, 0x14, 0x44, 0x29, 0x06, 0xd0, 0x31, 0x02, 0x50, 0x02, 0xc4, 0xd4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xd0, 0x8c,
        0x41, 0x22, 0x7c, 0xdf, 0x34, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x0d, 0x19, 0x28, 0x84, 0x66, 0x51, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xd0, 0x94, 0xc1, 0x52, 0x80, 0x01, 0x56, 0x8d, 0x18,
        0x18, 0x00, 0x08, 0x82, 0x01, 0x81, 0x06, 0x89, 0x18, 0x54, 0x40, 0x06, 0x3a, 0xdc, 0x10, 0x90, 0x01, 0x18, 0xcc, 0x32, 0x08, 0x41, 0x50, 0x03, 0x19, 0xc0, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18,
        0x34, 0x6a, 0x00, 0x29, 0x67, 0x10, 0x20, 0x23, 0x06, 0x0d, 0x00, 0x82, 0x60, 0xa0, 0xb0, 0x01, 0x14, 0x9c, 0xc1, 0x19, 0x9c, 0x81, 0x24, 0x49, 0xd2, 0x68, 0x42, 0x00, 0xcc, 0x12, 0x0c, 0x75,
        0xa0, 0x01, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x41, 0xe3, 0x06, 0x94, 0xa3, 0x06, 0x01, 0x33, 0x62, 0x80, 0x00, 0x20, 0x08, 0x06, 0x8f, 0x1b, 0x4c, 0xc1, 0x1a, 0xac, 0xc1, 0x68, 0x42, 0x00,
        0xcc, 0x12, 0x0c, 0x03, 0x15, 0x83, 0x12, 0x08, 0xc2, 0x2c, 0x43, 0x41, 0x28, 0xc5, 0xb4, 0x01, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x41, 0x33, 0x07, 0xd9, 0x84, 0x06, 0x41, 0x34, 0x62, 0xd0,
        0x00, 0x20, 0x08, 0x06, 0x4a, 0x1d, 0x64, 0x01, 0x1c, 0xc0, 0x01, 0x1c, 0x6c, 0xdb, 0xb6, 0x8d, 0x26, 0x04, 0xc0, 0x2c, 0x81, 0x51, 0x50, 0x1c, 0xc0, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x34,
        0x77, 0xd0, 0x5d, 0x6e, 0x10, 0x54, 0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0xf0, 0xdc, 0x01, 0x17, 0xd0, 0x01, 0x1d, 0x8c, 0x26, 0x04, 0xc0, 0x2c, 0x81, 0x31, 0x50, 0x31, 0x28, 0x84, 0x50, 0x54,
        0xd0, 0x5d, 0x05, 0x0b, 0x8c, 0x18, 0x20, 0x00, 0x08, 0x82, 0xc1, 0xc3, 0x07, 0x61, 0x80, 0xe5, 0x41, 0x1e, 0x8c, 0x26, 0x04, 0x40, 0x05, 0x83, 0x8c, 0x18, 0x34, 0x00, 0x08, 0x82, 0x81, 0x12,
        0x0a, 0x65, 0xc0, 0xf1, 0x01, 0x1f, 0xf0, 0xc1, 0x19, 0x9c, 0xc1, 0x19, 0x9c, 0xc1, 0x68, 0x42, 0x00, 0xd4, 0x10, 0x48, 0x75, 0x7a, 0x70, 0x23, 0x06, 0x0d, 0x00, 0x82, 0x60, 0x10, 0x89, 0x42,
        0x19, 0x84, 0x41, 0xb0, 0x06, 0xc2, 0x1a, 0xac, 0xc1, 0x1a, 0x8c, 0x01, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    if (!init_compute_test_context(&context))
        return;

    if (use_dxil && !context_supports_dxil(&context))
    {
        destroy_test_context(&context);
        return;
    }

    command_list = context.list;
    queue = context.queue;

    root_signature_desc.NumParameters = 3;
    root_signature_desc.Flags = 0;
    root_signature_desc.NumStaticSamplers = 0;
    root_signature_desc.pStaticSamplers = NULL;
    root_signature_desc.pParameters = root_parameters;

    root_parameters[0].ParameterType = D3D12_ROOT_PARAMETER_TYPE_DESCRIPTOR_TABLE;
    root_parameters[0].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[0].DescriptorTable.NumDescriptorRanges = 2;
    root_parameters[0].DescriptorTable.pDescriptorRanges = &descriptor_ranges[0];

    root_parameters[1].ParameterType = D3D12_ROOT_PARAMETER_TYPE_DESCRIPTOR_TABLE;
    root_parameters[1].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[1].DescriptorTable.NumDescriptorRanges = 2;
    root_parameters[1].DescriptorTable.pDescriptorRanges = &descriptor_ranges[2];

    root_parameters[2].ParameterType = D3D12_ROOT_PARAMETER_TYPE_UAV;
    root_parameters[2].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[2].Descriptor.RegisterSpace = 0;
    root_parameters[2].Descriptor.ShaderRegister = 0;

    /* Need two idential ranges so we can alias two different resource dimensions over same table. */
    for (i = 0; i < 4; i++)
    {
        descriptor_ranges[i].RegisterSpace = i + 1;
        descriptor_ranges[i].BaseShaderRegister = 4;
        descriptor_ranges[i].OffsetInDescriptorsFromTableStart = i >= 2 ? 1 : 0;
        descriptor_ranges[i].NumDescriptors = 64 * 1024;
        descriptor_ranges[i].RangeType = D3D12_DESCRIPTOR_RANGE_TYPE_SRV;
    }

    hr = create_root_signature(context.device, &root_signature_desc, &context.root_signature);
    ok(SUCCEEDED(hr), "Failed to create root signature, hr %#x.\n", hr);

    for (i = 0; i < 256; i++)
    {
        const UINT buffer_data[] = { i * 2, i * 2, i * 2, i * 2 };
        input_buffers[i] = create_default_buffer(context.device, sizeof(buffer_data), D3D12_RESOURCE_FLAG_NONE, D3D12_RESOURCE_STATE_COPY_DEST);
        upload_buffer_data(input_buffers[i], 0, sizeof(buffer_data), buffer_data, queue, command_list);
        reset_command_list(command_list, context.allocator);
        transition_resource_state(command_list, input_buffers[i], D3D12_RESOURCE_STATE_COPY_DEST, D3D12_RESOURCE_STATE_NON_PIXEL_SHADER_RESOURCE);
    }

    for (i = 0; i < 256; i++)
    {
        const UINT tex_data = i * 2 + 1;
        D3D12_SUBRESOURCE_DATA sub;
        sub.pData = &tex_data;
        sub.RowPitch = 1;
        sub.SlicePitch = 1;
        input_textures[i] = create_default_texture2d(context.device, 1, 1, 1, 1, DXGI_FORMAT_R32_UINT, D3D12_RESOURCE_FLAG_NONE, D3D12_RESOURCE_STATE_COPY_DEST);
        upload_texture_data(input_textures[i], &sub, 1, queue, command_list);
        reset_command_list(command_list, context.allocator);
        transition_resource_state(command_list, input_textures[i], D3D12_RESOURCE_STATE_COPY_DEST, D3D12_RESOURCE_STATE_NON_PIXEL_SHADER_RESOURCE);
    }
    output_buffer = create_default_buffer(context.device, 4 * 64, D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);

    context.pipeline_state = create_compute_pipeline_state(context.device,
        context.root_signature,
        shader_bytecode(use_dxil ? (const void*)cs_code_dxil : (const void*)cs_code_dxbc, use_dxil ? sizeof(cs_code_dxil) : sizeof(cs_code_dxbc)));

    heap = create_gpu_descriptor_heap(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV, 128 * 1024);
    cpu_handle = ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(heap);
    gpu_handle = ID3D12DescriptorHeap_GetGPUDescriptorHandleForHeapStart(heap);
    descriptor_size = ID3D12Device_GetDescriptorHandleIncrementSize(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV);
    for (i = 0; i < 512; i++)
    {
        D3D12_SHADER_RESOURCE_VIEW_DESC view;
        D3D12_CPU_DESCRIPTOR_HANDLE h = cpu_handle;
        view.Shader4ComponentMapping = D3D12_DEFAULT_SHADER_4_COMPONENT_MAPPING;
        h.ptr += (1024 + i) * descriptor_size;

        /* Every other resource is a buffer and texture SRV which are aliased over the same descriptor table range. */
        if (i & 1)
        {
            view.Format = DXGI_FORMAT_R32_UINT;
            view.ViewDimension = D3D12_SRV_DIMENSION_TEXTURE2D;
            view.Texture2D.MipLevels = 1;
            view.Texture2D.MostDetailedMip = 0;
            view.Texture2D.PlaneSlice = 0;
            view.Texture2D.ResourceMinLODClamp = 0;
            ID3D12Device_CreateShaderResourceView(context.device, input_textures[i >> 1], &view, h);
        }
        else
        {
            view.Format = DXGI_FORMAT_UNKNOWN;
            view.ViewDimension = D3D12_SRV_DIMENSION_BUFFER;
            view.Buffer.FirstElement = 0;
            view.Buffer.NumElements = 4;
            view.Buffer.StructureByteStride = 4;
            view.Buffer.Flags = 0;
            ID3D12Device_CreateShaderResourceView(context.device, input_buffers[i >> 1], &view, h);
        }
    }

    ID3D12GraphicsCommandList_SetComputeRootSignature(command_list, context.root_signature);
    ID3D12GraphicsCommandList_SetPipelineState(command_list, context.pipeline_state);
    ID3D12GraphicsCommandList_SetDescriptorHeaps(command_list, 1, &heap);
    {
        D3D12_GPU_DESCRIPTOR_HANDLE gpu = gpu_handle;
        gpu.ptr += 1024 * descriptor_size;
        ID3D12GraphicsCommandList_SetComputeRootDescriptorTable(command_list, 0, gpu);
        gpu.ptr += descriptor_size;
        ID3D12GraphicsCommandList_SetComputeRootDescriptorTable(command_list, 1, gpu);
    }
    ID3D12GraphicsCommandList_SetComputeRootUnorderedAccessView(command_list, 2, ID3D12Resource_GetGPUVirtualAddress(output_buffer));
    ID3D12GraphicsCommandList_Dispatch(command_list, 1, 1, 1);

    transition_resource_state(command_list, output_buffer, D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);
    get_buffer_readback_with_command_list(output_buffer, DXGI_FORMAT_UNKNOWN, &rb, queue, command_list);

    for (i = 0; i < 64; i++)
    {
        UINT value = get_readback_uint(&rb, i, 0, 0);
        UINT reference = (i + (i + 2) * 256) * 98 * 197;
        ok(value == reference, "Readback value is: %u\n", value);
    }

    release_resource_readback(&rb);
    reset_command_list(command_list, context.allocator);

    for (i = 0; i < 256; i++)
    {
        ID3D12Resource_Release(input_buffers[i]);
        ID3D12Resource_Release(input_textures[i]);
    }
    ID3D12Resource_Release(output_buffer);
    ID3D12DescriptorHeap_Release(heap);
    destroy_test_context(&context);
}

void test_bindless_srv_sm51(void)
{
    test_bindless_srv(false);
}

void test_bindless_srv_dxil(void)
{
    test_bindless_srv(true);
}

static void test_bindless_samplers(bool use_dxil)
{
    D3D12_ROOT_SIGNATURE_DESC root_signature_desc;
    D3D12_ROOT_PARAMETER root_parameters[3];
    D3D12_DESCRIPTOR_RANGE descriptor_ranges[2];
    ID3D12DescriptorHeap *heaps[2];

    ID3D12Resource* input_texture;
    ID3D12Resource* output_buffer;
    struct resource_readback rb;

    ID3D12GraphicsCommandList* command_list;
    D3D12_CPU_DESCRIPTOR_HANDLE cpu_handle;
    unsigned int i, descriptor_size;
    struct test_context context;
    ID3D12CommandQueue* queue;
    HRESULT hr;


#if 0
    Texture2D<float> Tex : register(t0);
    SamplerState Samp[] : register(s0);
    RWByteAddressBuffer OBuffer : register(u0);

    [numthreads(64, 1, 1)]
    void main(uint idx : SV_DispatchThreadID)
    {
        // Should alternate between wrap (sample 0), or clamp (sample 100).
        uint value = Tex.SampleLevel(Samp[NonUniformResourceIndex(idx)], 1.1.xx, 0.0);
        OBuffer.Store(4 * idx, value);
    }
#endif
    static const DWORD cs_code_dxbc[] =
    {
        0x43425844, 0x44f291da, 0x1146ced9, 0x71030deb, 0x28d62ae2, 0x00000001, 0x00000174, 0x00000003,
        0x0000002c, 0x0000003c, 0x0000004c, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x00000008, 0x00000000, 0x00000008, 0x58454853, 0x00000120, 0x00050051, 0x00000048, 0x0100086a,
        0x0600005a, 0x00306e46, 0x00000000, 0x00000000, 0xffffffff, 0x00000000, 0x07001858, 0x00307e46,
        0x00000000, 0x00000000, 0x00000000, 0x00005555, 0x00000000, 0x0600009d, 0x0031ee46, 0x00000000,
        0x00000000, 0x00000000, 0x00000000, 0x0200005f, 0x00020012, 0x02000068, 0x00000001, 0x0400009b,
        0x00000040, 0x00000001, 0x00000001, 0x04000036, 0x00100012, 0x00000000, 0x0002000a, 0x12000048,
        0x00100012, 0x00000000, 0x00004002, 0x3f8ccccd, 0x3f8ccccd, 0x00000000, 0x00000000, 0x00207e46,
        0x00000000, 0x00000000, 0x84206000, 0x00020001, 0x00000000, 0x0010000a, 0x00000000, 0x00004001,
        0x00000000, 0x0500001c, 0x00100012, 0x00000000, 0x0010000a, 0x00000000, 0x06000029, 0x00100022,
        0x00000000, 0x0002000a, 0x00004001, 0x00000002, 0x080000a6, 0x0021e012, 0x00000000, 0x00000000,
        0x0010001a, 0x00000000, 0x0010000a, 0x00000000, 0x0100003e,
    };
    static const BYTE cs_code_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0x08, 0x79, 0xe1, 0xcc, 0xec, 0xa8, 0x89, 0x07, 0x4c, 0x25, 0xe5, 0xf7, 0x5d, 0x5a, 0x63, 0x60, 0x01, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0xd8, 0x00, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x6c, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x20, 0x06, 0x00, 0x00,
        0x60, 0x00, 0x05, 0x00, 0x88, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x08, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00,
        0x7f, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c,
        0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20,
        0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x86, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff,
        0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
        0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x70, 0x23, 0x00,
        0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0x63, 0x0c, 0x22, 0x33, 0x00, 0x37, 0x0d, 0x97, 0x3f, 0x61, 0x0f, 0x21, 0xf9, 0x2b, 0x21, 0xad, 0xc4, 0xe4, 0x17, 0xb7, 0x8d, 0x0a, 0x63, 0x8c,
        0x19, 0x73, 0x04, 0x08, 0xa1, 0x7b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x10, 0x68, 0x86, 0x85, 0x40, 0x41, 0x2a, 0xcc, 0x19, 0x4a, 0x19, 0x63, 0xcc, 0x18, 0xc3, 0xd0, 0x9a, 0x23, 0x08,
        0x8a, 0xa1, 0x06, 0x1a, 0xa3, 0x91, 0x2b, 0x0a, 0x18, 0x6a, 0x8c, 0x31, 0xc6, 0x40, 0x04, 0x07, 0x02, 0x66, 0x0a, 0x83, 0x71, 0x60, 0x87, 0x70, 0x98, 0x87, 0x79, 0x70, 0x03, 0x5a, 0x28, 0x07,
        0x7c, 0xa0, 0x87, 0x7a, 0x90, 0x87, 0x72, 0x90, 0x03, 0x52, 0xe0, 0x03, 0x73, 0x60, 0x87, 0x77, 0x08, 0x07, 0x7a, 0xf0, 0x03, 0x3d, 0xd0, 0x83, 0x76, 0x48, 0x07, 0x78, 0x98, 0x87, 0x5f, 0xa0,
        0x87, 0x7c, 0x80, 0x87, 0x72, 0x40, 0xc1, 0x98, 0xc9, 0x1a, 0x07, 0x76, 0x08, 0x87, 0x79, 0x98, 0x07, 0x37, 0xa0, 0x85, 0x72, 0xc0, 0x07, 0x7a, 0xa8, 0x07, 0x79, 0x28, 0x07, 0x39, 0x20, 0x05,
        0x3e, 0x30, 0x07, 0x76, 0x78, 0x87, 0x70, 0xa0, 0x07, 0x3f, 0x40, 0x82, 0x99, 0x44, 0x4f, 0x93, 0xa6, 0x88, 0x12, 0x26, 0x7f, 0x85, 0x37, 0x6c, 0x22, 0xb4, 0x61, 0x88, 0x08, 0x49, 0xda, 0xa8,
        0xa2, 0x20, 0x22, 0x14, 0x0c, 0xb2, 0x37, 0x49, 0x53, 0x44, 0x09, 0x93, 0xcf, 0x02, 0xcc, 0xb3, 0x10, 0x11, 0x3b, 0x01, 0x13, 0x81, 0x82, 0x91, 0x00, 0x98, 0xf2, 0x1c, 0x01, 0x28, 0x00, 0x00,
        0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30,
        0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a,
        0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07,
        0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60,
        0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x04, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x16, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x38, 0x40, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4,
        0x81, 0x80, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x0b, 0x04, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47,
        0xc6, 0x04, 0x43, 0x1a, 0x25, 0x30, 0x02, 0x50, 0x08, 0xc5, 0x50, 0x04, 0x25, 0x51, 0x16, 0x35, 0x50, 0x80, 0x01, 0xd4, 0x46, 0x00, 0x68, 0x17, 0x08, 0xdd, 0x19, 0x00, 0xd2, 0x33, 0x00, 0x54,
        0x67, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x88, 0x0c, 0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26, 0x26, 0xc7,
        0x05, 0xa6, 0xc6, 0x05, 0x06, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x86, 0x62, 0x82, 0x30, 0x18, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x0c, 0xc7, 0x06, 0x61, 0x30, 0x28, 0x8c, 0xcd, 0x4d, 0x10, 0x06, 0x64,
        0xc3, 0x80, 0x24, 0xc4, 0x04, 0xa1, 0x92, 0x08, 0x4c, 0x10, 0x86, 0x64, 0x82, 0x30, 0x28, 0x1b, 0x04, 0xc2, 0xd9, 0x90, 0x10, 0x0b, 0x43, 0x10, 0x43, 0x43, 0x3c, 0x1b, 0x02, 0x68, 0x82, 0x70,
        0x41, 0x13, 0x84, 0x61, 0x99, 0x20, 0x34, 0xce, 0x86, 0x85, 0x90, 0x18, 0x82, 0x18, 0x26, 0x8a, 0xa2, 0x80, 0x0d, 0x41, 0x35, 0x41, 0xd0, 0xa2, 0x09, 0xc2, 0xc0, 0x6c, 0x40, 0x88, 0x8b, 0x21,
        0x08, 0x8c, 0x00, 0x36, 0x04, 0xd9, 0x06, 0x22, 0xb2, 0x00, 0x6d, 0x82, 0x20, 0x00, 0x1b, 0x80, 0x0d, 0x03, 0xd1, 0x75, 0x1b, 0x02, 0x6f, 0xc3, 0x30, 0x70, 0x1f, 0x89, 0xb6, 0xb0, 0x34, 0xb7,
        0x09, 0xc2, 0xf6, 0x4c, 0x10, 0x86, 0x66, 0xc3, 0x30, 0x06, 0xc3, 0xb0, 0x81, 0x20, 0xc4, 0xc0, 0x20, 0x83, 0x0d, 0x05, 0x17, 0x06, 0xc0, 0x56, 0x06, 0x55, 0xd8, 0xd8, 0xec, 0xda, 0x5c, 0xd2,
        0xc8, 0xca, 0xdc, 0xe8, 0xa6, 0x04, 0x41, 0x15, 0x32, 0x3c, 0x17, 0xbb, 0x32, 0xb9, 0xb9, 0xb4, 0x37, 0xb7, 0x29, 0x01, 0xd1, 0x84, 0x0c, 0xcf, 0xc5, 0x2e, 0x8c, 0xcd, 0xae, 0x4c, 0x6e, 0x4a,
        0x60, 0xd4, 0x21, 0xc3, 0x73, 0x99, 0x43, 0x0b, 0x23, 0x2b, 0x93, 0x6b, 0x7a, 0x23, 0x2b, 0x63, 0x9b, 0x12, 0x24, 0x65, 0xc8, 0xf0, 0x5c, 0xe4, 0xca, 0xe6, 0xde, 0xea, 0xe4, 0xc6, 0xca, 0xe6,
        0xa6, 0x04, 0x5a, 0x25, 0x32, 0x3c, 0x17, 0xba, 0x3c, 0xb8, 0xb2, 0x20, 0x37, 0xb7, 0x37, 0xba, 0x30, 0xba, 0xb4, 0x37, 0xb7, 0xb9, 0x29, 0xc1, 0x57, 0x87, 0x0c, 0xcf, 0xa5, 0xcc, 0x8d, 0x4e,
        0x2e, 0x0f, 0xea, 0x2d, 0xcd, 0x8d, 0x6e, 0x6e, 0x4a, 0x50, 0x06, 0x00, 0x79, 0x18, 0x00, 0x00, 0x51, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88,
        0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce,
        0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48,
        0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e,
        0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b,
        0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78,
        0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1,
        0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39,
        0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xcc, 0x21, 0x07, 0x7c, 0x70,
        0x03, 0x74, 0x60, 0x07, 0x37, 0x90, 0x87, 0x72, 0x98, 0x87, 0x77, 0xa8, 0x07, 0x79, 0x18, 0x87, 0x72, 0x70, 0x83, 0x70, 0xa0, 0x07, 0x7a, 0x90, 0x87, 0x74, 0x10, 0x87, 0x7a, 0xa0, 0x87, 0x72,
        0x98, 0xe1, 0x3c, 0xe4, 0x80, 0x0f, 0x6e, 0xe0, 0x0e, 0xef, 0xe0, 0x0e, 0xf5, 0xe0, 0x0e, 0xe9, 0x60, 0x0e, 0xef, 0x20, 0x0f, 0xed, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00,
        0x46, 0x50, 0x0d, 0x97, 0xef, 0x3c, 0x7e, 0x40, 0x15, 0x05, 0x11, 0xb1, 0x93, 0x13, 0x11, 0x3e, 0x72, 0xdb, 0x36, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44, 0x84, 0x40, 0x33,
        0x2c, 0x84, 0x09, 0x54, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f, 0x01, 0xcc, 0xb3, 0x10, 0x25, 0x51, 0x11, 0x8b, 0x5f, 0xdc, 0xb6, 0x05, 0x48, 0xc3, 0xe5, 0x3b, 0x8f, 0x3f, 0x1d, 0x11, 0x01, 0x0c, 0xe2,
        0xe0, 0x23, 0xb7, 0x6d, 0x00, 0x04, 0x03, 0x20, 0x0d, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x22, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x34, 0x66, 0x00, 0x4a, 0xae, 0x74, 0x03, 0x8a, 0xaf, 0x14, 0x03, 0xa8, 0xd5, 0x00, 0xa1, 0x32, 0x18, 0x01, 0x28, 0x01, 0x32, 0x23, 0x00, 0x33, 0x00, 0x63, 0x04, 0xad, 0x39, 0xe7, 0xf8, 0x07,
        0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xe0, 0x70, 0x0b, 0x91, 0x65, 0xd1, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x38, 0x1d, 0x63, 0x68, 0x9a, 0x34, 0x62, 0x60, 0x00, 0x20, 0x08, 0x06, 0x04, 0x18,
        0x30, 0x5b, 0x05, 0x1c, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x81, 0x03, 0x06, 0x8f, 0xd2, 0x05, 0xcb, 0x88, 0xc1, 0x03, 0x80, 0x20, 0x18, 0x2c, 0x62, 0xe0, 0x10, 0x81, 0x61, 0x1c, 0x47, 0x14,
        0x45, 0xc8, 0x68, 0x42, 0x00, 0x5c, 0x30, 0xa6, 0x0c, 0xef, 0x46, 0x0c, 0x1a, 0x00, 0x04, 0xc1, 0x00, 0x22, 0x83, 0x28, 0x09, 0x2a, 0xa1, 0xaa, 0xaa, 0x06, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    if (!init_compute_test_context(&context))
        return;

    if (use_dxil && !context_supports_dxil(&context))
    {
        destroy_test_context(&context);
        return;
    }

    command_list = context.list;
    queue = context.queue;

    root_signature_desc.NumParameters = 3;
    root_signature_desc.Flags = 0;
    root_signature_desc.NumStaticSamplers = 0;
    root_signature_desc.pStaticSamplers = NULL;
    root_signature_desc.pParameters = root_parameters;

    root_parameters[0].ParameterType = D3D12_ROOT_PARAMETER_TYPE_DESCRIPTOR_TABLE;
    root_parameters[0].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[0].DescriptorTable.NumDescriptorRanges = 1;
    root_parameters[0].DescriptorTable.pDescriptorRanges = &descriptor_ranges[0];

    root_parameters[1].ParameterType = D3D12_ROOT_PARAMETER_TYPE_DESCRIPTOR_TABLE;
    root_parameters[1].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[1].DescriptorTable.NumDescriptorRanges = 1;
    root_parameters[1].DescriptorTable.pDescriptorRanges = &descriptor_ranges[1];

    root_parameters[2].ParameterType = D3D12_ROOT_PARAMETER_TYPE_UAV;
    root_parameters[2].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[2].Descriptor.RegisterSpace = 0;
    root_parameters[2].Descriptor.ShaderRegister = 0;

    descriptor_ranges[0].RegisterSpace = 0;
    descriptor_ranges[0].BaseShaderRegister = 0;
    descriptor_ranges[0].OffsetInDescriptorsFromTableStart = 0;
    descriptor_ranges[0].NumDescriptors = 1;
    descriptor_ranges[0].RangeType = D3D12_DESCRIPTOR_RANGE_TYPE_SRV;

    descriptor_ranges[1].RegisterSpace = 0;
    descriptor_ranges[1].BaseShaderRegister = 0;
    descriptor_ranges[1].OffsetInDescriptorsFromTableStart = 0;
    descriptor_ranges[1].NumDescriptors = 1024;
    descriptor_ranges[1].RangeType = D3D12_DESCRIPTOR_RANGE_TYPE_SAMPLER;

    hr = create_root_signature(context.device, &root_signature_desc, &context.root_signature);
    ok(SUCCEEDED(hr), "Failed to create root signature, hr %#x.\n", hr);

    {
        const float tex_data[] = { 10, 100, 100, 100 };
        D3D12_SUBRESOURCE_DATA sub;
        sub.pData = tex_data;
        sub.RowPitch = 8;
        sub.SlicePitch = 8;
        input_texture = create_default_texture2d(context.device, 2, 2, 1, 1, DXGI_FORMAT_R32_FLOAT, D3D12_RESOURCE_FLAG_NONE, D3D12_RESOURCE_STATE_COPY_DEST);
        upload_texture_data(input_texture, &sub, 1, queue, command_list);
        reset_command_list(command_list, context.allocator);
        transition_resource_state(command_list, input_texture, D3D12_RESOURCE_STATE_COPY_DEST, D3D12_RESOURCE_STATE_NON_PIXEL_SHADER_RESOURCE);
    }

    output_buffer = create_default_buffer(context.device, 4 * 64, D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);

    context.pipeline_state = create_compute_pipeline_state(context.device,
        context.root_signature,
        shader_bytecode(use_dxil ? (const void*)cs_code_dxil : (const void*)cs_code_dxbc, use_dxil ? sizeof(cs_code_dxil) : sizeof(cs_code_dxbc)));

    heaps[0] = create_gpu_descriptor_heap(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV, 1);
    heaps[1] = create_gpu_descriptor_heap(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_SAMPLER, 1024);

    {
        D3D12_SHADER_RESOURCE_VIEW_DESC view;
        view.Shader4ComponentMapping = D3D12_DEFAULT_SHADER_4_COMPONENT_MAPPING;
        view.Format = DXGI_FORMAT_R32_FLOAT;
        view.ViewDimension = D3D12_SRV_DIMENSION_TEXTURE2D;
        view.Texture2D.MipLevels = 1;
        view.Texture2D.MostDetailedMip = 0;
        view.Texture2D.PlaneSlice = 0;
        view.Texture2D.ResourceMinLODClamp = 0;
        ID3D12Device_CreateShaderResourceView(context.device, input_texture, &view,
                ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(heaps[0]));
    }

    for (i = 0; i < 1024; i++)
    {
        D3D12_SAMPLER_DESC samp;
        memset(&samp, 0, sizeof(samp));
        samp.Filter = D3D12_FILTER_MIN_MAG_MIP_POINT;
        samp.AddressU = samp.AddressV = samp.AddressW = (i & 1) ? D3D12_TEXTURE_ADDRESS_MODE_CLAMP : D3D12_TEXTURE_ADDRESS_MODE_WRAP;
        cpu_handle = ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(heaps[1]);
        descriptor_size = ID3D12Device_GetDescriptorHandleIncrementSize(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_SAMPLER);
        cpu_handle.ptr += descriptor_size * i;
        ID3D12Device_CreateSampler(context.device, &samp, cpu_handle);
    }

    ID3D12GraphicsCommandList_SetComputeRootSignature(command_list, context.root_signature);
    ID3D12GraphicsCommandList_SetPipelineState(command_list, context.pipeline_state);
    ID3D12GraphicsCommandList_SetDescriptorHeaps(command_list, 2, heaps);
    ID3D12GraphicsCommandList_SetComputeRootDescriptorTable(command_list, 0, ID3D12DescriptorHeap_GetGPUDescriptorHandleForHeapStart(heaps[0]));
    ID3D12GraphicsCommandList_SetComputeRootDescriptorTable(command_list, 1, ID3D12DescriptorHeap_GetGPUDescriptorHandleForHeapStart(heaps[1]));
    ID3D12GraphicsCommandList_SetComputeRootUnorderedAccessView(command_list, 2, ID3D12Resource_GetGPUVirtualAddress(output_buffer));
    ID3D12GraphicsCommandList_Dispatch(command_list, 1, 1, 1);

    transition_resource_state(command_list, output_buffer, D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);
    get_buffer_readback_with_command_list(output_buffer, DXGI_FORMAT_UNKNOWN, &rb, queue, command_list);

    for (i = 0; i < 64; i++)
    {
        UINT value = get_readback_uint(&rb, i, 0, 0);
        UINT reference = (i & 1) ? 100 : 10;
        ok(value == reference, "Readback value for index %u is: %u\n", i, value);
    }

    release_resource_readback(&rb);
    reset_command_list(command_list, context.allocator);

    ID3D12Resource_Release(input_texture);
    ID3D12Resource_Release(output_buffer);
    ID3D12DescriptorHeap_Release(heaps[0]);
    ID3D12DescriptorHeap_Release(heaps[1]);
    destroy_test_context(&context);
}

void test_bindless_samplers_sm51(void)
{
    test_bindless_samplers(false);
}

void test_bindless_samplers_dxil(void)
{
    test_bindless_samplers(true);
}

void test_bindless_full_root_parameters_sm51(void)
{
    D3D12_ROOT_SIGNATURE_DESC root_signature_desc;
    D3D12_ROOT_PARAMETER root_parameters[63];
    D3D12_DESCRIPTOR_RANGE descriptor_ranges[63];
    ID3D12DescriptorHeap* heap;

    ID3D12Resource* input_buffers[1024];
    ID3D12Resource* output_buffer;
    struct resource_readback rb;

    ID3D12GraphicsCommandList* command_list;
    D3D12_CPU_DESCRIPTOR_HANDLE cpu_handle;
    D3D12_GPU_DESCRIPTOR_HANDLE gpu_handle;
    unsigned int i, descriptor_size;
    struct test_context context;
    ID3D12CommandQueue* queue;
    HRESULT hr;

    static const DWORD cs_code[] = {
#if 0
        #define R(x) StructuredBuffer<uint> Buffers##x[] : register(t0, space##x)
        R(0); R(1); R(2); R(3); R(4); R(5); R(6); R(7); R(8); R(9);
        R(10); R(11); R(12); R(13); R(14); R(15); R(16); R(17); R(18); R(19);
        R(20); R(21); R(22); R(23); R(24); R(25); R(26); R(27); R(28); R(29);
        R(30); R(31); R(32); R(33); R(34); R(35); R(36); R(37); R(38); R(39);
        R(40); R(41); R(42); R(43); R(44); R(45); R(46); R(47); R(48); R(49);
        R(50); R(51); R(52); R(53); R(54); R(55); R(56); R(57); R(58); R(59);
        R(60); R(61);
        #undef R

        RWByteAddressBuffer OBuffer : register(u0, space62);

        [numthreads(64, 1, 1)]
        void main(uint idx : SV_DispatchThreadID)
        {
            uint result = 0;
        #define R(x) result += Buffers##x[NonUniformResourceIndex(idx)].Load(0)
            R(0); R(1); R(2); R(3); R(4); R(5); R(6); R(7); R(8); R(9);
            R(10); R(11); R(12); R(13); R(14); R(15); R(16); R(17); R(18); R(19);
            R(20); R(21); R(22); R(23); R(24); R(25); R(26); R(27); R(28); R(29);
            R(30); R(31); R(32); R(33); R(34); R(35); R(36); R(37); R(38); R(39);
            R(40); R(41); R(42); R(43); R(44); R(45); R(46); R(47); R(48); R(49);
            R(50); R(51); R(52); R(53); R(54); R(55); R(56); R(57); R(58); R(59);
            R(60); R(61);
        #undef R
            OBuffer.Store(4 * idx, result);
        }
#endif
        0x43425844, 0x650a9e9e, 0x8d75a363, 0x0ea5772b, 0x31899012, 0x00000001, 0x000019f8, 0x00000003,
        0x0000002c, 0x0000003c, 0x0000004c, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x00000008, 0x00000000, 0x00000008, 0x58454853, 0x000019a4, 0x00050051, 0x00000669, 0x0100086a,
        0x070000a2, 0x00307e46, 0x00000000, 0x00000000, 0xffffffff, 0x00000004, 0x00000000, 0x070000a2,
        0x00307e46, 0x00000001, 0x00000000, 0xffffffff, 0x00000004, 0x00000001, 0x070000a2, 0x00307e46,
        0x00000002, 0x00000000, 0xffffffff, 0x00000004, 0x00000002, 0x070000a2, 0x00307e46, 0x00000003,
        0x00000000, 0xffffffff, 0x00000004, 0x00000003, 0x070000a2, 0x00307e46, 0x00000004, 0x00000000,
        0xffffffff, 0x00000004, 0x00000004, 0x070000a2, 0x00307e46, 0x00000005, 0x00000000, 0xffffffff,
        0x00000004, 0x00000005, 0x070000a2, 0x00307e46, 0x00000006, 0x00000000, 0xffffffff, 0x00000004,
        0x00000006, 0x070000a2, 0x00307e46, 0x00000007, 0x00000000, 0xffffffff, 0x00000004, 0x00000007,
        0x070000a2, 0x00307e46, 0x00000008, 0x00000000, 0xffffffff, 0x00000004, 0x00000008, 0x070000a2,
        0x00307e46, 0x00000009, 0x00000000, 0xffffffff, 0x00000004, 0x00000009, 0x070000a2, 0x00307e46,
        0x0000000a, 0x00000000, 0xffffffff, 0x00000004, 0x0000000a, 0x070000a2, 0x00307e46, 0x0000000b,
        0x00000000, 0xffffffff, 0x00000004, 0x0000000b, 0x070000a2, 0x00307e46, 0x0000000c, 0x00000000,
        0xffffffff, 0x00000004, 0x0000000c, 0x070000a2, 0x00307e46, 0x0000000d, 0x00000000, 0xffffffff,
        0x00000004, 0x0000000d, 0x070000a2, 0x00307e46, 0x0000000e, 0x00000000, 0xffffffff, 0x00000004,
        0x0000000e, 0x070000a2, 0x00307e46, 0x0000000f, 0x00000000, 0xffffffff, 0x00000004, 0x0000000f,
        0x070000a2, 0x00307e46, 0x00000010, 0x00000000, 0xffffffff, 0x00000004, 0x00000010, 0x070000a2,
        0x00307e46, 0x00000011, 0x00000000, 0xffffffff, 0x00000004, 0x00000011, 0x070000a2, 0x00307e46,
        0x00000012, 0x00000000, 0xffffffff, 0x00000004, 0x00000012, 0x070000a2, 0x00307e46, 0x00000013,
        0x00000000, 0xffffffff, 0x00000004, 0x00000013, 0x070000a2, 0x00307e46, 0x00000014, 0x00000000,
        0xffffffff, 0x00000004, 0x00000014, 0x070000a2, 0x00307e46, 0x00000015, 0x00000000, 0xffffffff,
        0x00000004, 0x00000015, 0x070000a2, 0x00307e46, 0x00000016, 0x00000000, 0xffffffff, 0x00000004,
        0x00000016, 0x070000a2, 0x00307e46, 0x00000017, 0x00000000, 0xffffffff, 0x00000004, 0x00000017,
        0x070000a2, 0x00307e46, 0x00000018, 0x00000000, 0xffffffff, 0x00000004, 0x00000018, 0x070000a2,
        0x00307e46, 0x00000019, 0x00000000, 0xffffffff, 0x00000004, 0x00000019, 0x070000a2, 0x00307e46,
        0x0000001a, 0x00000000, 0xffffffff, 0x00000004, 0x0000001a, 0x070000a2, 0x00307e46, 0x0000001b,
        0x00000000, 0xffffffff, 0x00000004, 0x0000001b, 0x070000a2, 0x00307e46, 0x0000001c, 0x00000000,
        0xffffffff, 0x00000004, 0x0000001c, 0x070000a2, 0x00307e46, 0x0000001d, 0x00000000, 0xffffffff,
        0x00000004, 0x0000001d, 0x070000a2, 0x00307e46, 0x0000001e, 0x00000000, 0xffffffff, 0x00000004,
        0x0000001e, 0x070000a2, 0x00307e46, 0x0000001f, 0x00000000, 0xffffffff, 0x00000004, 0x0000001f,
        0x070000a2, 0x00307e46, 0x00000020, 0x00000000, 0xffffffff, 0x00000004, 0x00000020, 0x070000a2,
        0x00307e46, 0x00000021, 0x00000000, 0xffffffff, 0x00000004, 0x00000021, 0x070000a2, 0x00307e46,
        0x00000022, 0x00000000, 0xffffffff, 0x00000004, 0x00000022, 0x070000a2, 0x00307e46, 0x00000023,
        0x00000000, 0xffffffff, 0x00000004, 0x00000023, 0x070000a2, 0x00307e46, 0x00000024, 0x00000000,
        0xffffffff, 0x00000004, 0x00000024, 0x070000a2, 0x00307e46, 0x00000025, 0x00000000, 0xffffffff,
        0x00000004, 0x00000025, 0x070000a2, 0x00307e46, 0x00000026, 0x00000000, 0xffffffff, 0x00000004,
        0x00000026, 0x070000a2, 0x00307e46, 0x00000027, 0x00000000, 0xffffffff, 0x00000004, 0x00000027,
        0x070000a2, 0x00307e46, 0x00000028, 0x00000000, 0xffffffff, 0x00000004, 0x00000028, 0x070000a2,
        0x00307e46, 0x00000029, 0x00000000, 0xffffffff, 0x00000004, 0x00000029, 0x070000a2, 0x00307e46,
        0x0000002a, 0x00000000, 0xffffffff, 0x00000004, 0x0000002a, 0x070000a2, 0x00307e46, 0x0000002b,
        0x00000000, 0xffffffff, 0x00000004, 0x0000002b, 0x070000a2, 0x00307e46, 0x0000002c, 0x00000000,
        0xffffffff, 0x00000004, 0x0000002c, 0x070000a2, 0x00307e46, 0x0000002d, 0x00000000, 0xffffffff,
        0x00000004, 0x0000002d, 0x070000a2, 0x00307e46, 0x0000002e, 0x00000000, 0xffffffff, 0x00000004,
        0x0000002e, 0x070000a2, 0x00307e46, 0x0000002f, 0x00000000, 0xffffffff, 0x00000004, 0x0000002f,
        0x070000a2, 0x00307e46, 0x00000030, 0x00000000, 0xffffffff, 0x00000004, 0x00000030, 0x070000a2,
        0x00307e46, 0x00000031, 0x00000000, 0xffffffff, 0x00000004, 0x00000031, 0x070000a2, 0x00307e46,
        0x00000032, 0x00000000, 0xffffffff, 0x00000004, 0x00000032, 0x070000a2, 0x00307e46, 0x00000033,
        0x00000000, 0xffffffff, 0x00000004, 0x00000033, 0x070000a2, 0x00307e46, 0x00000034, 0x00000000,
        0xffffffff, 0x00000004, 0x00000034, 0x070000a2, 0x00307e46, 0x00000035, 0x00000000, 0xffffffff,
        0x00000004, 0x00000035, 0x070000a2, 0x00307e46, 0x00000036, 0x00000000, 0xffffffff, 0x00000004,
        0x00000036, 0x070000a2, 0x00307e46, 0x00000037, 0x00000000, 0xffffffff, 0x00000004, 0x00000037,
        0x070000a2, 0x00307e46, 0x00000038, 0x00000000, 0xffffffff, 0x00000004, 0x00000038, 0x070000a2,
        0x00307e46, 0x00000039, 0x00000000, 0xffffffff, 0x00000004, 0x00000039, 0x070000a2, 0x00307e46,
        0x0000003a, 0x00000000, 0xffffffff, 0x00000004, 0x0000003a, 0x070000a2, 0x00307e46, 0x0000003b,
        0x00000000, 0xffffffff, 0x00000004, 0x0000003b, 0x070000a2, 0x00307e46, 0x0000003c, 0x00000000,
        0xffffffff, 0x00000004, 0x0000003c, 0x070000a2, 0x00307e46, 0x0000003d, 0x00000000, 0xffffffff,
        0x00000004, 0x0000003d, 0x0600009d, 0x0031ee46, 0x00000000, 0x00000000, 0x00000000, 0x0000003e,
        0x0200005f, 0x00020012, 0x02000068, 0x00000001, 0x0400009b, 0x00000040, 0x00000001, 0x00000001,
        0x04000036, 0x00100012, 0x00000000, 0x0002000a, 0x0c0000a7, 0x00100022, 0x00000000, 0x00004001,
        0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x00000000, 0x0010000a, 0x00000000,
        0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006,
        0x00020001, 0x00000001, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a,
        0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000,
        0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x00000002, 0x0010000a, 0x00000000, 0x0700001e,
        0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042,
        0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x00000003,
        0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a,
        0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000,
        0x84207006, 0x00020001, 0x00000004, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000,
        0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001,
        0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x00000005, 0x0010000a, 0x00000000,
        0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7,
        0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001,
        0x00000006, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000,
        0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001,
        0x00000000, 0x84207006, 0x00020001, 0x00000007, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022,
        0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000,
        0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x00000008, 0x0010000a,
        0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000,
        0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006,
        0x00020001, 0x00000009, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a,
        0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000,
        0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x0000000a, 0x0010000a, 0x00000000, 0x0700001e,
        0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042,
        0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x0000000b,
        0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a,
        0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000,
        0x84207006, 0x00020001, 0x0000000c, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000,
        0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001,
        0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x0000000d, 0x0010000a, 0x00000000,
        0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7,
        0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001,
        0x0000000e, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000,
        0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001,
        0x00000000, 0x84207006, 0x00020001, 0x0000000f, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022,
        0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000,
        0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x00000010, 0x0010000a,
        0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000,
        0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006,
        0x00020001, 0x00000011, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a,
        0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000,
        0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x00000012, 0x0010000a, 0x00000000, 0x0700001e,
        0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042,
        0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x00000013,
        0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a,
        0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000,
        0x84207006, 0x00020001, 0x00000014, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000,
        0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001,
        0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x00000015, 0x0010000a, 0x00000000,
        0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7,
        0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001,
        0x00000016, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000,
        0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001,
        0x00000000, 0x84207006, 0x00020001, 0x00000017, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022,
        0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000,
        0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x00000018, 0x0010000a,
        0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000,
        0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006,
        0x00020001, 0x00000019, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a,
        0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000,
        0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x0000001a, 0x0010000a, 0x00000000, 0x0700001e,
        0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042,
        0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x0000001b,
        0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a,
        0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000,
        0x84207006, 0x00020001, 0x0000001c, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000,
        0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001,
        0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x0000001d, 0x0010000a, 0x00000000,
        0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7,
        0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001,
        0x0000001e, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000,
        0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001,
        0x00000000, 0x84207006, 0x00020001, 0x0000001f, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022,
        0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000,
        0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x00000020, 0x0010000a,
        0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000,
        0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006,
        0x00020001, 0x00000021, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a,
        0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000,
        0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x00000022, 0x0010000a, 0x00000000, 0x0700001e,
        0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042,
        0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x00000023,
        0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a,
        0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000,
        0x84207006, 0x00020001, 0x00000024, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000,
        0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001,
        0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x00000025, 0x0010000a, 0x00000000,
        0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7,
        0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001,
        0x00000026, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000,
        0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001,
        0x00000000, 0x84207006, 0x00020001, 0x00000027, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022,
        0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000,
        0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x00000028, 0x0010000a,
        0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000,
        0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006,
        0x00020001, 0x00000029, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a,
        0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000,
        0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x0000002a, 0x0010000a, 0x00000000, 0x0700001e,
        0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042,
        0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x0000002b,
        0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a,
        0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000,
        0x84207006, 0x00020001, 0x0000002c, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000,
        0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001,
        0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x0000002d, 0x0010000a, 0x00000000,
        0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7,
        0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001,
        0x0000002e, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000,
        0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001,
        0x00000000, 0x84207006, 0x00020001, 0x0000002f, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022,
        0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000,
        0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x00000030, 0x0010000a,
        0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000,
        0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006,
        0x00020001, 0x00000031, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a,
        0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000,
        0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x00000032, 0x0010000a, 0x00000000, 0x0700001e,
        0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042,
        0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x00000033,
        0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a,
        0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000,
        0x84207006, 0x00020001, 0x00000034, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000,
        0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001,
        0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x00000035, 0x0010000a, 0x00000000,
        0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7,
        0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001,
        0x00000036, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000,
        0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001,
        0x00000000, 0x84207006, 0x00020001, 0x00000037, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022,
        0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000,
        0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x00000038, 0x0010000a,
        0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000,
        0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006,
        0x00020001, 0x00000039, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a,
        0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000,
        0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x0000003a, 0x0010000a, 0x00000000, 0x0700001e,
        0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100042,
        0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x0000003b,
        0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000, 0x0010002a, 0x00000000, 0x0010001a,
        0x00000000, 0x0c0000a7, 0x00100042, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000,
        0x84207006, 0x00020001, 0x0000003c, 0x0010000a, 0x00000000, 0x0700001e, 0x00100022, 0x00000000,
        0x0010002a, 0x00000000, 0x0010001a, 0x00000000, 0x0c0000a7, 0x00100012, 0x00000000, 0x00004001,
        0x00000000, 0x00004001, 0x00000000, 0x84207006, 0x00020001, 0x0000003d, 0x0010000a, 0x00000000,
        0x0700001e, 0x00100012, 0x00000000, 0x0010000a, 0x00000000, 0x0010001a, 0x00000000, 0x06000029,
        0x00100022, 0x00000000, 0x0002000a, 0x00004001, 0x00000002, 0x080000a6, 0x0021e012, 0x00000000,
        0x00000000, 0x0010001a, 0x00000000, 0x0010000a, 0x00000000, 0x0100003e,
    };

    if (!init_compute_test_context(&context))
        return;
    command_list = context.list;
    queue = context.queue;

    root_signature_desc.NumParameters = 63;
    root_signature_desc.Flags = 0;
    root_signature_desc.NumStaticSamplers = 0;
    root_signature_desc.pStaticSamplers = NULL;
    root_signature_desc.pParameters = root_parameters;

    for (i = 0; i < 62; i++)
    {
        root_parameters[i].ParameterType = D3D12_ROOT_PARAMETER_TYPE_DESCRIPTOR_TABLE;
        root_parameters[i].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
        root_parameters[i].DescriptorTable.NumDescriptorRanges = 1;
        root_parameters[i].DescriptorTable.pDescriptorRanges = &descriptor_ranges[i];

        descriptor_ranges[i].RegisterSpace = i;
        descriptor_ranges[i].BaseShaderRegister = 0;
        descriptor_ranges[i].OffsetInDescriptorsFromTableStart = 0;
        descriptor_ranges[i].NumDescriptors = 64 * 1024;
        descriptor_ranges[i].RangeType = D3D12_DESCRIPTOR_RANGE_TYPE_SRV;
    }

    root_parameters[62].ParameterType = D3D12_ROOT_PARAMETER_TYPE_UAV;
    root_parameters[62].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[62].Descriptor.RegisterSpace = 62;
    root_parameters[62].Descriptor.ShaderRegister = 0;

    hr = create_root_signature(context.device, &root_signature_desc, &context.root_signature);
    ok(SUCCEEDED(hr), "Failed to create root signature, hr %#x.\n", hr);

    for (i = 0; i < 1024; i++)
    {
        const UINT buffer_data[] = { i, i, i, i };
        input_buffers[i] = create_default_buffer(context.device, sizeof(buffer_data), D3D12_RESOURCE_FLAG_NONE, D3D12_RESOURCE_STATE_COPY_DEST);
        upload_buffer_data(input_buffers[i], 0, sizeof(buffer_data), buffer_data, queue, command_list);
        reset_command_list(command_list, context.allocator);
        transition_resource_state(command_list, input_buffers[i], D3D12_RESOURCE_STATE_COPY_DEST, D3D12_RESOURCE_STATE_NON_PIXEL_SHADER_RESOURCE);
    }
    output_buffer = create_default_buffer(context.device, 4 * 64, D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);

    context.pipeline_state = create_compute_pipeline_state(context.device,
        context.root_signature,
        shader_bytecode(cs_code, sizeof(cs_code)));

    heap = create_gpu_descriptor_heap(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV, 128 * 1024);
    cpu_handle = ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(heap);
    gpu_handle = ID3D12DescriptorHeap_GetGPUDescriptorHandleForHeapStart(heap);
    descriptor_size = ID3D12Device_GetDescriptorHandleIncrementSize(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV);
    for (i = 0; i < 1024; i++)
    {
        D3D12_CPU_DESCRIPTOR_HANDLE h = cpu_handle;
        D3D12_SHADER_RESOURCE_VIEW_DESC view;

        h.ptr += i * descriptor_size;

        view.Format = DXGI_FORMAT_UNKNOWN;
        view.ViewDimension = D3D12_SRV_DIMENSION_BUFFER;
        view.Shader4ComponentMapping = D3D12_DEFAULT_SHADER_4_COMPONENT_MAPPING;
        view.Buffer.FirstElement = 0;
        view.Buffer.NumElements = 4;
        view.Buffer.StructureByteStride = 4;
        view.Buffer.Flags = 0;
        ID3D12Device_CreateShaderResourceView(context.device, input_buffers[i], &view, h);
    }

    ID3D12GraphicsCommandList_SetComputeRootSignature(command_list, context.root_signature);
    ID3D12GraphicsCommandList_SetPipelineState(command_list, context.pipeline_state);
    ID3D12GraphicsCommandList_SetDescriptorHeaps(command_list, 1, &heap);

    for (i = 0; i < 62; i++)
    {
        D3D12_GPU_DESCRIPTOR_HANDLE gpu = gpu_handle;
        gpu.ptr += i * descriptor_size;
        ID3D12GraphicsCommandList_SetComputeRootDescriptorTable(command_list, i, gpu);
    }
    ID3D12GraphicsCommandList_SetComputeRootUnorderedAccessView(command_list, 62, ID3D12Resource_GetGPUVirtualAddress(output_buffer));
    ID3D12GraphicsCommandList_Dispatch(command_list, 1, 1, 1);

    transition_resource_state(command_list, output_buffer, D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);
    get_buffer_readback_with_command_list(output_buffer, DXGI_FORMAT_UNKNOWN, &rb, queue, command_list);

    for (i = 0; i < 64; i++)
    {
        UINT value = get_readback_uint(&rb, i, 0, 0);
        UINT reference = 62 * (i + (i + 61)) / 2;
        ok(value == reference, "Readback value is: %u\n", value);
    }

    release_resource_readback(&rb);
    reset_command_list(command_list, context.allocator);

    for (i = 0; i < 1024; i++)
        ID3D12Resource_Release(input_buffers[i]);
    ID3D12Resource_Release(output_buffer);
    ID3D12DescriptorHeap_Release(heap);
    destroy_test_context(&context);
}

static void test_bindless_cbv(bool use_dxil)
{
    D3D12_ROOT_SIGNATURE_DESC root_signature_desc;
    D3D12_ROOT_PARAMETER root_parameters[2];
    D3D12_DESCRIPTOR_RANGE descriptor_ranges;
    ID3D12DescriptorHeap* heap;

    ID3D12Resource* input_buffers[512];
    ID3D12Resource* output_buffer;
    struct resource_readback rb;

    ID3D12GraphicsCommandList* command_list;
    D3D12_CPU_DESCRIPTOR_HANDLE cpu_handle;
    D3D12_GPU_DESCRIPTOR_HANDLE gpu_handle;
    unsigned int i, descriptor_size;
    struct test_context context;
    ID3D12CommandQueue* queue;
    HRESULT hr;

#if 0
    struct Foo
    {
        uint value;
    };
    ConstantBuffer<Foo> CBVs[] : register(b2, space1);

    RWByteAddressBuffer RWBuf : register(u0);

    [numthreads(64, 1, 1)]
    void main(uint index : SV_DispatchThreadID)
    {
        uint value = CBVs[NonUniformResourceIndex(index)].value;
        RWBuf.Store(4 * index, value);
    }
#endif
    static const DWORD cs_code_dxbc[] =
    {
        0x43425844, 0x15b62447, 0xbdab3867, 0x4685eb54, 0xf1bf1b23, 0x00000001, 0x00000114, 0x00000003,
        0x0000002c, 0x0000003c, 0x0000004c, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x00000008, 0x00000000, 0x00000008, 0x58454853, 0x000000c0, 0x00050051, 0x00000030, 0x0100086a,
        0x07000859, 0x00308e46, 0x00000000, 0x00000002, 0xffffffff, 0x00000001, 0x00000001, 0x0600009d,
        0x0031ee46, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x0200005f, 0x00020012, 0x02000068,
        0x00000001, 0x0400009b, 0x00000040, 0x00000001, 0x00000001, 0x06000029, 0x00100012, 0x00000000,
        0x0002000a, 0x00004001, 0x00000002, 0x04000036, 0x00100022, 0x00000000, 0x0002000a, 0x0d0000a6,
        0x0021e012, 0x00000000, 0x00000000, 0x0010000a, 0x00000000, 0x8630800a, 0x00020001, 0x00000000,
        0x00000002, 0x0010001a, 0x00000000, 0x00000000, 0x0100003e,
    };
    static const BYTE cs_code_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0x0c, 0x9e, 0xac, 0x4e, 0xa3, 0x30, 0xba, 0xe4, 0x9e, 0x8a, 0x8e, 0x94, 0x0f, 0xcd, 0x88, 0x4b, 0x01, 0x00, 0x00, 0x00, 0x3c, 0x06, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x5c, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x6c, 0x05, 0x00, 0x00, 0x60, 0x00, 0x05, 0x00, 0x5b, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00,
        0x10, 0x00, 0x00, 0x00, 0x54, 0x05, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x52, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00,
        0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42,
        0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50,
        0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d,
        0x86, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42,
        0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3,
        0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x64, 0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0x63, 0x0c, 0x22, 0x47, 0x0d, 0x97, 0x3f, 0x61,
        0x0f, 0x21, 0xf9, 0xdc, 0x46, 0x15, 0x2b, 0x31, 0xf9, 0xc8, 0x6d, 0x23, 0x62, 0x8c, 0x31, 0xe6, 0x08, 0x10, 0x3a, 0xf7, 0x0c, 0x97, 0x3f, 0x61, 0x0f, 0x21, 0xf9, 0x21, 0xd0, 0x0c, 0x0b, 0x81,
        0x02, 0x54, 0x08, 0x33, 0xd2, 0x20, 0x35, 0x47, 0x10, 0x14, 0x23, 0x8d, 0x33, 0x06, 0xa3, 0x56, 0x14, 0x30, 0xd2, 0x18, 0x63, 0x8c, 0x71, 0xe8, 0x0d, 0x04, 0x9c, 0x26, 0x4d, 0x11, 0x25, 0x4c,
        0xfe, 0x0a, 0x6f, 0xd8, 0x44, 0x68, 0xc3, 0x10, 0x11, 0x92, 0xb4, 0x51, 0x45, 0x41, 0x44, 0x28, 0x18, 0x24, 0x4f, 0x91, 0xa6, 0x88, 0x12, 0x26, 0xff, 0x73, 0x1c, 0x14, 0x8c, 0x23, 0xb8, 0xed,
        0x95, 0x50, 0x40, 0xf3, 0xff, 0xff, 0xff, 0xff, 0xa1, 0x92, 0x9d, 0x23, 0x00, 0x05, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0,
        0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07,
        0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90,
        0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6,
        0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x04, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x14, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x18, 0xf2, 0x34, 0x40, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x79, 0x80, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x0b, 0x04, 0x00, 0x00,
        0x0b, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x25, 0x30, 0x02, 0x50, 0x08, 0xc5, 0x50, 0x16, 0x45, 0x50, 0x03, 0x05,
        0x18, 0x40, 0x6c, 0x04, 0x80, 0x70, 0x81, 0xd0, 0x9d, 0x01, 0xa0, 0x39, 0x03, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4,
        0x88, 0x0c, 0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26, 0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05, 0x06, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x86, 0x62, 0x82, 0x30, 0x18, 0x1b, 0x84, 0x81, 0x98, 0x20,
        0x0c, 0xc7, 0x06, 0x61, 0x30, 0x28, 0x8c, 0xcd, 0x4d, 0x10, 0x06, 0x64, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0x61, 0x82, 0x08, 0x4c, 0x10, 0x86, 0x64, 0x82, 0xc0, 0x34, 0x1b, 0x16, 0x62, 0x61, 0x08,
        0x62, 0x68, 0x1c, 0xc7, 0x01, 0x36, 0x04, 0xcf, 0x04, 0xe1, 0x7a, 0x26, 0x08, 0x83, 0x32, 0x41, 0x18, 0x96, 0x0d, 0x08, 0x11, 0x31, 0x83, 0x34, 0x19, 0xc0, 0x86, 0x80, 0xda, 0x40, 0x00, 0x50,
        0x05, 0x4c, 0x10, 0x04, 0x60, 0x03, 0xb0, 0x61, 0x20, 0x30, 0x6c, 0x43, 0x90, 0x6d, 0x18, 0x86, 0x4b, 0x23, 0xd1, 0x16, 0x96, 0xe6, 0x36, 0x41, 0xc0, 0x9c, 0x09, 0xc2, 0xc0, 0x6c, 0x18, 0xbc,
        0x61, 0xd8, 0x40, 0x10, 0x9d, 0xf1, 0x6d, 0x28, 0x2e, 0x0e, 0xb0, 0xc0, 0xa0, 0x0a, 0x1b, 0x9b, 0x5d, 0x9b, 0x4b, 0x1a, 0x59, 0x99, 0x1b, 0xdd, 0x94, 0x20, 0xa8, 0x42, 0x86, 0xe7, 0x62, 0x57,
        0x26, 0x37, 0x97, 0xf6, 0xe6, 0x36, 0x25, 0x20, 0x9a, 0x90, 0xe1, 0xb9, 0xd8, 0x85, 0xb1, 0xd9, 0x95, 0xc9, 0x4d, 0x09, 0x8c, 0x3a, 0x64, 0x78, 0x2e, 0x73, 0x68, 0x61, 0x64, 0x65, 0x72, 0x4d,
        0x6f, 0x64, 0x65, 0x6c, 0x53, 0x82, 0xa4, 0x0c, 0x19, 0x9e, 0x8b, 0x5c, 0xd9, 0xdc, 0x5b, 0x9d, 0xdc, 0x58, 0xd9, 0xdc, 0x94, 0xa0, 0xaa, 0x44, 0x86, 0xe7, 0x42, 0x97, 0x07, 0x57, 0x16, 0xe4,
        0xe6, 0xf6, 0x46, 0x17, 0x46, 0x97, 0xf6, 0xe6, 0x36, 0x37, 0x25, 0xd0, 0xea, 0x90, 0xe1, 0xb9, 0x94, 0xb9, 0xd1, 0xc9, 0xe5, 0x41, 0xbd, 0xa5, 0xb9, 0xd1, 0xcd, 0x4d, 0x09, 0xc0, 0x00, 0x00,
        0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73,
        0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b,
        0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20,
        0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61,
        0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87,
        0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98,
        0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61,
        0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b,
        0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xcc, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x74, 0x60, 0x07, 0x37, 0x90, 0x87, 0x72, 0x98, 0x87, 0x77, 0xa8,
        0x07, 0x79, 0x18, 0x87, 0x72, 0x70, 0x83, 0x70, 0xa0, 0x07, 0x7a, 0x90, 0x87, 0x74, 0x10, 0x87, 0x7a, 0xa0, 0x87, 0x72, 0x00, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
        0x46, 0x50, 0x0d, 0x97, 0xef, 0x3c, 0x7e, 0x40, 0x15, 0x05, 0x11, 0xb1, 0x93, 0x13, 0x11, 0x3e, 0x72, 0xdb, 0x26, 0xb0, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10, 0x50, 0x45, 0x41, 0x44, 0xa5, 0x03,
        0x0c, 0x25, 0x61, 0x00, 0x02, 0xe6, 0x23, 0xb7, 0x6d, 0x03, 0xd2, 0x70, 0xf9, 0xce, 0xe3, 0x0b, 0x11, 0x01, 0x4c, 0x44, 0x08, 0x34, 0xc3, 0x42, 0x58, 0x80, 0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0xd3,
        0x11, 0x11, 0xc0, 0x20, 0x0e, 0x3e, 0x72, 0xdb, 0x06, 0x40, 0x30, 0x00, 0xd2, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00,
        0x04, 0x00, 0x00, 0x00, 0x34, 0x66, 0x00, 0x4a, 0xae, 0x74, 0x03, 0xca, 0xae, 0x14, 0x03, 0xe8, 0x94, 0x40, 0x11, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xd0, 0x54, 0x86, 0x20, 0x49, 0xcb,
        0x88, 0x81, 0x01, 0x80, 0x20, 0x18, 0x10, 0x98, 0x31, 0x55, 0x00, 0xc1, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x34, 0x58, 0x42, 0x54, 0x81, 0x33, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0x8a, 0x86,
        0x04, 0xd6, 0x68, 0x42, 0x00, 0x54, 0x41, 0xdd, 0x88, 0x41, 0x03, 0x80, 0x20, 0x18, 0x3c, 0x9b, 0x72, 0x04, 0x8e, 0xe0, 0x38, 0x4e, 0x82, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    if (!init_compute_test_context(&context))
        return;

    if (use_dxil && !context_supports_dxil(&context))
    {
        destroy_test_context(&context);
        return;
    }

    command_list = context.list;
    queue = context.queue;

    root_signature_desc.NumParameters = 2;
    root_signature_desc.Flags = 0;
    root_signature_desc.NumStaticSamplers = 0;
    root_signature_desc.pStaticSamplers = NULL;
    root_signature_desc.pParameters = root_parameters;

    root_parameters[0].ParameterType = D3D12_ROOT_PARAMETER_TYPE_DESCRIPTOR_TABLE;
    root_parameters[0].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[0].DescriptorTable.NumDescriptorRanges = 1;
    root_parameters[0].DescriptorTable.pDescriptorRanges = &descriptor_ranges;

    root_parameters[1].ParameterType = D3D12_ROOT_PARAMETER_TYPE_UAV;
    root_parameters[1].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[1].Descriptor.RegisterSpace = 0;
    root_parameters[1].Descriptor.ShaderRegister = 0;

    descriptor_ranges.RegisterSpace = 1;
    descriptor_ranges.BaseShaderRegister = 2;
    descriptor_ranges.OffsetInDescriptorsFromTableStart = 1;
    descriptor_ranges.NumDescriptors = UINT_MAX;
    descriptor_ranges.RangeType = D3D12_DESCRIPTOR_RANGE_TYPE_CBV;

    hr = create_root_signature(context.device, &root_signature_desc, &context.root_signature);
    ok(SUCCEEDED(hr), "Failed to create root signature, hr %#x.\n", hr);

    for (i = 0; i < 512; i++)
    {
        const UINT buffer_data[] = { i, i, i, i };
        input_buffers[i] = create_default_buffer(context.device, D3D12_CONSTANT_BUFFER_DATA_PLACEMENT_ALIGNMENT, D3D12_RESOURCE_FLAG_NONE, D3D12_RESOURCE_STATE_COPY_DEST);
        upload_buffer_data(input_buffers[i], 0, sizeof(buffer_data), buffer_data, queue, command_list);
        reset_command_list(command_list, context.allocator);
        transition_resource_state(command_list, input_buffers[i], D3D12_RESOURCE_STATE_COPY_DEST, D3D12_RESOURCE_STATE_NON_PIXEL_SHADER_RESOURCE);
    }

    output_buffer = create_default_buffer(context.device, 4 * 256, D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);

    context.pipeline_state = create_compute_pipeline_state(context.device,
        context.root_signature,
        shader_bytecode(use_dxil ? (const void*)cs_code_dxil : (const void*)cs_code_dxbc, use_dxil ? sizeof(cs_code_dxil) : sizeof(cs_code_dxbc)));

    heap = create_gpu_descriptor_heap(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV, 800000);
    cpu_handle = ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(heap);
    gpu_handle = ID3D12DescriptorHeap_GetGPUDescriptorHandleForHeapStart(heap);
    descriptor_size = ID3D12Device_GetDescriptorHandleIncrementSize(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV);
    for (i = 0; i < 512; i++)
    {
        D3D12_CONSTANT_BUFFER_VIEW_DESC view;
        D3D12_CPU_DESCRIPTOR_HANDLE h = cpu_handle;
        view.BufferLocation = ID3D12Resource_GetGPUVirtualAddress(input_buffers[i]);
        view.SizeInBytes = D3D12_CONSTANT_BUFFER_DATA_PLACEMENT_ALIGNMENT;
        h.ptr += i * descriptor_size;
        ID3D12Device_CreateConstantBufferView(context.device, &view, h);
    }

    ID3D12GraphicsCommandList_SetComputeRootSignature(command_list, context.root_signature);
    ID3D12GraphicsCommandList_SetPipelineState(command_list, context.pipeline_state);
    ID3D12GraphicsCommandList_SetDescriptorHeaps(command_list, 1, &heap);
    ID3D12GraphicsCommandList_SetComputeRootDescriptorTable(command_list, 0, gpu_handle);
    ID3D12GraphicsCommandList_SetComputeRootUnorderedAccessView(command_list, 1, ID3D12Resource_GetGPUVirtualAddress(output_buffer));
    ID3D12GraphicsCommandList_Dispatch(command_list, 4, 1, 1);

    transition_resource_state(command_list, output_buffer, D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);
    get_buffer_readback_with_command_list(output_buffer, DXGI_FORMAT_UNKNOWN, &rb, queue, command_list);

    for (i = 0; i < 256; i++)
    {
        UINT value = get_readback_uint(&rb, i, 0, 0);
        UINT reference = i + 1;

        if (use_dxil && (i & 63) != 0)
        {
            /* DXC is bugged and does not emit NonUniformResourceIndex correctly for CBVs,
               so only check the first lane for correctness. */
            continue;
        }

        ok(value == reference, "Readback value for iteration %u is: %u\n", i, value);
    }

    release_resource_readback(&rb);
    reset_command_list(command_list, context.allocator);

    for (i = 0; i < 512; i++)
        ID3D12Resource_Release(input_buffers[i]);
    ID3D12Resource_Release(output_buffer);
    ID3D12DescriptorHeap_Release(heap);
    destroy_test_context(&context);
}

static void test_bindless_uav(bool use_dxil)
{
    D3D12_ROOT_SIGNATURE_DESC root_signature_desc;
    D3D12_ROOT_PARAMETER root_parameters[1];
    D3D12_DESCRIPTOR_RANGE descriptor_ranges[2];
    ID3D12DescriptorHeap *heap;

    ID3D12Resource *output_buffers[256];
    ID3D12Resource *output_textures[256];
    struct resource_readback rb;

    ID3D12GraphicsCommandList* command_list;
    D3D12_CPU_DESCRIPTOR_HANDLE cpu_handle;
    D3D12_GPU_DESCRIPTOR_HANDLE gpu_handle;
    unsigned int i, descriptor_size;
    struct test_context context;
    ID3D12CommandQueue* queue;
    HRESULT hr;

#if 0
    RWStructuredBuffer<uint> Buffers[] : register(u2, space1);
    RWTexture2D<uint> Textures[] : register(u2, space2);

    [numthreads(64, 1, 1)]
    void main(uint global_index : SV_DispatchThreadID, uint index : SV_GroupID)
    {
        // Need this branch or FXC refuses to compile. It doesn't understand we're writing to different resources.
        if (global_index < 512)
        {
            Buffers[NonUniformResourceIndex(global_index)][0] = global_index + 1;
            Textures[NonUniformResourceIndex(global_index)][int2(0, 0)] = 256 + global_index + 1;
        }
    }
#endif
    static const DWORD cs_code_dxbc[] =
    {
        0x43425844, 0x7f1b7f4d, 0xd41e1bd6, 0x1fdc0577, 0xc59de64e, 0x00000001, 0x00000184, 0x00000003,
        0x0000002c, 0x0000003c, 0x0000004c, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x00000008, 0x00000000, 0x00000008, 0x58454853, 0x00000130, 0x00050051, 0x0000004c, 0x0100086a,
        0x0700009e, 0x0031ee46, 0x00000000, 0x00000002, 0xffffffff, 0x00000004, 0x00000001, 0x0700189c,
        0x0031ee46, 0x00000001, 0x00000002, 0xffffffff, 0x00004444, 0x00000002, 0x0200005f, 0x00020012,
        0x02000068, 0x00000001, 0x0400009b, 0x00000040, 0x00000001, 0x00000001, 0x0600004f, 0x00100012,
        0x00000000, 0x0002000a, 0x00004001, 0x00000200, 0x0304001f, 0x0010000a, 0x00000000, 0x0900001e,
        0x00100032, 0x00000000, 0x00020006, 0x00004002, 0x00000001, 0x00000101, 0x00000000, 0x00000000,
        0x04000036, 0x00100042, 0x00000000, 0x0002000a, 0x0d0000a8, 0x8621e012, 0x00020001, 0x00000000,
        0x00000002, 0x0010002a, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x0010000a,
        0x00000000, 0x0e0000a4, 0x8621e0f2, 0x00020001, 0x00000001, 0x00000002, 0x0010002a, 0x00000000,
        0x00004002, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00100556, 0x00000000, 0x01000015,
        0x0100003e,
    };
    static const BYTE cs_code_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0xf6, 0x2f, 0x3f, 0xa0, 0x92, 0xa9, 0x88, 0x5d, 0x94, 0x68, 0xbb, 0xd7, 0x1a, 0x3f, 0x91, 0x74, 0x01, 0x00, 0x00, 0x00, 0xac, 0x06, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x5c, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x06, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0xdc, 0x05, 0x00, 0x00, 0x60, 0x00, 0x05, 0x00, 0x77, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00,
        0x10, 0x00, 0x00, 0x00, 0xc4, 0x05, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x6e, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00,
        0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42,
        0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50,
        0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d,
        0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42,
        0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x36, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3,
        0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x64, 0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0x63, 0x0c, 0x22, 0x73, 0x04, 0x08, 0x99, 0x7b,
        0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x10, 0x68, 0x86, 0x85, 0x40, 0xc1, 0x29, 0x0b, 0x18, 0x68, 0x8c, 0x31, 0xc6, 0x18, 0x86, 0xd2, 0x1c, 0x41, 0x50, 0x0c, 0x34, 0xcc, 0x18, 0x8b, 0x58,
        0x51, 0xc0, 0x40, 0x63, 0x8c, 0x31, 0x86, 0x21, 0x37, 0x10, 0x30, 0x93, 0x19, 0x8c, 0x03, 0x3b, 0x84, 0xc3, 0x3c, 0xcc, 0x83, 0x1b, 0xc8, 0xc2, 0x2d, 0xcc, 0x02, 0x3d, 0xc8, 0x43, 0x3d, 0x8c,
        0x03, 0x3d, 0xd4, 0x83, 0x3c, 0x94, 0x03, 0x39, 0x88, 0x42, 0x3d, 0x98, 0x83, 0x39, 0x94, 0x83, 0x3c, 0xf0, 0x41, 0x3d, 0xb8, 0xc3, 0x3c, 0xa4, 0xc3, 0x39, 0xb8, 0x43, 0x39, 0x90, 0x03, 0x18,
        0xa4, 0x83, 0x3b, 0xd0, 0x83, 0x1f, 0xa0, 0x60, 0x24, 0x40, 0x24, 0x39, 0xd3, 0x37, 0x0e, 0xec, 0x10, 0x0e, 0xf3, 0x30, 0x0f, 0x6e, 0x20, 0x0b, 0xb7, 0x40, 0x0b, 0xe5, 0x80, 0x0f, 0xf4, 0x50,
        0x0f, 0xf2, 0x50, 0x0e, 0x72, 0x40, 0x0a, 0x7c, 0x50, 0x0f, 0xee, 0x30, 0x0f, 0xe9, 0x70, 0x0e, 0xee, 0x50, 0x0e, 0xe4, 0x00, 0x06, 0xe9, 0xe0, 0x0e, 0xf4, 0xe0, 0x07, 0x28, 0x18, 0x09, 0x40,
        0xa9, 0xce, 0x11, 0x80, 0xc2, 0x14, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e,
        0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20,
        0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74,
        0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07,
        0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86,
        0x3c, 0x04, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x12, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x30, 0x40, 0x00, 0x0c, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x71, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x0b, 0x04, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14,
        0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x25, 0x30, 0x02, 0x50, 0x08, 0xc5, 0x50, 0x04, 0x35, 0x50, 0x18, 0xa5, 0x50, 0x80, 0x01, 0xb4, 0x46, 0x00, 0xe8, 0x16,
        0x08, 0xd9, 0x19, 0x00, 0x9a, 0x33, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x88, 0x0c, 0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44,
        0x06, 0x26, 0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05, 0x06, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x86, 0x62, 0x82, 0x30, 0x18, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x0c, 0xc7, 0x06, 0x61, 0x30, 0x28, 0x8c, 0xcd,
        0x4d, 0x10, 0x06, 0x64, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0x61, 0x8a, 0x08, 0x4c, 0x10, 0x86, 0x64, 0x82, 0x30, 0x28, 0x13, 0x84, 0x61, 0x99, 0x20, 0x2c, 0xce, 0x86, 0x85, 0x58, 0x98, 0xa1, 0x71,
        0x1e, 0x08, 0x82, 0x8e, 0x09, 0x82, 0x05, 0x4d, 0x10, 0x06, 0x66, 0x83, 0x40, 0x4c, 0x1b, 0x96, 0x41, 0x62, 0x9a, 0xc6, 0x69, 0x20, 0x08, 0xa2, 0x36, 0x08, 0x51, 0xb5, 0x81, 0x00, 0x2c, 0x00,
        0x98, 0x20, 0x08, 0xc0, 0x06, 0x60, 0xc3, 0x40, 0x64, 0xd9, 0x86, 0x40, 0xdb, 0x30, 0x0c, 0xd8, 0x46, 0xa2, 0x2d, 0x2c, 0xcd, 0x6d, 0x82, 0x70, 0x3d, 0x13, 0x84, 0xa1, 0xd9, 0x30, 0x7c, 0xc3,
        0xb0, 0x81, 0x20, 0x3c, 0x03, 0x0c, 0x36, 0x14, 0x58, 0x07, 0x5c, 0x61, 0x50, 0x85, 0x8d, 0xcd, 0xae, 0xcd, 0x25, 0x8d, 0xac, 0xcc, 0x8d, 0x6e, 0x4a, 0x10, 0x54, 0x21, 0xc3, 0x73, 0xb1, 0x2b,
        0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x9b, 0x12, 0x10, 0x4d, 0xc8, 0xf0, 0x5c, 0xec, 0xc2, 0xd8, 0xec, 0xca, 0xe4, 0xa6, 0x04, 0x46, 0x1d, 0x32, 0x3c, 0x97, 0x39, 0xb4, 0x30, 0xb2, 0x32, 0xb9, 0xa6,
        0x37, 0xb2, 0x32, 0xb6, 0x29, 0x41, 0x52, 0x86, 0x0c, 0xcf, 0x45, 0xae, 0x6c, 0xee, 0xad, 0x4e, 0x6e, 0xac, 0x6c, 0x6e, 0x4a, 0x60, 0x55, 0x22, 0xc3, 0x73, 0xa1, 0xcb, 0x83, 0x2b, 0x0b, 0x72,
        0x73, 0x7b, 0xa3, 0x0b, 0xa3, 0x4b, 0x7b, 0x73, 0x9b, 0x9b, 0x12, 0x6c, 0x75, 0xc8, 0xf0, 0x5c, 0xca, 0xdc, 0xe8, 0xe4, 0xf2, 0xa0, 0xde, 0xd2, 0xdc, 0xe8, 0xe6, 0xa6, 0x04, 0x61, 0x00, 0x00,
        0x79, 0x18, 0x00, 0x00, 0x51, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73,
        0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b,
        0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20,
        0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61,
        0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87,
        0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98,
        0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61,
        0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b,
        0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xcc, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x74, 0x60, 0x07, 0x37, 0x90, 0x87, 0x72, 0x98, 0x87, 0x77, 0xa8,
        0x07, 0x79, 0x18, 0x87, 0x72, 0x70, 0x83, 0x70, 0xa0, 0x07, 0x7a, 0x90, 0x87, 0x74, 0x10, 0x87, 0x7a, 0xa0, 0x87, 0x72, 0x98, 0xe1, 0x3c, 0xe4, 0x80, 0x0f, 0x6e, 0xe0, 0x0e, 0xef, 0xe0, 0x0e,
        0xf5, 0xe0, 0x0e, 0xe9, 0x60, 0x0e, 0xef, 0x20, 0x0f, 0xed, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x46, 0x50, 0x0d, 0x97, 0xef, 0x3c, 0x7e, 0x40, 0x15, 0x05, 0x11, 0xb1,
        0x93, 0x13, 0x11, 0x3e, 0x72, 0xdb, 0x36, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44, 0x84, 0x40, 0x33, 0x2c, 0x84, 0x09, 0x58, 0xc3, 0xe5, 0x3b, 0x8f, 0x3f, 0x11, 0xd7, 0x44,
        0x45, 0x04, 0x3b, 0x39, 0x11, 0xe1, 0x23, 0xb7, 0x6d, 0x01, 0xd2, 0x70, 0xf9, 0xce, 0xe3, 0x4f, 0x47, 0x44, 0x00, 0x83, 0x38, 0xf8, 0xc8, 0x6d, 0x1b, 0x00, 0xc1, 0x00, 0x48, 0x03, 0x00, 0x00,
        0x61, 0x20, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x13, 0x04, 0x43, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x34, 0x66, 0x00, 0x4a, 0xae, 0x74, 0x03, 0x0a, 0x10, 0xa1, 0x14, 0x03,
        0x4a, 0x90, 0xa0, 0x0c, 0x03, 0xc8, 0x94, 0x40, 0x79, 0xd0, 0xaa, 0x01, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0x40, 0x6c, 0x88, 0x35, 0xdc, 0x10, 0x20, 0x64, 0x30, 0xcb,
        0x10, 0x08, 0x41, 0x09, 0x19, 0xd4, 0x60, 0xc1, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x30, 0x5d, 0x73, 0x68, 0x41, 0x31, 0x62, 0xd0, 0x00, 0x20, 0x08, 0x06, 0x4e, 0xb7, 0x04, 0xdb, 0x36, 0x3c,
        0xcf, 0x83, 0x54, 0xa1, 0x40, 0x19, 0x19, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0x03, 0x06, 0x90, 0xe2, 0x05, 0xc8, 0x88, 0x81, 0x03, 0x80, 0x20, 0x18, 0x24, 0x62, 0xc0, 0x04, 0x9e, 0x27,
        0x0d, 0xc3, 0x30, 0x28, 0xb3, 0x04, 0x02, 0x02, 0x00, 0x00, 0x00, 0x00,
    };

    if (!init_compute_test_context(&context))
        return;

    if (use_dxil && !context_supports_dxil(&context))
    {
        destroy_test_context(&context);
        return;
    }

    command_list = context.list;
    queue = context.queue;

    root_signature_desc.NumParameters = 1;
    root_signature_desc.Flags = 0;
    root_signature_desc.NumStaticSamplers = 0;
    root_signature_desc.pStaticSamplers = NULL;
    root_signature_desc.pParameters = root_parameters;

    root_parameters[0].ParameterType = D3D12_ROOT_PARAMETER_TYPE_DESCRIPTOR_TABLE;
    root_parameters[0].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[0].DescriptorTable.NumDescriptorRanges = 2;
    root_parameters[0].DescriptorTable.pDescriptorRanges = descriptor_ranges;

    descriptor_ranges[0].RegisterSpace = 1;
    descriptor_ranges[0].BaseShaderRegister = 2;
    descriptor_ranges[0].OffsetInDescriptorsFromTableStart = 1;
    descriptor_ranges[0].NumDescriptors = UINT_MAX;
    descriptor_ranges[0].RangeType = D3D12_DESCRIPTOR_RANGE_TYPE_UAV;

    descriptor_ranges[1].RegisterSpace = 2;
    descriptor_ranges[1].BaseShaderRegister = 2;
    descriptor_ranges[1].OffsetInDescriptorsFromTableStart = 256 + 1;
    descriptor_ranges[1].NumDescriptors = UINT_MAX;
    descriptor_ranges[1].RangeType = D3D12_DESCRIPTOR_RANGE_TYPE_UAV;

    hr = create_root_signature(context.device, &root_signature_desc, &context.root_signature);
    ok(SUCCEEDED(hr), "Failed to create root signature, hr %#x.\n", hr);

    for (i = 0; i < 256; i++)
        output_buffers[i] = create_default_buffer(context.device, 256, D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);
    for (i = 0; i < 256; i++)
        output_textures[i] = create_default_texture2d(context.device, 1, 1, 1, 1, DXGI_FORMAT_R32_UINT, D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);

    context.pipeline_state = create_compute_pipeline_state(context.device,
        context.root_signature,
        shader_bytecode(use_dxil ? (const void*)cs_code_dxil : (const void*)cs_code_dxbc, use_dxil ? sizeof(cs_code_dxil) : sizeof(cs_code_dxbc)));

    heap = create_gpu_descriptor_heap(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV, 800000);
    cpu_handle = ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(heap);
    gpu_handle = ID3D12DescriptorHeap_GetGPUDescriptorHandleForHeapStart(heap);
    descriptor_size = ID3D12Device_GetDescriptorHandleIncrementSize(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV);

    for (i = 0; i < 256; i++)
    {
        D3D12_UNORDERED_ACCESS_VIEW_DESC view;
        D3D12_CPU_DESCRIPTOR_HANDLE h = cpu_handle;
        view.Format = DXGI_FORMAT_UNKNOWN;
        view.ViewDimension = D3D12_UAV_DIMENSION_BUFFER;
        view.Buffer.FirstElement = 0;
        view.Buffer.NumElements = 64;
        view.Buffer.StructureByteStride = 4;
        view.Buffer.CounterOffsetInBytes = 0;
        view.Buffer.Flags = 0;
        h.ptr += (i + 1) * descriptor_size;
        ID3D12Device_CreateUnorderedAccessView(context.device, output_buffers[i], NULL, &view, h);
    }

    for (i = 0; i < 256; i++)
    {
        D3D12_UNORDERED_ACCESS_VIEW_DESC view;
        D3D12_CPU_DESCRIPTOR_HANDLE h = cpu_handle;
        view.Format = DXGI_FORMAT_R32_UINT;
        view.ViewDimension = D3D12_UAV_DIMENSION_TEXTURE2D;
        view.Texture2D.MipSlice = 0;
        view.Texture2D.PlaneSlice = 0;
        h.ptr += (256 + i + 1) * descriptor_size;
        ID3D12Device_CreateUnorderedAccessView(context.device, output_textures[i], NULL, &view, h);
    }

    ID3D12GraphicsCommandList_SetComputeRootSignature(command_list, context.root_signature);
    ID3D12GraphicsCommandList_SetPipelineState(command_list, context.pipeline_state);
    ID3D12GraphicsCommandList_SetDescriptorHeaps(command_list, 1, &heap);
    ID3D12GraphicsCommandList_SetComputeRootDescriptorTable(command_list, 0, gpu_handle);
    ID3D12GraphicsCommandList_Dispatch(command_list, 4, 1, 1);

    for (i = 0; i < 256; i++)
        transition_resource_state(command_list, output_buffers[i], D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);
    for (i = 0; i < 256; i++)
        transition_resource_state(command_list, output_textures[i], D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);

    for (i = 0; i < 256; i++)
    {
        UINT value;
        UINT reference = i + 1;

        get_buffer_readback_with_command_list(output_buffers[i], DXGI_FORMAT_UNKNOWN, &rb, queue, command_list);
        value = get_readback_uint(&rb, 0, 0, 0);

        ok(value == reference, "Readback value for buffer iteration %u is: %u\n", i, value);

        release_resource_readback(&rb);
        reset_command_list(command_list, context.allocator);
    }

    for (i = 0; i < 256; i++)
    {
        UINT value;
        UINT reference = i + 1 + 256;
        get_texture_readback_with_command_list(output_textures[i], 0, &rb, queue, command_list);

        value = get_readback_uint(&rb, 0, 0, 0);
        ok(value == reference, "Readback value for texture iteration %u is: %u\n", i, value);

        release_resource_readback(&rb);
        reset_command_list(command_list, context.allocator);
    }

    for (i = 0; i < 256; i++)
        ID3D12Resource_Release(output_buffers[i]);
    for (i = 0; i < 256; i++)
        ID3D12Resource_Release(output_textures[i]);
    ID3D12DescriptorHeap_Release(heap);
    destroy_test_context(&context);
}

void test_bindless_cbv_sm51(void)
{
    test_bindless_cbv(false);
}

void test_bindless_cbv_dxil(void)
{
    test_bindless_cbv(true);
}

static void test_bindless_uav_counter(bool use_dxil)
{
    D3D12_ROOT_SIGNATURE_DESC root_signature_desc;
    D3D12_ROOT_PARAMETER root_parameters[1];
    D3D12_DESCRIPTOR_RANGE descriptor_ranges[1];
    ID3D12DescriptorHeap *heap, *cpu_heap;

    ID3D12Resource *output_buffers[256];
    struct resource_readback rb;

    ID3D12GraphicsCommandList* command_list;
    D3D12_CPU_DESCRIPTOR_HANDLE cpu_handle;
    D3D12_GPU_DESCRIPTOR_HANDLE gpu_handle;
    unsigned int i, descriptor_size;
    struct test_context context;
    ID3D12CommandQueue* queue;
    HRESULT hr;

#if 0
    RWStructuredBuffer<uint> Buffers[] : register(u2, space1);

    [numthreads(64, 1, 1)]
    void main(uint global_index : SV_DispatchThreadID, uint index : SV_GroupID)
    {
        // Need branch here or FXC complains about race condition.
        if (global_index < 512)
        {
            Buffers[NonUniformResourceIndex(global_index)][0] = global_index + 1;
            Buffers[NonUniformResourceIndex(global_index & ~3)].IncrementCounter();
        }
    }
#endif
    static const DWORD cs_code_dxbc[] =
    {
        0x43425844, 0x3282d7ba, 0xcefa5b07, 0xd66c504b, 0xe11b90a0, 0x00000001, 0x00000160, 0x00000003,
        0x0000002c, 0x0000003c, 0x0000004c, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x00000008, 0x00000000, 0x00000008, 0x58454853, 0x0000010c, 0x00050051, 0x00000043, 0x0100086a,
        0x0700009e, 0x0031ee46, 0x00000000, 0x00000002, 0xffffffff, 0x00000004, 0x00000001, 0x0200005f,
        0x00020012, 0x02000068, 0x00000001, 0x0400009b, 0x00000040, 0x00000001, 0x00000001, 0x0600004f,
        0x00100012, 0x00000000, 0x0002000a, 0x00004001, 0x00000200, 0x0304001f, 0x0010000a, 0x00000000,
        0x0600001e, 0x00100012, 0x00000000, 0x0002000a, 0x00004001, 0x00000001, 0x04000036, 0x00100022,
        0x00000000, 0x0002000a, 0x0d0000a8, 0x8621e012, 0x00020001, 0x00000000, 0x00000002, 0x0010001a,
        0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000, 0x0010000a, 0x00000000, 0x06000001,
        0x00100012, 0x00000000, 0x0002000a, 0x00004001, 0x007ffffc, 0x090000b2, 0x00100012, 0x00000000,
        0x8621e000, 0x00020001, 0x00000000, 0x00000002, 0x0010000a, 0x00000000, 0x01000015, 0x0100003e,
    };
    static const BYTE cs_code_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0x36, 0xf0, 0x07, 0x4c, 0xb5, 0xbe, 0x3c, 0x6f, 0xb8, 0xad, 0x74, 0x99, 0x8c, 0x91, 0x7a, 0xc8, 0x01, 0x00, 0x00, 0x00, 0x48, 0x06, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0xb8, 0x00, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x4c, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x88, 0x05, 0x00, 0x00,
        0x60, 0x00, 0x05, 0x00, 0x62, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x70, 0x05, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00,
        0x59, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c,
        0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20,
        0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f,
        0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
        0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x58, 0x23, 0x00,
        0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0x63, 0x0c, 0x22, 0x73, 0x04, 0x08, 0x99, 0x7b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x10, 0x68, 0x86, 0x85, 0x40, 0xc1, 0x29, 0xc4, 0x18,
        0xc8, 0x50, 0x9a, 0x23, 0x08, 0x8a, 0x81, 0x86, 0x19, 0x63, 0x11, 0x2b, 0x0a, 0x18, 0x68, 0x8c, 0x31, 0xc6, 0x30, 0xe4, 0x06, 0x02, 0x66, 0x32, 0x83, 0x71, 0x60, 0x87, 0x70, 0x98, 0x87, 0x79,
        0x70, 0x03, 0x59, 0xb8, 0x85, 0x59, 0xa0, 0x07, 0x79, 0xa8, 0x87, 0x71, 0xa0, 0x87, 0x7a, 0x90, 0x87, 0x72, 0x20, 0x07, 0x51, 0xa8, 0x07, 0x73, 0x30, 0x87, 0x72, 0x90, 0x07, 0x3e, 0xa8, 0x07,
        0x77, 0x98, 0x87, 0x74, 0x38, 0x07, 0x77, 0x28, 0x07, 0x72, 0x00, 0x83, 0x74, 0x70, 0x07, 0x7a, 0xf0, 0x03, 0x14, 0x8c, 0x04, 0x88, 0x24, 0xe7, 0x08, 0x40, 0x61, 0x0a, 0x00, 0x00, 0x00, 0x00,
        0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30,
        0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a,
        0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07,
        0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60,
        0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x04, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x12, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x30, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4,
        0x71, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x0b, 0x04, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47,
        0xc6, 0x04, 0x43, 0x1a, 0x25, 0x30, 0x02, 0x50, 0x08, 0xc5, 0x50, 0x04, 0x35, 0x50, 0x18, 0x05, 0x18, 0x40, 0x6b, 0x04, 0xa0, 0x06, 0x88, 0x16, 0x08, 0xcd, 0x19, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x79, 0x18, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x88, 0x0c, 0x6f, 0xec, 0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26, 0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05,
        0x06, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x86, 0x62, 0x82, 0x30, 0x18, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x0c, 0xc7, 0x06, 0x61, 0x30, 0x28, 0x8c, 0xcd, 0x4d, 0x10, 0x06, 0x64, 0xc3, 0x80, 0x24, 0xc4,
        0x04, 0x61, 0x82, 0x08, 0x4c, 0x10, 0x86, 0x64, 0x82, 0x30, 0x28, 0x13, 0x84, 0x61, 0x99, 0x20, 0x2c, 0xcd, 0x04, 0x61, 0x71, 0x36, 0x2c, 0xc4, 0xc2, 0x0c, 0x8d, 0xf3, 0x40, 0x11, 0x74, 0x6c,
        0x08, 0xa4, 0x0d, 0x04, 0x30, 0x01, 0xc0, 0x04, 0x41, 0x00, 0x36, 0x00, 0x1b, 0x06, 0xc2, 0xb2, 0x36, 0x04, 0xd7, 0x86, 0x61, 0xa8, 0x30, 0x12, 0x6d, 0x61, 0x69, 0x6e, 0x13, 0x04, 0xea, 0x99,
        0x20, 0x0c, 0xcc, 0x86, 0x81, 0x1b, 0x86, 0x0d, 0x04, 0xb1, 0x19, 0xdd, 0x86, 0xa2, 0xd2, 0x00, 0xca, 0xab, 0xc2, 0xc6, 0x66, 0xd7, 0xe6, 0x92, 0x46, 0x56, 0xe6, 0x46, 0x37, 0x25, 0x08, 0xaa,
        0x90, 0xe1, 0xb9, 0xd8, 0x95, 0xc9, 0xcd, 0xa5, 0xbd, 0xb9, 0x4d, 0x09, 0x88, 0x26, 0x64, 0x78, 0x2e, 0x76, 0x61, 0x6c, 0x76, 0x65, 0x72, 0x53, 0x02, 0xa3, 0x0e, 0x19, 0x9e, 0xcb, 0x1c, 0x5a,
        0x18, 0x59, 0x99, 0x5c, 0xd3, 0x1b, 0x59, 0x19, 0xdb, 0x94, 0x20, 0x29, 0x43, 0x86, 0xe7, 0x22, 0x57, 0x36, 0xf7, 0x56, 0x27, 0x37, 0x56, 0x36, 0x37, 0x25, 0x98, 0x2a, 0x91, 0xe1, 0xb9, 0xd0,
        0xe5, 0xc1, 0x95, 0x05, 0xb9, 0xb9, 0xbd, 0xd1, 0x85, 0xd1, 0xa5, 0xbd, 0xb9, 0xcd, 0x4d, 0x09, 0xb0, 0x3a, 0x64, 0x78, 0x2e, 0x65, 0x6e, 0x74, 0x72, 0x79, 0x50, 0x6f, 0x69, 0x6e, 0x74, 0x73,
        0x53, 0x02, 0x0f, 0x00, 0x79, 0x18, 0x00, 0x00, 0x51, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07,
        0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43,
        0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76,
        0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8,
        0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68,
        0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71,
        0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4,
        0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43,
        0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xcc, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x74, 0x60, 0x07, 0x37, 0x90, 0x87, 0x72,
        0x98, 0x87, 0x77, 0xa8, 0x07, 0x79, 0x18, 0x87, 0x72, 0x70, 0x83, 0x70, 0xa0, 0x07, 0x7a, 0x90, 0x87, 0x74, 0x10, 0x87, 0x7a, 0xa0, 0x87, 0x72, 0x98, 0xe1, 0x3c, 0xe4, 0x80, 0x0f, 0x6e, 0xe0,
        0x0e, 0xef, 0xe0, 0x0e, 0xf5, 0xe0, 0x0e, 0xe9, 0x60, 0x0e, 0xef, 0x20, 0x0f, 0xed, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x46, 0x50, 0x0d, 0x97, 0xef, 0x3c, 0x7e, 0x40,
        0x15, 0x05, 0x11, 0xb1, 0x93, 0x13, 0x11, 0x3e, 0x72, 0xdb, 0x26, 0x90, 0x0d, 0x97, 0xef, 0x3c, 0x7e, 0x40, 0x15, 0x05, 0x11, 0xb9, 0xcf, 0x00, 0x4c, 0x04, 0xe7, 0x50, 0xcd, 0x44, 0x44, 0x36,
        0x20, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44, 0x84, 0x40, 0x33, 0x2c, 0x84, 0x05, 0x48, 0xc3, 0xe5, 0x3b, 0x8f, 0x3f, 0x1d, 0x11, 0x01, 0x0c, 0xe2, 0xe0, 0x23, 0xb7, 0x6d, 0x00,
        0x04, 0x03, 0x20, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x00, 0x13, 0x04, 0x43, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x34, 0x66, 0x00, 0x4a,
        0xae, 0x74, 0x03, 0x0a, 0x10, 0xa1, 0x14, 0x03, 0x2a, 0xa1, 0x18, 0x03, 0xc8, 0x94, 0x00, 0x00, 0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0x40, 0x60, 0xc6, 0x34, 0xdc, 0x10, 0x18, 0x64, 0x30, 0xcb,
        0x10, 0x08, 0x41, 0x09, 0x16, 0xd4, 0x30, 0xc1, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x30, 0xda, 0x52, 0x5c, 0xc1, 0x33, 0x62, 0xd0, 0x00, 0x20, 0x08, 0x06, 0x8e, 0x96, 0x04, 0x18, 0x36, 0x34,
        0x4d, 0x63, 0x54, 0x81, 0x68, 0x05, 0x16, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xd3, 0x39, 0x88, 0x16, 0x48, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0x90, 0x7c, 0x4a, 0x90, 0xcc, 0x12, 0x08,
        0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    if (!init_compute_test_context(&context))
        return;

    if (use_dxil && !context_supports_dxil(&context))
    {
        destroy_test_context(&context);
        return;
    }

    command_list = context.list;
    queue = context.queue;

    root_signature_desc.NumParameters = 1;
    root_signature_desc.Flags = 0;
    root_signature_desc.NumStaticSamplers = 0;
    root_signature_desc.pStaticSamplers = NULL;
    root_signature_desc.pParameters = root_parameters;

    root_parameters[0].ParameterType = D3D12_ROOT_PARAMETER_TYPE_DESCRIPTOR_TABLE;
    root_parameters[0].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[0].DescriptorTable.NumDescriptorRanges = 1;
    root_parameters[0].DescriptorTable.pDescriptorRanges = descriptor_ranges;

    descriptor_ranges[0].RegisterSpace = 1;
    descriptor_ranges[0].BaseShaderRegister = 2;
    descriptor_ranges[0].OffsetInDescriptorsFromTableStart = 1;
    descriptor_ranges[0].NumDescriptors = UINT_MAX;
    descriptor_ranges[0].RangeType = D3D12_DESCRIPTOR_RANGE_TYPE_UAV;

    hr = create_root_signature(context.device, &root_signature_desc, &context.root_signature);
    ok(SUCCEEDED(hr), "Failed to create root signature, hr %#x.\n", hr);

    for (i = 0; i < 256; i++)
        output_buffers[i] = create_default_buffer(context.device, 256, D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);

    context.pipeline_state = create_compute_pipeline_state(context.device,
        context.root_signature,
        shader_bytecode(use_dxil ? (const void*)cs_code_dxil : (const void*)cs_code_dxbc, use_dxil ? sizeof(cs_code_dxil) : sizeof(cs_code_dxbc)));

    heap = create_gpu_descriptor_heap(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV, 800000);
    cpu_heap = create_cpu_descriptor_heap(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV, 800000);
    cpu_handle = ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(heap);
    gpu_handle = ID3D12DescriptorHeap_GetGPUDescriptorHandleForHeapStart(heap);
    descriptor_size = ID3D12Device_GetDescriptorHandleIncrementSize(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV);

    for (i = 0; i < 256; i++)
    {
        D3D12_UNORDERED_ACCESS_VIEW_DESC view;
        D3D12_CPU_DESCRIPTOR_HANDLE h = cpu_handle;

        view.Format = DXGI_FORMAT_UNKNOWN;
        view.ViewDimension = D3D12_UAV_DIMENSION_BUFFER;
        view.Buffer.FirstElement = 1;
        view.Buffer.NumElements = 63;
        view.Buffer.StructureByteStride = 4;
        view.Buffer.CounterOffsetInBytes = 0;
        view.Buffer.Flags = 0;
        h.ptr += (i + 1) * descriptor_size;
        ID3D12Device_CreateUnorderedAccessView(context.device, output_buffers[i], output_buffers[i ^ 1], &view, h);
    }

    /* Cannot UAV clear structured buffers, so use a separate raw byte address buffer for that. */
    for (i = 0; i < 256; i++)
    {
        D3D12_UNORDERED_ACCESS_VIEW_DESC view;
        D3D12_CPU_DESCRIPTOR_HANDLE cpu_h = cpu_handle;
        D3D12_GPU_DESCRIPTOR_HANDLE gpu_h = gpu_handle;
        static const UINT init_data[4] = { 10, 10, 10, 10 };

        cpu_h.ptr += (512 + i) * descriptor_size;
        gpu_h.ptr += (512 + i) * descriptor_size;

        view.Format = DXGI_FORMAT_R32_TYPELESS;
        view.ViewDimension = D3D12_UAV_DIMENSION_BUFFER;
        view.Buffer.FirstElement = 0;
        view.Buffer.NumElements = 64;
        view.Buffer.StructureByteStride = 0;
        view.Buffer.CounterOffsetInBytes = 0;
        view.Buffer.Flags = D3D12_BUFFER_UAV_FLAG_RAW;

        ID3D12Device_CreateUnorderedAccessView(context.device, output_buffers[i], NULL, &view, cpu_h);
        cpu_h = ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(cpu_heap);
        cpu_h.ptr += (512 + i) * descriptor_size;
        ID3D12Device_CreateUnorderedAccessView(context.device, output_buffers[i], NULL, &view, cpu_h);
        ID3D12GraphicsCommandList_ClearUnorderedAccessViewUint(command_list, gpu_h, cpu_h, output_buffers[i], init_data, 0, NULL);
    }

    uav_barrier(context.list, NULL);

    ID3D12GraphicsCommandList_SetComputeRootSignature(command_list, context.root_signature);
    ID3D12GraphicsCommandList_SetPipelineState(command_list, context.pipeline_state);
    ID3D12GraphicsCommandList_SetDescriptorHeaps(command_list, 1, &heap);
    ID3D12GraphicsCommandList_SetComputeRootDescriptorTable(command_list, 0, gpu_handle);
    ID3D12GraphicsCommandList_Dispatch(command_list, 4, 1, 1);

    for (i = 0; i < 256; i++)
        transition_resource_state(command_list, output_buffers[i], D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);

    for (i = 0; i < 256; i++)
    {
        UINT value;
        UINT reference = i + 1;
        get_buffer_readback_with_command_list(output_buffers[i], DXGI_FORMAT_UNKNOWN, &rb, queue, command_list);

        value = get_readback_uint(&rb, 1, 0, 0);
        ok(value == reference, "Readback value for buffer iteration %u is: %u\n", i, value);

        value = get_readback_uint(&rb, 0, 0, 0);
        reference = ((i ^ 1) & 3) == 0 ? 14 : 10;
        ok(value == reference, "Readback value for buffer counter iteration %u is: %u\n", i, value);

        release_resource_readback(&rb);
        reset_command_list(command_list, context.allocator);
    }

    for (i = 0; i < 256; i++)
        ID3D12Resource_Release(output_buffers[i]);
    ID3D12DescriptorHeap_Release(heap);
    ID3D12DescriptorHeap_Release(cpu_heap);
    destroy_test_context(&context);
}

void test_bindless_uav_sm51(void)
{
    test_bindless_uav(false);
}

void test_bindless_uav_dxil(void)
{
    test_bindless_uav(true);
}

void test_bindless_uav_counter_sm51(void)
{
    test_bindless_uav_counter(false);
}

void test_bindless_uav_counter_dxil(void)
{
    test_bindless_uav_counter(true);
}

static void test_bindless_bufinfo(bool use_dxil)
{
    D3D12_ROOT_SIGNATURE_DESC root_signature_desc;
    D3D12_ROOT_PARAMETER root_parameters[1];
    D3D12_DESCRIPTOR_RANGE descriptor_ranges[2];
    ID3D12DescriptorHeap *heap;

    ID3D12Resource *output_buffers[256];
    ID3D12Resource *output_textures[256];
    struct resource_readback rb;

    ID3D12GraphicsCommandList *command_list;
    D3D12_CPU_DESCRIPTOR_HANDLE cpu_handle;
    D3D12_GPU_DESCRIPTOR_HANDLE gpu_handle;
    unsigned int i, descriptor_size;
    struct test_context context;
    ID3D12CommandQueue *queue;
    HRESULT hr;

#if 0
    RWStructuredBuffer<uint> RWBuf[] : register(u0, space0);
    RWTexture2D<uint> RWTex[] : register(u0, space1);

    [numthreads(64, 1, 1)]
    void main(uint2 thr : SV_DispatchThreadID)
    {
        uint width, height, count, stride;
        RWTex[NonUniformResourceIndex(thr.x)].GetDimensions(width, height);

        if (thr.y == 0)
            RWTex[NonUniformResourceIndex(thr.x)][int2(0, 0)] = width;

        RWBuf[NonUniformResourceIndex(thr.x)].GetDimensions(count, stride);
        if (thr.y == 0)
            RWBuf[NonUniformResourceIndex(thr.x)][0] = count;
    }
#endif
    static const DWORD cs_code_dxbc[] =
    {
        0x43425844, 0xd997b29f, 0xbe9e2ef9, 0xe48c0c37, 0x0829289c, 0x00000001, 0x00000190, 0x00000003,
        0x0000002c, 0x0000003c, 0x0000004c, 0x4e475349, 0x00000008, 0x00000000, 0x00000008, 0x4e47534f,
        0x00000008, 0x00000000, 0x00000008, 0x58454853, 0x0000013c, 0x00050051, 0x0000004f, 0x0100086a,
        0x0700009e, 0x0031ee46, 0x00000000, 0x00000000, 0xffffffff, 0x00000004, 0x00000000, 0x0700189c,
        0x0031ee46, 0x00000001, 0x00000000, 0xffffffff, 0x00004444, 0x00000001, 0x0200005f, 0x00020032,
        0x02000068, 0x00000001, 0x0400009b, 0x00000040, 0x00000001, 0x00000001, 0x04000036, 0x00100012,
        0x00000000, 0x0002000a, 0x0a00103d, 0x00100022, 0x00000000, 0x00004001, 0x00000000, 0x8421ee16,
        0x00020001, 0x00000001, 0x0010000a, 0x00000000, 0x0200001f, 0x0002001a, 0x0d0000a4, 0x8421e0f2,
        0x00020001, 0x00000001, 0x0010000a, 0x00000000, 0x00004002, 0x00000000, 0x00000000, 0x00000000,
        0x00000000, 0x00100556, 0x00000000, 0x01000015, 0x08000079, 0x00100022, 0x00000000, 0x8421ee16,
        0x00020001, 0x00000000, 0x0010000a, 0x00000000, 0x0200001f, 0x0002001a, 0x0c0000a8, 0x8421e012,
        0x00020001, 0x00000000, 0x0010000a, 0x00000000, 0x00004001, 0x00000000, 0x00004001, 0x00000000,
        0x0010001a, 0x00000000, 0x01000015, 0x0100003e,
    };
    static const BYTE cs_code_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0x35, 0xf3, 0xf9, 0xfc, 0xcf, 0x21, 0x97, 0xf4, 0x9a, 0x42, 0x02, 0x24, 0x03, 0x10, 0x62, 0x4f, 0x01, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x5c, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x06, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x30, 0x06, 0x00, 0x00, 0x60, 0x00, 0x05, 0x00, 0x8c, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00,
        0x10, 0x00, 0x00, 0x00, 0x18, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x83, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00,
        0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42,
        0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50,
        0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d,
        0x86, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42,
        0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3,
        0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x70, 0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0xa0, 0x0c, 0x63, 0x0c, 0x22, 0x37, 0x0d, 0x97, 0x3f, 0x61,
        0x0f, 0x21, 0xf9, 0x1d, 0xc2, 0x10, 0x8d, 0x84, 0x38, 0x8d, 0x84, 0x88, 0x31, 0xc6, 0x98, 0x23, 0x40, 0xe8, 0xdc, 0x33, 0x5c, 0xfe, 0x84, 0x3d, 0x84, 0xe4, 0x87, 0x40, 0x33, 0x2c, 0x04, 0x0a,
        0x50, 0x21, 0xcc, 0x48, 0x83, 0x54, 0x59, 0xc0, 0x48, 0x63, 0x8c, 0x31, 0xc6, 0x38, 0xc4, 0xe6, 0x08, 0x82, 0x62, 0xa4, 0x71, 0xc6, 0x70, 0xf4, 0x8a, 0x02, 0x46, 0x1a, 0x63, 0x8c, 0x31, 0x0e,
        0xc5, 0x81, 0x80, 0x99, 0xcc, 0x60, 0x1c, 0xd8, 0x21, 0x1c, 0xe6, 0x61, 0x1e, 0xdc, 0x40, 0x16, 0x6e, 0x61, 0x16, 0xe8, 0x41, 0x1e, 0xea, 0x61, 0x1c, 0xe8, 0xa1, 0x1e, 0xe4, 0xa1, 0x1c, 0xc8,
        0x41, 0x14, 0xea, 0xc1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0x81, 0x0f, 0xea, 0xc1, 0x1d, 0xe6, 0x21, 0x1d, 0xce, 0xc1, 0x1d, 0xca, 0x81, 0x1c, 0xc0, 0x20, 0x1d, 0xdc, 0x81, 0x1e, 0xfc, 0x00, 0x05,
        0x23, 0x01, 0x28, 0xd5, 0x99, 0xbe, 0x71, 0x60, 0x87, 0x70, 0x98, 0x87, 0x79, 0x70, 0x03, 0x59, 0xb8, 0x05, 0x5a, 0x28, 0x07, 0x7c, 0xa0, 0x87, 0x7a, 0x90, 0x87, 0x72, 0x90, 0x03, 0x52, 0xe0,
        0x83, 0x7a, 0x70, 0x87, 0x79, 0x48, 0x87, 0x73, 0x70, 0x87, 0x72, 0x20, 0x07, 0x30, 0x48, 0x07, 0x77, 0xa0, 0x07, 0x3f, 0x40, 0xc1, 0x48, 0x80, 0x4b, 0x78, 0x8e, 0x00, 0x14, 0xa6, 0x00, 0x00,
        0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30,
        0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a,
        0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07,
        0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60,
        0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x04, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x14, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x30, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4,
        0x79, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x13, 0x01, 0x01, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x16, 0x08, 0x00, 0x0b, 0x00, 0x00, 0x00,
        0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x25, 0x30, 0x02, 0x50, 0x08, 0xc5, 0x50, 0x03, 0x85, 0x51, 0x04, 0xa5, 0x50, 0x80, 0x01, 0xe4,
        0x46, 0x00, 0x48, 0x17, 0x08, 0xe5, 0x19, 0x00, 0xb2, 0x33, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0xc4, 0x88, 0x0c, 0x6f, 0xec,
        0xed, 0x4d, 0x0c, 0x44, 0x06, 0x26, 0x26, 0xc7, 0x05, 0xa6, 0xc6, 0x05, 0x06, 0x66, 0x43, 0x10, 0x4c, 0x10, 0x06, 0x63, 0x82, 0x30, 0x1c, 0x1b, 0x84, 0x81, 0x98, 0x20, 0x0c, 0xc8, 0x06, 0x61,
        0x30, 0x28, 0x8c, 0xcd, 0x4d, 0x10, 0x86, 0x64, 0xc3, 0x80, 0x24, 0xc4, 0x04, 0xc1, 0x92, 0x08, 0x4c, 0x10, 0x06, 0x65, 0x82, 0x30, 0x2c, 0x13, 0x04, 0xe7, 0xd9, 0xb0, 0x10, 0x0b, 0x43, 0x10,
        0x8d, 0xf3, 0x3c, 0xcf, 0x31, 0x41, 0xc8, 0xa2, 0x09, 0xc2, 0xc0, 0x4c, 0x10, 0x86, 0x66, 0x83, 0x40, 0x4c, 0x1b, 0x96, 0x21, 0x62, 0x06, 0xa2, 0x91, 0x9e, 0xe7, 0xa1, 0x36, 0x08, 0x50, 0xb5,
        0x81, 0x00, 0x2c, 0x00, 0x98, 0x20, 0x08, 0xc0, 0x06, 0x60, 0xc3, 0x40, 0x64, 0xd9, 0x86, 0x40, 0xdb, 0x30, 0x0c, 0xd8, 0x46, 0xa2, 0x2d, 0x2c, 0xcd, 0x6d, 0x82, 0xa0, 0x41, 0x13, 0x84, 0xc1,
        0xd9, 0x30, 0x7c, 0xc3, 0xb0, 0x81, 0x20, 0x3c, 0x03, 0x0c, 0x36, 0x14, 0x58, 0x07, 0x5c, 0x61, 0x50, 0x85, 0x8d, 0xcd, 0xae, 0xcd, 0x25, 0x8d, 0xac, 0xcc, 0x8d, 0x6e, 0x4a, 0x10, 0x54, 0x21,
        0xc3, 0x73, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x9b, 0x12, 0x10, 0x4d, 0xc8, 0xf0, 0x5c, 0xec, 0xc2, 0xd8, 0xec, 0xca, 0xe4, 0xa6, 0x04, 0x46, 0x1d, 0x32, 0x3c, 0x97, 0x39, 0xb4, 0x30,
        0xb2, 0x32, 0xb9, 0xa6, 0x37, 0xb2, 0x32, 0xb6, 0x29, 0x41, 0x52, 0x86, 0x0c, 0xcf, 0x45, 0xae, 0x6c, 0xee, 0xad, 0x4e, 0x6e, 0xac, 0x6c, 0x6e, 0x4a, 0x60, 0x55, 0x22, 0xc3, 0x73, 0xa1, 0xcb,
        0x83, 0x2b, 0x0b, 0x72, 0x73, 0x7b, 0xa3, 0x0b, 0xa3, 0x4b, 0x7b, 0x73, 0x9b, 0x9b, 0x12, 0x6c, 0x75, 0xc8, 0xf0, 0x5c, 0xca, 0xdc, 0xe8, 0xe4, 0xf2, 0xa0, 0xde, 0xd2, 0xdc, 0xe8, 0xe6, 0xa6,
        0x04, 0x61, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x51, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07,
        0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43,
        0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76,
        0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8,
        0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68,
        0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71,
        0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4,
        0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43,
        0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xcc, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x74, 0x60, 0x07, 0x37, 0x90, 0x87, 0x72,
        0x98, 0x87, 0x77, 0xa8, 0x07, 0x79, 0x18, 0x87, 0x72, 0x70, 0x83, 0x70, 0xa0, 0x07, 0x7a, 0x90, 0x87, 0x74, 0x10, 0x87, 0x7a, 0xa0, 0x87, 0x72, 0x98, 0xe1, 0x3c, 0xe4, 0x80, 0x0f, 0x6e, 0xe0,
        0x0e, 0xef, 0xe0, 0x0e, 0xf5, 0xe0, 0x0e, 0xe9, 0x60, 0x0e, 0xef, 0x20, 0x0f, 0xed, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x17, 0x00, 0x00, 0x00, 0x56, 0x50, 0x0d, 0x97, 0xef, 0x3c, 0x7e, 0x40,
        0x15, 0x05, 0x11, 0xb1, 0x93, 0x13, 0x11, 0x3e, 0x72, 0xdb, 0x46, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44, 0x84, 0x40, 0x33, 0x2c, 0x84, 0x09, 0x4c, 0xc3, 0xe5, 0x3b, 0x8f,
        0x6f, 0x10, 0x53, 0x87, 0x30, 0x44, 0x23, 0x21, 0x4e, 0x23, 0xd9, 0x80, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0x13, 0x71, 0x4d, 0x54, 0x44, 0xb0, 0x93, 0x13, 0x11, 0x3e, 0x72, 0xdb, 0x16, 0x20, 0x0d,
        0x97, 0xef, 0x3c, 0xfe, 0x74, 0x44, 0x04, 0x30, 0x88, 0x83, 0x8f, 0xdc, 0xb6, 0x01, 0x10, 0x0c, 0x80, 0x34, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x13, 0x04, 0x43, 0x2c,
        0x10, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x34, 0x66, 0x00, 0x4a, 0x37, 0xa0, 0xe4, 0x0a, 0x32, 0xa0, 0x0c, 0x03, 0x4a, 0x31, 0x80, 0x4e, 0x09, 0x94, 0x07, 0xb9, 0x1a, 0x00, 0x00, 0x00,
        0x23, 0x06, 0x06, 0x00, 0x82, 0x60, 0x40, 0x6c, 0x48, 0x35, 0x62, 0x60, 0x00, 0x20, 0x08, 0x06, 0x04, 0x97, 0x5c, 0x25, 0x5c, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0xcf, 0xa6, 0x18, 0x59,
        0x40, 0x8c, 0x18, 0x1c, 0x00, 0x08, 0x82, 0x81, 0xe2, 0x29, 0x41, 0x36, 0x9a, 0x10, 0x00, 0xc3, 0x0d, 0xc5, 0x06, 0x06, 0xb3, 0x0c, 0x81, 0x10, 0x8c, 0x18, 0x38, 0x00, 0x08, 0x82, 0x01, 0x03,
        0x06, 0x0c, 0xc1, 0x71, 0x90, 0x20, 0x08, 0x42, 0x52, 0x07, 0x07, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0xf0, 0x80, 0xc1, 0xb3, 0x74, 0x41, 0x32, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0xca, 0x18,
        0x3c, 0x81, 0x34, 0x9a, 0x10, 0x00, 0x23, 0x06, 0x0d, 0x00, 0x82, 0x60, 0x10, 0x89, 0xc1, 0x33, 0x80, 0x01, 0x18, 0x04, 0x14, 0x45, 0x39, 0xb3, 0x04, 0x02, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    if (!init_compute_test_context(&context))
        return;

    if (use_dxil && !context_supports_dxil(&context))
    {
        destroy_test_context(&context);
        return;
    }

    command_list = context.list;
    queue = context.queue;

    root_signature_desc.NumParameters = 1;
    root_signature_desc.Flags = 0;
    root_signature_desc.NumStaticSamplers = 0;
    root_signature_desc.pStaticSamplers = NULL;
    root_signature_desc.pParameters = root_parameters;

    root_parameters[0].ParameterType = D3D12_ROOT_PARAMETER_TYPE_DESCRIPTOR_TABLE;
    root_parameters[0].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    root_parameters[0].DescriptorTable.NumDescriptorRanges = 2;
    root_parameters[0].DescriptorTable.pDescriptorRanges = descriptor_ranges;

    descriptor_ranges[0].RegisterSpace = 0;
    descriptor_ranges[0].BaseShaderRegister = 0;
    descriptor_ranges[0].OffsetInDescriptorsFromTableStart = 0;
    descriptor_ranges[0].NumDescriptors = UINT_MAX;
    descriptor_ranges[0].RangeType = D3D12_DESCRIPTOR_RANGE_TYPE_UAV;

    descriptor_ranges[1].RegisterSpace = 1;
    descriptor_ranges[1].BaseShaderRegister = 0;
    descriptor_ranges[1].OffsetInDescriptorsFromTableStart = 256;
    descriptor_ranges[1].NumDescriptors = UINT_MAX;
    descriptor_ranges[1].RangeType = D3D12_DESCRIPTOR_RANGE_TYPE_UAV;

    hr = create_root_signature(context.device, &root_signature_desc, &context.root_signature);
    ok(SUCCEEDED(hr), "Failed to create root signature, hr %#x.\n", hr);

    for (i = 0; i < 256; i++)
        output_buffers[i] = create_default_buffer(context.device, 4096, D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);
    for (i = 0; i < 256; i++)
        output_textures[i] = create_default_texture2d(context.device, i + 1, 1, 1, 1, DXGI_FORMAT_R32_UINT, D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);

    context.pipeline_state = create_compute_pipeline_state(context.device,
        context.root_signature,
        shader_bytecode(use_dxil ? (const void *)cs_code_dxil : (const void *)cs_code_dxbc, use_dxil ? sizeof(cs_code_dxil) : sizeof(cs_code_dxbc)));

    heap = create_gpu_descriptor_heap(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV, 800000);
    cpu_handle = ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(heap);
    gpu_handle = ID3D12DescriptorHeap_GetGPUDescriptorHandleForHeapStart(heap);
    descriptor_size = ID3D12Device_GetDescriptorHandleIncrementSize(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV);

    for (i = 0; i < 256; i++)
    {
        D3D12_UNORDERED_ACCESS_VIEW_DESC view;
        D3D12_CPU_DESCRIPTOR_HANDLE h = cpu_handle;
        view.Format = DXGI_FORMAT_UNKNOWN;
        view.ViewDimension = D3D12_UAV_DIMENSION_BUFFER;
        view.Buffer.FirstElement = i & 3; /* Unaligned FirstElement might affect bufinfo computation if we have to emulate alignment. */
        view.Buffer.NumElements = 1 + i;
        view.Buffer.StructureByteStride = 4;
        view.Buffer.CounterOffsetInBytes = 0;
        view.Buffer.Flags = 0;
        h.ptr += i * descriptor_size;
        ID3D12Device_CreateUnorderedAccessView(context.device, output_buffers[i], NULL, &view, h);
    }

    for (i = 0; i < 256; i++)
    {
        D3D12_UNORDERED_ACCESS_VIEW_DESC view;
        D3D12_CPU_DESCRIPTOR_HANDLE h = cpu_handle;
        view.Format = DXGI_FORMAT_R32_UINT;
        view.ViewDimension = D3D12_UAV_DIMENSION_TEXTURE2D;
        view.Texture2D.MipSlice = 0;
        view.Texture2D.PlaneSlice = 0;
        h.ptr += (256 + i) * descriptor_size;
        ID3D12Device_CreateUnorderedAccessView(context.device, output_textures[i], NULL, &view, h);
    }

    ID3D12GraphicsCommandList_SetComputeRootSignature(command_list, context.root_signature);
    ID3D12GraphicsCommandList_SetPipelineState(command_list, context.pipeline_state);
    ID3D12GraphicsCommandList_SetDescriptorHeaps(command_list, 1, &heap);
    ID3D12GraphicsCommandList_SetComputeRootDescriptorTable(command_list, 0, gpu_handle);
    ID3D12GraphicsCommandList_Dispatch(command_list, 4, 1, 1);

    for (i = 0; i < 256; i++)
        transition_resource_state(command_list, output_buffers[i], D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);
    for (i = 0; i < 256; i++)
        transition_resource_state(command_list, output_textures[i], D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);

    for (i = 0; i < 256; i++)
    {
        UINT value;
        UINT reference = i + 1;
        get_buffer_readback_with_command_list(output_buffers[i], DXGI_FORMAT_UNKNOWN, &rb, queue, command_list);

        value = get_readback_uint(&rb, i & 3, 0, 0);
        ok(value == reference, "Readback value for buffer iteration %u is: %u\n", i, value);

        release_resource_readback(&rb);
        reset_command_list(command_list, context.allocator);
    }

    for (i = 0; i < 256; i++)
    {
        UINT value;
        UINT reference = i + 1;
        get_texture_readback_with_command_list(output_textures[i], 0, &rb, queue, command_list);

        value = get_readback_uint(&rb, 0, 0, 0);
        ok(value == reference, "Readback value for texture iteration %u is: %u\n", i, value);

        release_resource_readback(&rb);
        reset_command_list(command_list, context.allocator);
    }

    for (i = 0; i < 256; i++)
        ID3D12Resource_Release(output_buffers[i]);
    for (i = 0; i < 256; i++)
        ID3D12Resource_Release(output_textures[i]);
    ID3D12DescriptorHeap_Release(heap);
    destroy_test_context(&context);
}

void test_bindless_bufinfo_sm51(void)
{
    test_bindless_bufinfo(false);
}

void test_bindless_bufinfo_dxil(void)
{
    test_bindless_bufinfo(true);
}

