/***************************************************************************
 *                                  _   _ ____  _
 *  Project                     ___| | | |  _ \| |
 *                             / __| | | | |_) | |
 *                            | (__| |_| |  _ <| |___
 *                             \___|\___/|_| \_\_____|
 *
 * Copyright (C) Daniel Stenberg, <daniel@haxx.se>, et al.
 *
 * This software is licensed as described in the file COPYING, which
 * you should have received as part of this distribution. The terms
 * are also available at https://curl.se/docs/copyright.html.
 *
 * You may opt to use, copy, modify, merge, publish, distribute and/or sell
 * copies of the Software, and permit persons to whom the Software is
 * furnished to do so, under the terms of the COPYING file.
 *
 * This software is distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY
 * KIND, either express or implied.
 *
 * SPDX-License-Identifier: curl
 *
 ***************************************************************************/

/*
 * Source file for all wolfSSL specific code for the TLS/SSL layer. No code
 * but vtls.c should ever call or use these functions.
 *
 */

#include "curl_setup.h"

#ifdef USE_WOLFSSL

#define WOLFSSL_OPTIONS_IGNORE_SYS
#include <wolfssl/options.h>
#include <wolfssl/version.h>

#if LIBWOLFSSL_VERSION_HEX < 0x03004006 /* wolfSSL 3.4.6 (2015) */
#error "wolfSSL version should be at least 3.4.6"
#endif

/* To determine what functions are available we rely on one or both of:
   - the user's options.h generated by wolfSSL
   - the symbols detected by curl's configure
   Since they are markedly different from one another, and one or the other may
   not be available, we do some checking below to bring things in sync. */

/* HAVE_ALPN is wolfSSL's build time symbol for enabling ALPN in options.h. */
#ifndef HAVE_ALPN
#ifdef HAVE_WOLFSSL_USEALPN
#define HAVE_ALPN
#endif
#endif

#include <limits.h>

#include "urldata.h"
#include "sendf.h"
#include "inet_pton.h"
#include "vtls.h"
#include "vtls_int.h"
#include "vtls_scache.h"
#include "keylog.h"
#include "parsedate.h"
#include "connect.h" /* for the connect timeout */
#include "select.h"
#include "strcase.h"
#include "x509asn1.h"
#include "curl_printf.h"
#include "multiif.h"

#include <wolfssl/ssl.h>
#include <wolfssl/error-ssl.h>
#include "wolfssl.h"

/* The last #include files should be: */
#include "curl_memory.h"
#include "memdebug.h"

#ifdef HAVE_WOLFSSL_CTX_GENERATEECHCONFIG
#define USE_ECH_WOLFSSL
#endif

/* KEEP_PEER_CERT is a product of the presence of build time symbol
   OPENSSL_EXTRA without NO_CERTS, depending on the version. KEEP_PEER_CERT is
   in wolfSSL's settings.h, and the latter two are build time symbols in
   options.h. */
#ifndef KEEP_PEER_CERT
#if defined(HAVE_WOLFSSL_GET_PEER_CERTIFICATE) || \
    (defined(OPENSSL_EXTRA) && !defined(NO_CERTS))
#define KEEP_PEER_CERT
#endif
#endif

#ifdef HAVE_WOLFSSL_BIO
#define USE_BIO_CHAIN
#ifdef HAVE_WOLFSSL_FULL_BIO
#define USE_FULL_BIO
#else /* HAVE_WOLFSSL_FULL_BIO */
#undef USE_FULL_BIO
#endif
/* wolfSSL 5.7.4 and older do not have these symbols, but only the
 * OpenSSL ones. */
#ifndef WOLFSSL_BIO_CTRL_GET_CLOSE
#define WOLFSSL_BIO_CTRL_GET_CLOSE    BIO_CTRL_GET_CLOSE
#define WOLFSSL_BIO_CTRL_SET_CLOSE    BIO_CTRL_SET_CLOSE
#define WOLFSSL_BIO_CTRL_FLUSH        BIO_CTRL_FLUSH
#define WOLFSSL_BIO_CTRL_DUP          BIO_CTRL_DUP
#define wolfSSL_BIO_set_retry_write   BIO_set_retry_write
#define wolfSSL_BIO_set_retry_read    BIO_set_retry_read
#endif /* !WOLFSSL_BIO_CTRL_GET_CLOSE */

#else /* HAVE_WOLFSSL_BIO */
#undef USE_BIO_CHAIN
#endif

#ifdef OPENSSL_EXTRA
/*
 * Availability note:
 * The TLS 1.3 secret callback (wolfSSL_set_tls13_secret_cb) was added in
 * wolfSSL 4.4.0, but requires the -DHAVE_SECRET_CALLBACK build option. If that
 * option is not set, then TLS 1.3 will not be logged.
 * For TLS 1.2 and before, we use wolfSSL_get_keys().
 * SSL_get_client_random and wolfSSL_get_keys require OPENSSL_EXTRA
 * (--enable-opensslextra or --enable-all).
 */
#if defined(HAVE_SECRET_CALLBACK) && defined(WOLFSSL_TLS13)
static int
wolfssl_tls13_secret_callback(SSL *ssl, int id, const unsigned char *secret,
                              int secretSz, void *ctx)
{
  const char *label;
  unsigned char client_random[SSL3_RANDOM_SIZE];
  (void)ctx;

  if(!ssl || !Curl_tls_keylog_enabled()) {
    return 0;
  }

  switch(id) {
  case CLIENT_EARLY_TRAFFIC_SECRET:
    label = "CLIENT_EARLY_TRAFFIC_SECRET";
    break;
  case CLIENT_HANDSHAKE_TRAFFIC_SECRET:
    label = "CLIENT_HANDSHAKE_TRAFFIC_SECRET";
    break;
  case SERVER_HANDSHAKE_TRAFFIC_SECRET:
    label = "SERVER_HANDSHAKE_TRAFFIC_SECRET";
    break;
  case CLIENT_TRAFFIC_SECRET:
    label = "CLIENT_TRAFFIC_SECRET_0";
    break;
  case SERVER_TRAFFIC_SECRET:
    label = "SERVER_TRAFFIC_SECRET_0";
    break;
  case EARLY_EXPORTER_SECRET:
    label = "EARLY_EXPORTER_SECRET";
    break;
  case EXPORTER_SECRET:
    label = "EXPORTER_SECRET";
    break;
  default:
    return 0;
  }

  if(SSL_get_client_random(ssl, client_random, SSL3_RANDOM_SIZE) == 0) {
    /* Should never happen as wolfSSL_KeepArrays() was called before. */
    return 0;
  }

  Curl_tls_keylog_write(label, client_random, secret, secretSz);
  return 0;
}
#endif /* defined(HAVE_SECRET_CALLBACK) && defined(WOLFSSL_TLS13) */

static void
wolfssl_log_tls12_secret(WOLFSSL *ssl)
{
  unsigned char *ms, *sr, *cr;
  unsigned int msLen, srLen, crLen, i, x = 0;

#if LIBWOLFSSL_VERSION_HEX >= 0x0300d000 /* >= 3.13.0 */
  /* wolfSSL_GetVersion is available since 3.13, we use it instead of
   * SSL_version since the latter relies on OPENSSL_ALL (--enable-opensslall or
   * --enable-all). Failing to perform this check could result in an unusable
   * key log line when TLS 1.3 is actually negotiated. */
  switch(wolfSSL_GetVersion(ssl)) {
  case WOLFSSL_SSLV3:
  case WOLFSSL_TLSV1:
  case WOLFSSL_TLSV1_1:
  case WOLFSSL_TLSV1_2:
    break;
  default:
    /* TLS 1.3 does not use this mechanism, the "master secret" returned below
     * is not directly usable. */
    return;
  }
#endif

  if(wolfSSL_get_keys(ssl, &ms, &msLen, &sr, &srLen, &cr, &crLen) !=
     WOLFSSL_SUCCESS) {
    return;
  }

  /* Check for a missing master secret and skip logging. That can happen if
   * curl rejects the server certificate and aborts the handshake.
   */
  for(i = 0; i < msLen; i++) {
    x |= ms[i];
  }
  if(x == 0) {
    return;
  }

  Curl_tls_keylog_write("CLIENT_RANDOM", cr, ms, msLen);
}
#endif /* OPENSSL_EXTRA */

static int wolfssl_do_file_type(const char *type)
{
  if(!type || !type[0])
    return WOLFSSL_FILETYPE_PEM;
  if(strcasecompare(type, "PEM"))
    return WOLFSSL_FILETYPE_PEM;
  if(strcasecompare(type, "DER"))
    return WOLFSSL_FILETYPE_ASN1;
  return -1;
}

#ifdef WOLFSSL_HAVE_KYBER
struct group_name_map {
  const word16 group;
  const char   *name;
};

static const struct group_name_map gnm[] = {
  { WOLFSSL_KYBER_LEVEL1, "KYBER_LEVEL1" },
  { WOLFSSL_KYBER_LEVEL3, "KYBER_LEVEL3" },
  { WOLFSSL_KYBER_LEVEL5, "KYBER_LEVEL5" },
  { WOLFSSL_P256_KYBER_LEVEL1, "P256_KYBER_LEVEL1" },
  { WOLFSSL_P384_KYBER_LEVEL3, "P384_KYBER_LEVEL3" },
  { WOLFSSL_P521_KYBER_LEVEL5, "P521_KYBER_LEVEL5" },
  { 0, NULL }
};
#endif

#ifdef USE_BIO_CHAIN

static int wolfssl_bio_cf_create(WOLFSSL_BIO *bio)
{
#ifdef USE_FULL_BIO
  wolfSSL_BIO_set_shutdown(bio, 1);
#endif
  wolfSSL_BIO_set_data(bio, NULL);
  return 1;
}

static int wolfssl_bio_cf_destroy(WOLFSSL_BIO *bio)
{
  if(!bio)
    return 0;
  return 1;
}

static long wolfssl_bio_cf_ctrl(WOLFSSL_BIO *bio, int cmd, long num, void *ptr)
{
  struct Curl_cfilter *cf = wolfSSL_BIO_get_data(bio);
  long ret = 1;

  (void)cf;
  (void)ptr;
  (void)num;
  switch(cmd) {
  case WOLFSSL_BIO_CTRL_GET_CLOSE:
#ifdef USE_FULL_BIO
    ret = (long)wolfSSL_BIO_get_shutdown(bio);
#else
    ret = 0;
#endif
    break;
  case WOLFSSL_BIO_CTRL_SET_CLOSE:
#ifdef USE_FULL_BIO
    wolfSSL_BIO_set_shutdown(bio, (int)num);
#endif
    break;
  case WOLFSSL_BIO_CTRL_FLUSH:
    /* we do no delayed writes, but if we ever would, this
     * needs to trigger it. */
    ret = 1;
    break;
  case WOLFSSL_BIO_CTRL_DUP:
    ret = 1;
    break;
#ifdef WOLFSSL_BIO_CTRL_EOF
  case WOLFSSL_BIO_CTRL_EOF:
    /* EOF has been reached on input? */
    return !cf->next || !cf->next->connected;
#endif
  default:
    ret = 0;
    break;
  }
  return ret;
}

static int wolfssl_bio_cf_out_write(WOLFSSL_BIO *bio,
                                    const char *buf, int blen)
{
  struct Curl_cfilter *cf = wolfSSL_BIO_get_data(bio);
  struct ssl_connect_data *connssl = cf->ctx;
  struct wolfssl_ctx *backend =
    (struct wolfssl_ctx *)connssl->backend;
  struct Curl_easy *data = CF_DATA_CURRENT(cf);
  ssize_t nwritten, skiplen = 0;
  CURLcode result = CURLE_OK;

  DEBUGASSERT(data);
  if(backend->shutting_down && backend->io_send_blocked_len &&
     (backend->io_send_blocked_len < blen)) {
    /* bug in wolfSSL: <https://github.com/wolfSSL/wolfssl/issues/7784>
     * It adds the close notify message again every time we retry
     * sending during shutdown. */
    CURL_TRC_CF(data, cf, "bio_write, shutdown restrict send of %d"
                " to %d bytes", blen, backend->io_send_blocked_len);
    skiplen = (ssize_t)(blen - backend->io_send_blocked_len);
    blen = backend->io_send_blocked_len;
  }
  nwritten = Curl_conn_cf_send(cf->next, data, buf, blen, FALSE, &result);
  backend->io_result = result;
  CURL_TRC_CF(data, cf, "bio_write(len=%d) -> %zd, %d",
              blen, nwritten, result);
#ifdef USE_FULL_BIO
  wolfSSL_BIO_clear_retry_flags(bio);
#endif
  if(nwritten < 0 && CURLE_AGAIN == result) {
    wolfSSL_BIO_set_retry_write(bio);
    if(backend->shutting_down && !backend->io_send_blocked_len)
      backend->io_send_blocked_len = blen;
  }
  else if(!result && skiplen)
    nwritten += skiplen;
  return (int)nwritten;
}

static int wolfssl_bio_cf_in_read(WOLFSSL_BIO *bio, char *buf, int blen)
{
  struct Curl_cfilter *cf = wolfSSL_BIO_get_data(bio);
  struct ssl_connect_data *connssl = cf->ctx;
  struct wolfssl_ctx *backend =
    (struct wolfssl_ctx *)connssl->backend;
  struct Curl_easy *data = CF_DATA_CURRENT(cf);
  ssize_t nread;
  CURLcode result = CURLE_OK;

  DEBUGASSERT(data);
  /* OpenSSL catches this case, so should we. */
  if(!buf)
    return 0;

  nread = Curl_conn_cf_recv(cf->next, data, buf, blen, &result);
  backend->io_result = result;
  CURL_TRC_CF(data, cf, "bio_read(len=%d) -> %zd, %d", blen, nread, result);
#ifdef USE_FULL_BIO
  wolfSSL_BIO_clear_retry_flags(bio);
#endif
  if(nread < 0 && CURLE_AGAIN == result)
    wolfSSL_BIO_set_retry_read(bio);
  else if(nread == 0)
    connssl->peer_closed = TRUE;
  return (int)nread;
}

static WOLFSSL_BIO_METHOD *wolfssl_bio_cf_method = NULL;

static void wolfssl_bio_cf_init_methods(void)
{
  wolfssl_bio_cf_method = wolfSSL_BIO_meth_new(WOLFSSL_BIO_MEMORY,
                                               "wolfSSL CF BIO");
  wolfSSL_BIO_meth_set_write(wolfssl_bio_cf_method, &wolfssl_bio_cf_out_write);
  wolfSSL_BIO_meth_set_read(wolfssl_bio_cf_method, &wolfssl_bio_cf_in_read);
  wolfSSL_BIO_meth_set_ctrl(wolfssl_bio_cf_method, &wolfssl_bio_cf_ctrl);
  wolfSSL_BIO_meth_set_create(wolfssl_bio_cf_method, &wolfssl_bio_cf_create);
  wolfSSL_BIO_meth_set_destroy(wolfssl_bio_cf_method, &wolfssl_bio_cf_destroy);
}

static void wolfssl_bio_cf_free_methods(void)
{
  wolfSSL_BIO_meth_free(wolfssl_bio_cf_method);
}

#else /* USE_BIO_CHAIN */

#define wolfssl_bio_cf_init_methods() Curl_nop_stmt
#define wolfssl_bio_cf_free_methods() Curl_nop_stmt

#endif /* !USE_BIO_CHAIN */

CURLcode Curl_wssl_cache_session(struct Curl_cfilter *cf,
                                 struct Curl_easy *data,
                                 const char *ssl_peer_key,
                                 WOLFSSL_SESSION *session,
                                 int ietf_tls_id,
                                 const char *alpn)
{
  CURLcode result = CURLE_OK;
  struct Curl_ssl_session *sc_session = NULL;
  unsigned char *sdata = NULL;
  unsigned int sdata_len;

  if(!session)
    goto out;

  sdata_len = wolfSSL_i2d_SSL_SESSION(session, NULL);
  if(sdata_len <= 0) {
    CURL_TRC_CF(data, cf, "fail to assess session length: %u", sdata_len);
    result = CURLE_FAILED_INIT;
    goto out;
  }
  sdata = calloc(1, sdata_len);
  if(!sdata) {
    failf(data, "unable to allocate session buffer of %u bytes", sdata_len);
    result = CURLE_OUT_OF_MEMORY;
    goto out;
  }
  sdata_len = wolfSSL_i2d_SSL_SESSION(session, &sdata);
  if(sdata_len <= 0) {
    CURL_TRC_CF(data, cf, "fail to serialize session: %u", sdata_len);
    result = CURLE_FAILED_INIT;
    goto out;
  }

  result = Curl_ssl_session_create(sdata, sdata_len,
                                   ietf_tls_id, alpn,
                                   (curl_off_t)time(NULL) +
                                   wolfSSL_SESSION_get_timeout(session), 0,
                                   &sc_session);
  sdata = NULL;  /* took ownership of sdata */
  if(!result) {
    result = Curl_ssl_scache_put(cf, data, ssl_peer_key, sc_session);
    /* took ownership of `sc_session` */
  }

out:
  free(sdata);
  return result;
}

static int wssl_vtls_new_session_cb(WOLFSSL *ssl, WOLFSSL_SESSION *session)
{
  struct Curl_cfilter *cf;

  cf = (struct Curl_cfilter*)wolfSSL_get_app_data(ssl);
  DEBUGASSERT(cf != NULL);
  if(cf && session) {
    struct ssl_connect_data *connssl = cf->ctx;
    struct Curl_easy *data = CF_DATA_CURRENT(cf);
    DEBUGASSERT(connssl);
    DEBUGASSERT(data);
    if(connssl && data) {
      (void)Curl_wssl_cache_session(cf, data, connssl->peer.scache_key,
                                    session, wolfSSL_version(ssl),
                                    connssl->negotiated.alpn);
    }
  }
  return 0;
}

CURLcode Curl_wssl_setup_session(struct Curl_cfilter *cf,
                                 struct Curl_easy *data,
                                 struct wolfssl_ctx *wss,
                                 const char *ssl_peer_key)
{
  struct Curl_ssl_session *sc_session = NULL;
  CURLcode result;

  result = Curl_ssl_scache_take(cf, data, ssl_peer_key, &sc_session);
  if(!result && sc_session && sc_session->sdata && sc_session->sdata_len) {
    WOLFSSL_SESSION *session;
    /* wolfSSL changes the passed pointer for whatever reasons, yikes */
    const unsigned char *sdata = sc_session->sdata;
    session = wolfSSL_d2i_SSL_SESSION(NULL, &sdata,
                                      (long)sc_session->sdata_len);
    if(session) {
      int ret = wolfSSL_set_session(wss->handle, session);
      if(ret != WOLFSSL_SUCCESS) {
        Curl_ssl_session_destroy(sc_session);
        sc_session = NULL;
        infof(data, "cached session not accepted (%d), "
              "removing from cache", ret);
      }
      else
        infof(data, "SSL reusing session ID");
      wolfSSL_SESSION_free(session);
    }
    else {
      failf(data, "could not decode previous session");
    }
  }
  Curl_ssl_scache_return(cf, data, ssl_peer_key, sc_session);
  return result;
}

static CURLcode wssl_populate_x509_store(struct Curl_cfilter *cf,
                                         struct Curl_easy *data,
                                         WOLFSSL_X509_STORE *store,
                                         struct wolfssl_ctx *wssl)
{
  struct ssl_primary_config *conn_config = Curl_ssl_cf_get_primary_config(cf);
  const struct curl_blob *ca_info_blob = conn_config->ca_info_blob;
  const char * const ssl_cafile =
    /* CURLOPT_CAINFO_BLOB overrides CURLOPT_CAINFO */
    (ca_info_blob ? NULL : conn_config->CAfile);
  const char * const ssl_capath = conn_config->CApath;
  struct ssl_config_data *ssl_config = Curl_ssl_cf_get_config(cf, data);
  bool imported_native_ca = FALSE;

#if !defined(NO_FILESYSTEM) && defined(WOLFSSL_SYS_CA_CERTS)
  /* load native CA certificates */
  if(ssl_config->native_ca_store) {
    if(wolfSSL_CTX_load_system_CA_certs(wssl->ctx) != WOLFSSL_SUCCESS) {
      infof(data, "error importing native CA store, continuing anyway");
    }
    else {
      imported_native_ca = TRUE;
      infof(data, "successfully imported native CA store");
      wssl->x509_store_setup = TRUE;
    }
  }
#endif /* !NO_FILESYSTEM */

  /* load certificate blob */
  if(ca_info_blob) {
    if(wolfSSL_CTX_load_verify_buffer(wssl->ctx, ca_info_blob->data,
                                      (long)ca_info_blob->len,
                                      WOLFSSL_FILETYPE_PEM) !=
       WOLFSSL_SUCCESS) {
      if(imported_native_ca) {
        infof(data, "error importing CA certificate blob, continuing anyway");
      }
      else {
        failf(data, "error importing CA certificate blob");
        return CURLE_SSL_CACERT_BADFILE;
      }
    }
    else {
      infof(data, "successfully imported CA certificate blob");
      wssl->x509_store_setup = TRUE;
    }
  }

#ifndef NO_FILESYSTEM
  /* load trusted cacert from file if not blob */

  CURL_TRC_CF(data, cf, "wssl_populate_x509_store, path=%s, blob=%d",
              ssl_cafile ? ssl_cafile : "none", !!ca_info_blob);
  if(!store)
    return CURLE_OUT_OF_MEMORY;

  if((ssl_cafile || ssl_capath) && (!wssl->x509_store_setup)) {
    int rc =
      wolfSSL_CTX_load_verify_locations_ex(wssl->ctx,
                                           ssl_cafile,
                                           ssl_capath,
                                           WOLFSSL_LOAD_FLAG_IGNORE_ERR);
    if(WOLFSSL_SUCCESS != rc) {
      if(conn_config->verifypeer) {
        /* Fail if we insist on successfully verifying the server. */
        failf(data, "error setting certificate verify locations:"
              " CAfile: %s CApath: %s",
              ssl_cafile ? ssl_cafile : "none",
              ssl_capath ? ssl_capath : "none");
        return CURLE_SSL_CACERT_BADFILE;
      }
      else {
        /* Just continue with a warning if no strict certificate
           verification is required. */
        infof(data, "error setting certificate verify locations,"
              " continuing anyway:");
      }
    }
    else {
      /* Everything is fine. */
      infof(data, "successfully set certificate verify locations:");
    }
    infof(data, " CAfile: %s", ssl_cafile ? ssl_cafile : "none");
    infof(data, " CApath: %s", ssl_capath ? ssl_capath : "none");
  }
#endif
  (void)store;
  wssl->x509_store_setup = TRUE;
  return CURLE_OK;
}

/* key to use at `multi->proto_hash` */
#define MPROTO_WSSL_X509_KEY   "tls:wssl:x509:share"

struct wssl_x509_share {
  char *CAfile;         /* CAfile path used to generate X509 store */
  WOLFSSL_X509_STORE *store; /* cached X509 store or NULL if none */
  struct curltime time; /* when the cached store was created */
};

static void wssl_x509_share_free(void *key, size_t key_len, void *p)
{
  struct wssl_x509_share *share = p;
  DEBUGASSERT(key_len == (sizeof(MPROTO_WSSL_X509_KEY)-1));
  DEBUGASSERT(!memcmp(MPROTO_WSSL_X509_KEY, key, key_len));
  (void)key;
  (void)key_len;
  if(share->store) {
    wolfSSL_X509_STORE_free(share->store);
  }
  free(share->CAfile);
  free(share);
}

static bool
wssl_cached_x509_store_expired(const struct Curl_easy *data,
                               const struct wssl_x509_share *mb)
{
  const struct ssl_general_config *cfg = &data->set.general_ssl;
  struct curltime now = Curl_now();
  timediff_t elapsed_ms = Curl_timediff(now, mb->time);
  timediff_t timeout_ms = cfg->ca_cache_timeout * (timediff_t)1000;

  if(timeout_ms < 0)
    return FALSE;

  return elapsed_ms >= timeout_ms;
}

static bool
wssl_cached_x509_store_different(struct Curl_cfilter *cf,
                                 const struct wssl_x509_share *mb)
{
  struct ssl_primary_config *conn_config = Curl_ssl_cf_get_primary_config(cf);
  if(!mb->CAfile || !conn_config->CAfile)
    return mb->CAfile != conn_config->CAfile;

  return strcmp(mb->CAfile, conn_config->CAfile);
}

static WOLFSSL_X509_STORE *wssl_get_cached_x509_store(struct Curl_cfilter *cf,
                                                  const struct Curl_easy *data)
{
  struct Curl_multi *multi = data->multi;
  struct wssl_x509_share *share;
  WOLFSSL_X509_STORE *store = NULL;

  DEBUGASSERT(multi);
  share = multi ? Curl_hash_pick(&multi->proto_hash,
                                 (void *)MPROTO_WSSL_X509_KEY,
                                 sizeof(MPROTO_WSSL_X509_KEY)-1) : NULL;
  if(share && share->store &&
     !wssl_cached_x509_store_expired(data, share) &&
     !wssl_cached_x509_store_different(cf, share)) {
    store = share->store;
  }

  return store;
}

static void wssl_set_cached_x509_store(struct Curl_cfilter *cf,
                                       const struct Curl_easy *data,
                                       WOLFSSL_X509_STORE *store)
{
  struct ssl_primary_config *conn_config = Curl_ssl_cf_get_primary_config(cf);
  struct Curl_multi *multi = data->multi;
  struct wssl_x509_share *share;

  DEBUGASSERT(multi);
  if(!multi)
    return;
  share = Curl_hash_pick(&multi->proto_hash,
                         (void *)MPROTO_WSSL_X509_KEY,
                         sizeof(MPROTO_WSSL_X509_KEY)-1);

  if(!share) {
    share = calloc(1, sizeof(*share));
    if(!share)
      return;
    if(!Curl_hash_add2(&multi->proto_hash,
                       (void *)MPROTO_WSSL_X509_KEY,
                       sizeof(MPROTO_WSSL_X509_KEY)-1,
                       share, wssl_x509_share_free)) {
      free(share);
      return;
    }
  }

  if(wolfSSL_X509_STORE_up_ref(store)) {
    char *CAfile = NULL;

    if(conn_config->CAfile) {
      CAfile = strdup(conn_config->CAfile);
      if(!CAfile) {
        wolfSSL_X509_STORE_free(store);
        return;
      }
    }

    if(share->store) {
      wolfSSL_X509_STORE_free(share->store);
      free(share->CAfile);
    }

    share->time = Curl_now();
    share->store = store;
    share->CAfile = CAfile;
  }
}

CURLcode Curl_wssl_setup_x509_store(struct Curl_cfilter *cf,
                                    struct Curl_easy *data,
                                    struct wolfssl_ctx *wssl)
{
  struct ssl_primary_config *conn_config = Curl_ssl_cf_get_primary_config(cf);
  struct ssl_config_data *ssl_config = Curl_ssl_cf_get_config(cf, data);
  CURLcode result = CURLE_OK;
  WOLFSSL_X509_STORE *cached_store;
  bool cache_criteria_met;

  /* Consider the X509 store cacheable if it comes exclusively from a CAfile,
     or no source is provided and we are falling back to wolfSSL's built-in
     default. */
  cache_criteria_met = (data->set.general_ssl.ca_cache_timeout != 0) &&
    conn_config->verifypeer &&
    !conn_config->CApath &&
    !conn_config->ca_info_blob &&
    !ssl_config->primary.CRLfile &&
    !ssl_config->native_ca_store;

  cached_store = cache_criteria_met ? wssl_get_cached_x509_store(cf, data)
                                    : NULL;
  if(cached_store && wolfSSL_CTX_get_cert_store(wssl->ctx) == cached_store) {
    /* The cached store is already in use, do nothing. */
  }
  else if(cached_store && wolfSSL_X509_STORE_up_ref(cached_store)) {
    wolfSSL_CTX_set_cert_store(wssl->ctx, cached_store);
  }
  else if(cache_criteria_met) {
    /* wolfSSL's initial store in CTX is not shareable by default.
     * Make a new one, suitable for adding to the cache. See #14278 */
    WOLFSSL_X509_STORE *store = wolfSSL_X509_STORE_new();
    if(!store) {
      failf(data, "SSL: could not create a X509 store");
      return CURLE_OUT_OF_MEMORY;
    }
    wolfSSL_CTX_set_cert_store(wssl->ctx, store);

    result = wssl_populate_x509_store(cf, data, store, wssl);
    if(!result) {
      wssl_set_cached_x509_store(cf, data, store);
    }
  }
  else {
   /* We never share the CTX's store, use it. */
   WOLFSSL_X509_STORE *store = wolfSSL_CTX_get_cert_store(wssl->ctx);
   result = wssl_populate_x509_store(cf, data, store, wssl);
  }

  return result;
}

#ifdef WOLFSSL_TLS13
static CURLcode
wssl_add_default_ciphers(bool tls13, struct dynbuf *buf)
{
  int i;
  char *str;

  for(i = 0; (str = wolfSSL_get_cipher_list(i)) != NULL; i++) {
    size_t n;
    if((strncmp(str, "TLS13", 5) == 0) != tls13)
      continue;

    /* if there already is data in the string, add colon separator */
    if(Curl_dyn_len(buf)) {
      CURLcode result = Curl_dyn_addn(buf, ":", 1);
      if(result)
        return result;
    }

    n = strlen(str);
    if(Curl_dyn_addn(buf, str, n))
      return CURLE_OUT_OF_MEMORY;
  }

  return CURLE_OK;
}
#endif

/* 4.2.0 (2019) */
#if LIBWOLFSSL_VERSION_HEX < 0x04002000 || !defined(OPENSSL_EXTRA)
static int
wssl_legacy_CTX_set_min_proto_version(WOLFSSL_CTX* ctx, int version)
{
  int res;
  switch(version) {
  default:
  case TLS1_VERSION:
    res = wolfSSL_CTX_SetMinVersion(ctx, WOLFSSL_TLSV1);
    if(res == WOLFSSL_SUCCESS)
      return res;
    FALLTHROUGH();
  case TLS1_1_VERSION:
    res = wolfSSL_CTX_SetMinVersion(ctx, WOLFSSL_TLSV1_1);
    if(res == WOLFSSL_SUCCESS)
      return res;
    FALLTHROUGH();
  case TLS1_2_VERSION:
    res = wolfSSL_CTX_SetMinVersion(ctx, WOLFSSL_TLSV1_2);
#ifdef WOLFSSL_TLS13
    if(res == WOLFSSL_SUCCESS)
      return res;
    FALLTHROUGH();
  case TLS1_3_VERSION:
    res = wolfSSL_CTX_SetMinVersion(ctx, WOLFSSL_TLSV1_3);
#endif
  }
  return res;
}
static int
wssl_legacy_CTX_set_max_proto_version(WOLFSSL_CTX* ctx, int version)
{
  (void) ctx, (void) version;
  return WOLFSSL_NOT_IMPLEMENTED;
}
#define wolfSSL_CTX_set_min_proto_version wssl_legacy_CTX_set_min_proto_version
#define wolfSSL_CTX_set_max_proto_version wssl_legacy_CTX_set_max_proto_version
#endif

/*
 * This function loads all the client/CA certificates and CRLs. Setup the TLS
 * layer and do all necessary magic.
 */
static CURLcode
wolfssl_connect_step1(struct Curl_cfilter *cf, struct Curl_easy *data)
{
  int res;
  char *curves;
  struct ssl_connect_data *connssl = cf->ctx;
  struct wolfssl_ctx *backend =
    (struct wolfssl_ctx *)connssl->backend;
  struct ssl_primary_config *conn_config = Curl_ssl_cf_get_primary_config(cf);
  const struct ssl_config_data *ssl_config = Curl_ssl_cf_get_config(cf, data);
  WOLFSSL_METHOD* req_method = NULL;
#ifdef WOLFSSL_HAVE_KYBER
  word16 pqkem = 0;
  size_t idx = 0;
#endif

  DEBUGASSERT(backend);

  if(connssl->state == ssl_connection_complete)
    return CURLE_OK;

#if LIBWOLFSSL_VERSION_HEX < 0x04002000 /* 4.2.0 (2019) */
  req_method = wolfSSLv23_client_method();
#else
  req_method = wolfTLS_client_method();
#endif
  if(!req_method) {
    failf(data, "wolfSSL: could not create a client method");
    return CURLE_OUT_OF_MEMORY;
  }

  if(backend->ctx)
    wolfSSL_CTX_free(backend->ctx);

  backend->ctx = wolfSSL_CTX_new(req_method);
  if(!backend->ctx) {
    failf(data, "wolfSSL: could not create a context");
    return CURLE_OUT_OF_MEMORY;
  }

  switch(conn_config->version) {
  case CURL_SSLVERSION_DEFAULT:
  case CURL_SSLVERSION_TLSv1:
  case CURL_SSLVERSION_TLSv1_0:
    res = wolfSSL_CTX_set_min_proto_version(backend->ctx, TLS1_VERSION);
    break;
  case CURL_SSLVERSION_TLSv1_1:
    res = wolfSSL_CTX_set_min_proto_version(backend->ctx, TLS1_1_VERSION);
    break;
  case CURL_SSLVERSION_TLSv1_2:
    res = wolfSSL_CTX_set_min_proto_version(backend->ctx, TLS1_2_VERSION);
    break;
#ifdef WOLFSSL_TLS13
  case CURL_SSLVERSION_TLSv1_3:
    res = wolfSSL_CTX_set_min_proto_version(backend->ctx, TLS1_3_VERSION);
    break;
#endif
  default:
    failf(data, "wolfSSL: unsupported minimum TLS version value");
    return CURLE_SSL_CONNECT_ERROR;
  }
  if(res != WOLFSSL_SUCCESS) {
    failf(data, "wolfSSL: failed set the minimum TLS version");
    return CURLE_SSL_CONNECT_ERROR;
  }

  switch(conn_config->version_max) {
#ifdef WOLFSSL_TLS13
  case CURL_SSLVERSION_MAX_TLSv1_3:
    res = wolfSSL_CTX_set_max_proto_version(backend->ctx, TLS1_3_VERSION);
    break;
#endif
  case CURL_SSLVERSION_MAX_TLSv1_2:
    res = wolfSSL_CTX_set_max_proto_version(backend->ctx, TLS1_2_VERSION);
    break;
  case CURL_SSLVERSION_MAX_TLSv1_1:
    res = wolfSSL_CTX_set_max_proto_version(backend->ctx, TLS1_1_VERSION);
    break;
  case CURL_SSLVERSION_MAX_TLSv1_0:
    res = wolfSSL_CTX_set_max_proto_version(backend->ctx, TLS1_VERSION);
    break;
  case CURL_SSLVERSION_MAX_DEFAULT:
  case CURL_SSLVERSION_MAX_NONE:
    res = WOLFSSL_SUCCESS;
    break;
  default:
    failf(data, "wolfSSL: unsupported maximum TLS version value");
    return CURLE_SSL_CONNECT_ERROR;
  }
  if(res != WOLFSSL_SUCCESS) {
    failf(data, "wolfSSL: failed set the maximum TLS version");
    return CURLE_SSL_CONNECT_ERROR;
  }

#ifndef WOLFSSL_TLS13
  {
    char *ciphers = conn_config->cipher_list;
    if(ciphers) {
      if(!SSL_CTX_set_cipher_list(backend->ctx, ciphers)) {
        failf(data, "failed setting cipher list: %s", ciphers);
        return CURLE_SSL_CIPHER;
      }
      infof(data, "Cipher selection: %s", ciphers);
    }
  }
#else
#define MAX_CIPHER_LEN 4096
  if(conn_config->cipher_list || conn_config->cipher_list13) {
    const char *ciphers12 = conn_config->cipher_list;
    const char *ciphers13 = conn_config->cipher_list13;
    struct dynbuf c;
    CURLcode result;
    Curl_dyn_init(&c, MAX_CIPHER_LEN);

    if(ciphers13)
      result = Curl_dyn_add(&c, ciphers13);
    else
      result = wssl_add_default_ciphers(TRUE, &c);

    if(!result) {
      if(ciphers12) {
        if(Curl_dyn_len(&c))
          result = Curl_dyn_addn(&c, ":", 1);
        if(!result)
          result = Curl_dyn_add(&c, ciphers12);
      }
      else
        result = wssl_add_default_ciphers(FALSE, &c);
    }
    if(result)
      return result;

    if(!wolfSSL_CTX_set_cipher_list(backend->ctx, Curl_dyn_ptr(&c))) {
      failf(data, "failed setting cipher list: %s", Curl_dyn_ptr(&c));
      Curl_dyn_free(&c);
      return CURLE_SSL_CIPHER;
    }
    infof(data, "Cipher selection: %s", Curl_dyn_ptr(&c));
    Curl_dyn_free(&c);
  }
#endif

  curves = conn_config->curves;
  if(curves) {

#ifdef WOLFSSL_HAVE_KYBER
    for(idx = 0; gnm[idx].name != NULL; idx++) {
      if(strncmp(curves, gnm[idx].name, strlen(gnm[idx].name)) == 0) {
        pqkem = gnm[idx].group;
        break;
      }
    }

    if(pqkem == 0)
#endif
    {
      if(!wolfSSL_CTX_set1_curves_list(backend->ctx, curves)) {
        failf(data, "failed setting curves list: '%s'", curves);
        return CURLE_SSL_CIPHER;
      }
    }
  }

  /* Load the client certificate, and private key */
#ifndef NO_FILESYSTEM
  if(ssl_config->primary.cert_blob || ssl_config->primary.clientcert) {
    const char *cert_file = ssl_config->primary.clientcert;
    const char *key_file = ssl_config->key;
    const struct curl_blob *cert_blob = ssl_config->primary.cert_blob;
    const struct curl_blob *key_blob = ssl_config->key_blob;
    int file_type = wolfssl_do_file_type(ssl_config->cert_type);
    int rc;

    switch(file_type) {
    case WOLFSSL_FILETYPE_PEM:
      rc = cert_blob ?
        wolfSSL_CTX_use_certificate_chain_buffer(backend->ctx,
                                                 cert_blob->data,
                                                 (long)cert_blob->len) :
        wolfSSL_CTX_use_certificate_chain_file(backend->ctx, cert_file);
      break;
    case WOLFSSL_FILETYPE_ASN1:
      rc = cert_blob ?
        wolfSSL_CTX_use_certificate_buffer(backend->ctx, cert_blob->data,
                                           (long)cert_blob->len, file_type) :
        wolfSSL_CTX_use_certificate_file(backend->ctx, cert_file, file_type);
      break;
    default:
      failf(data, "unknown cert type");
      return CURLE_BAD_FUNCTION_ARGUMENT;
    }
    if(rc != 1) {
      failf(data, "unable to use client certificate");
      return CURLE_SSL_CONNECT_ERROR;
    }

    if(!key_blob && !key_file) {
      key_blob = cert_blob;
      key_file = cert_file;
    }
    else
      file_type = wolfssl_do_file_type(ssl_config->key_type);

    rc = key_blob ?
      wolfSSL_CTX_use_PrivateKey_buffer(backend->ctx, key_blob->data,
                                        (long)key_blob->len, file_type) :
      wolfSSL_CTX_use_PrivateKey_file(backend->ctx, key_file, file_type);
    if(rc != 1) {
      failf(data, "unable to set private key");
      return CURLE_SSL_CONNECT_ERROR;
    }
  }
#else /* NO_FILESYSTEM */
  if(ssl_config->primary.cert_blob) {
    const struct curl_blob *cert_blob = ssl_config->primary.cert_blob;
    const struct curl_blob *key_blob = ssl_config->key_blob;
    int file_type = wolfssl_do_file_type(ssl_config->cert_type);
    int rc;

    switch(file_type) {
    case WOLFSSL_FILETYPE_PEM:
      rc = wolfSSL_CTX_use_certificate_chain_buffer(backend->ctx,
                                                    cert_blob->data,
                                                    (long)cert_blob->len);
      break;
    case WOLFSSL_FILETYPE_ASN1:
      rc = wolfSSL_CTX_use_certificate_buffer(backend->ctx, cert_blob->data,
                                              (long)cert_blob->len, file_type);
      break;
    default:
      failf(data, "unknown cert type");
      return CURLE_BAD_FUNCTION_ARGUMENT;
    }
    if(rc != 1) {
      failf(data, "unable to use client certificate");
      return CURLE_SSL_CONNECT_ERROR;
    }

    if(!key_blob)
      key_blob = cert_blob;
    else
      file_type = wolfssl_do_file_type(ssl_config->key_type);

    if(wolfSSL_CTX_use_PrivateKey_buffer(backend->ctx, key_blob->data,
                                         (long)key_blob->len,
                                         file_type) != 1) {
      failf(data, "unable to set private key");
      return CURLE_SSL_CONNECT_ERROR;
    }
  }
#endif /* !NO_FILESYSTEM */

  /* SSL always tries to verify the peer, this only says whether it should
   * fail to connect if the verification fails, or if it should continue
   * anyway. In the latter case the result of the verification is checked with
   * SSL_get_verify_result() below. */
  wolfSSL_CTX_set_verify(backend->ctx,
                         conn_config->verifypeer ? WOLFSSL_VERIFY_PEER :
                         WOLFSSL_VERIFY_NONE, NULL);

#ifdef HAVE_SNI
  if(connssl->peer.sni) {
    size_t sni_len = strlen(connssl->peer.sni);
    if((sni_len < USHRT_MAX)) {
      if(wolfSSL_CTX_UseSNI(backend->ctx, WOLFSSL_SNI_HOST_NAME,
                            connssl->peer.sni,
                            (unsigned short)sni_len) != 1) {
        failf(data, "Failed to set SNI");
        return CURLE_SSL_CONNECT_ERROR;
      }
    }
  }
#endif

  /* give application a chance to interfere with SSL set up. */
  if(data->set.ssl.fsslctx) {
    CURLcode result;
    if(!backend->x509_store_setup) {
      result = Curl_wssl_setup_x509_store(cf, data, backend);
      if(result)
        return result;
    }
    result = (*data->set.ssl.fsslctx)(data, backend->ctx,
                                      data->set.ssl.fsslctxp);
    if(result) {
      failf(data, "error signaled by ssl ctx callback");
      return result;
    }
  }
#ifdef NO_FILESYSTEM
  else if(conn_config->verifypeer) {
    failf(data, "SSL: Certificates cannot be loaded because wolfSSL was built"
          " with \"no filesystem\". Either disable peer verification"
          " (insecure) or if you are building an application with libcurl you"
          " can load certificates via CURLOPT_SSL_CTX_FUNCTION.");
    return CURLE_SSL_CONNECT_ERROR;
  }
#endif

  /* Let's make an SSL structure */
  if(backend->handle)
    wolfSSL_free(backend->handle);
  backend->handle = wolfSSL_new(backend->ctx);
  if(!backend->handle) {
    failf(data, "SSL: could not create a handle");
    return CURLE_OUT_OF_MEMORY;
  }

#ifdef WOLFSSL_HAVE_KYBER
  if(pqkem) {
    if(wolfSSL_UseKeyShare(backend->handle, pqkem) != WOLFSSL_SUCCESS) {
      failf(data, "unable to use PQ KEM");
    }
  }
#endif

#ifdef HAVE_ALPN
  if(connssl->alpn) {
    struct alpn_proto_buf proto;
    CURLcode result;

    result = Curl_alpn_to_proto_str(&proto, connssl->alpn);
    if(result ||
       wolfSSL_UseALPN(backend->handle,
                       (char *)proto.data, (unsigned int)proto.len,
                       WOLFSSL_ALPN_CONTINUE_ON_MISMATCH) != WOLFSSL_SUCCESS) {
      failf(data, "SSL: failed setting ALPN protocols");
      return CURLE_SSL_CONNECT_ERROR;
    }
    infof(data, VTLS_INFOF_ALPN_OFFER_1STR, proto.data);
  }
#endif /* HAVE_ALPN */

#ifdef OPENSSL_EXTRA
  if(Curl_tls_keylog_enabled()) {
    /* Ensure the Client Random is preserved. */
    wolfSSL_KeepArrays(backend->handle);
#if defined(HAVE_SECRET_CALLBACK) && defined(WOLFSSL_TLS13)
    wolfSSL_set_tls13_secret_cb(backend->handle,
                                wolfssl_tls13_secret_callback, NULL);
#endif
  }
#endif /* OPENSSL_EXTRA */

#ifdef HAVE_SECURE_RENEGOTIATION
  if(wolfSSL_UseSecureRenegotiation(backend->handle) != SSL_SUCCESS) {
    failf(data, "SSL: failed setting secure renegotiation");
    return CURLE_SSL_CONNECT_ERROR;
  }
#endif /* HAVE_SECURE_RENEGOTIATION */

  /* Check if there is a cached ID we can/should use here! */
  if(ssl_config->primary.cache_session) {
    /* Set session from cache if there is one */
    (void)Curl_wssl_setup_session(cf, data, backend, connssl->peer.scache_key);
    /* Register to get notified when a new session is received */
    wolfSSL_set_app_data(backend->handle, cf);
    wolfSSL_CTX_sess_set_new_cb(backend->ctx, wssl_vtls_new_session_cb);
  }

#ifdef USE_ECH_WOLFSSL
  if(ECH_ENABLED(data)) {
    int trying_ech_now = 0;

    if(data->set.str[STRING_ECH_PUBLIC]) {
      infof(data, "ECH: outername not (yet) supported with wolfSSL");
      return CURLE_SSL_CONNECT_ERROR;
    }
    if(data->set.tls_ech == CURLECH_GREASE) {
      infof(data, "ECH: GREASE'd ECH not yet supported for wolfSSL");
      return CURLE_SSL_CONNECT_ERROR;
    }
    if(data->set.tls_ech & CURLECH_CLA_CFG
       && data->set.str[STRING_ECH_CONFIG]) {
      char *b64val = data->set.str[STRING_ECH_CONFIG];
      word32 b64len = 0;

      b64len = (word32) strlen(b64val);
      if(b64len
         && wolfSSL_SetEchConfigsBase64(backend->handle, b64val, b64len)
              != WOLFSSL_SUCCESS) {
        if(data->set.tls_ech & CURLECH_HARD)
          return CURLE_SSL_CONNECT_ERROR;
      }
      else {
       trying_ech_now = 1;
       infof(data, "ECH: ECHConfig from command line");
      }
    }
    else {
      struct Curl_dns_entry *dns = NULL;

      dns = Curl_fetch_addr(data, connssl->peer.hostname, connssl->peer.port);
      if(!dns) {
        infof(data, "ECH: requested but no DNS info available");
        if(data->set.tls_ech & CURLECH_HARD)
          return CURLE_SSL_CONNECT_ERROR;
      }
      else {
        struct Curl_https_rrinfo *rinfo = NULL;

        rinfo = dns->hinfo;
        if(rinfo && rinfo->echconfiglist) {
          unsigned char *ecl = rinfo->echconfiglist;
          size_t elen = rinfo->echconfiglist_len;

          infof(data, "ECH: ECHConfig from DoH HTTPS RR");
          if(wolfSSL_SetEchConfigs(backend->handle, ecl, (word32) elen) !=
                WOLFSSL_SUCCESS) {
            infof(data, "ECH: wolfSSL_SetEchConfigs failed");
            if(data->set.tls_ech & CURLECH_HARD)
              return CURLE_SSL_CONNECT_ERROR;
          }
          else {
            trying_ech_now = 1;
            infof(data, "ECH: imported ECHConfigList of length %ld", elen);
          }
        }
        else {
          infof(data, "ECH: requested but no ECHConfig available");
          if(data->set.tls_ech & CURLECH_HARD)
            return CURLE_SSL_CONNECT_ERROR;
        }
        Curl_resolv_unlink(data, &dns);
      }
    }

    if(trying_ech_now && wolfSSL_set_min_proto_version(backend->handle,
                                                       TLS1_3_VERSION) != 1) {
      infof(data, "ECH: cannot force TLSv1.3 [ERROR]");
      return CURLE_SSL_CONNECT_ERROR;
    }

  }
#endif  /* USE_ECH_WOLFSSL */

#ifdef USE_BIO_CHAIN
  {
    WOLFSSL_BIO *bio;

    bio = wolfSSL_BIO_new(wolfssl_bio_cf_method);
    if(!bio)
      return CURLE_OUT_OF_MEMORY;

    wolfSSL_BIO_set_data(bio, cf);
    wolfSSL_set_bio(backend->handle, bio, bio);
  }
#else /* USE_BIO_CHAIN */
  /* pass the raw socket into the SSL layer */
  if(!wolfSSL_set_fd(backend->handle,
                     (int)Curl_conn_cf_get_socket(cf, data))) {
    failf(data, "SSL: SSL_set_fd failed");
    return CURLE_SSL_CONNECT_ERROR;
  }
#endif /* !USE_BIO_CHAIN */

  connssl->connecting_state = ssl_connect_2;
  return CURLE_OK;
}


static char *wolfssl_strerror(unsigned long error, char *buf,
                              unsigned long size)
{
  DEBUGASSERT(size > 40);
  *buf = '\0';

  wolfSSL_ERR_error_string_n(error, buf, size);

  if(!*buf) {
    const char *msg = error ? "Unknown error" : "No error";
    /* the string fits because the assert above assures this */
    strcpy(buf, msg);
  }

  return buf;
}


static CURLcode
wolfssl_connect_step2(struct Curl_cfilter *cf, struct Curl_easy *data)
{
  int ret = -1;
  struct ssl_connect_data *connssl = cf->ctx;
  struct wolfssl_ctx *backend =
    (struct wolfssl_ctx *)connssl->backend;
  struct ssl_primary_config *conn_config = Curl_ssl_cf_get_primary_config(cf);
#ifndef CURL_DISABLE_PROXY
  const char * const pinnedpubkey = Curl_ssl_cf_is_proxy(cf) ?
    data->set.str[STRING_SSL_PINNEDPUBLICKEY_PROXY] :
    data->set.str[STRING_SSL_PINNEDPUBLICKEY];
#else
  const char * const pinnedpubkey = data->set.str[STRING_SSL_PINNEDPUBLICKEY];
#endif

  DEBUGASSERT(backend);

  wolfSSL_ERR_clear_error();

  /* Enable RFC2818 checks */
  if(conn_config->verifyhost) {
    char *snihost = connssl->peer.sni ?
      connssl->peer.sni : connssl->peer.hostname;
    if(wolfSSL_check_domain_name(backend->handle, snihost) == WOLFSSL_FAILURE)
      return CURLE_SSL_CONNECT_ERROR;
  }

  if(!backend->x509_store_setup) {
    /* After having send off the ClientHello, we prepare the x509
     * store to verify the coming certificate from the server */
    CURLcode result;
    result = Curl_wssl_setup_x509_store(cf, data, backend);
    if(result)
      return result;
  }

  connssl->io_need = CURL_SSL_IO_NEED_NONE;
  ret = wolfSSL_connect(backend->handle);

#ifdef OPENSSL_EXTRA
  if(Curl_tls_keylog_enabled()) {
    /* If key logging is enabled, wait for the handshake to complete and then
     * proceed with logging secrets (for TLS 1.2 or older).
     *
     * During the handshake (ret==-1), wolfSSL_want_read() is true as it waits
     * for the server response. At that point the master secret is not yet
     * available, so we must not try to read it.
     * To log the secret on completion with a handshake failure, detect
     * completion via the observation that there is nothing to read or write.
     * Note that OpenSSL SSL_want_read() is always true here. If wolfSSL ever
     * changes, the worst case is that no key is logged on error.
     */
    if(ret == WOLFSSL_SUCCESS ||
       (!wolfSSL_want_read(backend->handle) &&
        !wolfSSL_want_write(backend->handle))) {
      wolfssl_log_tls12_secret(backend->handle);
      /* Client Random and master secrets are no longer needed, erase these.
       * Ignored while the handshake is still in progress. */
      wolfSSL_FreeArrays(backend->handle);
    }
  }
#endif  /* OPENSSL_EXTRA */

  if(ret != 1) {
    int detail = wolfSSL_get_error(backend->handle, ret);

    if(WOLFSSL_ERROR_WANT_READ == detail) {
      connssl->io_need = CURL_SSL_IO_NEED_RECV;
      return CURLE_OK;
    }
    else if(WOLFSSL_ERROR_WANT_WRITE == detail) {
      connssl->io_need = CURL_SSL_IO_NEED_SEND;
      return CURLE_OK;
    }
    /* There is no easy way to override only the CN matching.
     * This will enable the override of both mismatching SubjectAltNames
     * as also mismatching CN fields */
    else if(DOMAIN_NAME_MISMATCH == detail) {
#if 1
      failf(data, " subject alt name(s) or common name do not match \"%s\"",
            connssl->peer.dispname);
      return CURLE_PEER_FAILED_VERIFICATION;
#else
      /* When the wolfssl_check_domain_name() is used and you desire to
       * continue on a DOMAIN_NAME_MISMATCH, i.e. 'ssl_config.verifyhost
       * == 0', CyaSSL version 2.4.0 will fail with an INCOMPLETE_DATA
       * error. The only way to do this is currently to switch the
       * Wolfssl_check_domain_name() in and out based on the
       * 'ssl_config.verifyhost' value. */
      if(conn_config->verifyhost) {
        failf(data,
              " subject alt name(s) or common name do not match \"%s\"\n",
              connssl->dispname);
        return CURLE_PEER_FAILED_VERIFICATION;
      }
      else {
        infof(data,
              " subject alt name(s) and/or common name do not match \"%s\"",
              connssl->dispname);
        return CURLE_OK;
      }
#endif
    }
    else if(ASN_NO_SIGNER_E == detail) {
      if(conn_config->verifypeer) {
        failf(data, " CA signer not available for verification");
        return CURLE_SSL_CACERT_BADFILE;
      }
      else {
        /* Just continue with a warning if no strict certificate
           verification is required. */
        infof(data, "CA signer not available for verification, "
                    "continuing anyway");
      }
    }
    else if(ASN_AFTER_DATE_E == detail) {
      failf(data, "server verification failed: certificate has expired.");
      return CURLE_PEER_FAILED_VERIFICATION;
    }
    else if(ASN_BEFORE_DATE_E == detail) {
      failf(data, "server verification failed: certificate not valid yet.");
      return CURLE_PEER_FAILED_VERIFICATION;
    }
#ifdef USE_ECH_WOLFSSL
    else if(-1 == detail) {
      /* try access a retry_config ECHConfigList for tracing */
      byte echConfigs[1000];
      word32 echConfigsLen = 1000;
      int rv = 0;

      /* this currently does not produce the retry_configs */
      rv = wolfSSL_GetEchConfigs(backend->handle, echConfigs,
                                 &echConfigsLen);
      if(rv != WOLFSSL_SUCCESS) {
        infof(data, "Failed to get ECHConfigs");
      }
      else {
        char *b64str = NULL;
        size_t blen = 0;

        rv = Curl_base64_encode((const char *)echConfigs, echConfigsLen,
                                &b64str, &blen);
        if(!rv && b64str)
          infof(data, "ECH: (not yet) retry_configs %s", b64str);
        free(b64str);
      }
    }
#endif
    else if(backend->io_result == CURLE_AGAIN) {
      return CURLE_OK;
    }
    else {
      char error_buffer[256];
      failf(data, "SSL_connect failed with error %d: %s", detail,
            wolfssl_strerror((unsigned long)detail, error_buffer,
                             sizeof(error_buffer)));
      return CURLE_SSL_CONNECT_ERROR;
    }
  }

  if(pinnedpubkey) {
#ifdef KEEP_PEER_CERT
    WOLFSSL_X509 *x509;
    const char *x509_der;
    int x509_der_len;
    struct Curl_X509certificate x509_parsed;
    struct Curl_asn1Element *pubkey;
    CURLcode result;

    x509 = wolfSSL_get_peer_certificate(backend->handle);
    if(!x509) {
      failf(data, "SSL: failed retrieving server certificate");
      return CURLE_SSL_PINNEDPUBKEYNOTMATCH;
    }

    x509_der = (const char *)wolfSSL_X509_get_der(x509, &x509_der_len);
    if(!x509_der) {
      failf(data, "SSL: failed retrieving ASN.1 server certificate");
      return CURLE_SSL_PINNEDPUBKEYNOTMATCH;
    }

    memset(&x509_parsed, 0, sizeof(x509_parsed));
    if(Curl_parseX509(&x509_parsed, x509_der, x509_der + x509_der_len))
      return CURLE_SSL_PINNEDPUBKEYNOTMATCH;

    pubkey = &x509_parsed.subjectPublicKeyInfo;
    if(!pubkey->header || pubkey->end <= pubkey->header) {
      failf(data, "SSL: failed retrieving public key from server certificate");
      return CURLE_SSL_PINNEDPUBKEYNOTMATCH;
    }

    result = Curl_pin_peer_pubkey(data,
                                  pinnedpubkey,
                                  (const unsigned char *)pubkey->header,
                                  (size_t)(pubkey->end - pubkey->header));
    wolfSSL_FreeX509(x509);
    if(result) {
      failf(data, "SSL: public key does not match pinned public key");
      return result;
    }
#else
    failf(data, "Library lacks pinning support built-in");
    return CURLE_NOT_BUILT_IN;
#endif
  }

#ifdef HAVE_ALPN
  if(connssl->alpn) {
    int rc;
    char *protocol = NULL;
    unsigned short protocol_len = 0;

    rc = wolfSSL_ALPN_GetProtocol(backend->handle, &protocol, &protocol_len);

    if(rc == WOLFSSL_SUCCESS) {
      Curl_alpn_set_negotiated(cf, data, connssl,
                               (const unsigned char *)protocol, protocol_len);
    }
    else if(rc == WOLFSSL_ALPN_NOT_FOUND)
      Curl_alpn_set_negotiated(cf, data, connssl, NULL, 0);
    else {
      failf(data, "ALPN, failure getting protocol, error %d", rc);
      return CURLE_SSL_CONNECT_ERROR;
    }
  }
#endif /* HAVE_ALPN */

  connssl->connecting_state = ssl_connect_3;
#if (LIBWOLFSSL_VERSION_HEX >= 0x03009010)
  infof(data, "SSL connection using %s / %s",
        wolfSSL_get_version(backend->handle),
        wolfSSL_get_cipher_name(backend->handle));
#else
  infof(data, "SSL connected");
#endif

  return CURLE_OK;
}

static ssize_t wolfssl_send(struct Curl_cfilter *cf,
                            struct Curl_easy *data,
                            const void *mem,
                            size_t len,
                            CURLcode *curlcode)
{
  struct ssl_connect_data *connssl = cf->ctx;
  struct wolfssl_ctx *backend =
    (struct wolfssl_ctx *)connssl->backend;
  int memlen = (len > (size_t)INT_MAX) ? INT_MAX : (int)len;
  int rc;

  DEBUGASSERT(backend);

  wolfSSL_ERR_clear_error();

  rc = wolfSSL_write(backend->handle, mem, memlen);
  if(rc <= 0) {
    int err = wolfSSL_get_error(backend->handle, rc);

    switch(err) {
    case WOLFSSL_ERROR_WANT_READ:
    case WOLFSSL_ERROR_WANT_WRITE:
      /* there is data pending, re-invoke SSL_write() */
      CURL_TRC_CF(data, cf, "wolfssl_send(len=%zu) -> AGAIN", len);
      *curlcode = CURLE_AGAIN;
      return -1;
    default:
      if(backend->io_result == CURLE_AGAIN) {
        CURL_TRC_CF(data, cf, "wolfssl_send(len=%zu) -> AGAIN", len);
        *curlcode = CURLE_AGAIN;
        return -1;
      }
      CURL_TRC_CF(data, cf, "wolfssl_send(len=%zu) -> %d, %d", len, rc, err);
      {
        char error_buffer[256];
        failf(data, "SSL write: %s, errno %d",
              wolfssl_strerror((unsigned long)err, error_buffer,
                               sizeof(error_buffer)),
              SOCKERRNO);
      }
      *curlcode = CURLE_SEND_ERROR;
      return -1;
    }
  }
  CURL_TRC_CF(data, cf, "wolfssl_send(len=%zu) -> %d", len, rc);
  return rc;
}

static CURLcode wolfssl_shutdown(struct Curl_cfilter *cf,
                                 struct Curl_easy *data,
                                 bool send_shutdown, bool *done)
{
  struct ssl_connect_data *connssl = cf->ctx;
  struct wolfssl_ctx *wctx = (struct wolfssl_ctx *)connssl->backend;
  CURLcode result = CURLE_OK;
  char buf[1024];
  char error_buffer[256];
  int nread = -1, err;
  size_t i;
  int detail;

  DEBUGASSERT(wctx);
  if(!wctx->handle || cf->shutdown) {
    *done = TRUE;
    goto out;
  }

  wctx->shutting_down = TRUE;
  connssl->io_need = CURL_SSL_IO_NEED_NONE;
  *done = FALSE;
  if(!(wolfSSL_get_shutdown(wctx->handle) & WOLFSSL_SENT_SHUTDOWN)) {
    /* We have not started the shutdown from our side yet. Check
     * if the server already sent us one. */
    wolfSSL_ERR_clear_error();
    nread = wolfSSL_read(wctx->handle, buf, (int)sizeof(buf));
    err = wolfSSL_get_error(wctx->handle, nread);
    CURL_TRC_CF(data, cf, "wolfSSL_read, nread=%d, err=%d", nread, err);
    if(!nread && err == WOLFSSL_ERROR_ZERO_RETURN) {
      bool input_pending;
      /* Yes, it did. */
      if(!send_shutdown) {
        CURL_TRC_CF(data, cf, "SSL shutdown received, not sending");
        *done = TRUE;
        goto out;
      }
      else if(!cf->next->cft->is_alive(cf->next, data, &input_pending)) {
        /* Server closed the connection after its closy notify. It
         * seems not interested to see our close notify, so do not
         * send it. We are done. */
        CURL_TRC_CF(data, cf, "peer closed connection");
        connssl->peer_closed = TRUE;
        *done = TRUE;
        goto out;
      }
    }
  }

  /* SSL should now have started the shutdown from our side. Since it
   * was not complete, we are lacking the close notify from the server. */
  if(send_shutdown) {
    wolfSSL_ERR_clear_error();
    if(wolfSSL_shutdown(wctx->handle) == 1) {
      CURL_TRC_CF(data, cf, "SSL shutdown finished");
      *done = TRUE;
      goto out;
    }
    if(WOLFSSL_ERROR_WANT_WRITE == wolfSSL_get_error(wctx->handle, nread)) {
      CURL_TRC_CF(data, cf, "SSL shutdown still wants to send");
      connssl->io_need = CURL_SSL_IO_NEED_SEND;
      goto out;
    }
    /* Having sent the close notify, we use wolfSSL_read() to get the
     * missing close notify from the server. */
  }

  for(i = 0; i < 10; ++i) {
    wolfSSL_ERR_clear_error();
    nread = wolfSSL_read(wctx->handle, buf, (int)sizeof(buf));
    if(nread <= 0)
      break;
  }
  err = wolfSSL_get_error(wctx->handle, nread);
  switch(err) {
  case WOLFSSL_ERROR_ZERO_RETURN: /* no more data */
    CURL_TRC_CF(data, cf, "SSL shutdown received");
    *done = TRUE;
    break;
  case WOLFSSL_ERROR_NONE: /* just did not get anything */
  case WOLFSSL_ERROR_WANT_READ:
    /* SSL has send its notify and now wants to read the reply
     * from the server. We are not really interested in that. */
    CURL_TRC_CF(data, cf, "SSL shutdown sent, want receive");
    connssl->io_need = CURL_SSL_IO_NEED_RECV;
    break;
  case WOLFSSL_ERROR_WANT_WRITE:
    CURL_TRC_CF(data, cf, "SSL shutdown send blocked");
    connssl->io_need = CURL_SSL_IO_NEED_SEND;
    break;
  default:
    detail = wolfSSL_get_error(wctx->handle, err);
    CURL_TRC_CF(data, cf, "SSL shutdown, error: '%s'(%d)",
                wolfssl_strerror((unsigned long)err, error_buffer,
                                 sizeof(error_buffer)),
                detail);
    result = CURLE_RECV_ERROR;
    break;
  }

out:
  cf->shutdown = (result || *done);
  return result;
}

static void wolfssl_close(struct Curl_cfilter *cf, struct Curl_easy *data)
{
  struct ssl_connect_data *connssl = cf->ctx;
  struct wolfssl_ctx *backend =
    (struct wolfssl_ctx *)connssl->backend;

  (void) data;

  DEBUGASSERT(backend);

  if(backend->handle) {
    wolfSSL_free(backend->handle);
    backend->handle = NULL;
  }
  if(backend->ctx) {
    wolfSSL_CTX_free(backend->ctx);
    backend->ctx = NULL;
  }
}

static ssize_t wolfssl_recv(struct Curl_cfilter *cf,
                            struct Curl_easy *data,
                            char *buf, size_t blen,
                            CURLcode *curlcode)
{
  struct ssl_connect_data *connssl = cf->ctx;
  struct wolfssl_ctx *backend =
    (struct wolfssl_ctx *)connssl->backend;
  int buffsize = (blen > (size_t)INT_MAX) ? INT_MAX : (int)blen;
  int nread;

  DEBUGASSERT(backend);

  wolfSSL_ERR_clear_error();
  *curlcode = CURLE_OK;

  nread = wolfSSL_read(backend->handle, buf, buffsize);

  if(nread <= 0) {
    int err = wolfSSL_get_error(backend->handle, nread);

    switch(err) {
    case WOLFSSL_ERROR_ZERO_RETURN: /* no more data */
      CURL_TRC_CF(data, cf, "wolfssl_recv(len=%zu) -> CLOSED", blen);
      *curlcode = CURLE_OK;
      return 0;
    case WOLFSSL_ERROR_NONE:
    case WOLFSSL_ERROR_WANT_READ:
    case WOLFSSL_ERROR_WANT_WRITE:
      if(!backend->io_result && connssl->peer_closed) {
        CURL_TRC_CF(data, cf, "wolfssl_recv(len=%zu) -> CLOSED", blen);
        *curlcode = CURLE_OK;
        return 0;
      }
      /* there is data pending, re-invoke wolfSSL_read() */
      CURL_TRC_CF(data, cf, "wolfssl_recv(len=%zu) -> AGAIN", blen);
      *curlcode = CURLE_AGAIN;
      return -1;
    default:
      if(backend->io_result == CURLE_AGAIN) {
        CURL_TRC_CF(data, cf, "wolfssl_recv(len=%zu) -> AGAIN", blen);
        *curlcode = CURLE_AGAIN;
        return -1;
      }
      else if(!backend->io_result && connssl->peer_closed) {
        CURL_TRC_CF(data, cf, "wolfssl_recv(len=%zu) -> CLOSED", blen);
        *curlcode = CURLE_OK;
        return 0;
      }
      else {
        char error_buffer[256];
        failf(data, "SSL read: %s, errno %d",
              wolfssl_strerror((unsigned long)err, error_buffer,
                               sizeof(error_buffer)),
              SOCKERRNO);
      }
      *curlcode = CURLE_RECV_ERROR;
      return -1;
    }
  }
  CURL_TRC_CF(data, cf, "wolfssl_recv(len=%zu) -> %d", blen, nread);
  return nread;
}


size_t Curl_wssl_version(char *buffer, size_t size)
{
#if LIBWOLFSSL_VERSION_HEX >= 0x03006000
  return msnprintf(buffer, size, "wolfSSL/%s", wolfSSL_lib_version());
#elif defined(WOLFSSL_VERSION)
  return msnprintf(buffer, size, "wolfSSL/%s", WOLFSSL_VERSION);
#endif
}


static int wolfssl_init(void)
{
  int ret;

#ifdef OPENSSL_EXTRA
  Curl_tls_keylog_open();
#endif
  ret = (wolfSSL_Init() == WOLFSSL_SUCCESS);
  wolfssl_bio_cf_init_methods();
  return ret;
}


static void wolfssl_cleanup(void)
{
  wolfssl_bio_cf_free_methods();
  wolfSSL_Cleanup();
#ifdef OPENSSL_EXTRA
  Curl_tls_keylog_close();
#endif
}


static bool wolfssl_data_pending(struct Curl_cfilter *cf,
                                 const struct Curl_easy *data)
{
  struct ssl_connect_data *ctx = cf->ctx;
  struct wolfssl_ctx *backend;

  (void)data;
  DEBUGASSERT(ctx && ctx->backend);

  backend = (struct wolfssl_ctx *)ctx->backend;
  if(backend->handle)   /* SSL is in use */
    return wolfSSL_pending(backend->handle);
  else
    return FALSE;
}

static CURLcode
wolfssl_connect_common(struct Curl_cfilter *cf,
                       struct Curl_easy *data,
                       bool nonblocking,
                       bool *done)
{
  CURLcode result;
  struct ssl_connect_data *connssl = cf->ctx;
  curl_socket_t sockfd = Curl_conn_cf_get_socket(cf, data);

  /* check if the connection has already been established */
  if(ssl_connection_complete == connssl->state) {
    *done = TRUE;
    return CURLE_OK;
  }

  if(ssl_connect_1 == connssl->connecting_state) {
    /* Find out how much more time we are allowed */
    const timediff_t timeout_ms = Curl_timeleft(data, NULL, TRUE);

    if(timeout_ms < 0) {
      /* no need to continue if time already is up */
      failf(data, "SSL connection timeout");
      return CURLE_OPERATION_TIMEDOUT;
    }

    result = wolfssl_connect_step1(cf, data);
    if(result)
      return result;
  }

  while(ssl_connect_2 == connssl->connecting_state) {

    /* check allowed time left */
    const timediff_t timeout_ms = Curl_timeleft(data, NULL, TRUE);

    if(timeout_ms < 0) {
      /* no need to continue if time already is up */
      failf(data, "SSL connection timeout");
      return CURLE_OPERATION_TIMEDOUT;
    }

    /* if ssl is expecting something, check if it is available. */
    if(connssl->io_need) {
      curl_socket_t writefd = (connssl->io_need & CURL_SSL_IO_NEED_SEND) ?
        sockfd : CURL_SOCKET_BAD;
      curl_socket_t readfd = (connssl->io_need & CURL_SSL_IO_NEED_RECV) ?
        sockfd : CURL_SOCKET_BAD;
      int what = Curl_socket_check(readfd, CURL_SOCKET_BAD, writefd,
                                   nonblocking ? 0 : timeout_ms);
      if(what < 0) {
        /* fatal error */
        failf(data, "select/poll on SSL socket, errno: %d", SOCKERRNO);
        return CURLE_SSL_CONNECT_ERROR;
      }
      else if(0 == what) {
        if(nonblocking) {
          *done = FALSE;
          return CURLE_OK;
        }
        else {
          /* timeout */
          failf(data, "SSL connection timeout");
          return CURLE_OPERATION_TIMEDOUT;
        }
      }
      /* socket is readable or writable */
    }

    /* Run transaction, and return to the caller if it failed or if
     * this connection is part of a multi handle and this loop would
     * execute again. This permits the owner of a multi handle to
     * abort a connection attempt before step2 has completed while
     * ensuring that a client using select() or epoll() will always
     * have a valid fdset to wait on.
     */
    result = wolfssl_connect_step2(cf, data);
    if(result || (nonblocking && (ssl_connect_2 == connssl->connecting_state)))
      return result;
  } /* repeat step2 until all transactions are done. */

  if(ssl_connect_3 == connssl->connecting_state) {
    /* In other backends, this is where we verify the certificate, but
     * wolfSSL already does that as part of the handshake. */
    connssl->connecting_state = ssl_connect_done;
  }

  if(ssl_connect_done == connssl->connecting_state) {
    connssl->state = ssl_connection_complete;
    *done = TRUE;
  }
  else
    *done = FALSE;

  /* Reset our connect state machine */
  connssl->connecting_state = ssl_connect_1;

  return CURLE_OK;
}


static CURLcode wolfssl_connect_nonblocking(struct Curl_cfilter *cf,
                                            struct Curl_easy *data,
                                            bool *done)
{
  return wolfssl_connect_common(cf, data, TRUE, done);
}


static CURLcode wolfssl_connect(struct Curl_cfilter *cf,
                                struct Curl_easy *data)
{
  CURLcode result;
  bool done = FALSE;

  result = wolfssl_connect_common(cf, data, FALSE, &done);
  if(result)
    return result;

  DEBUGASSERT(done);

  return CURLE_OK;
}

static CURLcode wolfssl_random(struct Curl_easy *data,
                               unsigned char *entropy, size_t length)
{
  WC_RNG rng;
  (void)data;
  if(wc_InitRng(&rng))
    return CURLE_FAILED_INIT;
  if(length > UINT_MAX)
    return CURLE_FAILED_INIT;
  if(wc_RNG_GenerateBlock(&rng, entropy, (unsigned)length))
    return CURLE_FAILED_INIT;
  if(wc_FreeRng(&rng))
    return CURLE_FAILED_INIT;
  return CURLE_OK;
}

static CURLcode wolfssl_sha256sum(const unsigned char *tmp, /* input */
                                  size_t tmplen,
                                  unsigned char *sha256sum /* output */,
                                  size_t unused)
{
  wc_Sha256 SHA256pw;
  (void)unused;
  if(wc_InitSha256(&SHA256pw))
    return CURLE_FAILED_INIT;
  wc_Sha256Update(&SHA256pw, tmp, (word32)tmplen);
  wc_Sha256Final(&SHA256pw, sha256sum);
  return CURLE_OK;
}

static void *wolfssl_get_internals(struct ssl_connect_data *connssl,
                                   CURLINFO info UNUSED_PARAM)
{
  struct wolfssl_ctx *backend =
    (struct wolfssl_ctx *)connssl->backend;
  (void)info;
  DEBUGASSERT(backend);
  return backend->handle;
}

const struct Curl_ssl Curl_ssl_wolfssl = {
  { CURLSSLBACKEND_WOLFSSL, "wolfssl" }, /* info */

#ifdef KEEP_PEER_CERT
  SSLSUPP_PINNEDPUBKEY |
#endif
#ifdef USE_BIO_CHAIN
  SSLSUPP_HTTPS_PROXY |
#endif
  SSLSUPP_CA_PATH |
  SSLSUPP_CAINFO_BLOB |
#ifdef USE_ECH_WOLFSSL
  SSLSUPP_ECH |
#endif
  SSLSUPP_SSL_CTX |
#ifdef WOLFSSL_TLS13
  SSLSUPP_TLS13_CIPHERSUITES |
#endif
  SSLSUPP_CA_CACHE |
  SSLSUPP_CIPHER_LIST,

  sizeof(struct wolfssl_ctx),

  wolfssl_init,                    /* init */
  wolfssl_cleanup,                 /* cleanup */
  Curl_wssl_version,               /* version */
  wolfssl_shutdown,                /* shutdown */
  wolfssl_data_pending,            /* data_pending */
  wolfssl_random,                  /* random */
  NULL,                            /* cert_status_request */
  wolfssl_connect,                 /* connect */
  wolfssl_connect_nonblocking,     /* connect_nonblocking */
  Curl_ssl_adjust_pollset,         /* adjust_pollset */
  wolfssl_get_internals,           /* get_internals */
  wolfssl_close,                   /* close_one */
  NULL,                            /* close_all */
  NULL,                            /* set_engine */
  NULL,                            /* set_engine_default */
  NULL,                            /* engines_list */
  NULL,                            /* false_start */
  wolfssl_sha256sum,               /* sha256sum */
  wolfssl_recv,                    /* recv decrypted data */
  wolfssl_send,                    /* send data to encrypt */
  NULL,                            /* get_channel_binding */
};

#endif
