//===-- Half-precision 10^x - 1 function ----------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

#include "src/math/exp10m1f16.h"
#include "expxf16.h"
#include "hdr/errno_macros.h"
#include "hdr/fenv_macros.h"
#include "src/__support/FPUtil/FEnvImpl.h"
#include "src/__support/FPUtil/FPBits.h"
#include "src/__support/FPUtil/PolyEval.h"
#include "src/__support/FPUtil/cast.h"
#include "src/__support/FPUtil/except_value_utils.h"
#include "src/__support/FPUtil/multiply_add.h"
#include "src/__support/FPUtil/rounding_mode.h"
#include "src/__support/common.h"
#include "src/__support/macros/config.h"
#include "src/__support/macros/optimization.h"
#include "src/__support/macros/properties/cpu_features.h"

namespace LIBC_NAMESPACE_DECL {

static constexpr fputil::ExceptValues<float16, 3> EXP10M1F16_EXCEPTS_LO = {{
    // (input, RZ output, RU offset, RD offset, RN offset)
    // x = 0x1.5c4p-4, exp10m1f16(x) = 0x1.bacp-3 (RZ)
    {0x2d71U, 0x32ebU, 1U, 0U, 0U},
    // x = -0x1.5ep-13, exp10m1f16(x) = -0x1.92cp-12 (RZ)
    {0x8978U, 0x8e4bU, 0U, 1U, 0U},
    // x = -0x1.e2p-10, exp10m1f16(x) = -0x1.14cp-8 (RZ)
    {0x9788U, 0x9c53U, 0U, 1U, 0U},
}};

#ifdef LIBC_TARGET_CPU_HAS_FMA
static constexpr size_t N_EXP10M1F16_EXCEPTS_HI = 3;
#else
static constexpr size_t N_EXP10M1F16_EXCEPTS_HI = 6;
#endif

static constexpr fputil::ExceptValues<float16, N_EXP10M1F16_EXCEPTS_HI>
    EXP10M1F16_EXCEPTS_HI = {{
        // (input, RZ output, RU offset, RD offset, RN offset)
        // x = 0x1.8f4p-2, exp10m1f16(x) = 0x1.744p+0 (RZ)
        {0x363dU, 0x3dd1U, 1U, 0U, 0U},
        // x = 0x1.95cp-2, exp10m1f16(x) = 0x1.7d8p+0 (RZ)
        {0x3657U, 0x3df6U, 1U, 0U, 0U},
        // x = 0x1.d04p-2, exp10m1f16(x) = 0x1.d7p+0 (RZ)
        {0x3741U, 0x3f5cU, 1U, 0U, 1U},
#ifndef LIBC_TARGET_CPU_HAS_FMA
        // x = 0x1.0cp+1, exp10m1f16(x) = 0x1.ec4p+6 (RZ)
        {0x4030U, 0x57b1U, 1U, 0U, 1U},
        // x = 0x1.1b8p+1, exp10m1f16(x) = 0x1.45cp+7 (RZ)
        {0x406eU, 0x5917U, 1U, 0U, 1U},
        // x = 0x1.2f4p+2, exp10m1f16(x) = 0x1.ab8p+15 (RZ)
        {0x44bdU, 0x7aaeU, 1U, 0U, 1U},
#endif
    }};

LLVM_LIBC_FUNCTION(float16, exp10m1f16, (float16 x)) {
  using FPBits = fputil::FPBits<float16>;
  FPBits x_bits(x);

  uint16_t x_u = x_bits.uintval();
  uint16_t x_abs = x_u & 0x7fffU;

  // When |x| <= 2^(-3), or |x| >= 11 * log10(2), or x is NaN.
  if (LIBC_UNLIKELY(x_abs <= 0x3000U || x_abs >= 0x429fU)) {
    // exp10m1(NaN) = NaN
    if (x_bits.is_nan()) {
      if (x_bits.is_signaling_nan()) {
        fputil::raise_except_if_required(FE_INVALID);
        return FPBits::quiet_nan().get_val();
      }

      return x;
    }

    // When x >= 16 * log10(2).
    if (x_u >= 0x44d1U && x_bits.is_pos()) {
      // exp10m1(+inf) = +inf
      if (x_bits.is_inf())
        return FPBits::inf().get_val();

      switch (fputil::quick_get_round()) {
      case FE_TONEAREST:
      case FE_UPWARD:
        fputil::set_errno_if_required(ERANGE);
        fputil::raise_except_if_required(FE_OVERFLOW | FE_INEXACT);
        return FPBits::inf().get_val();
      default:
        return FPBits::max_normal().get_val();
      }
    }

    // When x < -11 * log10(2).
    if (x_u > 0xc29fU) {
      // exp10m1(-inf) = -1
      if (x_bits.is_inf())
        return FPBits::one(Sign::NEG).get_val();

      // When x >= -0x1.ce4p+1, round(10^x - 1, HP, RN) = -0x1.ffcp-1.
      if (x_u <= 0xc339U) {
        return fputil::round_result_slightly_down(
            fputil::cast<float16>(-0x1.ffcp-1));
      }

      // When x < -0x1.ce4p+1, round(10^x - 1, HP, RN) = -1.
      switch (fputil::quick_get_round()) {
      case FE_TONEAREST:
      case FE_DOWNWARD:
        return FPBits::one(Sign::NEG).get_val();
      default:
        return fputil::cast<float16>(-0x1.ffcp-1);
      }
    }

    // When |x| <= 2^(-3).
    if (x_abs <= 0x3000U) {
      if (LIBC_UNLIKELY(x_abs == 0))
        return x;

      if (auto r = EXP10M1F16_EXCEPTS_LO.lookup(x_u);
          LIBC_UNLIKELY(r.has_value()))
        return r.value();

      float xf = x;
      // Degree-5 minimax polynomial generated by Sollya with the following
      // commands:
      //   > display = hexadecimal;
      //   > P = fpminimax((10^x - 1)/x, 4, [|SG...|], [-2^-3, 2^-3]);
      //   > x * P;
      return fputil::cast<float16>(
          xf * fputil::polyeval(xf, 0x1.26bb1cp+1f, 0x1.5351c8p+1f,
                                0x1.04704p+1f, 0x1.2ce084p+0f, 0x1.14a6bep-1f));
    }
  }

  // When x is 1, 2, or 3. These are hard-to-round cases with exact results.
  // 10^4 - 1 = 9'999 is not exactly representable as a float16, but luckily the
  // polynomial approximation gives the correct result for x = 4 in all
  // rounding modes.
  if (LIBC_UNLIKELY((x_u & ~(0x3c00U | 0x4000U | 0x4200U | 0x4400U)) == 0)) {
    switch (x_u) {
    case 0x3c00U: // x = 1.0f16
      return fputil::cast<float16>(9.0);
    case 0x4000U: // x = 2.0f16
      return fputil::cast<float16>(99.0);
    case 0x4200U: // x = 3.0f16
      return fputil::cast<float16>(999.0);
    }
  }

  if (auto r = EXP10M1F16_EXCEPTS_HI.lookup(x_u); LIBC_UNLIKELY(r.has_value()))
    return r.value();

  // exp10(x) = exp2((hi + mid) * log2(10)) * exp10(lo)
  auto [exp2_hi_mid, exp10_lo] = exp10_range_reduction(x);
  // exp10m1(x) = exp2((hi + mid) * log2(lo)) * exp10(lo) - 1
  return fputil::cast<float16>(
      fputil::multiply_add(exp2_hi_mid, exp10_lo, -1.0f));
}

} // namespace LIBC_NAMESPACE_DECL
