// Based on the Chromium source. The Chromium source code is governed by a
// BSD-style license that can be found in the LICENSE file.
// https://source.chromium.org/chromium/chromium/src/+/main:ui/events/keycodes/dom/dom_code_data.inc

const std = @import("std");
const builtin = @import("builtin");
const Key = @import("key.zig").Key;

/// The full list of entries for the current platform.
pub const entries: []const Entry = entries: {
    const native_idx = switch (builtin.os.tag) {
        .ios, .macos => 4, // mac
        .windows => 3, // win
        .freebsd, .linux => 2, // xkb
        else => @compileError("unsupported platform"),
    };

    var result: [raw_entries.len]Entry = undefined;
    for (raw_entries, 0..) |raw, i| {
        @setEvalBranchQuota(10000);
        result[i] = .{
            .key = code_to_key.get(raw[5]) orelse .unidentified,
            .usb = raw[0],
            .code = raw[5],
            .native = raw[native_idx],
        };
    }

    const final = result;
    break :entries &final;
};

/// Entry contains the USB code, native keycode, and W3C dom code for
/// the current platform.
pub const Entry = struct {
    key: Key, // input key enum
    usb: u32, // USB HID usage code
    native: u32, // Native keycode
    code: []const u8, // W3C DOM code, static memory
};

/// A map from code to key. This isn't meant to be used at runtime
/// (though it could), so it isn't exported. It it used to build the
/// key value for Entry.
const code_to_key = code_to_key: {
    @setEvalBranchQuota(5000);
    break :code_to_key std.StaticStringMap(Key).initComptime(.{
        .{ "KeyA", .key_a },
        .{ "KeyB", .key_b },
        .{ "KeyC", .key_c },
        .{ "KeyD", .key_d },
        .{ "KeyE", .key_e },
        .{ "KeyF", .key_f },
        .{ "KeyG", .key_g },
        .{ "KeyH", .key_h },
        .{ "KeyI", .key_i },
        .{ "KeyJ", .key_j },
        .{ "KeyK", .key_k },
        .{ "KeyL", .key_l },
        .{ "KeyM", .key_m },
        .{ "KeyN", .key_n },
        .{ "KeyO", .key_o },
        .{ "KeyP", .key_p },
        .{ "KeyQ", .key_q },
        .{ "KeyR", .key_r },
        .{ "KeyS", .key_s },
        .{ "KeyT", .key_t },
        .{ "KeyU", .key_u },
        .{ "KeyV", .key_v },
        .{ "KeyW", .key_w },
        .{ "KeyX", .key_x },
        .{ "KeyY", .key_y },
        .{ "KeyZ", .key_z },
        .{ "Digit1", .digit_1 },
        .{ "Digit2", .digit_2 },
        .{ "Digit3", .digit_3 },
        .{ "Digit4", .digit_4 },
        .{ "Digit5", .digit_5 },
        .{ "Digit6", .digit_6 },
        .{ "Digit7", .digit_7 },
        .{ "Digit8", .digit_8 },
        .{ "Digit9", .digit_9 },
        .{ "Digit0", .digit_0 },
        .{ "Enter", .enter },
        .{ "Escape", .escape },
        .{ "Backspace", .backspace },
        .{ "Tab", .tab },
        .{ "Space", .space },
        .{ "Minus", .minus },
        .{ "Equal", .equal },
        .{ "BracketLeft", .bracket_left },
        .{ "BracketRight", .bracket_right },
        .{ "Backslash", .backslash },
        .{ "Semicolon", .semicolon },
        .{ "Quote", .quote },
        .{ "Backquote", .backquote },
        .{ "Comma", .comma },
        .{ "Period", .period },
        .{ "Slash", .slash },
        .{ "CapsLock", .caps_lock },
        .{ "F1", .f1 },
        .{ "F2", .f2 },
        .{ "F3", .f3 },
        .{ "F4", .f4 },
        .{ "F5", .f5 },
        .{ "F6", .f6 },
        .{ "F7", .f7 },
        .{ "F8", .f8 },
        .{ "F9", .f9 },
        .{ "F10", .f10 },
        .{ "F11", .f11 },
        .{ "F12", .f12 },
        .{ "F13", .f13 },
        .{ "F14", .f14 },
        .{ "F15", .f15 },
        .{ "F16", .f16 },
        .{ "F17", .f17 },
        .{ "F18", .f18 },
        .{ "F19", .f19 },
        .{ "F20", .f20 },
        .{ "F21", .f21 },
        .{ "F22", .f22 },
        .{ "F23", .f23 },
        .{ "F24", .f24 },
        .{ "PrintScreen", .print_screen },
        .{ "ScrollLock", .scroll_lock },
        .{ "Pause", .pause },
        .{ "Insert", .insert },
        .{ "Home", .home },
        .{ "PageUp", .page_up },
        .{ "Delete", .delete },
        .{ "End", .end },
        .{ "Copy", .copy },
        .{ "Cut", .cut },
        .{ "Paste", .paste },
        .{ "PageDown", .page_down },
        .{ "ArrowRight", .arrow_right },
        .{ "ArrowLeft", .arrow_left },
        .{ "ArrowDown", .arrow_down },
        .{ "ArrowUp", .arrow_up },
        .{ "NumLock", .num_lock },
        .{ "NumpadDivide", .numpad_divide },
        .{ "NumpadMultiply", .numpad_multiply },
        .{ "NumpadSubtract", .numpad_subtract },
        .{ "NumpadAdd", .numpad_add },
        .{ "NumpadEnter", .numpad_enter },
        .{ "Numpad1", .numpad_1 },
        .{ "Numpad2", .numpad_2 },
        .{ "Numpad3", .numpad_3 },
        .{ "Numpad4", .numpad_4 },
        .{ "Numpad5", .numpad_5 },
        .{ "Numpad6", .numpad_6 },
        .{ "Numpad7", .numpad_7 },
        .{ "Numpad8", .numpad_8 },
        .{ "Numpad9", .numpad_9 },
        .{ "Numpad0", .numpad_0 },
        .{ "NumpadDecimal", .numpad_decimal },
        .{ "NumpadEqual", .numpad_equal },
        .{ "ContextMenu", .context_menu },
        .{ "ControlLeft", .control_left },
        .{ "ShiftLeft", .shift_left },
        .{ "AltLeft", .alt_left },
        .{ "MetaLeft", .meta_left },
        .{ "ControlRight", .control_right },
        .{ "ShiftRight", .shift_right },
        .{ "AltRight", .alt_right },
        .{ "MetaRight", .meta_right },
    });
};

/// The codes for the table from the Chromium data set. These are ALL the
/// codes, not just the ones that are supported by the current platform.
/// These are `pub` but you shouldn't use this because it uses way more
/// memory than is necessary.
///
/// The format is: usb, evdev, xkb, win, mac, code
pub const RawEntry = struct { u32, u32, u32, u32, u32, []const u8 };

/// All of the full entries. This is marked pub but it should NOT be referenced
/// directly because it contains too much data for normal usage. Use `entries`
/// instead which contains just the relevant data for the target platform.
pub const raw_entries: []const RawEntry = &.{
    //   USB     evdev    XKB     Win     Mac   Code
    .{ 0x000000, 0x0000, 0x0000, 0x0000, 0xffff, "" },

    // =========================================
    // Non-USB codes
    // =========================================

    //   USB     evdev    XKB     Win     Mac   Code
    .{ 0x000010, 0x0000, 0x0000, 0x0000, 0xffff, "Hyper" },
    .{ 0x000011, 0x0000, 0x0000, 0x0000, 0xffff, "Super" },
    .{ 0x000012, 0x0000, 0x0000, 0x0000, 0xffff, "Fn" },
    .{ 0x000013, 0x0000, 0x0000, 0x0000, 0xffff, "FnLock" },
    .{ 0x000014, 0x0000, 0x0000, 0x0000, 0xffff, "Suspend" },
    .{ 0x000015, 0x0000, 0x0000, 0x0000, 0xffff, "Resume" },
    .{ 0x000016, 0x0000, 0x0000, 0x0000, 0xffff, "Turbo" },

    // =========================================
    // USB Usage Page 0x01: Generic Desktop Page
    // =========================================

    // Sleep could be encoded as USB#0c0032, but there's no corresponding WakeUp
    // in the 0x0c USB page.
    //   USB     evdev    XKB     Win     Mac
    .{ 0x010082, 0x008e, 0x0096, 0xe05f, 0xffff, "Sleep" },
    .{ 0x010083, 0x008f, 0x0097, 0xe063, 0xffff, "WakeUp" },
    .{ 0x0100a9, 0x00f8, 0x0100, 0x0000, 0xffff, "" },
    .{ 0x0100b5, 0x00e3, 0x00eb, 0x0000, 0xffff, "DisplayToggleIntExt" },

    // =========================================
    // USB Usage Page 0x07: Keyboard/Keypad Page
    // =========================================

    // TODO(garykac):
    // XKB#005c ISO Level3 Shift (AltGr)
    // XKB#005e <>||
    // XKB#006d Linefeed
    // XKB#008a SunProps cf. USB#0700a3 CrSel/Props
    // XKB#008e SunOpen
    // Mac#003f kVK_Function
    // Mac#000a kVK_ISO_Section (ISO keyboards only)
    // Mac#0066 kVK_JIS_Eisu (USB#07008a Henkan?)

    //   USB     evdev    XKB     Win     Mac
    .{ 0x070000, 0x0000, 0x0000, 0x0000, 0xffff, "" },
    .{ 0x070001, 0x0000, 0x0000, 0x00ff, 0xffff, "" },
    .{ 0x070002, 0x0000, 0x0000, 0x00fc, 0xffff, "" },
    .{ 0x070003, 0x0000, 0x0000, 0x0000, 0xffff, "" },
    .{ 0x070004, 0x001e, 0x0026, 0x001e, 0x0000, "KeyA" },
    .{ 0x070005, 0x0030, 0x0038, 0x0030, 0x000b, "KeyB" },
    .{ 0x070006, 0x002e, 0x0036, 0x002e, 0x0008, "KeyC" },
    .{ 0x070007, 0x0020, 0x0028, 0x0020, 0x0002, "KeyD" },

    .{ 0x070008, 0x0012, 0x001a, 0x0012, 0x000e, "KeyE" },
    .{ 0x070009, 0x0021, 0x0029, 0x0021, 0x0003, "KeyF" },
    .{ 0x07000a, 0x0022, 0x002a, 0x0022, 0x0005, "KeyG" },
    .{ 0x07000b, 0x0023, 0x002b, 0x0023, 0x0004, "KeyH" },
    .{ 0x07000c, 0x0017, 0x001f, 0x0017, 0x0022, "KeyI" },
    .{ 0x07000d, 0x0024, 0x002c, 0x0024, 0x0026, "KeyJ" },
    .{ 0x07000e, 0x0025, 0x002d, 0x0025, 0x0028, "KeyK" },
    .{ 0x07000f, 0x0026, 0x002e, 0x0026, 0x0025, "KeyL" },

    .{ 0x070010, 0x0032, 0x003a, 0x0032, 0x002e, "KeyM" },
    .{ 0x070011, 0x0031, 0x0039, 0x0031, 0x002d, "KeyN" },
    .{ 0x070012, 0x0018, 0x0020, 0x0018, 0x001f, "KeyO" },
    .{ 0x070013, 0x0019, 0x0021, 0x0019, 0x0023, "KeyP" },
    .{ 0x070014, 0x0010, 0x0018, 0x0010, 0x000c, "KeyQ" },
    .{ 0x070015, 0x0013, 0x001b, 0x0013, 0x000f, "KeyR" },
    .{ 0x070016, 0x001f, 0x0027, 0x001f, 0x0001, "KeyS" },
    .{ 0x070017, 0x0014, 0x001c, 0x0014, 0x0011, "KeyT" },

    .{ 0x070018, 0x0016, 0x001e, 0x0016, 0x0020, "KeyU" },
    .{ 0x070019, 0x002f, 0x0037, 0x002f, 0x0009, "KeyV" },
    .{ 0x07001a, 0x0011, 0x0019, 0x0011, 0x000d, "KeyW" },
    .{ 0x07001b, 0x002d, 0x0035, 0x002d, 0x0007, "KeyX" },
    .{ 0x07001c, 0x0015, 0x001d, 0x0015, 0x0010, "KeyY" },
    .{ 0x07001d, 0x002c, 0x0034, 0x002c, 0x0006, "KeyZ" },
    .{ 0x07001e, 0x0002, 0x000a, 0x0002, 0x0012, "Digit1" },
    .{ 0x07001f, 0x0003, 0x000b, 0x0003, 0x0013, "Digit2" },

    .{ 0x070020, 0x0004, 0x000c, 0x0004, 0x0014, "Digit3" },
    .{ 0x070021, 0x0005, 0x000d, 0x0005, 0x0015, "Digit4" },
    .{ 0x070022, 0x0006, 0x000e, 0x0006, 0x0017, "Digit5" },
    .{ 0x070023, 0x0007, 0x000f, 0x0007, 0x0016, "Digit6" },
    .{ 0x070024, 0x0008, 0x0010, 0x0008, 0x001a, "Digit7" },
    .{ 0x070025, 0x0009, 0x0011, 0x0009, 0x001c, "Digit8" },
    .{ 0x070026, 0x000a, 0x0012, 0x000a, 0x0019, "Digit9" },
    .{ 0x070027, 0x000b, 0x0013, 0x000b, 0x001d, "Digit0" },

    .{ 0x070028, 0x001c, 0x0024, 0x001c, 0x0024, "Enter" },
    .{ 0x070029, 0x0001, 0x0009, 0x0001, 0x0035, "Escape" },
    .{ 0x07002a, 0x000e, 0x0016, 0x000e, 0x0033, "Backspace" },
    .{ 0x07002b, 0x000f, 0x0017, 0x000f, 0x0030, "Tab" },
    .{ 0x07002c, 0x0039, 0x0041, 0x0039, 0x0031, "Space" },
    .{ 0x07002d, 0x000c, 0x0014, 0x000c, 0x001b, "Minus" },
    .{ 0x07002e, 0x000d, 0x0015, 0x000d, 0x0018, "Equal" },
    .{ 0x07002f, 0x001a, 0x0022, 0x001a, 0x0021, "BracketLeft" },

    .{ 0x070030, 0x001b, 0x0023, 0x001b, 0x001e, "BracketRight" },
    .{ 0x070031, 0x002b, 0x0033, 0x002b, 0x002a, "Backslash" },
    // USB#070032 never appears on keyboards that have USB#070031.
    // Platforms use the same scancode as for the two keys.
    // Hence this code can only be generated synthetically
    // (e.g. in a DOM Level 3 KeyboardEvent).
    // The keycap varies on international keyboards:
    //   Dan: '*  Dutch: <>  Ger: #'  UK: #~
    // TODO(garykac): Verify Mac intl keyboard.
    //.{ 0x070032, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    .{ 0x070033, 0x0027, 0x002f, 0x0027, 0x0029, "Semicolon" },
    .{ 0x070034, 0x0028, 0x0030, 0x0028, 0x0027, "Quote" },
    .{ 0x070035, 0x0029, 0x0031, 0x0029, 0x0032, "Backquote" },
    .{ 0x070036, 0x0033, 0x003b, 0x0033, 0x002b, "Comma" },
    .{ 0x070037, 0x0034, 0x003c, 0x0034, 0x002f, "Period" },

    .{ 0x070038, 0x0035, 0x003d, 0x0035, 0x002c, "Slash" },
    // TODO(garykac): CapsLock requires special handling for each platform.
    .{ 0x070039, 0x003a, 0x0042, 0x003a, 0x0039, "CapsLock" },
    .{ 0x07003a, 0x003b, 0x0043, 0x003b, 0x007a, "F1" },
    .{ 0x07003b, 0x003c, 0x0044, 0x003c, 0x0078, "F2" },
    .{ 0x07003c, 0x003d, 0x0045, 0x003d, 0x0063, "F3" },
    .{ 0x07003d, 0x003e, 0x0046, 0x003e, 0x0076, "F4" },
    .{ 0x07003e, 0x003f, 0x0047, 0x003f, 0x0060, "F5" },
    .{ 0x07003f, 0x0040, 0x0048, 0x0040, 0x0061, "F6" },

    .{ 0x070040, 0x0041, 0x0049, 0x0041, 0x0062, "F7" },
    .{ 0x070041, 0x0042, 0x004a, 0x0042, 0x0064, "F8" },
    .{ 0x070042, 0x0043, 0x004b, 0x0043, 0x0065, "F9" },
    .{ 0x070043, 0x0044, 0x004c, 0x0044, 0x006d, "F10" },
    .{ 0x070044, 0x0057, 0x005f, 0x0057, 0x0067, "F11" },
    .{ 0x070045, 0x0058, 0x0060, 0x0058, 0x006f, "F12" },
    // PrintScreen is effectively F13 on Mac OS X.
    .{ 0x070046, 0x0063, 0x006b, 0xe037, 0xffff, "PrintScreen" },
    .{ 0x070047, 0x0046, 0x004e, 0x0046, 0xffff, "ScrollLock" },

    .{ 0x070048, 0x0077, 0x007f, 0x0045, 0xffff, "Pause" },
    // USB#0x070049 Insert, labeled "Help/Insert" on Mac -- see note M1 at top.
    .{ 0x070049, 0x006e, 0x0076, 0xe052, 0x0072, "Insert" },
    .{ 0x07004a, 0x0066, 0x006e, 0xe047, 0x0073, "Home" },
    .{ 0x07004b, 0x0068, 0x0070, 0xe049, 0x0074, "PageUp" },
    // Delete (Forward Delete) named DEL because DELETE conflicts with <windows.h>
    .{ 0x07004c, 0x006f, 0x0077, 0xe053, 0x0075, "Delete" },
    .{ 0x07004d, 0x006b, 0x0073, 0xe04f, 0x0077, "End" },
    .{ 0x07004e, 0x006d, 0x0075, 0xe051, 0x0079, "PageDown" },
    .{ 0x07004f, 0x006a, 0x0072, 0xe04d, 0x007c, "ArrowRight" },

    .{ 0x070050, 0x0069, 0x0071, 0xe04b, 0x007b, "ArrowLeft" },
    .{ 0x070051, 0x006c, 0x0074, 0xe050, 0x007d, "ArrowDown" },
    .{ 0x070052, 0x0067, 0x006f, 0xe048, 0x007e, "ArrowUp" },
    .{ 0x070053, 0x0045, 0x004d, 0xe045, 0x0047, "NumLock" },
    .{ 0x070054, 0x0062, 0x006a, 0xe035, 0x004b, "NumpadDivide" },
    .{ 0x070055, 0x0037, 0x003f, 0x0037, 0x0043, "NumpadMultiply" },
    .{ 0x070056, 0x004a, 0x0052, 0x004a, 0x004e, "NumpadSubtract" },
    .{ 0x070057, 0x004e, 0x0056, 0x004e, 0x0045, "NumpadAdd" },

    .{ 0x070058, 0x0060, 0x0068, 0xe01c, 0x004c, "NumpadEnter" },
    .{ 0x070059, 0x004f, 0x0057, 0x004f, 0x0053, "Numpad1" }, // +End
    .{ 0x07005a, 0x0050, 0x0058, 0x0050, 0x0054, "Numpad2" }, // +Down
    .{ 0x07005b, 0x0051, 0x0059, 0x0051, 0x0055, "Numpad3" }, // +PageDown
    .{ 0x07005c, 0x004b, 0x0053, 0x004b, 0x0056, "Numpad4" }, // +Left
    .{ 0x07005d, 0x004c, 0x0054, 0x004c, 0x0057, "Numpad5" },
    .{ 0x07005e, 0x004d, 0x0055, 0x004d, 0x0058, "Numpad6" }, // +Right
    .{ 0x07005f, 0x0047, 0x004f, 0x0047, 0x0059, "Numpad7" }, // +Home
    .{ 0x070060, 0x0048, 0x0050, 0x0048, 0x005b, "Numpad8" }, // +Up
    .{ 0x070061, 0x0049, 0x0051, 0x0049, 0x005c, "Numpad9" }, // +PageUp
    .{ 0x070062, 0x0052, 0x005a, 0x0052, 0x0052, "Numpad0" }, // +Insert
    .{ 0x070063, 0x0053, 0x005b, 0x0053, 0x0041, "NumpadDecimal" }, // +Delete
    // USB#070064 is not present on US keyboard.
    // This key is typically located near LeftShift key.
    // The keycap varies on international keyboards:
    //   Dan: <> Dutch: ][ Ger: <> UK: \|
    .{ 0x070064, 0x0056, 0x005e, 0x0056, 0x000a, "IntlBackslash" },
    // USB#0x070065 Application Menu (next to RWin key) -- see note L2 at top.
    .{ 0x070065, 0x007f, 0x0087, 0xe05d, 0x006e, "ContextMenu" },
    .{ 0x070066, 0x0074, 0x007c, 0xe05e, 0xffff, "Power" },
    .{ 0x070067, 0x0075, 0x007d, 0x0059, 0x0051, "NumpadEqual" },

    .{ 0x070068, 0x00b7, 0x00bf, 0x0064, 0x0069, "F13" },
    .{ 0x070069, 0x00b8, 0x00c0, 0x0065, 0x006b, "F14" },
    .{ 0x07006a, 0x00b9, 0x00c1, 0x0066, 0x0071, "F15" },
    .{ 0x07006b, 0x00ba, 0x00c2, 0x0067, 0x006a, "F16" },
    .{ 0x07006c, 0x00bb, 0x00c3, 0x0068, 0x0040, "F17" },
    .{ 0x07006d, 0x00bc, 0x00c4, 0x0069, 0x004f, "F18" },
    .{ 0x07006e, 0x00bd, 0x00c5, 0x006a, 0x0050, "F19" },
    .{ 0x07006f, 0x00be, 0x00c6, 0x006b, 0x005a, "F20" },

    .{ 0x070070, 0x00bf, 0x00c7, 0x006c, 0xffff, "F21" },
    .{ 0x070071, 0x00c0, 0x00c8, 0x006d, 0xffff, "F22" },
    .{ 0x070072, 0x00c1, 0x00c9, 0x006e, 0xffff, "F23" },
    // USB#0x070073 -- see note W1 at top.
    .{ 0x070073, 0x00c2, 0x00ca, 0x0076, 0xffff, "F24" },
    .{ 0x070074, 0x0086, 0x008e, 0x0000, 0xffff, "Open" },
    // USB#0x070075 Help -- see note M1 at top.
    .{ 0x070075, 0x008a, 0x0092, 0xe03b, 0xffff, "Help" },
    // USB#0x070076 Keyboard Menu -- see note L2 at top.
    //.{ 0x070076, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    .{ 0x070077, 0x0084, 0x008c, 0x0000, 0xffff, "Select" },

    //.{ 0x070078, 0x0080, 0x0088, 0x0000, 0xffff, ""},
    .{ 0x070079, 0x0081, 0x0089, 0x0000, 0xffff, "Again" },
    .{ 0x07007a, 0x0083, 0x008b, 0xe008, 0xffff, "Undo" },
    .{ 0x07007b, 0x0089, 0x0091, 0xe017, 0xffff, "Cut" },
    .{ 0x07007c, 0x0085, 0x008d, 0xe018, 0xffff, "Copy" },
    .{ 0x07007d, 0x0087, 0x008f, 0xe00a, 0xffff, "Paste" },
    .{ 0x07007e, 0x0088, 0x0090, 0x0000, 0xffff, "Find" },
    .{ 0x07007f, 0x0071, 0x0079, 0xe020, 0x004a, "AudioVolumeMute" },

    .{ 0x070080, 0x0073, 0x007b, 0xe030, 0x0048, "AudioVolumeUp" },
    .{ 0x070081, 0x0072, 0x007a, 0xe02e, 0x0049, "AudioVolumeDown" },
    //.{ 0x070082, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x070083, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x070084, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    .{ 0x070085, 0x0079, 0x0081, 0x007e, 0x005f, "NumpadComma" },

    // International1
    // USB#070086 is used on AS/400 keyboards. Standard Keypad_= is USB#070067.
    //.{ 0x070086, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    // USB#070087 is used for Brazilian /? and Japanese _ 'ro'.
    .{ 0x070087, 0x0059, 0x0061, 0x0073, 0x005e, "IntlRo" },
    // International2
    // USB#070088 is used as Japanese Hiragana/Katakana key.
    .{ 0x070088, 0x005d, 0x0065, 0x0070, 0xffff, "KanaMode" },
    // International3
    // USB#070089 is used as Japanese Yen key.
    .{ 0x070089, 0x007c, 0x0084, 0x007d, 0x005d, "IntlYen" },
    // International4
    // USB#07008a is used as Japanese Henkan (Convert) key.
    .{ 0x07008a, 0x005c, 0x0064, 0x0079, 0xffff, "Convert" },
    // International5
    // USB#07008b is used as Japanese Muhenkan (No-convert) key.
    .{ 0x07008b, 0x005e, 0x0066, 0x007b, 0xffff, "NonConvert" },
    //.{ 0x07008c, 0x005f, 0x0067, 0x005c, 0xffff, ""},
    //.{ 0x07008d, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x07008e, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x07008f, 0x0000, 0x0000, 0x0000, 0xffff, ""},

    // LANG1
    // USB#070090 is used as Korean Hangul/English toggle key, and as the Kana key
    // on the Apple Japanese keyboard.
    .{ 0x070090, 0x007a, 0x0082, 0x0072, 0x0068, "Lang1" },
    // LANG2
    // USB#070091 is used as Korean Hanja conversion key, and as the Eisu key on
    // the Apple Japanese keyboard.
    .{ 0x070091, 0x007b, 0x0083, 0x0071, 0x0066, "Lang2" },
    // LANG3
    // USB#070092 is used as Japanese Katakana key.
    .{ 0x070092, 0x005a, 0x0062, 0x0078, 0xffff, "Lang3" },
    // LANG4
    // USB#070093 is used as Japanese Hiragana key.
    .{ 0x070093, 0x005b, 0x0063, 0x0077, 0xffff, "Lang4" },
    // LANG5
    // USB#070094 is used as Japanese Zenkaku/Hankaku (Fullwidth/halfwidth) key.
    // Not mapped on Windows -- see note W1 at top.
    .{ 0x070094, 0x0055, 0x005d, 0x0000, 0xffff, "Lang5" },
    //.{ 0x070095, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x070096, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x070097, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x070098, 0x0000, 0x0000, 0x0000, 0xffff, ""},

    //.{ 0x070099, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x07009a, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    .{ 0x07009b, 0x0000, 0x0000, 0x0000, 0xffff, "Abort" },
    // USB#0x07009c Keyboard Clear -- see note L1 at top.
    //.{ 0x07009c, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x07009d, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x07009e, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x07009f, 0x0000, 0x0000, 0x0000, 0xffff, ""},

    //.{ 0x0700a0, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700a1, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700a2, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    // USB#0x0700a3 Props -- see note L2 at top.
    .{ 0x0700a3, 0x0000, 0x0000, 0x0000, 0xffff, "Props" },
    //.{ 0x0700a4, 0x0000, 0x0000, 0x0000, 0xffff, ""},

    //.{ 0x0700b0, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700b1, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700b2, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700b3, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700b4, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700b5, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    .{ 0x0700b6, 0x00b3, 0x00bb, 0x0000, 0xffff, "NumpadParenLeft" },
    .{ 0x0700b7, 0x00b4, 0x00bc, 0x0000, 0xffff, "NumpadParenRight" },

    //.{ 0x0700b8, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700b9, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700ba, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    .{ 0x0700bb, 0x0000, 0x0000, 0x0000, 0xffff, "NumpadBackspace" },
    //.{ 0x0700bc, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700bd, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700be, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700bf, 0x0000, 0x0000, 0x0000, 0xffff, ""},

    //.{ 0x0700c0, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700c1, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700c2, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700c3, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700c4, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700c5, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700c6, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700c7, 0x0000, 0x0000, 0x0000, 0xffff, ""},

    //.{ 0x0700c8, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700c9, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700ca, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //           NUMPAD_DOUBLE_VERTICAL_BAR),  // Keypad_||
    //.{ 0x0700cb, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700cc, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700cd, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700ce, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700cf, 0x0000, 0x0000, 0x0000, 0xffff, ""},

    .{ 0x0700d0, 0x0000, 0x0000, 0x0000, 0xffff, "NumpadMemoryStore" },
    .{ 0x0700d1, 0x0000, 0x0000, 0x0000, 0xffff, "NumpadMemoryRecall" },
    .{ 0x0700d2, 0x0000, 0x0000, 0x0000, 0xffff, "NumpadMemoryClear" },
    .{ 0x0700d3, 0x0000, 0x0000, 0x0000, 0xffff, "NumpadMemoryAdd" },
    .{ 0x0700d4, 0x0000, 0x0000, 0x0000, 0xffff, "NumpadMemorySubtract" },
    //.{ 0x0700d5, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700d6, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    .{ 0x0700d7, 0x0076, 0x007e, 0x0000, 0xffff, "" },
    // USB#0x0700d8 Keypad Clear -- see note L1 at top.
    .{ 0x0700d8, 0x0000, 0x0000, 0x0000, 0xffff, "NumpadClear" },
    .{ 0x0700d9, 0x0000, 0x0000, 0x0000, 0xffff, "NumpadClearEntry" },
    //.{ 0x0700da, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700db, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700dc, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0700dd, 0x0000, 0x0000, 0x0000, 0xffff, ""},

    // USB#0700de - #0700df are reserved.
    .{ 0x0700e0, 0x001d, 0x0025, 0x001d, 0x003b, "ControlLeft" },
    .{ 0x0700e1, 0x002a, 0x0032, 0x002a, 0x0038, "ShiftLeft" },
    // USB#0700e2: left Alt key (Mac left Option key).
    .{ 0x0700e2, 0x0038, 0x0040, 0x0038, 0x003a, "AltLeft" },
    // USB#0700e3: left GUI key, e.g. Windows, Mac Command, ChromeOS Search.
    .{ 0x0700e3, 0x007d, 0x0085, 0xe05b, 0x0037, "MetaLeft" },
    .{ 0x0700e4, 0x0061, 0x0069, 0xe01d, 0x003e, "ControlRight" },
    .{ 0x0700e5, 0x0036, 0x003e, 0x0036, 0x003c, "ShiftRight" },
    // USB#0700e6: right Alt key (Mac right Option key).
    .{ 0x0700e6, 0x0064, 0x006c, 0xe038, 0x003d, "AltRight" },
    // USB#0700e7: right GUI key, e.g. Windows, Mac Command, ChromeOS Search.
    .{ 0x0700e7, 0x007e, 0x0086, 0xe05c, 0x0036, "MetaRight" },

    // USB#0700e8 - #07ffff are reserved

    // ==================================
    // USB Usage Page 0x0c: Consumer Page
    // ==================================
    // AL = Application Launch
    // AC = Application Control

    // TODO(garykac): Many XF86 keys have multiple scancodes mapping to them.
    // We need to map all of these into a canonical USB scancode without
    // confusing the reverse-lookup - most likely by simply returning the first
    // found match.

    // TODO(garykac): Find appropriate mappings for:
    // Win#e03c Music - USB#0c0193 is AL_AVCapturePlayback
    // Win#e064 Pictures
    // XKB#0080 XF86LaunchA
    // XKB#0099 XF86Send
    // XKB#009b XF86Xfer
    // XKB#009c XF86Launch1
    // XKB#009d XF86Launch2
    // XKB... remaining XF86 keys

    // KEY_BRIGHTNESS* added in Linux 3.16
    // http://www.usb.org/developers/hidpage/HUTRR41.pdf
    //
    // Keyboard backlight/illumination spec update.
    // https://www.usb.org/sites/default/files/hutrr73_-_fn_key_and_keyboard_backlight_brightness_0.pdf
    //   USB     evdev    XKB     Win     Mac   Code
    .{ 0x0c0060, 0x0166, 0x016e, 0x0000, 0xffff, "" },
    .{ 0x0c0061, 0x0172, 0x017a, 0x0000, 0xffff, "" },
    .{ 0x0c006f, 0x00e1, 0x00e9, 0x0000, 0xffff, "BrightnessUp" },
    .{ 0x0c0070, 0x00e0, 0x00e8, 0x0000, 0xffff, "BrightnessDown" },
    .{ 0x0c0072, 0x01af, 0x01b7, 0x0000, 0xffff, "" },
    .{ 0x0c0073, 0x0250, 0x0258, 0x0000, 0xffff, "" },
    .{ 0x0c0074, 0x0251, 0x0259, 0x0000, 0xffff, "" },
    .{ 0x0c0075, 0x00f4, 0x00fc, 0x0000, 0xffff, "" },
    .{ 0x0c0079, 0x00e6, 0x00ee, 0x0000, 0xffff, "" },
    .{ 0x0c007a, 0x00e5, 0x00ed, 0x0000, 0xffff, "" },
    .{ 0x0c007c, 0x00e4, 0x00ec, 0x0000, 0xffff, "" },
    .{ 0x0c0083, 0x0195, 0x019d, 0x0000, 0xffff, "" },
    .{ 0x0c008c, 0x00a9, 0x00b1, 0x0000, 0xffff, "" },
    .{ 0x0c008d, 0x016a, 0x0172, 0x0000, 0xffff, "" },
    .{ 0x0c0094, 0x00ae, 0x00b6, 0x0000, 0xffff, "" },
    .{ 0x0c009c, 0x019a, 0x01a2, 0x0000, 0xffff, "" },
    .{ 0x0c009d, 0x019b, 0x01a3, 0x0000, 0xffff, "" },

    //              USB     evdev    XKB     Win     Mac
    .{ 0x0c00b0, 0x00cf, 0x00d7, 0x0000, 0xffff, "MediaPlay" },
    .{ 0x0c00b1, 0x00c9, 0x00d1, 0x0000, 0xffff, "MediaPause" },
    .{ 0x0c00b2, 0x00a7, 0x00af, 0x0000, 0xffff, "MediaRecord" },
    .{ 0x0c00b3, 0x00d0, 0x00d8, 0x0000, 0xffff, "MediaFastForward" },
    .{ 0x0c00b4, 0x00a8, 0x00b0, 0x0000, 0xffff, "MediaRewind" },
    .{ 0x0c00b5, 0x00a3, 0x00ab, 0xe019, 0xffff, "MediaTrackNext" },
    .{ 0x0c00b6, 0x00a5, 0x00ad, 0xe010, 0xffff, "MediaTrackPrevious" },
    .{ 0x0c00b7, 0x00a6, 0x00ae, 0xe024, 0xffff, "MediaStop" },
    .{ 0x0c00b8, 0x00a1, 0x00a9, 0xe02c, 0xffff, "Eject" },
    .{ 0x0c00cd, 0x00a4, 0x00ac, 0xe022, 0xffff, "MediaPlayPause" },
    .{ 0x0c00cf, 0x0246, 0x024e, 0x0000, 0xffff, "" },
    .{ 0x0c00d8, 0x024a, 0x0252, 0x0000, 0xffff, "" },
    .{ 0x0c00d9, 0x0249, 0x0251, 0x0000, 0xffff, "" },
    .{ 0x0c00e5, 0x00d1, 0x00d9, 0x0000, 0xffff, "" },
    //.{ 0x0c00e6, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0c0150, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0c0151, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0c0152, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0c0153, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0c0154, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    //.{ 0x0c0155, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    // USB#0c0183: AL Consumer Control Configuration
    .{ 0x0c0183, 0x00ab, 0x00b3, 0xe06d, 0xffff, "MediaSelect" },
    .{ 0x0c0184, 0x01a5, 0x01ad, 0x0000, 0xffff, "" },
    .{ 0x0c0186, 0x01a7, 0x01af, 0x0000, 0xffff, "" },
    // USB#0x0c018a AL_EmailReader
    .{ 0x0c018a, 0x009b, 0x00a3, 0xe06c, 0xffff, "LaunchMail" },
    // USB#0x0c018d: AL Contacts/Address Book
    .{ 0x0c018d, 0x01ad, 0x01b5, 0x0000, 0xffff, "" },
    // USB#0x0c018e: AL Calendar/Schedule
    .{ 0x0c018e, 0x018d, 0x0195, 0x0000, 0xffff, "" },
    // USB#0x0c018f AL Task/Project Manager
    //.{ 0x0c018f, 0x0241, 0x0249, 0x0000, 0xffff, ""},
    // USB#0x0c0190: AL Log/Journal/Timecard
    //.{ 0x0c0190, 0x0242, 0x024a, 0x0000, 0xffff, ""},
    // USB#0x0c0192: AL_Calculator
    .{ 0x0c0192, 0x008c, 0x0094, 0xe021, 0xffff, "LaunchApp2" },
    // USB#0c0194: My Computer (AL_LocalMachineBrowser)
    .{ 0x0c0194, 0x0090, 0x0098, 0xe06b, 0xffff, "LaunchApp1" },
    .{ 0x0c0196, 0x0096, 0x009e, 0x0000, 0xffff, "" },
    .{ 0x0c019C, 0x01b1, 0x01b9, 0x0000, 0xffff, "" },
    // USB#0x0c019e: AL Terminal Lock/Screensaver
    .{ 0x0c019e, 0x0098, 0x00a0, 0x0000, 0xffff, "" },
    // USB#0x0c019f AL Control Panel
    .{ 0x0c019f, 0x0243, 0x024b, 0x0000, 0xffff, "LaunchControlPanel" },
    // USB#0x0c01a2: AL Select Task/Application
    .{ 0x0c01a2, 0x0244, 0x024c, 0x0000, 0xffff, "SelectTask" },
    // USB#0x0c01a7: AL_Documents
    .{ 0x0c01a7, 0x00eb, 0x00f3, 0x0000, 0xffff, "" },
    .{ 0x0c01ab, 0x01b0, 0x01b8, 0x0000, 0xffff, "" },
    // USB#0x0c01ae: AL Keyboard Layout
    .{ 0x0c01ae, 0x0176, 0x017e, 0x0000, 0xffff, "" },
    .{ 0x0c01b1, 0x0245, 0x024d, 0x0000, 0xffff, "LaunchScreenSaver" },
    .{ 0x0c01cb, 0x0247, 0x024f, 0x0000, 0xffff, "LaunchAssistant" },
    // USB#0c01b4: Home Directory (AL_FileBrowser) (Explorer)
    //.{ 0x0c01b4, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    // USB#0x0c01b7: AL Audio Browser
    .{ 0x0c01b7, 0x0188, 0x0190, 0x0000, 0xffff, "" },
    // USB#0x0c0201: AC New
    .{ 0x0c0201, 0x00b5, 0x00bd, 0x0000, 0xffff, "" },
    // USB#0x0c0203: AC Close
    .{ 0x0c0203, 0x00ce, 0x00d6, 0x0000, 0xffff, "" },
    // USB#0x0c0207: AC Close
    .{ 0x0c0207, 0x00ea, 0x00f2, 0x0000, 0xffff, "" },
    // USB#0x0c0208: AC Print
    .{ 0x0c0208, 0x00d2, 0x00da, 0x0000, 0xffff, "" },
    // USB#0x0c0221:  AC_Search
    .{ 0x0c0221, 0x00d9, 0x00e1, 0xe065, 0xffff, "BrowserSearch" },
    // USB#0x0c0223:  AC_Home
    .{ 0x0c0223, 0x00ac, 0x00b4, 0xe032, 0xffff, "BrowserHome" },
    // USB#0x0c0224:  AC_Back
    .{ 0x0c0224, 0x009e, 0x00a6, 0xe06a, 0xffff, "BrowserBack" },
    // USB#0x0c0225:  AC_Forward
    .{ 0x0c0225, 0x009f, 0x00a7, 0xe069, 0xffff, "BrowserForward" },
    // USB#0x0c0226:  AC_Stop
    .{ 0x0c0226, 0x0080, 0x0088, 0xe068, 0xffff, "BrowserStop" },
    // USB#0x0c0227:  AC_Refresh (Reload)
    .{ 0x0c0227, 0x00ad, 0x00b5, 0xe067, 0xffff, "BrowserRefresh" },
    // USB#0x0c022a:  AC_Bookmarks (Favorites)
    .{ 0x0c022a, 0x009c, 0x00a4, 0xe066, 0xffff, "BrowserFavorites" },
    .{ 0x0c022d, 0x01a2, 0x01aa, 0x0000, 0xffff, "" },
    .{ 0x0c022e, 0x01a3, 0x01ab, 0x0000, 0xffff, "" },
    // USB#0x0c0230:  AC Full Screen View
    //.{ 0x0c0230, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    // USB#0x0c0231:  AC Normal View
    //.{ 0x0c0231, 0x0000, 0x0000, 0x0000, 0xffff, ""},
    .{ 0x0c0232, 0x0174, 0x017c, 0x0000, 0xffff, "ZoomToggle" },
    // USB#0x0c0279:  AC Redo/Repeat
    .{ 0x0c0279, 0x00b6, 0x00be, 0x0000, 0xffff, "" },
    // USB#0x0c0289:  AC_Reply
    .{ 0x0c0289, 0x00e8, 0x00f0, 0x0000, 0xffff, "MailReply" },
    // USB#0x0c028b:  AC_ForwardMsg (MailForward)
    .{ 0x0c028b, 0x00e9, 0x00f1, 0x0000, 0xffff, "MailForward" },
    // USB#0x0c028c:  AC_Send
    .{ 0x0c028c, 0x00e7, 0x00ef, 0x0000, 0xffff, "MailSend" },
    // USB#0x0c029d:  AC Next Keyboard Layout Select
    .{ 0x0c029d, 0x0248, 0x0250, 0x0000, 0xffff, "KeyboardLayoutSelect" },
    .{ 0x0c029f, 0x0078, 0x0080, 0x0000, 0xffff, "ShowAllWindows" },
    .{ 0x0c02a2, 0x00cc, 0x00d4, 0x0000, 0xffff, "" },
    .{ 0x0c02d0, 0x0279, 0x0281, 0x0000, 0xffff, "" },
};

test {}
