searchData={"items":[{"type":"module","title":"crypto","doc":"Crypto Functions\n\nThis module provides a set of cryptographic functions.\n\n- **Hash functions** -\n\n  - **SHA1, SHA2** - [Secure Hash Standard (FIPS PUB180-4)](https://nvlpubs.nist.gov/nistpubs/FIPS/NIST.FIPS.180-4.pdf)\n  - **SHA3** - [SHA-3 Standard: Permutation-Based Hash and Extendable-Output Functions (FIPS PUB 202)](https://www.nist.gov/publications/sha-3-standard-permutation-based-hash-and-extendable-output-functions?pub_id=919061)\n\n  - **BLAKE2** - [BLAKE2 — fast secure hashing](https://blake2.net/)\n\n  - **SM3** - [The SM3 Hash Function (GM/T 0004-2012)](https://datatracker.ietf.org/doc/html/draft-sca-cfrg-sm3-02)\n\n  - **MD5** - [The MD5 Message Digest Algorithm (RFC 1321)](http://www.ietf.org/rfc/rfc1321.txt)\n\n  - **MD4** - [The MD4 Message Digest Algorithm (RFC 1320)](http://www.ietf.org/rfc/rfc1320.txt)\n\n- **MACs - Message Authentication Codes** -\n\n  - **Hmac functions** - [Keyed-Hashing for Message Authentication (RFC 2104)](http://www.ietf.org/rfc/rfc2104.txt)\n\n  - **Cmac functions** - [The AES-CMAC Algorithm (RFC 4493)](http://www.ietf.org/rfc/rfc4493.txt)\n\n  - **POLY1305** - [ChaCha20 and Poly1305 for IETF Protocols (RFC 7539)](http://www.ietf.org/rfc/rfc7539.txt)\n\n- **Symmetric Ciphers** - \n\n  - **DES, 3DES and AES** - [Block Cipher Techniques (NIST)](https://csrc.nist.gov/projects/block-cipher-techniques)\n\n  - **Blowfish** -\n    [Fast Software Encryption, Cambridge Security Workshop Proceedings (December 1993), Springer-Verlag, 1994, pp. 191-204.](https://www.schneier.com/academic/archives/1994/09/description_of_a_new.html)\n\n  - **Chacha20** - [ChaCha20 and Poly1305 for IETF Protocols (RFC 7539)](http://www.ietf.org/rfc/rfc7539.txt)\n\n  - **Chacha20_poly1305** - [ChaCha20 and Poly1305 for IETF Protocols (RFC 7539)](http://www.ietf.org/rfc/rfc7539.txt)\n\n  - **SM4** - [The SM4 Block Cipher Algorithm](https://www.iso.org/standard/81564.html)\n\n- **Modes** -\n  - **ECB, CBC, CFB, OFB and CTR** - [Recommendation for Block Cipher Modes of\n    Operation: Methods and Techniques (NIST SP 800-38A)](https://csrc.nist.gov/publications/detail/sp/800-38a/final)\n\n  - **GCM** - [Recommendation for Block Cipher Modes of Operation:\n    Galois/Counter Mode (GCM) and GMAC (NIST SP 800-38D)](https://csrc.nist.gov/publications/detail/sp/800-38d/final)\n\n  - **CCM** - [Recommendation for Block Cipher Modes of Operation: The CCM Mode\n    for Authentication and Confidentiality (NIST SP 800-38C)](https://nvlpubs.nist.gov/nistpubs/legacy/sp/nistspecialpublication800-38c.pdf)\n\n- **Asymmetric Ciphers - Public Key Techniques** -\n\n  - **RSA** - [PKCS #1: RSA Cryptography Specifications (RFC 3447)](http://www.ietf.org/rfc/rfc3447.txt)\n\n  - **DSS** - [Digital Signature Standard (DSS) (FIPS 186-4)](https://csrc.nist.gov/publications/detail/fips/186/4/final)\n\n  - **ECDSA** - [Elliptic Curve Digital Signature Algorithm (ECDSA)](http://csrc.nist.gov/groups/STM/cavp/documents/dss2/ecdsa2vs.pdf)\n\n  - **SRP** - [The SRP Authentication and Key Exchange System (RFC 2945)](http://www.ietf.org/rfc/rfc2945.txt)\n\n> #### Note {: .info }\n>\n> The actual supported algorithms and features depends on their availability in\n> the actual libcrypto used. See the [crypto (App)](crypto_app.md) about\n> dependencies.\n>\n> Enabling FIPS mode will also disable algorithms and features.\n\nThe [CRYPTO User's Guide](index.html) has more information on FIPS, Engines and\nAlgorithm Details like key lengths.","ref":"crypto.html"},{"type":"module","title":"Exceptions - crypto","doc":"[](){: #error_old }","ref":"crypto.html#module-exceptions"},{"type":"module","title":"Atoms - the older style - crypto","doc":"The exception `error:badarg` signifies that one or more arguments are of wrong\ndata type, or are otherwise badly formed.\n\nThe exception `error:notsup` signifies that the algorithm is known but is not\nsupported by current underlying libcrypto or explicitly disabled when building\nthat.\n\nFor a list of supported algorithms, see [supports(ciphers)](`supports/1`).\n\n[](){: #error_3tup }","ref":"crypto.html#module-atoms-the-older-style"},{"type":"module","title":"3-tuples - the new style - crypto","doc":"The exception is:\n\n```text\nerror:{Tag, C_FileInfo, Description}\n\nTag = badarg | notsup | error\nC_FileInfo = term()    % Usually only useful for the OTP maintainer\nDescription = string() % Clear text, sometimes only useful for the OTP maintainer\n```\n\nThe exception tags are:\n\n- **`badarg`** - Signifies that one or more arguments are of wrong data type or\n  are otherwise badly formed.\n\n- **`notsup`** - Signifies that the algorithm is known but is not supported by\n  current underlying libcrypto or explicitly disabled when building that one.\n\n- **`error`** - An error condition that should not occur, for example a memory\n  allocation failed or the underlying cryptolib returned an error code, for\n  example `\"Can't initialize context, step 1\"`. Those text usually needs\n  searching the C-code to be understood.\n\nUsually there are more information in the call stack about which argument caused\nthe exception and what the values where.\n\nTo catch the exception, use for example:\n\n```text\ntry crypto:crypto_init(Ciph, Key, IV, true)\n    catch\n        error:{Tag, _C_FileInfo, Description} ->\n            do_something(......)\n         .....\nend\n```","ref":"crypto.html#module-3-tuples-the-new-style"},{"type":"type","title":"crypto.blake2/0","doc":"","ref":"crypto.html#t:blake2/0"},{"type":"function","title":"crypto.bytes_to_integer/1","doc":"Convert binary representation, of an integer, to an Erlang integer.","ref":"crypto.html#bytes_to_integer/1"},{"type":"type","title":"crypto.cipher/0","doc":"","ref":"crypto.html#t:cipher/0"},{"type":"type","title":"crypto.cipher_aead/0","doc":"Ciphers known by the CRYPTO application.\n\nNote that this list might be reduced if the underlying libcrypto does not\nsupport all of them.","ref":"crypto.html#t:cipher_aead/0"},{"type":"function","title":"crypto.cipher_info/1","doc":"Get information about a cipher algorithm.\n\nReturns a map with information about block size, key length, IV length, aead\nsupport and possibly other properties of the cipher algorithm in question.\n\n> #### Note {: .info }\n>\n> The ciphers `aes_cbc`, `aes_cfb8`, `aes_cfb128`, `aes_ctr`, `aes_ecb`,\n> `aes_gcm` and `aes_ccm` has no keylength in the `Type` as opposed to for\n> example `aes_128_ctr`. They adapt to the length of the key provided in the\n> encrypt and decrypt function. Therefore it is impossible to return a valid\n> keylength in the map.\n>\n> Always use a `Type` with an explicit key length,\n\nFor a list of supported cipher algorithms, see\n[supports(ciphers)](`supports/1`).","ref":"crypto.html#cipher_info/1"},{"type":"type","title":"crypto.cipher_iv/0","doc":"","ref":"crypto.html#t:cipher_iv/0"},{"type":"type","title":"crypto.cipher_no_iv/0","doc":"","ref":"crypto.html#t:cipher_no_iv/0"},{"type":"type","title":"crypto.cmac_cipher_algorithm/0","doc":"","ref":"crypto.html#t:cmac_cipher_algorithm/0"},{"type":"type","title":"crypto.compatibility_only_hash/0","doc":"The `t:compatibility_only_hash/0` algorithms are recommended only for\ncompatibility with existing applications.","ref":"crypto.html#t:compatibility_only_hash/0"},{"type":"function","title":"crypto.compute_key/4","doc":"Compute the shared secret from the private key and the other party's public\nkey.\n\nSee also `public_key:compute_key/2`.\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.","ref":"crypto.html#compute_key/4"},{"type":"function","title":"crypto.crypto_final/1","doc":"Finalize a streaming encryptions or decryptions operation and delivers the final\nbytes of the final block.\n\nThe data returned from this function may be empty if no padding was enabled in\n`crypto_init/3` or `crypto_init/4`.\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.","ref":"crypto.html#crypto_final/1"},{"type":"function","title":"crypto.crypto_get_data/1","doc":"Return information about a `t:crypto_state/0`.\n\nThe information returned is a map, which currently contains at least:\n\n- **`size`** - The number of bytes encrypted or decrypted so far.\n\n- **`padding_size`** - After a call to `crypto_final/1` it contains the number\n  of bytes padded. Otherwise 0.\n\n- **`padding_type`** - The type of the padding as provided in the call to\n  `crypto_init/3` or `crypto_init/4`.\n\n- **`encrypt`** - Is `true` if encryption is performed. It is `false` otherwise.\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.","ref":"crypto.html#crypto_get_data/1"},{"type":"function","title":"crypto.crypto_init/3","doc":"Initialize the state for a streaming encryption or decryption\noperation.\n\nEquivalent to the call\n[`crypto_init(Cipher, Key, <<>>, FlagOrOptions)`](`crypto_init/4`). It is\nintended for ciphers without an IV (nounce).\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.","ref":"crypto.html#crypto_init/3"},{"type":"function","title":"crypto.crypto_init/4","doc":"Initialize the state for a streaming encryptions or decryptions operation.\n\nThe returned state should be used as argument to `crypto_update/2` and\n`crypto_final/1` to do the actual encryption or decryption.\n\nIf `IV = <<>>`, no IV is used. This is intended for ciphers without an IV\n(nounce). See `crypto_init/3`.\n\nFor encryption, set the `FlagOrOptions` to `true` or `[{encrypt,true}]`. For\ndecryption, set it to `false` or `[{encrypt,false}]`.\n\nPadding could be enabled with the option [\\{padding,Padding\\}](`t:padding/0`).\nThe [cryptolib_padding](`t:cryptolib_padding/0`) enables `pkcs_padding` or no\npadding (`none`). The paddings `zero` or `random` fills the last part of the\nlast block with zeroes or random bytes. If the last block is already full,\nnothing is added.\n\nIn decryption, the [cryptolib_padding](`t:cryptolib_padding/0`) removes such\npadding, if present. The [otp_padding](`t:otp_padding/0`) is not removed - it\nhas to be done elsewhere.\n\nIf padding is `{padding,none}` or not specified and the total data from all\nsubsequent [crypto_updates](`crypto_update/2`) does not fill the last block\nfully, that last data is lost. In case of `{padding,none}` there will be an\nerror in this case. If padding is not specified, the bytes of the unfilled block\nis silently discarded.\n\nThe actual padding is performed by `crypto_final/1`.\n\nFor blocksizes call `cipher_info/1`.\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.\n\nSee\n[examples in the User's Guide.](new_api.md#examples-of-crypto_init-4-and-crypto_update-2)","ref":"crypto.html#crypto_init/4"},{"type":"type","title":"crypto.crypto_integer/0","doc":"","ref":"crypto.html#t:crypto_integer/0"},{"type":"function","title":"crypto.crypto_one_time/4","doc":"Do a complete encrypt or decrypt of the full text.\n\nAs `crypto_one_time/5` but for ciphers without IVs.\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.","ref":"crypto.html#crypto_one_time/4"},{"type":"function","title":"crypto.crypto_one_time/5","doc":"Do a complete encrypt or decrypt of the full text.\n\nArgument `Data` is the text to be encrypted or decrypted.\n\nFor encryption, set the `FlagOrOptions` to `true`. For decryption, set it to\n`false`. For setting other options, see `crypto_init/4`.\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.\n\nSee [examples in the User's Guide.](new_api.md#example-of-crypto_one_time-5)","ref":"crypto.html#crypto_one_time/5"},{"type":"function","title":"crypto.crypto_one_time_aead/6","doc":"","ref":"crypto.html#crypto_one_time_aead/6"},{"type":"function","title":"crypto.crypto_one_time_aead/7","doc":"Do a complete encrypt or decrypt with an AEAD cipher of the full text.\n\nFor encryption, set the `EncryptFlag` to `true` and set the `TagOrTagLength` to\nthe wanted size (in bytes) of the tag, that is, the tag length. If the default\nlength is wanted, the `crypto_one_time_aead/6` form may be used.\n\nFor decryption, set the `EncryptFlag` to `false` and put the tag to be checked\nin the argument `TagOrTagLength`.\n\nAdditional Authentication Data (AAD) is plaintext data that will not be\nencrypted, but will be covered by authenticity protection. It should be provided\nthrough the `AAD` argument, but can be an empty binary as well (`<<>>`) if not\nneeded. In that case, a plain AE (Authenticated Encryption) is performed instead\nof AEAD (Authenticated Encryption with Associated Data). This function only\nsupports ciphers that can be used both with and without AAD.\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.\n\nSee\n[examples in the User's Guide.](new_api.md#example-of-crypto_one_time_aead-6)","ref":"crypto.html#crypto_one_time_aead/7"},{"type":"type","title":"crypto.crypto_opt/0","doc":"Selects encryption (`{encrypt,true}`) or decryption (`{encrypt,false}`).","ref":"crypto.html#t:crypto_opt/0"},{"type":"type","title":"crypto.crypto_opts/0","doc":"","ref":"crypto.html#t:crypto_opts/0"},{"type":"opaque","title":"crypto.crypto_state/0","doc":"","ref":"crypto.html#t:crypto_state/0"},{"type":"function","title":"crypto.crypto_update/2","doc":"Add data to a streaming encryption or decryption operation.\n\nIf the part is less than a number of full blocks, only the full blocks (possibly\nnone) are encrypted or decrypted and the remaining bytes are saved to the next\n`crypto_update` operation. The `State` should be created with `crypto_init/3` or\n`crypto_init/4`.\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.\n\nSee\n[examples in the User's Guide.](new_api.md#examples-of-crypto_init-4-and-crypto_update-2)","ref":"crypto.html#crypto_update/2"},{"type":"type","title":"crypto.cryptolib_padding/0","doc":"The `cryptolib_padding` are paddings that may be present in the underlying\ncryptolib linked to the Erlang/OTP crypto app.\n\nFor OpenSSL, see the [OpenSSL documentation](https://openssl.org). and find\n`EVP_CIPHER_CTX_set_padding()` in cryptolib for your linked version.","ref":"crypto.html#t:cryptolib_padding/0"},{"type":"type","title":"crypto.dh_params/0","doc":"```text\ndh_params() = [P, G] | [P, G, PrivateKeyBitLength]\n```","ref":"crypto.html#t:dh_params/0"},{"type":"type","title":"crypto.dh_private/0","doc":"","ref":"crypto.html#t:dh_private/0"},{"type":"type","title":"crypto.dh_public/0","doc":"","ref":"crypto.html#t:dh_public/0"},{"type":"type","title":"crypto.dss_digest_type/0","doc":"","ref":"crypto.html#t:dss_digest_type/0"},{"type":"type","title":"crypto.dss_private/0","doc":"```text\ndss_public() = [P, Q, G, Y]\n```\n\nWhere P, Q and G are the dss parameters and Y is the public key.\n\n```text\ndss_private() = [P, Q, G, X]\n```\n\nWhere P, Q and G are the dss parameters and X is the private key.","ref":"crypto.html#t:dss_private/0"},{"type":"type","title":"crypto.dss_public/0","doc":"","ref":"crypto.html#t:dss_public/0"},{"type":"type","title":"crypto.ec_basis/0","doc":"Curve definition details.","ref":"crypto.html#t:ec_basis/0"},{"type":"type","title":"crypto.ec_characteristic_two_field/0","doc":"","ref":"crypto.html#t:ec_characteristic_two_field/0"},{"type":"type","title":"crypto.ec_curve/0","doc":"Parametric curve definition.","ref":"crypto.html#t:ec_curve/0"},{"type":"function","title":"crypto.ec_curve/1","doc":"Return the defining parameters of a elliptic curve.","ref":"crypto.html#ec_curve/1"},{"type":"function","title":"crypto.ec_curves/0","doc":"Return all supported named elliptic curves.","ref":"crypto.html#ec_curves/0"},{"type":"type","title":"crypto.ec_explicit_curve/0","doc":"","ref":"crypto.html#t:ec_explicit_curve/0"},{"type":"type","title":"crypto.ec_field/0","doc":"","ref":"crypto.html#t:ec_field/0"},{"type":"type","title":"crypto.ec_named_curve/0","doc":"","ref":"crypto.html#t:ec_named_curve/0"},{"type":"type","title":"crypto.ec_prime_field/0","doc":"","ref":"crypto.html#t:ec_prime_field/0"},{"type":"type","title":"crypto.ecdh_params/0","doc":"","ref":"crypto.html#t:ecdh_params/0"},{"type":"type","title":"crypto.ecdh_private/0","doc":"","ref":"crypto.html#t:ecdh_private/0"},{"type":"type","title":"crypto.ecdh_public/0","doc":"","ref":"crypto.html#t:ecdh_public/0"},{"type":"type","title":"crypto.ecdsa_digest_type/0","doc":"","ref":"crypto.html#t:ecdsa_digest_type/0"},{"type":"type","title":"crypto.ecdsa_params/0","doc":"","ref":"crypto.html#t:ecdsa_params/0"},{"type":"type","title":"crypto.ecdsa_private/0","doc":"","ref":"crypto.html#t:ecdsa_private/0"},{"type":"type","title":"crypto.ecdsa_public/0","doc":"","ref":"crypto.html#t:ecdsa_public/0"},{"type":"type","title":"crypto.eddsa_params/0","doc":"","ref":"crypto.html#t:eddsa_params/0"},{"type":"type","title":"crypto.eddsa_private/0","doc":"","ref":"crypto.html#t:eddsa_private/0"},{"type":"type","title":"crypto.eddsa_public/0","doc":"","ref":"crypto.html#t:eddsa_public/0"},{"type":"type","title":"crypto.edwards_curve_dh/0","doc":"","ref":"crypto.html#t:edwards_curve_dh/0"},{"type":"type","title":"crypto.edwards_curve_ed/0","doc":"Note that some curves are disabled if FIPS is enabled.","ref":"crypto.html#t:edwards_curve_ed/0"},{"type":"function","title":"crypto.enable_fips_mode/1","doc":"Enable or disable FIPs mode.\n\nArgument `Enable` should be `true` to enable and `false` to disable FIPS mode.\nReturns `true` if the operation was successful or `false` otherwise.\n\nNote that to enable FIPS mode successfully, OTP must be built with the configure\noption `--enable-fips`, and the underlying libcrypto must also support FIPS.\n\nSee also `info_fips/0`.","ref":"crypto.html#enable_fips_mode/1"},{"type":"function","title":"crypto.engine_add/1","doc":"Add the engine to OpenSSL's internal list.\n\nThe function raises a `error:badarg` if the parameters are in wrong format. It\nmay also raise the exception `error:notsup` in case there is no engine support\nin the underlying OpenSSL implementation.","ref":"crypto.html#engine_add/1"},{"type":"function","title":"crypto.engine_by_id/1","doc":"Get a reference to an already loaded engine with `EngineId`. An error tuple is\nreturned if the engine can't be unloaded.\n\nThe function raises a `error:badarg` if the parameter is in wrong format. It may\nalso raise the exception `error:notsup` in case there is no engine support in\nthe underlying OpenSSL implementation.\n\nSee also the chapter [Engine Load](engine_load.md#engine_load) in the User's\nGuide.","ref":"crypto.html#engine_by_id/1"},{"type":"type","title":"crypto.engine_cmnd/0","doc":"Pre and Post commands for [engine_load/3 and /4](`engine_load/3`).","ref":"crypto.html#t:engine_cmnd/0"},{"type":"function","title":"crypto.engine_ctrl_cmd_string/3","doc":"Send ctrl commands to an OpenSSL engine.\n\nThis function is the same as calling `engine_ctrl_cmd_string/4` with `Optional`\nset to `false`.\n\nThe function raises a `error:badarg` if the parameters are in wrong format. It\nmay also raise the exception `error:notsup` in case there is no engine support\nin the underlying OpenSSL implementation.","ref":"crypto.html#engine_ctrl_cmd_string/3"},{"type":"function","title":"crypto.engine_ctrl_cmd_string/4","doc":"Send ctrl commands to an OpenSSL engine.\n\n`Optional` is a\nboolean argument that can relax the semantics of the function. If set to `true`\nit will only return failure if the ENGINE supported the given command name but\nfailed while executing it, if the ENGINE doesn't support the command name it\nwill simply return success without doing anything. In this case we assume the\nuser is only supplying commands specific to the given ENGINE so we set this to\n`false`.\n\nThe function raises a `error:badarg` if the parameters are in wrong format. It\nmay also raise the exception `error:notsup` in case there is no engine support\nin the underlying OpenSSL implementation.","ref":"crypto.html#engine_ctrl_cmd_string/4"},{"type":"function","title":"crypto.engine_get_all_methods/0","doc":"Return a list of all possible engine methods.\n\nMay raise exception `error:notsup` in case there is no engine support in the\nunderlying OpenSSL implementation.\n\nSee also the chapter [Engine Load](engine_load.md#engine_load) in the User's\nGuide.","ref":"crypto.html#engine_get_all_methods/0"},{"type":"function","title":"crypto.engine_get_id/1","doc":"Return the ID for the engine, or an empty binary if there is no id set.\n\nThe function raises a `error:badarg` if the parameters are in wrong format. It\nmay also raise the exception `error:notsup` in case there is no engine support\nin the underlying OpenSSL implementation.","ref":"crypto.html#engine_get_id/1"},{"type":"function","title":"crypto.engine_get_name/1","doc":"Return the name (eg a description) for the engine, or an empty binary if there\nis no name set.\n\nThe function raises a `error:badarg` if the parameters are in wrong format. It\nmay also raise the exception `error:notsup` in case there is no engine support\nin the underlying OpenSSL implementation.","ref":"crypto.html#engine_get_name/1"},{"type":"type","title":"crypto.engine_key_ref/0","doc":"","ref":"crypto.html#t:engine_key_ref/0"},{"type":"function","title":"crypto.engine_list/0","doc":"List the id's of all engines in OpenSSL's internal list.\n\nIt may also raise the exception `error:notsup` in case there is no engine\nsupport in the underlying OpenSSL implementation.\n\nSee also the chapter [Engine Load](engine_load.md#engine_load) in the User's\nGuide.\n\nMay raise exception `error:notsup` in case engine functionality is not supported\nby the underlying OpenSSL implementation.","ref":"crypto.html#engine_list/0"},{"type":"function","title":"crypto.engine_load/3","doc":"Load an OpenSSL engine.\n\nLoads the OpenSSL engine given by `EngineId` if it is available and intialize\nit. Returns `ok` and an engine handle, or if the engine can't be loaded an error\ntuple is returned.\n\nThe function raises a `error:badarg` if the parameters are in wrong format. It\nmay also raise the exception `error:notsup` in case there is no engine support\nin the underlying OpenSSL implementation.\n\nSee also the chapter [Engine Load](engine_load.md#engine_load) in the User's\nGuide.","ref":"crypto.html#engine_load/3"},{"type":"type","title":"crypto.engine_method_type/0","doc":"","ref":"crypto.html#t:engine_method_type/0"},{"type":"type","title":"crypto.engine_ref/0","doc":"The result of a call to `engine_load/3`.","ref":"crypto.html#t:engine_ref/0"},{"type":"function","title":"crypto.engine_register/2","doc":"Register engine to handle some type of methods, for example\nengine_method_digests.\n\nThe function raises a `error:badarg` if the parameters are in wrong format. It\nmay also raise the exception `error:notsup` in case there is no engine support\nin the underlying OpenSSL implementation.","ref":"crypto.html#engine_register/2"},{"type":"function","title":"crypto.engine_remove/1","doc":"Remove the engine from OpenSSL's internal list.\n\nThe function raises a `error:badarg` if the parameters are in wrong format. It\nmay also raise the exception `error:notsup` in case there is no engine support\nin the underlying OpenSSL implementation.","ref":"crypto.html#engine_remove/1"},{"type":"function","title":"crypto.engine_unload/1","doc":"Unload an OpenSSL engine.\n\nUnloads the OpenSSL engine given by `Engine`. An error tuple is returned if the\nengine can't be unloaded.\n\nThe function raises a `error:badarg` if the parameter is in wrong format. It may\nalso raise the exception `error:notsup` in case there is no engine support in\nthe underlying OpenSSL implementation.\n\nSee also the chapter [Engine Load](engine_load.md#engine_load) in the User's\nGuide.","ref":"crypto.html#engine_unload/1"},{"type":"function","title":"crypto.engine_unregister/2","doc":"Unregister engine so it don't handle some type of methods.\n\nThe function raises a `error:badarg` if the parameters are in wrong format. It\nmay also raise the exception `error:notsup` in case there is no engine support\nin the underlying OpenSSL implementation.","ref":"crypto.html#engine_unregister/2"},{"type":"function","title":"crypto.ensure_engine_loaded/2","doc":"Load a dynamic engine if not already done.\n\nLoada the engine given by `EngineId` and the path to the dynamic library\nimplementing the engine. An error tuple is returned if the engine can't be\nloaded.\n\nThis function differs from the normal engine_load in the sense that it also add\nthe engine id to OpenSSL's internal engine list. The difference between the\nfirst call and the following is that the first loads the engine with the\ndynamical engine and the following calls fetch it from the OpenSSL's engine\nlist. All references that is returned are equal.\n\nUse [`engine_unload/1`](`engine_unload/1`) function to remove the references.\nBut remember that [`engine_unload/1`](`engine_unload/1`) just removes the\nreferences to the engine and not the tag in OpenSSL's engine list. That has to\nbe done with the [`engine_remove/1`](`engine_remove/1`) function when needed\n(just called once, from any of the references you got).\n\nThe function raises a `error:badarg` if the parameters are in wrong format. It\nmay also raise the exception `error:notsup` in case there is no engine support\nin the underlying OpenSSL implementation.\n\nSee also the chapter [Engine Load](engine_load.md#engine_load) in the User's\nGuide.","ref":"crypto.html#ensure_engine_loaded/2"},{"type":"function","title":"crypto.exor/2","doc":"Perform bit-wise XOR (exclusive or) on the data supplied.\n\nThe two byte sequences mus be of equal length.","ref":"crypto.html#exor/2"},{"type":"function","title":"crypto.generate_key/2","doc":"","ref":"crypto.html#generate_key/2"},{"type":"function","title":"crypto.generate_key/3","doc":"Generate a public key.\n\nSee also `public_key:generate_key/1`.\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.\n\n> #### Note {: .info }\n>\n> If the linked version of cryptolib is OpenSSL 3.0\n>\n> - and the `Type` is `dh` (diffie-hellman)\n> - and the parameter `P` (in `t:dh_params/0`) is one of the MODP groups (see\n>   [RFC 3526](https://tools.ietf.org/html/rfc3526))\n> - and the optional `PrivateKeyBitLength` parameter (in `t:dh_params/0`) is\n>   present,\n>\n> then the optional key length parameter must be at least 224, 256, 302, 352 and\n> 400 for group sizes of 2048, 3072, 4096, 6144 and 8192, respectively.","ref":"crypto.html#generate_key/3"},{"type":"function","title":"crypto.hash/2","doc":"Compute a message digest.\n\nArgument `Type` is the digest type and argument `Data` is the full message.\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.","ref":"crypto.html#hash/2"},{"type":"type","title":"crypto.hash_algorithm/0","doc":"","ref":"crypto.html#t:hash_algorithm/0"},{"type":"function","title":"crypto.hash_equals/2","doc":"Compare two binaries in constant time, such as results of HMAC computations.\n\nReturns true if the binaries are identical, false if they are of the same length\nbut not identical. The function raises an `error:badarg` exception if the\nbinaries are of different size.","ref":"crypto.html#hash_equals/2"},{"type":"function","title":"crypto.hash_final/1","doc":"Finalize a streaming hash calculation.\n\nArgument `State` as returned from the last call to\n[hash_update](`hash_update/2`). The size of `Digest` is determined by\nthe type of hash function used to generate it.\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.","ref":"crypto.html#hash_final/1"},{"type":"function","title":"crypto.hash_info/1","doc":"Get information about a hash algorithm.\n\nReturns a map with information about block_size, size and possibly other\nproperties of the hash algorithm in question.\n\nFor a list of supported hash algorithms, see [supports(hashs)](`supports/1`).","ref":"crypto.html#hash_info/1"},{"type":"function","title":"crypto.hash_init/1","doc":"Initialize the state for a streaming hash digest calculation.\n\nArgument `Type` determines which digest to use. The returned state should be\nused as argument to `hash_update/2`.\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.","ref":"crypto.html#hash_init/1"},{"type":"opaque","title":"crypto.hash_state/0","doc":"","ref":"crypto.html#t:hash_state/0"},{"type":"function","title":"crypto.hash_update/2","doc":"Add data to a streaming digest calculation.\n\nUpdate the digest using the given `Data` of any length.\n\nArgument `State` must have been generated by [hash_init](`hash_init/1`) or a\nprevious call to this function.\n\nReturns `NewState` that must be passed into the next call to `hash_update/2` or\n`hash_final/1`.\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.","ref":"crypto.html#hash_update/2"},{"type":"function","title":"crypto.hash_xof/3","doc":"Compute a message digest for an `xof_algorithm`.\n\nArgument `Type` is the type of digest, `Data` is the full text and `Length` is\nthe digest length in bits.\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.\n\nMay raise exception `error:notsup` in case the chosen `Type` is not supported by\nthe underlying libcrypto implementation.","ref":"crypto.html#hash_xof/3"},{"type":"type","title":"crypto.hash_xof_algorithm/0","doc":"","ref":"crypto.html#t:hash_xof_algorithm/0"},{"type":"type","title":"crypto.hmac_hash_algorithm/0","doc":"","ref":"crypto.html#t:hmac_hash_algorithm/0"},{"type":"function","title":"crypto.info/0","doc":"Get information about crypto and the OpenSSL backend.\n\nReturns a map with information about the compilation and linking of crypto.\n\nExample:\n\n```erlang\n1> crypto:info().\n#{compile_type => normal,\n  cryptolib_version_compiled => \"OpenSSL 3.0.0 7 sep 2021\",\n  cryptolib_version_linked => \"OpenSSL 3.0.0 7 sep 2021\",\n  link_type => dynamic,\n  otp_crypto_version => \"5.0.2\"}\n2>\n```\n\nMore association types than documented may be present in the map.","ref":"crypto.html#info/0"},{"type":"function","title":"crypto.info_fips/0","doc":"Get information about the operating status of FIPS.\n\nReturns the FIPS operating status of crypto and the underlying libcrypto\nlibrary. If crypto was built with FIPS support this can be either `enabled`\n(when running in FIPS mode) or `not_enabled`. For other builds\nthis value is always `not_supported`.\n\nSee `enable_fips_mode/1` about how to enable FIPS mode.\n\n> #### Warning {: .warning }\n>\n> In FIPS mode all non-FIPS compliant algorithms are disabled and raise\n> exception `error:notsup`. Check [supports(ciphers)](`supports/1`) that in FIPS\n> mode returns the restricted list of available algorithms.","ref":"crypto.html#info_fips/0"},{"type":"function","title":"crypto.info_lib/0","doc":"Get the name and version of the libraries used by crypto.\n\n`Name` is the name of the library. `VerNum` is the numeric version according to\nthe library's own versioning scheme. `VerStr` contains a text variant of the\nversion.\n\n```erlang\n> info_lib().\n[{<<\"OpenSSL\">>,269484095,<<\"OpenSSL 1.1.0c  10 Nov 2016\"\">>}]\n```\n\n> #### Note {: .info }\n>\n> From OTP R16 the _numeric version_ represents the version of the OpenSSL\n> _header files_ (`openssl/opensslv.h`) used when crypto was compiled. The text\n> variant represents the libcrypto library used at runtime. In earlier OTP\n> versions both numeric and text was taken from the library.","ref":"crypto.html#info_lib/0"},{"type":"type","title":"crypto.key_id/0","doc":"Identifies the key to be used. The format depends on the loaded engine. It is\npassed to the `ENGINE_load_(private|public)_key` functions in libcrypto.","ref":"crypto.html#t:key_id/0"},{"type":"type","title":"crypto.key_integer/0","doc":"Always `t:binary/0` when used as return value","ref":"crypto.html#t:key_integer/0"},{"type":"function","title":"crypto.mac/3","doc":"Compute a `poly1305` MAC (Message Authentication Code).\n\nSame as [`mac(Type, undefined, Key, Data)`](`mac/4`).\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.","ref":"crypto.html#mac/3"},{"type":"function","title":"crypto.mac/4","doc":"Compute a MAC (Message Authentication Code).\n\nArgument `Type` is the type of MAC and `Data` is the full message.\n\n`SubType` depends on the MAC `Type`:\n\n- For `hmac` it is a hash algorithm, see\n  [Algorithm Details](algorithm_details.md#hmac) in the User's Guide.\n- For `cmac` it is a cipher suitable for cmac, see\n  [Algorithm Details](algorithm_details.md#cmac) in the User's Guide.\n- For `poly1305` it should be set to `undefined` or the [mac/2](`mac_init/2`)\n  function could be used instead, see\n  [Algorithm Details](algorithm_details.md#poly1305) in the User's Guide.\n\n`Key` is the authentication key with a length according to the `Type` and\n`SubType`. The key length could be found with the `hash_info/1` (`hmac`) for and\n`cipher_info/1` (`cmac`) functions. For `poly1305` the key length is 32 bytes.\nNote that the cryptographic quality of the key is not checked.\n\nThe `Mac` result will have a default length depending on the `Type` and\n`SubType`. To set a shorter length, use `macN/4` or `macN/5` instead. The\ndefault length is documented in\n[Algorithm Details](algorithm_details.md#message-authentication-codes-macs) in\nthe User's Guide.\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.","ref":"crypto.html#mac/4"},{"type":"function","title":"crypto.mac_final/1","doc":"Finalize a streaming MAC operation.\n\nArgument `State` is the state as returned by the last call to `mac_update/2`.\n\nThe `Mac` result will have a default length depending on the `Type` and `SubType` in the\n[mac_init/2,3](`mac_init/3`) call. To set a shorter length, use `mac_finalN/2`\ninstead. The default length is documented in\n[Algorithm Details](algorithm_details.md#message-authentication-codes-macs) in\nthe User's Guide.\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.","ref":"crypto.html#mac_final/1"},{"type":"function","title":"crypto.mac_finalN/2","doc":"Finalize a MAC operation with a custom length.\n\nArgument `State` is the state as returned by the last call to `mac_update/2`.\n\n`Mac` will be a binary with at most `MacLength` bytes. Note that if `MacLength`\nis greater than the actual number of bytes returned from the underlying hash,\nthe returned hash will have that shorter length instead.\n\nThe max `MacLength` is documented in\n[Algorithm Details](algorithm_details.md#message-authentication-codes-macs) in\nthe User's Guide.\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.","ref":"crypto.html#mac_finalN/2"},{"type":"function","title":"crypto.mac_init/2","doc":"Initialize a state for streaming `poly1305` MAC calculation.\n\nSame as [`mac_init(Type, undefined, Key)`](`mac_init/3`).\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.","ref":"crypto.html#mac_init/2"},{"type":"function","title":"crypto.mac_init/3","doc":"Initialize the state for streaming MAC calculation.\n\n`Type` determines which mac algorithm to use in the MAC operation.\n\n`SubType` depends on the MAC `Type`:\n\n- For `hmac` it is a hash algorithm, see\n  [Algorithm Details](algorithm_details.md#hmac) in the User's Guide.\n- For `cmac` it is a cipher suitable for cmac, see\n  [Algorithm Details](algorithm_details.md#cmac) in the User's Guide.\n- For `poly1305` it should be set to `undefined` or the [mac/2](`mac_init/2`)\n  function could be used instead, see\n  [Algorithm Details](algorithm_details.md#poly1305) in the User's Guide.\n\n`Key` is the authentication key with a length according to the `Type` and\n`SubType`. The key length could be found with the `hash_info/1` (`hmac`) for and\n`cipher_info/1` (`cmac`) functions. For `poly1305` the key length is 32 bytes.\nNote that the cryptographic quality of the key is not checked.\n\nThe returned `State` should be used in one or more subsequent calls to\n`mac_update/2`. The MAC value is finally returned by calling `mac_final/1` or\n`mac_finalN/2`.\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.\n\nSee\n[examples in the User's Guide.](new_api.md#example-of-mac_init-mac_update-and-mac_final)","ref":"crypto.html#mac_init/3"},{"type":"opaque","title":"crypto.mac_state/0","doc":"Contexts with an internal state that should not be manipulated but passed\nbetween function calls.","ref":"crypto.html#t:mac_state/0"},{"type":"function","title":"crypto.mac_update/2","doc":"Add data to a streaming MAC calculation.\n\nUpdate the MAC represented by `State0` using the given `Data` which could be of\nany length.\n\nThe `State0` is the State value originally from a MAC init function, that is\n`mac_init/2`, `mac_init/3` or the last call to `mac_update/2`. The value\n`State0` is returned unchanged by the function as a reference to a mutated\ninternal state. Hence, it is not possible to branch off a data stream by reusing\nold states.\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.","ref":"crypto.html#mac_update/2"},{"type":"function","title":"crypto.macN/4","doc":"Compute a `poly1305` MAC (Message Authentication Code) with a limited length.\n\nSame as [`macN(Type, undefined, Key, Data, MacLength)`](`macN/5`).\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.","ref":"crypto.html#macN/4"},{"type":"function","title":"crypto.macN/5","doc":"Compute a MAC (Message Authentication Code) with a limited length.\n\nWorks like `mac/3` and `mac/4` but `MacLength` will limit the size of the\nresultant `Mac` to at most `MacLength` bytes. Note that if `MacLength` is\ngreater than the actual number of bytes returned from the underlying hash, the\nreturned hash will have that shorter length instead.\n\nThe max `MacLength` is documented in\n[Algorithm Details](algorithm_details.md#message-authentication-codes-macs) in\nthe User's Guide.","ref":"crypto.html#macN/5"},{"type":"function","title":"crypto.mod_pow/3","doc":"Compute the function `N^P mod M`.","ref":"crypto.html#mod_pow/3"},{"type":"type","title":"crypto.otp_padding/0","doc":"Erlang/OTP adds a either padding of zeroes or padding with random bytes.","ref":"crypto.html#t:otp_padding/0"},{"type":"type","title":"crypto.padding/0","doc":"This option handles padding in the last block. If not set, no padding is done\nand any bytes in the last unfilled block is silently discarded.","ref":"crypto.html#t:padding/0"},{"type":"type","title":"crypto.password/0","doc":"The password of the key stored in an engine.","ref":"crypto.html#t:password/0"},{"type":"function","title":"crypto.pbkdf2_hmac/5","doc":"PKCS #5 PBKDF2 (Password-Based Key Derivation Function 2) in combination with\nHMAC.\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.","ref":"crypto.html#pbkdf2_hmac/5"},{"type":"type","title":"crypto.pk_encrypt_decrypt_algs/0","doc":"Algorithms for public key encrypt/decrypt. Only RSA is supported.","ref":"crypto.html#t:pk_encrypt_decrypt_algs/0"},{"type":"type","title":"crypto.pk_encrypt_decrypt_opts/0","doc":"","ref":"crypto.html#t:pk_encrypt_decrypt_opts/0"},{"type":"type","title":"crypto.pk_sign_verify_algs/0","doc":"Algorithms for sign and verify.","ref":"crypto.html#t:pk_sign_verify_algs/0"},{"type":"type","title":"crypto.pk_sign_verify_opts/0","doc":"","ref":"crypto.html#t:pk_sign_verify_opts/0"},{"type":"function","title":"crypto.private_decrypt/4","doc":"Decrypt using a private key.\n\nDecrypts the `CipherText`, encrypted with `public_encrypt/4` (or equivalent\nfunction) using the `PrivateKey`, and returns the plaintext (message digest).\nThis is a low level signature verification operation used for instance by older\nversions of the SSL protocol. See also\n[public_key:decrypt_private/2,3](`public_key:decrypt_private/2`)\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.\n\n> #### Warning {: .warning }\n>\n> This is a legacy function, for security reasons do not use with rsa_pkcs1_padding.","ref":"crypto.html#private_decrypt/4"},{"type":"function","title":"crypto.private_encrypt/4","doc":"Encrypt using a private key.\n\nEncrypts the `PlainText` using the `PrivateKey` and returns the ciphertext. This\nis a low level signature operation used for instance by older versions of the\nSSL protocol. See also\n[public_key:encrypt_private/2,3](`public_key:encrypt_private/2`)\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.\n\nPublic-key decryption using the private key. See also `crypto:private_decrypt/4`\n\n> #### Warning {: .warning }\n>\n> This is a legacy function, for security reasons do not use with rsa_pkcs1_padding.\n> For digital signatures use of [`sign/4`](`sign/4`) together\n> with [`verify/5`](`verify/5`) is the prefered solution.","ref":"crypto.html#private_encrypt/4"},{"type":"function","title":"crypto.privkey_to_pubkey/2","doc":"Fetch public key from a private key stored in an Engine.\n\nThe key must be of the type indicated by the Type parameter.","ref":"crypto.html#privkey_to_pubkey/2"},{"type":"function","title":"crypto.public_decrypt/4","doc":"Decrypt using a public key.\n\nDecrypts the `CipherText`, encrypted with `private_encrypt/4`(or equivalent\nfunction) using the `PublicKey`, and returns the plaintext (message digest).\nThis is a low level signature verification operation used for instance by older\nversions of the SSL protocol. See also\n[public_key:decrypt_public/2,3](`public_key:decrypt_public/2`)\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.\n\n> #### Warning {: .warning }\n>\n> This is a legacy function, for security reasons do not use with rsa_pkcs1_padding.\n> For digital signatures use of [`verify/5`](`verify/5`) together\n> with [`sign/4`](`sign/4`) is the prefered solution.","ref":"crypto.html#public_decrypt/4"},{"type":"function","title":"crypto.public_encrypt/4","doc":"Encrypt using a public key.\n\nEncrypts the `PlainText` (message digest) using the `PublicKey` and returns the\n`CipherText`. This is a low level signature operation used for instance by older\nversions of the SSL protocol. See also\n[public_key:encrypt_public/2,3](`public_key:encrypt_public/2`)\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.\n\n> #### Warning {: .warning }\n>\n> This is a legacy function, for security reasons do not use together with rsa_pkcs1_padding.","ref":"crypto.html#public_encrypt/4"},{"type":"type","title":"crypto.rand_cache_seed/0","doc":"","ref":"crypto.html#t:rand_cache_seed/0"},{"type":"function","title":"crypto.rand_seed/0","doc":"Create a state object for [random number generation](`m:rand`), in order to\ngenerate cryptographically strong random numbers (based on OpenSSL's\n`BN_rand_range`).\n\nSaves the state in the process dictionary before returning it as\nwell. See also `rand:seed/1` and `rand_seed_s/0`.\n\nWhen using the state object from this function the `m:rand` functions using it\nmay raise exception `error:low_entropy` in case the random generator failed due\nto lack of secure \"randomness\".\n\n_Example_\n\n```erlang\n_ = crypto:rand_seed(),\n_IntegerValue = rand:uniform(42), % [1; 42]\n_FloatValue = rand:uniform().     % [0.0; 1.0[\n```","ref":"crypto.html#rand_seed/0"},{"type":"function","title":"crypto.rand_seed/1","doc":"Set the seed for PRNG to the given binary.\n\nThis calls the RAND_seed function from openssl. Only use this if the system you\nare running on does not have enough \"randomness\" built in. Normally this is when\n`strong_rand_bytes/1` raises `error:low_entropy`.","ref":"crypto.html#rand_seed/1"},{"type":"function","title":"crypto.rand_seed_alg/1","doc":"Create a state object for [random number generation](`m:rand`), in order to\ngenerate cryptographically strong random numbers.\n\nSaves the state in the process dictionary before returning it as well. See also\n`rand:seed/1` and `rand_seed_alg_s/1`.\n\nWhen using the state object from this function the `m:rand` functions using it\nmay raise exception `error:low_entropy` in case the random generator failed due\nto lack of secure \"randomness\".\n\n_Example_\n\n```erlang\n_ = crypto:rand_seed_alg(crypto_cache),\n_IntegerValue = rand:uniform(42), % [1; 42]\n_FloatValue = rand:uniform().     % [0.0; 1.0[\n```","ref":"crypto.html#rand_seed_alg/1"},{"type":"function","title":"crypto.rand_seed_alg/2","doc":"Creates a state object for [random number generation](`m:rand`), in order to\ngenerate cryptographically unpredictable random numbers.\n\nSaves the state in the process dictionary before returning it as well. See also\n`rand_seed_alg_s/2`.\n\n_Example_\n\n```erlang\n_ = crypto:rand_seed_alg(crypto_aes, \"my seed\"),\nIntegerValue = rand:uniform(42), % [1; 42]\nFloatValue = rand:uniform(),     % [0.0; 1.0[\n_ = crypto:rand_seed_alg(crypto_aes, \"my seed\"),\nIntegerValue = rand:uniform(42), % Same values\nFloatValue = rand:uniform().     % again\n```","ref":"crypto.html#rand_seed_alg/2"},{"type":"function","title":"crypto.rand_seed_alg_s/1","doc":"Create a state object for [random number generation](`m:rand`), in order to\ngenerate cryptographically strongly random numbers.\n\nSee also `rand:seed_s/1`.\n\nIf `Alg` is `crypto` this function behaves exactly like `rand_seed_s/0`.\n\nIf `Alg` is `crypto_cache` this function fetches random data with OpenSSL's\n`RAND_bytes` and caches it for speed using an internal word size of 56 bits that\nmakes calculations fast on 64 bit machines.\n\nWhen using the state object from this function the `m:rand` functions using it\nmay raise exception `error:low_entropy` in case the random generator failed due\nto lack of secure \"randomness\".\n\nThe cache size can be changed from its default value using the\n[crypto app's ](crypto_app.md)configuration parameter `rand_cache_size`.\n\n> #### Note {: .info }\n>\n> The state returned from this function cannot be used to get a reproducible\n> random sequence as from the other `m:rand` functions, since reproducibility\n> does not match cryptographically safe.\n>\n> In fact since random data is cached some numbers may get reproduced if you\n> try, but this is unpredictable.\n>\n> The only supported usage is to generate one distinct random sequence from this\n> start state.","ref":"crypto.html#rand_seed_alg_s/1"},{"type":"function","title":"crypto.rand_seed_alg_s/2","doc":"Create a state object for [random number generation](`m:rand`), in order to\ngenerate cryptographically unpredictable random numbers.\n\nSee also `rand_seed_alg/1`.\n\nTo get a long period the Xoroshiro928 generator from the `m:rand` module is used\nas a counter (with period 2^928 - 1) and the generator states are scrambled\nthrough AES to create 58-bit pseudo random values.\n\nThe result should be statistically completely unpredictable random values, since\nthe scrambling is cryptographically strong and the period is ridiculously long.\nBut the generated numbers are not to be regarded as cryptographically strong\nsince there is no re-keying schedule.\n\n- If you need cryptographically strong random numbers use `rand_seed_alg_s/1`\n  with `Alg =:= crypto` or `Alg =:= crypto_cache`.\n- If you need to be able to repeat the sequence use this function.\n- If you do not need the statistical quality of this function, there are faster\n  algorithms in the `m:rand` module.\n\nThanks to the used generator the state object supports the\n[`rand:jump/0,1`](`rand:jump/0`) function with distance 2^512.\n\nNumbers are generated in batches and cached for speed reasons. The cache size\ncan be changed from its default value using the\n[crypto app's ](crypto_app.md)configuration parameter `rand_cache_size`.","ref":"crypto.html#rand_seed_alg_s/2"},{"type":"function","title":"crypto.rand_seed_s/0","doc":"Create a state object for [random number generation](`m:rand`), in order to\ngenerate cryptographically strongly random numbers (based on OpenSSL's\n`BN_rand_range`). See also `rand:seed_s/1`.\n\nWhen using the state object from this function the `m:rand` functions using it\nmay raise exception `error:low_entropy` in case the random generator failed due\nto lack of secure \"randomness\".\n\n> #### Note {: .info }\n>\n> The state returned from this function cannot be used to get a reproducible\n> random sequence as from the other `m:rand` functions, since reproducibility\n> does not match cryptographically safe.\n>\n> The only supported usage is to generate one distinct random sequence from this\n> start state.","ref":"crypto.html#rand_seed_s/0"},{"type":"function","title":"crypto.rand_uniform/2","doc":"Generate a random integer number.\n\nThe interval is `From =< N < To`. Uses the `crypto` library\npseudo-random number generator. `To` must be larger than `From`.","ref":"crypto.html#rand_uniform/2"},{"type":"type","title":"crypto.rsa_compat_opts/0","doc":"Those option forms are kept only for compatibility and should not be used in new\ncode.","ref":"crypto.html#t:rsa_compat_opts/0"},{"type":"type","title":"crypto.rsa_digest_type/0","doc":"","ref":"crypto.html#t:rsa_digest_type/0"},{"type":"type","title":"crypto.rsa_opt/0","doc":"","ref":"crypto.html#t:rsa_opt/0"},{"type":"type","title":"crypto.rsa_padding/0","doc":"Options for public key encrypt/decrypt. Only RSA is supported.\n\n> #### Warning {: .warning }\n>\n> The RSA options are experimental.\n>\n> The exact set of options and there syntax _may_ be changed without prior\n> notice.","ref":"crypto.html#t:rsa_padding/0"},{"type":"type","title":"crypto.rsa_params/0","doc":"```text\nrsa_public() = [E, N]\n```\n\n```erlang\nrsa_private() = [E, N, D] | [E, N, D, P1, P2, E1, E2, C]\n```\n\nWhere E is the public exponent, N is public modulus and D is the private\nexponent. The longer key format contains redundant information that will make\nthe calculation faster. P1 and P2 are first and second prime factors. E1 and E2\nare first and second exponents. C is the CRT coefficient. The terminology is\ntaken from [RFC 3447](http://www.ietf.org/rfc/rfc3447.txt).","ref":"crypto.html#t:rsa_params/0"},{"type":"type","title":"crypto.rsa_private/0","doc":"","ref":"crypto.html#t:rsa_private/0"},{"type":"type","title":"crypto.rsa_public/0","doc":"","ref":"crypto.html#t:rsa_public/0"},{"type":"type","title":"crypto.rsa_sign_verify_opt/0","doc":"","ref":"crypto.html#t:rsa_sign_verify_opt/0"},{"type":"type","title":"crypto.rsa_sign_verify_padding/0","doc":"Options for sign and verify.\n\n> #### Warning {: .warning }\n>\n> The RSA options are experimental.\n>\n> The exact set of options and there syntax _may_ be changed without prior\n> notice.","ref":"crypto.html#t:rsa_sign_verify_padding/0"},{"type":"type","title":"crypto.sha1/0","doc":"","ref":"crypto.html#t:sha1/0"},{"type":"type","title":"crypto.sha2/0","doc":"","ref":"crypto.html#t:sha2/0"},{"type":"type","title":"crypto.sha3/0","doc":"","ref":"crypto.html#t:sha3/0"},{"type":"type","title":"crypto.sha3_xof/0","doc":"","ref":"crypto.html#t:sha3_xof/0"},{"type":"function","title":"crypto.sign/4","doc":"","ref":"crypto.html#sign/4"},{"type":"function","title":"crypto.sign/5","doc":"Create a digital signature.\n\nThe msg is either the binary \"cleartext\" data to be signed or it is the hashed\nvalue of \"cleartext\" i.e. the digest (plaintext).\n\nAlgorithm `dss` can only be used together with digest type `sha`.\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.\n\nSee also `public_key:sign/3`.","ref":"crypto.html#sign/5"},{"type":"type","title":"crypto.srp_comp_params/0","doc":"","ref":"crypto.html#t:srp_comp_params/0"},{"type":"type","title":"crypto.srp_gen_params/0","doc":"","ref":"crypto.html#t:srp_gen_params/0"},{"type":"type","title":"crypto.srp_host_comp_params/0","doc":"Where Verifier is `v`, Generator is `g` and Prime is` N`, DerivedKey is `X`, and\nScrambler is `u` (optional will be generated if not provided) from\n[SRP design](http://srp.stanford.edu/design.html) Version = '3' | '6' | '6a'","ref":"crypto.html#t:srp_host_comp_params/0"},{"type":"type","title":"crypto.srp_host_gen_params/0","doc":"","ref":"crypto.html#t:srp_host_gen_params/0"},{"type":"type","title":"crypto.srp_private/0","doc":"```text\nsrp_public() = key_integer()\n```\n\nWhere is `A` or `B` from [SRP design](http://srp.stanford.edu/design.html)\n\n```text\nsrp_private() = key_integer()\n```\n\nWhere is `a` or `b` from [SRP design](http://srp.stanford.edu/design.html)","ref":"crypto.html#t:srp_private/0"},{"type":"type","title":"crypto.srp_public/0","doc":"","ref":"crypto.html#t:srp_public/0"},{"type":"type","title":"crypto.srp_user_comp_params/0","doc":"","ref":"crypto.html#t:srp_user_comp_params/0"},{"type":"type","title":"crypto.srp_user_gen_params/0","doc":"","ref":"crypto.html#t:srp_user_gen_params/0"},{"type":"function","title":"crypto.start/0","doc":"Use [`application:start(crypto)`](`application:start/1`) instead.\n\n> #### Warning {: .warning }\n>\n> This function does not work if FIPS mode is to be enabled. FIPS mode will be\n> disabled even if configuration parameter `fips_mode` is set to `true`. Use\n> [`application:start(crypto)`](`application:start/1`) instead.","ref":"crypto.html#start/0"},{"type":"function","title":"crypto.stop/0","doc":"Use [`application:stop(crypto)`](`application:stop/1`) instead.","ref":"crypto.html#stop/0"},{"type":"function","title":"crypto.strong_rand_bytes/1","doc":"Generate bytes with randomly uniform values 0..255.\n\nReturns the result in a binary with `N` bytes.\n\nUses a cryptographically secure PRNG seeded and periodically mixed with\noperating system provided entropy. By default this is the `RAND_bytes` method\nfrom OpenSSL.\n\nMay raise exception `error:low_entropy` in case the random generator failed due\nto lack of secure \"randomness\".","ref":"crypto.html#strong_rand_bytes/1"},{"type":"function","title":"crypto.supports/1","doc":"Get which crypto algorithms that are supported by the underlying libcrypto\nlibrary.\n\nSee `hash_info/1` and `cipher_info/1` for information about the hash and cipher\nalgorithms.","ref":"crypto.html#supports/1"},{"type":"function","title":"crypto.verify/5","doc":"","ref":"crypto.html#verify/5"},{"type":"function","title":"crypto.verify/6","doc":"Verify a digital signature.\n\nThe msg is either the binary \"cleartext\" data to be signed or it is the hashed\nvalue of \"cleartext\" i.e. the digest (plaintext).\n\nAlgorithm `dss` can only be used together with digest type `sha`.\n\nUses the [3-tuple style](`m:crypto#error_3tup`) for error handling.\n\nSee also `public_key:verify/4`.","ref":"crypto.html#verify/6"},{"type":"extras","title":"Crypto Application","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Crypto Application","ref":"crypto_app.html"},{"type":"extras","title":"Description - Crypto Application","doc":"The purpose of the Crypto application is to provide an Erlang API to\ncryptographic functions, see `m:crypto`. Note that the API is on a fairly low\nlevel and there are some corresponding API functions available in\n`m:public_key`, on a higher abstraction level, that uses the crypto application\nin its implementation.","ref":"crypto_app.html#description"},{"type":"extras","title":"DEPENDENCIES - Crypto Application","doc":"The current crypto implementation uses nifs to interface OpenSSLs crypto library\nand may work with limited functionality with as old versions as _OpenSSL_\n0\\.9.8c. FIPS mode support requires at least version 1.0.1 and a FIPS capable\nOpenSSL installation. We recommend using a version that is officially supported\nby the OpenSSL project. API compatible backends like LibreSSL should also work.\n\nThe crypto app is tested daily with at least one version of each of the OpenSSL\n1.0.1, 1.0.2, 1.1.0, 1.1.1 and 3.0. FIPS mode is also tested for 1.0.1, 1.0.2\nand 3.0.\n\nUsing OpenSSL 3.0 with Engines is supported since OTP 26.2.\n\nSource releases of OpenSSL can be downloaded from the\n[OpenSSL](http://www.openssl.org) project home page, or mirror sites listed\nthere.","ref":"crypto_app.html#dependencies"},{"type":"extras","title":"CONFIGURATION - Crypto Application","doc":"The following configuration parameters are defined for the crypto application.\nSee [`app(3)`](`e:kernel:app.md`) for more information about configuration\nparameters.\n\n- **`fips_mode = boolean()`** - Specifies whether to run crypto in FIPS mode.\n  This setting will take effect when the nif module is loaded. If FIPS mode is\n  requested but not available at run time the nif module and thus the crypto\n  module will fail to load. This mechanism prevents the accidental use of\n  non-validated algorithms.\n\n- **`rand_cache_size = integer()`** - Sets the cache size in bytes to use by\n  [`crypto:rand_seed_alg(crypto_cache)` ](`crypto:rand_seed_alg/1`)and\n  [`crypto:rand_seed_alg_s(crypto_cache)` ](`crypto:rand_seed_alg_s/1`). This\n  parameter is read when a seed function is called, and then kept in generators\n  state object. It has a rather small default value that causes reads of strong\n  random bytes about once per hundred calls for a random value. The set value is\n  rounded up to an integral number of words of the size these seed functions\n  use.","ref":"crypto_app.html#configuration"},{"type":"extras","title":"SEE ALSO - Crypto Application","doc":"application(3)","ref":"crypto_app.html#see-also"},{"type":"extras","title":"Crypto Release Notes","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Crypto Release Notes\n\nThis document describes the changes made to the Crypto application.","ref":"notes.html"},{"type":"extras","title":"Crypto 5.5.3 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-5-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- `crypto` will now work when ED25519 and X25519 are available while ED448 and X448 are not, which is the case in LibreSSL. This is necessary for supporting TLS1.3 using LibreSSL.\n\n  Own Id: OTP-19399 Aux Id: [GH-9000], [PR-9136]\n\n[GH-9000]: https://github.com/erlang/otp/issues/9000\n[PR-9136]: https://github.com/erlang/otp/pull/9136","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 5.5.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-5-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- `crypto:strong_rand_bytes/2` fixed to work on Ubuntu pro with installed FIPS support.\n\n  Own Id: OTP-19223 Aux Id: [PR-8800], [GH-8769]\n\n- Fixed `crypto:hash_final/1` for digest types `shake128` and `shake256` when using OpenSSL 3.4 or newer.\n\n  Own Id: OTP-19329 Aux Id: [GH-8997], [PR-9002]\n\n[PR-8800]: https://github.com/erlang/otp/pull/8800\n[GH-8769]: https://github.com/erlang/otp/issues/8769\n[GH-8997]: https://github.com/erlang/otp/issues/8997\n[PR-9002]: https://github.com/erlang/otp/pull/9002","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 5.5.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-5-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- crypto built with `--enable-fips` will now accept an OpenSSL 3 lib without fips provider as long as fips mode is not enabled.\n\n  Own Id: OTP-19212 Aux Id: [GH-8562]\n\n[GH-8562]: https://github.com/erlang/otp/issues/8562","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Added a warning in the documentation to avoid calling `crypto:start/0` as it does not work for FIPS mode. Use `application:start(crypto)` instead.\n\n  Own Id: OTP-19143\n\n- Deprecation of RSA encryption functions has been reverted, as there still exists legitimate use cases with other padding modes than PKCS-1.\n  \n  While use PCKS-1 padding with some versions of cryptolib could be considered secure, we still recommend using other algorithms that are less sensitive to oracle attacks.\n\n  Own Id: OTP-19163\n\n- Compiler warnings for some removed functions have been corrected to point out the correct replacement functions.\n\n  Own Id: OTP-19186 Aux Id: [PR-8709]\n\n[PR-8709]: https://github.com/erlang/otp/pull/8709","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 5.5 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-5"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- The documentation has been migrated to use Markdown and ExDoc.\n\n  Own Id: OTP-18955 Aux Id: [PR-8026]\n\n- Removed functions `crypto_dyn_iv_init/3` and `crypto_dyn_iv_update/3` which were marked as deprecated since OTP 25.\n\n  Own Id: OTP-18973\n\n- Add support for sm3 hash and hmac.\n\n  Own Id: OTP-18975 Aux Id: [PR-6658]\n\n- `OPENSSL_thread_stop`  is called when `crypto` is purged to not leak thread specific data.\n\n  Own Id: OTP-18978 Aux Id: [PR-7809]\n\n- Add SM4 block cipher implemented according to GB/T 32907-2016.\n\n  Own Id: OTP-19005 Aux Id: [PR-8168]\n\n- The existing function `ssl:key_exporter_materials/4` is now documented and supported.\n\n  *** POTENTIAL INCOMPATIBILITY ***\n\n  Own Id: OTP-19016 Aux Id: [PR-8233]\n\n- Due to another attack on PKCS #1 v1.5 padding, known as the Marvin attack, about which we were alerted by Hubert Kario from Red Hat. You can find more details about the attack at\n  https://people.redhat.com/~hkario/marvin/\n  Functions that may be vulnerable are now deprecated. \n  \n  Note that you might mitigate the problem \n  by using appropriate versions of OpenSSL together with our software, but we recommend not using them at all. \n  \n  Also avoid using TLS versions prior to TLS-1.2 (not supported by default) and\n  do not enable RSA-key exchange cipher suites (not supported by default).\n\n  *** POTENTIAL INCOMPATIBILITY ***\n\n  Own Id: OTP-19075\n\n[PR-8026]: https://github.com/erlang/otp/pull/8026\n[PR-6658]: https://github.com/erlang/otp/pull/6658\n[PR-7809]: https://github.com/erlang/otp/pull/7809\n[PR-8168]: https://github.com/erlang/otp/pull/8168\n[PR-8233]: https://github.com/erlang/otp/pull/8233","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 5.4.2.3 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-4-2-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"* Fixed bug that could cause blocking scheduler threads in `crypto:supports(curves)` if called the first time by more than one Erlang process. Bug exists only in OTP-26.2.5.4.\n\n  Own Id: OTP-19316 Aux Id: GH-8971, PR-8979\n* Fixed `crypto:hash_final/1` for digest types `shake128` and `shake256` when using OpenSSL 3.4 or newer.\n\n  Own Id: OTP-19329 Aux Id: GH-8997, PR-9002","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 5.4.2.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-4-2-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"* crypto built with `--enable-fips` will now accept an OpenSSL 3 lib without fips provider as long as fips mode is not enabled.\n\n  Own Id: OTP-19212 Aux Id: GH-8562, PR-8762\n* `crypto:strong_rand_bytes/2` fixed to work on Ubuntu pro with installed FIPS support.\n\n  Own Id: OTP-19223 Aux Id: PR-8800, GH-8769","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 5.4.2.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-4-2-1"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"* Add warning in documentation to avoid `crypto:start/0` as it does not work for FIPS mode. Use `application:start(crypto)` instead.\n\n  Own Id: OTP-19143","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 5.4.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-4-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"* Fix building with `--enable-fips` with OpenSSL 3 on MacOS.\n\n  Own Id: OTP-19038 Aux Id: GH-8271, PR-8277","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 5.4.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-4-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"* Fix compile error when OPENSSL_NO_DES is defined.\n\n  Own Id: OTP-18921\n* The function `crypto:pbkdf2_hmac` will no longer block the main schedulers. If the iteration count or block size parameters are such that the function is likely to take a long time to execute, the function will be scheduled to run on a dirty CPU scheduler.\n\n  Own Id: OTP-18996 Aux Id: PR-8173, PR-8174","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 5.4 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fixed some benign compile warnings on Windows.\n\n  Own Id: OTP-18895","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Enable engine support for OpenSSL versions 3.\n\n  Own Id: OTP-18832 Aux Id: PR-7763","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 5.3 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fix VM crash caused by `crypto being` purged and reloaded (by init:restart for\n  example) on OS with musl libc (such as Alpine linux).\n\n  Own Id: OTP-18670 Aux Id: GH-7436, PR-7450\n\n- Improved understanding of LibreSSL versions. Support chacha20 and\n  chacha20_poly1305 for LibreSSL 3.7. Reflect removal of support for the DSS/DSA\n  algorithm which was done in LibreSSL 2.6.1.\n\n  Own Id: OTP-18758 Aux Id: PR-7209","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- FIPS supported by `crypto` for OpenSSL 3.0._ and 3.1._.\n\n  Own Id: OTP-18666 Aux Id: PR-7392","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 5.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fix cmac_update aes_128_cbc for LibreSSL.\n\n  Own Id: OTP-18571","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Add support for SHAKE128 and SHAKE256.\n\n  Own Id: OTP-18204 Aux Id: PR-6203\n\n- Make the `-DOPENSSL_API_COMPAT` flag work without warnings.\n\n  Own Id: OTP-18206 Aux Id: PR-6167\n\n- Replace size/1 with either tuple_size/1 or byte_size/1\n\n  The [`size/1`](`size/1`) BIF is not optimized by the JIT, and its use can\n  result in worse types for Dialyzer.\n\n  When one knows that the value being tested must be a tuple,\n  [`tuple_size/1`](`tuple_size/1`) should always be preferred.\n\n  When one knows that the value being tested must be a binary,\n  [`byte_size/1`](`byte_size/1`) should be preferred. However,\n  [`byte_size/1`](`byte_size/1`) also accepts a bitstring (rounding up size to a\n  whole number of bytes), so one must make sure that the call to `byte_size/` is\n  preceded by a call to [`is_binary/1`](`is_binary/1`) to ensure that bitstrings\n  are rejected. Note that the compiler removes redundant calls to\n  [`is_binary/1`](`is_binary/1`), so if one is not sure whether previous code\n  had made sure that the argument is a binary, it does not harm to add an\n  [`is_binary/1`](`is_binary/1`) test immediately before the call to\n  [`byte_size/1`](`byte_size/1`).\n\n  Own Id: OTP-18405 Aux Id:\n  GH-6672,PR-6702,PR-6768,PR-6700,PR-6769,PR-6812,PR-6814\n\n- Handling of `on_load` modules during boot has been improved by adding an extra\n  step in the boot order for embedded mode that runs all `on_load` handlers,\n  instead of relying on explicit invocation of them, later, when the kernel\n  supervision tree starts.\n\n  This is mostly a code improvement and OTP internal simplification to avoid\n  future bugs and to simplify code maintenance.\n\n  Own Id: OTP-18447","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 5.1.4.3 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-1-4-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"* Fix building with `--enable-fips` with OpenSSL 3 on MacOS.\n\n  Own Id: OTP-19038 Aux Id: GH-8271, PR-8277","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 5.1.4.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-1-4-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"* The function `crypto:pbkdf2_hmac` will no longer block the main schedulers. If the iteration count or block size parameters are such that the function is likely to take a long time to execute, the function will be scheduled to run on a dirty CPU scheduler.\n\n  Own Id: OTP-18996 Aux Id: PR-8173, PR-8174","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 5.1.4.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-1-4-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fix VM crash caused by `crypto being` purged and reloaded (by init:restart for\n  example) on OS with musl libc (such as Alpine linux).\n\n  Own Id: OTP-18670 Aux Id: GH-7436, PR-7450","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 5.1.4 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-1-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- With this change, random errors are fixed for crypto:generate_key calls with\n  OpenSSL 3.\n\n  Own Id: OTP-18555","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 5.1.3 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-1-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- A user defined runtime library path configured using\n  `--with-ssl-rpath= ` could fail to be enabled.\n\n  Own Id: OTP-18384 Aux Id: PR-6596\n\n- Ensure that `configure` fails if a user defined runtime library path has been\n  passed by the user, but cannot set.\n\n  Own Id: OTP-18408","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 5.1.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-1-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fix configure with `--with-ssl` and `--disable-dynamic-ssl-lib` on Windows.\n\n  Own Id: OTP-18147 Aux Id: GH-6024, PR-6056\n\n- Remove all references correctly in the garbage collection if an engine handle\n  was not explicit unloaded.\n\n  Own Id: OTP-18152\n\n- Changed the behaviour of the engine load/unload functions\n\n  The engine load/unload functions have got changed semantics to get a more\n  consistent behaviour and work correct when variables are garbage collected.\n\n  The load functions now don't register the methods for the engine to replace.\n  That will now be handled with the new functions\n  engine_register/engine_unregister if needed.\n\n  Some functions are removed from the documentation and therefor the API, but\n  they are left in the code for compatibility.\n\n  - engine_load/4: is now the same as engine_load/3\n  - engine_unload/2: is now the same as engine_unload/1\n  - ensure_engine_loaded/3: is now the same as ensure_engine_loaded/2\n  - ensure_engine_unloaded/1, ensure_engine_unloaded/2: is now the same as\n    engine_unload/1\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-18172 Aux Id: ERIERL-826\n\n- Fixed a naming bug for AES-CFB and Blowfish-CFB/OFB when linked with OpenSSL\n  3.0 cryptolib.\n\n  Own Id: OTP-18200\n\n- Sign/verify does now behave as in OTP-24 and earlier for eddsa.\n\n  Own Id: OTP-18205 Aux Id: GH-6219","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Pass elliptic curve names from crypto.erl to crypto's nif.\n\n  Own Id: OTP-18037\n\n- The configure option `--disable-deprecated-warnings` is removed. It was used\n  for some releases when the support for OpenSSL 3.0 was not completed. It is\n  not needed in OTP 25.\n\n  Own Id: OTP-18133\n\n- Crypto is now considered to be usable with the OpenSSL 3.0 cryptolib for\n  production code.\n\n  ENGINE and FIPS are not yet fully functional.\n\n  Own Id: OTP-18153\n\n- Do not exit if the legacy provider is missing in libcrypto 3.0.\n\n  Own Id: OTP-18217","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 5.1.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-1-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Note in the documentation that MODP (rfc3526) groups and OpenSSL 3.0 can give\n  an error if a call to `crypto:generate_key/2` specifies a key length, and that\n  length is to small.\n\n  Own Id: OTP-18046","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- The cmac now uses only the 3.0 API\n\n  Own Id: OTP-18010\n\n- Documentation is now updated with which OpenSSL cryptolib versions that OTP\n  currently is tested.\n\n  Own Id: OTP-18132","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 5.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fix timing bug in ensure_engine_loaded\n\n  When two ensure_engine_loaded() calls were done in parallel there was a\n  possibility that a crypto lib function was called by both instead of just one\n  of them which resulted in an error. This is solved by moving the\n  implementation from erlang down into a NIF function that uses a mutex to\n  protect the sensitive part.\n\n  Own Id: OTP-17858 Aux Id: ERIERL-728\n\n- Remove faulty types `run_time_error()` and `descriptive_error()`.\n\n  Own Id: OTP-17984","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Input for `configure` scripts adapted to `autoconf` 2\\.71.\n\n  Own Id: OTP-17414 Aux Id: PR-4967\n\n- Add crypto:hash_equals/2\n\n  Own Id: OTP-17471 Aux Id: PR-4750\n\n- Add /opt/homebrew/opt/openssl to standard locations to search for OpenSSL\n  cryptolib.\n\n  Own Id: OTP-17561\n\n- `crypto_dyn_iv_init/3` and `crypto_dyn_iv_update/3` are deprecated.\n\n  Own Id: OTP-17870\n\n- The information in error messages are increased.\n\n  Previously the error was signaled with en error class exception `badarg`,\n  `notsup` or `error`, and also in some more ways like an other exception or a\n  return value in a non-standardized format.\n\n  Now it is an error-class exception\n  `{notsup|badarg|error, InfoFromCfile, Description::string()}`.\n\n  The `InfoFromCfile` is a term with name and line number of the C-file where\n  the error was found. This is primarily intended for a crypto maintainer or an\n  advanced user to find the cause of complicated errors - maybe in crypto\n  itself. The contents of that term might be changed in the future.\n\n  The `Description` is a clear text string that describes the error. In case of\n  `badarg` and `notsup` the intention is that it should help the user to find\n  the cause (\"Bad key size\" as an example). Specially for some `error` that are\n  unlikely, the string may not be possible to understand without deep knowledge\n  of the underlying cryptolib. Such messages are intended for a crypto\n  maintainer.\n\n  The first element on call stack (the `S` in\n  `try ... catch error:E:S .... end`) gives more information like the actual\n  argument list in the call of crypto and the argument number (if possible) in\n  the call to the NIF inside crypto.\n\n  The functions in crypto affected by this change are:\n\n  sign/4, sign/5, verify/5, verify/6,\n\n  generate_key/2, generate_key/3, compute_key/4,\n\n  hash/2, hash/4, hash_init/1, hash_update/4, hash_final/1,\n\n  mac/3,4, mac_init/3, mac_update/2, mac_final/2,\n\n  pbkdf2_hmac/5,\n\n  public_encrypt/4, private_decrypt/4, private_encrypt/4, public_decrypt/4\n\n  This schema was introduced earlier in:\n\n  crypto_init/3, crypto_init/4, crypto_update/2, crypto_final/1,\n  crypto_get_data/1,\n\n  crypto_one_time/4, crypto_one_time/5, crypto_one_time_aead/6,\n  crypto_one_time_aead/7\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-17965\n\n- Add Output Feedback mode (OFB) support for AES encryption / decryption for key\n  sizes of 128, 192 and 256 bits.\n\n  Own Id: OTP-18067 Aux Id: PR-5866\n\n- The cryptolib API deprecated in OpenSSL 3.0 is now no longer used with a few\n  exceptions listed below.\n\n  Although OpenSSL 3.0.x itself is stable, its usage in OTP/crypto should still\n  not be considered suitable for production code.\n\n  The use of ENGINEs is still disabled by default when using 3.0.\n\n  Deprecated functions are still called in the otp_test_engine.c (only used in\n  tests), in mac.c (EVP_PKEY_new_CMAC_key) and five function calls in ec.c\n  (EVP_PKEY_assign, EC_KEY_get_conv_form, EVP_PKEY_get1_EC_KEY,\n  EC_KEY_get0_group and EC_KEY_set_public_key).\n\n  Own Id: OTP-18086 Aux Id: OTP-16282, OTP-16643, OTP-16644, OTP-17701,\n  OTP-17702, OTP-17704","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 5.0.6.5 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-0-6-5"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"* The function `crypto:pbkdf2_hmac` will no longer block the main schedulers. If the iteration count or block size parameters are such that the function is likely to take a long time to execute, the function will be scheduled to run on a dirty CPU scheduler.\n\n  Own Id: OTP-18996 Aux Id: PR-8173, PR-8174","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 5.0.6.4 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-0-6-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fix VM crash caused by `crypto being` purged and reloaded (by init:restart for\n  example) on OS with musl libc (such as Alpine linux).\n\n  Own Id: OTP-18670 Aux Id: GH-7436, PR-7450","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 5.0.6.3 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-0-6-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Changed the behaviour of the engine load/unload functions\n\n  The engine load/unload functions have got changed semantics to get a more\n  consistent behaviour and work correct when variables are garbage collected.\n\n  The load functions now don't register the methods for the engine to replace.\n  That will now be handled with the new functions\n  engine_register/engine_unregister if needed.\n\n  Some functions functions are removed from the documentation and therefor the\n  API, but they are left in the code for compatibility.\n\n  - engine_load/4: is now the same as engine_load/3\n  - engine_unload/2: is now the same as engine_unload/1\n  - ensure_engine_loaded/3: is now the same as ensure_engine_loaded/2\n  - ensure_engine_unloaded/1, ensure_engine_unloaded/2: is now the same as\n    engine_unload/1\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-18172 Aux Id: ERIERL-826","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 5.0.6.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-0-6-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fix configure with `--with-ssl` and `--disable-dynamic-ssl-lib` on Windows.\n\n  Own Id: OTP-18147 Aux Id: GH-6024, PR-6056","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 5.0.6.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-0-6-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fix timing bug in ensure_engine_loaded\n\n  When two ensure_engine_loaded() calls were done in parallel there was a\n  possibility that a crypto lib function was called by both instead of just one\n  of them which resulted in an error. This is solved by moving the\n  implementation from erlang down into a NIF function that uses a mutex to\n  protect the sensitive part.\n\n  Own Id: OTP-17858 Aux Id: ERIERL-728","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 5.0.6 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-0-6"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- The crypto app in OTP can since OTP-24.2 be compiled, linked and used with the\n  new OpenSSL 3.0 cryptolib.\n\n  The crypto app has 3.0 support has been improved, but is still \\*\\*not\n  recommended\\*\\* for other usages than experiments and alpha testing. There are\n  not yet any guaranties that it works, not even together with other OTP\n  applications like for example SSL and SSH, although there are no known errors.\n\n  Since the previous release, OTP-24.2, the following improvements have been\n  done:\n\n  \\- It has been tested during nearly every nightly test on the OTP lab\n\n  \\- The hash algorithms `md4` and `ripemd160` have been enabled with OpenSSL\n  3.0.\n\n  \\- The ciphers `blowfish_cbc`, `blowfish_ecb`, `des_cbc`, `des_cfb`,\n  `des_ecb`, `rc2_cbc` and `rc4` have been enabled with OpenSSL 3.0.\n\n  Disabled or unsupported with OpenSSL 3.0 are still:\n\n  \\- ENGINE support\n\n  \\- FIPS mode\n\n  \\- Other providers than the built-in ones\n\n  \\- Compiling and linking with OpenSSL 3.0 cryptolib in compatibility modes\n  (for example to behave as 1.1.1)\n\n  and, the ciphers `blowfish_cfb64` and `blowfish_ofb64` are not supported and\n  will not be either.\n\n  Deprecated functions in the OpenSSL 3.0 cryptolib must not be disabled as\n  OTP/crypto still uses some of the deprecated API functions. The gcc flag\n  `-Wno-deprecated-declarations` is set to prevent deprecation warnings to be\n  printed when compiling.\n\n  Own Id: OTP-17812 Aux Id: OTP-16646, OTP-16282\n\n- Crypto is adapted to LibreSSL 3.5.0 on OpenBSD.\n\n  Own Id: OTP-17941 Aux Id: OTP-17942\n\n- New configure option ( `--disable-otp-test-engine`) to prohibit the build of\n  the OTP test engine used in some test suites.\n\n  The reason is that the test engine could be hard to compile on for instance\n  LibreSSL 3.5.0. For that particular cryptolib version (or higher), this\n  configure option is set automatically.\n\n  Own Id: OTP-17942 Aux Id: OTP-17941","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 5.0.5 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-0-5"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fixed the C-warning \"implicit declaration of function 'OpenSSL_version_num'\"\n  if compiling with an early LibreSSL version.\n\n  Own Id: OTP-17637\n\n- FIPS availability was not checked correctly for AEAD ciphers.\n\n  Own Id: OTP-17740\n\n- Fixed that cipher aliases (like aes_cbc etc) could be present even if the\n  aliased cipher(s) (like aes_128_cbc, aes_256_cbc,... etc) was missing.\n\n  Own Id: OTP-17741","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- The crypto app in OTP can now be compiled, linked and used with the new\n  OpenSSL 3.0 cryptolib.\n\n  It has not yet been extensively tested and is in this release \\*\\*not\n  recommended\\*\\* for other usages than experiments and alpha testing. There are\n  not yet any guaranties that it works, not even together with other OTP\n  applications like for example SSL and SSH, although there are no known errors.\n\n  Compiling and linking with OpenSSL 3.0 cryptolib in compatibility modes (for\n  example to behave as 1.1.1) are not tested. It is not tested with external\n  providers.\n\n  The support for FIPS mode does not yet work, and is disabled when compiled\n  with OpenSSL 3.0.\n\n  Deprecated functions in the OpenSSL 3.0 cryptolib must not be disabled as\n  OTP/crypto still uses some of the deprecated API functions. The gcc flag\n  `-Wno-deprecated-declarations` is set to prevent deprecation warnings to be\n  printed when compiling.\n\n  The hash algorithms `md4` and `ripemd160` are disabled temporarily when\n  compiled with OpenSSL 3.0.\n\n  The ciphers `blowfish_cbc`, `blowfish_cfb64`, `blowfish_ecb`,\n  `blowfish_ofb64`, `des_cbc`, `des_cfb`, `des_ecb`, `rc2_cbc` and `rc4` are\n  disabled temporarily when compiled with OpenSSL 3.0.\n\n  Own Id: OTP-16646 Aux Id: OTP-16282\n\n- The error handling in crypto is partly refactored using the new error\n  reporting support. Errors earlier propagated like exceptions are still so, but\n  when the failing function is called from the terminal - for example during\n  failure hunting - a more descriptive text is produced.\n\n  Own Id: OTP-17241\n\n- A new function `crypto:info/0` which presents some data about the compilation\n  and linkage of the crypto nif is added.\n\n  Own Id: OTP-17603\n\n- Added the `pbkdf2_hmac/5` function to the crypto module.\n\n  It calls the `PKCS5_PBKDF2_HMAC` function which implements PBKD2 with HMAC in\n  an efficient way.\n\n  Own Id: OTP-17808 Aux Id: PR-5421","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 5.0.4 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-0-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fixed minor memory leak at `crypto` module purge.\n\n  Own Id: OTP-17668 Aux Id: PR-5245\n\n- Fix possible inconsistency in fips mode when linking with some cryptolibs.\n\n  Own Id: OTP-17672","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 5.0.3 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-0-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fix bug in `crypto:ensure_engine_unloaded`. Also fixed minor memory leak\n  related to engine unloading.\n\n  Own Id: OTP-17593 Aux Id: ERIERL-679\n\n- Fixes that FIPS enable and disable (or vice versa) on Windows sometimes leads\n  to core dump at the time of process exit.\n\n  Own Id: OTP-17618 Aux Id: PR-5126, GH-4920","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Disable fips if cryptolib < 1.0.1 and OTP/crypto is configured with\n  --enable-fips\n\n  If not, there could be compiling or loading problems with antique OpenSSL\n  versions.\n\n  Own Id: OTP-17389","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 5.0.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-0-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- EC keys are now zero-padded to the expected length if needed.\n\n  Own Id: OTP-17442 Aux Id: GH-4861","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 5.0.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-0-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Removed a risk for coredump.\n\n  Own Id: OTP-17391 Aux Id: GH-4810\n\n- Fixed and documented the `DED_LDFLAGS_CONFTEST` configuration variable in\n  `$ERL_TOP/HOWTO/INSTALL.md`.\n\n  Own Id: OTP-17419 Aux Id: GH-4821\n\n- Commit of generated `configure` script.\n\n  Own Id: OTP-17420 Aux Id: OTP-17398, GH-4821","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 5.0 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-5-0"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Add `/usr/local/opt/openssl` to the openssl configure search path. This path\n  is where some tools on OS X place openssl.\n\n  Own Id: OTP-16882\n\n- Fix compiler warnings produced by the clang compiler.\n\n  Own Id: OTP-17105 Aux Id: PR-2872\n\n- The `configure` scripts in `crypto` and `erts` now fail if a requested feature\n  cannot be enabled.\n\n  Large parts of the `configure` script of `crypto` have been rewritten with\n  various improvements and bug fixes. It is now better at finding usable OpenSSL\n  libraries, but will in the following cases fail to detect OpenSSL libraries\n  where it previously sometimes detected the libraries by chance:\n\n  - OpenSSL installations with `include` directory and `lib` directory parts\n    installed in different base directories. In order to detect such\n    installations after this change, the user must explicitly specify the\n    locations using the\n    [`--with-ssl= `](`e:system:install.md#advanced-configuration-and-build-of-erlang-otp_configuring`)\n    and the\n    [`--with-ssl-incl= `](`e:system:install.md#advanced-configuration-and-build-of-erlang-otp_configuring`)\n    `configure` command line arguments.\n  - When building with old `gcc` compilers or other compilers on Debian\n    derivatives with multiarch directories under the `lib` directory. In order\n    to detect such installations after this change, the user must explicitly\n    specify the multiarch directory name using the\n    [`--with-ssl-lib-subdir=lib/ `](`e:system:install.md#advanced-configuration-and-build-of-erlang-otp_configuring`)\n    `configure` command line argument.\n\n  Own Id: OTP-17254 Aux Id: ERIERL-618, GH-4230\n\n- The value 'none' was missing in the specs of crypto:sign/4 and\n  crypto:verify/6.\n\n  Own Id: OTP-17312 Aux Id: PR-4723","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- The functions and cipher names that were deprecated in OTP-23.0 are now\n  removed.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-16656\n\n- Removed installed directory priv/obj/ containing superfluous object files.\n\n  Own Id: OTP-17001 Aux Id: PR-2852\n\n- TLS connections now support EdDSA certificates.\n\n  Own Id: OTP-17142 Aux Id: PR-4756, GH-4637, GH-4650\n\n- Add prop_aead attribute to map from crypto:cipher_info/1.\n\n  Own Id: OTP-17313 Aux Id: PR-4686","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 4.9.0.4 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-9-0-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Changed the behaviour of the engine load/unload functions\n\n  The engine load/unload functions have got changed semantics to get a more\n  consistent behaviour and work correct when variables are garbage collected.\n\n  The load functions now don't register the methods for the engine to replace.\n  That will now be handled with the new functions\n  engine_register/engine_unregister if needed.\n\n  Some functions functions are removed from the documentation and therefor the\n  API, but they are left in the code for compatibility.\n\n  - engine_load/4: is now the same as engine_load/3\n  - engine_unload/2: is now the same as engine_unload/1\n  - ensure_engine_loaded/3: is now the same as ensure_engine_loaded/2\n  - ensure_engine_unloaded/1, ensure_engine_unloaded/2: is now the same as\n    engine_unload/1\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-18172 Aux Id: ERIERL-826","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 4.9.0.3 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-9-0-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fix timing bug in ensure_engine_loaded\n\n  When two ensure_engine_loaded() calls were done in parallel there was a\n  possibility that a crypto lib function was called by both instead of just one\n  of them which resulted in an error. This is solved by moving the\n  implementation from erlang down into a NIF function that uses a mutex to\n  protect the sensitive part.\n\n  Own Id: OTP-17858 Aux Id: ERIERL-728\n\n- Remove all references correctly in the garbage collection if an engine handle\n  was not explicit unloaded.\n\n  Own Id: OTP-18152","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 4.9.0.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-9-0-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- EC keys are now zero-padded to the expected length if needed.\n\n  Own Id: OTP-17442 Aux Id: GH-4861","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 4.9.0.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-9-0-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Removed a risk for coredump.\n\n  Own Id: OTP-17391 Aux Id: GH-4810\n\n- Commit of generated `configure` script.\n\n  Own Id: OTP-17420 Aux Id: OTP-17398, GH-4821","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 4.9 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-9"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fix minor memory leaks in crypto ENGINE and robustify the code.\n\n  Own Id: OTP-17212\n\n- The otp_test_engine no longer fails if NO_EC\\* is set in the OpenSSL\n  configuration.\n\n  Own Id: OTP-17256 Aux Id: PR-4580, GH-4573","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Various address sanitizer support.\n\n  Own Id: OTP-16959 Aux Id: PR-2965\n\n- EVP is now disabled for OpenSSL cryptolib versions up to and including 1.0.2\n\n  Own Id: OTP-17116 Aux Id: PR-2972\n\n- Warning for unused C function removed\n\n  Own Id: OTP-17145 Aux Id: OTP-17105, PR-2872","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 4.8.3 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-8-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Adding missing flag in BN-calls in SRP.\n\n  Own Id: OTP-17107","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 4.8.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-8-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fixed usage of `AC_CONFIG_AUX_DIRS()` macros in configure script sources.\n\n  Own Id: OTP-17093 Aux Id: ERL-1447, PR-2948","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 4.8.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-8-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Build the supported curves cache in the NIF when crypto is loaded, no matter\n  how it is loaded.\n\n  This prevents a possible problem with different processes starting the crypto\n  application concurrently.\n\n  Own Id: OTP-16819 Aux Id: PR-2720\n\n- It is now possible to build with crypto and openssl gprof-enabled and\n  statically link them into the VM.\n\n  Own Id: OTP-17029","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Fixed performance loss in HMAC when using older OpenSSL due to mutex issues.\n\n  A workaround is implemented to allow fallback from using the EVP API for HMAC\n  operations. On some architectures this may improve the performance, especially\n  with old OpenSSL versions. This fallback to low-level functions is always\n  enabled for openssl versions before 1.0.2.\n\n  Own Id: OTP-17025 Aux Id: ERL-1400, PR-2877","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 4.8 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-8"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fix type spec bug in crypto for crypto_init and crypto:one_time\n\n  Own Id: OTP-16658 Aux Id: OTP-15884, ERL-1257\n\n- The deprecation message for crypto:rand_uniform/2 indicated a non-existent\n  function. The correct one (rand:uniform/1) is now suggested.\n\n  Own Id: OTP-16846 Aux Id: PR-2741","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Implemented a workaround to allow fallback from using the EVP API for\n  Diffie-Hellman key generation\n\n  Own Id: OTP-16771 Aux Id: ERIERL-509\n\n- The internal Diffie-Hellman high level API for key generation was slow in old\n  and by OpenSSL now unsupported cryptolib versions (1.0.1 and earlier).\n\n  If such a cryptolib is used anyhow, the low-level API is used internally in\n  the crypto application.\n\n  Own Id: OTP-16774","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 4.7 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-7"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Crypto reported unsupported elliptic curves as supported on e.g Fedora\n  distros.\n\n  Own Id: OTP-16579 Aux Id: ERL-825","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Support for ed25519 and ed448 added to `crypto:generate_key`.\n\n  Own Id: OTP-15967 Aux Id: PR-2329\n\n- The [new crypto functions api](new_api.md#the-new-api) (crypto_init,\n  crypto_update and crypto_one_time) has been updated.\n\n  There is now a function [`crypto_final/1`](`crypto:crypto_final/1`) and a\n  possibility to set options in [`crypto_init/3`](`crypto:crypto_init/3`) and\n  [`crypto_init/4`](`crypto:crypto_init/4`). See the manual for details.\n\n  Own Id: OTP-16160\n\n- As [announced](notes.md#crypto-4-5) in OTP 22.0, a New API was introduced in\n  CRYPTO. See the [_New and Old API_](new_api.md) chapter in the CRYPTO User's\n  Guide for more information and suggested replacement functions.\n\n  [The Old API](new_api.md#the-old-api) is now deprecated in OTP-23.0 and will\n  be removed in OTP-24.0.\n\n  This deprecation includes cipher names. See the section\n  [Retired cipher names](new_api.md#retired-cipher-names) in the crypto User's\n  Guide, chapter [The Old API](new_api.md#the-old-api).\n\n  Own Id: OTP-16232\n\n- Fix C-compilation without deprecated OpenSSL cryptolib APIs\n\n  Own Id: OTP-16369 Aux Id: PR-2474\n\n- Refactored the internal handling of deprecated and removed functions.\n\n  Own Id: OTP-16469\n\n- Added missing 'eddh' to [crypto:supports(public_keys)](`crypto:supports/1`).\n\n  Own Id: OTP-16583","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 4.6.5.4 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-6-5-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- EC keys are now zero-padded to the expected length if needed.\n\n  Own Id: OTP-17442 Aux Id: GH-4861","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 4.6.5.3 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-6-5-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Removed a risk for coredump.\n\n  Own Id: OTP-17391 Aux Id: GH-4810\n\n- Commit of generated `configure` script.\n\n  Own Id: OTP-17420 Aux Id: OTP-17398, GH-4821","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 4.6.5.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-6-5-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Adding missing flag in BN-calls in SRP.\n\n  Own Id: OTP-17107","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 4.6.5.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-6-5-1"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Implemented a workaround to allow fallback from using the EVP API for\n  Diffie-Hellman key generation\n\n  Own Id: OTP-16771 Aux Id: ERIERL-509","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 4.6.5 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-6-5"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fixed potential memory leaks involving calls to the crypto ng_api.\n\n  Own Id: OTP-16428 Aux Id: PR-2511","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 4.6.4 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-6-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Constant time comparisons added.\n\n  Own Id: OTP-16376","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 4.6.3 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-6-3"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- The ciphers aes_cfb8 and aes_cfb128 are now using the EVP interface. The\n  supported key lengths are 128, 192 and 256 bits.\n\n  Own Id: OTP-16133 Aux Id: PR-2407\n\n- The ciphers aes_cfb8 and aes_cfb128 are now available in FIPS enabled mode.\n\n  Own Id: OTP-16134 Aux Id: PR-2407","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 4.6.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-6-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- The AEAD tag was not previously checked on decrypt with chacha20_poly1305\n\n  Own Id: OTP-16242 Aux Id: ERL-1078","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 4.6.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-6-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- FIxed a bug if the erlang emulator was linked with a very old cryptolib\n  version (1.0.1 or earlier).\n\n  The bug now fixed could have triggered a core dump if an unknown cipher name\n  was used in crypto functions.\n\n  Own Id: OTP-16202","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 4.6 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-6"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- The implementation of `crypto_one_time/4` is adjusted to match the type\n  specification. The spec and the black-box behaviour of the function are\n  unchanged.\n\n  Some details: Both the spec and the implementation were correct seen\n  separately. But with both of them combined simultaneously with\n  `crypto_one_time/5` which was called by the implementation of\n  `crypto_one_time/4`, an (obvious) error was detected by a Dialyzer with more\n  thorough checking than usual.\n\n  Own Id: OTP-15884 Aux Id: ERL-974\n\n- When using crypto with FIPS mode enabled, the digests were not correctly\n  handled.\n\n  Own Id: OTP-15911\n\n- A memory leak in error handling code in `ng_crypto_init_nif` is fixed.\n\n  Own Id: OTP-15924\n\n- Fixed the broken static build of the crypto nifs\n\n  Own Id: OTP-15928 Aux Id: PR-2296","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- The Message Authentication Codes (MAC) CMAC, HMAC and Poly1305 are unified\n  into common functions in the New Crypto API. See the manual for CRYPTO.\n\n  Own Id: OTP-13872","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 4.5.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-5-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- The cipher aes-ctr was disabled by mistake in crypto:supports for cryptolibs\n  before 1.0.1. It worked however in the encrypt and decrypt functions.\n\n  Own Id: OTP-15829","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 4.5 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-5"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fixed a bug in error return for `crypto:poly1305/2`. It returned the atom\n  `notsup` instead of the exception `notsup`.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-15677\n\n- The cipher chacha20 was introduced in OpenSSL 1.1.0. However, it could in a\n  very odd situation, fail for versions less than OpenSSL 1.1.0d. It is\n  therefore disabled for those versions.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-15678","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- A new `rand` module algorithm, `exro928ss` (Xoroshiro928\\*\\*), has been\n  implemented. It has got a really long period and good statistical quality for\n  all output bits, while still being only about 50% slower than the default\n  algorithm.\n\n  The same generator is also used as a long period counter in a new `crypto`\n  plugin for the `rand` module, algorithm `crypto_aes`. This plugin uses AES-256\n  to scramble the counter which buries any detectable statistical artifacts.\n  Scrambling is done in chunks which are cached to get good amortized speed\n  (about half of the default algorithm).\n\n  Own Id: OTP-14461 Aux Id: PR-1857\n\n- Crypto's single C-file is split into multiple files. The different coding\n  styles in the different parts are unified into a single style.\n\n  Own Id: OTP-14732 Aux Id: PR-2068, PR-2095\n\n- Build configuration of the `crypto` application has been moved from the `erts`\n  application into the `crypto` application.\n\n  Own Id: OTP-15129\n\n- Adds two hash functions `blake2b` and `blake2s` (64 bit hash and 32 bit hash\n  respectively). These are modern and standard hash functions used in\n  blockchains and encrypted communication protocols. The hash functions are\n  available in OpenSSL since version 1.1.1.\n\n  Own Id: OTP-15564 Aux Id: PR-2129\n\n- A new API is implemented in crypto. See the CRYPTO user's guide, chapter _New\n  and Old API_ for more information.\n\n  The old api with the `crypto:block_*` and `crypto:stream_*` interfaces are\n  kept for compatibility, but implemented with the new api. Please note that\n  since the error checking is more thorough, there _might_ be arguments with for\n  example faulty lengths that are no longer accepted.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-15644 Aux Id: OTP-14732 , OTP-15451, PR-1857 , PR-2068, PR-2095\n\n- The new hash_info/1 and cipher_info/1 functions returns maps with information\n  about the hash or cipher in the argument.\n\n  Own Id: OTP-15655 Aux Id: PR-2173, ERL-864, PR-2186\n\n- Obey additional OpenSSL configure flags when compiling the C-part of the\n  CRYPTO application: `no-bf`, `no-blake2`, `no-chacha`, `no-cmac`, `no-dh`,\n  `no-dsa`, `no-md4`, `no-poly1305`, `no-rc2`, `no-rc4` and `no-rmd160`.\n\n  Own Id: OTP-15683\n\n- A new function `crypto:supports/1` is introduced. The single argument takes an\n  atom as argument: `hashes`, `public_keys`, `ciphers`, `macs`, `curves` or\n  `rsa_opts`. The return value is a list of supported algorithms.\n\n  The difference with the existing `crypto:supports/0` is, apart from the\n  argument and the return value, that the old function reports what is supported\n  by the old api, and the new function reports algorithms in the new api.\n\n  Own Id: OTP-15771","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 4.4.2.3 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-4-2-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Adding missing flag in BN-calls in SRP.\n\n  Own Id: OTP-17107","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 4.4.2.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-4-2-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Constant time comparisons added.\n\n  Own Id: OTP-16376","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 4.4.2.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-4-2-1"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- The ciphers aes_cfb8 and aes_cfb128 are now using the EVP interface. The\n  supported key lengths are 128, 192 and 256 bits.\n\n  Own Id: OTP-16133 Aux Id: PR-2407\n\n- The ciphers aes_cfb8 and aes_cfb128 are now available in FIPS enabled mode.\n\n  Own Id: OTP-16134 Aux Id: PR-2407","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 4.4.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-4-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fixed build link error on Windows. Unresolved symbol 'bcmp'.\n\n  Own Id: OTP-15750 Aux Id: ERL-905","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 4.4.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-4-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fixes a bug that caused `crypto:sign` and `crypto:verify` to return the error\n  message `badarg` instead of `notsup` in one case. That case was when signing\n  or verifying with eddsa keys (that is, ed15519 or ed448), but only when FIPS\n  was supported and enabled.\n\n  Own Id: OTP-15634","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Added a crypto benchmark test suite.\n\n  Own Id: OTP-15447","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 4.4 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Updated the RSA options part in the crypto application's C-code, documentation\n  and tests.\n\n  Own Id: OTP-15302","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Added ed25519 and ed448 sign/verify.\n\n  Requires OpenSSL 1.1.1 or higher as cryptolib under the OTP application\n  `crypto`.\n\n  Own Id: OTP-15419 Aux Id: OTP-15094\n\n- Fixed valgrind warnings.\n\n  Own Id: OTP-15467","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 4.3.3 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-3-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- The RSA options `rsa_mgf1_md`, `rsa_oaep_md`, and `rsa_oaep_label` were always\n  disabled. They will now be enabled when a suitable cryptolib is used.\n\n  They are still experimental and may change without prior notice.\n\n  Own Id: OTP-15212 Aux Id: ERL-675, PR1899, PR838\n\n- The ciphers `aes_ige256` and `blowfish_cbc` had naming issues in\n  `crypto:next_iv/2`.\n\n  Own Id: OTP-15283\n\n- the `RSA_SSLV23_PADDING` is disabled if LibreSSL is used as cryptlib. This is\n  due to compilation problems.\n\n  This will be investigated further in the future.\n\n  Own Id: OTP-15303","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- The supported named elliptic curves are now reported in `crypto:supports/0` in\n  a new entry tagged by `'curves'`.\n\n  The function `crypto:ec_curves/0` is kept for compatibility.\n\n  Own Id: OTP-14717 Aux Id: OTP-15244\n\n- The typing in the CRYPTO and PUBLIC_KEY applications are reworked and a few\n  mistakes are corrected.\n\n  The documentation is now generated from the typing and some clarifications are\n  made.\n\n  A new chapter on Algorithm Details such as key sizes and availability is added\n  to the CRYPTO User's Guide.\n\n  Own Id: OTP-15134\n\n- Support for SHA3 both as a separate hash and in HMAC is now available if\n  OpenSSL 1.1.1 or higher is used as cryptolib.\n\n  Available lengths are reported in the `'hashs'` entry in `crypto:supports/0`\n  as `sha3_*`.\n\n  Own Id: OTP-15153\n\n- The mac algorithm `poly1305` and the cipher algorithm `chacha20` are now\n  supported if OpenSSL 1.1.1 or higher is used as cryptolib.\n\n  Own Id: OTP-15164 Aux Id: OTP-15209\n\n- The key exchange Edward curves `x25519` and `x448` are now supported if\n  OpenSSL 1.1.1 or higher is used as cryptolib.\n\n  Own Id: OTP-15240 Aux Id: OTP-15133\n\n- The supported RSA options for sign/verify and encrypt/decrypt are now reported\n  in `crypto:supports/0` in a new entry tagged by '`rsa_opts`'.\n\n  The exakt set is still experimental and may change without prior notice.\n\n  Own Id: OTP-15260\n\n- The cipher `aes_ccm` is added.\n\n  Own Id: OTP-15286","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 4.3.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-3-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Update the crypto engine functions to handle multiple loads of an engine.\n\n  `engine_load/3/4` is updated so it doesn't add the engine ID to OpenSSLs\n  internal list of engines which makes it possible to run the engine_load more\n  than once if it doesn't contain global data.\n\n  Added `ensure_engine_loaded/2/3` which guarantees that the engine just is\n  loaded once and the following calls just returns a reference to it. This is\n  done by add the ID to the internal OpenSSL list and check if it is already\n  registered when the function is called.\n\n  Added `ensure_engine_unloaded/1/2` to unload engines loaded with\n  ensure_engine_loaded.\n\n  Then some more utility functions are added.\n\n  `engine_add/1`, adds the engine to OpenSSL internal list\n\n  `engine_remove/1`, remove the engine from OpenSSL internal list\n\n  `engine_get_id/1`, fetch the engines id\n\n  `engine_get_name/1`, fetch the engine name\n\n  Own Id: OTP-15233","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 4.3.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-3-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fixed a node crash in `crypto:compute_key(ecdh, ...)` when passing a wrongly\n  typed Others argument.\n\n  Own Id: OTP-15194 Aux Id: ERL-673","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 4.3 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Removed two undocumented and erroneous functions\n  (`crypto:dh_generate_parameters/2` and `crypto:dh_check/1`).\n\n  Own Id: OTP-14956 Aux Id: ERL-579\n\n- Fixed bug causing VM crash if doing runtime upgrade of a crypto module built\n  against OpenSSL older than 0.9.8h. Bug exists since OTP-20.2.\n\n  Own Id: OTP-15088","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- A new `rand` plugin algorithm has been implemented in `crypto`, that is:\n  `crypto_cache`. It uses strong random bytes as randomness source and caches\n  them to get good speed. See `crypto:rand_seed_alg/1`.\n\n  Own Id: OTP-13370 Aux Id: PR-1573\n\n- Diffie-Hellman key functions are re-written with the EVP_PKEY api.\n\n  Own Id: OTP-14864","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 4.2.2.4 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-2-2-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Constant time comparisons added.\n\n  Own Id: OTP-16376","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 4.2.2.3 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-2-2-3"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- The ciphers aes_cfb8 and aes_cfb128 are now using the EVP interface. The\n  supported key lengths are 128, 192 and 256 bits.\n\n  Own Id: OTP-16133 Aux Id: PR-2407","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 4.2.2.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-2-2-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fixed a node crash in `crypto:compute_key(ecdh, ...)` when passing a wrongly\n  typed Others argument.\n\n  Own Id: OTP-15194 Aux Id: ERL-673","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 4.2.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-2-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- If OPENSSL_NO_EC was set, the compilation of the crypto nifs failed.\n\n  Own Id: OTP-15073\n\n- C-compile errors for LibreSSL 2.7.0 - 2.7.2 fixed\n\n  Own Id: OTP-15074 Aux Id: ERL-618","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 4.2.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-2-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fix build error caused by removed RSA padding functions in LibreSSL >= 2.6.1\n\n  Own Id: OTP-14873","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 4.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- The compatibility function `void HMAC_CTX_free` in `crypto.c` erroneously\n  tried to return a value.\n\n  Own Id: OTP-14720","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Rewrite public and private key encode/decode with EVP api. New RSA padding\n  options added. This is a modified half of PR-838.\n\n  Own Id: OTP-14446\n\n- The crypto API is extended to use private/public keys stored in an Engine for\n  sign/verify or encrypt/decrypt operations.\n\n  The ssl application provides an API to use this new engine concept in TLS.\n\n  Own Id: OTP-14448\n\n- Add support to plug in alternative implementations for some or all of the\n  cryptographic operations supported by the OpenSSL Engine API. When configured\n  appropriately, OpenSSL calls the engine's implementation of these operations\n  instead of its own.\n\n  Own Id: OTP-14567\n\n- Replaced a call of the OpenSSL deprecated function `DH_generate_parameters` in\n  `crypto.c`.\n\n  Own Id: OTP-14639\n\n- Documentation added about how to use keys stored in an Engine.\n\n  Own Id: OTP-14735 Aux Id: OTP-14448\n\n- Add engine\\_ ctrl_cmd_string/3,4 the OpenSSL Engine support in crypto.\n\n  Own Id: OTP-14801","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 4.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- On macOS, `crypto` would crash if `observer` had been started before `crypto`.\n  On the beta for macOS 10.13 (High Sierra), `crypto` would crash. Both of those\n  bugs have been fixed.\n\n  Own Id: OTP-14499 Aux Id: ERL-251 ERL-439","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Extend crypto:sign, crypto:verify, public_key:sign and public_key:verify with:\n\n  \\* support for RSASSA-PS padding for signatures and for saltlength setting  \n  \\* X9.31 RSA padding.  \n  \\* sha, sha224, sha256, sha384, and sha512 for dss signatures as mentioned in\n  NIST SP 800-57 Part 1.  \n  \\* ripemd160 to be used for rsa signatures.\n\n  This is a manual merge of half of the pull request 838 by potatosalad from\n  Sept 2015.\n\n  Own Id: OTP-13704 Aux Id: PR838\n\n- A new tuple in `crypto:supports/0` reports supported MAC algorithms.\n\n  Own Id: OTP-14504","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 4.0 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-4-0"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- LibreSSL can now be used by the modernized crypto app.\n\n  Own Id: OTP-14247\n\n- Add compile option `-compile(no_native)` in modules with `on_load` directive\n  which is not yet supported by HiPE.\n\n  Own Id: OTP-14316 Aux Id: PR-1390\n\n- Fix a bug in aes cfb128 function introduced by the bug fix in GitHub pull\n  request [\\#1393](https://github.com/erlang/otp/pull/1393).\n\n  Own Id: OTP-14435 Aux Id: PR-1462, PR-1393, OTP-14313","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Add basic support for CMAC\n\n  Own Id: OTP-13779 Aux Id: ERL-82 PR-1138\n\n- Removed functions deprecated in crypto-3.0 first released in OTP-R16B01\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-13873\n\n- The `crypto` application now supports OpenSSL 1.1.\n\n  Own Id: OTP-13900\n\n- Allow Erlang/OTP to use OpenSSL in FIPS-140 mode, in order to satisfy specific\n  security requirements (mostly by different parts of the US federal\n  government).\n\n  See the new crypto users guide \"FIPS mode\" chapter about building and using\n  the FIPS support which is disabled by default.\n\n  (Thanks to dszoboszlay and legoscia)\n\n  Own Id: OTP-13921 Aux Id: PR-1180\n\n- Crypto chacha20-poly1305 as in RFC 7539 enabled for OpenSSL >= 1.1.\n\n  Thanks to mururu.\n\n  Own Id: OTP-14092 Aux Id: PR-1291\n\n- RSA key generation added to `crypto:generate_key/2`. Thanks to wiml.\n\n  An interface is also added to `public_key:generate_key/1`.\n\n  Own Id: OTP-14140 Aux Id: ERL-165, PR-1299\n\n- Raised minimum requirement for OpenSSL version to OpenSSL-0.9.8.c although we\n  recommend a much higher version, that is a version that is still maintained\n  officially by the OpenSSL project. Note that using such an old version may\n  restrict the crypto algorithms supported.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-14171\n\n- Deprecate crypto:rand_uniform/2 as it is not cryptographically strong\n\n  Own Id: OTP-14274\n\n- The Crypto application now supports generation of cryptographically strong\n  random numbers (floats < 1.0 and integer arbitrary ranges) as a plugin to the\n  'rand' module.\n\n  Own Id: OTP-14317 Aux Id: PR-1372\n\n- This replaces the hard coded test values for AES, CMAC and GCM ciphers with\n  the full validation set from NIST's CAVP program.\n\n  Own Id: OTP-14436 Aux Id: PR-1396","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 3.7.4 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-3-7-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fix a bug with AES CFB 128 for 192 and 256 bit keys. Thanks to kellymclaughlin\n  \\!\n\n  Own Id: OTP-14313 Aux Id: PR-1393","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 3.7.3 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-3-7-3"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- The implementation of the key exchange algorithms\n  diffie-hellman-group-exchange-sha\\* are optimized, up to a factor of 11 for\n  the slowest ( = biggest and safest) group size.\n\n  Own Id: OTP-14169 Aux Id: seq-13261","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 3.7.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-3-7-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- The crypto application has been fixed to not use RC2 against OpenSSL built\n  with RC2 disabled.\n\n  Own Id: OTP-13895 Aux Id: PR-1163\n\n- The crypto application has been fixed to not use RC4 against OpenSSL built\n  with RC4 disabled.\n\n  Own Id: OTP-13896 Aux Id: PR-1169","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- To ease troubleshooting, `erlang:load_nif/2` now includes the return value\n  from a failed call to load/reload/upgrade in the text part of the error tuple.\n  The `crypto` NIF makes use of this feature by returning the source line\n  where/if the initialization fails.\n\n  Own Id: OTP-13951","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 3.7.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-3-7-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Crypto has been fixed to work against OpenSSL versions with disabled DES\n  ciphers. Correct spelling of cipher algorithm 'des3_cfb' has been introduced;\n  the previous misspeling still works.\n\n  Own Id: OTP-13783 Aux Id: ERL-203\n\n- The size of an internal array in crypto has been fixed to not segfault when\n  having all possible ciphers. Bug fix by Duncan Overbruck.\n\n  Own Id: OTP-13789 Aux Id: PR-1140","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 3.7 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-3-7"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Refactor `crypto` to use the EVP interface of OpenSSL, which is the\n  recommended interface that also enables access to hardware acceleration for\n  some operations.\n\n  Own Id: OTP-12217\n\n- Add support for 192-bit keys for the `aes_cbc` cipher.\n\n  Own Id: OTP-13206 Aux Id: pr 832\n\n- Add support for 192-bit keys for `aes_ecb`.\n\n  Own Id: OTP-13207 Aux Id: pr829\n\n- Deprecate the function `crypto:rand_bytes` and make sure that\n  `crypto:strong_rand_bytes` is used in all places that are cryptographically\n  significant.\n\n  Own Id: OTP-13214\n\n- Enable AES-GCM encryption/decryption to change the tag length between 1 to 16\n  bytes.\n\n  Own Id: OTP-13483 Aux Id: PR-998","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 3.6.3 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-3-6-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fix bug for `aes_ecb` block crypto when data is larger than 16 bytes.\n\n  Own Id: OTP-13249\n\n- Improve portability of ECC tests in Crypto and SSL for \"exotic\" OpenSSL\n  versions.\n\n  Own Id: OTP-13311","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 3.6.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-3-6-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Small documentation fixes\n\n  Own Id: OTP-13017","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 3.6.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-3-6-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Make `crypto:ec_curves/0` return empty list if elliptic curve is not supported\n  at all.\n\n  Own Id: OTP-12944","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 3.6 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-3-6"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Enhance crypto:generate_key to calculate ECC public keys from private key.\n\n  Own Id: OTP-12394\n\n- Fix bug in `crypto:generate_key` for `ecdh` that could cause VM crash for\n  faulty input.\n\n  Own Id: OTP-12733","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Use the EVP API for AES-CBC crypto to enables the use of hardware acceleration\n  for AES-CBC crypto on newer Intel CPUs (AES-NI), among other platforms.\n\n  Own Id: OTP-12380\n\n- Add AES ECB block encryption.\n\n  Own Id: OTP-12403","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 3.5 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-3-5"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Extend block_encrypt/decrypt for aes_cfb8 and aes_cfb128 to accept keys of\n  length 128, 192 and 256 bits. Before only 128 bit keys were accepted.\n\n  Own Id: OTP-12467","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 3.4.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-3-4-2"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Add configure option --with-ssl-incl=PATH to support OpenSSL installations\n  with headers and libraries at different places.\n\n  Own Id: OTP-12215 Aux Id: seq12700\n\n- Add configure option --with-ssl-rpath to control which runtime library path to\n  use for dynamic linkage toward OpenSSL.\n\n  Own Id: OTP-12316 Aux Id: seq12753","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 3.4.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-3-4-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Make `crypto` verify major version number of OpenSSL header files and runtime\n  library. Loading of `crypto` will fail if there is a version mismatch.\n\n  Own Id: OTP-12146 Aux Id: seq12700","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 3.4 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-3-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fix memory leak in `crypto:hmac_init/upgrade/final` functions for all data and\n  in `crypto:hmac/3/4` for data larger than 20000 bytes. Bug exists since OTP\n  17.0.\n\n  Own Id: OTP-11953\n\n- Fix memory leak in `crypto` for elliptic curve.\n\n  Own Id: OTP-11999","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Add `aes_cfb8` cypher to `crypto:block_encrypt` and `block_decrypt`.\n\n  Own Id: OTP-11911","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 3.3 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-3-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fix memory leaks and invalid deallocations in `mod_pow`, `mod_exp` and\n  `generate_key(srp,...)` when bad arguments are passed. (Thanks to Florian\n  Zumbiehi)\n\n  Own Id: OTP-11550\n\n- Correction of the word 'ChipherText' throughout the documentation (Thanks to\n  Andrew Tunnell-Jones)\n\n  Own Id: OTP-11609\n\n- Fix fatal bug when using a hmac context variable in more than one call to\n  `hmac_update` or `hmac_final`. The reuse of hmac contexts has never worked as\n  the underlying OpenSSL implementation does not support it. It is now\n  documented as having undefined behaviour, but it does not crash or corrupt the\n  VM anymore.\n\n  Own Id: OTP-11724\n\n- Crypto handles out-of-memory with a controlled abort instead of\n  crash/corruption. (Thanks to Florian Zumbiehi)\n\n  Own Id: OTP-11725\n\n- Application upgrade (appup) files are corrected for the following\n  applications:\n\n  `asn1, common_test, compiler, crypto, debugger, dialyzer, edoc, eldap, erl_docgen, et, eunit, gs, hipe, inets, observer, odbc, os_mon, otp_mibs, parsetools, percept, public_key, reltool, runtime_tools, ssh, syntax_tools, test_server, tools, typer, webtool, wx, xmerl`\n\n  A new test utility for testing appup files is added to test_server. This is\n  now used by most applications in OTP.\n\n  (Thanks to Tobias Schlager)\n\n  Own Id: OTP-11744","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- By giving --enable-static-\\{nifs,drivers\\} to configure it is now possible to\n  statically linking of nifs and drivers to the main Erlang VM binary. At the\n  moment only the asn1 and crypto nifs of the Erlang/OTP nifs and drivers have\n  been prepared to be statically linked. For more details see the Installation\n  Guide in the System documentation.\n\n  Own Id: OTP-11258\n\n- Add IGE mode for AES cipher in crypto (Thanks to Yura Beznos).\n\n  Own Id: OTP-11522\n\n- Moved elliptic curve definition from the crypto NIF/OpenSSL into Erlang code,\n  adds the RFC-5639 brainpool curves and makes TLS use them (RFC-7027).\n\n  Thanks to Andreas Schultz\n\n  Own Id: OTP-11578\n\n- Remove all obsolete application processes from crypto and make it into a pure\n  library application.\n\n  Own Id: OTP-11619","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 3.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-3-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fix uninitialized pointers in crypto (Thanks to Anthony Ramine)\n\n  Own Id: OTP-11510","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 3.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-3-1"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Refactor ecdsa cipher to simplify code and improve performance.\n\n  Own Id: OTP-11320","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 3.0 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-3-0"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Integrate elliptic curve contribution from Andreas Schultz\n\n  In order to be able to support elliptic curve cipher suites in SSL/TLS,\n  additions to handle elliptic curve infrastructure has been added to public_key\n  and crypto.\n\n  This also has resulted in a rewrite of the crypto API to gain consistency and\n  remove unnecessary overhead. All OTP applications using crypto has been\n  updated to use the new API.\n\n  Impact: Elliptic curve cryptography (ECC) offers equivalent security with\n  smaller key sizes than other public key algorithms. Smaller key sizes result\n  in savings for power, memory, bandwidth, and computational cost that make ECC\n  especially attractive for constrained environments.\n\n  Own Id: OTP-11009\n\n- Fixed a spelling mistake in crypto docs. Thanks to Klaus Trainer\n\n  Own Id: OTP-11058","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Known Bugs and Problems - Crypto Release Notes","doc":"- Make the crypto functions interruptible by chunking input when it is very\n  large and bumping reductions in the nifs.\n\n  Not yet implemented for block_encrypt|decrypt/4\n\n  Impact: Individual calls to crypto functions may take longer time but over all\n  system performance should improve as crypto calls will not become throughput\n  bottlenecks.\n\n  Own Id: OTP-11142","ref":"notes.html#known-bugs-and-problems"},{"type":"extras","title":"Crypto 2.3 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-2-3"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Enable runtime upgrade of crypto including the OpenSSL library used by crypto.\n\n  Own Id: OTP-10596\n\n- Improve documentation and tests for hmac functions in crypto. Thanks to Daniel\n  White\n\n  Own Id: OTP-10640\n\n- Added ripemd160 support to crypto. Thanks to Michael Loftis\n\n  Own Id: OTP-10667","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 2.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-2-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Remove unnecessary dependency to libssl from crypto NIF library. This\n  dependency was introduced by accident in R14B04.\n\n  Own Id: OTP-10064","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Add crypto and public_key support for the hash functions SHA224, SHA256,\n  SHA384 and SHA512 and also hmac and rsa_sign/verify support using these hash\n  functions. Thanks to Andreas Schultz for making a prototype.\n\n  Own Id: OTP-9908\n\n- Optimize RSA private key handling in `crypto` and `public_key`.\n\n  Own Id: OTP-10065\n\n- Make `crypto:aes_cfb_128_encrypt` and `crypto:aes_cfb_128_decrypt` handle data\n  and cipher with arbitrary length. (Thanks to Stefan Zegenhagen)\n\n  Own Id: OTP-10136","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 2.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-2-1"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- public_key, ssl and crypto now supports PKCS-8\n\n  Own Id: OTP-9312\n\n- Erlang/OTP can now be built using parallel make if you limit the number of\n  jobs, for instance using '`make -j6`' or '`make -j10`'. '`make -j`' does not\n  work at the moment because of some missing dependencies.\n\n  Own Id: OTP-9451\n\n- Add DES and Triple DES cipher feedback (CFB) mode functions to `crypto`.\n  (Thanks to Paul Guyot)\n\n  Own Id: OTP-9640\n\n- Add sha256, sha384 and sha512 support for `crypto:rsa_verify`.\n\n  Own Id: OTP-9778","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 2.0.4 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-2-0-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- `crypto:rand_uniform` works correctly for negative integers. Fails with\n  `badarg` exception for invalid ranges (when `Hi =< Lo`) instead of returning\n  incorrect output.\n\n  Own Id: OTP-9526\n\n- Fix win32 OpenSSL static linking (Thanks to Dave Cottlehuber)\n\n  Own Id: OTP-9532","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 2.0.3 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-2-0-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Various small documentation fixes (Thanks to Bernard Duggan)\n\n  Own Id: OTP-9172","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- New `crypto` support for streaming of AES CTR and HMAC. (Thanks to Travis\n  Jensen)\n\n  Own Id: OTP-9275\n\n- Due to standard library DLL mismatches between versions of OpenSSL and\n  Erlang/OTP, OpenSSL is now linked statically to the crypto driver on Windows.\n  This fixes problems starting crypto when running Erlang as a service on all\n  Windows versions.\n\n  Own Id: OTP-9280","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 2.0.2.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-2-0-2-2"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Strengthened random number generation. (Thanks to Geoff Cant)\n\n  Own Id: OTP-9225","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 2.0.2.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-2-0-2-1"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Misc. Updates.\n\n  Own Id: OTP-9132","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 2.0.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-2-0-2"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- AES CTR encryption support in `crypto`.\n\n  Own Id: OTP-8752 Aux Id: seq11642","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 2.0.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-2-0-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Crypto dialyzer type error in md5_mac and sha_mac.\n\n  Own Id: OTP-8718\n\n- RC4 stream cipher didn't work. This since the new NIF implementation of\n  `crypto:rc4_encrypt_with_state/2` introduced in `crypto-2.0` didn't return an\n  updated state. (Thanks to Paul Guyot)\n\n  Own Id: OTP-8781\n\n- A number of memory leaks in the crypto NIF library have been fixed.\n\n  Own Id: OTP-8810","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Added erlang:system_info(build_type) which makes it easier to chose drivers,\n  NIF libraries, etc based on build type of the runtime system.\n\n  The NIF library for crypto can now be built for valgrind and/or debug as\n  separate NIF libraries that will be automatically loaded if the runtime system\n  has been built with a matching build type.\n\n  Own Id: OTP-8760","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 2.0 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-2-0"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- crypto application changed to use NIFs instead of driver.\n\n  Own Id: OTP-8333\n\n- des_ecb_encrypt/2 and des_ecb_decrypt/2 has been added to the crypto module.\n  The crypto:md4/1 function has been documented.\n\n  Own Id: OTP-8551\n\n- The undocumented, unsupported, and deprecated function `lists:flat_length/1`\n  has been removed.\n\n  Own Id: OTP-8584\n\n- New variants of `crypto:dss_sign` and `crypto:dss_verify` with an extra\n  argument to control how the digest is calculated.\n\n  Own Id: OTP-8700","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 1.6.4 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-1-6-4"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Cross compilation improvements and other build system improvements.\n\n  Most notable:\n\n  - Lots of cross compilation improvements. The old cross compilation support\n    was more or less non-existing as well as broken. Please, note that the cross\n    compilation support should still be considered as experimental. Also note\n    that old cross compilation configurations cannot be used without\n    modifications. For more information on cross compiling Erlang/OTP see the\n    `$ERL_TOP/INSTALL-CROSS.md` file.\n  - Support for staged install using\n    [DESTDIR](http://www.gnu.org/prep/standards/html_node/DESTDIR.html). The old\n    broken `INSTALL_PREFIX` has also been fixed. For more information see the\n    `$ERL_TOP/INSTALL.md` file.\n  - Documentation of the `release` target of the top `Makefile`. For more\n    information see the `$ERL_TOP/INSTALL.md` file.\n  - `make install` now by default creates relative symbolic links instead of\n    absolute ones. For more information see the `$ERL_TOP/INSTALL.md` file.\n  - `$ERL_TOP/configure --help=recursive` now works and prints help for all\n    applications with `configure` scripts.\n  - Doing `make install`, or `make release` directly after `make all` no longer\n    triggers miscellaneous rebuilds.\n  - Existing bootstrap system is now used when doing `make install`, or\n    `make release` without a preceding `make all`.\n  - The `crypto` and `ssl` applications use the same runtime library path when\n    dynamically linking against `libssl.so` and `libcrypto.so`. The runtime\n    library search path has also been extended.\n  - The `configure` scripts of `erl_interface` and `odbc` now search for thread\n    libraries and thread library quirks the same way as ERTS do.\n  - The `configure` script of the `odbc` application now also looks for odbc\n    libraries in `lib64` and `lib/64` directories when building on a 64-bit\n    system.\n  - The `config.h.in` file in the `erl_interface` application is now\n    automatically generated in instead of statically updated which reduces the\n    risk of `configure` tests without any effect.\n\n  (Thanks to Henrik Riomar for suggestions and testing)\n\n  (Thanks to Winston Smith for the AVR32-Linux cross configuration and testing)\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-8323\n\n- The crypto module now supports Blowfish in ECB, CBC and OFB modes. (Thanks to\n  Paul Oliver.)\n\n  Own Id: OTP-8331\n\n- The documentation is now possible to build in an open source environment after\n  a number of bugs are fixed and some features are added in the documentation\n  build process.\n\n  \\- The arity calculation is updated.\n\n  \\- The module prefix used in the function names for bif's are removed in the\n  generated links so the links will look like\n  \"http://www.erlang.org/doc/man/erlang.html#append_element-2\" instead of\n  \"http://www.erlang.org/doc/man/erlang.html#erlang:append_element-2\".\n\n  \\- Enhanced the menu positioning in the html documentation when a new page is\n  loaded.\n\n  \\- A number of corrections in the generation of man pages (thanks to Sergei\n  Golovan)\n\n  \\- The legal notice is taken from the xml book file so OTP's build process can\n  be used for non OTP applications.\n\n  Own Id: OTP-8343","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 1.6.3 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-1-6-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Suppressed false valgrind errors caused by libcrypto using uninitialized data\n  as entropy.\n\n  Own Id: OTP-8200","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- The documentation is now built with open source tools (xsltproc and fop) that\n  exists on most platforms. One visible change is that the frames are removed.\n\n  Own Id: OTP-8201\n\n- When the crypto application failed to load the OpenSSL/LibEAY shared object,\n  error indication was sparse. Now a more specific error message is sent to the\n  error logger.\n\n  Own Id: OTP-8281","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 1.6.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-1-6-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Fixed emulator crash caused by crypto using an old openssl version that did\n  not cope with large file descriptors.\n\n  Own Id: OTP-8261 Aux Id: seq11434","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 1.6.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-1-6-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- `Makefile.in` has been updated to use the LDFLAGS environment variable (if\n  set). (Thanks to Davide Pesavento.)\n\n  Own Id: OTP-8157","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Support for Blowfish cfb64 added to `crypto`.\n\n  Own Id: OTP-8096\n\n- New function `crypto:aes_cbc_ivec`\n\n  Own Id: OTP-8141","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 1.6 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-1-6"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- The `dh_compute_key` sometimes returned a SharedSecret of incorrect size.\n\n  Own Id: OTP-7674","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Optimization for drivers by creating small binaries direct on process heap.\n\n  Own Id: OTP-7762","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 1.5.3 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-1-5-3"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Added new functions: dss_verify/3, rsa_verify/3, rsa_verify/4, dss_sign/2,\n  rsa_sign/2, rsa_sign/3, rsa_public_encrypt, rsa_private_decrypt/3,\n  rsa_private_encrypt/3, rsa_public_decrypt/3, dh_generate_key/1,\n  dh_generate_key/2, dh_compute_key/3.\n\n  Own Id: OTP-7545","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 1.5.2.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-1-5-2-1"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Minor performance optimization.\n\n  Own Id: OTP-7521","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 1.5.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-1-5-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- ./configure has been improved to find 64-bit OpenSSL libraries.\n\n  Own Id: OTP-7270","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- crypto and zlib drivers improved to allow concurrent smp access.\n\n  Own Id: OTP-7262","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 1.5.1.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-1-5-1-1"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- The linked in driver for the crypto application is now linked statically\n  against the OpenSSL libraries, to avoid installation and runtime problems in\n  connection to the OpenSSL library locations.\n\n  Own Id: OTP-6680\n\n- Minor Makefile changes.\n\n  Own Id: OTP-6689","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 1.5 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-1-5"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- It is now explicitly checked at start-up that the crypto driver is properly\n  loaded (Thanks to Claes Wikstrom).\n\n  Own Id: OTP-6109","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 1.4 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-1-4"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- The previously undocumented and UNSUPPORTED `ssh` application has been updated\n  and documented. This release of the `ssh` application is still considered to\n  be a beta release and (if necessary) there could still be changes in its API\n  before it reaches 1.0.\n\n  Also, more cryptographic algorithms have been added to the `crypto`\n  application.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-5631","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 1.3 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-1-3"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Added support for RFC 3826 - The Advanced Encryption Standard (AES) Cipher\n  Algorithm in the SNMP User-based Security Model.  \n  Martin Björklund","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 1.2.3 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-1-2-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- Linked in drivers in the crypto, and asn1 applications are now compiled with\n  the -D_THREAD_SAFE and -D_REENTRANT switches on unix when the emulator has\n  thread support enabled.\n\n  Linked in drivers on MacOSX are not compiled with the undocumented -lbundle1.o\n  switch anymore. Thanks to Sean Hinde who sent us a patch.\n\n  Linked in driver in crypto, and port programs in ssl, now compiles on OSF1.\n\n  Minor makefile improvements in runtime_tools.\n\n  Own Id: OTP-5346","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 1.2.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-1-2-2"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Corrected error handling. If the port to the driver that crypto uses is\n  unexpectedly closed (which should not happen during normal operation of\n  crypto), crypto will terminate immediately (rather than crashing the next time\n  crypto is used). Also corrected build problems on Mac OS X.\n\n  Own Id: OTP-5279","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 1.2.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-1-2-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- It was not possible in R9 to relink the crypto driver. The object file was\n  missing as well as an example makefile. The crypto driver object file is now\n  released with the application (installed in priv/obj). An example makefile has\n  also been added to the priv/obj directory. The makefile serves as an example\n  of how to relink the driver on Unix (crypto_drv.so) or Windows\n  (crypto_drv.dll).\n\n  Own Id: OTP-4828 Aux Id: seq8193","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 1.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-1-2"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- Previous versions of Crypto where delivered with statically linked binaries\n  based on SSLeay. That is not longer the case. The current version of Crypto\n  requires dynamically linked OpenSSL libraries that the user has to install.\n  The library needed is `libcrypto.so` (Unix) or `libeay32.[lib|dll]` (Win32).\n  For further details see the crypto(6) application manual page.\n- This version of Crypto uses the new DES interface of OpenSSL 0.9.7, which is\n  not backward compatible with earlier versions of OpenSSL.\n- The functions `des_ede3_cbc_encrypt/5` and `des_ede3_cbc_decrypt/5` have been\n  renamed to `des3_cbc_encrypt/5` and `des3_cbc_decrypt/5`, respectively. The\n  old functions have been retained (they are deprecated and not listed in the\n  crypto(3) manual page).","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Reported Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- The start of crypto failed on Windows, due to erroneous addition of a DES3\n  algorithm.\n\n  Own Id: OTP-4684  \n  Aux Id: seq7864","ref":"notes.html#reported-fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 1.1.3 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-1-1-3"},{"type":"extras","title":"Reported Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- To obtain backward compatibility with the old SSLeay package, and with earlier\n  versions of OpenSSL, the macro OPENSSL_DES_LIBDES_COMPATIBILITY has been added\n  to `crypto_drv.c`. This is of importance only for the open source version of\n  Crypto.","ref":"notes.html#reported-fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 1.1.2 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-1-1-2"},{"type":"extras","title":"Reported Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- In the manual page `m:crypto` the function names `md5_finish` and `sha_finish`\n  have been changed to `md5_final` and `sha_final` to correctly document the\n  implementation.\n\n  Own Id: OTP-3409","ref":"notes.html#reported-fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 1.1.1 - Crypto Release Notes","doc":"Code replacement in runtime is supported. Upgrade can be done from from version\n1.1 and downgrade to version 1.1.","ref":"notes.html#crypto-1-1-1"},{"type":"extras","title":"Improvements and New Features - Crypto Release Notes","doc":"- The driver part of the Crypto application has been updated to use the\n  erl_driver header file. Version 1.1.1 requires emulator version 4.9.1 or\n  later.","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Crypto 1.1 - Crypto Release Notes","doc":"","ref":"notes.html#crypto-1-1"},{"type":"extras","title":"Reported Fixed Bugs and Malfunctions - Crypto Release Notes","doc":"- On Windows the crypto_drv was incorrectly linked to static run-time libraries\n  instead of dynamic ones.\n\n  Own Id: OTP-3240","ref":"notes.html#reported-fixed-bugs-and-malfunctions"},{"type":"extras","title":"Crypto 1.0 - Crypto Release Notes","doc":"New application.","ref":"notes.html#crypto-1-0"},{"type":"extras","title":"Licenses","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Licenses\n\n[](){: #licenses } This chapter contains in extenso versions of the OpenSSL and\nSSLeay licenses.","ref":"licenses.html"},{"type":"extras","title":"OpenSSL License - Licenses","doc":"```c\n/* ====================================================================\n * Copyright (c) 1998-2011 The OpenSSL Project.  All rights reserved.\n *\n * Redistribution and use in source and binary forms, with or without\n * modification, are permitted provided that the following conditions\n * are met:\n *\n * 1. Redistributions of source code must retain the above copyright\n *    notice, this list of conditions and the following disclaimer.\n *\n * 2. Redistributions in binary form must reproduce the above copyright\n *    notice, this list of conditions and the following disclaimer in\n *    the documentation and/or other materials provided with the\n *    distribution.\n *\n * 3. All advertising materials mentioning features or use of this\n *    software must display the following acknowledgment:\n *    \"This product includes software developed by the OpenSSL Project\n *    for use in the OpenSSL Toolkit. (http://www.openssl.org/)\"\n *\n * 4. The names \"OpenSSL Toolkit\" and \"OpenSSL Project\" must not be used to\n *    endorse or promote products derived from this software without\n *    prior written permission. For written permission, please contact\n *    openssl-core@openssl.org.\n *\n * 5. Products derived from this software may not be called \"OpenSSL\"\n *    nor may \"OpenSSL\" appear in their names without prior written\n *    permission of the OpenSSL Project.\n *\n * 6. Redistributions of any form whatsoever must retain the following\n *    acknowledgment:\n *    \"This product includes software developed by the OpenSSL Project\n *    for use in the OpenSSL Toolkit (http://www.openssl.org/)\"\n *\n * THIS SOFTWARE IS PROVIDED BY THE OpenSSL PROJECT ``AS IS'' AND ANY\n * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE\n * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR\n * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE OpenSSL PROJECT OR\n * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,\n * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT\n * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;\n * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)\n * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,\n * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)\n * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED\n * OF THE POSSIBILITY OF SUCH DAMAGE.\n * ====================================================================\n *\n * This product includes cryptographic software written by Eric Young\n * (eay@cryptsoft.com).  This product includes software written by Tim\n * Hudson (tjh@cryptsoft.com).\n *\n */\n```","ref":"licenses.html#openssl-license"},{"type":"extras","title":"SSLeay License - Licenses","doc":"```c\n/* Copyright (C) 1995-1998 Eric Young (eay@cryptsoft.com)\n * All rights reserved.\n *\n * This package is an SSL implementation written\n * by Eric Young (eay@cryptsoft.com).\n * The implementation was written so as to conform with Netscapes SSL.\n *\n * This library is free for commercial and non-commercial use as long as\n * the following conditions are adhered to.  The following conditions\n * apply to all code found in this distribution, be it the RC4, RSA,\n * lhash, DES, etc., code; not just the SSL code.  The SSL documentation\n * included with this distribution is covered by the same copyright terms\n * except that the holder is Tim Hudson (tjh@cryptsoft.com).\n *\n * Copyright remains Eric Young's, and as such any Copyright notices in\n * the code are not to be removed.\n * If this package is used in a product, Eric Young should be given attribution\n * as the author of the parts of the library used.\n * This can be in the form of a textual message at program startup or\n * in documentation (online or textual) provided with the package.\n *\n * Redistribution and use in source and binary forms, with or without\n * modification, are permitted provided that the following conditions\n * are met:\n * 1. Redistributions of source code must retain the copyright\n *    notice, this list of conditions and the following disclaimer.\n * 2. Redistributions in binary form must reproduce the above copyright\n *    notice, this list of conditions and the following disclaimer in the\n *    documentation and/or other materials provided with the distribution.\n * 3. All advertising materials mentioning features or use of this software\n *    must display the following acknowledgement:\n *    \"This product includes cryptographic software written by\n *     Eric Young (eay@cryptsoft.com)\"\n *    The word 'cryptographic' can be left out if the routines from the library\n *    being used are not cryptographic related :-).\n * 4. If you include any Windows specific code (or a derivative thereof) from\n *    the apps directory (application code) you must include an acknowledgement:\n *    \"This product includes software written by Tim Hudson (tjh@cryptsoft.com)\"\n *\n * THIS SOFTWARE IS PROVIDED BY ERIC YOUNG ``AS IS'' AND\n * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE\n * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE\n * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE\n * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL\n * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS\n * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)\n * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT\n * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY\n * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF\n * SUCH DAMAGE.\n *\n * The licence and distribution terms for any publicly available version or\n * derivative of this code cannot be changed.  i.e. this code cannot simply be\n * copied and put under another distribution licence\n * [including the GNU Public Licence.]\n */\n```","ref":"licenses.html#ssleay-license"},{"type":"extras","title":"FIPS mode","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# FIPS mode\n\n[](){: #fips } This chapter describes FIPS mode support in the crypto\napplication.","ref":"fips.html"},{"type":"extras","title":"Background - FIPS mode","doc":"OpenSSL can be built to provide FIPS 140-2 validated cryptographic services. It\nis not the OpenSSL application that is validated, but a special software\ncomponent called the OpenSSL FIPS Object Module. However applications do not use\nthis Object Module directly, but through the regular API of the OpenSSL library.\n\nThe crypto application supports using OpenSSL in FIPS mode. In this scenario\nonly the validated algorithms provided by the Object Module are accessible,\nother algorithms usually available in OpenSSL (like md5) or implemented in the\nErlang code (like SRP) are disabled.","ref":"fips.html#background"},{"type":"extras","title":"Enabling FIPS mode - FIPS mode","doc":"1. Build or install the FIPS Object Module and a FIPS enabled OpenSSL library.\n\nYou should read and precisely follow the instructions of the\n[Security Policy](http://csrc.nist.gov/groups/STM/cmvp/documents/140-1/140sp/140sp1747.pdf)\nand [User Guide](https://www.openssl.org/docs/fips/UserGuide-2.0.pdf).\n\n> #### Warning {: .warning }\n>\n> It is very easy to build a working OpenSSL FIPS Object Module and library from\n> the source. However it _does not_ qualify as FIPS 140-2 validated if the\n> numerous restrictions in the Security Policy are not properly followed.\n\n1. Configure and build Erlang/OTP with FIPS support:\n\n```text\n$ cd $ERL_TOP\n$ ./otp_build configure --enable-fips\n...\nchecking for FIPS_mode_set... yes\n...\n$ make\n```\n\nIf `FIPS_mode_set` returns `no` the OpenSSL library is not FIPS enabled and\ncrypto won't support FIPS mode either.\n\n1. Set the `fips_mode` configuration setting of the crypto application to `true`\n   _before loading the crypto module_.\n\nThe best place is in the `sys.config` system configuration file of the release.\n\n1. Start and use the crypto application as usual. However take care to avoid the\n   non-FIPS validated algorithms, they will all throw exception `not_supported`.\n\nEntering and leaving FIPS mode on a node already running crypto is not\nsupported. The reason is that OpenSSL is designed to prevent an application\nrequesting FIPS mode to end up accidentally running in non-FIPS mode. If\nentering FIPS mode fails (e.g. the Object Module is not found or is compromised)\nany subsequent use of the OpenSSL API would terminate the emulator.\n\nAn on-the-fly FIPS mode change would thus have to be performed in a critical\nsection protected from any concurrently running crypto operations. Furthermore\nin case of failure all crypto calls would have to be disabled from the Erlang or\nnif code. This would be too much effort put into this not too important feature.","ref":"fips.html#enabling-fips-mode"},{"type":"extras","title":"Incompatibilities with regular builds - FIPS mode","doc":"The Erlang API of the crypto application is identical regardless of building\nwith or without FIPS support. However the nif code internally uses a different\nOpenSSL API.\n\nThis means that the context (an opaque type) returned from streaming crypto\nfunctions (`hash_(init|update|final)`, `hmac_(init|update|final)` and\n`stream_(init|encrypt|decrypt)`) is different and incompatible with regular\nbuilds when compiling crypto with FIPS support.","ref":"fips.html#incompatibilities-with-regular-builds"},{"type":"extras","title":"Common caveats - FIPS mode","doc":"In FIPS mode non-validated algorithms are disabled. This may cause some\nunexpected problems in application relying on crypto.\n\n> #### Warning {: .warning }\n>\n> Do not try to work around these problems by using alternative implementations\n> of the missing algorithms\\! An application can only claim to be using a FIPS\n> 140-2 validated cryptographic module if it uses it exclusively for every\n> cryptographic operation.","ref":"fips.html#common-caveats"},{"type":"extras","title":"Restrictions on key sizes - FIPS mode","doc":"Although public key algorithms are supported in FIPS mode they can only be used\nwith secure key sizes. The Security Policy requires the following minimum\nvalues:\n\n- **RSA** - 1024 bit\n\n- **DSS** - 1024 bit\n\n- **EC algorithms** - 160 bit","ref":"fips.html#restrictions-on-key-sizes"},{"type":"extras","title":"Restrictions on elliptic curves - FIPS mode","doc":"The Erlang API allows using arbitrary curve parameters, but in FIPS mode only\nthose allowed by the Security Policy shall be used.","ref":"fips.html#restrictions-on-elliptic-curves"},{"type":"extras","title":"Avoid md5 for hashing - FIPS mode","doc":"Md5 is a popular choice as a hash function, but it is not secure enough to be\nvalidated. Try to use sha instead wherever possible.\n\nFor exceptional, non-cryptographic use cases one may consider switching to\n`erlang:md5/1` as well.","ref":"fips.html#avoid-md5-for-hashing"},{"type":"extras","title":"Certificates and encrypted keys - FIPS mode","doc":"As md5 is not available in FIPS mode it is only possible to use certificates\nthat were signed using sha hashing. When validating an entire certificate chain\nall certificates (including the root CA's) must comply with this rule.\n\nFor similar dependency on the md5 and des algorithms most encrypted private keys\nin PEM format do not work either. However, the PBES2 encryption scheme allows\nthe use of stronger FIPS verified algorithms which is a viable alternative.","ref":"fips.html#certificates-and-encrypted-keys"},{"type":"extras","title":"SNMP v3 limitations - FIPS mode","doc":"It is only possible to use `usmHMACSHAAuthProtocol` and `usmAesCfb128Protocol`\nfor authentication and privacy respectively in FIPS mode. The snmp application\nhowever won't restrict selecting disabled protocols in any way, and using them\nwould result in run time crashes.","ref":"fips.html#snmp-v3-limitations"},{"type":"extras","title":"TLS 1.2 is required - FIPS mode","doc":"All SSL and TLS versions prior to TLS 1.2 use a combination of md5 and sha1\nhashes in the handshake for various purposes:\n\n- Authenticating the integrity of the handshake messages.\n- In the exchange of DH parameters in cipher suites providing non-anonymous PFS\n  (perfect forward secrecy).\n- In the PRF (pseud-random function) to generate keying materials in cipher\n  suites not using PFS.\n\nOpenSSL handles these corner cases in FIPS mode, however the Erlang crypto and\nssl applications are not prepared for them and therefore you are limited to TLS\n1.2 in FIPS mode.\n\nOn the other hand it worth mentioning that at least all cipher suites that would\nrely on non-validated algorithms are automatically disabled in FIPS mode.\n\n> #### Note {: .info }\n>\n> Certificates using weak (md5) digests may also cause problems in TLS. Although\n> TLS 1.2 has an extension for specifying which type of signatures are accepted,\n> and in FIPS mode the ssl application will use it properly, most TLS\n> implementations ignore this extension and simply send whatever certificates\n> they were configured with.","ref":"fips.html#tls-1-2-is-required"},{"type":"extras","title":"Engine Load","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Engine Load\n\n[](){: #engine_load } This chapter describes the support for loading encryption\nengines in the crypto application.","ref":"engine_load.html"},{"type":"extras","title":"Background - Engine Load","doc":"OpenSSL exposes an Engine API, which makes it possible to plug in alternative\nimplementations for some or all of the cryptographic operations implemented by\nOpenSSL. When configured appropriately, OpenSSL calls the engine's\nimplementation of these operations instead of its own.\n\nTypically, OpenSSL engines provide a hardware implementation of specific\ncryptographic operations. The hardware implementation usually offers improved\nperformance over its software-based counterpart, which is known as cryptographic\nacceleration.\n\n> #### Note {: .info }\n>\n> The file name requirement on the engine dynamic library can differ between SSL\n> versions.","ref":"engine_load.html#background"},{"type":"extras","title":"Use Cases - Engine Load","doc":"","ref":"engine_load.html#use-cases"},{"type":"extras","title":"Dynamically load an engine from default directory - Engine Load","doc":"If the engine is located in the OpenSSL/LibreSSL installation `engines`\ndirectory.\n\n```erlang\n1> {ok, Engine} = crypto:engine_load(<<\"otp_test_engine\">>, [], []).\n {ok, #Ref}\n```","ref":"engine_load.html#dynamically-load-an-engine-from-default-directory"},{"type":"extras","title":"Load an engine with the dynamic engine - Engine Load","doc":"Load an engine with the help of the dynamic engine by giving the path to the\nlibrary.\n\n```erlang\n 2> {ok, Engine} = crypto:engine_load(<<\"dynamic\">>,\n                                      [{<<\"SO_PATH\">>,\n                                        <<\"/some/path/otp_test_engine.so\">>},\n                                       {<<\"ID\">>, <<\"MD5\">>},\n                                       <<\"LOAD\">>],\n                                      []).\n {ok, #Ref}\n```","ref":"engine_load.html#load-an-engine-with-the-dynamic-engine"},{"type":"extras","title":"Load an engine and replace some methods - Engine Load","doc":"Load an engine with the help of the dynamic engine and just replace some engine\nmethods.\n\n```erlang\n 3> {ok, Engine} = crypto:engine_load(<<\"dynamic\">>,\n                                      [{<<\"SO_PATH\">>,\n                                        <<\"/some/path/otp_test_engine.so\">>},\n                                       {<<\"ID\">>, <<\"MD5\">>},\n                                       <<\"LOAD\">>],\n                                      []).\n{ok, #Ref}\n4> ok = crypto:engine_register(Engine, [engine_method_digests]).\nok\n```","ref":"engine_load.html#load-an-engine-and-replace-some-methods"},{"type":"extras","title":"Load with the ensure loaded function - Engine Load","doc":"This function makes sure the engine is loaded just once and the ID is added to\nthe internal engine list of OpenSSL. The following calls to the function will\ncheck if the ID is loaded and then just get a new reference to the engine.\n\n```erlang\n 5> {ok, Engine} = crypto:ensure_engine_loaded(<<\"MD5\">>,\n                                               <<\"/some/path/otp_test_engine.so\">>).\n {ok, #Ref}\n```\n\nTo remove the tag from the OpenSSL engine list use `crypto:engine_remove/1`.\n\n```erlang\n 6> crypto:engine_remove(Engine).\n ok\n```\n\nTo unload it use `crypto:engine_unload/1` which removes the references to the\nengine.\n\n```erlang\n 6> crypto:engine_unload(Engine).\n ok\n```","ref":"engine_load.html#load-with-the-ensure-loaded-function"},{"type":"extras","title":"List all engines currently loaded - Engine Load","doc":"```erlang\n 8> crypto:engine_list().\n[<<\"dynamic\">>, <<\"MD5\">>]\n```","ref":"engine_load.html#list-all-engines-currently-loaded"},{"type":"extras","title":"Engine Stored Keys","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Engine Stored Keys\n\n[](){: #engine_key } This chapter describes the support in the crypto\napplication for using public and private keys stored in encryption engines.","ref":"engine_keys.html"},{"type":"extras","title":"Background - Engine Stored Keys","doc":"[OpenSSL](https://www.openssl.org/) exposes an Engine API, which makes it\npossible to plug in alternative implementations for some of the cryptographic\noperations implemented by OpenSSL. See the chapter\n[Engine Load](engine_load.md#engine_load) for details and how to load an Engine.\n\nAn engine could among other tasks provide a storage for private or public keys.\nSuch a storage could be made safer than the normal file system. Those techniques\nare not described in this User's Guide. Here we concentrate on how to use\nprivate or public keys stored in such an engine.\n\nThe storage engine must call `ENGINE_set_load_privkey_function` and\n`ENGINE_set_load_pubkey_function`. See the OpenSSL cryptolib's\n[manpages](https://www.openssl.org/docs/manpages.html).\n\nOTP/Crypto requires that the user provides two or three items of information\nabout the key. The application used by the user is usually on a higher level,\nfor example in [SSL](`t:ssl:key/0`). If using the crypto application directly,\nit is required that:\n\n- an Engine is loaded, see the chapter on\n  [Engine Load](engine_load.md#engine_load) or the\n  [Reference Manual](`crypto:engine_load/3`)\n- a reference to a key in the Engine is available. This should be an Erlang\n  string or binary and depends on the Engine loaded\n- an Erlang map is constructed with the Engine reference, the key reference and\n  possibly a key passphrase if needed by the Engine. See the\n  [Reference Manual](`t:crypto:engine_key_ref/0`) for details of the map.","ref":"engine_keys.html#background"},{"type":"extras","title":"Use Cases - Engine Stored Keys","doc":"","ref":"engine_keys.html#use-cases"},{"type":"extras","title":"Sign with an engine stored private key - Engine Stored Keys","doc":"This example shows how to construct a key reference that is used in a sign\noperation. The actual key is stored in the engine that is loaded at prompt 1.\n\n```erlang\n1> {ok, EngineRef} = crypto:engine_load(....).\n...\n{ok,#Ref<0.2399045421.3028942852.173962>}\n2> PrivKey = #{engine => EngineRef,\n               key_id => \"id of the private key in Engine\"}.\n...\n3> Signature = crypto:sign(rsa, sha, <<\"The message\">>, PrivKey).\n<<65,6,125,254,54,233,84,77,83,63,168,28,169,214,121,76,\n  207,177,124,183,156,185,160,243,36,79,125,230,231,...>>\n```","ref":"engine_keys.html#sign-with-an-engine-stored-private-key"},{"type":"extras","title":"Verify with an engine stored public key - Engine Stored Keys","doc":"Here the signature and message in the last example is verifyed using the public\nkey. The public key is stored in an engine, only to exemplify that it is\npossible. The public key could of course be handled openly as usual.\n\n```erlang\n4> PublicKey = #{engine => EngineRef,\n                 key_id => \"id of the public key in Engine\"}.\n...\n5> crypto:verify(rsa, sha, <<\"The message\">>, Signature, PublicKey).\ntrue\n6>\n```","ref":"engine_keys.html#verify-with-an-engine-stored-public-key"},{"type":"extras","title":"Using a password protected private key - Engine Stored Keys","doc":"The same example as the first sign example, except that a password protects the\nkey down in the Engine.\n\n```erlang\n6> PrivKeyPwd = #{engine => EngineRef,\n                  key_id => \"id of the pwd protected private key in Engine\",\n\t\t  password => \"password\"}.\n...\n7> crypto:sign(rsa, sha, <<\"The message\">>, PrivKeyPwd).\n<<140,80,168,101,234,211,146,183,231,190,160,82,85,163,\n  175,106,77,241,141,120,72,149,181,181,194,154,175,76,\n  223,...>>\n8>\n```","ref":"engine_keys.html#using-a-password-protected-private-key"},{"type":"extras","title":"Algorithm Details","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Algorithm Details\n\nThis chapter describes details of algorithms in the crypto application.\n\nThe tables only documents the supported cryptos and key lengths. The user should\nnot draw any conclusion on security from the supplied tables.","ref":"algorithm_details.html"},{"type":"extras","title":"Ciphers - Algorithm Details","doc":"A [cipher](`t:crypto:cipher/0`) in the [new api](new_api.md#the-new-api) is\ncategorized as either [cipher_no_iv()](`t:crypto:cipher_no_iv/0`),\n[cipher_iv()](`t:crypto:cipher_iv/0`) or\n[cipher_aead()](`t:crypto:cipher_aead/0`). The letters IV are short for\n_Initialization Vector_ and AEAD is an abbreviation of _Authenticated Encryption\nwith Associated Data_.\n\nDue to irregular naming conventions, some cipher names in the old api are\nsubstituted by new names in the new api. For a list of retired names, see\n[Retired cipher names](new_api.md#retired-cipher-names).\n\nTo dynamically check availability, check that the name in the _Cipher and Mode_\ncolumn is present in the list returned by\n[crypto:supports(ciphers)](`crypto:supports/1`).","ref":"algorithm_details.html#ciphers"},{"type":"extras","title":"Ciphers without an IV - cipher_no_iv() - Algorithm Details","doc":"To be used with:\n\n- [crypto_one_time/4](`crypto:crypto_one_time/4`)\n- [crypto_init/3](`crypto:crypto_init/3`)\n\nThe ciphers are:\n\n| **Cipher and Mode** | **Key length** **\\[bytes]** | **Block size** **\\[bytes]** |\n| ------------------- | --------------------------- | --------------------------- |\n| `aes_128_ecb`       | 16                          | 16                          |\n| `aes_192_ecb`       | 24                          | 16                          |\n| `aes_256_ecb`       | 32                          | 16                          |\n| `blowfish_ecb`      | 16                          |  8                          |\n| `des_ecb`           |  8                          |  8                          |\n| `rc4`               | 16                          |  1                          |\n| `sm4_ecb`           | 16                          | 16                          |\n\n_Table: Ciphers without IV_","ref":"algorithm_details.html#ciphers-without-an-iv-cipher_no_iv"},{"type":"extras","title":"Ciphers with an IV - cipher_iv() - Algorithm Details","doc":"To be used with:\n\n- [crypto_one_time/5](`crypto:crypto_one_time/5`)\n- [crypto_init/4](`crypto:crypto_init/4`)\n\nThe ciphers are:\n\n| **Cipher and Mode** | **Key length** **\\[bytes]** | **IV length** **\\[bytes]** | **Block size** **\\[bytes]** | **Limited to** **OpenSSL versions** |\n| ------------------- | --------------------------- | -------------------------- | --------------------------- | ----------------------------------- |\n| `aes_128_cbc`       | 16                          | 16                         | 16                          |                                     |\n| `aes_192_cbc`       | 24                          | 16                         | 16                          |                                     |\n| `aes_256_cbc`       | 32                          | 16                         | 16                          |                                     |\n| `aes_128_cfb8`      | 16                          | 16                         |  1                          |                                     |\n| `aes_192_cfb8`      | 24                          | 16                         |  1                          |                                     |\n| `aes_256_cfb8`      | 32                          | 16                         |  1                          |                                     |\n| `aes_128_cfb128`    | 16                          | 16                         |  1                          |                                     |\n| `aes_192_cfb128`    | 24                          | 16                         |  1                          |                                     |\n| `aes_256_cfb128`    | 32                          | 16                         |  1                          |                                     |\n| `aes_128_ctr`       | 16                          | 16                         |  1                          |                                     |\n| `aes_192_ctr`       | 24                          | 16                         |  1                          |                                     |\n| `aes_256_ctr`       | 32                          | 16                         |  1                          |                                     |\n| `aes_128_ofb`       | 16                          | 16                         |  1                          |                                     |\n| `aes_192_ofb`       | 24                          | 16                         |  1                          |                                     |\n| `aes_256_ofb`       | 32                          | 16                         |  1                          |                                     |\n| `blowfish_cbc`      | 16                          |  8                         |  8                          |                                     |\n| `blowfish_cfb64`    | 16                          |  8                         |  1                          |                                     |\n| `blowfish_ofb64`    | 16                          |  8                         |  1                          |                                     |\n| `chacha20`          | 32                          | 16                         |  1                          | ≥1.1.0d                             |\n| `des_cbc`           |  8                          |  8                         |  8                          |                                     |\n| `des_ede3_cbc`      | 24                          |  8                         |  8                          |                                     |\n| `des_cfb`           |  8                          |  8                         |  1                          |                                     |\n| `des_ede3_cfb`      | 24                          |  8                         |  1                          |                                     |\n| `rc2_cbc`           | 16                          |  8                         |  8                          |                                     |\n| `sm4_cbc`           | 16                          | 16                         | 16                          | ≥1.1.1                              |\n| `sm4_cfb`           | 16                          | 16                         | 16                          | ≥1.1.1                              |\n| `sm4_ofb`           | 16                          | 16                         | 16                          | ≥1.1.1                              |\n| `sm4_ctr`           | 16                          | 16                         | 16                          | ≥1.1.1                              |\n\n_Table: Ciphers with IV_","ref":"algorithm_details.html#ciphers-with-an-iv-cipher_iv"},{"type":"extras","title":"Ciphers with AEAD - cipher_aead() - Algorithm Details","doc":"To be used with:\n\n- [crypto_one_time_aead/6](`crypto:crypto_one_time_aead/6`)\n- [crypto_one_time_aead/7](`crypto:crypto_one_time_aead/7`)\n\nThe ciphers are:\n\n| **Cipher and Mode** | **Key length** **\\[bytes]** | **IV length** **\\[bytes]** | **AAD length** **\\[bytes]** | **Tag length** **\\[bytes]** | **Block size** **\\[bytes]** | **Limited to** **OpenSSL versions** |\n| ------------------- | --------------------------- | -------------------------- | --------------------------- | --------------------------- | --------------------------- | ----------------------------------- |\n| `aes_128_ccm`       | 16                          | 7-13                       | any                         | even 4-16 default: 12       | any                         | ≥1.0.1                              |\n| `aes_192_ccm`       | 24                          | 7-13                       | any                         | even 4-16 default: 12       | any                         | ≥1.0.1                              |\n| `aes_256_ccm`       | 32                          | 7-13                       | any                         | even 4-16 default: 12       | any                         | ≥1.0.1                              |\n| `aes_128_gcm`       | 16                          | ≥1                         | any                         | 1-16 default: 16            | any                         | ≥1.0.1                              |\n| `aes_192_gcm`       | 24                          | ≥1                         | any                         | 1-16 default: 16            | any                         | ≥1.0.1                              |\n| `aes_256_gcm`       | 32                          | ≥1                         | any                         | 1-16 default: 16            | any                         | ≥1.0.1                              |\n| `chacha20_poly1305` | 32                          | 1-16                       | any                         | 16                          | any                         | ≥1.1.0                              |\n| `sm4_gcm`           | 16                          | 12                         | any                         | 16                          | any                         | ≥3.1.0                              |\n| `sm4_ccm`           | 16                          | 12                         | any                         | 16                          | any                         | ≥3.1.0                              |\n\n_Table: AEAD ciphers_","ref":"algorithm_details.html#ciphers-with-aead-cipher_aead"},{"type":"extras","title":"Message Authentication Codes (MACs) - Algorithm Details","doc":"To be used in [mac/4](`crypto:mac/4`) and\n[related functions](new_api.md#macs-message-authentication-codes).","ref":"algorithm_details.html#message-authentication-codes-macs"},{"type":"extras","title":"CMAC - Algorithm Details","doc":"CMAC with the following ciphers are available with OpenSSL 1.0.1 or later if not\ndisabled by configuration.\n\nTo dynamically check availability, check that the name `cmac` is present in the\nlist returned by [crypto:supports(macs)](`crypto:supports/1`). Also check that\nthe name in the _Cipher and Mode_ column is present in the list returned by\n[crypto:supports(ciphers)](`crypto:supports/1`).\n\n| **Cipher and Mode** | **Key length** **\\[bytes]** | **Max Mac Length** **(= default length)** **\\[bytes]** |\n| ------------------- | --------------------------- | ------------------------------------------------------ |\n| `aes_128_cbc`       | 16                          | 16                                                     |\n| `aes_192_cbc`       | 24                          | 16                                                     |\n| `aes_256_cbc`       | 32                          | 16                                                     |\n| `aes_128_ecb`       | 16                          | 16                                                     |\n| `aes_192_ecb`       | 24                          | 16                                                     |\n| `aes_256_ecb`       | 32                          | 16                                                     |\n| `blowfish_cbc`      | 16                          |  8                                                     |\n| `blowfish_ecb`      | 16                          |  8                                                     |\n| `des_cbc`           |  8                          |  8                                                     |\n| `des_ecb`           |  8                          |  8                                                     |\n| `des_ede3_cbc`      | 24                          |  8                                                     |\n| `rc2_cbc`           | 16                          |  8                                                     |\n\n_Table: CMAC cipher key lengths_","ref":"algorithm_details.html#cmac"},{"type":"extras","title":"HMAC - Algorithm Details","doc":"Available in all OpenSSL compatible with Erlang CRYPTO if not disabled by\nconfiguration.\n\nTo dynamically check availability, check that the name `hmac` is present in the\nlist returned by [crypto:supports(macs)](`crypto:supports/1`) and that the hash\nname is present in the list returned by\n[crypto:supports(hashs)](`crypto:supports/1`).\n\n| **Hash**    | **Max Mac Length** **(= default length)** **\\[bytes]** |\n| ----------- | ------------------------------------------------------ |\n| `sha`       | 20                                                     |\n| `sha224`    | 28                                                     |\n| `sha256`    | 32                                                     |\n| `sha384`    | 48                                                     |\n| `sha512`    | 64                                                     |\n| `sha3_224`  | 28                                                     |\n| `sha3_256`  | 32                                                     |\n| `sha3_384`  | 48                                                     |\n| `sha3_512`  | 64                                                     |\n| `shake128`  | 64                                                     |\n| `shake256`  | 64                                                     |\n| `blake2b`   | 64                                                     |\n| `blake2s`   | 32                                                     |\n| `md4`       | 16                                                     |\n| `md5`       | 16                                                     |\n| `ripemd160` | 20                                                     |\n\n_Table: HMAC output sizes_","ref":"algorithm_details.html#hmac"},{"type":"extras","title":"POLY1305 - Algorithm Details","doc":"POLY1305 is available with OpenSSL 1.1.1 or later if not disabled by\nconfiguration.\n\nTo dynamically check availability, check that the name `poly1305` is present in\nthe list returned by [crypto:supports(macs)](`crypto:supports/1`).\n\nThe poly1305 mac wants an 32 bytes key and produces a 16 byte MAC by default.","ref":"algorithm_details.html#poly1305"},{"type":"extras","title":"Hash - Algorithm Details","doc":"To dynamically check availability, check that the wanted name in the _Names_\ncolumn is present in the list returned by\n[crypto:supports(hashs)](`crypto:supports/1`).\n\n| **Type** | **Names**                                                  | **Limited to** **OpenSSL versions** |\n| -------- | ---------------------------------------------------------- | ----------------------------------- |\n| SHA1     | sha                                                        |                                     |\n| SHA2     | sha224, sha256, sha384, sha512                             |                                     |\n| SHA3     | sha3_224, sha3_256, sha3_384, sha3_512, shake128, shake256 | ≥1.1.1                              |\n| SM3      | sm3                                                        | ≥1.1.1                              |\n| MD4      | md4                                                        |                                     |\n| MD5      | md5                                                        |                                     |\n| RIPEMD   | ripemd160                                                  |                                     |","ref":"algorithm_details.html#hash"},{"type":"extras","title":"Public Key Cryptography - Algorithm Details","doc":"","ref":"algorithm_details.html#public-key-cryptography"},{"type":"extras","title":"RSA - Algorithm Details","doc":"RSA is available with all OpenSSL versions compatible with Erlang CRYPTO if not\ndisabled by configuration. To dynamically check availability, check that the\natom `rsa` is present in the list returned by\n[crypto:supports(public_keys)](`crypto:supports/1`).\n\n> #### Warning {: .warning }\n>\n> The RSA options are experimental.\n>\n> The exact set of options and there syntax _may_ be changed without prior\n> notice.\n\n| **Option**                                                                                                            | **sign/verify**   | **public encrypt** **private decrypt** | **private encrypt** **public decrypt** |\n| --------------------------------------------------------------------------------------------------------------------- | ----------------- | -------------------------------------- | -------------------------------------- |\n| \\{rsa_padding,rsa_x931_padding\\}                                                                                      | x                 |                                        | x                                      |\n| \\{rsa_padding,rsa_pkcs1_padding\\}                                                                                     | x                 | x                                      | x                                      |\n| \\{rsa_padding,rsa_pkcs1_pss_padding\\} \\{rsa_pss_saltlen, -2..\\} \\{rsa_mgf1_md, atom()\\}                               | x (2) x (2) x (2) |                                        |                                        |\n| \\{rsa_padding,rsa_pkcs1_oaep_padding\\} \\{rsa_mgf1_md, atom()\\} \\{rsa_oaep_label, binary()\\}\\} \\{rsa_oaep_md, atom()\\} |                   | x (2) x (2) x (3) x (3)                |                                        |\n| \\{rsa_padding,rsa_no_padding\\}                                                                                        | x (1)             |                                        |                                        |\n\nNotes:\n\n1. (1) OpenSSL ≤ 1.0.0\n1. (2) OpenSSL ≥ 1.0.1\n1. (3) OpenSSL ≥ 1.1.0","ref":"algorithm_details.html#rsa"},{"type":"extras","title":"DSS - Algorithm Details","doc":"DSS is available with OpenSSL versions compatible with Erlang CRYPTO if not\ndisabled by configuration. To dynamically check availability, check that the\natom `dss` is present in the list returned by\n[crypto:supports(public_keys)](`crypto:supports/1`).","ref":"algorithm_details.html#dss"},{"type":"extras","title":"ECDSA - Algorithm Details","doc":"ECDSA is available with OpenSSL 0.9.8o or later if not disabled by\nconfiguration. To dynamically check availability, check that the atom `ecdsa` is\npresent in the list returned by\n[crypto:supports(public_keys)](`crypto:supports/1`). If the atom `ec_gf2m` also\nis present, the characteristic two field curves are available.\n\nThe actual supported named curves could be checked by examining the list\nreturned by [crypto:supports(curves)](`crypto:supports/1`).","ref":"algorithm_details.html#ecdsa"},{"type":"extras","title":"EdDSA - Algorithm Details","doc":"EdDSA is available with OpenSSL 1.1.1 or later if not disabled by configuration.\nTo dynamically check availability, check that the atom `eddsa` is present in the\nlist returned by [crypto:supports(public_keys)](`crypto:supports/1`).\n\nSupport for the curves ed25519 and ed448 is implemented. The actual supported\nnamed curves could be checked by examining the list with the list returned by\n[crypto:supports(curves)](`crypto:supports/1`).","ref":"algorithm_details.html#eddsa"},{"type":"extras","title":"Diffie-Hellman - Algorithm Details","doc":"Diffie-Hellman computations are available with OpenSSL versions compatible with\nErlang CRYPTO if not disabled by configuration. To dynamically check\navailability, check that the atom `dh` is present in the list returned by\n[crypto:supports(public_keys)](`crypto:supports/1`).","ref":"algorithm_details.html#diffie-hellman"},{"type":"extras","title":"Elliptic Curve Diffie-Hellman - Algorithm Details","doc":"Elliptic Curve Diffie-Hellman is available with OpenSSL 0.9.8o or later if not\ndisabled by configuration. To dynamically check availability, check that the\natom `ecdh` is present in the list returned by\n[crypto:supports(public_keys)](`crypto:supports/1`).\n\nThe Edward curves `x25519` and `x448` are supported with OpenSSL 1.1.1 or later\nif not disabled by configuration.\n\nThe actual supported named curves could be checked by examining the list\nreturned by [crypto:supports(curves)](`crypto:supports/1`).","ref":"algorithm_details.html#elliptic-curve-diffie-hellman"},{"type":"extras","title":"New and Old API","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# New and Old API\n\nThis chapter describes the new api to encryption and decryption.","ref":"new_api.html"},{"type":"extras","title":"Background - New and Old API","doc":"The CRYPTO app has evolved during its lifetime. Since also the OpenSSL cryptolib\nhas changed the API several times, there are parts of the CRYPTO app that uses a\nvery old one internally and other parts that uses the latest one. The internal\ndefinitions of e.g cipher names was a bit hard to maintain.\n\nIt turned out that using the old api in the new way (more about that later), and\nstill keep it backwards compatible, was not possible. Specially as more\nprecision in the error messages is desired it could not be combined with the old\nstandard.\n\nTherefore the old api (see next section) is kept for now but internally\nimplemented with new primitives.","ref":"new_api.html#background"},{"type":"extras","title":"The old API - New and Old API","doc":"The old functions - deprecated from 23.0 and removed from OTP 24.0 - are for\nciphers:\n\n- `block_encrypt/3`\n- `block_encrypt/4`\n- `block_decrypt/3`\n- `block_decrypt/4`\n- `stream_init/2`\n- `stream_init/3`\n- `stream_encrypt/2`\n- `stream_decrypt/2`\n- `next_iv/2`\n- `next_iv/3`\n\nfor lists of supported algorithms:\n\n- `supports/0`\n\nand for MACs (Message Authentication Codes):\n\n- `cmac/3`\n- `cmac/4`\n- `hmac/3`\n- `hmac/4`\n- `hmac_init/2`\n- `hmac_update/2`\n- `hmac_final/1`\n- `hmac_final_n/2`\n- `poly1305/2`","ref":"new_api.html#the-old-api"},{"type":"extras","title":"The new API - New and Old API","doc":"","ref":"new_api.html#the-new-api"},{"type":"extras","title":"Encryption and decryption - New and Old API","doc":"The new functions for encrypting or decrypting one single binary are:\n\n- [crypto_one_time/4](`crypto:crypto_one_time/4`)\n- [crypto_one_time/5](`crypto:crypto_one_time/5`)\n- [crypto_one_time_aead/6](`crypto:crypto_one_time_aead/6`)\n- [crypto_one_time_aead/7](`crypto:crypto_one_time_aead/7`)\n\nIn those functions the internal crypto state is first created and initialized\nwith the cipher type, the key and possibly other data. Then the single binary is\nencrypted or decrypted, the crypto state is de-allocated and the result of the\ncrypto operation is returned.\n\nThe `crypto_one_time_aead` functions are for the ciphers of mode `ccm` or `gcm`,\nand for the cipher `chacha20-poly1305`.\n\nFor repeated encryption or decryption of a text divided in parts, where the\ninternal crypto state is initialized once, and then many binaries are encrypted\nor decrypted with the same state, the functions are:\n\n- [crypto_init/4](`crypto:crypto_init/4`)\n- [crypto_init/3](`crypto:crypto_init/3`)\n- [crypto_update/2](`crypto:crypto_update/2`)\n- [crypto_final/1](`crypto:crypto_final/1`)\n\nThe `crypto_init` initialies an internal cipher state, and one or more calls of\n`crypto_update` does the actual encryption or decryption. Note that AEAD ciphers\ncan't be handled this way due to their nature.\n\nAn example of where those functions are needed, is when handling the TLS\nprotocol.\n\nIf padding was not enabled, the call to\n[crypto_final/1](`crypto:crypto_final/1`) may be excluded.\n\nFor information about available algorithms, use:\n\n- [supports/1](`crypto:supports/1`)\n- [hash_info/1](`crypto:hash_info/1`)\n- [cipher_info/1](`crypto:cipher_info/1`)\n\nThe `next_iv/2` and `next_iv/3` are not needed since the `crypto_init` and\n`crypto_update` includes this functionality.","ref":"new_api.html#encryption-and-decryption"},{"type":"extras","title":"MACs (Message Authentication Codes) - New and Old API","doc":"The new functions for calculating a MAC of a single piece of text are:\n\n- [mac/3](`crypto:mac/3`)\n- [mac/4](`crypto:mac/4`)\n- [macN/4](`crypto:macN/4`)\n- [macN/5](`crypto:macN/5`)\n\nFor calculating a MAC of a text divided in parts use:\n\n- [mac_init/2](`crypto:mac_init/2`)\n- [mac_init/3](`crypto:mac_init/3`)\n- [mac_update/2](`crypto:mac_update/2`)\n- [mac_final/1](`crypto:mac_final/1`)\n- [mac_finalN/2](`crypto:mac_finalN/2`)","ref":"new_api.html#macs-message-authentication-codes"},{"type":"extras","title":"Examples of the new api - New and Old API","doc":"","ref":"new_api.html#examples-of-the-new-api"},{"type":"extras","title":"Examples of crypto_init/4 and crypto_update/2 - New and Old API","doc":"The functions [crypto_init/4](`crypto:crypto_init/4`) and\n[crypto_update/2](`crypto:crypto_update/2`) are intended to be used for\nencrypting or decrypting a sequence of blocks. First one call of `crypto_init/4`\ninitialises the crypto context. One or more calls `crypto_update/2` does the\nactual encryption or decryption for each block.\n\nThis example shows first the encryption of two blocks and then decryptions of\nthe cipher text, but divided into three blocks just to show that it is possible\nto divide the plain text and cipher text differently for some ciphers:\n\n```erlang\n\t1> crypto:start().\n\tok\n\t2> Key = <<1:128>>.\n\t<<0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1>>\n\t3> IV = <<0:128>>.\n\t<<0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0>>\n\t4> StateEnc = crypto:crypto_init(aes_128_ctr, Key, IV, true). % encrypt -> true\n\t#Ref<0.3768901617.1128660993.124047>\n\t5> crypto:crypto_update(StateEnc, <<\"First bytes\">>).\n\t<<67,44,216,166,25,130,203,5,66,6,162>>\n\t6> crypto:crypto_update(StateEnc, <<\"Second bytes\">>).\n\t<<16,79,94,115,234,197,94,253,16,144,151,41>>\n\t7>\n\t7> StateDec = crypto:crypto_init(aes_128_ctr, Key, IV, false). % decrypt -> false\n\t#Ref<0.3768901617.1128660994.124255>\n\t8> crypto:crypto_update(StateDec, <<67,44,216,166,25,130,203>>).\n\t<<\"First b\">>\n\t9> crypto:crypto_update(StateDec, <<5,66,6,162,16,79,94,115,234,197,\n        94,253,16,144,151>>).\n\t<<\"ytesSecond byte\">>\n\t10> crypto:crypto_update(StateDec, <<41>>).\n\t<<\"s\">>\n\t11>\n```\n\nNote that the internal data that the `StateEnc` and `StateDec` references are\ndestructivly updated by the calls to\n[crypto_update/2](`crypto:crypto_update/2`). This is to gain time in the calls\nof the nifs interfacing the cryptolib. In a loop where the state is saved in the\nloop's state, it also saves one update of the loop state per crypto operation.\n\nFor example, a simple server receiving text parts to encrypt and send the result\nback to the one who sent them (the `Requester`):\n\n```erlang\n\tencode(Crypto, Key, IV) ->\n\tcrypto_loop(crypto:crypto_init(Crypto, Key, IV, true)).\n\n\tcrypto_loop(State) ->\n\treceive\n        {Text, Requester} ->\n        Requester ! crypto:crypto_update(State, Text),\n\tloop(State)\n\tend.\n```","ref":"new_api.html#examples-of-crypto_init-4-and-crypto_update-2"},{"type":"extras","title":"Example of crypto_one_time/5 - New and Old API","doc":"The same example as in the\n[previous section](new_api.md#examples-of-crypto_init-4-and-crypto_update-2),\nbut now with one call to [crypto_one_time/5](`crypto:crypto_one_time/5`):\n\n```erlang\n\t1> Key = <<1:128>>.\n\t<<0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1>>\n\t2> IV = <<0:128>>.\n\t<<0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0>>\n\t3> Txt = [<<\"First bytes\">>,<<\"Second bytes\">>].\n\t[<<\"First bytes\">>,<<\"Second bytes\">>]\n\t4> crypto:crypto_one_time(aes_128_ctr, Key, IV, Txt, true).\n\t<<67,44,216,166,25,130,203,5,66,6,162,16,79,94,115,234,\n\t197,94,253,16,144,151,41>>\n\t5>\n```\n\nThe `[<<\"First bytes\">>,<<\"Second bytes\">>]` could of course have been one\nsingle binary: `<<\"First bytesSecond bytes\">>`.","ref":"new_api.html#example-of-crypto_one_time-5"},{"type":"extras","title":"Example of crypto_one_time_aead/6 - New and Old API","doc":"The same example as in the\n[previous section](new_api.md#example-of-crypto_one_time-5), but now with one\ncall to [crypto_one_time_aead/6](`crypto:crypto_one_time_aead/6`):\n\n```erlang\n\t1> Key = <<1:128>>.\n\t<<0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1>>\n\t2> IV = <<0:128>>.\n\t<<0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0>>\n\t3> Txt = [<<\"First bytes\">>,<<\"Second bytes\">>].\n\t[<<\"First bytes\">>,<<\"Second bytes\">>]\n\t4> AAD = <<\"Some additional auth data\">>.\n\t<<\"Some additional auth data\">>\n\t5> crypto:crypto_one_time_aead(aes_128_gcm, Key, IV, Txt, AAD, true).\n\t{<<240,130,38,96,130,241,189,52,3,190,179,213,132,1,72,\n\t192,103,176,90,104,15,71,158>>,\n\t<<131,47,45,91,142,85,9,244,21,141,214,71,31,135,2,155>>}\n\t6>\n```\n\nThe `[<<\"First bytes\">>,<<\"Second bytes\">>]` could of course have been one\nsingle binary: `<<\"First bytesSecond bytes\">>`.","ref":"new_api.html#example-of-crypto_one_time_aead-6"},{"type":"extras","title":"Example of mac_init mac_update and mac_final - New and Old API","doc":"```erlang\n\t1> Key = <<1:128>>.\n\t<<0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1>>\n\t2> StateMac = crypto:mac_init(cmac, aes_128_cbc, Key).\n\t#Ref<0.2424664121.2781478916.232610>\n\t3> crypto:mac_update(StateMac, <<\"First bytes\">>).\n\t#Ref<0.2424664121.2781478916.232610>\n\t4> crypto:mac_update(StateMac, \" \").\n\t#Ref<0.2424664121.2781478916.232610>\n\t5> crypto:mac_update(StateMac, <<\"last bytes\">>).\n\t#Ref<0.2424664121.2781478916.232610>\n\t6> crypto:mac_final(StateMac).\n\t<<68,191,219,128,84,77,11,193,197,238,107,6,214,141,160,\n\t249>>\n\t7>\n```\n\nand compare the result with a single calculation just for this example:\n\n```erlang\n\t7> crypto:mac(cmac, aes_128_cbc, Key, \"First bytes last bytes\").\n\t<<68,191,219,128,84,77,11,193,197,238,107,6,214,141,160,\n\t249>>\n\t8> v(7) == v(6).\n\ttrue\n\t9>\n```","ref":"new_api.html#example-of-mac_init-mac_update-and-mac_final"},{"type":"extras","title":"Retired cipher names - New and Old API","doc":"This table lists the retired cipher names in the first column and suggests names\nto replace them with in the second column.\n\nThe new names follows the OpenSSL libcrypto names. The format is\nALGORITM_KEYSIZE_MODE.\n\nExamples of algorithms are aes, chacha20 and des. The keysize is the number of\nbits and examples of the mode are cbc, ctr and gcm. The mode may be followed by\na number depending on the mode. An example is the ccm mode which has a variant\ncalled ccm8 where the so called tag has a length of eight bits.\n\nThe old names had by time lost any common naming convention which the new names\nnow introduces. The new names include the key length which improves the error\nchecking in the lower levels of the crypto application.\n\n| **Instead of:** | **Use:**                                         |\n| --------------- | ------------------------------------------------ |\n| `aes_cbc128`    | `aes_128_cbc`                                    |\n| `aes_cbc256`    | `aes_256_cbc`                                    |\n| `aes_cbc`       | `aes_128_cbc, aes_192_cbc, aes_256_cbc`          |\n| `aes_ccm`       | `aes_128_ccm, aes_192_ccm, aes_256_ccm`          |\n| `aes_cfb128`    | `aes_128_cfb128, aes_192_cfb128, aes_256_cfb128` |\n| `aes_cfb8`      | `aes_128_cfb8, aes_192_cfb8, aes_256_cfb8`       |\n| `aes_ctr`       | `aes_128_ctr, aes_192_ctr, aes_256_ctr`          |\n| `aes_gcm`       | `aes_128_gcm, aes_192_gcm, aes_256_gcm`          |\n| `des3_cbc`      | `des_ede3_cbc`                                   |\n| `des3_cbf`      | `des_ede3_cfb`                                   |\n| `des3_cfb`      | `des_ede3_cfb`                                   |\n| `des_ede3`      | `des_ede3_cbc`                                   |\n| `des_ede3_cbf`  | `des_ede3_cfb`                                   |","ref":"new_api.html#retired-cipher-names"}],"proglang":"erlang","content_type":"text/markdown","producer":{"name":"ex_doc","version":"0.37.0-rc.2"}}