//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
// G4ParticleChange
//
// Class description:
//
// Concrete class for ParticleChange containing the results after
// invocation of a physics process.
// This includes final states of parent particle (momentum, energy,
// etc) and secondary particles generated by the interaction.
// The tracking assumes that all the values of energy and momentum
// are in global reference system, therefore all the needed Lorentz
// transformations must have been already computed when filling the
// data-members of this class.
//
// IMPORTANT NOTE: Despite the name, what this class stores/returns
// through its methods, are the "FINAL" values of the Position,
// Momentum, etc.

// Author: Hisaya Kurashige, 23 March 1998  
// --------------------------------------------------------------------
#ifndef G4ParticleChange_hh
#define G4ParticleChange_hh 1

#include "globals.hh"
#include "G4ios.hh"
#include "G4ThreeVector.hh"
#include "G4VParticleChange.hh"

class G4DynamicParticle;

class G4ParticleChange : public G4VParticleChange
{
  public:

    G4ParticleChange();
      // Default constructor

    ~G4ParticleChange() override = default;
      // Destructor

    G4ParticleChange(const G4ParticleChange& right) = delete;
    G4ParticleChange& operator=(const G4ParticleChange& right) = delete;

  // --- the following methods are for updating G4Step -----
  // Return the pointer to G4Step after updating the Step information
  // by using final state of the track given by a physics process

    G4Step* UpdateStepForAlongStep(G4Step* Step) override;
      // A physics process gives the final state of the particle
      // relative to the initial state at the beginning of the Step,
      // i.e., based on information of G4Track (or equivalently
      // the PreStepPoint).
      // In this method, the differences (delta) between these two states
      // are calculated, and are accumulated in PostStepPoint.
      // Take note that the return type of GetMomentumChange() is a
      // pointer to G4ParticleMomentum. Also it is a normalized
      // momentum vector

    G4Step* UpdateStepForAtRest(G4Step* Step) override;
    G4Step* UpdateStepForPostStep(G4Step* Step) override;
      // A physics process gives the final state of the particle
      // based on information of G4Track (or equivalently the PreStepPoint)

    void Initialize(const G4Track&) override;
      // Initialize all propoerties by using G4Track information

  // --- methods to keep information of the final state ---
  //
  //  The ProposeXXX methods store (and return in GetXXX methods)
  //  the "FINAL" values of the Position, Momentum, etc.

    inline const G4ThreeVector* GetMomentumDirection() const;
    inline void ProposeMomentumDirection(G4double Px, G4double Py, G4double Pz);
    inline void ProposeMomentumDirection(const G4ThreeVector& Pfinal);
      // Get/Propose the MomentumDirection vector: it is the final momentum
      // direction

    inline const G4ThreeVector* GetPolarization() const;
    inline void ProposePolarization(G4double Px, G4double Py, G4double Pz);
    inline void ProposePolarization(const G4ThreeVector& finalPoralization);
      // Get/Propose the final Polarization vector

    inline G4double GetEnergy() const;
    inline void ProposeEnergy(G4double finalEnergy);
      // Get/Propose the final kinetic energy of the current particle

    inline G4double GetVelocity() const;
    inline void ProposeVelocity(G4double finalVelocity);
      // Get/Propose the final velocity of the current particle

    inline G4double GetProperTime() const;
    inline void ProposeProperTime(G4double finalProperTime);
      // Get/Propose the final ProperTime

    inline const G4ThreeVector* GetPosition() const;
    inline void ProposePosition(G4double x, G4double y, G4double z);
    inline void ProposePosition(const G4ThreeVector& finalPosition);
      // Get/Propose the final position of the current particle

    inline void ProposeGlobalTime(G4double t);
    inline void ProposeLocalTime(G4double t);
      // Get/Propose the final global/local Time.
      // NOTE: DO NOT INVOKE both methods in a step
      //       Each method affects both local and global time

    inline G4double GetGlobalTime(G4double timeDelay = 0.0) const;
    inline G4double GetLocalTime(G4double timeDelay = 0.0) const;
      // Convert the time delay to the glocbal/local time.
      // Can get the final global/local time without argument

    inline G4double GetMass() const;
    inline void ProposeMass(G4double finalMass);
      // Get/Propose the final dynamical mass in G4DynamicParticle

    inline G4double GetCharge() const;
    inline void ProposeCharge(G4double finalCharge);
      // Get/Propose the final dynamical charge in G4DynamicParticle

    inline G4double GetMagneticMoment() const;
    inline void ProposeMagneticMoment(G4double finalMagneticMoment);
      // Get/Propose the final MagneticMoment in G4DynamicParticle

    inline G4ThreeVector
    GetGlobalPosition(const G4ThreeVector& displacement) const;
      // Convert the position displacement to the global position

    inline G4ThreeVector CalcMomentum(G4double energy, G4ThreeVector direction,
                                      G4double mass) const;
      // Calculate momentum by using Energy, Momentum Direction, and Mass

  // --- methods for adding secondaries ---

    void AddSecondary(G4Track* aSecondary);
      // Add a secondary particle to theListOfSecondaries

    void AddSecondary(G4DynamicParticle* aSecondary,
                      G4bool IsGoodForTracking = false);
      // Add a secondary particle to theListOfSecondaries.
      // Position and time are same as thePositionChange and theTimeChange

    void AddSecondary(G4DynamicParticle* aSecondary, G4ThreeVector position,
                      G4bool IsGoodForTracking = false);
      // Add a secondary particle to theListOfSecondaries.
      // Global time are same as theTimeChange and theTimeChange

    void AddSecondary(G4DynamicParticle* aSecondary, G4double time,
                      G4bool IsGoodForTracking = false);
      // Add a secondary particle to theListOfSecondaries.
      // Position and are same as thePositionChange

  // --- Dump and debug methods ---

    void DumpInfo() const override;

  protected:

    G4Step* UpdateStepInfo(G4Step* Step);
      // Update the G4Step specific attributes
      // (i.e. SteppingControl, LocalEnergyDeposit, and TrueStepLength)

    G4ThreeVector theMomentumDirectionChange;
      // It is the vector containing the final momentum direction
      // after the invoked process. The application of the change
      // of the momentum direction of the particle is not done here.
      // The responsibility to apply the change is up to the entity
      // which invoked the process

    G4ThreeVector thePolarizationChange;
      // The changed (final) polarization of a given track

    G4double theEnergyChange = 0.0;
      // The final kinetic energy of the current track

    G4double theVelocityChange = 0.0;
    G4bool isVelocityChanged = false;
      // The final velocity of the current track

    G4ThreeVector thePositionChange;
      // The changed (final) position of a given track

    G4double theGlobalTime0 = 0.0;
      // The global time at Initial
    G4double theLocalTime0 = 0.0;
      // The local time at Initial

    G4double theTimeChange = 0.0;
      // The change of local time of a given particle

    G4double theProperTimeChange = 0.0;
      // The changed (final) proper time of a given track

    G4double theMassChange = 0.0;
      // The Changed (final) mass of a given track

    G4double theChargeChange = 0.0;
     // The Changed (final) charge of a given track

    G4double theMagneticMomentChange = 0.0;
     // The Changed (final) MagneticMoment of a given track
};

#include "G4ParticleChange.icc"

#endif
