﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/application-signals/ApplicationSignalsRequest.h>
#include <aws/application-signals/ApplicationSignals_EXPORTS.h>
#include <aws/application-signals/model/AuditTarget.h>
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace ApplicationSignals {
namespace Model {

/**
 */
class ListAuditFindingsRequest : public ApplicationSignalsRequest {
 public:
  AWS_APPLICATIONSIGNALS_API ListAuditFindingsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListAuditFindings"; }

  AWS_APPLICATIONSIGNALS_API Aws::String SerializePayload() const override;

  AWS_APPLICATIONSIGNALS_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The start time for the audit findings query. Only findings created on or
   * after this time will be included in the results. Specify the time as the number
   * of milliseconds since January 1, 1970, 00:00:00 UTC.</p>
   */
  inline const Aws::Utils::DateTime& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = Aws::Utils::DateTime>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = Aws::Utils::DateTime>
  ListAuditFindingsRequest& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The end time for the audit findings query. Only findings created before this
   * time will be included in the results. Specify the time as the number of
   * milliseconds since January 1, 1970, 00:00:00 UTC.</p>
   */
  inline const Aws::Utils::DateTime& GetEndTime() const { return m_endTime; }
  inline bool EndTimeHasBeenSet() const { return m_endTimeHasBeenSet; }
  template <typename EndTimeT = Aws::Utils::DateTime>
  void SetEndTime(EndTimeT&& value) {
    m_endTimeHasBeenSet = true;
    m_endTime = std::forward<EndTimeT>(value);
  }
  template <typename EndTimeT = Aws::Utils::DateTime>
  ListAuditFindingsRequest& WithEndTime(EndTimeT&& value) {
    SetEndTime(std::forward<EndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An array of auditor names to filter the findings. Only findings generated by
   * the specified auditors will be returned. When not specified, findings from all
   * auditors are included except canary.</p>
   */
  inline const Aws::Vector<Aws::String>& GetAuditors() const { return m_auditors; }
  inline bool AuditorsHasBeenSet() const { return m_auditorsHasBeenSet; }
  template <typename AuditorsT = Aws::Vector<Aws::String>>
  void SetAuditors(AuditorsT&& value) {
    m_auditorsHasBeenSet = true;
    m_auditors = std::forward<AuditorsT>(value);
  }
  template <typename AuditorsT = Aws::Vector<Aws::String>>
  ListAuditFindingsRequest& WithAuditors(AuditorsT&& value) {
    SetAuditors(std::forward<AuditorsT>(value));
    return *this;
  }
  template <typename AuditorsT = Aws::String>
  ListAuditFindingsRequest& AddAuditors(AuditorsT&& value) {
    m_auditorsHasBeenSet = true;
    m_auditors.emplace_back(std::forward<AuditorsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An array of audit target specifications to filter the findings. Only findings
   * related to the specified targets (such as specific services, SLOs, operations or
   * canary) will be returned.</p>
   */
  inline const Aws::Vector<AuditTarget>& GetAuditTargets() const { return m_auditTargets; }
  inline bool AuditTargetsHasBeenSet() const { return m_auditTargetsHasBeenSet; }
  template <typename AuditTargetsT = Aws::Vector<AuditTarget>>
  void SetAuditTargets(AuditTargetsT&& value) {
    m_auditTargetsHasBeenSet = true;
    m_auditTargets = std::forward<AuditTargetsT>(value);
  }
  template <typename AuditTargetsT = Aws::Vector<AuditTarget>>
  ListAuditFindingsRequest& WithAuditTargets(AuditTargetsT&& value) {
    SetAuditTargets(std::forward<AuditTargetsT>(value));
    return *this;
  }
  template <typename AuditTargetsT = AuditTarget>
  ListAuditFindingsRequest& AddAuditTargets(AuditTargetsT&& value) {
    m_auditTargetsHasBeenSet = true;
    m_auditTargets.emplace_back(std::forward<AuditTargetsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token for the next set of results. Use this token to retrieve additional
   * pages of audit findings when the result set is large.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListAuditFindingsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of audit findings to return in a single request. Valid
   * range is 1 to 100. If not specified, defaults to 50.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListAuditFindingsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::Utils::DateTime m_startTime{};
  bool m_startTimeHasBeenSet = false;

  Aws::Utils::DateTime m_endTime{};
  bool m_endTimeHasBeenSet = false;

  Aws::Vector<Aws::String> m_auditors;
  bool m_auditorsHasBeenSet = false;

  Aws::Vector<AuditTarget> m_auditTargets;
  bool m_auditTargetsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace ApplicationSignals
}  // namespace Aws
