;;;-*- Mode: Lisp; Package: CCL -*-
;;;
;;; Copyright 2007-2009 Clozure Associates
;;;
;;; Licensed under the Apache License, Version 2.0 (the "License");
;;; you may not use this file except in compliance with the License.
;;; You may obtain a copy of the License at
;;;
;;;     http://www.apache.org/licenses/LICENSE-2.0
;;;
;;; Unless required by applicable law or agreed to in writing, software
;;; distributed under the License is distributed on an "AS IS" BASIS,
;;; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;;; See the License for the specific language governing permissions and
;;; limitations under the License.

(in-package "CCL")

;;; converting tables
(defstatic *gb2312-to-unicode-map* nil)
(defstatic *unicode-to-gb2312-map* nil)
(defstatic *cp936-to-unicode-map* nil)
(defstatic *unicode-to-cp936-map* nil)   

;;; setup the hash tables
(let ((gb2312-to-unicode-table
       (mapcar (lambda (pair)
                 (cons (+ (car pair) #x8080) (cdr pair)))
               '(
                 ;; gb2312-to-unicode-table
                 (#x80   . #x20AC)      ; EURO SIGN

                 (#x2121 . #x3000)      ; IDEOGRAPHIC SPACE
                 (#x2122 . #x3001)      ; IDEOGRAPHIC COMMA
                 (#x2123 . #x3002)      ; IDEOGRAPHIC FULL STOP
                 (#x2124 . #x30FB)      ; KATAKANA MIDDLE DOT
                 (#x2125 . #x02C9)      ; MODIFIER LETTER MACRON (Mandarin Chinese first tone)
                 (#x2126 . #x02C7)      ; CARON (Mandarin Chinese third tone)
                 (#x2127 . #x00A8)      ; DIAERESIS
                 (#x2128 . #x3003)      ; DITTO MARK
                 (#x2129 . #x3005)      ; IDEOGRAPHIC ITERATION MARK
                 (#x212A . #x2015)      ; HORIZONTAL BAR
                 (#x212B . #xFF5E)      ; FULLWIDTH TILDE
                 (#x212C . #x2016)      ; DOUBLE VERTICAL LINE
                 (#x212D . #x2026)      ; HORIZONTAL ELLIPSIS
                 (#x212E . #x2018)      ; LEFT SINGLE QUOTATION MARK
                 (#x212F . #x2019)      ; RIGHT SINGLE QUOTATION MARK
                 (#x2130 . #x201C)      ; LEFT DOUBLE QUOTATION MARK
                 (#x2131 . #x201D)      ; RIGHT DOUBLE QUOTATION MARK
                 (#x2132 . #x3014)      ; LEFT TORTOISE SHELL BRACKET
                 (#x2133 . #x3015)      ; RIGHT TORTOISE SHELL BRACKET
                 (#x2134 . #x3008)      ; LEFT ANGLE BRACKET
                 (#x2135 . #x3009)      ; RIGHT ANGLE BRACKET
                 (#x2136 . #x300A)      ; LEFT DOUBLE ANGLE BRACKET
                 (#x2137 . #x300B)      ; RIGHT DOUBLE ANGLE BRACKET
                 (#x2138 . #x300C)      ; LEFT CORNER BRACKET
                 (#x2139 . #x300D)      ; RIGHT CORNER BRACKET
                 (#x213A . #x300E)      ; LEFT WHITE CORNER BRACKET
                 (#x213B . #x300F)      ; RIGHT WHITE CORNER BRACKET
                 (#x213C . #x3016)      ; LEFT WHITE LENTICULAR BRACKET
                 (#x213D . #x3017)      ; RIGHT WHITE LENTICULAR BRACKET
                 (#x213E . #x3010)      ; LEFT BLACK LENTICULAR BRACKET
                 (#x213F . #x3011)      ; RIGHT BLACK LENTICULAR BRACKET
                 (#x2140 . #x00B1)      ; PLUS-MINUS SIGN
                 (#x2141 . #x00D7)      ; MULTIPLICATION SIGN
                 (#x2142 . #x00F7)      ; DIVISION SIGN
                 (#x2143 . #x2236)      ; RATIO
                 (#x2144 . #x2227)      ; LOGICAL AND
                 (#x2145 . #x2228)      ; LOGICAL OR
                 (#x2146 . #x2211)      ; N-ARY SUMMATION
                 (#x2147 . #x220F)      ; N-ARY PRODUCT
                 (#x2148 . #x222A)      ; UNION
                 (#x2149 . #x2229)      ; INTERSECTION
                 (#x214A . #x2208)      ; ELEMENT OF
                 (#x214B . #x2237)      ; PROPORTION
                 (#x214C . #x221A)      ; SQUARE ROOT
                 (#x214D . #x22A5)      ; UP TACK
                 (#x214E . #x2225)      ; PARALLEL TO
                 (#x214F . #x2220)      ; ANGLE
                 (#x2150 . #x2312)      ; ARC
                 (#x2151 . #x2299)      ; CIRCLED DOT OPERATOR
                 (#x2152 . #x222B)      ; INTEGRAL
                 (#x2153 . #x222E)      ; CONTOUR INTEGRAL
                 (#x2154 . #x2261)      ; IDENTICAL TO
                 (#x2155 . #x224C)      ; ALL EQUAL TO
                 (#x2156 . #x2248)      ; ALMOST EQUAL TO
                 (#x2157 . #x223D)      ; REVERSED TILDE
                 (#x2158 . #x221D)      ; PROPORTIONAL TO
                 (#x2159 . #x2260)      ; NOT EQUAL TO
                 (#x215A . #x226E)      ; NOT LESS-THAN
                 (#x215B . #x226F)      ; NOT GREATER-THAN
                 (#x215C . #x2264)      ; LESS-THAN OR EQUAL TO
                 (#x215D . #x2265)      ; GREATER-THAN OR EQUAL TO
                 (#x215E . #x221E)      ; INFINITY
                 (#x215F . #x2235)      ; BECAUSE
                 (#x2160 . #x2234)      ; THEREFORE
                 (#x2161 . #x2642)      ; MALE SIGN
                 (#x2162 . #x2640)      ; FEMALE SIGN
                 (#x2163 . #x00B0)      ; DEGREE SIGN
                 (#x2164 . #x2032)      ; PRIME
                 (#x2165 . #x2033)      ; DOUBLE PRIME
                 (#x2166 . #x2103)      ; DEGREE CELSIUS
                 (#x2167 . #xFF04)      ; FULLWIDTH DOLLAR SIGN
                 (#x2168 . #x00A4)      ; CURRENCY SIGN
                 (#x2169 . #xFFE0)      ; FULLWIDTH CENT SIGN
                 (#x216A . #xFFE1)      ; FULLWIDTH POUND SIGN
                 (#x216B . #x2030)      ; PER MILLE SIGN
                 (#x216C . #x00A7)      ; SECTION SIGN
                 (#x216D . #x2116)      ; NUMERO SIGN
                 (#x216E . #x2606)      ; WHITE STAR
                 (#x216F . #x2605)      ; BLACK STAR
                 (#x2170 . #x25CB)      ; WHITE CIRCLE
                 (#x2171 . #x25CF)      ; BLACK CIRCLE
                 (#x2172 . #x25CE)      ; BULLSEYE
                 (#x2173 . #x25C7)      ; WHITE DIAMOND
                 (#x2174 . #x25C6)      ; BLACK DIAMOND
                 (#x2175 . #x25A1)      ; WHITE SQUARE
                 (#x2176 . #x25A0)      ; BLACK SQUARE
                 (#x2177 . #x25B3)      ; WHITE UP-POINTING TRIANGLE
                 (#x2178 . #x25B2)      ; BLACK UP-POINTING TRIANGLE
                 (#x2179 . #x203B)      ; REFERENCE MARK
                 (#x217A . #x2192)      ; RIGHTWARDS ARROW
                 (#x217B . #x2190)      ; LEFTWARDS ARROW
                 (#x217C . #x2191)      ; UPWARDS ARROW
                 (#x217D . #x2193)      ; DOWNWARDS ARROW
                 (#x217E . #x3013)      ; GETA MARK
                 (#x2231 . #x2488)      ; DIGIT ONE FULL STOP
                 (#x2232 . #x2489)      ; DIGIT TWO FULL STOP
                 (#x2233 . #x248A)      ; DIGIT THREE FULL STOP
                 (#x2234 . #x248B)      ; DIGIT FOUR FULL STOP
                 (#x2235 . #x248C)      ; DIGIT FIVE FULL STOP
                 (#x2236 . #x248D)      ; DIGIT SIX FULL STOP
                 (#x2237 . #x248E)      ; DIGIT SEVEN FULL STOP
                 (#x2238 . #x248F)      ; DIGIT EIGHT FULL STOP
                 (#x2239 . #x2490)      ; DIGIT NINE FULL STOP
                 (#x223A . #x2491)      ; NUMBER TEN FULL STOP
                 (#x223B . #x2492)      ; NUMBER ELEVEN FULL STOP
                 (#x223C . #x2493)      ; NUMBER TWELVE FULL STOP
                 (#x223D . #x2494)      ; NUMBER THIRTEEN FULL STOP
                 (#x223E . #x2495)      ; NUMBER FOURTEEN FULL STOP
                 (#x223F . #x2496)      ; NUMBER FIFTEEN FULL STOP
                 (#x2240 . #x2497)      ; NUMBER SIXTEEN FULL STOP
                 (#x2241 . #x2498)      ; NUMBER SEVENTEEN FULL STOP
                 (#x2242 . #x2499)      ; NUMBER EIGHTEEN FULL STOP
                 (#x2243 . #x249A)      ; NUMBER NINETEEN FULL STOP
                 (#x2244 . #x249B)      ; NUMBER TWENTY FULL STOP
                 (#x2245 . #x2474)      ; PARENTHESIZED DIGIT ONE
                 (#x2246 . #x2475)      ; PARENTHESIZED DIGIT TWO
                 (#x2247 . #x2476)      ; PARENTHESIZED DIGIT THREE
                 (#x2248 . #x2477)      ; PARENTHESIZED DIGIT FOUR
                 (#x2249 . #x2478)      ; PARENTHESIZED DIGIT FIVE
                 (#x224A . #x2479)      ; PARENTHESIZED DIGIT SIX
                 (#x224B . #x247A)      ; PARENTHESIZED DIGIT SEVEN
                 (#x224C . #x247B)      ; PARENTHESIZED DIGIT EIGHT
                 (#x224D . #x247C)      ; PARENTHESIZED DIGIT NINE
                 (#x224E . #x247D)      ; PARENTHESIZED NUMBER TEN
                 (#x224F . #x247E)      ; PARENTHESIZED NUMBER ELEVEN
                 (#x2250 . #x247F)      ; PARENTHESIZED NUMBER TWELVE
                 (#x2251 . #x2480)      ; PARENTHESIZED NUMBER THIRTEEN
                 (#x2252 . #x2481)      ; PARENTHESIZED NUMBER FOURTEEN
                 (#x2253 . #x2482)      ; PARENTHESIZED NUMBER FIFTEEN
                 (#x2254 . #x2483)      ; PARENTHESIZED NUMBER SIXTEEN
                 (#x2255 . #x2484)      ; PARENTHESIZED NUMBER SEVENTEEN
                 (#x2256 . #x2485)      ; PARENTHESIZED NUMBER EIGHTEEN
                 (#x2257 . #x2486)      ; PARENTHESIZED NUMBER NINETEEN
                 (#x2258 . #x2487)      ; PARENTHESIZED NUMBER TWENTY
                 (#x2259 . #x2460)      ; CIRCLED DIGIT ONE
                 (#x225A . #x2461)      ; CIRCLED DIGIT TWO
                 (#x225B . #x2462)      ; CIRCLED DIGIT THREE
                 (#x225C . #x2463)      ; CIRCLED DIGIT FOUR
                 (#x225D . #x2464)      ; CIRCLED DIGIT FIVE
                 (#x225E . #x2465)      ; CIRCLED DIGIT SIX
                 (#x225F . #x2466)      ; CIRCLED DIGIT SEVEN
                 (#x2260 . #x2467)      ; CIRCLED DIGIT EIGHT
                 (#x2261 . #x2468)      ; CIRCLED DIGIT NINE
                 (#x2262 . #x2469)      ; CIRCLED NUMBER TEN
                 (#x2265 . #x3220)      ; PARENTHESIZED IDEOGRAPH ONE
                 (#x2266 . #x3221)      ; PARENTHESIZED IDEOGRAPH TWO
                 (#x2267 . #x3222)      ; PARENTHESIZED IDEOGRAPH THREE
                 (#x2268 . #x3223)      ; PARENTHESIZED IDEOGRAPH FOUR
                 (#x2269 . #x3224)      ; PARENTHESIZED IDEOGRAPH FIVE
                 (#x226A . #x3225)      ; PARENTHESIZED IDEOGRAPH SIX
                 (#x226B . #x3226)      ; PARENTHESIZED IDEOGRAPH SEVEN
                 (#x226C . #x3227)      ; PARENTHESIZED IDEOGRAPH EIGHT
                 (#x226D . #x3228)      ; PARENTHESIZED IDEOGRAPH NINE
                 (#x226E . #x3229)      ; PARENTHESIZED IDEOGRAPH TEN
                 (#x2271 . #x2160)      ; ROMAN NUMERAL ONE
                 (#x2272 . #x2161)      ; ROMAN NUMERAL TWO
                 (#x2273 . #x2162)      ; ROMAN NUMERAL THREE
                 (#x2274 . #x2163)      ; ROMAN NUMERAL FOUR
                 (#x2275 . #x2164)      ; ROMAN NUMERAL FIVE
                 (#x2276 . #x2165)      ; ROMAN NUMERAL SIX
                 (#x2277 . #x2166)      ; ROMAN NUMERAL SEVEN
                 (#x2278 . #x2167)      ; ROMAN NUMERAL EIGHT
                 (#x2279 . #x2168)      ; ROMAN NUMERAL NINE
                 (#x227A . #x2169)      ; ROMAN NUMERAL TEN
                 (#x227B . #x216A)      ; ROMAN NUMERAL ELEVEN
                 (#x227C . #x216B)      ; ROMAN NUMERAL TWELVE
                 (#x2321 . #xFF01)      ; FULLWIDTH EXCLAMATION MARK
                 (#x2322 . #xFF02)      ; FULLWIDTH QUOTATION MARK
                 (#x2323 . #xFF03)      ; FULLWIDTH NUMBER SIGN
                 (#x2324 . #xFFE5)      ; FULLWIDTH YEN SIGN
                 (#x2325 . #xFF05)      ; FULLWIDTH PERCENT SIGN
                 (#x2326 . #xFF06)      ; FULLWIDTH AMPERSAND
                 (#x2327 . #xFF07)      ; FULLWIDTH APOSTROPHE
                 (#x2328 . #xFF08)      ; FULLWIDTH LEFT PARENTHESIS
                 (#x2329 . #xFF09)      ; FULLWIDTH RIGHT PARENTHESIS
                 (#x232A . #xFF0A)      ; FULLWIDTH ASTERISK
                 (#x232B . #xFF0B)      ; FULLWIDTH PLUS SIGN
                 (#x232C . #xFF0C)      ; FULLWIDTH COMMA
                 (#x232D . #xFF0D)      ; FULLWIDTH HYPHEN-MINUS
                 (#x232E . #xFF0E)      ; FULLWIDTH FULL STOP
                 (#x232F . #xFF0F)      ; FULLWIDTH SOLIDUS
                 (#x2330 . #xFF10)      ; FULLWIDTH DIGIT ZERO
                 (#x2331 . #xFF11)      ; FULLWIDTH DIGIT ONE
                 (#x2332 . #xFF12)      ; FULLWIDTH DIGIT TWO
                 (#x2333 . #xFF13)      ; FULLWIDTH DIGIT THREE
                 (#x2334 . #xFF14)      ; FULLWIDTH DIGIT FOUR
                 (#x2335 . #xFF15)      ; FULLWIDTH DIGIT FIVE
                 (#x2336 . #xFF16)      ; FULLWIDTH DIGIT SIX
                 (#x2337 . #xFF17)      ; FULLWIDTH DIGIT SEVEN
                 (#x2338 . #xFF18)      ; FULLWIDTH DIGIT EIGHT
                 (#x2339 . #xFF19)      ; FULLWIDTH DIGIT NINE
                 (#x233A . #xFF1A)      ; FULLWIDTH COLON
                 (#x233B . #xFF1B)      ; FULLWIDTH SEMICOLON
                 (#x233C . #xFF1C)      ; FULLWIDTH LESS-THAN SIGN
                 (#x233D . #xFF1D)      ; FULLWIDTH EQUALS SIGN
                 (#x233E . #xFF1E)      ; FULLWIDTH GREATER-THAN SIGN
                 (#x233F . #xFF1F)      ; FULLWIDTH QUESTION MARK
                 (#x2340 . #xFF20)      ; FULLWIDTH COMMERCIAL AT
                 (#x2341 . #xFF21)      ; FULLWIDTH LATIN CAPITAL LETTER A
                 (#x2342 . #xFF22)      ; FULLWIDTH LATIN CAPITAL LETTER B
                 (#x2343 . #xFF23)      ; FULLWIDTH LATIN CAPITAL LETTER C
                 (#x2344 . #xFF24)      ; FULLWIDTH LATIN CAPITAL LETTER D
                 (#x2345 . #xFF25)      ; FULLWIDTH LATIN CAPITAL LETTER E
                 (#x2346 . #xFF26)      ; FULLWIDTH LATIN CAPITAL LETTER F
                 (#x2347 . #xFF27)      ; FULLWIDTH LATIN CAPITAL LETTER G
                 (#x2348 . #xFF28)      ; FULLWIDTH LATIN CAPITAL LETTER H
                 (#x2349 . #xFF29)      ; FULLWIDTH LATIN CAPITAL LETTER I
                 (#x234A . #xFF2A)      ; FULLWIDTH LATIN CAPITAL LETTER J
                 (#x234B . #xFF2B)      ; FULLWIDTH LATIN CAPITAL LETTER K
                 (#x234C . #xFF2C)      ; FULLWIDTH LATIN CAPITAL LETTER L
                 (#x234D . #xFF2D)      ; FULLWIDTH LATIN CAPITAL LETTER M
                 (#x234E . #xFF2E)      ; FULLWIDTH LATIN CAPITAL LETTER N
                 (#x234F . #xFF2F)      ; FULLWIDTH LATIN CAPITAL LETTER O
                 (#x2350 . #xFF30)      ; FULLWIDTH LATIN CAPITAL LETTER P
                 (#x2351 . #xFF31)      ; FULLWIDTH LATIN CAPITAL LETTER Q
                 (#x2352 . #xFF32)      ; FULLWIDTH LATIN CAPITAL LETTER R
                 (#x2353 . #xFF33)      ; FULLWIDTH LATIN CAPITAL LETTER S
                 (#x2354 . #xFF34)      ; FULLWIDTH LATIN CAPITAL LETTER T
                 (#x2355 . #xFF35)      ; FULLWIDTH LATIN CAPITAL LETTER U
                 (#x2356 . #xFF36)      ; FULLWIDTH LATIN CAPITAL LETTER V
                 (#x2357 . #xFF37)      ; FULLWIDTH LATIN CAPITAL LETTER W
                 (#x2358 . #xFF38)      ; FULLWIDTH LATIN CAPITAL LETTER X
                 (#x2359 . #xFF39)      ; FULLWIDTH LATIN CAPITAL LETTER Y
                 (#x235A . #xFF3A)      ; FULLWIDTH LATIN CAPITAL LETTER Z
                 (#x235B . #xFF3B)      ; FULLWIDTH LEFT SQUARE BRACKET
                 (#x235C . #xFF3C)      ; FULLWIDTH REVERSE SOLIDUS
                 (#x235D . #xFF3D)      ; FULLWIDTH RIGHT SQUARE BRACKET
                 (#x235E . #xFF3E)      ; FULLWIDTH CIRCUMFLEX ACCENT
                 (#x235F . #xFF3F)      ; FULLWIDTH LOW LINE
                 (#x2360 . #xFF40)      ; FULLWIDTH GRAVE ACCENT
                 (#x2361 . #xFF41)      ; FULLWIDTH LATIN SMALL LETTER A
                 (#x2362 . #xFF42)      ; FULLWIDTH LATIN SMALL LETTER B
                 (#x2363 . #xFF43)      ; FULLWIDTH LATIN SMALL LETTER C
                 (#x2364 . #xFF44)      ; FULLWIDTH LATIN SMALL LETTER D
                 (#x2365 . #xFF45)      ; FULLWIDTH LATIN SMALL LETTER E
                 (#x2366 . #xFF46)      ; FULLWIDTH LATIN SMALL LETTER F
                 (#x2367 . #xFF47)      ; FULLWIDTH LATIN SMALL LETTER G
                 (#x2368 . #xFF48)      ; FULLWIDTH LATIN SMALL LETTER H
                 (#x2369 . #xFF49)      ; FULLWIDTH LATIN SMALL LETTER I
                 (#x236A . #xFF4A)      ; FULLWIDTH LATIN SMALL LETTER J
                 (#x236B . #xFF4B)      ; FULLWIDTH LATIN SMALL LETTER K
                 (#x236C . #xFF4C)      ; FULLWIDTH LATIN SMALL LETTER L
                 (#x236D . #xFF4D)      ; FULLWIDTH LATIN SMALL LETTER M
                 (#x236E . #xFF4E)      ; FULLWIDTH LATIN SMALL LETTER N
                 (#x236F . #xFF4F)      ; FULLWIDTH LATIN SMALL LETTER O
                 (#x2370 . #xFF50)      ; FULLWIDTH LATIN SMALL LETTER P
                 (#x2371 . #xFF51)      ; FULLWIDTH LATIN SMALL LETTER Q
                 (#x2372 . #xFF52)      ; FULLWIDTH LATIN SMALL LETTER R
                 (#x2373 . #xFF53)      ; FULLWIDTH LATIN SMALL LETTER S
                 (#x2374 . #xFF54)      ; FULLWIDTH LATIN SMALL LETTER T
                 (#x2375 . #xFF55)      ; FULLWIDTH LATIN SMALL LETTER U
                 (#x2376 . #xFF56)      ; FULLWIDTH LATIN SMALL LETTER V
                 (#x2377 . #xFF57)      ; FULLWIDTH LATIN SMALL LETTER W
                 (#x2378 . #xFF58)      ; FULLWIDTH LATIN SMALL LETTER X
                 (#x2379 . #xFF59)      ; FULLWIDTH LATIN SMALL LETTER Y
                 (#x237A . #xFF5A)      ; FULLWIDTH LATIN SMALL LETTER Z
                 (#x237B . #xFF5B)      ; FULLWIDTH LEFT CURLY BRACKET
                 (#x237C . #xFF5C)      ; FULLWIDTH VERTICAL LINE
                 (#x237D . #xFF5D)      ; FULLWIDTH RIGHT CURLY BRACKET
                 (#x237E . #xFFE3)      ; FULLWIDTH MACRON
                 (#x2421 . #x3041)      ; HIRAGANA LETTER SMALL A
                 (#x2422 . #x3042)      ; HIRAGANA LETTER A
                 (#x2423 . #x3043)      ; HIRAGANA LETTER SMALL I
                 (#x2424 . #x3044)      ; HIRAGANA LETTER I
                 (#x2425 . #x3045)      ; HIRAGANA LETTER SMALL U
                 (#x2426 . #x3046)      ; HIRAGANA LETTER U
                 (#x2427 . #x3047)      ; HIRAGANA LETTER SMALL E
                 (#x2428 . #x3048)      ; HIRAGANA LETTER E
                 (#x2429 . #x3049)      ; HIRAGANA LETTER SMALL O
                 (#x242A . #x304A)      ; HIRAGANA LETTER O
                 (#x242B . #x304B)      ; HIRAGANA LETTER KA
                 (#x242C . #x304C)      ; HIRAGANA LETTER GA
                 (#x242D . #x304D)      ; HIRAGANA LETTER KI
                 (#x242E . #x304E)      ; HIRAGANA LETTER GI
                 (#x242F . #x304F)      ; HIRAGANA LETTER KU
                 (#x2430 . #x3050)      ; HIRAGANA LETTER GU
                 (#x2431 . #x3051)      ; HIRAGANA LETTER KE
                 (#x2432 . #x3052)      ; HIRAGANA LETTER GE
                 (#x2433 . #x3053)      ; HIRAGANA LETTER KO
                 (#x2434 . #x3054)      ; HIRAGANA LETTER GO
                 (#x2435 . #x3055)      ; HIRAGANA LETTER SA
                 (#x2436 . #x3056)      ; HIRAGANA LETTER ZA
                 (#x2437 . #x3057)      ; HIRAGANA LETTER SI
                 (#x2438 . #x3058)      ; HIRAGANA LETTER ZI
                 (#x2439 . #x3059)      ; HIRAGANA LETTER SU
                 (#x243A . #x305A)      ; HIRAGANA LETTER ZU
                 (#x243B . #x305B)      ; HIRAGANA LETTER SE
                 (#x243C . #x305C)      ; HIRAGANA LETTER ZE
                 (#x243D . #x305D)      ; HIRAGANA LETTER SO
                 (#x243E . #x305E)      ; HIRAGANA LETTER ZO
                 (#x243F . #x305F)      ; HIRAGANA LETTER TA
                 (#x2440 . #x3060)      ; HIRAGANA LETTER DA
                 (#x2441 . #x3061)      ; HIRAGANA LETTER TI
                 (#x2442 . #x3062)      ; HIRAGANA LETTER DI
                 (#x2443 . #x3063)      ; HIRAGANA LETTER SMALL TU
                 (#x2444 . #x3064)      ; HIRAGANA LETTER TU
                 (#x2445 . #x3065)      ; HIRAGANA LETTER DU
                 (#x2446 . #x3066)      ; HIRAGANA LETTER TE
                 (#x2447 . #x3067)      ; HIRAGANA LETTER DE
                 (#x2448 . #x3068)      ; HIRAGANA LETTER TO
                 (#x2449 . #x3069)      ; HIRAGANA LETTER DO
                 (#x244A . #x306A)      ; HIRAGANA LETTER NA
                 (#x244B . #x306B)      ; HIRAGANA LETTER NI
                 (#x244C . #x306C)      ; HIRAGANA LETTER NU
                 (#x244D . #x306D)      ; HIRAGANA LETTER NE
                 (#x244E . #x306E)      ; HIRAGANA LETTER NO
                 (#x244F . #x306F)      ; HIRAGANA LETTER HA
                 (#x2450 . #x3070)      ; HIRAGANA LETTER BA
                 (#x2451 . #x3071)      ; HIRAGANA LETTER PA
                 (#x2452 . #x3072)      ; HIRAGANA LETTER HI
                 (#x2453 . #x3073)      ; HIRAGANA LETTER BI
                 (#x2454 . #x3074)      ; HIRAGANA LETTER PI
                 (#x2455 . #x3075)      ; HIRAGANA LETTER HU
                 (#x2456 . #x3076)      ; HIRAGANA LETTER BU
                 (#x2457 . #x3077)      ; HIRAGANA LETTER PU
                 (#x2458 . #x3078)      ; HIRAGANA LETTER HE
                 (#x2459 . #x3079)      ; HIRAGANA LETTER BE
                 (#x245A . #x307A)      ; HIRAGANA LETTER PE
                 (#x245B . #x307B)      ; HIRAGANA LETTER HO
                 (#x245C . #x307C)      ; HIRAGANA LETTER BO
                 (#x245D . #x307D)      ; HIRAGANA LETTER PO
                 (#x245E . #x307E)      ; HIRAGANA LETTER MA
                 (#x245F . #x307F)      ; HIRAGANA LETTER MI
                 (#x2460 . #x3080)      ; HIRAGANA LETTER MU
                 (#x2461 . #x3081)      ; HIRAGANA LETTER ME
                 (#x2462 . #x3082)      ; HIRAGANA LETTER MO
                 (#x2463 . #x3083)      ; HIRAGANA LETTER SMALL YA
                 (#x2464 . #x3084)      ; HIRAGANA LETTER YA
                 (#x2465 . #x3085)      ; HIRAGANA LETTER SMALL YU
                 (#x2466 . #x3086)      ; HIRAGANA LETTER YU
                 (#x2467 . #x3087)      ; HIRAGANA LETTER SMALL YO
                 (#x2468 . #x3088)      ; HIRAGANA LETTER YO
                 (#x2469 . #x3089)      ; HIRAGANA LETTER RA
                 (#x246A . #x308A)      ; HIRAGANA LETTER RI
                 (#x246B . #x308B)      ; HIRAGANA LETTER RU
                 (#x246C . #x308C)      ; HIRAGANA LETTER RE
                 (#x246D . #x308D)      ; HIRAGANA LETTER RO
                 (#x246E . #x308E)      ; HIRAGANA LETTER SMALL WA
                 (#x246F . #x308F)      ; HIRAGANA LETTER WA
                 (#x2470 . #x3090)      ; HIRAGANA LETTER WI
                 (#x2471 . #x3091)      ; HIRAGANA LETTER WE
                 (#x2472 . #x3092)      ; HIRAGANA LETTER WO
                 (#x2473 . #x3093)      ; HIRAGANA LETTER N
                 (#x2521 . #x30A1)      ; KATAKANA LETTER SMALL A
                 (#x2522 . #x30A2)      ; KATAKANA LETTER A
                 (#x2523 . #x30A3)      ; KATAKANA LETTER SMALL I
                 (#x2524 . #x30A4)      ; KATAKANA LETTER I
                 (#x2525 . #x30A5)      ; KATAKANA LETTER SMALL U
                 (#x2526 . #x30A6)      ; KATAKANA LETTER U
                 (#x2527 . #x30A7)      ; KATAKANA LETTER SMALL E
                 (#x2528 . #x30A8)      ; KATAKANA LETTER E
                 (#x2529 . #x30A9)      ; KATAKANA LETTER SMALL O
                 (#x252A . #x30AA)      ; KATAKANA LETTER O
                 (#x252B . #x30AB)      ; KATAKANA LETTER KA
                 (#x252C . #x30AC)      ; KATAKANA LETTER GA
                 (#x252D . #x30AD)      ; KATAKANA LETTER KI
                 (#x252E . #x30AE)      ; KATAKANA LETTER GI
                 (#x252F . #x30AF)      ; KATAKANA LETTER KU
                 (#x2530 . #x30B0)      ; KATAKANA LETTER GU
                 (#x2531 . #x30B1)      ; KATAKANA LETTER KE
                 (#x2532 . #x30B2)      ; KATAKANA LETTER GE
                 (#x2533 . #x30B3)      ; KATAKANA LETTER KO
                 (#x2534 . #x30B4)      ; KATAKANA LETTER GO
                 (#x2535 . #x30B5)      ; KATAKANA LETTER SA
                 (#x2536 . #x30B6)      ; KATAKANA LETTER ZA
                 (#x2537 . #x30B7)      ; KATAKANA LETTER SI
                 (#x2538 . #x30B8)      ; KATAKANA LETTER ZI
                 (#x2539 . #x30B9)      ; KATAKANA LETTER SU
                 (#x253A . #x30BA)      ; KATAKANA LETTER ZU
                 (#x253B . #x30BB)      ; KATAKANA LETTER SE
                 (#x253C . #x30BC)      ; KATAKANA LETTER ZE
                 (#x253D . #x30BD)      ; KATAKANA LETTER SO
                 (#x253E . #x30BE)      ; KATAKANA LETTER ZO
                 (#x253F . #x30BF)      ; KATAKANA LETTER TA
                 (#x2540 . #x30C0)      ; KATAKANA LETTER DA
                 (#x2541 . #x30C1)      ; KATAKANA LETTER TI
                 (#x2542 . #x30C2)      ; KATAKANA LETTER DI
                 (#x2543 . #x30C3)      ; KATAKANA LETTER SMALL TU
                 (#x2544 . #x30C4)      ; KATAKANA LETTER TU
                 (#x2545 . #x30C5)      ; KATAKANA LETTER DU
                 (#x2546 . #x30C6)      ; KATAKANA LETTER TE
                 (#x2547 . #x30C7)      ; KATAKANA LETTER DE
                 (#x2548 . #x30C8)      ; KATAKANA LETTER TO
                 (#x2549 . #x30C9)      ; KATAKANA LETTER DO
                 (#x254A . #x30CA)      ; KATAKANA LETTER NA
                 (#x254B . #x30CB)      ; KATAKANA LETTER NI
                 (#x254C . #x30CC)      ; KATAKANA LETTER NU
                 (#x254D . #x30CD)      ; KATAKANA LETTER NE
                 (#x254E . #x30CE)      ; KATAKANA LETTER NO
                 (#x254F . #x30CF)      ; KATAKANA LETTER HA
                 (#x2550 . #x30D0)      ; KATAKANA LETTER BA
                 (#x2551 . #x30D1)      ; KATAKANA LETTER PA
                 (#x2552 . #x30D2)      ; KATAKANA LETTER HI
                 (#x2553 . #x30D3)      ; KATAKANA LETTER BI
                 (#x2554 . #x30D4)      ; KATAKANA LETTER PI
                 (#x2555 . #x30D5)      ; KATAKANA LETTER HU
                 (#x2556 . #x30D6)      ; KATAKANA LETTER BU
                 (#x2557 . #x30D7)      ; KATAKANA LETTER PU
                 (#x2558 . #x30D8)      ; KATAKANA LETTER HE
                 (#x2559 . #x30D9)      ; KATAKANA LETTER BE
                 (#x255A . #x30DA)      ; KATAKANA LETTER PE
                 (#x255B . #x30DB)      ; KATAKANA LETTER HO
                 (#x255C . #x30DC)      ; KATAKANA LETTER BO
                 (#x255D . #x30DD)      ; KATAKANA LETTER PO
                 (#x255E . #x30DE)      ; KATAKANA LETTER MA
                 (#x255F . #x30DF)      ; KATAKANA LETTER MI
                 (#x2560 . #x30E0)      ; KATAKANA LETTER MU
                 (#x2561 . #x30E1)      ; KATAKANA LETTER ME
                 (#x2562 . #x30E2)      ; KATAKANA LETTER MO
                 (#x2563 . #x30E3)      ; KATAKANA LETTER SMALL YA
                 (#x2564 . #x30E4)      ; KATAKANA LETTER YA
                 (#x2565 . #x30E5)      ; KATAKANA LETTER SMALL YU
                 (#x2566 . #x30E6)      ; KATAKANA LETTER YU
                 (#x2567 . #x30E7)      ; KATAKANA LETTER SMALL YO
                 (#x2568 . #x30E8)      ; KATAKANA LETTER YO
                 (#x2569 . #x30E9)      ; KATAKANA LETTER RA
                 (#x256A . #x30EA)      ; KATAKANA LETTER RI
                 (#x256B . #x30EB)      ; KATAKANA LETTER RU
                 (#x256C . #x30EC)      ; KATAKANA LETTER RE
                 (#x256D . #x30ED)      ; KATAKANA LETTER RO
                 (#x256E . #x30EE)      ; KATAKANA LETTER SMALL WA
                 (#x256F . #x30EF)      ; KATAKANA LETTER WA
                 (#x2570 . #x30F0)      ; KATAKANA LETTER WI
                 (#x2571 . #x30F1)      ; KATAKANA LETTER WE
                 (#x2572 . #x30F2)      ; KATAKANA LETTER WO
                 (#x2573 . #x30F3)      ; KATAKANA LETTER N
                 (#x2574 . #x30F4)      ; KATAKANA LETTER VU
                 (#x2575 . #x30F5)      ; KATAKANA LETTER SMALL KA
                 (#x2576 . #x30F6)      ; KATAKANA LETTER SMALL KE
                 (#x2621 . #x0391)      ; GREEK CAPITAL LETTER ALPHA
                 (#x2622 . #x0392)      ; GREEK CAPITAL LETTER BETA
                 (#x2623 . #x0393)      ; GREEK CAPITAL LETTER GAMMA
                 (#x2624 . #x0394)      ; GREEK CAPITAL LETTER DELTA
                 (#x2625 . #x0395)      ; GREEK CAPITAL LETTER EPSILON
                 (#x2626 . #x0396)      ; GREEK CAPITAL LETTER ZETA
                 (#x2627 . #x0397)      ; GREEK CAPITAL LETTER ETA
                 (#x2628 . #x0398)      ; GREEK CAPITAL LETTER THETA
                 (#x2629 . #x0399)      ; GREEK CAPITAL LETTER IOTA
                 (#x262A . #x039A)      ; GREEK CAPITAL LETTER KAPPA
                 (#x262B . #x039B)      ; GREEK CAPITAL LETTER LAMDA
                 (#x262C . #x039C)      ; GREEK CAPITAL LETTER MU
                 (#x262D . #x039D)      ; GREEK CAPITAL LETTER NU
                 (#x262E . #x039E)      ; GREEK CAPITAL LETTER XI
                 (#x262F . #x039F)      ; GREEK CAPITAL LETTER OMICRON
                 (#x2630 . #x03A0)      ; GREEK CAPITAL LETTER PI
                 (#x2631 . #x03A1)      ; GREEK CAPITAL LETTER RHO
                 (#x2632 . #x03A3)      ; GREEK CAPITAL LETTER SIGMA
                 (#x2633 . #x03A4)      ; GREEK CAPITAL LETTER TAU
                 (#x2634 . #x03A5)      ; GREEK CAPITAL LETTER UPSILON
                 (#x2635 . #x03A6)      ; GREEK CAPITAL LETTER PHI
                 (#x2636 . #x03A7)      ; GREEK CAPITAL LETTER CHI
                 (#x2637 . #x03A8)      ; GREEK CAPITAL LETTER PSI
                 (#x2638 . #x03A9)      ; GREEK CAPITAL LETTER OMEGA
                 (#x2641 . #x03B1)      ; GREEK SMALL LETTER ALPHA
                 (#x2642 . #x03B2)      ; GREEK SMALL LETTER BETA
                 (#x2643 . #x03B3)      ; GREEK SMALL LETTER GAMMA
                 (#x2644 . #x03B4)      ; GREEK SMALL LETTER DELTA
                 (#x2645 . #x03B5)      ; GREEK SMALL LETTER EPSILON
                 (#x2646 . #x03B6)      ; GREEK SMALL LETTER ZETA
                 (#x2647 . #x03B7)      ; GREEK SMALL LETTER ETA
                 (#x2648 . #x03B8)      ; GREEK SMALL LETTER THETA
                 (#x2649 . #x03B9)      ; GREEK SMALL LETTER IOTA
                 (#x264A . #x03BA)      ; GREEK SMALL LETTER KAPPA
                 (#x264B . #x03BB)      ; GREEK SMALL LETTER LAMDA
                 (#x264C . #x03BC)      ; GREEK SMALL LETTER MU
                 (#x264D . #x03BD)      ; GREEK SMALL LETTER NU
                 (#x264E . #x03BE)      ; GREEK SMALL LETTER XI
                 (#x264F . #x03BF)      ; GREEK SMALL LETTER OMICRON
                 (#x2650 . #x03C0)      ; GREEK SMALL LETTER PI
                 (#x2651 . #x03C1)      ; GREEK SMALL LETTER RHO
                 (#x2652 . #x03C3)      ; GREEK SMALL LETTER SIGMA
                 (#x2653 . #x03C4)      ; GREEK SMALL LETTER TAU
                 (#x2654 . #x03C5)      ; GREEK SMALL LETTER UPSILON
                 (#x2655 . #x03C6)      ; GREEK SMALL LETTER PHI
                 (#x2656 . #x03C7)      ; GREEK SMALL LETTER CHI
                 (#x2657 . #x03C8)      ; GREEK SMALL LETTER PSI
                 (#x2658 . #x03C9)      ; GREEK SMALL LETTER OMEGA
                 (#x2721 . #x0410)      ; CYRILLIC CAPITAL LETTER A
                 (#x2722 . #x0411)      ; CYRILLIC CAPITAL LETTER BE
                 (#x2723 . #x0412)      ; CYRILLIC CAPITAL LETTER VE
                 (#x2724 . #x0413)      ; CYRILLIC CAPITAL LETTER GHE
                 (#x2725 . #x0414)      ; CYRILLIC CAPITAL LETTER DE
                 (#x2726 . #x0415)      ; CYRILLIC CAPITAL LETTER IE
                 (#x2727 . #x0401)      ; CYRILLIC CAPITAL LETTER IO
                 (#x2728 . #x0416)      ; CYRILLIC CAPITAL LETTER ZHE
                 (#x2729 . #x0417)      ; CYRILLIC CAPITAL LETTER ZE
                 (#x272A . #x0418)      ; CYRILLIC CAPITAL LETTER I
                 (#x272B . #x0419)      ; CYRILLIC CAPITAL LETTER SHORT I
                 (#x272C . #x041A)      ; CYRILLIC CAPITAL LETTER KA
                 (#x272D . #x041B)      ; CYRILLIC CAPITAL LETTER EL
                 (#x272E . #x041C)      ; CYRILLIC CAPITAL LETTER EM
                 (#x272F . #x041D)      ; CYRILLIC CAPITAL LETTER EN
                 (#x2730 . #x041E)      ; CYRILLIC CAPITAL LETTER O
                 (#x2731 . #x041F)      ; CYRILLIC CAPITAL LETTER PE
                 (#x2732 . #x0420)      ; CYRILLIC CAPITAL LETTER ER
                 (#x2733 . #x0421)      ; CYRILLIC CAPITAL LETTER ES
                 (#x2734 . #x0422)      ; CYRILLIC CAPITAL LETTER TE
                 (#x2735 . #x0423)      ; CYRILLIC CAPITAL LETTER U
                 (#x2736 . #x0424)      ; CYRILLIC CAPITAL LETTER EF
                 (#x2737 . #x0425)      ; CYRILLIC CAPITAL LETTER HA
                 (#x2738 . #x0426)      ; CYRILLIC CAPITAL LETTER TSE
                 (#x2739 . #x0427)      ; CYRILLIC CAPITAL LETTER CHE
                 (#x273A . #x0428)      ; CYRILLIC CAPITAL LETTER SHA
                 (#x273B . #x0429)      ; CYRILLIC CAPITAL LETTER SHCHA
                 (#x273C . #x042A)      ; CYRILLIC CAPITAL LETTER HARD SIGN
                 (#x273D . #x042B)      ; CYRILLIC CAPITAL LETTER YERU
                 (#x273E . #x042C)      ; CYRILLIC CAPITAL LETTER SOFT SIGN
                 (#x273F . #x042D)      ; CYRILLIC CAPITAL LETTER E
                 (#x2740 . #x042E)      ; CYRILLIC CAPITAL LETTER YU
                 (#x2741 . #x042F)      ; CYRILLIC CAPITAL LETTER YA
                 (#x2751 . #x0430)      ; CYRILLIC SMALL LETTER A
                 (#x2752 . #x0431)      ; CYRILLIC SMALL LETTER BE
                 (#x2753 . #x0432)      ; CYRILLIC SMALL LETTER VE
                 (#x2754 . #x0433)      ; CYRILLIC SMALL LETTER GHE
                 (#x2755 . #x0434)      ; CYRILLIC SMALL LETTER DE
                 (#x2756 . #x0435)      ; CYRILLIC SMALL LETTER IE
                 (#x2757 . #x0451)      ; CYRILLIC SMALL LETTER IO
                 (#x2758 . #x0436)      ; CYRILLIC SMALL LETTER ZHE
                 (#x2759 . #x0437)      ; CYRILLIC SMALL LETTER ZE
                 (#x275A . #x0438)      ; CYRILLIC SMALL LETTER I
                 (#x275B . #x0439)      ; CYRILLIC SMALL LETTER SHORT I
                 (#x275C . #x043A)      ; CYRILLIC SMALL LETTER KA
                 (#x275D . #x043B)      ; CYRILLIC SMALL LETTER EL
                 (#x275E . #x043C)      ; CYRILLIC SMALL LETTER EM
                 (#x275F . #x043D)      ; CYRILLIC SMALL LETTER EN
                 (#x2760 . #x043E)      ; CYRILLIC SMALL LETTER O
                 (#x2761 . #x043F)      ; CYRILLIC SMALL LETTER PE
                 (#x2762 . #x0440)      ; CYRILLIC SMALL LETTER ER
                 (#x2763 . #x0441)      ; CYRILLIC SMALL LETTER ES
                 (#x2764 . #x0442)      ; CYRILLIC SMALL LETTER TE
                 (#x2765 . #x0443)      ; CYRILLIC SMALL LETTER U
                 (#x2766 . #x0444)      ; CYRILLIC SMALL LETTER EF
                 (#x2767 . #x0445)      ; CYRILLIC SMALL LETTER HA
                 (#x2768 . #x0446)      ; CYRILLIC SMALL LETTER TSE
                 (#x2769 . #x0447)      ; CYRILLIC SMALL LETTER CHE
                 (#x276A . #x0448)      ; CYRILLIC SMALL LETTER SHA
                 (#x276B . #x0449)      ; CYRILLIC SMALL LETTER SHCHA
                 (#x276C . #x044A)      ; CYRILLIC SMALL LETTER HARD SIGN
                 (#x276D . #x044B)      ; CYRILLIC SMALL LETTER YERU
                 (#x276E . #x044C)      ; CYRILLIC SMALL LETTER SOFT SIGN
                 (#x276F . #x044D)      ; CYRILLIC SMALL LETTER E
                 (#x2770 . #x044E)      ; CYRILLIC SMALL LETTER YU
                 (#x2771 . #x044F)      ; CYRILLIC SMALL LETTER YA
                 (#x2821 . #x0101)      ; LATIN SMALL LETTER A WITH MACRON
                 (#x2822 . #x00E1)      ; LATIN SMALL LETTER A WITH ACUTE
                 (#x2823 . #x01CE)      ; LATIN SMALL LETTER A WITH CARON
                 (#x2824 . #x00E0)      ; LATIN SMALL LETTER A WITH GRAVE
                 (#x2825 . #x0113)      ; LATIN SMALL LETTER E WITH MACRON
                 (#x2826 . #x00E9)      ; LATIN SMALL LETTER E WITH ACUTE
                 (#x2827 . #x011B)      ; LATIN SMALL LETTER E WITH CARON
                 (#x2828 . #x00E8)      ; LATIN SMALL LETTER E WITH GRAVE
                 (#x2829 . #x012B)      ; LATIN SMALL LETTER I WITH MACRON
                 (#x282A . #x00ED)      ; LATIN SMALL LETTER I WITH ACUTE
                 (#x282B . #x01D0)      ; LATIN SMALL LETTER I WITH CARON
                 (#x282C . #x00EC)      ; LATIN SMALL LETTER I WITH GRAVE
                 (#x282D . #x014D)      ; LATIN SMALL LETTER O WITH MACRON
                 (#x282E . #x00F3)      ; LATIN SMALL LETTER O WITH ACUTE
                 (#x282F . #x01D2)      ; LATIN SMALL LETTER O WITH CARON
                 (#x2830 . #x00F2)      ; LATIN SMALL LETTER O WITH GRAVE
                 (#x2831 . #x016B)      ; LATIN SMALL LETTER U WITH MACRON
                 (#x2832 . #x00FA)      ; LATIN SMALL LETTER U WITH ACUTE
                 (#x2833 . #x01D4)      ; LATIN SMALL LETTER U WITH CARON
                 (#x2834 . #x00F9)      ; LATIN SMALL LETTER U WITH GRAVE
                 (#x2835 . #x01D6)      ; LATIN SMALL LETTER U WITH DIAERESIS AND MACRON
                 (#x2836 . #x01D8)      ; LATIN SMALL LETTER U WITH DIAERESIS AND ACUTE
                 (#x2837 . #x01DA)      ; LATIN SMALL LETTER U WITH DIAERESIS AND CARON
                 (#x2838 . #x01DC)      ; LATIN SMALL LETTER U WITH DIAERESIS AND GRAVE
                 (#x2839 . #x00FC)      ; LATIN SMALL LETTER U WITH DIAERESIS
                 (#x283A . #x00EA)      ; LATIN SMALL LETTER E WITH CIRCUMFLEX
                 (#x2845 . #x3105)      ; BOPOMOFO LETTER B
                 (#x2846 . #x3106)      ; BOPOMOFO LETTER P
                 (#x2847 . #x3107)      ; BOPOMOFO LETTER M
                 (#x2848 . #x3108)      ; BOPOMOFO LETTER F
                 (#x2849 . #x3109)      ; BOPOMOFO LETTER D
                 (#x284A . #x310A)      ; BOPOMOFO LETTER T
                 (#x284B . #x310B)      ; BOPOMOFO LETTER N
                 (#x284C . #x310C)      ; BOPOMOFO LETTER L
                 (#x284D . #x310D)      ; BOPOMOFO LETTER G
                 (#x284E . #x310E)      ; BOPOMOFO LETTER K
                 (#x284F . #x310F)      ; BOPOMOFO LETTER H
                 (#x2850 . #x3110)      ; BOPOMOFO LETTER J
                 (#x2851 . #x3111)      ; BOPOMOFO LETTER Q
                 (#x2852 . #x3112)      ; BOPOMOFO LETTER X
                 (#x2853 . #x3113)      ; BOPOMOFO LETTER ZH
                 (#x2854 . #x3114)      ; BOPOMOFO LETTER CH
                 (#x2855 . #x3115)      ; BOPOMOFO LETTER SH
                 (#x2856 . #x3116)      ; BOPOMOFO LETTER R
                 (#x2857 . #x3117)      ; BOPOMOFO LETTER Z
                 (#x2858 . #x3118)      ; BOPOMOFO LETTER C
                 (#x2859 . #x3119)      ; BOPOMOFO LETTER S
                 (#x285A . #x311A)      ; BOPOMOFO LETTER A
                 (#x285B . #x311B)      ; BOPOMOFO LETTER O
                 (#x285C . #x311C)      ; BOPOMOFO LETTER E
                 (#x285D . #x311D)      ; BOPOMOFO LETTER EH
                 (#x285E . #x311E)      ; BOPOMOFO LETTER AI
                 (#x285F . #x311F)      ; BOPOMOFO LETTER EI
                 (#x2860 . #x3120)      ; BOPOMOFO LETTER AU
                 (#x2861 . #x3121)      ; BOPOMOFO LETTER OU
                 (#x2862 . #x3122)      ; BOPOMOFO LETTER AN
                 (#x2863 . #x3123)      ; BOPOMOFO LETTER EN
                 (#x2864 . #x3124)      ; BOPOMOFO LETTER ANG
                 (#x2865 . #x3125)      ; BOPOMOFO LETTER ENG
                 (#x2866 . #x3126)      ; BOPOMOFO LETTER ER
                 (#x2867 . #x3127)      ; BOPOMOFO LETTER I
                 (#x2868 . #x3128)      ; BOPOMOFO LETTER U
                 (#x2869 . #x3129)      ; BOPOMOFO LETTER IU
                 (#x2924 . #x2500)      ; BOX DRAWINGS LIGHT HORIZONTAL
                 (#x2925 . #x2501)      ; BOX DRAWINGS HEAVY HORIZONTAL
                 (#x2926 . #x2502)      ; BOX DRAWINGS LIGHT VERTICAL
                 (#x2927 . #x2503)      ; BOX DRAWINGS HEAVY VERTICAL
                 (#x2928 . #x2504)      ; BOX DRAWINGS LIGHT TRIPLE DASH HORIZONTAL
                 (#x2929 . #x2505)      ; BOX DRAWINGS HEAVY TRIPLE DASH HORIZONTAL
                 (#x292A . #x2506)      ; BOX DRAWINGS LIGHT TRIPLE DASH VERTICAL
                 (#x292B . #x2507)      ; BOX DRAWINGS HEAVY TRIPLE DASH VERTICAL
                 (#x292C . #x2508)      ; BOX DRAWINGS LIGHT QUADRUPLE DASH HORIZONTAL
                 (#x292D . #x2509)      ; BOX DRAWINGS HEAVY QUADRUPLE DASH HORIZONTAL
                 (#x292E . #x250A)      ; BOX DRAWINGS LIGHT QUADRUPLE DASH VERTICAL
                 (#x292F . #x250B)      ; BOX DRAWINGS HEAVY QUADRUPLE DASH VERTICAL
                 (#x2930 . #x250C)      ; BOX DRAWINGS LIGHT DOWN AND RIGHT
                 (#x2931 . #x250D)      ; BOX DRAWINGS DOWN LIGHT AND RIGHT HEAVY
                 (#x2932 . #x250E)      ; BOX DRAWINGS DOWN HEAVY AND RIGHT LIGHT
                 (#x2933 . #x250F)      ; BOX DRAWINGS HEAVY DOWN AND RIGHT
                 (#x2934 . #x2510)      ; BOX DRAWINGS LIGHT DOWN AND LEFT
                 (#x2935 . #x2511)      ; BOX DRAWINGS DOWN LIGHT AND LEFT HEAVY
                 (#x2936 . #x2512)      ; BOX DRAWINGS DOWN HEAVY AND LEFT LIGHT
                 (#x2937 . #x2513)      ; BOX DRAWINGS HEAVY DOWN AND LEFT
                 (#x2938 . #x2514)      ; BOX DRAWINGS LIGHT UP AND RIGHT
                 (#x2939 . #x2515)      ; BOX DRAWINGS UP LIGHT AND RIGHT HEAVY
                 (#x293A . #x2516)      ; BOX DRAWINGS UP HEAVY AND RIGHT LIGHT
                 (#x293B . #x2517)      ; BOX DRAWINGS HEAVY UP AND RIGHT
                 (#x293C . #x2518)      ; BOX DRAWINGS LIGHT UP AND LEFT
                 (#x293D . #x2519)      ; BOX DRAWINGS UP LIGHT AND LEFT HEAVY
                 (#x293E . #x251A)      ; BOX DRAWINGS UP HEAVY AND LEFT LIGHT
                 (#x293F . #x251B)      ; BOX DRAWINGS HEAVY UP AND LEFT
                 (#x2940 . #x251C)      ; BOX DRAWINGS LIGHT VERTICAL AND RIGHT
                 (#x2941 . #x251D)      ; BOX DRAWINGS VERTICAL LIGHT AND RIGHT HEAVY
                 (#x2942 . #x251E)      ; BOX DRAWINGS UP HEAVY AND RIGHT DOWN LIGHT
                 (#x2943 . #x251F)      ; BOX DRAWINGS DOWN HEAVY AND RIGHT UP LIGHT
                 (#x2944 . #x2520)      ; BOX DRAWINGS VERTICAL HEAVY AND RIGHT LIGHT
                 (#x2945 . #x2521)      ; BOX DRAWINGS DOWN LIGHT AND RIGHT UP HEAVY
                 (#x2946 . #x2522)      ; BOX DRAWINGS UP LIGHT AND RIGHT DOWN HEAVY
                 (#x2947 . #x2523)      ; BOX DRAWINGS HEAVY VERTICAL AND RIGHT
                 (#x2948 . #x2524)      ; BOX DRAWINGS LIGHT VERTICAL AND LEFT
                 (#x2949 . #x2525)      ; BOX DRAWINGS VERTICAL LIGHT AND LEFT HEAVY
                 (#x294A . #x2526)      ; BOX DRAWINGS UP HEAVY AND LEFT DOWN LIGHT
                 (#x294B . #x2527)      ; BOX DRAWINGS DOWN HEAVY AND LEFT UP LIGHT
                 (#x294C . #x2528)      ; BOX DRAWINGS VERTICAL HEAVY AND LEFT LIGHT
                 (#x294D . #x2529)      ; BOX DRAWINGS DOWN LIGHT AND LEFT UP HEAVY
                 (#x294E . #x252A)      ; BOX DRAWINGS UP LIGHT AND LEFT DOWN HEAVY
                 (#x294F . #x252B)      ; BOX DRAWINGS HEAVY VERTICAL AND LEFT
                 (#x2950 . #x252C)      ; BOX DRAWINGS LIGHT DOWN AND HORIZONTAL
                 (#x2951 . #x252D)      ; BOX DRAWINGS LEFT HEAVY AND RIGHT DOWN LIGHT
                 (#x2952 . #x252E)      ; BOX DRAWINGS RIGHT HEAVY AND LEFT DOWN LIGHT
                 (#x2953 . #x252F)      ; BOX DRAWINGS DOWN LIGHT AND HORIZONTAL HEAVY
                 (#x2954 . #x2530)      ; BOX DRAWINGS DOWN HEAVY AND HORIZONTAL LIGHT
                 (#x2955 . #x2531)      ; BOX DRAWINGS RIGHT LIGHT AND LEFT DOWN HEAVY
                 (#x2956 . #x2532)      ; BOX DRAWINGS LEFT LIGHT AND RIGHT DOWN HEAVY
                 (#x2957 . #x2533)      ; BOX DRAWINGS HEAVY DOWN AND HORIZONTAL
                 (#x2958 . #x2534)      ; BOX DRAWINGS LIGHT UP AND HORIZONTAL
                 (#x2959 . #x2535)      ; BOX DRAWINGS LEFT HEAVY AND RIGHT UP LIGHT
                 (#x295A . #x2536)      ; BOX DRAWINGS RIGHT HEAVY AND LEFT UP LIGHT
                 (#x295B . #x2537)      ; BOX DRAWINGS UP LIGHT AND HORIZONTAL HEAVY
                 (#x295C . #x2538)      ; BOX DRAWINGS UP HEAVY AND HORIZONTAL LIGHT
                 (#x295D . #x2539)      ; BOX DRAWINGS RIGHT LIGHT AND LEFT UP HEAVY
                 (#x295E . #x253A)      ; BOX DRAWINGS LEFT LIGHT AND RIGHT UP HEAVY
                 (#x295F . #x253B)      ; BOX DRAWINGS HEAVY UP AND HORIZONTAL
                 (#x2960 . #x253C)      ; BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL
                 (#x2961 . #x253D)      ; BOX DRAWINGS LEFT HEAVY AND RIGHT VERTICAL LIGHT
                 (#x2962 . #x253E)      ; BOX DRAWINGS RIGHT HEAVY AND LEFT VERTICAL LIGHT
                 (#x2963 . #x253F)      ; BOX DRAWINGS VERTICAL LIGHT AND HORIZONTAL HEAVY
                 (#x2964 . #x2540)      ; BOX DRAWINGS UP HEAVY AND DOWN HORIZONTAL LIGHT
                 (#x2965 . #x2541)      ; BOX DRAWINGS DOWN HEAVY AND UP HORIZONTAL LIGHT
                 (#x2966 . #x2542)      ; BOX DRAWINGS VERTICAL HEAVY AND HORIZONTAL LIGHT
                 (#x2967 . #x2543)      ; BOX DRAWINGS LEFT UP HEAVY AND RIGHT DOWN LIGHT
                 (#x2968 . #x2544)      ; BOX DRAWINGS RIGHT UP HEAVY AND LEFT DOWN LIGHT
                 (#x2969 . #x2545)      ; BOX DRAWINGS LEFT DOWN HEAVY AND RIGHT UP LIGHT
                 (#x296A . #x2546)      ; BOX DRAWINGS RIGHT DOWN HEAVY AND LEFT UP LIGHT
                 (#x296B . #x2547)      ; BOX DRAWINGS DOWN LIGHT AND UP HORIZONTAL HEAVY
                 (#x296C . #x2548)      ; BOX DRAWINGS UP LIGHT AND DOWN HORIZONTAL HEAVY
                 (#x296D . #x2549)      ; BOX DRAWINGS RIGHT LIGHT AND LEFT VERTICAL HEAVY
                 (#x296E . #x254A)      ; BOX DRAWINGS LEFT LIGHT AND RIGHT VERTICAL HEAVY
                 (#x296F . #x254B)      ; BOX DRAWINGS HEAVY VERTICAL AND HORIZONTAL
                 (#x3021 . #x554A)      ; <CJK>
                 (#x3022 . #x963F)      ; <CJK>
                 (#x3023 . #x57C3)      ; <CJK>
                 (#x3024 . #x6328)      ; <CJK>
                 (#x3025 . #x54CE)      ; <CJK>
                 (#x3026 . #x5509)      ; <CJK>
                 (#x3027 . #x54C0)      ; <CJK>
                 (#x3028 . #x7691)      ; <CJK>
                 (#x3029 . #x764C)      ; <CJK>
                 (#x302A . #x853C)      ; <CJK>
                 (#x302B . #x77EE)      ; <CJK>
                 (#x302C . #x827E)      ; <CJK>
                 (#x302D . #x788D)      ; <CJK>
                 (#x302E . #x7231)      ; <CJK>
                 (#x302F . #x9698)      ; <CJK>
                 (#x3030 . #x978D)      ; <CJK>
                 (#x3031 . #x6C28)      ; <CJK>
                 (#x3032 . #x5B89)      ; <CJK>
                 (#x3033 . #x4FFA)      ; <CJK>
                 (#x3034 . #x6309)      ; <CJK>
                 (#x3035 . #x6697)      ; <CJK>
                 (#x3036 . #x5CB8)      ; <CJK>
                 (#x3037 . #x80FA)      ; <CJK>
                 (#x3038 . #x6848)      ; <CJK>
                 (#x3039 . #x80AE)      ; <CJK>
                 (#x303A . #x6602)      ; <CJK>
                 (#x303B . #x76CE)      ; <CJK>
                 (#x303C . #x51F9)      ; <CJK>
                 (#x303D . #x6556)      ; <CJK>
                 (#x303E . #x71AC)      ; <CJK>
                 (#x303F . #x7FF1)      ; <CJK>
                 (#x3040 . #x8884)      ; <CJK>
                 (#x3041 . #x50B2)      ; <CJK>
                 (#x3042 . #x5965)      ; <CJK>
                 (#x3043 . #x61CA)      ; <CJK>
                 (#x3044 . #x6FB3)      ; <CJK>
                 (#x3045 . #x82AD)      ; <CJK>
                 (#x3046 . #x634C)      ; <CJK>
                 (#x3047 . #x6252)      ; <CJK>
                 (#x3048 . #x53ED)      ; <CJK>
                 (#x3049 . #x5427)      ; <CJK>
                 (#x304A . #x7B06)      ; <CJK>
                 (#x304B . #x516B)      ; <CJK>
                 (#x304C . #x75A4)      ; <CJK>
                 (#x304D . #x5DF4)      ; <CJK>
                 (#x304E . #x62D4)      ; <CJK>
                 (#x304F . #x8DCB)      ; <CJK>
                 (#x3050 . #x9776)      ; <CJK>
                 (#x3051 . #x628A)      ; <CJK>
                 (#x3052 . #x8019)      ; <CJK>
                 (#x3053 . #x575D)      ; <CJK>
                 (#x3054 . #x9738)      ; <CJK>
                 (#x3055 . #x7F62)      ; <CJK>
                 (#x3056 . #x7238)      ; <CJK>
                 (#x3057 . #x767D)      ; <CJK>
                 (#x3058 . #x67CF)      ; <CJK>
                 (#x3059 . #x767E)      ; <CJK>
                 (#x305A . #x6446)      ; <CJK>
                 (#x305B . #x4F70)      ; <CJK>
                 (#x305C . #x8D25)      ; <CJK>
                 (#x305D . #x62DC)      ; <CJK>
                 (#x305E . #x7A17)      ; <CJK>
                 (#x305F . #x6591)      ; <CJK>
                 (#x3060 . #x73ED)      ; <CJK>
                 (#x3061 . #x642C)      ; <CJK>
                 (#x3062 . #x6273)      ; <CJK>
                 (#x3063 . #x822C)      ; <CJK>
                 (#x3064 . #x9881)      ; <CJK>
                 (#x3065 . #x677F)      ; <CJK>
                 (#x3066 . #x7248)      ; <CJK>
                 (#x3067 . #x626E)      ; <CJK>
                 (#x3068 . #x62CC)      ; <CJK>
                 (#x3069 . #x4F34)      ; <CJK>
                 (#x306A . #x74E3)      ; <CJK>
                 (#x306B . #x534A)      ; <CJK>
                 (#x306C . #x529E)      ; <CJK>
                 (#x306D . #x7ECA)      ; <CJK>
                 (#x306E . #x90A6)      ; <CJK>
                 (#x306F . #x5E2E)      ; <CJK>
                 (#x3070 . #x6886)      ; <CJK>
                 (#x3071 . #x699C)      ; <CJK>
                 (#x3072 . #x8180)      ; <CJK>
                 (#x3073 . #x7ED1)      ; <CJK>
                 (#x3074 . #x68D2)      ; <CJK>
                 (#x3075 . #x78C5)      ; <CJK>
                 (#x3076 . #x868C)      ; <CJK>
                 (#x3077 . #x9551)      ; <CJK>
                 (#x3078 . #x508D)      ; <CJK>
                 (#x3079 . #x8C24)      ; <CJK>
                 (#x307A . #x82DE)      ; <CJK>
                 (#x307B . #x80DE)      ; <CJK>
                 (#x307C . #x5305)      ; <CJK>
                 (#x307D . #x8912)      ; <CJK>
                 (#x307E . #x5265)      ; <CJK>
                 (#x3121 . #x8584)      ; <CJK>
                 (#x3122 . #x96F9)      ; <CJK>
                 (#x3123 . #x4FDD)      ; <CJK>
                 (#x3124 . #x5821)      ; <CJK>
                 (#x3125 . #x9971)      ; <CJK>
                 (#x3126 . #x5B9D)      ; <CJK>
                 (#x3127 . #x62B1)      ; <CJK>
                 (#x3128 . #x62A5)      ; <CJK>
                 (#x3129 . #x66B4)      ; <CJK>
                 (#x312A . #x8C79)      ; <CJK>
                 (#x312B . #x9C8D)      ; <CJK>
                 (#x312C . #x7206)      ; <CJK>
                 (#x312D . #x676F)      ; <CJK>
                 (#x312E . #x7891)      ; <CJK>
                 (#x312F . #x60B2)      ; <CJK>
                 (#x3130 . #x5351)      ; <CJK>
                 (#x3131 . #x5317)      ; <CJK>
                 (#x3132 . #x8F88)      ; <CJK>
                 (#x3133 . #x80CC)      ; <CJK>
                 (#x3134 . #x8D1D)      ; <CJK>
                 (#x3135 . #x94A1)      ; <CJK>
                 (#x3136 . #x500D)      ; <CJK>
                 (#x3137 . #x72C8)      ; <CJK>
                 (#x3138 . #x5907)      ; <CJK>
                 (#x3139 . #x60EB)      ; <CJK>
                 (#x313A . #x7119)      ; <CJK>
                 (#x313B . #x88AB)      ; <CJK>
                 (#x313C . #x5954)      ; <CJK>
                 (#x313D . #x82EF)      ; <CJK>
                 (#x313E . #x672C)      ; <CJK>
                 (#x313F . #x7B28)      ; <CJK>
                 (#x3140 . #x5D29)      ; <CJK>
                 (#x3141 . #x7EF7)      ; <CJK>
                 (#x3142 . #x752D)      ; <CJK>
                 (#x3143 . #x6CF5)      ; <CJK>
                 (#x3144 . #x8E66)      ; <CJK>
                 (#x3145 . #x8FF8)      ; <CJK>
                 (#x3146 . #x903C)      ; <CJK>
                 (#x3147 . #x9F3B)      ; <CJK>
                 (#x3148 . #x6BD4)      ; <CJK>
                 (#x3149 . #x9119)      ; <CJK>
                 (#x314A . #x7B14)      ; <CJK>
                 (#x314B . #x5F7C)      ; <CJK>
                 (#x314C . #x78A7)      ; <CJK>
                 (#x314D . #x84D6)      ; <CJK>
                 (#x314E . #x853D)      ; <CJK>
                 (#x314F . #x6BD5)      ; <CJK>
                 (#x3150 . #x6BD9)      ; <CJK>
                 (#x3151 . #x6BD6)      ; <CJK>
                 (#x3152 . #x5E01)      ; <CJK>
                 (#x3153 . #x5E87)      ; <CJK>
                 (#x3154 . #x75F9)      ; <CJK>
                 (#x3155 . #x95ED)      ; <CJK>
                 (#x3156 . #x655D)      ; <CJK>
                 (#x3157 . #x5F0A)      ; <CJK>
                 (#x3158 . #x5FC5)      ; <CJK>
                 (#x3159 . #x8F9F)      ; <CJK>
                 (#x315A . #x58C1)      ; <CJK>
                 (#x315B . #x81C2)      ; <CJK>
                 (#x315C . #x907F)      ; <CJK>
                 (#x315D . #x965B)      ; <CJK>
                 (#x315E . #x97AD)      ; <CJK>
                 (#x315F . #x8FB9)      ; <CJK>
                 (#x3160 . #x7F16)      ; <CJK>
                 (#x3161 . #x8D2C)      ; <CJK>
                 (#x3162 . #x6241)      ; <CJK>
                 (#x3163 . #x4FBF)      ; <CJK>
                 (#x3164 . #x53D8)      ; <CJK>
                 (#x3165 . #x535E)      ; <CJK>
                 (#x3166 . #x8FA8)      ; <CJK>
                 (#x3167 . #x8FA9)      ; <CJK>
                 (#x3168 . #x8FAB)      ; <CJK>
                 (#x3169 . #x904D)      ; <CJK>
                 (#x316A . #x6807)      ; <CJK>
                 (#x316B . #x5F6A)      ; <CJK>
                 (#x316C . #x8198)      ; <CJK>
                 (#x316D . #x8868)      ; <CJK>
                 (#x316E . #x9CD6)      ; <CJK>
                 (#x316F . #x618B)      ; <CJK>
                 (#x3170 . #x522B)      ; <CJK>
                 (#x3171 . #x762A)      ; <CJK>
                 (#x3172 . #x5F6C)      ; <CJK>
                 (#x3173 . #x658C)      ; <CJK>
                 (#x3174 . #x6FD2)      ; <CJK>
                 (#x3175 . #x6EE8)      ; <CJK>
                 (#x3176 . #x5BBE)      ; <CJK>
                 (#x3177 . #x6448)      ; <CJK>
                 (#x3178 . #x5175)      ; <CJK>
                 (#x3179 . #x51B0)      ; <CJK>
                 (#x317A . #x67C4)      ; <CJK>
                 (#x317B . #x4E19)      ; <CJK>
                 (#x317C . #x79C9)      ; <CJK>
                 (#x317D . #x997C)      ; <CJK>
                 (#x317E . #x70B3)      ; <CJK>
                 (#x3221 . #x75C5)      ; <CJK>
                 (#x3222 . #x5E76)      ; <CJK>
                 (#x3223 . #x73BB)      ; <CJK>
                 (#x3224 . #x83E0)      ; <CJK>
                 (#x3225 . #x64AD)      ; <CJK>
                 (#x3226 . #x62E8)      ; <CJK>
                 (#x3227 . #x94B5)      ; <CJK>
                 (#x3228 . #x6CE2)      ; <CJK>
                 (#x3229 . #x535A)      ; <CJK>
                 (#x322A . #x52C3)      ; <CJK>
                 (#x322B . #x640F)      ; <CJK>
                 (#x322C . #x94C2)      ; <CJK>
                 (#x322D . #x7B94)      ; <CJK>
                 (#x322E . #x4F2F)      ; <CJK>
                 (#x322F . #x5E1B)      ; <CJK>
                 (#x3230 . #x8236)      ; <CJK>
                 (#x3231 . #x8116)      ; <CJK>
                 (#x3232 . #x818A)      ; <CJK>
                 (#x3233 . #x6E24)      ; <CJK>
                 (#x3234 . #x6CCA)      ; <CJK>
                 (#x3235 . #x9A73)      ; <CJK>
                 (#x3236 . #x6355)      ; <CJK>
                 (#x3237 . #x535C)      ; <CJK>
                 (#x3238 . #x54FA)      ; <CJK>
                 (#x3239 . #x8865)      ; <CJK>
                 (#x323A . #x57E0)      ; <CJK>
                 (#x323B . #x4E0D)      ; <CJK>
                 (#x323C . #x5E03)      ; <CJK>
                 (#x323D . #x6B65)      ; <CJK>
                 (#x323E . #x7C3F)      ; <CJK>
                 (#x323F . #x90E8)      ; <CJK>
                 (#x3240 . #x6016)      ; <CJK>
                 (#x3241 . #x64E6)      ; <CJK>
                 (#x3242 . #x731C)      ; <CJK>
                 (#x3243 . #x88C1)      ; <CJK>
                 (#x3244 . #x6750)      ; <CJK>
                 (#x3245 . #x624D)      ; <CJK>
                 (#x3246 . #x8D22)      ; <CJK>
                 (#x3247 . #x776C)      ; <CJK>
                 (#x3248 . #x8E29)      ; <CJK>
                 (#x3249 . #x91C7)      ; <CJK>
                 (#x324A . #x5F69)      ; <CJK>
                 (#x324B . #x83DC)      ; <CJK>
                 (#x324C . #x8521)      ; <CJK>
                 (#x324D . #x9910)      ; <CJK>
                 (#x324E . #x53C2)      ; <CJK>
                 (#x324F . #x8695)      ; <CJK>
                 (#x3250 . #x6B8B)      ; <CJK>
                 (#x3251 . #x60ED)      ; <CJK>
                 (#x3252 . #x60E8)      ; <CJK>
                 (#x3253 . #x707F)      ; <CJK>
                 (#x3254 . #x82CD)      ; <CJK>
                 (#x3255 . #x8231)      ; <CJK>
                 (#x3256 . #x4ED3)      ; <CJK>
                 (#x3257 . #x6CA7)      ; <CJK>
                 (#x3258 . #x85CF)      ; <CJK>
                 (#x3259 . #x64CD)      ; <CJK>
                 (#x325A . #x7CD9)      ; <CJK>
                 (#x325B . #x69FD)      ; <CJK>
                 (#x325C . #x66F9)      ; <CJK>
                 (#x325D . #x8349)      ; <CJK>
                 (#x325E . #x5395)      ; <CJK>
                 (#x325F . #x7B56)      ; <CJK>
                 (#x3260 . #x4FA7)      ; <CJK>
                 (#x3261 . #x518C)      ; <CJK>
                 (#x3262 . #x6D4B)      ; <CJK>
                 (#x3263 . #x5C42)      ; <CJK>
                 (#x3264 . #x8E6D)      ; <CJK>
                 (#x3265 . #x63D2)      ; <CJK>
                 (#x3266 . #x53C9)      ; <CJK>
                 (#x3267 . #x832C)      ; <CJK>
                 (#x3268 . #x8336)      ; <CJK>
                 (#x3269 . #x67E5)      ; <CJK>
                 (#x326A . #x78B4)      ; <CJK>
                 (#x326B . #x643D)      ; <CJK>
                 (#x326C . #x5BDF)      ; <CJK>
                 (#x326D . #x5C94)      ; <CJK>
                 (#x326E . #x5DEE)      ; <CJK>
                 (#x326F . #x8BE7)      ; <CJK>
                 (#x3270 . #x62C6)      ; <CJK>
                 (#x3271 . #x67F4)      ; <CJK>
                 (#x3272 . #x8C7A)      ; <CJK>
                 (#x3273 . #x6400)      ; <CJK>
                 (#x3274 . #x63BA)      ; <CJK>
                 (#x3275 . #x8749)      ; <CJK>
                 (#x3276 . #x998B)      ; <CJK>
                 (#x3277 . #x8C17)      ; <CJK>
                 (#x3278 . #x7F20)      ; <CJK>
                 (#x3279 . #x94F2)      ; <CJK>
                 (#x327A . #x4EA7)      ; <CJK>
                 (#x327B . #x9610)      ; <CJK>
                 (#x327C . #x98A4)      ; <CJK>
                 (#x327D . #x660C)      ; <CJK>
                 (#x327E . #x7316)      ; <CJK>
                 (#x3321 . #x573A)      ; <CJK>
                 (#x3322 . #x5C1D)      ; <CJK>
                 (#x3323 . #x5E38)      ; <CJK>
                 (#x3324 . #x957F)      ; <CJK>
                 (#x3325 . #x507F)      ; <CJK>
                 (#x3326 . #x80A0)      ; <CJK>
                 (#x3327 . #x5382)      ; <CJK>
                 (#x3328 . #x655E)      ; <CJK>
                 (#x3329 . #x7545)      ; <CJK>
                 (#x332A . #x5531)      ; <CJK>
                 (#x332B . #x5021)      ; <CJK>
                 (#x332C . #x8D85)      ; <CJK>
                 (#x332D . #x6284)      ; <CJK>
                 (#x332E . #x949E)      ; <CJK>
                 (#x332F . #x671D)      ; <CJK>
                 (#x3330 . #x5632)      ; <CJK>
                 (#x3331 . #x6F6E)      ; <CJK>
                 (#x3332 . #x5DE2)      ; <CJK>
                 (#x3333 . #x5435)      ; <CJK>
                 (#x3334 . #x7092)      ; <CJK>
                 (#x3335 . #x8F66)      ; <CJK>
                 (#x3336 . #x626F)      ; <CJK>
                 (#x3337 . #x64A4)      ; <CJK>
                 (#x3338 . #x63A3)      ; <CJK>
                 (#x3339 . #x5F7B)      ; <CJK>
                 (#x333A . #x6F88)      ; <CJK>
                 (#x333B . #x90F4)      ; <CJK>
                 (#x333C . #x81E3)      ; <CJK>
                 (#x333D . #x8FB0)      ; <CJK>
                 (#x333E . #x5C18)      ; <CJK>
                 (#x333F . #x6668)      ; <CJK>
                 (#x3340 . #x5FF1)      ; <CJK>
                 (#x3341 . #x6C89)      ; <CJK>
                 (#x3342 . #x9648)      ; <CJK>
                 (#x3343 . #x8D81)      ; <CJK>
                 (#x3344 . #x886C)      ; <CJK>
                 (#x3345 . #x6491)      ; <CJK>
                 (#x3346 . #x79F0)      ; <CJK>
                 (#x3347 . #x57CE)      ; <CJK>
                 (#x3348 . #x6A59)      ; <CJK>
                 (#x3349 . #x6210)      ; <CJK>
                 (#x334A . #x5448)      ; <CJK>
                 (#x334B . #x4E58)      ; <CJK>
                 (#x334C . #x7A0B)      ; <CJK>
                 (#x334D . #x60E9)      ; <CJK>
                 (#x334E . #x6F84)      ; <CJK>
                 (#x334F . #x8BDA)      ; <CJK>
                 (#x3350 . #x627F)      ; <CJK>
                 (#x3351 . #x901E)      ; <CJK>
                 (#x3352 . #x9A8B)      ; <CJK>
                 (#x3353 . #x79E4)      ; <CJK>
                 (#x3354 . #x5403)      ; <CJK>
                 (#x3355 . #x75F4)      ; <CJK>
                 (#x3356 . #x6301)      ; <CJK>
                 (#x3357 . #x5319)      ; <CJK>
                 (#x3358 . #x6C60)      ; <CJK>
                 (#x3359 . #x8FDF)      ; <CJK>
                 (#x335A . #x5F1B)      ; <CJK>
                 (#x335B . #x9A70)      ; <CJK>
                 (#x335C . #x803B)      ; <CJK>
                 (#x335D . #x9F7F)      ; <CJK>
                 (#x335E . #x4F88)      ; <CJK>
                 (#x335F . #x5C3A)      ; <CJK>
                 (#x3360 . #x8D64)      ; <CJK>
                 (#x3361 . #x7FC5)      ; <CJK>
                 (#x3362 . #x65A5)      ; <CJK>
                 (#x3363 . #x70BD)      ; <CJK>
                 (#x3364 . #x5145)      ; <CJK>
                 (#x3365 . #x51B2)      ; <CJK>
                 (#x3366 . #x866B)      ; <CJK>
                 (#x3367 . #x5D07)      ; <CJK>
                 (#x3368 . #x5BA0)      ; <CJK>
                 (#x3369 . #x62BD)      ; <CJK>
                 (#x336A . #x916C)      ; <CJK>
                 (#x336B . #x7574)      ; <CJK>
                 (#x336C . #x8E0C)      ; <CJK>
                 (#x336D . #x7A20)      ; <CJK>
                 (#x336E . #x6101)      ; <CJK>
                 (#x336F . #x7B79)      ; <CJK>
                 (#x3370 . #x4EC7)      ; <CJK>
                 (#x3371 . #x7EF8)      ; <CJK>
                 (#x3372 . #x7785)      ; <CJK>
                 (#x3373 . #x4E11)      ; <CJK>
                 (#x3374 . #x81ED)      ; <CJK>
                 (#x3375 . #x521D)      ; <CJK>
                 (#x3376 . #x51FA)      ; <CJK>
                 (#x3377 . #x6A71)      ; <CJK>
                 (#x3378 . #x53A8)      ; <CJK>
                 (#x3379 . #x8E87)      ; <CJK>
                 (#x337A . #x9504)      ; <CJK>
                 (#x337B . #x96CF)      ; <CJK>
                 (#x337C . #x6EC1)      ; <CJK>
                 (#x337D . #x9664)      ; <CJK>
                 (#x337E . #x695A)      ; <CJK>
                 (#x3421 . #x7840)      ; <CJK>
                 (#x3422 . #x50A8)      ; <CJK>
                 (#x3423 . #x77D7)      ; <CJK>
                 (#x3424 . #x6410)      ; <CJK>
                 (#x3425 . #x89E6)      ; <CJK>
                 (#x3426 . #x5904)      ; <CJK>
                 (#x3427 . #x63E3)      ; <CJK>
                 (#x3428 . #x5DDD)      ; <CJK>
                 (#x3429 . #x7A7F)      ; <CJK>
                 (#x342A . #x693D)      ; <CJK>
                 (#x342B . #x4F20)      ; <CJK>
                 (#x342C . #x8239)      ; <CJK>
                 (#x342D . #x5598)      ; <CJK>
                 (#x342E . #x4E32)      ; <CJK>
                 (#x342F . #x75AE)      ; <CJK>
                 (#x3430 . #x7A97)      ; <CJK>
                 (#x3431 . #x5E62)      ; <CJK>
                 (#x3432 . #x5E8A)      ; <CJK>
                 (#x3433 . #x95EF)      ; <CJK>
                 (#x3434 . #x521B)      ; <CJK>
                 (#x3435 . #x5439)      ; <CJK>
                 (#x3436 . #x708A)      ; <CJK>
                 (#x3437 . #x6376)      ; <CJK>
                 (#x3438 . #x9524)      ; <CJK>
                 (#x3439 . #x5782)      ; <CJK>
                 (#x343A . #x6625)      ; <CJK>
                 (#x343B . #x693F)      ; <CJK>
                 (#x343C . #x9187)      ; <CJK>
                 (#x343D . #x5507)      ; <CJK>
                 (#x343E . #x6DF3)      ; <CJK>
                 (#x343F . #x7EAF)      ; <CJK>
                 (#x3440 . #x8822)      ; <CJK>
                 (#x3441 . #x6233)      ; <CJK>
                 (#x3442 . #x7EF0)      ; <CJK>
                 (#x3443 . #x75B5)      ; <CJK>
                 (#x3444 . #x8328)      ; <CJK>
                 (#x3445 . #x78C1)      ; <CJK>
                 (#x3446 . #x96CC)      ; <CJK>
                 (#x3447 . #x8F9E)      ; <CJK>
                 (#x3448 . #x6148)      ; <CJK>
                 (#x3449 . #x74F7)      ; <CJK>
                 (#x344A . #x8BCD)      ; <CJK>
                 (#x344B . #x6B64)      ; <CJK>
                 (#x344C . #x523A)      ; <CJK>
                 (#x344D . #x8D50)      ; <CJK>
                 (#x344E . #x6B21)      ; <CJK>
                 (#x344F . #x806A)      ; <CJK>
                 (#x3450 . #x8471)      ; <CJK>
                 (#x3451 . #x56F1)      ; <CJK>
                 (#x3452 . #x5306)      ; <CJK>
                 (#x3453 . #x4ECE)      ; <CJK>
                 (#x3454 . #x4E1B)      ; <CJK>
                 (#x3455 . #x51D1)      ; <CJK>
                 (#x3456 . #x7C97)      ; <CJK>
                 (#x3457 . #x918B)      ; <CJK>
                 (#x3458 . #x7C07)      ; <CJK>
                 (#x3459 . #x4FC3)      ; <CJK>
                 (#x345A . #x8E7F)      ; <CJK>
                 (#x345B . #x7BE1)      ; <CJK>
                 (#x345C . #x7A9C)      ; <CJK>
                 (#x345D . #x6467)      ; <CJK>
                 (#x345E . #x5D14)      ; <CJK>
                 (#x345F . #x50AC)      ; <CJK>
                 (#x3460 . #x8106)      ; <CJK>
                 (#x3461 . #x7601)      ; <CJK>
                 (#x3462 . #x7CB9)      ; <CJK>
                 (#x3463 . #x6DEC)      ; <CJK>
                 (#x3464 . #x7FE0)      ; <CJK>
                 (#x3465 . #x6751)      ; <CJK>
                 (#x3466 . #x5B58)      ; <CJK>
                 (#x3467 . #x5BF8)      ; <CJK>
                 (#x3468 . #x78CB)      ; <CJK>
                 (#x3469 . #x64AE)      ; <CJK>
                 (#x346A . #x6413)      ; <CJK>
                 (#x346B . #x63AA)      ; <CJK>
                 (#x346C . #x632B)      ; <CJK>
                 (#x346D . #x9519)      ; <CJK>
                 (#x346E . #x642D)      ; <CJK>
                 (#x346F . #x8FBE)      ; <CJK>
                 (#x3470 . #x7B54)      ; <CJK>
                 (#x3471 . #x7629)      ; <CJK>
                 (#x3472 . #x6253)      ; <CJK>
                 (#x3473 . #x5927)      ; <CJK>
                 (#x3474 . #x5446)      ; <CJK>
                 (#x3475 . #x6B79)      ; <CJK>
                 (#x3476 . #x50A3)      ; <CJK>
                 (#x3477 . #x6234)      ; <CJK>
                 (#x3478 . #x5E26)      ; <CJK>
                 (#x3479 . #x6B86)      ; <CJK>
                 (#x347A . #x4EE3)      ; <CJK>
                 (#x347B . #x8D37)      ; <CJK>
                 (#x347C . #x888B)      ; <CJK>
                 (#x347D . #x5F85)      ; <CJK>
                 (#x347E . #x902E)      ; <CJK>
                 (#x3521 . #x6020)      ; <CJK>
                 (#x3522 . #x803D)      ; <CJK>
                 (#x3523 . #x62C5)      ; <CJK>
                 (#x3524 . #x4E39)      ; <CJK>
                 (#x3525 . #x5355)      ; <CJK>
                 (#x3526 . #x90F8)      ; <CJK>
                 (#x3527 . #x63B8)      ; <CJK>
                 (#x3528 . #x80C6)      ; <CJK>
                 (#x3529 . #x65E6)      ; <CJK>
                 (#x352A . #x6C2E)      ; <CJK>
                 (#x352B . #x4F46)      ; <CJK>
                 (#x352C . #x60EE)      ; <CJK>
                 (#x352D . #x6DE1)      ; <CJK>
                 (#x352E . #x8BDE)      ; <CJK>
                 (#x352F . #x5F39)      ; <CJK>
                 (#x3530 . #x86CB)      ; <CJK>
                 (#x3531 . #x5F53)      ; <CJK>
                 (#x3532 . #x6321)      ; <CJK>
                 (#x3533 . #x515A)      ; <CJK>
                 (#x3534 . #x8361)      ; <CJK>
                 (#x3535 . #x6863)      ; <CJK>
                 (#x3536 . #x5200)      ; <CJK>
                 (#x3537 . #x6363)      ; <CJK>
                 (#x3538 . #x8E48)      ; <CJK>
                 (#x3539 . #x5012)      ; <CJK>
                 (#x353A . #x5C9B)      ; <CJK>
                 (#x353B . #x7977)      ; <CJK>
                 (#x353C . #x5BFC)      ; <CJK>
                 (#x353D . #x5230)      ; <CJK>
                 (#x353E . #x7A3B)      ; <CJK>
                 (#x353F . #x60BC)      ; <CJK>
                 (#x3540 . #x9053)      ; <CJK>
                 (#x3541 . #x76D7)      ; <CJK>
                 (#x3542 . #x5FB7)      ; <CJK>
                 (#x3543 . #x5F97)      ; <CJK>
                 (#x3544 . #x7684)      ; <CJK>
                 (#x3545 . #x8E6C)      ; <CJK>
                 (#x3546 . #x706F)      ; <CJK>
                 (#x3547 . #x767B)      ; <CJK>
                 (#x3548 . #x7B49)      ; <CJK>
                 (#x3549 . #x77AA)      ; <CJK>
                 (#x354A . #x51F3)      ; <CJK>
                 (#x354B . #x9093)      ; <CJK>
                 (#x354C . #x5824)      ; <CJK>
                 (#x354D . #x4F4E)      ; <CJK>
                 (#x354E . #x6EF4)      ; <CJK>
                 (#x354F . #x8FEA)      ; <CJK>
                 (#x3550 . #x654C)      ; <CJK>
                 (#x3551 . #x7B1B)      ; <CJK>
                 (#x3552 . #x72C4)      ; <CJK>
                 (#x3553 . #x6DA4)      ; <CJK>
                 (#x3554 . #x7FDF)      ; <CJK>
                 (#x3555 . #x5AE1)      ; <CJK>
                 (#x3556 . #x62B5)      ; <CJK>
                 (#x3557 . #x5E95)      ; <CJK>
                 (#x3558 . #x5730)      ; <CJK>
                 (#x3559 . #x8482)      ; <CJK>
                 (#x355A . #x7B2C)      ; <CJK>
                 (#x355B . #x5E1D)      ; <CJK>
                 (#x355C . #x5F1F)      ; <CJK>
                 (#x355D . #x9012)      ; <CJK>
                 (#x355E . #x7F14)      ; <CJK>
                 (#x355F . #x98A0)      ; <CJK>
                 (#x3560 . #x6382)      ; <CJK>
                 (#x3561 . #x6EC7)      ; <CJK>
                 (#x3562 . #x7898)      ; <CJK>
                 (#x3563 . #x70B9)      ; <CJK>
                 (#x3564 . #x5178)      ; <CJK>
                 (#x3565 . #x975B)      ; <CJK>
                 (#x3566 . #x57AB)      ; <CJK>
                 (#x3567 . #x7535)      ; <CJK>
                 (#x3568 . #x4F43)      ; <CJK>
                 (#x3569 . #x7538)      ; <CJK>
                 (#x356A . #x5E97)      ; <CJK>
                 (#x356B . #x60E6)      ; <CJK>
                 (#x356C . #x5960)      ; <CJK>
                 (#x356D . #x6DC0)      ; <CJK>
                 (#x356E . #x6BBF)      ; <CJK>
                 (#x356F . #x7889)      ; <CJK>
                 (#x3570 . #x53FC)      ; <CJK>
                 (#x3571 . #x96D5)      ; <CJK>
                 (#x3572 . #x51CB)      ; <CJK>
                 (#x3573 . #x5201)      ; <CJK>
                 (#x3574 . #x6389)      ; <CJK>
                 (#x3575 . #x540A)      ; <CJK>
                 (#x3576 . #x9493)      ; <CJK>
                 (#x3577 . #x8C03)      ; <CJK>
                 (#x3578 . #x8DCC)      ; <CJK>
                 (#x3579 . #x7239)      ; <CJK>
                 (#x357A . #x789F)      ; <CJK>
                 (#x357B . #x8776)      ; <CJK>
                 (#x357C . #x8FED)      ; <CJK>
                 (#x357D . #x8C0D)      ; <CJK>
                 (#x357E . #x53E0)      ; <CJK>
                 (#x3621 . #x4E01)      ; <CJK>
                 (#x3622 . #x76EF)      ; <CJK>
                 (#x3623 . #x53EE)      ; <CJK>
                 (#x3624 . #x9489)      ; <CJK>
                 (#x3625 . #x9876)      ; <CJK>
                 (#x3626 . #x9F0E)      ; <CJK>
                 (#x3627 . #x952D)      ; <CJK>
                 (#x3628 . #x5B9A)      ; <CJK>
                 (#x3629 . #x8BA2)      ; <CJK>
                 (#x362A . #x4E22)      ; <CJK>
                 (#x362B . #x4E1C)      ; <CJK>
                 (#x362C . #x51AC)      ; <CJK>
                 (#x362D . #x8463)      ; <CJK>
                 (#x362E . #x61C2)      ; <CJK>
                 (#x362F . #x52A8)      ; <CJK>
                 (#x3630 . #x680B)      ; <CJK>
                 (#x3631 . #x4F97)      ; <CJK>
                 (#x3632 . #x606B)      ; <CJK>
                 (#x3633 . #x51BB)      ; <CJK>
                 (#x3634 . #x6D1E)      ; <CJK>
                 (#x3635 . #x515C)      ; <CJK>
                 (#x3636 . #x6296)      ; <CJK>
                 (#x3637 . #x6597)      ; <CJK>
                 (#x3638 . #x9661)      ; <CJK>
                 (#x3639 . #x8C46)      ; <CJK>
                 (#x363A . #x9017)      ; <CJK>
                 (#x363B . #x75D8)      ; <CJK>
                 (#x363C . #x90FD)      ; <CJK>
                 (#x363D . #x7763)      ; <CJK>
                 (#x363E . #x6BD2)      ; <CJK>
                 (#x363F . #x728A)      ; <CJK>
                 (#x3640 . #x72EC)      ; <CJK>
                 (#x3641 . #x8BFB)      ; <CJK>
                 (#x3642 . #x5835)      ; <CJK>
                 (#x3643 . #x7779)      ; <CJK>
                 (#x3644 . #x8D4C)      ; <CJK>
                 (#x3645 . #x675C)      ; <CJK>
                 (#x3646 . #x9540)      ; <CJK>
                 (#x3647 . #x809A)      ; <CJK>
                 (#x3648 . #x5EA6)      ; <CJK>
                 (#x3649 . #x6E21)      ; <CJK>
                 (#x364A . #x5992)      ; <CJK>
                 (#x364B . #x7AEF)      ; <CJK>
                 (#x364C . #x77ED)      ; <CJK>
                 (#x364D . #x953B)      ; <CJK>
                 (#x364E . #x6BB5)      ; <CJK>
                 (#x364F . #x65AD)      ; <CJK>
                 (#x3650 . #x7F0E)      ; <CJK>
                 (#x3651 . #x5806)      ; <CJK>
                 (#x3652 . #x5151)      ; <CJK>
                 (#x3653 . #x961F)      ; <CJK>
                 (#x3654 . #x5BF9)      ; <CJK>
                 (#x3655 . #x58A9)      ; <CJK>
                 (#x3656 . #x5428)      ; <CJK>
                 (#x3657 . #x8E72)      ; <CJK>
                 (#x3658 . #x6566)      ; <CJK>
                 (#x3659 . #x987F)      ; <CJK>
                 (#x365A . #x56E4)      ; <CJK>
                 (#x365B . #x949D)      ; <CJK>
                 (#x365C . #x76FE)      ; <CJK>
                 (#x365D . #x9041)      ; <CJK>
                 (#x365E . #x6387)      ; <CJK>
                 (#x365F . #x54C6)      ; <CJK>
                 (#x3660 . #x591A)      ; <CJK>
                 (#x3661 . #x593A)      ; <CJK>
                 (#x3662 . #x579B)      ; <CJK>
                 (#x3663 . #x8EB2)      ; <CJK>
                 (#x3664 . #x6735)      ; <CJK>
                 (#x3665 . #x8DFA)      ; <CJK>
                 (#x3666 . #x8235)      ; <CJK>
                 (#x3667 . #x5241)      ; <CJK>
                 (#x3668 . #x60F0)      ; <CJK>
                 (#x3669 . #x5815)      ; <CJK>
                 (#x366A . #x86FE)      ; <CJK>
                 (#x366B . #x5CE8)      ; <CJK>
                 (#x366C . #x9E45)      ; <CJK>
                 (#x366D . #x4FC4)      ; <CJK>
                 (#x366E . #x989D)      ; <CJK>
                 (#x366F . #x8BB9)      ; <CJK>
                 (#x3670 . #x5A25)      ; <CJK>
                 (#x3671 . #x6076)      ; <CJK>
                 (#x3672 . #x5384)      ; <CJK>
                 (#x3673 . #x627C)      ; <CJK>
                 (#x3674 . #x904F)      ; <CJK>
                 (#x3675 . #x9102)      ; <CJK>
                 (#x3676 . #x997F)      ; <CJK>
                 (#x3677 . #x6069)      ; <CJK>
                 (#x3678 . #x800C)      ; <CJK>
                 (#x3679 . #x513F)      ; <CJK>
                 (#x367A . #x8033)      ; <CJK>
                 (#x367B . #x5C14)      ; <CJK>
                 (#x367C . #x9975)      ; <CJK>
                 (#x367D . #x6D31)      ; <CJK>
                 (#x367E . #x4E8C)      ; <CJK>
                 (#x3721 . #x8D30)      ; <CJK>
                 (#x3722 . #x53D1)      ; <CJK>
                 (#x3723 . #x7F5A)      ; <CJK>
                 (#x3724 . #x7B4F)      ; <CJK>
                 (#x3725 . #x4F10)      ; <CJK>
                 (#x3726 . #x4E4F)      ; <CJK>
                 (#x3727 . #x9600)      ; <CJK>
                 (#x3728 . #x6CD5)      ; <CJK>
                 (#x3729 . #x73D0)      ; <CJK>
                 (#x372A . #x85E9)      ; <CJK>
                 (#x372B . #x5E06)      ; <CJK>
                 (#x372C . #x756A)      ; <CJK>
                 (#x372D . #x7FFB)      ; <CJK>
                 (#x372E . #x6A0A)      ; <CJK>
                 (#x372F . #x77FE)      ; <CJK>
                 (#x3730 . #x9492)      ; <CJK>
                 (#x3731 . #x7E41)      ; <CJK>
                 (#x3732 . #x51E1)      ; <CJK>
                 (#x3733 . #x70E6)      ; <CJK>
                 (#x3734 . #x53CD)      ; <CJK>
                 (#x3735 . #x8FD4)      ; <CJK>
                 (#x3736 . #x8303)      ; <CJK>
                 (#x3737 . #x8D29)      ; <CJK>
                 (#x3738 . #x72AF)      ; <CJK>
                 (#x3739 . #x996D)      ; <CJK>
                 (#x373A . #x6CDB)      ; <CJK>
                 (#x373B . #x574A)      ; <CJK>
                 (#x373C . #x82B3)      ; <CJK>
                 (#x373D . #x65B9)      ; <CJK>
                 (#x373E . #x80AA)      ; <CJK>
                 (#x373F . #x623F)      ; <CJK>
                 (#x3740 . #x9632)      ; <CJK>
                 (#x3741 . #x59A8)      ; <CJK>
                 (#x3742 . #x4EFF)      ; <CJK>
                 (#x3743 . #x8BBF)      ; <CJK>
                 (#x3744 . #x7EBA)      ; <CJK>
                 (#x3745 . #x653E)      ; <CJK>
                 (#x3746 . #x83F2)      ; <CJK>
                 (#x3747 . #x975E)      ; <CJK>
                 (#x3748 . #x5561)      ; <CJK>
                 (#x3749 . #x98DE)      ; <CJK>
                 (#x374A . #x80A5)      ; <CJK>
                 (#x374B . #x532A)      ; <CJK>
                 (#x374C . #x8BFD)      ; <CJK>
                 (#x374D . #x5420)      ; <CJK>
                 (#x374E . #x80BA)      ; <CJK>
                 (#x374F . #x5E9F)      ; <CJK>
                 (#x3750 . #x6CB8)      ; <CJK>
                 (#x3751 . #x8D39)      ; <CJK>
                 (#x3752 . #x82AC)      ; <CJK>
                 (#x3753 . #x915A)      ; <CJK>
                 (#x3754 . #x5429)      ; <CJK>
                 (#x3755 . #x6C1B)      ; <CJK>
                 (#x3756 . #x5206)      ; <CJK>
                 (#x3757 . #x7EB7)      ; <CJK>
                 (#x3758 . #x575F)      ; <CJK>
                 (#x3759 . #x711A)      ; <CJK>
                 (#x375A . #x6C7E)      ; <CJK>
                 (#x375B . #x7C89)      ; <CJK>
                 (#x375C . #x594B)      ; <CJK>
                 (#x375D . #x4EFD)      ; <CJK>
                 (#x375E . #x5FFF)      ; <CJK>
                 (#x375F . #x6124)      ; <CJK>
                 (#x3760 . #x7CAA)      ; <CJK>
                 (#x3761 . #x4E30)      ; <CJK>
                 (#x3762 . #x5C01)      ; <CJK>
                 (#x3763 . #x67AB)      ; <CJK>
                 (#x3764 . #x8702)      ; <CJK>
                 (#x3765 . #x5CF0)      ; <CJK>
                 (#x3766 . #x950B)      ; <CJK>
                 (#x3767 . #x98CE)      ; <CJK>
                 (#x3768 . #x75AF)      ; <CJK>
                 (#x3769 . #x70FD)      ; <CJK>
                 (#x376A . #x9022)      ; <CJK>
                 (#x376B . #x51AF)      ; <CJK>
                 (#x376C . #x7F1D)      ; <CJK>
                 (#x376D . #x8BBD)      ; <CJK>
                 (#x376E . #x5949)      ; <CJK>
                 (#x376F . #x51E4)      ; <CJK>
                 (#x3770 . #x4F5B)      ; <CJK>
                 (#x3771 . #x5426)      ; <CJK>
                 (#x3772 . #x592B)      ; <CJK>
                 (#x3773 . #x6577)      ; <CJK>
                 (#x3774 . #x80A4)      ; <CJK>
                 (#x3775 . #x5B75)      ; <CJK>
                 (#x3776 . #x6276)      ; <CJK>
                 (#x3777 . #x62C2)      ; <CJK>
                 (#x3778 . #x8F90)      ; <CJK>
                 (#x3779 . #x5E45)      ; <CJK>
                 (#x377A . #x6C1F)      ; <CJK>
                 (#x377B . #x7B26)      ; <CJK>
                 (#x377C . #x4F0F)      ; <CJK>
                 (#x377D . #x4FD8)      ; <CJK>
                 (#x377E . #x670D)      ; <CJK>
                 (#x3821 . #x6D6E)      ; <CJK>
                 (#x3822 . #x6DAA)      ; <CJK>
                 (#x3823 . #x798F)      ; <CJK>
                 (#x3824 . #x88B1)      ; <CJK>
                 (#x3825 . #x5F17)      ; <CJK>
                 (#x3826 . #x752B)      ; <CJK>
                 (#x3827 . #x629A)      ; <CJK>
                 (#x3828 . #x8F85)      ; <CJK>
                 (#x3829 . #x4FEF)      ; <CJK>
                 (#x382A . #x91DC)      ; <CJK>
                 (#x382B . #x65A7)      ; <CJK>
                 (#x382C . #x812F)      ; <CJK>
                 (#x382D . #x8151)      ; <CJK>
                 (#x382E . #x5E9C)      ; <CJK>
                 (#x382F . #x8150)      ; <CJK>
                 (#x3830 . #x8D74)      ; <CJK>
                 (#x3831 . #x526F)      ; <CJK>
                 (#x3832 . #x8986)      ; <CJK>
                 (#x3833 . #x8D4B)      ; <CJK>
                 (#x3834 . #x590D)      ; <CJK>
                 (#x3835 . #x5085)      ; <CJK>
                 (#x3836 . #x4ED8)      ; <CJK>
                 (#x3837 . #x961C)      ; <CJK>
                 (#x3838 . #x7236)      ; <CJK>
                 (#x3839 . #x8179)      ; <CJK>
                 (#x383A . #x8D1F)      ; <CJK>
                 (#x383B . #x5BCC)      ; <CJK>
                 (#x383C . #x8BA3)      ; <CJK>
                 (#x383D . #x9644)      ; <CJK>
                 (#x383E . #x5987)      ; <CJK>
                 (#x383F . #x7F1A)      ; <CJK>
                 (#x3840 . #x5490)      ; <CJK>
                 (#x3841 . #x5676)      ; <CJK>
                 (#x3842 . #x560E)      ; <CJK>
                 (#x3843 . #x8BE5)      ; <CJK>
                 (#x3844 . #x6539)      ; <CJK>
                 (#x3845 . #x6982)      ; <CJK>
                 (#x3846 . #x9499)      ; <CJK>
                 (#x3847 . #x76D6)      ; <CJK>
                 (#x3848 . #x6E89)      ; <CJK>
                 (#x3849 . #x5E72)      ; <CJK>
                 (#x384A . #x7518)      ; <CJK>
                 (#x384B . #x6746)      ; <CJK>
                 (#x384C . #x67D1)      ; <CJK>
                 (#x384D . #x7AFF)      ; <CJK>
                 (#x384E . #x809D)      ; <CJK>
                 (#x384F . #x8D76)      ; <CJK>
                 (#x3850 . #x611F)      ; <CJK>
                 (#x3851 . #x79C6)      ; <CJK>
                 (#x3852 . #x6562)      ; <CJK>
                 (#x3853 . #x8D63)      ; <CJK>
                 (#x3854 . #x5188)      ; <CJK>
                 (#x3855 . #x521A)      ; <CJK>
                 (#x3856 . #x94A2)      ; <CJK>
                 (#x3857 . #x7F38)      ; <CJK>
                 (#x3858 . #x809B)      ; <CJK>
                 (#x3859 . #x7EB2)      ; <CJK>
                 (#x385A . #x5C97)      ; <CJK>
                 (#x385B . #x6E2F)      ; <CJK>
                 (#x385C . #x6760)      ; <CJK>
                 (#x385D . #x7BD9)      ; <CJK>
                 (#x385E . #x768B)      ; <CJK>
                 (#x385F . #x9AD8)      ; <CJK>
                 (#x3860 . #x818F)      ; <CJK>
                 (#x3861 . #x7F94)      ; <CJK>
                 (#x3862 . #x7CD5)      ; <CJK>
                 (#x3863 . #x641E)      ; <CJK>
                 (#x3864 . #x9550)      ; <CJK>
                 (#x3865 . #x7A3F)      ; <CJK>
                 (#x3866 . #x544A)      ; <CJK>
                 (#x3867 . #x54E5)      ; <CJK>
                 (#x3868 . #x6B4C)      ; <CJK>
                 (#x3869 . #x6401)      ; <CJK>
                 (#x386A . #x6208)      ; <CJK>
                 (#x386B . #x9E3D)      ; <CJK>
                 (#x386C . #x80F3)      ; <CJK>
                 (#x386D . #x7599)      ; <CJK>
                 (#x386E . #x5272)      ; <CJK>
                 (#x386F . #x9769)      ; <CJK>
                 (#x3870 . #x845B)      ; <CJK>
                 (#x3871 . #x683C)      ; <CJK>
                 (#x3872 . #x86E4)      ; <CJK>
                 (#x3873 . #x9601)      ; <CJK>
                 (#x3874 . #x9694)      ; <CJK>
                 (#x3875 . #x94EC)      ; <CJK>
                 (#x3876 . #x4E2A)      ; <CJK>
                 (#x3877 . #x5404)      ; <CJK>
                 (#x3878 . #x7ED9)      ; <CJK>
                 (#x3879 . #x6839)      ; <CJK>
                 (#x387A . #x8DDF)      ; <CJK>
                 (#x387B . #x8015)      ; <CJK>
                 (#x387C . #x66F4)      ; <CJK>
                 (#x387D . #x5E9A)      ; <CJK>
                 (#x387E . #x7FB9)      ; <CJK>
                 (#x3921 . #x57C2)      ; <CJK>
                 (#x3922 . #x803F)      ; <CJK>
                 (#x3923 . #x6897)      ; <CJK>
                 (#x3924 . #x5DE5)      ; <CJK>
                 (#x3925 . #x653B)      ; <CJK>
                 (#x3926 . #x529F)      ; <CJK>
                 (#x3927 . #x606D)      ; <CJK>
                 (#x3928 . #x9F9A)      ; <CJK>
                 (#x3929 . #x4F9B)      ; <CJK>
                 (#x392A . #x8EAC)      ; <CJK>
                 (#x392B . #x516C)      ; <CJK>
                 (#x392C . #x5BAB)      ; <CJK>
                 (#x392D . #x5F13)      ; <CJK>
                 (#x392E . #x5DE9)      ; <CJK>
                 (#x392F . #x6C5E)      ; <CJK>
                 (#x3930 . #x62F1)      ; <CJK>
                 (#x3931 . #x8D21)      ; <CJK>
                 (#x3932 . #x5171)      ; <CJK>
                 (#x3933 . #x94A9)      ; <CJK>
                 (#x3934 . #x52FE)      ; <CJK>
                 (#x3935 . #x6C9F)      ; <CJK>
                 (#x3936 . #x82DF)      ; <CJK>
                 (#x3937 . #x72D7)      ; <CJK>
                 (#x3938 . #x57A2)      ; <CJK>
                 (#x3939 . #x6784)      ; <CJK>
                 (#x393A . #x8D2D)      ; <CJK>
                 (#x393B . #x591F)      ; <CJK>
                 (#x393C . #x8F9C)      ; <CJK>
                 (#x393D . #x83C7)      ; <CJK>
                 (#x393E . #x5495)      ; <CJK>
                 (#x393F . #x7B8D)      ; <CJK>
                 (#x3940 . #x4F30)      ; <CJK>
                 (#x3941 . #x6CBD)      ; <CJK>
                 (#x3942 . #x5B64)      ; <CJK>
                 (#x3943 . #x59D1)      ; <CJK>
                 (#x3944 . #x9F13)      ; <CJK>
                 (#x3945 . #x53E4)      ; <CJK>
                 (#x3946 . #x86CA)      ; <CJK>
                 (#x3947 . #x9AA8)      ; <CJK>
                 (#x3948 . #x8C37)      ; <CJK>
                 (#x3949 . #x80A1)      ; <CJK>
                 (#x394A . #x6545)      ; <CJK>
                 (#x394B . #x987E)      ; <CJK>
                 (#x394C . #x56FA)      ; <CJK>
                 (#x394D . #x96C7)      ; <CJK>
                 (#x394E . #x522E)      ; <CJK>
                 (#x394F . #x74DC)      ; <CJK>
                 (#x3950 . #x5250)      ; <CJK>
                 (#x3951 . #x5BE1)      ; <CJK>
                 (#x3952 . #x6302)      ; <CJK>
                 (#x3953 . #x8902)      ; <CJK>
                 (#x3954 . #x4E56)      ; <CJK>
                 (#x3955 . #x62D0)      ; <CJK>
                 (#x3956 . #x602A)      ; <CJK>
                 (#x3957 . #x68FA)      ; <CJK>
                 (#x3958 . #x5173)      ; <CJK>
                 (#x3959 . #x5B98)      ; <CJK>
                 (#x395A . #x51A0)      ; <CJK>
                 (#x395B . #x89C2)      ; <CJK>
                 (#x395C . #x7BA1)      ; <CJK>
                 (#x395D . #x9986)      ; <CJK>
                 (#x395E . #x7F50)      ; <CJK>
                 (#x395F . #x60EF)      ; <CJK>
                 (#x3960 . #x704C)      ; <CJK>
                 (#x3961 . #x8D2F)      ; <CJK>
                 (#x3962 . #x5149)      ; <CJK>
                 (#x3963 . #x5E7F)      ; <CJK>
                 (#x3964 . #x901B)      ; <CJK>
                 (#x3965 . #x7470)      ; <CJK>
                 (#x3966 . #x89C4)      ; <CJK>
                 (#x3967 . #x572D)      ; <CJK>
                 (#x3968 . #x7845)      ; <CJK>
                 (#x3969 . #x5F52)      ; <CJK>
                 (#x396A . #x9F9F)      ; <CJK>
                 (#x396B . #x95FA)      ; <CJK>
                 (#x396C . #x8F68)      ; <CJK>
                 (#x396D . #x9B3C)      ; <CJK>
                 (#x396E . #x8BE1)      ; <CJK>
                 (#x396F . #x7678)      ; <CJK>
                 (#x3970 . #x6842)      ; <CJK>
                 (#x3971 . #x67DC)      ; <CJK>
                 (#x3972 . #x8DEA)      ; <CJK>
                 (#x3973 . #x8D35)      ; <CJK>
                 (#x3974 . #x523D)      ; <CJK>
                 (#x3975 . #x8F8A)      ; <CJK>
                 (#x3976 . #x6EDA)      ; <CJK>
                 (#x3977 . #x68CD)      ; <CJK>
                 (#x3978 . #x9505)      ; <CJK>
                 (#x3979 . #x90ED)      ; <CJK>
                 (#x397A . #x56FD)      ; <CJK>
                 (#x397B . #x679C)      ; <CJK>
                 (#x397C . #x88F9)      ; <CJK>
                 (#x397D . #x8FC7)      ; <CJK>
                 (#x397E . #x54C8)      ; <CJK>
                 (#x3A21 . #x9AB8)      ; <CJK>
                 (#x3A22 . #x5B69)      ; <CJK>
                 (#x3A23 . #x6D77)      ; <CJK>
                 (#x3A24 . #x6C26)      ; <CJK>
                 (#x3A25 . #x4EA5)      ; <CJK>
                 (#x3A26 . #x5BB3)      ; <CJK>
                 (#x3A27 . #x9A87)      ; <CJK>
                 (#x3A28 . #x9163)      ; <CJK>
                 (#x3A29 . #x61A8)      ; <CJK>
                 (#x3A2A . #x90AF)      ; <CJK>
                 (#x3A2B . #x97E9)      ; <CJK>
                 (#x3A2C . #x542B)      ; <CJK>
                 (#x3A2D . #x6DB5)      ; <CJK>
                 (#x3A2E . #x5BD2)      ; <CJK>
                 (#x3A2F . #x51FD)      ; <CJK>
                 (#x3A30 . #x558A)      ; <CJK>
                 (#x3A31 . #x7F55)      ; <CJK>
                 (#x3A32 . #x7FF0)      ; <CJK>
                 (#x3A33 . #x64BC)      ; <CJK>
                 (#x3A34 . #x634D)      ; <CJK>
                 (#x3A35 . #x65F1)      ; <CJK>
                 (#x3A36 . #x61BE)      ; <CJK>
                 (#x3A37 . #x608D)      ; <CJK>
                 (#x3A38 . #x710A)      ; <CJK>
                 (#x3A39 . #x6C57)      ; <CJK>
                 (#x3A3A . #x6C49)      ; <CJK>
                 (#x3A3B . #x592F)      ; <CJK>
                 (#x3A3C . #x676D)      ; <CJK>
                 (#x3A3D . #x822A)      ; <CJK>
                 (#x3A3E . #x58D5)      ; <CJK>
                 (#x3A3F . #x568E)      ; <CJK>
                 (#x3A40 . #x8C6A)      ; <CJK>
                 (#x3A41 . #x6BEB)      ; <CJK>
                 (#x3A42 . #x90DD)      ; <CJK>
                 (#x3A43 . #x597D)      ; <CJK>
                 (#x3A44 . #x8017)      ; <CJK>
                 (#x3A45 . #x53F7)      ; <CJK>
                 (#x3A46 . #x6D69)      ; <CJK>
                 (#x3A47 . #x5475)      ; <CJK>
                 (#x3A48 . #x559D)      ; <CJK>
                 (#x3A49 . #x8377)      ; <CJK>
                 (#x3A4A . #x83CF)      ; <CJK>
                 (#x3A4B . #x6838)      ; <CJK>
                 (#x3A4C . #x79BE)      ; <CJK>
                 (#x3A4D . #x548C)      ; <CJK>
                 (#x3A4E . #x4F55)      ; <CJK>
                 (#x3A4F . #x5408)      ; <CJK>
                 (#x3A50 . #x76D2)      ; <CJK>
                 (#x3A51 . #x8C89)      ; <CJK>
                 (#x3A52 . #x9602)      ; <CJK>
                 (#x3A53 . #x6CB3)      ; <CJK>
                 (#x3A54 . #x6DB8)      ; <CJK>
                 (#x3A55 . #x8D6B)      ; <CJK>
                 (#x3A56 . #x8910)      ; <CJK>
                 (#x3A57 . #x9E64)      ; <CJK>
                 (#x3A58 . #x8D3A)      ; <CJK>
                 (#x3A59 . #x563F)      ; <CJK>
                 (#x3A5A . #x9ED1)      ; <CJK>
                 (#x3A5B . #x75D5)      ; <CJK>
                 (#x3A5C . #x5F88)      ; <CJK>
                 (#x3A5D . #x72E0)      ; <CJK>
                 (#x3A5E . #x6068)      ; <CJK>
                 (#x3A5F . #x54FC)      ; <CJK>
                 (#x3A60 . #x4EA8)      ; <CJK>
                 (#x3A61 . #x6A2A)      ; <CJK>
                 (#x3A62 . #x8861)      ; <CJK>
                 (#x3A63 . #x6052)      ; <CJK>
                 (#x3A64 . #x8F70)      ; <CJK>
                 (#x3A65 . #x54C4)      ; <CJK>
                 (#x3A66 . #x70D8)      ; <CJK>
                 (#x3A67 . #x8679)      ; <CJK>
                 (#x3A68 . #x9E3F)      ; <CJK>
                 (#x3A69 . #x6D2A)      ; <CJK>
                 (#x3A6A . #x5B8F)      ; <CJK>
                 (#x3A6B . #x5F18)      ; <CJK>
                 (#x3A6C . #x7EA2)      ; <CJK>
                 (#x3A6D . #x5589)      ; <CJK>
                 (#x3A6E . #x4FAF)      ; <CJK>
                 (#x3A6F . #x7334)      ; <CJK>
                 (#x3A70 . #x543C)      ; <CJK>
                 (#x3A71 . #x539A)      ; <CJK>
                 (#x3A72 . #x5019)      ; <CJK>
                 (#x3A73 . #x540E)      ; <CJK>
                 (#x3A74 . #x547C)      ; <CJK>
                 (#x3A75 . #x4E4E)      ; <CJK>
                 (#x3A76 . #x5FFD)      ; <CJK>
                 (#x3A77 . #x745A)      ; <CJK>
                 (#x3A78 . #x58F6)      ; <CJK>
                 (#x3A79 . #x846B)      ; <CJK>
                 (#x3A7A . #x80E1)      ; <CJK>
                 (#x3A7B . #x8774)      ; <CJK>
                 (#x3A7C . #x72D0)      ; <CJK>
                 (#x3A7D . #x7CCA)      ; <CJK>
                 (#x3A7E . #x6E56)      ; <CJK>
                 (#x3B21 . #x5F27)      ; <CJK>
                 (#x3B22 . #x864E)      ; <CJK>
                 (#x3B23 . #x552C)      ; <CJK>
                 (#x3B24 . #x62A4)      ; <CJK>
                 (#x3B25 . #x4E92)      ; <CJK>
                 (#x3B26 . #x6CAA)      ; <CJK>
                 (#x3B27 . #x6237)      ; <CJK>
                 (#x3B28 . #x82B1)      ; <CJK>
                 (#x3B29 . #x54D7)      ; <CJK>
                 (#x3B2A . #x534E)      ; <CJK>
                 (#x3B2B . #x733E)      ; <CJK>
                 (#x3B2C . #x6ED1)      ; <CJK>
                 (#x3B2D . #x753B)      ; <CJK>
                 (#x3B2E . #x5212)      ; <CJK>
                 (#x3B2F . #x5316)      ; <CJK>
                 (#x3B30 . #x8BDD)      ; <CJK>
                 (#x3B31 . #x69D0)      ; <CJK>
                 (#x3B32 . #x5F8A)      ; <CJK>
                 (#x3B33 . #x6000)      ; <CJK>
                 (#x3B34 . #x6DEE)      ; <CJK>
                 (#x3B35 . #x574F)      ; <CJK>
                 (#x3B36 . #x6B22)      ; <CJK>
                 (#x3B37 . #x73AF)      ; <CJK>
                 (#x3B38 . #x6853)      ; <CJK>
                 (#x3B39 . #x8FD8)      ; <CJK>
                 (#x3B3A . #x7F13)      ; <CJK>
                 (#x3B3B . #x6362)      ; <CJK>
                 (#x3B3C . #x60A3)      ; <CJK>
                 (#x3B3D . #x5524)      ; <CJK>
                 (#x3B3E . #x75EA)      ; <CJK>
                 (#x3B3F . #x8C62)      ; <CJK>
                 (#x3B40 . #x7115)      ; <CJK>
                 (#x3B41 . #x6DA3)      ; <CJK>
                 (#x3B42 . #x5BA6)      ; <CJK>
                 (#x3B43 . #x5E7B)      ; <CJK>
                 (#x3B44 . #x8352)      ; <CJK>
                 (#x3B45 . #x614C)      ; <CJK>
                 (#x3B46 . #x9EC4)      ; <CJK>
                 (#x3B47 . #x78FA)      ; <CJK>
                 (#x3B48 . #x8757)      ; <CJK>
                 (#x3B49 . #x7C27)      ; <CJK>
                 (#x3B4A . #x7687)      ; <CJK>
                 (#x3B4B . #x51F0)      ; <CJK>
                 (#x3B4C . #x60F6)      ; <CJK>
                 (#x3B4D . #x714C)      ; <CJK>
                 (#x3B4E . #x6643)      ; <CJK>
                 (#x3B4F . #x5E4C)      ; <CJK>
                 (#x3B50 . #x604D)      ; <CJK>
                 (#x3B51 . #x8C0E)      ; <CJK>
                 (#x3B52 . #x7070)      ; <CJK>
                 (#x3B53 . #x6325)      ; <CJK>
                 (#x3B54 . #x8F89)      ; <CJK>
                 (#x3B55 . #x5FBD)      ; <CJK>
                 (#x3B56 . #x6062)      ; <CJK>
                 (#x3B57 . #x86D4)      ; <CJK>
                 (#x3B58 . #x56DE)      ; <CJK>
                 (#x3B59 . #x6BC1)      ; <CJK>
                 (#x3B5A . #x6094)      ; <CJK>
                 (#x3B5B . #x6167)      ; <CJK>
                 (#x3B5C . #x5349)      ; <CJK>
                 (#x3B5D . #x60E0)      ; <CJK>
                 (#x3B5E . #x6666)      ; <CJK>
                 (#x3B5F . #x8D3F)      ; <CJK>
                 (#x3B60 . #x79FD)      ; <CJK>
                 (#x3B61 . #x4F1A)      ; <CJK>
                 (#x3B62 . #x70E9)      ; <CJK>
                 (#x3B63 . #x6C47)      ; <CJK>
                 (#x3B64 . #x8BB3)      ; <CJK>
                 (#x3B65 . #x8BF2)      ; <CJK>
                 (#x3B66 . #x7ED8)      ; <CJK>
                 (#x3B67 . #x8364)      ; <CJK>
                 (#x3B68 . #x660F)      ; <CJK>
                 (#x3B69 . #x5A5A)      ; <CJK>
                 (#x3B6A . #x9B42)      ; <CJK>
                 (#x3B6B . #x6D51)      ; <CJK>
                 (#x3B6C . #x6DF7)      ; <CJK>
                 (#x3B6D . #x8C41)      ; <CJK>
                 (#x3B6E . #x6D3B)      ; <CJK>
                 (#x3B6F . #x4F19)      ; <CJK>
                 (#x3B70 . #x706B)      ; <CJK>
                 (#x3B71 . #x83B7)      ; <CJK>
                 (#x3B72 . #x6216)      ; <CJK>
                 (#x3B73 . #x60D1)      ; <CJK>
                 (#x3B74 . #x970D)      ; <CJK>
                 (#x3B75 . #x8D27)      ; <CJK>
                 (#x3B76 . #x7978)      ; <CJK>
                 (#x3B77 . #x51FB)      ; <CJK>
                 (#x3B78 . #x573E)      ; <CJK>
                 (#x3B79 . #x57FA)      ; <CJK>
                 (#x3B7A . #x673A)      ; <CJK>
                 (#x3B7B . #x7578)      ; <CJK>
                 (#x3B7C . #x7A3D)      ; <CJK>
                 (#x3B7D . #x79EF)      ; <CJK>
                 (#x3B7E . #x7B95)      ; <CJK>
                 (#x3C21 . #x808C)      ; <CJK>
                 (#x3C22 . #x9965)      ; <CJK>
                 (#x3C23 . #x8FF9)      ; <CJK>
                 (#x3C24 . #x6FC0)      ; <CJK>
                 (#x3C25 . #x8BA5)      ; <CJK>
                 (#x3C26 . #x9E21)      ; <CJK>
                 (#x3C27 . #x59EC)      ; <CJK>
                 (#x3C28 . #x7EE9)      ; <CJK>
                 (#x3C29 . #x7F09)      ; <CJK>
                 (#x3C2A . #x5409)      ; <CJK>
                 (#x3C2B . #x6781)      ; <CJK>
                 (#x3C2C . #x68D8)      ; <CJK>
                 (#x3C2D . #x8F91)      ; <CJK>
                 (#x3C2E . #x7C4D)      ; <CJK>
                 (#x3C2F . #x96C6)      ; <CJK>
                 (#x3C30 . #x53CA)      ; <CJK>
                 (#x3C31 . #x6025)      ; <CJK>
                 (#x3C32 . #x75BE)      ; <CJK>
                 (#x3C33 . #x6C72)      ; <CJK>
                 (#x3C34 . #x5373)      ; <CJK>
                 (#x3C35 . #x5AC9)      ; <CJK>
                 (#x3C36 . #x7EA7)      ; <CJK>
                 (#x3C37 . #x6324)      ; <CJK>
                 (#x3C38 . #x51E0)      ; <CJK>
                 (#x3C39 . #x810A)      ; <CJK>
                 (#x3C3A . #x5DF1)      ; <CJK>
                 (#x3C3B . #x84DF)      ; <CJK>
                 (#x3C3C . #x6280)      ; <CJK>
                 (#x3C3D . #x5180)      ; <CJK>
                 (#x3C3E . #x5B63)      ; <CJK>
                 (#x3C3F . #x4F0E)      ; <CJK>
                 (#x3C40 . #x796D)      ; <CJK>
                 (#x3C41 . #x5242)      ; <CJK>
                 (#x3C42 . #x60B8)      ; <CJK>
                 (#x3C43 . #x6D4E)      ; <CJK>
                 (#x3C44 . #x5BC4)      ; <CJK>
                 (#x3C45 . #x5BC2)      ; <CJK>
                 (#x3C46 . #x8BA1)      ; <CJK>
                 (#x3C47 . #x8BB0)      ; <CJK>
                 (#x3C48 . #x65E2)      ; <CJK>
                 (#x3C49 . #x5FCC)      ; <CJK>
                 (#x3C4A . #x9645)      ; <CJK>
                 (#x3C4B . #x5993)      ; <CJK>
                 (#x3C4C . #x7EE7)      ; <CJK>
                 (#x3C4D . #x7EAA)      ; <CJK>
                 (#x3C4E . #x5609)      ; <CJK>
                 (#x3C4F . #x67B7)      ; <CJK>
                 (#x3C50 . #x5939)      ; <CJK>
                 (#x3C51 . #x4F73)      ; <CJK>
                 (#x3C52 . #x5BB6)      ; <CJK>
                 (#x3C53 . #x52A0)      ; <CJK>
                 (#x3C54 . #x835A)      ; <CJK>
                 (#x3C55 . #x988A)      ; <CJK>
                 (#x3C56 . #x8D3E)      ; <CJK>
                 (#x3C57 . #x7532)      ; <CJK>
                 (#x3C58 . #x94BE)      ; <CJK>
                 (#x3C59 . #x5047)      ; <CJK>
                 (#x3C5A . #x7A3C)      ; <CJK>
                 (#x3C5B . #x4EF7)      ; <CJK>
                 (#x3C5C . #x67B6)      ; <CJK>
                 (#x3C5D . #x9A7E)      ; <CJK>
                 (#x3C5E . #x5AC1)      ; <CJK>
                 (#x3C5F . #x6B7C)      ; <CJK>
                 (#x3C60 . #x76D1)      ; <CJK>
                 (#x3C61 . #x575A)      ; <CJK>
                 (#x3C62 . #x5C16)      ; <CJK>
                 (#x3C63 . #x7B3A)      ; <CJK>
                 (#x3C64 . #x95F4)      ; <CJK>
                 (#x3C65 . #x714E)      ; <CJK>
                 (#x3C66 . #x517C)      ; <CJK>
                 (#x3C67 . #x80A9)      ; <CJK>
                 (#x3C68 . #x8270)      ; <CJK>
                 (#x3C69 . #x5978)      ; <CJK>
                 (#x3C6A . #x7F04)      ; <CJK>
                 (#x3C6B . #x8327)      ; <CJK>
                 (#x3C6C . #x68C0)      ; <CJK>
                 (#x3C6D . #x67EC)      ; <CJK>
                 (#x3C6E . #x78B1)      ; <CJK>
                 (#x3C6F . #x7877)      ; <CJK>
                 (#x3C70 . #x62E3)      ; <CJK>
                 (#x3C71 . #x6361)      ; <CJK>
                 (#x3C72 . #x7B80)      ; <CJK>
                 (#x3C73 . #x4FED)      ; <CJK>
                 (#x3C74 . #x526A)      ; <CJK>
                 (#x3C75 . #x51CF)      ; <CJK>
                 (#x3C76 . #x8350)      ; <CJK>
                 (#x3C77 . #x69DB)      ; <CJK>
                 (#x3C78 . #x9274)      ; <CJK>
                 (#x3C79 . #x8DF5)      ; <CJK>
                 (#x3C7A . #x8D31)      ; <CJK>
                 (#x3C7B . #x89C1)      ; <CJK>
                 (#x3C7C . #x952E)      ; <CJK>
                 (#x3C7D . #x7BAD)      ; <CJK>
                 (#x3C7E . #x4EF6)      ; <CJK>
                 (#x3D21 . #x5065)      ; <CJK>
                 (#x3D22 . #x8230)      ; <CJK>
                 (#x3D23 . #x5251)      ; <CJK>
                 (#x3D24 . #x996F)      ; <CJK>
                 (#x3D25 . #x6E10)      ; <CJK>
                 (#x3D26 . #x6E85)      ; <CJK>
                 (#x3D27 . #x6DA7)      ; <CJK>
                 (#x3D28 . #x5EFA)      ; <CJK>
                 (#x3D29 . #x50F5)      ; <CJK>
                 (#x3D2A . #x59DC)      ; <CJK>
                 (#x3D2B . #x5C06)      ; <CJK>
                 (#x3D2C . #x6D46)      ; <CJK>
                 (#x3D2D . #x6C5F)      ; <CJK>
                 (#x3D2E . #x7586)      ; <CJK>
                 (#x3D2F . #x848B)      ; <CJK>
                 (#x3D30 . #x6868)      ; <CJK>
                 (#x3D31 . #x5956)      ; <CJK>
                 (#x3D32 . #x8BB2)      ; <CJK>
                 (#x3D33 . #x5320)      ; <CJK>
                 (#x3D34 . #x9171)      ; <CJK>
                 (#x3D35 . #x964D)      ; <CJK>
                 (#x3D36 . #x8549)      ; <CJK>
                 (#x3D37 . #x6912)      ; <CJK>
                 (#x3D38 . #x7901)      ; <CJK>
                 (#x3D39 . #x7126)      ; <CJK>
                 (#x3D3A . #x80F6)      ; <CJK>
                 (#x3D3B . #x4EA4)      ; <CJK>
                 (#x3D3C . #x90CA)      ; <CJK>
                 (#x3D3D . #x6D47)      ; <CJK>
                 (#x3D3E . #x9A84)      ; <CJK>
                 (#x3D3F . #x5A07)      ; <CJK>
                 (#x3D40 . #x56BC)      ; <CJK>
                 (#x3D41 . #x6405)      ; <CJK>
                 (#x3D42 . #x94F0)      ; <CJK>
                 (#x3D43 . #x77EB)      ; <CJK>
                 (#x3D44 . #x4FA5)      ; <CJK>
                 (#x3D45 . #x811A)      ; <CJK>
                 (#x3D46 . #x72E1)      ; <CJK>
                 (#x3D47 . #x89D2)      ; <CJK>
                 (#x3D48 . #x997A)      ; <CJK>
                 (#x3D49 . #x7F34)      ; <CJK>
                 (#x3D4A . #x7EDE)      ; <CJK>
                 (#x3D4B . #x527F)      ; <CJK>
                 (#x3D4C . #x6559)      ; <CJK>
                 (#x3D4D . #x9175)      ; <CJK>
                 (#x3D4E . #x8F7F)      ; <CJK>
                 (#x3D4F . #x8F83)      ; <CJK>
                 (#x3D50 . #x53EB)      ; <CJK>
                 (#x3D51 . #x7A96)      ; <CJK>
                 (#x3D52 . #x63ED)      ; <CJK>
                 (#x3D53 . #x63A5)      ; <CJK>
                 (#x3D54 . #x7686)      ; <CJK>
                 (#x3D55 . #x79F8)      ; <CJK>
                 (#x3D56 . #x8857)      ; <CJK>
                 (#x3D57 . #x9636)      ; <CJK>
                 (#x3D58 . #x622A)      ; <CJK>
                 (#x3D59 . #x52AB)      ; <CJK>
                 (#x3D5A . #x8282)      ; <CJK>
                 (#x3D5B . #x6854)      ; <CJK>
                 (#x3D5C . #x6770)      ; <CJK>
                 (#x3D5D . #x6377)      ; <CJK>
                 (#x3D5E . #x776B)      ; <CJK>
                 (#x3D5F . #x7AED)      ; <CJK>
                 (#x3D60 . #x6D01)      ; <CJK>
                 (#x3D61 . #x7ED3)      ; <CJK>
                 (#x3D62 . #x89E3)      ; <CJK>
                 (#x3D63 . #x59D0)      ; <CJK>
                 (#x3D64 . #x6212)      ; <CJK>
                 (#x3D65 . #x85C9)      ; <CJK>
                 (#x3D66 . #x82A5)      ; <CJK>
                 (#x3D67 . #x754C)      ; <CJK>
                 (#x3D68 . #x501F)      ; <CJK>
                 (#x3D69 . #x4ECB)      ; <CJK>
                 (#x3D6A . #x75A5)      ; <CJK>
                 (#x3D6B . #x8BEB)      ; <CJK>
                 (#x3D6C . #x5C4A)      ; <CJK>
                 (#x3D6D . #x5DFE)      ; <CJK>
                 (#x3D6E . #x7B4B)      ; <CJK>
                 (#x3D6F . #x65A4)      ; <CJK>
                 (#x3D70 . #x91D1)      ; <CJK>
                 (#x3D71 . #x4ECA)      ; <CJK>
                 (#x3D72 . #x6D25)      ; <CJK>
                 (#x3D73 . #x895F)      ; <CJK>
                 (#x3D74 . #x7D27)      ; <CJK>
                 (#x3D75 . #x9526)      ; <CJK>
                 (#x3D76 . #x4EC5)      ; <CJK>
                 (#x3D77 . #x8C28)      ; <CJK>
                 (#x3D78 . #x8FDB)      ; <CJK>
                 (#x3D79 . #x9773)      ; <CJK>
                 (#x3D7A . #x664B)      ; <CJK>
                 (#x3D7B . #x7981)      ; <CJK>
                 (#x3D7C . #x8FD1)      ; <CJK>
                 (#x3D7D . #x70EC)      ; <CJK>
                 (#x3D7E . #x6D78)      ; <CJK>
                 (#x3E21 . #x5C3D)      ; <CJK>
                 (#x3E22 . #x52B2)      ; <CJK>
                 (#x3E23 . #x8346)      ; <CJK>
                 (#x3E24 . #x5162)      ; <CJK>
                 (#x3E25 . #x830E)      ; <CJK>
                 (#x3E26 . #x775B)      ; <CJK>
                 (#x3E27 . #x6676)      ; <CJK>
                 (#x3E28 . #x9CB8)      ; <CJK>
                 (#x3E29 . #x4EAC)      ; <CJK>
                 (#x3E2A . #x60CA)      ; <CJK>
                 (#x3E2B . #x7CBE)      ; <CJK>
                 (#x3E2C . #x7CB3)      ; <CJK>
                 (#x3E2D . #x7ECF)      ; <CJK>
                 (#x3E2E . #x4E95)      ; <CJK>
                 (#x3E2F . #x8B66)      ; <CJK>
                 (#x3E30 . #x666F)      ; <CJK>
                 (#x3E31 . #x9888)      ; <CJK>
                 (#x3E32 . #x9759)      ; <CJK>
                 (#x3E33 . #x5883)      ; <CJK>
                 (#x3E34 . #x656C)      ; <CJK>
                 (#x3E35 . #x955C)      ; <CJK>
                 (#x3E36 . #x5F84)      ; <CJK>
                 (#x3E37 . #x75C9)      ; <CJK>
                 (#x3E38 . #x9756)      ; <CJK>
                 (#x3E39 . #x7ADF)      ; <CJK>
                 (#x3E3A . #x7ADE)      ; <CJK>
                 (#x3E3B . #x51C0)      ; <CJK>
                 (#x3E3C . #x70AF)      ; <CJK>
                 (#x3E3D . #x7A98)      ; <CJK>
                 (#x3E3E . #x63EA)      ; <CJK>
                 (#x3E3F . #x7A76)      ; <CJK>
                 (#x3E40 . #x7EA0)      ; <CJK>
                 (#x3E41 . #x7396)      ; <CJK>
                 (#x3E42 . #x97ED)      ; <CJK>
                 (#x3E43 . #x4E45)      ; <CJK>
                 (#x3E44 . #x7078)      ; <CJK>
                 (#x3E45 . #x4E5D)      ; <CJK>
                 (#x3E46 . #x9152)      ; <CJK>
                 (#x3E47 . #x53A9)      ; <CJK>
                 (#x3E48 . #x6551)      ; <CJK>
                 (#x3E49 . #x65E7)      ; <CJK>
                 (#x3E4A . #x81FC)      ; <CJK>
                 (#x3E4B . #x8205)      ; <CJK>
                 (#x3E4C . #x548E)      ; <CJK>
                 (#x3E4D . #x5C31)      ; <CJK>
                 (#x3E4E . #x759A)      ; <CJK>
                 (#x3E4F . #x97A0)      ; <CJK>
                 (#x3E50 . #x62D8)      ; <CJK>
                 (#x3E51 . #x72D9)      ; <CJK>
                 (#x3E52 . #x75BD)      ; <CJK>
                 (#x3E53 . #x5C45)      ; <CJK>
                 (#x3E54 . #x9A79)      ; <CJK>
                 (#x3E55 . #x83CA)      ; <CJK>
                 (#x3E56 . #x5C40)      ; <CJK>
                 (#x3E57 . #x5480)      ; <CJK>
                 (#x3E58 . #x77E9)      ; <CJK>
                 (#x3E59 . #x4E3E)      ; <CJK>
                 (#x3E5A . #x6CAE)      ; <CJK>
                 (#x3E5B . #x805A)      ; <CJK>
                 (#x3E5C . #x62D2)      ; <CJK>
                 (#x3E5D . #x636E)      ; <CJK>
                 (#x3E5E . #x5DE8)      ; <CJK>
                 (#x3E5F . #x5177)      ; <CJK>
                 (#x3E60 . #x8DDD)      ; <CJK>
                 (#x3E61 . #x8E1E)      ; <CJK>
                 (#x3E62 . #x952F)      ; <CJK>
                 (#x3E63 . #x4FF1)      ; <CJK>
                 (#x3E64 . #x53E5)      ; <CJK>
                 (#x3E65 . #x60E7)      ; <CJK>
                 (#x3E66 . #x70AC)      ; <CJK>
                 (#x3E67 . #x5267)      ; <CJK>
                 (#x3E68 . #x6350)      ; <CJK>
                 (#x3E69 . #x9E43)      ; <CJK>
                 (#x3E6A . #x5A1F)      ; <CJK>
                 (#x3E6B . #x5026)      ; <CJK>
                 (#x3E6C . #x7737)      ; <CJK>
                 (#x3E6D . #x5377)      ; <CJK>
                 (#x3E6E . #x7EE2)      ; <CJK>
                 (#x3E6F . #x6485)      ; <CJK>
                 (#x3E70 . #x652B)      ; <CJK>
                 (#x3E71 . #x6289)      ; <CJK>
                 (#x3E72 . #x6398)      ; <CJK>
                 (#x3E73 . #x5014)      ; <CJK>
                 (#x3E74 . #x7235)      ; <CJK>
                 (#x3E75 . #x89C9)      ; <CJK>
                 (#x3E76 . #x51B3)      ; <CJK>
                 (#x3E77 . #x8BC0)      ; <CJK>
                 (#x3E78 . #x7EDD)      ; <CJK>
                 (#x3E79 . #x5747)      ; <CJK>
                 (#x3E7A . #x83CC)      ; <CJK>
                 (#x3E7B . #x94A7)      ; <CJK>
                 (#x3E7C . #x519B)      ; <CJK>
                 (#x3E7D . #x541B)      ; <CJK>
                 (#x3E7E . #x5CFB)      ; <CJK>
                 (#x3F21 . #x4FCA)      ; <CJK>
                 (#x3F22 . #x7AE3)      ; <CJK>
                 (#x3F23 . #x6D5A)      ; <CJK>
                 (#x3F24 . #x90E1)      ; <CJK>
                 (#x3F25 . #x9A8F)      ; <CJK>
                 (#x3F26 . #x5580)      ; <CJK>
                 (#x3F27 . #x5496)      ; <CJK>
                 (#x3F28 . #x5361)      ; <CJK>
                 (#x3F29 . #x54AF)      ; <CJK>
                 (#x3F2A . #x5F00)      ; <CJK>
                 (#x3F2B . #x63E9)      ; <CJK>
                 (#x3F2C . #x6977)      ; <CJK>
                 (#x3F2D . #x51EF)      ; <CJK>
                 (#x3F2E . #x6168)      ; <CJK>
                 (#x3F2F . #x520A)      ; <CJK>
                 (#x3F30 . #x582A)      ; <CJK>
                 (#x3F31 . #x52D8)      ; <CJK>
                 (#x3F32 . #x574E)      ; <CJK>
                 (#x3F33 . #x780D)      ; <CJK>
                 (#x3F34 . #x770B)      ; <CJK>
                 (#x3F35 . #x5EB7)      ; <CJK>
                 (#x3F36 . #x6177)      ; <CJK>
                 (#x3F37 . #x7CE0)      ; <CJK>
                 (#x3F38 . #x625B)      ; <CJK>
                 (#x3F39 . #x6297)      ; <CJK>
                 (#x3F3A . #x4EA2)      ; <CJK>
                 (#x3F3B . #x7095)      ; <CJK>
                 (#x3F3C . #x8003)      ; <CJK>
                 (#x3F3D . #x62F7)      ; <CJK>
                 (#x3F3E . #x70E4)      ; <CJK>
                 (#x3F3F . #x9760)      ; <CJK>
                 (#x3F40 . #x5777)      ; <CJK>
                 (#x3F41 . #x82DB)      ; <CJK>
                 (#x3F42 . #x67EF)      ; <CJK>
                 (#x3F43 . #x68F5)      ; <CJK>
                 (#x3F44 . #x78D5)      ; <CJK>
                 (#x3F45 . #x9897)      ; <CJK>
                 (#x3F46 . #x79D1)      ; <CJK>
                 (#x3F47 . #x58F3)      ; <CJK>
                 (#x3F48 . #x54B3)      ; <CJK>
                 (#x3F49 . #x53EF)      ; <CJK>
                 (#x3F4A . #x6E34)      ; <CJK>
                 (#x3F4B . #x514B)      ; <CJK>
                 (#x3F4C . #x523B)      ; <CJK>
                 (#x3F4D . #x5BA2)      ; <CJK>
                 (#x3F4E . #x8BFE)      ; <CJK>
                 (#x3F4F . #x80AF)      ; <CJK>
                 (#x3F50 . #x5543)      ; <CJK>
                 (#x3F51 . #x57A6)      ; <CJK>
                 (#x3F52 . #x6073)      ; <CJK>
                 (#x3F53 . #x5751)      ; <CJK>
                 (#x3F54 . #x542D)      ; <CJK>
                 (#x3F55 . #x7A7A)      ; <CJK>
                 (#x3F56 . #x6050)      ; <CJK>
                 (#x3F57 . #x5B54)      ; <CJK>
                 (#x3F58 . #x63A7)      ; <CJK>
                 (#x3F59 . #x62A0)      ; <CJK>
                 (#x3F5A . #x53E3)      ; <CJK>
                 (#x3F5B . #x6263)      ; <CJK>
                 (#x3F5C . #x5BC7)      ; <CJK>
                 (#x3F5D . #x67AF)      ; <CJK>
                 (#x3F5E . #x54ED)      ; <CJK>
                 (#x3F5F . #x7A9F)      ; <CJK>
                 (#x3F60 . #x82E6)      ; <CJK>
                 (#x3F61 . #x9177)      ; <CJK>
                 (#x3F62 . #x5E93)      ; <CJK>
                 (#x3F63 . #x88E4)      ; <CJK>
                 (#x3F64 . #x5938)      ; <CJK>
                 (#x3F65 . #x57AE)      ; <CJK>
                 (#x3F66 . #x630E)      ; <CJK>
                 (#x3F67 . #x8DE8)      ; <CJK>
                 (#x3F68 . #x80EF)      ; <CJK>
                 (#x3F69 . #x5757)      ; <CJK>
                 (#x3F6A . #x7B77)      ; <CJK>
                 (#x3F6B . #x4FA9)      ; <CJK>
                 (#x3F6C . #x5FEB)      ; <CJK>
                 (#x3F6D . #x5BBD)      ; <CJK>
                 (#x3F6E . #x6B3E)      ; <CJK>
                 (#x3F6F . #x5321)      ; <CJK>
                 (#x3F70 . #x7B50)      ; <CJK>
                 (#x3F71 . #x72C2)      ; <CJK>
                 (#x3F72 . #x6846)      ; <CJK>
                 (#x3F73 . #x77FF)      ; <CJK>
                 (#x3F74 . #x7736)      ; <CJK>
                 (#x3F75 . #x65F7)      ; <CJK>
                 (#x3F76 . #x51B5)      ; <CJK>
                 (#x3F77 . #x4E8F)      ; <CJK>
                 (#x3F78 . #x76D4)      ; <CJK>
                 (#x3F79 . #x5CBF)      ; <CJK>
                 (#x3F7A . #x7AA5)      ; <CJK>
                 (#x3F7B . #x8475)      ; <CJK>
                 (#x3F7C . #x594E)      ; <CJK>
                 (#x3F7D . #x9B41)      ; <CJK>
                 (#x3F7E . #x5080)      ; <CJK>
                 (#x4021 . #x9988)      ; <CJK>
                 (#x4022 . #x6127)      ; <CJK>
                 (#x4023 . #x6E83)      ; <CJK>
                 (#x4024 . #x5764)      ; <CJK>
                 (#x4025 . #x6606)      ; <CJK>
                 (#x4026 . #x6346)      ; <CJK>
                 (#x4027 . #x56F0)      ; <CJK>
                 (#x4028 . #x62EC)      ; <CJK>
                 (#x4029 . #x6269)      ; <CJK>
                 (#x402A . #x5ED3)      ; <CJK>
                 (#x402B . #x9614)      ; <CJK>
                 (#x402C . #x5783)      ; <CJK>
                 (#x402D . #x62C9)      ; <CJK>
                 (#x402E . #x5587)      ; <CJK>
                 (#x402F . #x8721)      ; <CJK>
                 (#x4030 . #x814A)      ; <CJK>
                 (#x4031 . #x8FA3)      ; <CJK>
                 (#x4032 . #x5566)      ; <CJK>
                 (#x4033 . #x83B1)      ; <CJK>
                 (#x4034 . #x6765)      ; <CJK>
                 (#x4035 . #x8D56)      ; <CJK>
                 (#x4036 . #x84DD)      ; <CJK>
                 (#x4037 . #x5A6A)      ; <CJK>
                 (#x4038 . #x680F)      ; <CJK>
                 (#x4039 . #x62E6)      ; <CJK>
                 (#x403A . #x7BEE)      ; <CJK>
                 (#x403B . #x9611)      ; <CJK>
                 (#x403C . #x5170)      ; <CJK>
                 (#x403D . #x6F9C)      ; <CJK>
                 (#x403E . #x8C30)      ; <CJK>
                 (#x403F . #x63FD)      ; <CJK>
                 (#x4040 . #x89C8)      ; <CJK>
                 (#x4041 . #x61D2)      ; <CJK>
                 (#x4042 . #x7F06)      ; <CJK>
                 (#x4043 . #x70C2)      ; <CJK>
                 (#x4044 . #x6EE5)      ; <CJK>
                 (#x4045 . #x7405)      ; <CJK>
                 (#x4046 . #x6994)      ; <CJK>
                 (#x4047 . #x72FC)      ; <CJK>
                 (#x4048 . #x5ECA)      ; <CJK>
                 (#x4049 . #x90CE)      ; <CJK>
                 (#x404A . #x6717)      ; <CJK>
                 (#x404B . #x6D6A)      ; <CJK>
                 (#x404C . #x635E)      ; <CJK>
                 (#x404D . #x52B3)      ; <CJK>
                 (#x404E . #x7262)      ; <CJK>
                 (#x404F . #x8001)      ; <CJK>
                 (#x4050 . #x4F6C)      ; <CJK>
                 (#x4051 . #x59E5)      ; <CJK>
                 (#x4052 . #x916A)      ; <CJK>
                 (#x4053 . #x70D9)      ; <CJK>
                 (#x4054 . #x6D9D)      ; <CJK>
                 (#x4055 . #x52D2)      ; <CJK>
                 (#x4056 . #x4E50)      ; <CJK>
                 (#x4057 . #x96F7)      ; <CJK>
                 (#x4058 . #x956D)      ; <CJK>
                 (#x4059 . #x857E)      ; <CJK>
                 (#x405A . #x78CA)      ; <CJK>
                 (#x405B . #x7D2F)      ; <CJK>
                 (#x405C . #x5121)      ; <CJK>
                 (#x405D . #x5792)      ; <CJK>
                 (#x405E . #x64C2)      ; <CJK>
                 (#x405F . #x808B)      ; <CJK>
                 (#x4060 . #x7C7B)      ; <CJK>
                 (#x4061 . #x6CEA)      ; <CJK>
                 (#x4062 . #x68F1)      ; <CJK>
                 (#x4063 . #x695E)      ; <CJK>
                 (#x4064 . #x51B7)      ; <CJK>
                 (#x4065 . #x5398)      ; <CJK>
                 (#x4066 . #x68A8)      ; <CJK>
                 (#x4067 . #x7281)      ; <CJK>
                 (#x4068 . #x9ECE)      ; <CJK>
                 (#x4069 . #x7BF1)      ; <CJK>
                 (#x406A . #x72F8)      ; <CJK>
                 (#x406B . #x79BB)      ; <CJK>
                 (#x406C . #x6F13)      ; <CJK>
                 (#x406D . #x7406)      ; <CJK>
                 (#x406E . #x674E)      ; <CJK>
                 (#x406F . #x91CC)      ; <CJK>
                 (#x4070 . #x9CA4)      ; <CJK>
                 (#x4071 . #x793C)      ; <CJK>
                 (#x4072 . #x8389)      ; <CJK>
                 (#x4073 . #x8354)      ; <CJK>
                 (#x4074 . #x540F)      ; <CJK>
                 (#x4075 . #x6817)      ; <CJK>
                 (#x4076 . #x4E3D)      ; <CJK>
                 (#x4077 . #x5389)      ; <CJK>
                 (#x4078 . #x52B1)      ; <CJK>
                 (#x4079 . #x783E)      ; <CJK>
                 (#x407A . #x5386)      ; <CJK>
                 (#x407B . #x5229)      ; <CJK>
                 (#x407C . #x5088)      ; <CJK>
                 (#x407D . #x4F8B)      ; <CJK>
                 (#x407E . #x4FD0)      ; <CJK>
                 (#x4121 . #x75E2)      ; <CJK>
                 (#x4122 . #x7ACB)      ; <CJK>
                 (#x4123 . #x7C92)      ; <CJK>
                 (#x4124 . #x6CA5)      ; <CJK>
                 (#x4125 . #x96B6)      ; <CJK>
                 (#x4126 . #x529B)      ; <CJK>
                 (#x4127 . #x7483)      ; <CJK>
                 (#x4128 . #x54E9)      ; <CJK>
                 (#x4129 . #x4FE9)      ; <CJK>
                 (#x412A . #x8054)      ; <CJK>
                 (#x412B . #x83B2)      ; <CJK>
                 (#x412C . #x8FDE)      ; <CJK>
                 (#x412D . #x9570)      ; <CJK>
                 (#x412E . #x5EC9)      ; <CJK>
                 (#x412F . #x601C)      ; <CJK>
                 (#x4130 . #x6D9F)      ; <CJK>
                 (#x4131 . #x5E18)      ; <CJK>
                 (#x4132 . #x655B)      ; <CJK>
                 (#x4133 . #x8138)      ; <CJK>
                 (#x4134 . #x94FE)      ; <CJK>
                 (#x4135 . #x604B)      ; <CJK>
                 (#x4136 . #x70BC)      ; <CJK>
                 (#x4137 . #x7EC3)      ; <CJK>
                 (#x4138 . #x7CAE)      ; <CJK>
                 (#x4139 . #x51C9)      ; <CJK>
                 (#x413A . #x6881)      ; <CJK>
                 (#x413B . #x7CB1)      ; <CJK>
                 (#x413C . #x826F)      ; <CJK>
                 (#x413D . #x4E24)      ; <CJK>
                 (#x413E . #x8F86)      ; <CJK>
                 (#x413F . #x91CF)      ; <CJK>
                 (#x4140 . #x667E)      ; <CJK>
                 (#x4141 . #x4EAE)      ; <CJK>
                 (#x4142 . #x8C05)      ; <CJK>
                 (#x4143 . #x64A9)      ; <CJK>
                 (#x4144 . #x804A)      ; <CJK>
                 (#x4145 . #x50DA)      ; <CJK>
                 (#x4146 . #x7597)      ; <CJK>
                 (#x4147 . #x71CE)      ; <CJK>
                 (#x4148 . #x5BE5)      ; <CJK>
                 (#x4149 . #x8FBD)      ; <CJK>
                 (#x414A . #x6F66)      ; <CJK>
                 (#x414B . #x4E86)      ; <CJK>
                 (#x414C . #x6482)      ; <CJK>
                 (#x414D . #x9563)      ; <CJK>
                 (#x414E . #x5ED6)      ; <CJK>
                 (#x414F . #x6599)      ; <CJK>
                 (#x4150 . #x5217)      ; <CJK>
                 (#x4151 . #x88C2)      ; <CJK>
                 (#x4152 . #x70C8)      ; <CJK>
                 (#x4153 . #x52A3)      ; <CJK>
                 (#x4154 . #x730E)      ; <CJK>
                 (#x4155 . #x7433)      ; <CJK>
                 (#x4156 . #x6797)      ; <CJK>
                 (#x4157 . #x78F7)      ; <CJK>
                 (#x4158 . #x9716)      ; <CJK>
                 (#x4159 . #x4E34)      ; <CJK>
                 (#x415A . #x90BB)      ; <CJK>
                 (#x415B . #x9CDE)      ; <CJK>
                 (#x415C . #x6DCB)      ; <CJK>
                 (#x415D . #x51DB)      ; <CJK>
                 (#x415E . #x8D41)      ; <CJK>
                 (#x415F . #x541D)      ; <CJK>
                 (#x4160 . #x62CE)      ; <CJK>
                 (#x4161 . #x73B2)      ; <CJK>
                 (#x4162 . #x83F1)      ; <CJK>
                 (#x4163 . #x96F6)      ; <CJK>
                 (#x4164 . #x9F84)      ; <CJK>
                 (#x4165 . #x94C3)      ; <CJK>
                 (#x4166 . #x4F36)      ; <CJK>
                 (#x4167 . #x7F9A)      ; <CJK>
                 (#x4168 . #x51CC)      ; <CJK>
                 (#x4169 . #x7075)      ; <CJK>
                 (#x416A . #x9675)      ; <CJK>
                 (#x416B . #x5CAD)      ; <CJK>
                 (#x416C . #x9886)      ; <CJK>
                 (#x416D . #x53E6)      ; <CJK>
                 (#x416E . #x4EE4)      ; <CJK>
                 (#x416F . #x6E9C)      ; <CJK>
                 (#x4170 . #x7409)      ; <CJK>
                 (#x4171 . #x69B4)      ; <CJK>
                 (#x4172 . #x786B)      ; <CJK>
                 (#x4173 . #x998F)      ; <CJK>
                 (#x4174 . #x7559)      ; <CJK>
                 (#x4175 . #x5218)      ; <CJK>
                 (#x4176 . #x7624)      ; <CJK>
                 (#x4177 . #x6D41)      ; <CJK>
                 (#x4178 . #x67F3)      ; <CJK>
                 (#x4179 . #x516D)      ; <CJK>
                 (#x417A . #x9F99)      ; <CJK>
                 (#x417B . #x804B)      ; <CJK>
                 (#x417C . #x5499)      ; <CJK>
                 (#x417D . #x7B3C)      ; <CJK>
                 (#x417E . #x7ABF)      ; <CJK>
                 (#x4221 . #x9686)      ; <CJK>
                 (#x4222 . #x5784)      ; <CJK>
                 (#x4223 . #x62E2)      ; <CJK>
                 (#x4224 . #x9647)      ; <CJK>
                 (#x4225 . #x697C)      ; <CJK>
                 (#x4226 . #x5A04)      ; <CJK>
                 (#x4227 . #x6402)      ; <CJK>
                 (#x4228 . #x7BD3)      ; <CJK>
                 (#x4229 . #x6F0F)      ; <CJK>
                 (#x422A . #x964B)      ; <CJK>
                 (#x422B . #x82A6)      ; <CJK>
                 (#x422C . #x5362)      ; <CJK>
                 (#x422D . #x9885)      ; <CJK>
                 (#x422E . #x5E90)      ; <CJK>
                 (#x422F . #x7089)      ; <CJK>
                 (#x4230 . #x63B3)      ; <CJK>
                 (#x4231 . #x5364)      ; <CJK>
                 (#x4232 . #x864F)      ; <CJK>
                 (#x4233 . #x9C81)      ; <CJK>
                 (#x4234 . #x9E93)      ; <CJK>
                 (#x4235 . #x788C)      ; <CJK>
                 (#x4236 . #x9732)      ; <CJK>
                 (#x4237 . #x8DEF)      ; <CJK>
                 (#x4238 . #x8D42)      ; <CJK>
                 (#x4239 . #x9E7F)      ; <CJK>
                 (#x423A . #x6F5E)      ; <CJK>
                 (#x423B . #x7984)      ; <CJK>
                 (#x423C . #x5F55)      ; <CJK>
                 (#x423D . #x9646)      ; <CJK>
                 (#x423E . #x622E)      ; <CJK>
                 (#x423F . #x9A74)      ; <CJK>
                 (#x4240 . #x5415)      ; <CJK>
                 (#x4241 . #x94DD)      ; <CJK>
                 (#x4242 . #x4FA3)      ; <CJK>
                 (#x4243 . #x65C5)      ; <CJK>
                 (#x4244 . #x5C65)      ; <CJK>
                 (#x4245 . #x5C61)      ; <CJK>
                 (#x4246 . #x7F15)      ; <CJK>
                 (#x4247 . #x8651)      ; <CJK>
                 (#x4248 . #x6C2F)      ; <CJK>
                 (#x4249 . #x5F8B)      ; <CJK>
                 (#x424A . #x7387)      ; <CJK>
                 (#x424B . #x6EE4)      ; <CJK>
                 (#x424C . #x7EFF)      ; <CJK>
                 (#x424D . #x5CE6)      ; <CJK>
                 (#x424E . #x631B)      ; <CJK>
                 (#x424F . #x5B6A)      ; <CJK>
                 (#x4250 . #x6EE6)      ; <CJK>
                 (#x4251 . #x5375)      ; <CJK>
                 (#x4252 . #x4E71)      ; <CJK>
                 (#x4253 . #x63A0)      ; <CJK>
                 (#x4254 . #x7565)      ; <CJK>
                 (#x4255 . #x62A1)      ; <CJK>
                 (#x4256 . #x8F6E)      ; <CJK>
                 (#x4257 . #x4F26)      ; <CJK>
                 (#x4258 . #x4ED1)      ; <CJK>
                 (#x4259 . #x6CA6)      ; <CJK>
                 (#x425A . #x7EB6)      ; <CJK>
                 (#x425B . #x8BBA)      ; <CJK>
                 (#x425C . #x841D)      ; <CJK>
                 (#x425D . #x87BA)      ; <CJK>
                 (#x425E . #x7F57)      ; <CJK>
                 (#x425F . #x903B)      ; <CJK>
                 (#x4260 . #x9523)      ; <CJK>
                 (#x4261 . #x7BA9)      ; <CJK>
                 (#x4262 . #x9AA1)      ; <CJK>
                 (#x4263 . #x88F8)      ; <CJK>
                 (#x4264 . #x843D)      ; <CJK>
                 (#x4265 . #x6D1B)      ; <CJK>
                 (#x4266 . #x9A86)      ; <CJK>
                 (#x4267 . #x7EDC)      ; <CJK>
                 (#x4268 . #x5988)      ; <CJK>
                 (#x4269 . #x9EBB)      ; <CJK>
                 (#x426A . #x739B)      ; <CJK>
                 (#x426B . #x7801)      ; <CJK>
                 (#x426C . #x8682)      ; <CJK>
                 (#x426D . #x9A6C)      ; <CJK>
                 (#x426E . #x9A82)      ; <CJK>
                 (#x426F . #x561B)      ; <CJK>
                 (#x4270 . #x5417)      ; <CJK>
                 (#x4271 . #x57CB)      ; <CJK>
                 (#x4272 . #x4E70)      ; <CJK>
                 (#x4273 . #x9EA6)      ; <CJK>
                 (#x4274 . #x5356)      ; <CJK>
                 (#x4275 . #x8FC8)      ; <CJK>
                 (#x4276 . #x8109)      ; <CJK>
                 (#x4277 . #x7792)      ; <CJK>
                 (#x4278 . #x9992)      ; <CJK>
                 (#x4279 . #x86EE)      ; <CJK>
                 (#x427A . #x6EE1)      ; <CJK>
                 (#x427B . #x8513)      ; <CJK>
                 (#x427C . #x66FC)      ; <CJK>
                 (#x427D . #x6162)      ; <CJK>
                 (#x427E . #x6F2B)      ; <CJK>
                 (#x4321 . #x8C29)      ; <CJK>
                 (#x4322 . #x8292)      ; <CJK>
                 (#x4323 . #x832B)      ; <CJK>
                 (#x4324 . #x76F2)      ; <CJK>
                 (#x4325 . #x6C13)      ; <CJK>
                 (#x4326 . #x5FD9)      ; <CJK>
                 (#x4327 . #x83BD)      ; <CJK>
                 (#x4328 . #x732B)      ; <CJK>
                 (#x4329 . #x8305)      ; <CJK>
                 (#x432A . #x951A)      ; <CJK>
                 (#x432B . #x6BDB)      ; <CJK>
                 (#x432C . #x77DB)      ; <CJK>
                 (#x432D . #x94C6)      ; <CJK>
                 (#x432E . #x536F)      ; <CJK>
                 (#x432F . #x8302)      ; <CJK>
                 (#x4330 . #x5192)      ; <CJK>
                 (#x4331 . #x5E3D)      ; <CJK>
                 (#x4332 . #x8C8C)      ; <CJK>
                 (#x4333 . #x8D38)      ; <CJK>
                 (#x4334 . #x4E48)      ; <CJK>
                 (#x4335 . #x73AB)      ; <CJK>
                 (#x4336 . #x679A)      ; <CJK>
                 (#x4337 . #x6885)      ; <CJK>
                 (#x4338 . #x9176)      ; <CJK>
                 (#x4339 . #x9709)      ; <CJK>
                 (#x433A . #x7164)      ; <CJK>
                 (#x433B . #x6CA1)      ; <CJK>
                 (#x433C . #x7709)      ; <CJK>
                 (#x433D . #x5A92)      ; <CJK>
                 (#x433E . #x9541)      ; <CJK>
                 (#x433F . #x6BCF)      ; <CJK>
                 (#x4340 . #x7F8E)      ; <CJK>
                 (#x4341 . #x6627)      ; <CJK>
                 (#x4342 . #x5BD0)      ; <CJK>
                 (#x4343 . #x59B9)      ; <CJK>
                 (#x4344 . #x5A9A)      ; <CJK>
                 (#x4345 . #x95E8)      ; <CJK>
                 (#x4346 . #x95F7)      ; <CJK>
                 (#x4347 . #x4EEC)      ; <CJK>
                 (#x4348 . #x840C)      ; <CJK>
                 (#x4349 . #x8499)      ; <CJK>
                 (#x434A . #x6AAC)      ; <CJK>
                 (#x434B . #x76DF)      ; <CJK>
                 (#x434C . #x9530)      ; <CJK>
                 (#x434D . #x731B)      ; <CJK>
                 (#x434E . #x68A6)      ; <CJK>
                 (#x434F . #x5B5F)      ; <CJK>
                 (#x4350 . #x772F)      ; <CJK>
                 (#x4351 . #x919A)      ; <CJK>
                 (#x4352 . #x9761)      ; <CJK>
                 (#x4353 . #x7CDC)      ; <CJK>
                 (#x4354 . #x8FF7)      ; <CJK>
                 (#x4355 . #x8C1C)      ; <CJK>
                 (#x4356 . #x5F25)      ; <CJK>
                 (#x4357 . #x7C73)      ; <CJK>
                 (#x4358 . #x79D8)      ; <CJK>
                 (#x4359 . #x89C5)      ; <CJK>
                 (#x435A . #x6CCC)      ; <CJK>
                 (#x435B . #x871C)      ; <CJK>
                 (#x435C . #x5BC6)      ; <CJK>
                 (#x435D . #x5E42)      ; <CJK>
                 (#x435E . #x68C9)      ; <CJK>
                 (#x435F . #x7720)      ; <CJK>
                 (#x4360 . #x7EF5)      ; <CJK>
                 (#x4361 . #x5195)      ; <CJK>
                 (#x4362 . #x514D)      ; <CJK>
                 (#x4363 . #x52C9)      ; <CJK>
                 (#x4364 . #x5A29)      ; <CJK>
                 (#x4365 . #x7F05)      ; <CJK>
                 (#x4366 . #x9762)      ; <CJK>
                 (#x4367 . #x82D7)      ; <CJK>
                 (#x4368 . #x63CF)      ; <CJK>
                 (#x4369 . #x7784)      ; <CJK>
                 (#x436A . #x85D0)      ; <CJK>
                 (#x436B . #x79D2)      ; <CJK>
                 (#x436C . #x6E3A)      ; <CJK>
                 (#x436D . #x5E99)      ; <CJK>
                 (#x436E . #x5999)      ; <CJK>
                 (#x436F . #x8511)      ; <CJK>
                 (#x4370 . #x706D)      ; <CJK>
                 (#x4371 . #x6C11)      ; <CJK>
                 (#x4372 . #x62BF)      ; <CJK>
                 (#x4373 . #x76BF)      ; <CJK>
                 (#x4374 . #x654F)      ; <CJK>
                 (#x4375 . #x60AF)      ; <CJK>
                 (#x4376 . #x95FD)      ; <CJK>
                 (#x4377 . #x660E)      ; <CJK>
                 (#x4378 . #x879F)      ; <CJK>
                 (#x4379 . #x9E23)      ; <CJK>
                 (#x437A . #x94ED)      ; <CJK>
                 (#x437B . #x540D)      ; <CJK>
                 (#x437C . #x547D)      ; <CJK>
                 (#x437D . #x8C2C)      ; <CJK>
                 (#x437E . #x6478)      ; <CJK>
                 (#x4421 . #x6479)      ; <CJK>
                 (#x4422 . #x8611)      ; <CJK>
                 (#x4423 . #x6A21)      ; <CJK>
                 (#x4424 . #x819C)      ; <CJK>
                 (#x4425 . #x78E8)      ; <CJK>
                 (#x4426 . #x6469)      ; <CJK>
                 (#x4427 . #x9B54)      ; <CJK>
                 (#x4428 . #x62B9)      ; <CJK>
                 (#x4429 . #x672B)      ; <CJK>
                 (#x442A . #x83AB)      ; <CJK>
                 (#x442B . #x58A8)      ; <CJK>
                 (#x442C . #x9ED8)      ; <CJK>
                 (#x442D . #x6CAB)      ; <CJK>
                 (#x442E . #x6F20)      ; <CJK>
                 (#x442F . #x5BDE)      ; <CJK>
                 (#x4430 . #x964C)      ; <CJK>
                 (#x4431 . #x8C0B)      ; <CJK>
                 (#x4432 . #x725F)      ; <CJK>
                 (#x4433 . #x67D0)      ; <CJK>
                 (#x4434 . #x62C7)      ; <CJK>
                 (#x4435 . #x7261)      ; <CJK>
                 (#x4436 . #x4EA9)      ; <CJK>
                 (#x4437 . #x59C6)      ; <CJK>
                 (#x4438 . #x6BCD)      ; <CJK>
                 (#x4439 . #x5893)      ; <CJK>
                 (#x443A . #x66AE)      ; <CJK>
                 (#x443B . #x5E55)      ; <CJK>
                 (#x443C . #x52DF)      ; <CJK>
                 (#x443D . #x6155)      ; <CJK>
                 (#x443E . #x6728)      ; <CJK>
                 (#x443F . #x76EE)      ; <CJK>
                 (#x4440 . #x7766)      ; <CJK>
                 (#x4441 . #x7267)      ; <CJK>
                 (#x4442 . #x7A46)      ; <CJK>
                 (#x4443 . #x62FF)      ; <CJK>
                 (#x4444 . #x54EA)      ; <CJK>
                 (#x4445 . #x5450)      ; <CJK>
                 (#x4446 . #x94A0)      ; <CJK>
                 (#x4447 . #x90A3)      ; <CJK>
                 (#x4448 . #x5A1C)      ; <CJK>
                 (#x4449 . #x7EB3)      ; <CJK>
                 (#x444A . #x6C16)      ; <CJK>
                 (#x444B . #x4E43)      ; <CJK>
                 (#x444C . #x5976)      ; <CJK>
                 (#x444D . #x8010)      ; <CJK>
                 (#x444E . #x5948)      ; <CJK>
                 (#x444F . #x5357)      ; <CJK>
                 (#x4450 . #x7537)      ; <CJK>
                 (#x4451 . #x96BE)      ; <CJK>
                 (#x4452 . #x56CA)      ; <CJK>
                 (#x4453 . #x6320)      ; <CJK>
                 (#x4454 . #x8111)      ; <CJK>
                 (#x4455 . #x607C)      ; <CJK>
                 (#x4456 . #x95F9)      ; <CJK>
                 (#x4457 . #x6DD6)      ; <CJK>
                 (#x4458 . #x5462)      ; <CJK>
                 (#x4459 . #x9981)      ; <CJK>
                 (#x445A . #x5185)      ; <CJK>
                 (#x445B . #x5AE9)      ; <CJK>
                 (#x445C . #x80FD)      ; <CJK>
                 (#x445D . #x59AE)      ; <CJK>
                 (#x445E . #x9713)      ; <CJK>
                 (#x445F . #x502A)      ; <CJK>
                 (#x4460 . #x6CE5)      ; <CJK>
                 (#x4461 . #x5C3C)      ; <CJK>
                 (#x4462 . #x62DF)      ; <CJK>
                 (#x4463 . #x4F60)      ; <CJK>
                 (#x4464 . #x533F)      ; <CJK>
                 (#x4465 . #x817B)      ; <CJK>
                 (#x4466 . #x9006)      ; <CJK>
                 (#x4467 . #x6EBA)      ; <CJK>
                 (#x4468 . #x852B)      ; <CJK>
                 (#x4469 . #x62C8)      ; <CJK>
                 (#x446A . #x5E74)      ; <CJK>
                 (#x446B . #x78BE)      ; <CJK>
                 (#x446C . #x64B5)      ; <CJK>
                 (#x446D . #x637B)      ; <CJK>
                 (#x446E . #x5FF5)      ; <CJK>
                 (#x446F . #x5A18)      ; <CJK>
                 (#x4470 . #x917F)      ; <CJK>
                 (#x4471 . #x9E1F)      ; <CJK>
                 (#x4472 . #x5C3F)      ; <CJK>
                 (#x4473 . #x634F)      ; <CJK>
                 (#x4474 . #x8042)      ; <CJK>
                 (#x4475 . #x5B7D)      ; <CJK>
                 (#x4476 . #x556E)      ; <CJK>
                 (#x4477 . #x954A)      ; <CJK>
                 (#x4478 . #x954D)      ; <CJK>
                 (#x4479 . #x6D85)      ; <CJK>
                 (#x447A . #x60A8)      ; <CJK>
                 (#x447B . #x67E0)      ; <CJK>
                 (#x447C . #x72DE)      ; <CJK>
                 (#x447D . #x51DD)      ; <CJK>
                 (#x447E . #x5B81)      ; <CJK>
                 (#x4521 . #x62E7)      ; <CJK>
                 (#x4522 . #x6CDE)      ; <CJK>
                 (#x4523 . #x725B)      ; <CJK>
                 (#x4524 . #x626D)      ; <CJK>
                 (#x4525 . #x94AE)      ; <CJK>
                 (#x4526 . #x7EBD)      ; <CJK>
                 (#x4527 . #x8113)      ; <CJK>
                 (#x4528 . #x6D53)      ; <CJK>
                 (#x4529 . #x519C)      ; <CJK>
                 (#x452A . #x5F04)      ; <CJK>
                 (#x452B . #x5974)      ; <CJK>
                 (#x452C . #x52AA)      ; <CJK>
                 (#x452D . #x6012)      ; <CJK>
                 (#x452E . #x5973)      ; <CJK>
                 (#x452F . #x6696)      ; <CJK>
                 (#x4530 . #x8650)      ; <CJK>
                 (#x4531 . #x759F)      ; <CJK>
                 (#x4532 . #x632A)      ; <CJK>
                 (#x4533 . #x61E6)      ; <CJK>
                 (#x4534 . #x7CEF)      ; <CJK>
                 (#x4535 . #x8BFA)      ; <CJK>
                 (#x4536 . #x54E6)      ; <CJK>
                 (#x4537 . #x6B27)      ; <CJK>
                 (#x4538 . #x9E25)      ; <CJK>
                 (#x4539 . #x6BB4)      ; <CJK>
                 (#x453A . #x85D5)      ; <CJK>
                 (#x453B . #x5455)      ; <CJK>
                 (#x453C . #x5076)      ; <CJK>
                 (#x453D . #x6CA4)      ; <CJK>
                 (#x453E . #x556A)      ; <CJK>
                 (#x453F . #x8DB4)      ; <CJK>
                 (#x4540 . #x722C)      ; <CJK>
                 (#x4541 . #x5E15)      ; <CJK>
                 (#x4542 . #x6015)      ; <CJK>
                 (#x4543 . #x7436)      ; <CJK>
                 (#x4544 . #x62CD)      ; <CJK>
                 (#x4545 . #x6392)      ; <CJK>
                 (#x4546 . #x724C)      ; <CJK>
                 (#x4547 . #x5F98)      ; <CJK>
                 (#x4548 . #x6E43)      ; <CJK>
                 (#x4549 . #x6D3E)      ; <CJK>
                 (#x454A . #x6500)      ; <CJK>
                 (#x454B . #x6F58)      ; <CJK>
                 (#x454C . #x76D8)      ; <CJK>
                 (#x454D . #x78D0)      ; <CJK>
                 (#x454E . #x76FC)      ; <CJK>
                 (#x454F . #x7554)      ; <CJK>
                 (#x4550 . #x5224)      ; <CJK>
                 (#x4551 . #x53DB)      ; <CJK>
                 (#x4552 . #x4E53)      ; <CJK>
                 (#x4553 . #x5E9E)      ; <CJK>
                 (#x4554 . #x65C1)      ; <CJK>
                 (#x4555 . #x802A)      ; <CJK>
                 (#x4556 . #x80D6)      ; <CJK>
                 (#x4557 . #x629B)      ; <CJK>
                 (#x4558 . #x5486)      ; <CJK>
                 (#x4559 . #x5228)      ; <CJK>
                 (#x455A . #x70AE)      ; <CJK>
                 (#x455B . #x888D)      ; <CJK>
                 (#x455C . #x8DD1)      ; <CJK>
                 (#x455D . #x6CE1)      ; <CJK>
                 (#x455E . #x5478)      ; <CJK>
                 (#x455F . #x80DA)      ; <CJK>
                 (#x4560 . #x57F9)      ; <CJK>
                 (#x4561 . #x88F4)      ; <CJK>
                 (#x4562 . #x8D54)      ; <CJK>
                 (#x4563 . #x966A)      ; <CJK>
                 (#x4564 . #x914D)      ; <CJK>
                 (#x4565 . #x4F69)      ; <CJK>
                 (#x4566 . #x6C9B)      ; <CJK>
                 (#x4567 . #x55B7)      ; <CJK>
                 (#x4568 . #x76C6)      ; <CJK>
                 (#x4569 . #x7830)      ; <CJK>
                 (#x456A . #x62A8)      ; <CJK>
                 (#x456B . #x70F9)      ; <CJK>
                 (#x456C . #x6F8E)      ; <CJK>
                 (#x456D . #x5F6D)      ; <CJK>
                 (#x456E . #x84EC)      ; <CJK>
                 (#x456F . #x68DA)      ; <CJK>
                 (#x4570 . #x787C)      ; <CJK>
                 (#x4571 . #x7BF7)      ; <CJK>
                 (#x4572 . #x81A8)      ; <CJK>
                 (#x4573 . #x670B)      ; <CJK>
                 (#x4574 . #x9E4F)      ; <CJK>
                 (#x4575 . #x6367)      ; <CJK>
                 (#x4576 . #x78B0)      ; <CJK>
                 (#x4577 . #x576F)      ; <CJK>
                 (#x4578 . #x7812)      ; <CJK>
                 (#x4579 . #x9739)      ; <CJK>
                 (#x457A . #x6279)      ; <CJK>
                 (#x457B . #x62AB)      ; <CJK>
                 (#x457C . #x5288)      ; <CJK>
                 (#x457D . #x7435)      ; <CJK>
                 (#x457E . #x6BD7)      ; <CJK>
                 (#x4621 . #x5564)      ; <CJK>
                 (#x4622 . #x813E)      ; <CJK>
                 (#x4623 . #x75B2)      ; <CJK>
                 (#x4624 . #x76AE)      ; <CJK>
                 (#x4625 . #x5339)      ; <CJK>
                 (#x4626 . #x75DE)      ; <CJK>
                 (#x4627 . #x50FB)      ; <CJK>
                 (#x4628 . #x5C41)      ; <CJK>
                 (#x4629 . #x8B6C)      ; <CJK>
                 (#x462A . #x7BC7)      ; <CJK>
                 (#x462B . #x504F)      ; <CJK>
                 (#x462C . #x7247)      ; <CJK>
                 (#x462D . #x9A97)      ; <CJK>
                 (#x462E . #x98D8)      ; <CJK>
                 (#x462F . #x6F02)      ; <CJK>
                 (#x4630 . #x74E2)      ; <CJK>
                 (#x4631 . #x7968)      ; <CJK>
                 (#x4632 . #x6487)      ; <CJK>
                 (#x4633 . #x77A5)      ; <CJK>
                 (#x4634 . #x62FC)      ; <CJK>
                 (#x4635 . #x9891)      ; <CJK>
                 (#x4636 . #x8D2B)      ; <CJK>
                 (#x4637 . #x54C1)      ; <CJK>
                 (#x4638 . #x8058)      ; <CJK>
                 (#x4639 . #x4E52)      ; <CJK>
                 (#x463A . #x576A)      ; <CJK>
                 (#x463B . #x82F9)      ; <CJK>
                 (#x463C . #x840D)      ; <CJK>
                 (#x463D . #x5E73)      ; <CJK>
                 (#x463E . #x51ED)      ; <CJK>
                 (#x463F . #x74F6)      ; <CJK>
                 (#x4640 . #x8BC4)      ; <CJK>
                 (#x4641 . #x5C4F)      ; <CJK>
                 (#x4642 . #x5761)      ; <CJK>
                 (#x4643 . #x6CFC)      ; <CJK>
                 (#x4644 . #x9887)      ; <CJK>
                 (#x4645 . #x5A46)      ; <CJK>
                 (#x4646 . #x7834)      ; <CJK>
                 (#x4647 . #x9B44)      ; <CJK>
                 (#x4648 . #x8FEB)      ; <CJK>
                 (#x4649 . #x7C95)      ; <CJK>
                 (#x464A . #x5256)      ; <CJK>
                 (#x464B . #x6251)      ; <CJK>
                 (#x464C . #x94FA)      ; <CJK>
                 (#x464D . #x4EC6)      ; <CJK>
                 (#x464E . #x8386)      ; <CJK>
                 (#x464F . #x8461)      ; <CJK>
                 (#x4650 . #x83E9)      ; <CJK>
                 (#x4651 . #x84B2)      ; <CJK>
                 (#x4652 . #x57D4)      ; <CJK>
                 (#x4653 . #x6734)      ; <CJK>
                 (#x4654 . #x5703)      ; <CJK>
                 (#x4655 . #x666E)      ; <CJK>
                 (#x4656 . #x6D66)      ; <CJK>
                 (#x4657 . #x8C31)      ; <CJK>
                 (#x4658 . #x66DD)      ; <CJK>
                 (#x4659 . #x7011)      ; <CJK>
                 (#x465A . #x671F)      ; <CJK>
                 (#x465B . #x6B3A)      ; <CJK>
                 (#x465C . #x6816)      ; <CJK>
                 (#x465D . #x621A)      ; <CJK>
                 (#x465E . #x59BB)      ; <CJK>
                 (#x465F . #x4E03)      ; <CJK>
                 (#x4660 . #x51C4)      ; <CJK>
                 (#x4661 . #x6F06)      ; <CJK>
                 (#x4662 . #x67D2)      ; <CJK>
                 (#x4663 . #x6C8F)      ; <CJK>
                 (#x4664 . #x5176)      ; <CJK>
                 (#x4665 . #x68CB)      ; <CJK>
                 (#x4666 . #x5947)      ; <CJK>
                 (#x4667 . #x6B67)      ; <CJK>
                 (#x4668 . #x7566)      ; <CJK>
                 (#x4669 . #x5D0E)      ; <CJK>
                 (#x466A . #x8110)      ; <CJK>
                 (#x466B . #x9F50)      ; <CJK>
                 (#x466C . #x65D7)      ; <CJK>
                 (#x466D . #x7948)      ; <CJK>
                 (#x466E . #x7941)      ; <CJK>
                 (#x466F . #x9A91)      ; <CJK>
                 (#x4670 . #x8D77)      ; <CJK>
                 (#x4671 . #x5C82)      ; <CJK>
                 (#x4672 . #x4E5E)      ; <CJK>
                 (#x4673 . #x4F01)      ; <CJK>
                 (#x4674 . #x542F)      ; <CJK>
                 (#x4675 . #x5951)      ; <CJK>
                 (#x4676 . #x780C)      ; <CJK>
                 (#x4677 . #x5668)      ; <CJK>
                 (#x4678 . #x6C14)      ; <CJK>
                 (#x4679 . #x8FC4)      ; <CJK>
                 (#x467A . #x5F03)      ; <CJK>
                 (#x467B . #x6C7D)      ; <CJK>
                 (#x467C . #x6CE3)      ; <CJK>
                 (#x467D . #x8BAB)      ; <CJK>
                 (#x467E . #x6390)      ; <CJK>
                 (#x4721 . #x6070)      ; <CJK>
                 (#x4722 . #x6D3D)      ; <CJK>
                 (#x4723 . #x7275)      ; <CJK>
                 (#x4724 . #x6266)      ; <CJK>
                 (#x4725 . #x948E)      ; <CJK>
                 (#x4726 . #x94C5)      ; <CJK>
                 (#x4727 . #x5343)      ; <CJK>
                 (#x4728 . #x8FC1)      ; <CJK>
                 (#x4729 . #x7B7E)      ; <CJK>
                 (#x472A . #x4EDF)      ; <CJK>
                 (#x472B . #x8C26)      ; <CJK>
                 (#x472C . #x4E7E)      ; <CJK>
                 (#x472D . #x9ED4)      ; <CJK>
                 (#x472E . #x94B1)      ; <CJK>
                 (#x472F . #x94B3)      ; <CJK>
                 (#x4730 . #x524D)      ; <CJK>
                 (#x4731 . #x6F5C)      ; <CJK>
                 (#x4732 . #x9063)      ; <CJK>
                 (#x4733 . #x6D45)      ; <CJK>
                 (#x4734 . #x8C34)      ; <CJK>
                 (#x4735 . #x5811)      ; <CJK>
                 (#x4736 . #x5D4C)      ; <CJK>
                 (#x4737 . #x6B20)      ; <CJK>
                 (#x4738 . #x6B49)      ; <CJK>
                 (#x4739 . #x67AA)      ; <CJK>
                 (#x473A . #x545B)      ; <CJK>
                 (#x473B . #x8154)      ; <CJK>
                 (#x473C . #x7F8C)      ; <CJK>
                 (#x473D . #x5899)      ; <CJK>
                 (#x473E . #x8537)      ; <CJK>
                 (#x473F . #x5F3A)      ; <CJK>
                 (#x4740 . #x62A2)      ; <CJK>
                 (#x4741 . #x6A47)      ; <CJK>
                 (#x4742 . #x9539)      ; <CJK>
                 (#x4743 . #x6572)      ; <CJK>
                 (#x4744 . #x6084)      ; <CJK>
                 (#x4745 . #x6865)      ; <CJK>
                 (#x4746 . #x77A7)      ; <CJK>
                 (#x4747 . #x4E54)      ; <CJK>
                 (#x4748 . #x4FA8)      ; <CJK>
                 (#x4749 . #x5DE7)      ; <CJK>
                 (#x474A . #x9798)      ; <CJK>
                 (#x474B . #x64AC)      ; <CJK>
                 (#x474C . #x7FD8)      ; <CJK>
                 (#x474D . #x5CED)      ; <CJK>
                 (#x474E . #x4FCF)      ; <CJK>
                 (#x474F . #x7A8D)      ; <CJK>
                 (#x4750 . #x5207)      ; <CJK>
                 (#x4751 . #x8304)      ; <CJK>
                 (#x4752 . #x4E14)      ; <CJK>
                 (#x4753 . #x602F)      ; <CJK>
                 (#x4754 . #x7A83)      ; <CJK>
                 (#x4755 . #x94A6)      ; <CJK>
                 (#x4756 . #x4FB5)      ; <CJK>
                 (#x4757 . #x4EB2)      ; <CJK>
                 (#x4758 . #x79E6)      ; <CJK>
                 (#x4759 . #x7434)      ; <CJK>
                 (#x475A . #x52E4)      ; <CJK>
                 (#x475B . #x82B9)      ; <CJK>
                 (#x475C . #x64D2)      ; <CJK>
                 (#x475D . #x79BD)      ; <CJK>
                 (#x475E . #x5BDD)      ; <CJK>
                 (#x475F . #x6C81)      ; <CJK>
                 (#x4760 . #x9752)      ; <CJK>
                 (#x4761 . #x8F7B)      ; <CJK>
                 (#x4762 . #x6C22)      ; <CJK>
                 (#x4763 . #x503E)      ; <CJK>
                 (#x4764 . #x537F)      ; <CJK>
                 (#x4765 . #x6E05)      ; <CJK>
                 (#x4766 . #x64CE)      ; <CJK>
                 (#x4767 . #x6674)      ; <CJK>
                 (#x4768 . #x6C30)      ; <CJK>
                 (#x4769 . #x60C5)      ; <CJK>
                 (#x476A . #x9877)      ; <CJK>
                 (#x476B . #x8BF7)      ; <CJK>
                 (#x476C . #x5E86)      ; <CJK>
                 (#x476D . #x743C)      ; <CJK>
                 (#x476E . #x7A77)      ; <CJK>
                 (#x476F . #x79CB)      ; <CJK>
                 (#x4770 . #x4E18)      ; <CJK>
                 (#x4771 . #x90B1)      ; <CJK>
                 (#x4772 . #x7403)      ; <CJK>
                 (#x4773 . #x6C42)      ; <CJK>
                 (#x4774 . #x56DA)      ; <CJK>
                 (#x4775 . #x914B)      ; <CJK>
                 (#x4776 . #x6CC5)      ; <CJK>
                 (#x4777 . #x8D8B)      ; <CJK>
                 (#x4778 . #x533A)      ; <CJK>
                 (#x4779 . #x86C6)      ; <CJK>
                 (#x477A . #x66F2)      ; <CJK>
                 (#x477B . #x8EAF)      ; <CJK>
                 (#x477C . #x5C48)      ; <CJK>
                 (#x477D . #x9A71)      ; <CJK>
                 (#x477E . #x6E20)      ; <CJK>
                 (#x4821 . #x53D6)      ; <CJK>
                 (#x4822 . #x5A36)      ; <CJK>
                 (#x4823 . #x9F8B)      ; <CJK>
                 (#x4824 . #x8DA3)      ; <CJK>
                 (#x4825 . #x53BB)      ; <CJK>
                 (#x4826 . #x5708)      ; <CJK>
                 (#x4827 . #x98A7)      ; <CJK>
                 (#x4828 . #x6743)      ; <CJK>
                 (#x4829 . #x919B)      ; <CJK>
                 (#x482A . #x6CC9)      ; <CJK>
                 (#x482B . #x5168)      ; <CJK>
                 (#x482C . #x75CA)      ; <CJK>
                 (#x482D . #x62F3)      ; <CJK>
                 (#x482E . #x72AC)      ; <CJK>
                 (#x482F . #x5238)      ; <CJK>
                 (#x4830 . #x529D)      ; <CJK>
                 (#x4831 . #x7F3A)      ; <CJK>
                 (#x4832 . #x7094)      ; <CJK>
                 (#x4833 . #x7638)      ; <CJK>
                 (#x4834 . #x5374)      ; <CJK>
                 (#x4835 . #x9E4A)      ; <CJK>
                 (#x4836 . #x69B7)      ; <CJK>
                 (#x4837 . #x786E)      ; <CJK>
                 (#x4838 . #x96C0)      ; <CJK>
                 (#x4839 . #x88D9)      ; <CJK>
                 (#x483A . #x7FA4)      ; <CJK>
                 (#x483B . #x7136)      ; <CJK>
                 (#x483C . #x71C3)      ; <CJK>
                 (#x483D . #x5189)      ; <CJK>
                 (#x483E . #x67D3)      ; <CJK>
                 (#x483F . #x74E4)      ; <CJK>
                 (#x4840 . #x58E4)      ; <CJK>
                 (#x4841 . #x6518)      ; <CJK>
                 (#x4842 . #x56B7)      ; <CJK>
                 (#x4843 . #x8BA9)      ; <CJK>
                 (#x4844 . #x9976)      ; <CJK>
                 (#x4845 . #x6270)      ; <CJK>
                 (#x4846 . #x7ED5)      ; <CJK>
                 (#x4847 . #x60F9)      ; <CJK>
                 (#x4848 . #x70ED)      ; <CJK>
                 (#x4849 . #x58EC)      ; <CJK>
                 (#x484A . #x4EC1)      ; <CJK>
                 (#x484B . #x4EBA)      ; <CJK>
                 (#x484C . #x5FCD)      ; <CJK>
                 (#x484D . #x97E7)      ; <CJK>
                 (#x484E . #x4EFB)      ; <CJK>
                 (#x484F . #x8BA4)      ; <CJK>
                 (#x4850 . #x5203)      ; <CJK>
                 (#x4851 . #x598A)      ; <CJK>
                 (#x4852 . #x7EAB)      ; <CJK>
                 (#x4853 . #x6254)      ; <CJK>
                 (#x4854 . #x4ECD)      ; <CJK>
                 (#x4855 . #x65E5)      ; <CJK>
                 (#x4856 . #x620E)      ; <CJK>
                 (#x4857 . #x8338)      ; <CJK>
                 (#x4858 . #x84C9)      ; <CJK>
                 (#x4859 . #x8363)      ; <CJK>
                 (#x485A . #x878D)      ; <CJK>
                 (#x485B . #x7194)      ; <CJK>
                 (#x485C . #x6EB6)      ; <CJK>
                 (#x485D . #x5BB9)      ; <CJK>
                 (#x485E . #x7ED2)      ; <CJK>
                 (#x485F . #x5197)      ; <CJK>
                 (#x4860 . #x63C9)      ; <CJK>
                 (#x4861 . #x67D4)      ; <CJK>
                 (#x4862 . #x8089)      ; <CJK>
                 (#x4863 . #x8339)      ; <CJK>
                 (#x4864 . #x8815)      ; <CJK>
                 (#x4865 . #x5112)      ; <CJK>
                 (#x4866 . #x5B7A)      ; <CJK>
                 (#x4867 . #x5982)      ; <CJK>
                 (#x4868 . #x8FB1)      ; <CJK>
                 (#x4869 . #x4E73)      ; <CJK>
                 (#x486A . #x6C5D)      ; <CJK>
                 (#x486B . #x5165)      ; <CJK>
                 (#x486C . #x8925)      ; <CJK>
                 (#x486D . #x8F6F)      ; <CJK>
                 (#x486E . #x962E)      ; <CJK>
                 (#x486F . #x854A)      ; <CJK>
                 (#x4870 . #x745E)      ; <CJK>
                 (#x4871 . #x9510)      ; <CJK>
                 (#x4872 . #x95F0)      ; <CJK>
                 (#x4873 . #x6DA6)      ; <CJK>
                 (#x4874 . #x82E5)      ; <CJK>
                 (#x4875 . #x5F31)      ; <CJK>
                 (#x4876 . #x6492)      ; <CJK>
                 (#x4877 . #x6D12)      ; <CJK>
                 (#x4878 . #x8428)      ; <CJK>
                 (#x4879 . #x816E)      ; <CJK>
                 (#x487A . #x9CC3)      ; <CJK>
                 (#x487B . #x585E)      ; <CJK>
                 (#x487C . #x8D5B)      ; <CJK>
                 (#x487D . #x4E09)      ; <CJK>
                 (#x487E . #x53C1)      ; <CJK>
                 (#x4921 . #x4F1E)      ; <CJK>
                 (#x4922 . #x6563)      ; <CJK>
                 (#x4923 . #x6851)      ; <CJK>
                 (#x4924 . #x55D3)      ; <CJK>
                 (#x4925 . #x4E27)      ; <CJK>
                 (#x4926 . #x6414)      ; <CJK>
                 (#x4927 . #x9A9A)      ; <CJK>
                 (#x4928 . #x626B)      ; <CJK>
                 (#x4929 . #x5AC2)      ; <CJK>
                 (#x492A . #x745F)      ; <CJK>
                 (#x492B . #x8272)      ; <CJK>
                 (#x492C . #x6DA9)      ; <CJK>
                 (#x492D . #x68EE)      ; <CJK>
                 (#x492E . #x50E7)      ; <CJK>
                 (#x492F . #x838E)      ; <CJK>
                 (#x4930 . #x7802)      ; <CJK>
                 (#x4931 . #x6740)      ; <CJK>
                 (#x4932 . #x5239)      ; <CJK>
                 (#x4933 . #x6C99)      ; <CJK>
                 (#x4934 . #x7EB1)      ; <CJK>
                 (#x4935 . #x50BB)      ; <CJK>
                 (#x4936 . #x5565)      ; <CJK>
                 (#x4937 . #x715E)      ; <CJK>
                 (#x4938 . #x7B5B)      ; <CJK>
                 (#x4939 . #x6652)      ; <CJK>
                 (#x493A . #x73CA)      ; <CJK>
                 (#x493B . #x82EB)      ; <CJK>
                 (#x493C . #x6749)      ; <CJK>
                 (#x493D . #x5C71)      ; <CJK>
                 (#x493E . #x5220)      ; <CJK>
                 (#x493F . #x717D)      ; <CJK>
                 (#x4940 . #x886B)      ; <CJK>
                 (#x4941 . #x95EA)      ; <CJK>
                 (#x4942 . #x9655)      ; <CJK>
                 (#x4943 . #x64C5)      ; <CJK>
                 (#x4944 . #x8D61)      ; <CJK>
                 (#x4945 . #x81B3)      ; <CJK>
                 (#x4946 . #x5584)      ; <CJK>
                 (#x4947 . #x6C55)      ; <CJK>
                 (#x4948 . #x6247)      ; <CJK>
                 (#x4949 . #x7F2E)      ; <CJK>
                 (#x494A . #x5892)      ; <CJK>
                 (#x494B . #x4F24)      ; <CJK>
                 (#x494C . #x5546)      ; <CJK>
                 (#x494D . #x8D4F)      ; <CJK>
                 (#x494E . #x664C)      ; <CJK>
                 (#x494F . #x4E0A)      ; <CJK>
                 (#x4950 . #x5C1A)      ; <CJK>
                 (#x4951 . #x88F3)      ; <CJK>
                 (#x4952 . #x68A2)      ; <CJK>
                 (#x4953 . #x634E)      ; <CJK>
                 (#x4954 . #x7A0D)      ; <CJK>
                 (#x4955 . #x70E7)      ; <CJK>
                 (#x4956 . #x828D)      ; <CJK>
                 (#x4957 . #x52FA)      ; <CJK>
                 (#x4958 . #x97F6)      ; <CJK>
                 (#x4959 . #x5C11)      ; <CJK>
                 (#x495A . #x54E8)      ; <CJK>
                 (#x495B . #x90B5)      ; <CJK>
                 (#x495C . #x7ECD)      ; <CJK>
                 (#x495D . #x5962)      ; <CJK>
                 (#x495E . #x8D4A)      ; <CJK>
                 (#x495F . #x86C7)      ; <CJK>
                 (#x4960 . #x820C)      ; <CJK>
                 (#x4961 . #x820D)      ; <CJK>
                 (#x4962 . #x8D66)      ; <CJK>
                 (#x4963 . #x6444)      ; <CJK>
                 (#x4964 . #x5C04)      ; <CJK>
                 (#x4965 . #x6151)      ; <CJK>
                 (#x4966 . #x6D89)      ; <CJK>
                 (#x4967 . #x793E)      ; <CJK>
                 (#x4968 . #x8BBE)      ; <CJK>
                 (#x4969 . #x7837)      ; <CJK>
                 (#x496A . #x7533)      ; <CJK>
                 (#x496B . #x547B)      ; <CJK>
                 (#x496C . #x4F38)      ; <CJK>
                 (#x496D . #x8EAB)      ; <CJK>
                 (#x496E . #x6DF1)      ; <CJK>
                 (#x496F . #x5A20)      ; <CJK>
                 (#x4970 . #x7EC5)      ; <CJK>
                 (#x4971 . #x795E)      ; <CJK>
                 (#x4972 . #x6C88)      ; <CJK>
                 (#x4973 . #x5BA1)      ; <CJK>
                 (#x4974 . #x5A76)      ; <CJK>
                 (#x4975 . #x751A)      ; <CJK>
                 (#x4976 . #x80BE)      ; <CJK>
                 (#x4977 . #x614E)      ; <CJK>
                 (#x4978 . #x6E17)      ; <CJK>
                 (#x4979 . #x58F0)      ; <CJK>
                 (#x497A . #x751F)      ; <CJK>
                 (#x497B . #x7525)      ; <CJK>
                 (#x497C . #x7272)      ; <CJK>
                 (#x497D . #x5347)      ; <CJK>
                 (#x497E . #x7EF3)      ; <CJK>
                 (#x4A21 . #x7701)      ; <CJK>
                 (#x4A22 . #x76DB)      ; <CJK>
                 (#x4A23 . #x5269)      ; <CJK>
                 (#x4A24 . #x80DC)      ; <CJK>
                 (#x4A25 . #x5723)      ; <CJK>
                 (#x4A26 . #x5E08)      ; <CJK>
                 (#x4A27 . #x5931)      ; <CJK>
                 (#x4A28 . #x72EE)      ; <CJK>
                 (#x4A29 . #x65BD)      ; <CJK>
                 (#x4A2A . #x6E7F)      ; <CJK>
                 (#x4A2B . #x8BD7)      ; <CJK>
                 (#x4A2C . #x5C38)      ; <CJK>
                 (#x4A2D . #x8671)      ; <CJK>
                 (#x4A2E . #x5341)      ; <CJK>
                 (#x4A2F . #x77F3)      ; <CJK>
                 (#x4A30 . #x62FE)      ; <CJK>
                 (#x4A31 . #x65F6)      ; <CJK>
                 (#x4A32 . #x4EC0)      ; <CJK>
                 (#x4A33 . #x98DF)      ; <CJK>
                 (#x4A34 . #x8680)      ; <CJK>
                 (#x4A35 . #x5B9E)      ; <CJK>
                 (#x4A36 . #x8BC6)      ; <CJK>
                 (#x4A37 . #x53F2)      ; <CJK>
                 (#x4A38 . #x77E2)      ; <CJK>
                 (#x4A39 . #x4F7F)      ; <CJK>
                 (#x4A3A . #x5C4E)      ; <CJK>
                 (#x4A3B . #x9A76)      ; <CJK>
                 (#x4A3C . #x59CB)      ; <CJK>
                 (#x4A3D . #x5F0F)      ; <CJK>
                 (#x4A3E . #x793A)      ; <CJK>
                 (#x4A3F . #x58EB)      ; <CJK>
                 (#x4A40 . #x4E16)      ; <CJK>
                 (#x4A41 . #x67FF)      ; <CJK>
                 (#x4A42 . #x4E8B)      ; <CJK>
                 (#x4A43 . #x62ED)      ; <CJK>
                 (#x4A44 . #x8A93)      ; <CJK>
                 (#x4A45 . #x901D)      ; <CJK>
                 (#x4A46 . #x52BF)      ; <CJK>
                 (#x4A47 . #x662F)      ; <CJK>
                 (#x4A48 . #x55DC)      ; <CJK>
                 (#x4A49 . #x566C)      ; <CJK>
                 (#x4A4A . #x9002)      ; <CJK>
                 (#x4A4B . #x4ED5)      ; <CJK>
                 (#x4A4C . #x4F8D)      ; <CJK>
                 (#x4A4D . #x91CA)      ; <CJK>
                 (#x4A4E . #x9970)      ; <CJK>
                 (#x4A4F . #x6C0F)      ; <CJK>
                 (#x4A50 . #x5E02)      ; <CJK>
                 (#x4A51 . #x6043)      ; <CJK>
                 (#x4A52 . #x5BA4)      ; <CJK>
                 (#x4A53 . #x89C6)      ; <CJK>
                 (#x4A54 . #x8BD5)      ; <CJK>
                 (#x4A55 . #x6536)      ; <CJK>
                 (#x4A56 . #x624B)      ; <CJK>
                 (#x4A57 . #x9996)      ; <CJK>
                 (#x4A58 . #x5B88)      ; <CJK>
                 (#x4A59 . #x5BFF)      ; <CJK>
                 (#x4A5A . #x6388)      ; <CJK>
                 (#x4A5B . #x552E)      ; <CJK>
                 (#x4A5C . #x53D7)      ; <CJK>
                 (#x4A5D . #x7626)      ; <CJK>
                 (#x4A5E . #x517D)      ; <CJK>
                 (#x4A5F . #x852C)      ; <CJK>
                 (#x4A60 . #x67A2)      ; <CJK>
                 (#x4A61 . #x68B3)      ; <CJK>
                 (#x4A62 . #x6B8A)      ; <CJK>
                 (#x4A63 . #x6292)      ; <CJK>
                 (#x4A64 . #x8F93)      ; <CJK>
                 (#x4A65 . #x53D4)      ; <CJK>
                 (#x4A66 . #x8212)      ; <CJK>
                 (#x4A67 . #x6DD1)      ; <CJK>
                 (#x4A68 . #x758F)      ; <CJK>
                 (#x4A69 . #x4E66)      ; <CJK>
                 (#x4A6A . #x8D4E)      ; <CJK>
                 (#x4A6B . #x5B70)      ; <CJK>
                 (#x4A6C . #x719F)      ; <CJK>
                 (#x4A6D . #x85AF)      ; <CJK>
                 (#x4A6E . #x6691)      ; <CJK>
                 (#x4A6F . #x66D9)      ; <CJK>
                 (#x4A70 . #x7F72)      ; <CJK>
                 (#x4A71 . #x8700)      ; <CJK>
                 (#x4A72 . #x9ECD)      ; <CJK>
                 (#x4A73 . #x9F20)      ; <CJK>
                 (#x4A74 . #x5C5E)      ; <CJK>
                 (#x4A75 . #x672F)      ; <CJK>
                 (#x4A76 . #x8FF0)      ; <CJK>
                 (#x4A77 . #x6811)      ; <CJK>
                 (#x4A78 . #x675F)      ; <CJK>
                 (#x4A79 . #x620D)      ; <CJK>
                 (#x4A7A . #x7AD6)      ; <CJK>
                 (#x4A7B . #x5885)      ; <CJK>
                 (#x4A7C . #x5EB6)      ; <CJK>
                 (#x4A7D . #x6570)      ; <CJK>
                 (#x4A7E . #x6F31)      ; <CJK>
                 (#x4B21 . #x6055)      ; <CJK>
                 (#x4B22 . #x5237)      ; <CJK>
                 (#x4B23 . #x800D)      ; <CJK>
                 (#x4B24 . #x6454)      ; <CJK>
                 (#x4B25 . #x8870)      ; <CJK>
                 (#x4B26 . #x7529)      ; <CJK>
                 (#x4B27 . #x5E05)      ; <CJK>
                 (#x4B28 . #x6813)      ; <CJK>
                 (#x4B29 . #x62F4)      ; <CJK>
                 (#x4B2A . #x971C)      ; <CJK>
                 (#x4B2B . #x53CC)      ; <CJK>
                 (#x4B2C . #x723D)      ; <CJK>
                 (#x4B2D . #x8C01)      ; <CJK>
                 (#x4B2E . #x6C34)      ; <CJK>
                 (#x4B2F . #x7761)      ; <CJK>
                 (#x4B30 . #x7A0E)      ; <CJK>
                 (#x4B31 . #x542E)      ; <CJK>
                 (#x4B32 . #x77AC)      ; <CJK>
                 (#x4B33 . #x987A)      ; <CJK>
                 (#x4B34 . #x821C)      ; <CJK>
                 (#x4B35 . #x8BF4)      ; <CJK>
                 (#x4B36 . #x7855)      ; <CJK>
                 (#x4B37 . #x6714)      ; <CJK>
                 (#x4B38 . #x70C1)      ; <CJK>
                 (#x4B39 . #x65AF)      ; <CJK>
                 (#x4B3A . #x6495)      ; <CJK>
                 (#x4B3B . #x5636)      ; <CJK>
                 (#x4B3C . #x601D)      ; <CJK>
                 (#x4B3D . #x79C1)      ; <CJK>
                 (#x4B3E . #x53F8)      ; <CJK>
                 (#x4B3F . #x4E1D)      ; <CJK>
                 (#x4B40 . #x6B7B)      ; <CJK>
                 (#x4B41 . #x8086)      ; <CJK>
                 (#x4B42 . #x5BFA)      ; <CJK>
                 (#x4B43 . #x55E3)      ; <CJK>
                 (#x4B44 . #x56DB)      ; <CJK>
                 (#x4B45 . #x4F3A)      ; <CJK>
                 (#x4B46 . #x4F3C)      ; <CJK>
                 (#x4B47 . #x9972)      ; <CJK>
                 (#x4B48 . #x5DF3)      ; <CJK>
                 (#x4B49 . #x677E)      ; <CJK>
                 (#x4B4A . #x8038)      ; <CJK>
                 (#x4B4B . #x6002)      ; <CJK>
                 (#x4B4C . #x9882)      ; <CJK>
                 (#x4B4D . #x9001)      ; <CJK>
                 (#x4B4E . #x5B8B)      ; <CJK>
                 (#x4B4F . #x8BBC)      ; <CJK>
                 (#x4B50 . #x8BF5)      ; <CJK>
                 (#x4B51 . #x641C)      ; <CJK>
                 (#x4B52 . #x8258)      ; <CJK>
                 (#x4B53 . #x64DE)      ; <CJK>
                 (#x4B54 . #x55FD)      ; <CJK>
                 (#x4B55 . #x82CF)      ; <CJK>
                 (#x4B56 . #x9165)      ; <CJK>
                 (#x4B57 . #x4FD7)      ; <CJK>
                 (#x4B58 . #x7D20)      ; <CJK>
                 (#x4B59 . #x901F)      ; <CJK>
                 (#x4B5A . #x7C9F)      ; <CJK>
                 (#x4B5B . #x50F3)      ; <CJK>
                 (#x4B5C . #x5851)      ; <CJK>
                 (#x4B5D . #x6EAF)      ; <CJK>
                 (#x4B5E . #x5BBF)      ; <CJK>
                 (#x4B5F . #x8BC9)      ; <CJK>
                 (#x4B60 . #x8083)      ; <CJK>
                 (#x4B61 . #x9178)      ; <CJK>
                 (#x4B62 . #x849C)      ; <CJK>
                 (#x4B63 . #x7B97)      ; <CJK>
                 (#x4B64 . #x867D)      ; <CJK>
                 (#x4B65 . #x968B)      ; <CJK>
                 (#x4B66 . #x968F)      ; <CJK>
                 (#x4B67 . #x7EE5)      ; <CJK>
                 (#x4B68 . #x9AD3)      ; <CJK>
                 (#x4B69 . #x788E)      ; <CJK>
                 (#x4B6A . #x5C81)      ; <CJK>
                 (#x4B6B . #x7A57)      ; <CJK>
                 (#x4B6C . #x9042)      ; <CJK>
                 (#x4B6D . #x96A7)      ; <CJK>
                 (#x4B6E . #x795F)      ; <CJK>
                 (#x4B6F . #x5B59)      ; <CJK>
                 (#x4B70 . #x635F)      ; <CJK>
                 (#x4B71 . #x7B0B)      ; <CJK>
                 (#x4B72 . #x84D1)      ; <CJK>
                 (#x4B73 . #x68AD)      ; <CJK>
                 (#x4B74 . #x5506)      ; <CJK>
                 (#x4B75 . #x7F29)      ; <CJK>
                 (#x4B76 . #x7410)      ; <CJK>
                 (#x4B77 . #x7D22)      ; <CJK>
                 (#x4B78 . #x9501)      ; <CJK>
                 (#x4B79 . #x6240)      ; <CJK>
                 (#x4B7A . #x584C)      ; <CJK>
                 (#x4B7B . #x4ED6)      ; <CJK>
                 (#x4B7C . #x5B83)      ; <CJK>
                 (#x4B7D . #x5979)      ; <CJK>
                 (#x4B7E . #x5854)      ; <CJK>
                 (#x4C21 . #x736D)      ; <CJK>
                 (#x4C22 . #x631E)      ; <CJK>
                 (#x4C23 . #x8E4B)      ; <CJK>
                 (#x4C24 . #x8E0F)      ; <CJK>
                 (#x4C25 . #x80CE)      ; <CJK>
                 (#x4C26 . #x82D4)      ; <CJK>
                 (#x4C27 . #x62AC)      ; <CJK>
                 (#x4C28 . #x53F0)      ; <CJK>
                 (#x4C29 . #x6CF0)      ; <CJK>
                 (#x4C2A . #x915E)      ; <CJK>
                 (#x4C2B . #x592A)      ; <CJK>
                 (#x4C2C . #x6001)      ; <CJK>
                 (#x4C2D . #x6C70)      ; <CJK>
                 (#x4C2E . #x574D)      ; <CJK>
                 (#x4C2F . #x644A)      ; <CJK>
                 (#x4C30 . #x8D2A)      ; <CJK>
                 (#x4C31 . #x762B)      ; <CJK>
                 (#x4C32 . #x6EE9)      ; <CJK>
                 (#x4C33 . #x575B)      ; <CJK>
                 (#x4C34 . #x6A80)      ; <CJK>
                 (#x4C35 . #x75F0)      ; <CJK>
                 (#x4C36 . #x6F6D)      ; <CJK>
                 (#x4C37 . #x8C2D)      ; <CJK>
                 (#x4C38 . #x8C08)      ; <CJK>
                 (#x4C39 . #x5766)      ; <CJK>
                 (#x4C3A . #x6BEF)      ; <CJK>
                 (#x4C3B . #x8892)      ; <CJK>
                 (#x4C3C . #x78B3)      ; <CJK>
                 (#x4C3D . #x63A2)      ; <CJK>
                 (#x4C3E . #x53F9)      ; <CJK>
                 (#x4C3F . #x70AD)      ; <CJK>
                 (#x4C40 . #x6C64)      ; <CJK>
                 (#x4C41 . #x5858)      ; <CJK>
                 (#x4C42 . #x642A)      ; <CJK>
                 (#x4C43 . #x5802)      ; <CJK>
                 (#x4C44 . #x68E0)      ; <CJK>
                 (#x4C45 . #x819B)      ; <CJK>
                 (#x4C46 . #x5510)      ; <CJK>
                 (#x4C47 . #x7CD6)      ; <CJK>
                 (#x4C48 . #x5018)      ; <CJK>
                 (#x4C49 . #x8EBA)      ; <CJK>
                 (#x4C4A . #x6DCC)      ; <CJK>
                 (#x4C4B . #x8D9F)      ; <CJK>
                 (#x4C4C . #x70EB)      ; <CJK>
                 (#x4C4D . #x638F)      ; <CJK>
                 (#x4C4E . #x6D9B)      ; <CJK>
                 (#x4C4F . #x6ED4)      ; <CJK>
                 (#x4C50 . #x7EE6)      ; <CJK>
                 (#x4C51 . #x8404)      ; <CJK>
                 (#x4C52 . #x6843)      ; <CJK>
                 (#x4C53 . #x9003)      ; <CJK>
                 (#x4C54 . #x6DD8)      ; <CJK>
                 (#x4C55 . #x9676)      ; <CJK>
                 (#x4C56 . #x8BA8)      ; <CJK>
                 (#x4C57 . #x5957)      ; <CJK>
                 (#x4C58 . #x7279)      ; <CJK>
                 (#x4C59 . #x85E4)      ; <CJK>
                 (#x4C5A . #x817E)      ; <CJK>
                 (#x4C5B . #x75BC)      ; <CJK>
                 (#x4C5C . #x8A8A)      ; <CJK>
                 (#x4C5D . #x68AF)      ; <CJK>
                 (#x4C5E . #x5254)      ; <CJK>
                 (#x4C5F . #x8E22)      ; <CJK>
                 (#x4C60 . #x9511)      ; <CJK>
                 (#x4C61 . #x63D0)      ; <CJK>
                 (#x4C62 . #x9898)      ; <CJK>
                 (#x4C63 . #x8E44)      ; <CJK>
                 (#x4C64 . #x557C)      ; <CJK>
                 (#x4C65 . #x4F53)      ; <CJK>
                 (#x4C66 . #x66FF)      ; <CJK>
                 (#x4C67 . #x568F)      ; <CJK>
                 (#x4C68 . #x60D5)      ; <CJK>
                 (#x4C69 . #x6D95)      ; <CJK>
                 (#x4C6A . #x5243)      ; <CJK>
                 (#x4C6B . #x5C49)      ; <CJK>
                 (#x4C6C . #x5929)      ; <CJK>
                 (#x4C6D . #x6DFB)      ; <CJK>
                 (#x4C6E . #x586B)      ; <CJK>
                 (#x4C6F . #x7530)      ; <CJK>
                 (#x4C70 . #x751C)      ; <CJK>
                 (#x4C71 . #x606C)      ; <CJK>
                 (#x4C72 . #x8214)      ; <CJK>
                 (#x4C73 . #x8146)      ; <CJK>
                 (#x4C74 . #x6311)      ; <CJK>
                 (#x4C75 . #x6761)      ; <CJK>
                 (#x4C76 . #x8FE2)      ; <CJK>
                 (#x4C77 . #x773A)      ; <CJK>
                 (#x4C78 . #x8DF3)      ; <CJK>
                 (#x4C79 . #x8D34)      ; <CJK>
                 (#x4C7A . #x94C1)      ; <CJK>
                 (#x4C7B . #x5E16)      ; <CJK>
                 (#x4C7C . #x5385)      ; <CJK>
                 (#x4C7D . #x542C)      ; <CJK>
                 (#x4C7E . #x70C3)      ; <CJK>
                 (#x4D21 . #x6C40)      ; <CJK>
                 (#x4D22 . #x5EF7)      ; <CJK>
                 (#x4D23 . #x505C)      ; <CJK>
                 (#x4D24 . #x4EAD)      ; <CJK>
                 (#x4D25 . #x5EAD)      ; <CJK>
                 (#x4D26 . #x633A)      ; <CJK>
                 (#x4D27 . #x8247)      ; <CJK>
                 (#x4D28 . #x901A)      ; <CJK>
                 (#x4D29 . #x6850)      ; <CJK>
                 (#x4D2A . #x916E)      ; <CJK>
                 (#x4D2B . #x77B3)      ; <CJK>
                 (#x4D2C . #x540C)      ; <CJK>
                 (#x4D2D . #x94DC)      ; <CJK>
                 (#x4D2E . #x5F64)      ; <CJK>
                 (#x4D2F . #x7AE5)      ; <CJK>
                 (#x4D30 . #x6876)      ; <CJK>
                 (#x4D31 . #x6345)      ; <CJK>
                 (#x4D32 . #x7B52)      ; <CJK>
                 (#x4D33 . #x7EDF)      ; <CJK>
                 (#x4D34 . #x75DB)      ; <CJK>
                 (#x4D35 . #x5077)      ; <CJK>
                 (#x4D36 . #x6295)      ; <CJK>
                 (#x4D37 . #x5934)      ; <CJK>
                 (#x4D38 . #x900F)      ; <CJK>
                 (#x4D39 . #x51F8)      ; <CJK>
                 (#x4D3A . #x79C3)      ; <CJK>
                 (#x4D3B . #x7A81)      ; <CJK>
                 (#x4D3C . #x56FE)      ; <CJK>
                 (#x4D3D . #x5F92)      ; <CJK>
                 (#x4D3E . #x9014)      ; <CJK>
                 (#x4D3F . #x6D82)      ; <CJK>
                 (#x4D40 . #x5C60)      ; <CJK>
                 (#x4D41 . #x571F)      ; <CJK>
                 (#x4D42 . #x5410)      ; <CJK>
                 (#x4D43 . #x5154)      ; <CJK>
                 (#x4D44 . #x6E4D)      ; <CJK>
                 (#x4D45 . #x56E2)      ; <CJK>
                 (#x4D46 . #x63A8)      ; <CJK>
                 (#x4D47 . #x9893)      ; <CJK>
                 (#x4D48 . #x817F)      ; <CJK>
                 (#x4D49 . #x8715)      ; <CJK>
                 (#x4D4A . #x892A)      ; <CJK>
                 (#x4D4B . #x9000)      ; <CJK>
                 (#x4D4C . #x541E)      ; <CJK>
                 (#x4D4D . #x5C6F)      ; <CJK>
                 (#x4D4E . #x81C0)      ; <CJK>
                 (#x4D4F . #x62D6)      ; <CJK>
                 (#x4D50 . #x6258)      ; <CJK>
                 (#x4D51 . #x8131)      ; <CJK>
                 (#x4D52 . #x9E35)      ; <CJK>
                 (#x4D53 . #x9640)      ; <CJK>
                 (#x4D54 . #x9A6E)      ; <CJK>
                 (#x4D55 . #x9A7C)      ; <CJK>
                 (#x4D56 . #x692D)      ; <CJK>
                 (#x4D57 . #x59A5)      ; <CJK>
                 (#x4D58 . #x62D3)      ; <CJK>
                 (#x4D59 . #x553E)      ; <CJK>
                 (#x4D5A . #x6316)      ; <CJK>
                 (#x4D5B . #x54C7)      ; <CJK>
                 (#x4D5C . #x86D9)      ; <CJK>
                 (#x4D5D . #x6D3C)      ; <CJK>
                 (#x4D5E . #x5A03)      ; <CJK>
                 (#x4D5F . #x74E6)      ; <CJK>
                 (#x4D60 . #x889C)      ; <CJK>
                 (#x4D61 . #x6B6A)      ; <CJK>
                 (#x4D62 . #x5916)      ; <CJK>
                 (#x4D63 . #x8C4C)      ; <CJK>
                 (#x4D64 . #x5F2F)      ; <CJK>
                 (#x4D65 . #x6E7E)      ; <CJK>
                 (#x4D66 . #x73A9)      ; <CJK>
                 (#x4D67 . #x987D)      ; <CJK>
                 (#x4D68 . #x4E38)      ; <CJK>
                 (#x4D69 . #x70F7)      ; <CJK>
                 (#x4D6A . #x5B8C)      ; <CJK>
                 (#x4D6B . #x7897)      ; <CJK>
                 (#x4D6C . #x633D)      ; <CJK>
                 (#x4D6D . #x665A)      ; <CJK>
                 (#x4D6E . #x7696)      ; <CJK>
                 (#x4D6F . #x60CB)      ; <CJK>
                 (#x4D70 . #x5B9B)      ; <CJK>
                 (#x4D71 . #x5A49)      ; <CJK>
                 (#x4D72 . #x4E07)      ; <CJK>
                 (#x4D73 . #x8155)      ; <CJK>
                 (#x4D74 . #x6C6A)      ; <CJK>
                 (#x4D75 . #x738B)      ; <CJK>
                 (#x4D76 . #x4EA1)      ; <CJK>
                 (#x4D77 . #x6789)      ; <CJK>
                 (#x4D78 . #x7F51)      ; <CJK>
                 (#x4D79 . #x5F80)      ; <CJK>
                 (#x4D7A . #x65FA)      ; <CJK>
                 (#x4D7B . #x671B)      ; <CJK>
                 (#x4D7C . #x5FD8)      ; <CJK>
                 (#x4D7D . #x5984)      ; <CJK>
                 (#x4D7E . #x5A01)      ; <CJK>
                 (#x4E21 . #x5DCD)      ; <CJK>
                 (#x4E22 . #x5FAE)      ; <CJK>
                 (#x4E23 . #x5371)      ; <CJK>
                 (#x4E24 . #x97E6)      ; <CJK>
                 (#x4E25 . #x8FDD)      ; <CJK>
                 (#x4E26 . #x6845)      ; <CJK>
                 (#x4E27 . #x56F4)      ; <CJK>
                 (#x4E28 . #x552F)      ; <CJK>
                 (#x4E29 . #x60DF)      ; <CJK>
                 (#x4E2A . #x4E3A)      ; <CJK>
                 (#x4E2B . #x6F4D)      ; <CJK>
                 (#x4E2C . #x7EF4)      ; <CJK>
                 (#x4E2D . #x82C7)      ; <CJK>
                 (#x4E2E . #x840E)      ; <CJK>
                 (#x4E2F . #x59D4)      ; <CJK>
                 (#x4E30 . #x4F1F)      ; <CJK>
                 (#x4E31 . #x4F2A)      ; <CJK>
                 (#x4E32 . #x5C3E)      ; <CJK>
                 (#x4E33 . #x7EAC)      ; <CJK>
                 (#x4E34 . #x672A)      ; <CJK>
                 (#x4E35 . #x851A)      ; <CJK>
                 (#x4E36 . #x5473)      ; <CJK>
                 (#x4E37 . #x754F)      ; <CJK>
                 (#x4E38 . #x80C3)      ; <CJK>
                 (#x4E39 . #x5582)      ; <CJK>
                 (#x4E3A . #x9B4F)      ; <CJK>
                 (#x4E3B . #x4F4D)      ; <CJK>
                 (#x4E3C . #x6E2D)      ; <CJK>
                 (#x4E3D . #x8C13)      ; <CJK>
                 (#x4E3E . #x5C09)      ; <CJK>
                 (#x4E3F . #x6170)      ; <CJK>
                 (#x4E40 . #x536B)      ; <CJK>
                 (#x4E41 . #x761F)      ; <CJK>
                 (#x4E42 . #x6E29)      ; <CJK>
                 (#x4E43 . #x868A)      ; <CJK>
                 (#x4E44 . #x6587)      ; <CJK>
                 (#x4E45 . #x95FB)      ; <CJK>
                 (#x4E46 . #x7EB9)      ; <CJK>
                 (#x4E47 . #x543B)      ; <CJK>
                 (#x4E48 . #x7A33)      ; <CJK>
                 (#x4E49 . #x7D0A)      ; <CJK>
                 (#x4E4A . #x95EE)      ; <CJK>
                 (#x4E4B . #x55E1)      ; <CJK>
                 (#x4E4C . #x7FC1)      ; <CJK>
                 (#x4E4D . #x74EE)      ; <CJK>
                 (#x4E4E . #x631D)      ; <CJK>
                 (#x4E4F . #x8717)      ; <CJK>
                 (#x4E50 . #x6DA1)      ; <CJK>
                 (#x4E51 . #x7A9D)      ; <CJK>
                 (#x4E52 . #x6211)      ; <CJK>
                 (#x4E53 . #x65A1)      ; <CJK>
                 (#x4E54 . #x5367)      ; <CJK>
                 (#x4E55 . #x63E1)      ; <CJK>
                 (#x4E56 . #x6C83)      ; <CJK>
                 (#x4E57 . #x5DEB)      ; <CJK>
                 (#x4E58 . #x545C)      ; <CJK>
                 (#x4E59 . #x94A8)      ; <CJK>
                 (#x4E5A . #x4E4C)      ; <CJK>
                 (#x4E5B . #x6C61)      ; <CJK>
                 (#x4E5C . #x8BEC)      ; <CJK>
                 (#x4E5D . #x5C4B)      ; <CJK>
                 (#x4E5E . #x65E0)      ; <CJK>
                 (#x4E5F . #x829C)      ; <CJK>
                 (#x4E60 . #x68A7)      ; <CJK>
                 (#x4E61 . #x543E)      ; <CJK>
                 (#x4E62 . #x5434)      ; <CJK>
                 (#x4E63 . #x6BCB)      ; <CJK>
                 (#x4E64 . #x6B66)      ; <CJK>
                 (#x4E65 . #x4E94)      ; <CJK>
                 (#x4E66 . #x6342)      ; <CJK>
                 (#x4E67 . #x5348)      ; <CJK>
                 (#x4E68 . #x821E)      ; <CJK>
                 (#x4E69 . #x4F0D)      ; <CJK>
                 (#x4E6A . #x4FAE)      ; <CJK>
                 (#x4E6B . #x575E)      ; <CJK>
                 (#x4E6C . #x620A)      ; <CJK>
                 (#x4E6D . #x96FE)      ; <CJK>
                 (#x4E6E . #x6664)      ; <CJK>
                 (#x4E6F . #x7269)      ; <CJK>
                 (#x4E70 . #x52FF)      ; <CJK>
                 (#x4E71 . #x52A1)      ; <CJK>
                 (#x4E72 . #x609F)      ; <CJK>
                 (#x4E73 . #x8BEF)      ; <CJK>
                 (#x4E74 . #x6614)      ; <CJK>
                 (#x4E75 . #x7199)      ; <CJK>
                 (#x4E76 . #x6790)      ; <CJK>
                 (#x4E77 . #x897F)      ; <CJK>
                 (#x4E78 . #x7852)      ; <CJK>
                 (#x4E79 . #x77FD)      ; <CJK>
                 (#x4E7A . #x6670)      ; <CJK>
                 (#x4E7B . #x563B)      ; <CJK>
                 (#x4E7C . #x5438)      ; <CJK>
                 (#x4E7D . #x9521)      ; <CJK>
                 (#x4E7E . #x727A)      ; <CJK>
                 (#x4F21 . #x7A00)      ; <CJK>
                 (#x4F22 . #x606F)      ; <CJK>
                 (#x4F23 . #x5E0C)      ; <CJK>
                 (#x4F24 . #x6089)      ; <CJK>
                 (#x4F25 . #x819D)      ; <CJK>
                 (#x4F26 . #x5915)      ; <CJK>
                 (#x4F27 . #x60DC)      ; <CJK>
                 (#x4F28 . #x7184)      ; <CJK>
                 (#x4F29 . #x70EF)      ; <CJK>
                 (#x4F2A . #x6EAA)      ; <CJK>
                 (#x4F2B . #x6C50)      ; <CJK>
                 (#x4F2C . #x7280)      ; <CJK>
                 (#x4F2D . #x6A84)      ; <CJK>
                 (#x4F2E . #x88AD)      ; <CJK>
                 (#x4F2F . #x5E2D)      ; <CJK>
                 (#x4F30 . #x4E60)      ; <CJK>
                 (#x4F31 . #x5AB3)      ; <CJK>
                 (#x4F32 . #x559C)      ; <CJK>
                 (#x4F33 . #x94E3)      ; <CJK>
                 (#x4F34 . #x6D17)      ; <CJK>
                 (#x4F35 . #x7CFB)      ; <CJK>
                 (#x4F36 . #x9699)      ; <CJK>
                 (#x4F37 . #x620F)      ; <CJK>
                 (#x4F38 . #x7EC6)      ; <CJK>
                 (#x4F39 . #x778E)      ; <CJK>
                 (#x4F3A . #x867E)      ; <CJK>
                 (#x4F3B . #x5323)      ; <CJK>
                 (#x4F3C . #x971E)      ; <CJK>
                 (#x4F3D . #x8F96)      ; <CJK>
                 (#x4F3E . #x6687)      ; <CJK>
                 (#x4F3F . #x5CE1)      ; <CJK>
                 (#x4F40 . #x4FA0)      ; <CJK>
                 (#x4F41 . #x72ED)      ; <CJK>
                 (#x4F42 . #x4E0B)      ; <CJK>
                 (#x4F43 . #x53A6)      ; <CJK>
                 (#x4F44 . #x590F)      ; <CJK>
                 (#x4F45 . #x5413)      ; <CJK>
                 (#x4F46 . #x6380)      ; <CJK>
                 (#x4F47 . #x9528)      ; <CJK>
                 (#x4F48 . #x5148)      ; <CJK>
                 (#x4F49 . #x4ED9)      ; <CJK>
                 (#x4F4A . #x9C9C)      ; <CJK>
                 (#x4F4B . #x7EA4)      ; <CJK>
                 (#x4F4C . #x54B8)      ; <CJK>
                 (#x4F4D . #x8D24)      ; <CJK>
                 (#x4F4E . #x8854)      ; <CJK>
                 (#x4F4F . #x8237)      ; <CJK>
                 (#x4F50 . #x95F2)      ; <CJK>
                 (#x4F51 . #x6D8E)      ; <CJK>
                 (#x4F52 . #x5F26)      ; <CJK>
                 (#x4F53 . #x5ACC)      ; <CJK>
                 (#x4F54 . #x663E)      ; <CJK>
                 (#x4F55 . #x9669)      ; <CJK>
                 (#x4F56 . #x73B0)      ; <CJK>
                 (#x4F57 . #x732E)      ; <CJK>
                 (#x4F58 . #x53BF)      ; <CJK>
                 (#x4F59 . #x817A)      ; <CJK>
                 (#x4F5A . #x9985)      ; <CJK>
                 (#x4F5B . #x7FA1)      ; <CJK>
                 (#x4F5C . #x5BAA)      ; <CJK>
                 (#x4F5D . #x9677)      ; <CJK>
                 (#x4F5E . #x9650)      ; <CJK>
                 (#x4F5F . #x7EBF)      ; <CJK>
                 (#x4F60 . #x76F8)      ; <CJK>
                 (#x4F61 . #x53A2)      ; <CJK>
                 (#x4F62 . #x9576)      ; <CJK>
                 (#x4F63 . #x9999)      ; <CJK>
                 (#x4F64 . #x7BB1)      ; <CJK>
                 (#x4F65 . #x8944)      ; <CJK>
                 (#x4F66 . #x6E58)      ; <CJK>
                 (#x4F67 . #x4E61)      ; <CJK>
                 (#x4F68 . #x7FD4)      ; <CJK>
                 (#x4F69 . #x7965)      ; <CJK>
                 (#x4F6A . #x8BE6)      ; <CJK>
                 (#x4F6B . #x60F3)      ; <CJK>
                 (#x4F6C . #x54CD)      ; <CJK>
                 (#x4F6D . #x4EAB)      ; <CJK>
                 (#x4F6E . #x9879)      ; <CJK>
                 (#x4F6F . #x5DF7)      ; <CJK>
                 (#x4F70 . #x6A61)      ; <CJK>
                 (#x4F71 . #x50CF)      ; <CJK>
                 (#x4F72 . #x5411)      ; <CJK>
                 (#x4F73 . #x8C61)      ; <CJK>
                 (#x4F74 . #x8427)      ; <CJK>
                 (#x4F75 . #x785D)      ; <CJK>
                 (#x4F76 . #x9704)      ; <CJK>
                 (#x4F77 . #x524A)      ; <CJK>
                 (#x4F78 . #x54EE)      ; <CJK>
                 (#x4F79 . #x56A3)      ; <CJK>
                 (#x4F7A . #x9500)      ; <CJK>
                 (#x4F7B . #x6D88)      ; <CJK>
                 (#x4F7C . #x5BB5)      ; <CJK>
                 (#x4F7D . #x6DC6)      ; <CJK>
                 (#x4F7E . #x6653)      ; <CJK>
                 (#x5021 . #x5C0F)      ; <CJK>
                 (#x5022 . #x5B5D)      ; <CJK>
                 (#x5023 . #x6821)      ; <CJK>
                 (#x5024 . #x8096)      ; <CJK>
                 (#x5025 . #x5578)      ; <CJK>
                 (#x5026 . #x7B11)      ; <CJK>
                 (#x5027 . #x6548)      ; <CJK>
                 (#x5028 . #x6954)      ; <CJK>
                 (#x5029 . #x4E9B)      ; <CJK>
                 (#x502A . #x6B47)      ; <CJK>
                 (#x502B . #x874E)      ; <CJK>
                 (#x502C . #x978B)      ; <CJK>
                 (#x502D . #x534F)      ; <CJK>
                 (#x502E . #x631F)      ; <CJK>
                 (#x502F . #x643A)      ; <CJK>
                 (#x5030 . #x90AA)      ; <CJK>
                 (#x5031 . #x659C)      ; <CJK>
                 (#x5032 . #x80C1)      ; <CJK>
                 (#x5033 . #x8C10)      ; <CJK>
                 (#x5034 . #x5199)      ; <CJK>
                 (#x5035 . #x68B0)      ; <CJK>
                 (#x5036 . #x5378)      ; <CJK>
                 (#x5037 . #x87F9)      ; <CJK>
                 (#x5038 . #x61C8)      ; <CJK>
                 (#x5039 . #x6CC4)      ; <CJK>
                 (#x503A . #x6CFB)      ; <CJK>
                 (#x503B . #x8C22)      ; <CJK>
                 (#x503C . #x5C51)      ; <CJK>
                 (#x503D . #x85AA)      ; <CJK>
                 (#x503E . #x82AF)      ; <CJK>
                 (#x503F . #x950C)      ; <CJK>
                 (#x5040 . #x6B23)      ; <CJK>
                 (#x5041 . #x8F9B)      ; <CJK>
                 (#x5042 . #x65B0)      ; <CJK>
                 (#x5043 . #x5FFB)      ; <CJK>
                 (#x5044 . #x5FC3)      ; <CJK>
                 (#x5045 . #x4FE1)      ; <CJK>
                 (#x5046 . #x8845)      ; <CJK>
                 (#x5047 . #x661F)      ; <CJK>
                 (#x5048 . #x8165)      ; <CJK>
                 (#x5049 . #x7329)      ; <CJK>
                 (#x504A . #x60FA)      ; <CJK>
                 (#x504B . #x5174)      ; <CJK>
                 (#x504C . #x5211)      ; <CJK>
                 (#x504D . #x578B)      ; <CJK>
                 (#x504E . #x5F62)      ; <CJK>
                 (#x504F . #x90A2)      ; <CJK>
                 (#x5050 . #x884C)      ; <CJK>
                 (#x5051 . #x9192)      ; <CJK>
                 (#x5052 . #x5E78)      ; <CJK>
                 (#x5053 . #x674F)      ; <CJK>
                 (#x5054 . #x6027)      ; <CJK>
                 (#x5055 . #x59D3)      ; <CJK>
                 (#x5056 . #x5144)      ; <CJK>
                 (#x5057 . #x51F6)      ; <CJK>
                 (#x5058 . #x80F8)      ; <CJK>
                 (#x5059 . #x5308)      ; <CJK>
                 (#x505A . #x6C79)      ; <CJK>
                 (#x505B . #x96C4)      ; <CJK>
                 (#x505C . #x718A)      ; <CJK>
                 (#x505D . #x4F11)      ; <CJK>
                 (#x505E . #x4FEE)      ; <CJK>
                 (#x505F . #x7F9E)      ; <CJK>
                 (#x5060 . #x673D)      ; <CJK>
                 (#x5061 . #x55C5)      ; <CJK>
                 (#x5062 . #x9508)      ; <CJK>
                 (#x5063 . #x79C0)      ; <CJK>
                 (#x5064 . #x8896)      ; <CJK>
                 (#x5065 . #x7EE3)      ; <CJK>
                 (#x5066 . #x589F)      ; <CJK>
                 (#x5067 . #x620C)      ; <CJK>
                 (#x5068 . #x9700)      ; <CJK>
                 (#x5069 . #x865A)      ; <CJK>
                 (#x506A . #x5618)      ; <CJK>
                 (#x506B . #x987B)      ; <CJK>
                 (#x506C . #x5F90)      ; <CJK>
                 (#x506D . #x8BB8)      ; <CJK>
                 (#x506E . #x84C4)      ; <CJK>
                 (#x506F . #x9157)      ; <CJK>
                 (#x5070 . #x53D9)      ; <CJK>
                 (#x5071 . #x65ED)      ; <CJK>
                 (#x5072 . #x5E8F)      ; <CJK>
                 (#x5073 . #x755C)      ; <CJK>
                 (#x5074 . #x6064)      ; <CJK>
                 (#x5075 . #x7D6E)      ; <CJK>
                 (#x5076 . #x5A7F)      ; <CJK>
                 (#x5077 . #x7EEA)      ; <CJK>
                 (#x5078 . #x7EED)      ; <CJK>
                 (#x5079 . #x8F69)      ; <CJK>
                 (#x507A . #x55A7)      ; <CJK>
                 (#x507B . #x5BA3)      ; <CJK>
                 (#x507C . #x60AC)      ; <CJK>
                 (#x507D . #x65CB)      ; <CJK>
                 (#x507E . #x7384)      ; <CJK>
                 (#x5121 . #x9009)      ; <CJK>
                 (#x5122 . #x7663)      ; <CJK>
                 (#x5123 . #x7729)      ; <CJK>
                 (#x5124 . #x7EDA)      ; <CJK>
                 (#x5125 . #x9774)      ; <CJK>
                 (#x5126 . #x859B)      ; <CJK>
                 (#x5127 . #x5B66)      ; <CJK>
                 (#x5128 . #x7A74)      ; <CJK>
                 (#x5129 . #x96EA)      ; <CJK>
                 (#x512A . #x8840)      ; <CJK>
                 (#x512B . #x52CB)      ; <CJK>
                 (#x512C . #x718F)      ; <CJK>
                 (#x512D . #x5FAA)      ; <CJK>
                 (#x512E . #x65EC)      ; <CJK>
                 (#x512F . #x8BE2)      ; <CJK>
                 (#x5130 . #x5BFB)      ; <CJK>
                 (#x5131 . #x9A6F)      ; <CJK>
                 (#x5132 . #x5DE1)      ; <CJK>
                 (#x5133 . #x6B89)      ; <CJK>
                 (#x5134 . #x6C5B)      ; <CJK>
                 (#x5135 . #x8BAD)      ; <CJK>
                 (#x5136 . #x8BAF)      ; <CJK>
                 (#x5137 . #x900A)      ; <CJK>
                 (#x5138 . #x8FC5)      ; <CJK>
                 (#x5139 . #x538B)      ; <CJK>
                 (#x513A . #x62BC)      ; <CJK>
                 (#x513B . #x9E26)      ; <CJK>
                 (#x513C . #x9E2D)      ; <CJK>
                 (#x513D . #x5440)      ; <CJK>
                 (#x513E . #x4E2B)      ; <CJK>
                 (#x513F . #x82BD)      ; <CJK>
                 (#x5140 . #x7259)      ; <CJK>
                 (#x5141 . #x869C)      ; <CJK>
                 (#x5142 . #x5D16)      ; <CJK>
                 (#x5143 . #x8859)      ; <CJK>
                 (#x5144 . #x6DAF)      ; <CJK>
                 (#x5145 . #x96C5)      ; <CJK>
                 (#x5146 . #x54D1)      ; <CJK>
                 (#x5147 . #x4E9A)      ; <CJK>
                 (#x5148 . #x8BB6)      ; <CJK>
                 (#x5149 . #x7109)      ; <CJK>
                 (#x514A . #x54BD)      ; <CJK>
                 (#x514B . #x9609)      ; <CJK>
                 (#x514C . #x70DF)      ; <CJK>
                 (#x514D . #x6DF9)      ; <CJK>
                 (#x514E . #x76D0)      ; <CJK>
                 (#x514F . #x4E25)      ; <CJK>
                 (#x5150 . #x7814)      ; <CJK>
                 (#x5151 . #x8712)      ; <CJK>
                 (#x5152 . #x5CA9)      ; <CJK>
                 (#x5153 . #x5EF6)      ; <CJK>
                 (#x5154 . #x8A00)      ; <CJK>
                 (#x5155 . #x989C)      ; <CJK>
                 (#x5156 . #x960E)      ; <CJK>
                 (#x5157 . #x708E)      ; <CJK>
                 (#x5158 . #x6CBF)      ; <CJK>
                 (#x5159 . #x5944)      ; <CJK>
                 (#x515A . #x63A9)      ; <CJK>
                 (#x515B . #x773C)      ; <CJK>
                 (#x515C . #x884D)      ; <CJK>
                 (#x515D . #x6F14)      ; <CJK>
                 (#x515E . #x8273)      ; <CJK>
                 (#x515F . #x5830)      ; <CJK>
                 (#x5160 . #x71D5)      ; <CJK>
                 (#x5161 . #x538C)      ; <CJK>
                 (#x5162 . #x781A)      ; <CJK>
                 (#x5163 . #x96C1)      ; <CJK>
                 (#x5164 . #x5501)      ; <CJK>
                 (#x5165 . #x5F66)      ; <CJK>
                 (#x5166 . #x7130)      ; <CJK>
                 (#x5167 . #x5BB4)      ; <CJK>
                 (#x5168 . #x8C1A)      ; <CJK>
                 (#x5169 . #x9A8C)      ; <CJK>
                 (#x516A . #x6B83)      ; <CJK>
                 (#x516B . #x592E)      ; <CJK>
                 (#x516C . #x9E2F)      ; <CJK>
                 (#x516D . #x79E7)      ; <CJK>
                 (#x516E . #x6768)      ; <CJK>
                 (#x516F . #x626C)      ; <CJK>
                 (#x5170 . #x4F6F)      ; <CJK>
                 (#x5171 . #x75A1)      ; <CJK>
                 (#x5172 . #x7F8A)      ; <CJK>
                 (#x5173 . #x6D0B)      ; <CJK>
                 (#x5174 . #x9633)      ; <CJK>
                 (#x5175 . #x6C27)      ; <CJK>
                 (#x5176 . #x4EF0)      ; <CJK>
                 (#x5177 . #x75D2)      ; <CJK>
                 (#x5178 . #x517B)      ; <CJK>
                 (#x5179 . #x6837)      ; <CJK>
                 (#x517A . #x6F3E)      ; <CJK>
                 (#x517B . #x9080)      ; <CJK>
                 (#x517C . #x8170)      ; <CJK>
                 (#x517D . #x5996)      ; <CJK>
                 (#x517E . #x7476)      ; <CJK>
                 (#x5221 . #x6447)      ; <CJK>
                 (#x5222 . #x5C27)      ; <CJK>
                 (#x5223 . #x9065)      ; <CJK>
                 (#x5224 . #x7A91)      ; <CJK>
                 (#x5225 . #x8C23)      ; <CJK>
                 (#x5226 . #x59DA)      ; <CJK>
                 (#x5227 . #x54AC)      ; <CJK>
                 (#x5228 . #x8200)      ; <CJK>
                 (#x5229 . #x836F)      ; <CJK>
                 (#x522A . #x8981)      ; <CJK>
                 (#x522B . #x8000)      ; <CJK>
                 (#x522C . #x6930)      ; <CJK>
                 (#x522D . #x564E)      ; <CJK>
                 (#x522E . #x8036)      ; <CJK>
                 (#x522F . #x7237)      ; <CJK>
                 (#x5230 . #x91CE)      ; <CJK>
                 (#x5231 . #x51B6)      ; <CJK>
                 (#x5232 . #x4E5F)      ; <CJK>
                 (#x5233 . #x9875)      ; <CJK>
                 (#x5234 . #x6396)      ; <CJK>
                 (#x5235 . #x4E1A)      ; <CJK>
                 (#x5236 . #x53F6)      ; <CJK>
                 (#x5237 . #x66F3)      ; <CJK>
                 (#x5238 . #x814B)      ; <CJK>
                 (#x5239 . #x591C)      ; <CJK>
                 (#x523A . #x6DB2)      ; <CJK>
                 (#x523B . #x4E00)      ; <CJK>
                 (#x523C . #x58F9)      ; <CJK>
                 (#x523D . #x533B)      ; <CJK>
                 (#x523E . #x63D6)      ; <CJK>
                 (#x523F . #x94F1)      ; <CJK>
                 (#x5240 . #x4F9D)      ; <CJK>
                 (#x5241 . #x4F0A)      ; <CJK>
                 (#x5242 . #x8863)      ; <CJK>
                 (#x5243 . #x9890)      ; <CJK>
                 (#x5244 . #x5937)      ; <CJK>
                 (#x5245 . #x9057)      ; <CJK>
                 (#x5246 . #x79FB)      ; <CJK>
                 (#x5247 . #x4EEA)      ; <CJK>
                 (#x5248 . #x80F0)      ; <CJK>
                 (#x5249 . #x7591)      ; <CJK>
                 (#x524A . #x6C82)      ; <CJK>
                 (#x524B . #x5B9C)      ; <CJK>
                 (#x524C . #x59E8)      ; <CJK>
                 (#x524D . #x5F5D)      ; <CJK>
                 (#x524E . #x6905)      ; <CJK>
                 (#x524F . #x8681)      ; <CJK>
                 (#x5250 . #x501A)      ; <CJK>
                 (#x5251 . #x5DF2)      ; <CJK>
                 (#x5252 . #x4E59)      ; <CJK>
                 (#x5253 . #x77E3)      ; <CJK>
                 (#x5254 . #x4EE5)      ; <CJK>
                 (#x5255 . #x827A)      ; <CJK>
                 (#x5256 . #x6291)      ; <CJK>
                 (#x5257 . #x6613)      ; <CJK>
                 (#x5258 . #x9091)      ; <CJK>
                 (#x5259 . #x5C79)      ; <CJK>
                 (#x525A . #x4EBF)      ; <CJK>
                 (#x525B . #x5F79)      ; <CJK>
                 (#x525C . #x81C6)      ; <CJK>
                 (#x525D . #x9038)      ; <CJK>
                 (#x525E . #x8084)      ; <CJK>
                 (#x525F . #x75AB)      ; <CJK>
                 (#x5260 . #x4EA6)      ; <CJK>
                 (#x5261 . #x88D4)      ; <CJK>
                 (#x5262 . #x610F)      ; <CJK>
                 (#x5263 . #x6BC5)      ; <CJK>
                 (#x5264 . #x5FC6)      ; <CJK>
                 (#x5265 . #x4E49)      ; <CJK>
                 (#x5266 . #x76CA)      ; <CJK>
                 (#x5267 . #x6EA2)      ; <CJK>
                 (#x5268 . #x8BE3)      ; <CJK>
                 (#x5269 . #x8BAE)      ; <CJK>
                 (#x526A . #x8C0A)      ; <CJK>
                 (#x526B . #x8BD1)      ; <CJK>
                 (#x526C . #x5F02)      ; <CJK>
                 (#x526D . #x7FFC)      ; <CJK>
                 (#x526E . #x7FCC)      ; <CJK>
                 (#x526F . #x7ECE)      ; <CJK>
                 (#x5270 . #x8335)      ; <CJK>
                 (#x5271 . #x836B)      ; <CJK>
                 (#x5272 . #x56E0)      ; <CJK>
                 (#x5273 . #x6BB7)      ; <CJK>
                 (#x5274 . #x97F3)      ; <CJK>
                 (#x5275 . #x9634)      ; <CJK>
                 (#x5276 . #x59FB)      ; <CJK>
                 (#x5277 . #x541F)      ; <CJK>
                 (#x5278 . #x94F6)      ; <CJK>
                 (#x5279 . #x6DEB)      ; <CJK>
                 (#x527A . #x5BC5)      ; <CJK>
                 (#x527B . #x996E)      ; <CJK>
                 (#x527C . #x5C39)      ; <CJK>
                 (#x527D . #x5F15)      ; <CJK>
                 (#x527E . #x9690)      ; <CJK>
                 (#x5321 . #x5370)      ; <CJK>
                 (#x5322 . #x82F1)      ; <CJK>
                 (#x5323 . #x6A31)      ; <CJK>
                 (#x5324 . #x5A74)      ; <CJK>
                 (#x5325 . #x9E70)      ; <CJK>
                 (#x5326 . #x5E94)      ; <CJK>
                 (#x5327 . #x7F28)      ; <CJK>
                 (#x5328 . #x83B9)      ; <CJK>
                 (#x5329 . #x8424)      ; <CJK>
                 (#x532A . #x8425)      ; <CJK>
                 (#x532B . #x8367)      ; <CJK>
                 (#x532C . #x8747)      ; <CJK>
                 (#x532D . #x8FCE)      ; <CJK>
                 (#x532E . #x8D62)      ; <CJK>
                 (#x532F . #x76C8)      ; <CJK>
                 (#x5330 . #x5F71)      ; <CJK>
                 (#x5331 . #x9896)      ; <CJK>
                 (#x5332 . #x786C)      ; <CJK>
                 (#x5333 . #x6620)      ; <CJK>
                 (#x5334 . #x54DF)      ; <CJK>
                 (#x5335 . #x62E5)      ; <CJK>
                 (#x5336 . #x4F63)      ; <CJK>
                 (#x5337 . #x81C3)      ; <CJK>
                 (#x5338 . #x75C8)      ; <CJK>
                 (#x5339 . #x5EB8)      ; <CJK>
                 (#x533A . #x96CD)      ; <CJK>
                 (#x533B . #x8E0A)      ; <CJK>
                 (#x533C . #x86F9)      ; <CJK>
                 (#x533D . #x548F)      ; <CJK>
                 (#x533E . #x6CF3)      ; <CJK>
                 (#x533F . #x6D8C)      ; <CJK>
                 (#x5340 . #x6C38)      ; <CJK>
                 (#x5341 . #x607F)      ; <CJK>
                 (#x5342 . #x52C7)      ; <CJK>
                 (#x5343 . #x7528)      ; <CJK>
                 (#x5344 . #x5E7D)      ; <CJK>
                 (#x5345 . #x4F18)      ; <CJK>
                 (#x5346 . #x60A0)      ; <CJK>
                 (#x5347 . #x5FE7)      ; <CJK>
                 (#x5348 . #x5C24)      ; <CJK>
                 (#x5349 . #x7531)      ; <CJK>
                 (#x534A . #x90AE)      ; <CJK>
                 (#x534B . #x94C0)      ; <CJK>
                 (#x534C . #x72B9)      ; <CJK>
                 (#x534D . #x6CB9)      ; <CJK>
                 (#x534E . #x6E38)      ; <CJK>
                 (#x534F . #x9149)      ; <CJK>
                 (#x5350 . #x6709)      ; <CJK>
                 (#x5351 . #x53CB)      ; <CJK>
                 (#x5352 . #x53F3)      ; <CJK>
                 (#x5353 . #x4F51)      ; <CJK>
                 (#x5354 . #x91C9)      ; <CJK>
                 (#x5355 . #x8BF1)      ; <CJK>
                 (#x5356 . #x53C8)      ; <CJK>
                 (#x5357 . #x5E7C)      ; <CJK>
                 (#x5358 . #x8FC2)      ; <CJK>
                 (#x5359 . #x6DE4)      ; <CJK>
                 (#x535A . #x4E8E)      ; <CJK>
                 (#x535B . #x76C2)      ; <CJK>
                 (#x535C . #x6986)      ; <CJK>
                 (#x535D . #x865E)      ; <CJK>
                 (#x535E . #x611A)      ; <CJK>
                 (#x535F . #x8206)      ; <CJK>
                 (#x5360 . #x4F59)      ; <CJK>
                 (#x5361 . #x4FDE)      ; <CJK>
                 (#x5362 . #x903E)      ; <CJK>
                 (#x5363 . #x9C7C)      ; <CJK>
                 (#x5364 . #x6109)      ; <CJK>
                 (#x5365 . #x6E1D)      ; <CJK>
                 (#x5366 . #x6E14)      ; <CJK>
                 (#x5367 . #x9685)      ; <CJK>
                 (#x5368 . #x4E88)      ; <CJK>
                 (#x5369 . #x5A31)      ; <CJK>
                 (#x536A . #x96E8)      ; <CJK>
                 (#x536B . #x4E0E)      ; <CJK>
                 (#x536C . #x5C7F)      ; <CJK>
                 (#x536D . #x79B9)      ; <CJK>
                 (#x536E . #x5B87)      ; <CJK>
                 (#x536F . #x8BED)      ; <CJK>
                 (#x5370 . #x7FBD)      ; <CJK>
                 (#x5371 . #x7389)      ; <CJK>
                 (#x5372 . #x57DF)      ; <CJK>
                 (#x5373 . #x828B)      ; <CJK>
                 (#x5374 . #x90C1)      ; <CJK>
                 (#x5375 . #x5401)      ; <CJK>
                 (#x5376 . #x9047)      ; <CJK>
                 (#x5377 . #x55BB)      ; <CJK>
                 (#x5378 . #x5CEA)      ; <CJK>
                 (#x5379 . #x5FA1)      ; <CJK>
                 (#x537A . #x6108)      ; <CJK>
                 (#x537B . #x6B32)      ; <CJK>
                 (#x537C . #x72F1)      ; <CJK>
                 (#x537D . #x80B2)      ; <CJK>
                 (#x537E . #x8A89)      ; <CJK>
                 (#x5421 . #x6D74)      ; <CJK>
                 (#x5422 . #x5BD3)      ; <CJK>
                 (#x5423 . #x88D5)      ; <CJK>
                 (#x5424 . #x9884)      ; <CJK>
                 (#x5425 . #x8C6B)      ; <CJK>
                 (#x5426 . #x9A6D)      ; <CJK>
                 (#x5427 . #x9E33)      ; <CJK>
                 (#x5428 . #x6E0A)      ; <CJK>
                 (#x5429 . #x51A4)      ; <CJK>
                 (#x542A . #x5143)      ; <CJK>
                 (#x542B . #x57A3)      ; <CJK>
                 (#x542C . #x8881)      ; <CJK>
                 (#x542D . #x539F)      ; <CJK>
                 (#x542E . #x63F4)      ; <CJK>
                 (#x542F . #x8F95)      ; <CJK>
                 (#x5430 . #x56ED)      ; <CJK>
                 (#x5431 . #x5458)      ; <CJK>
                 (#x5432 . #x5706)      ; <CJK>
                 (#x5433 . #x733F)      ; <CJK>
                 (#x5434 . #x6E90)      ; <CJK>
                 (#x5435 . #x7F18)      ; <CJK>
                 (#x5436 . #x8FDC)      ; <CJK>
                 (#x5437 . #x82D1)      ; <CJK>
                 (#x5438 . #x613F)      ; <CJK>
                 (#x5439 . #x6028)      ; <CJK>
                 (#x543A . #x9662)      ; <CJK>
                 (#x543B . #x66F0)      ; <CJK>
                 (#x543C . #x7EA6)      ; <CJK>
                 (#x543D . #x8D8A)      ; <CJK>
                 (#x543E . #x8DC3)      ; <CJK>
                 (#x543F . #x94A5)      ; <CJK>
                 (#x5440 . #x5CB3)      ; <CJK>
                 (#x5441 . #x7CA4)      ; <CJK>
                 (#x5442 . #x6708)      ; <CJK>
                 (#x5443 . #x60A6)      ; <CJK>
                 (#x5444 . #x9605)      ; <CJK>
                 (#x5445 . #x8018)      ; <CJK>
                 (#x5446 . #x4E91)      ; <CJK>
                 (#x5447 . #x90E7)      ; <CJK>
                 (#x5448 . #x5300)      ; <CJK>
                 (#x5449 . #x9668)      ; <CJK>
                 (#x544A . #x5141)      ; <CJK>
                 (#x544B . #x8FD0)      ; <CJK>
                 (#x544C . #x8574)      ; <CJK>
                 (#x544D . #x915D)      ; <CJK>
                 (#x544E . #x6655)      ; <CJK>
                 (#x544F . #x97F5)      ; <CJK>
                 (#x5450 . #x5B55)      ; <CJK>
                 (#x5451 . #x531D)      ; <CJK>
                 (#x5452 . #x7838)      ; <CJK>
                 (#x5453 . #x6742)      ; <CJK>
                 (#x5454 . #x683D)      ; <CJK>
                 (#x5455 . #x54C9)      ; <CJK>
                 (#x5456 . #x707E)      ; <CJK>
                 (#x5457 . #x5BB0)      ; <CJK>
                 (#x5458 . #x8F7D)      ; <CJK>
                 (#x5459 . #x518D)      ; <CJK>
                 (#x545A . #x5728)      ; <CJK>
                 (#x545B . #x54B1)      ; <CJK>
                 (#x545C . #x6512)      ; <CJK>
                 (#x545D . #x6682)      ; <CJK>
                 (#x545E . #x8D5E)      ; <CJK>
                 (#x545F . #x8D43)      ; <CJK>
                 (#x5460 . #x810F)      ; <CJK>
                 (#x5461 . #x846C)      ; <CJK>
                 (#x5462 . #x906D)      ; <CJK>
                 (#x5463 . #x7CDF)      ; <CJK>
                 (#x5464 . #x51FF)      ; <CJK>
                 (#x5465 . #x85FB)      ; <CJK>
                 (#x5466 . #x67A3)      ; <CJK>
                 (#x5467 . #x65E9)      ; <CJK>
                 (#x5468 . #x6FA1)      ; <CJK>
                 (#x5469 . #x86A4)      ; <CJK>
                 (#x546A . #x8E81)      ; <CJK>
                 (#x546B . #x566A)      ; <CJK>
                 (#x546C . #x9020)      ; <CJK>
                 (#x546D . #x7682)      ; <CJK>
                 (#x546E . #x7076)      ; <CJK>
                 (#x546F . #x71E5)      ; <CJK>
                 (#x5470 . #x8D23)      ; <CJK>
                 (#x5471 . #x62E9)      ; <CJK>
                 (#x5472 . #x5219)      ; <CJK>
                 (#x5473 . #x6CFD)      ; <CJK>
                 (#x5474 . #x8D3C)      ; <CJK>
                 (#x5475 . #x600E)      ; <CJK>
                 (#x5476 . #x589E)      ; <CJK>
                 (#x5477 . #x618E)      ; <CJK>
                 (#x5478 . #x66FE)      ; <CJK>
                 (#x5479 . #x8D60)      ; <CJK>
                 (#x547A . #x624E)      ; <CJK>
                 (#x547B . #x55B3)      ; <CJK>
                 (#x547C . #x6E23)      ; <CJK>
                 (#x547D . #x672D)      ; <CJK>
                 (#x547E . #x8F67)      ; <CJK>
                 (#x5521 . #x94E1)      ; <CJK>
                 (#x5522 . #x95F8)      ; <CJK>
                 (#x5523 . #x7728)      ; <CJK>
                 (#x5524 . #x6805)      ; <CJK>
                 (#x5525 . #x69A8)      ; <CJK>
                 (#x5526 . #x548B)      ; <CJK>
                 (#x5527 . #x4E4D)      ; <CJK>
                 (#x5528 . #x70B8)      ; <CJK>
                 (#x5529 . #x8BC8)      ; <CJK>
                 (#x552A . #x6458)      ; <CJK>
                 (#x552B . #x658B)      ; <CJK>
                 (#x552C . #x5B85)      ; <CJK>
                 (#x552D . #x7A84)      ; <CJK>
                 (#x552E . #x503A)      ; <CJK>
                 (#x552F . #x5BE8)      ; <CJK>
                 (#x5530 . #x77BB)      ; <CJK>
                 (#x5531 . #x6BE1)      ; <CJK>
                 (#x5532 . #x8A79)      ; <CJK>
                 (#x5533 . #x7C98)      ; <CJK>
                 (#x5534 . #x6CBE)      ; <CJK>
                 (#x5535 . #x76CF)      ; <CJK>
                 (#x5536 . #x65A9)      ; <CJK>
                 (#x5537 . #x8F97)      ; <CJK>
                 (#x5538 . #x5D2D)      ; <CJK>
                 (#x5539 . #x5C55)      ; <CJK>
                 (#x553A . #x8638)      ; <CJK>
                 (#x553B . #x6808)      ; <CJK>
                 (#x553C . #x5360)      ; <CJK>
                 (#x553D . #x6218)      ; <CJK>
                 (#x553E . #x7AD9)      ; <CJK>
                 (#x553F . #x6E5B)      ; <CJK>
                 (#x5540 . #x7EFD)      ; <CJK>
                 (#x5541 . #x6A1F)      ; <CJK>
                 (#x5542 . #x7AE0)      ; <CJK>
                 (#x5543 . #x5F70)      ; <CJK>
                 (#x5544 . #x6F33)      ; <CJK>
                 (#x5545 . #x5F20)      ; <CJK>
                 (#x5546 . #x638C)      ; <CJK>
                 (#x5547 . #x6DA8)      ; <CJK>
                 (#x5548 . #x6756)      ; <CJK>
                 (#x5549 . #x4E08)      ; <CJK>
                 (#x554A . #x5E10)      ; <CJK>
                 (#x554B . #x8D26)      ; <CJK>
                 (#x554C . #x4ED7)      ; <CJK>
                 (#x554D . #x80C0)      ; <CJK>
                 (#x554E . #x7634)      ; <CJK>
                 (#x554F . #x969C)      ; <CJK>
                 (#x5550 . #x62DB)      ; <CJK>
                 (#x5551 . #x662D)      ; <CJK>
                 (#x5552 . #x627E)      ; <CJK>
                 (#x5553 . #x6CBC)      ; <CJK>
                 (#x5554 . #x8D75)      ; <CJK>
                 (#x5555 . #x7167)      ; <CJK>
                 (#x5556 . #x7F69)      ; <CJK>
                 (#x5557 . #x5146)      ; <CJK>
                 (#x5558 . #x8087)      ; <CJK>
                 (#x5559 . #x53EC)      ; <CJK>
                 (#x555A . #x906E)      ; <CJK>
                 (#x555B . #x6298)      ; <CJK>
                 (#x555C . #x54F2)      ; <CJK>
                 (#x555D . #x86F0)      ; <CJK>
                 (#x555E . #x8F99)      ; <CJK>
                 (#x555F . #x8005)      ; <CJK>
                 (#x5560 . #x9517)      ; <CJK>
                 (#x5561 . #x8517)      ; <CJK>
                 (#x5562 . #x8FD9)      ; <CJK>
                 (#x5563 . #x6D59)      ; <CJK>
                 (#x5564 . #x73CD)      ; <CJK>
                 (#x5565 . #x659F)      ; <CJK>
                 (#x5566 . #x771F)      ; <CJK>
                 (#x5567 . #x7504)      ; <CJK>
                 (#x5568 . #x7827)      ; <CJK>
                 (#x5569 . #x81FB)      ; <CJK>
                 (#x556A . #x8D1E)      ; <CJK>
                 (#x556B . #x9488)      ; <CJK>
                 (#x556C . #x4FA6)      ; <CJK>
                 (#x556D . #x6795)      ; <CJK>
                 (#x556E . #x75B9)      ; <CJK>
                 (#x556F . #x8BCA)      ; <CJK>
                 (#x5570 . #x9707)      ; <CJK>
                 (#x5571 . #x632F)      ; <CJK>
                 (#x5572 . #x9547)      ; <CJK>
                 (#x5573 . #x9635)      ; <CJK>
                 (#x5574 . #x84B8)      ; <CJK>
                 (#x5575 . #x6323)      ; <CJK>
                 (#x5576 . #x7741)      ; <CJK>
                 (#x5577 . #x5F81)      ; <CJK>
                 (#x5578 . #x72F0)      ; <CJK>
                 (#x5579 . #x4E89)      ; <CJK>
                 (#x557A . #x6014)      ; <CJK>
                 (#x557B . #x6574)      ; <CJK>
                 (#x557C . #x62EF)      ; <CJK>
                 (#x557D . #x6B63)      ; <CJK>
                 (#x557E . #x653F)      ; <CJK>
                 (#x5621 . #x5E27)      ; <CJK>
                 (#x5622 . #x75C7)      ; <CJK>
                 (#x5623 . #x90D1)      ; <CJK>
                 (#x5624 . #x8BC1)      ; <CJK>
                 (#x5625 . #x829D)      ; <CJK>
                 (#x5626 . #x679D)      ; <CJK>
                 (#x5627 . #x652F)      ; <CJK>
                 (#x5628 . #x5431)      ; <CJK>
                 (#x5629 . #x8718)      ; <CJK>
                 (#x562A . #x77E5)      ; <CJK>
                 (#x562B . #x80A2)      ; <CJK>
                 (#x562C . #x8102)      ; <CJK>
                 (#x562D . #x6C41)      ; <CJK>
                 (#x562E . #x4E4B)      ; <CJK>
                 (#x562F . #x7EC7)      ; <CJK>
                 (#x5630 . #x804C)      ; <CJK>
                 (#x5631 . #x76F4)      ; <CJK>
                 (#x5632 . #x690D)      ; <CJK>
                 (#x5633 . #x6B96)      ; <CJK>
                 (#x5634 . #x6267)      ; <CJK>
                 (#x5635 . #x503C)      ; <CJK>
                 (#x5636 . #x4F84)      ; <CJK>
                 (#x5637 . #x5740)      ; <CJK>
                 (#x5638 . #x6307)      ; <CJK>
                 (#x5639 . #x6B62)      ; <CJK>
                 (#x563A . #x8DBE)      ; <CJK>
                 (#x563B . #x53EA)      ; <CJK>
                 (#x563C . #x65E8)      ; <CJK>
                 (#x563D . #x7EB8)      ; <CJK>
                 (#x563E . #x5FD7)      ; <CJK>
                 (#x563F . #x631A)      ; <CJK>
                 (#x5640 . #x63B7)      ; <CJK>
                 (#x5641 . #x81F3)      ; <CJK>
                 (#x5642 . #x81F4)      ; <CJK>
                 (#x5643 . #x7F6E)      ; <CJK>
                 (#x5644 . #x5E1C)      ; <CJK>
                 (#x5645 . #x5CD9)      ; <CJK>
                 (#x5646 . #x5236)      ; <CJK>
                 (#x5647 . #x667A)      ; <CJK>
                 (#x5648 . #x79E9)      ; <CJK>
                 (#x5649 . #x7A1A)      ; <CJK>
                 (#x564A . #x8D28)      ; <CJK>
                 (#x564B . #x7099)      ; <CJK>
                 (#x564C . #x75D4)      ; <CJK>
                 (#x564D . #x6EDE)      ; <CJK>
                 (#x564E . #x6CBB)      ; <CJK>
                 (#x564F . #x7A92)      ; <CJK>
                 (#x5650 . #x4E2D)      ; <CJK>
                 (#x5651 . #x76C5)      ; <CJK>
                 (#x5652 . #x5FE0)      ; <CJK>
                 (#x5653 . #x949F)      ; <CJK>
                 (#x5654 . #x8877)      ; <CJK>
                 (#x5655 . #x7EC8)      ; <CJK>
                 (#x5656 . #x79CD)      ; <CJK>
                 (#x5657 . #x80BF)      ; <CJK>
                 (#x5658 . #x91CD)      ; <CJK>
                 (#x5659 . #x4EF2)      ; <CJK>
                 (#x565A . #x4F17)      ; <CJK>
                 (#x565B . #x821F)      ; <CJK>
                 (#x565C . #x5468)      ; <CJK>
                 (#x565D . #x5DDE)      ; <CJK>
                 (#x565E . #x6D32)      ; <CJK>
                 (#x565F . #x8BCC)      ; <CJK>
                 (#x5660 . #x7CA5)      ; <CJK>
                 (#x5661 . #x8F74)      ; <CJK>
                 (#x5662 . #x8098)      ; <CJK>
                 (#x5663 . #x5E1A)      ; <CJK>
                 (#x5664 . #x5492)      ; <CJK>
                 (#x5665 . #x76B1)      ; <CJK>
                 (#x5666 . #x5B99)      ; <CJK>
                 (#x5667 . #x663C)      ; <CJK>
                 (#x5668 . #x9AA4)      ; <CJK>
                 (#x5669 . #x73E0)      ; <CJK>
                 (#x566A . #x682A)      ; <CJK>
                 (#x566B . #x86DB)      ; <CJK>
                 (#x566C . #x6731)      ; <CJK>
                 (#x566D . #x732A)      ; <CJK>
                 (#x566E . #x8BF8)      ; <CJK>
                 (#x566F . #x8BDB)      ; <CJK>
                 (#x5670 . #x9010)      ; <CJK>
                 (#x5671 . #x7AF9)      ; <CJK>
                 (#x5672 . #x70DB)      ; <CJK>
                 (#x5673 . #x716E)      ; <CJK>
                 (#x5674 . #x62C4)      ; <CJK>
                 (#x5675 . #x77A9)      ; <CJK>
                 (#x5676 . #x5631)      ; <CJK>
                 (#x5677 . #x4E3B)      ; <CJK>
                 (#x5678 . #x8457)      ; <CJK>
                 (#x5679 . #x67F1)      ; <CJK>
                 (#x567A . #x52A9)      ; <CJK>
                 (#x567B . #x86C0)      ; <CJK>
                 (#x567C . #x8D2E)      ; <CJK>
                 (#x567D . #x94F8)      ; <CJK>
                 (#x567E . #x7B51)      ; <CJK>
                 (#x5721 . #x4F4F)      ; <CJK>
                 (#x5722 . #x6CE8)      ; <CJK>
                 (#x5723 . #x795D)      ; <CJK>
                 (#x5724 . #x9A7B)      ; <CJK>
                 (#x5725 . #x6293)      ; <CJK>
                 (#x5726 . #x722A)      ; <CJK>
                 (#x5727 . #x62FD)      ; <CJK>
                 (#x5728 . #x4E13)      ; <CJK>
                 (#x5729 . #x7816)      ; <CJK>
                 (#x572A . #x8F6C)      ; <CJK>
                 (#x572B . #x64B0)      ; <CJK>
                 (#x572C . #x8D5A)      ; <CJK>
                 (#x572D . #x7BC6)      ; <CJK>
                 (#x572E . #x6869)      ; <CJK>
                 (#x572F . #x5E84)      ; <CJK>
                 (#x5730 . #x88C5)      ; <CJK>
                 (#x5731 . #x5986)      ; <CJK>
                 (#x5732 . #x649E)      ; <CJK>
                 (#x5733 . #x58EE)      ; <CJK>
                 (#x5734 . #x72B6)      ; <CJK>
                 (#x5735 . #x690E)      ; <CJK>
                 (#x5736 . #x9525)      ; <CJK>
                 (#x5737 . #x8FFD)      ; <CJK>
                 (#x5738 . #x8D58)      ; <CJK>
                 (#x5739 . #x5760)      ; <CJK>
                 (#x573A . #x7F00)      ; <CJK>
                 (#x573B . #x8C06)      ; <CJK>
                 (#x573C . #x51C6)      ; <CJK>
                 (#x573D . #x6349)      ; <CJK>
                 (#x573E . #x62D9)      ; <CJK>
                 (#x573F . #x5353)      ; <CJK>
                 (#x5740 . #x684C)      ; <CJK>
                 (#x5741 . #x7422)      ; <CJK>
                 (#x5742 . #x8301)      ; <CJK>
                 (#x5743 . #x914C)      ; <CJK>
                 (#x5744 . #x5544)      ; <CJK>
                 (#x5745 . #x7740)      ; <CJK>
                 (#x5746 . #x707C)      ; <CJK>
                 (#x5747 . #x6D4A)      ; <CJK>
                 (#x5748 . #x5179)      ; <CJK>
                 (#x5749 . #x54A8)      ; <CJK>
                 (#x574A . #x8D44)      ; <CJK>
                 (#x574B . #x59FF)      ; <CJK>
                 (#x574C . #x6ECB)      ; <CJK>
                 (#x574D . #x6DC4)      ; <CJK>
                 (#x574E . #x5B5C)      ; <CJK>
                 (#x574F . #x7D2B)      ; <CJK>
                 (#x5750 . #x4ED4)      ; <CJK>
                 (#x5751 . #x7C7D)      ; <CJK>
                 (#x5752 . #x6ED3)      ; <CJK>
                 (#x5753 . #x5B50)      ; <CJK>
                 (#x5754 . #x81EA)      ; <CJK>
                 (#x5755 . #x6E0D)      ; <CJK>
                 (#x5756 . #x5B57)      ; <CJK>
                 (#x5757 . #x9B03)      ; <CJK>
                 (#x5758 . #x68D5)      ; <CJK>
                 (#x5759 . #x8E2A)      ; <CJK>
                 (#x575A . #x5B97)      ; <CJK>
                 (#x575B . #x7EFC)      ; <CJK>
                 (#x575C . #x603B)      ; <CJK>
                 (#x575D . #x7EB5)      ; <CJK>
                 (#x575E . #x90B9)      ; <CJK>
                 (#x575F . #x8D70)      ; <CJK>
                 (#x5760 . #x594F)      ; <CJK>
                 (#x5761 . #x63CD)      ; <CJK>
                 (#x5762 . #x79DF)      ; <CJK>
                 (#x5763 . #x8DB3)      ; <CJK>
                 (#x5764 . #x5352)      ; <CJK>
                 (#x5765 . #x65CF)      ; <CJK>
                 (#x5766 . #x7956)      ; <CJK>
                 (#x5767 . #x8BC5)      ; <CJK>
                 (#x5768 . #x963B)      ; <CJK>
                 (#x5769 . #x7EC4)      ; <CJK>
                 (#x576A . #x94BB)      ; <CJK>
                 (#x576B . #x7E82)      ; <CJK>
                 (#x576C . #x5634)      ; <CJK>
                 (#x576D . #x9189)      ; <CJK>
                 (#x576E . #x6700)      ; <CJK>
                 (#x576F . #x7F6A)      ; <CJK>
                 (#x5770 . #x5C0A)      ; <CJK>
                 (#x5771 . #x9075)      ; <CJK>
                 (#x5772 . #x6628)      ; <CJK>
                 (#x5773 . #x5DE6)      ; <CJK>
                 (#x5774 . #x4F50)      ; <CJK>
                 (#x5775 . #x67DE)      ; <CJK>
                 (#x5776 . #x505A)      ; <CJK>
                 (#x5777 . #x4F5C)      ; <CJK>
                 (#x5778 . #x5750)      ; <CJK>
                 (#x5779 . #x5EA7)      ; <CJK>
                 (#x5821 . #x4E8D)      ; <CJK>
                 (#x5822 . #x4E0C)      ; <CJK>
                 (#x5823 . #x5140)      ; <CJK>
                 (#x5824 . #x4E10)      ; <CJK>
                 (#x5825 . #x5EFF)      ; <CJK>
                 (#x5826 . #x5345)      ; <CJK>
                 (#x5827 . #x4E15)      ; <CJK>
                 (#x5828 . #x4E98)      ; <CJK>
                 (#x5829 . #x4E1E)      ; <CJK>
                 (#x582A . #x9B32)      ; <CJK>
                 (#x582B . #x5B6C)      ; <CJK>
                 (#x582C . #x5669)      ; <CJK>
                 (#x582D . #x4E28)      ; <CJK>
                 (#x582E . #x79BA)      ; <CJK>
                 (#x582F . #x4E3F)      ; <CJK>
                 (#x5830 . #x5315)      ; <CJK>
                 (#x5831 . #x4E47)      ; <CJK>
                 (#x5832 . #x592D)      ; <CJK>
                 (#x5833 . #x723B)      ; <CJK>
                 (#x5834 . #x536E)      ; <CJK>
                 (#x5835 . #x6C10)      ; <CJK>
                 (#x5836 . #x56DF)      ; <CJK>
                 (#x5837 . #x80E4)      ; <CJK>
                 (#x5838 . #x9997)      ; <CJK>
                 (#x5839 . #x6BD3)      ; <CJK>
                 (#x583A . #x777E)      ; <CJK>
                 (#x583B . #x9F17)      ; <CJK>
                 (#x583C . #x4E36)      ; <CJK>
                 (#x583D . #x4E9F)      ; <CJK>
                 (#x583E . #x9F10)      ; <CJK>
                 (#x583F . #x4E5C)      ; <CJK>
                 (#x5840 . #x4E69)      ; <CJK>
                 (#x5841 . #x4E93)      ; <CJK>
                 (#x5842 . #x8288)      ; <CJK>
                 (#x5843 . #x5B5B)      ; <CJK>
                 (#x5844 . #x556C)      ; <CJK>
                 (#x5845 . #x560F)      ; <CJK>
                 (#x5846 . #x4EC4)      ; <CJK>
                 (#x5847 . #x538D)      ; <CJK>
                 (#x5848 . #x539D)      ; <CJK>
                 (#x5849 . #x53A3)      ; <CJK>
                 (#x584A . #x53A5)      ; <CJK>
                 (#x584B . #x53AE)      ; <CJK>
                 (#x584C . #x9765)      ; <CJK>
                 (#x584D . #x8D5D)      ; <CJK>
                 (#x584E . #x531A)      ; <CJK>
                 (#x584F . #x53F5)      ; <CJK>
                 (#x5850 . #x5326)      ; <CJK>
                 (#x5851 . #x532E)      ; <CJK>
                 (#x5852 . #x533E)      ; <CJK>
                 (#x5853 . #x8D5C)      ; <CJK>
                 (#x5854 . #x5366)      ; <CJK>
                 (#x5855 . #x5363)      ; <CJK>
                 (#x5856 . #x5202)      ; <CJK>
                 (#x5857 . #x5208)      ; <CJK>
                 (#x5858 . #x520E)      ; <CJK>
                 (#x5859 . #x522D)      ; <CJK>
                 (#x585A . #x5233)      ; <CJK>
                 (#x585B . #x523F)      ; <CJK>
                 (#x585C . #x5240)      ; <CJK>
                 (#x585D . #x524C)      ; <CJK>
                 (#x585E . #x525E)      ; <CJK>
                 (#x585F . #x5261)      ; <CJK>
                 (#x5860 . #x525C)      ; <CJK>
                 (#x5861 . #x84AF)      ; <CJK>
                 (#x5862 . #x527D)      ; <CJK>
                 (#x5863 . #x5282)      ; <CJK>
                 (#x5864 . #x5281)      ; <CJK>
                 (#x5865 . #x5290)      ; <CJK>
                 (#x5866 . #x5293)      ; <CJK>
                 (#x5867 . #x5182)      ; <CJK>
                 (#x5868 . #x7F54)      ; <CJK>
                 (#x5869 . #x4EBB)      ; <CJK>
                 (#x586A . #x4EC3)      ; <CJK>
                 (#x586B . #x4EC9)      ; <CJK>
                 (#x586C . #x4EC2)      ; <CJK>
                 (#x586D . #x4EE8)      ; <CJK>
                 (#x586E . #x4EE1)      ; <CJK>
                 (#x586F . #x4EEB)      ; <CJK>
                 (#x5870 . #x4EDE)      ; <CJK>
                 (#x5871 . #x4F1B)      ; <CJK>
                 (#x5872 . #x4EF3)      ; <CJK>
                 (#x5873 . #x4F22)      ; <CJK>
                 (#x5874 . #x4F64)      ; <CJK>
                 (#x5875 . #x4EF5)      ; <CJK>
                 (#x5876 . #x4F25)      ; <CJK>
                 (#x5877 . #x4F27)      ; <CJK>
                 (#x5878 . #x4F09)      ; <CJK>
                 (#x5879 . #x4F2B)      ; <CJK>
                 (#x587A . #x4F5E)      ; <CJK>
                 (#x587B . #x4F67)      ; <CJK>
                 (#x587C . #x6538)      ; <CJK>
                 (#x587D . #x4F5A)      ; <CJK>
                 (#x587E . #x4F5D)      ; <CJK>
                 (#x5921 . #x4F5F)      ; <CJK>
                 (#x5922 . #x4F57)      ; <CJK>
                 (#x5923 . #x4F32)      ; <CJK>
                 (#x5924 . #x4F3D)      ; <CJK>
                 (#x5925 . #x4F76)      ; <CJK>
                 (#x5926 . #x4F74)      ; <CJK>
                 (#x5927 . #x4F91)      ; <CJK>
                 (#x5928 . #x4F89)      ; <CJK>
                 (#x5929 . #x4F83)      ; <CJK>
                 (#x592A . #x4F8F)      ; <CJK>
                 (#x592B . #x4F7E)      ; <CJK>
                 (#x592C . #x4F7B)      ; <CJK>
                 (#x592D . #x4FAA)      ; <CJK>
                 (#x592E . #x4F7C)      ; <CJK>
                 (#x592F . #x4FAC)      ; <CJK>
                 (#x5930 . #x4F94)      ; <CJK>
                 (#x5931 . #x4FE6)      ; <CJK>
                 (#x5932 . #x4FE8)      ; <CJK>
                 (#x5933 . #x4FEA)      ; <CJK>
                 (#x5934 . #x4FC5)      ; <CJK>
                 (#x5935 . #x4FDA)      ; <CJK>
                 (#x5936 . #x4FE3)      ; <CJK>
                 (#x5937 . #x4FDC)      ; <CJK>
                 (#x5938 . #x4FD1)      ; <CJK>
                 (#x5939 . #x4FDF)      ; <CJK>
                 (#x593A . #x4FF8)      ; <CJK>
                 (#x593B . #x5029)      ; <CJK>
                 (#x593C . #x504C)      ; <CJK>
                 (#x593D . #x4FF3)      ; <CJK>
                 (#x593E . #x502C)      ; <CJK>
                 (#x593F . #x500F)      ; <CJK>
                 (#x5940 . #x502E)      ; <CJK>
                 (#x5941 . #x502D)      ; <CJK>
                 (#x5942 . #x4FFE)      ; <CJK>
                 (#x5943 . #x501C)      ; <CJK>
                 (#x5944 . #x500C)      ; <CJK>
                 (#x5945 . #x5025)      ; <CJK>
                 (#x5946 . #x5028)      ; <CJK>
                 (#x5947 . #x507E)      ; <CJK>
                 (#x5948 . #x5043)      ; <CJK>
                 (#x5949 . #x5055)      ; <CJK>
                 (#x594A . #x5048)      ; <CJK>
                 (#x594B . #x504E)      ; <CJK>
                 (#x594C . #x506C)      ; <CJK>
                 (#x594D . #x507B)      ; <CJK>
                 (#x594E . #x50A5)      ; <CJK>
                 (#x594F . #x50A7)      ; <CJK>
                 (#x5950 . #x50A9)      ; <CJK>
                 (#x5951 . #x50BA)      ; <CJK>
                 (#x5952 . #x50D6)      ; <CJK>
                 (#x5953 . #x5106)      ; <CJK>
                 (#x5954 . #x50ED)      ; <CJK>
                 (#x5955 . #x50EC)      ; <CJK>
                 (#x5956 . #x50E6)      ; <CJK>
                 (#x5957 . #x50EE)      ; <CJK>
                 (#x5958 . #x5107)      ; <CJK>
                 (#x5959 . #x510B)      ; <CJK>
                 (#x595A . #x4EDD)      ; <CJK>
                 (#x595B . #x6C3D)      ; <CJK>
                 (#x595C . #x4F58)      ; <CJK>
                 (#x595D . #x4F65)      ; <CJK>
                 (#x595E . #x4FCE)      ; <CJK>
                 (#x595F . #x9FA0)      ; <CJK>
                 (#x5960 . #x6C46)      ; <CJK>
                 (#x5961 . #x7C74)      ; <CJK>
                 (#x5962 . #x516E)      ; <CJK>
                 (#x5963 . #x5DFD)      ; <CJK>
                 (#x5964 . #x9EC9)      ; <CJK>
                 (#x5965 . #x9998)      ; <CJK>
                 (#x5966 . #x5181)      ; <CJK>
                 (#x5967 . #x5914)      ; <CJK>
                 (#x5968 . #x52F9)      ; <CJK>
                 (#x5969 . #x530D)      ; <CJK>
                 (#x596A . #x8A07)      ; <CJK>
                 (#x596B . #x5310)      ; <CJK>
                 (#x596C . #x51EB)      ; <CJK>
                 (#x596D . #x5919)      ; <CJK>
                 (#x596E . #x5155)      ; <CJK>
                 (#x596F . #x4EA0)      ; <CJK>
                 (#x5970 . #x5156)      ; <CJK>
                 (#x5971 . #x4EB3)      ; <CJK>
                 (#x5972 . #x886E)      ; <CJK>
                 (#x5973 . #x88A4)      ; <CJK>
                 (#x5974 . #x4EB5)      ; <CJK>
                 (#x5975 . #x8114)      ; <CJK>
                 (#x5976 . #x88D2)      ; <CJK>
                 (#x5977 . #x7980)      ; <CJK>
                 (#x5978 . #x5B34)      ; <CJK>
                 (#x5979 . #x8803)      ; <CJK>
                 (#x597A . #x7FB8)      ; <CJK>
                 (#x597B . #x51AB)      ; <CJK>
                 (#x597C . #x51B1)      ; <CJK>
                 (#x597D . #x51BD)      ; <CJK>
                 (#x597E . #x51BC)      ; <CJK>
                 (#x5A21 . #x51C7)      ; <CJK>
                 (#x5A22 . #x5196)      ; <CJK>
                 (#x5A23 . #x51A2)      ; <CJK>
                 (#x5A24 . #x51A5)      ; <CJK>
                 (#x5A25 . #x8BA0)      ; <CJK>
                 (#x5A26 . #x8BA6)      ; <CJK>
                 (#x5A27 . #x8BA7)      ; <CJK>
                 (#x5A28 . #x8BAA)      ; <CJK>
                 (#x5A29 . #x8BB4)      ; <CJK>
                 (#x5A2A . #x8BB5)      ; <CJK>
                 (#x5A2B . #x8BB7)      ; <CJK>
                 (#x5A2C . #x8BC2)      ; <CJK>
                 (#x5A2D . #x8BC3)      ; <CJK>
                 (#x5A2E . #x8BCB)      ; <CJK>
                 (#x5A2F . #x8BCF)      ; <CJK>
                 (#x5A30 . #x8BCE)      ; <CJK>
                 (#x5A31 . #x8BD2)      ; <CJK>
                 (#x5A32 . #x8BD3)      ; <CJK>
                 (#x5A33 . #x8BD4)      ; <CJK>
                 (#x5A34 . #x8BD6)      ; <CJK>
                 (#x5A35 . #x8BD8)      ; <CJK>
                 (#x5A36 . #x8BD9)      ; <CJK>
                 (#x5A37 . #x8BDC)      ; <CJK>
                 (#x5A38 . #x8BDF)      ; <CJK>
                 (#x5A39 . #x8BE0)      ; <CJK>
                 (#x5A3A . #x8BE4)      ; <CJK>
                 (#x5A3B . #x8BE8)      ; <CJK>
                 (#x5A3C . #x8BE9)      ; <CJK>
                 (#x5A3D . #x8BEE)      ; <CJK>
                 (#x5A3E . #x8BF0)      ; <CJK>
                 (#x5A3F . #x8BF3)      ; <CJK>
                 (#x5A40 . #x8BF6)      ; <CJK>
                 (#x5A41 . #x8BF9)      ; <CJK>
                 (#x5A42 . #x8BFC)      ; <CJK>
                 (#x5A43 . #x8BFF)      ; <CJK>
                 (#x5A44 . #x8C00)      ; <CJK>
                 (#x5A45 . #x8C02)      ; <CJK>
                 (#x5A46 . #x8C04)      ; <CJK>
                 (#x5A47 . #x8C07)      ; <CJK>
                 (#x5A48 . #x8C0C)      ; <CJK>
                 (#x5A49 . #x8C0F)      ; <CJK>
                 (#x5A4A . #x8C11)      ; <CJK>
                 (#x5A4B . #x8C12)      ; <CJK>
                 (#x5A4C . #x8C14)      ; <CJK>
                 (#x5A4D . #x8C15)      ; <CJK>
                 (#x5A4E . #x8C16)      ; <CJK>
                 (#x5A4F . #x8C19)      ; <CJK>
                 (#x5A50 . #x8C1B)      ; <CJK>
                 (#x5A51 . #x8C18)      ; <CJK>
                 (#x5A52 . #x8C1D)      ; <CJK>
                 (#x5A53 . #x8C1F)      ; <CJK>
                 (#x5A54 . #x8C20)      ; <CJK>
                 (#x5A55 . #x8C21)      ; <CJK>
                 (#x5A56 . #x8C25)      ; <CJK>
                 (#x5A57 . #x8C27)      ; <CJK>
                 (#x5A58 . #x8C2A)      ; <CJK>
                 (#x5A59 . #x8C2B)      ; <CJK>
                 (#x5A5A . #x8C2E)      ; <CJK>
                 (#x5A5B . #x8C2F)      ; <CJK>
                 (#x5A5C . #x8C32)      ; <CJK>
                 (#x5A5D . #x8C33)      ; <CJK>
                 (#x5A5E . #x8C35)      ; <CJK>
                 (#x5A5F . #x8C36)      ; <CJK>
                 (#x5A60 . #x5369)      ; <CJK>
                 (#x5A61 . #x537A)      ; <CJK>
                 (#x5A62 . #x961D)      ; <CJK>
                 (#x5A63 . #x9622)      ; <CJK>
                 (#x5A64 . #x9621)      ; <CJK>
                 (#x5A65 . #x9631)      ; <CJK>
                 (#x5A66 . #x962A)      ; <CJK>
                 (#x5A67 . #x963D)      ; <CJK>
                 (#x5A68 . #x963C)      ; <CJK>
                 (#x5A69 . #x9642)      ; <CJK>
                 (#x5A6A . #x9649)      ; <CJK>
                 (#x5A6B . #x9654)      ; <CJK>
                 (#x5A6C . #x965F)      ; <CJK>
                 (#x5A6D . #x9667)      ; <CJK>
                 (#x5A6E . #x966C)      ; <CJK>
                 (#x5A6F . #x9672)      ; <CJK>
                 (#x5A70 . #x9674)      ; <CJK>
                 (#x5A71 . #x9688)      ; <CJK>
                 (#x5A72 . #x968D)      ; <CJK>
                 (#x5A73 . #x9697)      ; <CJK>
                 (#x5A74 . #x96B0)      ; <CJK>
                 (#x5A75 . #x9097)      ; <CJK>
                 (#x5A76 . #x909B)      ; <CJK>
                 (#x5A77 . #x909D)      ; <CJK>
                 (#x5A78 . #x9099)      ; <CJK>
                 (#x5A79 . #x90AC)      ; <CJK>
                 (#x5A7A . #x90A1)      ; <CJK>
                 (#x5A7B . #x90B4)      ; <CJK>
                 (#x5A7C . #x90B3)      ; <CJK>
                 (#x5A7D . #x90B6)      ; <CJK>
                 (#x5A7E . #x90BA)      ; <CJK>
                 (#x5B21 . #x90B8)      ; <CJK>
                 (#x5B22 . #x90B0)      ; <CJK>
                 (#x5B23 . #x90CF)      ; <CJK>
                 (#x5B24 . #x90C5)      ; <CJK>
                 (#x5B25 . #x90BE)      ; <CJK>
                 (#x5B26 . #x90D0)      ; <CJK>
                 (#x5B27 . #x90C4)      ; <CJK>
                 (#x5B28 . #x90C7)      ; <CJK>
                 (#x5B29 . #x90D3)      ; <CJK>
                 (#x5B2A . #x90E6)      ; <CJK>
                 (#x5B2B . #x90E2)      ; <CJK>
                 (#x5B2C . #x90DC)      ; <CJK>
                 (#x5B2D . #x90D7)      ; <CJK>
                 (#x5B2E . #x90DB)      ; <CJK>
                 (#x5B2F . #x90EB)      ; <CJK>
                 (#x5B30 . #x90EF)      ; <CJK>
                 (#x5B31 . #x90FE)      ; <CJK>
                 (#x5B32 . #x9104)      ; <CJK>
                 (#x5B33 . #x9122)      ; <CJK>
                 (#x5B34 . #x911E)      ; <CJK>
                 (#x5B35 . #x9123)      ; <CJK>
                 (#x5B36 . #x9131)      ; <CJK>
                 (#x5B37 . #x912F)      ; <CJK>
                 (#x5B38 . #x9139)      ; <CJK>
                 (#x5B39 . #x9143)      ; <CJK>
                 (#x5B3A . #x9146)      ; <CJK>
                 (#x5B3B . #x520D)      ; <CJK>
                 (#x5B3C . #x5942)      ; <CJK>
                 (#x5B3D . #x52A2)      ; <CJK>
                 (#x5B3E . #x52AC)      ; <CJK>
                 (#x5B3F . #x52AD)      ; <CJK>
                 (#x5B40 . #x52BE)      ; <CJK>
                 (#x5B41 . #x54FF)      ; <CJK>
                 (#x5B42 . #x52D0)      ; <CJK>
                 (#x5B43 . #x52D6)      ; <CJK>
                 (#x5B44 . #x52F0)      ; <CJK>
                 (#x5B45 . #x53DF)      ; <CJK>
                 (#x5B46 . #x71EE)      ; <CJK>
                 (#x5B47 . #x77CD)      ; <CJK>
                 (#x5B48 . #x5EF4)      ; <CJK>
                 (#x5B49 . #x51F5)      ; <CJK>
                 (#x5B4A . #x51FC)      ; <CJK>
                 (#x5B4B . #x9B2F)      ; <CJK>
                 (#x5B4C . #x53B6)      ; <CJK>
                 (#x5B4D . #x5F01)      ; <CJK>
                 (#x5B4E . #x755A)      ; <CJK>
                 (#x5B4F . #x5DEF)      ; <CJK>
                 (#x5B50 . #x574C)      ; <CJK>
                 (#x5B51 . #x57A9)      ; <CJK>
                 (#x5B52 . #x57A1)      ; <CJK>
                 (#x5B53 . #x587E)      ; <CJK>
                 (#x5B54 . #x58BC)      ; <CJK>
                 (#x5B55 . #x58C5)      ; <CJK>
                 (#x5B56 . #x58D1)      ; <CJK>
                 (#x5B57 . #x5729)      ; <CJK>
                 (#x5B58 . #x572C)      ; <CJK>
                 (#x5B59 . #x572A)      ; <CJK>
                 (#x5B5A . #x5733)      ; <CJK>
                 (#x5B5B . #x5739)      ; <CJK>
                 (#x5B5C . #x572E)      ; <CJK>
                 (#x5B5D . #x572F)      ; <CJK>
                 (#x5B5E . #x575C)      ; <CJK>
                 (#x5B5F . #x573B)      ; <CJK>
                 (#x5B60 . #x5742)      ; <CJK>
                 (#x5B61 . #x5769)      ; <CJK>
                 (#x5B62 . #x5785)      ; <CJK>
                 (#x5B63 . #x576B)      ; <CJK>
                 (#x5B64 . #x5786)      ; <CJK>
                 (#x5B65 . #x577C)      ; <CJK>
                 (#x5B66 . #x577B)      ; <CJK>
                 (#x5B67 . #x5768)      ; <CJK>
                 (#x5B68 . #x576D)      ; <CJK>
                 (#x5B69 . #x5776)      ; <CJK>
                 (#x5B6A . #x5773)      ; <CJK>
                 (#x5B6B . #x57AD)      ; <CJK>
                 (#x5B6C . #x57A4)      ; <CJK>
                 (#x5B6D . #x578C)      ; <CJK>
                 (#x5B6E . #x57B2)      ; <CJK>
                 (#x5B6F . #x57CF)      ; <CJK>
                 (#x5B70 . #x57A7)      ; <CJK>
                 (#x5B71 . #x57B4)      ; <CJK>
                 (#x5B72 . #x5793)      ; <CJK>
                 (#x5B73 . #x57A0)      ; <CJK>
                 (#x5B74 . #x57D5)      ; <CJK>
                 (#x5B75 . #x57D8)      ; <CJK>
                 (#x5B76 . #x57DA)      ; <CJK>
                 (#x5B77 . #x57D9)      ; <CJK>
                 (#x5B78 . #x57D2)      ; <CJK>
                 (#x5B79 . #x57B8)      ; <CJK>
                 (#x5B7A . #x57F4)      ; <CJK>
                 (#x5B7B . #x57EF)      ; <CJK>
                 (#x5B7C . #x57F8)      ; <CJK>
                 (#x5B7D . #x57E4)      ; <CJK>
                 (#x5B7E . #x57DD)      ; <CJK>
                 (#x5C21 . #x580B)      ; <CJK>
                 (#x5C22 . #x580D)      ; <CJK>
                 (#x5C23 . #x57FD)      ; <CJK>
                 (#x5C24 . #x57ED)      ; <CJK>
                 (#x5C25 . #x5800)      ; <CJK>
                 (#x5C26 . #x581E)      ; <CJK>
                 (#x5C27 . #x5819)      ; <CJK>
                 (#x5C28 . #x5844)      ; <CJK>
                 (#x5C29 . #x5820)      ; <CJK>
                 (#x5C2A . #x5865)      ; <CJK>
                 (#x5C2B . #x586C)      ; <CJK>
                 (#x5C2C . #x5881)      ; <CJK>
                 (#x5C2D . #x5889)      ; <CJK>
                 (#x5C2E . #x589A)      ; <CJK>
                 (#x5C2F . #x5880)      ; <CJK>
                 (#x5C30 . #x99A8)      ; <CJK>
                 (#x5C31 . #x9F19)      ; <CJK>
                 (#x5C32 . #x61FF)      ; <CJK>
                 (#x5C33 . #x8279)      ; <CJK>
                 (#x5C34 . #x827D)      ; <CJK>
                 (#x5C35 . #x827F)      ; <CJK>
                 (#x5C36 . #x828F)      ; <CJK>
                 (#x5C37 . #x828A)      ; <CJK>
                 (#x5C38 . #x82A8)      ; <CJK>
                 (#x5C39 . #x8284)      ; <CJK>
                 (#x5C3A . #x828E)      ; <CJK>
                 (#x5C3B . #x8291)      ; <CJK>
                 (#x5C3C . #x8297)      ; <CJK>
                 (#x5C3D . #x8299)      ; <CJK>
                 (#x5C3E . #x82AB)      ; <CJK>
                 (#x5C3F . #x82B8)      ; <CJK>
                 (#x5C40 . #x82BE)      ; <CJK>
                 (#x5C41 . #x82B0)      ; <CJK>
                 (#x5C42 . #x82C8)      ; <CJK>
                 (#x5C43 . #x82CA)      ; <CJK>
                 (#x5C44 . #x82E3)      ; <CJK>
                 (#x5C45 . #x8298)      ; <CJK>
                 (#x5C46 . #x82B7)      ; <CJK>
                 (#x5C47 . #x82AE)      ; <CJK>
                 (#x5C48 . #x82CB)      ; <CJK>
                 (#x5C49 . #x82CC)      ; <CJK>
                 (#x5C4A . #x82C1)      ; <CJK>
                 (#x5C4B . #x82A9)      ; <CJK>
                 (#x5C4C . #x82B4)      ; <CJK>
                 (#x5C4D . #x82A1)      ; <CJK>
                 (#x5C4E . #x82AA)      ; <CJK>
                 (#x5C4F . #x829F)      ; <CJK>
                 (#x5C50 . #x82C4)      ; <CJK>
                 (#x5C51 . #x82CE)      ; <CJK>
                 (#x5C52 . #x82A4)      ; <CJK>
                 (#x5C53 . #x82E1)      ; <CJK>
                 (#x5C54 . #x8309)      ; <CJK>
                 (#x5C55 . #x82F7)      ; <CJK>
                 (#x5C56 . #x82E4)      ; <CJK>
                 (#x5C57 . #x830F)      ; <CJK>
                 (#x5C58 . #x8307)      ; <CJK>
                 (#x5C59 . #x82DC)      ; <CJK>
                 (#x5C5A . #x82F4)      ; <CJK>
                 (#x5C5B . #x82D2)      ; <CJK>
                 (#x5C5C . #x82D8)      ; <CJK>
                 (#x5C5D . #x830C)      ; <CJK>
                 (#x5C5E . #x82FB)      ; <CJK>
                 (#x5C5F . #x82D3)      ; <CJK>
                 (#x5C60 . #x8311)      ; <CJK>
                 (#x5C61 . #x831A)      ; <CJK>
                 (#x5C62 . #x8306)      ; <CJK>
                 (#x5C63 . #x8314)      ; <CJK>
                 (#x5C64 . #x8315)      ; <CJK>
                 (#x5C65 . #x82E0)      ; <CJK>
                 (#x5C66 . #x82D5)      ; <CJK>
                 (#x5C67 . #x831C)      ; <CJK>
                 (#x5C68 . #x8351)      ; <CJK>
                 (#x5C69 . #x835B)      ; <CJK>
                 (#x5C6A . #x835C)      ; <CJK>
                 (#x5C6B . #x8308)      ; <CJK>
                 (#x5C6C . #x8392)      ; <CJK>
                 (#x5C6D . #x833C)      ; <CJK>
                 (#x5C6E . #x8334)      ; <CJK>
                 (#x5C6F . #x8331)      ; <CJK>
                 (#x5C70 . #x839B)      ; <CJK>
                 (#x5C71 . #x835E)      ; <CJK>
                 (#x5C72 . #x832F)      ; <CJK>
                 (#x5C73 . #x834F)      ; <CJK>
                 (#x5C74 . #x8347)      ; <CJK>
                 (#x5C75 . #x8343)      ; <CJK>
                 (#x5C76 . #x835F)      ; <CJK>
                 (#x5C77 . #x8340)      ; <CJK>
                 (#x5C78 . #x8317)      ; <CJK>
                 (#x5C79 . #x8360)      ; <CJK>
                 (#x5C7A . #x832D)      ; <CJK>
                 (#x5C7B . #x833A)      ; <CJK>
                 (#x5C7C . #x8333)      ; <CJK>
                 (#x5C7D . #x8366)      ; <CJK>
                 (#x5C7E . #x8365)      ; <CJK>
                 (#x5D21 . #x8368)      ; <CJK>
                 (#x5D22 . #x831B)      ; <CJK>
                 (#x5D23 . #x8369)      ; <CJK>
                 (#x5D24 . #x836C)      ; <CJK>
                 (#x5D25 . #x836A)      ; <CJK>
                 (#x5D26 . #x836D)      ; <CJK>
                 (#x5D27 . #x836E)      ; <CJK>
                 (#x5D28 . #x83B0)      ; <CJK>
                 (#x5D29 . #x8378)      ; <CJK>
                 (#x5D2A . #x83B3)      ; <CJK>
                 (#x5D2B . #x83B4)      ; <CJK>
                 (#x5D2C . #x83A0)      ; <CJK>
                 (#x5D2D . #x83AA)      ; <CJK>
                 (#x5D2E . #x8393)      ; <CJK>
                 (#x5D2F . #x839C)      ; <CJK>
                 (#x5D30 . #x8385)      ; <CJK>
                 (#x5D31 . #x837C)      ; <CJK>
                 (#x5D32 . #x83B6)      ; <CJK>
                 (#x5D33 . #x83A9)      ; <CJK>
                 (#x5D34 . #x837D)      ; <CJK>
                 (#x5D35 . #x83B8)      ; <CJK>
                 (#x5D36 . #x837B)      ; <CJK>
                 (#x5D37 . #x8398)      ; <CJK>
                 (#x5D38 . #x839E)      ; <CJK>
                 (#x5D39 . #x83A8)      ; <CJK>
                 (#x5D3A . #x83BA)      ; <CJK>
                 (#x5D3B . #x83BC)      ; <CJK>
                 (#x5D3C . #x83C1)      ; <CJK>
                 (#x5D3D . #x8401)      ; <CJK>
                 (#x5D3E . #x83E5)      ; <CJK>
                 (#x5D3F . #x83D8)      ; <CJK>
                 (#x5D40 . #x5807)      ; <CJK>
                 (#x5D41 . #x8418)      ; <CJK>
                 (#x5D42 . #x840B)      ; <CJK>
                 (#x5D43 . #x83DD)      ; <CJK>
                 (#x5D44 . #x83FD)      ; <CJK>
                 (#x5D45 . #x83D6)      ; <CJK>
                 (#x5D46 . #x841C)      ; <CJK>
                 (#x5D47 . #x8438)      ; <CJK>
                 (#x5D48 . #x8411)      ; <CJK>
                 (#x5D49 . #x8406)      ; <CJK>
                 (#x5D4A . #x83D4)      ; <CJK>
                 (#x5D4B . #x83DF)      ; <CJK>
                 (#x5D4C . #x840F)      ; <CJK>
                 (#x5D4D . #x8403)      ; <CJK>
                 (#x5D4E . #x83F8)      ; <CJK>
                 (#x5D4F . #x83F9)      ; <CJK>
                 (#x5D50 . #x83EA)      ; <CJK>
                 (#x5D51 . #x83C5)      ; <CJK>
                 (#x5D52 . #x83C0)      ; <CJK>
                 (#x5D53 . #x8426)      ; <CJK>
                 (#x5D54 . #x83F0)      ; <CJK>
                 (#x5D55 . #x83E1)      ; <CJK>
                 (#x5D56 . #x845C)      ; <CJK>
                 (#x5D57 . #x8451)      ; <CJK>
                 (#x5D58 . #x845A)      ; <CJK>
                 (#x5D59 . #x8459)      ; <CJK>
                 (#x5D5A . #x8473)      ; <CJK>
                 (#x5D5B . #x8487)      ; <CJK>
                 (#x5D5C . #x8488)      ; <CJK>
                 (#x5D5D . #x847A)      ; <CJK>
                 (#x5D5E . #x8489)      ; <CJK>
                 (#x5D5F . #x8478)      ; <CJK>
                 (#x5D60 . #x843C)      ; <CJK>
                 (#x5D61 . #x8446)      ; <CJK>
                 (#x5D62 . #x8469)      ; <CJK>
                 (#x5D63 . #x8476)      ; <CJK>
                 (#x5D64 . #x848C)      ; <CJK>
                 (#x5D65 . #x848E)      ; <CJK>
                 (#x5D66 . #x8431)      ; <CJK>
                 (#x5D67 . #x846D)      ; <CJK>
                 (#x5D68 . #x84C1)      ; <CJK>
                 (#x5D69 . #x84CD)      ; <CJK>
                 (#x5D6A . #x84D0)      ; <CJK>
                 (#x5D6B . #x84E6)      ; <CJK>
                 (#x5D6C . #x84BD)      ; <CJK>
                 (#x5D6D . #x84D3)      ; <CJK>
                 (#x5D6E . #x84CA)      ; <CJK>
                 (#x5D6F . #x84BF)      ; <CJK>
                 (#x5D70 . #x84BA)      ; <CJK>
                 (#x5D71 . #x84E0)      ; <CJK>
                 (#x5D72 . #x84A1)      ; <CJK>
                 (#x5D73 . #x84B9)      ; <CJK>
                 (#x5D74 . #x84B4)      ; <CJK>
                 (#x5D75 . #x8497)      ; <CJK>
                 (#x5D76 . #x84E5)      ; <CJK>
                 (#x5D77 . #x84E3)      ; <CJK>
                 (#x5D78 . #x850C)      ; <CJK>
                 (#x5D79 . #x750D)      ; <CJK>
                 (#x5D7A . #x8538)      ; <CJK>
                 (#x5D7B . #x84F0)      ; <CJK>
                 (#x5D7C . #x8539)      ; <CJK>
                 (#x5D7D . #x851F)      ; <CJK>
                 (#x5D7E . #x853A)      ; <CJK>
                 (#x5E21 . #x8556)      ; <CJK>
                 (#x5E22 . #x853B)      ; <CJK>
                 (#x5E23 . #x84FF)      ; <CJK>
                 (#x5E24 . #x84FC)      ; <CJK>
                 (#x5E25 . #x8559)      ; <CJK>
                 (#x5E26 . #x8548)      ; <CJK>
                 (#x5E27 . #x8568)      ; <CJK>
                 (#x5E28 . #x8564)      ; <CJK>
                 (#x5E29 . #x855E)      ; <CJK>
                 (#x5E2A . #x857A)      ; <CJK>
                 (#x5E2B . #x77A2)      ; <CJK>
                 (#x5E2C . #x8543)      ; <CJK>
                 (#x5E2D . #x8572)      ; <CJK>
                 (#x5E2E . #x857B)      ; <CJK>
                 (#x5E2F . #x85A4)      ; <CJK>
                 (#x5E30 . #x85A8)      ; <CJK>
                 (#x5E31 . #x8587)      ; <CJK>
                 (#x5E32 . #x858F)      ; <CJK>
                 (#x5E33 . #x8579)      ; <CJK>
                 (#x5E34 . #x85AE)      ; <CJK>
                 (#x5E35 . #x859C)      ; <CJK>
                 (#x5E36 . #x8585)      ; <CJK>
                 (#x5E37 . #x85B9)      ; <CJK>
                 (#x5E38 . #x85B7)      ; <CJK>
                 (#x5E39 . #x85B0)      ; <CJK>
                 (#x5E3A . #x85D3)      ; <CJK>
                 (#x5E3B . #x85C1)      ; <CJK>
                 (#x5E3C . #x85DC)      ; <CJK>
                 (#x5E3D . #x85FF)      ; <CJK>
                 (#x5E3E . #x8627)      ; <CJK>
                 (#x5E3F . #x8605)      ; <CJK>
                 (#x5E40 . #x8629)      ; <CJK>
                 (#x5E41 . #x8616)      ; <CJK>
                 (#x5E42 . #x863C)      ; <CJK>
                 (#x5E43 . #x5EFE)      ; <CJK>
                 (#x5E44 . #x5F08)      ; <CJK>
                 (#x5E45 . #x593C)      ; <CJK>
                 (#x5E46 . #x5941)      ; <CJK>
                 (#x5E47 . #x8037)      ; <CJK>
                 (#x5E48 . #x5955)      ; <CJK>
                 (#x5E49 . #x595A)      ; <CJK>
                 (#x5E4A . #x5958)      ; <CJK>
                 (#x5E4B . #x530F)      ; <CJK>
                 (#x5E4C . #x5C22)      ; <CJK>
                 (#x5E4D . #x5C25)      ; <CJK>
                 (#x5E4E . #x5C2C)      ; <CJK>
                 (#x5E4F . #x5C34)      ; <CJK>
                 (#x5E50 . #x624C)      ; <CJK>
                 (#x5E51 . #x626A)      ; <CJK>
                 (#x5E52 . #x629F)      ; <CJK>
                 (#x5E53 . #x62BB)      ; <CJK>
                 (#x5E54 . #x62CA)      ; <CJK>
                 (#x5E55 . #x62DA)      ; <CJK>
                 (#x5E56 . #x62D7)      ; <CJK>
                 (#x5E57 . #x62EE)      ; <CJK>
                 (#x5E58 . #x6322)      ; <CJK>
                 (#x5E59 . #x62F6)      ; <CJK>
                 (#x5E5A . #x6339)      ; <CJK>
                 (#x5E5B . #x634B)      ; <CJK>
                 (#x5E5C . #x6343)      ; <CJK>
                 (#x5E5D . #x63AD)      ; <CJK>
                 (#x5E5E . #x63F6)      ; <CJK>
                 (#x5E5F . #x6371)      ; <CJK>
                 (#x5E60 . #x637A)      ; <CJK>
                 (#x5E61 . #x638E)      ; <CJK>
                 (#x5E62 . #x63B4)      ; <CJK>
                 (#x5E63 . #x636D)      ; <CJK>
                 (#x5E64 . #x63AC)      ; <CJK>
                 (#x5E65 . #x638A)      ; <CJK>
                 (#x5E66 . #x6369)      ; <CJK>
                 (#x5E67 . #x63AE)      ; <CJK>
                 (#x5E68 . #x63BC)      ; <CJK>
                 (#x5E69 . #x63F2)      ; <CJK>
                 (#x5E6A . #x63F8)      ; <CJK>
                 (#x5E6B . #x63E0)      ; <CJK>
                 (#x5E6C . #x63FF)      ; <CJK>
                 (#x5E6D . #x63C4)      ; <CJK>
                 (#x5E6E . #x63DE)      ; <CJK>
                 (#x5E6F . #x63CE)      ; <CJK>
                 (#x5E70 . #x6452)      ; <CJK>
                 (#x5E71 . #x63C6)      ; <CJK>
                 (#x5E72 . #x63BE)      ; <CJK>
                 (#x5E73 . #x6445)      ; <CJK>
                 (#x5E74 . #x6441)      ; <CJK>
                 (#x5E75 . #x640B)      ; <CJK>
                 (#x5E76 . #x641B)      ; <CJK>
                 (#x5E77 . #x6420)      ; <CJK>
                 (#x5E78 . #x640C)      ; <CJK>
                 (#x5E79 . #x6426)      ; <CJK>
                 (#x5E7A . #x6421)      ; <CJK>
                 (#x5E7B . #x645E)      ; <CJK>
                 (#x5E7C . #x6484)      ; <CJK>
                 (#x5E7D . #x646D)      ; <CJK>
                 (#x5E7E . #x6496)      ; <CJK>
                 (#x5F21 . #x647A)      ; <CJK>
                 (#x5F22 . #x64B7)      ; <CJK>
                 (#x5F23 . #x64B8)      ; <CJK>
                 (#x5F24 . #x6499)      ; <CJK>
                 (#x5F25 . #x64BA)      ; <CJK>
                 (#x5F26 . #x64C0)      ; <CJK>
                 (#x5F27 . #x64D0)      ; <CJK>
                 (#x5F28 . #x64D7)      ; <CJK>
                 (#x5F29 . #x64E4)      ; <CJK>
                 (#x5F2A . #x64E2)      ; <CJK>
                 (#x5F2B . #x6509)      ; <CJK>
                 (#x5F2C . #x6525)      ; <CJK>
                 (#x5F2D . #x652E)      ; <CJK>
                 (#x5F2E . #x5F0B)      ; <CJK>
                 (#x5F2F . #x5FD2)      ; <CJK>
                 (#x5F30 . #x7519)      ; <CJK>
                 (#x5F31 . #x5F11)      ; <CJK>
                 (#x5F32 . #x535F)      ; <CJK>
                 (#x5F33 . #x53F1)      ; <CJK>
                 (#x5F34 . #x53FD)      ; <CJK>
                 (#x5F35 . #x53E9)      ; <CJK>
                 (#x5F36 . #x53E8)      ; <CJK>
                 (#x5F37 . #x53FB)      ; <CJK>
                 (#x5F38 . #x5412)      ; <CJK>
                 (#x5F39 . #x5416)      ; <CJK>
                 (#x5F3A . #x5406)      ; <CJK>
                 (#x5F3B . #x544B)      ; <CJK>
                 (#x5F3C . #x5452)      ; <CJK>
                 (#x5F3D . #x5453)      ; <CJK>
                 (#x5F3E . #x5454)      ; <CJK>
                 (#x5F3F . #x5456)      ; <CJK>
                 (#x5F40 . #x5443)      ; <CJK>
                 (#x5F41 . #x5421)      ; <CJK>
                 (#x5F42 . #x5457)      ; <CJK>
                 (#x5F43 . #x5459)      ; <CJK>
                 (#x5F44 . #x5423)      ; <CJK>
                 (#x5F45 . #x5432)      ; <CJK>
                 (#x5F46 . #x5482)      ; <CJK>
                 (#x5F47 . #x5494)      ; <CJK>
                 (#x5F48 . #x5477)      ; <CJK>
                 (#x5F49 . #x5471)      ; <CJK>
                 (#x5F4A . #x5464)      ; <CJK>
                 (#x5F4B . #x549A)      ; <CJK>
                 (#x5F4C . #x549B)      ; <CJK>
                 (#x5F4D . #x5484)      ; <CJK>
                 (#x5F4E . #x5476)      ; <CJK>
                 (#x5F4F . #x5466)      ; <CJK>
                 (#x5F50 . #x549D)      ; <CJK>
                 (#x5F51 . #x54D0)      ; <CJK>
                 (#x5F52 . #x54AD)      ; <CJK>
                 (#x5F53 . #x54C2)      ; <CJK>
                 (#x5F54 . #x54B4)      ; <CJK>
                 (#x5F55 . #x54D2)      ; <CJK>
                 (#x5F56 . #x54A7)      ; <CJK>
                 (#x5F57 . #x54A6)      ; <CJK>
                 (#x5F58 . #x54D3)      ; <CJK>
                 (#x5F59 . #x54D4)      ; <CJK>
                 (#x5F5A . #x5472)      ; <CJK>
                 (#x5F5B . #x54A3)      ; <CJK>
                 (#x5F5C . #x54D5)      ; <CJK>
                 (#x5F5D . #x54BB)      ; <CJK>
                 (#x5F5E . #x54BF)      ; <CJK>
                 (#x5F5F . #x54CC)      ; <CJK>
                 (#x5F60 . #x54D9)      ; <CJK>
                 (#x5F61 . #x54DA)      ; <CJK>
                 (#x5F62 . #x54DC)      ; <CJK>
                 (#x5F63 . #x54A9)      ; <CJK>
                 (#x5F64 . #x54AA)      ; <CJK>
                 (#x5F65 . #x54A4)      ; <CJK>
                 (#x5F66 . #x54DD)      ; <CJK>
                 (#x5F67 . #x54CF)      ; <CJK>
                 (#x5F68 . #x54DE)      ; <CJK>
                 (#x5F69 . #x551B)      ; <CJK>
                 (#x5F6A . #x54E7)      ; <CJK>
                 (#x5F6B . #x5520)      ; <CJK>
                 (#x5F6C . #x54FD)      ; <CJK>
                 (#x5F6D . #x5514)      ; <CJK>
                 (#x5F6E . #x54F3)      ; <CJK>
                 (#x5F6F . #x5522)      ; <CJK>
                 (#x5F70 . #x5523)      ; <CJK>
                 (#x5F71 . #x550F)      ; <CJK>
                 (#x5F72 . #x5511)      ; <CJK>
                 (#x5F73 . #x5527)      ; <CJK>
                 (#x5F74 . #x552A)      ; <CJK>
                 (#x5F75 . #x5567)      ; <CJK>
                 (#x5F76 . #x558F)      ; <CJK>
                 (#x5F77 . #x55B5)      ; <CJK>
                 (#x5F78 . #x5549)      ; <CJK>
                 (#x5F79 . #x556D)      ; <CJK>
                 (#x5F7A . #x5541)      ; <CJK>
                 (#x5F7B . #x5555)      ; <CJK>
                 (#x5F7C . #x553F)      ; <CJK>
                 (#x5F7D . #x5550)      ; <CJK>
                 (#x5F7E . #x553C)      ; <CJK>
                 (#x6021 . #x5537)      ; <CJK>
                 (#x6022 . #x5556)      ; <CJK>
                 (#x6023 . #x5575)      ; <CJK>
                 (#x6024 . #x5576)      ; <CJK>
                 (#x6025 . #x5577)      ; <CJK>
                 (#x6026 . #x5533)      ; <CJK>
                 (#x6027 . #x5530)      ; <CJK>
                 (#x6028 . #x555C)      ; <CJK>
                 (#x6029 . #x558B)      ; <CJK>
                 (#x602A . #x55D2)      ; <CJK>
                 (#x602B . #x5583)      ; <CJK>
                 (#x602C . #x55B1)      ; <CJK>
                 (#x602D . #x55B9)      ; <CJK>
                 (#x602E . #x5588)      ; <CJK>
                 (#x602F . #x5581)      ; <CJK>
                 (#x6030 . #x559F)      ; <CJK>
                 (#x6031 . #x557E)      ; <CJK>
                 (#x6032 . #x55D6)      ; <CJK>
                 (#x6033 . #x5591)      ; <CJK>
                 (#x6034 . #x557B)      ; <CJK>
                 (#x6035 . #x55DF)      ; <CJK>
                 (#x6036 . #x55BD)      ; <CJK>
                 (#x6037 . #x55BE)      ; <CJK>
                 (#x6038 . #x5594)      ; <CJK>
                 (#x6039 . #x5599)      ; <CJK>
                 (#x603A . #x55EA)      ; <CJK>
                 (#x603B . #x55F7)      ; <CJK>
                 (#x603C . #x55C9)      ; <CJK>
                 (#x603D . #x561F)      ; <CJK>
                 (#x603E . #x55D1)      ; <CJK>
                 (#x603F . #x55EB)      ; <CJK>
                 (#x6040 . #x55EC)      ; <CJK>
                 (#x6041 . #x55D4)      ; <CJK>
                 (#x6042 . #x55E6)      ; <CJK>
                 (#x6043 . #x55DD)      ; <CJK>
                 (#x6044 . #x55C4)      ; <CJK>
                 (#x6045 . #x55EF)      ; <CJK>
                 (#x6046 . #x55E5)      ; <CJK>
                 (#x6047 . #x55F2)      ; <CJK>
                 (#x6048 . #x55F3)      ; <CJK>
                 (#x6049 . #x55CC)      ; <CJK>
                 (#x604A . #x55CD)      ; <CJK>
                 (#x604B . #x55E8)      ; <CJK>
                 (#x604C . #x55F5)      ; <CJK>
                 (#x604D . #x55E4)      ; <CJK>
                 (#x604E . #x8F94)      ; <CJK>
                 (#x604F . #x561E)      ; <CJK>
                 (#x6050 . #x5608)      ; <CJK>
                 (#x6051 . #x560C)      ; <CJK>
                 (#x6052 . #x5601)      ; <CJK>
                 (#x6053 . #x5624)      ; <CJK>
                 (#x6054 . #x5623)      ; <CJK>
                 (#x6055 . #x55FE)      ; <CJK>
                 (#x6056 . #x5600)      ; <CJK>
                 (#x6057 . #x5627)      ; <CJK>
                 (#x6058 . #x562D)      ; <CJK>
                 (#x6059 . #x5658)      ; <CJK>
                 (#x605A . #x5639)      ; <CJK>
                 (#x605B . #x5657)      ; <CJK>
                 (#x605C . #x562C)      ; <CJK>
                 (#x605D . #x564D)      ; <CJK>
                 (#x605E . #x5662)      ; <CJK>
                 (#x605F . #x5659)      ; <CJK>
                 (#x6060 . #x565C)      ; <CJK>
                 (#x6061 . #x564C)      ; <CJK>
                 (#x6062 . #x5654)      ; <CJK>
                 (#x6063 . #x5686)      ; <CJK>
                 (#x6064 . #x5664)      ; <CJK>
                 (#x6065 . #x5671)      ; <CJK>
                 (#x6066 . #x566B)      ; <CJK>
                 (#x6067 . #x567B)      ; <CJK>
                 (#x6068 . #x567C)      ; <CJK>
                 (#x6069 . #x5685)      ; <CJK>
                 (#x606A . #x5693)      ; <CJK>
                 (#x606B . #x56AF)      ; <CJK>
                 (#x606C . #x56D4)      ; <CJK>
                 (#x606D . #x56D7)      ; <CJK>
                 (#x606E . #x56DD)      ; <CJK>
                 (#x606F . #x56E1)      ; <CJK>
                 (#x6070 . #x56F5)      ; <CJK>
                 (#x6071 . #x56EB)      ; <CJK>
                 (#x6072 . #x56F9)      ; <CJK>
                 (#x6073 . #x56FF)      ; <CJK>
                 (#x6074 . #x5704)      ; <CJK>
                 (#x6075 . #x570A)      ; <CJK>
                 (#x6076 . #x5709)      ; <CJK>
                 (#x6077 . #x571C)      ; <CJK>
                 (#x6078 . #x5E0F)      ; <CJK>
                 (#x6079 . #x5E19)      ; <CJK>
                 (#x607A . #x5E14)      ; <CJK>
                 (#x607B . #x5E11)      ; <CJK>
                 (#x607C . #x5E31)      ; <CJK>
                 (#x607D . #x5E3B)      ; <CJK>
                 (#x607E . #x5E3C)      ; <CJK>
                 (#x6121 . #x5E37)      ; <CJK>
                 (#x6122 . #x5E44)      ; <CJK>
                 (#x6123 . #x5E54)      ; <CJK>
                 (#x6124 . #x5E5B)      ; <CJK>
                 (#x6125 . #x5E5E)      ; <CJK>
                 (#x6126 . #x5E61)      ; <CJK>
                 (#x6127 . #x5C8C)      ; <CJK>
                 (#x6128 . #x5C7A)      ; <CJK>
                 (#x6129 . #x5C8D)      ; <CJK>
                 (#x612A . #x5C90)      ; <CJK>
                 (#x612B . #x5C96)      ; <CJK>
                 (#x612C . #x5C88)      ; <CJK>
                 (#x612D . #x5C98)      ; <CJK>
                 (#x612E . #x5C99)      ; <CJK>
                 (#x612F . #x5C91)      ; <CJK>
                 (#x6130 . #x5C9A)      ; <CJK>
                 (#x6131 . #x5C9C)      ; <CJK>
                 (#x6132 . #x5CB5)      ; <CJK>
                 (#x6133 . #x5CA2)      ; <CJK>
                 (#x6134 . #x5CBD)      ; <CJK>
                 (#x6135 . #x5CAC)      ; <CJK>
                 (#x6136 . #x5CAB)      ; <CJK>
                 (#x6137 . #x5CB1)      ; <CJK>
                 (#x6138 . #x5CA3)      ; <CJK>
                 (#x6139 . #x5CC1)      ; <CJK>
                 (#x613A . #x5CB7)      ; <CJK>
                 (#x613B . #x5CC4)      ; <CJK>
                 (#x613C . #x5CD2)      ; <CJK>
                 (#x613D . #x5CE4)      ; <CJK>
                 (#x613E . #x5CCB)      ; <CJK>
                 (#x613F . #x5CE5)      ; <CJK>
                 (#x6140 . #x5D02)      ; <CJK>
                 (#x6141 . #x5D03)      ; <CJK>
                 (#x6142 . #x5D27)      ; <CJK>
                 (#x6143 . #x5D26)      ; <CJK>
                 (#x6144 . #x5D2E)      ; <CJK>
                 (#x6145 . #x5D24)      ; <CJK>
                 (#x6146 . #x5D1E)      ; <CJK>
                 (#x6147 . #x5D06)      ; <CJK>
                 (#x6148 . #x5D1B)      ; <CJK>
                 (#x6149 . #x5D58)      ; <CJK>
                 (#x614A . #x5D3E)      ; <CJK>
                 (#x614B . #x5D34)      ; <CJK>
                 (#x614C . #x5D3D)      ; <CJK>
                 (#x614D . #x5D6C)      ; <CJK>
                 (#x614E . #x5D5B)      ; <CJK>
                 (#x614F . #x5D6F)      ; <CJK>
                 (#x6150 . #x5D5D)      ; <CJK>
                 (#x6151 . #x5D6B)      ; <CJK>
                 (#x6152 . #x5D4B)      ; <CJK>
                 (#x6153 . #x5D4A)      ; <CJK>
                 (#x6154 . #x5D69)      ; <CJK>
                 (#x6155 . #x5D74)      ; <CJK>
                 (#x6156 . #x5D82)      ; <CJK>
                 (#x6157 . #x5D99)      ; <CJK>
                 (#x6158 . #x5D9D)      ; <CJK>
                 (#x6159 . #x8C73)      ; <CJK>
                 (#x615A . #x5DB7)      ; <CJK>
                 (#x615B . #x5DC5)      ; <CJK>
                 (#x615C . #x5F73)      ; <CJK>
                 (#x615D . #x5F77)      ; <CJK>
                 (#x615E . #x5F82)      ; <CJK>
                 (#x615F . #x5F87)      ; <CJK>
                 (#x6160 . #x5F89)      ; <CJK>
                 (#x6161 . #x5F8C)      ; <CJK>
                 (#x6162 . #x5F95)      ; <CJK>
                 (#x6163 . #x5F99)      ; <CJK>
                 (#x6164 . #x5F9C)      ; <CJK>
                 (#x6165 . #x5FA8)      ; <CJK>
                 (#x6166 . #x5FAD)      ; <CJK>
                 (#x6167 . #x5FB5)      ; <CJK>
                 (#x6168 . #x5FBC)      ; <CJK>
                 (#x6169 . #x8862)      ; <CJK>
                 (#x616A . #x5F61)      ; <CJK>
                 (#x616B . #x72AD)      ; <CJK>
                 (#x616C . #x72B0)      ; <CJK>
                 (#x616D . #x72B4)      ; <CJK>
                 (#x616E . #x72B7)      ; <CJK>
                 (#x616F . #x72B8)      ; <CJK>
                 (#x6170 . #x72C3)      ; <CJK>
                 (#x6171 . #x72C1)      ; <CJK>
                 (#x6172 . #x72CE)      ; <CJK>
                 (#x6173 . #x72CD)      ; <CJK>
                 (#x6174 . #x72D2)      ; <CJK>
                 (#x6175 . #x72E8)      ; <CJK>
                 (#x6176 . #x72EF)      ; <CJK>
                 (#x6177 . #x72E9)      ; <CJK>
                 (#x6178 . #x72F2)      ; <CJK>
                 (#x6179 . #x72F4)      ; <CJK>
                 (#x617A . #x72F7)      ; <CJK>
                 (#x617B . #x7301)      ; <CJK>
                 (#x617C . #x72F3)      ; <CJK>
                 (#x617D . #x7303)      ; <CJK>
                 (#x617E . #x72FA)      ; <CJK>
                 (#x6221 . #x72FB)      ; <CJK>
                 (#x6222 . #x7317)      ; <CJK>
                 (#x6223 . #x7313)      ; <CJK>
                 (#x6224 . #x7321)      ; <CJK>
                 (#x6225 . #x730A)      ; <CJK>
                 (#x6226 . #x731E)      ; <CJK>
                 (#x6227 . #x731D)      ; <CJK>
                 (#x6228 . #x7315)      ; <CJK>
                 (#x6229 . #x7322)      ; <CJK>
                 (#x622A . #x7339)      ; <CJK>
                 (#x622B . #x7325)      ; <CJK>
                 (#x622C . #x732C)      ; <CJK>
                 (#x622D . #x7338)      ; <CJK>
                 (#x622E . #x7331)      ; <CJK>
                 (#x622F . #x7350)      ; <CJK>
                 (#x6230 . #x734D)      ; <CJK>
                 (#x6231 . #x7357)      ; <CJK>
                 (#x6232 . #x7360)      ; <CJK>
                 (#x6233 . #x736C)      ; <CJK>
                 (#x6234 . #x736F)      ; <CJK>
                 (#x6235 . #x737E)      ; <CJK>
                 (#x6236 . #x821B)      ; <CJK>
                 (#x6237 . #x5925)      ; <CJK>
                 (#x6238 . #x98E7)      ; <CJK>
                 (#x6239 . #x5924)      ; <CJK>
                 (#x623A . #x5902)      ; <CJK>
                 (#x623B . #x9963)      ; <CJK>
                 (#x623C . #x9967)      ; <CJK>
                 (#x623D . #x9968)      ; <CJK>
                 (#x623E . #x9969)      ; <CJK>
                 (#x623F . #x996A)      ; <CJK>
                 (#x6240 . #x996B)      ; <CJK>
                 (#x6241 . #x996C)      ; <CJK>
                 (#x6242 . #x9974)      ; <CJK>
                 (#x6243 . #x9977)      ; <CJK>
                 (#x6244 . #x997D)      ; <CJK>
                 (#x6245 . #x9980)      ; <CJK>
                 (#x6246 . #x9984)      ; <CJK>
                 (#x6247 . #x9987)      ; <CJK>
                 (#x6248 . #x998A)      ; <CJK>
                 (#x6249 . #x998D)      ; <CJK>
                 (#x624A . #x9990)      ; <CJK>
                 (#x624B . #x9991)      ; <CJK>
                 (#x624C . #x9993)      ; <CJK>
                 (#x624D . #x9994)      ; <CJK>
                 (#x624E . #x9995)      ; <CJK>
                 (#x624F . #x5E80)      ; <CJK>
                 (#x6250 . #x5E91)      ; <CJK>
                 (#x6251 . #x5E8B)      ; <CJK>
                 (#x6252 . #x5E96)      ; <CJK>
                 (#x6253 . #x5EA5)      ; <CJK>
                 (#x6254 . #x5EA0)      ; <CJK>
                 (#x6255 . #x5EB9)      ; <CJK>
                 (#x6256 . #x5EB5)      ; <CJK>
                 (#x6257 . #x5EBE)      ; <CJK>
                 (#x6258 . #x5EB3)      ; <CJK>
                 (#x6259 . #x8D53)      ; <CJK>
                 (#x625A . #x5ED2)      ; <CJK>
                 (#x625B . #x5ED1)      ; <CJK>
                 (#x625C . #x5EDB)      ; <CJK>
                 (#x625D . #x5EE8)      ; <CJK>
                 (#x625E . #x5EEA)      ; <CJK>
                 (#x625F . #x81BA)      ; <CJK>
                 (#x6260 . #x5FC4)      ; <CJK>
                 (#x6261 . #x5FC9)      ; <CJK>
                 (#x6262 . #x5FD6)      ; <CJK>
                 (#x6263 . #x5FCF)      ; <CJK>
                 (#x6264 . #x6003)      ; <CJK>
                 (#x6265 . #x5FEE)      ; <CJK>
                 (#x6266 . #x6004)      ; <CJK>
                 (#x6267 . #x5FE1)      ; <CJK>
                 (#x6268 . #x5FE4)      ; <CJK>
                 (#x6269 . #x5FFE)      ; <CJK>
                 (#x626A . #x6005)      ; <CJK>
                 (#x626B . #x6006)      ; <CJK>
                 (#x626C . #x5FEA)      ; <CJK>
                 (#x626D . #x5FED)      ; <CJK>
                 (#x626E . #x5FF8)      ; <CJK>
                 (#x626F . #x6019)      ; <CJK>
                 (#x6270 . #x6035)      ; <CJK>
                 (#x6271 . #x6026)      ; <CJK>
                 (#x6272 . #x601B)      ; <CJK>
                 (#x6273 . #x600F)      ; <CJK>
                 (#x6274 . #x600D)      ; <CJK>
                 (#x6275 . #x6029)      ; <CJK>
                 (#x6276 . #x602B)      ; <CJK>
                 (#x6277 . #x600A)      ; <CJK>
                 (#x6278 . #x603F)      ; <CJK>
                 (#x6279 . #x6021)      ; <CJK>
                 (#x627A . #x6078)      ; <CJK>
                 (#x627B . #x6079)      ; <CJK>
                 (#x627C . #x607B)      ; <CJK>
                 (#x627D . #x607A)      ; <CJK>
                 (#x627E . #x6042)      ; <CJK>
                 (#x6321 . #x606A)      ; <CJK>
                 (#x6322 . #x607D)      ; <CJK>
                 (#x6323 . #x6096)      ; <CJK>
                 (#x6324 . #x609A)      ; <CJK>
                 (#x6325 . #x60AD)      ; <CJK>
                 (#x6326 . #x609D)      ; <CJK>
                 (#x6327 . #x6083)      ; <CJK>
                 (#x6328 . #x6092)      ; <CJK>
                 (#x6329 . #x608C)      ; <CJK>
                 (#x632A . #x609B)      ; <CJK>
                 (#x632B . #x60EC)      ; <CJK>
                 (#x632C . #x60BB)      ; <CJK>
                 (#x632D . #x60B1)      ; <CJK>
                 (#x632E . #x60DD)      ; <CJK>
                 (#x632F . #x60D8)      ; <CJK>
                 (#x6330 . #x60C6)      ; <CJK>
                 (#x6331 . #x60DA)      ; <CJK>
                 (#x6332 . #x60B4)      ; <CJK>
                 (#x6333 . #x6120)      ; <CJK>
                 (#x6334 . #x6126)      ; <CJK>
                 (#x6335 . #x6115)      ; <CJK>
                 (#x6336 . #x6123)      ; <CJK>
                 (#x6337 . #x60F4)      ; <CJK>
                 (#x6338 . #x6100)      ; <CJK>
                 (#x6339 . #x610E)      ; <CJK>
                 (#x633A . #x612B)      ; <CJK>
                 (#x633B . #x614A)      ; <CJK>
                 (#x633C . #x6175)      ; <CJK>
                 (#x633D . #x61AC)      ; <CJK>
                 (#x633E . #x6194)      ; <CJK>
                 (#x633F . #x61A7)      ; <CJK>
                 (#x6340 . #x61B7)      ; <CJK>
                 (#x6341 . #x61D4)      ; <CJK>
                 (#x6342 . #x61F5)      ; <CJK>
                 (#x6343 . #x5FDD)      ; <CJK>
                 (#x6344 . #x96B3)      ; <CJK>
                 (#x6345 . #x95E9)      ; <CJK>
                 (#x6346 . #x95EB)      ; <CJK>
                 (#x6347 . #x95F1)      ; <CJK>
                 (#x6348 . #x95F3)      ; <CJK>
                 (#x6349 . #x95F5)      ; <CJK>
                 (#x634A . #x95F6)      ; <CJK>
                 (#x634B . #x95FC)      ; <CJK>
                 (#x634C . #x95FE)      ; <CJK>
                 (#x634D . #x9603)      ; <CJK>
                 (#x634E . #x9604)      ; <CJK>
                 (#x634F . #x9606)      ; <CJK>
                 (#x6350 . #x9608)      ; <CJK>
                 (#x6351 . #x960A)      ; <CJK>
                 (#x6352 . #x960B)      ; <CJK>
                 (#x6353 . #x960C)      ; <CJK>
                 (#x6354 . #x960D)      ; <CJK>
                 (#x6355 . #x960F)      ; <CJK>
                 (#x6356 . #x9612)      ; <CJK>
                 (#x6357 . #x9615)      ; <CJK>
                 (#x6358 . #x9616)      ; <CJK>
                 (#x6359 . #x9617)      ; <CJK>
                 (#x635A . #x9619)      ; <CJK>
                 (#x635B . #x961A)      ; <CJK>
                 (#x635C . #x4E2C)      ; <CJK>
                 (#x635D . #x723F)      ; <CJK>
                 (#x635E . #x6215)      ; <CJK>
                 (#x635F . #x6C35)      ; <CJK>
                 (#x6360 . #x6C54)      ; <CJK>
                 (#x6361 . #x6C5C)      ; <CJK>
                 (#x6362 . #x6C4A)      ; <CJK>
                 (#x6363 . #x6CA3)      ; <CJK>
                 (#x6364 . #x6C85)      ; <CJK>
                 (#x6365 . #x6C90)      ; <CJK>
                 (#x6366 . #x6C94)      ; <CJK>
                 (#x6367 . #x6C8C)      ; <CJK>
                 (#x6368 . #x6C68)      ; <CJK>
                 (#x6369 . #x6C69)      ; <CJK>
                 (#x636A . #x6C74)      ; <CJK>
                 (#x636B . #x6C76)      ; <CJK>
                 (#x636C . #x6C86)      ; <CJK>
                 (#x636D . #x6CA9)      ; <CJK>
                 (#x636E . #x6CD0)      ; <CJK>
                 (#x636F . #x6CD4)      ; <CJK>
                 (#x6370 . #x6CAD)      ; <CJK>
                 (#x6371 . #x6CF7)      ; <CJK>
                 (#x6372 . #x6CF8)      ; <CJK>
                 (#x6373 . #x6CF1)      ; <CJK>
                 (#x6374 . #x6CD7)      ; <CJK>
                 (#x6375 . #x6CB2)      ; <CJK>
                 (#x6376 . #x6CE0)      ; <CJK>
                 (#x6377 . #x6CD6)      ; <CJK>
                 (#x6378 . #x6CFA)      ; <CJK>
                 (#x6379 . #x6CEB)      ; <CJK>
                 (#x637A . #x6CEE)      ; <CJK>
                 (#x637B . #x6CB1)      ; <CJK>
                 (#x637C . #x6CD3)      ; <CJK>
                 (#x637D . #x6CEF)      ; <CJK>
                 (#x637E . #x6CFE)      ; <CJK>
                 (#x6421 . #x6D39)      ; <CJK>
                 (#x6422 . #x6D27)      ; <CJK>
                 (#x6423 . #x6D0C)      ; <CJK>
                 (#x6424 . #x6D43)      ; <CJK>
                 (#x6425 . #x6D48)      ; <CJK>
                 (#x6426 . #x6D07)      ; <CJK>
                 (#x6427 . #x6D04)      ; <CJK>
                 (#x6428 . #x6D19)      ; <CJK>
                 (#x6429 . #x6D0E)      ; <CJK>
                 (#x642A . #x6D2B)      ; <CJK>
                 (#x642B . #x6D4D)      ; <CJK>
                 (#x642C . #x6D2E)      ; <CJK>
                 (#x642D . #x6D35)      ; <CJK>
                 (#x642E . #x6D1A)      ; <CJK>
                 (#x642F . #x6D4F)      ; <CJK>
                 (#x6430 . #x6D52)      ; <CJK>
                 (#x6431 . #x6D54)      ; <CJK>
                 (#x6432 . #x6D33)      ; <CJK>
                 (#x6433 . #x6D91)      ; <CJK>
                 (#x6434 . #x6D6F)      ; <CJK>
                 (#x6435 . #x6D9E)      ; <CJK>
                 (#x6436 . #x6DA0)      ; <CJK>
                 (#x6437 . #x6D5E)      ; <CJK>
                 (#x6438 . #x6D93)      ; <CJK>
                 (#x6439 . #x6D94)      ; <CJK>
                 (#x643A . #x6D5C)      ; <CJK>
                 (#x643B . #x6D60)      ; <CJK>
                 (#x643C . #x6D7C)      ; <CJK>
                 (#x643D . #x6D63)      ; <CJK>
                 (#x643E . #x6E1A)      ; <CJK>
                 (#x643F . #x6DC7)      ; <CJK>
                 (#x6440 . #x6DC5)      ; <CJK>
                 (#x6441 . #x6DDE)      ; <CJK>
                 (#x6442 . #x6E0E)      ; <CJK>
                 (#x6443 . #x6DBF)      ; <CJK>
                 (#x6444 . #x6DE0)      ; <CJK>
                 (#x6445 . #x6E11)      ; <CJK>
                 (#x6446 . #x6DE6)      ; <CJK>
                 (#x6447 . #x6DDD)      ; <CJK>
                 (#x6448 . #x6DD9)      ; <CJK>
                 (#x6449 . #x6E16)      ; <CJK>
                 (#x644A . #x6DAB)      ; <CJK>
                 (#x644B . #x6E0C)      ; <CJK>
                 (#x644C . #x6DAE)      ; <CJK>
                 (#x644D . #x6E2B)      ; <CJK>
                 (#x644E . #x6E6E)      ; <CJK>
                 (#x644F . #x6E4E)      ; <CJK>
                 (#x6450 . #x6E6B)      ; <CJK>
                 (#x6451 . #x6EB2)      ; <CJK>
                 (#x6452 . #x6E5F)      ; <CJK>
                 (#x6453 . #x6E86)      ; <CJK>
                 (#x6454 . #x6E53)      ; <CJK>
                 (#x6455 . #x6E54)      ; <CJK>
                 (#x6456 . #x6E32)      ; <CJK>
                 (#x6457 . #x6E25)      ; <CJK>
                 (#x6458 . #x6E44)      ; <CJK>
                 (#x6459 . #x6EDF)      ; <CJK>
                 (#x645A . #x6EB1)      ; <CJK>
                 (#x645B . #x6E98)      ; <CJK>
                 (#x645C . #x6EE0)      ; <CJK>
                 (#x645D . #x6F2D)      ; <CJK>
                 (#x645E . #x6EE2)      ; <CJK>
                 (#x645F . #x6EA5)      ; <CJK>
                 (#x6460 . #x6EA7)      ; <CJK>
                 (#x6461 . #x6EBD)      ; <CJK>
                 (#x6462 . #x6EBB)      ; <CJK>
                 (#x6463 . #x6EB7)      ; <CJK>
                 (#x6464 . #x6ED7)      ; <CJK>
                 (#x6465 . #x6EB4)      ; <CJK>
                 (#x6466 . #x6ECF)      ; <CJK>
                 (#x6467 . #x6E8F)      ; <CJK>
                 (#x6468 . #x6EC2)      ; <CJK>
                 (#x6469 . #x6E9F)      ; <CJK>
                 (#x646A . #x6F62)      ; <CJK>
                 (#x646B . #x6F46)      ; <CJK>
                 (#x646C . #x6F47)      ; <CJK>
                 (#x646D . #x6F24)      ; <CJK>
                 (#x646E . #x6F15)      ; <CJK>
                 (#x646F . #x6EF9)      ; <CJK>
                 (#x6470 . #x6F2F)      ; <CJK>
                 (#x6471 . #x6F36)      ; <CJK>
                 (#x6472 . #x6F4B)      ; <CJK>
                 (#x6473 . #x6F74)      ; <CJK>
                 (#x6474 . #x6F2A)      ; <CJK>
                 (#x6475 . #x6F09)      ; <CJK>
                 (#x6476 . #x6F29)      ; <CJK>
                 (#x6477 . #x6F89)      ; <CJK>
                 (#x6478 . #x6F8D)      ; <CJK>
                 (#x6479 . #x6F8C)      ; <CJK>
                 (#x647A . #x6F78)      ; <CJK>
                 (#x647B . #x6F72)      ; <CJK>
                 (#x647C . #x6F7C)      ; <CJK>
                 (#x647D . #x6F7A)      ; <CJK>
                 (#x647E . #x6FD1)      ; <CJK>
                 (#x6521 . #x6FC9)      ; <CJK>
                 (#x6522 . #x6FA7)      ; <CJK>
                 (#x6523 . #x6FB9)      ; <CJK>
                 (#x6524 . #x6FB6)      ; <CJK>
                 (#x6525 . #x6FC2)      ; <CJK>
                 (#x6526 . #x6FE1)      ; <CJK>
                 (#x6527 . #x6FEE)      ; <CJK>
                 (#x6528 . #x6FDE)      ; <CJK>
                 (#x6529 . #x6FE0)      ; <CJK>
                 (#x652A . #x6FEF)      ; <CJK>
                 (#x652B . #x701A)      ; <CJK>
                 (#x652C . #x7023)      ; <CJK>
                 (#x652D . #x701B)      ; <CJK>
                 (#x652E . #x7039)      ; <CJK>
                 (#x652F . #x7035)      ; <CJK>
                 (#x6530 . #x704F)      ; <CJK>
                 (#x6531 . #x705E)      ; <CJK>
                 (#x6532 . #x5B80)      ; <CJK>
                 (#x6533 . #x5B84)      ; <CJK>
                 (#x6534 . #x5B95)      ; <CJK>
                 (#x6535 . #x5B93)      ; <CJK>
                 (#x6536 . #x5BA5)      ; <CJK>
                 (#x6537 . #x5BB8)      ; <CJK>
                 (#x6538 . #x752F)      ; <CJK>
                 (#x6539 . #x9A9E)      ; <CJK>
                 (#x653A . #x6434)      ; <CJK>
                 (#x653B . #x5BE4)      ; <CJK>
                 (#x653C . #x5BEE)      ; <CJK>
                 (#x653D . #x8930)      ; <CJK>
                 (#x653E . #x5BF0)      ; <CJK>
                 (#x653F . #x8E47)      ; <CJK>
                 (#x6540 . #x8B07)      ; <CJK>
                 (#x6541 . #x8FB6)      ; <CJK>
                 (#x6542 . #x8FD3)      ; <CJK>
                 (#x6543 . #x8FD5)      ; <CJK>
                 (#x6544 . #x8FE5)      ; <CJK>
                 (#x6545 . #x8FEE)      ; <CJK>
                 (#x6546 . #x8FE4)      ; <CJK>
                 (#x6547 . #x8FE9)      ; <CJK>
                 (#x6548 . #x8FE6)      ; <CJK>
                 (#x6549 . #x8FF3)      ; <CJK>
                 (#x654A . #x8FE8)      ; <CJK>
                 (#x654B . #x9005)      ; <CJK>
                 (#x654C . #x9004)      ; <CJK>
                 (#x654D . #x900B)      ; <CJK>
                 (#x654E . #x9026)      ; <CJK>
                 (#x654F . #x9011)      ; <CJK>
                 (#x6550 . #x900D)      ; <CJK>
                 (#x6551 . #x9016)      ; <CJK>
                 (#x6552 . #x9021)      ; <CJK>
                 (#x6553 . #x9035)      ; <CJK>
                 (#x6554 . #x9036)      ; <CJK>
                 (#x6555 . #x902D)      ; <CJK>
                 (#x6556 . #x902F)      ; <CJK>
                 (#x6557 . #x9044)      ; <CJK>
                 (#x6558 . #x9051)      ; <CJK>
                 (#x6559 . #x9052)      ; <CJK>
                 (#x655A . #x9050)      ; <CJK>
                 (#x655B . #x9068)      ; <CJK>
                 (#x655C . #x9058)      ; <CJK>
                 (#x655D . #x9062)      ; <CJK>
                 (#x655E . #x905B)      ; <CJK>
                 (#x655F . #x66B9)      ; <CJK>
                 (#x6560 . #x9074)      ; <CJK>
                 (#x6561 . #x907D)      ; <CJK>
                 (#x6562 . #x9082)      ; <CJK>
                 (#x6563 . #x9088)      ; <CJK>
                 (#x6564 . #x9083)      ; <CJK>
                 (#x6565 . #x908B)      ; <CJK>
                 (#x6566 . #x5F50)      ; <CJK>
                 (#x6567 . #x5F57)      ; <CJK>
                 (#x6568 . #x5F56)      ; <CJK>
                 (#x6569 . #x5F58)      ; <CJK>
                 (#x656A . #x5C3B)      ; <CJK>
                 (#x656B . #x54AB)      ; <CJK>
                 (#x656C . #x5C50)      ; <CJK>
                 (#x656D . #x5C59)      ; <CJK>
                 (#x656E . #x5B71)      ; <CJK>
                 (#x656F . #x5C63)      ; <CJK>
                 (#x6570 . #x5C66)      ; <CJK>
                 (#x6571 . #x7FBC)      ; <CJK>
                 (#x6572 . #x5F2A)      ; <CJK>
                 (#x6573 . #x5F29)      ; <CJK>
                 (#x6574 . #x5F2D)      ; <CJK>
                 (#x6575 . #x8274)      ; <CJK>
                 (#x6576 . #x5F3C)      ; <CJK>
                 (#x6577 . #x9B3B)      ; <CJK>
                 (#x6578 . #x5C6E)      ; <CJK>
                 (#x6579 . #x5981)      ; <CJK>
                 (#x657A . #x5983)      ; <CJK>
                 (#x657B . #x598D)      ; <CJK>
                 (#x657C . #x59A9)      ; <CJK>
                 (#x657D . #x59AA)      ; <CJK>
                 (#x657E . #x59A3)      ; <CJK>
                 (#x6621 . #x5997)      ; <CJK>
                 (#x6622 . #x59CA)      ; <CJK>
                 (#x6623 . #x59AB)      ; <CJK>
                 (#x6624 . #x599E)      ; <CJK>
                 (#x6625 . #x59A4)      ; <CJK>
                 (#x6626 . #x59D2)      ; <CJK>
                 (#x6627 . #x59B2)      ; <CJK>
                 (#x6628 . #x59AF)      ; <CJK>
                 (#x6629 . #x59D7)      ; <CJK>
                 (#x662A . #x59BE)      ; <CJK>
                 (#x662B . #x5A05)      ; <CJK>
                 (#x662C . #x5A06)      ; <CJK>
                 (#x662D . #x59DD)      ; <CJK>
                 (#x662E . #x5A08)      ; <CJK>
                 (#x662F . #x59E3)      ; <CJK>
                 (#x6630 . #x59D8)      ; <CJK>
                 (#x6631 . #x59F9)      ; <CJK>
                 (#x6632 . #x5A0C)      ; <CJK>
                 (#x6633 . #x5A09)      ; <CJK>
                 (#x6634 . #x5A32)      ; <CJK>
                 (#x6635 . #x5A34)      ; <CJK>
                 (#x6636 . #x5A11)      ; <CJK>
                 (#x6637 . #x5A23)      ; <CJK>
                 (#x6638 . #x5A13)      ; <CJK>
                 (#x6639 . #x5A40)      ; <CJK>
                 (#x663A . #x5A67)      ; <CJK>
                 (#x663B . #x5A4A)      ; <CJK>
                 (#x663C . #x5A55)      ; <CJK>
                 (#x663D . #x5A3C)      ; <CJK>
                 (#x663E . #x5A62)      ; <CJK>
                 (#x663F . #x5A75)      ; <CJK>
                 (#x6640 . #x80EC)      ; <CJK>
                 (#x6641 . #x5AAA)      ; <CJK>
                 (#x6642 . #x5A9B)      ; <CJK>
                 (#x6643 . #x5A77)      ; <CJK>
                 (#x6644 . #x5A7A)      ; <CJK>
                 (#x6645 . #x5ABE)      ; <CJK>
                 (#x6646 . #x5AEB)      ; <CJK>
                 (#x6647 . #x5AB2)      ; <CJK>
                 (#x6648 . #x5AD2)      ; <CJK>
                 (#x6649 . #x5AD4)      ; <CJK>
                 (#x664A . #x5AB8)      ; <CJK>
                 (#x664B . #x5AE0)      ; <CJK>
                 (#x664C . #x5AE3)      ; <CJK>
                 (#x664D . #x5AF1)      ; <CJK>
                 (#x664E . #x5AD6)      ; <CJK>
                 (#x664F . #x5AE6)      ; <CJK>
                 (#x6650 . #x5AD8)      ; <CJK>
                 (#x6651 . #x5ADC)      ; <CJK>
                 (#x6652 . #x5B09)      ; <CJK>
                 (#x6653 . #x5B17)      ; <CJK>
                 (#x6654 . #x5B16)      ; <CJK>
                 (#x6655 . #x5B32)      ; <CJK>
                 (#x6656 . #x5B37)      ; <CJK>
                 (#x6657 . #x5B40)      ; <CJK>
                 (#x6658 . #x5C15)      ; <CJK>
                 (#x6659 . #x5C1C)      ; <CJK>
                 (#x665A . #x5B5A)      ; <CJK>
                 (#x665B . #x5B65)      ; <CJK>
                 (#x665C . #x5B73)      ; <CJK>
                 (#x665D . #x5B51)      ; <CJK>
                 (#x665E . #x5B53)      ; <CJK>
                 (#x665F . #x5B62)      ; <CJK>
                 (#x6660 . #x9A75)      ; <CJK>
                 (#x6661 . #x9A77)      ; <CJK>
                 (#x6662 . #x9A78)      ; <CJK>
                 (#x6663 . #x9A7A)      ; <CJK>
                 (#x6664 . #x9A7F)      ; <CJK>
                 (#x6665 . #x9A7D)      ; <CJK>
                 (#x6666 . #x9A80)      ; <CJK>
                 (#x6667 . #x9A81)      ; <CJK>
                 (#x6668 . #x9A85)      ; <CJK>
                 (#x6669 . #x9A88)      ; <CJK>
                 (#x666A . #x9A8A)      ; <CJK>
                 (#x666B . #x9A90)      ; <CJK>
                 (#x666C . #x9A92)      ; <CJK>
                 (#x666D . #x9A93)      ; <CJK>
                 (#x666E . #x9A96)      ; <CJK>
                 (#x666F . #x9A98)      ; <CJK>
                 (#x6670 . #x9A9B)      ; <CJK>
                 (#x6671 . #x9A9C)      ; <CJK>
                 (#x6672 . #x9A9D)      ; <CJK>
                 (#x6673 . #x9A9F)      ; <CJK>
                 (#x6674 . #x9AA0)      ; <CJK>
                 (#x6675 . #x9AA2)      ; <CJK>
                 (#x6676 . #x9AA3)      ; <CJK>
                 (#x6677 . #x9AA5)      ; <CJK>
                 (#x6678 . #x9AA7)      ; <CJK>
                 (#x6679 . #x7E9F)      ; <CJK>
                 (#x667A . #x7EA1)      ; <CJK>
                 (#x667B . #x7EA3)      ; <CJK>
                 (#x667C . #x7EA5)      ; <CJK>
                 (#x667D . #x7EA8)      ; <CJK>
                 (#x667E . #x7EA9)      ; <CJK>
                 (#x6721 . #x7EAD)      ; <CJK>
                 (#x6722 . #x7EB0)      ; <CJK>
                 (#x6723 . #x7EBE)      ; <CJK>
                 (#x6724 . #x7EC0)      ; <CJK>
                 (#x6725 . #x7EC1)      ; <CJK>
                 (#x6726 . #x7EC2)      ; <CJK>
                 (#x6727 . #x7EC9)      ; <CJK>
                 (#x6728 . #x7ECB)      ; <CJK>
                 (#x6729 . #x7ECC)      ; <CJK>
                 (#x672A . #x7ED0)      ; <CJK>
                 (#x672B . #x7ED4)      ; <CJK>
                 (#x672C . #x7ED7)      ; <CJK>
                 (#x672D . #x7EDB)      ; <CJK>
                 (#x672E . #x7EE0)      ; <CJK>
                 (#x672F . #x7EE1)      ; <CJK>
                 (#x6730 . #x7EE8)      ; <CJK>
                 (#x6731 . #x7EEB)      ; <CJK>
                 (#x6732 . #x7EEE)      ; <CJK>
                 (#x6733 . #x7EEF)      ; <CJK>
                 (#x6734 . #x7EF1)      ; <CJK>
                 (#x6735 . #x7EF2)      ; <CJK>
                 (#x6736 . #x7F0D)      ; <CJK>
                 (#x6737 . #x7EF6)      ; <CJK>
                 (#x6738 . #x7EFA)      ; <CJK>
                 (#x6739 . #x7EFB)      ; <CJK>
                 (#x673A . #x7EFE)      ; <CJK>
                 (#x673B . #x7F01)      ; <CJK>
                 (#x673C . #x7F02)      ; <CJK>
                 (#x673D . #x7F03)      ; <CJK>
                 (#x673E . #x7F07)      ; <CJK>
                 (#x673F . #x7F08)      ; <CJK>
                 (#x6740 . #x7F0B)      ; <CJK>
                 (#x6741 . #x7F0C)      ; <CJK>
                 (#x6742 . #x7F0F)      ; <CJK>
                 (#x6743 . #x7F11)      ; <CJK>
                 (#x6744 . #x7F12)      ; <CJK>
                 (#x6745 . #x7F17)      ; <CJK>
                 (#x6746 . #x7F19)      ; <CJK>
                 (#x6747 . #x7F1C)      ; <CJK>
                 (#x6748 . #x7F1B)      ; <CJK>
                 (#x6749 . #x7F1F)      ; <CJK>
                 (#x674A . #x7F21)      ; <CJK>
                 (#x674B . #x7F22)      ; <CJK>
                 (#x674C . #x7F23)      ; <CJK>
                 (#x674D . #x7F24)      ; <CJK>
                 (#x674E . #x7F25)      ; <CJK>
                 (#x674F . #x7F26)      ; <CJK>
                 (#x6750 . #x7F27)      ; <CJK>
                 (#x6751 . #x7F2A)      ; <CJK>
                 (#x6752 . #x7F2B)      ; <CJK>
                 (#x6753 . #x7F2C)      ; <CJK>
                 (#x6754 . #x7F2D)      ; <CJK>
                 (#x6755 . #x7F2F)      ; <CJK>
                 (#x6756 . #x7F30)      ; <CJK>
                 (#x6757 . #x7F31)      ; <CJK>
                 (#x6758 . #x7F32)      ; <CJK>
                 (#x6759 . #x7F33)      ; <CJK>
                 (#x675A . #x7F35)      ; <CJK>
                 (#x675B . #x5E7A)      ; <CJK>
                 (#x675C . #x757F)      ; <CJK>
                 (#x675D . #x5DDB)      ; <CJK>
                 (#x675E . #x753E)      ; <CJK>
                 (#x675F . #x9095)      ; <CJK>
                 (#x6760 . #x738E)      ; <CJK>
                 (#x6761 . #x7391)      ; <CJK>
                 (#x6762 . #x73AE)      ; <CJK>
                 (#x6763 . #x73A2)      ; <CJK>
                 (#x6764 . #x739F)      ; <CJK>
                 (#x6765 . #x73CF)      ; <CJK>
                 (#x6766 . #x73C2)      ; <CJK>
                 (#x6767 . #x73D1)      ; <CJK>
                 (#x6768 . #x73B7)      ; <CJK>
                 (#x6769 . #x73B3)      ; <CJK>
                 (#x676A . #x73C0)      ; <CJK>
                 (#x676B . #x73C9)      ; <CJK>
                 (#x676C . #x73C8)      ; <CJK>
                 (#x676D . #x73E5)      ; <CJK>
                 (#x676E . #x73D9)      ; <CJK>
                 (#x676F . #x987C)      ; <CJK>
                 (#x6770 . #x740A)      ; <CJK>
                 (#x6771 . #x73E9)      ; <CJK>
                 (#x6772 . #x73E7)      ; <CJK>
                 (#x6773 . #x73DE)      ; <CJK>
                 (#x6774 . #x73BA)      ; <CJK>
                 (#x6775 . #x73F2)      ; <CJK>
                 (#x6776 . #x740F)      ; <CJK>
                 (#x6777 . #x742A)      ; <CJK>
                 (#x6778 . #x745B)      ; <CJK>
                 (#x6779 . #x7426)      ; <CJK>
                 (#x677A . #x7425)      ; <CJK>
                 (#x677B . #x7428)      ; <CJK>
                 (#x677C . #x7430)      ; <CJK>
                 (#x677D . #x742E)      ; <CJK>
                 (#x677E . #x742C)      ; <CJK>
                 (#x6821 . #x741B)      ; <CJK>
                 (#x6822 . #x741A)      ; <CJK>
                 (#x6823 . #x7441)      ; <CJK>
                 (#x6824 . #x745C)      ; <CJK>
                 (#x6825 . #x7457)      ; <CJK>
                 (#x6826 . #x7455)      ; <CJK>
                 (#x6827 . #x7459)      ; <CJK>
                 (#x6828 . #x7477)      ; <CJK>
                 (#x6829 . #x746D)      ; <CJK>
                 (#x682A . #x747E)      ; <CJK>
                 (#x682B . #x749C)      ; <CJK>
                 (#x682C . #x748E)      ; <CJK>
                 (#x682D . #x7480)      ; <CJK>
                 (#x682E . #x7481)      ; <CJK>
                 (#x682F . #x7487)      ; <CJK>
                 (#x6830 . #x748B)      ; <CJK>
                 (#x6831 . #x749E)      ; <CJK>
                 (#x6832 . #x74A8)      ; <CJK>
                 (#x6833 . #x74A9)      ; <CJK>
                 (#x6834 . #x7490)      ; <CJK>
                 (#x6835 . #x74A7)      ; <CJK>
                 (#x6836 . #x74D2)      ; <CJK>
                 (#x6837 . #x74BA)      ; <CJK>
                 (#x6838 . #x97EA)      ; <CJK>
                 (#x6839 . #x97EB)      ; <CJK>
                 (#x683A . #x97EC)      ; <CJK>
                 (#x683B . #x674C)      ; <CJK>
                 (#x683C . #x6753)      ; <CJK>
                 (#x683D . #x675E)      ; <CJK>
                 (#x683E . #x6748)      ; <CJK>
                 (#x683F . #x6769)      ; <CJK>
                 (#x6840 . #x67A5)      ; <CJK>
                 (#x6841 . #x6787)      ; <CJK>
                 (#x6842 . #x676A)      ; <CJK>
                 (#x6843 . #x6773)      ; <CJK>
                 (#x6844 . #x6798)      ; <CJK>
                 (#x6845 . #x67A7)      ; <CJK>
                 (#x6846 . #x6775)      ; <CJK>
                 (#x6847 . #x67A8)      ; <CJK>
                 (#x6848 . #x679E)      ; <CJK>
                 (#x6849 . #x67AD)      ; <CJK>
                 (#x684A . #x678B)      ; <CJK>
                 (#x684B . #x6777)      ; <CJK>
                 (#x684C . #x677C)      ; <CJK>
                 (#x684D . #x67F0)      ; <CJK>
                 (#x684E . #x6809)      ; <CJK>
                 (#x684F . #x67D8)      ; <CJK>
                 (#x6850 . #x680A)      ; <CJK>
                 (#x6851 . #x67E9)      ; <CJK>
                 (#x6852 . #x67B0)      ; <CJK>
                 (#x6853 . #x680C)      ; <CJK>
                 (#x6854 . #x67D9)      ; <CJK>
                 (#x6855 . #x67B5)      ; <CJK>
                 (#x6856 . #x67DA)      ; <CJK>
                 (#x6857 . #x67B3)      ; <CJK>
                 (#x6858 . #x67DD)      ; <CJK>
                 (#x6859 . #x6800)      ; <CJK>
                 (#x685A . #x67C3)      ; <CJK>
                 (#x685B . #x67B8)      ; <CJK>
                 (#x685C . #x67E2)      ; <CJK>
                 (#x685D . #x680E)      ; <CJK>
                 (#x685E . #x67C1)      ; <CJK>
                 (#x685F . #x67FD)      ; <CJK>
                 (#x6860 . #x6832)      ; <CJK>
                 (#x6861 . #x6833)      ; <CJK>
                 (#x6862 . #x6860)      ; <CJK>
                 (#x6863 . #x6861)      ; <CJK>
                 (#x6864 . #x684E)      ; <CJK>
                 (#x6865 . #x6862)      ; <CJK>
                 (#x6866 . #x6844)      ; <CJK>
                 (#x6867 . #x6864)      ; <CJK>
                 (#x6868 . #x6883)      ; <CJK>
                 (#x6869 . #x681D)      ; <CJK>
                 (#x686A . #x6855)      ; <CJK>
                 (#x686B . #x6866)      ; <CJK>
                 (#x686C . #x6841)      ; <CJK>
                 (#x686D . #x6867)      ; <CJK>
                 (#x686E . #x6840)      ; <CJK>
                 (#x686F . #x683E)      ; <CJK>
                 (#x6870 . #x684A)      ; <CJK>
                 (#x6871 . #x6849)      ; <CJK>
                 (#x6872 . #x6829)      ; <CJK>
                 (#x6873 . #x68B5)      ; <CJK>
                 (#x6874 . #x688F)      ; <CJK>
                 (#x6875 . #x6874)      ; <CJK>
                 (#x6876 . #x6877)      ; <CJK>
                 (#x6877 . #x6893)      ; <CJK>
                 (#x6878 . #x686B)      ; <CJK>
                 (#x6879 . #x68C2)      ; <CJK>
                 (#x687A . #x696E)      ; <CJK>
                 (#x687B . #x68FC)      ; <CJK>
                 (#x687C . #x691F)      ; <CJK>
                 (#x687D . #x6920)      ; <CJK>
                 (#x687E . #x68F9)      ; <CJK>
                 (#x6921 . #x6924)      ; <CJK>
                 (#x6922 . #x68F0)      ; <CJK>
                 (#x6923 . #x690B)      ; <CJK>
                 (#x6924 . #x6901)      ; <CJK>
                 (#x6925 . #x6957)      ; <CJK>
                 (#x6926 . #x68E3)      ; <CJK>
                 (#x6927 . #x6910)      ; <CJK>
                 (#x6928 . #x6971)      ; <CJK>
                 (#x6929 . #x6939)      ; <CJK>
                 (#x692A . #x6960)      ; <CJK>
                 (#x692B . #x6942)      ; <CJK>
                 (#x692C . #x695D)      ; <CJK>
                 (#x692D . #x6984)      ; <CJK>
                 (#x692E . #x696B)      ; <CJK>
                 (#x692F . #x6980)      ; <CJK>
                 (#x6930 . #x6998)      ; <CJK>
                 (#x6931 . #x6978)      ; <CJK>
                 (#x6932 . #x6934)      ; <CJK>
                 (#x6933 . #x69CC)      ; <CJK>
                 (#x6934 . #x6987)      ; <CJK>
                 (#x6935 . #x6988)      ; <CJK>
                 (#x6936 . #x69CE)      ; <CJK>
                 (#x6937 . #x6989)      ; <CJK>
                 (#x6938 . #x6966)      ; <CJK>
                 (#x6939 . #x6963)      ; <CJK>
                 (#x693A . #x6979)      ; <CJK>
                 (#x693B . #x699B)      ; <CJK>
                 (#x693C . #x69A7)      ; <CJK>
                 (#x693D . #x69BB)      ; <CJK>
                 (#x693E . #x69AB)      ; <CJK>
                 (#x693F . #x69AD)      ; <CJK>
                 (#x6940 . #x69D4)      ; <CJK>
                 (#x6941 . #x69B1)      ; <CJK>
                 (#x6942 . #x69C1)      ; <CJK>
                 (#x6943 . #x69CA)      ; <CJK>
                 (#x6944 . #x69DF)      ; <CJK>
                 (#x6945 . #x6995)      ; <CJK>
                 (#x6946 . #x69E0)      ; <CJK>
                 (#x6947 . #x698D)      ; <CJK>
                 (#x6948 . #x69FF)      ; <CJK>
                 (#x6949 . #x6A2F)      ; <CJK>
                 (#x694A . #x69ED)      ; <CJK>
                 (#x694B . #x6A17)      ; <CJK>
                 (#x694C . #x6A18)      ; <CJK>
                 (#x694D . #x6A65)      ; <CJK>
                 (#x694E . #x69F2)      ; <CJK>
                 (#x694F . #x6A44)      ; <CJK>
                 (#x6950 . #x6A3E)      ; <CJK>
                 (#x6951 . #x6AA0)      ; <CJK>
                 (#x6952 . #x6A50)      ; <CJK>
                 (#x6953 . #x6A5B)      ; <CJK>
                 (#x6954 . #x6A35)      ; <CJK>
                 (#x6955 . #x6A8E)      ; <CJK>
                 (#x6956 . #x6A79)      ; <CJK>
                 (#x6957 . #x6A3D)      ; <CJK>
                 (#x6958 . #x6A28)      ; <CJK>
                 (#x6959 . #x6A58)      ; <CJK>
                 (#x695A . #x6A7C)      ; <CJK>
                 (#x695B . #x6A91)      ; <CJK>
                 (#x695C . #x6A90)      ; <CJK>
                 (#x695D . #x6AA9)      ; <CJK>
                 (#x695E . #x6A97)      ; <CJK>
                 (#x695F . #x6AAB)      ; <CJK>
                 (#x6960 . #x7337)      ; <CJK>
                 (#x6961 . #x7352)      ; <CJK>
                 (#x6962 . #x6B81)      ; <CJK>
                 (#x6963 . #x6B82)      ; <CJK>
                 (#x6964 . #x6B87)      ; <CJK>
                 (#x6965 . #x6B84)      ; <CJK>
                 (#x6966 . #x6B92)      ; <CJK>
                 (#x6967 . #x6B93)      ; <CJK>
                 (#x6968 . #x6B8D)      ; <CJK>
                 (#x6969 . #x6B9A)      ; <CJK>
                 (#x696A . #x6B9B)      ; <CJK>
                 (#x696B . #x6BA1)      ; <CJK>
                 (#x696C . #x6BAA)      ; <CJK>
                 (#x696D . #x8F6B)      ; <CJK>
                 (#x696E . #x8F6D)      ; <CJK>
                 (#x696F . #x8F71)      ; <CJK>
                 (#x6970 . #x8F72)      ; <CJK>
                 (#x6971 . #x8F73)      ; <CJK>
                 (#x6972 . #x8F75)      ; <CJK>
                 (#x6973 . #x8F76)      ; <CJK>
                 (#x6974 . #x8F78)      ; <CJK>
                 (#x6975 . #x8F77)      ; <CJK>
                 (#x6976 . #x8F79)      ; <CJK>
                 (#x6977 . #x8F7A)      ; <CJK>
                 (#x6978 . #x8F7C)      ; <CJK>
                 (#x6979 . #x8F7E)      ; <CJK>
                 (#x697A . #x8F81)      ; <CJK>
                 (#x697B . #x8F82)      ; <CJK>
                 (#x697C . #x8F84)      ; <CJK>
                 (#x697D . #x8F87)      ; <CJK>
                 (#x697E . #x8F8B)      ; <CJK>
                 (#x6A21 . #x8F8D)      ; <CJK>
                 (#x6A22 . #x8F8E)      ; <CJK>
                 (#x6A23 . #x8F8F)      ; <CJK>
                 (#x6A24 . #x8F98)      ; <CJK>
                 (#x6A25 . #x8F9A)      ; <CJK>
                 (#x6A26 . #x8ECE)      ; <CJK>
                 (#x6A27 . #x620B)      ; <CJK>
                 (#x6A28 . #x6217)      ; <CJK>
                 (#x6A29 . #x621B)      ; <CJK>
                 (#x6A2A . #x621F)      ; <CJK>
                 (#x6A2B . #x6222)      ; <CJK>
                 (#x6A2C . #x6221)      ; <CJK>
                 (#x6A2D . #x6225)      ; <CJK>
                 (#x6A2E . #x6224)      ; <CJK>
                 (#x6A2F . #x622C)      ; <CJK>
                 (#x6A30 . #x81E7)      ; <CJK>
                 (#x6A31 . #x74EF)      ; <CJK>
                 (#x6A32 . #x74F4)      ; <CJK>
                 (#x6A33 . #x74FF)      ; <CJK>
                 (#x6A34 . #x750F)      ; <CJK>
                 (#x6A35 . #x7511)      ; <CJK>
                 (#x6A36 . #x7513)      ; <CJK>
                 (#x6A37 . #x6534)      ; <CJK>
                 (#x6A38 . #x65EE)      ; <CJK>
                 (#x6A39 . #x65EF)      ; <CJK>
                 (#x6A3A . #x65F0)      ; <CJK>
                 (#x6A3B . #x660A)      ; <CJK>
                 (#x6A3C . #x6619)      ; <CJK>
                 (#x6A3D . #x6772)      ; <CJK>
                 (#x6A3E . #x6603)      ; <CJK>
                 (#x6A3F . #x6615)      ; <CJK>
                 (#x6A40 . #x6600)      ; <CJK>
                 (#x6A41 . #x7085)      ; <CJK>
                 (#x6A42 . #x66F7)      ; <CJK>
                 (#x6A43 . #x661D)      ; <CJK>
                 (#x6A44 . #x6634)      ; <CJK>
                 (#x6A45 . #x6631)      ; <CJK>
                 (#x6A46 . #x6636)      ; <CJK>
                 (#x6A47 . #x6635)      ; <CJK>
                 (#x6A48 . #x8006)      ; <CJK>
                 (#x6A49 . #x665F)      ; <CJK>
                 (#x6A4A . #x6654)      ; <CJK>
                 (#x6A4B . #x6641)      ; <CJK>
                 (#x6A4C . #x664F)      ; <CJK>
                 (#x6A4D . #x6656)      ; <CJK>
                 (#x6A4E . #x6661)      ; <CJK>
                 (#x6A4F . #x6657)      ; <CJK>
                 (#x6A50 . #x6677)      ; <CJK>
                 (#x6A51 . #x6684)      ; <CJK>
                 (#x6A52 . #x668C)      ; <CJK>
                 (#x6A53 . #x66A7)      ; <CJK>
                 (#x6A54 . #x669D)      ; <CJK>
                 (#x6A55 . #x66BE)      ; <CJK>
                 (#x6A56 . #x66DB)      ; <CJK>
                 (#x6A57 . #x66DC)      ; <CJK>
                 (#x6A58 . #x66E6)      ; <CJK>
                 (#x6A59 . #x66E9)      ; <CJK>
                 (#x6A5A . #x8D32)      ; <CJK>
                 (#x6A5B . #x8D33)      ; <CJK>
                 (#x6A5C . #x8D36)      ; <CJK>
                 (#x6A5D . #x8D3B)      ; <CJK>
                 (#x6A5E . #x8D3D)      ; <CJK>
                 (#x6A5F . #x8D40)      ; <CJK>
                 (#x6A60 . #x8D45)      ; <CJK>
                 (#x6A61 . #x8D46)      ; <CJK>
                 (#x6A62 . #x8D48)      ; <CJK>
                 (#x6A63 . #x8D49)      ; <CJK>
                 (#x6A64 . #x8D47)      ; <CJK>
                 (#x6A65 . #x8D4D)      ; <CJK>
                 (#x6A66 . #x8D55)      ; <CJK>
                 (#x6A67 . #x8D59)      ; <CJK>
                 (#x6A68 . #x89C7)      ; <CJK>
                 (#x6A69 . #x89CA)      ; <CJK>
                 (#x6A6A . #x89CB)      ; <CJK>
                 (#x6A6B . #x89CC)      ; <CJK>
                 (#x6A6C . #x89CE)      ; <CJK>
                 (#x6A6D . #x89CF)      ; <CJK>
                 (#x6A6E . #x89D0)      ; <CJK>
                 (#x6A6F . #x89D1)      ; <CJK>
                 (#x6A70 . #x726E)      ; <CJK>
                 (#x6A71 . #x729F)      ; <CJK>
                 (#x6A72 . #x725D)      ; <CJK>
                 (#x6A73 . #x7266)      ; <CJK>
                 (#x6A74 . #x726F)      ; <CJK>
                 (#x6A75 . #x727E)      ; <CJK>
                 (#x6A76 . #x727F)      ; <CJK>
                 (#x6A77 . #x7284)      ; <CJK>
                 (#x6A78 . #x728B)      ; <CJK>
                 (#x6A79 . #x728D)      ; <CJK>
                 (#x6A7A . #x728F)      ; <CJK>
                 (#x6A7B . #x7292)      ; <CJK>
                 (#x6A7C . #x6308)      ; <CJK>
                 (#x6A7D . #x6332)      ; <CJK>
                 (#x6A7E . #x63B0)      ; <CJK>
                 (#x6B21 . #x643F)      ; <CJK>
                 (#x6B22 . #x64D8)      ; <CJK>
                 (#x6B23 . #x8004)      ; <CJK>
                 (#x6B24 . #x6BEA)      ; <CJK>
                 (#x6B25 . #x6BF3)      ; <CJK>
                 (#x6B26 . #x6BFD)      ; <CJK>
                 (#x6B27 . #x6BF5)      ; <CJK>
                 (#x6B28 . #x6BF9)      ; <CJK>
                 (#x6B29 . #x6C05)      ; <CJK>
                 (#x6B2A . #x6C07)      ; <CJK>
                 (#x6B2B . #x6C06)      ; <CJK>
                 (#x6B2C . #x6C0D)      ; <CJK>
                 (#x6B2D . #x6C15)      ; <CJK>
                 (#x6B2E . #x6C18)      ; <CJK>
                 (#x6B2F . #x6C19)      ; <CJK>
                 (#x6B30 . #x6C1A)      ; <CJK>
                 (#x6B31 . #x6C21)      ; <CJK>
                 (#x6B32 . #x6C29)      ; <CJK>
                 (#x6B33 . #x6C24)      ; <CJK>
                 (#x6B34 . #x6C2A)      ; <CJK>
                 (#x6B35 . #x6C32)      ; <CJK>
                 (#x6B36 . #x6535)      ; <CJK>
                 (#x6B37 . #x6555)      ; <CJK>
                 (#x6B38 . #x656B)      ; <CJK>
                 (#x6B39 . #x724D)      ; <CJK>
                 (#x6B3A . #x7252)      ; <CJK>
                 (#x6B3B . #x7256)      ; <CJK>
                 (#x6B3C . #x7230)      ; <CJK>
                 (#x6B3D . #x8662)      ; <CJK>
                 (#x6B3E . #x5216)      ; <CJK>
                 (#x6B3F . #x809F)      ; <CJK>
                 (#x6B40 . #x809C)      ; <CJK>
                 (#x6B41 . #x8093)      ; <CJK>
                 (#x6B42 . #x80BC)      ; <CJK>
                 (#x6B43 . #x670A)      ; <CJK>
                 (#x6B44 . #x80BD)      ; <CJK>
                 (#x6B45 . #x80B1)      ; <CJK>
                 (#x6B46 . #x80AB)      ; <CJK>
                 (#x6B47 . #x80AD)      ; <CJK>
                 (#x6B48 . #x80B4)      ; <CJK>
                 (#x6B49 . #x80B7)      ; <CJK>
                 (#x6B4A . #x80E7)      ; <CJK>
                 (#x6B4B . #x80E8)      ; <CJK>
                 (#x6B4C . #x80E9)      ; <CJK>
                 (#x6B4D . #x80EA)      ; <CJK>
                 (#x6B4E . #x80DB)      ; <CJK>
                 (#x6B4F . #x80C2)      ; <CJK>
                 (#x6B50 . #x80C4)      ; <CJK>
                 (#x6B51 . #x80D9)      ; <CJK>
                 (#x6B52 . #x80CD)      ; <CJK>
                 (#x6B53 . #x80D7)      ; <CJK>
                 (#x6B54 . #x6710)      ; <CJK>
                 (#x6B55 . #x80DD)      ; <CJK>
                 (#x6B56 . #x80EB)      ; <CJK>
                 (#x6B57 . #x80F1)      ; <CJK>
                 (#x6B58 . #x80F4)      ; <CJK>
                 (#x6B59 . #x80ED)      ; <CJK>
                 (#x6B5A . #x810D)      ; <CJK>
                 (#x6B5B . #x810E)      ; <CJK>
                 (#x6B5C . #x80F2)      ; <CJK>
                 (#x6B5D . #x80FC)      ; <CJK>
                 (#x6B5E . #x6715)      ; <CJK>
                 (#x6B5F . #x8112)      ; <CJK>
                 (#x6B60 . #x8C5A)      ; <CJK>
                 (#x6B61 . #x8136)      ; <CJK>
                 (#x6B62 . #x811E)      ; <CJK>
                 (#x6B63 . #x812C)      ; <CJK>
                 (#x6B64 . #x8118)      ; <CJK>
                 (#x6B65 . #x8132)      ; <CJK>
                 (#x6B66 . #x8148)      ; <CJK>
                 (#x6B67 . #x814C)      ; <CJK>
                 (#x6B68 . #x8153)      ; <CJK>
                 (#x6B69 . #x8174)      ; <CJK>
                 (#x6B6A . #x8159)      ; <CJK>
                 (#x6B6B . #x815A)      ; <CJK>
                 (#x6B6C . #x8171)      ; <CJK>
                 (#x6B6D . #x8160)      ; <CJK>
                 (#x6B6E . #x8169)      ; <CJK>
                 (#x6B6F . #x817C)      ; <CJK>
                 (#x6B70 . #x817D)      ; <CJK>
                 (#x6B71 . #x816D)      ; <CJK>
                 (#x6B72 . #x8167)      ; <CJK>
                 (#x6B73 . #x584D)      ; <CJK>
                 (#x6B74 . #x5AB5)      ; <CJK>
                 (#x6B75 . #x8188)      ; <CJK>
                 (#x6B76 . #x8182)      ; <CJK>
                 (#x6B77 . #x8191)      ; <CJK>
                 (#x6B78 . #x6ED5)      ; <CJK>
                 (#x6B79 . #x81A3)      ; <CJK>
                 (#x6B7A . #x81AA)      ; <CJK>
                 (#x6B7B . #x81CC)      ; <CJK>
                 (#x6B7C . #x6726)      ; <CJK>
                 (#x6B7D . #x81CA)      ; <CJK>
                 (#x6B7E . #x81BB)      ; <CJK>
                 (#x6C21 . #x81C1)      ; <CJK>
                 (#x6C22 . #x81A6)      ; <CJK>
                 (#x6C23 . #x6B24)      ; <CJK>
                 (#x6C24 . #x6B37)      ; <CJK>
                 (#x6C25 . #x6B39)      ; <CJK>
                 (#x6C26 . #x6B43)      ; <CJK>
                 (#x6C27 . #x6B46)      ; <CJK>
                 (#x6C28 . #x6B59)      ; <CJK>
                 (#x6C29 . #x98D1)      ; <CJK>
                 (#x6C2A . #x98D2)      ; <CJK>
                 (#x6C2B . #x98D3)      ; <CJK>
                 (#x6C2C . #x98D5)      ; <CJK>
                 (#x6C2D . #x98D9)      ; <CJK>
                 (#x6C2E . #x98DA)      ; <CJK>
                 (#x6C2F . #x6BB3)      ; <CJK>
                 (#x6C30 . #x5F40)      ; <CJK>
                 (#x6C31 . #x6BC2)      ; <CJK>
                 (#x6C32 . #x89F3)      ; <CJK>
                 (#x6C33 . #x6590)      ; <CJK>
                 (#x6C34 . #x9F51)      ; <CJK>
                 (#x6C35 . #x6593)      ; <CJK>
                 (#x6C36 . #x65BC)      ; <CJK>
                 (#x6C37 . #x65C6)      ; <CJK>
                 (#x6C38 . #x65C4)      ; <CJK>
                 (#x6C39 . #x65C3)      ; <CJK>
                 (#x6C3A . #x65CC)      ; <CJK>
                 (#x6C3B . #x65CE)      ; <CJK>
                 (#x6C3C . #x65D2)      ; <CJK>
                 (#x6C3D . #x65D6)      ; <CJK>
                 (#x6C3E . #x7080)      ; <CJK>
                 (#x6C3F . #x709C)      ; <CJK>
                 (#x6C40 . #x7096)      ; <CJK>
                 (#x6C41 . #x709D)      ; <CJK>
                 (#x6C42 . #x70BB)      ; <CJK>
                 (#x6C43 . #x70C0)      ; <CJK>
                 (#x6C44 . #x70B7)      ; <CJK>
                 (#x6C45 . #x70AB)      ; <CJK>
                 (#x6C46 . #x70B1)      ; <CJK>
                 (#x6C47 . #x70E8)      ; <CJK>
                 (#x6C48 . #x70CA)      ; <CJK>
                 (#x6C49 . #x7110)      ; <CJK>
                 (#x6C4A . #x7113)      ; <CJK>
                 (#x6C4B . #x7116)      ; <CJK>
                 (#x6C4C . #x712F)      ; <CJK>
                 (#x6C4D . #x7131)      ; <CJK>
                 (#x6C4E . #x7173)      ; <CJK>
                 (#x6C4F . #x715C)      ; <CJK>
                 (#x6C50 . #x7168)      ; <CJK>
                 (#x6C51 . #x7145)      ; <CJK>
                 (#x6C52 . #x7172)      ; <CJK>
                 (#x6C53 . #x714A)      ; <CJK>
                 (#x6C54 . #x7178)      ; <CJK>
                 (#x6C55 . #x717A)      ; <CJK>
                 (#x6C56 . #x7198)      ; <CJK>
                 (#x6C57 . #x71B3)      ; <CJK>
                 (#x6C58 . #x71B5)      ; <CJK>
                 (#x6C59 . #x71A8)      ; <CJK>
                 (#x6C5A . #x71A0)      ; <CJK>
                 (#x6C5B . #x71E0)      ; <CJK>
                 (#x6C5C . #x71D4)      ; <CJK>
                 (#x6C5D . #x71E7)      ; <CJK>
                 (#x6C5E . #x71F9)      ; <CJK>
                 (#x6C5F . #x721D)      ; <CJK>
                 (#x6C60 . #x7228)      ; <CJK>
                 (#x6C61 . #x706C)      ; <CJK>
                 (#x6C62 . #x7118)      ; <CJK>
                 (#x6C63 . #x7166)      ; <CJK>
                 (#x6C64 . #x71B9)      ; <CJK>
                 (#x6C65 . #x623E)      ; <CJK>
                 (#x6C66 . #x623D)      ; <CJK>
                 (#x6C67 . #x6243)      ; <CJK>
                 (#x6C68 . #x6248)      ; <CJK>
                 (#x6C69 . #x6249)      ; <CJK>
                 (#x6C6A . #x793B)      ; <CJK>
                 (#x6C6B . #x7940)      ; <CJK>
                 (#x6C6C . #x7946)      ; <CJK>
                 (#x6C6D . #x7949)      ; <CJK>
                 (#x6C6E . #x795B)      ; <CJK>
                 (#x6C6F . #x795C)      ; <CJK>
                 (#x6C70 . #x7953)      ; <CJK>
                 (#x6C71 . #x795A)      ; <CJK>
                 (#x6C72 . #x7962)      ; <CJK>
                 (#x6C73 . #x7957)      ; <CJK>
                 (#x6C74 . #x7960)      ; <CJK>
                 (#x6C75 . #x796F)      ; <CJK>
                 (#x6C76 . #x7967)      ; <CJK>
                 (#x6C77 . #x797A)      ; <CJK>
                 (#x6C78 . #x7985)      ; <CJK>
                 (#x6C79 . #x798A)      ; <CJK>
                 (#x6C7A . #x799A)      ; <CJK>
                 (#x6C7B . #x79A7)      ; <CJK>
                 (#x6C7C . #x79B3)      ; <CJK>
                 (#x6C7D . #x5FD1)      ; <CJK>
                 (#x6C7E . #x5FD0)      ; <CJK>
                 (#x6D21 . #x603C)      ; <CJK>
                 (#x6D22 . #x605D)      ; <CJK>
                 (#x6D23 . #x605A)      ; <CJK>
                 (#x6D24 . #x6067)      ; <CJK>
                 (#x6D25 . #x6041)      ; <CJK>
                 (#x6D26 . #x6059)      ; <CJK>
                 (#x6D27 . #x6063)      ; <CJK>
                 (#x6D28 . #x60AB)      ; <CJK>
                 (#x6D29 . #x6106)      ; <CJK>
                 (#x6D2A . #x610D)      ; <CJK>
                 (#x6D2B . #x615D)      ; <CJK>
                 (#x6D2C . #x61A9)      ; <CJK>
                 (#x6D2D . #x619D)      ; <CJK>
                 (#x6D2E . #x61CB)      ; <CJK>
                 (#x6D2F . #x61D1)      ; <CJK>
                 (#x6D30 . #x6206)      ; <CJK>
                 (#x6D31 . #x8080)      ; <CJK>
                 (#x6D32 . #x807F)      ; <CJK>
                 (#x6D33 . #x6C93)      ; <CJK>
                 (#x6D34 . #x6CF6)      ; <CJK>
                 (#x6D35 . #x6DFC)      ; <CJK>
                 (#x6D36 . #x77F6)      ; <CJK>
                 (#x6D37 . #x77F8)      ; <CJK>
                 (#x6D38 . #x7800)      ; <CJK>
                 (#x6D39 . #x7809)      ; <CJK>
                 (#x6D3A . #x7817)      ; <CJK>
                 (#x6D3B . #x7818)      ; <CJK>
                 (#x6D3C . #x7811)      ; <CJK>
                 (#x6D3D . #x65AB)      ; <CJK>
                 (#x6D3E . #x782D)      ; <CJK>
                 (#x6D3F . #x781C)      ; <CJK>
                 (#x6D40 . #x781D)      ; <CJK>
                 (#x6D41 . #x7839)      ; <CJK>
                 (#x6D42 . #x783A)      ; <CJK>
                 (#x6D43 . #x783B)      ; <CJK>
                 (#x6D44 . #x781F)      ; <CJK>
                 (#x6D45 . #x783C)      ; <CJK>
                 (#x6D46 . #x7825)      ; <CJK>
                 (#x6D47 . #x782C)      ; <CJK>
                 (#x6D48 . #x7823)      ; <CJK>
                 (#x6D49 . #x7829)      ; <CJK>
                 (#x6D4A . #x784E)      ; <CJK>
                 (#x6D4B . #x786D)      ; <CJK>
                 (#x6D4C . #x7856)      ; <CJK>
                 (#x6D4D . #x7857)      ; <CJK>
                 (#x6D4E . #x7826)      ; <CJK>
                 (#x6D4F . #x7850)      ; <CJK>
                 (#x6D50 . #x7847)      ; <CJK>
                 (#x6D51 . #x784C)      ; <CJK>
                 (#x6D52 . #x786A)      ; <CJK>
                 (#x6D53 . #x789B)      ; <CJK>
                 (#x6D54 . #x7893)      ; <CJK>
                 (#x6D55 . #x789A)      ; <CJK>
                 (#x6D56 . #x7887)      ; <CJK>
                 (#x6D57 . #x789C)      ; <CJK>
                 (#x6D58 . #x78A1)      ; <CJK>
                 (#x6D59 . #x78A3)      ; <CJK>
                 (#x6D5A . #x78B2)      ; <CJK>
                 (#x6D5B . #x78B9)      ; <CJK>
                 (#x6D5C . #x78A5)      ; <CJK>
                 (#x6D5D . #x78D4)      ; <CJK>
                 (#x6D5E . #x78D9)      ; <CJK>
                 (#x6D5F . #x78C9)      ; <CJK>
                 (#x6D60 . #x78EC)      ; <CJK>
                 (#x6D61 . #x78F2)      ; <CJK>
                 (#x6D62 . #x7905)      ; <CJK>
                 (#x6D63 . #x78F4)      ; <CJK>
                 (#x6D64 . #x7913)      ; <CJK>
                 (#x6D65 . #x7924)      ; <CJK>
                 (#x6D66 . #x791E)      ; <CJK>
                 (#x6D67 . #x7934)      ; <CJK>
                 (#x6D68 . #x9F9B)      ; <CJK>
                 (#x6D69 . #x9EF9)      ; <CJK>
                 (#x6D6A . #x9EFB)      ; <CJK>
                 (#x6D6B . #x9EFC)      ; <CJK>
                 (#x6D6C . #x76F1)      ; <CJK>
                 (#x6D6D . #x7704)      ; <CJK>
                 (#x6D6E . #x770D)      ; <CJK>
                 (#x6D6F . #x76F9)      ; <CJK>
                 (#x6D70 . #x7707)      ; <CJK>
                 (#x6D71 . #x7708)      ; <CJK>
                 (#x6D72 . #x771A)      ; <CJK>
                 (#x6D73 . #x7722)      ; <CJK>
                 (#x6D74 . #x7719)      ; <CJK>
                 (#x6D75 . #x772D)      ; <CJK>
                 (#x6D76 . #x7726)      ; <CJK>
                 (#x6D77 . #x7735)      ; <CJK>
                 (#x6D78 . #x7738)      ; <CJK>
                 (#x6D79 . #x7750)      ; <CJK>
                 (#x6D7A . #x7751)      ; <CJK>
                 (#x6D7B . #x7747)      ; <CJK>
                 (#x6D7C . #x7743)      ; <CJK>
                 (#x6D7D . #x775A)      ; <CJK>
                 (#x6D7E . #x7768)      ; <CJK>
                 (#x6E21 . #x7762)      ; <CJK>
                 (#x6E22 . #x7765)      ; <CJK>
                 (#x6E23 . #x777F)      ; <CJK>
                 (#x6E24 . #x778D)      ; <CJK>
                 (#x6E25 . #x777D)      ; <CJK>
                 (#x6E26 . #x7780)      ; <CJK>
                 (#x6E27 . #x778C)      ; <CJK>
                 (#x6E28 . #x7791)      ; <CJK>
                 (#x6E29 . #x779F)      ; <CJK>
                 (#x6E2A . #x77A0)      ; <CJK>
                 (#x6E2B . #x77B0)      ; <CJK>
                 (#x6E2C . #x77B5)      ; <CJK>
                 (#x6E2D . #x77BD)      ; <CJK>
                 (#x6E2E . #x753A)      ; <CJK>
                 (#x6E2F . #x7540)      ; <CJK>
                 (#x6E30 . #x754E)      ; <CJK>
                 (#x6E31 . #x754B)      ; <CJK>
                 (#x6E32 . #x7548)      ; <CJK>
                 (#x6E33 . #x755B)      ; <CJK>
                 (#x6E34 . #x7572)      ; <CJK>
                 (#x6E35 . #x7579)      ; <CJK>
                 (#x6E36 . #x7583)      ; <CJK>
                 (#x6E37 . #x7F58)      ; <CJK>
                 (#x6E38 . #x7F61)      ; <CJK>
                 (#x6E39 . #x7F5F)      ; <CJK>
                 (#x6E3A . #x8A48)      ; <CJK>
                 (#x6E3B . #x7F68)      ; <CJK>
                 (#x6E3C . #x7F74)      ; <CJK>
                 (#x6E3D . #x7F71)      ; <CJK>
                 (#x6E3E . #x7F79)      ; <CJK>
                 (#x6E3F . #x7F81)      ; <CJK>
                 (#x6E40 . #x7F7E)      ; <CJK>
                 (#x6E41 . #x76CD)      ; <CJK>
                 (#x6E42 . #x76E5)      ; <CJK>
                 (#x6E43 . #x8832)      ; <CJK>
                 (#x6E44 . #x9485)      ; <CJK>
                 (#x6E45 . #x9486)      ; <CJK>
                 (#x6E46 . #x9487)      ; <CJK>
                 (#x6E47 . #x948B)      ; <CJK>
                 (#x6E48 . #x948A)      ; <CJK>
                 (#x6E49 . #x948C)      ; <CJK>
                 (#x6E4A . #x948D)      ; <CJK>
                 (#x6E4B . #x948F)      ; <CJK>
                 (#x6E4C . #x9490)      ; <CJK>
                 (#x6E4D . #x9494)      ; <CJK>
                 (#x6E4E . #x9497)      ; <CJK>
                 (#x6E4F . #x9495)      ; <CJK>
                 (#x6E50 . #x949A)      ; <CJK>
                 (#x6E51 . #x949B)      ; <CJK>
                 (#x6E52 . #x949C)      ; <CJK>
                 (#x6E53 . #x94A3)      ; <CJK>
                 (#x6E54 . #x94A4)      ; <CJK>
                 (#x6E55 . #x94AB)      ; <CJK>
                 (#x6E56 . #x94AA)      ; <CJK>
                 (#x6E57 . #x94AD)      ; <CJK>
                 (#x6E58 . #x94AC)      ; <CJK>
                 (#x6E59 . #x94AF)      ; <CJK>
                 (#x6E5A . #x94B0)      ; <CJK>
                 (#x6E5B . #x94B2)      ; <CJK>
                 (#x6E5C . #x94B4)      ; <CJK>
                 (#x6E5D . #x94B6)      ; <CJK>
                 (#x6E5E . #x94B7)      ; <CJK>
                 (#x6E5F . #x94B8)      ; <CJK>
                 (#x6E60 . #x94B9)      ; <CJK>
                 (#x6E61 . #x94BA)      ; <CJK>
                 (#x6E62 . #x94BC)      ; <CJK>
                 (#x6E63 . #x94BD)      ; <CJK>
                 (#x6E64 . #x94BF)      ; <CJK>
                 (#x6E65 . #x94C4)      ; <CJK>
                 (#x6E66 . #x94C8)      ; <CJK>
                 (#x6E67 . #x94C9)      ; <CJK>
                 (#x6E68 . #x94CA)      ; <CJK>
                 (#x6E69 . #x94CB)      ; <CJK>
                 (#x6E6A . #x94CC)      ; <CJK>
                 (#x6E6B . #x94CD)      ; <CJK>
                 (#x6E6C . #x94CE)      ; <CJK>
                 (#x6E6D . #x94D0)      ; <CJK>
                 (#x6E6E . #x94D1)      ; <CJK>
                 (#x6E6F . #x94D2)      ; <CJK>
                 (#x6E70 . #x94D5)      ; <CJK>
                 (#x6E71 . #x94D6)      ; <CJK>
                 (#x6E72 . #x94D7)      ; <CJK>
                 (#x6E73 . #x94D9)      ; <CJK>
                 (#x6E74 . #x94D8)      ; <CJK>
                 (#x6E75 . #x94DB)      ; <CJK>
                 (#x6E76 . #x94DE)      ; <CJK>
                 (#x6E77 . #x94DF)      ; <CJK>
                 (#x6E78 . #x94E0)      ; <CJK>
                 (#x6E79 . #x94E2)      ; <CJK>
                 (#x6E7A . #x94E4)      ; <CJK>
                 (#x6E7B . #x94E5)      ; <CJK>
                 (#x6E7C . #x94E7)      ; <CJK>
                 (#x6E7D . #x94E8)      ; <CJK>
                 (#x6E7E . #x94EA)      ; <CJK>
                 (#x6F21 . #x94E9)      ; <CJK>
                 (#x6F22 . #x94EB)      ; <CJK>
                 (#x6F23 . #x94EE)      ; <CJK>
                 (#x6F24 . #x94EF)      ; <CJK>
                 (#x6F25 . #x94F3)      ; <CJK>
                 (#x6F26 . #x94F4)      ; <CJK>
                 (#x6F27 . #x94F5)      ; <CJK>
                 (#x6F28 . #x94F7)      ; <CJK>
                 (#x6F29 . #x94F9)      ; <CJK>
                 (#x6F2A . #x94FC)      ; <CJK>
                 (#x6F2B . #x94FD)      ; <CJK>
                 (#x6F2C . #x94FF)      ; <CJK>
                 (#x6F2D . #x9503)      ; <CJK>
                 (#x6F2E . #x9502)      ; <CJK>
                 (#x6F2F . #x9506)      ; <CJK>
                 (#x6F30 . #x9507)      ; <CJK>
                 (#x6F31 . #x9509)      ; <CJK>
                 (#x6F32 . #x950A)      ; <CJK>
                 (#x6F33 . #x950D)      ; <CJK>
                 (#x6F34 . #x950E)      ; <CJK>
                 (#x6F35 . #x950F)      ; <CJK>
                 (#x6F36 . #x9512)      ; <CJK>
                 (#x6F37 . #x9513)      ; <CJK>
                 (#x6F38 . #x9514)      ; <CJK>
                 (#x6F39 . #x9515)      ; <CJK>
                 (#x6F3A . #x9516)      ; <CJK>
                 (#x6F3B . #x9518)      ; <CJK>
                 (#x6F3C . #x951B)      ; <CJK>
                 (#x6F3D . #x951D)      ; <CJK>
                 (#x6F3E . #x951E)      ; <CJK>
                 (#x6F3F . #x951F)      ; <CJK>
                 (#x6F40 . #x9522)      ; <CJK>
                 (#x6F41 . #x952A)      ; <CJK>
                 (#x6F42 . #x952B)      ; <CJK>
                 (#x6F43 . #x9529)      ; <CJK>
                 (#x6F44 . #x952C)      ; <CJK>
                 (#x6F45 . #x9531)      ; <CJK>
                 (#x6F46 . #x9532)      ; <CJK>
                 (#x6F47 . #x9534)      ; <CJK>
                 (#x6F48 . #x9536)      ; <CJK>
                 (#x6F49 . #x9537)      ; <CJK>
                 (#x6F4A . #x9538)      ; <CJK>
                 (#x6F4B . #x953C)      ; <CJK>
                 (#x6F4C . #x953E)      ; <CJK>
                 (#x6F4D . #x953F)      ; <CJK>
                 (#x6F4E . #x9542)      ; <CJK>
                 (#x6F4F . #x9535)      ; <CJK>
                 (#x6F50 . #x9544)      ; <CJK>
                 (#x6F51 . #x9545)      ; <CJK>
                 (#x6F52 . #x9546)      ; <CJK>
                 (#x6F53 . #x9549)      ; <CJK>
                 (#x6F54 . #x954C)      ; <CJK>
                 (#x6F55 . #x954E)      ; <CJK>
                 (#x6F56 . #x954F)      ; <CJK>
                 (#x6F57 . #x9552)      ; <CJK>
                 (#x6F58 . #x9553)      ; <CJK>
                 (#x6F59 . #x9554)      ; <CJK>
                 (#x6F5A . #x9556)      ; <CJK>
                 (#x6F5B . #x9557)      ; <CJK>
                 (#x6F5C . #x9558)      ; <CJK>
                 (#x6F5D . #x9559)      ; <CJK>
                 (#x6F5E . #x955B)      ; <CJK>
                 (#x6F5F . #x955E)      ; <CJK>
                 (#x6F60 . #x955F)      ; <CJK>
                 (#x6F61 . #x955D)      ; <CJK>
                 (#x6F62 . #x9561)      ; <CJK>
                 (#x6F63 . #x9562)      ; <CJK>
                 (#x6F64 . #x9564)      ; <CJK>
                 (#x6F65 . #x9565)      ; <CJK>
                 (#x6F66 . #x9566)      ; <CJK>
                 (#x6F67 . #x9567)      ; <CJK>
                 (#x6F68 . #x9568)      ; <CJK>
                 (#x6F69 . #x9569)      ; <CJK>
                 (#x6F6A . #x956A)      ; <CJK>
                 (#x6F6B . #x956B)      ; <CJK>
                 (#x6F6C . #x956C)      ; <CJK>
                 (#x6F6D . #x956F)      ; <CJK>
                 (#x6F6E . #x9571)      ; <CJK>
                 (#x6F6F . #x9572)      ; <CJK>
                 (#x6F70 . #x9573)      ; <CJK>
                 (#x6F71 . #x953A)      ; <CJK>
                 (#x6F72 . #x77E7)      ; <CJK>
                 (#x6F73 . #x77EC)      ; <CJK>
                 (#x6F74 . #x96C9)      ; <CJK>
                 (#x6F75 . #x79D5)      ; <CJK>
                 (#x6F76 . #x79ED)      ; <CJK>
                 (#x6F77 . #x79E3)      ; <CJK>
                 (#x6F78 . #x79EB)      ; <CJK>
                 (#x6F79 . #x7A06)      ; <CJK>
                 (#x6F7A . #x5D47)      ; <CJK>
                 (#x6F7B . #x7A03)      ; <CJK>
                 (#x6F7C . #x7A02)      ; <CJK>
                 (#x6F7D . #x7A1E)      ; <CJK>
                 (#x6F7E . #x7A14)      ; <CJK>
                 (#x7021 . #x7A39)      ; <CJK>
                 (#x7022 . #x7A37)      ; <CJK>
                 (#x7023 . #x7A51)      ; <CJK>
                 (#x7024 . #x9ECF)      ; <CJK>
                 (#x7025 . #x99A5)      ; <CJK>
                 (#x7026 . #x7A70)      ; <CJK>
                 (#x7027 . #x7688)      ; <CJK>
                 (#x7028 . #x768E)      ; <CJK>
                 (#x7029 . #x7693)      ; <CJK>
                 (#x702A . #x7699)      ; <CJK>
                 (#x702B . #x76A4)      ; <CJK>
                 (#x702C . #x74DE)      ; <CJK>
                 (#x702D . #x74E0)      ; <CJK>
                 (#x702E . #x752C)      ; <CJK>
                 (#x702F . #x9E20)      ; <CJK>
                 (#x7030 . #x9E22)      ; <CJK>
                 (#x7031 . #x9E28)      ; <CJK>
                 (#x7032 . #x9E29)      ; <CJK>
                 (#x7033 . #x9E2A)      ; <CJK>
                 (#x7034 . #x9E2B)      ; <CJK>
                 (#x7035 . #x9E2C)      ; <CJK>
                 (#x7036 . #x9E32)      ; <CJK>
                 (#x7037 . #x9E31)      ; <CJK>
                 (#x7038 . #x9E36)      ; <CJK>
                 (#x7039 . #x9E38)      ; <CJK>
                 (#x703A . #x9E37)      ; <CJK>
                 (#x703B . #x9E39)      ; <CJK>
                 (#x703C . #x9E3A)      ; <CJK>
                 (#x703D . #x9E3E)      ; <CJK>
                 (#x703E . #x9E41)      ; <CJK>
                 (#x703F . #x9E42)      ; <CJK>
                 (#x7040 . #x9E44)      ; <CJK>
                 (#x7041 . #x9E46)      ; <CJK>
                 (#x7042 . #x9E47)      ; <CJK>
                 (#x7043 . #x9E48)      ; <CJK>
                 (#x7044 . #x9E49)      ; <CJK>
                 (#x7045 . #x9E4B)      ; <CJK>
                 (#x7046 . #x9E4C)      ; <CJK>
                 (#x7047 . #x9E4E)      ; <CJK>
                 (#x7048 . #x9E51)      ; <CJK>
                 (#x7049 . #x9E55)      ; <CJK>
                 (#x704A . #x9E57)      ; <CJK>
                 (#x704B . #x9E5A)      ; <CJK>
                 (#x704C . #x9E5B)      ; <CJK>
                 (#x704D . #x9E5C)      ; <CJK>
                 (#x704E . #x9E5E)      ; <CJK>
                 (#x704F . #x9E63)      ; <CJK>
                 (#x7050 . #x9E66)      ; <CJK>
                 (#x7051 . #x9E67)      ; <CJK>
                 (#x7052 . #x9E68)      ; <CJK>
                 (#x7053 . #x9E69)      ; <CJK>
                 (#x7054 . #x9E6A)      ; <CJK>
                 (#x7055 . #x9E6B)      ; <CJK>
                 (#x7056 . #x9E6C)      ; <CJK>
                 (#x7057 . #x9E71)      ; <CJK>
                 (#x7058 . #x9E6D)      ; <CJK>
                 (#x7059 . #x9E73)      ; <CJK>
                 (#x705A . #x7592)      ; <CJK>
                 (#x705B . #x7594)      ; <CJK>
                 (#x705C . #x7596)      ; <CJK>
                 (#x705D . #x75A0)      ; <CJK>
                 (#x705E . #x759D)      ; <CJK>
                 (#x705F . #x75AC)      ; <CJK>
                 (#x7060 . #x75A3)      ; <CJK>
                 (#x7061 . #x75B3)      ; <CJK>
                 (#x7062 . #x75B4)      ; <CJK>
                 (#x7063 . #x75B8)      ; <CJK>
                 (#x7064 . #x75C4)      ; <CJK>
                 (#x7065 . #x75B1)      ; <CJK>
                 (#x7066 . #x75B0)      ; <CJK>
                 (#x7067 . #x75C3)      ; <CJK>
                 (#x7068 . #x75C2)      ; <CJK>
                 (#x7069 . #x75D6)      ; <CJK>
                 (#x706A . #x75CD)      ; <CJK>
                 (#x706B . #x75E3)      ; <CJK>
                 (#x706C . #x75E8)      ; <CJK>
                 (#x706D . #x75E6)      ; <CJK>
                 (#x706E . #x75E4)      ; <CJK>
                 (#x706F . #x75EB)      ; <CJK>
                 (#x7070 . #x75E7)      ; <CJK>
                 (#x7071 . #x7603)      ; <CJK>
                 (#x7072 . #x75F1)      ; <CJK>
                 (#x7073 . #x75FC)      ; <CJK>
                 (#x7074 . #x75FF)      ; <CJK>
                 (#x7075 . #x7610)      ; <CJK>
                 (#x7076 . #x7600)      ; <CJK>
                 (#x7077 . #x7605)      ; <CJK>
                 (#x7078 . #x760C)      ; <CJK>
                 (#x7079 . #x7617)      ; <CJK>
                 (#x707A . #x760A)      ; <CJK>
                 (#x707B . #x7625)      ; <CJK>
                 (#x707C . #x7618)      ; <CJK>
                 (#x707D . #x7615)      ; <CJK>
                 (#x707E . #x7619)      ; <CJK>
                 (#x7121 . #x761B)      ; <CJK>
                 (#x7122 . #x763C)      ; <CJK>
                 (#x7123 . #x7622)      ; <CJK>
                 (#x7124 . #x7620)      ; <CJK>
                 (#x7125 . #x7640)      ; <CJK>
                 (#x7126 . #x762D)      ; <CJK>
                 (#x7127 . #x7630)      ; <CJK>
                 (#x7128 . #x763F)      ; <CJK>
                 (#x7129 . #x7635)      ; <CJK>
                 (#x712A . #x7643)      ; <CJK>
                 (#x712B . #x763E)      ; <CJK>
                 (#x712C . #x7633)      ; <CJK>
                 (#x712D . #x764D)      ; <CJK>
                 (#x712E . #x765E)      ; <CJK>
                 (#x712F . #x7654)      ; <CJK>
                 (#x7130 . #x765C)      ; <CJK>
                 (#x7131 . #x7656)      ; <CJK>
                 (#x7132 . #x766B)      ; <CJK>
                 (#x7133 . #x766F)      ; <CJK>
                 (#x7134 . #x7FCA)      ; <CJK>
                 (#x7135 . #x7AE6)      ; <CJK>
                 (#x7136 . #x7A78)      ; <CJK>
                 (#x7137 . #x7A79)      ; <CJK>
                 (#x7138 . #x7A80)      ; <CJK>
                 (#x7139 . #x7A86)      ; <CJK>
                 (#x713A . #x7A88)      ; <CJK>
                 (#x713B . #x7A95)      ; <CJK>
                 (#x713C . #x7AA6)      ; <CJK>
                 (#x713D . #x7AA0)      ; <CJK>
                 (#x713E . #x7AAC)      ; <CJK>
                 (#x713F . #x7AA8)      ; <CJK>
                 (#x7140 . #x7AAD)      ; <CJK>
                 (#x7141 . #x7AB3)      ; <CJK>
                 (#x7142 . #x8864)      ; <CJK>
                 (#x7143 . #x8869)      ; <CJK>
                 (#x7144 . #x8872)      ; <CJK>
                 (#x7145 . #x887D)      ; <CJK>
                 (#x7146 . #x887F)      ; <CJK>
                 (#x7147 . #x8882)      ; <CJK>
                 (#x7148 . #x88A2)      ; <CJK>
                 (#x7149 . #x88C6)      ; <CJK>
                 (#x714A . #x88B7)      ; <CJK>
                 (#x714B . #x88BC)      ; <CJK>
                 (#x714C . #x88C9)      ; <CJK>
                 (#x714D . #x88E2)      ; <CJK>
                 (#x714E . #x88CE)      ; <CJK>
                 (#x714F . #x88E3)      ; <CJK>
                 (#x7150 . #x88E5)      ; <CJK>
                 (#x7151 . #x88F1)      ; <CJK>
                 (#x7152 . #x891A)      ; <CJK>
                 (#x7153 . #x88FC)      ; <CJK>
                 (#x7154 . #x88E8)      ; <CJK>
                 (#x7155 . #x88FE)      ; <CJK>
                 (#x7156 . #x88F0)      ; <CJK>
                 (#x7157 . #x8921)      ; <CJK>
                 (#x7158 . #x8919)      ; <CJK>
                 (#x7159 . #x8913)      ; <CJK>
                 (#x715A . #x891B)      ; <CJK>
                 (#x715B . #x890A)      ; <CJK>
                 (#x715C . #x8934)      ; <CJK>
                 (#x715D . #x892B)      ; <CJK>
                 (#x715E . #x8936)      ; <CJK>
                 (#x715F . #x8941)      ; <CJK>
                 (#x7160 . #x8966)      ; <CJK>
                 (#x7161 . #x897B)      ; <CJK>
                 (#x7162 . #x758B)      ; <CJK>
                 (#x7163 . #x80E5)      ; <CJK>
                 (#x7164 . #x76B2)      ; <CJK>
                 (#x7165 . #x76B4)      ; <CJK>
                 (#x7166 . #x77DC)      ; <CJK>
                 (#x7167 . #x8012)      ; <CJK>
                 (#x7168 . #x8014)      ; <CJK>
                 (#x7169 . #x8016)      ; <CJK>
                 (#x716A . #x801C)      ; <CJK>
                 (#x716B . #x8020)      ; <CJK>
                 (#x716C . #x8022)      ; <CJK>
                 (#x716D . #x8025)      ; <CJK>
                 (#x716E . #x8026)      ; <CJK>
                 (#x716F . #x8027)      ; <CJK>
                 (#x7170 . #x8029)      ; <CJK>
                 (#x7171 . #x8028)      ; <CJK>
                 (#x7172 . #x8031)      ; <CJK>
                 (#x7173 . #x800B)      ; <CJK>
                 (#x7174 . #x8035)      ; <CJK>
                 (#x7175 . #x8043)      ; <CJK>
                 (#x7176 . #x8046)      ; <CJK>
                 (#x7177 . #x804D)      ; <CJK>
                 (#x7178 . #x8052)      ; <CJK>
                 (#x7179 . #x8069)      ; <CJK>
                 (#x717A . #x8071)      ; <CJK>
                 (#x717B . #x8983)      ; <CJK>
                 (#x717C . #x9878)      ; <CJK>
                 (#x717D . #x9880)      ; <CJK>
                 (#x717E . #x9883)      ; <CJK>
                 (#x7221 . #x9889)      ; <CJK>
                 (#x7222 . #x988C)      ; <CJK>
                 (#x7223 . #x988D)      ; <CJK>
                 (#x7224 . #x988F)      ; <CJK>
                 (#x7225 . #x9894)      ; <CJK>
                 (#x7226 . #x989A)      ; <CJK>
                 (#x7227 . #x989B)      ; <CJK>
                 (#x7228 . #x989E)      ; <CJK>
                 (#x7229 . #x989F)      ; <CJK>
                 (#x722A . #x98A1)      ; <CJK>
                 (#x722B . #x98A2)      ; <CJK>
                 (#x722C . #x98A5)      ; <CJK>
                 (#x722D . #x98A6)      ; <CJK>
                 (#x722E . #x864D)      ; <CJK>
                 (#x722F . #x8654)      ; <CJK>
                 (#x7230 . #x866C)      ; <CJK>
                 (#x7231 . #x866E)      ; <CJK>
                 (#x7232 . #x867F)      ; <CJK>
                 (#x7233 . #x867A)      ; <CJK>
                 (#x7234 . #x867C)      ; <CJK>
                 (#x7235 . #x867B)      ; <CJK>
                 (#x7236 . #x86A8)      ; <CJK>
                 (#x7237 . #x868D)      ; <CJK>
                 (#x7238 . #x868B)      ; <CJK>
                 (#x7239 . #x86AC)      ; <CJK>
                 (#x723A . #x869D)      ; <CJK>
                 (#x723B . #x86A7)      ; <CJK>
                 (#x723C . #x86A3)      ; <CJK>
                 (#x723D . #x86AA)      ; <CJK>
                 (#x723E . #x8693)      ; <CJK>
                 (#x723F . #x86A9)      ; <CJK>
                 (#x7240 . #x86B6)      ; <CJK>
                 (#x7241 . #x86C4)      ; <CJK>
                 (#x7242 . #x86B5)      ; <CJK>
                 (#x7243 . #x86CE)      ; <CJK>
                 (#x7244 . #x86B0)      ; <CJK>
                 (#x7245 . #x86BA)      ; <CJK>
                 (#x7246 . #x86B1)      ; <CJK>
                 (#x7247 . #x86AF)      ; <CJK>
                 (#x7248 . #x86C9)      ; <CJK>
                 (#x7249 . #x86CF)      ; <CJK>
                 (#x724A . #x86B4)      ; <CJK>
                 (#x724B . #x86E9)      ; <CJK>
                 (#x724C . #x86F1)      ; <CJK>
                 (#x724D . #x86F2)      ; <CJK>
                 (#x724E . #x86ED)      ; <CJK>
                 (#x724F . #x86F3)      ; <CJK>
                 (#x7250 . #x86D0)      ; <CJK>
                 (#x7251 . #x8713)      ; <CJK>
                 (#x7252 . #x86DE)      ; <CJK>
                 (#x7253 . #x86F4)      ; <CJK>
                 (#x7254 . #x86DF)      ; <CJK>
                 (#x7255 . #x86D8)      ; <CJK>
                 (#x7256 . #x86D1)      ; <CJK>
                 (#x7257 . #x8703)      ; <CJK>
                 (#x7258 . #x8707)      ; <CJK>
                 (#x7259 . #x86F8)      ; <CJK>
                 (#x725A . #x8708)      ; <CJK>
                 (#x725B . #x870A)      ; <CJK>
                 (#x725C . #x870D)      ; <CJK>
                 (#x725D . #x8709)      ; <CJK>
                 (#x725E . #x8723)      ; <CJK>
                 (#x725F . #x873B)      ; <CJK>
                 (#x7260 . #x871E)      ; <CJK>
                 (#x7261 . #x8725)      ; <CJK>
                 (#x7262 . #x872E)      ; <CJK>
                 (#x7263 . #x871A)      ; <CJK>
                 (#x7264 . #x873E)      ; <CJK>
                 (#x7265 . #x8748)      ; <CJK>
                 (#x7266 . #x8734)      ; <CJK>
                 (#x7267 . #x8731)      ; <CJK>
                 (#x7268 . #x8729)      ; <CJK>
                 (#x7269 . #x8737)      ; <CJK>
                 (#x726A . #x873F)      ; <CJK>
                 (#x726B . #x8782)      ; <CJK>
                 (#x726C . #x8722)      ; <CJK>
                 (#x726D . #x877D)      ; <CJK>
                 (#x726E . #x877E)      ; <CJK>
                 (#x726F . #x877B)      ; <CJK>
                 (#x7270 . #x8760)      ; <CJK>
                 (#x7271 . #x8770)      ; <CJK>
                 (#x7272 . #x874C)      ; <CJK>
                 (#x7273 . #x876E)      ; <CJK>
                 (#x7274 . #x878B)      ; <CJK>
                 (#x7275 . #x8753)      ; <CJK>
                 (#x7276 . #x8763)      ; <CJK>
                 (#x7277 . #x877C)      ; <CJK>
                 (#x7278 . #x8764)      ; <CJK>
                 (#x7279 . #x8759)      ; <CJK>
                 (#x727A . #x8765)      ; <CJK>
                 (#x727B . #x8793)      ; <CJK>
                 (#x727C . #x87AF)      ; <CJK>
                 (#x727D . #x87A8)      ; <CJK>
                 (#x727E . #x87D2)      ; <CJK>
                 (#x7321 . #x87C6)      ; <CJK>
                 (#x7322 . #x8788)      ; <CJK>
                 (#x7323 . #x8785)      ; <CJK>
                 (#x7324 . #x87AD)      ; <CJK>
                 (#x7325 . #x8797)      ; <CJK>
                 (#x7326 . #x8783)      ; <CJK>
                 (#x7327 . #x87AB)      ; <CJK>
                 (#x7328 . #x87E5)      ; <CJK>
                 (#x7329 . #x87AC)      ; <CJK>
                 (#x732A . #x87B5)      ; <CJK>
                 (#x732B . #x87B3)      ; <CJK>
                 (#x732C . #x87CB)      ; <CJK>
                 (#x732D . #x87D3)      ; <CJK>
                 (#x732E . #x87BD)      ; <CJK>
                 (#x732F . #x87D1)      ; <CJK>
                 (#x7330 . #x87C0)      ; <CJK>
                 (#x7331 . #x87CA)      ; <CJK>
                 (#x7332 . #x87DB)      ; <CJK>
                 (#x7333 . #x87EA)      ; <CJK>
                 (#x7334 . #x87E0)      ; <CJK>
                 (#x7335 . #x87EE)      ; <CJK>
                 (#x7336 . #x8816)      ; <CJK>
                 (#x7337 . #x8813)      ; <CJK>
                 (#x7338 . #x87FE)      ; <CJK>
                 (#x7339 . #x880A)      ; <CJK>
                 (#x733A . #x881B)      ; <CJK>
                 (#x733B . #x8821)      ; <CJK>
                 (#x733C . #x8839)      ; <CJK>
                 (#x733D . #x883C)      ; <CJK>
                 (#x733E . #x7F36)      ; <CJK>
                 (#x733F . #x7F42)      ; <CJK>
                 (#x7340 . #x7F44)      ; <CJK>
                 (#x7341 . #x7F45)      ; <CJK>
                 (#x7342 . #x8210)      ; <CJK>
                 (#x7343 . #x7AFA)      ; <CJK>
                 (#x7344 . #x7AFD)      ; <CJK>
                 (#x7345 . #x7B08)      ; <CJK>
                 (#x7346 . #x7B03)      ; <CJK>
                 (#x7347 . #x7B04)      ; <CJK>
                 (#x7348 . #x7B15)      ; <CJK>
                 (#x7349 . #x7B0A)      ; <CJK>
                 (#x734A . #x7B2B)      ; <CJK>
                 (#x734B . #x7B0F)      ; <CJK>
                 (#x734C . #x7B47)      ; <CJK>
                 (#x734D . #x7B38)      ; <CJK>
                 (#x734E . #x7B2A)      ; <CJK>
                 (#x734F . #x7B19)      ; <CJK>
                 (#x7350 . #x7B2E)      ; <CJK>
                 (#x7351 . #x7B31)      ; <CJK>
                 (#x7352 . #x7B20)      ; <CJK>
                 (#x7353 . #x7B25)      ; <CJK>
                 (#x7354 . #x7B24)      ; <CJK>
                 (#x7355 . #x7B33)      ; <CJK>
                 (#x7356 . #x7B3E)      ; <CJK>
                 (#x7357 . #x7B1E)      ; <CJK>
                 (#x7358 . #x7B58)      ; <CJK>
                 (#x7359 . #x7B5A)      ; <CJK>
                 (#x735A . #x7B45)      ; <CJK>
                 (#x735B . #x7B75)      ; <CJK>
                 (#x735C . #x7B4C)      ; <CJK>
                 (#x735D . #x7B5D)      ; <CJK>
                 (#x735E . #x7B60)      ; <CJK>
                 (#x735F . #x7B6E)      ; <CJK>
                 (#x7360 . #x7B7B)      ; <CJK>
                 (#x7361 . #x7B62)      ; <CJK>
                 (#x7362 . #x7B72)      ; <CJK>
                 (#x7363 . #x7B71)      ; <CJK>
                 (#x7364 . #x7B90)      ; <CJK>
                 (#x7365 . #x7BA6)      ; <CJK>
                 (#x7366 . #x7BA7)      ; <CJK>
                 (#x7367 . #x7BB8)      ; <CJK>
                 (#x7368 . #x7BAC)      ; <CJK>
                 (#x7369 . #x7B9D)      ; <CJK>
                 (#x736A . #x7BA8)      ; <CJK>
                 (#x736B . #x7B85)      ; <CJK>
                 (#x736C . #x7BAA)      ; <CJK>
                 (#x736D . #x7B9C)      ; <CJK>
                 (#x736E . #x7BA2)      ; <CJK>
                 (#x736F . #x7BAB)      ; <CJK>
                 (#x7370 . #x7BB4)      ; <CJK>
                 (#x7371 . #x7BD1)      ; <CJK>
                 (#x7372 . #x7BC1)      ; <CJK>
                 (#x7373 . #x7BCC)      ; <CJK>
                 (#x7374 . #x7BDD)      ; <CJK>
                 (#x7375 . #x7BDA)      ; <CJK>
                 (#x7376 . #x7BE5)      ; <CJK>
                 (#x7377 . #x7BE6)      ; <CJK>
                 (#x7378 . #x7BEA)      ; <CJK>
                 (#x7379 . #x7C0C)      ; <CJK>
                 (#x737A . #x7BFE)      ; <CJK>
                 (#x737B . #x7BFC)      ; <CJK>
                 (#x737C . #x7C0F)      ; <CJK>
                 (#x737D . #x7C16)      ; <CJK>
                 (#x737E . #x7C0B)      ; <CJK>
                 (#x7421 . #x7C1F)      ; <CJK>
                 (#x7422 . #x7C2A)      ; <CJK>
                 (#x7423 . #x7C26)      ; <CJK>
                 (#x7424 . #x7C38)      ; <CJK>
                 (#x7425 . #x7C41)      ; <CJK>
                 (#x7426 . #x7C40)      ; <CJK>
                 (#x7427 . #x81FE)      ; <CJK>
                 (#x7428 . #x8201)      ; <CJK>
                 (#x7429 . #x8202)      ; <CJK>
                 (#x742A . #x8204)      ; <CJK>
                 (#x742B . #x81EC)      ; <CJK>
                 (#x742C . #x8844)      ; <CJK>
                 (#x742D . #x8221)      ; <CJK>
                 (#x742E . #x8222)      ; <CJK>
                 (#x742F . #x8223)      ; <CJK>
                 (#x7430 . #x822D)      ; <CJK>
                 (#x7431 . #x822F)      ; <CJK>
                 (#x7432 . #x8228)      ; <CJK>
                 (#x7433 . #x822B)      ; <CJK>
                 (#x7434 . #x8238)      ; <CJK>
                 (#x7435 . #x823B)      ; <CJK>
                 (#x7436 . #x8233)      ; <CJK>
                 (#x7437 . #x8234)      ; <CJK>
                 (#x7438 . #x823E)      ; <CJK>
                 (#x7439 . #x8244)      ; <CJK>
                 (#x743A . #x8249)      ; <CJK>
                 (#x743B . #x824B)      ; <CJK>
                 (#x743C . #x824F)      ; <CJK>
                 (#x743D . #x825A)      ; <CJK>
                 (#x743E . #x825F)      ; <CJK>
                 (#x743F . #x8268)      ; <CJK>
                 (#x7440 . #x887E)      ; <CJK>
                 (#x7441 . #x8885)      ; <CJK>
                 (#x7442 . #x8888)      ; <CJK>
                 (#x7443 . #x88D8)      ; <CJK>
                 (#x7444 . #x88DF)      ; <CJK>
                 (#x7445 . #x895E)      ; <CJK>
                 (#x7446 . #x7F9D)      ; <CJK>
                 (#x7447 . #x7F9F)      ; <CJK>
                 (#x7448 . #x7FA7)      ; <CJK>
                 (#x7449 . #x7FAF)      ; <CJK>
                 (#x744A . #x7FB0)      ; <CJK>
                 (#x744B . #x7FB2)      ; <CJK>
                 (#x744C . #x7C7C)      ; <CJK>
                 (#x744D . #x6549)      ; <CJK>
                 (#x744E . #x7C91)      ; <CJK>
                 (#x744F . #x7C9D)      ; <CJK>
                 (#x7450 . #x7C9C)      ; <CJK>
                 (#x7451 . #x7C9E)      ; <CJK>
                 (#x7452 . #x7CA2)      ; <CJK>
                 (#x7453 . #x7CB2)      ; <CJK>
                 (#x7454 . #x7CBC)      ; <CJK>
                 (#x7455 . #x7CBD)      ; <CJK>
                 (#x7456 . #x7CC1)      ; <CJK>
                 (#x7457 . #x7CC7)      ; <CJK>
                 (#x7458 . #x7CCC)      ; <CJK>
                 (#x7459 . #x7CCD)      ; <CJK>
                 (#x745A . #x7CC8)      ; <CJK>
                 (#x745B . #x7CC5)      ; <CJK>
                 (#x745C . #x7CD7)      ; <CJK>
                 (#x745D . #x7CE8)      ; <CJK>
                 (#x745E . #x826E)      ; <CJK>
                 (#x745F . #x66A8)      ; <CJK>
                 (#x7460 . #x7FBF)      ; <CJK>
                 (#x7461 . #x7FCE)      ; <CJK>
                 (#x7462 . #x7FD5)      ; <CJK>
                 (#x7463 . #x7FE5)      ; <CJK>
                 (#x7464 . #x7FE1)      ; <CJK>
                 (#x7465 . #x7FE6)      ; <CJK>
                 (#x7466 . #x7FE9)      ; <CJK>
                 (#x7467 . #x7FEE)      ; <CJK>
                 (#x7468 . #x7FF3)      ; <CJK>
                 (#x7469 . #x7CF8)      ; <CJK>
                 (#x746A . #x7D77)      ; <CJK>
                 (#x746B . #x7DA6)      ; <CJK>
                 (#x746C . #x7DAE)      ; <CJK>
                 (#x746D . #x7E47)      ; <CJK>
                 (#x746E . #x7E9B)      ; <CJK>
                 (#x746F . #x9EB8)      ; <CJK>
                 (#x7470 . #x9EB4)      ; <CJK>
                 (#x7471 . #x8D73)      ; <CJK>
                 (#x7472 . #x8D84)      ; <CJK>
                 (#x7473 . #x8D94)      ; <CJK>
                 (#x7474 . #x8D91)      ; <CJK>
                 (#x7475 . #x8DB1)      ; <CJK>
                 (#x7476 . #x8D67)      ; <CJK>
                 (#x7477 . #x8D6D)      ; <CJK>
                 (#x7478 . #x8C47)      ; <CJK>
                 (#x7479 . #x8C49)      ; <CJK>
                 (#x747A . #x914A)      ; <CJK>
                 (#x747B . #x9150)      ; <CJK>
                 (#x747C . #x914E)      ; <CJK>
                 (#x747D . #x914F)      ; <CJK>
                 (#x747E . #x9164)      ; <CJK>
                 (#x7521 . #x9162)      ; <CJK>
                 (#x7522 . #x9161)      ; <CJK>
                 (#x7523 . #x9170)      ; <CJK>
                 (#x7524 . #x9169)      ; <CJK>
                 (#x7525 . #x916F)      ; <CJK>
                 (#x7526 . #x917D)      ; <CJK>
                 (#x7527 . #x917E)      ; <CJK>
                 (#x7528 . #x9172)      ; <CJK>
                 (#x7529 . #x9174)      ; <CJK>
                 (#x752A . #x9179)      ; <CJK>
                 (#x752B . #x918C)      ; <CJK>
                 (#x752C . #x9185)      ; <CJK>
                 (#x752D . #x9190)      ; <CJK>
                 (#x752E . #x918D)      ; <CJK>
                 (#x752F . #x9191)      ; <CJK>
                 (#x7530 . #x91A2)      ; <CJK>
                 (#x7531 . #x91A3)      ; <CJK>
                 (#x7532 . #x91AA)      ; <CJK>
                 (#x7533 . #x91AD)      ; <CJK>
                 (#x7534 . #x91AE)      ; <CJK>
                 (#x7535 . #x91AF)      ; <CJK>
                 (#x7536 . #x91B5)      ; <CJK>
                 (#x7537 . #x91B4)      ; <CJK>
                 (#x7538 . #x91BA)      ; <CJK>
                 (#x7539 . #x8C55)      ; <CJK>
                 (#x753A . #x9E7E)      ; <CJK>
                 (#x753B . #x8DB8)      ; <CJK>
                 (#x753C . #x8DEB)      ; <CJK>
                 (#x753D . #x8E05)      ; <CJK>
                 (#x753E . #x8E59)      ; <CJK>
                 (#x753F . #x8E69)      ; <CJK>
                 (#x7540 . #x8DB5)      ; <CJK>
                 (#x7541 . #x8DBF)      ; <CJK>
                 (#x7542 . #x8DBC)      ; <CJK>
                 (#x7543 . #x8DBA)      ; <CJK>
                 (#x7544 . #x8DC4)      ; <CJK>
                 (#x7545 . #x8DD6)      ; <CJK>
                 (#x7546 . #x8DD7)      ; <CJK>
                 (#x7547 . #x8DDA)      ; <CJK>
                 (#x7548 . #x8DDE)      ; <CJK>
                 (#x7549 . #x8DCE)      ; <CJK>
                 (#x754A . #x8DCF)      ; <CJK>
                 (#x754B . #x8DDB)      ; <CJK>
                 (#x754C . #x8DC6)      ; <CJK>
                 (#x754D . #x8DEC)      ; <CJK>
                 (#x754E . #x8DF7)      ; <CJK>
                 (#x754F . #x8DF8)      ; <CJK>
                 (#x7550 . #x8DE3)      ; <CJK>
                 (#x7551 . #x8DF9)      ; <CJK>
                 (#x7552 . #x8DFB)      ; <CJK>
                 (#x7553 . #x8DE4)      ; <CJK>
                 (#x7554 . #x8E09)      ; <CJK>
                 (#x7555 . #x8DFD)      ; <CJK>
                 (#x7556 . #x8E14)      ; <CJK>
                 (#x7557 . #x8E1D)      ; <CJK>
                 (#x7558 . #x8E1F)      ; <CJK>
                 (#x7559 . #x8E2C)      ; <CJK>
                 (#x755A . #x8E2E)      ; <CJK>
                 (#x755B . #x8E23)      ; <CJK>
                 (#x755C . #x8E2F)      ; <CJK>
                 (#x755D . #x8E3A)      ; <CJK>
                 (#x755E . #x8E40)      ; <CJK>
                 (#x755F . #x8E39)      ; <CJK>
                 (#x7560 . #x8E35)      ; <CJK>
                 (#x7561 . #x8E3D)      ; <CJK>
                 (#x7562 . #x8E31)      ; <CJK>
                 (#x7563 . #x8E49)      ; <CJK>
                 (#x7564 . #x8E41)      ; <CJK>
                 (#x7565 . #x8E42)      ; <CJK>
                 (#x7566 . #x8E51)      ; <CJK>
                 (#x7567 . #x8E52)      ; <CJK>
                 (#x7568 . #x8E4A)      ; <CJK>
                 (#x7569 . #x8E70)      ; <CJK>
                 (#x756A . #x8E76)      ; <CJK>
                 (#x756B . #x8E7C)      ; <CJK>
                 (#x756C . #x8E6F)      ; <CJK>
                 (#x756D . #x8E74)      ; <CJK>
                 (#x756E . #x8E85)      ; <CJK>
                 (#x756F . #x8E8F)      ; <CJK>
                 (#x7570 . #x8E94)      ; <CJK>
                 (#x7571 . #x8E90)      ; <CJK>
                 (#x7572 . #x8E9C)      ; <CJK>
                 (#x7573 . #x8E9E)      ; <CJK>
                 (#x7574 . #x8C78)      ; <CJK>
                 (#x7575 . #x8C82)      ; <CJK>
                 (#x7576 . #x8C8A)      ; <CJK>
                 (#x7577 . #x8C85)      ; <CJK>
                 (#x7578 . #x8C98)      ; <CJK>
                 (#x7579 . #x8C94)      ; <CJK>
                 (#x757A . #x659B)      ; <CJK>
                 (#x757B . #x89D6)      ; <CJK>
                 (#x757C . #x89DE)      ; <CJK>
                 (#x757D . #x89DA)      ; <CJK>
                 (#x757E . #x89DC)      ; <CJK>
                 (#x7621 . #x89E5)      ; <CJK>
                 (#x7622 . #x89EB)      ; <CJK>
                 (#x7623 . #x89EF)      ; <CJK>
                 (#x7624 . #x8A3E)      ; <CJK>
                 (#x7625 . #x8B26)      ; <CJK>
                 (#x7626 . #x9753)      ; <CJK>
                 (#x7627 . #x96E9)      ; <CJK>
                 (#x7628 . #x96F3)      ; <CJK>
                 (#x7629 . #x96EF)      ; <CJK>
                 (#x762A . #x9706)      ; <CJK>
                 (#x762B . #x9701)      ; <CJK>
                 (#x762C . #x9708)      ; <CJK>
                 (#x762D . #x970F)      ; <CJK>
                 (#x762E . #x970E)      ; <CJK>
                 (#x762F . #x972A)      ; <CJK>
                 (#x7630 . #x972D)      ; <CJK>
                 (#x7631 . #x9730)      ; <CJK>
                 (#x7632 . #x973E)      ; <CJK>
                 (#x7633 . #x9F80)      ; <CJK>
                 (#x7634 . #x9F83)      ; <CJK>
                 (#x7635 . #x9F85)      ; <CJK>
                 (#x7636 . #x9F86)      ; <CJK>
                 (#x7637 . #x9F87)      ; <CJK>
                 (#x7638 . #x9F88)      ; <CJK>
                 (#x7639 . #x9F89)      ; <CJK>
                 (#x763A . #x9F8A)      ; <CJK>
                 (#x763B . #x9F8C)      ; <CJK>
                 (#x763C . #x9EFE)      ; <CJK>
                 (#x763D . #x9F0B)      ; <CJK>
                 (#x763E . #x9F0D)      ; <CJK>
                 (#x763F . #x96B9)      ; <CJK>
                 (#x7640 . #x96BC)      ; <CJK>
                 (#x7641 . #x96BD)      ; <CJK>
                 (#x7642 . #x96CE)      ; <CJK>
                 (#x7643 . #x96D2)      ; <CJK>
                 (#x7644 . #x77BF)      ; <CJK>
                 (#x7645 . #x96E0)      ; <CJK>
                 (#x7646 . #x928E)      ; <CJK>
                 (#x7647 . #x92AE)      ; <CJK>
                 (#x7648 . #x92C8)      ; <CJK>
                 (#x7649 . #x933E)      ; <CJK>
                 (#x764A . #x936A)      ; <CJK>
                 (#x764B . #x93CA)      ; <CJK>
                 (#x764C . #x938F)      ; <CJK>
                 (#x764D . #x943E)      ; <CJK>
                 (#x764E . #x946B)      ; <CJK>
                 (#x764F . #x9C7F)      ; <CJK>
                 (#x7650 . #x9C82)      ; <CJK>
                 (#x7651 . #x9C85)      ; <CJK>
                 (#x7652 . #x9C86)      ; <CJK>
                 (#x7653 . #x9C87)      ; <CJK>
                 (#x7654 . #x9C88)      ; <CJK>
                 (#x7655 . #x7A23)      ; <CJK>
                 (#x7656 . #x9C8B)      ; <CJK>
                 (#x7657 . #x9C8E)      ; <CJK>
                 (#x7658 . #x9C90)      ; <CJK>
                 (#x7659 . #x9C91)      ; <CJK>
                 (#x765A . #x9C92)      ; <CJK>
                 (#x765B . #x9C94)      ; <CJK>
                 (#x765C . #x9C95)      ; <CJK>
                 (#x765D . #x9C9A)      ; <CJK>
                 (#x765E . #x9C9B)      ; <CJK>
                 (#x765F . #x9C9E)      ; <CJK>
                 (#x7660 . #x9C9F)      ; <CJK>
                 (#x7661 . #x9CA0)      ; <CJK>
                 (#x7662 . #x9CA1)      ; <CJK>
                 (#x7663 . #x9CA2)      ; <CJK>
                 (#x7664 . #x9CA3)      ; <CJK>
                 (#x7665 . #x9CA5)      ; <CJK>
                 (#x7666 . #x9CA6)      ; <CJK>
                 (#x7667 . #x9CA7)      ; <CJK>
                 (#x7668 . #x9CA8)      ; <CJK>
                 (#x7669 . #x9CA9)      ; <CJK>
                 (#x766A . #x9CAB)      ; <CJK>
                 (#x766B . #x9CAD)      ; <CJK>
                 (#x766C . #x9CAE)      ; <CJK>
                 (#x766D . #x9CB0)      ; <CJK>
                 (#x766E . #x9CB1)      ; <CJK>
                 (#x766F . #x9CB2)      ; <CJK>
                 (#x7670 . #x9CB3)      ; <CJK>
                 (#x7671 . #x9CB4)      ; <CJK>
                 (#x7672 . #x9CB5)      ; <CJK>
                 (#x7673 . #x9CB6)      ; <CJK>
                 (#x7674 . #x9CB7)      ; <CJK>
                 (#x7675 . #x9CBA)      ; <CJK>
                 (#x7676 . #x9CBB)      ; <CJK>
                 (#x7677 . #x9CBC)      ; <CJK>
                 (#x7678 . #x9CBD)      ; <CJK>
                 (#x7679 . #x9CC4)      ; <CJK>
                 (#x767A . #x9CC5)      ; <CJK>
                 (#x767B . #x9CC6)      ; <CJK>
                 (#x767C . #x9CC7)      ; <CJK>
                 (#x767D . #x9CCA)      ; <CJK>
                 (#x767E . #x9CCB)      ; <CJK>
                 (#x7721 . #x9CCC)      ; <CJK>
                 (#x7722 . #x9CCD)      ; <CJK>
                 (#x7723 . #x9CCE)      ; <CJK>
                 (#x7724 . #x9CCF)      ; <CJK>
                 (#x7725 . #x9CD0)      ; <CJK>
                 (#x7726 . #x9CD3)      ; <CJK>
                 (#x7727 . #x9CD4)      ; <CJK>
                 (#x7728 . #x9CD5)      ; <CJK>
                 (#x7729 . #x9CD7)      ; <CJK>
                 (#x772A . #x9CD8)      ; <CJK>
                 (#x772B . #x9CD9)      ; <CJK>
                 (#x772C . #x9CDC)      ; <CJK>
                 (#x772D . #x9CDD)      ; <CJK>
                 (#x772E . #x9CDF)      ; <CJK>
                 (#x772F . #x9CE2)      ; <CJK>
                 (#x7730 . #x977C)      ; <CJK>
                 (#x7731 . #x9785)      ; <CJK>
                 (#x7732 . #x9791)      ; <CJK>
                 (#x7733 . #x9792)      ; <CJK>
                 (#x7734 . #x9794)      ; <CJK>
                 (#x7735 . #x97AF)      ; <CJK>
                 (#x7736 . #x97AB)      ; <CJK>
                 (#x7737 . #x97A3)      ; <CJK>
                 (#x7738 . #x97B2)      ; <CJK>
                 (#x7739 . #x97B4)      ; <CJK>
                 (#x773A . #x9AB1)      ; <CJK>
                 (#x773B . #x9AB0)      ; <CJK>
                 (#x773C . #x9AB7)      ; <CJK>
                 (#x773D . #x9E58)      ; <CJK>
                 (#x773E . #x9AB6)      ; <CJK>
                 (#x773F . #x9ABA)      ; <CJK>
                 (#x7740 . #x9ABC)      ; <CJK>
                 (#x7741 . #x9AC1)      ; <CJK>
                 (#x7742 . #x9AC0)      ; <CJK>
                 (#x7743 . #x9AC5)      ; <CJK>
                 (#x7744 . #x9AC2)      ; <CJK>
                 (#x7745 . #x9ACB)      ; <CJK>
                 (#x7746 . #x9ACC)      ; <CJK>
                 (#x7747 . #x9AD1)      ; <CJK>
                 (#x7748 . #x9B45)      ; <CJK>
                 (#x7749 . #x9B43)      ; <CJK>
                 (#x774A . #x9B47)      ; <CJK>
                 (#x774B . #x9B49)      ; <CJK>
                 (#x774C . #x9B48)      ; <CJK>
                 (#x774D . #x9B4D)      ; <CJK>
                 (#x774E . #x9B51)      ; <CJK>
                 (#x774F . #x98E8)      ; <CJK>
                 (#x7750 . #x990D)      ; <CJK>
                 (#x7751 . #x992E)      ; <CJK>
                 (#x7752 . #x9955)      ; <CJK>
                 (#x7753 . #x9954)      ; <CJK>
                 (#x7754 . #x9ADF)      ; <CJK>
                 (#x7755 . #x9AE1)      ; <CJK>
                 (#x7756 . #x9AE6)      ; <CJK>
                 (#x7757 . #x9AEF)      ; <CJK>
                 (#x7758 . #x9AEB)      ; <CJK>
                 (#x7759 . #x9AFB)      ; <CJK>
                 (#x775A . #x9AED)      ; <CJK>
                 (#x775B . #x9AF9)      ; <CJK>
                 (#x775C . #x9B08)      ; <CJK>
                 (#x775D . #x9B0F)      ; <CJK>
                 (#x775E . #x9B13)      ; <CJK>
                 (#x775F . #x9B1F)      ; <CJK>
                 (#x7760 . #x9B23)      ; <CJK>
                 (#x7761 . #x9EBD)      ; <CJK>
                 (#x7762 . #x9EBE)      ; <CJK>
                 (#x7763 . #x7E3B)      ; <CJK>
                 (#x7764 . #x9E82)      ; <CJK>
                 (#x7765 . #x9E87)      ; <CJK>
                 (#x7766 . #x9E88)      ; <CJK>
                 (#x7767 . #x9E8B)      ; <CJK>
                 (#x7768 . #x9E92)      ; <CJK>
                 (#x7769 . #x93D6)      ; <CJK>
                 (#x776A . #x9E9D)      ; <CJK>
                 (#x776B . #x9E9F)      ; <CJK>xx
                 (#x776C . #x9EDB)      ; <CJK>
                 (#x776D . #x9EDC)      ; <CJK>
                 (#x776E . #x9EDD)      ; <CJK>
                 (#x776F . #x9EE0)      ; <CJK>
                 (#x7770 . #x9EDF)      ; <CJK>
                 (#x7771 . #x9EE2)      ; <CJK>
                 (#x7772 . #x9EE9)      ; <CJK>
                 (#x7773 . #x9EE7)      ; <CJK>
                 (#x7774 . #x9EE5)      ; <CJK>
                 (#x7775 . #x9EEA)      ; <CJK>
                 (#x7776 . #x9EEF)      ; <CJK>
                 (#x7777 . #x9F22)      ; <CJK>
                 (#x7778 . #x9F2C)      ; <CJK>
                 (#x7779 . #x9F2F)      ; <CJK>
                 (#x777A . #x9F39)      ; <CJK>
                 (#x777B . #x9F37)      ; <CJK>
                 (#x777C . #x9F3D)      ; <CJK>
                 (#x777D . #x9F3E)      ; <CJK>
                 (#x777E . #x9F44)      ; <CJK>
                 ;; end of gb2312-to-unicode-table
                 )))
      (cp936-to-unicode-table '(
                                ;; cp936-to-unicode-table
                                (#x80     . #x20AC) ;EURO SIGN
                                (#x8140 . #x4E02)	;CJK UNIFIED IDEOGRAPH
                                (#x8141 . #x4E04)	;CJK UNIFIED IDEOGRAPH
                                (#x8142 . #x4E05)	;CJK UNIFIED IDEOGRAPH
                                (#x8143 . #x4E06)	;CJK UNIFIED IDEOGRAPH
                                (#x8144 . #x4E0F)	;CJK UNIFIED IDEOGRAPH
                                (#x8145 . #x4E12)	;CJK UNIFIED IDEOGRAPH
                                (#x8146 . #x4E17)	;CJK UNIFIED IDEOGRAPH
                                (#x8147 . #x4E1F)	;CJK UNIFIED IDEOGRAPH
                                (#x8148 . #x4E20)	;CJK UNIFIED IDEOGRAPH
                                (#x8149 . #x4E21)	;CJK UNIFIED IDEOGRAPH
                                (#x814A . #x4E23)	;CJK UNIFIED IDEOGRAPH
                                (#x814B . #x4E26)	;CJK UNIFIED IDEOGRAPH
                                (#x814C . #x4E29)	;CJK UNIFIED IDEOGRAPH
                                (#x814D . #x4E2E)	;CJK UNIFIED IDEOGRAPH
                                (#x814E . #x4E2F)	;CJK UNIFIED IDEOGRAPH
                                (#x814F . #x4E31)	;CJK UNIFIED IDEOGRAPH
                                (#x8150 . #x4E33)	;CJK UNIFIED IDEOGRAPH
                                (#x8151 . #x4E35)	;CJK UNIFIED IDEOGRAPH
                                (#x8152 . #x4E37)	;CJK UNIFIED IDEOGRAPH
                                (#x8153 . #x4E3C)	;CJK UNIFIED IDEOGRAPH
                                (#x8154 . #x4E40)	;CJK UNIFIED IDEOGRAPH
                                (#x8155 . #x4E41)	;CJK UNIFIED IDEOGRAPH
                                (#x8156 . #x4E42)	;CJK UNIFIED IDEOGRAPH
                                (#x8157 . #x4E44)	;CJK UNIFIED IDEOGRAPH
                                (#x8158 . #x4E46)	;CJK UNIFIED IDEOGRAPH
                                (#x8159 . #x4E4A)	;CJK UNIFIED IDEOGRAPH
                                (#x815A . #x4E51)	;CJK UNIFIED IDEOGRAPH
                                (#x815B . #x4E55)	;CJK UNIFIED IDEOGRAPH
                                (#x815C . #x4E57)	;CJK UNIFIED IDEOGRAPH
                                (#x815D . #x4E5A)	;CJK UNIFIED IDEOGRAPH
                                (#x815E . #x4E5B)	;CJK UNIFIED IDEOGRAPH
                                (#x815F . #x4E62)	;CJK UNIFIED IDEOGRAPH
                                (#x8160 . #x4E63)	;CJK UNIFIED IDEOGRAPH
                                (#x8161 . #x4E64)	;CJK UNIFIED IDEOGRAPH
                                (#x8162 . #x4E65)	;CJK UNIFIED IDEOGRAPH
                                (#x8163 . #x4E67)	;CJK UNIFIED IDEOGRAPH
                                (#x8164 . #x4E68)	;CJK UNIFIED IDEOGRAPH
                                (#x8165 . #x4E6A)	;CJK UNIFIED IDEOGRAPH
                                (#x8166 . #x4E6B)	;CJK UNIFIED IDEOGRAPH
                                (#x8167 . #x4E6C)	;CJK UNIFIED IDEOGRAPH
                                (#x8168 . #x4E6D)	;CJK UNIFIED IDEOGRAPH
                                (#x8169 . #x4E6E)	;CJK UNIFIED IDEOGRAPH
                                (#x816A . #x4E6F)	;CJK UNIFIED IDEOGRAPH
                                (#x816B . #x4E72)	;CJK UNIFIED IDEOGRAPH
                                (#x816C . #x4E74)	;CJK UNIFIED IDEOGRAPH
                                (#x816D . #x4E75)	;CJK UNIFIED IDEOGRAPH
                                (#x816E . #x4E76)	;CJK UNIFIED IDEOGRAPH
                                (#x816F . #x4E77)	;CJK UNIFIED IDEOGRAPH
                                (#x8170 . #x4E78)	;CJK UNIFIED IDEOGRAPH
                                (#x8171 . #x4E79)	;CJK UNIFIED IDEOGRAPH
                                (#x8172 . #x4E7A)	;CJK UNIFIED IDEOGRAPH
                                (#x8173 . #x4E7B)	;CJK UNIFIED IDEOGRAPH
                                (#x8174 . #x4E7C)	;CJK UNIFIED IDEOGRAPH
                                (#x8175 . #x4E7D)	;CJK UNIFIED IDEOGRAPH
                                (#x8176 . #x4E7F)	;CJK UNIFIED IDEOGRAPH
                                (#x8177 . #x4E80)	;CJK UNIFIED IDEOGRAPH
                                (#x8178 . #x4E81)	;CJK UNIFIED IDEOGRAPH
                                (#x8179 . #x4E82)	;CJK UNIFIED IDEOGRAPH
                                (#x817A . #x4E83)	;CJK UNIFIED IDEOGRAPH
                                (#x817B . #x4E84)	;CJK UNIFIED IDEOGRAPH
                                (#x817C . #x4E85)	;CJK UNIFIED IDEOGRAPH
                                (#x817D . #x4E87)	;CJK UNIFIED IDEOGRAPH
                                (#x817E . #x4E8A)	;CJK UNIFIED IDEOGRAPH
                                (#x8180 . #x4E90)	;CJK UNIFIED IDEOGRAPH
                                (#x8181 . #x4E96)	;CJK UNIFIED IDEOGRAPH
                                (#x8182 . #x4E97)	;CJK UNIFIED IDEOGRAPH
                                (#x8183 . #x4E99)	;CJK UNIFIED IDEOGRAPH
                                (#x8184 . #x4E9C)	;CJK UNIFIED IDEOGRAPH
                                (#x8185 . #x4E9D)	;CJK UNIFIED IDEOGRAPH
                                (#x8186 . #x4E9E)	;CJK UNIFIED IDEOGRAPH
                                (#x8187 . #x4EA3)	;CJK UNIFIED IDEOGRAPH
                                (#x8188 . #x4EAA)	;CJK UNIFIED IDEOGRAPH
                                (#x8189 . #x4EAF)	;CJK UNIFIED IDEOGRAPH
                                (#x818A . #x4EB0)	;CJK UNIFIED IDEOGRAPH
                                (#x818B . #x4EB1)	;CJK UNIFIED IDEOGRAPH
                                (#x818C . #x4EB4)	;CJK UNIFIED IDEOGRAPH
                                (#x818D . #x4EB6)	;CJK UNIFIED IDEOGRAPH
                                (#x818E . #x4EB7)	;CJK UNIFIED IDEOGRAPH
                                (#x818F . #x4EB8)	;CJK UNIFIED IDEOGRAPH
                                (#x8190 . #x4EB9)	;CJK UNIFIED IDEOGRAPH
                                (#x8191 . #x4EBC)	;CJK UNIFIED IDEOGRAPH
                                (#x8192 . #x4EBD)	;CJK UNIFIED IDEOGRAPH
                                (#x8193 . #x4EBE)	;CJK UNIFIED IDEOGRAPH
                                (#x8194 . #x4EC8)	;CJK UNIFIED IDEOGRAPH
                                (#x8195 . #x4ECC)	;CJK UNIFIED IDEOGRAPH
                                (#x8196 . #x4ECF)	;CJK UNIFIED IDEOGRAPH
                                (#x8197 . #x4ED0)	;CJK UNIFIED IDEOGRAPH
                                (#x8198 . #x4ED2)	;CJK UNIFIED IDEOGRAPH
                                (#x8199 . #x4EDA)	;CJK UNIFIED IDEOGRAPH
                                (#x819A . #x4EDB)	;CJK UNIFIED IDEOGRAPH
                                (#x819B . #x4EDC)	;CJK UNIFIED IDEOGRAPH
                                (#x819C . #x4EE0)	;CJK UNIFIED IDEOGRAPH
                                (#x819D . #x4EE2)	;CJK UNIFIED IDEOGRAPH
                                (#x819E . #x4EE6)	;CJK UNIFIED IDEOGRAPH
                                (#x819F . #x4EE7)	;CJK UNIFIED IDEOGRAPH
                                (#x81A0 . #x4EE9)	;CJK UNIFIED IDEOGRAPH
                                (#x81A1 . #x4EED)	;CJK UNIFIED IDEOGRAPH
                                (#x81A2 . #x4EEE)	;CJK UNIFIED IDEOGRAPH
                                (#x81A3 . #x4EEF)	;CJK UNIFIED IDEOGRAPH
                                (#x81A4 . #x4EF1)	;CJK UNIFIED IDEOGRAPH
                                (#x81A5 . #x4EF4)	;CJK UNIFIED IDEOGRAPH
                                (#x81A6 . #x4EF8)	;CJK UNIFIED IDEOGRAPH
                                (#x81A7 . #x4EF9)	;CJK UNIFIED IDEOGRAPH
                                (#x81A8 . #x4EFA)	;CJK UNIFIED IDEOGRAPH
                                (#x81A9 . #x4EFC)	;CJK UNIFIED IDEOGRAPH
                                (#x81AA . #x4EFE)	;CJK UNIFIED IDEOGRAPH
                                (#x81AB . #x4F00)	;CJK UNIFIED IDEOGRAPH
                                (#x81AC . #x4F02)	;CJK UNIFIED IDEOGRAPH
                                (#x81AD . #x4F03)	;CJK UNIFIED IDEOGRAPH
                                (#x81AE . #x4F04)	;CJK UNIFIED IDEOGRAPH
                                (#x81AF . #x4F05)	;CJK UNIFIED IDEOGRAPH
                                (#x81B0 . #x4F06)	;CJK UNIFIED IDEOGRAPH
                                (#x81B1 . #x4F07)	;CJK UNIFIED IDEOGRAPH
                                (#x81B2 . #x4F08)	;CJK UNIFIED IDEOGRAPH
                                (#x81B3 . #x4F0B)	;CJK UNIFIED IDEOGRAPH
                                (#x81B4 . #x4F0C)	;CJK UNIFIED IDEOGRAPH
                                (#x81B5 . #x4F12)	;CJK UNIFIED IDEOGRAPH
                                (#x81B6 . #x4F13)	;CJK UNIFIED IDEOGRAPH
                                (#x81B7 . #x4F14)	;CJK UNIFIED IDEOGRAPH
                                (#x81B8 . #x4F15)	;CJK UNIFIED IDEOGRAPH
                                (#x81B9 . #x4F16)	;CJK UNIFIED IDEOGRAPH
                                (#x81BA . #x4F1C)	;CJK UNIFIED IDEOGRAPH
                                (#x81BB . #x4F1D)	;CJK UNIFIED IDEOGRAPH
                                (#x81BC . #x4F21)	;CJK UNIFIED IDEOGRAPH
                                (#x81BD . #x4F23)	;CJK UNIFIED IDEOGRAPH
                                (#x81BE . #x4F28)	;CJK UNIFIED IDEOGRAPH
                                (#x81BF . #x4F29)	;CJK UNIFIED IDEOGRAPH
                                (#x81C0 . #x4F2C)	;CJK UNIFIED IDEOGRAPH
                                (#x81C1 . #x4F2D)	;CJK UNIFIED IDEOGRAPH
                                (#x81C2 . #x4F2E)	;CJK UNIFIED IDEOGRAPH
                                (#x81C3 . #x4F31)	;CJK UNIFIED IDEOGRAPH
                                (#x81C4 . #x4F33)	;CJK UNIFIED IDEOGRAPH
                                (#x81C5 . #x4F35)	;CJK UNIFIED IDEOGRAPH
                                (#x81C6 . #x4F37)	;CJK UNIFIED IDEOGRAPH
                                (#x81C7 . #x4F39)	;CJK UNIFIED IDEOGRAPH
                                (#x81C8 . #x4F3B)	;CJK UNIFIED IDEOGRAPH
                                (#x81C9 . #x4F3E)	;CJK UNIFIED IDEOGRAPH
                                (#x81CA . #x4F3F)	;CJK UNIFIED IDEOGRAPH
                                (#x81CB . #x4F40)	;CJK UNIFIED IDEOGRAPH
                                (#x81CC . #x4F41)	;CJK UNIFIED IDEOGRAPH
                                (#x81CD . #x4F42)	;CJK UNIFIED IDEOGRAPH
                                (#x81CE . #x4F44)	;CJK UNIFIED IDEOGRAPH
                                (#x81CF . #x4F45)	;CJK UNIFIED IDEOGRAPH
                                (#x81D0 . #x4F47)	;CJK UNIFIED IDEOGRAPH
                                (#x81D1 . #x4F48)	;CJK UNIFIED IDEOGRAPH
                                (#x81D2 . #x4F49)	;CJK UNIFIED IDEOGRAPH
                                (#x81D3 . #x4F4A)	;CJK UNIFIED IDEOGRAPH
                                (#x81D4 . #x4F4B)	;CJK UNIFIED IDEOGRAPH
                                (#x81D5 . #x4F4C)	;CJK UNIFIED IDEOGRAPH
                                (#x81D6 . #x4F52)	;CJK UNIFIED IDEOGRAPH
                                (#x81D7 . #x4F54)	;CJK UNIFIED IDEOGRAPH
                                (#x81D8 . #x4F56)	;CJK UNIFIED IDEOGRAPH
                                (#x81D9 . #x4F61)	;CJK UNIFIED IDEOGRAPH
                                (#x81DA . #x4F62)	;CJK UNIFIED IDEOGRAPH
                                (#x81DB . #x4F66)	;CJK UNIFIED IDEOGRAPH
                                (#x81DC . #x4F68)	;CJK UNIFIED IDEOGRAPH
                                (#x81DD . #x4F6A)	;CJK UNIFIED IDEOGRAPH
                                (#x81DE . #x4F6B)	;CJK UNIFIED IDEOGRAPH
                                (#x81DF . #x4F6D)	;CJK UNIFIED IDEOGRAPH
                                (#x81E0 . #x4F6E)	;CJK UNIFIED IDEOGRAPH
                                (#x81E1 . #x4F71)	;CJK UNIFIED IDEOGRAPH
                                (#x81E2 . #x4F72)	;CJK UNIFIED IDEOGRAPH
                                (#x81E3 . #x4F75)	;CJK UNIFIED IDEOGRAPH
                                (#x81E4 . #x4F77)	;CJK UNIFIED IDEOGRAPH
                                (#x81E5 . #x4F78)	;CJK UNIFIED IDEOGRAPH
                                (#x81E6 . #x4F79)	;CJK UNIFIED IDEOGRAPH
                                (#x81E7 . #x4F7A)	;CJK UNIFIED IDEOGRAPH
                                (#x81E8 . #x4F7D)	;CJK UNIFIED IDEOGRAPH
                                (#x81E9 . #x4F80)	;CJK UNIFIED IDEOGRAPH
                                (#x81EA . #x4F81)	;CJK UNIFIED IDEOGRAPH
                                (#x81EB . #x4F82)	;CJK UNIFIED IDEOGRAPH
                                (#x81EC . #x4F85)	;CJK UNIFIED IDEOGRAPH
                                (#x81ED . #x4F86)	;CJK UNIFIED IDEOGRAPH
                                (#x81EE . #x4F87)	;CJK UNIFIED IDEOGRAPH
                                (#x81EF . #x4F8A)	;CJK UNIFIED IDEOGRAPH
                                (#x81F0 . #x4F8C)	;CJK UNIFIED IDEOGRAPH
                                (#x81F1 . #x4F8E)	;CJK UNIFIED IDEOGRAPH
                                (#x81F2 . #x4F90)	;CJK UNIFIED IDEOGRAPH
                                (#x81F3 . #x4F92)	;CJK UNIFIED IDEOGRAPH
                                (#x81F4 . #x4F93)	;CJK UNIFIED IDEOGRAPH
                                (#x81F5 . #x4F95)	;CJK UNIFIED IDEOGRAPH
                                (#x81F6 . #x4F96)	;CJK UNIFIED IDEOGRAPH
                                (#x81F7 . #x4F98)	;CJK UNIFIED IDEOGRAPH
                                (#x81F8 . #x4F99)	;CJK UNIFIED IDEOGRAPH
                                (#x81F9 . #x4F9A)	;CJK UNIFIED IDEOGRAPH
                                (#x81FA . #x4F9C)	;CJK UNIFIED IDEOGRAPH
                                (#x81FB . #x4F9E)	;CJK UNIFIED IDEOGRAPH
                                (#x81FC . #x4F9F)	;CJK UNIFIED IDEOGRAPH
                                (#x81FD . #x4FA1)	;CJK UNIFIED IDEOGRAPH
                                (#x81FE . #x4FA2)	;CJK UNIFIED IDEOGRAPH
                                (#x8240 . #x4FA4)	;CJK UNIFIED IDEOGRAPH
                                (#x8241 . #x4FAB)	;CJK UNIFIED IDEOGRAPH
                                (#x8242 . #x4FAD)	;CJK UNIFIED IDEOGRAPH
                                (#x8243 . #x4FB0)	;CJK UNIFIED IDEOGRAPH
                                (#x8244 . #x4FB1)	;CJK UNIFIED IDEOGRAPH
                                (#x8245 . #x4FB2)	;CJK UNIFIED IDEOGRAPH
                                (#x8246 . #x4FB3)	;CJK UNIFIED IDEOGRAPH
                                (#x8247 . #x4FB4)	;CJK UNIFIED IDEOGRAPH
                                (#x8248 . #x4FB6)	;CJK UNIFIED IDEOGRAPH
                                (#x8249 . #x4FB7)	;CJK UNIFIED IDEOGRAPH
                                (#x824A . #x4FB8)	;CJK UNIFIED IDEOGRAPH
                                (#x824B . #x4FB9)	;CJK UNIFIED IDEOGRAPH
                                (#x824C . #x4FBA)	;CJK UNIFIED IDEOGRAPH
                                (#x824D . #x4FBB)	;CJK UNIFIED IDEOGRAPH
                                (#x824E . #x4FBC)	;CJK UNIFIED IDEOGRAPH
                                (#x824F . #x4FBD)	;CJK UNIFIED IDEOGRAPH
                                (#x8250 . #x4FBE)	;CJK UNIFIED IDEOGRAPH
                                (#x8251 . #x4FC0)	;CJK UNIFIED IDEOGRAPH
                                (#x8252 . #x4FC1)	;CJK UNIFIED IDEOGRAPH
                                (#x8253 . #x4FC2)	;CJK UNIFIED IDEOGRAPH
                                (#x8254 . #x4FC6)	;CJK UNIFIED IDEOGRAPH
                                (#x8255 . #x4FC7)	;CJK UNIFIED IDEOGRAPH
                                (#x8256 . #x4FC8)	;CJK UNIFIED IDEOGRAPH
                                (#x8257 . #x4FC9)	;CJK UNIFIED IDEOGRAPH
                                (#x8258 . #x4FCB)	;CJK UNIFIED IDEOGRAPH
                                (#x8259 . #x4FCC)	;CJK UNIFIED IDEOGRAPH
                                (#x825A . #x4FCD)	;CJK UNIFIED IDEOGRAPH
                                (#x825B . #x4FD2)	;CJK UNIFIED IDEOGRAPH
                                (#x825C . #x4FD3)	;CJK UNIFIED IDEOGRAPH
                                (#x825D . #x4FD4)	;CJK UNIFIED IDEOGRAPH
                                (#x825E . #x4FD5)	;CJK UNIFIED IDEOGRAPH
                                (#x825F . #x4FD6)	;CJK UNIFIED IDEOGRAPH
                                (#x8260 . #x4FD9)	;CJK UNIFIED IDEOGRAPH
                                (#x8261 . #x4FDB)	;CJK UNIFIED IDEOGRAPH
                                (#x8262 . #x4FE0)	;CJK UNIFIED IDEOGRAPH
                                (#x8263 . #x4FE2)	;CJK UNIFIED IDEOGRAPH
                                (#x8264 . #x4FE4)	;CJK UNIFIED IDEOGRAPH
                                (#x8265 . #x4FE5)	;CJK UNIFIED IDEOGRAPH
                                (#x8266 . #x4FE7)	;CJK UNIFIED IDEOGRAPH
                                (#x8267 . #x4FEB)	;CJK UNIFIED IDEOGRAPH
                                (#x8268 . #x4FEC)	;CJK UNIFIED IDEOGRAPH
                                (#x8269 . #x4FF0)	;CJK UNIFIED IDEOGRAPH
                                (#x826A . #x4FF2)	;CJK UNIFIED IDEOGRAPH
                                (#x826B . #x4FF4)	;CJK UNIFIED IDEOGRAPH
                                (#x826C . #x4FF5)	;CJK UNIFIED IDEOGRAPH
                                (#x826D . #x4FF6)	;CJK UNIFIED IDEOGRAPH
                                (#x826E . #x4FF7)	;CJK UNIFIED IDEOGRAPH
                                (#x826F . #x4FF9)	;CJK UNIFIED IDEOGRAPH
                                (#x8270 . #x4FFB)	;CJK UNIFIED IDEOGRAPH
                                (#x8271 . #x4FFC)	;CJK UNIFIED IDEOGRAPH
                                (#x8272 . #x4FFD)	;CJK UNIFIED IDEOGRAPH
                                (#x8273 . #x4FFF)	;CJK UNIFIED IDEOGRAPH
                                (#x8274 . #x5000)	;CJK UNIFIED IDEOGRAPH
                                (#x8275 . #x5001)	;CJK UNIFIED IDEOGRAPH
                                (#x8276 . #x5002)	;CJK UNIFIED IDEOGRAPH
                                (#x8277 . #x5003)	;CJK UNIFIED IDEOGRAPH
                                (#x8278 . #x5004)	;CJK UNIFIED IDEOGRAPH
                                (#x8279 . #x5005)	;CJK UNIFIED IDEOGRAPH
                                (#x827A . #x5006)	;CJK UNIFIED IDEOGRAPH
                                (#x827B . #x5007)	;CJK UNIFIED IDEOGRAPH
                                (#x827C . #x5008)	;CJK UNIFIED IDEOGRAPH
                                (#x827D . #x5009)	;CJK UNIFIED IDEOGRAPH
                                (#x827E . #x500A)	;CJK UNIFIED IDEOGRAPH
                                (#x8280 . #x500B)	;CJK UNIFIED IDEOGRAPH
                                (#x8281 . #x500E)	;CJK UNIFIED IDEOGRAPH
                                (#x8282 . #x5010)	;CJK UNIFIED IDEOGRAPH
                                (#x8283 . #x5011)	;CJK UNIFIED IDEOGRAPH
                                (#x8284 . #x5013)	;CJK UNIFIED IDEOGRAPH
                                (#x8285 . #x5015)	;CJK UNIFIED IDEOGRAPH
                                (#x8286 . #x5016)	;CJK UNIFIED IDEOGRAPH
                                (#x8287 . #x5017)	;CJK UNIFIED IDEOGRAPH
                                (#x8288 . #x501B)	;CJK UNIFIED IDEOGRAPH
                                (#x8289 . #x501D)	;CJK UNIFIED IDEOGRAPH
                                (#x828A . #x501E)	;CJK UNIFIED IDEOGRAPH
                                (#x828B . #x5020)	;CJK UNIFIED IDEOGRAPH
                                (#x828C . #x5022)	;CJK UNIFIED IDEOGRAPH
                                (#x828D . #x5023)	;CJK UNIFIED IDEOGRAPH
                                (#x828E . #x5024)	;CJK UNIFIED IDEOGRAPH
                                (#x828F . #x5027)	;CJK UNIFIED IDEOGRAPH
                                (#x8290 . #x502B)	;CJK UNIFIED IDEOGRAPH
                                (#x8291 . #x502F)	;CJK UNIFIED IDEOGRAPH
                                (#x8292 . #x5030)	;CJK UNIFIED IDEOGRAPH
                                (#x8293 . #x5031)	;CJK UNIFIED IDEOGRAPH
                                (#x8294 . #x5032)	;CJK UNIFIED IDEOGRAPH
                                (#x8295 . #x5033)	;CJK UNIFIED IDEOGRAPH
                                (#x8296 . #x5034)	;CJK UNIFIED IDEOGRAPH
                                (#x8297 . #x5035)	;CJK UNIFIED IDEOGRAPH
                                (#x8298 . #x5036)	;CJK UNIFIED IDEOGRAPH
                                (#x8299 . #x5037)	;CJK UNIFIED IDEOGRAPH
                                (#x829A . #x5038)	;CJK UNIFIED IDEOGRAPH
                                (#x829B . #x5039)	;CJK UNIFIED IDEOGRAPH
                                (#x829C . #x503B)	;CJK UNIFIED IDEOGRAPH
                                (#x829D . #x503D)	;CJK UNIFIED IDEOGRAPH
                                (#x829E . #x503F)	;CJK UNIFIED IDEOGRAPH
                                (#x829F . #x5040)	;CJK UNIFIED IDEOGRAPH
                                (#x82A0 . #x5041)	;CJK UNIFIED IDEOGRAPH
                                (#x82A1 . #x5042)	;CJK UNIFIED IDEOGRAPH
                                (#x82A2 . #x5044)	;CJK UNIFIED IDEOGRAPH
                                (#x82A3 . #x5045)	;CJK UNIFIED IDEOGRAPH
                                (#x82A4 . #x5046)	;CJK UNIFIED IDEOGRAPH
                                (#x82A5 . #x5049)	;CJK UNIFIED IDEOGRAPH
                                (#x82A6 . #x504A)	;CJK UNIFIED IDEOGRAPH
                                (#x82A7 . #x504B)	;CJK UNIFIED IDEOGRAPH
                                (#x82A8 . #x504D)	;CJK UNIFIED IDEOGRAPH
                                (#x82A9 . #x5050)	;CJK UNIFIED IDEOGRAPH
                                (#x82AA . #x5051)	;CJK UNIFIED IDEOGRAPH
                                (#x82AB . #x5052)	;CJK UNIFIED IDEOGRAPH
                                (#x82AC . #x5053)	;CJK UNIFIED IDEOGRAPH
                                (#x82AD . #x5054)	;CJK UNIFIED IDEOGRAPH
                                (#x82AE . #x5056)	;CJK UNIFIED IDEOGRAPH
                                (#x82AF . #x5057)	;CJK UNIFIED IDEOGRAPH
                                (#x82B0 . #x5058)	;CJK UNIFIED IDEOGRAPH
                                (#x82B1 . #x5059)	;CJK UNIFIED IDEOGRAPH
                                (#x82B2 . #x505B)	;CJK UNIFIED IDEOGRAPH
                                (#x82B3 . #x505D)	;CJK UNIFIED IDEOGRAPH
                                (#x82B4 . #x505E)	;CJK UNIFIED IDEOGRAPH
                                (#x82B5 . #x505F)	;CJK UNIFIED IDEOGRAPH
                                (#x82B6 . #x5060)	;CJK UNIFIED IDEOGRAPH
                                (#x82B7 . #x5061)	;CJK UNIFIED IDEOGRAPH
                                (#x82B8 . #x5062)	;CJK UNIFIED IDEOGRAPH
                                (#x82B9 . #x5063)	;CJK UNIFIED IDEOGRAPH
                                (#x82BA . #x5064)	;CJK UNIFIED IDEOGRAPH
                                (#x82BB . #x5066)	;CJK UNIFIED IDEOGRAPH
                                (#x82BC . #x5067)	;CJK UNIFIED IDEOGRAPH
                                (#x82BD . #x5068)	;CJK UNIFIED IDEOGRAPH
                                (#x82BE . #x5069)	;CJK UNIFIED IDEOGRAPH
                                (#x82BF . #x506A)	;CJK UNIFIED IDEOGRAPH
                                (#x82C0 . #x506B)	;CJK UNIFIED IDEOGRAPH
                                (#x82C1 . #x506D)	;CJK UNIFIED IDEOGRAPH
                                (#x82C2 . #x506E)	;CJK UNIFIED IDEOGRAPH
                                (#x82C3 . #x506F)	;CJK UNIFIED IDEOGRAPH
                                (#x82C4 . #x5070)	;CJK UNIFIED IDEOGRAPH
                                (#x82C5 . #x5071)	;CJK UNIFIED IDEOGRAPH
                                (#x82C6 . #x5072)	;CJK UNIFIED IDEOGRAPH
                                (#x82C7 . #x5073)	;CJK UNIFIED IDEOGRAPH
                                (#x82C8 . #x5074)	;CJK UNIFIED IDEOGRAPH
                                (#x82C9 . #x5075)	;CJK UNIFIED IDEOGRAPH
                                (#x82CA . #x5078)	;CJK UNIFIED IDEOGRAPH
                                (#x82CB . #x5079)	;CJK UNIFIED IDEOGRAPH
                                (#x82CC . #x507A)	;CJK UNIFIED IDEOGRAPH
                                (#x82CD . #x507C)	;CJK UNIFIED IDEOGRAPH
                                (#x82CE . #x507D)	;CJK UNIFIED IDEOGRAPH
                                (#x82CF . #x5081)	;CJK UNIFIED IDEOGRAPH
                                (#x82D0 . #x5082)	;CJK UNIFIED IDEOGRAPH
                                (#x82D1 . #x5083)	;CJK UNIFIED IDEOGRAPH
                                (#x82D2 . #x5084)	;CJK UNIFIED IDEOGRAPH
                                (#x82D3 . #x5086)	;CJK UNIFIED IDEOGRAPH
                                (#x82D4 . #x5087)	;CJK UNIFIED IDEOGRAPH
                                (#x82D5 . #x5089)	;CJK UNIFIED IDEOGRAPH
                                (#x82D6 . #x508A)	;CJK UNIFIED IDEOGRAPH
                                (#x82D7 . #x508B)	;CJK UNIFIED IDEOGRAPH
                                (#x82D8 . #x508C)	;CJK UNIFIED IDEOGRAPH
                                (#x82D9 . #x508E)	;CJK UNIFIED IDEOGRAPH
                                (#x82DA . #x508F)	;CJK UNIFIED IDEOGRAPH
                                (#x82DB . #x5090)	;CJK UNIFIED IDEOGRAPH
                                (#x82DC . #x5091)	;CJK UNIFIED IDEOGRAPH
                                (#x82DD . #x5092)	;CJK UNIFIED IDEOGRAPH
                                (#x82DE . #x5093)	;CJK UNIFIED IDEOGRAPH
                                (#x82DF . #x5094)	;CJK UNIFIED IDEOGRAPH
                                (#x82E0 . #x5095)	;CJK UNIFIED IDEOGRAPH
                                (#x82E1 . #x5096)	;CJK UNIFIED IDEOGRAPH
                                (#x82E2 . #x5097)	;CJK UNIFIED IDEOGRAPH
                                (#x82E3 . #x5098)	;CJK UNIFIED IDEOGRAPH
                                (#x82E4 . #x5099)	;CJK UNIFIED IDEOGRAPH
                                (#x82E5 . #x509A)	;CJK UNIFIED IDEOGRAPH
                                (#x82E6 . #x509B)	;CJK UNIFIED IDEOGRAPH
                                (#x82E7 . #x509C)	;CJK UNIFIED IDEOGRAPH
                                (#x82E8 . #x509D)	;CJK UNIFIED IDEOGRAPH
                                (#x82E9 . #x509E)	;CJK UNIFIED IDEOGRAPH
                                (#x82EA . #x509F)	;CJK UNIFIED IDEOGRAPH
                                (#x82EB . #x50A0)	;CJK UNIFIED IDEOGRAPH
                                (#x82EC . #x50A1)	;CJK UNIFIED IDEOGRAPH
                                (#x82ED . #x50A2)	;CJK UNIFIED IDEOGRAPH
                                (#x82EE . #x50A4)	;CJK UNIFIED IDEOGRAPH
                                (#x82EF . #x50A6)	;CJK UNIFIED IDEOGRAPH
                                (#x82F0 . #x50AA)	;CJK UNIFIED IDEOGRAPH
                                (#x82F1 . #x50AB)	;CJK UNIFIED IDEOGRAPH
                                (#x82F2 . #x50AD)	;CJK UNIFIED IDEOGRAPH
                                (#x82F3 . #x50AE)	;CJK UNIFIED IDEOGRAPH
                                (#x82F4 . #x50AF)	;CJK UNIFIED IDEOGRAPH
                                (#x82F5 . #x50B0)	;CJK UNIFIED IDEOGRAPH
                                (#x82F6 . #x50B1)	;CJK UNIFIED IDEOGRAPH
                                (#x82F7 . #x50B3)	;CJK UNIFIED IDEOGRAPH
                                (#x82F8 . #x50B4)	;CJK UNIFIED IDEOGRAPH
                                (#x82F9 . #x50B5)	;CJK UNIFIED IDEOGRAPH
                                (#x82FA . #x50B6)	;CJK UNIFIED IDEOGRAPH
                                (#x82FB . #x50B7)	;CJK UNIFIED IDEOGRAPH
                                (#x82FC . #x50B8)	;CJK UNIFIED IDEOGRAPH
                                (#x82FD . #x50B9)	;CJK UNIFIED IDEOGRAPH
                                (#x82FE . #x50BC)	;CJK UNIFIED IDEOGRAPH
                                (#x8340 . #x50BD)	;CJK UNIFIED IDEOGRAPH
                                (#x8341 . #x50BE)	;CJK UNIFIED IDEOGRAPH
                                (#x8342 . #x50BF)	;CJK UNIFIED IDEOGRAPH
                                (#x8343 . #x50C0)	;CJK UNIFIED IDEOGRAPH
                                (#x8344 . #x50C1)	;CJK UNIFIED IDEOGRAPH
                                (#x8345 . #x50C2)	;CJK UNIFIED IDEOGRAPH
                                (#x8346 . #x50C3)	;CJK UNIFIED IDEOGRAPH
                                (#x8347 . #x50C4)	;CJK UNIFIED IDEOGRAPH
                                (#x8348 . #x50C5)	;CJK UNIFIED IDEOGRAPH
                                (#x8349 . #x50C6)	;CJK UNIFIED IDEOGRAPH
                                (#x834A . #x50C7)	;CJK UNIFIED IDEOGRAPH
                                (#x834B . #x50C8)	;CJK UNIFIED IDEOGRAPH
                                (#x834C . #x50C9)	;CJK UNIFIED IDEOGRAPH
                                (#x834D . #x50CA)	;CJK UNIFIED IDEOGRAPH
                                (#x834E . #x50CB)	;CJK UNIFIED IDEOGRAPH
                                (#x834F . #x50CC)	;CJK UNIFIED IDEOGRAPH
                                (#x8350 . #x50CD)	;CJK UNIFIED IDEOGRAPH
                                (#x8351 . #x50CE)	;CJK UNIFIED IDEOGRAPH
                                (#x8352 . #x50D0)	;CJK UNIFIED IDEOGRAPH
                                (#x8353 . #x50D1)	;CJK UNIFIED IDEOGRAPH
                                (#x8354 . #x50D2)	;CJK UNIFIED IDEOGRAPH
                                (#x8355 . #x50D3)	;CJK UNIFIED IDEOGRAPH
                                (#x8356 . #x50D4)	;CJK UNIFIED IDEOGRAPH
                                (#x8357 . #x50D5)	;CJK UNIFIED IDEOGRAPH
                                (#x8358 . #x50D7)	;CJK UNIFIED IDEOGRAPH
                                (#x8359 . #x50D8)	;CJK UNIFIED IDEOGRAPH
                                (#x835A . #x50D9)	;CJK UNIFIED IDEOGRAPH
                                (#x835B . #x50DB)	;CJK UNIFIED IDEOGRAPH
                                (#x835C . #x50DC)	;CJK UNIFIED IDEOGRAPH
                                (#x835D . #x50DD)	;CJK UNIFIED IDEOGRAPH
                                (#x835E . #x50DE)	;CJK UNIFIED IDEOGRAPH
                                (#x835F . #x50DF)	;CJK UNIFIED IDEOGRAPH
                                (#x8360 . #x50E0)	;CJK UNIFIED IDEOGRAPH
                                (#x8361 . #x50E1)	;CJK UNIFIED IDEOGRAPH
                                (#x8362 . #x50E2)	;CJK UNIFIED IDEOGRAPH
                                (#x8363 . #x50E3)	;CJK UNIFIED IDEOGRAPH
                                (#x8364 . #x50E4)	;CJK UNIFIED IDEOGRAPH
                                (#x8365 . #x50E5)	;CJK UNIFIED IDEOGRAPH
                                (#x8366 . #x50E8)	;CJK UNIFIED IDEOGRAPH
                                (#x8367 . #x50E9)	;CJK UNIFIED IDEOGRAPH
                                (#x8368 . #x50EA)	;CJK UNIFIED IDEOGRAPH
                                (#x8369 . #x50EB)	;CJK UNIFIED IDEOGRAPH
                                (#x836A . #x50EF)	;CJK UNIFIED IDEOGRAPH
                                (#x836B . #x50F0)	;CJK UNIFIED IDEOGRAPH
                                (#x836C . #x50F1)	;CJK UNIFIED IDEOGRAPH
                                (#x836D . #x50F2)	;CJK UNIFIED IDEOGRAPH
                                (#x836E . #x50F4)	;CJK UNIFIED IDEOGRAPH
                                (#x836F . #x50F6)	;CJK UNIFIED IDEOGRAPH
                                (#x8370 . #x50F7)	;CJK UNIFIED IDEOGRAPH
                                (#x8371 . #x50F8)	;CJK UNIFIED IDEOGRAPH
                                (#x8372 . #x50F9)	;CJK UNIFIED IDEOGRAPH
                                (#x8373 . #x50FA)	;CJK UNIFIED IDEOGRAPH
                                (#x8374 . #x50FC)	;CJK UNIFIED IDEOGRAPH
                                (#x8375 . #x50FD)	;CJK UNIFIED IDEOGRAPH
                                (#x8376 . #x50FE)	;CJK UNIFIED IDEOGRAPH
                                (#x8377 . #x50FF)	;CJK UNIFIED IDEOGRAPH
                                (#x8378 . #x5100)	;CJK UNIFIED IDEOGRAPH
                                (#x8379 . #x5101)	;CJK UNIFIED IDEOGRAPH
                                (#x837A . #x5102)	;CJK UNIFIED IDEOGRAPH
                                (#x837B . #x5103)	;CJK UNIFIED IDEOGRAPH
                                (#x837C . #x5104)	;CJK UNIFIED IDEOGRAPH
                                (#x837D . #x5105)	;CJK UNIFIED IDEOGRAPH
                                (#x837E . #x5108)	;CJK UNIFIED IDEOGRAPH
                                (#x8380 . #x5109)	;CJK UNIFIED IDEOGRAPH
                                (#x8381 . #x510A)	;CJK UNIFIED IDEOGRAPH
                                (#x8382 . #x510C)	;CJK UNIFIED IDEOGRAPH
                                (#x8383 . #x510D)	;CJK UNIFIED IDEOGRAPH
                                (#x8384 . #x510E)	;CJK UNIFIED IDEOGRAPH
                                (#x8385 . #x510F)	;CJK UNIFIED IDEOGRAPH
                                (#x8386 . #x5110)	;CJK UNIFIED IDEOGRAPH
                                (#x8387 . #x5111)	;CJK UNIFIED IDEOGRAPH
                                (#x8388 . #x5113)	;CJK UNIFIED IDEOGRAPH
                                (#x8389 . #x5114)	;CJK UNIFIED IDEOGRAPH
                                (#x838A . #x5115)	;CJK UNIFIED IDEOGRAPH
                                (#x838B . #x5116)	;CJK UNIFIED IDEOGRAPH
                                (#x838C . #x5117)	;CJK UNIFIED IDEOGRAPH
                                (#x838D . #x5118)	;CJK UNIFIED IDEOGRAPH
                                (#x838E . #x5119)	;CJK UNIFIED IDEOGRAPH
                                (#x838F . #x511A)	;CJK UNIFIED IDEOGRAPH
                                (#x8390 . #x511B)	;CJK UNIFIED IDEOGRAPH
                                (#x8391 . #x511C)	;CJK UNIFIED IDEOGRAPH
                                (#x8392 . #x511D)	;CJK UNIFIED IDEOGRAPH
                                (#x8393 . #x511E)	;CJK UNIFIED IDEOGRAPH
                                (#x8394 . #x511F)	;CJK UNIFIED IDEOGRAPH
                                (#x8395 . #x5120)	;CJK UNIFIED IDEOGRAPH
                                (#x8396 . #x5122)	;CJK UNIFIED IDEOGRAPH
                                (#x8397 . #x5123)	;CJK UNIFIED IDEOGRAPH
                                (#x8398 . #x5124)	;CJK UNIFIED IDEOGRAPH
                                (#x8399 . #x5125)	;CJK UNIFIED IDEOGRAPH
                                (#x839A . #x5126)	;CJK UNIFIED IDEOGRAPH
                                (#x839B . #x5127)	;CJK UNIFIED IDEOGRAPH
                                (#x839C . #x5128)	;CJK UNIFIED IDEOGRAPH
                                (#x839D . #x5129)	;CJK UNIFIED IDEOGRAPH
                                (#x839E . #x512A)	;CJK UNIFIED IDEOGRAPH
                                (#x839F . #x512B)	;CJK UNIFIED IDEOGRAPH
                                (#x83A0 . #x512C)	;CJK UNIFIED IDEOGRAPH
                                (#x83A1 . #x512D)	;CJK UNIFIED IDEOGRAPH
                                (#x83A2 . #x512E)	;CJK UNIFIED IDEOGRAPH
                                (#x83A3 . #x512F)	;CJK UNIFIED IDEOGRAPH
                                (#x83A4 . #x5130)	;CJK UNIFIED IDEOGRAPH
                                (#x83A5 . #x5131)	;CJK UNIFIED IDEOGRAPH
                                (#x83A6 . #x5132)	;CJK UNIFIED IDEOGRAPH
                                (#x83A7 . #x5133)	;CJK UNIFIED IDEOGRAPH
                                (#x83A8 . #x5134)	;CJK UNIFIED IDEOGRAPH
                                (#x83A9 . #x5135)	;CJK UNIFIED IDEOGRAPH
                                (#x83AA . #x5136)	;CJK UNIFIED IDEOGRAPH
                                (#x83AB . #x5137)	;CJK UNIFIED IDEOGRAPH
                                (#x83AC . #x5138)	;CJK UNIFIED IDEOGRAPH
                                (#x83AD . #x5139)	;CJK UNIFIED IDEOGRAPH
                                (#x83AE . #x513A)	;CJK UNIFIED IDEOGRAPH
                                (#x83AF . #x513B)	;CJK UNIFIED IDEOGRAPH
                                (#x83B0 . #x513C)	;CJK UNIFIED IDEOGRAPH
                                (#x83B1 . #x513D)	;CJK UNIFIED IDEOGRAPH
                                (#x83B2 . #x513E)	;CJK UNIFIED IDEOGRAPH
                                (#x83B3 . #x5142)	;CJK UNIFIED IDEOGRAPH
                                (#x83B4 . #x5147)	;CJK UNIFIED IDEOGRAPH
                                (#x83B5 . #x514A)	;CJK UNIFIED IDEOGRAPH
                                (#x83B6 . #x514C)	;CJK UNIFIED IDEOGRAPH
                                (#x83B7 . #x514E)	;CJK UNIFIED IDEOGRAPH
                                (#x83B8 . #x514F)	;CJK UNIFIED IDEOGRAPH
                                (#x83B9 . #x5150)	;CJK UNIFIED IDEOGRAPH
                                (#x83BA . #x5152)	;CJK UNIFIED IDEOGRAPH
                                (#x83BB . #x5153)	;CJK UNIFIED IDEOGRAPH
                                (#x83BC . #x5157)	;CJK UNIFIED IDEOGRAPH
                                (#x83BD . #x5158)	;CJK UNIFIED IDEOGRAPH
                                (#x83BE . #x5159)	;CJK UNIFIED IDEOGRAPH
                                (#x83BF . #x515B)	;CJK UNIFIED IDEOGRAPH
                                (#x83C0 . #x515D)	;CJK UNIFIED IDEOGRAPH
                                (#x83C1 . #x515E)	;CJK UNIFIED IDEOGRAPH
                                (#x83C2 . #x515F)	;CJK UNIFIED IDEOGRAPH
                                (#x83C3 . #x5160)	;CJK UNIFIED IDEOGRAPH
                                (#x83C4 . #x5161)	;CJK UNIFIED IDEOGRAPH
                                (#x83C5 . #x5163)	;CJK UNIFIED IDEOGRAPH
                                (#x83C6 . #x5164)	;CJK UNIFIED IDEOGRAPH
                                (#x83C7 . #x5166)	;CJK UNIFIED IDEOGRAPH
                                (#x83C8 . #x5167)	;CJK UNIFIED IDEOGRAPH
                                (#x83C9 . #x5169)	;CJK UNIFIED IDEOGRAPH
                                (#x83CA . #x516A)	;CJK UNIFIED IDEOGRAPH
                                (#x83CB . #x516F)	;CJK UNIFIED IDEOGRAPH
                                (#x83CC . #x5172)	;CJK UNIFIED IDEOGRAPH
                                (#x83CD . #x517A)	;CJK UNIFIED IDEOGRAPH
                                (#x83CE . #x517E)	;CJK UNIFIED IDEOGRAPH
                                (#x83CF . #x517F)	;CJK UNIFIED IDEOGRAPH
                                (#x83D0 . #x5183)	;CJK UNIFIED IDEOGRAPH
                                (#x83D1 . #x5184)	;CJK UNIFIED IDEOGRAPH
                                (#x83D2 . #x5186)	;CJK UNIFIED IDEOGRAPH
                                (#x83D3 . #x5187)	;CJK UNIFIED IDEOGRAPH
                                (#x83D4 . #x518A)	;CJK UNIFIED IDEOGRAPH
                                (#x83D5 . #x518B)	;CJK UNIFIED IDEOGRAPH
                                (#x83D6 . #x518E)	;CJK UNIFIED IDEOGRAPH
                                (#x83D7 . #x518F)	;CJK UNIFIED IDEOGRAPH
                                (#x83D8 . #x5190)	;CJK UNIFIED IDEOGRAPH
                                (#x83D9 . #x5191)	;CJK UNIFIED IDEOGRAPH
                                (#x83DA . #x5193)	;CJK UNIFIED IDEOGRAPH
                                (#x83DB . #x5194)	;CJK UNIFIED IDEOGRAPH
                                (#x83DC . #x5198)	;CJK UNIFIED IDEOGRAPH
                                (#x83DD . #x519A)	;CJK UNIFIED IDEOGRAPH
                                (#x83DE . #x519D)	;CJK UNIFIED IDEOGRAPH
                                (#x83DF . #x519E)	;CJK UNIFIED IDEOGRAPH
                                (#x83E0 . #x519F)	;CJK UNIFIED IDEOGRAPH
                                (#x83E1 . #x51A1)	;CJK UNIFIED IDEOGRAPH
                                (#x83E2 . #x51A3)	;CJK UNIFIED IDEOGRAPH
                                (#x83E3 . #x51A6)	;CJK UNIFIED IDEOGRAPH
                                (#x83E4 . #x51A7)	;CJK UNIFIED IDEOGRAPH
                                (#x83E5 . #x51A8)	;CJK UNIFIED IDEOGRAPH
                                (#x83E6 . #x51A9)	;CJK UNIFIED IDEOGRAPH
                                (#x83E7 . #x51AA)	;CJK UNIFIED IDEOGRAPH
                                (#x83E8 . #x51AD)	;CJK UNIFIED IDEOGRAPH
                                (#x83E9 . #x51AE)	;CJK UNIFIED IDEOGRAPH
                                (#x83EA . #x51B4)	;CJK UNIFIED IDEOGRAPH
                                (#x83EB . #x51B8)	;CJK UNIFIED IDEOGRAPH
                                (#x83EC . #x51B9)	;CJK UNIFIED IDEOGRAPH
                                (#x83ED . #x51BA)	;CJK UNIFIED IDEOGRAPH
                                (#x83EE . #x51BE)	;CJK UNIFIED IDEOGRAPH
                                (#x83EF . #x51BF)	;CJK UNIFIED IDEOGRAPH
                                (#x83F0 . #x51C1)	;CJK UNIFIED IDEOGRAPH
                                (#x83F1 . #x51C2)	;CJK UNIFIED IDEOGRAPH
                                (#x83F2 . #x51C3)	;CJK UNIFIED IDEOGRAPH
                                (#x83F3 . #x51C5)	;CJK UNIFIED IDEOGRAPH
                                (#x83F4 . #x51C8)	;CJK UNIFIED IDEOGRAPH
                                (#x83F5 . #x51CA)	;CJK UNIFIED IDEOGRAPH
                                (#x83F6 . #x51CD)	;CJK UNIFIED IDEOGRAPH
                                (#x83F7 . #x51CE)	;CJK UNIFIED IDEOGRAPH
                                (#x83F8 . #x51D0)	;CJK UNIFIED IDEOGRAPH
                                (#x83F9 . #x51D2)	;CJK UNIFIED IDEOGRAPH
                                (#x83FA . #x51D3)	;CJK UNIFIED IDEOGRAPH
                                (#x83FB . #x51D4)	;CJK UNIFIED IDEOGRAPH
                                (#x83FC . #x51D5)	;CJK UNIFIED IDEOGRAPH
                                (#x83FD . #x51D6)	;CJK UNIFIED IDEOGRAPH
                                (#x83FE . #x51D7)	;CJK UNIFIED IDEOGRAPH
                                (#x8440 . #x51D8)	;CJK UNIFIED IDEOGRAPH
                                (#x8441 . #x51D9)	;CJK UNIFIED IDEOGRAPH
                                (#x8442 . #x51DA)	;CJK UNIFIED IDEOGRAPH
                                (#x8443 . #x51DC)	;CJK UNIFIED IDEOGRAPH
                                (#x8444 . #x51DE)	;CJK UNIFIED IDEOGRAPH
                                (#x8445 . #x51DF)	;CJK UNIFIED IDEOGRAPH
                                (#x8446 . #x51E2)	;CJK UNIFIED IDEOGRAPH
                                (#x8447 . #x51E3)	;CJK UNIFIED IDEOGRAPH
                                (#x8448 . #x51E5)	;CJK UNIFIED IDEOGRAPH
                                (#x8449 . #x51E6)	;CJK UNIFIED IDEOGRAPH
                                (#x844A . #x51E7)	;CJK UNIFIED IDEOGRAPH
                                (#x844B . #x51E8)	;CJK UNIFIED IDEOGRAPH
                                (#x844C . #x51E9)	;CJK UNIFIED IDEOGRAPH
                                (#x844D . #x51EA)	;CJK UNIFIED IDEOGRAPH
                                (#x844E . #x51EC)	;CJK UNIFIED IDEOGRAPH
                                (#x844F . #x51EE)	;CJK UNIFIED IDEOGRAPH
                                (#x8450 . #x51F1)	;CJK UNIFIED IDEOGRAPH
                                (#x8451 . #x51F2)	;CJK UNIFIED IDEOGRAPH
                                (#x8452 . #x51F4)	;CJK UNIFIED IDEOGRAPH
                                (#x8453 . #x51F7)	;CJK UNIFIED IDEOGRAPH
                                (#x8454 . #x51FE)	;CJK UNIFIED IDEOGRAPH
                                (#x8455 . #x5204)	;CJK UNIFIED IDEOGRAPH
                                (#x8456 . #x5205)	;CJK UNIFIED IDEOGRAPH
                                (#x8457 . #x5209)	;CJK UNIFIED IDEOGRAPH
                                (#x8458 . #x520B)	;CJK UNIFIED IDEOGRAPH
                                (#x8459 . #x520C)	;CJK UNIFIED IDEOGRAPH
                                (#x845A . #x520F)	;CJK UNIFIED IDEOGRAPH
                                (#x845B . #x5210)	;CJK UNIFIED IDEOGRAPH
                                (#x845C . #x5213)	;CJK UNIFIED IDEOGRAPH
                                (#x845D . #x5214)	;CJK UNIFIED IDEOGRAPH
                                (#x845E . #x5215)	;CJK UNIFIED IDEOGRAPH
                                (#x845F . #x521C)	;CJK UNIFIED IDEOGRAPH
                                (#x8460 . #x521E)	;CJK UNIFIED IDEOGRAPH
                                (#x8461 . #x521F)	;CJK UNIFIED IDEOGRAPH
                                (#x8462 . #x5221)	;CJK UNIFIED IDEOGRAPH
                                (#x8463 . #x5222)	;CJK UNIFIED IDEOGRAPH
                                (#x8464 . #x5223)	;CJK UNIFIED IDEOGRAPH
                                (#x8465 . #x5225)	;CJK UNIFIED IDEOGRAPH
                                (#x8466 . #x5226)	;CJK UNIFIED IDEOGRAPH
                                (#x8467 . #x5227)	;CJK UNIFIED IDEOGRAPH
                                (#x8468 . #x522A)	;CJK UNIFIED IDEOGRAPH
                                (#x8469 . #x522C)	;CJK UNIFIED IDEOGRAPH
                                (#x846A . #x522F)	;CJK UNIFIED IDEOGRAPH
                                (#x846B . #x5231)	;CJK UNIFIED IDEOGRAPH
                                (#x846C . #x5232)	;CJK UNIFIED IDEOGRAPH
                                (#x846D . #x5234)	;CJK UNIFIED IDEOGRAPH
                                (#x846E . #x5235)	;CJK UNIFIED IDEOGRAPH
                                (#x846F . #x523C)	;CJK UNIFIED IDEOGRAPH
                                (#x8470 . #x523E)	;CJK UNIFIED IDEOGRAPH
                                (#x8471 . #x5244)	;CJK UNIFIED IDEOGRAPH
                                (#x8472 . #x5245)	;CJK UNIFIED IDEOGRAPH
                                (#x8473 . #x5246)	;CJK UNIFIED IDEOGRAPH
                                (#x8474 . #x5247)	;CJK UNIFIED IDEOGRAPH
                                (#x8475 . #x5248)	;CJK UNIFIED IDEOGRAPH
                                (#x8476 . #x5249)	;CJK UNIFIED IDEOGRAPH
                                (#x8477 . #x524B)	;CJK UNIFIED IDEOGRAPH
                                (#x8478 . #x524E)	;CJK UNIFIED IDEOGRAPH
                                (#x8479 . #x524F)	;CJK UNIFIED IDEOGRAPH
                                (#x847A . #x5252)	;CJK UNIFIED IDEOGRAPH
                                (#x847B . #x5253)	;CJK UNIFIED IDEOGRAPH
                                (#x847C . #x5255)	;CJK UNIFIED IDEOGRAPH
                                (#x847D . #x5257)	;CJK UNIFIED IDEOGRAPH
                                (#x847E . #x5258)	;CJK UNIFIED IDEOGRAPH
                                (#x8480 . #x5259)	;CJK UNIFIED IDEOGRAPH
                                (#x8481 . #x525A)	;CJK UNIFIED IDEOGRAPH
                                (#x8482 . #x525B)	;CJK UNIFIED IDEOGRAPH
                                (#x8483 . #x525D)	;CJK UNIFIED IDEOGRAPH
                                (#x8484 . #x525F)	;CJK UNIFIED IDEOGRAPH
                                (#x8485 . #x5260)	;CJK UNIFIED IDEOGRAPH
                                (#x8486 . #x5262)	;CJK UNIFIED IDEOGRAPH
                                (#x8487 . #x5263)	;CJK UNIFIED IDEOGRAPH
                                (#x8488 . #x5264)	;CJK UNIFIED IDEOGRAPH
                                (#x8489 . #x5266)	;CJK UNIFIED IDEOGRAPH
                                (#x848A . #x5268)	;CJK UNIFIED IDEOGRAPH
                                (#x848B . #x526B)	;CJK UNIFIED IDEOGRAPH
                                (#x848C . #x526C)	;CJK UNIFIED IDEOGRAPH
                                (#x848D . #x526D)	;CJK UNIFIED IDEOGRAPH
                                (#x848E . #x526E)	;CJK UNIFIED IDEOGRAPH
                                (#x848F . #x5270)	;CJK UNIFIED IDEOGRAPH
                                (#x8490 . #x5271)	;CJK UNIFIED IDEOGRAPH
                                (#x8491 . #x5273)	;CJK UNIFIED IDEOGRAPH
                                (#x8492 . #x5274)	;CJK UNIFIED IDEOGRAPH
                                (#x8493 . #x5275)	;CJK UNIFIED IDEOGRAPH
                                (#x8494 . #x5276)	;CJK UNIFIED IDEOGRAPH
                                (#x8495 . #x5277)	;CJK UNIFIED IDEOGRAPH
                                (#x8496 . #x5278)	;CJK UNIFIED IDEOGRAPH
                                (#x8497 . #x5279)	;CJK UNIFIED IDEOGRAPH
                                (#x8498 . #x527A)	;CJK UNIFIED IDEOGRAPH
                                (#x8499 . #x527B)	;CJK UNIFIED IDEOGRAPH
                                (#x849A . #x527C)	;CJK UNIFIED IDEOGRAPH
                                (#x849B . #x527E)	;CJK UNIFIED IDEOGRAPH
                                (#x849C . #x5280)	;CJK UNIFIED IDEOGRAPH
                                (#x849D . #x5283)	;CJK UNIFIED IDEOGRAPH
                                (#x849E . #x5284)	;CJK UNIFIED IDEOGRAPH
                                (#x849F . #x5285)	;CJK UNIFIED IDEOGRAPH
                                (#x84A0 . #x5286)	;CJK UNIFIED IDEOGRAPH
                                (#x84A1 . #x5287)	;CJK UNIFIED IDEOGRAPH
                                (#x84A2 . #x5289)	;CJK UNIFIED IDEOGRAPH
                                (#x84A3 . #x528A)	;CJK UNIFIED IDEOGRAPH
                                (#x84A4 . #x528B)	;CJK UNIFIED IDEOGRAPH
                                (#x84A5 . #x528C)	;CJK UNIFIED IDEOGRAPH
                                (#x84A6 . #x528D)	;CJK UNIFIED IDEOGRAPH
                                (#x84A7 . #x528E)	;CJK UNIFIED IDEOGRAPH
                                (#x84A8 . #x528F)	;CJK UNIFIED IDEOGRAPH
                                (#x84A9 . #x5291)	;CJK UNIFIED IDEOGRAPH
                                (#x84AA . #x5292)	;CJK UNIFIED IDEOGRAPH
                                (#x84AB . #x5294)	;CJK UNIFIED IDEOGRAPH
                                (#x84AC . #x5295)	;CJK UNIFIED IDEOGRAPH
                                (#x84AD . #x5296)	;CJK UNIFIED IDEOGRAPH
                                (#x84AE . #x5297)	;CJK UNIFIED IDEOGRAPH
                                (#x84AF . #x5298)	;CJK UNIFIED IDEOGRAPH
                                (#x84B0 . #x5299)	;CJK UNIFIED IDEOGRAPH
                                (#x84B1 . #x529A)	;CJK UNIFIED IDEOGRAPH
                                (#x84B2 . #x529C)	;CJK UNIFIED IDEOGRAPH
                                (#x84B3 . #x52A4)	;CJK UNIFIED IDEOGRAPH
                                (#x84B4 . #x52A5)	;CJK UNIFIED IDEOGRAPH
                                (#x84B5 . #x52A6)	;CJK UNIFIED IDEOGRAPH
                                (#x84B6 . #x52A7)	;CJK UNIFIED IDEOGRAPH
                                (#x84B7 . #x52AE)	;CJK UNIFIED IDEOGRAPH
                                (#x84B8 . #x52AF)	;CJK UNIFIED IDEOGRAPH
                                (#x84B9 . #x52B0)	;CJK UNIFIED IDEOGRAPH
                                (#x84BA . #x52B4)	;CJK UNIFIED IDEOGRAPH
                                (#x84BB . #x52B5)	;CJK UNIFIED IDEOGRAPH
                                (#x84BC . #x52B6)	;CJK UNIFIED IDEOGRAPH
                                (#x84BD . #x52B7)	;CJK UNIFIED IDEOGRAPH
                                (#x84BE . #x52B8)	;CJK UNIFIED IDEOGRAPH
                                (#x84BF . #x52B9)	;CJK UNIFIED IDEOGRAPH
                                (#x84C0 . #x52BA)	;CJK UNIFIED IDEOGRAPH
                                (#x84C1 . #x52BB)	;CJK UNIFIED IDEOGRAPH
                                (#x84C2 . #x52BC)	;CJK UNIFIED IDEOGRAPH
                                (#x84C3 . #x52BD)	;CJK UNIFIED IDEOGRAPH
                                (#x84C4 . #x52C0)	;CJK UNIFIED IDEOGRAPH
                                (#x84C5 . #x52C1)	;CJK UNIFIED IDEOGRAPH
                                (#x84C6 . #x52C2)	;CJK UNIFIED IDEOGRAPH
                                (#x84C7 . #x52C4)	;CJK UNIFIED IDEOGRAPH
                                (#x84C8 . #x52C5)	;CJK UNIFIED IDEOGRAPH
                                (#x84C9 . #x52C6)	;CJK UNIFIED IDEOGRAPH
                                (#x84CA . #x52C8)	;CJK UNIFIED IDEOGRAPH
                                (#x84CB . #x52CA)	;CJK UNIFIED IDEOGRAPH
                                (#x84CC . #x52CC)	;CJK UNIFIED IDEOGRAPH
                                (#x84CD . #x52CD)	;CJK UNIFIED IDEOGRAPH
                                (#x84CE . #x52CE)	;CJK UNIFIED IDEOGRAPH
                                (#x84CF . #x52CF)	;CJK UNIFIED IDEOGRAPH
                                (#x84D0 . #x52D1)	;CJK UNIFIED IDEOGRAPH
                                (#x84D1 . #x52D3)	;CJK UNIFIED IDEOGRAPH
                                (#x84D2 . #x52D4)	;CJK UNIFIED IDEOGRAPH
                                (#x84D3 . #x52D5)	;CJK UNIFIED IDEOGRAPH
                                (#x84D4 . #x52D7)	;CJK UNIFIED IDEOGRAPH
                                (#x84D5 . #x52D9)	;CJK UNIFIED IDEOGRAPH
                                (#x84D6 . #x52DA)	;CJK UNIFIED IDEOGRAPH
                                (#x84D7 . #x52DB)	;CJK UNIFIED IDEOGRAPH
                                (#x84D8 . #x52DC)	;CJK UNIFIED IDEOGRAPH
                                (#x84D9 . #x52DD)	;CJK UNIFIED IDEOGRAPH
                                (#x84DA . #x52DE)	;CJK UNIFIED IDEOGRAPH
                                (#x84DB . #x52E0)	;CJK UNIFIED IDEOGRAPH
                                (#x84DC . #x52E1)	;CJK UNIFIED IDEOGRAPH
                                (#x84DD . #x52E2)	;CJK UNIFIED IDEOGRAPH
                                (#x84DE . #x52E3)	;CJK UNIFIED IDEOGRAPH
                                (#x84DF . #x52E5)	;CJK UNIFIED IDEOGRAPH
                                (#x84E0 . #x52E6)	;CJK UNIFIED IDEOGRAPH
                                (#x84E1 . #x52E7)	;CJK UNIFIED IDEOGRAPH
                                (#x84E2 . #x52E8)	;CJK UNIFIED IDEOGRAPH
                                (#x84E3 . #x52E9)	;CJK UNIFIED IDEOGRAPH
                                (#x84E4 . #x52EA)	;CJK UNIFIED IDEOGRAPH
                                (#x84E5 . #x52EB)	;CJK UNIFIED IDEOGRAPH
                                (#x84E6 . #x52EC)	;CJK UNIFIED IDEOGRAPH
                                (#x84E7 . #x52ED)	;CJK UNIFIED IDEOGRAPH
                                (#x84E8 . #x52EE)	;CJK UNIFIED IDEOGRAPH
                                (#x84E9 . #x52EF)	;CJK UNIFIED IDEOGRAPH
                                (#x84EA . #x52F1)	;CJK UNIFIED IDEOGRAPH
                                (#x84EB . #x52F2)	;CJK UNIFIED IDEOGRAPH
                                (#x84EC . #x52F3)	;CJK UNIFIED IDEOGRAPH
                                (#x84ED . #x52F4)	;CJK UNIFIED IDEOGRAPH
                                (#x84EE . #x52F5)	;CJK UNIFIED IDEOGRAPH
                                (#x84EF . #x52F6)	;CJK UNIFIED IDEOGRAPH
                                (#x84F0 . #x52F7)	;CJK UNIFIED IDEOGRAPH
                                (#x84F1 . #x52F8)	;CJK UNIFIED IDEOGRAPH
                                (#x84F2 . #x52FB)	;CJK UNIFIED IDEOGRAPH
                                (#x84F3 . #x52FC)	;CJK UNIFIED IDEOGRAPH
                                (#x84F4 . #x52FD)	;CJK UNIFIED IDEOGRAPH
                                (#x84F5 . #x5301)	;CJK UNIFIED IDEOGRAPH
                                (#x84F6 . #x5302)	;CJK UNIFIED IDEOGRAPH
                                (#x84F7 . #x5303)	;CJK UNIFIED IDEOGRAPH
                                (#x84F8 . #x5304)	;CJK UNIFIED IDEOGRAPH
                                (#x84F9 . #x5307)	;CJK UNIFIED IDEOGRAPH
                                (#x84FA . #x5309)	;CJK UNIFIED IDEOGRAPH
                                (#x84FB . #x530A)	;CJK UNIFIED IDEOGRAPH
                                (#x84FC . #x530B)	;CJK UNIFIED IDEOGRAPH
                                (#x84FD . #x530C)	;CJK UNIFIED IDEOGRAPH
                                (#x84FE . #x530E)	;CJK UNIFIED IDEOGRAPH
                                (#x8540 . #x5311)	;CJK UNIFIED IDEOGRAPH
                                (#x8541 . #x5312)	;CJK UNIFIED IDEOGRAPH
                                (#x8542 . #x5313)	;CJK UNIFIED IDEOGRAPH
                                (#x8543 . #x5314)	;CJK UNIFIED IDEOGRAPH
                                (#x8544 . #x5318)	;CJK UNIFIED IDEOGRAPH
                                (#x8545 . #x531B)	;CJK UNIFIED IDEOGRAPH
                                (#x8546 . #x531C)	;CJK UNIFIED IDEOGRAPH
                                (#x8547 . #x531E)	;CJK UNIFIED IDEOGRAPH
                                (#x8548 . #x531F)	;CJK UNIFIED IDEOGRAPH
                                (#x8549 . #x5322)	;CJK UNIFIED IDEOGRAPH
                                (#x854A . #x5324)	;CJK UNIFIED IDEOGRAPH
                                (#x854B . #x5325)	;CJK UNIFIED IDEOGRAPH
                                (#x854C . #x5327)	;CJK UNIFIED IDEOGRAPH
                                (#x854D . #x5328)	;CJK UNIFIED IDEOGRAPH
                                (#x854E . #x5329)	;CJK UNIFIED IDEOGRAPH
                                (#x854F . #x532B)	;CJK UNIFIED IDEOGRAPH
                                (#x8550 . #x532C)	;CJK UNIFIED IDEOGRAPH
                                (#x8551 . #x532D)	;CJK UNIFIED IDEOGRAPH
                                (#x8552 . #x532F)	;CJK UNIFIED IDEOGRAPH
                                (#x8553 . #x5330)	;CJK UNIFIED IDEOGRAPH
                                (#x8554 . #x5331)	;CJK UNIFIED IDEOGRAPH
                                (#x8555 . #x5332)	;CJK UNIFIED IDEOGRAPH
                                (#x8556 . #x5333)	;CJK UNIFIED IDEOGRAPH
                                (#x8557 . #x5334)	;CJK UNIFIED IDEOGRAPH
                                (#x8558 . #x5335)	;CJK UNIFIED IDEOGRAPH
                                (#x8559 . #x5336)	;CJK UNIFIED IDEOGRAPH
                                (#x855A . #x5337)	;CJK UNIFIED IDEOGRAPH
                                (#x855B . #x5338)	;CJK UNIFIED IDEOGRAPH
                                (#x855C . #x533C)	;CJK UNIFIED IDEOGRAPH
                                (#x855D . #x533D)	;CJK UNIFIED IDEOGRAPH
                                (#x855E . #x5340)	;CJK UNIFIED IDEOGRAPH
                                (#x855F . #x5342)	;CJK UNIFIED IDEOGRAPH
                                (#x8560 . #x5344)	;CJK UNIFIED IDEOGRAPH
                                (#x8561 . #x5346)	;CJK UNIFIED IDEOGRAPH
                                (#x8562 . #x534B)	;CJK UNIFIED IDEOGRAPH
                                (#x8563 . #x534C)	;CJK UNIFIED IDEOGRAPH
                                (#x8564 . #x534D)	;CJK UNIFIED IDEOGRAPH
                                (#x8565 . #x5350)	;CJK UNIFIED IDEOGRAPH
                                (#x8566 . #x5354)	;CJK UNIFIED IDEOGRAPH
                                (#x8567 . #x5358)	;CJK UNIFIED IDEOGRAPH
                                (#x8568 . #x5359)	;CJK UNIFIED IDEOGRAPH
                                (#x8569 . #x535B)	;CJK UNIFIED IDEOGRAPH
                                (#x856A . #x535D)	;CJK UNIFIED IDEOGRAPH
                                (#x856B . #x5365)	;CJK UNIFIED IDEOGRAPH
                                (#x856C . #x5368)	;CJK UNIFIED IDEOGRAPH
                                (#x856D . #x536A)	;CJK UNIFIED IDEOGRAPH
                                (#x856E . #x536C)	;CJK UNIFIED IDEOGRAPH
                                (#x856F . #x536D)	;CJK UNIFIED IDEOGRAPH
                                (#x8570 . #x5372)	;CJK UNIFIED IDEOGRAPH
                                (#x8571 . #x5376)	;CJK UNIFIED IDEOGRAPH
                                (#x8572 . #x5379)	;CJK UNIFIED IDEOGRAPH
                                (#x8573 . #x537B)	;CJK UNIFIED IDEOGRAPH
                                (#x8574 . #x537C)	;CJK UNIFIED IDEOGRAPH
                                (#x8575 . #x537D)	;CJK UNIFIED IDEOGRAPH
                                (#x8576 . #x537E)	;CJK UNIFIED IDEOGRAPH
                                (#x8577 . #x5380)	;CJK UNIFIED IDEOGRAPH
                                (#x8578 . #x5381)	;CJK UNIFIED IDEOGRAPH
                                (#x8579 . #x5383)	;CJK UNIFIED IDEOGRAPH
                                (#x857A . #x5387)	;CJK UNIFIED IDEOGRAPH
                                (#x857B . #x5388)	;CJK UNIFIED IDEOGRAPH
                                (#x857C . #x538A)	;CJK UNIFIED IDEOGRAPH
                                (#x857D . #x538E)	;CJK UNIFIED IDEOGRAPH
                                (#x857E . #x538F)	;CJK UNIFIED IDEOGRAPH
                                (#x8580 . #x5390)	;CJK UNIFIED IDEOGRAPH
                                (#x8581 . #x5391)	;CJK UNIFIED IDEOGRAPH
                                (#x8582 . #x5392)	;CJK UNIFIED IDEOGRAPH
                                (#x8583 . #x5393)	;CJK UNIFIED IDEOGRAPH
                                (#x8584 . #x5394)	;CJK UNIFIED IDEOGRAPH
                                (#x8585 . #x5396)	;CJK UNIFIED IDEOGRAPH
                                (#x8586 . #x5397)	;CJK UNIFIED IDEOGRAPH
                                (#x8587 . #x5399)	;CJK UNIFIED IDEOGRAPH
                                (#x8588 . #x539B)	;CJK UNIFIED IDEOGRAPH
                                (#x8589 . #x539C)	;CJK UNIFIED IDEOGRAPH
                                (#x858A . #x539E)	;CJK UNIFIED IDEOGRAPH
                                (#x858B . #x53A0)	;CJK UNIFIED IDEOGRAPH
                                (#x858C . #x53A1)	;CJK UNIFIED IDEOGRAPH
                                (#x858D . #x53A4)	;CJK UNIFIED IDEOGRAPH
                                (#x858E . #x53A7)	;CJK UNIFIED IDEOGRAPH
                                (#x858F . #x53AA)	;CJK UNIFIED IDEOGRAPH
                                (#x8590 . #x53AB)	;CJK UNIFIED IDEOGRAPH
                                (#x8591 . #x53AC)	;CJK UNIFIED IDEOGRAPH
                                (#x8592 . #x53AD)	;CJK UNIFIED IDEOGRAPH
                                (#x8593 . #x53AF)	;CJK UNIFIED IDEOGRAPH
                                (#x8594 . #x53B0)	;CJK UNIFIED IDEOGRAPH
                                (#x8595 . #x53B1)	;CJK UNIFIED IDEOGRAPH
                                (#x8596 . #x53B2)	;CJK UNIFIED IDEOGRAPH
                                (#x8597 . #x53B3)	;CJK UNIFIED IDEOGRAPH
                                (#x8598 . #x53B4)	;CJK UNIFIED IDEOGRAPH
                                (#x8599 . #x53B5)	;CJK UNIFIED IDEOGRAPH
                                (#x859A . #x53B7)	;CJK UNIFIED IDEOGRAPH
                                (#x859B . #x53B8)	;CJK UNIFIED IDEOGRAPH
                                (#x859C . #x53B9)	;CJK UNIFIED IDEOGRAPH
                                (#x859D . #x53BA)	;CJK UNIFIED IDEOGRAPH
                                (#x859E . #x53BC)	;CJK UNIFIED IDEOGRAPH
                                (#x859F . #x53BD)	;CJK UNIFIED IDEOGRAPH
                                (#x85A0 . #x53BE)	;CJK UNIFIED IDEOGRAPH
                                (#x85A1 . #x53C0)	;CJK UNIFIED IDEOGRAPH
                                (#x85A2 . #x53C3)	;CJK UNIFIED IDEOGRAPH
                                (#x85A3 . #x53C4)	;CJK UNIFIED IDEOGRAPH
                                (#x85A4 . #x53C5)	;CJK UNIFIED IDEOGRAPH
                                (#x85A5 . #x53C6)	;CJK UNIFIED IDEOGRAPH
                                (#x85A6 . #x53C7)	;CJK UNIFIED IDEOGRAPH
                                (#x85A7 . #x53CE)	;CJK UNIFIED IDEOGRAPH
                                (#x85A8 . #x53CF)	;CJK UNIFIED IDEOGRAPH
                                (#x85A9 . #x53D0)	;CJK UNIFIED IDEOGRAPH
                                (#x85AA . #x53D2)	;CJK UNIFIED IDEOGRAPH
                                (#x85AB . #x53D3)	;CJK UNIFIED IDEOGRAPH
                                (#x85AC . #x53D5)	;CJK UNIFIED IDEOGRAPH
                                (#x85AD . #x53DA)	;CJK UNIFIED IDEOGRAPH
                                (#x85AE . #x53DC)	;CJK UNIFIED IDEOGRAPH
                                (#x85AF . #x53DD)	;CJK UNIFIED IDEOGRAPH
                                (#x85B0 . #x53DE)	;CJK UNIFIED IDEOGRAPH
                                (#x85B1 . #x53E1)	;CJK UNIFIED IDEOGRAPH
                                (#x85B2 . #x53E2)	;CJK UNIFIED IDEOGRAPH
                                (#x85B3 . #x53E7)	;CJK UNIFIED IDEOGRAPH
                                (#x85B4 . #x53F4)	;CJK UNIFIED IDEOGRAPH
                                (#x85B5 . #x53FA)	;CJK UNIFIED IDEOGRAPH
                                (#x85B6 . #x53FE)	;CJK UNIFIED IDEOGRAPH
                                (#x85B7 . #x53FF)	;CJK UNIFIED IDEOGRAPH
                                (#x85B8 . #x5400)	;CJK UNIFIED IDEOGRAPH
                                (#x85B9 . #x5402)	;CJK UNIFIED IDEOGRAPH
                                (#x85BA . #x5405)	;CJK UNIFIED IDEOGRAPH
                                (#x85BB . #x5407)	;CJK UNIFIED IDEOGRAPH
                                (#x85BC . #x540B)	;CJK UNIFIED IDEOGRAPH
                                (#x85BD . #x5414)	;CJK UNIFIED IDEOGRAPH
                                (#x85BE . #x5418)	;CJK UNIFIED IDEOGRAPH
                                (#x85BF . #x5419)	;CJK UNIFIED IDEOGRAPH
                                (#x85C0 . #x541A)	;CJK UNIFIED IDEOGRAPH
                                (#x85C1 . #x541C)	;CJK UNIFIED IDEOGRAPH
                                (#x85C2 . #x5422)	;CJK UNIFIED IDEOGRAPH
                                (#x85C3 . #x5424)	;CJK UNIFIED IDEOGRAPH
                                (#x85C4 . #x5425)	;CJK UNIFIED IDEOGRAPH
                                (#x85C5 . #x542A)	;CJK UNIFIED IDEOGRAPH
                                (#x85C6 . #x5430)	;CJK UNIFIED IDEOGRAPH
                                (#x85C7 . #x5433)	;CJK UNIFIED IDEOGRAPH
                                (#x85C8 . #x5436)	;CJK UNIFIED IDEOGRAPH
                                (#x85C9 . #x5437)	;CJK UNIFIED IDEOGRAPH
                                (#x85CA . #x543A)	;CJK UNIFIED IDEOGRAPH
                                (#x85CB . #x543D)	;CJK UNIFIED IDEOGRAPH
                                (#x85CC . #x543F)	;CJK UNIFIED IDEOGRAPH
                                (#x85CD . #x5441)	;CJK UNIFIED IDEOGRAPH
                                (#x85CE . #x5442)	;CJK UNIFIED IDEOGRAPH
                                (#x85CF . #x5444)	;CJK UNIFIED IDEOGRAPH
                                (#x85D0 . #x5445)	;CJK UNIFIED IDEOGRAPH
                                (#x85D1 . #x5447)	;CJK UNIFIED IDEOGRAPH
                                (#x85D2 . #x5449)	;CJK UNIFIED IDEOGRAPH
                                (#x85D3 . #x544C)	;CJK UNIFIED IDEOGRAPH
                                (#x85D4 . #x544D)	;CJK UNIFIED IDEOGRAPH
                                (#x85D5 . #x544E)	;CJK UNIFIED IDEOGRAPH
                                (#x85D6 . #x544F)	;CJK UNIFIED IDEOGRAPH
                                (#x85D7 . #x5451)	;CJK UNIFIED IDEOGRAPH
                                (#x85D8 . #x545A)	;CJK UNIFIED IDEOGRAPH
                                (#x85D9 . #x545D)	;CJK UNIFIED IDEOGRAPH
                                (#x85DA . #x545E)	;CJK UNIFIED IDEOGRAPH
                                (#x85DB . #x545F)	;CJK UNIFIED IDEOGRAPH
                                (#x85DC . #x5460)	;CJK UNIFIED IDEOGRAPH
                                (#x85DD . #x5461)	;CJK UNIFIED IDEOGRAPH
                                (#x85DE . #x5463)	;CJK UNIFIED IDEOGRAPH
                                (#x85DF . #x5465)	;CJK UNIFIED IDEOGRAPH
                                (#x85E0 . #x5467)	;CJK UNIFIED IDEOGRAPH
                                (#x85E1 . #x5469)	;CJK UNIFIED IDEOGRAPH
                                (#x85E2 . #x546A)	;CJK UNIFIED IDEOGRAPH
                                (#x85E3 . #x546B)	;CJK UNIFIED IDEOGRAPH
                                (#x85E4 . #x546C)	;CJK UNIFIED IDEOGRAPH
                                (#x85E5 . #x546D)	;CJK UNIFIED IDEOGRAPH
                                (#x85E6 . #x546E)	;CJK UNIFIED IDEOGRAPH
                                (#x85E7 . #x546F)	;CJK UNIFIED IDEOGRAPH
                                (#x85E8 . #x5470)	;CJK UNIFIED IDEOGRAPH
                                (#x85E9 . #x5474)	;CJK UNIFIED IDEOGRAPH
                                (#x85EA . #x5479)	;CJK UNIFIED IDEOGRAPH
                                (#x85EB . #x547A)	;CJK UNIFIED IDEOGRAPH
                                (#x85EC . #x547E)	;CJK UNIFIED IDEOGRAPH
                                (#x85ED . #x547F)	;CJK UNIFIED IDEOGRAPH
                                (#x85EE . #x5481)	;CJK UNIFIED IDEOGRAPH
                                (#x85EF . #x5483)	;CJK UNIFIED IDEOGRAPH
                                (#x85F0 . #x5485)	;CJK UNIFIED IDEOGRAPH
                                (#x85F1 . #x5487)	;CJK UNIFIED IDEOGRAPH
                                (#x85F2 . #x5488)	;CJK UNIFIED IDEOGRAPH
                                (#x85F3 . #x5489)	;CJK UNIFIED IDEOGRAPH
                                (#x85F4 . #x548A)	;CJK UNIFIED IDEOGRAPH
                                (#x85F5 . #x548D)	;CJK UNIFIED IDEOGRAPH
                                (#x85F6 . #x5491)	;CJK UNIFIED IDEOGRAPH
                                (#x85F7 . #x5493)	;CJK UNIFIED IDEOGRAPH
                                (#x85F8 . #x5497)	;CJK UNIFIED IDEOGRAPH
                                (#x85F9 . #x5498)	;CJK UNIFIED IDEOGRAPH
                                (#x85FA . #x549C)	;CJK UNIFIED IDEOGRAPH
                                (#x85FB . #x549E)	;CJK UNIFIED IDEOGRAPH
                                (#x85FC . #x549F)	;CJK UNIFIED IDEOGRAPH
                                (#x85FD . #x54A0)	;CJK UNIFIED IDEOGRAPH
                                (#x85FE . #x54A1)	;CJK UNIFIED IDEOGRAPH
                                (#x8640 . #x54A2)	;CJK UNIFIED IDEOGRAPH
                                (#x8641 . #x54A5)	;CJK UNIFIED IDEOGRAPH
                                (#x8642 . #x54AE)	;CJK UNIFIED IDEOGRAPH
                                (#x8643 . #x54B0)	;CJK UNIFIED IDEOGRAPH
                                (#x8644 . #x54B2)	;CJK UNIFIED IDEOGRAPH
                                (#x8645 . #x54B5)	;CJK UNIFIED IDEOGRAPH
                                (#x8646 . #x54B6)	;CJK UNIFIED IDEOGRAPH
                                (#x8647 . #x54B7)	;CJK UNIFIED IDEOGRAPH
                                (#x8648 . #x54B9)	;CJK UNIFIED IDEOGRAPH
                                (#x8649 . #x54BA)	;CJK UNIFIED IDEOGRAPH
                                (#x864A . #x54BC)	;CJK UNIFIED IDEOGRAPH
                                (#x864B . #x54BE)	;CJK UNIFIED IDEOGRAPH
                                (#x864C . #x54C3)	;CJK UNIFIED IDEOGRAPH
                                (#x864D . #x54C5)	;CJK UNIFIED IDEOGRAPH
                                (#x864E . #x54CA)	;CJK UNIFIED IDEOGRAPH
                                (#x864F . #x54CB)	;CJK UNIFIED IDEOGRAPH
                                (#x8650 . #x54D6)	;CJK UNIFIED IDEOGRAPH
                                (#x8651 . #x54D8)	;CJK UNIFIED IDEOGRAPH
                                (#x8652 . #x54DB)	;CJK UNIFIED IDEOGRAPH
                                (#x8653 . #x54E0)	;CJK UNIFIED IDEOGRAPH
                                (#x8654 . #x54E1)	;CJK UNIFIED IDEOGRAPH
                                (#x8655 . #x54E2)	;CJK UNIFIED IDEOGRAPH
                                (#x8656 . #x54E3)	;CJK UNIFIED IDEOGRAPH
                                (#x8657 . #x54E4)	;CJK UNIFIED IDEOGRAPH
                                (#x8658 . #x54EB)	;CJK UNIFIED IDEOGRAPH
                                (#x8659 . #x54EC)	;CJK UNIFIED IDEOGRAPH
                                (#x865A . #x54EF)	;CJK UNIFIED IDEOGRAPH
                                (#x865B . #x54F0)	;CJK UNIFIED IDEOGRAPH
                                (#x865C . #x54F1)	;CJK UNIFIED IDEOGRAPH
                                (#x865D . #x54F4)	;CJK UNIFIED IDEOGRAPH
                                (#x865E . #x54F5)	;CJK UNIFIED IDEOGRAPH
                                (#x865F . #x54F6)	;CJK UNIFIED IDEOGRAPH
                                (#x8660 . #x54F7)	;CJK UNIFIED IDEOGRAPH
                                (#x8661 . #x54F8)	;CJK UNIFIED IDEOGRAPH
                                (#x8662 . #x54F9)	;CJK UNIFIED IDEOGRAPH
                                (#x8663 . #x54FB)	;CJK UNIFIED IDEOGRAPH
                                (#x8664 . #x54FE)	;CJK UNIFIED IDEOGRAPH
                                (#x8665 . #x5500)	;CJK UNIFIED IDEOGRAPH
                                (#x8666 . #x5502)	;CJK UNIFIED IDEOGRAPH
                                (#x8667 . #x5503)	;CJK UNIFIED IDEOGRAPH
                                (#x8668 . #x5504)	;CJK UNIFIED IDEOGRAPH
                                (#x8669 . #x5505)	;CJK UNIFIED IDEOGRAPH
                                (#x866A . #x5508)	;CJK UNIFIED IDEOGRAPH
                                (#x866B . #x550A)	;CJK UNIFIED IDEOGRAPH
                                (#x866C . #x550B)	;CJK UNIFIED IDEOGRAPH
                                (#x866D . #x550C)	;CJK UNIFIED IDEOGRAPH
                                (#x866E . #x550D)	;CJK UNIFIED IDEOGRAPH
                                (#x866F . #x550E)	;CJK UNIFIED IDEOGRAPH
                                (#x8670 . #x5512)	;CJK UNIFIED IDEOGRAPH
                                (#x8671 . #x5513)	;CJK UNIFIED IDEOGRAPH
                                (#x8672 . #x5515)	;CJK UNIFIED IDEOGRAPH
                                (#x8673 . #x5516)	;CJK UNIFIED IDEOGRAPH
                                (#x8674 . #x5517)	;CJK UNIFIED IDEOGRAPH
                                (#x8675 . #x5518)	;CJK UNIFIED IDEOGRAPH
                                (#x8676 . #x5519)	;CJK UNIFIED IDEOGRAPH
                                (#x8677 . #x551A)	;CJK UNIFIED IDEOGRAPH
                                (#x8678 . #x551C)	;CJK UNIFIED IDEOGRAPH
                                (#x8679 . #x551D)	;CJK UNIFIED IDEOGRAPH
                                (#x867A . #x551E)	;CJK UNIFIED IDEOGRAPH
                                (#x867B . #x551F)	;CJK UNIFIED IDEOGRAPH
                                (#x867C . #x5521)	;CJK UNIFIED IDEOGRAPH
                                (#x867D . #x5525)	;CJK UNIFIED IDEOGRAPH
                                (#x867E . #x5526)	;CJK UNIFIED IDEOGRAPH
                                (#x8680 . #x5528)	;CJK UNIFIED IDEOGRAPH
                                (#x8681 . #x5529)	;CJK UNIFIED IDEOGRAPH
                                (#x8682 . #x552B)	;CJK UNIFIED IDEOGRAPH
                                (#x8683 . #x552D)	;CJK UNIFIED IDEOGRAPH
                                (#x8684 . #x5532)	;CJK UNIFIED IDEOGRAPH
                                (#x8685 . #x5534)	;CJK UNIFIED IDEOGRAPH
                                (#x8686 . #x5535)	;CJK UNIFIED IDEOGRAPH
                                (#x8687 . #x5536)	;CJK UNIFIED IDEOGRAPH
                                (#x8688 . #x5538)	;CJK UNIFIED IDEOGRAPH
                                (#x8689 . #x5539)	;CJK UNIFIED IDEOGRAPH
                                (#x868A . #x553A)	;CJK UNIFIED IDEOGRAPH
                                (#x868B . #x553B)	;CJK UNIFIED IDEOGRAPH
                                (#x868C . #x553D)	;CJK UNIFIED IDEOGRAPH
                                (#x868D . #x5540)	;CJK UNIFIED IDEOGRAPH
                                (#x868E . #x5542)	;CJK UNIFIED IDEOGRAPH
                                (#x868F . #x5545)	;CJK UNIFIED IDEOGRAPH
                                (#x8690 . #x5547)	;CJK UNIFIED IDEOGRAPH
                                (#x8691 . #x5548)	;CJK UNIFIED IDEOGRAPH
                                (#x8692 . #x554B)	;CJK UNIFIED IDEOGRAPH
                                (#x8693 . #x554C)	;CJK UNIFIED IDEOGRAPH
                                (#x8694 . #x554D)	;CJK UNIFIED IDEOGRAPH
                                (#x8695 . #x554E)	;CJK UNIFIED IDEOGRAPH
                                (#x8696 . #x554F)	;CJK UNIFIED IDEOGRAPH
                                (#x8697 . #x5551)	;CJK UNIFIED IDEOGRAPH
                                (#x8698 . #x5552)	;CJK UNIFIED IDEOGRAPH
                                (#x8699 . #x5553)	;CJK UNIFIED IDEOGRAPH
                                (#x869A . #x5554)	;CJK UNIFIED IDEOGRAPH
                                (#x869B . #x5557)	;CJK UNIFIED IDEOGRAPH
                                (#x869C . #x5558)	;CJK UNIFIED IDEOGRAPH
                                (#x869D . #x5559)	;CJK UNIFIED IDEOGRAPH
                                (#x869E . #x555A)	;CJK UNIFIED IDEOGRAPH
                                (#x869F . #x555B)	;CJK UNIFIED IDEOGRAPH
                                (#x86A0 . #x555D)	;CJK UNIFIED IDEOGRAPH
                                (#x86A1 . #x555E)	;CJK UNIFIED IDEOGRAPH
                                (#x86A2 . #x555F)	;CJK UNIFIED IDEOGRAPH
                                (#x86A3 . #x5560)	;CJK UNIFIED IDEOGRAPH
                                (#x86A4 . #x5562)	;CJK UNIFIED IDEOGRAPH
                                (#x86A5 . #x5563)	;CJK UNIFIED IDEOGRAPH
                                (#x86A6 . #x5568)	;CJK UNIFIED IDEOGRAPH
                                (#x86A7 . #x5569)	;CJK UNIFIED IDEOGRAPH
                                (#x86A8 . #x556B)	;CJK UNIFIED IDEOGRAPH
                                (#x86A9 . #x556F)	;CJK UNIFIED IDEOGRAPH
                                (#x86AA . #x5570)	;CJK UNIFIED IDEOGRAPH
                                (#x86AB . #x5571)	;CJK UNIFIED IDEOGRAPH
                                (#x86AC . #x5572)	;CJK UNIFIED IDEOGRAPH
                                (#x86AD . #x5573)	;CJK UNIFIED IDEOGRAPH
                                (#x86AE . #x5574)	;CJK UNIFIED IDEOGRAPH
                                (#x86AF . #x5579)	;CJK UNIFIED IDEOGRAPH
                                (#x86B0 . #x557A)	;CJK UNIFIED IDEOGRAPH
                                (#x86B1 . #x557D)	;CJK UNIFIED IDEOGRAPH
                                (#x86B2 . #x557F)	;CJK UNIFIED IDEOGRAPH
                                (#x86B3 . #x5585)	;CJK UNIFIED IDEOGRAPH
                                (#x86B4 . #x5586)	;CJK UNIFIED IDEOGRAPH
                                (#x86B5 . #x558C)	;CJK UNIFIED IDEOGRAPH
                                (#x86B6 . #x558D)	;CJK UNIFIED IDEOGRAPH
                                (#x86B7 . #x558E)	;CJK UNIFIED IDEOGRAPH
                                (#x86B8 . #x5590)	;CJK UNIFIED IDEOGRAPH
                                (#x86B9 . #x5592)	;CJK UNIFIED IDEOGRAPH
                                (#x86BA . #x5593)	;CJK UNIFIED IDEOGRAPH
                                (#x86BB . #x5595)	;CJK UNIFIED IDEOGRAPH
                                (#x86BC . #x5596)	;CJK UNIFIED IDEOGRAPH
                                (#x86BD . #x5597)	;CJK UNIFIED IDEOGRAPH
                                (#x86BE . #x559A)	;CJK UNIFIED IDEOGRAPH
                                (#x86BF . #x559B)	;CJK UNIFIED IDEOGRAPH
                                (#x86C0 . #x559E)	;CJK UNIFIED IDEOGRAPH
                                (#x86C1 . #x55A0)	;CJK UNIFIED IDEOGRAPH
                                (#x86C2 . #x55A1)	;CJK UNIFIED IDEOGRAPH
                                (#x86C3 . #x55A2)	;CJK UNIFIED IDEOGRAPH
                                (#x86C4 . #x55A3)	;CJK UNIFIED IDEOGRAPH
                                (#x86C5 . #x55A4)	;CJK UNIFIED IDEOGRAPH
                                (#x86C6 . #x55A5)	;CJK UNIFIED IDEOGRAPH
                                (#x86C7 . #x55A6)	;CJK UNIFIED IDEOGRAPH
                                (#x86C8 . #x55A8)	;CJK UNIFIED IDEOGRAPH
                                (#x86C9 . #x55A9)	;CJK UNIFIED IDEOGRAPH
                                (#x86CA . #x55AA)	;CJK UNIFIED IDEOGRAPH
                                (#x86CB . #x55AB)	;CJK UNIFIED IDEOGRAPH
                                (#x86CC . #x55AC)	;CJK UNIFIED IDEOGRAPH
                                (#x86CD . #x55AD)	;CJK UNIFIED IDEOGRAPH
                                (#x86CE . #x55AE)	;CJK UNIFIED IDEOGRAPH
                                (#x86CF . #x55AF)	;CJK UNIFIED IDEOGRAPH
                                (#x86D0 . #x55B0)	;CJK UNIFIED IDEOGRAPH
                                (#x86D1 . #x55B2)	;CJK UNIFIED IDEOGRAPH
                                (#x86D2 . #x55B4)	;CJK UNIFIED IDEOGRAPH
                                (#x86D3 . #x55B6)	;CJK UNIFIED IDEOGRAPH
                                (#x86D4 . #x55B8)	;CJK UNIFIED IDEOGRAPH
                                (#x86D5 . #x55BA)	;CJK UNIFIED IDEOGRAPH
                                (#x86D6 . #x55BC)	;CJK UNIFIED IDEOGRAPH
                                (#x86D7 . #x55BF)	;CJK UNIFIED IDEOGRAPH
                                (#x86D8 . #x55C0)	;CJK UNIFIED IDEOGRAPH
                                (#x86D9 . #x55C1)	;CJK UNIFIED IDEOGRAPH
                                (#x86DA . #x55C2)	;CJK UNIFIED IDEOGRAPH
                                (#x86DB . #x55C3)	;CJK UNIFIED IDEOGRAPH
                                (#x86DC . #x55C6)	;CJK UNIFIED IDEOGRAPH
                                (#x86DD . #x55C7)	;CJK UNIFIED IDEOGRAPH
                                (#x86DE . #x55C8)	;CJK UNIFIED IDEOGRAPH
                                (#x86DF . #x55CA)	;CJK UNIFIED IDEOGRAPH
                                (#x86E0 . #x55CB)	;CJK UNIFIED IDEOGRAPH
                                (#x86E1 . #x55CE)	;CJK UNIFIED IDEOGRAPH
                                (#x86E2 . #x55CF)	;CJK UNIFIED IDEOGRAPH
                                (#x86E3 . #x55D0)	;CJK UNIFIED IDEOGRAPH
                                (#x86E4 . #x55D5)	;CJK UNIFIED IDEOGRAPH
                                (#x86E5 . #x55D7)	;CJK UNIFIED IDEOGRAPH
                                (#x86E6 . #x55D8)	;CJK UNIFIED IDEOGRAPH
                                (#x86E7 . #x55D9)	;CJK UNIFIED IDEOGRAPH
                                (#x86E8 . #x55DA)	;CJK UNIFIED IDEOGRAPH
                                (#x86E9 . #x55DB)	;CJK UNIFIED IDEOGRAPH
                                (#x86EA . #x55DE)	;CJK UNIFIED IDEOGRAPH
                                (#x86EB . #x55E0)	;CJK UNIFIED IDEOGRAPH
                                (#x86EC . #x55E2)	;CJK UNIFIED IDEOGRAPH
                                (#x86ED . #x55E7)	;CJK UNIFIED IDEOGRAPH
                                (#x86EE . #x55E9)	;CJK UNIFIED IDEOGRAPH
                                (#x86EF . #x55ED)	;CJK UNIFIED IDEOGRAPH
                                (#x86F0 . #x55EE)	;CJK UNIFIED IDEOGRAPH
                                (#x86F1 . #x55F0)	;CJK UNIFIED IDEOGRAPH
                                (#x86F2 . #x55F1)	;CJK UNIFIED IDEOGRAPH
                                (#x86F3 . #x55F4)	;CJK UNIFIED IDEOGRAPH
                                (#x86F4 . #x55F6)	;CJK UNIFIED IDEOGRAPH
                                (#x86F5 . #x55F8)	;CJK UNIFIED IDEOGRAPH
                                (#x86F6 . #x55F9)	;CJK UNIFIED IDEOGRAPH
                                (#x86F7 . #x55FA)	;CJK UNIFIED IDEOGRAPH
                                (#x86F8 . #x55FB)	;CJK UNIFIED IDEOGRAPH
                                (#x86F9 . #x55FC)	;CJK UNIFIED IDEOGRAPH
                                (#x86FA . #x55FF)	;CJK UNIFIED IDEOGRAPH
                                (#x86FB . #x5602)	;CJK UNIFIED IDEOGRAPH
                                (#x86FC . #x5603)	;CJK UNIFIED IDEOGRAPH
                                (#x86FD . #x5604)	;CJK UNIFIED IDEOGRAPH
                                (#x86FE . #x5605)	;CJK UNIFIED IDEOGRAPH
                                (#x8740 . #x5606)	;CJK UNIFIED IDEOGRAPH
                                (#x8741 . #x5607)	;CJK UNIFIED IDEOGRAPH
                                (#x8742 . #x560A)	;CJK UNIFIED IDEOGRAPH
                                (#x8743 . #x560B)	;CJK UNIFIED IDEOGRAPH
                                (#x8744 . #x560D)	;CJK UNIFIED IDEOGRAPH
                                (#x8745 . #x5610)	;CJK UNIFIED IDEOGRAPH
                                (#x8746 . #x5611)	;CJK UNIFIED IDEOGRAPH
                                (#x8747 . #x5612)	;CJK UNIFIED IDEOGRAPH
                                (#x8748 . #x5613)	;CJK UNIFIED IDEOGRAPH
                                (#x8749 . #x5614)	;CJK UNIFIED IDEOGRAPH
                                (#x874A . #x5615)	;CJK UNIFIED IDEOGRAPH
                                (#x874B . #x5616)	;CJK UNIFIED IDEOGRAPH
                                (#x874C . #x5617)	;CJK UNIFIED IDEOGRAPH
                                (#x874D . #x5619)	;CJK UNIFIED IDEOGRAPH
                                (#x874E . #x561A)	;CJK UNIFIED IDEOGRAPH
                                (#x874F . #x561C)	;CJK UNIFIED IDEOGRAPH
                                (#x8750 . #x561D)	;CJK UNIFIED IDEOGRAPH
                                (#x8751 . #x5620)	;CJK UNIFIED IDEOGRAPH
                                (#x8752 . #x5621)	;CJK UNIFIED IDEOGRAPH
                                (#x8753 . #x5622)	;CJK UNIFIED IDEOGRAPH
                                (#x8754 . #x5625)	;CJK UNIFIED IDEOGRAPH
                                (#x8755 . #x5626)	;CJK UNIFIED IDEOGRAPH
                                (#x8756 . #x5628)	;CJK UNIFIED IDEOGRAPH
                                (#x8757 . #x5629)	;CJK UNIFIED IDEOGRAPH
                                (#x8758 . #x562A)	;CJK UNIFIED IDEOGRAPH
                                (#x8759 . #x562B)	;CJK UNIFIED IDEOGRAPH
                                (#x875A . #x562E)	;CJK UNIFIED IDEOGRAPH
                                (#x875B . #x562F)	;CJK UNIFIED IDEOGRAPH
                                (#x875C . #x5630)	;CJK UNIFIED IDEOGRAPH
                                (#x875D . #x5633)	;CJK UNIFIED IDEOGRAPH
                                (#x875E . #x5635)	;CJK UNIFIED IDEOGRAPH
                                (#x875F . #x5637)	;CJK UNIFIED IDEOGRAPH
                                (#x8760 . #x5638)	;CJK UNIFIED IDEOGRAPH
                                (#x8761 . #x563A)	;CJK UNIFIED IDEOGRAPH
                                (#x8762 . #x563C)	;CJK UNIFIED IDEOGRAPH
                                (#x8763 . #x563D)	;CJK UNIFIED IDEOGRAPH
                                (#x8764 . #x563E)	;CJK UNIFIED IDEOGRAPH
                                (#x8765 . #x5640)	;CJK UNIFIED IDEOGRAPH
                                (#x8766 . #x5641)	;CJK UNIFIED IDEOGRAPH
                                (#x8767 . #x5642)	;CJK UNIFIED IDEOGRAPH
                                (#x8768 . #x5643)	;CJK UNIFIED IDEOGRAPH
                                (#x8769 . #x5644)	;CJK UNIFIED IDEOGRAPH
                                (#x876A . #x5645)	;CJK UNIFIED IDEOGRAPH
                                (#x876B . #x5646)	;CJK UNIFIED IDEOGRAPH
                                (#x876C . #x5647)	;CJK UNIFIED IDEOGRAPH
                                (#x876D . #x5648)	;CJK UNIFIED IDEOGRAPH
                                (#x876E . #x5649)	;CJK UNIFIED IDEOGRAPH
                                (#x876F . #x564A)	;CJK UNIFIED IDEOGRAPH
                                (#x8770 . #x564B)	;CJK UNIFIED IDEOGRAPH
                                (#x8771 . #x564F)	;CJK UNIFIED IDEOGRAPH
                                (#x8772 . #x5650)	;CJK UNIFIED IDEOGRAPH
                                (#x8773 . #x5651)	;CJK UNIFIED IDEOGRAPH
                                (#x8774 . #x5652)	;CJK UNIFIED IDEOGRAPH
                                (#x8775 . #x5653)	;CJK UNIFIED IDEOGRAPH
                                (#x8776 . #x5655)	;CJK UNIFIED IDEOGRAPH
                                (#x8777 . #x5656)	;CJK UNIFIED IDEOGRAPH
                                (#x8778 . #x565A)	;CJK UNIFIED IDEOGRAPH
                                (#x8779 . #x565B)	;CJK UNIFIED IDEOGRAPH
                                (#x877A . #x565D)	;CJK UNIFIED IDEOGRAPH
                                (#x877B . #x565E)	;CJK UNIFIED IDEOGRAPH
                                (#x877C . #x565F)	;CJK UNIFIED IDEOGRAPH
                                (#x877D . #x5660)	;CJK UNIFIED IDEOGRAPH
                                (#x877E . #x5661)	;CJK UNIFIED IDEOGRAPH
                                (#x8780 . #x5663)	;CJK UNIFIED IDEOGRAPH
                                (#x8781 . #x5665)	;CJK UNIFIED IDEOGRAPH
                                (#x8782 . #x5666)	;CJK UNIFIED IDEOGRAPH
                                (#x8783 . #x5667)	;CJK UNIFIED IDEOGRAPH
                                (#x8784 . #x566D)	;CJK UNIFIED IDEOGRAPH
                                (#x8785 . #x566E)	;CJK UNIFIED IDEOGRAPH
                                (#x8786 . #x566F)	;CJK UNIFIED IDEOGRAPH
                                (#x8787 . #x5670)	;CJK UNIFIED IDEOGRAPH
                                (#x8788 . #x5672)	;CJK UNIFIED IDEOGRAPH
                                (#x8789 . #x5673)	;CJK UNIFIED IDEOGRAPH
                                (#x878A . #x5674)	;CJK UNIFIED IDEOGRAPH
                                (#x878B . #x5675)	;CJK UNIFIED IDEOGRAPH
                                (#x878C . #x5677)	;CJK UNIFIED IDEOGRAPH
                                (#x878D . #x5678)	;CJK UNIFIED IDEOGRAPH
                                (#x878E . #x5679)	;CJK UNIFIED IDEOGRAPH
                                (#x878F . #x567A)	;CJK UNIFIED IDEOGRAPH
                                (#x8790 . #x567D)	;CJK UNIFIED IDEOGRAPH
                                (#x8791 . #x567E)	;CJK UNIFIED IDEOGRAPH
                                (#x8792 . #x567F)	;CJK UNIFIED IDEOGRAPH
                                (#x8793 . #x5680)	;CJK UNIFIED IDEOGRAPH
                                (#x8794 . #x5681)	;CJK UNIFIED IDEOGRAPH
                                (#x8795 . #x5682)	;CJK UNIFIED IDEOGRAPH
                                (#x8796 . #x5683)	;CJK UNIFIED IDEOGRAPH
                                (#x8797 . #x5684)	;CJK UNIFIED IDEOGRAPH
                                (#x8798 . #x5687)	;CJK UNIFIED IDEOGRAPH
                                (#x8799 . #x5688)	;CJK UNIFIED IDEOGRAPH
                                (#x879A . #x5689)	;CJK UNIFIED IDEOGRAPH
                                (#x879B . #x568A)	;CJK UNIFIED IDEOGRAPH
                                (#x879C . #x568B)	;CJK UNIFIED IDEOGRAPH
                                (#x879D . #x568C)	;CJK UNIFIED IDEOGRAPH
                                (#x879E . #x568D)	;CJK UNIFIED IDEOGRAPH
                                (#x879F . #x5690)	;CJK UNIFIED IDEOGRAPH
                                (#x87A0 . #x5691)	;CJK UNIFIED IDEOGRAPH
                                (#x87A1 . #x5692)	;CJK UNIFIED IDEOGRAPH
                                (#x87A2 . #x5694)	;CJK UNIFIED IDEOGRAPH
                                (#x87A3 . #x5695)	;CJK UNIFIED IDEOGRAPH
                                (#x87A4 . #x5696)	;CJK UNIFIED IDEOGRAPH
                                (#x87A5 . #x5697)	;CJK UNIFIED IDEOGRAPH
                                (#x87A6 . #x5698)	;CJK UNIFIED IDEOGRAPH
                                (#x87A7 . #x5699)	;CJK UNIFIED IDEOGRAPH
                                (#x87A8 . #x569A)	;CJK UNIFIED IDEOGRAPH
                                (#x87A9 . #x569B)	;CJK UNIFIED IDEOGRAPH
                                (#x87AA . #x569C)	;CJK UNIFIED IDEOGRAPH
                                (#x87AB . #x569D)	;CJK UNIFIED IDEOGRAPH
                                (#x87AC . #x569E)	;CJK UNIFIED IDEOGRAPH
                                (#x87AD . #x569F)	;CJK UNIFIED IDEOGRAPH
                                (#x87AE . #x56A0)	;CJK UNIFIED IDEOGRAPH
                                (#x87AF . #x56A1)	;CJK UNIFIED IDEOGRAPH
                                (#x87B0 . #x56A2)	;CJK UNIFIED IDEOGRAPH
                                (#x87B1 . #x56A4)	;CJK UNIFIED IDEOGRAPH
                                (#x87B2 . #x56A5)	;CJK UNIFIED IDEOGRAPH
                                (#x87B3 . #x56A6)	;CJK UNIFIED IDEOGRAPH
                                (#x87B4 . #x56A7)	;CJK UNIFIED IDEOGRAPH
                                (#x87B5 . #x56A8)	;CJK UNIFIED IDEOGRAPH
                                (#x87B6 . #x56A9)	;CJK UNIFIED IDEOGRAPH
                                (#x87B7 . #x56AA)	;CJK UNIFIED IDEOGRAPH
                                (#x87B8 . #x56AB)	;CJK UNIFIED IDEOGRAPH
                                (#x87B9 . #x56AC)	;CJK UNIFIED IDEOGRAPH
                                (#x87BA . #x56AD)	;CJK UNIFIED IDEOGRAPH
                                (#x87BB . #x56AE)	;CJK UNIFIED IDEOGRAPH
                                (#x87BC . #x56B0)	;CJK UNIFIED IDEOGRAPH
                                (#x87BD . #x56B1)	;CJK UNIFIED IDEOGRAPH
                                (#x87BE . #x56B2)	;CJK UNIFIED IDEOGRAPH
                                (#x87BF . #x56B3)	;CJK UNIFIED IDEOGRAPH
                                (#x87C0 . #x56B4)	;CJK UNIFIED IDEOGRAPH
                                (#x87C1 . #x56B5)	;CJK UNIFIED IDEOGRAPH
                                (#x87C2 . #x56B6)	;CJK UNIFIED IDEOGRAPH
                                (#x87C3 . #x56B8)	;CJK UNIFIED IDEOGRAPH
                                (#x87C4 . #x56B9)	;CJK UNIFIED IDEOGRAPH
                                (#x87C5 . #x56BA)	;CJK UNIFIED IDEOGRAPH
                                (#x87C6 . #x56BB)	;CJK UNIFIED IDEOGRAPH
                                (#x87C7 . #x56BD)	;CJK UNIFIED IDEOGRAPH
                                (#x87C8 . #x56BE)	;CJK UNIFIED IDEOGRAPH
                                (#x87C9 . #x56BF)	;CJK UNIFIED IDEOGRAPH
                                (#x87CA . #x56C0)	;CJK UNIFIED IDEOGRAPH
                                (#x87CB . #x56C1)	;CJK UNIFIED IDEOGRAPH
                                (#x87CC . #x56C2)	;CJK UNIFIED IDEOGRAPH
                                (#x87CD . #x56C3)	;CJK UNIFIED IDEOGRAPH
                                (#x87CE . #x56C4)	;CJK UNIFIED IDEOGRAPH
                                (#x87CF . #x56C5)	;CJK UNIFIED IDEOGRAPH
                                (#x87D0 . #x56C6)	;CJK UNIFIED IDEOGRAPH
                                (#x87D1 . #x56C7)	;CJK UNIFIED IDEOGRAPH
                                (#x87D2 . #x56C8)	;CJK UNIFIED IDEOGRAPH
                                (#x87D3 . #x56C9)	;CJK UNIFIED IDEOGRAPH
                                (#x87D4 . #x56CB)	;CJK UNIFIED IDEOGRAPH
                                (#x87D5 . #x56CC)	;CJK UNIFIED IDEOGRAPH
                                (#x87D6 . #x56CD)	;CJK UNIFIED IDEOGRAPH
                                (#x87D7 . #x56CE)	;CJK UNIFIED IDEOGRAPH
                                (#x87D8 . #x56CF)	;CJK UNIFIED IDEOGRAPH
                                (#x87D9 . #x56D0)	;CJK UNIFIED IDEOGRAPH
                                (#x87DA . #x56D1)	;CJK UNIFIED IDEOGRAPH
                                (#x87DB . #x56D2)	;CJK UNIFIED IDEOGRAPH
                                (#x87DC . #x56D3)	;CJK UNIFIED IDEOGRAPH
                                (#x87DD . #x56D5)	;CJK UNIFIED IDEOGRAPH
                                (#x87DE . #x56D6)	;CJK UNIFIED IDEOGRAPH
                                (#x87DF . #x56D8)	;CJK UNIFIED IDEOGRAPH
                                (#x87E0 . #x56D9)	;CJK UNIFIED IDEOGRAPH
                                (#x87E1 . #x56DC)	;CJK UNIFIED IDEOGRAPH
                                (#x87E2 . #x56E3)	;CJK UNIFIED IDEOGRAPH
                                (#x87E3 . #x56E5)	;CJK UNIFIED IDEOGRAPH
                                (#x87E4 . #x56E6)	;CJK UNIFIED IDEOGRAPH
                                (#x87E5 . #x56E7)	;CJK UNIFIED IDEOGRAPH
                                (#x87E6 . #x56E8)	;CJK UNIFIED IDEOGRAPH
                                (#x87E7 . #x56E9)	;CJK UNIFIED IDEOGRAPH
                                (#x87E8 . #x56EA)	;CJK UNIFIED IDEOGRAPH
                                (#x87E9 . #x56EC)	;CJK UNIFIED IDEOGRAPH
                                (#x87EA . #x56EE)	;CJK UNIFIED IDEOGRAPH
                                (#x87EB . #x56EF)	;CJK UNIFIED IDEOGRAPH
                                (#x87EC . #x56F2)	;CJK UNIFIED IDEOGRAPH
                                (#x87ED . #x56F3)	;CJK UNIFIED IDEOGRAPH
                                (#x87EE . #x56F6)	;CJK UNIFIED IDEOGRAPH
                                (#x87EF . #x56F7)	;CJK UNIFIED IDEOGRAPH
                                (#x87F0 . #x56F8)	;CJK UNIFIED IDEOGRAPH
                                (#x87F1 . #x56FB)	;CJK UNIFIED IDEOGRAPH
                                (#x87F2 . #x56FC)	;CJK UNIFIED IDEOGRAPH
                                (#x87F3 . #x5700)	;CJK UNIFIED IDEOGRAPH
                                (#x87F4 . #x5701)	;CJK UNIFIED IDEOGRAPH
                                (#x87F5 . #x5702)	;CJK UNIFIED IDEOGRAPH
                                (#x87F6 . #x5705)	;CJK UNIFIED IDEOGRAPH
                                (#x87F7 . #x5707)	;CJK UNIFIED IDEOGRAPH
                                (#x87F8 . #x570B)	;CJK UNIFIED IDEOGRAPH
                                (#x87F9 . #x570C)	;CJK UNIFIED IDEOGRAPH
                                (#x87FA . #x570D)	;CJK UNIFIED IDEOGRAPH
                                (#x87FB . #x570E)	;CJK UNIFIED IDEOGRAPH
                                (#x87FC . #x570F)	;CJK UNIFIED IDEOGRAPH
                                (#x87FD . #x5710)	;CJK UNIFIED IDEOGRAPH
                                (#x87FE . #x5711)	;CJK UNIFIED IDEOGRAPH
                                (#x8840 . #x5712)	;CJK UNIFIED IDEOGRAPH
                                (#x8841 . #x5713)	;CJK UNIFIED IDEOGRAPH
                                (#x8842 . #x5714)	;CJK UNIFIED IDEOGRAPH
                                (#x8843 . #x5715)	;CJK UNIFIED IDEOGRAPH
                                (#x8844 . #x5716)	;CJK UNIFIED IDEOGRAPH
                                (#x8845 . #x5717)	;CJK UNIFIED IDEOGRAPH
                                (#x8846 . #x5718)	;CJK UNIFIED IDEOGRAPH
                                (#x8847 . #x5719)	;CJK UNIFIED IDEOGRAPH
                                (#x8848 . #x571A)	;CJK UNIFIED IDEOGRAPH
                                (#x8849 . #x571B)	;CJK UNIFIED IDEOGRAPH
                                (#x884A . #x571D)	;CJK UNIFIED IDEOGRAPH
                                (#x884B . #x571E)	;CJK UNIFIED IDEOGRAPH
                                (#x884C . #x5720)	;CJK UNIFIED IDEOGRAPH
                                (#x884D . #x5721)	;CJK UNIFIED IDEOGRAPH
                                (#x884E . #x5722)	;CJK UNIFIED IDEOGRAPH
                                (#x884F . #x5724)	;CJK UNIFIED IDEOGRAPH
                                (#x8850 . #x5725)	;CJK UNIFIED IDEOGRAPH
                                (#x8851 . #x5726)	;CJK UNIFIED IDEOGRAPH
                                (#x8852 . #x5727)	;CJK UNIFIED IDEOGRAPH
                                (#x8853 . #x572B)	;CJK UNIFIED IDEOGRAPH
                                (#x8854 . #x5731)	;CJK UNIFIED IDEOGRAPH
                                (#x8855 . #x5732)	;CJK UNIFIED IDEOGRAPH
                                (#x8856 . #x5734)	;CJK UNIFIED IDEOGRAPH
                                (#x8857 . #x5735)	;CJK UNIFIED IDEOGRAPH
                                (#x8858 . #x5736)	;CJK UNIFIED IDEOGRAPH
                                (#x8859 . #x5737)	;CJK UNIFIED IDEOGRAPH
                                (#x885A . #x5738)	;CJK UNIFIED IDEOGRAPH
                                (#x885B . #x573C)	;CJK UNIFIED IDEOGRAPH
                                (#x885C . #x573D)	;CJK UNIFIED IDEOGRAPH
                                (#x885D . #x573F)	;CJK UNIFIED IDEOGRAPH
                                (#x885E . #x5741)	;CJK UNIFIED IDEOGRAPH
                                (#x885F . #x5743)	;CJK UNIFIED IDEOGRAPH
                                (#x8860 . #x5744)	;CJK UNIFIED IDEOGRAPH
                                (#x8861 . #x5745)	;CJK UNIFIED IDEOGRAPH
                                (#x8862 . #x5746)	;CJK UNIFIED IDEOGRAPH
                                (#x8863 . #x5748)	;CJK UNIFIED IDEOGRAPH
                                (#x8864 . #x5749)	;CJK UNIFIED IDEOGRAPH
                                (#x8865 . #x574B)	;CJK UNIFIED IDEOGRAPH
                                (#x8866 . #x5752)	;CJK UNIFIED IDEOGRAPH
                                (#x8867 . #x5753)	;CJK UNIFIED IDEOGRAPH
                                (#x8868 . #x5754)	;CJK UNIFIED IDEOGRAPH
                                (#x8869 . #x5755)	;CJK UNIFIED IDEOGRAPH
                                (#x886A . #x5756)	;CJK UNIFIED IDEOGRAPH
                                (#x886B . #x5758)	;CJK UNIFIED IDEOGRAPH
                                (#x886C . #x5759)	;CJK UNIFIED IDEOGRAPH
                                (#x886D . #x5762)	;CJK UNIFIED IDEOGRAPH
                                (#x886E . #x5763)	;CJK UNIFIED IDEOGRAPH
                                (#x886F . #x5765)	;CJK UNIFIED IDEOGRAPH
                                (#x8870 . #x5767)	;CJK UNIFIED IDEOGRAPH
                                (#x8871 . #x576C)	;CJK UNIFIED IDEOGRAPH
                                (#x8872 . #x576E)	;CJK UNIFIED IDEOGRAPH
                                (#x8873 . #x5770)	;CJK UNIFIED IDEOGRAPH
                                (#x8874 . #x5771)	;CJK UNIFIED IDEOGRAPH
                                (#x8875 . #x5772)	;CJK UNIFIED IDEOGRAPH
                                (#x8876 . #x5774)	;CJK UNIFIED IDEOGRAPH
                                (#x8877 . #x5775)	;CJK UNIFIED IDEOGRAPH
                                (#x8878 . #x5778)	;CJK UNIFIED IDEOGRAPH
                                (#x8879 . #x5779)	;CJK UNIFIED IDEOGRAPH
                                (#x887A . #x577A)	;CJK UNIFIED IDEOGRAPH
                                (#x887B . #x577D)	;CJK UNIFIED IDEOGRAPH
                                (#x887C . #x577E)	;CJK UNIFIED IDEOGRAPH
                                (#x887D . #x577F)	;CJK UNIFIED IDEOGRAPH
                                (#x887E . #x5780)	;CJK UNIFIED IDEOGRAPH
                                (#x8880 . #x5781)	;CJK UNIFIED IDEOGRAPH
                                (#x8881 . #x5787)	;CJK UNIFIED IDEOGRAPH
                                (#x8882 . #x5788)	;CJK UNIFIED IDEOGRAPH
                                (#x8883 . #x5789)	;CJK UNIFIED IDEOGRAPH
                                (#x8884 . #x578A)	;CJK UNIFIED IDEOGRAPH
                                (#x8885 . #x578D)	;CJK UNIFIED IDEOGRAPH
                                (#x8886 . #x578E)	;CJK UNIFIED IDEOGRAPH
                                (#x8887 . #x578F)	;CJK UNIFIED IDEOGRAPH
                                (#x8888 . #x5790)	;CJK UNIFIED IDEOGRAPH
                                (#x8889 . #x5791)	;CJK UNIFIED IDEOGRAPH
                                (#x888A . #x5794)	;CJK UNIFIED IDEOGRAPH
                                (#x888B . #x5795)	;CJK UNIFIED IDEOGRAPH
                                (#x888C . #x5796)	;CJK UNIFIED IDEOGRAPH
                                (#x888D . #x5797)	;CJK UNIFIED IDEOGRAPH
                                (#x888E . #x5798)	;CJK UNIFIED IDEOGRAPH
                                (#x888F . #x5799)	;CJK UNIFIED IDEOGRAPH
                                (#x8890 . #x579A)	;CJK UNIFIED IDEOGRAPH
                                (#x8891 . #x579C)	;CJK UNIFIED IDEOGRAPH
                                (#x8892 . #x579D)	;CJK UNIFIED IDEOGRAPH
                                (#x8893 . #x579E)	;CJK UNIFIED IDEOGRAPH
                                (#x8894 . #x579F)	;CJK UNIFIED IDEOGRAPH
                                (#x8895 . #x57A5)	;CJK UNIFIED IDEOGRAPH
                                (#x8896 . #x57A8)	;CJK UNIFIED IDEOGRAPH
                                (#x8897 . #x57AA)	;CJK UNIFIED IDEOGRAPH
                                (#x8898 . #x57AC)	;CJK UNIFIED IDEOGRAPH
                                (#x8899 . #x57AF)	;CJK UNIFIED IDEOGRAPH
                                (#x889A . #x57B0)	;CJK UNIFIED IDEOGRAPH
                                (#x889B . #x57B1)	;CJK UNIFIED IDEOGRAPH
                                (#x889C . #x57B3)	;CJK UNIFIED IDEOGRAPH
                                (#x889D . #x57B5)	;CJK UNIFIED IDEOGRAPH
                                (#x889E . #x57B6)	;CJK UNIFIED IDEOGRAPH
                                (#x889F . #x57B7)	;CJK UNIFIED IDEOGRAPH
                                (#x88A0 . #x57B9)	;CJK UNIFIED IDEOGRAPH
                                (#x88A1 . #x57BA)	;CJK UNIFIED IDEOGRAPH
                                (#x88A2 . #x57BB)	;CJK UNIFIED IDEOGRAPH
                                (#x88A3 . #x57BC)	;CJK UNIFIED IDEOGRAPH
                                (#x88A4 . #x57BD)	;CJK UNIFIED IDEOGRAPH
                                (#x88A5 . #x57BE)	;CJK UNIFIED IDEOGRAPH
                                (#x88A6 . #x57BF)	;CJK UNIFIED IDEOGRAPH
                                (#x88A7 . #x57C0)	;CJK UNIFIED IDEOGRAPH
                                (#x88A8 . #x57C1)	;CJK UNIFIED IDEOGRAPH
                                (#x88A9 . #x57C4)	;CJK UNIFIED IDEOGRAPH
                                (#x88AA . #x57C5)	;CJK UNIFIED IDEOGRAPH
                                (#x88AB . #x57C6)	;CJK UNIFIED IDEOGRAPH
                                (#x88AC . #x57C7)	;CJK UNIFIED IDEOGRAPH
                                (#x88AD . #x57C8)	;CJK UNIFIED IDEOGRAPH
                                (#x88AE . #x57C9)	;CJK UNIFIED IDEOGRAPH
                                (#x88AF . #x57CA)	;CJK UNIFIED IDEOGRAPH
                                (#x88B0 . #x57CC)	;CJK UNIFIED IDEOGRAPH
                                (#x88B1 . #x57CD)	;CJK UNIFIED IDEOGRAPH
                                (#x88B2 . #x57D0)	;CJK UNIFIED IDEOGRAPH
                                (#x88B3 . #x57D1)	;CJK UNIFIED IDEOGRAPH
                                (#x88B4 . #x57D3)	;CJK UNIFIED IDEOGRAPH
                                (#x88B5 . #x57D6)	;CJK UNIFIED IDEOGRAPH
                                (#x88B6 . #x57D7)	;CJK UNIFIED IDEOGRAPH
                                (#x88B7 . #x57DB)	;CJK UNIFIED IDEOGRAPH
                                (#x88B8 . #x57DC)	;CJK UNIFIED IDEOGRAPH
                                (#x88B9 . #x57DE)	;CJK UNIFIED IDEOGRAPH
                                (#x88BA . #x57E1)	;CJK UNIFIED IDEOGRAPH
                                (#x88BB . #x57E2)	;CJK UNIFIED IDEOGRAPH
                                (#x88BC . #x57E3)	;CJK UNIFIED IDEOGRAPH
                                (#x88BD . #x57E5)	;CJK UNIFIED IDEOGRAPH
                                (#x88BE . #x57E6)	;CJK UNIFIED IDEOGRAPH
                                (#x88BF . #x57E7)	;CJK UNIFIED IDEOGRAPH
                                (#x88C0 . #x57E8)	;CJK UNIFIED IDEOGRAPH
                                (#x88C1 . #x57E9)	;CJK UNIFIED IDEOGRAPH
                                (#x88C2 . #x57EA)	;CJK UNIFIED IDEOGRAPH
                                (#x88C3 . #x57EB)	;CJK UNIFIED IDEOGRAPH
                                (#x88C4 . #x57EC)	;CJK UNIFIED IDEOGRAPH
                                (#x88C5 . #x57EE)	;CJK UNIFIED IDEOGRAPH
                                (#x88C6 . #x57F0)	;CJK UNIFIED IDEOGRAPH
                                (#x88C7 . #x57F1)	;CJK UNIFIED IDEOGRAPH
                                (#x88C8 . #x57F2)	;CJK UNIFIED IDEOGRAPH
                                (#x88C9 . #x57F3)	;CJK UNIFIED IDEOGRAPH
                                (#x88CA . #x57F5)	;CJK UNIFIED IDEOGRAPH
                                (#x88CB . #x57F6)	;CJK UNIFIED IDEOGRAPH
                                (#x88CC . #x57F7)	;CJK UNIFIED IDEOGRAPH
                                (#x88CD . #x57FB)	;CJK UNIFIED IDEOGRAPH
                                (#x88CE . #x57FC)	;CJK UNIFIED IDEOGRAPH
                                (#x88CF . #x57FE)	;CJK UNIFIED IDEOGRAPH
                                (#x88D0 . #x57FF)	;CJK UNIFIED IDEOGRAPH
                                (#x88D1 . #x5801)	;CJK UNIFIED IDEOGRAPH
                                (#x88D2 . #x5803)	;CJK UNIFIED IDEOGRAPH
                                (#x88D3 . #x5804)	;CJK UNIFIED IDEOGRAPH
                                (#x88D4 . #x5805)	;CJK UNIFIED IDEOGRAPH
                                (#x88D5 . #x5808)	;CJK UNIFIED IDEOGRAPH
                                (#x88D6 . #x5809)	;CJK UNIFIED IDEOGRAPH
                                (#x88D7 . #x580A)	;CJK UNIFIED IDEOGRAPH
                                (#x88D8 . #x580C)	;CJK UNIFIED IDEOGRAPH
                                (#x88D9 . #x580E)	;CJK UNIFIED IDEOGRAPH
                                (#x88DA . #x580F)	;CJK UNIFIED IDEOGRAPH
                                (#x88DB . #x5810)	;CJK UNIFIED IDEOGRAPH
                                (#x88DC . #x5812)	;CJK UNIFIED IDEOGRAPH
                                (#x88DD . #x5813)	;CJK UNIFIED IDEOGRAPH
                                (#x88DE . #x5814)	;CJK UNIFIED IDEOGRAPH
                                (#x88DF . #x5816)	;CJK UNIFIED IDEOGRAPH
                                (#x88E0 . #x5817)	;CJK UNIFIED IDEOGRAPH
                                (#x88E1 . #x5818)	;CJK UNIFIED IDEOGRAPH
                                (#x88E2 . #x581A)	;CJK UNIFIED IDEOGRAPH
                                (#x88E3 . #x581B)	;CJK UNIFIED IDEOGRAPH
                                (#x88E4 . #x581C)	;CJK UNIFIED IDEOGRAPH
                                (#x88E5 . #x581D)	;CJK UNIFIED IDEOGRAPH
                                (#x88E6 . #x581F)	;CJK UNIFIED IDEOGRAPH
                                (#x88E7 . #x5822)	;CJK UNIFIED IDEOGRAPH
                                (#x88E8 . #x5823)	;CJK UNIFIED IDEOGRAPH
                                (#x88E9 . #x5825)	;CJK UNIFIED IDEOGRAPH
                                (#x88EA . #x5826)	;CJK UNIFIED IDEOGRAPH
                                (#x88EB . #x5827)	;CJK UNIFIED IDEOGRAPH
                                (#x88EC . #x5828)	;CJK UNIFIED IDEOGRAPH
                                (#x88ED . #x5829)	;CJK UNIFIED IDEOGRAPH
                                (#x88EE . #x582B)	;CJK UNIFIED IDEOGRAPH
                                (#x88EF . #x582C)	;CJK UNIFIED IDEOGRAPH
                                (#x88F0 . #x582D)	;CJK UNIFIED IDEOGRAPH
                                (#x88F1 . #x582E)	;CJK UNIFIED IDEOGRAPH
                                (#x88F2 . #x582F)	;CJK UNIFIED IDEOGRAPH
                                (#x88F3 . #x5831)	;CJK UNIFIED IDEOGRAPH
                                (#x88F4 . #x5832)	;CJK UNIFIED IDEOGRAPH
                                (#x88F5 . #x5833)	;CJK UNIFIED IDEOGRAPH
                                (#x88F6 . #x5834)	;CJK UNIFIED IDEOGRAPH
                                (#x88F7 . #x5836)	;CJK UNIFIED IDEOGRAPH
                                (#x88F8 . #x5837)	;CJK UNIFIED IDEOGRAPH
                                (#x88F9 . #x5838)	;CJK UNIFIED IDEOGRAPH
                                (#x88FA . #x5839)	;CJK UNIFIED IDEOGRAPH
                                (#x88FB . #x583A)	;CJK UNIFIED IDEOGRAPH
                                (#x88FC . #x583B)	;CJK UNIFIED IDEOGRAPH
                                (#x88FD . #x583C)	;CJK UNIFIED IDEOGRAPH
                                (#x88FE . #x583D)	;CJK UNIFIED IDEOGRAPH
                                (#x8940 . #x583E)	;CJK UNIFIED IDEOGRAPH
                                (#x8941 . #x583F)	;CJK UNIFIED IDEOGRAPH
                                (#x8942 . #x5840)	;CJK UNIFIED IDEOGRAPH
                                (#x8943 . #x5841)	;CJK UNIFIED IDEOGRAPH
                                (#x8944 . #x5842)	;CJK UNIFIED IDEOGRAPH
                                (#x8945 . #x5843)	;CJK UNIFIED IDEOGRAPH
                                (#x8946 . #x5845)	;CJK UNIFIED IDEOGRAPH
                                (#x8947 . #x5846)	;CJK UNIFIED IDEOGRAPH
                                (#x8948 . #x5847)	;CJK UNIFIED IDEOGRAPH
                                (#x8949 . #x5848)	;CJK UNIFIED IDEOGRAPH
                                (#x894A . #x5849)	;CJK UNIFIED IDEOGRAPH
                                (#x894B . #x584A)	;CJK UNIFIED IDEOGRAPH
                                (#x894C . #x584B)	;CJK UNIFIED IDEOGRAPH
                                (#x894D . #x584E)	;CJK UNIFIED IDEOGRAPH
                                (#x894E . #x584F)	;CJK UNIFIED IDEOGRAPH
                                (#x894F . #x5850)	;CJK UNIFIED IDEOGRAPH
                                (#x8950 . #x5852)	;CJK UNIFIED IDEOGRAPH
                                (#x8951 . #x5853)	;CJK UNIFIED IDEOGRAPH
                                (#x8952 . #x5855)	;CJK UNIFIED IDEOGRAPH
                                (#x8953 . #x5856)	;CJK UNIFIED IDEOGRAPH
                                (#x8954 . #x5857)	;CJK UNIFIED IDEOGRAPH
                                (#x8955 . #x5859)	;CJK UNIFIED IDEOGRAPH
                                (#x8956 . #x585A)	;CJK UNIFIED IDEOGRAPH
                                (#x8957 . #x585B)	;CJK UNIFIED IDEOGRAPH
                                (#x8958 . #x585C)	;CJK UNIFIED IDEOGRAPH
                                (#x8959 . #x585D)	;CJK UNIFIED IDEOGRAPH
                                (#x895A . #x585F)	;CJK UNIFIED IDEOGRAPH
                                (#x895B . #x5860)	;CJK UNIFIED IDEOGRAPH
                                (#x895C . #x5861)	;CJK UNIFIED IDEOGRAPH
                                (#x895D . #x5862)	;CJK UNIFIED IDEOGRAPH
                                (#x895E . #x5863)	;CJK UNIFIED IDEOGRAPH
                                (#x895F . #x5864)	;CJK UNIFIED IDEOGRAPH
                                (#x8960 . #x5866)	;CJK UNIFIED IDEOGRAPH
                                (#x8961 . #x5867)	;CJK UNIFIED IDEOGRAPH
                                (#x8962 . #x5868)	;CJK UNIFIED IDEOGRAPH
                                (#x8963 . #x5869)	;CJK UNIFIED IDEOGRAPH
                                (#x8964 . #x586A)	;CJK UNIFIED IDEOGRAPH
                                (#x8965 . #x586D)	;CJK UNIFIED IDEOGRAPH
                                (#x8966 . #x586E)	;CJK UNIFIED IDEOGRAPH
                                (#x8967 . #x586F)	;CJK UNIFIED IDEOGRAPH
                                (#x8968 . #x5870)	;CJK UNIFIED IDEOGRAPH
                                (#x8969 . #x5871)	;CJK UNIFIED IDEOGRAPH
                                (#x896A . #x5872)	;CJK UNIFIED IDEOGRAPH
                                (#x896B . #x5873)	;CJK UNIFIED IDEOGRAPH
                                (#x896C . #x5874)	;CJK UNIFIED IDEOGRAPH
                                (#x896D . #x5875)	;CJK UNIFIED IDEOGRAPH
                                (#x896E . #x5876)	;CJK UNIFIED IDEOGRAPH
                                (#x896F . #x5877)	;CJK UNIFIED IDEOGRAPH
                                (#x8970 . #x5878)	;CJK UNIFIED IDEOGRAPH
                                (#x8971 . #x5879)	;CJK UNIFIED IDEOGRAPH
                                (#x8972 . #x587A)	;CJK UNIFIED IDEOGRAPH
                                (#x8973 . #x587B)	;CJK UNIFIED IDEOGRAPH
                                (#x8974 . #x587C)	;CJK UNIFIED IDEOGRAPH
                                (#x8975 . #x587D)	;CJK UNIFIED IDEOGRAPH
                                (#x8976 . #x587F)	;CJK UNIFIED IDEOGRAPH
                                (#x8977 . #x5882)	;CJK UNIFIED IDEOGRAPH
                                (#x8978 . #x5884)	;CJK UNIFIED IDEOGRAPH
                                (#x8979 . #x5886)	;CJK UNIFIED IDEOGRAPH
                                (#x897A . #x5887)	;CJK UNIFIED IDEOGRAPH
                                (#x897B . #x5888)	;CJK UNIFIED IDEOGRAPH
                                (#x897C . #x588A)	;CJK UNIFIED IDEOGRAPH
                                (#x897D . #x588B)	;CJK UNIFIED IDEOGRAPH
                                (#x897E . #x588C)	;CJK UNIFIED IDEOGRAPH
                                (#x8980 . #x588D)	;CJK UNIFIED IDEOGRAPH
                                (#x8981 . #x588E)	;CJK UNIFIED IDEOGRAPH
                                (#x8982 . #x588F)	;CJK UNIFIED IDEOGRAPH
                                (#x8983 . #x5890)	;CJK UNIFIED IDEOGRAPH
                                (#x8984 . #x5891)	;CJK UNIFIED IDEOGRAPH
                                (#x8985 . #x5894)	;CJK UNIFIED IDEOGRAPH
                                (#x8986 . #x5895)	;CJK UNIFIED IDEOGRAPH
                                (#x8987 . #x5896)	;CJK UNIFIED IDEOGRAPH
                                (#x8988 . #x5897)	;CJK UNIFIED IDEOGRAPH
                                (#x8989 . #x5898)	;CJK UNIFIED IDEOGRAPH
                                (#x898A . #x589B)	;CJK UNIFIED IDEOGRAPH
                                (#x898B . #x589C)	;CJK UNIFIED IDEOGRAPH
                                (#x898C . #x589D)	;CJK UNIFIED IDEOGRAPH
                                (#x898D . #x58A0)	;CJK UNIFIED IDEOGRAPH
                                (#x898E . #x58A1)	;CJK UNIFIED IDEOGRAPH
                                (#x898F . #x58A2)	;CJK UNIFIED IDEOGRAPH
                                (#x8990 . #x58A3)	;CJK UNIFIED IDEOGRAPH
                                (#x8991 . #x58A4)	;CJK UNIFIED IDEOGRAPH
                                (#x8992 . #x58A5)	;CJK UNIFIED IDEOGRAPH
                                (#x8993 . #x58A6)	;CJK UNIFIED IDEOGRAPH
                                (#x8994 . #x58A7)	;CJK UNIFIED IDEOGRAPH
                                (#x8995 . #x58AA)	;CJK UNIFIED IDEOGRAPH
                                (#x8996 . #x58AB)	;CJK UNIFIED IDEOGRAPH
                                (#x8997 . #x58AC)	;CJK UNIFIED IDEOGRAPH
                                (#x8998 . #x58AD)	;CJK UNIFIED IDEOGRAPH
                                (#x8999 . #x58AE)	;CJK UNIFIED IDEOGRAPH
                                (#x899A . #x58AF)	;CJK UNIFIED IDEOGRAPH
                                (#x899B . #x58B0)	;CJK UNIFIED IDEOGRAPH
                                (#x899C . #x58B1)	;CJK UNIFIED IDEOGRAPH
                                (#x899D . #x58B2)	;CJK UNIFIED IDEOGRAPH
                                (#x899E . #x58B3)	;CJK UNIFIED IDEOGRAPH
                                (#x899F . #x58B4)	;CJK UNIFIED IDEOGRAPH
                                (#x89A0 . #x58B5)	;CJK UNIFIED IDEOGRAPH
                                (#x89A1 . #x58B6)	;CJK UNIFIED IDEOGRAPH
                                (#x89A2 . #x58B7)	;CJK UNIFIED IDEOGRAPH
                                (#x89A3 . #x58B8)	;CJK UNIFIED IDEOGRAPH
                                (#x89A4 . #x58B9)	;CJK UNIFIED IDEOGRAPH
                                (#x89A5 . #x58BA)	;CJK UNIFIED IDEOGRAPH
                                (#x89A6 . #x58BB)	;CJK UNIFIED IDEOGRAPH
                                (#x89A7 . #x58BD)	;CJK UNIFIED IDEOGRAPH
                                (#x89A8 . #x58BE)	;CJK UNIFIED IDEOGRAPH
                                (#x89A9 . #x58BF)	;CJK UNIFIED IDEOGRAPH
                                (#x89AA . #x58C0)	;CJK UNIFIED IDEOGRAPH
                                (#x89AB . #x58C2)	;CJK UNIFIED IDEOGRAPH
                                (#x89AC . #x58C3)	;CJK UNIFIED IDEOGRAPH
                                (#x89AD . #x58C4)	;CJK UNIFIED IDEOGRAPH
                                (#x89AE . #x58C6)	;CJK UNIFIED IDEOGRAPH
                                (#x89AF . #x58C7)	;CJK UNIFIED IDEOGRAPH
                                (#x89B0 . #x58C8)	;CJK UNIFIED IDEOGRAPH
                                (#x89B1 . #x58C9)	;CJK UNIFIED IDEOGRAPH
                                (#x89B2 . #x58CA)	;CJK UNIFIED IDEOGRAPH
                                (#x89B3 . #x58CB)	;CJK UNIFIED IDEOGRAPH
                                (#x89B4 . #x58CC)	;CJK UNIFIED IDEOGRAPH
                                (#x89B5 . #x58CD)	;CJK UNIFIED IDEOGRAPH
                                (#x89B6 . #x58CE)	;CJK UNIFIED IDEOGRAPH
                                (#x89B7 . #x58CF)	;CJK UNIFIED IDEOGRAPH
                                (#x89B8 . #x58D0)	;CJK UNIFIED IDEOGRAPH
                                (#x89B9 . #x58D2)	;CJK UNIFIED IDEOGRAPH
                                (#x89BA . #x58D3)	;CJK UNIFIED IDEOGRAPH
                                (#x89BB . #x58D4)	;CJK UNIFIED IDEOGRAPH
                                (#x89BC . #x58D6)	;CJK UNIFIED IDEOGRAPH
                                (#x89BD . #x58D7)	;CJK UNIFIED IDEOGRAPH
                                (#x89BE . #x58D8)	;CJK UNIFIED IDEOGRAPH
                                (#x89BF . #x58D9)	;CJK UNIFIED IDEOGRAPH
                                (#x89C0 . #x58DA)	;CJK UNIFIED IDEOGRAPH
                                (#x89C1 . #x58DB)	;CJK UNIFIED IDEOGRAPH
                                (#x89C2 . #x58DC)	;CJK UNIFIED IDEOGRAPH
                                (#x89C3 . #x58DD)	;CJK UNIFIED IDEOGRAPH
                                (#x89C4 . #x58DE)	;CJK UNIFIED IDEOGRAPH
                                (#x89C5 . #x58DF)	;CJK UNIFIED IDEOGRAPH
                                (#x89C6 . #x58E0)	;CJK UNIFIED IDEOGRAPH
                                (#x89C7 . #x58E1)	;CJK UNIFIED IDEOGRAPH
                                (#x89C8 . #x58E2)	;CJK UNIFIED IDEOGRAPH
                                (#x89C9 . #x58E3)	;CJK UNIFIED IDEOGRAPH
                                (#x89CA . #x58E5)	;CJK UNIFIED IDEOGRAPH
                                (#x89CB . #x58E6)	;CJK UNIFIED IDEOGRAPH
                                (#x89CC . #x58E7)	;CJK UNIFIED IDEOGRAPH
                                (#x89CD . #x58E8)	;CJK UNIFIED IDEOGRAPH
                                (#x89CE . #x58E9)	;CJK UNIFIED IDEOGRAPH
                                (#x89CF . #x58EA)	;CJK UNIFIED IDEOGRAPH
                                (#x89D0 . #x58ED)	;CJK UNIFIED IDEOGRAPH
                                (#x89D1 . #x58EF)	;CJK UNIFIED IDEOGRAPH
                                (#x89D2 . #x58F1)	;CJK UNIFIED IDEOGRAPH
                                (#x89D3 . #x58F2)	;CJK UNIFIED IDEOGRAPH
                                (#x89D4 . #x58F4)	;CJK UNIFIED IDEOGRAPH
                                (#x89D5 . #x58F5)	;CJK UNIFIED IDEOGRAPH
                                (#x89D6 . #x58F7)	;CJK UNIFIED IDEOGRAPH
                                (#x89D7 . #x58F8)	;CJK UNIFIED IDEOGRAPH
                                (#x89D8 . #x58FA)	;CJK UNIFIED IDEOGRAPH
                                (#x89D9 . #x58FB)	;CJK UNIFIED IDEOGRAPH
                                (#x89DA . #x58FC)	;CJK UNIFIED IDEOGRAPH
                                (#x89DB . #x58FD)	;CJK UNIFIED IDEOGRAPH
                                (#x89DC . #x58FE)	;CJK UNIFIED IDEOGRAPH
                                (#x89DD . #x58FF)	;CJK UNIFIED IDEOGRAPH
                                (#x89DE . #x5900)	;CJK UNIFIED IDEOGRAPH
                                (#x89DF . #x5901)	;CJK UNIFIED IDEOGRAPH
                                (#x89E0 . #x5903)	;CJK UNIFIED IDEOGRAPH
                                (#x89E1 . #x5905)	;CJK UNIFIED IDEOGRAPH
                                (#x89E2 . #x5906)	;CJK UNIFIED IDEOGRAPH
                                (#x89E3 . #x5908)	;CJK UNIFIED IDEOGRAPH
                                (#x89E4 . #x5909)	;CJK UNIFIED IDEOGRAPH
                                (#x89E5 . #x590A)	;CJK UNIFIED IDEOGRAPH
                                (#x89E6 . #x590B)	;CJK UNIFIED IDEOGRAPH
                                (#x89E7 . #x590C)	;CJK UNIFIED IDEOGRAPH
                                (#x89E8 . #x590E)	;CJK UNIFIED IDEOGRAPH
                                (#x89E9 . #x5910)	;CJK UNIFIED IDEOGRAPH
                                (#x89EA . #x5911)	;CJK UNIFIED IDEOGRAPH
                                (#x89EB . #x5912)	;CJK UNIFIED IDEOGRAPH
                                (#x89EC . #x5913)	;CJK UNIFIED IDEOGRAPH
                                (#x89ED . #x5917)	;CJK UNIFIED IDEOGRAPH
                                (#x89EE . #x5918)	;CJK UNIFIED IDEOGRAPH
                                (#x89EF . #x591B)	;CJK UNIFIED IDEOGRAPH
                                (#x89F0 . #x591D)	;CJK UNIFIED IDEOGRAPH
                                (#x89F1 . #x591E)	;CJK UNIFIED IDEOGRAPH
                                (#x89F2 . #x5920)	;CJK UNIFIED IDEOGRAPH
                                (#x89F3 . #x5921)	;CJK UNIFIED IDEOGRAPH
                                (#x89F4 . #x5922)	;CJK UNIFIED IDEOGRAPH
                                (#x89F5 . #x5923)	;CJK UNIFIED IDEOGRAPH
                                (#x89F6 . #x5926)	;CJK UNIFIED IDEOGRAPH
                                (#x89F7 . #x5928)	;CJK UNIFIED IDEOGRAPH
                                (#x89F8 . #x592C)	;CJK UNIFIED IDEOGRAPH
                                (#x89F9 . #x5930)	;CJK UNIFIED IDEOGRAPH
                                (#x89FA . #x5932)	;CJK UNIFIED IDEOGRAPH
                                (#x89FB . #x5933)	;CJK UNIFIED IDEOGRAPH
                                (#x89FC . #x5935)	;CJK UNIFIED IDEOGRAPH
                                (#x89FD . #x5936)	;CJK UNIFIED IDEOGRAPH
                                (#x89FE . #x593B)	;CJK UNIFIED IDEOGRAPH
                                (#x8A40 . #x593D)	;CJK UNIFIED IDEOGRAPH
                                (#x8A41 . #x593E)	;CJK UNIFIED IDEOGRAPH
                                (#x8A42 . #x593F)	;CJK UNIFIED IDEOGRAPH
                                (#x8A43 . #x5940)	;CJK UNIFIED IDEOGRAPH
                                (#x8A44 . #x5943)	;CJK UNIFIED IDEOGRAPH
                                (#x8A45 . #x5945)	;CJK UNIFIED IDEOGRAPH
                                (#x8A46 . #x5946)	;CJK UNIFIED IDEOGRAPH
                                (#x8A47 . #x594A)	;CJK UNIFIED IDEOGRAPH
                                (#x8A48 . #x594C)	;CJK UNIFIED IDEOGRAPH
                                (#x8A49 . #x594D)	;CJK UNIFIED IDEOGRAPH
                                (#x8A4A . #x5950)	;CJK UNIFIED IDEOGRAPH
                                (#x8A4B . #x5952)	;CJK UNIFIED IDEOGRAPH
                                (#x8A4C . #x5953)	;CJK UNIFIED IDEOGRAPH
                                (#x8A4D . #x5959)	;CJK UNIFIED IDEOGRAPH
                                (#x8A4E . #x595B)	;CJK UNIFIED IDEOGRAPH
                                (#x8A4F . #x595C)	;CJK UNIFIED IDEOGRAPH
                                (#x8A50 . #x595D)	;CJK UNIFIED IDEOGRAPH
                                (#x8A51 . #x595E)	;CJK UNIFIED IDEOGRAPH
                                (#x8A52 . #x595F)	;CJK UNIFIED IDEOGRAPH
                                (#x8A53 . #x5961)	;CJK UNIFIED IDEOGRAPH
                                (#x8A54 . #x5963)	;CJK UNIFIED IDEOGRAPH
                                (#x8A55 . #x5964)	;CJK UNIFIED IDEOGRAPH
                                (#x8A56 . #x5966)	;CJK UNIFIED IDEOGRAPH
                                (#x8A57 . #x5967)	;CJK UNIFIED IDEOGRAPH
                                (#x8A58 . #x5968)	;CJK UNIFIED IDEOGRAPH
                                (#x8A59 . #x5969)	;CJK UNIFIED IDEOGRAPH
                                (#x8A5A . #x596A)	;CJK UNIFIED IDEOGRAPH
                                (#x8A5B . #x596B)	;CJK UNIFIED IDEOGRAPH
                                (#x8A5C . #x596C)	;CJK UNIFIED IDEOGRAPH
                                (#x8A5D . #x596D)	;CJK UNIFIED IDEOGRAPH
                                (#x8A5E . #x596E)	;CJK UNIFIED IDEOGRAPH
                                (#x8A5F . #x596F)	;CJK UNIFIED IDEOGRAPH
                                (#x8A60 . #x5970)	;CJK UNIFIED IDEOGRAPH
                                (#x8A61 . #x5971)	;CJK UNIFIED IDEOGRAPH
                                (#x8A62 . #x5972)	;CJK UNIFIED IDEOGRAPH
                                (#x8A63 . #x5975)	;CJK UNIFIED IDEOGRAPH
                                (#x8A64 . #x5977)	;CJK UNIFIED IDEOGRAPH
                                (#x8A65 . #x597A)	;CJK UNIFIED IDEOGRAPH
                                (#x8A66 . #x597B)	;CJK UNIFIED IDEOGRAPH
                                (#x8A67 . #x597C)	;CJK UNIFIED IDEOGRAPH
                                (#x8A68 . #x597E)	;CJK UNIFIED IDEOGRAPH
                                (#x8A69 . #x597F)	;CJK UNIFIED IDEOGRAPH
                                (#x8A6A . #x5980)	;CJK UNIFIED IDEOGRAPH
                                (#x8A6B . #x5985)	;CJK UNIFIED IDEOGRAPH
                                (#x8A6C . #x5989)	;CJK UNIFIED IDEOGRAPH
                                (#x8A6D . #x598B)	;CJK UNIFIED IDEOGRAPH
                                (#x8A6E . #x598C)	;CJK UNIFIED IDEOGRAPH
                                (#x8A6F . #x598E)	;CJK UNIFIED IDEOGRAPH
                                (#x8A70 . #x598F)	;CJK UNIFIED IDEOGRAPH
                                (#x8A71 . #x5990)	;CJK UNIFIED IDEOGRAPH
                                (#x8A72 . #x5991)	;CJK UNIFIED IDEOGRAPH
                                (#x8A73 . #x5994)	;CJK UNIFIED IDEOGRAPH
                                (#x8A74 . #x5995)	;CJK UNIFIED IDEOGRAPH
                                (#x8A75 . #x5998)	;CJK UNIFIED IDEOGRAPH
                                (#x8A76 . #x599A)	;CJK UNIFIED IDEOGRAPH
                                (#x8A77 . #x599B)	;CJK UNIFIED IDEOGRAPH
                                (#x8A78 . #x599C)	;CJK UNIFIED IDEOGRAPH
                                (#x8A79 . #x599D)	;CJK UNIFIED IDEOGRAPH
                                (#x8A7A . #x599F)	;CJK UNIFIED IDEOGRAPH
                                (#x8A7B . #x59A0)	;CJK UNIFIED IDEOGRAPH
                                (#x8A7C . #x59A1)	;CJK UNIFIED IDEOGRAPH
                                (#x8A7D . #x59A2)	;CJK UNIFIED IDEOGRAPH
                                (#x8A7E . #x59A6)	;CJK UNIFIED IDEOGRAPH
                                (#x8A80 . #x59A7)	;CJK UNIFIED IDEOGRAPH
                                (#x8A81 . #x59AC)	;CJK UNIFIED IDEOGRAPH
                                (#x8A82 . #x59AD)	;CJK UNIFIED IDEOGRAPH
                                (#x8A83 . #x59B0)	;CJK UNIFIED IDEOGRAPH
                                (#x8A84 . #x59B1)	;CJK UNIFIED IDEOGRAPH
                                (#x8A85 . #x59B3)	;CJK UNIFIED IDEOGRAPH
                                (#x8A86 . #x59B4)	;CJK UNIFIED IDEOGRAPH
                                (#x8A87 . #x59B5)	;CJK UNIFIED IDEOGRAPH
                                (#x8A88 . #x59B6)	;CJK UNIFIED IDEOGRAPH
                                (#x8A89 . #x59B7)	;CJK UNIFIED IDEOGRAPH
                                (#x8A8A . #x59B8)	;CJK UNIFIED IDEOGRAPH
                                (#x8A8B . #x59BA)	;CJK UNIFIED IDEOGRAPH
                                (#x8A8C . #x59BC)	;CJK UNIFIED IDEOGRAPH
                                (#x8A8D . #x59BD)	;CJK UNIFIED IDEOGRAPH
                                (#x8A8E . #x59BF)	;CJK UNIFIED IDEOGRAPH
                                (#x8A8F . #x59C0)	;CJK UNIFIED IDEOGRAPH
                                (#x8A90 . #x59C1)	;CJK UNIFIED IDEOGRAPH
                                (#x8A91 . #x59C2)	;CJK UNIFIED IDEOGRAPH
                                (#x8A92 . #x59C3)	;CJK UNIFIED IDEOGRAPH
                                (#x8A93 . #x59C4)	;CJK UNIFIED IDEOGRAPH
                                (#x8A94 . #x59C5)	;CJK UNIFIED IDEOGRAPH
                                (#x8A95 . #x59C7)	;CJK UNIFIED IDEOGRAPH
                                (#x8A96 . #x59C8)	;CJK UNIFIED IDEOGRAPH
                                (#x8A97 . #x59C9)	;CJK UNIFIED IDEOGRAPH
                                (#x8A98 . #x59CC)	;CJK UNIFIED IDEOGRAPH
                                (#x8A99 . #x59CD)	;CJK UNIFIED IDEOGRAPH
                                (#x8A9A . #x59CE)	;CJK UNIFIED IDEOGRAPH
                                (#x8A9B . #x59CF)	;CJK UNIFIED IDEOGRAPH
                                (#x8A9C . #x59D5)	;CJK UNIFIED IDEOGRAPH
                                (#x8A9D . #x59D6)	;CJK UNIFIED IDEOGRAPH
                                (#x8A9E . #x59D9)	;CJK UNIFIED IDEOGRAPH
                                (#x8A9F . #x59DB)	;CJK UNIFIED IDEOGRAPH
                                (#x8AA0 . #x59DE)	;CJK UNIFIED IDEOGRAPH
                                (#x8AA1 . #x59DF)	;CJK UNIFIED IDEOGRAPH
                                (#x8AA2 . #x59E0)	;CJK UNIFIED IDEOGRAPH
                                (#x8AA3 . #x59E1)	;CJK UNIFIED IDEOGRAPH
                                (#x8AA4 . #x59E2)	;CJK UNIFIED IDEOGRAPH
                                (#x8AA5 . #x59E4)	;CJK UNIFIED IDEOGRAPH
                                (#x8AA6 . #x59E6)	;CJK UNIFIED IDEOGRAPH
                                (#x8AA7 . #x59E7)	;CJK UNIFIED IDEOGRAPH
                                (#x8AA8 . #x59E9)	;CJK UNIFIED IDEOGRAPH
                                (#x8AA9 . #x59EA)	;CJK UNIFIED IDEOGRAPH
                                (#x8AAA . #x59EB)	;CJK UNIFIED IDEOGRAPH
                                (#x8AAB . #x59ED)	;CJK UNIFIED IDEOGRAPH
                                (#x8AAC . #x59EE)	;CJK UNIFIED IDEOGRAPH
                                (#x8AAD . #x59EF)	;CJK UNIFIED IDEOGRAPH
                                (#x8AAE . #x59F0)	;CJK UNIFIED IDEOGRAPH
                                (#x8AAF . #x59F1)	;CJK UNIFIED IDEOGRAPH
                                (#x8AB0 . #x59F2)	;CJK UNIFIED IDEOGRAPH
                                (#x8AB1 . #x59F3)	;CJK UNIFIED IDEOGRAPH
                                (#x8AB2 . #x59F4)	;CJK UNIFIED IDEOGRAPH
                                (#x8AB3 . #x59F5)	;CJK UNIFIED IDEOGRAPH
                                (#x8AB4 . #x59F6)	;CJK UNIFIED IDEOGRAPH
                                (#x8AB5 . #x59F7)	;CJK UNIFIED IDEOGRAPH
                                (#x8AB6 . #x59F8)	;CJK UNIFIED IDEOGRAPH
                                (#x8AB7 . #x59FA)	;CJK UNIFIED IDEOGRAPH
                                (#x8AB8 . #x59FC)	;CJK UNIFIED IDEOGRAPH
                                (#x8AB9 . #x59FD)	;CJK UNIFIED IDEOGRAPH
                                (#x8ABA . #x59FE)	;CJK UNIFIED IDEOGRAPH
                                (#x8ABB . #x5A00)	;CJK UNIFIED IDEOGRAPH
                                (#x8ABC . #x5A02)	;CJK UNIFIED IDEOGRAPH
                                (#x8ABD . #x5A0A)	;CJK UNIFIED IDEOGRAPH
                                (#x8ABE . #x5A0B)	;CJK UNIFIED IDEOGRAPH
                                (#x8ABF . #x5A0D)	;CJK UNIFIED IDEOGRAPH
                                (#x8AC0 . #x5A0E)	;CJK UNIFIED IDEOGRAPH
                                (#x8AC1 . #x5A0F)	;CJK UNIFIED IDEOGRAPH
                                (#x8AC2 . #x5A10)	;CJK UNIFIED IDEOGRAPH
                                (#x8AC3 . #x5A12)	;CJK UNIFIED IDEOGRAPH
                                (#x8AC4 . #x5A14)	;CJK UNIFIED IDEOGRAPH
                                (#x8AC5 . #x5A15)	;CJK UNIFIED IDEOGRAPH
                                (#x8AC6 . #x5A16)	;CJK UNIFIED IDEOGRAPH
                                (#x8AC7 . #x5A17)	;CJK UNIFIED IDEOGRAPH
                                (#x8AC8 . #x5A19)	;CJK UNIFIED IDEOGRAPH
                                (#x8AC9 . #x5A1A)	;CJK UNIFIED IDEOGRAPH
                                (#x8ACA . #x5A1B)	;CJK UNIFIED IDEOGRAPH
                                (#x8ACB . #x5A1D)	;CJK UNIFIED IDEOGRAPH
                                (#x8ACC . #x5A1E)	;CJK UNIFIED IDEOGRAPH
                                (#x8ACD . #x5A21)	;CJK UNIFIED IDEOGRAPH
                                (#x8ACE . #x5A22)	;CJK UNIFIED IDEOGRAPH
                                (#x8ACF . #x5A24)	;CJK UNIFIED IDEOGRAPH
                                (#x8AD0 . #x5A26)	;CJK UNIFIED IDEOGRAPH
                                (#x8AD1 . #x5A27)	;CJK UNIFIED IDEOGRAPH
                                (#x8AD2 . #x5A28)	;CJK UNIFIED IDEOGRAPH
                                (#x8AD3 . #x5A2A)	;CJK UNIFIED IDEOGRAPH
                                (#x8AD4 . #x5A2B)	;CJK UNIFIED IDEOGRAPH
                                (#x8AD5 . #x5A2C)	;CJK UNIFIED IDEOGRAPH
                                (#x8AD6 . #x5A2D)	;CJK UNIFIED IDEOGRAPH
                                (#x8AD7 . #x5A2E)	;CJK UNIFIED IDEOGRAPH
                                (#x8AD8 . #x5A2F)	;CJK UNIFIED IDEOGRAPH
                                (#x8AD9 . #x5A30)	;CJK UNIFIED IDEOGRAPH
                                (#x8ADA . #x5A33)	;CJK UNIFIED IDEOGRAPH
                                (#x8ADB . #x5A35)	;CJK UNIFIED IDEOGRAPH
                                (#x8ADC . #x5A37)	;CJK UNIFIED IDEOGRAPH
                                (#x8ADD . #x5A38)	;CJK UNIFIED IDEOGRAPH
                                (#x8ADE . #x5A39)	;CJK UNIFIED IDEOGRAPH
                                (#x8ADF . #x5A3A)	;CJK UNIFIED IDEOGRAPH
                                (#x8AE0 . #x5A3B)	;CJK UNIFIED IDEOGRAPH
                                (#x8AE1 . #x5A3D)	;CJK UNIFIED IDEOGRAPH
                                (#x8AE2 . #x5A3E)	;CJK UNIFIED IDEOGRAPH
                                (#x8AE3 . #x5A3F)	;CJK UNIFIED IDEOGRAPH
                                (#x8AE4 . #x5A41)	;CJK UNIFIED IDEOGRAPH
                                (#x8AE5 . #x5A42)	;CJK UNIFIED IDEOGRAPH
                                (#x8AE6 . #x5A43)	;CJK UNIFIED IDEOGRAPH
                                (#x8AE7 . #x5A44)	;CJK UNIFIED IDEOGRAPH
                                (#x8AE8 . #x5A45)	;CJK UNIFIED IDEOGRAPH
                                (#x8AE9 . #x5A47)	;CJK UNIFIED IDEOGRAPH
                                (#x8AEA . #x5A48)	;CJK UNIFIED IDEOGRAPH
                                (#x8AEB . #x5A4B)	;CJK UNIFIED IDEOGRAPH
                                (#x8AEC . #x5A4C)	;CJK UNIFIED IDEOGRAPH
                                (#x8AED . #x5A4D)	;CJK UNIFIED IDEOGRAPH
                                (#x8AEE . #x5A4E)	;CJK UNIFIED IDEOGRAPH
                                (#x8AEF . #x5A4F)	;CJK UNIFIED IDEOGRAPH
                                (#x8AF0 . #x5A50)	;CJK UNIFIED IDEOGRAPH
                                (#x8AF1 . #x5A51)	;CJK UNIFIED IDEOGRAPH
                                (#x8AF2 . #x5A52)	;CJK UNIFIED IDEOGRAPH
                                (#x8AF3 . #x5A53)	;CJK UNIFIED IDEOGRAPH
                                (#x8AF4 . #x5A54)	;CJK UNIFIED IDEOGRAPH
                                (#x8AF5 . #x5A56)	;CJK UNIFIED IDEOGRAPH
                                (#x8AF6 . #x5A57)	;CJK UNIFIED IDEOGRAPH
                                (#x8AF7 . #x5A58)	;CJK UNIFIED IDEOGRAPH
                                (#x8AF8 . #x5A59)	;CJK UNIFIED IDEOGRAPH
                                (#x8AF9 . #x5A5B)	;CJK UNIFIED IDEOGRAPH
                                (#x8AFA . #x5A5C)	;CJK UNIFIED IDEOGRAPH
                                (#x8AFB . #x5A5D)	;CJK UNIFIED IDEOGRAPH
                                (#x8AFC . #x5A5E)	;CJK UNIFIED IDEOGRAPH
                                (#x8AFD . #x5A5F)	;CJK UNIFIED IDEOGRAPH
                                (#x8AFE . #x5A60)	;CJK UNIFIED IDEOGRAPH
                                (#x8B40 . #x5A61)	;CJK UNIFIED IDEOGRAPH
                                (#x8B41 . #x5A63)	;CJK UNIFIED IDEOGRAPH
                                (#x8B42 . #x5A64)	;CJK UNIFIED IDEOGRAPH
                                (#x8B43 . #x5A65)	;CJK UNIFIED IDEOGRAPH
                                (#x8B44 . #x5A66)	;CJK UNIFIED IDEOGRAPH
                                (#x8B45 . #x5A68)	;CJK UNIFIED IDEOGRAPH
                                (#x8B46 . #x5A69)	;CJK UNIFIED IDEOGRAPH
                                (#x8B47 . #x5A6B)	;CJK UNIFIED IDEOGRAPH
                                (#x8B48 . #x5A6C)	;CJK UNIFIED IDEOGRAPH
                                (#x8B49 . #x5A6D)	;CJK UNIFIED IDEOGRAPH
                                (#x8B4A . #x5A6E)	;CJK UNIFIED IDEOGRAPH
                                (#x8B4B . #x5A6F)	;CJK UNIFIED IDEOGRAPH
                                (#x8B4C . #x5A70)	;CJK UNIFIED IDEOGRAPH
                                (#x8B4D . #x5A71)	;CJK UNIFIED IDEOGRAPH
                                (#x8B4E . #x5A72)	;CJK UNIFIED IDEOGRAPH
                                (#x8B4F . #x5A73)	;CJK UNIFIED IDEOGRAPH
                                (#x8B50 . #x5A78)	;CJK UNIFIED IDEOGRAPH
                                (#x8B51 . #x5A79)	;CJK UNIFIED IDEOGRAPH
                                (#x8B52 . #x5A7B)	;CJK UNIFIED IDEOGRAPH
                                (#x8B53 . #x5A7C)	;CJK UNIFIED IDEOGRAPH
                                (#x8B54 . #x5A7D)	;CJK UNIFIED IDEOGRAPH
                                (#x8B55 . #x5A7E)	;CJK UNIFIED IDEOGRAPH
                                (#x8B56 . #x5A80)	;CJK UNIFIED IDEOGRAPH
                                (#x8B57 . #x5A81)	;CJK UNIFIED IDEOGRAPH
                                (#x8B58 . #x5A82)	;CJK UNIFIED IDEOGRAPH
                                (#x8B59 . #x5A83)	;CJK UNIFIED IDEOGRAPH
                                (#x8B5A . #x5A84)	;CJK UNIFIED IDEOGRAPH
                                (#x8B5B . #x5A85)	;CJK UNIFIED IDEOGRAPH
                                (#x8B5C . #x5A86)	;CJK UNIFIED IDEOGRAPH
                                (#x8B5D . #x5A87)	;CJK UNIFIED IDEOGRAPH
                                (#x8B5E . #x5A88)	;CJK UNIFIED IDEOGRAPH
                                (#x8B5F . #x5A89)	;CJK UNIFIED IDEOGRAPH
                                (#x8B60 . #x5A8A)	;CJK UNIFIED IDEOGRAPH
                                (#x8B61 . #x5A8B)	;CJK UNIFIED IDEOGRAPH
                                (#x8B62 . #x5A8C)	;CJK UNIFIED IDEOGRAPH
                                (#x8B63 . #x5A8D)	;CJK UNIFIED IDEOGRAPH
                                (#x8B64 . #x5A8E)	;CJK UNIFIED IDEOGRAPH
                                (#x8B65 . #x5A8F)	;CJK UNIFIED IDEOGRAPH
                                (#x8B66 . #x5A90)	;CJK UNIFIED IDEOGRAPH
                                (#x8B67 . #x5A91)	;CJK UNIFIED IDEOGRAPH
                                (#x8B68 . #x5A93)	;CJK UNIFIED IDEOGRAPH
                                (#x8B69 . #x5A94)	;CJK UNIFIED IDEOGRAPH
                                (#x8B6A . #x5A95)	;CJK UNIFIED IDEOGRAPH
                                (#x8B6B . #x5A96)	;CJK UNIFIED IDEOGRAPH
                                (#x8B6C . #x5A97)	;CJK UNIFIED IDEOGRAPH
                                (#x8B6D . #x5A98)	;CJK UNIFIED IDEOGRAPH
                                (#x8B6E . #x5A99)	;CJK UNIFIED IDEOGRAPH
                                (#x8B6F . #x5A9C)	;CJK UNIFIED IDEOGRAPH
                                (#x8B70 . #x5A9D)	;CJK UNIFIED IDEOGRAPH
                                (#x8B71 . #x5A9E)	;CJK UNIFIED IDEOGRAPH
                                (#x8B72 . #x5A9F)	;CJK UNIFIED IDEOGRAPH
                                (#x8B73 . #x5AA0)	;CJK UNIFIED IDEOGRAPH
                                (#x8B74 . #x5AA1)	;CJK UNIFIED IDEOGRAPH
                                (#x8B75 . #x5AA2)	;CJK UNIFIED IDEOGRAPH
                                (#x8B76 . #x5AA3)	;CJK UNIFIED IDEOGRAPH
                                (#x8B77 . #x5AA4)	;CJK UNIFIED IDEOGRAPH
                                (#x8B78 . #x5AA5)	;CJK UNIFIED IDEOGRAPH
                                (#x8B79 . #x5AA6)	;CJK UNIFIED IDEOGRAPH
                                (#x8B7A . #x5AA7)	;CJK UNIFIED IDEOGRAPH
                                (#x8B7B . #x5AA8)	;CJK UNIFIED IDEOGRAPH
                                (#x8B7C . #x5AA9)	;CJK UNIFIED IDEOGRAPH
                                (#x8B7D . #x5AAB)	;CJK UNIFIED IDEOGRAPH
                                (#x8B7E . #x5AAC)	;CJK UNIFIED IDEOGRAPH
                                (#x8B80 . #x5AAD)	;CJK UNIFIED IDEOGRAPH
                                (#x8B81 . #x5AAE)	;CJK UNIFIED IDEOGRAPH
                                (#x8B82 . #x5AAF)	;CJK UNIFIED IDEOGRAPH
                                (#x8B83 . #x5AB0)	;CJK UNIFIED IDEOGRAPH
                                (#x8B84 . #x5AB1)	;CJK UNIFIED IDEOGRAPH
                                (#x8B85 . #x5AB4)	;CJK UNIFIED IDEOGRAPH
                                (#x8B86 . #x5AB6)	;CJK UNIFIED IDEOGRAPH
                                (#x8B87 . #x5AB7)	;CJK UNIFIED IDEOGRAPH
                                (#x8B88 . #x5AB9)	;CJK UNIFIED IDEOGRAPH
                                (#x8B89 . #x5ABA)	;CJK UNIFIED IDEOGRAPH
                                (#x8B8A . #x5ABB)	;CJK UNIFIED IDEOGRAPH
                                (#x8B8B . #x5ABC)	;CJK UNIFIED IDEOGRAPH
                                (#x8B8C . #x5ABD)	;CJK UNIFIED IDEOGRAPH
                                (#x8B8D . #x5ABF)	;CJK UNIFIED IDEOGRAPH
                                (#x8B8E . #x5AC0)	;CJK UNIFIED IDEOGRAPH
                                (#x8B8F . #x5AC3)	;CJK UNIFIED IDEOGRAPH
                                (#x8B90 . #x5AC4)	;CJK UNIFIED IDEOGRAPH
                                (#x8B91 . #x5AC5)	;CJK UNIFIED IDEOGRAPH
                                (#x8B92 . #x5AC6)	;CJK UNIFIED IDEOGRAPH
                                (#x8B93 . #x5AC7)	;CJK UNIFIED IDEOGRAPH
                                (#x8B94 . #x5AC8)	;CJK UNIFIED IDEOGRAPH
                                (#x8B95 . #x5ACA)	;CJK UNIFIED IDEOGRAPH
                                (#x8B96 . #x5ACB)	;CJK UNIFIED IDEOGRAPH
                                (#x8B97 . #x5ACD)	;CJK UNIFIED IDEOGRAPH
                                (#x8B98 . #x5ACE)	;CJK UNIFIED IDEOGRAPH
                                (#x8B99 . #x5ACF)	;CJK UNIFIED IDEOGRAPH
                                (#x8B9A . #x5AD0)	;CJK UNIFIED IDEOGRAPH
                                (#x8B9B . #x5AD1)	;CJK UNIFIED IDEOGRAPH
                                (#x8B9C . #x5AD3)	;CJK UNIFIED IDEOGRAPH
                                (#x8B9D . #x5AD5)	;CJK UNIFIED IDEOGRAPH
                                (#x8B9E . #x5AD7)	;CJK UNIFIED IDEOGRAPH
                                (#x8B9F . #x5AD9)	;CJK UNIFIED IDEOGRAPH
                                (#x8BA0 . #x5ADA)	;CJK UNIFIED IDEOGRAPH
                                (#x8BA1 . #x5ADB)	;CJK UNIFIED IDEOGRAPH
                                (#x8BA2 . #x5ADD)	;CJK UNIFIED IDEOGRAPH
                                (#x8BA3 . #x5ADE)	;CJK UNIFIED IDEOGRAPH
                                (#x8BA4 . #x5ADF)	;CJK UNIFIED IDEOGRAPH
                                (#x8BA5 . #x5AE2)	;CJK UNIFIED IDEOGRAPH
                                (#x8BA6 . #x5AE4)	;CJK UNIFIED IDEOGRAPH
                                (#x8BA7 . #x5AE5)	;CJK UNIFIED IDEOGRAPH
                                (#x8BA8 . #x5AE7)	;CJK UNIFIED IDEOGRAPH
                                (#x8BA9 . #x5AE8)	;CJK UNIFIED IDEOGRAPH
                                (#x8BAA . #x5AEA)	;CJK UNIFIED IDEOGRAPH
                                (#x8BAB . #x5AEC)	;CJK UNIFIED IDEOGRAPH
                                (#x8BAC . #x5AED)	;CJK UNIFIED IDEOGRAPH
                                (#x8BAD . #x5AEE)	;CJK UNIFIED IDEOGRAPH
                                (#x8BAE . #x5AEF)	;CJK UNIFIED IDEOGRAPH
                                (#x8BAF . #x5AF0)	;CJK UNIFIED IDEOGRAPH
                                (#x8BB0 . #x5AF2)	;CJK UNIFIED IDEOGRAPH
                                (#x8BB1 . #x5AF3)	;CJK UNIFIED IDEOGRAPH
                                (#x8BB2 . #x5AF4)	;CJK UNIFIED IDEOGRAPH
                                (#x8BB3 . #x5AF5)	;CJK UNIFIED IDEOGRAPH
                                (#x8BB4 . #x5AF6)	;CJK UNIFIED IDEOGRAPH
                                (#x8BB5 . #x5AF7)	;CJK UNIFIED IDEOGRAPH
                                (#x8BB6 . #x5AF8)	;CJK UNIFIED IDEOGRAPH
                                (#x8BB7 . #x5AF9)	;CJK UNIFIED IDEOGRAPH
                                (#x8BB8 . #x5AFA)	;CJK UNIFIED IDEOGRAPH
                                (#x8BB9 . #x5AFB)	;CJK UNIFIED IDEOGRAPH
                                (#x8BBA . #x5AFC)	;CJK UNIFIED IDEOGRAPH
                                (#x8BBB . #x5AFD)	;CJK UNIFIED IDEOGRAPH
                                (#x8BBC . #x5AFE)	;CJK UNIFIED IDEOGRAPH
                                (#x8BBD . #x5AFF)	;CJK UNIFIED IDEOGRAPH
                                (#x8BBE . #x5B00)	;CJK UNIFIED IDEOGRAPH
                                (#x8BBF . #x5B01)	;CJK UNIFIED IDEOGRAPH
                                (#x8BC0 . #x5B02)	;CJK UNIFIED IDEOGRAPH
                                (#x8BC1 . #x5B03)	;CJK UNIFIED IDEOGRAPH
                                (#x8BC2 . #x5B04)	;CJK UNIFIED IDEOGRAPH
                                (#x8BC3 . #x5B05)	;CJK UNIFIED IDEOGRAPH
                                (#x8BC4 . #x5B06)	;CJK UNIFIED IDEOGRAPH
                                (#x8BC5 . #x5B07)	;CJK UNIFIED IDEOGRAPH
                                (#x8BC6 . #x5B08)	;CJK UNIFIED IDEOGRAPH
                                (#x8BC7 . #x5B0A)	;CJK UNIFIED IDEOGRAPH
                                (#x8BC8 . #x5B0B)	;CJK UNIFIED IDEOGRAPH
                                (#x8BC9 . #x5B0C)	;CJK UNIFIED IDEOGRAPH
                                (#x8BCA . #x5B0D)	;CJK UNIFIED IDEOGRAPH
                                (#x8BCB . #x5B0E)	;CJK UNIFIED IDEOGRAPH
                                (#x8BCC . #x5B0F)	;CJK UNIFIED IDEOGRAPH
                                (#x8BCD . #x5B10)	;CJK UNIFIED IDEOGRAPH
                                (#x8BCE . #x5B11)	;CJK UNIFIED IDEOGRAPH
                                (#x8BCF . #x5B12)	;CJK UNIFIED IDEOGRAPH
                                (#x8BD0 . #x5B13)	;CJK UNIFIED IDEOGRAPH
                                (#x8BD1 . #x5B14)	;CJK UNIFIED IDEOGRAPH
                                (#x8BD2 . #x5B15)	;CJK UNIFIED IDEOGRAPH
                                (#x8BD3 . #x5B18)	;CJK UNIFIED IDEOGRAPH
                                (#x8BD4 . #x5B19)	;CJK UNIFIED IDEOGRAPH
                                (#x8BD5 . #x5B1A)	;CJK UNIFIED IDEOGRAPH
                                (#x8BD6 . #x5B1B)	;CJK UNIFIED IDEOGRAPH
                                (#x8BD7 . #x5B1C)	;CJK UNIFIED IDEOGRAPH
                                (#x8BD8 . #x5B1D)	;CJK UNIFIED IDEOGRAPH
                                (#x8BD9 . #x5B1E)	;CJK UNIFIED IDEOGRAPH
                                (#x8BDA . #x5B1F)	;CJK UNIFIED IDEOGRAPH
                                (#x8BDB . #x5B20)	;CJK UNIFIED IDEOGRAPH
                                (#x8BDC . #x5B21)	;CJK UNIFIED IDEOGRAPH
                                (#x8BDD . #x5B22)	;CJK UNIFIED IDEOGRAPH
                                (#x8BDE . #x5B23)	;CJK UNIFIED IDEOGRAPH
                                (#x8BDF . #x5B24)	;CJK UNIFIED IDEOGRAPH
                                (#x8BE0 . #x5B25)	;CJK UNIFIED IDEOGRAPH
                                (#x8BE1 . #x5B26)	;CJK UNIFIED IDEOGRAPH
                                (#x8BE2 . #x5B27)	;CJK UNIFIED IDEOGRAPH
                                (#x8BE3 . #x5B28)	;CJK UNIFIED IDEOGRAPH
                                (#x8BE4 . #x5B29)	;CJK UNIFIED IDEOGRAPH
                                (#x8BE5 . #x5B2A)	;CJK UNIFIED IDEOGRAPH
                                (#x8BE6 . #x5B2B)	;CJK UNIFIED IDEOGRAPH
                                (#x8BE7 . #x5B2C)	;CJK UNIFIED IDEOGRAPH
                                (#x8BE8 . #x5B2D)	;CJK UNIFIED IDEOGRAPH
                                (#x8BE9 . #x5B2E)	;CJK UNIFIED IDEOGRAPH
                                (#x8BEA . #x5B2F)	;CJK UNIFIED IDEOGRAPH
                                (#x8BEB . #x5B30)	;CJK UNIFIED IDEOGRAPH
                                (#x8BEC . #x5B31)	;CJK UNIFIED IDEOGRAPH
                                (#x8BED . #x5B33)	;CJK UNIFIED IDEOGRAPH
                                (#x8BEE . #x5B35)	;CJK UNIFIED IDEOGRAPH
                                (#x8BEF . #x5B36)	;CJK UNIFIED IDEOGRAPH
                                (#x8BF0 . #x5B38)	;CJK UNIFIED IDEOGRAPH
                                (#x8BF1 . #x5B39)	;CJK UNIFIED IDEOGRAPH
                                (#x8BF2 . #x5B3A)	;CJK UNIFIED IDEOGRAPH
                                (#x8BF3 . #x5B3B)	;CJK UNIFIED IDEOGRAPH
                                (#x8BF4 . #x5B3C)	;CJK UNIFIED IDEOGRAPH
                                (#x8BF5 . #x5B3D)	;CJK UNIFIED IDEOGRAPH
                                (#x8BF6 . #x5B3E)	;CJK UNIFIED IDEOGRAPH
                                (#x8BF7 . #x5B3F)	;CJK UNIFIED IDEOGRAPH
                                (#x8BF8 . #x5B41)	;CJK UNIFIED IDEOGRAPH
                                (#x8BF9 . #x5B42)	;CJK UNIFIED IDEOGRAPH
                                (#x8BFA . #x5B43)	;CJK UNIFIED IDEOGRAPH
                                (#x8BFB . #x5B44)	;CJK UNIFIED IDEOGRAPH
                                (#x8BFC . #x5B45)	;CJK UNIFIED IDEOGRAPH
                                (#x8BFD . #x5B46)	;CJK UNIFIED IDEOGRAPH
                                (#x8BFE . #x5B47)	;CJK UNIFIED IDEOGRAPH
                                (#x8C40 . #x5B48)	;CJK UNIFIED IDEOGRAPH
                                (#x8C41 . #x5B49)	;CJK UNIFIED IDEOGRAPH
                                (#x8C42 . #x5B4A)	;CJK UNIFIED IDEOGRAPH
                                (#x8C43 . #x5B4B)	;CJK UNIFIED IDEOGRAPH
                                (#x8C44 . #x5B4C)	;CJK UNIFIED IDEOGRAPH
                                (#x8C45 . #x5B4D)	;CJK UNIFIED IDEOGRAPH
                                (#x8C46 . #x5B4E)	;CJK UNIFIED IDEOGRAPH
                                (#x8C47 . #x5B4F)	;CJK UNIFIED IDEOGRAPH
                                (#x8C48 . #x5B52)	;CJK UNIFIED IDEOGRAPH
                                (#x8C49 . #x5B56)	;CJK UNIFIED IDEOGRAPH
                                (#x8C4A . #x5B5E)	;CJK UNIFIED IDEOGRAPH
                                (#x8C4B . #x5B60)	;CJK UNIFIED IDEOGRAPH
                                (#x8C4C . #x5B61)	;CJK UNIFIED IDEOGRAPH
                                (#x8C4D . #x5B67)	;CJK UNIFIED IDEOGRAPH
                                (#x8C4E . #x5B68)	;CJK UNIFIED IDEOGRAPH
                                (#x8C4F . #x5B6B)	;CJK UNIFIED IDEOGRAPH
                                (#x8C50 . #x5B6D)	;CJK UNIFIED IDEOGRAPH
                                (#x8C51 . #x5B6E)	;CJK UNIFIED IDEOGRAPH
                                (#x8C52 . #x5B6F)	;CJK UNIFIED IDEOGRAPH
                                (#x8C53 . #x5B72)	;CJK UNIFIED IDEOGRAPH
                                (#x8C54 . #x5B74)	;CJK UNIFIED IDEOGRAPH
                                (#x8C55 . #x5B76)	;CJK UNIFIED IDEOGRAPH
                                (#x8C56 . #x5B77)	;CJK UNIFIED IDEOGRAPH
                                (#x8C57 . #x5B78)	;CJK UNIFIED IDEOGRAPH
                                (#x8C58 . #x5B79)	;CJK UNIFIED IDEOGRAPH
                                (#x8C59 . #x5B7B)	;CJK UNIFIED IDEOGRAPH
                                (#x8C5A . #x5B7C)	;CJK UNIFIED IDEOGRAPH
                                (#x8C5B . #x5B7E)	;CJK UNIFIED IDEOGRAPH
                                (#x8C5C . #x5B7F)	;CJK UNIFIED IDEOGRAPH
                                (#x8C5D . #x5B82)	;CJK UNIFIED IDEOGRAPH
                                (#x8C5E . #x5B86)	;CJK UNIFIED IDEOGRAPH
                                (#x8C5F . #x5B8A)	;CJK UNIFIED IDEOGRAPH
                                (#x8C60 . #x5B8D)	;CJK UNIFIED IDEOGRAPH
                                (#x8C61 . #x5B8E)	;CJK UNIFIED IDEOGRAPH
                                (#x8C62 . #x5B90)	;CJK UNIFIED IDEOGRAPH
                                (#x8C63 . #x5B91)	;CJK UNIFIED IDEOGRAPH
                                (#x8C64 . #x5B92)	;CJK UNIFIED IDEOGRAPH
                                (#x8C65 . #x5B94)	;CJK UNIFIED IDEOGRAPH
                                (#x8C66 . #x5B96)	;CJK UNIFIED IDEOGRAPH
                                (#x8C67 . #x5B9F)	;CJK UNIFIED IDEOGRAPH
                                (#x8C68 . #x5BA7)	;CJK UNIFIED IDEOGRAPH
                                (#x8C69 . #x5BA8)	;CJK UNIFIED IDEOGRAPH
                                (#x8C6A . #x5BA9)	;CJK UNIFIED IDEOGRAPH
                                (#x8C6B . #x5BAC)	;CJK UNIFIED IDEOGRAPH
                                (#x8C6C . #x5BAD)	;CJK UNIFIED IDEOGRAPH
                                (#x8C6D . #x5BAE)	;CJK UNIFIED IDEOGRAPH
                                (#x8C6E . #x5BAF)	;CJK UNIFIED IDEOGRAPH
                                (#x8C6F . #x5BB1)	;CJK UNIFIED IDEOGRAPH
                                (#x8C70 . #x5BB2)	;CJK UNIFIED IDEOGRAPH
                                (#x8C71 . #x5BB7)	;CJK UNIFIED IDEOGRAPH
                                (#x8C72 . #x5BBA)	;CJK UNIFIED IDEOGRAPH
                                (#x8C73 . #x5BBB)	;CJK UNIFIED IDEOGRAPH
                                (#x8C74 . #x5BBC)	;CJK UNIFIED IDEOGRAPH
                                (#x8C75 . #x5BC0)	;CJK UNIFIED IDEOGRAPH
                                (#x8C76 . #x5BC1)	;CJK UNIFIED IDEOGRAPH
                                (#x8C77 . #x5BC3)	;CJK UNIFIED IDEOGRAPH
                                (#x8C78 . #x5BC8)	;CJK UNIFIED IDEOGRAPH
                                (#x8C79 . #x5BC9)	;CJK UNIFIED IDEOGRAPH
                                (#x8C7A . #x5BCA)	;CJK UNIFIED IDEOGRAPH
                                (#x8C7B . #x5BCB)	;CJK UNIFIED IDEOGRAPH
                                (#x8C7C . #x5BCD)	;CJK UNIFIED IDEOGRAPH
                                (#x8C7D . #x5BCE)	;CJK UNIFIED IDEOGRAPH
                                (#x8C7E . #x5BCF)	;CJK UNIFIED IDEOGRAPH
                                (#x8C80 . #x5BD1)	;CJK UNIFIED IDEOGRAPH
                                (#x8C81 . #x5BD4)	;CJK UNIFIED IDEOGRAPH
                                (#x8C82 . #x5BD5)	;CJK UNIFIED IDEOGRAPH
                                (#x8C83 . #x5BD6)	;CJK UNIFIED IDEOGRAPH
                                (#x8C84 . #x5BD7)	;CJK UNIFIED IDEOGRAPH
                                (#x8C85 . #x5BD8)	;CJK UNIFIED IDEOGRAPH
                                (#x8C86 . #x5BD9)	;CJK UNIFIED IDEOGRAPH
                                (#x8C87 . #x5BDA)	;CJK UNIFIED IDEOGRAPH
                                (#x8C88 . #x5BDB)	;CJK UNIFIED IDEOGRAPH
                                (#x8C89 . #x5BDC)	;CJK UNIFIED IDEOGRAPH
                                (#x8C8A . #x5BE0)	;CJK UNIFIED IDEOGRAPH
                                (#x8C8B . #x5BE2)	;CJK UNIFIED IDEOGRAPH
                                (#x8C8C . #x5BE3)	;CJK UNIFIED IDEOGRAPH
                                (#x8C8D . #x5BE6)	;CJK UNIFIED IDEOGRAPH
                                (#x8C8E . #x5BE7)	;CJK UNIFIED IDEOGRAPH
                                (#x8C8F . #x5BE9)	;CJK UNIFIED IDEOGRAPH
                                (#x8C90 . #x5BEA)	;CJK UNIFIED IDEOGRAPH
                                (#x8C91 . #x5BEB)	;CJK UNIFIED IDEOGRAPH
                                (#x8C92 . #x5BEC)	;CJK UNIFIED IDEOGRAPH
                                (#x8C93 . #x5BED)	;CJK UNIFIED IDEOGRAPH
                                (#x8C94 . #x5BEF)	;CJK UNIFIED IDEOGRAPH
                                (#x8C95 . #x5BF1)	;CJK UNIFIED IDEOGRAPH
                                (#x8C96 . #x5BF2)	;CJK UNIFIED IDEOGRAPH
                                (#x8C97 . #x5BF3)	;CJK UNIFIED IDEOGRAPH
                                (#x8C98 . #x5BF4)	;CJK UNIFIED IDEOGRAPH
                                (#x8C99 . #x5BF5)	;CJK UNIFIED IDEOGRAPH
                                (#x8C9A . #x5BF6)	;CJK UNIFIED IDEOGRAPH
                                (#x8C9B . #x5BF7)	;CJK UNIFIED IDEOGRAPH
                                (#x8C9C . #x5BFD)	;CJK UNIFIED IDEOGRAPH
                                (#x8C9D . #x5BFE)	;CJK UNIFIED IDEOGRAPH
                                (#x8C9E . #x5C00)	;CJK UNIFIED IDEOGRAPH
                                (#x8C9F . #x5C02)	;CJK UNIFIED IDEOGRAPH
                                (#x8CA0 . #x5C03)	;CJK UNIFIED IDEOGRAPH
                                (#x8CA1 . #x5C05)	;CJK UNIFIED IDEOGRAPH
                                (#x8CA2 . #x5C07)	;CJK UNIFIED IDEOGRAPH
                                (#x8CA3 . #x5C08)	;CJK UNIFIED IDEOGRAPH
                                (#x8CA4 . #x5C0B)	;CJK UNIFIED IDEOGRAPH
                                (#x8CA5 . #x5C0C)	;CJK UNIFIED IDEOGRAPH
                                (#x8CA6 . #x5C0D)	;CJK UNIFIED IDEOGRAPH
                                (#x8CA7 . #x5C0E)	;CJK UNIFIED IDEOGRAPH
                                (#x8CA8 . #x5C10)	;CJK UNIFIED IDEOGRAPH
                                (#x8CA9 . #x5C12)	;CJK UNIFIED IDEOGRAPH
                                (#x8CAA . #x5C13)	;CJK UNIFIED IDEOGRAPH
                                (#x8CAB . #x5C17)	;CJK UNIFIED IDEOGRAPH
                                (#x8CAC . #x5C19)	;CJK UNIFIED IDEOGRAPH
                                (#x8CAD . #x5C1B)	;CJK UNIFIED IDEOGRAPH
                                (#x8CAE . #x5C1E)	;CJK UNIFIED IDEOGRAPH
                                (#x8CAF . #x5C1F)	;CJK UNIFIED IDEOGRAPH
                                (#x8CB0 . #x5C20)	;CJK UNIFIED IDEOGRAPH
                                (#x8CB1 . #x5C21)	;CJK UNIFIED IDEOGRAPH
                                (#x8CB2 . #x5C23)	;CJK UNIFIED IDEOGRAPH
                                (#x8CB3 . #x5C26)	;CJK UNIFIED IDEOGRAPH
                                (#x8CB4 . #x5C28)	;CJK UNIFIED IDEOGRAPH
                                (#x8CB5 . #x5C29)	;CJK UNIFIED IDEOGRAPH
                                (#x8CB6 . #x5C2A)	;CJK UNIFIED IDEOGRAPH
                                (#x8CB7 . #x5C2B)	;CJK UNIFIED IDEOGRAPH
                                (#x8CB8 . #x5C2D)	;CJK UNIFIED IDEOGRAPH
                                (#x8CB9 . #x5C2E)	;CJK UNIFIED IDEOGRAPH
                                (#x8CBA . #x5C2F)	;CJK UNIFIED IDEOGRAPH
                                (#x8CBB . #x5C30)	;CJK UNIFIED IDEOGRAPH
                                (#x8CBC . #x5C32)	;CJK UNIFIED IDEOGRAPH
                                (#x8CBD . #x5C33)	;CJK UNIFIED IDEOGRAPH
                                (#x8CBE . #x5C35)	;CJK UNIFIED IDEOGRAPH
                                (#x8CBF . #x5C36)	;CJK UNIFIED IDEOGRAPH
                                (#x8CC0 . #x5C37)	;CJK UNIFIED IDEOGRAPH
                                (#x8CC1 . #x5C43)	;CJK UNIFIED IDEOGRAPH
                                (#x8CC2 . #x5C44)	;CJK UNIFIED IDEOGRAPH
                                (#x8CC3 . #x5C46)	;CJK UNIFIED IDEOGRAPH
                                (#x8CC4 . #x5C47)	;CJK UNIFIED IDEOGRAPH
                                (#x8CC5 . #x5C4C)	;CJK UNIFIED IDEOGRAPH
                                (#x8CC6 . #x5C4D)	;CJK UNIFIED IDEOGRAPH
                                (#x8CC7 . #x5C52)	;CJK UNIFIED IDEOGRAPH
                                (#x8CC8 . #x5C53)	;CJK UNIFIED IDEOGRAPH
                                (#x8CC9 . #x5C54)	;CJK UNIFIED IDEOGRAPH
                                (#x8CCA . #x5C56)	;CJK UNIFIED IDEOGRAPH
                                (#x8CCB . #x5C57)	;CJK UNIFIED IDEOGRAPH
                                (#x8CCC . #x5C58)	;CJK UNIFIED IDEOGRAPH
                                (#x8CCD . #x5C5A)	;CJK UNIFIED IDEOGRAPH
                                (#x8CCE . #x5C5B)	;CJK UNIFIED IDEOGRAPH
                                (#x8CCF . #x5C5C)	;CJK UNIFIED IDEOGRAPH
                                (#x8CD0 . #x5C5D)	;CJK UNIFIED IDEOGRAPH
                                (#x8CD1 . #x5C5F)	;CJK UNIFIED IDEOGRAPH
                                (#x8CD2 . #x5C62)	;CJK UNIFIED IDEOGRAPH
                                (#x8CD3 . #x5C64)	;CJK UNIFIED IDEOGRAPH
                                (#x8CD4 . #x5C67)	;CJK UNIFIED IDEOGRAPH
                                (#x8CD5 . #x5C68)	;CJK UNIFIED IDEOGRAPH
                                (#x8CD6 . #x5C69)	;CJK UNIFIED IDEOGRAPH
                                (#x8CD7 . #x5C6A)	;CJK UNIFIED IDEOGRAPH
                                (#x8CD8 . #x5C6B)	;CJK UNIFIED IDEOGRAPH
                                (#x8CD9 . #x5C6C)	;CJK UNIFIED IDEOGRAPH
                                (#x8CDA . #x5C6D)	;CJK UNIFIED IDEOGRAPH
                                (#x8CDB . #x5C70)	;CJK UNIFIED IDEOGRAPH
                                (#x8CDC . #x5C72)	;CJK UNIFIED IDEOGRAPH
                                (#x8CDD . #x5C73)	;CJK UNIFIED IDEOGRAPH
                                (#x8CDE . #x5C74)	;CJK UNIFIED IDEOGRAPH
                                (#x8CDF . #x5C75)	;CJK UNIFIED IDEOGRAPH
                                (#x8CE0 . #x5C76)	;CJK UNIFIED IDEOGRAPH
                                (#x8CE1 . #x5C77)	;CJK UNIFIED IDEOGRAPH
                                (#x8CE2 . #x5C78)	;CJK UNIFIED IDEOGRAPH
                                (#x8CE3 . #x5C7B)	;CJK UNIFIED IDEOGRAPH
                                (#x8CE4 . #x5C7C)	;CJK UNIFIED IDEOGRAPH
                                (#x8CE5 . #x5C7D)	;CJK UNIFIED IDEOGRAPH
                                (#x8CE6 . #x5C7E)	;CJK UNIFIED IDEOGRAPH
                                (#x8CE7 . #x5C80)	;CJK UNIFIED IDEOGRAPH
                                (#x8CE8 . #x5C83)	;CJK UNIFIED IDEOGRAPH
                                (#x8CE9 . #x5C84)	;CJK UNIFIED IDEOGRAPH
                                (#x8CEA . #x5C85)	;CJK UNIFIED IDEOGRAPH
                                (#x8CEB . #x5C86)	;CJK UNIFIED IDEOGRAPH
                                (#x8CEC . #x5C87)	;CJK UNIFIED IDEOGRAPH
                                (#x8CED . #x5C89)	;CJK UNIFIED IDEOGRAPH
                                (#x8CEE . #x5C8A)	;CJK UNIFIED IDEOGRAPH
                                (#x8CEF . #x5C8B)	;CJK UNIFIED IDEOGRAPH
                                (#x8CF0 . #x5C8E)	;CJK UNIFIED IDEOGRAPH
                                (#x8CF1 . #x5C8F)	;CJK UNIFIED IDEOGRAPH
                                (#x8CF2 . #x5C92)	;CJK UNIFIED IDEOGRAPH
                                (#x8CF3 . #x5C93)	;CJK UNIFIED IDEOGRAPH
                                (#x8CF4 . #x5C95)	;CJK UNIFIED IDEOGRAPH
                                (#x8CF5 . #x5C9D)	;CJK UNIFIED IDEOGRAPH
                                (#x8CF6 . #x5C9E)	;CJK UNIFIED IDEOGRAPH
                                (#x8CF7 . #x5C9F)	;CJK UNIFIED IDEOGRAPH
                                (#x8CF8 . #x5CA0)	;CJK UNIFIED IDEOGRAPH
                                (#x8CF9 . #x5CA1)	;CJK UNIFIED IDEOGRAPH
                                (#x8CFA . #x5CA4)	;CJK UNIFIED IDEOGRAPH
                                (#x8CFB . #x5CA5)	;CJK UNIFIED IDEOGRAPH
                                (#x8CFC . #x5CA6)	;CJK UNIFIED IDEOGRAPH
                                (#x8CFD . #x5CA7)	;CJK UNIFIED IDEOGRAPH
                                (#x8CFE . #x5CA8)	;CJK UNIFIED IDEOGRAPH
                                (#x8D40 . #x5CAA)	;CJK UNIFIED IDEOGRAPH
                                (#x8D41 . #x5CAE)	;CJK UNIFIED IDEOGRAPH
                                (#x8D42 . #x5CAF)	;CJK UNIFIED IDEOGRAPH
                                (#x8D43 . #x5CB0)	;CJK UNIFIED IDEOGRAPH
                                (#x8D44 . #x5CB2)	;CJK UNIFIED IDEOGRAPH
                                (#x8D45 . #x5CB4)	;CJK UNIFIED IDEOGRAPH
                                (#x8D46 . #x5CB6)	;CJK UNIFIED IDEOGRAPH
                                (#x8D47 . #x5CB9)	;CJK UNIFIED IDEOGRAPH
                                (#x8D48 . #x5CBA)	;CJK UNIFIED IDEOGRAPH
                                (#x8D49 . #x5CBB)	;CJK UNIFIED IDEOGRAPH
                                (#x8D4A . #x5CBC)	;CJK UNIFIED IDEOGRAPH
                                (#x8D4B . #x5CBE)	;CJK UNIFIED IDEOGRAPH
                                (#x8D4C . #x5CC0)	;CJK UNIFIED IDEOGRAPH
                                (#x8D4D . #x5CC2)	;CJK UNIFIED IDEOGRAPH
                                (#x8D4E . #x5CC3)	;CJK UNIFIED IDEOGRAPH
                                (#x8D4F . #x5CC5)	;CJK UNIFIED IDEOGRAPH
                                (#x8D50 . #x5CC6)	;CJK UNIFIED IDEOGRAPH
                                (#x8D51 . #x5CC7)	;CJK UNIFIED IDEOGRAPH
                                (#x8D52 . #x5CC8)	;CJK UNIFIED IDEOGRAPH
                                (#x8D53 . #x5CC9)	;CJK UNIFIED IDEOGRAPH
                                (#x8D54 . #x5CCA)	;CJK UNIFIED IDEOGRAPH
                                (#x8D55 . #x5CCC)	;CJK UNIFIED IDEOGRAPH
                                (#x8D56 . #x5CCD)	;CJK UNIFIED IDEOGRAPH
                                (#x8D57 . #x5CCE)	;CJK UNIFIED IDEOGRAPH
                                (#x8D58 . #x5CCF)	;CJK UNIFIED IDEOGRAPH
                                (#x8D59 . #x5CD0)	;CJK UNIFIED IDEOGRAPH
                                (#x8D5A . #x5CD1)	;CJK UNIFIED IDEOGRAPH
                                (#x8D5B . #x5CD3)	;CJK UNIFIED IDEOGRAPH
                                (#x8D5C . #x5CD4)	;CJK UNIFIED IDEOGRAPH
                                (#x8D5D . #x5CD5)	;CJK UNIFIED IDEOGRAPH
                                (#x8D5E . #x5CD6)	;CJK UNIFIED IDEOGRAPH
                                (#x8D5F . #x5CD7)	;CJK UNIFIED IDEOGRAPH
                                (#x8D60 . #x5CD8)	;CJK UNIFIED IDEOGRAPH
                                (#x8D61 . #x5CDA)	;CJK UNIFIED IDEOGRAPH
                                (#x8D62 . #x5CDB)	;CJK UNIFIED IDEOGRAPH
                                (#x8D63 . #x5CDC)	;CJK UNIFIED IDEOGRAPH
                                (#x8D64 . #x5CDD)	;CJK UNIFIED IDEOGRAPH
                                (#x8D65 . #x5CDE)	;CJK UNIFIED IDEOGRAPH
                                (#x8D66 . #x5CDF)	;CJK UNIFIED IDEOGRAPH
                                (#x8D67 . #x5CE0)	;CJK UNIFIED IDEOGRAPH
                                (#x8D68 . #x5CE2)	;CJK UNIFIED IDEOGRAPH
                                (#x8D69 . #x5CE3)	;CJK UNIFIED IDEOGRAPH
                                (#x8D6A . #x5CE7)	;CJK UNIFIED IDEOGRAPH
                                (#x8D6B . #x5CE9)	;CJK UNIFIED IDEOGRAPH
                                (#x8D6C . #x5CEB)	;CJK UNIFIED IDEOGRAPH
                                (#x8D6D . #x5CEC)	;CJK UNIFIED IDEOGRAPH
                                (#x8D6E . #x5CEE)	;CJK UNIFIED IDEOGRAPH
                                (#x8D6F . #x5CEF)	;CJK UNIFIED IDEOGRAPH
                                (#x8D70 . #x5CF1)	;CJK UNIFIED IDEOGRAPH
                                (#x8D71 . #x5CF2)	;CJK UNIFIED IDEOGRAPH
                                (#x8D72 . #x5CF3)	;CJK UNIFIED IDEOGRAPH
                                (#x8D73 . #x5CF4)	;CJK UNIFIED IDEOGRAPH
                                (#x8D74 . #x5CF5)	;CJK UNIFIED IDEOGRAPH
                                (#x8D75 . #x5CF6)	;CJK UNIFIED IDEOGRAPH
                                (#x8D76 . #x5CF7)	;CJK UNIFIED IDEOGRAPH
                                (#x8D77 . #x5CF8)	;CJK UNIFIED IDEOGRAPH
                                (#x8D78 . #x5CF9)	;CJK UNIFIED IDEOGRAPH
                                (#x8D79 . #x5CFA)	;CJK UNIFIED IDEOGRAPH
                                (#x8D7A . #x5CFC)	;CJK UNIFIED IDEOGRAPH
                                (#x8D7B . #x5CFD)	;CJK UNIFIED IDEOGRAPH
                                (#x8D7C . #x5CFE)	;CJK UNIFIED IDEOGRAPH
                                (#x8D7D . #x5CFF)	;CJK UNIFIED IDEOGRAPH
                                (#x8D7E . #x5D00)	;CJK UNIFIED IDEOGRAPH
                                (#x8D80 . #x5D01)	;CJK UNIFIED IDEOGRAPH
                                (#x8D81 . #x5D04)	;CJK UNIFIED IDEOGRAPH
                                (#x8D82 . #x5D05)	;CJK UNIFIED IDEOGRAPH
                                (#x8D83 . #x5D08)	;CJK UNIFIED IDEOGRAPH
                                (#x8D84 . #x5D09)	;CJK UNIFIED IDEOGRAPH
                                (#x8D85 . #x5D0A)	;CJK UNIFIED IDEOGRAPH
                                (#x8D86 . #x5D0B)	;CJK UNIFIED IDEOGRAPH
                                (#x8D87 . #x5D0C)	;CJK UNIFIED IDEOGRAPH
                                (#x8D88 . #x5D0D)	;CJK UNIFIED IDEOGRAPH
                                (#x8D89 . #x5D0F)	;CJK UNIFIED IDEOGRAPH
                                (#x8D8A . #x5D10)	;CJK UNIFIED IDEOGRAPH
                                (#x8D8B . #x5D11)	;CJK UNIFIED IDEOGRAPH
                                (#x8D8C . #x5D12)	;CJK UNIFIED IDEOGRAPH
                                (#x8D8D . #x5D13)	;CJK UNIFIED IDEOGRAPH
                                (#x8D8E . #x5D15)	;CJK UNIFIED IDEOGRAPH
                                (#x8D8F . #x5D17)	;CJK UNIFIED IDEOGRAPH
                                (#x8D90 . #x5D18)	;CJK UNIFIED IDEOGRAPH
                                (#x8D91 . #x5D19)	;CJK UNIFIED IDEOGRAPH
                                (#x8D92 . #x5D1A)	;CJK UNIFIED IDEOGRAPH
                                (#x8D93 . #x5D1C)	;CJK UNIFIED IDEOGRAPH
                                (#x8D94 . #x5D1D)	;CJK UNIFIED IDEOGRAPH
                                (#x8D95 . #x5D1F)	;CJK UNIFIED IDEOGRAPH
                                (#x8D96 . #x5D20)	;CJK UNIFIED IDEOGRAPH
                                (#x8D97 . #x5D21)	;CJK UNIFIED IDEOGRAPH
                                (#x8D98 . #x5D22)	;CJK UNIFIED IDEOGRAPH
                                (#x8D99 . #x5D23)	;CJK UNIFIED IDEOGRAPH
                                (#x8D9A . #x5D25)	;CJK UNIFIED IDEOGRAPH
                                (#x8D9B . #x5D28)	;CJK UNIFIED IDEOGRAPH
                                (#x8D9C . #x5D2A)	;CJK UNIFIED IDEOGRAPH
                                (#x8D9D . #x5D2B)	;CJK UNIFIED IDEOGRAPH
                                (#x8D9E . #x5D2C)	;CJK UNIFIED IDEOGRAPH
                                (#x8D9F . #x5D2F)	;CJK UNIFIED IDEOGRAPH
                                (#x8DA0 . #x5D30)	;CJK UNIFIED IDEOGRAPH
                                (#x8DA1 . #x5D31)	;CJK UNIFIED IDEOGRAPH
                                (#x8DA2 . #x5D32)	;CJK UNIFIED IDEOGRAPH
                                (#x8DA3 . #x5D33)	;CJK UNIFIED IDEOGRAPH
                                (#x8DA4 . #x5D35)	;CJK UNIFIED IDEOGRAPH
                                (#x8DA5 . #x5D36)	;CJK UNIFIED IDEOGRAPH
                                (#x8DA6 . #x5D37)	;CJK UNIFIED IDEOGRAPH
                                (#x8DA7 . #x5D38)	;CJK UNIFIED IDEOGRAPH
                                (#x8DA8 . #x5D39)	;CJK UNIFIED IDEOGRAPH
                                (#x8DA9 . #x5D3A)	;CJK UNIFIED IDEOGRAPH
                                (#x8DAA . #x5D3B)	;CJK UNIFIED IDEOGRAPH
                                (#x8DAB . #x5D3C)	;CJK UNIFIED IDEOGRAPH
                                (#x8DAC . #x5D3F)	;CJK UNIFIED IDEOGRAPH
                                (#x8DAD . #x5D40)	;CJK UNIFIED IDEOGRAPH
                                (#x8DAE . #x5D41)	;CJK UNIFIED IDEOGRAPH
                                (#x8DAF . #x5D42)	;CJK UNIFIED IDEOGRAPH
                                (#x8DB0 . #x5D43)	;CJK UNIFIED IDEOGRAPH
                                (#x8DB1 . #x5D44)	;CJK UNIFIED IDEOGRAPH
                                (#x8DB2 . #x5D45)	;CJK UNIFIED IDEOGRAPH
                                (#x8DB3 . #x5D46)	;CJK UNIFIED IDEOGRAPH
                                (#x8DB4 . #x5D48)	;CJK UNIFIED IDEOGRAPH
                                (#x8DB5 . #x5D49)	;CJK UNIFIED IDEOGRAPH
                                (#x8DB6 . #x5D4D)	;CJK UNIFIED IDEOGRAPH
                                (#x8DB7 . #x5D4E)	;CJK UNIFIED IDEOGRAPH
                                (#x8DB8 . #x5D4F)	;CJK UNIFIED IDEOGRAPH
                                (#x8DB9 . #x5D50)	;CJK UNIFIED IDEOGRAPH
                                (#x8DBA . #x5D51)	;CJK UNIFIED IDEOGRAPH
                                (#x8DBB . #x5D52)	;CJK UNIFIED IDEOGRAPH
                                (#x8DBC . #x5D53)	;CJK UNIFIED IDEOGRAPH
                                (#x8DBD . #x5D54)	;CJK UNIFIED IDEOGRAPH
                                (#x8DBE . #x5D55)	;CJK UNIFIED IDEOGRAPH
                                (#x8DBF . #x5D56)	;CJK UNIFIED IDEOGRAPH
                                (#x8DC0 . #x5D57)	;CJK UNIFIED IDEOGRAPH
                                (#x8DC1 . #x5D59)	;CJK UNIFIED IDEOGRAPH
                                (#x8DC2 . #x5D5A)	;CJK UNIFIED IDEOGRAPH
                                (#x8DC3 . #x5D5C)	;CJK UNIFIED IDEOGRAPH
                                (#x8DC4 . #x5D5E)	;CJK UNIFIED IDEOGRAPH
                                (#x8DC5 . #x5D5F)	;CJK UNIFIED IDEOGRAPH
                                (#x8DC6 . #x5D60)	;CJK UNIFIED IDEOGRAPH
                                (#x8DC7 . #x5D61)	;CJK UNIFIED IDEOGRAPH
                                (#x8DC8 . #x5D62)	;CJK UNIFIED IDEOGRAPH
                                (#x8DC9 . #x5D63)	;CJK UNIFIED IDEOGRAPH
                                (#x8DCA . #x5D64)	;CJK UNIFIED IDEOGRAPH
                                (#x8DCB . #x5D65)	;CJK UNIFIED IDEOGRAPH
                                (#x8DCC . #x5D66)	;CJK UNIFIED IDEOGRAPH
                                (#x8DCD . #x5D67)	;CJK UNIFIED IDEOGRAPH
                                (#x8DCE . #x5D68)	;CJK UNIFIED IDEOGRAPH
                                (#x8DCF . #x5D6A)	;CJK UNIFIED IDEOGRAPH
                                (#x8DD0 . #x5D6D)	;CJK UNIFIED IDEOGRAPH
                                (#x8DD1 . #x5D6E)	;CJK UNIFIED IDEOGRAPH
                                (#x8DD2 . #x5D70)	;CJK UNIFIED IDEOGRAPH
                                (#x8DD3 . #x5D71)	;CJK UNIFIED IDEOGRAPH
                                (#x8DD4 . #x5D72)	;CJK UNIFIED IDEOGRAPH
                                (#x8DD5 . #x5D73)	;CJK UNIFIED IDEOGRAPH
                                (#x8DD6 . #x5D75)	;CJK UNIFIED IDEOGRAPH
                                (#x8DD7 . #x5D76)	;CJK UNIFIED IDEOGRAPH
                                (#x8DD8 . #x5D77)	;CJK UNIFIED IDEOGRAPH
                                (#x8DD9 . #x5D78)	;CJK UNIFIED IDEOGRAPH
                                (#x8DDA . #x5D79)	;CJK UNIFIED IDEOGRAPH
                                (#x8DDB . #x5D7A)	;CJK UNIFIED IDEOGRAPH
                                (#x8DDC . #x5D7B)	;CJK UNIFIED IDEOGRAPH
                                (#x8DDD . #x5D7C)	;CJK UNIFIED IDEOGRAPH
                                (#x8DDE . #x5D7D)	;CJK UNIFIED IDEOGRAPH
                                (#x8DDF . #x5D7E)	;CJK UNIFIED IDEOGRAPH
                                (#x8DE0 . #x5D7F)	;CJK UNIFIED IDEOGRAPH
                                (#x8DE1 . #x5D80)	;CJK UNIFIED IDEOGRAPH
                                (#x8DE2 . #x5D81)	;CJK UNIFIED IDEOGRAPH
                                (#x8DE3 . #x5D83)	;CJK UNIFIED IDEOGRAPH
                                (#x8DE4 . #x5D84)	;CJK UNIFIED IDEOGRAPH
                                (#x8DE5 . #x5D85)	;CJK UNIFIED IDEOGRAPH
                                (#x8DE6 . #x5D86)	;CJK UNIFIED IDEOGRAPH
                                (#x8DE7 . #x5D87)	;CJK UNIFIED IDEOGRAPH
                                (#x8DE8 . #x5D88)	;CJK UNIFIED IDEOGRAPH
                                (#x8DE9 . #x5D89)	;CJK UNIFIED IDEOGRAPH
                                (#x8DEA . #x5D8A)	;CJK UNIFIED IDEOGRAPH
                                (#x8DEB . #x5D8B)	;CJK UNIFIED IDEOGRAPH
                                (#x8DEC . #x5D8C)	;CJK UNIFIED IDEOGRAPH
                                (#x8DED . #x5D8D)	;CJK UNIFIED IDEOGRAPH
                                (#x8DEE . #x5D8E)	;CJK UNIFIED IDEOGRAPH
                                (#x8DEF . #x5D8F)	;CJK UNIFIED IDEOGRAPH
                                (#x8DF0 . #x5D90)	;CJK UNIFIED IDEOGRAPH
                                (#x8DF1 . #x5D91)	;CJK UNIFIED IDEOGRAPH
                                (#x8DF2 . #x5D92)	;CJK UNIFIED IDEOGRAPH
                                (#x8DF3 . #x5D93)	;CJK UNIFIED IDEOGRAPH
                                (#x8DF4 . #x5D94)	;CJK UNIFIED IDEOGRAPH
                                (#x8DF5 . #x5D95)	;CJK UNIFIED IDEOGRAPH
                                (#x8DF6 . #x5D96)	;CJK UNIFIED IDEOGRAPH
                                (#x8DF7 . #x5D97)	;CJK UNIFIED IDEOGRAPH
                                (#x8DF8 . #x5D98)	;CJK UNIFIED IDEOGRAPH
                                (#x8DF9 . #x5D9A)	;CJK UNIFIED IDEOGRAPH
                                (#x8DFA . #x5D9B)	;CJK UNIFIED IDEOGRAPH
                                (#x8DFB . #x5D9C)	;CJK UNIFIED IDEOGRAPH
                                (#x8DFC . #x5D9E)	;CJK UNIFIED IDEOGRAPH
                                (#x8DFD . #x5D9F)	;CJK UNIFIED IDEOGRAPH
                                (#x8DFE . #x5DA0)	;CJK UNIFIED IDEOGRAPH
                                (#x8E40 . #x5DA1)	;CJK UNIFIED IDEOGRAPH
                                (#x8E41 . #x5DA2)	;CJK UNIFIED IDEOGRAPH
                                (#x8E42 . #x5DA3)	;CJK UNIFIED IDEOGRAPH
                                (#x8E43 . #x5DA4)	;CJK UNIFIED IDEOGRAPH
                                (#x8E44 . #x5DA5)	;CJK UNIFIED IDEOGRAPH
                                (#x8E45 . #x5DA6)	;CJK UNIFIED IDEOGRAPH
                                (#x8E46 . #x5DA7)	;CJK UNIFIED IDEOGRAPH
                                (#x8E47 . #x5DA8)	;CJK UNIFIED IDEOGRAPH
                                (#x8E48 . #x5DA9)	;CJK UNIFIED IDEOGRAPH
                                (#x8E49 . #x5DAA)	;CJK UNIFIED IDEOGRAPH
                                (#x8E4A . #x5DAB)	;CJK UNIFIED IDEOGRAPH
                                (#x8E4B . #x5DAC)	;CJK UNIFIED IDEOGRAPH
                                (#x8E4C . #x5DAD)	;CJK UNIFIED IDEOGRAPH
                                (#x8E4D . #x5DAE)	;CJK UNIFIED IDEOGRAPH
                                (#x8E4E . #x5DAF)	;CJK UNIFIED IDEOGRAPH
                                (#x8E4F . #x5DB0)	;CJK UNIFIED IDEOGRAPH
                                (#x8E50 . #x5DB1)	;CJK UNIFIED IDEOGRAPH
                                (#x8E51 . #x5DB2)	;CJK UNIFIED IDEOGRAPH
                                (#x8E52 . #x5DB3)	;CJK UNIFIED IDEOGRAPH
                                (#x8E53 . #x5DB4)	;CJK UNIFIED IDEOGRAPH
                                (#x8E54 . #x5DB5)	;CJK UNIFIED IDEOGRAPH
                                (#x8E55 . #x5DB6)	;CJK UNIFIED IDEOGRAPH
                                (#x8E56 . #x5DB8)	;CJK UNIFIED IDEOGRAPH
                                (#x8E57 . #x5DB9)	;CJK UNIFIED IDEOGRAPH
                                (#x8E58 . #x5DBA)	;CJK UNIFIED IDEOGRAPH
                                (#x8E59 . #x5DBB)	;CJK UNIFIED IDEOGRAPH
                                (#x8E5A . #x5DBC)	;CJK UNIFIED IDEOGRAPH
                                (#x8E5B . #x5DBD)	;CJK UNIFIED IDEOGRAPH
                                (#x8E5C . #x5DBE)	;CJK UNIFIED IDEOGRAPH
                                (#x8E5D . #x5DBF)	;CJK UNIFIED IDEOGRAPH
                                (#x8E5E . #x5DC0)	;CJK UNIFIED IDEOGRAPH
                                (#x8E5F . #x5DC1)	;CJK UNIFIED IDEOGRAPH
                                (#x8E60 . #x5DC2)	;CJK UNIFIED IDEOGRAPH
                                (#x8E61 . #x5DC3)	;CJK UNIFIED IDEOGRAPH
                                (#x8E62 . #x5DC4)	;CJK UNIFIED IDEOGRAPH
                                (#x8E63 . #x5DC6)	;CJK UNIFIED IDEOGRAPH
                                (#x8E64 . #x5DC7)	;CJK UNIFIED IDEOGRAPH
                                (#x8E65 . #x5DC8)	;CJK UNIFIED IDEOGRAPH
                                (#x8E66 . #x5DC9)	;CJK UNIFIED IDEOGRAPH
                                (#x8E67 . #x5DCA)	;CJK UNIFIED IDEOGRAPH
                                (#x8E68 . #x5DCB)	;CJK UNIFIED IDEOGRAPH
                                (#x8E69 . #x5DCC)	;CJK UNIFIED IDEOGRAPH
                                (#x8E6A . #x5DCE)	;CJK UNIFIED IDEOGRAPH
                                (#x8E6B . #x5DCF)	;CJK UNIFIED IDEOGRAPH
                                (#x8E6C . #x5DD0)	;CJK UNIFIED IDEOGRAPH
                                (#x8E6D . #x5DD1)	;CJK UNIFIED IDEOGRAPH
                                (#x8E6E . #x5DD2)	;CJK UNIFIED IDEOGRAPH
                                (#x8E6F . #x5DD3)	;CJK UNIFIED IDEOGRAPH
                                (#x8E70 . #x5DD4)	;CJK UNIFIED IDEOGRAPH
                                (#x8E71 . #x5DD5)	;CJK UNIFIED IDEOGRAPH
                                (#x8E72 . #x5DD6)	;CJK UNIFIED IDEOGRAPH
                                (#x8E73 . #x5DD7)	;CJK UNIFIED IDEOGRAPH
                                (#x8E74 . #x5DD8)	;CJK UNIFIED IDEOGRAPH
                                (#x8E75 . #x5DD9)	;CJK UNIFIED IDEOGRAPH
                                (#x8E76 . #x5DDA)	;CJK UNIFIED IDEOGRAPH
                                (#x8E77 . #x5DDC)	;CJK UNIFIED IDEOGRAPH
                                (#x8E78 . #x5DDF)	;CJK UNIFIED IDEOGRAPH
                                (#x8E79 . #x5DE0)	;CJK UNIFIED IDEOGRAPH
                                (#x8E7A . #x5DE3)	;CJK UNIFIED IDEOGRAPH
                                (#x8E7B . #x5DE4)	;CJK UNIFIED IDEOGRAPH
                                (#x8E7C . #x5DEA)	;CJK UNIFIED IDEOGRAPH
                                (#x8E7D . #x5DEC)	;CJK UNIFIED IDEOGRAPH
                                (#x8E7E . #x5DED)	;CJK UNIFIED IDEOGRAPH
                                (#x8E80 . #x5DF0)	;CJK UNIFIED IDEOGRAPH
                                (#x8E81 . #x5DF5)	;CJK UNIFIED IDEOGRAPH
                                (#x8E82 . #x5DF6)	;CJK UNIFIED IDEOGRAPH
                                (#x8E83 . #x5DF8)	;CJK UNIFIED IDEOGRAPH
                                (#x8E84 . #x5DF9)	;CJK UNIFIED IDEOGRAPH
                                (#x8E85 . #x5DFA)	;CJK UNIFIED IDEOGRAPH
                                (#x8E86 . #x5DFB)	;CJK UNIFIED IDEOGRAPH
                                (#x8E87 . #x5DFC)	;CJK UNIFIED IDEOGRAPH
                                (#x8E88 . #x5DFF)	;CJK UNIFIED IDEOGRAPH
                                (#x8E89 . #x5E00)	;CJK UNIFIED IDEOGRAPH
                                (#x8E8A . #x5E04)	;CJK UNIFIED IDEOGRAPH
                                (#x8E8B . #x5E07)	;CJK UNIFIED IDEOGRAPH
                                (#x8E8C . #x5E09)	;CJK UNIFIED IDEOGRAPH
                                (#x8E8D . #x5E0A)	;CJK UNIFIED IDEOGRAPH
                                (#x8E8E . #x5E0B)	;CJK UNIFIED IDEOGRAPH
                                (#x8E8F . #x5E0D)	;CJK UNIFIED IDEOGRAPH
                                (#x8E90 . #x5E0E)	;CJK UNIFIED IDEOGRAPH
                                (#x8E91 . #x5E12)	;CJK UNIFIED IDEOGRAPH
                                (#x8E92 . #x5E13)	;CJK UNIFIED IDEOGRAPH
                                (#x8E93 . #x5E17)	;CJK UNIFIED IDEOGRAPH
                                (#x8E94 . #x5E1E)	;CJK UNIFIED IDEOGRAPH
                                (#x8E95 . #x5E1F)	;CJK UNIFIED IDEOGRAPH
                                (#x8E96 . #x5E20)	;CJK UNIFIED IDEOGRAPH
                                (#x8E97 . #x5E21)	;CJK UNIFIED IDEOGRAPH
                                (#x8E98 . #x5E22)	;CJK UNIFIED IDEOGRAPH
                                (#x8E99 . #x5E23)	;CJK UNIFIED IDEOGRAPH
                                (#x8E9A . #x5E24)	;CJK UNIFIED IDEOGRAPH
                                (#x8E9B . #x5E25)	;CJK UNIFIED IDEOGRAPH
                                (#x8E9C . #x5E28)	;CJK UNIFIED IDEOGRAPH
                                (#x8E9D . #x5E29)	;CJK UNIFIED IDEOGRAPH
                                (#x8E9E . #x5E2A)	;CJK UNIFIED IDEOGRAPH
                                (#x8E9F . #x5E2B)	;CJK UNIFIED IDEOGRAPH
                                (#x8EA0 . #x5E2C)	;CJK UNIFIED IDEOGRAPH
                                (#x8EA1 . #x5E2F)	;CJK UNIFIED IDEOGRAPH
                                (#x8EA2 . #x5E30)	;CJK UNIFIED IDEOGRAPH
                                (#x8EA3 . #x5E32)	;CJK UNIFIED IDEOGRAPH
                                (#x8EA4 . #x5E33)	;CJK UNIFIED IDEOGRAPH
                                (#x8EA5 . #x5E34)	;CJK UNIFIED IDEOGRAPH
                                (#x8EA6 . #x5E35)	;CJK UNIFIED IDEOGRAPH
                                (#x8EA7 . #x5E36)	;CJK UNIFIED IDEOGRAPH
                                (#x8EA8 . #x5E39)	;CJK UNIFIED IDEOGRAPH
                                (#x8EA9 . #x5E3A)	;CJK UNIFIED IDEOGRAPH
                                (#x8EAA . #x5E3E)	;CJK UNIFIED IDEOGRAPH
                                (#x8EAB . #x5E3F)	;CJK UNIFIED IDEOGRAPH
                                (#x8EAC . #x5E40)	;CJK UNIFIED IDEOGRAPH
                                (#x8EAD . #x5E41)	;CJK UNIFIED IDEOGRAPH
                                (#x8EAE . #x5E43)	;CJK UNIFIED IDEOGRAPH
                                (#x8EAF . #x5E46)	;CJK UNIFIED IDEOGRAPH
                                (#x8EB0 . #x5E47)	;CJK UNIFIED IDEOGRAPH
                                (#x8EB1 . #x5E48)	;CJK UNIFIED IDEOGRAPH
                                (#x8EB2 . #x5E49)	;CJK UNIFIED IDEOGRAPH
                                (#x8EB3 . #x5E4A)	;CJK UNIFIED IDEOGRAPH
                                (#x8EB4 . #x5E4B)	;CJK UNIFIED IDEOGRAPH
                                (#x8EB5 . #x5E4D)	;CJK UNIFIED IDEOGRAPH
                                (#x8EB6 . #x5E4E)	;CJK UNIFIED IDEOGRAPH
                                (#x8EB7 . #x5E4F)	;CJK UNIFIED IDEOGRAPH
                                (#x8EB8 . #x5E50)	;CJK UNIFIED IDEOGRAPH
                                (#x8EB9 . #x5E51)	;CJK UNIFIED IDEOGRAPH
                                (#x8EBA . #x5E52)	;CJK UNIFIED IDEOGRAPH
                                (#x8EBB . #x5E53)	;CJK UNIFIED IDEOGRAPH
                                (#x8EBC . #x5E56)	;CJK UNIFIED IDEOGRAPH
                                (#x8EBD . #x5E57)	;CJK UNIFIED IDEOGRAPH
                                (#x8EBE . #x5E58)	;CJK UNIFIED IDEOGRAPH
                                (#x8EBF . #x5E59)	;CJK UNIFIED IDEOGRAPH
                                (#x8EC0 . #x5E5A)	;CJK UNIFIED IDEOGRAPH
                                (#x8EC1 . #x5E5C)	;CJK UNIFIED IDEOGRAPH
                                (#x8EC2 . #x5E5D)	;CJK UNIFIED IDEOGRAPH
                                (#x8EC3 . #x5E5F)	;CJK UNIFIED IDEOGRAPH
                                (#x8EC4 . #x5E60)	;CJK UNIFIED IDEOGRAPH
                                (#x8EC5 . #x5E63)	;CJK UNIFIED IDEOGRAPH
                                (#x8EC6 . #x5E64)	;CJK UNIFIED IDEOGRAPH
                                (#x8EC7 . #x5E65)	;CJK UNIFIED IDEOGRAPH
                                (#x8EC8 . #x5E66)	;CJK UNIFIED IDEOGRAPH
                                (#x8EC9 . #x5E67)	;CJK UNIFIED IDEOGRAPH
                                (#x8ECA . #x5E68)	;CJK UNIFIED IDEOGRAPH
                                (#x8ECB . #x5E69)	;CJK UNIFIED IDEOGRAPH
                                (#x8ECC . #x5E6A)	;CJK UNIFIED IDEOGRAPH
                                (#x8ECD . #x5E6B)	;CJK UNIFIED IDEOGRAPH
                                (#x8ECE . #x5E6C)	;CJK UNIFIED IDEOGRAPH
                                (#x8ECF . #x5E6D)	;CJK UNIFIED IDEOGRAPH
                                (#x8ED0 . #x5E6E)	;CJK UNIFIED IDEOGRAPH
                                (#x8ED1 . #x5E6F)	;CJK UNIFIED IDEOGRAPH
                                (#x8ED2 . #x5E70)	;CJK UNIFIED IDEOGRAPH
                                (#x8ED3 . #x5E71)	;CJK UNIFIED IDEOGRAPH
                                (#x8ED4 . #x5E75)	;CJK UNIFIED IDEOGRAPH
                                (#x8ED5 . #x5E77)	;CJK UNIFIED IDEOGRAPH
                                (#x8ED6 . #x5E79)	;CJK UNIFIED IDEOGRAPH
                                (#x8ED7 . #x5E7E)	;CJK UNIFIED IDEOGRAPH
                                (#x8ED8 . #x5E81)	;CJK UNIFIED IDEOGRAPH
                                (#x8ED9 . #x5E82)	;CJK UNIFIED IDEOGRAPH
                                (#x8EDA . #x5E83)	;CJK UNIFIED IDEOGRAPH
                                (#x8EDB . #x5E85)	;CJK UNIFIED IDEOGRAPH
                                (#x8EDC . #x5E88)	;CJK UNIFIED IDEOGRAPH
                                (#x8EDD . #x5E89)	;CJK UNIFIED IDEOGRAPH
                                (#x8EDE . #x5E8C)	;CJK UNIFIED IDEOGRAPH
                                (#x8EDF . #x5E8D)	;CJK UNIFIED IDEOGRAPH
                                (#x8EE0 . #x5E8E)	;CJK UNIFIED IDEOGRAPH
                                (#x8EE1 . #x5E92)	;CJK UNIFIED IDEOGRAPH
                                (#x8EE2 . #x5E98)	;CJK UNIFIED IDEOGRAPH
                                (#x8EE3 . #x5E9B)	;CJK UNIFIED IDEOGRAPH
                                (#x8EE4 . #x5E9D)	;CJK UNIFIED IDEOGRAPH
                                (#x8EE5 . #x5EA1)	;CJK UNIFIED IDEOGRAPH
                                (#x8EE6 . #x5EA2)	;CJK UNIFIED IDEOGRAPH
                                (#x8EE7 . #x5EA3)	;CJK UNIFIED IDEOGRAPH
                                (#x8EE8 . #x5EA4)	;CJK UNIFIED IDEOGRAPH
                                (#x8EE9 . #x5EA8)	;CJK UNIFIED IDEOGRAPH
                                (#x8EEA . #x5EA9)	;CJK UNIFIED IDEOGRAPH
                                (#x8EEB . #x5EAA)	;CJK UNIFIED IDEOGRAPH
                                (#x8EEC . #x5EAB)	;CJK UNIFIED IDEOGRAPH
                                (#x8EED . #x5EAC)	;CJK UNIFIED IDEOGRAPH
                                (#x8EEE . #x5EAE)	;CJK UNIFIED IDEOGRAPH
                                (#x8EEF . #x5EAF)	;CJK UNIFIED IDEOGRAPH
                                (#x8EF0 . #x5EB0)	;CJK UNIFIED IDEOGRAPH
                                (#x8EF1 . #x5EB1)	;CJK UNIFIED IDEOGRAPH
                                (#x8EF2 . #x5EB2)	;CJK UNIFIED IDEOGRAPH
                                (#x8EF3 . #x5EB4)	;CJK UNIFIED IDEOGRAPH
                                (#x8EF4 . #x5EBA)	;CJK UNIFIED IDEOGRAPH
                                (#x8EF5 . #x5EBB)	;CJK UNIFIED IDEOGRAPH
                                (#x8EF6 . #x5EBC)	;CJK UNIFIED IDEOGRAPH
                                (#x8EF7 . #x5EBD)	;CJK UNIFIED IDEOGRAPH
                                (#x8EF8 . #x5EBF)	;CJK UNIFIED IDEOGRAPH
                                (#x8EF9 . #x5EC0)	;CJK UNIFIED IDEOGRAPH
                                (#x8EFA . #x5EC1)	;CJK UNIFIED IDEOGRAPH
                                (#x8EFB . #x5EC2)	;CJK UNIFIED IDEOGRAPH
                                (#x8EFC . #x5EC3)	;CJK UNIFIED IDEOGRAPH
                                (#x8EFD . #x5EC4)	;CJK UNIFIED IDEOGRAPH
                                (#x8EFE . #x5EC5)	;CJK UNIFIED IDEOGRAPH
                                (#x8F40 . #x5EC6)	;CJK UNIFIED IDEOGRAPH
                                (#x8F41 . #x5EC7)	;CJK UNIFIED IDEOGRAPH
                                (#x8F42 . #x5EC8)	;CJK UNIFIED IDEOGRAPH
                                (#x8F43 . #x5ECB)	;CJK UNIFIED IDEOGRAPH
                                (#x8F44 . #x5ECC)	;CJK UNIFIED IDEOGRAPH
                                (#x8F45 . #x5ECD)	;CJK UNIFIED IDEOGRAPH
                                (#x8F46 . #x5ECE)	;CJK UNIFIED IDEOGRAPH
                                (#x8F47 . #x5ECF)	;CJK UNIFIED IDEOGRAPH
                                (#x8F48 . #x5ED0)	;CJK UNIFIED IDEOGRAPH
                                (#x8F49 . #x5ED4)	;CJK UNIFIED IDEOGRAPH
                                (#x8F4A . #x5ED5)	;CJK UNIFIED IDEOGRAPH
                                (#x8F4B . #x5ED7)	;CJK UNIFIED IDEOGRAPH
                                (#x8F4C . #x5ED8)	;CJK UNIFIED IDEOGRAPH
                                (#x8F4D . #x5ED9)	;CJK UNIFIED IDEOGRAPH
                                (#x8F4E . #x5EDA)	;CJK UNIFIED IDEOGRAPH
                                (#x8F4F . #x5EDC)	;CJK UNIFIED IDEOGRAPH
                                (#x8F50 . #x5EDD)	;CJK UNIFIED IDEOGRAPH
                                (#x8F51 . #x5EDE)	;CJK UNIFIED IDEOGRAPH
                                (#x8F52 . #x5EDF)	;CJK UNIFIED IDEOGRAPH
                                (#x8F53 . #x5EE0)	;CJK UNIFIED IDEOGRAPH
                                (#x8F54 . #x5EE1)	;CJK UNIFIED IDEOGRAPH
                                (#x8F55 . #x5EE2)	;CJK UNIFIED IDEOGRAPH
                                (#x8F56 . #x5EE3)	;CJK UNIFIED IDEOGRAPH
                                (#x8F57 . #x5EE4)	;CJK UNIFIED IDEOGRAPH
                                (#x8F58 . #x5EE5)	;CJK UNIFIED IDEOGRAPH
                                (#x8F59 . #x5EE6)	;CJK UNIFIED IDEOGRAPH
                                (#x8F5A . #x5EE7)	;CJK UNIFIED IDEOGRAPH
                                (#x8F5B . #x5EE9)	;CJK UNIFIED IDEOGRAPH
                                (#x8F5C . #x5EEB)	;CJK UNIFIED IDEOGRAPH
                                (#x8F5D . #x5EEC)	;CJK UNIFIED IDEOGRAPH
                                (#x8F5E . #x5EED)	;CJK UNIFIED IDEOGRAPH
                                (#x8F5F . #x5EEE)	;CJK UNIFIED IDEOGRAPH
                                (#x8F60 . #x5EEF)	;CJK UNIFIED IDEOGRAPH
                                (#x8F61 . #x5EF0)	;CJK UNIFIED IDEOGRAPH
                                (#x8F62 . #x5EF1)	;CJK UNIFIED IDEOGRAPH
                                (#x8F63 . #x5EF2)	;CJK UNIFIED IDEOGRAPH
                                (#x8F64 . #x5EF3)	;CJK UNIFIED IDEOGRAPH
                                (#x8F65 . #x5EF5)	;CJK UNIFIED IDEOGRAPH
                                (#x8F66 . #x5EF8)	;CJK UNIFIED IDEOGRAPH
                                (#x8F67 . #x5EF9)	;CJK UNIFIED IDEOGRAPH
                                (#x8F68 . #x5EFB)	;CJK UNIFIED IDEOGRAPH
                                (#x8F69 . #x5EFC)	;CJK UNIFIED IDEOGRAPH
                                (#x8F6A . #x5EFD)	;CJK UNIFIED IDEOGRAPH
                                (#x8F6B . #x5F05)	;CJK UNIFIED IDEOGRAPH
                                (#x8F6C . #x5F06)	;CJK UNIFIED IDEOGRAPH
                                (#x8F6D . #x5F07)	;CJK UNIFIED IDEOGRAPH
                                (#x8F6E . #x5F09)	;CJK UNIFIED IDEOGRAPH
                                (#x8F6F . #x5F0C)	;CJK UNIFIED IDEOGRAPH
                                (#x8F70 . #x5F0D)	;CJK UNIFIED IDEOGRAPH
                                (#x8F71 . #x5F0E)	;CJK UNIFIED IDEOGRAPH
                                (#x8F72 . #x5F10)	;CJK UNIFIED IDEOGRAPH
                                (#x8F73 . #x5F12)	;CJK UNIFIED IDEOGRAPH
                                (#x8F74 . #x5F14)	;CJK UNIFIED IDEOGRAPH
                                (#x8F75 . #x5F16)	;CJK UNIFIED IDEOGRAPH
                                (#x8F76 . #x5F19)	;CJK UNIFIED IDEOGRAPH
                                (#x8F77 . #x5F1A)	;CJK UNIFIED IDEOGRAPH
                                (#x8F78 . #x5F1C)	;CJK UNIFIED IDEOGRAPH
                                (#x8F79 . #x5F1D)	;CJK UNIFIED IDEOGRAPH
                                (#x8F7A . #x5F1E)	;CJK UNIFIED IDEOGRAPH
                                (#x8F7B . #x5F21)	;CJK UNIFIED IDEOGRAPH
                                (#x8F7C . #x5F22)	;CJK UNIFIED IDEOGRAPH
                                (#x8F7D . #x5F23)	;CJK UNIFIED IDEOGRAPH
                                (#x8F7E . #x5F24)	;CJK UNIFIED IDEOGRAPH
                                (#x8F80 . #x5F28)	;CJK UNIFIED IDEOGRAPH
                                (#x8F81 . #x5F2B)	;CJK UNIFIED IDEOGRAPH
                                (#x8F82 . #x5F2C)	;CJK UNIFIED IDEOGRAPH
                                (#x8F83 . #x5F2E)	;CJK UNIFIED IDEOGRAPH
                                (#x8F84 . #x5F30)	;CJK UNIFIED IDEOGRAPH
                                (#x8F85 . #x5F32)	;CJK UNIFIED IDEOGRAPH
                                (#x8F86 . #x5F33)	;CJK UNIFIED IDEOGRAPH
                                (#x8F87 . #x5F34)	;CJK UNIFIED IDEOGRAPH
                                (#x8F88 . #x5F35)	;CJK UNIFIED IDEOGRAPH
                                (#x8F89 . #x5F36)	;CJK UNIFIED IDEOGRAPH
                                (#x8F8A . #x5F37)	;CJK UNIFIED IDEOGRAPH
                                (#x8F8B . #x5F38)	;CJK UNIFIED IDEOGRAPH
                                (#x8F8C . #x5F3B)	;CJK UNIFIED IDEOGRAPH
                                (#x8F8D . #x5F3D)	;CJK UNIFIED IDEOGRAPH
                                (#x8F8E . #x5F3E)	;CJK UNIFIED IDEOGRAPH
                                (#x8F8F . #x5F3F)	;CJK UNIFIED IDEOGRAPH
                                (#x8F90 . #x5F41)	;CJK UNIFIED IDEOGRAPH
                                (#x8F91 . #x5F42)	;CJK UNIFIED IDEOGRAPH
                                (#x8F92 . #x5F43)	;CJK UNIFIED IDEOGRAPH
                                (#x8F93 . #x5F44)	;CJK UNIFIED IDEOGRAPH
                                (#x8F94 . #x5F45)	;CJK UNIFIED IDEOGRAPH
                                (#x8F95 . #x5F46)	;CJK UNIFIED IDEOGRAPH
                                (#x8F96 . #x5F47)	;CJK UNIFIED IDEOGRAPH
                                (#x8F97 . #x5F48)	;CJK UNIFIED IDEOGRAPH
                                (#x8F98 . #x5F49)	;CJK UNIFIED IDEOGRAPH
                                (#x8F99 . #x5F4A)	;CJK UNIFIED IDEOGRAPH
                                (#x8F9A . #x5F4B)	;CJK UNIFIED IDEOGRAPH
                                (#x8F9B . #x5F4C)	;CJK UNIFIED IDEOGRAPH
                                (#x8F9C . #x5F4D)	;CJK UNIFIED IDEOGRAPH
                                (#x8F9D . #x5F4E)	;CJK UNIFIED IDEOGRAPH
                                (#x8F9E . #x5F4F)	;CJK UNIFIED IDEOGRAPH
                                (#x8F9F . #x5F51)	;CJK UNIFIED IDEOGRAPH
                                (#x8FA0 . #x5F54)	;CJK UNIFIED IDEOGRAPH
                                (#x8FA1 . #x5F59)	;CJK UNIFIED IDEOGRAPH
                                (#x8FA2 . #x5F5A)	;CJK UNIFIED IDEOGRAPH
                                (#x8FA3 . #x5F5B)	;CJK UNIFIED IDEOGRAPH
                                (#x8FA4 . #x5F5C)	;CJK UNIFIED IDEOGRAPH
                                (#x8FA5 . #x5F5E)	;CJK UNIFIED IDEOGRAPH
                                (#x8FA6 . #x5F5F)	;CJK UNIFIED IDEOGRAPH
                                (#x8FA7 . #x5F60)	;CJK UNIFIED IDEOGRAPH
                                (#x8FA8 . #x5F63)	;CJK UNIFIED IDEOGRAPH
                                (#x8FA9 . #x5F65)	;CJK UNIFIED IDEOGRAPH
                                (#x8FAA . #x5F67)	;CJK UNIFIED IDEOGRAPH
                                (#x8FAB . #x5F68)	;CJK UNIFIED IDEOGRAPH
                                (#x8FAC . #x5F6B)	;CJK UNIFIED IDEOGRAPH
                                (#x8FAD . #x5F6E)	;CJK UNIFIED IDEOGRAPH
                                (#x8FAE . #x5F6F)	;CJK UNIFIED IDEOGRAPH
                                (#x8FAF . #x5F72)	;CJK UNIFIED IDEOGRAPH
                                (#x8FB0 . #x5F74)	;CJK UNIFIED IDEOGRAPH
                                (#x8FB1 . #x5F75)	;CJK UNIFIED IDEOGRAPH
                                (#x8FB2 . #x5F76)	;CJK UNIFIED IDEOGRAPH
                                (#x8FB3 . #x5F78)	;CJK UNIFIED IDEOGRAPH
                                (#x8FB4 . #x5F7A)	;CJK UNIFIED IDEOGRAPH
                                (#x8FB5 . #x5F7D)	;CJK UNIFIED IDEOGRAPH
                                (#x8FB6 . #x5F7E)	;CJK UNIFIED IDEOGRAPH
                                (#x8FB7 . #x5F7F)	;CJK UNIFIED IDEOGRAPH
                                (#x8FB8 . #x5F83)	;CJK UNIFIED IDEOGRAPH
                                (#x8FB9 . #x5F86)	;CJK UNIFIED IDEOGRAPH
                                (#x8FBA . #x5F8D)	;CJK UNIFIED IDEOGRAPH
                                (#x8FBB . #x5F8E)	;CJK UNIFIED IDEOGRAPH
                                (#x8FBC . #x5F8F)	;CJK UNIFIED IDEOGRAPH
                                (#x8FBD . #x5F91)	;CJK UNIFIED IDEOGRAPH
                                (#x8FBE . #x5F93)	;CJK UNIFIED IDEOGRAPH
                                (#x8FBF . #x5F94)	;CJK UNIFIED IDEOGRAPH
                                (#x8FC0 . #x5F96)	;CJK UNIFIED IDEOGRAPH
                                (#x8FC1 . #x5F9A)	;CJK UNIFIED IDEOGRAPH
                                (#x8FC2 . #x5F9B)	;CJK UNIFIED IDEOGRAPH
                                (#x8FC3 . #x5F9D)	;CJK UNIFIED IDEOGRAPH
                                (#x8FC4 . #x5F9E)	;CJK UNIFIED IDEOGRAPH
                                (#x8FC5 . #x5F9F)	;CJK UNIFIED IDEOGRAPH
                                (#x8FC6 . #x5FA0)	;CJK UNIFIED IDEOGRAPH
                                (#x8FC7 . #x5FA2)	;CJK UNIFIED IDEOGRAPH
                                (#x8FC8 . #x5FA3)	;CJK UNIFIED IDEOGRAPH
                                (#x8FC9 . #x5FA4)	;CJK UNIFIED IDEOGRAPH
                                (#x8FCA . #x5FA5)	;CJK UNIFIED IDEOGRAPH
                                (#x8FCB . #x5FA6)	;CJK UNIFIED IDEOGRAPH
                                (#x8FCC . #x5FA7)	;CJK UNIFIED IDEOGRAPH
                                (#x8FCD . #x5FA9)	;CJK UNIFIED IDEOGRAPH
                                (#x8FCE . #x5FAB)	;CJK UNIFIED IDEOGRAPH
                                (#x8FCF . #x5FAC)	;CJK UNIFIED IDEOGRAPH
                                (#x8FD0 . #x5FAF)	;CJK UNIFIED IDEOGRAPH
                                (#x8FD1 . #x5FB0)	;CJK UNIFIED IDEOGRAPH
                                (#x8FD2 . #x5FB1)	;CJK UNIFIED IDEOGRAPH
                                (#x8FD3 . #x5FB2)	;CJK UNIFIED IDEOGRAPH
                                (#x8FD4 . #x5FB3)	;CJK UNIFIED IDEOGRAPH
                                (#x8FD5 . #x5FB4)	;CJK UNIFIED IDEOGRAPH
                                (#x8FD6 . #x5FB6)	;CJK UNIFIED IDEOGRAPH
                                (#x8FD7 . #x5FB8)	;CJK UNIFIED IDEOGRAPH
                                (#x8FD8 . #x5FB9)	;CJK UNIFIED IDEOGRAPH
                                (#x8FD9 . #x5FBA)	;CJK UNIFIED IDEOGRAPH
                                (#x8FDA . #x5FBB)	;CJK UNIFIED IDEOGRAPH
                                (#x8FDB . #x5FBE)	;CJK UNIFIED IDEOGRAPH
                                (#x8FDC . #x5FBF)	;CJK UNIFIED IDEOGRAPH
                                (#x8FDD . #x5FC0)	;CJK UNIFIED IDEOGRAPH
                                (#x8FDE . #x5FC1)	;CJK UNIFIED IDEOGRAPH
                                (#x8FDF . #x5FC2)	;CJK UNIFIED IDEOGRAPH
                                (#x8FE0 . #x5FC7)	;CJK UNIFIED IDEOGRAPH
                                (#x8FE1 . #x5FC8)	;CJK UNIFIED IDEOGRAPH
                                (#x8FE2 . #x5FCA)	;CJK UNIFIED IDEOGRAPH
                                (#x8FE3 . #x5FCB)	;CJK UNIFIED IDEOGRAPH
                                (#x8FE4 . #x5FCE)	;CJK UNIFIED IDEOGRAPH
                                (#x8FE5 . #x5FD3)	;CJK UNIFIED IDEOGRAPH
                                (#x8FE6 . #x5FD4)	;CJK UNIFIED IDEOGRAPH
                                (#x8FE7 . #x5FD5)	;CJK UNIFIED IDEOGRAPH
                                (#x8FE8 . #x5FDA)	;CJK UNIFIED IDEOGRAPH
                                (#x8FE9 . #x5FDB)	;CJK UNIFIED IDEOGRAPH
                                (#x8FEA . #x5FDC)	;CJK UNIFIED IDEOGRAPH
                                (#x8FEB . #x5FDE)	;CJK UNIFIED IDEOGRAPH
                                (#x8FEC . #x5FDF)	;CJK UNIFIED IDEOGRAPH
                                (#x8FED . #x5FE2)	;CJK UNIFIED IDEOGRAPH
                                (#x8FEE . #x5FE3)	;CJK UNIFIED IDEOGRAPH
                                (#x8FEF . #x5FE5)	;CJK UNIFIED IDEOGRAPH
                                (#x8FF0 . #x5FE6)	;CJK UNIFIED IDEOGRAPH
                                (#x8FF1 . #x5FE8)	;CJK UNIFIED IDEOGRAPH
                                (#x8FF2 . #x5FE9)	;CJK UNIFIED IDEOGRAPH
                                (#x8FF3 . #x5FEC)	;CJK UNIFIED IDEOGRAPH
                                (#x8FF4 . #x5FEF)	;CJK UNIFIED IDEOGRAPH
                                (#x8FF5 . #x5FF0)	;CJK UNIFIED IDEOGRAPH
                                (#x8FF6 . #x5FF2)	;CJK UNIFIED IDEOGRAPH
                                (#x8FF7 . #x5FF3)	;CJK UNIFIED IDEOGRAPH
                                (#x8FF8 . #x5FF4)	;CJK UNIFIED IDEOGRAPH
                                (#x8FF9 . #x5FF6)	;CJK UNIFIED IDEOGRAPH
                                (#x8FFA . #x5FF7)	;CJK UNIFIED IDEOGRAPH
                                (#x8FFB . #x5FF9)	;CJK UNIFIED IDEOGRAPH
                                (#x8FFC . #x5FFA)	;CJK UNIFIED IDEOGRAPH
                                (#x8FFD . #x5FFC)	;CJK UNIFIED IDEOGRAPH
                                (#x8FFE . #x6007)	;CJK UNIFIED IDEOGRAPH
                                (#x9040 . #x6008)	;CJK UNIFIED IDEOGRAPH
                                (#x9041 . #x6009)	;CJK UNIFIED IDEOGRAPH
                                (#x9042 . #x600B)	;CJK UNIFIED IDEOGRAPH
                                (#x9043 . #x600C)	;CJK UNIFIED IDEOGRAPH
                                (#x9044 . #x6010)	;CJK UNIFIED IDEOGRAPH
                                (#x9045 . #x6011)	;CJK UNIFIED IDEOGRAPH
                                (#x9046 . #x6013)	;CJK UNIFIED IDEOGRAPH
                                (#x9047 . #x6017)	;CJK UNIFIED IDEOGRAPH
                                (#x9048 . #x6018)	;CJK UNIFIED IDEOGRAPH
                                (#x9049 . #x601A)	;CJK UNIFIED IDEOGRAPH
                                (#x904A . #x601E)	;CJK UNIFIED IDEOGRAPH
                                (#x904B . #x601F)	;CJK UNIFIED IDEOGRAPH
                                (#x904C . #x6022)	;CJK UNIFIED IDEOGRAPH
                                (#x904D . #x6023)	;CJK UNIFIED IDEOGRAPH
                                (#x904E . #x6024)	;CJK UNIFIED IDEOGRAPH
                                (#x904F . #x602C)	;CJK UNIFIED IDEOGRAPH
                                (#x9050 . #x602D)	;CJK UNIFIED IDEOGRAPH
                                (#x9051 . #x602E)	;CJK UNIFIED IDEOGRAPH
                                (#x9052 . #x6030)	;CJK UNIFIED IDEOGRAPH
                                (#x9053 . #x6031)	;CJK UNIFIED IDEOGRAPH
                                (#x9054 . #x6032)	;CJK UNIFIED IDEOGRAPH
                                (#x9055 . #x6033)	;CJK UNIFIED IDEOGRAPH
                                (#x9056 . #x6034)	;CJK UNIFIED IDEOGRAPH
                                (#x9057 . #x6036)	;CJK UNIFIED IDEOGRAPH
                                (#x9058 . #x6037)	;CJK UNIFIED IDEOGRAPH
                                (#x9059 . #x6038)	;CJK UNIFIED IDEOGRAPH
                                (#x905A . #x6039)	;CJK UNIFIED IDEOGRAPH
                                (#x905B . #x603A)	;CJK UNIFIED IDEOGRAPH
                                (#x905C . #x603D)	;CJK UNIFIED IDEOGRAPH
                                (#x905D . #x603E)	;CJK UNIFIED IDEOGRAPH
                                (#x905E . #x6040)	;CJK UNIFIED IDEOGRAPH
                                (#x905F . #x6044)	;CJK UNIFIED IDEOGRAPH
                                (#x9060 . #x6045)	;CJK UNIFIED IDEOGRAPH
                                (#x9061 . #x6046)	;CJK UNIFIED IDEOGRAPH
                                (#x9062 . #x6047)	;CJK UNIFIED IDEOGRAPH
                                (#x9063 . #x6048)	;CJK UNIFIED IDEOGRAPH
                                (#x9064 . #x6049)	;CJK UNIFIED IDEOGRAPH
                                (#x9065 . #x604A)	;CJK UNIFIED IDEOGRAPH
                                (#x9066 . #x604C)	;CJK UNIFIED IDEOGRAPH
                                (#x9067 . #x604E)	;CJK UNIFIED IDEOGRAPH
                                (#x9068 . #x604F)	;CJK UNIFIED IDEOGRAPH
                                (#x9069 . #x6051)	;CJK UNIFIED IDEOGRAPH
                                (#x906A . #x6053)	;CJK UNIFIED IDEOGRAPH
                                (#x906B . #x6054)	;CJK UNIFIED IDEOGRAPH
                                (#x906C . #x6056)	;CJK UNIFIED IDEOGRAPH
                                (#x906D . #x6057)	;CJK UNIFIED IDEOGRAPH
                                (#x906E . #x6058)	;CJK UNIFIED IDEOGRAPH
                                (#x906F . #x605B)	;CJK UNIFIED IDEOGRAPH
                                (#x9070 . #x605C)	;CJK UNIFIED IDEOGRAPH
                                (#x9071 . #x605E)	;CJK UNIFIED IDEOGRAPH
                                (#x9072 . #x605F)	;CJK UNIFIED IDEOGRAPH
                                (#x9073 . #x6060)	;CJK UNIFIED IDEOGRAPH
                                (#x9074 . #x6061)	;CJK UNIFIED IDEOGRAPH
                                (#x9075 . #x6065)	;CJK UNIFIED IDEOGRAPH
                                (#x9076 . #x6066)	;CJK UNIFIED IDEOGRAPH
                                (#x9077 . #x606E)	;CJK UNIFIED IDEOGRAPH
                                (#x9078 . #x6071)	;CJK UNIFIED IDEOGRAPH
                                (#x9079 . #x6072)	;CJK UNIFIED IDEOGRAPH
                                (#x907A . #x6074)	;CJK UNIFIED IDEOGRAPH
                                (#x907B . #x6075)	;CJK UNIFIED IDEOGRAPH
                                (#x907C . #x6077)	;CJK UNIFIED IDEOGRAPH
                                (#x907D . #x607E)	;CJK UNIFIED IDEOGRAPH
                                (#x907E . #x6080)	;CJK UNIFIED IDEOGRAPH
                                (#x9080 . #x6081)	;CJK UNIFIED IDEOGRAPH
                                (#x9081 . #x6082)	;CJK UNIFIED IDEOGRAPH
                                (#x9082 . #x6085)	;CJK UNIFIED IDEOGRAPH
                                (#x9083 . #x6086)	;CJK UNIFIED IDEOGRAPH
                                (#x9084 . #x6087)	;CJK UNIFIED IDEOGRAPH
                                (#x9085 . #x6088)	;CJK UNIFIED IDEOGRAPH
                                (#x9086 . #x608A)	;CJK UNIFIED IDEOGRAPH
                                (#x9087 . #x608B)	;CJK UNIFIED IDEOGRAPH
                                (#x9088 . #x608E)	;CJK UNIFIED IDEOGRAPH
                                (#x9089 . #x608F)	;CJK UNIFIED IDEOGRAPH
                                (#x908A . #x6090)	;CJK UNIFIED IDEOGRAPH
                                (#x908B . #x6091)	;CJK UNIFIED IDEOGRAPH
                                (#x908C . #x6093)	;CJK UNIFIED IDEOGRAPH
                                (#x908D . #x6095)	;CJK UNIFIED IDEOGRAPH
                                (#x908E . #x6097)	;CJK UNIFIED IDEOGRAPH
                                (#x908F . #x6098)	;CJK UNIFIED IDEOGRAPH
                                (#x9090 . #x6099)	;CJK UNIFIED IDEOGRAPH
                                (#x9091 . #x609C)	;CJK UNIFIED IDEOGRAPH
                                (#x9092 . #x609E)	;CJK UNIFIED IDEOGRAPH
                                (#x9093 . #x60A1)	;CJK UNIFIED IDEOGRAPH
                                (#x9094 . #x60A2)	;CJK UNIFIED IDEOGRAPH
                                (#x9095 . #x60A4)	;CJK UNIFIED IDEOGRAPH
                                (#x9096 . #x60A5)	;CJK UNIFIED IDEOGRAPH
                                (#x9097 . #x60A7)	;CJK UNIFIED IDEOGRAPH
                                (#x9098 . #x60A9)	;CJK UNIFIED IDEOGRAPH
                                (#x9099 . #x60AA)	;CJK UNIFIED IDEOGRAPH
                                (#x909A . #x60AE)	;CJK UNIFIED IDEOGRAPH
                                (#x909B . #x60B0)	;CJK UNIFIED IDEOGRAPH
                                (#x909C . #x60B3)	;CJK UNIFIED IDEOGRAPH
                                (#x909D . #x60B5)	;CJK UNIFIED IDEOGRAPH
                                (#x909E . #x60B6)	;CJK UNIFIED IDEOGRAPH
                                (#x909F . #x60B7)	;CJK UNIFIED IDEOGRAPH
                                (#x90A0 . #x60B9)	;CJK UNIFIED IDEOGRAPH
                                (#x90A1 . #x60BA)	;CJK UNIFIED IDEOGRAPH
                                (#x90A2 . #x60BD)	;CJK UNIFIED IDEOGRAPH
                                (#x90A3 . #x60BE)	;CJK UNIFIED IDEOGRAPH
                                (#x90A4 . #x60BF)	;CJK UNIFIED IDEOGRAPH
                                (#x90A5 . #x60C0)	;CJK UNIFIED IDEOGRAPH
                                (#x90A6 . #x60C1)	;CJK UNIFIED IDEOGRAPH
                                (#x90A7 . #x60C2)	;CJK UNIFIED IDEOGRAPH
                                (#x90A8 . #x60C3)	;CJK UNIFIED IDEOGRAPH
                                (#x90A9 . #x60C4)	;CJK UNIFIED IDEOGRAPH
                                (#x90AA . #x60C7)	;CJK UNIFIED IDEOGRAPH
                                (#x90AB . #x60C8)	;CJK UNIFIED IDEOGRAPH
                                (#x90AC . #x60C9)	;CJK UNIFIED IDEOGRAPH
                                (#x90AD . #x60CC)	;CJK UNIFIED IDEOGRAPH
                                (#x90AE . #x60CD)	;CJK UNIFIED IDEOGRAPH
                                (#x90AF . #x60CE)	;CJK UNIFIED IDEOGRAPH
                                (#x90B0 . #x60CF)	;CJK UNIFIED IDEOGRAPH
                                (#x90B1 . #x60D0)	;CJK UNIFIED IDEOGRAPH
                                (#x90B2 . #x60D2)	;CJK UNIFIED IDEOGRAPH
                                (#x90B3 . #x60D3)	;CJK UNIFIED IDEOGRAPH
                                (#x90B4 . #x60D4)	;CJK UNIFIED IDEOGRAPH
                                (#x90B5 . #x60D6)	;CJK UNIFIED IDEOGRAPH
                                (#x90B6 . #x60D7)	;CJK UNIFIED IDEOGRAPH
                                (#x90B7 . #x60D9)	;CJK UNIFIED IDEOGRAPH
                                (#x90B8 . #x60DB)	;CJK UNIFIED IDEOGRAPH
                                (#x90B9 . #x60DE)	;CJK UNIFIED IDEOGRAPH
                                (#x90BA . #x60E1)	;CJK UNIFIED IDEOGRAPH
                                (#x90BB . #x60E2)	;CJK UNIFIED IDEOGRAPH
                                (#x90BC . #x60E3)	;CJK UNIFIED IDEOGRAPH
                                (#x90BD . #x60E4)	;CJK UNIFIED IDEOGRAPH
                                (#x90BE . #x60E5)	;CJK UNIFIED IDEOGRAPH
                                (#x90BF . #x60EA)	;CJK UNIFIED IDEOGRAPH
                                (#x90C0 . #x60F1)	;CJK UNIFIED IDEOGRAPH
                                (#x90C1 . #x60F2)	;CJK UNIFIED IDEOGRAPH
                                (#x90C2 . #x60F5)	;CJK UNIFIED IDEOGRAPH
                                (#x90C3 . #x60F7)	;CJK UNIFIED IDEOGRAPH
                                (#x90C4 . #x60F8)	;CJK UNIFIED IDEOGRAPH
                                (#x90C5 . #x60FB)	;CJK UNIFIED IDEOGRAPH
                                (#x90C6 . #x60FC)	;CJK UNIFIED IDEOGRAPH
                                (#x90C7 . #x60FD)	;CJK UNIFIED IDEOGRAPH
                                (#x90C8 . #x60FE)	;CJK UNIFIED IDEOGRAPH
                                (#x90C9 . #x60FF)	;CJK UNIFIED IDEOGRAPH
                                (#x90CA . #x6102)	;CJK UNIFIED IDEOGRAPH
                                (#x90CB . #x6103)	;CJK UNIFIED IDEOGRAPH
                                (#x90CC . #x6104)	;CJK UNIFIED IDEOGRAPH
                                (#x90CD . #x6105)	;CJK UNIFIED IDEOGRAPH
                                (#x90CE . #x6107)	;CJK UNIFIED IDEOGRAPH
                                (#x90CF . #x610A)	;CJK UNIFIED IDEOGRAPH
                                (#x90D0 . #x610B)	;CJK UNIFIED IDEOGRAPH
                                (#x90D1 . #x610C)	;CJK UNIFIED IDEOGRAPH
                                (#x90D2 . #x6110)	;CJK UNIFIED IDEOGRAPH
                                (#x90D3 . #x6111)	;CJK UNIFIED IDEOGRAPH
                                (#x90D4 . #x6112)	;CJK UNIFIED IDEOGRAPH
                                (#x90D5 . #x6113)	;CJK UNIFIED IDEOGRAPH
                                (#x90D6 . #x6114)	;CJK UNIFIED IDEOGRAPH
                                (#x90D7 . #x6116)	;CJK UNIFIED IDEOGRAPH
                                (#x90D8 . #x6117)	;CJK UNIFIED IDEOGRAPH
                                (#x90D9 . #x6118)	;CJK UNIFIED IDEOGRAPH
                                (#x90DA . #x6119)	;CJK UNIFIED IDEOGRAPH
                                (#x90DB . #x611B)	;CJK UNIFIED IDEOGRAPH
                                (#x90DC . #x611C)	;CJK UNIFIED IDEOGRAPH
                                (#x90DD . #x611D)	;CJK UNIFIED IDEOGRAPH
                                (#x90DE . #x611E)	;CJK UNIFIED IDEOGRAPH
                                (#x90DF . #x6121)	;CJK UNIFIED IDEOGRAPH
                                (#x90E0 . #x6122)	;CJK UNIFIED IDEOGRAPH
                                (#x90E1 . #x6125)	;CJK UNIFIED IDEOGRAPH
                                (#x90E2 . #x6128)	;CJK UNIFIED IDEOGRAPH
                                (#x90E3 . #x6129)	;CJK UNIFIED IDEOGRAPH
                                (#x90E4 . #x612A)	;CJK UNIFIED IDEOGRAPH
                                (#x90E5 . #x612C)	;CJK UNIFIED IDEOGRAPH
                                (#x90E6 . #x612D)	;CJK UNIFIED IDEOGRAPH
                                (#x90E7 . #x612E)	;CJK UNIFIED IDEOGRAPH
                                (#x90E8 . #x612F)	;CJK UNIFIED IDEOGRAPH
                                (#x90E9 . #x6130)	;CJK UNIFIED IDEOGRAPH
                                (#x90EA . #x6131)	;CJK UNIFIED IDEOGRAPH
                                (#x90EB . #x6132)	;CJK UNIFIED IDEOGRAPH
                                (#x90EC . #x6133)	;CJK UNIFIED IDEOGRAPH
                                (#x90ED . #x6134)	;CJK UNIFIED IDEOGRAPH
                                (#x90EE . #x6135)	;CJK UNIFIED IDEOGRAPH
                                (#x90EF . #x6136)	;CJK UNIFIED IDEOGRAPH
                                (#x90F0 . #x6137)	;CJK UNIFIED IDEOGRAPH
                                (#x90F1 . #x6138)	;CJK UNIFIED IDEOGRAPH
                                (#x90F2 . #x6139)	;CJK UNIFIED IDEOGRAPH
                                (#x90F3 . #x613A)	;CJK UNIFIED IDEOGRAPH
                                (#x90F4 . #x613B)	;CJK UNIFIED IDEOGRAPH
                                (#x90F5 . #x613C)	;CJK UNIFIED IDEOGRAPH
                                (#x90F6 . #x613D)	;CJK UNIFIED IDEOGRAPH
                                (#x90F7 . #x613E)	;CJK UNIFIED IDEOGRAPH
                                (#x90F8 . #x6140)	;CJK UNIFIED IDEOGRAPH
                                (#x90F9 . #x6141)	;CJK UNIFIED IDEOGRAPH
                                (#x90FA . #x6142)	;CJK UNIFIED IDEOGRAPH
                                (#x90FB . #x6143)	;CJK UNIFIED IDEOGRAPH
                                (#x90FC . #x6144)	;CJK UNIFIED IDEOGRAPH
                                (#x90FD . #x6145)	;CJK UNIFIED IDEOGRAPH
                                (#x90FE . #x6146)	;CJK UNIFIED IDEOGRAPH
                                (#x9140 . #x6147)	;CJK UNIFIED IDEOGRAPH
                                (#x9141 . #x6149)	;CJK UNIFIED IDEOGRAPH
                                (#x9142 . #x614B)	;CJK UNIFIED IDEOGRAPH
                                (#x9143 . #x614D)	;CJK UNIFIED IDEOGRAPH
                                (#x9144 . #x614F)	;CJK UNIFIED IDEOGRAPH
                                (#x9145 . #x6150)	;CJK UNIFIED IDEOGRAPH
                                (#x9146 . #x6152)	;CJK UNIFIED IDEOGRAPH
                                (#x9147 . #x6153)	;CJK UNIFIED IDEOGRAPH
                                (#x9148 . #x6154)	;CJK UNIFIED IDEOGRAPH
                                (#x9149 . #x6156)	;CJK UNIFIED IDEOGRAPH
                                (#x914A . #x6157)	;CJK UNIFIED IDEOGRAPH
                                (#x914B . #x6158)	;CJK UNIFIED IDEOGRAPH
                                (#x914C . #x6159)	;CJK UNIFIED IDEOGRAPH
                                (#x914D . #x615A)	;CJK UNIFIED IDEOGRAPH
                                (#x914E . #x615B)	;CJK UNIFIED IDEOGRAPH
                                (#x914F . #x615C)	;CJK UNIFIED IDEOGRAPH
                                (#x9150 . #x615E)	;CJK UNIFIED IDEOGRAPH
                                (#x9151 . #x615F)	;CJK UNIFIED IDEOGRAPH
                                (#x9152 . #x6160)	;CJK UNIFIED IDEOGRAPH
                                (#x9153 . #x6161)	;CJK UNIFIED IDEOGRAPH
                                (#x9154 . #x6163)	;CJK UNIFIED IDEOGRAPH
                                (#x9155 . #x6164)	;CJK UNIFIED IDEOGRAPH
                                (#x9156 . #x6165)	;CJK UNIFIED IDEOGRAPH
                                (#x9157 . #x6166)	;CJK UNIFIED IDEOGRAPH
                                (#x9158 . #x6169)	;CJK UNIFIED IDEOGRAPH
                                (#x9159 . #x616A)	;CJK UNIFIED IDEOGRAPH
                                (#x915A . #x616B)	;CJK UNIFIED IDEOGRAPH
                                (#x915B . #x616C)	;CJK UNIFIED IDEOGRAPH
                                (#x915C . #x616D)	;CJK UNIFIED IDEOGRAPH
                                (#x915D . #x616E)	;CJK UNIFIED IDEOGRAPH
                                (#x915E . #x616F)	;CJK UNIFIED IDEOGRAPH
                                (#x915F . #x6171)	;CJK UNIFIED IDEOGRAPH
                                (#x9160 . #x6172)	;CJK UNIFIED IDEOGRAPH
                                (#x9161 . #x6173)	;CJK UNIFIED IDEOGRAPH
                                (#x9162 . #x6174)	;CJK UNIFIED IDEOGRAPH
                                (#x9163 . #x6176)	;CJK UNIFIED IDEOGRAPH
                                (#x9164 . #x6178)	;CJK UNIFIED IDEOGRAPH
                                (#x9165 . #x6179)	;CJK UNIFIED IDEOGRAPH
                                (#x9166 . #x617A)	;CJK UNIFIED IDEOGRAPH
                                (#x9167 . #x617B)	;CJK UNIFIED IDEOGRAPH
                                (#x9168 . #x617C)	;CJK UNIFIED IDEOGRAPH
                                (#x9169 . #x617D)	;CJK UNIFIED IDEOGRAPH
                                (#x916A . #x617E)	;CJK UNIFIED IDEOGRAPH
                                (#x916B . #x617F)	;CJK UNIFIED IDEOGRAPH
                                (#x916C . #x6180)	;CJK UNIFIED IDEOGRAPH
                                (#x916D . #x6181)	;CJK UNIFIED IDEOGRAPH
                                (#x916E . #x6182)	;CJK UNIFIED IDEOGRAPH
                                (#x916F . #x6183)	;CJK UNIFIED IDEOGRAPH
                                (#x9170 . #x6184)	;CJK UNIFIED IDEOGRAPH
                                (#x9171 . #x6185)	;CJK UNIFIED IDEOGRAPH
                                (#x9172 . #x6186)	;CJK UNIFIED IDEOGRAPH
                                (#x9173 . #x6187)	;CJK UNIFIED IDEOGRAPH
                                (#x9174 . #x6188)	;CJK UNIFIED IDEOGRAPH
                                (#x9175 . #x6189)	;CJK UNIFIED IDEOGRAPH
                                (#x9176 . #x618A)	;CJK UNIFIED IDEOGRAPH
                                (#x9177 . #x618C)	;CJK UNIFIED IDEOGRAPH
                                (#x9178 . #x618D)	;CJK UNIFIED IDEOGRAPH
                                (#x9179 . #x618F)	;CJK UNIFIED IDEOGRAPH
                                (#x917A . #x6190)	;CJK UNIFIED IDEOGRAPH
                                (#x917B . #x6191)	;CJK UNIFIED IDEOGRAPH
                                (#x917C . #x6192)	;CJK UNIFIED IDEOGRAPH
                                (#x917D . #x6193)	;CJK UNIFIED IDEOGRAPH
                                (#x917E . #x6195)	;CJK UNIFIED IDEOGRAPH
                                (#x9180 . #x6196)	;CJK UNIFIED IDEOGRAPH
                                (#x9181 . #x6197)	;CJK UNIFIED IDEOGRAPH
                                (#x9182 . #x6198)	;CJK UNIFIED IDEOGRAPH
                                (#x9183 . #x6199)	;CJK UNIFIED IDEOGRAPH
                                (#x9184 . #x619A)	;CJK UNIFIED IDEOGRAPH
                                (#x9185 . #x619B)	;CJK UNIFIED IDEOGRAPH
                                (#x9186 . #x619C)	;CJK UNIFIED IDEOGRAPH
                                (#x9187 . #x619E)	;CJK UNIFIED IDEOGRAPH
                                (#x9188 . #x619F)	;CJK UNIFIED IDEOGRAPH
                                (#x9189 . #x61A0)	;CJK UNIFIED IDEOGRAPH
                                (#x918A . #x61A1)	;CJK UNIFIED IDEOGRAPH
                                (#x918B . #x61A2)	;CJK UNIFIED IDEOGRAPH
                                (#x918C . #x61A3)	;CJK UNIFIED IDEOGRAPH
                                (#x918D . #x61A4)	;CJK UNIFIED IDEOGRAPH
                                (#x918E . #x61A5)	;CJK UNIFIED IDEOGRAPH
                                (#x918F . #x61A6)	;CJK UNIFIED IDEOGRAPH
                                (#x9190 . #x61AA)	;CJK UNIFIED IDEOGRAPH
                                (#x9191 . #x61AB)	;CJK UNIFIED IDEOGRAPH
                                (#x9192 . #x61AD)	;CJK UNIFIED IDEOGRAPH
                                (#x9193 . #x61AE)	;CJK UNIFIED IDEOGRAPH
                                (#x9194 . #x61AF)	;CJK UNIFIED IDEOGRAPH
                                (#x9195 . #x61B0)	;CJK UNIFIED IDEOGRAPH
                                (#x9196 . #x61B1)	;CJK UNIFIED IDEOGRAPH
                                (#x9197 . #x61B2)	;CJK UNIFIED IDEOGRAPH
                                (#x9198 . #x61B3)	;CJK UNIFIED IDEOGRAPH
                                (#x9199 . #x61B4)	;CJK UNIFIED IDEOGRAPH
                                (#x919A . #x61B5)	;CJK UNIFIED IDEOGRAPH
                                (#x919B . #x61B6)	;CJK UNIFIED IDEOGRAPH
                                (#x919C . #x61B8)	;CJK UNIFIED IDEOGRAPH
                                (#x919D . #x61B9)	;CJK UNIFIED IDEOGRAPH
                                (#x919E . #x61BA)	;CJK UNIFIED IDEOGRAPH
                                (#x919F . #x61BB)	;CJK UNIFIED IDEOGRAPH
                                (#x91A0 . #x61BC)	;CJK UNIFIED IDEOGRAPH
                                (#x91A1 . #x61BD)	;CJK UNIFIED IDEOGRAPH
                                (#x91A2 . #x61BF)	;CJK UNIFIED IDEOGRAPH
                                (#x91A3 . #x61C0)	;CJK UNIFIED IDEOGRAPH
                                (#x91A4 . #x61C1)	;CJK UNIFIED IDEOGRAPH
                                (#x91A5 . #x61C3)	;CJK UNIFIED IDEOGRAPH
                                (#x91A6 . #x61C4)	;CJK UNIFIED IDEOGRAPH
                                (#x91A7 . #x61C5)	;CJK UNIFIED IDEOGRAPH
                                (#x91A8 . #x61C6)	;CJK UNIFIED IDEOGRAPH
                                (#x91A9 . #x61C7)	;CJK UNIFIED IDEOGRAPH
                                (#x91AA . #x61C9)	;CJK UNIFIED IDEOGRAPH
                                (#x91AB . #x61CC)	;CJK UNIFIED IDEOGRAPH
                                (#x91AC . #x61CD)	;CJK UNIFIED IDEOGRAPH
                                (#x91AD . #x61CE)	;CJK UNIFIED IDEOGRAPH
                                (#x91AE . #x61CF)	;CJK UNIFIED IDEOGRAPH
                                (#x91AF . #x61D0)	;CJK UNIFIED IDEOGRAPH
                                (#x91B0 . #x61D3)	;CJK UNIFIED IDEOGRAPH
                                (#x91B1 . #x61D5)	;CJK UNIFIED IDEOGRAPH
                                (#x91B2 . #x61D6)	;CJK UNIFIED IDEOGRAPH
                                (#x91B3 . #x61D7)	;CJK UNIFIED IDEOGRAPH
                                (#x91B4 . #x61D8)	;CJK UNIFIED IDEOGRAPH
                                (#x91B5 . #x61D9)	;CJK UNIFIED IDEOGRAPH
                                (#x91B6 . #x61DA)	;CJK UNIFIED IDEOGRAPH
                                (#x91B7 . #x61DB)	;CJK UNIFIED IDEOGRAPH
                                (#x91B8 . #x61DC)	;CJK UNIFIED IDEOGRAPH
                                (#x91B9 . #x61DD)	;CJK UNIFIED IDEOGRAPH
                                (#x91BA . #x61DE)	;CJK UNIFIED IDEOGRAPH
                                (#x91BB . #x61DF)	;CJK UNIFIED IDEOGRAPH
                                (#x91BC . #x61E0)	;CJK UNIFIED IDEOGRAPH
                                (#x91BD . #x61E1)	;CJK UNIFIED IDEOGRAPH
                                (#x91BE . #x61E2)	;CJK UNIFIED IDEOGRAPH
                                (#x91BF . #x61E3)	;CJK UNIFIED IDEOGRAPH
                                (#x91C0 . #x61E4)	;CJK UNIFIED IDEOGRAPH
                                (#x91C1 . #x61E5)	;CJK UNIFIED IDEOGRAPH
                                (#x91C2 . #x61E7)	;CJK UNIFIED IDEOGRAPH
                                (#x91C3 . #x61E8)	;CJK UNIFIED IDEOGRAPH
                                (#x91C4 . #x61E9)	;CJK UNIFIED IDEOGRAPH
                                (#x91C5 . #x61EA)	;CJK UNIFIED IDEOGRAPH
                                (#x91C6 . #x61EB)	;CJK UNIFIED IDEOGRAPH
                                (#x91C7 . #x61EC)	;CJK UNIFIED IDEOGRAPH
                                (#x91C8 . #x61ED)	;CJK UNIFIED IDEOGRAPH
                                (#x91C9 . #x61EE)	;CJK UNIFIED IDEOGRAPH
                                (#x91CA . #x61EF)	;CJK UNIFIED IDEOGRAPH
                                (#x91CB . #x61F0)	;CJK UNIFIED IDEOGRAPH
                                (#x91CC . #x61F1)	;CJK UNIFIED IDEOGRAPH
                                (#x91CD . #x61F2)	;CJK UNIFIED IDEOGRAPH
                                (#x91CE . #x61F3)	;CJK UNIFIED IDEOGRAPH
                                (#x91CF . #x61F4)	;CJK UNIFIED IDEOGRAPH
                                (#x91D0 . #x61F6)	;CJK UNIFIED IDEOGRAPH
                                (#x91D1 . #x61F7)	;CJK UNIFIED IDEOGRAPH
                                (#x91D2 . #x61F8)	;CJK UNIFIED IDEOGRAPH
                                (#x91D3 . #x61F9)	;CJK UNIFIED IDEOGRAPH
                                (#x91D4 . #x61FA)	;CJK UNIFIED IDEOGRAPH
                                (#x91D5 . #x61FB)	;CJK UNIFIED IDEOGRAPH
                                (#x91D6 . #x61FC)	;CJK UNIFIED IDEOGRAPH
                                (#x91D7 . #x61FD)	;CJK UNIFIED IDEOGRAPH
                                (#x91D8 . #x61FE)	;CJK UNIFIED IDEOGRAPH
                                (#x91D9 . #x6200)	;CJK UNIFIED IDEOGRAPH
                                (#x91DA . #x6201)	;CJK UNIFIED IDEOGRAPH
                                (#x91DB . #x6202)	;CJK UNIFIED IDEOGRAPH
                                (#x91DC . #x6203)	;CJK UNIFIED IDEOGRAPH
                                (#x91DD . #x6204)	;CJK UNIFIED IDEOGRAPH
                                (#x91DE . #x6205)	;CJK UNIFIED IDEOGRAPH
                                (#x91DF . #x6207)	;CJK UNIFIED IDEOGRAPH
                                (#x91E0 . #x6209)	;CJK UNIFIED IDEOGRAPH
                                (#x91E1 . #x6213)	;CJK UNIFIED IDEOGRAPH
                                (#x91E2 . #x6214)	;CJK UNIFIED IDEOGRAPH
                                (#x91E3 . #x6219)	;CJK UNIFIED IDEOGRAPH
                                (#x91E4 . #x621C)	;CJK UNIFIED IDEOGRAPH
                                (#x91E5 . #x621D)	;CJK UNIFIED IDEOGRAPH
                                (#x91E6 . #x621E)	;CJK UNIFIED IDEOGRAPH
                                (#x91E7 . #x6220)	;CJK UNIFIED IDEOGRAPH
                                (#x91E8 . #x6223)	;CJK UNIFIED IDEOGRAPH
                                (#x91E9 . #x6226)	;CJK UNIFIED IDEOGRAPH
                                (#x91EA . #x6227)	;CJK UNIFIED IDEOGRAPH
                                (#x91EB . #x6228)	;CJK UNIFIED IDEOGRAPH
                                (#x91EC . #x6229)	;CJK UNIFIED IDEOGRAPH
                                (#x91ED . #x622B)	;CJK UNIFIED IDEOGRAPH
                                (#x91EE . #x622D)	;CJK UNIFIED IDEOGRAPH
                                (#x91EF . #x622F)	;CJK UNIFIED IDEOGRAPH
                                (#x91F0 . #x6230)	;CJK UNIFIED IDEOGRAPH
                                (#x91F1 . #x6231)	;CJK UNIFIED IDEOGRAPH
                                (#x91F2 . #x6232)	;CJK UNIFIED IDEOGRAPH
                                (#x91F3 . #x6235)	;CJK UNIFIED IDEOGRAPH
                                (#x91F4 . #x6236)	;CJK UNIFIED IDEOGRAPH
                                (#x91F5 . #x6238)	;CJK UNIFIED IDEOGRAPH
                                (#x91F6 . #x6239)	;CJK UNIFIED IDEOGRAPH
                                (#x91F7 . #x623A)	;CJK UNIFIED IDEOGRAPH
                                (#x91F8 . #x623B)	;CJK UNIFIED IDEOGRAPH
                                (#x91F9 . #x623C)	;CJK UNIFIED IDEOGRAPH
                                (#x91FA . #x6242)	;CJK UNIFIED IDEOGRAPH
                                (#x91FB . #x6244)	;CJK UNIFIED IDEOGRAPH
                                (#x91FC . #x6245)	;CJK UNIFIED IDEOGRAPH
                                (#x91FD . #x6246)	;CJK UNIFIED IDEOGRAPH
                                (#x91FE . #x624A)	;CJK UNIFIED IDEOGRAPH
                                (#x9240 . #x624F)	;CJK UNIFIED IDEOGRAPH
                                (#x9241 . #x6250)	;CJK UNIFIED IDEOGRAPH
                                (#x9242 . #x6255)	;CJK UNIFIED IDEOGRAPH
                                (#x9243 . #x6256)	;CJK UNIFIED IDEOGRAPH
                                (#x9244 . #x6257)	;CJK UNIFIED IDEOGRAPH
                                (#x9245 . #x6259)	;CJK UNIFIED IDEOGRAPH
                                (#x9246 . #x625A)	;CJK UNIFIED IDEOGRAPH
                                (#x9247 . #x625C)	;CJK UNIFIED IDEOGRAPH
                                (#x9248 . #x625D)	;CJK UNIFIED IDEOGRAPH
                                (#x9249 . #x625E)	;CJK UNIFIED IDEOGRAPH
                                (#x924A . #x625F)	;CJK UNIFIED IDEOGRAPH
                                (#x924B . #x6260)	;CJK UNIFIED IDEOGRAPH
                                (#x924C . #x6261)	;CJK UNIFIED IDEOGRAPH
                                (#x924D . #x6262)	;CJK UNIFIED IDEOGRAPH
                                (#x924E . #x6264)	;CJK UNIFIED IDEOGRAPH
                                (#x924F . #x6265)	;CJK UNIFIED IDEOGRAPH
                                (#x9250 . #x6268)	;CJK UNIFIED IDEOGRAPH
                                (#x9251 . #x6271)	;CJK UNIFIED IDEOGRAPH
                                (#x9252 . #x6272)	;CJK UNIFIED IDEOGRAPH
                                (#x9253 . #x6274)	;CJK UNIFIED IDEOGRAPH
                                (#x9254 . #x6275)	;CJK UNIFIED IDEOGRAPH
                                (#x9255 . #x6277)	;CJK UNIFIED IDEOGRAPH
                                (#x9256 . #x6278)	;CJK UNIFIED IDEOGRAPH
                                (#x9257 . #x627A)	;CJK UNIFIED IDEOGRAPH
                                (#x9258 . #x627B)	;CJK UNIFIED IDEOGRAPH
                                (#x9259 . #x627D)	;CJK UNIFIED IDEOGRAPH
                                (#x925A . #x6281)	;CJK UNIFIED IDEOGRAPH
                                (#x925B . #x6282)	;CJK UNIFIED IDEOGRAPH
                                (#x925C . #x6283)	;CJK UNIFIED IDEOGRAPH
                                (#x925D . #x6285)	;CJK UNIFIED IDEOGRAPH
                                (#x925E . #x6286)	;CJK UNIFIED IDEOGRAPH
                                (#x925F . #x6287)	;CJK UNIFIED IDEOGRAPH
                                (#x9260 . #x6288)	;CJK UNIFIED IDEOGRAPH
                                (#x9261 . #x628B)	;CJK UNIFIED IDEOGRAPH
                                (#x9262 . #x628C)	;CJK UNIFIED IDEOGRAPH
                                (#x9263 . #x628D)	;CJK UNIFIED IDEOGRAPH
                                (#x9264 . #x628E)	;CJK UNIFIED IDEOGRAPH
                                (#x9265 . #x628F)	;CJK UNIFIED IDEOGRAPH
                                (#x9266 . #x6290)	;CJK UNIFIED IDEOGRAPH
                                (#x9267 . #x6294)	;CJK UNIFIED IDEOGRAPH
                                (#x9268 . #x6299)	;CJK UNIFIED IDEOGRAPH
                                (#x9269 . #x629C)	;CJK UNIFIED IDEOGRAPH
                                (#x926A . #x629D)	;CJK UNIFIED IDEOGRAPH
                                (#x926B . #x629E)	;CJK UNIFIED IDEOGRAPH
                                (#x926C . #x62A3)	;CJK UNIFIED IDEOGRAPH
                                (#x926D . #x62A6)	;CJK UNIFIED IDEOGRAPH
                                (#x926E . #x62A7)	;CJK UNIFIED IDEOGRAPH
                                (#x926F . #x62A9)	;CJK UNIFIED IDEOGRAPH
                                (#x9270 . #x62AA)	;CJK UNIFIED IDEOGRAPH
                                (#x9271 . #x62AD)	;CJK UNIFIED IDEOGRAPH
                                (#x9272 . #x62AE)	;CJK UNIFIED IDEOGRAPH
                                (#x9273 . #x62AF)	;CJK UNIFIED IDEOGRAPH
                                (#x9274 . #x62B0)	;CJK UNIFIED IDEOGRAPH
                                (#x9275 . #x62B2)	;CJK UNIFIED IDEOGRAPH
                                (#x9276 . #x62B3)	;CJK UNIFIED IDEOGRAPH
                                (#x9277 . #x62B4)	;CJK UNIFIED IDEOGRAPH
                                (#x9278 . #x62B6)	;CJK UNIFIED IDEOGRAPH
                                (#x9279 . #x62B7)	;CJK UNIFIED IDEOGRAPH
                                (#x927A . #x62B8)	;CJK UNIFIED IDEOGRAPH
                                (#x927B . #x62BA)	;CJK UNIFIED IDEOGRAPH
                                (#x927C . #x62BE)	;CJK UNIFIED IDEOGRAPH
                                (#x927D . #x62C0)	;CJK UNIFIED IDEOGRAPH
                                (#x927E . #x62C1)	;CJK UNIFIED IDEOGRAPH
                                (#x9280 . #x62C3)	;CJK UNIFIED IDEOGRAPH
                                (#x9281 . #x62CB)	;CJK UNIFIED IDEOGRAPH
                                (#x9282 . #x62CF)	;CJK UNIFIED IDEOGRAPH
                                (#x9283 . #x62D1)	;CJK UNIFIED IDEOGRAPH
                                (#x9284 . #x62D5)	;CJK UNIFIED IDEOGRAPH
                                (#x9285 . #x62DD)	;CJK UNIFIED IDEOGRAPH
                                (#x9286 . #x62DE)	;CJK UNIFIED IDEOGRAPH
                                (#x9287 . #x62E0)	;CJK UNIFIED IDEOGRAPH
                                (#x9288 . #x62E1)	;CJK UNIFIED IDEOGRAPH
                                (#x9289 . #x62E4)	;CJK UNIFIED IDEOGRAPH
                                (#x928A . #x62EA)	;CJK UNIFIED IDEOGRAPH
                                (#x928B . #x62EB)	;CJK UNIFIED IDEOGRAPH
                                (#x928C . #x62F0)	;CJK UNIFIED IDEOGRAPH
                                (#x928D . #x62F2)	;CJK UNIFIED IDEOGRAPH
                                (#x928E . #x62F5)	;CJK UNIFIED IDEOGRAPH
                                (#x928F . #x62F8)	;CJK UNIFIED IDEOGRAPH
                                (#x9290 . #x62F9)	;CJK UNIFIED IDEOGRAPH
                                (#x9291 . #x62FA)	;CJK UNIFIED IDEOGRAPH
                                (#x9292 . #x62FB)	;CJK UNIFIED IDEOGRAPH
                                (#x9293 . #x6300)	;CJK UNIFIED IDEOGRAPH
                                (#x9294 . #x6303)	;CJK UNIFIED IDEOGRAPH
                                (#x9295 . #x6304)	;CJK UNIFIED IDEOGRAPH
                                (#x9296 . #x6305)	;CJK UNIFIED IDEOGRAPH
                                (#x9297 . #x6306)	;CJK UNIFIED IDEOGRAPH
                                (#x9298 . #x630A)	;CJK UNIFIED IDEOGRAPH
                                (#x9299 . #x630B)	;CJK UNIFIED IDEOGRAPH
                                (#x929A . #x630C)	;CJK UNIFIED IDEOGRAPH
                                (#x929B . #x630D)	;CJK UNIFIED IDEOGRAPH
                                (#x929C . #x630F)	;CJK UNIFIED IDEOGRAPH
                                (#x929D . #x6310)	;CJK UNIFIED IDEOGRAPH
                                (#x929E . #x6312)	;CJK UNIFIED IDEOGRAPH
                                (#x929F . #x6313)	;CJK UNIFIED IDEOGRAPH
                                (#x92A0 . #x6314)	;CJK UNIFIED IDEOGRAPH
                                (#x92A1 . #x6315)	;CJK UNIFIED IDEOGRAPH
                                (#x92A2 . #x6317)	;CJK UNIFIED IDEOGRAPH
                                (#x92A3 . #x6318)	;CJK UNIFIED IDEOGRAPH
                                (#x92A4 . #x6319)	;CJK UNIFIED IDEOGRAPH
                                (#x92A5 . #x631C)	;CJK UNIFIED IDEOGRAPH
                                (#x92A6 . #x6326)	;CJK UNIFIED IDEOGRAPH
                                (#x92A7 . #x6327)	;CJK UNIFIED IDEOGRAPH
                                (#x92A8 . #x6329)	;CJK UNIFIED IDEOGRAPH
                                (#x92A9 . #x632C)	;CJK UNIFIED IDEOGRAPH
                                (#x92AA . #x632D)	;CJK UNIFIED IDEOGRAPH
                                (#x92AB . #x632E)	;CJK UNIFIED IDEOGRAPH
                                (#x92AC . #x6330)	;CJK UNIFIED IDEOGRAPH
                                (#x92AD . #x6331)	;CJK UNIFIED IDEOGRAPH
                                (#x92AE . #x6333)	;CJK UNIFIED IDEOGRAPH
                                (#x92AF . #x6334)	;CJK UNIFIED IDEOGRAPH
                                (#x92B0 . #x6335)	;CJK UNIFIED IDEOGRAPH
                                (#x92B1 . #x6336)	;CJK UNIFIED IDEOGRAPH
                                (#x92B2 . #x6337)	;CJK UNIFIED IDEOGRAPH
                                (#x92B3 . #x6338)	;CJK UNIFIED IDEOGRAPH
                                (#x92B4 . #x633B)	;CJK UNIFIED IDEOGRAPH
                                (#x92B5 . #x633C)	;CJK UNIFIED IDEOGRAPH
                                (#x92B6 . #x633E)	;CJK UNIFIED IDEOGRAPH
                                (#x92B7 . #x633F)	;CJK UNIFIED IDEOGRAPH
                                (#x92B8 . #x6340)	;CJK UNIFIED IDEOGRAPH
                                (#x92B9 . #x6341)	;CJK UNIFIED IDEOGRAPH
                                (#x92BA . #x6344)	;CJK UNIFIED IDEOGRAPH
                                (#x92BB . #x6347)	;CJK UNIFIED IDEOGRAPH
                                (#x92BC . #x6348)	;CJK UNIFIED IDEOGRAPH
                                (#x92BD . #x634A)	;CJK UNIFIED IDEOGRAPH
                                (#x92BE . #x6351)	;CJK UNIFIED IDEOGRAPH
                                (#x92BF . #x6352)	;CJK UNIFIED IDEOGRAPH
                                (#x92C0 . #x6353)	;CJK UNIFIED IDEOGRAPH
                                (#x92C1 . #x6354)	;CJK UNIFIED IDEOGRAPH
                                (#x92C2 . #x6356)	;CJK UNIFIED IDEOGRAPH
                                (#x92C3 . #x6357)	;CJK UNIFIED IDEOGRAPH
                                (#x92C4 . #x6358)	;CJK UNIFIED IDEOGRAPH
                                (#x92C5 . #x6359)	;CJK UNIFIED IDEOGRAPH
                                (#x92C6 . #x635A)	;CJK UNIFIED IDEOGRAPH
                                (#x92C7 . #x635B)	;CJK UNIFIED IDEOGRAPH
                                (#x92C8 . #x635C)	;CJK UNIFIED IDEOGRAPH
                                (#x92C9 . #x635D)	;CJK UNIFIED IDEOGRAPH
                                (#x92CA . #x6360)	;CJK UNIFIED IDEOGRAPH
                                (#x92CB . #x6364)	;CJK UNIFIED IDEOGRAPH
                                (#x92CC . #x6365)	;CJK UNIFIED IDEOGRAPH
                                (#x92CD . #x6366)	;CJK UNIFIED IDEOGRAPH
                                (#x92CE . #x6368)	;CJK UNIFIED IDEOGRAPH
                                (#x92CF . #x636A)	;CJK UNIFIED IDEOGRAPH
                                (#x92D0 . #x636B)	;CJK UNIFIED IDEOGRAPH
                                (#x92D1 . #x636C)	;CJK UNIFIED IDEOGRAPH
                                (#x92D2 . #x636F)	;CJK UNIFIED IDEOGRAPH
                                (#x92D3 . #x6370)	;CJK UNIFIED IDEOGRAPH
                                (#x92D4 . #x6372)	;CJK UNIFIED IDEOGRAPH
                                (#x92D5 . #x6373)	;CJK UNIFIED IDEOGRAPH
                                (#x92D6 . #x6374)	;CJK UNIFIED IDEOGRAPH
                                (#x92D7 . #x6375)	;CJK UNIFIED IDEOGRAPH
                                (#x92D8 . #x6378)	;CJK UNIFIED IDEOGRAPH
                                (#x92D9 . #x6379)	;CJK UNIFIED IDEOGRAPH
                                (#x92DA . #x637C)	;CJK UNIFIED IDEOGRAPH
                                (#x92DB . #x637D)	;CJK UNIFIED IDEOGRAPH
                                (#x92DC . #x637E)	;CJK UNIFIED IDEOGRAPH
                                (#x92DD . #x637F)	;CJK UNIFIED IDEOGRAPH
                                (#x92DE . #x6381)	;CJK UNIFIED IDEOGRAPH
                                (#x92DF . #x6383)	;CJK UNIFIED IDEOGRAPH
                                (#x92E0 . #x6384)	;CJK UNIFIED IDEOGRAPH
                                (#x92E1 . #x6385)	;CJK UNIFIED IDEOGRAPH
                                (#x92E2 . #x6386)	;CJK UNIFIED IDEOGRAPH
                                (#x92E3 . #x638B)	;CJK UNIFIED IDEOGRAPH
                                (#x92E4 . #x638D)	;CJK UNIFIED IDEOGRAPH
                                (#x92E5 . #x6391)	;CJK UNIFIED IDEOGRAPH
                                (#x92E6 . #x6393)	;CJK UNIFIED IDEOGRAPH
                                (#x92E7 . #x6394)	;CJK UNIFIED IDEOGRAPH
                                (#x92E8 . #x6395)	;CJK UNIFIED IDEOGRAPH
                                (#x92E9 . #x6397)	;CJK UNIFIED IDEOGRAPH
                                (#x92EA . #x6399)	;CJK UNIFIED IDEOGRAPH
                                (#x92EB . #x639A)	;CJK UNIFIED IDEOGRAPH
                                (#x92EC . #x639B)	;CJK UNIFIED IDEOGRAPH
                                (#x92ED . #x639C)	;CJK UNIFIED IDEOGRAPH
                                (#x92EE . #x639D)	;CJK UNIFIED IDEOGRAPH
                                (#x92EF . #x639E)	;CJK UNIFIED IDEOGRAPH
                                (#x92F0 . #x639F)	;CJK UNIFIED IDEOGRAPH
                                (#x92F1 . #x63A1)	;CJK UNIFIED IDEOGRAPH
                                (#x92F2 . #x63A4)	;CJK UNIFIED IDEOGRAPH
                                (#x92F3 . #x63A6)	;CJK UNIFIED IDEOGRAPH
                                (#x92F4 . #x63AB)	;CJK UNIFIED IDEOGRAPH
                                (#x92F5 . #x63AF)	;CJK UNIFIED IDEOGRAPH
                                (#x92F6 . #x63B1)	;CJK UNIFIED IDEOGRAPH
                                (#x92F7 . #x63B2)	;CJK UNIFIED IDEOGRAPH
                                (#x92F8 . #x63B5)	;CJK UNIFIED IDEOGRAPH
                                (#x92F9 . #x63B6)	;CJK UNIFIED IDEOGRAPH
                                (#x92FA . #x63B9)	;CJK UNIFIED IDEOGRAPH
                                (#x92FB . #x63BB)	;CJK UNIFIED IDEOGRAPH
                                (#x92FC . #x63BD)	;CJK UNIFIED IDEOGRAPH
                                (#x92FD . #x63BF)	;CJK UNIFIED IDEOGRAPH
                                (#x92FE . #x63C0)	;CJK UNIFIED IDEOGRAPH
                                (#x9340 . #x63C1)	;CJK UNIFIED IDEOGRAPH
                                (#x9341 . #x63C2)	;CJK UNIFIED IDEOGRAPH
                                (#x9342 . #x63C3)	;CJK UNIFIED IDEOGRAPH
                                (#x9343 . #x63C5)	;CJK UNIFIED IDEOGRAPH
                                (#x9344 . #x63C7)	;CJK UNIFIED IDEOGRAPH
                                (#x9345 . #x63C8)	;CJK UNIFIED IDEOGRAPH
                                (#x9346 . #x63CA)	;CJK UNIFIED IDEOGRAPH
                                (#x9347 . #x63CB)	;CJK UNIFIED IDEOGRAPH
                                (#x9348 . #x63CC)	;CJK UNIFIED IDEOGRAPH
                                (#x9349 . #x63D1)	;CJK UNIFIED IDEOGRAPH
                                (#x934A . #x63D3)	;CJK UNIFIED IDEOGRAPH
                                (#x934B . #x63D4)	;CJK UNIFIED IDEOGRAPH
                                (#x934C . #x63D5)	;CJK UNIFIED IDEOGRAPH
                                (#x934D . #x63D7)	;CJK UNIFIED IDEOGRAPH
                                (#x934E . #x63D8)	;CJK UNIFIED IDEOGRAPH
                                (#x934F . #x63D9)	;CJK UNIFIED IDEOGRAPH
                                (#x9350 . #x63DA)	;CJK UNIFIED IDEOGRAPH
                                (#x9351 . #x63DB)	;CJK UNIFIED IDEOGRAPH
                                (#x9352 . #x63DC)	;CJK UNIFIED IDEOGRAPH
                                (#x9353 . #x63DD)	;CJK UNIFIED IDEOGRAPH
                                (#x9354 . #x63DF)	;CJK UNIFIED IDEOGRAPH
                                (#x9355 . #x63E2)	;CJK UNIFIED IDEOGRAPH
                                (#x9356 . #x63E4)	;CJK UNIFIED IDEOGRAPH
                                (#x9357 . #x63E5)	;CJK UNIFIED IDEOGRAPH
                                (#x9358 . #x63E6)	;CJK UNIFIED IDEOGRAPH
                                (#x9359 . #x63E7)	;CJK UNIFIED IDEOGRAPH
                                (#x935A . #x63E8)	;CJK UNIFIED IDEOGRAPH
                                (#x935B . #x63EB)	;CJK UNIFIED IDEOGRAPH
                                (#x935C . #x63EC)	;CJK UNIFIED IDEOGRAPH
                                (#x935D . #x63EE)	;CJK UNIFIED IDEOGRAPH
                                (#x935E . #x63EF)	;CJK UNIFIED IDEOGRAPH
                                (#x935F . #x63F0)	;CJK UNIFIED IDEOGRAPH
                                (#x9360 . #x63F1)	;CJK UNIFIED IDEOGRAPH
                                (#x9361 . #x63F3)	;CJK UNIFIED IDEOGRAPH
                                (#x9362 . #x63F5)	;CJK UNIFIED IDEOGRAPH
                                (#x9363 . #x63F7)	;CJK UNIFIED IDEOGRAPH
                                (#x9364 . #x63F9)	;CJK UNIFIED IDEOGRAPH
                                (#x9365 . #x63FA)	;CJK UNIFIED IDEOGRAPH
                                (#x9366 . #x63FB)	;CJK UNIFIED IDEOGRAPH
                                (#x9367 . #x63FC)	;CJK UNIFIED IDEOGRAPH
                                (#x9368 . #x63FE)	;CJK UNIFIED IDEOGRAPH
                                (#x9369 . #x6403)	;CJK UNIFIED IDEOGRAPH
                                (#x936A . #x6404)	;CJK UNIFIED IDEOGRAPH
                                (#x936B . #x6406)	;CJK UNIFIED IDEOGRAPH
                                (#x936C . #x6407)	;CJK UNIFIED IDEOGRAPH
                                (#x936D . #x6408)	;CJK UNIFIED IDEOGRAPH
                                (#x936E . #x6409)	;CJK UNIFIED IDEOGRAPH
                                (#x936F . #x640A)	;CJK UNIFIED IDEOGRAPH
                                (#x9370 . #x640D)	;CJK UNIFIED IDEOGRAPH
                                (#x9371 . #x640E)	;CJK UNIFIED IDEOGRAPH
                                (#x9372 . #x6411)	;CJK UNIFIED IDEOGRAPH
                                (#x9373 . #x6412)	;CJK UNIFIED IDEOGRAPH
                                (#x9374 . #x6415)	;CJK UNIFIED IDEOGRAPH
                                (#x9375 . #x6416)	;CJK UNIFIED IDEOGRAPH
                                (#x9376 . #x6417)	;CJK UNIFIED IDEOGRAPH
                                (#x9377 . #x6418)	;CJK UNIFIED IDEOGRAPH
                                (#x9378 . #x6419)	;CJK UNIFIED IDEOGRAPH
                                (#x9379 . #x641A)	;CJK UNIFIED IDEOGRAPH
                                (#x937A . #x641D)	;CJK UNIFIED IDEOGRAPH
                                (#x937B . #x641F)	;CJK UNIFIED IDEOGRAPH
                                (#x937C . #x6422)	;CJK UNIFIED IDEOGRAPH
                                (#x937D . #x6423)	;CJK UNIFIED IDEOGRAPH
                                (#x937E . #x6424)	;CJK UNIFIED IDEOGRAPH
                                (#x9380 . #x6425)	;CJK UNIFIED IDEOGRAPH
                                (#x9381 . #x6427)	;CJK UNIFIED IDEOGRAPH
                                (#x9382 . #x6428)	;CJK UNIFIED IDEOGRAPH
                                (#x9383 . #x6429)	;CJK UNIFIED IDEOGRAPH
                                (#x9384 . #x642B)	;CJK UNIFIED IDEOGRAPH
                                (#x9385 . #x642E)	;CJK UNIFIED IDEOGRAPH
                                (#x9386 . #x642F)	;CJK UNIFIED IDEOGRAPH
                                (#x9387 . #x6430)	;CJK UNIFIED IDEOGRAPH
                                (#x9388 . #x6431)	;CJK UNIFIED IDEOGRAPH
                                (#x9389 . #x6432)	;CJK UNIFIED IDEOGRAPH
                                (#x938A . #x6433)	;CJK UNIFIED IDEOGRAPH
                                (#x938B . #x6435)	;CJK UNIFIED IDEOGRAPH
                                (#x938C . #x6436)	;CJK UNIFIED IDEOGRAPH
                                (#x938D . #x6437)	;CJK UNIFIED IDEOGRAPH
                                (#x938E . #x6438)	;CJK UNIFIED IDEOGRAPH
                                (#x938F . #x6439)	;CJK UNIFIED IDEOGRAPH
                                (#x9390 . #x643B)	;CJK UNIFIED IDEOGRAPH
                                (#x9391 . #x643C)	;CJK UNIFIED IDEOGRAPH
                                (#x9392 . #x643E)	;CJK UNIFIED IDEOGRAPH
                                (#x9393 . #x6440)	;CJK UNIFIED IDEOGRAPH
                                (#x9394 . #x6442)	;CJK UNIFIED IDEOGRAPH
                                (#x9395 . #x6443)	;CJK UNIFIED IDEOGRAPH
                                (#x9396 . #x6449)	;CJK UNIFIED IDEOGRAPH
                                (#x9397 . #x644B)	;CJK UNIFIED IDEOGRAPH
                                (#x9398 . #x644C)	;CJK UNIFIED IDEOGRAPH
                                (#x9399 . #x644D)	;CJK UNIFIED IDEOGRAPH
                                (#x939A . #x644E)	;CJK UNIFIED IDEOGRAPH
                                (#x939B . #x644F)	;CJK UNIFIED IDEOGRAPH
                                (#x939C . #x6450)	;CJK UNIFIED IDEOGRAPH
                                (#x939D . #x6451)	;CJK UNIFIED IDEOGRAPH
                                (#x939E . #x6453)	;CJK UNIFIED IDEOGRAPH
                                (#x939F . #x6455)	;CJK UNIFIED IDEOGRAPH
                                (#x93A0 . #x6456)	;CJK UNIFIED IDEOGRAPH
                                (#x93A1 . #x6457)	;CJK UNIFIED IDEOGRAPH
                                (#x93A2 . #x6459)	;CJK UNIFIED IDEOGRAPH
                                (#x93A3 . #x645A)	;CJK UNIFIED IDEOGRAPH
                                (#x93A4 . #x645B)	;CJK UNIFIED IDEOGRAPH
                                (#x93A5 . #x645C)	;CJK UNIFIED IDEOGRAPH
                                (#x93A6 . #x645D)	;CJK UNIFIED IDEOGRAPH
                                (#x93A7 . #x645F)	;CJK UNIFIED IDEOGRAPH
                                (#x93A8 . #x6460)	;CJK UNIFIED IDEOGRAPH
                                (#x93A9 . #x6461)	;CJK UNIFIED IDEOGRAPH
                                (#x93AA . #x6462)	;CJK UNIFIED IDEOGRAPH
                                (#x93AB . #x6463)	;CJK UNIFIED IDEOGRAPH
                                (#x93AC . #x6464)	;CJK UNIFIED IDEOGRAPH
                                (#x93AD . #x6465)	;CJK UNIFIED IDEOGRAPH
                                (#x93AE . #x6466)	;CJK UNIFIED IDEOGRAPH
                                (#x93AF . #x6468)	;CJK UNIFIED IDEOGRAPH
                                (#x93B0 . #x646A)	;CJK UNIFIED IDEOGRAPH
                                (#x93B1 . #x646B)	;CJK UNIFIED IDEOGRAPH
                                (#x93B2 . #x646C)	;CJK UNIFIED IDEOGRAPH
                                (#x93B3 . #x646E)	;CJK UNIFIED IDEOGRAPH
                                (#x93B4 . #x646F)	;CJK UNIFIED IDEOGRAPH
                                (#x93B5 . #x6470)	;CJK UNIFIED IDEOGRAPH
                                (#x93B6 . #x6471)	;CJK UNIFIED IDEOGRAPH
                                (#x93B7 . #x6472)	;CJK UNIFIED IDEOGRAPH
                                (#x93B8 . #x6473)	;CJK UNIFIED IDEOGRAPH
                                (#x93B9 . #x6474)	;CJK UNIFIED IDEOGRAPH
                                (#x93BA . #x6475)	;CJK UNIFIED IDEOGRAPH
                                (#x93BB . #x6476)	;CJK UNIFIED IDEOGRAPH
                                (#x93BC . #x6477)	;CJK UNIFIED IDEOGRAPH
                                (#x93BD . #x647B)	;CJK UNIFIED IDEOGRAPH
                                (#x93BE . #x647C)	;CJK UNIFIED IDEOGRAPH
                                (#x93BF . #x647D)	;CJK UNIFIED IDEOGRAPH
                                (#x93C0 . #x647E)	;CJK UNIFIED IDEOGRAPH
                                (#x93C1 . #x647F)	;CJK UNIFIED IDEOGRAPH
                                (#x93C2 . #x6480)	;CJK UNIFIED IDEOGRAPH
                                (#x93C3 . #x6481)	;CJK UNIFIED IDEOGRAPH
                                (#x93C4 . #x6483)	;CJK UNIFIED IDEOGRAPH
                                (#x93C5 . #x6486)	;CJK UNIFIED IDEOGRAPH
                                (#x93C6 . #x6488)	;CJK UNIFIED IDEOGRAPH
                                (#x93C7 . #x6489)	;CJK UNIFIED IDEOGRAPH
                                (#x93C8 . #x648A)	;CJK UNIFIED IDEOGRAPH
                                (#x93C9 . #x648B)	;CJK UNIFIED IDEOGRAPH
                                (#x93CA . #x648C)	;CJK UNIFIED IDEOGRAPH
                                (#x93CB . #x648D)	;CJK UNIFIED IDEOGRAPH
                                (#x93CC . #x648E)	;CJK UNIFIED IDEOGRAPH
                                (#x93CD . #x648F)	;CJK UNIFIED IDEOGRAPH
                                (#x93CE . #x6490)	;CJK UNIFIED IDEOGRAPH
                                (#x93CF . #x6493)	;CJK UNIFIED IDEOGRAPH
                                (#x93D0 . #x6494)	;CJK UNIFIED IDEOGRAPH
                                (#x93D1 . #x6497)	;CJK UNIFIED IDEOGRAPH
                                (#x93D2 . #x6498)	;CJK UNIFIED IDEOGRAPH
                                (#x93D3 . #x649A)	;CJK UNIFIED IDEOGRAPH
                                (#x93D4 . #x649B)	;CJK UNIFIED IDEOGRAPH
                                (#x93D5 . #x649C)	;CJK UNIFIED IDEOGRAPH
                                (#x93D6 . #x649D)	;CJK UNIFIED IDEOGRAPH
                                (#x93D7 . #x649F)	;CJK UNIFIED IDEOGRAPH
                                (#x93D8 . #x64A0)	;CJK UNIFIED IDEOGRAPH
                                (#x93D9 . #x64A1)	;CJK UNIFIED IDEOGRAPH
                                (#x93DA . #x64A2)	;CJK UNIFIED IDEOGRAPH
                                (#x93DB . #x64A3)	;CJK UNIFIED IDEOGRAPH
                                (#x93DC . #x64A5)	;CJK UNIFIED IDEOGRAPH
                                (#x93DD . #x64A6)	;CJK UNIFIED IDEOGRAPH
                                (#x93DE . #x64A7)	;CJK UNIFIED IDEOGRAPH
                                (#x93DF . #x64A8)	;CJK UNIFIED IDEOGRAPH
                                (#x93E0 . #x64AA)	;CJK UNIFIED IDEOGRAPH
                                (#x93E1 . #x64AB)	;CJK UNIFIED IDEOGRAPH
                                (#x93E2 . #x64AF)	;CJK UNIFIED IDEOGRAPH
                                (#x93E3 . #x64B1)	;CJK UNIFIED IDEOGRAPH
                                (#x93E4 . #x64B2)	;CJK UNIFIED IDEOGRAPH
                                (#x93E5 . #x64B3)	;CJK UNIFIED IDEOGRAPH
                                (#x93E6 . #x64B4)	;CJK UNIFIED IDEOGRAPH
                                (#x93E7 . #x64B6)	;CJK UNIFIED IDEOGRAPH
                                (#x93E8 . #x64B9)	;CJK UNIFIED IDEOGRAPH
                                (#x93E9 . #x64BB)	;CJK UNIFIED IDEOGRAPH
                                (#x93EA . #x64BD)	;CJK UNIFIED IDEOGRAPH
                                (#x93EB . #x64BE)	;CJK UNIFIED IDEOGRAPH
                                (#x93EC . #x64BF)	;CJK UNIFIED IDEOGRAPH
                                (#x93ED . #x64C1)	;CJK UNIFIED IDEOGRAPH
                                (#x93EE . #x64C3)	;CJK UNIFIED IDEOGRAPH
                                (#x93EF . #x64C4)	;CJK UNIFIED IDEOGRAPH
                                (#x93F0 . #x64C6)	;CJK UNIFIED IDEOGRAPH
                                (#x93F1 . #x64C7)	;CJK UNIFIED IDEOGRAPH
                                (#x93F2 . #x64C8)	;CJK UNIFIED IDEOGRAPH
                                (#x93F3 . #x64C9)	;CJK UNIFIED IDEOGRAPH
                                (#x93F4 . #x64CA)	;CJK UNIFIED IDEOGRAPH
                                (#x93F5 . #x64CB)	;CJK UNIFIED IDEOGRAPH
                                (#x93F6 . #x64CC)	;CJK UNIFIED IDEOGRAPH
                                (#x93F7 . #x64CF)	;CJK UNIFIED IDEOGRAPH
                                (#x93F8 . #x64D1)	;CJK UNIFIED IDEOGRAPH
                                (#x93F9 . #x64D3)	;CJK UNIFIED IDEOGRAPH
                                (#x93FA . #x64D4)	;CJK UNIFIED IDEOGRAPH
                                (#x93FB . #x64D5)	;CJK UNIFIED IDEOGRAPH
                                (#x93FC . #x64D6)	;CJK UNIFIED IDEOGRAPH
                                (#x93FD . #x64D9)	;CJK UNIFIED IDEOGRAPH
                                (#x93FE . #x64DA)	;CJK UNIFIED IDEOGRAPH
                                (#x9440 . #x64DB)	;CJK UNIFIED IDEOGRAPH
                                (#x9441 . #x64DC)	;CJK UNIFIED IDEOGRAPH
                                (#x9442 . #x64DD)	;CJK UNIFIED IDEOGRAPH
                                (#x9443 . #x64DF)	;CJK UNIFIED IDEOGRAPH
                                (#x9444 . #x64E0)	;CJK UNIFIED IDEOGRAPH
                                (#x9445 . #x64E1)	;CJK UNIFIED IDEOGRAPH
                                (#x9446 . #x64E3)	;CJK UNIFIED IDEOGRAPH
                                (#x9447 . #x64E5)	;CJK UNIFIED IDEOGRAPH
                                (#x9448 . #x64E7)	;CJK UNIFIED IDEOGRAPH
                                (#x9449 . #x64E8)	;CJK UNIFIED IDEOGRAPH
                                (#x944A . #x64E9)	;CJK UNIFIED IDEOGRAPH
                                (#x944B . #x64EA)	;CJK UNIFIED IDEOGRAPH
                                (#x944C . #x64EB)	;CJK UNIFIED IDEOGRAPH
                                (#x944D . #x64EC)	;CJK UNIFIED IDEOGRAPH
                                (#x944E . #x64ED)	;CJK UNIFIED IDEOGRAPH
                                (#x944F . #x64EE)	;CJK UNIFIED IDEOGRAPH
                                (#x9450 . #x64EF)	;CJK UNIFIED IDEOGRAPH
                                (#x9451 . #x64F0)	;CJK UNIFIED IDEOGRAPH
                                (#x9452 . #x64F1)	;CJK UNIFIED IDEOGRAPH
                                (#x9453 . #x64F2)	;CJK UNIFIED IDEOGRAPH
                                (#x9454 . #x64F3)	;CJK UNIFIED IDEOGRAPH
                                (#x9455 . #x64F4)	;CJK UNIFIED IDEOGRAPH
                                (#x9456 . #x64F5)	;CJK UNIFIED IDEOGRAPH
                                (#x9457 . #x64F6)	;CJK UNIFIED IDEOGRAPH
                                (#x9458 . #x64F7)	;CJK UNIFIED IDEOGRAPH
                                (#x9459 . #x64F8)	;CJK UNIFIED IDEOGRAPH
                                (#x945A . #x64F9)	;CJK UNIFIED IDEOGRAPH
                                (#x945B . #x64FA)	;CJK UNIFIED IDEOGRAPH
                                (#x945C . #x64FB)	;CJK UNIFIED IDEOGRAPH
                                (#x945D . #x64FC)	;CJK UNIFIED IDEOGRAPH
                                (#x945E . #x64FD)	;CJK UNIFIED IDEOGRAPH
                                (#x945F . #x64FE)	;CJK UNIFIED IDEOGRAPH
                                (#x9460 . #x64FF)	;CJK UNIFIED IDEOGRAPH
                                (#x9461 . #x6501)	;CJK UNIFIED IDEOGRAPH
                                (#x9462 . #x6502)	;CJK UNIFIED IDEOGRAPH
                                (#x9463 . #x6503)	;CJK UNIFIED IDEOGRAPH
                                (#x9464 . #x6504)	;CJK UNIFIED IDEOGRAPH
                                (#x9465 . #x6505)	;CJK UNIFIED IDEOGRAPH
                                (#x9466 . #x6506)	;CJK UNIFIED IDEOGRAPH
                                (#x9467 . #x6507)	;CJK UNIFIED IDEOGRAPH
                                (#x9468 . #x6508)	;CJK UNIFIED IDEOGRAPH
                                (#x9469 . #x650A)	;CJK UNIFIED IDEOGRAPH
                                (#x946A . #x650B)	;CJK UNIFIED IDEOGRAPH
                                (#x946B . #x650C)	;CJK UNIFIED IDEOGRAPH
                                (#x946C . #x650D)	;CJK UNIFIED IDEOGRAPH
                                (#x946D . #x650E)	;CJK UNIFIED IDEOGRAPH
                                (#x946E . #x650F)	;CJK UNIFIED IDEOGRAPH
                                (#x946F . #x6510)	;CJK UNIFIED IDEOGRAPH
                                (#x9470 . #x6511)	;CJK UNIFIED IDEOGRAPH
                                (#x9471 . #x6513)	;CJK UNIFIED IDEOGRAPH
                                (#x9472 . #x6514)	;CJK UNIFIED IDEOGRAPH
                                (#x9473 . #x6515)	;CJK UNIFIED IDEOGRAPH
                                (#x9474 . #x6516)	;CJK UNIFIED IDEOGRAPH
                                (#x9475 . #x6517)	;CJK UNIFIED IDEOGRAPH
                                (#x9476 . #x6519)	;CJK UNIFIED IDEOGRAPH
                                (#x9477 . #x651A)	;CJK UNIFIED IDEOGRAPH
                                (#x9478 . #x651B)	;CJK UNIFIED IDEOGRAPH
                                (#x9479 . #x651C)	;CJK UNIFIED IDEOGRAPH
                                (#x947A . #x651D)	;CJK UNIFIED IDEOGRAPH
                                (#x947B . #x651E)	;CJK UNIFIED IDEOGRAPH
                                (#x947C . #x651F)	;CJK UNIFIED IDEOGRAPH
                                (#x947D . #x6520)	;CJK UNIFIED IDEOGRAPH
                                (#x947E . #x6521)	;CJK UNIFIED IDEOGRAPH
                                (#x9480 . #x6522)	;CJK UNIFIED IDEOGRAPH
                                (#x9481 . #x6523)	;CJK UNIFIED IDEOGRAPH
                                (#x9482 . #x6524)	;CJK UNIFIED IDEOGRAPH
                                (#x9483 . #x6526)	;CJK UNIFIED IDEOGRAPH
                                (#x9484 . #x6527)	;CJK UNIFIED IDEOGRAPH
                                (#x9485 . #x6528)	;CJK UNIFIED IDEOGRAPH
                                (#x9486 . #x6529)	;CJK UNIFIED IDEOGRAPH
                                (#x9487 . #x652A)	;CJK UNIFIED IDEOGRAPH
                                (#x9488 . #x652C)	;CJK UNIFIED IDEOGRAPH
                                (#x9489 . #x652D)	;CJK UNIFIED IDEOGRAPH
                                (#x948A . #x6530)	;CJK UNIFIED IDEOGRAPH
                                (#x948B . #x6531)	;CJK UNIFIED IDEOGRAPH
                                (#x948C . #x6532)	;CJK UNIFIED IDEOGRAPH
                                (#x948D . #x6533)	;CJK UNIFIED IDEOGRAPH
                                (#x948E . #x6537)	;CJK UNIFIED IDEOGRAPH
                                (#x948F . #x653A)	;CJK UNIFIED IDEOGRAPH
                                (#x9490 . #x653C)	;CJK UNIFIED IDEOGRAPH
                                (#x9491 . #x653D)	;CJK UNIFIED IDEOGRAPH
                                (#x9492 . #x6540)	;CJK UNIFIED IDEOGRAPH
                                (#x9493 . #x6541)	;CJK UNIFIED IDEOGRAPH
                                (#x9494 . #x6542)	;CJK UNIFIED IDEOGRAPH
                                (#x9495 . #x6543)	;CJK UNIFIED IDEOGRAPH
                                (#x9496 . #x6544)	;CJK UNIFIED IDEOGRAPH
                                (#x9497 . #x6546)	;CJK UNIFIED IDEOGRAPH
                                (#x9498 . #x6547)	;CJK UNIFIED IDEOGRAPH
                                (#x9499 . #x654A)	;CJK UNIFIED IDEOGRAPH
                                (#x949A . #x654B)	;CJK UNIFIED IDEOGRAPH
                                (#x949B . #x654D)	;CJK UNIFIED IDEOGRAPH
                                (#x949C . #x654E)	;CJK UNIFIED IDEOGRAPH
                                (#x949D . #x6550)	;CJK UNIFIED IDEOGRAPH
                                (#x949E . #x6552)	;CJK UNIFIED IDEOGRAPH
                                (#x949F . #x6553)	;CJK UNIFIED IDEOGRAPH
                                (#x94A0 . #x6554)	;CJK UNIFIED IDEOGRAPH
                                (#x94A1 . #x6557)	;CJK UNIFIED IDEOGRAPH
                                (#x94A2 . #x6558)	;CJK UNIFIED IDEOGRAPH
                                (#x94A3 . #x655A)	;CJK UNIFIED IDEOGRAPH
                                (#x94A4 . #x655C)	;CJK UNIFIED IDEOGRAPH
                                (#x94A5 . #x655F)	;CJK UNIFIED IDEOGRAPH
                                (#x94A6 . #x6560)	;CJK UNIFIED IDEOGRAPH
                                (#x94A7 . #x6561)	;CJK UNIFIED IDEOGRAPH
                                (#x94A8 . #x6564)	;CJK UNIFIED IDEOGRAPH
                                (#x94A9 . #x6565)	;CJK UNIFIED IDEOGRAPH
                                (#x94AA . #x6567)	;CJK UNIFIED IDEOGRAPH
                                (#x94AB . #x6568)	;CJK UNIFIED IDEOGRAPH
                                (#x94AC . #x6569)	;CJK UNIFIED IDEOGRAPH
                                (#x94AD . #x656A)	;CJK UNIFIED IDEOGRAPH
                                (#x94AE . #x656D)	;CJK UNIFIED IDEOGRAPH
                                (#x94AF . #x656E)	;CJK UNIFIED IDEOGRAPH
                                (#x94B0 . #x656F)	;CJK UNIFIED IDEOGRAPH
                                (#x94B1 . #x6571)	;CJK UNIFIED IDEOGRAPH
                                (#x94B2 . #x6573)	;CJK UNIFIED IDEOGRAPH
                                (#x94B3 . #x6575)	;CJK UNIFIED IDEOGRAPH
                                (#x94B4 . #x6576)	;CJK UNIFIED IDEOGRAPH
                                (#x94B5 . #x6578)	;CJK UNIFIED IDEOGRAPH
                                (#x94B6 . #x6579)	;CJK UNIFIED IDEOGRAPH
                                (#x94B7 . #x657A)	;CJK UNIFIED IDEOGRAPH
                                (#x94B8 . #x657B)	;CJK UNIFIED IDEOGRAPH
                                (#x94B9 . #x657C)	;CJK UNIFIED IDEOGRAPH
                                (#x94BA . #x657D)	;CJK UNIFIED IDEOGRAPH
                                (#x94BB . #x657E)	;CJK UNIFIED IDEOGRAPH
                                (#x94BC . #x657F)	;CJK UNIFIED IDEOGRAPH
                                (#x94BD . #x6580)	;CJK UNIFIED IDEOGRAPH
                                (#x94BE . #x6581)	;CJK UNIFIED IDEOGRAPH
                                (#x94BF . #x6582)	;CJK UNIFIED IDEOGRAPH
                                (#x94C0 . #x6583)	;CJK UNIFIED IDEOGRAPH
                                (#x94C1 . #x6584)	;CJK UNIFIED IDEOGRAPH
                                (#x94C2 . #x6585)	;CJK UNIFIED IDEOGRAPH
                                (#x94C3 . #x6586)	;CJK UNIFIED IDEOGRAPH
                                (#x94C4 . #x6588)	;CJK UNIFIED IDEOGRAPH
                                (#x94C5 . #x6589)	;CJK UNIFIED IDEOGRAPH
                                (#x94C6 . #x658A)	;CJK UNIFIED IDEOGRAPH
                                (#x94C7 . #x658D)	;CJK UNIFIED IDEOGRAPH
                                (#x94C8 . #x658E)	;CJK UNIFIED IDEOGRAPH
                                (#x94C9 . #x658F)	;CJK UNIFIED IDEOGRAPH
                                (#x94CA . #x6592)	;CJK UNIFIED IDEOGRAPH
                                (#x94CB . #x6594)	;CJK UNIFIED IDEOGRAPH
                                (#x94CC . #x6595)	;CJK UNIFIED IDEOGRAPH
                                (#x94CD . #x6596)	;CJK UNIFIED IDEOGRAPH
                                (#x94CE . #x6598)	;CJK UNIFIED IDEOGRAPH
                                (#x94CF . #x659A)	;CJK UNIFIED IDEOGRAPH
                                (#x94D0 . #x659D)	;CJK UNIFIED IDEOGRAPH
                                (#x94D1 . #x659E)	;CJK UNIFIED IDEOGRAPH
                                (#x94D2 . #x65A0)	;CJK UNIFIED IDEOGRAPH
                                (#x94D3 . #x65A2)	;CJK UNIFIED IDEOGRAPH
                                (#x94D4 . #x65A3)	;CJK UNIFIED IDEOGRAPH
                                (#x94D5 . #x65A6)	;CJK UNIFIED IDEOGRAPH
                                (#x94D6 . #x65A8)	;CJK UNIFIED IDEOGRAPH
                                (#x94D7 . #x65AA)	;CJK UNIFIED IDEOGRAPH
                                (#x94D8 . #x65AC)	;CJK UNIFIED IDEOGRAPH
                                (#x94D9 . #x65AE)	;CJK UNIFIED IDEOGRAPH
                                (#x94DA . #x65B1)	;CJK UNIFIED IDEOGRAPH
                                (#x94DB . #x65B2)	;CJK UNIFIED IDEOGRAPH
                                (#x94DC . #x65B3)	;CJK UNIFIED IDEOGRAPH
                                (#x94DD . #x65B4)	;CJK UNIFIED IDEOGRAPH
                                (#x94DE . #x65B5)	;CJK UNIFIED IDEOGRAPH
                                (#x94DF . #x65B6)	;CJK UNIFIED IDEOGRAPH
                                (#x94E0 . #x65B7)	;CJK UNIFIED IDEOGRAPH
                                (#x94E1 . #x65B8)	;CJK UNIFIED IDEOGRAPH
                                (#x94E2 . #x65BA)	;CJK UNIFIED IDEOGRAPH
                                (#x94E3 . #x65BB)	;CJK UNIFIED IDEOGRAPH
                                (#x94E4 . #x65BE)	;CJK UNIFIED IDEOGRAPH
                                (#x94E5 . #x65BF)	;CJK UNIFIED IDEOGRAPH
                                (#x94E6 . #x65C0)	;CJK UNIFIED IDEOGRAPH
                                (#x94E7 . #x65C2)	;CJK UNIFIED IDEOGRAPH
                                (#x94E8 . #x65C7)	;CJK UNIFIED IDEOGRAPH
                                (#x94E9 . #x65C8)	;CJK UNIFIED IDEOGRAPH
                                (#x94EA . #x65C9)	;CJK UNIFIED IDEOGRAPH
                                (#x94EB . #x65CA)	;CJK UNIFIED IDEOGRAPH
                                (#x94EC . #x65CD)	;CJK UNIFIED IDEOGRAPH
                                (#x94ED . #x65D0)	;CJK UNIFIED IDEOGRAPH
                                (#x94EE . #x65D1)	;CJK UNIFIED IDEOGRAPH
                                (#x94EF . #x65D3)	;CJK UNIFIED IDEOGRAPH
                                (#x94F0 . #x65D4)	;CJK UNIFIED IDEOGRAPH
                                (#x94F1 . #x65D5)	;CJK UNIFIED IDEOGRAPH
                                (#x94F2 . #x65D8)	;CJK UNIFIED IDEOGRAPH
                                (#x94F3 . #x65D9)	;CJK UNIFIED IDEOGRAPH
                                (#x94F4 . #x65DA)	;CJK UNIFIED IDEOGRAPH
                                (#x94F5 . #x65DB)	;CJK UNIFIED IDEOGRAPH
                                (#x94F6 . #x65DC)	;CJK UNIFIED IDEOGRAPH
                                (#x94F7 . #x65DD)	;CJK UNIFIED IDEOGRAPH
                                (#x94F8 . #x65DE)	;CJK UNIFIED IDEOGRAPH
                                (#x94F9 . #x65DF)	;CJK UNIFIED IDEOGRAPH
                                (#x94FA . #x65E1)	;CJK UNIFIED IDEOGRAPH
                                (#x94FB . #x65E3)	;CJK UNIFIED IDEOGRAPH
                                (#x94FC . #x65E4)	;CJK UNIFIED IDEOGRAPH
                                (#x94FD . #x65EA)	;CJK UNIFIED IDEOGRAPH
                                (#x94FE . #x65EB)	;CJK UNIFIED IDEOGRAPH
                                (#x9540 . #x65F2)	;CJK UNIFIED IDEOGRAPH
                                (#x9541 . #x65F3)	;CJK UNIFIED IDEOGRAPH
                                (#x9542 . #x65F4)	;CJK UNIFIED IDEOGRAPH
                                (#x9543 . #x65F5)	;CJK UNIFIED IDEOGRAPH
                                (#x9544 . #x65F8)	;CJK UNIFIED IDEOGRAPH
                                (#x9545 . #x65F9)	;CJK UNIFIED IDEOGRAPH
                                (#x9546 . #x65FB)	;CJK UNIFIED IDEOGRAPH
                                (#x9547 . #x65FC)	;CJK UNIFIED IDEOGRAPH
                                (#x9548 . #x65FD)	;CJK UNIFIED IDEOGRAPH
                                (#x9549 . #x65FE)	;CJK UNIFIED IDEOGRAPH
                                (#x954A . #x65FF)	;CJK UNIFIED IDEOGRAPH
                                (#x954B . #x6601)	;CJK UNIFIED IDEOGRAPH
                                (#x954C . #x6604)	;CJK UNIFIED IDEOGRAPH
                                (#x954D . #x6605)	;CJK UNIFIED IDEOGRAPH
                                (#x954E . #x6607)	;CJK UNIFIED IDEOGRAPH
                                (#x954F . #x6608)	;CJK UNIFIED IDEOGRAPH
                                (#x9550 . #x6609)	;CJK UNIFIED IDEOGRAPH
                                (#x9551 . #x660B)	;CJK UNIFIED IDEOGRAPH
                                (#x9552 . #x660D)	;CJK UNIFIED IDEOGRAPH
                                (#x9553 . #x6610)	;CJK UNIFIED IDEOGRAPH
                                (#x9554 . #x6611)	;CJK UNIFIED IDEOGRAPH
                                (#x9555 . #x6612)	;CJK UNIFIED IDEOGRAPH
                                (#x9556 . #x6616)	;CJK UNIFIED IDEOGRAPH
                                (#x9557 . #x6617)	;CJK UNIFIED IDEOGRAPH
                                (#x9558 . #x6618)	;CJK UNIFIED IDEOGRAPH
                                (#x9559 . #x661A)	;CJK UNIFIED IDEOGRAPH
                                (#x955A . #x661B)	;CJK UNIFIED IDEOGRAPH
                                (#x955B . #x661C)	;CJK UNIFIED IDEOGRAPH
                                (#x955C . #x661E)	;CJK UNIFIED IDEOGRAPH
                                (#x955D . #x6621)	;CJK UNIFIED IDEOGRAPH
                                (#x955E . #x6622)	;CJK UNIFIED IDEOGRAPH
                                (#x955F . #x6623)	;CJK UNIFIED IDEOGRAPH
                                (#x9560 . #x6624)	;CJK UNIFIED IDEOGRAPH
                                (#x9561 . #x6626)	;CJK UNIFIED IDEOGRAPH
                                (#x9562 . #x6629)	;CJK UNIFIED IDEOGRAPH
                                (#x9563 . #x662A)	;CJK UNIFIED IDEOGRAPH
                                (#x9564 . #x662B)	;CJK UNIFIED IDEOGRAPH
                                (#x9565 . #x662C)	;CJK UNIFIED IDEOGRAPH
                                (#x9566 . #x662E)	;CJK UNIFIED IDEOGRAPH
                                (#x9567 . #x6630)	;CJK UNIFIED IDEOGRAPH
                                (#x9568 . #x6632)	;CJK UNIFIED IDEOGRAPH
                                (#x9569 . #x6633)	;CJK UNIFIED IDEOGRAPH
                                (#x956A . #x6637)	;CJK UNIFIED IDEOGRAPH
                                (#x956B . #x6638)	;CJK UNIFIED IDEOGRAPH
                                (#x956C . #x6639)	;CJK UNIFIED IDEOGRAPH
                                (#x956D . #x663A)	;CJK UNIFIED IDEOGRAPH
                                (#x956E . #x663B)	;CJK UNIFIED IDEOGRAPH
                                (#x956F . #x663D)	;CJK UNIFIED IDEOGRAPH
                                (#x9570 . #x663F)	;CJK UNIFIED IDEOGRAPH
                                (#x9571 . #x6640)	;CJK UNIFIED IDEOGRAPH
                                (#x9572 . #x6642)	;CJK UNIFIED IDEOGRAPH
                                (#x9573 . #x6644)	;CJK UNIFIED IDEOGRAPH
                                (#x9574 . #x6645)	;CJK UNIFIED IDEOGRAPH
                                (#x9575 . #x6646)	;CJK UNIFIED IDEOGRAPH
                                (#x9576 . #x6647)	;CJK UNIFIED IDEOGRAPH
                                (#x9577 . #x6648)	;CJK UNIFIED IDEOGRAPH
                                (#x9578 . #x6649)	;CJK UNIFIED IDEOGRAPH
                                (#x9579 . #x664A)	;CJK UNIFIED IDEOGRAPH
                                (#x957A . #x664D)	;CJK UNIFIED IDEOGRAPH
                                (#x957B . #x664E)	;CJK UNIFIED IDEOGRAPH
                                (#x957C . #x6650)	;CJK UNIFIED IDEOGRAPH
                                (#x957D . #x6651)	;CJK UNIFIED IDEOGRAPH
                                (#x957E . #x6658)	;CJK UNIFIED IDEOGRAPH
                                (#x9580 . #x6659)	;CJK UNIFIED IDEOGRAPH
                                (#x9581 . #x665B)	;CJK UNIFIED IDEOGRAPH
                                (#x9582 . #x665C)	;CJK UNIFIED IDEOGRAPH
                                (#x9583 . #x665D)	;CJK UNIFIED IDEOGRAPH
                                (#x9584 . #x665E)	;CJK UNIFIED IDEOGRAPH
                                (#x9585 . #x6660)	;CJK UNIFIED IDEOGRAPH
                                (#x9586 . #x6662)	;CJK UNIFIED IDEOGRAPH
                                (#x9587 . #x6663)	;CJK UNIFIED IDEOGRAPH
                                (#x9588 . #x6665)	;CJK UNIFIED IDEOGRAPH
                                (#x9589 . #x6667)	;CJK UNIFIED IDEOGRAPH
                                (#x958A . #x6669)	;CJK UNIFIED IDEOGRAPH
                                (#x958B . #x666A)	;CJK UNIFIED IDEOGRAPH
                                (#x958C . #x666B)	;CJK UNIFIED IDEOGRAPH
                                (#x958D . #x666C)	;CJK UNIFIED IDEOGRAPH
                                (#x958E . #x666D)	;CJK UNIFIED IDEOGRAPH
                                (#x958F . #x6671)	;CJK UNIFIED IDEOGRAPH
                                (#x9590 . #x6672)	;CJK UNIFIED IDEOGRAPH
                                (#x9591 . #x6673)	;CJK UNIFIED IDEOGRAPH
                                (#x9592 . #x6675)	;CJK UNIFIED IDEOGRAPH
                                (#x9593 . #x6678)	;CJK UNIFIED IDEOGRAPH
                                (#x9594 . #x6679)	;CJK UNIFIED IDEOGRAPH
                                (#x9595 . #x667B)	;CJK UNIFIED IDEOGRAPH
                                (#x9596 . #x667C)	;CJK UNIFIED IDEOGRAPH
                                (#x9597 . #x667D)	;CJK UNIFIED IDEOGRAPH
                                (#x9598 . #x667F)	;CJK UNIFIED IDEOGRAPH
                                (#x9599 . #x6680)	;CJK UNIFIED IDEOGRAPH
                                (#x959A . #x6681)	;CJK UNIFIED IDEOGRAPH
                                (#x959B . #x6683)	;CJK UNIFIED IDEOGRAPH
                                (#x959C . #x6685)	;CJK UNIFIED IDEOGRAPH
                                (#x959D . #x6686)	;CJK UNIFIED IDEOGRAPH
                                (#x959E . #x6688)	;CJK UNIFIED IDEOGRAPH
                                (#x959F . #x6689)	;CJK UNIFIED IDEOGRAPH
                                (#x95A0 . #x668A)	;CJK UNIFIED IDEOGRAPH
                                (#x95A1 . #x668B)	;CJK UNIFIED IDEOGRAPH
                                (#x95A2 . #x668D)	;CJK UNIFIED IDEOGRAPH
                                (#x95A3 . #x668E)	;CJK UNIFIED IDEOGRAPH
                                (#x95A4 . #x668F)	;CJK UNIFIED IDEOGRAPH
                                (#x95A5 . #x6690)	;CJK UNIFIED IDEOGRAPH
                                (#x95A6 . #x6692)	;CJK UNIFIED IDEOGRAPH
                                (#x95A7 . #x6693)	;CJK UNIFIED IDEOGRAPH
                                (#x95A8 . #x6694)	;CJK UNIFIED IDEOGRAPH
                                (#x95A9 . #x6695)	;CJK UNIFIED IDEOGRAPH
                                (#x95AA . #x6698)	;CJK UNIFIED IDEOGRAPH
                                (#x95AB . #x6699)	;CJK UNIFIED IDEOGRAPH
                                (#x95AC . #x669A)	;CJK UNIFIED IDEOGRAPH
                                (#x95AD . #x669B)	;CJK UNIFIED IDEOGRAPH
                                (#x95AE . #x669C)	;CJK UNIFIED IDEOGRAPH
                                (#x95AF . #x669E)	;CJK UNIFIED IDEOGRAPH
                                (#x95B0 . #x669F)	;CJK UNIFIED IDEOGRAPH
                                (#x95B1 . #x66A0)	;CJK UNIFIED IDEOGRAPH
                                (#x95B2 . #x66A1)	;CJK UNIFIED IDEOGRAPH
                                (#x95B3 . #x66A2)	;CJK UNIFIED IDEOGRAPH
                                (#x95B4 . #x66A3)	;CJK UNIFIED IDEOGRAPH
                                (#x95B5 . #x66A4)	;CJK UNIFIED IDEOGRAPH
                                (#x95B6 . #x66A5)	;CJK UNIFIED IDEOGRAPH
                                (#x95B7 . #x66A6)	;CJK UNIFIED IDEOGRAPH
                                (#x95B8 . #x66A9)	;CJK UNIFIED IDEOGRAPH
                                (#x95B9 . #x66AA)	;CJK UNIFIED IDEOGRAPH
                                (#x95BA . #x66AB)	;CJK UNIFIED IDEOGRAPH
                                (#x95BB . #x66AC)	;CJK UNIFIED IDEOGRAPH
                                (#x95BC . #x66AD)	;CJK UNIFIED IDEOGRAPH
                                (#x95BD . #x66AF)	;CJK UNIFIED IDEOGRAPH
                                (#x95BE . #x66B0)	;CJK UNIFIED IDEOGRAPH
                                (#x95BF . #x66B1)	;CJK UNIFIED IDEOGRAPH
                                (#x95C0 . #x66B2)	;CJK UNIFIED IDEOGRAPH
                                (#x95C1 . #x66B3)	;CJK UNIFIED IDEOGRAPH
                                (#x95C2 . #x66B5)	;CJK UNIFIED IDEOGRAPH
                                (#x95C3 . #x66B6)	;CJK UNIFIED IDEOGRAPH
                                (#x95C4 . #x66B7)	;CJK UNIFIED IDEOGRAPH
                                (#x95C5 . #x66B8)	;CJK UNIFIED IDEOGRAPH
                                (#x95C6 . #x66BA)	;CJK UNIFIED IDEOGRAPH
                                (#x95C7 . #x66BB)	;CJK UNIFIED IDEOGRAPH
                                (#x95C8 . #x66BC)	;CJK UNIFIED IDEOGRAPH
                                (#x95C9 . #x66BD)	;CJK UNIFIED IDEOGRAPH
                                (#x95CA . #x66BF)	;CJK UNIFIED IDEOGRAPH
                                (#x95CB . #x66C0)	;CJK UNIFIED IDEOGRAPH
                                (#x95CC . #x66C1)	;CJK UNIFIED IDEOGRAPH
                                (#x95CD . #x66C2)	;CJK UNIFIED IDEOGRAPH
                                (#x95CE . #x66C3)	;CJK UNIFIED IDEOGRAPH
                                (#x95CF . #x66C4)	;CJK UNIFIED IDEOGRAPH
                                (#x95D0 . #x66C5)	;CJK UNIFIED IDEOGRAPH
                                (#x95D1 . #x66C6)	;CJK UNIFIED IDEOGRAPH
                                (#x95D2 . #x66C7)	;CJK UNIFIED IDEOGRAPH
                                (#x95D3 . #x66C8)	;CJK UNIFIED IDEOGRAPH
                                (#x95D4 . #x66C9)	;CJK UNIFIED IDEOGRAPH
                                (#x95D5 . #x66CA)	;CJK UNIFIED IDEOGRAPH
                                (#x95D6 . #x66CB)	;CJK UNIFIED IDEOGRAPH
                                (#x95D7 . #x66CC)	;CJK UNIFIED IDEOGRAPH
                                (#x95D8 . #x66CD)	;CJK UNIFIED IDEOGRAPH
                                (#x95D9 . #x66CE)	;CJK UNIFIED IDEOGRAPH
                                (#x95DA . #x66CF)	;CJK UNIFIED IDEOGRAPH
                                (#x95DB . #x66D0)	;CJK UNIFIED IDEOGRAPH
                                (#x95DC . #x66D1)	;CJK UNIFIED IDEOGRAPH
                                (#x95DD . #x66D2)	;CJK UNIFIED IDEOGRAPH
                                (#x95DE . #x66D3)	;CJK UNIFIED IDEOGRAPH
                                (#x95DF . #x66D4)	;CJK UNIFIED IDEOGRAPH
                                (#x95E0 . #x66D5)	;CJK UNIFIED IDEOGRAPH
                                (#x95E1 . #x66D6)	;CJK UNIFIED IDEOGRAPH
                                (#x95E2 . #x66D7)	;CJK UNIFIED IDEOGRAPH
                                (#x95E3 . #x66D8)	;CJK UNIFIED IDEOGRAPH
                                (#x95E4 . #x66DA)	;CJK UNIFIED IDEOGRAPH
                                (#x95E5 . #x66DE)	;CJK UNIFIED IDEOGRAPH
                                (#x95E6 . #x66DF)	;CJK UNIFIED IDEOGRAPH
                                (#x95E7 . #x66E0)	;CJK UNIFIED IDEOGRAPH
                                (#x95E8 . #x66E1)	;CJK UNIFIED IDEOGRAPH
                                (#x95E9 . #x66E2)	;CJK UNIFIED IDEOGRAPH
                                (#x95EA . #x66E3)	;CJK UNIFIED IDEOGRAPH
                                (#x95EB . #x66E4)	;CJK UNIFIED IDEOGRAPH
                                (#x95EC . #x66E5)	;CJK UNIFIED IDEOGRAPH
                                (#x95ED . #x66E7)	;CJK UNIFIED IDEOGRAPH
                                (#x95EE . #x66E8)	;CJK UNIFIED IDEOGRAPH
                                (#x95EF . #x66EA)	;CJK UNIFIED IDEOGRAPH
                                (#x95F0 . #x66EB)	;CJK UNIFIED IDEOGRAPH
                                (#x95F1 . #x66EC)	;CJK UNIFIED IDEOGRAPH
                                (#x95F2 . #x66ED)	;CJK UNIFIED IDEOGRAPH
                                (#x95F3 . #x66EE)	;CJK UNIFIED IDEOGRAPH
                                (#x95F4 . #x66EF)	;CJK UNIFIED IDEOGRAPH
                                (#x95F5 . #x66F1)	;CJK UNIFIED IDEOGRAPH
                                (#x95F6 . #x66F5)	;CJK UNIFIED IDEOGRAPH
                                (#x95F7 . #x66F6)	;CJK UNIFIED IDEOGRAPH
                                (#x95F8 . #x66F8)	;CJK UNIFIED IDEOGRAPH
                                (#x95F9 . #x66FA)	;CJK UNIFIED IDEOGRAPH
                                (#x95FA . #x66FB)	;CJK UNIFIED IDEOGRAPH
                                (#x95FB . #x66FD)	;CJK UNIFIED IDEOGRAPH
                                (#x95FC . #x6701)	;CJK UNIFIED IDEOGRAPH
                                (#x95FD . #x6702)	;CJK UNIFIED IDEOGRAPH
                                (#x95FE . #x6703)	;CJK UNIFIED IDEOGRAPH
                                (#x9640 . #x6704)	;CJK UNIFIED IDEOGRAPH
                                (#x9641 . #x6705)	;CJK UNIFIED IDEOGRAPH
                                (#x9642 . #x6706)	;CJK UNIFIED IDEOGRAPH
                                (#x9643 . #x6707)	;CJK UNIFIED IDEOGRAPH
                                (#x9644 . #x670C)	;CJK UNIFIED IDEOGRAPH
                                (#x9645 . #x670E)	;CJK UNIFIED IDEOGRAPH
                                (#x9646 . #x670F)	;CJK UNIFIED IDEOGRAPH
                                (#x9647 . #x6711)	;CJK UNIFIED IDEOGRAPH
                                (#x9648 . #x6712)	;CJK UNIFIED IDEOGRAPH
                                (#x9649 . #x6713)	;CJK UNIFIED IDEOGRAPH
                                (#x964A . #x6716)	;CJK UNIFIED IDEOGRAPH
                                (#x964B . #x6718)	;CJK UNIFIED IDEOGRAPH
                                (#x964C . #x6719)	;CJK UNIFIED IDEOGRAPH
                                (#x964D . #x671A)	;CJK UNIFIED IDEOGRAPH
                                (#x964E . #x671C)	;CJK UNIFIED IDEOGRAPH
                                (#x964F . #x671E)	;CJK UNIFIED IDEOGRAPH
                                (#x9650 . #x6720)	;CJK UNIFIED IDEOGRAPH
                                (#x9651 . #x6721)	;CJK UNIFIED IDEOGRAPH
                                (#x9652 . #x6722)	;CJK UNIFIED IDEOGRAPH
                                (#x9653 . #x6723)	;CJK UNIFIED IDEOGRAPH
                                (#x9654 . #x6724)	;CJK UNIFIED IDEOGRAPH
                                (#x9655 . #x6725)	;CJK UNIFIED IDEOGRAPH
                                (#x9656 . #x6727)	;CJK UNIFIED IDEOGRAPH
                                (#x9657 . #x6729)	;CJK UNIFIED IDEOGRAPH
                                (#x9658 . #x672E)	;CJK UNIFIED IDEOGRAPH
                                (#x9659 . #x6730)	;CJK UNIFIED IDEOGRAPH
                                (#x965A . #x6732)	;CJK UNIFIED IDEOGRAPH
                                (#x965B . #x6733)	;CJK UNIFIED IDEOGRAPH
                                (#x965C . #x6736)	;CJK UNIFIED IDEOGRAPH
                                (#x965D . #x6737)	;CJK UNIFIED IDEOGRAPH
                                (#x965E . #x6738)	;CJK UNIFIED IDEOGRAPH
                                (#x965F . #x6739)	;CJK UNIFIED IDEOGRAPH
                                (#x9660 . #x673B)	;CJK UNIFIED IDEOGRAPH
                                (#x9661 . #x673C)	;CJK UNIFIED IDEOGRAPH
                                (#x9662 . #x673E)	;CJK UNIFIED IDEOGRAPH
                                (#x9663 . #x673F)	;CJK UNIFIED IDEOGRAPH
                                (#x9664 . #x6741)	;CJK UNIFIED IDEOGRAPH
                                (#x9665 . #x6744)	;CJK UNIFIED IDEOGRAPH
                                (#x9666 . #x6745)	;CJK UNIFIED IDEOGRAPH
                                (#x9667 . #x6747)	;CJK UNIFIED IDEOGRAPH
                                (#x9668 . #x674A)	;CJK UNIFIED IDEOGRAPH
                                (#x9669 . #x674B)	;CJK UNIFIED IDEOGRAPH
                                (#x966A . #x674D)	;CJK UNIFIED IDEOGRAPH
                                (#x966B . #x6752)	;CJK UNIFIED IDEOGRAPH
                                (#x966C . #x6754)	;CJK UNIFIED IDEOGRAPH
                                (#x966D . #x6755)	;CJK UNIFIED IDEOGRAPH
                                (#x966E . #x6757)	;CJK UNIFIED IDEOGRAPH
                                (#x966F . #x6758)	;CJK UNIFIED IDEOGRAPH
                                (#x9670 . #x6759)	;CJK UNIFIED IDEOGRAPH
                                (#x9671 . #x675A)	;CJK UNIFIED IDEOGRAPH
                                (#x9672 . #x675B)	;CJK UNIFIED IDEOGRAPH
                                (#x9673 . #x675D)	;CJK UNIFIED IDEOGRAPH
                                (#x9674 . #x6762)	;CJK UNIFIED IDEOGRAPH
                                (#x9675 . #x6763)	;CJK UNIFIED IDEOGRAPH
                                (#x9676 . #x6764)	;CJK UNIFIED IDEOGRAPH
                                (#x9677 . #x6766)	;CJK UNIFIED IDEOGRAPH
                                (#x9678 . #x6767)	;CJK UNIFIED IDEOGRAPH
                                (#x9679 . #x676B)	;CJK UNIFIED IDEOGRAPH
                                (#x967A . #x676C)	;CJK UNIFIED IDEOGRAPH
                                (#x967B . #x676E)	;CJK UNIFIED IDEOGRAPH
                                (#x967C . #x6771)	;CJK UNIFIED IDEOGRAPH
                                (#x967D . #x6774)	;CJK UNIFIED IDEOGRAPH
                                (#x967E . #x6776)	;CJK UNIFIED IDEOGRAPH
                                (#x9680 . #x6778)	;CJK UNIFIED IDEOGRAPH
                                (#x9681 . #x6779)	;CJK UNIFIED IDEOGRAPH
                                (#x9682 . #x677A)	;CJK UNIFIED IDEOGRAPH
                                (#x9683 . #x677B)	;CJK UNIFIED IDEOGRAPH
                                (#x9684 . #x677D)	;CJK UNIFIED IDEOGRAPH
                                (#x9685 . #x6780)	;CJK UNIFIED IDEOGRAPH
                                (#x9686 . #x6782)	;CJK UNIFIED IDEOGRAPH
                                (#x9687 . #x6783)	;CJK UNIFIED IDEOGRAPH
                                (#x9688 . #x6785)	;CJK UNIFIED IDEOGRAPH
                                (#x9689 . #x6786)	;CJK UNIFIED IDEOGRAPH
                                (#x968A . #x6788)	;CJK UNIFIED IDEOGRAPH
                                (#x968B . #x678A)	;CJK UNIFIED IDEOGRAPH
                                (#x968C . #x678C)	;CJK UNIFIED IDEOGRAPH
                                (#x968D . #x678D)	;CJK UNIFIED IDEOGRAPH
                                (#x968E . #x678E)	;CJK UNIFIED IDEOGRAPH
                                (#x968F . #x678F)	;CJK UNIFIED IDEOGRAPH
                                (#x9690 . #x6791)	;CJK UNIFIED IDEOGRAPH
                                (#x9691 . #x6792)	;CJK UNIFIED IDEOGRAPH
                                (#x9692 . #x6793)	;CJK UNIFIED IDEOGRAPH
                                (#x9693 . #x6794)	;CJK UNIFIED IDEOGRAPH
                                (#x9694 . #x6796)	;CJK UNIFIED IDEOGRAPH
                                (#x9695 . #x6799)	;CJK UNIFIED IDEOGRAPH
                                (#x9696 . #x679B)	;CJK UNIFIED IDEOGRAPH
                                (#x9697 . #x679F)	;CJK UNIFIED IDEOGRAPH
                                (#x9698 . #x67A0)	;CJK UNIFIED IDEOGRAPH
                                (#x9699 . #x67A1)	;CJK UNIFIED IDEOGRAPH
                                (#x969A . #x67A4)	;CJK UNIFIED IDEOGRAPH
                                (#x969B . #x67A6)	;CJK UNIFIED IDEOGRAPH
                                (#x969C . #x67A9)	;CJK UNIFIED IDEOGRAPH
                                (#x969D . #x67AC)	;CJK UNIFIED IDEOGRAPH
                                (#x969E . #x67AE)	;CJK UNIFIED IDEOGRAPH
                                (#x969F . #x67B1)	;CJK UNIFIED IDEOGRAPH
                                (#x96A0 . #x67B2)	;CJK UNIFIED IDEOGRAPH
                                (#x96A1 . #x67B4)	;CJK UNIFIED IDEOGRAPH
                                (#x96A2 . #x67B9)	;CJK UNIFIED IDEOGRAPH
                                (#x96A3 . #x67BA)	;CJK UNIFIED IDEOGRAPH
                                (#x96A4 . #x67BB)	;CJK UNIFIED IDEOGRAPH
                                (#x96A5 . #x67BC)	;CJK UNIFIED IDEOGRAPH
                                (#x96A6 . #x67BD)	;CJK UNIFIED IDEOGRAPH
                                (#x96A7 . #x67BE)	;CJK UNIFIED IDEOGRAPH
                                (#x96A8 . #x67BF)	;CJK UNIFIED IDEOGRAPH
                                (#x96A9 . #x67C0)	;CJK UNIFIED IDEOGRAPH
                                (#x96AA . #x67C2)	;CJK UNIFIED IDEOGRAPH
                                (#x96AB . #x67C5)	;CJK UNIFIED IDEOGRAPH
                                (#x96AC . #x67C6)	;CJK UNIFIED IDEOGRAPH
                                (#x96AD . #x67C7)	;CJK UNIFIED IDEOGRAPH
                                (#x96AE . #x67C8)	;CJK UNIFIED IDEOGRAPH
                                (#x96AF . #x67C9)	;CJK UNIFIED IDEOGRAPH
                                (#x96B0 . #x67CA)	;CJK UNIFIED IDEOGRAPH
                                (#x96B1 . #x67CB)	;CJK UNIFIED IDEOGRAPH
                                (#x96B2 . #x67CC)	;CJK UNIFIED IDEOGRAPH
                                (#x96B3 . #x67CD)	;CJK UNIFIED IDEOGRAPH
                                (#x96B4 . #x67CE)	;CJK UNIFIED IDEOGRAPH
                                (#x96B5 . #x67D5)	;CJK UNIFIED IDEOGRAPH
                                (#x96B6 . #x67D6)	;CJK UNIFIED IDEOGRAPH
                                (#x96B7 . #x67D7)	;CJK UNIFIED IDEOGRAPH
                                (#x96B8 . #x67DB)	;CJK UNIFIED IDEOGRAPH
                                (#x96B9 . #x67DF)	;CJK UNIFIED IDEOGRAPH
                                (#x96BA . #x67E1)	;CJK UNIFIED IDEOGRAPH
                                (#x96BB . #x67E3)	;CJK UNIFIED IDEOGRAPH
                                (#x96BC . #x67E4)	;CJK UNIFIED IDEOGRAPH
                                (#x96BD . #x67E6)	;CJK UNIFIED IDEOGRAPH
                                (#x96BE . #x67E7)	;CJK UNIFIED IDEOGRAPH
                                (#x96BF . #x67E8)	;CJK UNIFIED IDEOGRAPH
                                (#x96C0 . #x67EA)	;CJK UNIFIED IDEOGRAPH
                                (#x96C1 . #x67EB)	;CJK UNIFIED IDEOGRAPH
                                (#x96C2 . #x67ED)	;CJK UNIFIED IDEOGRAPH
                                (#x96C3 . #x67EE)	;CJK UNIFIED IDEOGRAPH
                                (#x96C4 . #x67F2)	;CJK UNIFIED IDEOGRAPH
                                (#x96C5 . #x67F5)	;CJK UNIFIED IDEOGRAPH
                                (#x96C6 . #x67F6)	;CJK UNIFIED IDEOGRAPH
                                (#x96C7 . #x67F7)	;CJK UNIFIED IDEOGRAPH
                                (#x96C8 . #x67F8)	;CJK UNIFIED IDEOGRAPH
                                (#x96C9 . #x67F9)	;CJK UNIFIED IDEOGRAPH
                                (#x96CA . #x67FA)	;CJK UNIFIED IDEOGRAPH
                                (#x96CB . #x67FB)	;CJK UNIFIED IDEOGRAPH
                                (#x96CC . #x67FC)	;CJK UNIFIED IDEOGRAPH
                                (#x96CD . #x67FE)	;CJK UNIFIED IDEOGRAPH
                                (#x96CE . #x6801)	;CJK UNIFIED IDEOGRAPH
                                (#x96CF . #x6802)	;CJK UNIFIED IDEOGRAPH
                                (#x96D0 . #x6803)	;CJK UNIFIED IDEOGRAPH
                                (#x96D1 . #x6804)	;CJK UNIFIED IDEOGRAPH
                                (#x96D2 . #x6806)	;CJK UNIFIED IDEOGRAPH
                                (#x96D3 . #x680D)	;CJK UNIFIED IDEOGRAPH
                                (#x96D4 . #x6810)	;CJK UNIFIED IDEOGRAPH
                                (#x96D5 . #x6812)	;CJK UNIFIED IDEOGRAPH
                                (#x96D6 . #x6814)	;CJK UNIFIED IDEOGRAPH
                                (#x96D7 . #x6815)	;CJK UNIFIED IDEOGRAPH
                                (#x96D8 . #x6818)	;CJK UNIFIED IDEOGRAPH
                                (#x96D9 . #x6819)	;CJK UNIFIED IDEOGRAPH
                                (#x96DA . #x681A)	;CJK UNIFIED IDEOGRAPH
                                (#x96DB . #x681B)	;CJK UNIFIED IDEOGRAPH
                                (#x96DC . #x681C)	;CJK UNIFIED IDEOGRAPH
                                (#x96DD . #x681E)	;CJK UNIFIED IDEOGRAPH
                                (#x96DE . #x681F)	;CJK UNIFIED IDEOGRAPH
                                (#x96DF . #x6820)	;CJK UNIFIED IDEOGRAPH
                                (#x96E0 . #x6822)	;CJK UNIFIED IDEOGRAPH
                                (#x96E1 . #x6823)	;CJK UNIFIED IDEOGRAPH
                                (#x96E2 . #x6824)	;CJK UNIFIED IDEOGRAPH
                                (#x96E3 . #x6825)	;CJK UNIFIED IDEOGRAPH
                                (#x96E4 . #x6826)	;CJK UNIFIED IDEOGRAPH
                                (#x96E5 . #x6827)	;CJK UNIFIED IDEOGRAPH
                                (#x96E6 . #x6828)	;CJK UNIFIED IDEOGRAPH
                                (#x96E7 . #x682B)	;CJK UNIFIED IDEOGRAPH
                                (#x96E8 . #x682C)	;CJK UNIFIED IDEOGRAPH
                                (#x96E9 . #x682D)	;CJK UNIFIED IDEOGRAPH
                                (#x96EA . #x682E)	;CJK UNIFIED IDEOGRAPH
                                (#x96EB . #x682F)	;CJK UNIFIED IDEOGRAPH
                                (#x96EC . #x6830)	;CJK UNIFIED IDEOGRAPH
                                (#x96ED . #x6831)	;CJK UNIFIED IDEOGRAPH
                                (#x96EE . #x6834)	;CJK UNIFIED IDEOGRAPH
                                (#x96EF . #x6835)	;CJK UNIFIED IDEOGRAPH
                                (#x96F0 . #x6836)	;CJK UNIFIED IDEOGRAPH
                                (#x96F1 . #x683A)	;CJK UNIFIED IDEOGRAPH
                                (#x96F2 . #x683B)	;CJK UNIFIED IDEOGRAPH
                                (#x96F3 . #x683F)	;CJK UNIFIED IDEOGRAPH
                                (#x96F4 . #x6847)	;CJK UNIFIED IDEOGRAPH
                                (#x96F5 . #x684B)	;CJK UNIFIED IDEOGRAPH
                                (#x96F6 . #x684D)	;CJK UNIFIED IDEOGRAPH
                                (#x96F7 . #x684F)	;CJK UNIFIED IDEOGRAPH
                                (#x96F8 . #x6852)	;CJK UNIFIED IDEOGRAPH
                                (#x96F9 . #x6856)	;CJK UNIFIED IDEOGRAPH
                                (#x96FA . #x6857)	;CJK UNIFIED IDEOGRAPH
                                (#x96FB . #x6858)	;CJK UNIFIED IDEOGRAPH
                                (#x96FC . #x6859)	;CJK UNIFIED IDEOGRAPH
                                (#x96FD . #x685A)	;CJK UNIFIED IDEOGRAPH
                                (#x96FE . #x685B)	;CJK UNIFIED IDEOGRAPH
                                (#x9740 . #x685C)	;CJK UNIFIED IDEOGRAPH
                                (#x9741 . #x685D)	;CJK UNIFIED IDEOGRAPH
                                (#x9742 . #x685E)	;CJK UNIFIED IDEOGRAPH
                                (#x9743 . #x685F)	;CJK UNIFIED IDEOGRAPH
                                (#x9744 . #x686A)	;CJK UNIFIED IDEOGRAPH
                                (#x9745 . #x686C)	;CJK UNIFIED IDEOGRAPH
                                (#x9746 . #x686D)	;CJK UNIFIED IDEOGRAPH
                                (#x9747 . #x686E)	;CJK UNIFIED IDEOGRAPH
                                (#x9748 . #x686F)	;CJK UNIFIED IDEOGRAPH
                                (#x9749 . #x6870)	;CJK UNIFIED IDEOGRAPH
                                (#x974A . #x6871)	;CJK UNIFIED IDEOGRAPH
                                (#x974B . #x6872)	;CJK UNIFIED IDEOGRAPH
                                (#x974C . #x6873)	;CJK UNIFIED IDEOGRAPH
                                (#x974D . #x6875)	;CJK UNIFIED IDEOGRAPH
                                (#x974E . #x6878)	;CJK UNIFIED IDEOGRAPH
                                (#x974F . #x6879)	;CJK UNIFIED IDEOGRAPH
                                (#x9750 . #x687A)	;CJK UNIFIED IDEOGRAPH
                                (#x9751 . #x687B)	;CJK UNIFIED IDEOGRAPH
                                (#x9752 . #x687C)	;CJK UNIFIED IDEOGRAPH
                                (#x9753 . #x687D)	;CJK UNIFIED IDEOGRAPH
                                (#x9754 . #x687E)	;CJK UNIFIED IDEOGRAPH
                                (#x9755 . #x687F)	;CJK UNIFIED IDEOGRAPH
                                (#x9756 . #x6880)	;CJK UNIFIED IDEOGRAPH
                                (#x9757 . #x6882)	;CJK UNIFIED IDEOGRAPH
                                (#x9758 . #x6884)	;CJK UNIFIED IDEOGRAPH
                                (#x9759 . #x6887)	;CJK UNIFIED IDEOGRAPH
                                (#x975A . #x6888)	;CJK UNIFIED IDEOGRAPH
                                (#x975B . #x6889)	;CJK UNIFIED IDEOGRAPH
                                (#x975C . #x688A)	;CJK UNIFIED IDEOGRAPH
                                (#x975D . #x688B)	;CJK UNIFIED IDEOGRAPH
                                (#x975E . #x688C)	;CJK UNIFIED IDEOGRAPH
                                (#x975F . #x688D)	;CJK UNIFIED IDEOGRAPH
                                (#x9760 . #x688E)	;CJK UNIFIED IDEOGRAPH
                                (#x9761 . #x6890)	;CJK UNIFIED IDEOGRAPH
                                (#x9762 . #x6891)	;CJK UNIFIED IDEOGRAPH
                                (#x9763 . #x6892)	;CJK UNIFIED IDEOGRAPH
                                (#x9764 . #x6894)	;CJK UNIFIED IDEOGRAPH
                                (#x9765 . #x6895)	;CJK UNIFIED IDEOGRAPH
                                (#x9766 . #x6896)	;CJK UNIFIED IDEOGRAPH
                                (#x9767 . #x6898)	;CJK UNIFIED IDEOGRAPH
                                (#x9768 . #x6899)	;CJK UNIFIED IDEOGRAPH
                                (#x9769 . #x689A)	;CJK UNIFIED IDEOGRAPH
                                (#x976A . #x689B)	;CJK UNIFIED IDEOGRAPH
                                (#x976B . #x689C)	;CJK UNIFIED IDEOGRAPH
                                (#x976C . #x689D)	;CJK UNIFIED IDEOGRAPH
                                (#x976D . #x689E)	;CJK UNIFIED IDEOGRAPH
                                (#x976E . #x689F)	;CJK UNIFIED IDEOGRAPH
                                (#x976F . #x68A0)	;CJK UNIFIED IDEOGRAPH
                                (#x9770 . #x68A1)	;CJK UNIFIED IDEOGRAPH
                                (#x9771 . #x68A3)	;CJK UNIFIED IDEOGRAPH
                                (#x9772 . #x68A4)	;CJK UNIFIED IDEOGRAPH
                                (#x9773 . #x68A5)	;CJK UNIFIED IDEOGRAPH
                                (#x9774 . #x68A9)	;CJK UNIFIED IDEOGRAPH
                                (#x9775 . #x68AA)	;CJK UNIFIED IDEOGRAPH
                                (#x9776 . #x68AB)	;CJK UNIFIED IDEOGRAPH
                                (#x9777 . #x68AC)	;CJK UNIFIED IDEOGRAPH
                                (#x9778 . #x68AE)	;CJK UNIFIED IDEOGRAPH
                                (#x9779 . #x68B1)	;CJK UNIFIED IDEOGRAPH
                                (#x977A . #x68B2)	;CJK UNIFIED IDEOGRAPH
                                (#x977B . #x68B4)	;CJK UNIFIED IDEOGRAPH
                                (#x977C . #x68B6)	;CJK UNIFIED IDEOGRAPH
                                (#x977D . #x68B7)	;CJK UNIFIED IDEOGRAPH
                                (#x977E . #x68B8)	;CJK UNIFIED IDEOGRAPH
                                (#x9780 . #x68B9)	;CJK UNIFIED IDEOGRAPH
                                (#x9781 . #x68BA)	;CJK UNIFIED IDEOGRAPH
                                (#x9782 . #x68BB)	;CJK UNIFIED IDEOGRAPH
                                (#x9783 . #x68BC)	;CJK UNIFIED IDEOGRAPH
                                (#x9784 . #x68BD)	;CJK UNIFIED IDEOGRAPH
                                (#x9785 . #x68BE)	;CJK UNIFIED IDEOGRAPH
                                (#x9786 . #x68BF)	;CJK UNIFIED IDEOGRAPH
                                (#x9787 . #x68C1)	;CJK UNIFIED IDEOGRAPH
                                (#x9788 . #x68C3)	;CJK UNIFIED IDEOGRAPH
                                (#x9789 . #x68C4)	;CJK UNIFIED IDEOGRAPH
                                (#x978A . #x68C5)	;CJK UNIFIED IDEOGRAPH
                                (#x978B . #x68C6)	;CJK UNIFIED IDEOGRAPH
                                (#x978C . #x68C7)	;CJK UNIFIED IDEOGRAPH
                                (#x978D . #x68C8)	;CJK UNIFIED IDEOGRAPH
                                (#x978E . #x68CA)	;CJK UNIFIED IDEOGRAPH
                                (#x978F . #x68CC)	;CJK UNIFIED IDEOGRAPH
                                (#x9790 . #x68CE)	;CJK UNIFIED IDEOGRAPH
                                (#x9791 . #x68CF)	;CJK UNIFIED IDEOGRAPH
                                (#x9792 . #x68D0)	;CJK UNIFIED IDEOGRAPH
                                (#x9793 . #x68D1)	;CJK UNIFIED IDEOGRAPH
                                (#x9794 . #x68D3)	;CJK UNIFIED IDEOGRAPH
                                (#x9795 . #x68D4)	;CJK UNIFIED IDEOGRAPH
                                (#x9796 . #x68D6)	;CJK UNIFIED IDEOGRAPH
                                (#x9797 . #x68D7)	;CJK UNIFIED IDEOGRAPH
                                (#x9798 . #x68D9)	;CJK UNIFIED IDEOGRAPH
                                (#x9799 . #x68DB)	;CJK UNIFIED IDEOGRAPH
                                (#x979A . #x68DC)	;CJK UNIFIED IDEOGRAPH
                                (#x979B . #x68DD)	;CJK UNIFIED IDEOGRAPH
                                (#x979C . #x68DE)	;CJK UNIFIED IDEOGRAPH
                                (#x979D . #x68DF)	;CJK UNIFIED IDEOGRAPH
                                (#x979E . #x68E1)	;CJK UNIFIED IDEOGRAPH
                                (#x979F . #x68E2)	;CJK UNIFIED IDEOGRAPH
                                (#x97A0 . #x68E4)	;CJK UNIFIED IDEOGRAPH
                                (#x97A1 . #x68E5)	;CJK UNIFIED IDEOGRAPH
                                (#x97A2 . #x68E6)	;CJK UNIFIED IDEOGRAPH
                                (#x97A3 . #x68E7)	;CJK UNIFIED IDEOGRAPH
                                (#x97A4 . #x68E8)	;CJK UNIFIED IDEOGRAPH
                                (#x97A5 . #x68E9)	;CJK UNIFIED IDEOGRAPH
                                (#x97A6 . #x68EA)	;CJK UNIFIED IDEOGRAPH
                                (#x97A7 . #x68EB)	;CJK UNIFIED IDEOGRAPH
                                (#x97A8 . #x68EC)	;CJK UNIFIED IDEOGRAPH
                                (#x97A9 . #x68ED)	;CJK UNIFIED IDEOGRAPH
                                (#x97AA . #x68EF)	;CJK UNIFIED IDEOGRAPH
                                (#x97AB . #x68F2)	;CJK UNIFIED IDEOGRAPH
                                (#x97AC . #x68F3)	;CJK UNIFIED IDEOGRAPH
                                (#x97AD . #x68F4)	;CJK UNIFIED IDEOGRAPH
                                (#x97AE . #x68F6)	;CJK UNIFIED IDEOGRAPH
                                (#x97AF . #x68F7)	;CJK UNIFIED IDEOGRAPH
                                (#x97B0 . #x68F8)	;CJK UNIFIED IDEOGRAPH
                                (#x97B1 . #x68FB)	;CJK UNIFIED IDEOGRAPH
                                (#x97B2 . #x68FD)	;CJK UNIFIED IDEOGRAPH
                                (#x97B3 . #x68FE)	;CJK UNIFIED IDEOGRAPH
                                (#x97B4 . #x68FF)	;CJK UNIFIED IDEOGRAPH
                                (#x97B5 . #x6900)	;CJK UNIFIED IDEOGRAPH
                                (#x97B6 . #x6902)	;CJK UNIFIED IDEOGRAPH
                                (#x97B7 . #x6903)	;CJK UNIFIED IDEOGRAPH
                                (#x97B8 . #x6904)	;CJK UNIFIED IDEOGRAPH
                                (#x97B9 . #x6906)	;CJK UNIFIED IDEOGRAPH
                                (#x97BA . #x6907)	;CJK UNIFIED IDEOGRAPH
                                (#x97BB . #x6908)	;CJK UNIFIED IDEOGRAPH
                                (#x97BC . #x6909)	;CJK UNIFIED IDEOGRAPH
                                (#x97BD . #x690A)	;CJK UNIFIED IDEOGRAPH
                                (#x97BE . #x690C)	;CJK UNIFIED IDEOGRAPH
                                (#x97BF . #x690F)	;CJK UNIFIED IDEOGRAPH
                                (#x97C0 . #x6911)	;CJK UNIFIED IDEOGRAPH
                                (#x97C1 . #x6913)	;CJK UNIFIED IDEOGRAPH
                                (#x97C2 . #x6914)	;CJK UNIFIED IDEOGRAPH
                                (#x97C3 . #x6915)	;CJK UNIFIED IDEOGRAPH
                                (#x97C4 . #x6916)	;CJK UNIFIED IDEOGRAPH
                                (#x97C5 . #x6917)	;CJK UNIFIED IDEOGRAPH
                                (#x97C6 . #x6918)	;CJK UNIFIED IDEOGRAPH
                                (#x97C7 . #x6919)	;CJK UNIFIED IDEOGRAPH
                                (#x97C8 . #x691A)	;CJK UNIFIED IDEOGRAPH
                                (#x97C9 . #x691B)	;CJK UNIFIED IDEOGRAPH
                                (#x97CA . #x691C)	;CJK UNIFIED IDEOGRAPH
                                (#x97CB . #x691D)	;CJK UNIFIED IDEOGRAPH
                                (#x97CC . #x691E)	;CJK UNIFIED IDEOGRAPH
                                (#x97CD . #x6921)	;CJK UNIFIED IDEOGRAPH
                                (#x97CE . #x6922)	;CJK UNIFIED IDEOGRAPH
                                (#x97CF . #x6923)	;CJK UNIFIED IDEOGRAPH
                                (#x97D0 . #x6925)	;CJK UNIFIED IDEOGRAPH
                                (#x97D1 . #x6926)	;CJK UNIFIED IDEOGRAPH
                                (#x97D2 . #x6927)	;CJK UNIFIED IDEOGRAPH
                                (#x97D3 . #x6928)	;CJK UNIFIED IDEOGRAPH
                                (#x97D4 . #x6929)	;CJK UNIFIED IDEOGRAPH
                                (#x97D5 . #x692A)	;CJK UNIFIED IDEOGRAPH
                                (#x97D6 . #x692B)	;CJK UNIFIED IDEOGRAPH
                                (#x97D7 . #x692C)	;CJK UNIFIED IDEOGRAPH
                                (#x97D8 . #x692E)	;CJK UNIFIED IDEOGRAPH
                                (#x97D9 . #x692F)	;CJK UNIFIED IDEOGRAPH
                                (#x97DA . #x6931)	;CJK UNIFIED IDEOGRAPH
                                (#x97DB . #x6932)	;CJK UNIFIED IDEOGRAPH
                                (#x97DC . #x6933)	;CJK UNIFIED IDEOGRAPH
                                (#x97DD . #x6935)	;CJK UNIFIED IDEOGRAPH
                                (#x97DE . #x6936)	;CJK UNIFIED IDEOGRAPH
                                (#x97DF . #x6937)	;CJK UNIFIED IDEOGRAPH
                                (#x97E0 . #x6938)	;CJK UNIFIED IDEOGRAPH
                                (#x97E1 . #x693A)	;CJK UNIFIED IDEOGRAPH
                                (#x97E2 . #x693B)	;CJK UNIFIED IDEOGRAPH
                                (#x97E3 . #x693C)	;CJK UNIFIED IDEOGRAPH
                                (#x97E4 . #x693E)	;CJK UNIFIED IDEOGRAPH
                                (#x97E5 . #x6940)	;CJK UNIFIED IDEOGRAPH
                                (#x97E6 . #x6941)	;CJK UNIFIED IDEOGRAPH
                                (#x97E7 . #x6943)	;CJK UNIFIED IDEOGRAPH
                                (#x97E8 . #x6944)	;CJK UNIFIED IDEOGRAPH
                                (#x97E9 . #x6945)	;CJK UNIFIED IDEOGRAPH
                                (#x97EA . #x6946)	;CJK UNIFIED IDEOGRAPH
                                (#x97EB . #x6947)	;CJK UNIFIED IDEOGRAPH
                                (#x97EC . #x6948)	;CJK UNIFIED IDEOGRAPH
                                (#x97ED . #x6949)	;CJK UNIFIED IDEOGRAPH
                                (#x97EE . #x694A)	;CJK UNIFIED IDEOGRAPH
                                (#x97EF . #x694B)	;CJK UNIFIED IDEOGRAPH
                                (#x97F0 . #x694C)	;CJK UNIFIED IDEOGRAPH
                                (#x97F1 . #x694D)	;CJK UNIFIED IDEOGRAPH
                                (#x97F2 . #x694E)	;CJK UNIFIED IDEOGRAPH
                                (#x97F3 . #x694F)	;CJK UNIFIED IDEOGRAPH
                                (#x97F4 . #x6950)	;CJK UNIFIED IDEOGRAPH
                                (#x97F5 . #x6951)	;CJK UNIFIED IDEOGRAPH
                                (#x97F6 . #x6952)	;CJK UNIFIED IDEOGRAPH
                                (#x97F7 . #x6953)	;CJK UNIFIED IDEOGRAPH
                                (#x97F8 . #x6955)	;CJK UNIFIED IDEOGRAPH
                                (#x97F9 . #x6956)	;CJK UNIFIED IDEOGRAPH
                                (#x97FA . #x6958)	;CJK UNIFIED IDEOGRAPH
                                (#x97FB . #x6959)	;CJK UNIFIED IDEOGRAPH
                                (#x97FC . #x695B)	;CJK UNIFIED IDEOGRAPH
                                (#x97FD . #x695C)	;CJK UNIFIED IDEOGRAPH
                                (#x97FE . #x695F)	;CJK UNIFIED IDEOGRAPH
                                (#x9840 . #x6961)	;CJK UNIFIED IDEOGRAPH
                                (#x9841 . #x6962)	;CJK UNIFIED IDEOGRAPH
                                (#x9842 . #x6964)	;CJK UNIFIED IDEOGRAPH
                                (#x9843 . #x6965)	;CJK UNIFIED IDEOGRAPH
                                (#x9844 . #x6967)	;CJK UNIFIED IDEOGRAPH
                                (#x9845 . #x6968)	;CJK UNIFIED IDEOGRAPH
                                (#x9846 . #x6969)	;CJK UNIFIED IDEOGRAPH
                                (#x9847 . #x696A)	;CJK UNIFIED IDEOGRAPH
                                (#x9848 . #x696C)	;CJK UNIFIED IDEOGRAPH
                                (#x9849 . #x696D)	;CJK UNIFIED IDEOGRAPH
                                (#x984A . #x696F)	;CJK UNIFIED IDEOGRAPH
                                (#x984B . #x6970)	;CJK UNIFIED IDEOGRAPH
                                (#x984C . #x6972)	;CJK UNIFIED IDEOGRAPH
                                (#x984D . #x6973)	;CJK UNIFIED IDEOGRAPH
                                (#x984E . #x6974)	;CJK UNIFIED IDEOGRAPH
                                (#x984F . #x6975)	;CJK UNIFIED IDEOGRAPH
                                (#x9850 . #x6976)	;CJK UNIFIED IDEOGRAPH
                                (#x9851 . #x697A)	;CJK UNIFIED IDEOGRAPH
                                (#x9852 . #x697B)	;CJK UNIFIED IDEOGRAPH
                                (#x9853 . #x697D)	;CJK UNIFIED IDEOGRAPH
                                (#x9854 . #x697E)	;CJK UNIFIED IDEOGRAPH
                                (#x9855 . #x697F)	;CJK UNIFIED IDEOGRAPH
                                (#x9856 . #x6981)	;CJK UNIFIED IDEOGRAPH
                                (#x9857 . #x6983)	;CJK UNIFIED IDEOGRAPH
                                (#x9858 . #x6985)	;CJK UNIFIED IDEOGRAPH
                                (#x9859 . #x698A)	;CJK UNIFIED IDEOGRAPH
                                (#x985A . #x698B)	;CJK UNIFIED IDEOGRAPH
                                (#x985B . #x698C)	;CJK UNIFIED IDEOGRAPH
                                (#x985C . #x698E)	;CJK UNIFIED IDEOGRAPH
                                (#x985D . #x698F)	;CJK UNIFIED IDEOGRAPH
                                (#x985E . #x6990)	;CJK UNIFIED IDEOGRAPH
                                (#x985F . #x6991)	;CJK UNIFIED IDEOGRAPH
                                (#x9860 . #x6992)	;CJK UNIFIED IDEOGRAPH
                                (#x9861 . #x6993)	;CJK UNIFIED IDEOGRAPH
                                (#x9862 . #x6996)	;CJK UNIFIED IDEOGRAPH
                                (#x9863 . #x6997)	;CJK UNIFIED IDEOGRAPH
                                (#x9864 . #x6999)	;CJK UNIFIED IDEOGRAPH
                                (#x9865 . #x699A)	;CJK UNIFIED IDEOGRAPH
                                (#x9866 . #x699D)	;CJK UNIFIED IDEOGRAPH
                                (#x9867 . #x699E)	;CJK UNIFIED IDEOGRAPH
                                (#x9868 . #x699F)	;CJK UNIFIED IDEOGRAPH
                                (#x9869 . #x69A0)	;CJK UNIFIED IDEOGRAPH
                                (#x986A . #x69A1)	;CJK UNIFIED IDEOGRAPH
                                (#x986B . #x69A2)	;CJK UNIFIED IDEOGRAPH
                                (#x986C . #x69A3)	;CJK UNIFIED IDEOGRAPH
                                (#x986D . #x69A4)	;CJK UNIFIED IDEOGRAPH
                                (#x986E . #x69A5)	;CJK UNIFIED IDEOGRAPH
                                (#x986F . #x69A6)	;CJK UNIFIED IDEOGRAPH
                                (#x9870 . #x69A9)	;CJK UNIFIED IDEOGRAPH
                                (#x9871 . #x69AA)	;CJK UNIFIED IDEOGRAPH
                                (#x9872 . #x69AC)	;CJK UNIFIED IDEOGRAPH
                                (#x9873 . #x69AE)	;CJK UNIFIED IDEOGRAPH
                                (#x9874 . #x69AF)	;CJK UNIFIED IDEOGRAPH
                                (#x9875 . #x69B0)	;CJK UNIFIED IDEOGRAPH
                                (#x9876 . #x69B2)	;CJK UNIFIED IDEOGRAPH
                                (#x9877 . #x69B3)	;CJK UNIFIED IDEOGRAPH
                                (#x9878 . #x69B5)	;CJK UNIFIED IDEOGRAPH
                                (#x9879 . #x69B6)	;CJK UNIFIED IDEOGRAPH
                                (#x987A . #x69B8)	;CJK UNIFIED IDEOGRAPH
                                (#x987B . #x69B9)	;CJK UNIFIED IDEOGRAPH
                                (#x987C . #x69BA)	;CJK UNIFIED IDEOGRAPH
                                (#x987D . #x69BC)	;CJK UNIFIED IDEOGRAPH
                                (#x987E . #x69BD)	;CJK UNIFIED IDEOGRAPH
                                (#x9880 . #x69BE)	;CJK UNIFIED IDEOGRAPH
                                (#x9881 . #x69BF)	;CJK UNIFIED IDEOGRAPH
                                (#x9882 . #x69C0)	;CJK UNIFIED IDEOGRAPH
                                (#x9883 . #x69C2)	;CJK UNIFIED IDEOGRAPH
                                (#x9884 . #x69C3)	;CJK UNIFIED IDEOGRAPH
                                (#x9885 . #x69C4)	;CJK UNIFIED IDEOGRAPH
                                (#x9886 . #x69C5)	;CJK UNIFIED IDEOGRAPH
                                (#x9887 . #x69C6)	;CJK UNIFIED IDEOGRAPH
                                (#x9888 . #x69C7)	;CJK UNIFIED IDEOGRAPH
                                (#x9889 . #x69C8)	;CJK UNIFIED IDEOGRAPH
                                (#x988A . #x69C9)	;CJK UNIFIED IDEOGRAPH
                                (#x988B . #x69CB)	;CJK UNIFIED IDEOGRAPH
                                (#x988C . #x69CD)	;CJK UNIFIED IDEOGRAPH
                                (#x988D . #x69CF)	;CJK UNIFIED IDEOGRAPH
                                (#x988E . #x69D1)	;CJK UNIFIED IDEOGRAPH
                                (#x988F . #x69D2)	;CJK UNIFIED IDEOGRAPH
                                (#x9890 . #x69D3)	;CJK UNIFIED IDEOGRAPH
                                (#x9891 . #x69D5)	;CJK UNIFIED IDEOGRAPH
                                (#x9892 . #x69D6)	;CJK UNIFIED IDEOGRAPH
                                (#x9893 . #x69D7)	;CJK UNIFIED IDEOGRAPH
                                (#x9894 . #x69D8)	;CJK UNIFIED IDEOGRAPH
                                (#x9895 . #x69D9)	;CJK UNIFIED IDEOGRAPH
                                (#x9896 . #x69DA)	;CJK UNIFIED IDEOGRAPH
                                (#x9897 . #x69DC)	;CJK UNIFIED IDEOGRAPH
                                (#x9898 . #x69DD)	;CJK UNIFIED IDEOGRAPH
                                (#x9899 . #x69DE)	;CJK UNIFIED IDEOGRAPH
                                (#x989A . #x69E1)	;CJK UNIFIED IDEOGRAPH
                                (#x989B . #x69E2)	;CJK UNIFIED IDEOGRAPH
                                (#x989C . #x69E3)	;CJK UNIFIED IDEOGRAPH
                                (#x989D . #x69E4)	;CJK UNIFIED IDEOGRAPH
                                (#x989E . #x69E5)	;CJK UNIFIED IDEOGRAPH
                                (#x989F . #x69E6)	;CJK UNIFIED IDEOGRAPH
                                (#x98A0 . #x69E7)	;CJK UNIFIED IDEOGRAPH
                                (#x98A1 . #x69E8)	;CJK UNIFIED IDEOGRAPH
                                (#x98A2 . #x69E9)	;CJK UNIFIED IDEOGRAPH
                                (#x98A3 . #x69EA)	;CJK UNIFIED IDEOGRAPH
                                (#x98A4 . #x69EB)	;CJK UNIFIED IDEOGRAPH
                                (#x98A5 . #x69EC)	;CJK UNIFIED IDEOGRAPH
                                (#x98A6 . #x69EE)	;CJK UNIFIED IDEOGRAPH
                                (#x98A7 . #x69EF)	;CJK UNIFIED IDEOGRAPH
                                (#x98A8 . #x69F0)	;CJK UNIFIED IDEOGRAPH
                                (#x98A9 . #x69F1)	;CJK UNIFIED IDEOGRAPH
                                (#x98AA . #x69F3)	;CJK UNIFIED IDEOGRAPH
                                (#x98AB . #x69F4)	;CJK UNIFIED IDEOGRAPH
                                (#x98AC . #x69F5)	;CJK UNIFIED IDEOGRAPH
                                (#x98AD . #x69F6)	;CJK UNIFIED IDEOGRAPH
                                (#x98AE . #x69F7)	;CJK UNIFIED IDEOGRAPH
                                (#x98AF . #x69F8)	;CJK UNIFIED IDEOGRAPH
                                (#x98B0 . #x69F9)	;CJK UNIFIED IDEOGRAPH
                                (#x98B1 . #x69FA)	;CJK UNIFIED IDEOGRAPH
                                (#x98B2 . #x69FB)	;CJK UNIFIED IDEOGRAPH
                                (#x98B3 . #x69FC)	;CJK UNIFIED IDEOGRAPH
                                (#x98B4 . #x69FE)	;CJK UNIFIED IDEOGRAPH
                                (#x98B5 . #x6A00)	;CJK UNIFIED IDEOGRAPH
                                (#x98B6 . #x6A01)	;CJK UNIFIED IDEOGRAPH
                                (#x98B7 . #x6A02)	;CJK UNIFIED IDEOGRAPH
                                (#x98B8 . #x6A03)	;CJK UNIFIED IDEOGRAPH
                                (#x98B9 . #x6A04)	;CJK UNIFIED IDEOGRAPH
                                (#x98BA . #x6A05)	;CJK UNIFIED IDEOGRAPH
                                (#x98BB . #x6A06)	;CJK UNIFIED IDEOGRAPH
                                (#x98BC . #x6A07)	;CJK UNIFIED IDEOGRAPH
                                (#x98BD . #x6A08)	;CJK UNIFIED IDEOGRAPH
                                (#x98BE . #x6A09)	;CJK UNIFIED IDEOGRAPH
                                (#x98BF . #x6A0B)	;CJK UNIFIED IDEOGRAPH
                                (#x98C0 . #x6A0C)	;CJK UNIFIED IDEOGRAPH
                                (#x98C1 . #x6A0D)	;CJK UNIFIED IDEOGRAPH
                                (#x98C2 . #x6A0E)	;CJK UNIFIED IDEOGRAPH
                                (#x98C3 . #x6A0F)	;CJK UNIFIED IDEOGRAPH
                                (#x98C4 . #x6A10)	;CJK UNIFIED IDEOGRAPH
                                (#x98C5 . #x6A11)	;CJK UNIFIED IDEOGRAPH
                                (#x98C6 . #x6A12)	;CJK UNIFIED IDEOGRAPH
                                (#x98C7 . #x6A13)	;CJK UNIFIED IDEOGRAPH
                                (#x98C8 . #x6A14)	;CJK UNIFIED IDEOGRAPH
                                (#x98C9 . #x6A15)	;CJK UNIFIED IDEOGRAPH
                                (#x98CA . #x6A16)	;CJK UNIFIED IDEOGRAPH
                                (#x98CB . #x6A19)	;CJK UNIFIED IDEOGRAPH
                                (#x98CC . #x6A1A)	;CJK UNIFIED IDEOGRAPH
                                (#x98CD . #x6A1B)	;CJK UNIFIED IDEOGRAPH
                                (#x98CE . #x6A1C)	;CJK UNIFIED IDEOGRAPH
                                (#x98CF . #x6A1D)	;CJK UNIFIED IDEOGRAPH
                                (#x98D0 . #x6A1E)	;CJK UNIFIED IDEOGRAPH
                                (#x98D1 . #x6A20)	;CJK UNIFIED IDEOGRAPH
                                (#x98D2 . #x6A22)	;CJK UNIFIED IDEOGRAPH
                                (#x98D3 . #x6A23)	;CJK UNIFIED IDEOGRAPH
                                (#x98D4 . #x6A24)	;CJK UNIFIED IDEOGRAPH
                                (#x98D5 . #x6A25)	;CJK UNIFIED IDEOGRAPH
                                (#x98D6 . #x6A26)	;CJK UNIFIED IDEOGRAPH
                                (#x98D7 . #x6A27)	;CJK UNIFIED IDEOGRAPH
                                (#x98D8 . #x6A29)	;CJK UNIFIED IDEOGRAPH
                                (#x98D9 . #x6A2B)	;CJK UNIFIED IDEOGRAPH
                                (#x98DA . #x6A2C)	;CJK UNIFIED IDEOGRAPH
                                (#x98DB . #x6A2D)	;CJK UNIFIED IDEOGRAPH
                                (#x98DC . #x6A2E)	;CJK UNIFIED IDEOGRAPH
                                (#x98DD . #x6A30)	;CJK UNIFIED IDEOGRAPH
                                (#x98DE . #x6A32)	;CJK UNIFIED IDEOGRAPH
                                (#x98DF . #x6A33)	;CJK UNIFIED IDEOGRAPH
                                (#x98E0 . #x6A34)	;CJK UNIFIED IDEOGRAPH
                                (#x98E1 . #x6A36)	;CJK UNIFIED IDEOGRAPH
                                (#x98E2 . #x6A37)	;CJK UNIFIED IDEOGRAPH
                                (#x98E3 . #x6A38)	;CJK UNIFIED IDEOGRAPH
                                (#x98E4 . #x6A39)	;CJK UNIFIED IDEOGRAPH
                                (#x98E5 . #x6A3A)	;CJK UNIFIED IDEOGRAPH
                                (#x98E6 . #x6A3B)	;CJK UNIFIED IDEOGRAPH
                                (#x98E7 . #x6A3C)	;CJK UNIFIED IDEOGRAPH
                                (#x98E8 . #x6A3F)	;CJK UNIFIED IDEOGRAPH
                                (#x98E9 . #x6A40)	;CJK UNIFIED IDEOGRAPH
                                (#x98EA . #x6A41)	;CJK UNIFIED IDEOGRAPH
                                (#x98EB . #x6A42)	;CJK UNIFIED IDEOGRAPH
                                (#x98EC . #x6A43)	;CJK UNIFIED IDEOGRAPH
                                (#x98ED . #x6A45)	;CJK UNIFIED IDEOGRAPH
                                (#x98EE . #x6A46)	;CJK UNIFIED IDEOGRAPH
                                (#x98EF . #x6A48)	;CJK UNIFIED IDEOGRAPH
                                (#x98F0 . #x6A49)	;CJK UNIFIED IDEOGRAPH
                                (#x98F1 . #x6A4A)	;CJK UNIFIED IDEOGRAPH
                                (#x98F2 . #x6A4B)	;CJK UNIFIED IDEOGRAPH
                                (#x98F3 . #x6A4C)	;CJK UNIFIED IDEOGRAPH
                                (#x98F4 . #x6A4D)	;CJK UNIFIED IDEOGRAPH
                                (#x98F5 . #x6A4E)	;CJK UNIFIED IDEOGRAPH
                                (#x98F6 . #x6A4F)	;CJK UNIFIED IDEOGRAPH
                                (#x98F7 . #x6A51)	;CJK UNIFIED IDEOGRAPH
                                (#x98F8 . #x6A52)	;CJK UNIFIED IDEOGRAPH
                                (#x98F9 . #x6A53)	;CJK UNIFIED IDEOGRAPH
                                (#x98FA . #x6A54)	;CJK UNIFIED IDEOGRAPH
                                (#x98FB . #x6A55)	;CJK UNIFIED IDEOGRAPH
                                (#x98FC . #x6A56)	;CJK UNIFIED IDEOGRAPH
                                (#x98FD . #x6A57)	;CJK UNIFIED IDEOGRAPH
                                (#x98FE . #x6A5A)	;CJK UNIFIED IDEOGRAPH
                                (#x9940 . #x6A5C)	;CJK UNIFIED IDEOGRAPH
                                (#x9941 . #x6A5D)	;CJK UNIFIED IDEOGRAPH
                                (#x9942 . #x6A5E)	;CJK UNIFIED IDEOGRAPH
                                (#x9943 . #x6A5F)	;CJK UNIFIED IDEOGRAPH
                                (#x9944 . #x6A60)	;CJK UNIFIED IDEOGRAPH
                                (#x9945 . #x6A62)	;CJK UNIFIED IDEOGRAPH
                                (#x9946 . #x6A63)	;CJK UNIFIED IDEOGRAPH
                                (#x9947 . #x6A64)	;CJK UNIFIED IDEOGRAPH
                                (#x9948 . #x6A66)	;CJK UNIFIED IDEOGRAPH
                                (#x9949 . #x6A67)	;CJK UNIFIED IDEOGRAPH
                                (#x994A . #x6A68)	;CJK UNIFIED IDEOGRAPH
                                (#x994B . #x6A69)	;CJK UNIFIED IDEOGRAPH
                                (#x994C . #x6A6A)	;CJK UNIFIED IDEOGRAPH
                                (#x994D . #x6A6B)	;CJK UNIFIED IDEOGRAPH
                                (#x994E . #x6A6C)	;CJK UNIFIED IDEOGRAPH
                                (#x994F . #x6A6D)	;CJK UNIFIED IDEOGRAPH
                                (#x9950 . #x6A6E)	;CJK UNIFIED IDEOGRAPH
                                (#x9951 . #x6A6F)	;CJK UNIFIED IDEOGRAPH
                                (#x9952 . #x6A70)	;CJK UNIFIED IDEOGRAPH
                                (#x9953 . #x6A72)	;CJK UNIFIED IDEOGRAPH
                                (#x9954 . #x6A73)	;CJK UNIFIED IDEOGRAPH
                                (#x9955 . #x6A74)	;CJK UNIFIED IDEOGRAPH
                                (#x9956 . #x6A75)	;CJK UNIFIED IDEOGRAPH
                                (#x9957 . #x6A76)	;CJK UNIFIED IDEOGRAPH
                                (#x9958 . #x6A77)	;CJK UNIFIED IDEOGRAPH
                                (#x9959 . #x6A78)	;CJK UNIFIED IDEOGRAPH
                                (#x995A . #x6A7A)	;CJK UNIFIED IDEOGRAPH
                                (#x995B . #x6A7B)	;CJK UNIFIED IDEOGRAPH
                                (#x995C . #x6A7D)	;CJK UNIFIED IDEOGRAPH
                                (#x995D . #x6A7E)	;CJK UNIFIED IDEOGRAPH
                                (#x995E . #x6A7F)	;CJK UNIFIED IDEOGRAPH
                                (#x995F . #x6A81)	;CJK UNIFIED IDEOGRAPH
                                (#x9960 . #x6A82)	;CJK UNIFIED IDEOGRAPH
                                (#x9961 . #x6A83)	;CJK UNIFIED IDEOGRAPH
                                (#x9962 . #x6A85)	;CJK UNIFIED IDEOGRAPH
                                (#x9963 . #x6A86)	;CJK UNIFIED IDEOGRAPH
                                (#x9964 . #x6A87)	;CJK UNIFIED IDEOGRAPH
                                (#x9965 . #x6A88)	;CJK UNIFIED IDEOGRAPH
                                (#x9966 . #x6A89)	;CJK UNIFIED IDEOGRAPH
                                (#x9967 . #x6A8A)	;CJK UNIFIED IDEOGRAPH
                                (#x9968 . #x6A8B)	;CJK UNIFIED IDEOGRAPH
                                (#x9969 . #x6A8C)	;CJK UNIFIED IDEOGRAPH
                                (#x996A . #x6A8D)	;CJK UNIFIED IDEOGRAPH
                                (#x996B . #x6A8F)	;CJK UNIFIED IDEOGRAPH
                                (#x996C . #x6A92)	;CJK UNIFIED IDEOGRAPH
                                (#x996D . #x6A93)	;CJK UNIFIED IDEOGRAPH
                                (#x996E . #x6A94)	;CJK UNIFIED IDEOGRAPH
                                (#x996F . #x6A95)	;CJK UNIFIED IDEOGRAPH
                                (#x9970 . #x6A96)	;CJK UNIFIED IDEOGRAPH
                                (#x9971 . #x6A98)	;CJK UNIFIED IDEOGRAPH
                                (#x9972 . #x6A99)	;CJK UNIFIED IDEOGRAPH
                                (#x9973 . #x6A9A)	;CJK UNIFIED IDEOGRAPH
                                (#x9974 . #x6A9B)	;CJK UNIFIED IDEOGRAPH
                                (#x9975 . #x6A9C)	;CJK UNIFIED IDEOGRAPH
                                (#x9976 . #x6A9D)	;CJK UNIFIED IDEOGRAPH
                                (#x9977 . #x6A9E)	;CJK UNIFIED IDEOGRAPH
                                (#x9978 . #x6A9F)	;CJK UNIFIED IDEOGRAPH
                                (#x9979 . #x6AA1)	;CJK UNIFIED IDEOGRAPH
                                (#x997A . #x6AA2)	;CJK UNIFIED IDEOGRAPH
                                (#x997B . #x6AA3)	;CJK UNIFIED IDEOGRAPH
                                (#x997C . #x6AA4)	;CJK UNIFIED IDEOGRAPH
                                (#x997D . #x6AA5)	;CJK UNIFIED IDEOGRAPH
                                (#x997E . #x6AA6)	;CJK UNIFIED IDEOGRAPH
                                (#x9980 . #x6AA7)	;CJK UNIFIED IDEOGRAPH
                                (#x9981 . #x6AA8)	;CJK UNIFIED IDEOGRAPH
                                (#x9982 . #x6AAA)	;CJK UNIFIED IDEOGRAPH
                                (#x9983 . #x6AAD)	;CJK UNIFIED IDEOGRAPH
                                (#x9984 . #x6AAE)	;CJK UNIFIED IDEOGRAPH
                                (#x9985 . #x6AAF)	;CJK UNIFIED IDEOGRAPH
                                (#x9986 . #x6AB0)	;CJK UNIFIED IDEOGRAPH
                                (#x9987 . #x6AB1)	;CJK UNIFIED IDEOGRAPH
                                (#x9988 . #x6AB2)	;CJK UNIFIED IDEOGRAPH
                                (#x9989 . #x6AB3)	;CJK UNIFIED IDEOGRAPH
                                (#x998A . #x6AB4)	;CJK UNIFIED IDEOGRAPH
                                (#x998B . #x6AB5)	;CJK UNIFIED IDEOGRAPH
                                (#x998C . #x6AB6)	;CJK UNIFIED IDEOGRAPH
                                (#x998D . #x6AB7)	;CJK UNIFIED IDEOGRAPH
                                (#x998E . #x6AB8)	;CJK UNIFIED IDEOGRAPH
                                (#x998F . #x6AB9)	;CJK UNIFIED IDEOGRAPH
                                (#x9990 . #x6ABA)	;CJK UNIFIED IDEOGRAPH
                                (#x9991 . #x6ABB)	;CJK UNIFIED IDEOGRAPH
                                (#x9992 . #x6ABC)	;CJK UNIFIED IDEOGRAPH
                                (#x9993 . #x6ABD)	;CJK UNIFIED IDEOGRAPH
                                (#x9994 . #x6ABE)	;CJK UNIFIED IDEOGRAPH
                                (#x9995 . #x6ABF)	;CJK UNIFIED IDEOGRAPH
                                (#x9996 . #x6AC0)	;CJK UNIFIED IDEOGRAPH
                                (#x9997 . #x6AC1)	;CJK UNIFIED IDEOGRAPH
                                (#x9998 . #x6AC2)	;CJK UNIFIED IDEOGRAPH
                                (#x9999 . #x6AC3)	;CJK UNIFIED IDEOGRAPH
                                (#x999A . #x6AC4)	;CJK UNIFIED IDEOGRAPH
                                (#x999B . #x6AC5)	;CJK UNIFIED IDEOGRAPH
                                (#x999C . #x6AC6)	;CJK UNIFIED IDEOGRAPH
                                (#x999D . #x6AC7)	;CJK UNIFIED IDEOGRAPH
                                (#x999E . #x6AC8)	;CJK UNIFIED IDEOGRAPH
                                (#x999F . #x6AC9)	;CJK UNIFIED IDEOGRAPH
                                (#x99A0 . #x6ACA)	;CJK UNIFIED IDEOGRAPH
                                (#x99A1 . #x6ACB)	;CJK UNIFIED IDEOGRAPH
                                (#x99A2 . #x6ACC)	;CJK UNIFIED IDEOGRAPH
                                (#x99A3 . #x6ACD)	;CJK UNIFIED IDEOGRAPH
                                (#x99A4 . #x6ACE)	;CJK UNIFIED IDEOGRAPH
                                (#x99A5 . #x6ACF)	;CJK UNIFIED IDEOGRAPH
                                (#x99A6 . #x6AD0)	;CJK UNIFIED IDEOGRAPH
                                (#x99A7 . #x6AD1)	;CJK UNIFIED IDEOGRAPH
                                (#x99A8 . #x6AD2)	;CJK UNIFIED IDEOGRAPH
                                (#x99A9 . #x6AD3)	;CJK UNIFIED IDEOGRAPH
                                (#x99AA . #x6AD4)	;CJK UNIFIED IDEOGRAPH
                                (#x99AB . #x6AD5)	;CJK UNIFIED IDEOGRAPH
                                (#x99AC . #x6AD6)	;CJK UNIFIED IDEOGRAPH
                                (#x99AD . #x6AD7)	;CJK UNIFIED IDEOGRAPH
                                (#x99AE . #x6AD8)	;CJK UNIFIED IDEOGRAPH
                                (#x99AF . #x6AD9)	;CJK UNIFIED IDEOGRAPH
                                (#x99B0 . #x6ADA)	;CJK UNIFIED IDEOGRAPH
                                (#x99B1 . #x6ADB)	;CJK UNIFIED IDEOGRAPH
                                (#x99B2 . #x6ADC)	;CJK UNIFIED IDEOGRAPH
                                (#x99B3 . #x6ADD)	;CJK UNIFIED IDEOGRAPH
                                (#x99B4 . #x6ADE)	;CJK UNIFIED IDEOGRAPH
                                (#x99B5 . #x6ADF)	;CJK UNIFIED IDEOGRAPH
                                (#x99B6 . #x6AE0)	;CJK UNIFIED IDEOGRAPH
                                (#x99B7 . #x6AE1)	;CJK UNIFIED IDEOGRAPH
                                (#x99B8 . #x6AE2)	;CJK UNIFIED IDEOGRAPH
                                (#x99B9 . #x6AE3)	;CJK UNIFIED IDEOGRAPH
                                (#x99BA . #x6AE4)	;CJK UNIFIED IDEOGRAPH
                                (#x99BB . #x6AE5)	;CJK UNIFIED IDEOGRAPH
                                (#x99BC . #x6AE6)	;CJK UNIFIED IDEOGRAPH
                                (#x99BD . #x6AE7)	;CJK UNIFIED IDEOGRAPH
                                (#x99BE . #x6AE8)	;CJK UNIFIED IDEOGRAPH
                                (#x99BF . #x6AE9)	;CJK UNIFIED IDEOGRAPH
                                (#x99C0 . #x6AEA)	;CJK UNIFIED IDEOGRAPH
                                (#x99C1 . #x6AEB)	;CJK UNIFIED IDEOGRAPH
                                (#x99C2 . #x6AEC)	;CJK UNIFIED IDEOGRAPH
                                (#x99C3 . #x6AED)	;CJK UNIFIED IDEOGRAPH
                                (#x99C4 . #x6AEE)	;CJK UNIFIED IDEOGRAPH
                                (#x99C5 . #x6AEF)	;CJK UNIFIED IDEOGRAPH
                                (#x99C6 . #x6AF0)	;CJK UNIFIED IDEOGRAPH
                                (#x99C7 . #x6AF1)	;CJK UNIFIED IDEOGRAPH
                                (#x99C8 . #x6AF2)	;CJK UNIFIED IDEOGRAPH
                                (#x99C9 . #x6AF3)	;CJK UNIFIED IDEOGRAPH
                                (#x99CA . #x6AF4)	;CJK UNIFIED IDEOGRAPH
                                (#x99CB . #x6AF5)	;CJK UNIFIED IDEOGRAPH
                                (#x99CC . #x6AF6)	;CJK UNIFIED IDEOGRAPH
                                (#x99CD . #x6AF7)	;CJK UNIFIED IDEOGRAPH
                                (#x99CE . #x6AF8)	;CJK UNIFIED IDEOGRAPH
                                (#x99CF . #x6AF9)	;CJK UNIFIED IDEOGRAPH
                                (#x99D0 . #x6AFA)	;CJK UNIFIED IDEOGRAPH
                                (#x99D1 . #x6AFB)	;CJK UNIFIED IDEOGRAPH
                                (#x99D2 . #x6AFC)	;CJK UNIFIED IDEOGRAPH
                                (#x99D3 . #x6AFD)	;CJK UNIFIED IDEOGRAPH
                                (#x99D4 . #x6AFE)	;CJK UNIFIED IDEOGRAPH
                                (#x99D5 . #x6AFF)	;CJK UNIFIED IDEOGRAPH
                                (#x99D6 . #x6B00)	;CJK UNIFIED IDEOGRAPH
                                (#x99D7 . #x6B01)	;CJK UNIFIED IDEOGRAPH
                                (#x99D8 . #x6B02)	;CJK UNIFIED IDEOGRAPH
                                (#x99D9 . #x6B03)	;CJK UNIFIED IDEOGRAPH
                                (#x99DA . #x6B04)	;CJK UNIFIED IDEOGRAPH
                                (#x99DB . #x6B05)	;CJK UNIFIED IDEOGRAPH
                                (#x99DC . #x6B06)	;CJK UNIFIED IDEOGRAPH
                                (#x99DD . #x6B07)	;CJK UNIFIED IDEOGRAPH
                                (#x99DE . #x6B08)	;CJK UNIFIED IDEOGRAPH
                                (#x99DF . #x6B09)	;CJK UNIFIED IDEOGRAPH
                                (#x99E0 . #x6B0A)	;CJK UNIFIED IDEOGRAPH
                                (#x99E1 . #x6B0B)	;CJK UNIFIED IDEOGRAPH
                                (#x99E2 . #x6B0C)	;CJK UNIFIED IDEOGRAPH
                                (#x99E3 . #x6B0D)	;CJK UNIFIED IDEOGRAPH
                                (#x99E4 . #x6B0E)	;CJK UNIFIED IDEOGRAPH
                                (#x99E5 . #x6B0F)	;CJK UNIFIED IDEOGRAPH
                                (#x99E6 . #x6B10)	;CJK UNIFIED IDEOGRAPH
                                (#x99E7 . #x6B11)	;CJK UNIFIED IDEOGRAPH
                                (#x99E8 . #x6B12)	;CJK UNIFIED IDEOGRAPH
                                (#x99E9 . #x6B13)	;CJK UNIFIED IDEOGRAPH
                                (#x99EA . #x6B14)	;CJK UNIFIED IDEOGRAPH
                                (#x99EB . #x6B15)	;CJK UNIFIED IDEOGRAPH
                                (#x99EC . #x6B16)	;CJK UNIFIED IDEOGRAPH
                                (#x99ED . #x6B17)	;CJK UNIFIED IDEOGRAPH
                                (#x99EE . #x6B18)	;CJK UNIFIED IDEOGRAPH
                                (#x99EF . #x6B19)	;CJK UNIFIED IDEOGRAPH
                                (#x99F0 . #x6B1A)	;CJK UNIFIED IDEOGRAPH
                                (#x99F1 . #x6B1B)	;CJK UNIFIED IDEOGRAPH
                                (#x99F2 . #x6B1C)	;CJK UNIFIED IDEOGRAPH
                                (#x99F3 . #x6B1D)	;CJK UNIFIED IDEOGRAPH
                                (#x99F4 . #x6B1E)	;CJK UNIFIED IDEOGRAPH
                                (#x99F5 . #x6B1F)	;CJK UNIFIED IDEOGRAPH
                                (#x99F6 . #x6B25)	;CJK UNIFIED IDEOGRAPH
                                (#x99F7 . #x6B26)	;CJK UNIFIED IDEOGRAPH
                                (#x99F8 . #x6B28)	;CJK UNIFIED IDEOGRAPH
                                (#x99F9 . #x6B29)	;CJK UNIFIED IDEOGRAPH
                                (#x99FA . #x6B2A)	;CJK UNIFIED IDEOGRAPH
                                (#x99FB . #x6B2B)	;CJK UNIFIED IDEOGRAPH
                                (#x99FC . #x6B2C)	;CJK UNIFIED IDEOGRAPH
                                (#x99FD . #x6B2D)	;CJK UNIFIED IDEOGRAPH
                                (#x99FE . #x6B2E)	;CJK UNIFIED IDEOGRAPH
                                (#x9A40 . #x6B2F)	;CJK UNIFIED IDEOGRAPH
                                (#x9A41 . #x6B30)	;CJK UNIFIED IDEOGRAPH
                                (#x9A42 . #x6B31)	;CJK UNIFIED IDEOGRAPH
                                (#x9A43 . #x6B33)	;CJK UNIFIED IDEOGRAPH
                                (#x9A44 . #x6B34)	;CJK UNIFIED IDEOGRAPH
                                (#x9A45 . #x6B35)	;CJK UNIFIED IDEOGRAPH
                                (#x9A46 . #x6B36)	;CJK UNIFIED IDEOGRAPH
                                (#x9A47 . #x6B38)	;CJK UNIFIED IDEOGRAPH
                                (#x9A48 . #x6B3B)	;CJK UNIFIED IDEOGRAPH
                                (#x9A49 . #x6B3C)	;CJK UNIFIED IDEOGRAPH
                                (#x9A4A . #x6B3D)	;CJK UNIFIED IDEOGRAPH
                                (#x9A4B . #x6B3F)	;CJK UNIFIED IDEOGRAPH
                                (#x9A4C . #x6B40)	;CJK UNIFIED IDEOGRAPH
                                (#x9A4D . #x6B41)	;CJK UNIFIED IDEOGRAPH
                                (#x9A4E . #x6B42)	;CJK UNIFIED IDEOGRAPH
                                (#x9A4F . #x6B44)	;CJK UNIFIED IDEOGRAPH
                                (#x9A50 . #x6B45)	;CJK UNIFIED IDEOGRAPH
                                (#x9A51 . #x6B48)	;CJK UNIFIED IDEOGRAPH
                                (#x9A52 . #x6B4A)	;CJK UNIFIED IDEOGRAPH
                                (#x9A53 . #x6B4B)	;CJK UNIFIED IDEOGRAPH
                                (#x9A54 . #x6B4D)	;CJK UNIFIED IDEOGRAPH
                                (#x9A55 . #x6B4E)	;CJK UNIFIED IDEOGRAPH
                                (#x9A56 . #x6B4F)	;CJK UNIFIED IDEOGRAPH
                                (#x9A57 . #x6B50)	;CJK UNIFIED IDEOGRAPH
                                (#x9A58 . #x6B51)	;CJK UNIFIED IDEOGRAPH
                                (#x9A59 . #x6B52)	;CJK UNIFIED IDEOGRAPH
                                (#x9A5A . #x6B53)	;CJK UNIFIED IDEOGRAPH
                                (#x9A5B . #x6B54)	;CJK UNIFIED IDEOGRAPH
                                (#x9A5C . #x6B55)	;CJK UNIFIED IDEOGRAPH
                                (#x9A5D . #x6B56)	;CJK UNIFIED IDEOGRAPH
                                (#x9A5E . #x6B57)	;CJK UNIFIED IDEOGRAPH
                                (#x9A5F . #x6B58)	;CJK UNIFIED IDEOGRAPH
                                (#x9A60 . #x6B5A)	;CJK UNIFIED IDEOGRAPH
                                (#x9A61 . #x6B5B)	;CJK UNIFIED IDEOGRAPH
                                (#x9A62 . #x6B5C)	;CJK UNIFIED IDEOGRAPH
                                (#x9A63 . #x6B5D)	;CJK UNIFIED IDEOGRAPH
                                (#x9A64 . #x6B5E)	;CJK UNIFIED IDEOGRAPH
                                (#x9A65 . #x6B5F)	;CJK UNIFIED IDEOGRAPH
                                (#x9A66 . #x6B60)	;CJK UNIFIED IDEOGRAPH
                                (#x9A67 . #x6B61)	;CJK UNIFIED IDEOGRAPH
                                (#x9A68 . #x6B68)	;CJK UNIFIED IDEOGRAPH
                                (#x9A69 . #x6B69)	;CJK UNIFIED IDEOGRAPH
                                (#x9A6A . #x6B6B)	;CJK UNIFIED IDEOGRAPH
                                (#x9A6B . #x6B6C)	;CJK UNIFIED IDEOGRAPH
                                (#x9A6C . #x6B6D)	;CJK UNIFIED IDEOGRAPH
                                (#x9A6D . #x6B6E)	;CJK UNIFIED IDEOGRAPH
                                (#x9A6E . #x6B6F)	;CJK UNIFIED IDEOGRAPH
                                (#x9A6F . #x6B70)	;CJK UNIFIED IDEOGRAPH
                                (#x9A70 . #x6B71)	;CJK UNIFIED IDEOGRAPH
                                (#x9A71 . #x6B72)	;CJK UNIFIED IDEOGRAPH
                                (#x9A72 . #x6B73)	;CJK UNIFIED IDEOGRAPH
                                (#x9A73 . #x6B74)	;CJK UNIFIED IDEOGRAPH
                                (#x9A74 . #x6B75)	;CJK UNIFIED IDEOGRAPH
                                (#x9A75 . #x6B76)	;CJK UNIFIED IDEOGRAPH
                                (#x9A76 . #x6B77)	;CJK UNIFIED IDEOGRAPH
                                (#x9A77 . #x6B78)	;CJK UNIFIED IDEOGRAPH
                                (#x9A78 . #x6B7A)	;CJK UNIFIED IDEOGRAPH
                                (#x9A79 . #x6B7D)	;CJK UNIFIED IDEOGRAPH
                                (#x9A7A . #x6B7E)	;CJK UNIFIED IDEOGRAPH
                                (#x9A7B . #x6B7F)	;CJK UNIFIED IDEOGRAPH
                                (#x9A7C . #x6B80)	;CJK UNIFIED IDEOGRAPH
                                (#x9A7D . #x6B85)	;CJK UNIFIED IDEOGRAPH
                                (#x9A7E . #x6B88)	;CJK UNIFIED IDEOGRAPH
                                (#x9A80 . #x6B8C)	;CJK UNIFIED IDEOGRAPH
                                (#x9A81 . #x6B8E)	;CJK UNIFIED IDEOGRAPH
                                (#x9A82 . #x6B8F)	;CJK UNIFIED IDEOGRAPH
                                (#x9A83 . #x6B90)	;CJK UNIFIED IDEOGRAPH
                                (#x9A84 . #x6B91)	;CJK UNIFIED IDEOGRAPH
                                (#x9A85 . #x6B94)	;CJK UNIFIED IDEOGRAPH
                                (#x9A86 . #x6B95)	;CJK UNIFIED IDEOGRAPH
                                (#x9A87 . #x6B97)	;CJK UNIFIED IDEOGRAPH
                                (#x9A88 . #x6B98)	;CJK UNIFIED IDEOGRAPH
                                (#x9A89 . #x6B99)	;CJK UNIFIED IDEOGRAPH
                                (#x9A8A . #x6B9C)	;CJK UNIFIED IDEOGRAPH
                                (#x9A8B . #x6B9D)	;CJK UNIFIED IDEOGRAPH
                                (#x9A8C . #x6B9E)	;CJK UNIFIED IDEOGRAPH
                                (#x9A8D . #x6B9F)	;CJK UNIFIED IDEOGRAPH
                                (#x9A8E . #x6BA0)	;CJK UNIFIED IDEOGRAPH
                                (#x9A8F . #x6BA2)	;CJK UNIFIED IDEOGRAPH
                                (#x9A90 . #x6BA3)	;CJK UNIFIED IDEOGRAPH
                                (#x9A91 . #x6BA4)	;CJK UNIFIED IDEOGRAPH
                                (#x9A92 . #x6BA5)	;CJK UNIFIED IDEOGRAPH
                                (#x9A93 . #x6BA6)	;CJK UNIFIED IDEOGRAPH
                                (#x9A94 . #x6BA7)	;CJK UNIFIED IDEOGRAPH
                                (#x9A95 . #x6BA8)	;CJK UNIFIED IDEOGRAPH
                                (#x9A96 . #x6BA9)	;CJK UNIFIED IDEOGRAPH
                                (#x9A97 . #x6BAB)	;CJK UNIFIED IDEOGRAPH
                                (#x9A98 . #x6BAC)	;CJK UNIFIED IDEOGRAPH
                                (#x9A99 . #x6BAD)	;CJK UNIFIED IDEOGRAPH
                                (#x9A9A . #x6BAE)	;CJK UNIFIED IDEOGRAPH
                                (#x9A9B . #x6BAF)	;CJK UNIFIED IDEOGRAPH
                                (#x9A9C . #x6BB0)	;CJK UNIFIED IDEOGRAPH
                                (#x9A9D . #x6BB1)	;CJK UNIFIED IDEOGRAPH
                                (#x9A9E . #x6BB2)	;CJK UNIFIED IDEOGRAPH
                                (#x9A9F . #x6BB6)	;CJK UNIFIED IDEOGRAPH
                                (#x9AA0 . #x6BB8)	;CJK UNIFIED IDEOGRAPH
                                (#x9AA1 . #x6BB9)	;CJK UNIFIED IDEOGRAPH
                                (#x9AA2 . #x6BBA)	;CJK UNIFIED IDEOGRAPH
                                (#x9AA3 . #x6BBB)	;CJK UNIFIED IDEOGRAPH
                                (#x9AA4 . #x6BBC)	;CJK UNIFIED IDEOGRAPH
                                (#x9AA5 . #x6BBD)	;CJK UNIFIED IDEOGRAPH
                                (#x9AA6 . #x6BBE)	;CJK UNIFIED IDEOGRAPH
                                (#x9AA7 . #x6BC0)	;CJK UNIFIED IDEOGRAPH
                                (#x9AA8 . #x6BC3)	;CJK UNIFIED IDEOGRAPH
                                (#x9AA9 . #x6BC4)	;CJK UNIFIED IDEOGRAPH
                                (#x9AAA . #x6BC6)	;CJK UNIFIED IDEOGRAPH
                                (#x9AAB . #x6BC7)	;CJK UNIFIED IDEOGRAPH
                                (#x9AAC . #x6BC8)	;CJK UNIFIED IDEOGRAPH
                                (#x9AAD . #x6BC9)	;CJK UNIFIED IDEOGRAPH
                                (#x9AAE . #x6BCA)	;CJK UNIFIED IDEOGRAPH
                                (#x9AAF . #x6BCC)	;CJK UNIFIED IDEOGRAPH
                                (#x9AB0 . #x6BCE)	;CJK UNIFIED IDEOGRAPH
                                (#x9AB1 . #x6BD0)	;CJK UNIFIED IDEOGRAPH
                                (#x9AB2 . #x6BD1)	;CJK UNIFIED IDEOGRAPH
                                (#x9AB3 . #x6BD8)	;CJK UNIFIED IDEOGRAPH
                                (#x9AB4 . #x6BDA)	;CJK UNIFIED IDEOGRAPH
                                (#x9AB5 . #x6BDC)	;CJK UNIFIED IDEOGRAPH
                                (#x9AB6 . #x6BDD)	;CJK UNIFIED IDEOGRAPH
                                (#x9AB7 . #x6BDE)	;CJK UNIFIED IDEOGRAPH
                                (#x9AB8 . #x6BDF)	;CJK UNIFIED IDEOGRAPH
                                (#x9AB9 . #x6BE0)	;CJK UNIFIED IDEOGRAPH
                                (#x9ABA . #x6BE2)	;CJK UNIFIED IDEOGRAPH
                                (#x9ABB . #x6BE3)	;CJK UNIFIED IDEOGRAPH
                                (#x9ABC . #x6BE4)	;CJK UNIFIED IDEOGRAPH
                                (#x9ABD . #x6BE5)	;CJK UNIFIED IDEOGRAPH
                                (#x9ABE . #x6BE6)	;CJK UNIFIED IDEOGRAPH
                                (#x9ABF . #x6BE7)	;CJK UNIFIED IDEOGRAPH
                                (#x9AC0 . #x6BE8)	;CJK UNIFIED IDEOGRAPH
                                (#x9AC1 . #x6BE9)	;CJK UNIFIED IDEOGRAPH
                                (#x9AC2 . #x6BEC)	;CJK UNIFIED IDEOGRAPH
                                (#x9AC3 . #x6BED)	;CJK UNIFIED IDEOGRAPH
                                (#x9AC4 . #x6BEE)	;CJK UNIFIED IDEOGRAPH
                                (#x9AC5 . #x6BF0)	;CJK UNIFIED IDEOGRAPH
                                (#x9AC6 . #x6BF1)	;CJK UNIFIED IDEOGRAPH
                                (#x9AC7 . #x6BF2)	;CJK UNIFIED IDEOGRAPH
                                (#x9AC8 . #x6BF4)	;CJK UNIFIED IDEOGRAPH
                                (#x9AC9 . #x6BF6)	;CJK UNIFIED IDEOGRAPH
                                (#x9ACA . #x6BF7)	;CJK UNIFIED IDEOGRAPH
                                (#x9ACB . #x6BF8)	;CJK UNIFIED IDEOGRAPH
                                (#x9ACC . #x6BFA)	;CJK UNIFIED IDEOGRAPH
                                (#x9ACD . #x6BFB)	;CJK UNIFIED IDEOGRAPH
                                (#x9ACE . #x6BFC)	;CJK UNIFIED IDEOGRAPH
                                (#x9ACF . #x6BFE)	;CJK UNIFIED IDEOGRAPH
                                (#x9AD0 . #x6BFF)	;CJK UNIFIED IDEOGRAPH
                                (#x9AD1 . #x6C00)	;CJK UNIFIED IDEOGRAPH
                                (#x9AD2 . #x6C01)	;CJK UNIFIED IDEOGRAPH
                                (#x9AD3 . #x6C02)	;CJK UNIFIED IDEOGRAPH
                                (#x9AD4 . #x6C03)	;CJK UNIFIED IDEOGRAPH
                                (#x9AD5 . #x6C04)	;CJK UNIFIED IDEOGRAPH
                                (#x9AD6 . #x6C08)	;CJK UNIFIED IDEOGRAPH
                                (#x9AD7 . #x6C09)	;CJK UNIFIED IDEOGRAPH
                                (#x9AD8 . #x6C0A)	;CJK UNIFIED IDEOGRAPH
                                (#x9AD9 . #x6C0B)	;CJK UNIFIED IDEOGRAPH
                                (#x9ADA . #x6C0C)	;CJK UNIFIED IDEOGRAPH
                                (#x9ADB . #x6C0E)	;CJK UNIFIED IDEOGRAPH
                                (#x9ADC . #x6C12)	;CJK UNIFIED IDEOGRAPH
                                (#x9ADD . #x6C17)	;CJK UNIFIED IDEOGRAPH
                                (#x9ADE . #x6C1C)	;CJK UNIFIED IDEOGRAPH
                                (#x9ADF . #x6C1D)	;CJK UNIFIED IDEOGRAPH
                                (#x9AE0 . #x6C1E)	;CJK UNIFIED IDEOGRAPH
                                (#x9AE1 . #x6C20)	;CJK UNIFIED IDEOGRAPH
                                (#x9AE2 . #x6C23)	;CJK UNIFIED IDEOGRAPH
                                (#x9AE3 . #x6C25)	;CJK UNIFIED IDEOGRAPH
                                (#x9AE4 . #x6C2B)	;CJK UNIFIED IDEOGRAPH
                                (#x9AE5 . #x6C2C)	;CJK UNIFIED IDEOGRAPH
                                (#x9AE6 . #x6C2D)	;CJK UNIFIED IDEOGRAPH
                                (#x9AE7 . #x6C31)	;CJK UNIFIED IDEOGRAPH
                                (#x9AE8 . #x6C33)	;CJK UNIFIED IDEOGRAPH
                                (#x9AE9 . #x6C36)	;CJK UNIFIED IDEOGRAPH
                                (#x9AEA . #x6C37)	;CJK UNIFIED IDEOGRAPH
                                (#x9AEB . #x6C39)	;CJK UNIFIED IDEOGRAPH
                                (#x9AEC . #x6C3A)	;CJK UNIFIED IDEOGRAPH
                                (#x9AED . #x6C3B)	;CJK UNIFIED IDEOGRAPH
                                (#x9AEE . #x6C3C)	;CJK UNIFIED IDEOGRAPH
                                (#x9AEF . #x6C3E)	;CJK UNIFIED IDEOGRAPH
                                (#x9AF0 . #x6C3F)	;CJK UNIFIED IDEOGRAPH
                                (#x9AF1 . #x6C43)	;CJK UNIFIED IDEOGRAPH
                                (#x9AF2 . #x6C44)	;CJK UNIFIED IDEOGRAPH
                                (#x9AF3 . #x6C45)	;CJK UNIFIED IDEOGRAPH
                                (#x9AF4 . #x6C48)	;CJK UNIFIED IDEOGRAPH
                                (#x9AF5 . #x6C4B)	;CJK UNIFIED IDEOGRAPH
                                (#x9AF6 . #x6C4C)	;CJK UNIFIED IDEOGRAPH
                                (#x9AF7 . #x6C4D)	;CJK UNIFIED IDEOGRAPH
                                (#x9AF8 . #x6C4E)	;CJK UNIFIED IDEOGRAPH
                                (#x9AF9 . #x6C4F)	;CJK UNIFIED IDEOGRAPH
                                (#x9AFA . #x6C51)	;CJK UNIFIED IDEOGRAPH
                                (#x9AFB . #x6C52)	;CJK UNIFIED IDEOGRAPH
                                (#x9AFC . #x6C53)	;CJK UNIFIED IDEOGRAPH
                                (#x9AFD . #x6C56)	;CJK UNIFIED IDEOGRAPH
                                (#x9AFE . #x6C58)	;CJK UNIFIED IDEOGRAPH
                                (#x9B40 . #x6C59)	;CJK UNIFIED IDEOGRAPH
                                (#x9B41 . #x6C5A)	;CJK UNIFIED IDEOGRAPH
                                (#x9B42 . #x6C62)	;CJK UNIFIED IDEOGRAPH
                                (#x9B43 . #x6C63)	;CJK UNIFIED IDEOGRAPH
                                (#x9B44 . #x6C65)	;CJK UNIFIED IDEOGRAPH
                                (#x9B45 . #x6C66)	;CJK UNIFIED IDEOGRAPH
                                (#x9B46 . #x6C67)	;CJK UNIFIED IDEOGRAPH
                                (#x9B47 . #x6C6B)	;CJK UNIFIED IDEOGRAPH
                                (#x9B48 . #x6C6C)	;CJK UNIFIED IDEOGRAPH
                                (#x9B49 . #x6C6D)	;CJK UNIFIED IDEOGRAPH
                                (#x9B4A . #x6C6E)	;CJK UNIFIED IDEOGRAPH
                                (#x9B4B . #x6C6F)	;CJK UNIFIED IDEOGRAPH
                                (#x9B4C . #x6C71)	;CJK UNIFIED IDEOGRAPH
                                (#x9B4D . #x6C73)	;CJK UNIFIED IDEOGRAPH
                                (#x9B4E . #x6C75)	;CJK UNIFIED IDEOGRAPH
                                (#x9B4F . #x6C77)	;CJK UNIFIED IDEOGRAPH
                                (#x9B50 . #x6C78)	;CJK UNIFIED IDEOGRAPH
                                (#x9B51 . #x6C7A)	;CJK UNIFIED IDEOGRAPH
                                (#x9B52 . #x6C7B)	;CJK UNIFIED IDEOGRAPH
                                (#x9B53 . #x6C7C)	;CJK UNIFIED IDEOGRAPH
                                (#x9B54 . #x6C7F)	;CJK UNIFIED IDEOGRAPH
                                (#x9B55 . #x6C80)	;CJK UNIFIED IDEOGRAPH
                                (#x9B56 . #x6C84)	;CJK UNIFIED IDEOGRAPH
                                (#x9B57 . #x6C87)	;CJK UNIFIED IDEOGRAPH
                                (#x9B58 . #x6C8A)	;CJK UNIFIED IDEOGRAPH
                                (#x9B59 . #x6C8B)	;CJK UNIFIED IDEOGRAPH
                                (#x9B5A . #x6C8D)	;CJK UNIFIED IDEOGRAPH
                                (#x9B5B . #x6C8E)	;CJK UNIFIED IDEOGRAPH
                                (#x9B5C . #x6C91)	;CJK UNIFIED IDEOGRAPH
                                (#x9B5D . #x6C92)	;CJK UNIFIED IDEOGRAPH
                                (#x9B5E . #x6C95)	;CJK UNIFIED IDEOGRAPH
                                (#x9B5F . #x6C96)	;CJK UNIFIED IDEOGRAPH
                                (#x9B60 . #x6C97)	;CJK UNIFIED IDEOGRAPH
                                (#x9B61 . #x6C98)	;CJK UNIFIED IDEOGRAPH
                                (#x9B62 . #x6C9A)	;CJK UNIFIED IDEOGRAPH
                                (#x9B63 . #x6C9C)	;CJK UNIFIED IDEOGRAPH
                                (#x9B64 . #x6C9D)	;CJK UNIFIED IDEOGRAPH
                                (#x9B65 . #x6C9E)	;CJK UNIFIED IDEOGRAPH
                                (#x9B66 . #x6CA0)	;CJK UNIFIED IDEOGRAPH
                                (#x9B67 . #x6CA2)	;CJK UNIFIED IDEOGRAPH
                                (#x9B68 . #x6CA8)	;CJK UNIFIED IDEOGRAPH
                                (#x9B69 . #x6CAC)	;CJK UNIFIED IDEOGRAPH
                                (#x9B6A . #x6CAF)	;CJK UNIFIED IDEOGRAPH
                                (#x9B6B . #x6CB0)	;CJK UNIFIED IDEOGRAPH
                                (#x9B6C . #x6CB4)	;CJK UNIFIED IDEOGRAPH
                                (#x9B6D . #x6CB5)	;CJK UNIFIED IDEOGRAPH
                                (#x9B6E . #x6CB6)	;CJK UNIFIED IDEOGRAPH
                                (#x9B6F . #x6CB7)	;CJK UNIFIED IDEOGRAPH
                                (#x9B70 . #x6CBA)	;CJK UNIFIED IDEOGRAPH
                                (#x9B71 . #x6CC0)	;CJK UNIFIED IDEOGRAPH
                                (#x9B72 . #x6CC1)	;CJK UNIFIED IDEOGRAPH
                                (#x9B73 . #x6CC2)	;CJK UNIFIED IDEOGRAPH
                                (#x9B74 . #x6CC3)	;CJK UNIFIED IDEOGRAPH
                                (#x9B75 . #x6CC6)	;CJK UNIFIED IDEOGRAPH
                                (#x9B76 . #x6CC7)	;CJK UNIFIED IDEOGRAPH
                                (#x9B77 . #x6CC8)	;CJK UNIFIED IDEOGRAPH
                                (#x9B78 . #x6CCB)	;CJK UNIFIED IDEOGRAPH
                                (#x9B79 . #x6CCD)	;CJK UNIFIED IDEOGRAPH
                                (#x9B7A . #x6CCE)	;CJK UNIFIED IDEOGRAPH
                                (#x9B7B . #x6CCF)	;CJK UNIFIED IDEOGRAPH
                                (#x9B7C . #x6CD1)	;CJK UNIFIED IDEOGRAPH
                                (#x9B7D . #x6CD2)	;CJK UNIFIED IDEOGRAPH
                                (#x9B7E . #x6CD8)	;CJK UNIFIED IDEOGRAPH
                                (#x9B80 . #x6CD9)	;CJK UNIFIED IDEOGRAPH
                                (#x9B81 . #x6CDA)	;CJK UNIFIED IDEOGRAPH
                                (#x9B82 . #x6CDC)	;CJK UNIFIED IDEOGRAPH
                                (#x9B83 . #x6CDD)	;CJK UNIFIED IDEOGRAPH
                                (#x9B84 . #x6CDF)	;CJK UNIFIED IDEOGRAPH
                                (#x9B85 . #x6CE4)	;CJK UNIFIED IDEOGRAPH
                                (#x9B86 . #x6CE6)	;CJK UNIFIED IDEOGRAPH
                                (#x9B87 . #x6CE7)	;CJK UNIFIED IDEOGRAPH
                                (#x9B88 . #x6CE9)	;CJK UNIFIED IDEOGRAPH
                                (#x9B89 . #x6CEC)	;CJK UNIFIED IDEOGRAPH
                                (#x9B8A . #x6CED)	;CJK UNIFIED IDEOGRAPH
                                (#x9B8B . #x6CF2)	;CJK UNIFIED IDEOGRAPH
                                (#x9B8C . #x6CF4)	;CJK UNIFIED IDEOGRAPH
                                (#x9B8D . #x6CF9)	;CJK UNIFIED IDEOGRAPH
                                (#x9B8E . #x6CFF)	;CJK UNIFIED IDEOGRAPH
                                (#x9B8F . #x6D00)	;CJK UNIFIED IDEOGRAPH
                                (#x9B90 . #x6D02)	;CJK UNIFIED IDEOGRAPH
                                (#x9B91 . #x6D03)	;CJK UNIFIED IDEOGRAPH
                                (#x9B92 . #x6D05)	;CJK UNIFIED IDEOGRAPH
                                (#x9B93 . #x6D06)	;CJK UNIFIED IDEOGRAPH
                                (#x9B94 . #x6D08)	;CJK UNIFIED IDEOGRAPH
                                (#x9B95 . #x6D09)	;CJK UNIFIED IDEOGRAPH
                                (#x9B96 . #x6D0A)	;CJK UNIFIED IDEOGRAPH
                                (#x9B97 . #x6D0D)	;CJK UNIFIED IDEOGRAPH
                                (#x9B98 . #x6D0F)	;CJK UNIFIED IDEOGRAPH
                                (#x9B99 . #x6D10)	;CJK UNIFIED IDEOGRAPH
                                (#x9B9A . #x6D11)	;CJK UNIFIED IDEOGRAPH
                                (#x9B9B . #x6D13)	;CJK UNIFIED IDEOGRAPH
                                (#x9B9C . #x6D14)	;CJK UNIFIED IDEOGRAPH
                                (#x9B9D . #x6D15)	;CJK UNIFIED IDEOGRAPH
                                (#x9B9E . #x6D16)	;CJK UNIFIED IDEOGRAPH
                                (#x9B9F . #x6D18)	;CJK UNIFIED IDEOGRAPH
                                (#x9BA0 . #x6D1C)	;CJK UNIFIED IDEOGRAPH
                                (#x9BA1 . #x6D1D)	;CJK UNIFIED IDEOGRAPH
                                (#x9BA2 . #x6D1F)	;CJK UNIFIED IDEOGRAPH
                                (#x9BA3 . #x6D20)	;CJK UNIFIED IDEOGRAPH
                                (#x9BA4 . #x6D21)	;CJK UNIFIED IDEOGRAPH
                                (#x9BA5 . #x6D22)	;CJK UNIFIED IDEOGRAPH
                                (#x9BA6 . #x6D23)	;CJK UNIFIED IDEOGRAPH
                                (#x9BA7 . #x6D24)	;CJK UNIFIED IDEOGRAPH
                                (#x9BA8 . #x6D26)	;CJK UNIFIED IDEOGRAPH
                                (#x9BA9 . #x6D28)	;CJK UNIFIED IDEOGRAPH
                                (#x9BAA . #x6D29)	;CJK UNIFIED IDEOGRAPH
                                (#x9BAB . #x6D2C)	;CJK UNIFIED IDEOGRAPH
                                (#x9BAC . #x6D2D)	;CJK UNIFIED IDEOGRAPH
                                (#x9BAD . #x6D2F)	;CJK UNIFIED IDEOGRAPH
                                (#x9BAE . #x6D30)	;CJK UNIFIED IDEOGRAPH
                                (#x9BAF . #x6D34)	;CJK UNIFIED IDEOGRAPH
                                (#x9BB0 . #x6D36)	;CJK UNIFIED IDEOGRAPH
                                (#x9BB1 . #x6D37)	;CJK UNIFIED IDEOGRAPH
                                (#x9BB2 . #x6D38)	;CJK UNIFIED IDEOGRAPH
                                (#x9BB3 . #x6D3A)	;CJK UNIFIED IDEOGRAPH
                                (#x9BB4 . #x6D3F)	;CJK UNIFIED IDEOGRAPH
                                (#x9BB5 . #x6D40)	;CJK UNIFIED IDEOGRAPH
                                (#x9BB6 . #x6D42)	;CJK UNIFIED IDEOGRAPH
                                (#x9BB7 . #x6D44)	;CJK UNIFIED IDEOGRAPH
                                (#x9BB8 . #x6D49)	;CJK UNIFIED IDEOGRAPH
                                (#x9BB9 . #x6D4C)	;CJK UNIFIED IDEOGRAPH
                                (#x9BBA . #x6D50)	;CJK UNIFIED IDEOGRAPH
                                (#x9BBB . #x6D55)	;CJK UNIFIED IDEOGRAPH
                                (#x9BBC . #x6D56)	;CJK UNIFIED IDEOGRAPH
                                (#x9BBD . #x6D57)	;CJK UNIFIED IDEOGRAPH
                                (#x9BBE . #x6D58)	;CJK UNIFIED IDEOGRAPH
                                (#x9BBF . #x6D5B)	;CJK UNIFIED IDEOGRAPH
                                (#x9BC0 . #x6D5D)	;CJK UNIFIED IDEOGRAPH
                                (#x9BC1 . #x6D5F)	;CJK UNIFIED IDEOGRAPH
                                (#x9BC2 . #x6D61)	;CJK UNIFIED IDEOGRAPH
                                (#x9BC3 . #x6D62)	;CJK UNIFIED IDEOGRAPH
                                (#x9BC4 . #x6D64)	;CJK UNIFIED IDEOGRAPH
                                (#x9BC5 . #x6D65)	;CJK UNIFIED IDEOGRAPH
                                (#x9BC6 . #x6D67)	;CJK UNIFIED IDEOGRAPH
                                (#x9BC7 . #x6D68)	;CJK UNIFIED IDEOGRAPH
                                (#x9BC8 . #x6D6B)	;CJK UNIFIED IDEOGRAPH
                                (#x9BC9 . #x6D6C)	;CJK UNIFIED IDEOGRAPH
                                (#x9BCA . #x6D6D)	;CJK UNIFIED IDEOGRAPH
                                (#x9BCB . #x6D70)	;CJK UNIFIED IDEOGRAPH
                                (#x9BCC . #x6D71)	;CJK UNIFIED IDEOGRAPH
                                (#x9BCD . #x6D72)	;CJK UNIFIED IDEOGRAPH
                                (#x9BCE . #x6D73)	;CJK UNIFIED IDEOGRAPH
                                (#x9BCF . #x6D75)	;CJK UNIFIED IDEOGRAPH
                                (#x9BD0 . #x6D76)	;CJK UNIFIED IDEOGRAPH
                                (#x9BD1 . #x6D79)	;CJK UNIFIED IDEOGRAPH
                                (#x9BD2 . #x6D7A)	;CJK UNIFIED IDEOGRAPH
                                (#x9BD3 . #x6D7B)	;CJK UNIFIED IDEOGRAPH
                                (#x9BD4 . #x6D7D)	;CJK UNIFIED IDEOGRAPH
                                (#x9BD5 . #x6D7E)	;CJK UNIFIED IDEOGRAPH
                                (#x9BD6 . #x6D7F)	;CJK UNIFIED IDEOGRAPH
                                (#x9BD7 . #x6D80)	;CJK UNIFIED IDEOGRAPH
                                (#x9BD8 . #x6D81)	;CJK UNIFIED IDEOGRAPH
                                (#x9BD9 . #x6D83)	;CJK UNIFIED IDEOGRAPH
                                (#x9BDA . #x6D84)	;CJK UNIFIED IDEOGRAPH
                                (#x9BDB . #x6D86)	;CJK UNIFIED IDEOGRAPH
                                (#x9BDC . #x6D87)	;CJK UNIFIED IDEOGRAPH
                                (#x9BDD . #x6D8A)	;CJK UNIFIED IDEOGRAPH
                                (#x9BDE . #x6D8B)	;CJK UNIFIED IDEOGRAPH
                                (#x9BDF . #x6D8D)	;CJK UNIFIED IDEOGRAPH
                                (#x9BE0 . #x6D8F)	;CJK UNIFIED IDEOGRAPH
                                (#x9BE1 . #x6D90)	;CJK UNIFIED IDEOGRAPH
                                (#x9BE2 . #x6D92)	;CJK UNIFIED IDEOGRAPH
                                (#x9BE3 . #x6D96)	;CJK UNIFIED IDEOGRAPH
                                (#x9BE4 . #x6D97)	;CJK UNIFIED IDEOGRAPH
                                (#x9BE5 . #x6D98)	;CJK UNIFIED IDEOGRAPH
                                (#x9BE6 . #x6D99)	;CJK UNIFIED IDEOGRAPH
                                (#x9BE7 . #x6D9A)	;CJK UNIFIED IDEOGRAPH
                                (#x9BE8 . #x6D9C)	;CJK UNIFIED IDEOGRAPH
                                (#x9BE9 . #x6DA2)	;CJK UNIFIED IDEOGRAPH
                                (#x9BEA . #x6DA5)	;CJK UNIFIED IDEOGRAPH
                                (#x9BEB . #x6DAC)	;CJK UNIFIED IDEOGRAPH
                                (#x9BEC . #x6DAD)	;CJK UNIFIED IDEOGRAPH
                                (#x9BED . #x6DB0)	;CJK UNIFIED IDEOGRAPH
                                (#x9BEE . #x6DB1)	;CJK UNIFIED IDEOGRAPH
                                (#x9BEF . #x6DB3)	;CJK UNIFIED IDEOGRAPH
                                (#x9BF0 . #x6DB4)	;CJK UNIFIED IDEOGRAPH
                                (#x9BF1 . #x6DB6)	;CJK UNIFIED IDEOGRAPH
                                (#x9BF2 . #x6DB7)	;CJK UNIFIED IDEOGRAPH
                                (#x9BF3 . #x6DB9)	;CJK UNIFIED IDEOGRAPH
                                (#x9BF4 . #x6DBA)	;CJK UNIFIED IDEOGRAPH
                                (#x9BF5 . #x6DBB)	;CJK UNIFIED IDEOGRAPH
                                (#x9BF6 . #x6DBC)	;CJK UNIFIED IDEOGRAPH
                                (#x9BF7 . #x6DBD)	;CJK UNIFIED IDEOGRAPH
                                (#x9BF8 . #x6DBE)	;CJK UNIFIED IDEOGRAPH
                                (#x9BF9 . #x6DC1)	;CJK UNIFIED IDEOGRAPH
                                (#x9BFA . #x6DC2)	;CJK UNIFIED IDEOGRAPH
                                (#x9BFB . #x6DC3)	;CJK UNIFIED IDEOGRAPH
                                (#x9BFC . #x6DC8)	;CJK UNIFIED IDEOGRAPH
                                (#x9BFD . #x6DC9)	;CJK UNIFIED IDEOGRAPH
                                (#x9BFE . #x6DCA)	;CJK UNIFIED IDEOGRAPH
                                (#x9C40 . #x6DCD)	;CJK UNIFIED IDEOGRAPH
                                (#x9C41 . #x6DCE)	;CJK UNIFIED IDEOGRAPH
                                (#x9C42 . #x6DCF)	;CJK UNIFIED IDEOGRAPH
                                (#x9C43 . #x6DD0)	;CJK UNIFIED IDEOGRAPH
                                (#x9C44 . #x6DD2)	;CJK UNIFIED IDEOGRAPH
                                (#x9C45 . #x6DD3)	;CJK UNIFIED IDEOGRAPH
                                (#x9C46 . #x6DD4)	;CJK UNIFIED IDEOGRAPH
                                (#x9C47 . #x6DD5)	;CJK UNIFIED IDEOGRAPH
                                (#x9C48 . #x6DD7)	;CJK UNIFIED IDEOGRAPH
                                (#x9C49 . #x6DDA)	;CJK UNIFIED IDEOGRAPH
                                (#x9C4A . #x6DDB)	;CJK UNIFIED IDEOGRAPH
                                (#x9C4B . #x6DDC)	;CJK UNIFIED IDEOGRAPH
                                (#x9C4C . #x6DDF)	;CJK UNIFIED IDEOGRAPH
                                (#x9C4D . #x6DE2)	;CJK UNIFIED IDEOGRAPH
                                (#x9C4E . #x6DE3)	;CJK UNIFIED IDEOGRAPH
                                (#x9C4F . #x6DE5)	;CJK UNIFIED IDEOGRAPH
                                (#x9C50 . #x6DE7)	;CJK UNIFIED IDEOGRAPH
                                (#x9C51 . #x6DE8)	;CJK UNIFIED IDEOGRAPH
                                (#x9C52 . #x6DE9)	;CJK UNIFIED IDEOGRAPH
                                (#x9C53 . #x6DEA)	;CJK UNIFIED IDEOGRAPH
                                (#x9C54 . #x6DED)	;CJK UNIFIED IDEOGRAPH
                                (#x9C55 . #x6DEF)	;CJK UNIFIED IDEOGRAPH
                                (#x9C56 . #x6DF0)	;CJK UNIFIED IDEOGRAPH
                                (#x9C57 . #x6DF2)	;CJK UNIFIED IDEOGRAPH
                                (#x9C58 . #x6DF4)	;CJK UNIFIED IDEOGRAPH
                                (#x9C59 . #x6DF5)	;CJK UNIFIED IDEOGRAPH
                                (#x9C5A . #x6DF6)	;CJK UNIFIED IDEOGRAPH
                                (#x9C5B . #x6DF8)	;CJK UNIFIED IDEOGRAPH
                                (#x9C5C . #x6DFA)	;CJK UNIFIED IDEOGRAPH
                                (#x9C5D . #x6DFD)	;CJK UNIFIED IDEOGRAPH
                                (#x9C5E . #x6DFE)	;CJK UNIFIED IDEOGRAPH
                                (#x9C5F . #x6DFF)	;CJK UNIFIED IDEOGRAPH
                                (#x9C60 . #x6E00)	;CJK UNIFIED IDEOGRAPH
                                (#x9C61 . #x6E01)	;CJK UNIFIED IDEOGRAPH
                                (#x9C62 . #x6E02)	;CJK UNIFIED IDEOGRAPH
                                (#x9C63 . #x6E03)	;CJK UNIFIED IDEOGRAPH
                                (#x9C64 . #x6E04)	;CJK UNIFIED IDEOGRAPH
                                (#x9C65 . #x6E06)	;CJK UNIFIED IDEOGRAPH
                                (#x9C66 . #x6E07)	;CJK UNIFIED IDEOGRAPH
                                (#x9C67 . #x6E08)	;CJK UNIFIED IDEOGRAPH
                                (#x9C68 . #x6E09)	;CJK UNIFIED IDEOGRAPH
                                (#x9C69 . #x6E0B)	;CJK UNIFIED IDEOGRAPH
                                (#x9C6A . #x6E0F)	;CJK UNIFIED IDEOGRAPH
                                (#x9C6B . #x6E12)	;CJK UNIFIED IDEOGRAPH
                                (#x9C6C . #x6E13)	;CJK UNIFIED IDEOGRAPH
                                (#x9C6D . #x6E15)	;CJK UNIFIED IDEOGRAPH
                                (#x9C6E . #x6E18)	;CJK UNIFIED IDEOGRAPH
                                (#x9C6F . #x6E19)	;CJK UNIFIED IDEOGRAPH
                                (#x9C70 . #x6E1B)	;CJK UNIFIED IDEOGRAPH
                                (#x9C71 . #x6E1C)	;CJK UNIFIED IDEOGRAPH
                                (#x9C72 . #x6E1E)	;CJK UNIFIED IDEOGRAPH
                                (#x9C73 . #x6E1F)	;CJK UNIFIED IDEOGRAPH
                                (#x9C74 . #x6E22)	;CJK UNIFIED IDEOGRAPH
                                (#x9C75 . #x6E26)	;CJK UNIFIED IDEOGRAPH
                                (#x9C76 . #x6E27)	;CJK UNIFIED IDEOGRAPH
                                (#x9C77 . #x6E28)	;CJK UNIFIED IDEOGRAPH
                                (#x9C78 . #x6E2A)	;CJK UNIFIED IDEOGRAPH
                                (#x9C79 . #x6E2C)	;CJK UNIFIED IDEOGRAPH
                                (#x9C7A . #x6E2E)	;CJK UNIFIED IDEOGRAPH
                                (#x9C7B . #x6E30)	;CJK UNIFIED IDEOGRAPH
                                (#x9C7C . #x6E31)	;CJK UNIFIED IDEOGRAPH
                                (#x9C7D . #x6E33)	;CJK UNIFIED IDEOGRAPH
                                (#x9C7E . #x6E35)	;CJK UNIFIED IDEOGRAPH
                                (#x9C80 . #x6E36)	;CJK UNIFIED IDEOGRAPH
                                (#x9C81 . #x6E37)	;CJK UNIFIED IDEOGRAPH
                                (#x9C82 . #x6E39)	;CJK UNIFIED IDEOGRAPH
                                (#x9C83 . #x6E3B)	;CJK UNIFIED IDEOGRAPH
                                (#x9C84 . #x6E3C)	;CJK UNIFIED IDEOGRAPH
                                (#x9C85 . #x6E3D)	;CJK UNIFIED IDEOGRAPH
                                (#x9C86 . #x6E3E)	;CJK UNIFIED IDEOGRAPH
                                (#x9C87 . #x6E3F)	;CJK UNIFIED IDEOGRAPH
                                (#x9C88 . #x6E40)	;CJK UNIFIED IDEOGRAPH
                                (#x9C89 . #x6E41)	;CJK UNIFIED IDEOGRAPH
                                (#x9C8A . #x6E42)	;CJK UNIFIED IDEOGRAPH
                                (#x9C8B . #x6E45)	;CJK UNIFIED IDEOGRAPH
                                (#x9C8C . #x6E46)	;CJK UNIFIED IDEOGRAPH
                                (#x9C8D . #x6E47)	;CJK UNIFIED IDEOGRAPH
                                (#x9C8E . #x6E48)	;CJK UNIFIED IDEOGRAPH
                                (#x9C8F . #x6E49)	;CJK UNIFIED IDEOGRAPH
                                (#x9C90 . #x6E4A)	;CJK UNIFIED IDEOGRAPH
                                (#x9C91 . #x6E4B)	;CJK UNIFIED IDEOGRAPH
                                (#x9C92 . #x6E4C)	;CJK UNIFIED IDEOGRAPH
                                (#x9C93 . #x6E4F)	;CJK UNIFIED IDEOGRAPH
                                (#x9C94 . #x6E50)	;CJK UNIFIED IDEOGRAPH
                                (#x9C95 . #x6E51)	;CJK UNIFIED IDEOGRAPH
                                (#x9C96 . #x6E52)	;CJK UNIFIED IDEOGRAPH
                                (#x9C97 . #x6E55)	;CJK UNIFIED IDEOGRAPH
                                (#x9C98 . #x6E57)	;CJK UNIFIED IDEOGRAPH
                                (#x9C99 . #x6E59)	;CJK UNIFIED IDEOGRAPH
                                (#x9C9A . #x6E5A)	;CJK UNIFIED IDEOGRAPH
                                (#x9C9B . #x6E5C)	;CJK UNIFIED IDEOGRAPH
                                (#x9C9C . #x6E5D)	;CJK UNIFIED IDEOGRAPH
                                (#x9C9D . #x6E5E)	;CJK UNIFIED IDEOGRAPH
                                (#x9C9E . #x6E60)	;CJK UNIFIED IDEOGRAPH
                                (#x9C9F . #x6E61)	;CJK UNIFIED IDEOGRAPH
                                (#x9CA0 . #x6E62)	;CJK UNIFIED IDEOGRAPH
                                (#x9CA1 . #x6E63)	;CJK UNIFIED IDEOGRAPH
                                (#x9CA2 . #x6E64)	;CJK UNIFIED IDEOGRAPH
                                (#x9CA3 . #x6E65)	;CJK UNIFIED IDEOGRAPH
                                (#x9CA4 . #x6E66)	;CJK UNIFIED IDEOGRAPH
                                (#x9CA5 . #x6E67)	;CJK UNIFIED IDEOGRAPH
                                (#x9CA6 . #x6E68)	;CJK UNIFIED IDEOGRAPH
                                (#x9CA7 . #x6E69)	;CJK UNIFIED IDEOGRAPH
                                (#x9CA8 . #x6E6A)	;CJK UNIFIED IDEOGRAPH
                                (#x9CA9 . #x6E6C)	;CJK UNIFIED IDEOGRAPH
                                (#x9CAA . #x6E6D)	;CJK UNIFIED IDEOGRAPH
                                (#x9CAB . #x6E6F)	;CJK UNIFIED IDEOGRAPH
                                (#x9CAC . #x6E70)	;CJK UNIFIED IDEOGRAPH
                                (#x9CAD . #x6E71)	;CJK UNIFIED IDEOGRAPH
                                (#x9CAE . #x6E72)	;CJK UNIFIED IDEOGRAPH
                                (#x9CAF . #x6E73)	;CJK UNIFIED IDEOGRAPH
                                (#x9CB0 . #x6E74)	;CJK UNIFIED IDEOGRAPH
                                (#x9CB1 . #x6E75)	;CJK UNIFIED IDEOGRAPH
                                (#x9CB2 . #x6E76)	;CJK UNIFIED IDEOGRAPH
                                (#x9CB3 . #x6E77)	;CJK UNIFIED IDEOGRAPH
                                (#x9CB4 . #x6E78)	;CJK UNIFIED IDEOGRAPH
                                (#x9CB5 . #x6E79)	;CJK UNIFIED IDEOGRAPH
                                (#x9CB6 . #x6E7A)	;CJK UNIFIED IDEOGRAPH
                                (#x9CB7 . #x6E7B)	;CJK UNIFIED IDEOGRAPH
                                (#x9CB8 . #x6E7C)	;CJK UNIFIED IDEOGRAPH
                                (#x9CB9 . #x6E7D)	;CJK UNIFIED IDEOGRAPH
                                (#x9CBA . #x6E80)	;CJK UNIFIED IDEOGRAPH
                                (#x9CBB . #x6E81)	;CJK UNIFIED IDEOGRAPH
                                (#x9CBC . #x6E82)	;CJK UNIFIED IDEOGRAPH
                                (#x9CBD . #x6E84)	;CJK UNIFIED IDEOGRAPH
                                (#x9CBE . #x6E87)	;CJK UNIFIED IDEOGRAPH
                                (#x9CBF . #x6E88)	;CJK UNIFIED IDEOGRAPH
                                (#x9CC0 . #x6E8A)	;CJK UNIFIED IDEOGRAPH
                                (#x9CC1 . #x6E8B)	;CJK UNIFIED IDEOGRAPH
                                (#x9CC2 . #x6E8C)	;CJK UNIFIED IDEOGRAPH
                                (#x9CC3 . #x6E8D)	;CJK UNIFIED IDEOGRAPH
                                (#x9CC4 . #x6E8E)	;CJK UNIFIED IDEOGRAPH
                                (#x9CC5 . #x6E91)	;CJK UNIFIED IDEOGRAPH
                                (#x9CC6 . #x6E92)	;CJK UNIFIED IDEOGRAPH
                                (#x9CC7 . #x6E93)	;CJK UNIFIED IDEOGRAPH
                                (#x9CC8 . #x6E94)	;CJK UNIFIED IDEOGRAPH
                                (#x9CC9 . #x6E95)	;CJK UNIFIED IDEOGRAPH
                                (#x9CCA . #x6E96)	;CJK UNIFIED IDEOGRAPH
                                (#x9CCB . #x6E97)	;CJK UNIFIED IDEOGRAPH
                                (#x9CCC . #x6E99)	;CJK UNIFIED IDEOGRAPH
                                (#x9CCD . #x6E9A)	;CJK UNIFIED IDEOGRAPH
                                (#x9CCE . #x6E9B)	;CJK UNIFIED IDEOGRAPH
                                (#x9CCF . #x6E9D)	;CJK UNIFIED IDEOGRAPH
                                (#x9CD0 . #x6E9E)	;CJK UNIFIED IDEOGRAPH
                                (#x9CD1 . #x6EA0)	;CJK UNIFIED IDEOGRAPH
                                (#x9CD2 . #x6EA1)	;CJK UNIFIED IDEOGRAPH
                                (#x9CD3 . #x6EA3)	;CJK UNIFIED IDEOGRAPH
                                (#x9CD4 . #x6EA4)	;CJK UNIFIED IDEOGRAPH
                                (#x9CD5 . #x6EA6)	;CJK UNIFIED IDEOGRAPH
                                (#x9CD6 . #x6EA8)	;CJK UNIFIED IDEOGRAPH
                                (#x9CD7 . #x6EA9)	;CJK UNIFIED IDEOGRAPH
                                (#x9CD8 . #x6EAB)	;CJK UNIFIED IDEOGRAPH
                                (#x9CD9 . #x6EAC)	;CJK UNIFIED IDEOGRAPH
                                (#x9CDA . #x6EAD)	;CJK UNIFIED IDEOGRAPH
                                (#x9CDB . #x6EAE)	;CJK UNIFIED IDEOGRAPH
                                (#x9CDC . #x6EB0)	;CJK UNIFIED IDEOGRAPH
                                (#x9CDD . #x6EB3)	;CJK UNIFIED IDEOGRAPH
                                (#x9CDE . #x6EB5)	;CJK UNIFIED IDEOGRAPH
                                (#x9CDF . #x6EB8)	;CJK UNIFIED IDEOGRAPH
                                (#x9CE0 . #x6EB9)	;CJK UNIFIED IDEOGRAPH
                                (#x9CE1 . #x6EBC)	;CJK UNIFIED IDEOGRAPH
                                (#x9CE2 . #x6EBE)	;CJK UNIFIED IDEOGRAPH
                                (#x9CE3 . #x6EBF)	;CJK UNIFIED IDEOGRAPH
                                (#x9CE4 . #x6EC0)	;CJK UNIFIED IDEOGRAPH
                                (#x9CE5 . #x6EC3)	;CJK UNIFIED IDEOGRAPH
                                (#x9CE6 . #x6EC4)	;CJK UNIFIED IDEOGRAPH
                                (#x9CE7 . #x6EC5)	;CJK UNIFIED IDEOGRAPH
                                (#x9CE8 . #x6EC6)	;CJK UNIFIED IDEOGRAPH
                                (#x9CE9 . #x6EC8)	;CJK UNIFIED IDEOGRAPH
                                (#x9CEA . #x6EC9)	;CJK UNIFIED IDEOGRAPH
                                (#x9CEB . #x6ECA)	;CJK UNIFIED IDEOGRAPH
                                (#x9CEC . #x6ECC)	;CJK UNIFIED IDEOGRAPH
                                (#x9CED . #x6ECD)	;CJK UNIFIED IDEOGRAPH
                                (#x9CEE . #x6ECE)	;CJK UNIFIED IDEOGRAPH
                                (#x9CEF . #x6ED0)	;CJK UNIFIED IDEOGRAPH
                                (#x9CF0 . #x6ED2)	;CJK UNIFIED IDEOGRAPH
                                (#x9CF1 . #x6ED6)	;CJK UNIFIED IDEOGRAPH
                                (#x9CF2 . #x6ED8)	;CJK UNIFIED IDEOGRAPH
                                (#x9CF3 . #x6ED9)	;CJK UNIFIED IDEOGRAPH
                                (#x9CF4 . #x6EDB)	;CJK UNIFIED IDEOGRAPH
                                (#x9CF5 . #x6EDC)	;CJK UNIFIED IDEOGRAPH
                                (#x9CF6 . #x6EDD)	;CJK UNIFIED IDEOGRAPH
                                (#x9CF7 . #x6EE3)	;CJK UNIFIED IDEOGRAPH
                                (#x9CF8 . #x6EE7)	;CJK UNIFIED IDEOGRAPH
                                (#x9CF9 . #x6EEA)	;CJK UNIFIED IDEOGRAPH
                                (#x9CFA . #x6EEB)	;CJK UNIFIED IDEOGRAPH
                                (#x9CFB . #x6EEC)	;CJK UNIFIED IDEOGRAPH
                                (#x9CFC . #x6EED)	;CJK UNIFIED IDEOGRAPH
                                (#x9CFD . #x6EEE)	;CJK UNIFIED IDEOGRAPH
                                (#x9CFE . #x6EEF)	;CJK UNIFIED IDEOGRAPH
                                (#x9D40 . #x6EF0)	;CJK UNIFIED IDEOGRAPH
                                (#x9D41 . #x6EF1)	;CJK UNIFIED IDEOGRAPH
                                (#x9D42 . #x6EF2)	;CJK UNIFIED IDEOGRAPH
                                (#x9D43 . #x6EF3)	;CJK UNIFIED IDEOGRAPH
                                (#x9D44 . #x6EF5)	;CJK UNIFIED IDEOGRAPH
                                (#x9D45 . #x6EF6)	;CJK UNIFIED IDEOGRAPH
                                (#x9D46 . #x6EF7)	;CJK UNIFIED IDEOGRAPH
                                (#x9D47 . #x6EF8)	;CJK UNIFIED IDEOGRAPH
                                (#x9D48 . #x6EFA)	;CJK UNIFIED IDEOGRAPH
                                (#x9D49 . #x6EFB)	;CJK UNIFIED IDEOGRAPH
                                (#x9D4A . #x6EFC)	;CJK UNIFIED IDEOGRAPH
                                (#x9D4B . #x6EFD)	;CJK UNIFIED IDEOGRAPH
                                (#x9D4C . #x6EFE)	;CJK UNIFIED IDEOGRAPH
                                (#x9D4D . #x6EFF)	;CJK UNIFIED IDEOGRAPH
                                (#x9D4E . #x6F00)	;CJK UNIFIED IDEOGRAPH
                                (#x9D4F . #x6F01)	;CJK UNIFIED IDEOGRAPH
                                (#x9D50 . #x6F03)	;CJK UNIFIED IDEOGRAPH
                                (#x9D51 . #x6F04)	;CJK UNIFIED IDEOGRAPH
                                (#x9D52 . #x6F05)	;CJK UNIFIED IDEOGRAPH
                                (#x9D53 . #x6F07)	;CJK UNIFIED IDEOGRAPH
                                (#x9D54 . #x6F08)	;CJK UNIFIED IDEOGRAPH
                                (#x9D55 . #x6F0A)	;CJK UNIFIED IDEOGRAPH
                                (#x9D56 . #x6F0B)	;CJK UNIFIED IDEOGRAPH
                                (#x9D57 . #x6F0C)	;CJK UNIFIED IDEOGRAPH
                                (#x9D58 . #x6F0D)	;CJK UNIFIED IDEOGRAPH
                                (#x9D59 . #x6F0E)	;CJK UNIFIED IDEOGRAPH
                                (#x9D5A . #x6F10)	;CJK UNIFIED IDEOGRAPH
                                (#x9D5B . #x6F11)	;CJK UNIFIED IDEOGRAPH
                                (#x9D5C . #x6F12)	;CJK UNIFIED IDEOGRAPH
                                (#x9D5D . #x6F16)	;CJK UNIFIED IDEOGRAPH
                                (#x9D5E . #x6F17)	;CJK UNIFIED IDEOGRAPH
                                (#x9D5F . #x6F18)	;CJK UNIFIED IDEOGRAPH
                                (#x9D60 . #x6F19)	;CJK UNIFIED IDEOGRAPH
                                (#x9D61 . #x6F1A)	;CJK UNIFIED IDEOGRAPH
                                (#x9D62 . #x6F1B)	;CJK UNIFIED IDEOGRAPH
                                (#x9D63 . #x6F1C)	;CJK UNIFIED IDEOGRAPH
                                (#x9D64 . #x6F1D)	;CJK UNIFIED IDEOGRAPH
                                (#x9D65 . #x6F1E)	;CJK UNIFIED IDEOGRAPH
                                (#x9D66 . #x6F1F)	;CJK UNIFIED IDEOGRAPH
                                (#x9D67 . #x6F21)	;CJK UNIFIED IDEOGRAPH
                                (#x9D68 . #x6F22)	;CJK UNIFIED IDEOGRAPH
                                (#x9D69 . #x6F23)	;CJK UNIFIED IDEOGRAPH
                                (#x9D6A . #x6F25)	;CJK UNIFIED IDEOGRAPH
                                (#x9D6B . #x6F26)	;CJK UNIFIED IDEOGRAPH
                                (#x9D6C . #x6F27)	;CJK UNIFIED IDEOGRAPH
                                (#x9D6D . #x6F28)	;CJK UNIFIED IDEOGRAPH
                                (#x9D6E . #x6F2C)	;CJK UNIFIED IDEOGRAPH
                                (#x9D6F . #x6F2E)	;CJK UNIFIED IDEOGRAPH
                                (#x9D70 . #x6F30)	;CJK UNIFIED IDEOGRAPH
                                (#x9D71 . #x6F32)	;CJK UNIFIED IDEOGRAPH
                                (#x9D72 . #x6F34)	;CJK UNIFIED IDEOGRAPH
                                (#x9D73 . #x6F35)	;CJK UNIFIED IDEOGRAPH
                                (#x9D74 . #x6F37)	;CJK UNIFIED IDEOGRAPH
                                (#x9D75 . #x6F38)	;CJK UNIFIED IDEOGRAPH
                                (#x9D76 . #x6F39)	;CJK UNIFIED IDEOGRAPH
                                (#x9D77 . #x6F3A)	;CJK UNIFIED IDEOGRAPH
                                (#x9D78 . #x6F3B)	;CJK UNIFIED IDEOGRAPH
                                (#x9D79 . #x6F3C)	;CJK UNIFIED IDEOGRAPH
                                (#x9D7A . #x6F3D)	;CJK UNIFIED IDEOGRAPH
                                (#x9D7B . #x6F3F)	;CJK UNIFIED IDEOGRAPH
                                (#x9D7C . #x6F40)	;CJK UNIFIED IDEOGRAPH
                                (#x9D7D . #x6F41)	;CJK UNIFIED IDEOGRAPH
                                (#x9D7E . #x6F42)	;CJK UNIFIED IDEOGRAPH
                                (#x9D80 . #x6F43)	;CJK UNIFIED IDEOGRAPH
                                (#x9D81 . #x6F44)	;CJK UNIFIED IDEOGRAPH
                                (#x9D82 . #x6F45)	;CJK UNIFIED IDEOGRAPH
                                (#x9D83 . #x6F48)	;CJK UNIFIED IDEOGRAPH
                                (#x9D84 . #x6F49)	;CJK UNIFIED IDEOGRAPH
                                (#x9D85 . #x6F4A)	;CJK UNIFIED IDEOGRAPH
                                (#x9D86 . #x6F4C)	;CJK UNIFIED IDEOGRAPH
                                (#x9D87 . #x6F4E)	;CJK UNIFIED IDEOGRAPH
                                (#x9D88 . #x6F4F)	;CJK UNIFIED IDEOGRAPH
                                (#x9D89 . #x6F50)	;CJK UNIFIED IDEOGRAPH
                                (#x9D8A . #x6F51)	;CJK UNIFIED IDEOGRAPH
                                (#x9D8B . #x6F52)	;CJK UNIFIED IDEOGRAPH
                                (#x9D8C . #x6F53)	;CJK UNIFIED IDEOGRAPH
                                (#x9D8D . #x6F54)	;CJK UNIFIED IDEOGRAPH
                                (#x9D8E . #x6F55)	;CJK UNIFIED IDEOGRAPH
                                (#x9D8F . #x6F56)	;CJK UNIFIED IDEOGRAPH
                                (#x9D90 . #x6F57)	;CJK UNIFIED IDEOGRAPH
                                (#x9D91 . #x6F59)	;CJK UNIFIED IDEOGRAPH
                                (#x9D92 . #x6F5A)	;CJK UNIFIED IDEOGRAPH
                                (#x9D93 . #x6F5B)	;CJK UNIFIED IDEOGRAPH
                                (#x9D94 . #x6F5D)	;CJK UNIFIED IDEOGRAPH
                                (#x9D95 . #x6F5F)	;CJK UNIFIED IDEOGRAPH
                                (#x9D96 . #x6F60)	;CJK UNIFIED IDEOGRAPH
                                (#x9D97 . #x6F61)	;CJK UNIFIED IDEOGRAPH
                                (#x9D98 . #x6F63)	;CJK UNIFIED IDEOGRAPH
                                (#x9D99 . #x6F64)	;CJK UNIFIED IDEOGRAPH
                                (#x9D9A . #x6F65)	;CJK UNIFIED IDEOGRAPH
                                (#x9D9B . #x6F67)	;CJK UNIFIED IDEOGRAPH
                                (#x9D9C . #x6F68)	;CJK UNIFIED IDEOGRAPH
                                (#x9D9D . #x6F69)	;CJK UNIFIED IDEOGRAPH
                                (#x9D9E . #x6F6A)	;CJK UNIFIED IDEOGRAPH
                                (#x9D9F . #x6F6B)	;CJK UNIFIED IDEOGRAPH
                                (#x9DA0 . #x6F6C)	;CJK UNIFIED IDEOGRAPH
                                (#x9DA1 . #x6F6F)	;CJK UNIFIED IDEOGRAPH
                                (#x9DA2 . #x6F70)	;CJK UNIFIED IDEOGRAPH
                                (#x9DA3 . #x6F71)	;CJK UNIFIED IDEOGRAPH
                                (#x9DA4 . #x6F73)	;CJK UNIFIED IDEOGRAPH
                                (#x9DA5 . #x6F75)	;CJK UNIFIED IDEOGRAPH
                                (#x9DA6 . #x6F76)	;CJK UNIFIED IDEOGRAPH
                                (#x9DA7 . #x6F77)	;CJK UNIFIED IDEOGRAPH
                                (#x9DA8 . #x6F79)	;CJK UNIFIED IDEOGRAPH
                                (#x9DA9 . #x6F7B)	;CJK UNIFIED IDEOGRAPH
                                (#x9DAA . #x6F7D)	;CJK UNIFIED IDEOGRAPH
                                (#x9DAB . #x6F7E)	;CJK UNIFIED IDEOGRAPH
                                (#x9DAC . #x6F7F)	;CJK UNIFIED IDEOGRAPH
                                (#x9DAD . #x6F80)	;CJK UNIFIED IDEOGRAPH
                                (#x9DAE . #x6F81)	;CJK UNIFIED IDEOGRAPH
                                (#x9DAF . #x6F82)	;CJK UNIFIED IDEOGRAPH
                                (#x9DB0 . #x6F83)	;CJK UNIFIED IDEOGRAPH
                                (#x9DB1 . #x6F85)	;CJK UNIFIED IDEOGRAPH
                                (#x9DB2 . #x6F86)	;CJK UNIFIED IDEOGRAPH
                                (#x9DB3 . #x6F87)	;CJK UNIFIED IDEOGRAPH
                                (#x9DB4 . #x6F8A)	;CJK UNIFIED IDEOGRAPH
                                (#x9DB5 . #x6F8B)	;CJK UNIFIED IDEOGRAPH
                                (#x9DB6 . #x6F8F)	;CJK UNIFIED IDEOGRAPH
                                (#x9DB7 . #x6F90)	;CJK UNIFIED IDEOGRAPH
                                (#x9DB8 . #x6F91)	;CJK UNIFIED IDEOGRAPH
                                (#x9DB9 . #x6F92)	;CJK UNIFIED IDEOGRAPH
                                (#x9DBA . #x6F93)	;CJK UNIFIED IDEOGRAPH
                                (#x9DBB . #x6F94)	;CJK UNIFIED IDEOGRAPH
                                (#x9DBC . #x6F95)	;CJK UNIFIED IDEOGRAPH
                                (#x9DBD . #x6F96)	;CJK UNIFIED IDEOGRAPH
                                (#x9DBE . #x6F97)	;CJK UNIFIED IDEOGRAPH
                                (#x9DBF . #x6F98)	;CJK UNIFIED IDEOGRAPH
                                (#x9DC0 . #x6F99)	;CJK UNIFIED IDEOGRAPH
                                (#x9DC1 . #x6F9A)	;CJK UNIFIED IDEOGRAPH
                                (#x9DC2 . #x6F9B)	;CJK UNIFIED IDEOGRAPH
                                (#x9DC3 . #x6F9D)	;CJK UNIFIED IDEOGRAPH
                                (#x9DC4 . #x6F9E)	;CJK UNIFIED IDEOGRAPH
                                (#x9DC5 . #x6F9F)	;CJK UNIFIED IDEOGRAPH
                                (#x9DC6 . #x6FA0)	;CJK UNIFIED IDEOGRAPH
                                (#x9DC7 . #x6FA2)	;CJK UNIFIED IDEOGRAPH
                                (#x9DC8 . #x6FA3)	;CJK UNIFIED IDEOGRAPH
                                (#x9DC9 . #x6FA4)	;CJK UNIFIED IDEOGRAPH
                                (#x9DCA . #x6FA5)	;CJK UNIFIED IDEOGRAPH
                                (#x9DCB . #x6FA6)	;CJK UNIFIED IDEOGRAPH
                                (#x9DCC . #x6FA8)	;CJK UNIFIED IDEOGRAPH
                                (#x9DCD . #x6FA9)	;CJK UNIFIED IDEOGRAPH
                                (#x9DCE . #x6FAA)	;CJK UNIFIED IDEOGRAPH
                                (#x9DCF . #x6FAB)	;CJK UNIFIED IDEOGRAPH
                                (#x9DD0 . #x6FAC)	;CJK UNIFIED IDEOGRAPH
                                (#x9DD1 . #x6FAD)	;CJK UNIFIED IDEOGRAPH
                                (#x9DD2 . #x6FAE)	;CJK UNIFIED IDEOGRAPH
                                (#x9DD3 . #x6FAF)	;CJK UNIFIED IDEOGRAPH
                                (#x9DD4 . #x6FB0)	;CJK UNIFIED IDEOGRAPH
                                (#x9DD5 . #x6FB1)	;CJK UNIFIED IDEOGRAPH
                                (#x9DD6 . #x6FB2)	;CJK UNIFIED IDEOGRAPH
                                (#x9DD7 . #x6FB4)	;CJK UNIFIED IDEOGRAPH
                                (#x9DD8 . #x6FB5)	;CJK UNIFIED IDEOGRAPH
                                (#x9DD9 . #x6FB7)	;CJK UNIFIED IDEOGRAPH
                                (#x9DDA . #x6FB8)	;CJK UNIFIED IDEOGRAPH
                                (#x9DDB . #x6FBA)	;CJK UNIFIED IDEOGRAPH
                                (#x9DDC . #x6FBB)	;CJK UNIFIED IDEOGRAPH
                                (#x9DDD . #x6FBC)	;CJK UNIFIED IDEOGRAPH
                                (#x9DDE . #x6FBD)	;CJK UNIFIED IDEOGRAPH
                                (#x9DDF . #x6FBE)	;CJK UNIFIED IDEOGRAPH
                                (#x9DE0 . #x6FBF)	;CJK UNIFIED IDEOGRAPH
                                (#x9DE1 . #x6FC1)	;CJK UNIFIED IDEOGRAPH
                                (#x9DE2 . #x6FC3)	;CJK UNIFIED IDEOGRAPH
                                (#x9DE3 . #x6FC4)	;CJK UNIFIED IDEOGRAPH
                                (#x9DE4 . #x6FC5)	;CJK UNIFIED IDEOGRAPH
                                (#x9DE5 . #x6FC6)	;CJK UNIFIED IDEOGRAPH
                                (#x9DE6 . #x6FC7)	;CJK UNIFIED IDEOGRAPH
                                (#x9DE7 . #x6FC8)	;CJK UNIFIED IDEOGRAPH
                                (#x9DE8 . #x6FCA)	;CJK UNIFIED IDEOGRAPH
                                (#x9DE9 . #x6FCB)	;CJK UNIFIED IDEOGRAPH
                                (#x9DEA . #x6FCC)	;CJK UNIFIED IDEOGRAPH
                                (#x9DEB . #x6FCD)	;CJK UNIFIED IDEOGRAPH
                                (#x9DEC . #x6FCE)	;CJK UNIFIED IDEOGRAPH
                                (#x9DED . #x6FCF)	;CJK UNIFIED IDEOGRAPH
                                (#x9DEE . #x6FD0)	;CJK UNIFIED IDEOGRAPH
                                (#x9DEF . #x6FD3)	;CJK UNIFIED IDEOGRAPH
                                (#x9DF0 . #x6FD4)	;CJK UNIFIED IDEOGRAPH
                                (#x9DF1 . #x6FD5)	;CJK UNIFIED IDEOGRAPH
                                (#x9DF2 . #x6FD6)	;CJK UNIFIED IDEOGRAPH
                                (#x9DF3 . #x6FD7)	;CJK UNIFIED IDEOGRAPH
                                (#x9DF4 . #x6FD8)	;CJK UNIFIED IDEOGRAPH
                                (#x9DF5 . #x6FD9)	;CJK UNIFIED IDEOGRAPH
                                (#x9DF6 . #x6FDA)	;CJK UNIFIED IDEOGRAPH
                                (#x9DF7 . #x6FDB)	;CJK UNIFIED IDEOGRAPH
                                (#x9DF8 . #x6FDC)	;CJK UNIFIED IDEOGRAPH
                                (#x9DF9 . #x6FDD)	;CJK UNIFIED IDEOGRAPH
                                (#x9DFA . #x6FDF)	;CJK UNIFIED IDEOGRAPH
                                (#x9DFB . #x6FE2)	;CJK UNIFIED IDEOGRAPH
                                (#x9DFC . #x6FE3)	;CJK UNIFIED IDEOGRAPH
                                (#x9DFD . #x6FE4)	;CJK UNIFIED IDEOGRAPH
                                (#x9DFE . #x6FE5)	;CJK UNIFIED IDEOGRAPH
                                (#x9E40 . #x6FE6)	;CJK UNIFIED IDEOGRAPH
                                (#x9E41 . #x6FE7)	;CJK UNIFIED IDEOGRAPH
                                (#x9E42 . #x6FE8)	;CJK UNIFIED IDEOGRAPH
                                (#x9E43 . #x6FE9)	;CJK UNIFIED IDEOGRAPH
                                (#x9E44 . #x6FEA)	;CJK UNIFIED IDEOGRAPH
                                (#x9E45 . #x6FEB)	;CJK UNIFIED IDEOGRAPH
                                (#x9E46 . #x6FEC)	;CJK UNIFIED IDEOGRAPH
                                (#x9E47 . #x6FED)	;CJK UNIFIED IDEOGRAPH
                                (#x9E48 . #x6FF0)	;CJK UNIFIED IDEOGRAPH
                                (#x9E49 . #x6FF1)	;CJK UNIFIED IDEOGRAPH
                                (#x9E4A . #x6FF2)	;CJK UNIFIED IDEOGRAPH
                                (#x9E4B . #x6FF3)	;CJK UNIFIED IDEOGRAPH
                                (#x9E4C . #x6FF4)	;CJK UNIFIED IDEOGRAPH
                                (#x9E4D . #x6FF5)	;CJK UNIFIED IDEOGRAPH
                                (#x9E4E . #x6FF6)	;CJK UNIFIED IDEOGRAPH
                                (#x9E4F . #x6FF7)	;CJK UNIFIED IDEOGRAPH
                                (#x9E50 . #x6FF8)	;CJK UNIFIED IDEOGRAPH
                                (#x9E51 . #x6FF9)	;CJK UNIFIED IDEOGRAPH
                                (#x9E52 . #x6FFA)	;CJK UNIFIED IDEOGRAPH
                                (#x9E53 . #x6FFB)	;CJK UNIFIED IDEOGRAPH
                                (#x9E54 . #x6FFC)	;CJK UNIFIED IDEOGRAPH
                                (#x9E55 . #x6FFD)	;CJK UNIFIED IDEOGRAPH
                                (#x9E56 . #x6FFE)	;CJK UNIFIED IDEOGRAPH
                                (#x9E57 . #x6FFF)	;CJK UNIFIED IDEOGRAPH
                                (#x9E58 . #x7000)	;CJK UNIFIED IDEOGRAPH
                                (#x9E59 . #x7001)	;CJK UNIFIED IDEOGRAPH
                                (#x9E5A . #x7002)	;CJK UNIFIED IDEOGRAPH
                                (#x9E5B . #x7003)	;CJK UNIFIED IDEOGRAPH
                                (#x9E5C . #x7004)	;CJK UNIFIED IDEOGRAPH
                                (#x9E5D . #x7005)	;CJK UNIFIED IDEOGRAPH
                                (#x9E5E . #x7006)	;CJK UNIFIED IDEOGRAPH
                                (#x9E5F . #x7007)	;CJK UNIFIED IDEOGRAPH
                                (#x9E60 . #x7008)	;CJK UNIFIED IDEOGRAPH
                                (#x9E61 . #x7009)	;CJK UNIFIED IDEOGRAPH
                                (#x9E62 . #x700A)	;CJK UNIFIED IDEOGRAPH
                                (#x9E63 . #x700B)	;CJK UNIFIED IDEOGRAPH
                                (#x9E64 . #x700C)	;CJK UNIFIED IDEOGRAPH
                                (#x9E65 . #x700D)	;CJK UNIFIED IDEOGRAPH
                                (#x9E66 . #x700E)	;CJK UNIFIED IDEOGRAPH
                                (#x9E67 . #x700F)	;CJK UNIFIED IDEOGRAPH
                                (#x9E68 . #x7010)	;CJK UNIFIED IDEOGRAPH
                                (#x9E69 . #x7012)	;CJK UNIFIED IDEOGRAPH
                                (#x9E6A . #x7013)	;CJK UNIFIED IDEOGRAPH
                                (#x9E6B . #x7014)	;CJK UNIFIED IDEOGRAPH
                                (#x9E6C . #x7015)	;CJK UNIFIED IDEOGRAPH
                                (#x9E6D . #x7016)	;CJK UNIFIED IDEOGRAPH
                                (#x9E6E . #x7017)	;CJK UNIFIED IDEOGRAPH
                                (#x9E6F . #x7018)	;CJK UNIFIED IDEOGRAPH
                                (#x9E70 . #x7019)	;CJK UNIFIED IDEOGRAPH
                                (#x9E71 . #x701C)	;CJK UNIFIED IDEOGRAPH
                                (#x9E72 . #x701D)	;CJK UNIFIED IDEOGRAPH
                                (#x9E73 . #x701E)	;CJK UNIFIED IDEOGRAPH
                                (#x9E74 . #x701F)	;CJK UNIFIED IDEOGRAPH
                                (#x9E75 . #x7020)	;CJK UNIFIED IDEOGRAPH
                                (#x9E76 . #x7021)	;CJK UNIFIED IDEOGRAPH
                                (#x9E77 . #x7022)	;CJK UNIFIED IDEOGRAPH
                                (#x9E78 . #x7024)	;CJK UNIFIED IDEOGRAPH
                                (#x9E79 . #x7025)	;CJK UNIFIED IDEOGRAPH
                                (#x9E7A . #x7026)	;CJK UNIFIED IDEOGRAPH
                                (#x9E7B . #x7027)	;CJK UNIFIED IDEOGRAPH
                                (#x9E7C . #x7028)	;CJK UNIFIED IDEOGRAPH
                                (#x9E7D . #x7029)	;CJK UNIFIED IDEOGRAPH
                                (#x9E7E . #x702A)	;CJK UNIFIED IDEOGRAPH
                                (#x9E80 . #x702B)	;CJK UNIFIED IDEOGRAPH
                                (#x9E81 . #x702C)	;CJK UNIFIED IDEOGRAPH
                                (#x9E82 . #x702D)	;CJK UNIFIED IDEOGRAPH
                                (#x9E83 . #x702E)	;CJK UNIFIED IDEOGRAPH
                                (#x9E84 . #x702F)	;CJK UNIFIED IDEOGRAPH
                                (#x9E85 . #x7030)	;CJK UNIFIED IDEOGRAPH
                                (#x9E86 . #x7031)	;CJK UNIFIED IDEOGRAPH
                                (#x9E87 . #x7032)	;CJK UNIFIED IDEOGRAPH
                                (#x9E88 . #x7033)	;CJK UNIFIED IDEOGRAPH
                                (#x9E89 . #x7034)	;CJK UNIFIED IDEOGRAPH
                                (#x9E8A . #x7036)	;CJK UNIFIED IDEOGRAPH
                                (#x9E8B . #x7037)	;CJK UNIFIED IDEOGRAPH
                                (#x9E8C . #x7038)	;CJK UNIFIED IDEOGRAPH
                                (#x9E8D . #x703A)	;CJK UNIFIED IDEOGRAPH
                                (#x9E8E . #x703B)	;CJK UNIFIED IDEOGRAPH
                                (#x9E8F . #x703C)	;CJK UNIFIED IDEOGRAPH
                                (#x9E90 . #x703D)	;CJK UNIFIED IDEOGRAPH
                                (#x9E91 . #x703E)	;CJK UNIFIED IDEOGRAPH
                                (#x9E92 . #x703F)	;CJK UNIFIED IDEOGRAPH
                                (#x9E93 . #x7040)	;CJK UNIFIED IDEOGRAPH
                                (#x9E94 . #x7041)	;CJK UNIFIED IDEOGRAPH
                                (#x9E95 . #x7042)	;CJK UNIFIED IDEOGRAPH
                                (#x9E96 . #x7043)	;CJK UNIFIED IDEOGRAPH
                                (#x9E97 . #x7044)	;CJK UNIFIED IDEOGRAPH
                                (#x9E98 . #x7045)	;CJK UNIFIED IDEOGRAPH
                                (#x9E99 . #x7046)	;CJK UNIFIED IDEOGRAPH
                                (#x9E9A . #x7047)	;CJK UNIFIED IDEOGRAPH
                                (#x9E9B . #x7048)	;CJK UNIFIED IDEOGRAPH
                                (#x9E9C . #x7049)	;CJK UNIFIED IDEOGRAPH
                                (#x9E9D . #x704A)	;CJK UNIFIED IDEOGRAPH
                                (#x9E9E . #x704B)	;CJK UNIFIED IDEOGRAPH
                                (#x9E9F . #x704D)	;CJK UNIFIED IDEOGRAPH
                                (#x9EA0 . #x704E)	;CJK UNIFIED IDEOGRAPH
                                (#x9EA1 . #x7050)	;CJK UNIFIED IDEOGRAPH
                                (#x9EA2 . #x7051)	;CJK UNIFIED IDEOGRAPH
                                (#x9EA3 . #x7052)	;CJK UNIFIED IDEOGRAPH
                                (#x9EA4 . #x7053)	;CJK UNIFIED IDEOGRAPH
                                (#x9EA5 . #x7054)	;CJK UNIFIED IDEOGRAPH
                                (#x9EA6 . #x7055)	;CJK UNIFIED IDEOGRAPH
                                (#x9EA7 . #x7056)	;CJK UNIFIED IDEOGRAPH
                                (#x9EA8 . #x7057)	;CJK UNIFIED IDEOGRAPH
                                (#x9EA9 . #x7058)	;CJK UNIFIED IDEOGRAPH
                                (#x9EAA . #x7059)	;CJK UNIFIED IDEOGRAPH
                                (#x9EAB . #x705A)	;CJK UNIFIED IDEOGRAPH
                                (#x9EAC . #x705B)	;CJK UNIFIED IDEOGRAPH
                                (#x9EAD . #x705C)	;CJK UNIFIED IDEOGRAPH
                                (#x9EAE . #x705D)	;CJK UNIFIED IDEOGRAPH
                                (#x9EAF . #x705F)	;CJK UNIFIED IDEOGRAPH
                                (#x9EB0 . #x7060)	;CJK UNIFIED IDEOGRAPH
                                (#x9EB1 . #x7061)	;CJK UNIFIED IDEOGRAPH
                                (#x9EB2 . #x7062)	;CJK UNIFIED IDEOGRAPH
                                (#x9EB3 . #x7063)	;CJK UNIFIED IDEOGRAPH
                                (#x9EB4 . #x7064)	;CJK UNIFIED IDEOGRAPH
                                (#x9EB5 . #x7065)	;CJK UNIFIED IDEOGRAPH
                                (#x9EB6 . #x7066)	;CJK UNIFIED IDEOGRAPH
                                (#x9EB7 . #x7067)	;CJK UNIFIED IDEOGRAPH
                                (#x9EB8 . #x7068)	;CJK UNIFIED IDEOGRAPH
                                (#x9EB9 . #x7069)	;CJK UNIFIED IDEOGRAPH
                                (#x9EBA . #x706A)	;CJK UNIFIED IDEOGRAPH
                                (#x9EBB . #x706E)	;CJK UNIFIED IDEOGRAPH
                                (#x9EBC . #x7071)	;CJK UNIFIED IDEOGRAPH
                                (#x9EBD . #x7072)	;CJK UNIFIED IDEOGRAPH
                                (#x9EBE . #x7073)	;CJK UNIFIED IDEOGRAPH
                                (#x9EBF . #x7074)	;CJK UNIFIED IDEOGRAPH
                                (#x9EC0 . #x7077)	;CJK UNIFIED IDEOGRAPH
                                (#x9EC1 . #x7079)	;CJK UNIFIED IDEOGRAPH
                                (#x9EC2 . #x707A)	;CJK UNIFIED IDEOGRAPH
                                (#x9EC3 . #x707B)	;CJK UNIFIED IDEOGRAPH
                                (#x9EC4 . #x707D)	;CJK UNIFIED IDEOGRAPH
                                (#x9EC5 . #x7081)	;CJK UNIFIED IDEOGRAPH
                                (#x9EC6 . #x7082)	;CJK UNIFIED IDEOGRAPH
                                (#x9EC7 . #x7083)	;CJK UNIFIED IDEOGRAPH
                                (#x9EC8 . #x7084)	;CJK UNIFIED IDEOGRAPH
                                (#x9EC9 . #x7086)	;CJK UNIFIED IDEOGRAPH
                                (#x9ECA . #x7087)	;CJK UNIFIED IDEOGRAPH
                                (#x9ECB . #x7088)	;CJK UNIFIED IDEOGRAPH
                                (#x9ECC . #x708B)	;CJK UNIFIED IDEOGRAPH
                                (#x9ECD . #x708C)	;CJK UNIFIED IDEOGRAPH
                                (#x9ECE . #x708D)	;CJK UNIFIED IDEOGRAPH
                                (#x9ECF . #x708F)	;CJK UNIFIED IDEOGRAPH
                                (#x9ED0 . #x7090)	;CJK UNIFIED IDEOGRAPH
                                (#x9ED1 . #x7091)	;CJK UNIFIED IDEOGRAPH
                                (#x9ED2 . #x7093)	;CJK UNIFIED IDEOGRAPH
                                (#x9ED3 . #x7097)	;CJK UNIFIED IDEOGRAPH
                                (#x9ED4 . #x7098)	;CJK UNIFIED IDEOGRAPH
                                (#x9ED5 . #x709A)	;CJK UNIFIED IDEOGRAPH
                                (#x9ED6 . #x709B)	;CJK UNIFIED IDEOGRAPH
                                (#x9ED7 . #x709E)	;CJK UNIFIED IDEOGRAPH
                                (#x9ED8 . #x709F)	;CJK UNIFIED IDEOGRAPH
                                (#x9ED9 . #x70A0)	;CJK UNIFIED IDEOGRAPH
                                (#x9EDA . #x70A1)	;CJK UNIFIED IDEOGRAPH
                                (#x9EDB . #x70A2)	;CJK UNIFIED IDEOGRAPH
                                (#x9EDC . #x70A3)	;CJK UNIFIED IDEOGRAPH
                                (#x9EDD . #x70A4)	;CJK UNIFIED IDEOGRAPH
                                (#x9EDE . #x70A5)	;CJK UNIFIED IDEOGRAPH
                                (#x9EDF . #x70A6)	;CJK UNIFIED IDEOGRAPH
                                (#x9EE0 . #x70A7)	;CJK UNIFIED IDEOGRAPH
                                (#x9EE1 . #x70A8)	;CJK UNIFIED IDEOGRAPH
                                (#x9EE2 . #x70A9)	;CJK UNIFIED IDEOGRAPH
                                (#x9EE3 . #x70AA)	;CJK UNIFIED IDEOGRAPH
                                (#x9EE4 . #x70B0)	;CJK UNIFIED IDEOGRAPH
                                (#x9EE5 . #x70B2)	;CJK UNIFIED IDEOGRAPH
                                (#x9EE6 . #x70B4)	;CJK UNIFIED IDEOGRAPH
                                (#x9EE7 . #x70B5)	;CJK UNIFIED IDEOGRAPH
                                (#x9EE8 . #x70B6)	;CJK UNIFIED IDEOGRAPH
                                (#x9EE9 . #x70BA)	;CJK UNIFIED IDEOGRAPH
                                (#x9EEA . #x70BE)	;CJK UNIFIED IDEOGRAPH
                                (#x9EEB . #x70BF)	;CJK UNIFIED IDEOGRAPH
                                (#x9EEC . #x70C4)	;CJK UNIFIED IDEOGRAPH
                                (#x9EED . #x70C5)	;CJK UNIFIED IDEOGRAPH
                                (#x9EEE . #x70C6)	;CJK UNIFIED IDEOGRAPH
                                (#x9EEF . #x70C7)	;CJK UNIFIED IDEOGRAPH
                                (#x9EF0 . #x70C9)	;CJK UNIFIED IDEOGRAPH
                                (#x9EF1 . #x70CB)	;CJK UNIFIED IDEOGRAPH
                                (#x9EF2 . #x70CC)	;CJK UNIFIED IDEOGRAPH
                                (#x9EF3 . #x70CD)	;CJK UNIFIED IDEOGRAPH
                                (#x9EF4 . #x70CE)	;CJK UNIFIED IDEOGRAPH
                                (#x9EF5 . #x70CF)	;CJK UNIFIED IDEOGRAPH
                                (#x9EF6 . #x70D0)	;CJK UNIFIED IDEOGRAPH
                                (#x9EF7 . #x70D1)	;CJK UNIFIED IDEOGRAPH
                                (#x9EF8 . #x70D2)	;CJK UNIFIED IDEOGRAPH
                                (#x9EF9 . #x70D3)	;CJK UNIFIED IDEOGRAPH
                                (#x9EFA . #x70D4)	;CJK UNIFIED IDEOGRAPH
                                (#x9EFB . #x70D5)	;CJK UNIFIED IDEOGRAPH
                                (#x9EFC . #x70D6)	;CJK UNIFIED IDEOGRAPH
                                (#x9EFD . #x70D7)	;CJK UNIFIED IDEOGRAPH
                                (#x9EFE . #x70DA)	;CJK UNIFIED IDEOGRAPH
                                (#x9F40 . #x70DC)	;CJK UNIFIED IDEOGRAPH
                                (#x9F41 . #x70DD)	;CJK UNIFIED IDEOGRAPH
                                (#x9F42 . #x70DE)	;CJK UNIFIED IDEOGRAPH
                                (#x9F43 . #x70E0)	;CJK UNIFIED IDEOGRAPH
                                (#x9F44 . #x70E1)	;CJK UNIFIED IDEOGRAPH
                                (#x9F45 . #x70E2)	;CJK UNIFIED IDEOGRAPH
                                (#x9F46 . #x70E3)	;CJK UNIFIED IDEOGRAPH
                                (#x9F47 . #x70E5)	;CJK UNIFIED IDEOGRAPH
                                (#x9F48 . #x70EA)	;CJK UNIFIED IDEOGRAPH
                                (#x9F49 . #x70EE)	;CJK UNIFIED IDEOGRAPH
                                (#x9F4A . #x70F0)	;CJK UNIFIED IDEOGRAPH
                                (#x9F4B . #x70F1)	;CJK UNIFIED IDEOGRAPH
                                (#x9F4C . #x70F2)	;CJK UNIFIED IDEOGRAPH
                                (#x9F4D . #x70F3)	;CJK UNIFIED IDEOGRAPH
                                (#x9F4E . #x70F4)	;CJK UNIFIED IDEOGRAPH
                                (#x9F4F . #x70F5)	;CJK UNIFIED IDEOGRAPH
                                (#x9F50 . #x70F6)	;CJK UNIFIED IDEOGRAPH
                                (#x9F51 . #x70F8)	;CJK UNIFIED IDEOGRAPH
                                (#x9F52 . #x70FA)	;CJK UNIFIED IDEOGRAPH
                                (#x9F53 . #x70FB)	;CJK UNIFIED IDEOGRAPH
                                (#x9F54 . #x70FC)	;CJK UNIFIED IDEOGRAPH
                                (#x9F55 . #x70FE)	;CJK UNIFIED IDEOGRAPH
                                (#x9F56 . #x70FF)	;CJK UNIFIED IDEOGRAPH
                                (#x9F57 . #x7100)	;CJK UNIFIED IDEOGRAPH
                                (#x9F58 . #x7101)	;CJK UNIFIED IDEOGRAPH
                                (#x9F59 . #x7102)	;CJK UNIFIED IDEOGRAPH
                                (#x9F5A . #x7103)	;CJK UNIFIED IDEOGRAPH
                                (#x9F5B . #x7104)	;CJK UNIFIED IDEOGRAPH
                                (#x9F5C . #x7105)	;CJK UNIFIED IDEOGRAPH
                                (#x9F5D . #x7106)	;CJK UNIFIED IDEOGRAPH
                                (#x9F5E . #x7107)	;CJK UNIFIED IDEOGRAPH
                                (#x9F5F . #x7108)	;CJK UNIFIED IDEOGRAPH
                                (#x9F60 . #x710B)	;CJK UNIFIED IDEOGRAPH
                                (#x9F61 . #x710C)	;CJK UNIFIED IDEOGRAPH
                                (#x9F62 . #x710D)	;CJK UNIFIED IDEOGRAPH
                                (#x9F63 . #x710E)	;CJK UNIFIED IDEOGRAPH
                                (#x9F64 . #x710F)	;CJK UNIFIED IDEOGRAPH
                                (#x9F65 . #x7111)	;CJK UNIFIED IDEOGRAPH
                                (#x9F66 . #x7112)	;CJK UNIFIED IDEOGRAPH
                                (#x9F67 . #x7114)	;CJK UNIFIED IDEOGRAPH
                                (#x9F68 . #x7117)	;CJK UNIFIED IDEOGRAPH
                                (#x9F69 . #x711B)	;CJK UNIFIED IDEOGRAPH
                                (#x9F6A . #x711C)	;CJK UNIFIED IDEOGRAPH
                                (#x9F6B . #x711D)	;CJK UNIFIED IDEOGRAPH
                                (#x9F6C . #x711E)	;CJK UNIFIED IDEOGRAPH
                                (#x9F6D . #x711F)	;CJK UNIFIED IDEOGRAPH
                                (#x9F6E . #x7120)	;CJK UNIFIED IDEOGRAPH
                                (#x9F6F . #x7121)	;CJK UNIFIED IDEOGRAPH
                                (#x9F70 . #x7122)	;CJK UNIFIED IDEOGRAPH
                                (#x9F71 . #x7123)	;CJK UNIFIED IDEOGRAPH
                                (#x9F72 . #x7124)	;CJK UNIFIED IDEOGRAPH
                                (#x9F73 . #x7125)	;CJK UNIFIED IDEOGRAPH
                                (#x9F74 . #x7127)	;CJK UNIFIED IDEOGRAPH
                                (#x9F75 . #x7128)	;CJK UNIFIED IDEOGRAPH
                                (#x9F76 . #x7129)	;CJK UNIFIED IDEOGRAPH
                                (#x9F77 . #x712A)	;CJK UNIFIED IDEOGRAPH
                                (#x9F78 . #x712B)	;CJK UNIFIED IDEOGRAPH
                                (#x9F79 . #x712C)	;CJK UNIFIED IDEOGRAPH
                                (#x9F7A . #x712D)	;CJK UNIFIED IDEOGRAPH
                                (#x9F7B . #x712E)	;CJK UNIFIED IDEOGRAPH
                                (#x9F7C . #x7132)	;CJK UNIFIED IDEOGRAPH
                                (#x9F7D . #x7133)	;CJK UNIFIED IDEOGRAPH
                                (#x9F7E . #x7134)	;CJK UNIFIED IDEOGRAPH
                                (#x9F80 . #x7135)	;CJK UNIFIED IDEOGRAPH
                                (#x9F81 . #x7137)	;CJK UNIFIED IDEOGRAPH
                                (#x9F82 . #x7138)	;CJK UNIFIED IDEOGRAPH
                                (#x9F83 . #x7139)	;CJK UNIFIED IDEOGRAPH
                                (#x9F84 . #x713A)	;CJK UNIFIED IDEOGRAPH
                                (#x9F85 . #x713B)	;CJK UNIFIED IDEOGRAPH
                                (#x9F86 . #x713C)	;CJK UNIFIED IDEOGRAPH
                                (#x9F87 . #x713D)	;CJK UNIFIED IDEOGRAPH
                                (#x9F88 . #x713E)	;CJK UNIFIED IDEOGRAPH
                                (#x9F89 . #x713F)	;CJK UNIFIED IDEOGRAPH
                                (#x9F8A . #x7140)	;CJK UNIFIED IDEOGRAPH
                                (#x9F8B . #x7141)	;CJK UNIFIED IDEOGRAPH
                                (#x9F8C . #x7142)	;CJK UNIFIED IDEOGRAPH
                                (#x9F8D . #x7143)	;CJK UNIFIED IDEOGRAPH
                                (#x9F8E . #x7144)	;CJK UNIFIED IDEOGRAPH
                                (#x9F8F . #x7146)	;CJK UNIFIED IDEOGRAPH
                                (#x9F90 . #x7147)	;CJK UNIFIED IDEOGRAPH
                                (#x9F91 . #x7148)	;CJK UNIFIED IDEOGRAPH
                                (#x9F92 . #x7149)	;CJK UNIFIED IDEOGRAPH
                                (#x9F93 . #x714B)	;CJK UNIFIED IDEOGRAPH
                                (#x9F94 . #x714D)	;CJK UNIFIED IDEOGRAPH
                                (#x9F95 . #x714F)	;CJK UNIFIED IDEOGRAPH
                                (#x9F96 . #x7150)	;CJK UNIFIED IDEOGRAPH
                                (#x9F97 . #x7151)	;CJK UNIFIED IDEOGRAPH
                                (#x9F98 . #x7152)	;CJK UNIFIED IDEOGRAPH
                                (#x9F99 . #x7153)	;CJK UNIFIED IDEOGRAPH
                                (#x9F9A . #x7154)	;CJK UNIFIED IDEOGRAPH
                                (#x9F9B . #x7155)	;CJK UNIFIED IDEOGRAPH
                                (#x9F9C . #x7156)	;CJK UNIFIED IDEOGRAPH
                                (#x9F9D . #x7157)	;CJK UNIFIED IDEOGRAPH
                                (#x9F9E . #x7158)	;CJK UNIFIED IDEOGRAPH
                                (#x9F9F . #x7159)	;CJK UNIFIED IDEOGRAPH
                                (#x9FA0 . #x715A)	;CJK UNIFIED IDEOGRAPH
                                (#x9FA1 . #x715B)	;CJK UNIFIED IDEOGRAPH
                                (#x9FA2 . #x715D)	;CJK UNIFIED IDEOGRAPH
                                (#x9FA3 . #x715F)	;CJK UNIFIED IDEOGRAPH
                                (#x9FA4 . #x7160)	;CJK UNIFIED IDEOGRAPH
                                (#x9FA5 . #x7161)	;CJK UNIFIED IDEOGRAPH
                                (#x9FA6 . #x7162)	;CJK UNIFIED IDEOGRAPH
                                (#x9FA7 . #x7163)	;CJK UNIFIED IDEOGRAPH
                                (#x9FA8 . #x7165)	;CJK UNIFIED IDEOGRAPH
                                (#x9FA9 . #x7169)	;CJK UNIFIED IDEOGRAPH
                                (#x9FAA . #x716A)	;CJK UNIFIED IDEOGRAPH
                                (#x9FAB . #x716B)	;CJK UNIFIED IDEOGRAPH
                                (#x9FAC . #x716C)	;CJK UNIFIED IDEOGRAPH
                                (#x9FAD . #x716D)	;CJK UNIFIED IDEOGRAPH
                                (#x9FAE . #x716F)	;CJK UNIFIED IDEOGRAPH
                                (#x9FAF . #x7170)	;CJK UNIFIED IDEOGRAPH
                                (#x9FB0 . #x7171)	;CJK UNIFIED IDEOGRAPH
                                (#x9FB1 . #x7174)	;CJK UNIFIED IDEOGRAPH
                                (#x9FB2 . #x7175)	;CJK UNIFIED IDEOGRAPH
                                (#x9FB3 . #x7176)	;CJK UNIFIED IDEOGRAPH
                                (#x9FB4 . #x7177)	;CJK UNIFIED IDEOGRAPH
                                (#x9FB5 . #x7179)	;CJK UNIFIED IDEOGRAPH
                                (#x9FB6 . #x717B)	;CJK UNIFIED IDEOGRAPH
                                (#x9FB7 . #x717C)	;CJK UNIFIED IDEOGRAPH
                                (#x9FB8 . #x717E)	;CJK UNIFIED IDEOGRAPH
                                (#x9FB9 . #x717F)	;CJK UNIFIED IDEOGRAPH
                                (#x9FBA . #x7180)	;CJK UNIFIED IDEOGRAPH
                                (#x9FBB . #x7181)	;CJK UNIFIED IDEOGRAPH
                                (#x9FBC . #x7182)	;CJK UNIFIED IDEOGRAPH
                                (#x9FBD . #x7183)	;CJK UNIFIED IDEOGRAPH
                                (#x9FBE . #x7185)	;CJK UNIFIED IDEOGRAPH
                                (#x9FBF . #x7186)	;CJK UNIFIED IDEOGRAPH
                                (#x9FC0 . #x7187)	;CJK UNIFIED IDEOGRAPH
                                (#x9FC1 . #x7188)	;CJK UNIFIED IDEOGRAPH
                                (#x9FC2 . #x7189)	;CJK UNIFIED IDEOGRAPH
                                (#x9FC3 . #x718B)	;CJK UNIFIED IDEOGRAPH
                                (#x9FC4 . #x718C)	;CJK UNIFIED IDEOGRAPH
                                (#x9FC5 . #x718D)	;CJK UNIFIED IDEOGRAPH
                                (#x9FC6 . #x718E)	;CJK UNIFIED IDEOGRAPH
                                (#x9FC7 . #x7190)	;CJK UNIFIED IDEOGRAPH
                                (#x9FC8 . #x7191)	;CJK UNIFIED IDEOGRAPH
                                (#x9FC9 . #x7192)	;CJK UNIFIED IDEOGRAPH
                                (#x9FCA . #x7193)	;CJK UNIFIED IDEOGRAPH
                                (#x9FCB . #x7195)	;CJK UNIFIED IDEOGRAPH
                                (#x9FCC . #x7196)	;CJK UNIFIED IDEOGRAPH
                                (#x9FCD . #x7197)	;CJK UNIFIED IDEOGRAPH
                                (#x9FCE . #x719A)	;CJK UNIFIED IDEOGRAPH
                                (#x9FCF . #x719B)	;CJK UNIFIED IDEOGRAPH
                                (#x9FD0 . #x719C)	;CJK UNIFIED IDEOGRAPH
                                (#x9FD1 . #x719D)	;CJK UNIFIED IDEOGRAPH
                                (#x9FD2 . #x719E)	;CJK UNIFIED IDEOGRAPH
                                (#x9FD3 . #x71A1)	;CJK UNIFIED IDEOGRAPH
                                (#x9FD4 . #x71A2)	;CJK UNIFIED IDEOGRAPH
                                (#x9FD5 . #x71A3)	;CJK UNIFIED IDEOGRAPH
                                (#x9FD6 . #x71A4)	;CJK UNIFIED IDEOGRAPH
                                (#x9FD7 . #x71A5)	;CJK UNIFIED IDEOGRAPH
                                (#x9FD8 . #x71A6)	;CJK UNIFIED IDEOGRAPH
                                (#x9FD9 . #x71A7)	;CJK UNIFIED IDEOGRAPH
                                (#x9FDA . #x71A9)	;CJK UNIFIED IDEOGRAPH
                                (#x9FDB . #x71AA)	;CJK UNIFIED IDEOGRAPH
                                (#x9FDC . #x71AB)	;CJK UNIFIED IDEOGRAPH
                                (#x9FDD . #x71AD)	;CJK UNIFIED IDEOGRAPH
                                (#x9FDE . #x71AE)	;CJK UNIFIED IDEOGRAPH
                                (#x9FDF . #x71AF)	;CJK UNIFIED IDEOGRAPH
                                (#x9FE0 . #x71B0)	;CJK UNIFIED IDEOGRAPH
                                (#x9FE1 . #x71B1)	;CJK UNIFIED IDEOGRAPH
                                (#x9FE2 . #x71B2)	;CJK UNIFIED IDEOGRAPH
                                (#x9FE3 . #x71B4)	;CJK UNIFIED IDEOGRAPH
                                (#x9FE4 . #x71B6)	;CJK UNIFIED IDEOGRAPH
                                (#x9FE5 . #x71B7)	;CJK UNIFIED IDEOGRAPH
                                (#x9FE6 . #x71B8)	;CJK UNIFIED IDEOGRAPH
                                (#x9FE7 . #x71BA)	;CJK UNIFIED IDEOGRAPH
                                (#x9FE8 . #x71BB)	;CJK UNIFIED IDEOGRAPH
                                (#x9FE9 . #x71BC)	;CJK UNIFIED IDEOGRAPH
                                (#x9FEA . #x71BD)	;CJK UNIFIED IDEOGRAPH
                                (#x9FEB . #x71BE)	;CJK UNIFIED IDEOGRAPH
                                (#x9FEC . #x71BF)	;CJK UNIFIED IDEOGRAPH
                                (#x9FED . #x71C0)	;CJK UNIFIED IDEOGRAPH
                                (#x9FEE . #x71C1)	;CJK UNIFIED IDEOGRAPH
                                (#x9FEF . #x71C2)	;CJK UNIFIED IDEOGRAPH
                                (#x9FF0 . #x71C4)	;CJK UNIFIED IDEOGRAPH
                                (#x9FF1 . #x71C5)	;CJK UNIFIED IDEOGRAPH
                                (#x9FF2 . #x71C6)	;CJK UNIFIED IDEOGRAPH
                                (#x9FF3 . #x71C7)	;CJK UNIFIED IDEOGRAPH
                                (#x9FF4 . #x71C8)	;CJK UNIFIED IDEOGRAPH
                                (#x9FF5 . #x71C9)	;CJK UNIFIED IDEOGRAPH
                                (#x9FF6 . #x71CA)	;CJK UNIFIED IDEOGRAPH
                                (#x9FF7 . #x71CB)	;CJK UNIFIED IDEOGRAPH
                                (#x9FF8 . #x71CC)	;CJK UNIFIED IDEOGRAPH
                                (#x9FF9 . #x71CD)	;CJK UNIFIED IDEOGRAPH
                                (#x9FFA . #x71CF)	;CJK UNIFIED IDEOGRAPH
                                (#x9FFB . #x71D0)	;CJK UNIFIED IDEOGRAPH
                                (#x9FFC . #x71D1)	;CJK UNIFIED IDEOGRAPH
                                (#x9FFD . #x71D2)	;CJK UNIFIED IDEOGRAPH
                                (#x9FFE . #x71D3)	;CJK UNIFIED IDEOGRAPH
                                (#xA040 . #x71D6)	;CJK UNIFIED IDEOGRAPH
                                (#xA041 . #x71D7)	;CJK UNIFIED IDEOGRAPH
                                (#xA042 . #x71D8)	;CJK UNIFIED IDEOGRAPH
                                (#xA043 . #x71D9)	;CJK UNIFIED IDEOGRAPH
                                (#xA044 . #x71DA)	;CJK UNIFIED IDEOGRAPH
                                (#xA045 . #x71DB)	;CJK UNIFIED IDEOGRAPH
                                (#xA046 . #x71DC)	;CJK UNIFIED IDEOGRAPH
                                (#xA047 . #x71DD)	;CJK UNIFIED IDEOGRAPH
                                (#xA048 . #x71DE)	;CJK UNIFIED IDEOGRAPH
                                (#xA049 . #x71DF)	;CJK UNIFIED IDEOGRAPH
                                (#xA04A . #x71E1)	;CJK UNIFIED IDEOGRAPH
                                (#xA04B . #x71E2)	;CJK UNIFIED IDEOGRAPH
                                (#xA04C . #x71E3)	;CJK UNIFIED IDEOGRAPH
                                (#xA04D . #x71E4)	;CJK UNIFIED IDEOGRAPH
                                (#xA04E . #x71E6)	;CJK UNIFIED IDEOGRAPH
                                (#xA04F . #x71E8)	;CJK UNIFIED IDEOGRAPH
                                (#xA050 . #x71E9)	;CJK UNIFIED IDEOGRAPH
                                (#xA051 . #x71EA)	;CJK UNIFIED IDEOGRAPH
                                (#xA052 . #x71EB)	;CJK UNIFIED IDEOGRAPH
                                (#xA053 . #x71EC)	;CJK UNIFIED IDEOGRAPH
                                (#xA054 . #x71ED)	;CJK UNIFIED IDEOGRAPH
                                (#xA055 . #x71EF)	;CJK UNIFIED IDEOGRAPH
                                (#xA056 . #x71F0)	;CJK UNIFIED IDEOGRAPH
                                (#xA057 . #x71F1)	;CJK UNIFIED IDEOGRAPH
                                (#xA058 . #x71F2)	;CJK UNIFIED IDEOGRAPH
                                (#xA059 . #x71F3)	;CJK UNIFIED IDEOGRAPH
                                (#xA05A . #x71F4)	;CJK UNIFIED IDEOGRAPH
                                (#xA05B . #x71F5)	;CJK UNIFIED IDEOGRAPH
                                (#xA05C . #x71F6)	;CJK UNIFIED IDEOGRAPH
                                (#xA05D . #x71F7)	;CJK UNIFIED IDEOGRAPH
                                (#xA05E . #x71F8)	;CJK UNIFIED IDEOGRAPH
                                (#xA05F . #x71FA)	;CJK UNIFIED IDEOGRAPH
                                (#xA060 . #x71FB)	;CJK UNIFIED IDEOGRAPH
                                (#xA061 . #x71FC)	;CJK UNIFIED IDEOGRAPH
                                (#xA062 . #x71FD)	;CJK UNIFIED IDEOGRAPH
                                (#xA063 . #x71FE)	;CJK UNIFIED IDEOGRAPH
                                (#xA064 . #x71FF)	;CJK UNIFIED IDEOGRAPH
                                (#xA065 . #x7200)	;CJK UNIFIED IDEOGRAPH
                                (#xA066 . #x7201)	;CJK UNIFIED IDEOGRAPH
                                (#xA067 . #x7202)	;CJK UNIFIED IDEOGRAPH
                                (#xA068 . #x7203)	;CJK UNIFIED IDEOGRAPH
                                (#xA069 . #x7204)	;CJK UNIFIED IDEOGRAPH
                                (#xA06A . #x7205)	;CJK UNIFIED IDEOGRAPH
                                (#xA06B . #x7207)	;CJK UNIFIED IDEOGRAPH
                                (#xA06C . #x7208)	;CJK UNIFIED IDEOGRAPH
                                (#xA06D . #x7209)	;CJK UNIFIED IDEOGRAPH
                                (#xA06E . #x720A)	;CJK UNIFIED IDEOGRAPH
                                (#xA06F . #x720B)	;CJK UNIFIED IDEOGRAPH
                                (#xA070 . #x720C)	;CJK UNIFIED IDEOGRAPH
                                (#xA071 . #x720D)	;CJK UNIFIED IDEOGRAPH
                                (#xA072 . #x720E)	;CJK UNIFIED IDEOGRAPH
                                (#xA073 . #x720F)	;CJK UNIFIED IDEOGRAPH
                                (#xA074 . #x7210)	;CJK UNIFIED IDEOGRAPH
                                (#xA075 . #x7211)	;CJK UNIFIED IDEOGRAPH
                                (#xA076 . #x7212)	;CJK UNIFIED IDEOGRAPH
                                (#xA077 . #x7213)	;CJK UNIFIED IDEOGRAPH
                                (#xA078 . #x7214)	;CJK UNIFIED IDEOGRAPH
                                (#xA079 . #x7215)	;CJK UNIFIED IDEOGRAPH
                                (#xA07A . #x7216)	;CJK UNIFIED IDEOGRAPH
                                (#xA07B . #x7217)	;CJK UNIFIED IDEOGRAPH
                                (#xA07C . #x7218)	;CJK UNIFIED IDEOGRAPH
                                (#xA07D . #x7219)	;CJK UNIFIED IDEOGRAPH
                                (#xA07E . #x721A)	;CJK UNIFIED IDEOGRAPH
                                (#xA080 . #x721B)	;CJK UNIFIED IDEOGRAPH
                                (#xA081 . #x721C)	;CJK UNIFIED IDEOGRAPH
                                (#xA082 . #x721E)	;CJK UNIFIED IDEOGRAPH
                                (#xA083 . #x721F)	;CJK UNIFIED IDEOGRAPH
                                (#xA084 . #x7220)	;CJK UNIFIED IDEOGRAPH
                                (#xA085 . #x7221)	;CJK UNIFIED IDEOGRAPH
                                (#xA086 . #x7222)	;CJK UNIFIED IDEOGRAPH
                                (#xA087 . #x7223)	;CJK UNIFIED IDEOGRAPH
                                (#xA088 . #x7224)	;CJK UNIFIED IDEOGRAPH
                                (#xA089 . #x7225)	;CJK UNIFIED IDEOGRAPH
                                (#xA08A . #x7226)	;CJK UNIFIED IDEOGRAPH
                                (#xA08B . #x7227)	;CJK UNIFIED IDEOGRAPH
                                (#xA08C . #x7229)	;CJK UNIFIED IDEOGRAPH
                                (#xA08D . #x722B)	;CJK UNIFIED IDEOGRAPH
                                (#xA08E . #x722D)	;CJK UNIFIED IDEOGRAPH
                                (#xA08F . #x722E)	;CJK UNIFIED IDEOGRAPH
                                (#xA090 . #x722F)	;CJK UNIFIED IDEOGRAPH
                                (#xA091 . #x7232)	;CJK UNIFIED IDEOGRAPH
                                (#xA092 . #x7233)	;CJK UNIFIED IDEOGRAPH
                                (#xA093 . #x7234)	;CJK UNIFIED IDEOGRAPH
                                (#xA094 . #x723A)	;CJK UNIFIED IDEOGRAPH
                                (#xA095 . #x723C)	;CJK UNIFIED IDEOGRAPH
                                (#xA096 . #x723E)	;CJK UNIFIED IDEOGRAPH
                                (#xA097 . #x7240)	;CJK UNIFIED IDEOGRAPH
                                (#xA098 . #x7241)	;CJK UNIFIED IDEOGRAPH
                                (#xA099 . #x7242)	;CJK UNIFIED IDEOGRAPH
                                (#xA09A . #x7243)	;CJK UNIFIED IDEOGRAPH
                                (#xA09B . #x7244)	;CJK UNIFIED IDEOGRAPH
                                (#xA09C . #x7245)	;CJK UNIFIED IDEOGRAPH
                                (#xA09D . #x7246)	;CJK UNIFIED IDEOGRAPH
                                (#xA09E . #x7249)	;CJK UNIFIED IDEOGRAPH
                                (#xA09F . #x724A)	;CJK UNIFIED IDEOGRAPH
                                (#xA0A0 . #x724B)	;CJK UNIFIED IDEOGRAPH
                                (#xA0A1 . #x724E)	;CJK UNIFIED IDEOGRAPH
                                (#xA0A2 . #x724F)	;CJK UNIFIED IDEOGRAPH
                                (#xA0A3 . #x7250)	;CJK UNIFIED IDEOGRAPH
                                (#xA0A4 . #x7251)	;CJK UNIFIED IDEOGRAPH
                                (#xA0A5 . #x7253)	;CJK UNIFIED IDEOGRAPH
                                (#xA0A6 . #x7254)	;CJK UNIFIED IDEOGRAPH
                                (#xA0A7 . #x7255)	;CJK UNIFIED IDEOGRAPH
                                (#xA0A8 . #x7257)	;CJK UNIFIED IDEOGRAPH
                                (#xA0A9 . #x7258)	;CJK UNIFIED IDEOGRAPH
                                (#xA0AA . #x725A)	;CJK UNIFIED IDEOGRAPH
                                (#xA0AB . #x725C)	;CJK UNIFIED IDEOGRAPH
                                (#xA0AC . #x725E)	;CJK UNIFIED IDEOGRAPH
                                (#xA0AD . #x7260)	;CJK UNIFIED IDEOGRAPH
                                (#xA0AE . #x7263)	;CJK UNIFIED IDEOGRAPH
                                (#xA0AF . #x7264)	;CJK UNIFIED IDEOGRAPH
                                (#xA0B0 . #x7265)	;CJK UNIFIED IDEOGRAPH
                                (#xA0B1 . #x7268)	;CJK UNIFIED IDEOGRAPH
                                (#xA0B2 . #x726A)	;CJK UNIFIED IDEOGRAPH
                                (#xA0B3 . #x726B)	;CJK UNIFIED IDEOGRAPH
                                (#xA0B4 . #x726C)	;CJK UNIFIED IDEOGRAPH
                                (#xA0B5 . #x726D)	;CJK UNIFIED IDEOGRAPH
                                (#xA0B6 . #x7270)	;CJK UNIFIED IDEOGRAPH
                                (#xA0B7 . #x7271)	;CJK UNIFIED IDEOGRAPH
                                (#xA0B8 . #x7273)	;CJK UNIFIED IDEOGRAPH
                                (#xA0B9 . #x7274)	;CJK UNIFIED IDEOGRAPH
                                (#xA0BA . #x7276)	;CJK UNIFIED IDEOGRAPH
                                (#xA0BB . #x7277)	;CJK UNIFIED IDEOGRAPH
                                (#xA0BC . #x7278)	;CJK UNIFIED IDEOGRAPH
                                (#xA0BD . #x727B)	;CJK UNIFIED IDEOGRAPH
                                (#xA0BE . #x727C)	;CJK UNIFIED IDEOGRAPH
                                (#xA0BF . #x727D)	;CJK UNIFIED IDEOGRAPH
                                (#xA0C0 . #x7282)	;CJK UNIFIED IDEOGRAPH
                                (#xA0C1 . #x7283)	;CJK UNIFIED IDEOGRAPH
                                (#xA0C2 . #x7285)	;CJK UNIFIED IDEOGRAPH
                                (#xA0C3 . #x7286)	;CJK UNIFIED IDEOGRAPH
                                (#xA0C4 . #x7287)	;CJK UNIFIED IDEOGRAPH
                                (#xA0C5 . #x7288)	;CJK UNIFIED IDEOGRAPH
                                (#xA0C6 . #x7289)	;CJK UNIFIED IDEOGRAPH
                                (#xA0C7 . #x728C)	;CJK UNIFIED IDEOGRAPH
                                (#xA0C8 . #x728E)	;CJK UNIFIED IDEOGRAPH
                                (#xA0C9 . #x7290)	;CJK UNIFIED IDEOGRAPH
                                (#xA0CA . #x7291)	;CJK UNIFIED IDEOGRAPH
                                (#xA0CB . #x7293)	;CJK UNIFIED IDEOGRAPH
                                (#xA0CC . #x7294)	;CJK UNIFIED IDEOGRAPH
                                (#xA0CD . #x7295)	;CJK UNIFIED IDEOGRAPH
                                (#xA0CE . #x7296)	;CJK UNIFIED IDEOGRAPH
                                (#xA0CF . #x7297)	;CJK UNIFIED IDEOGRAPH
                                (#xA0D0 . #x7298)	;CJK UNIFIED IDEOGRAPH
                                (#xA0D1 . #x7299)	;CJK UNIFIED IDEOGRAPH
                                (#xA0D2 . #x729A)	;CJK UNIFIED IDEOGRAPH
                                (#xA0D3 . #x729B)	;CJK UNIFIED IDEOGRAPH
                                (#xA0D4 . #x729C)	;CJK UNIFIED IDEOGRAPH
                                (#xA0D5 . #x729D)	;CJK UNIFIED IDEOGRAPH
                                (#xA0D6 . #x729E)	;CJK UNIFIED IDEOGRAPH
                                (#xA0D7 . #x72A0)	;CJK UNIFIED IDEOGRAPH
                                (#xA0D8 . #x72A1)	;CJK UNIFIED IDEOGRAPH
                                (#xA0D9 . #x72A2)	;CJK UNIFIED IDEOGRAPH
                                (#xA0DA . #x72A3)	;CJK UNIFIED IDEOGRAPH
                                (#xA0DB . #x72A4)	;CJK UNIFIED IDEOGRAPH
                                (#xA0DC . #x72A5)	;CJK UNIFIED IDEOGRAPH
                                (#xA0DD . #x72A6)	;CJK UNIFIED IDEOGRAPH
                                (#xA0DE . #x72A7)	;CJK UNIFIED IDEOGRAPH
                                (#xA0DF . #x72A8)	;CJK UNIFIED IDEOGRAPH
                                (#xA0E0 . #x72A9)	;CJK UNIFIED IDEOGRAPH
                                (#xA0E1 . #x72AA)	;CJK UNIFIED IDEOGRAPH
                                (#xA0E2 . #x72AB)	;CJK UNIFIED IDEOGRAPH
                                (#xA0E3 . #x72AE)	;CJK UNIFIED IDEOGRAPH
                                (#xA0E4 . #x72B1)	;CJK UNIFIED IDEOGRAPH
                                (#xA0E5 . #x72B2)	;CJK UNIFIED IDEOGRAPH
                                (#xA0E6 . #x72B3)	;CJK UNIFIED IDEOGRAPH
                                (#xA0E7 . #x72B5)	;CJK UNIFIED IDEOGRAPH
                                (#xA0E8 . #x72BA)	;CJK UNIFIED IDEOGRAPH
                                (#xA0E9 . #x72BB)	;CJK UNIFIED IDEOGRAPH
                                (#xA0EA . #x72BC)	;CJK UNIFIED IDEOGRAPH
                                (#xA0EB . #x72BD)	;CJK UNIFIED IDEOGRAPH
                                (#xA0EC . #x72BE)	;CJK UNIFIED IDEOGRAPH
                                (#xA0ED . #x72BF)	;CJK UNIFIED IDEOGRAPH
                                (#xA0EE . #x72C0)	;CJK UNIFIED IDEOGRAPH
                                (#xA0EF . #x72C5)	;CJK UNIFIED IDEOGRAPH
                                (#xA0F0 . #x72C6)	;CJK UNIFIED IDEOGRAPH
                                (#xA0F1 . #x72C7)	;CJK UNIFIED IDEOGRAPH
                                (#xA0F2 . #x72C9)	;CJK UNIFIED IDEOGRAPH
                                (#xA0F3 . #x72CA)	;CJK UNIFIED IDEOGRAPH
                                (#xA0F4 . #x72CB)	;CJK UNIFIED IDEOGRAPH
                                (#xA0F5 . #x72CC)	;CJK UNIFIED IDEOGRAPH
                                (#xA0F6 . #x72CF)	;CJK UNIFIED IDEOGRAPH
                                (#xA0F7 . #x72D1)	;CJK UNIFIED IDEOGRAPH
                                (#xA0F8 . #x72D3)	;CJK UNIFIED IDEOGRAPH
                                (#xA0F9 . #x72D4)	;CJK UNIFIED IDEOGRAPH
                                (#xA0FA . #x72D5)	;CJK UNIFIED IDEOGRAPH
                                (#xA0FB . #x72D6)	;CJK UNIFIED IDEOGRAPH
                                (#xA0FC . #x72D8)	;CJK UNIFIED IDEOGRAPH
                                (#xA0FD . #x72DA)	;CJK UNIFIED IDEOGRAPH
                                (#xA0FE . #x72DB)	;CJK UNIFIED IDEOGRAPH
                                (#xA1A1 . #x3000)	;IDEOGRAPHIC SPACE
                                (#xA1A2 . #x3001)	;IDEOGRAPHIC COMMA
                                (#xA1A3 . #x3002)	;IDEOGRAPHIC FULL STOP
                                (#xA1A4 . #x00B7)	;MIDDLE DOT
                                (#xA1A5 . #x02C9)	;MODIFIER LETTER MACRON
                                (#xA1A6 . #x02C7)	;CARON
                                (#xA1A7 . #x00A8)	;DIAERESIS
                                (#xA1A8 . #x3003)	;DITTO MARK
                                (#xA1A9 . #x3005)	;IDEOGRAPHIC ITERATION MARK
                                (#xA1AA . #x2014)	;EM DASH
                                (#xA1AB . #xFF5E)	;FULLWIDTH TILDE
                                (#xA1AC . #x2016)	;DOUBLE VERTICAL LINE
                                (#xA1AD . #x2026)	;HORIZONTAL ELLIPSIS
                                (#xA1AE . #x2018)	;LEFT SINGLE QUOTATION MARK
                                (#xA1AF . #x2019)	;RIGHT SINGLE QUOTATION MARK
                                (#xA1B0 . #x201C)	;LEFT DOUBLE QUOTATION MARK
                                (#xA1B1 . #x201D)	;RIGHT DOUBLE QUOTATION MARK
                                (#xA1B2 . #x3014)	;LEFT TORTOISE SHELL BRACKET
                                (#xA1B3 . #x3015)	;RIGHT TORTOISE SHELL BRACKET
                                (#xA1B4 . #x3008)	;LEFT ANGLE BRACKET
                                (#xA1B5 . #x3009)	;RIGHT ANGLE BRACKET
                                (#xA1B6 . #x300A)	;LEFT DOUBLE ANGLE BRACKET
                                (#xA1B7 . #x300B)	;RIGHT DOUBLE ANGLE BRACKET
                                (#xA1B8 . #x300C)	;LEFT CORNER BRACKET
                                (#xA1B9 . #x300D)	;RIGHT CORNER BRACKET
                                (#xA1BA . #x300E)	;LEFT WHITE CORNER BRACKET
                                (#xA1BB . #x300F)	;RIGHT WHITE CORNER BRACKET
                                (#xA1BC . #x3016)	;LEFT WHITE LENTICULAR BRACKET
                                (#xA1BD . #x3017)	;RIGHT WHITE LENTICULAR BRACKET
                                (#xA1BE . #x3010)	;LEFT BLACK LENTICULAR BRACKET
                                (#xA1BF . #x3011)	;RIGHT BLACK LENTICULAR BRACKET
                                (#xA1C0 . #x00B1)	;PLUS-MINUS SIGN
                                (#xA1C1 . #x00D7)	;MULTIPLICATION SIGN
                                (#xA1C2 . #x00F7)	;DIVISION SIGN
                                (#xA1C3 . #x2236)	;RATIO
                                (#xA1C4 . #x2227)	;LOGICAL AND
                                (#xA1C5 . #x2228)	;LOGICAL OR
                                (#xA1C6 . #x2211)	;N-ARY SUMMATION
                                (#xA1C7 . #x220F)	;N-ARY PRODUCT
                                (#xA1C8 . #x222A)	;UNION
                                (#xA1C9 . #x2229)	;INTERSECTION
                                (#xA1CA . #x2208)	;ELEMENT OF
                                (#xA1CB . #x2237)	;PROPORTION
                                (#xA1CC . #x221A)	;SQUARE ROOT
                                (#xA1CD . #x22A5)	;UP TACK
                                (#xA1CE . #x2225)	;PARALLEL TO
                                (#xA1CF . #x2220)	;ANGLE
                                (#xA1D0 . #x2312)	;ARC
                                (#xA1D1 . #x2299)	;CIRCLED DOT OPERATOR
                                (#xA1D2 . #x222B)	;INTEGRAL
                                (#xA1D3 . #x222E)	;CONTOUR INTEGRAL
                                (#xA1D4 . #x2261)	;IDENTICAL TO
                                (#xA1D5 . #x224C)	;ALL EQUAL TO
                                (#xA1D6 . #x2248)	;ALMOST EQUAL TO
                                (#xA1D7 . #x223D)	;REVERSED TILDE
                                (#xA1D8 . #x221D)	;PROPORTIONAL TO
                                (#xA1D9 . #x2260)	;NOT EQUAL TO
                                (#xA1DA . #x226E)	;NOT LESS-THAN
                                (#xA1DB . #x226F)	;NOT GREATER-THAN
                                (#xA1DC . #x2264)	;LESS-THAN OR EQUAL TO
                                (#xA1DD . #x2265)	;GREATER-THAN OR EQUAL TO
                                (#xA1DE . #x221E)	;INFINITY
                                (#xA1DF . #x2235)	;BECAUSE
                                (#xA1E0 . #x2234)	;THEREFORE
                                (#xA1E1 . #x2642)	;MALE SIGN
                                (#xA1E2 . #x2640)	;FEMALE SIGN
                                (#xA1E3 . #x00B0)	;DEGREE SIGN
                                (#xA1E4 . #x2032)	;PRIME
                                (#xA1E5 . #x2033)	;DOUBLE PRIME
                                (#xA1E6 . #x2103)	;DEGREE CELSIUS
                                (#xA1E7 . #xFF04)	;FULLWIDTH DOLLAR SIGN
                                (#xA1E8 . #x00A4)	;CURRENCY SIGN
                                (#xA1E9 . #xFFE0)	;FULLWIDTH CENT SIGN
                                (#xA1EA . #xFFE1)	;FULLWIDTH POUND SIGN
                                (#xA1EB . #x2030)	;PER MILLE SIGN
                                (#xA1EC . #x00A7)	;SECTION SIGN
                                (#xA1ED . #x2116)	;NUMERO SIGN
                                (#xA1EE . #x2606)	;WHITE STAR
                                (#xA1EF . #x2605)	;BLACK STAR
                                (#xA1F0 . #x25CB)	;WHITE CIRCLE
                                (#xA1F1 . #x25CF)	;BLACK CIRCLE
                                (#xA1F2 . #x25CE)	;BULLSEYE
                                (#xA1F3 . #x25C7)	;WHITE DIAMOND
                                (#xA1F4 . #x25C6)	;BLACK DIAMOND
                                (#xA1F5 . #x25A1)	;WHITE SQUARE
                                (#xA1F6 . #x25A0)	;BLACK SQUARE
                                (#xA1F7 . #x25B3)	;WHITE UP-POINTING TRIANGLE
                                (#xA1F8 . #x25B2)	;BLACK UP-POINTING TRIANGLE
                                (#xA1F9 . #x203B)	;REFERENCE MARK
                                (#xA1FA . #x2192)	;RIGHTWARDS ARROW
                                (#xA1FB . #x2190)	;LEFTWARDS ARROW
                                (#xA1FC . #x2191)	;UPWARDS ARROW
                                (#xA1FD . #x2193)	;DOWNWARDS ARROW
                                (#xA1FE . #x3013)	;GETA MARK
                                (#xA2A1 . #x2170)	;SMALL ROMAN NUMERAL ONE
                                (#xA2A2 . #x2171)	;SMALL ROMAN NUMERAL TWO
                                (#xA2A3 . #x2172)	;SMALL ROMAN NUMERAL THREE
                                (#xA2A4 . #x2173)	;SMALL ROMAN NUMERAL FOUR
                                (#xA2A5 . #x2174)	;SMALL ROMAN NUMERAL FIVE
                                (#xA2A6 . #x2175)	;SMALL ROMAN NUMERAL SIX
                                (#xA2A7 . #x2176)	;SMALL ROMAN NUMERAL SEVEN
                                (#xA2A8 . #x2177)	;SMALL ROMAN NUMERAL EIGHT
                                (#xA2A9 . #x2178)	;SMALL ROMAN NUMERAL NINE
                                (#xA2AA . #x2179)	;SMALL ROMAN NUMERAL TEN
                                (#xA2B1 . #x2488)	;DIGIT ONE FULL STOP
                                (#xA2B2 . #x2489)	;DIGIT TWO FULL STOP
                                (#xA2B3 . #x248A)	;DIGIT THREE FULL STOP
                                (#xA2B4 . #x248B)	;DIGIT FOUR FULL STOP
                                (#xA2B5 . #x248C)	;DIGIT FIVE FULL STOP
                                (#xA2B6 . #x248D)	;DIGIT SIX FULL STOP
                                (#xA2B7 . #x248E)	;DIGIT SEVEN FULL STOP
                                (#xA2B8 . #x248F)	;DIGIT EIGHT FULL STOP
                                (#xA2B9 . #x2490)	;DIGIT NINE FULL STOP
                                (#xA2BA . #x2491)	;NUMBER TEN FULL STOP
                                (#xA2BB . #x2492)	;NUMBER ELEVEN FULL STOP
                                (#xA2BC . #x2493)	;NUMBER TWELVE FULL STOP
                                (#xA2BD . #x2494)	;NUMBER THIRTEEN FULL STOP
                                (#xA2BE . #x2495)	;NUMBER FOURTEEN FULL STOP
                                (#xA2BF . #x2496)	;NUMBER FIFTEEN FULL STOP
                                (#xA2C0 . #x2497)	;NUMBER SIXTEEN FULL STOP
                                (#xA2C1 . #x2498)	;NUMBER SEVENTEEN FULL STOP
                                (#xA2C2 . #x2499)	;NUMBER EIGHTEEN FULL STOP
                                (#xA2C3 . #x249A)	;NUMBER NINETEEN FULL STOP
                                (#xA2C4 . #x249B)	;NUMBER TWENTY FULL STOP
                                (#xA2C5 . #x2474)	;PARENTHESIZED DIGIT ONE
                                (#xA2C6 . #x2475)	;PARENTHESIZED DIGIT TWO
                                (#xA2C7 . #x2476)	;PARENTHESIZED DIGIT THREE
                                (#xA2C8 . #x2477)	;PARENTHESIZED DIGIT FOUR
                                (#xA2C9 . #x2478)	;PARENTHESIZED DIGIT FIVE
                                (#xA2CA . #x2479)	;PARENTHESIZED DIGIT SIX
                                (#xA2CB . #x247A)	;PARENTHESIZED DIGIT SEVEN
                                (#xA2CC . #x247B)	;PARENTHESIZED DIGIT EIGHT
                                (#xA2CD . #x247C)	;PARENTHESIZED DIGIT NINE
                                (#xA2CE . #x247D)	;PARENTHESIZED NUMBER TEN
                                (#xA2CF . #x247E)	;PARENTHESIZED NUMBER ELEVEN
                                (#xA2D0 . #x247F)	;PARENTHESIZED NUMBER TWELVE
                                (#xA2D1 . #x2480)	;PARENTHESIZED NUMBER THIRTEEN
                                (#xA2D2 . #x2481)	;PARENTHESIZED NUMBER FOURTEEN
                                (#xA2D3 . #x2482)	;PARENTHESIZED NUMBER FIFTEEN
                                (#xA2D4 . #x2483)	;PARENTHESIZED NUMBER SIXTEEN
                                (#xA2D5 . #x2484)	;PARENTHESIZED NUMBER SEVENTEEN
                                (#xA2D6 . #x2485)	;PARENTHESIZED NUMBER EIGHTEEN
                                (#xA2D7 . #x2486)	;PARENTHESIZED NUMBER NINETEEN
                                (#xA2D8 . #x2487)	;PARENTHESIZED NUMBER TWENTY
                                (#xA2D9 . #x2460)	;CIRCLED DIGIT ONE
                                (#xA2DA . #x2461)	;CIRCLED DIGIT TWO
                                (#xA2DB . #x2462)	;CIRCLED DIGIT THREE
                                (#xA2DC . #x2463)	;CIRCLED DIGIT FOUR
                                (#xA2DD . #x2464)	;CIRCLED DIGIT FIVE
                                (#xA2DE . #x2465)	;CIRCLED DIGIT SIX
                                (#xA2DF . #x2466)	;CIRCLED DIGIT SEVEN
                                (#xA2E0 . #x2467)	;CIRCLED DIGIT EIGHT
                                (#xA2E1 . #x2468)	;CIRCLED DIGIT NINE
                                (#xA2E2 . #x2469)	;CIRCLED NUMBER TEN
                                (#xA2E5 . #x3220)	;PARENTHESIZED IDEOGRAPH ONE
                                (#xA2E6 . #x3221)	;PARENTHESIZED IDEOGRAPH TWO
                                (#xA2E7 . #x3222)	;PARENTHESIZED IDEOGRAPH THREE
                                (#xA2E8 . #x3223)	;PARENTHESIZED IDEOGRAPH FOUR
                                (#xA2E9 . #x3224)	;PARENTHESIZED IDEOGRAPH FIVE
                                (#xA2EA . #x3225)	;PARENTHESIZED IDEOGRAPH SIX
                                (#xA2EB . #x3226)	;PARENTHESIZED IDEOGRAPH SEVEN
                                (#xA2EC . #x3227)	;PARENTHESIZED IDEOGRAPH EIGHT
                                (#xA2ED . #x3228)	;PARENTHESIZED IDEOGRAPH NINE
                                (#xA2EE . #x3229)	;PARENTHESIZED IDEOGRAPH TEN
                                (#xA2F1 . #x2160)	;ROMAN NUMERAL ONE
                                (#xA2F2 . #x2161)	;ROMAN NUMERAL TWO
                                (#xA2F3 . #x2162)	;ROMAN NUMERAL THREE
                                (#xA2F4 . #x2163)	;ROMAN NUMERAL FOUR
                                (#xA2F5 . #x2164)	;ROMAN NUMERAL FIVE
                                (#xA2F6 . #x2165)	;ROMAN NUMERAL SIX
                                (#xA2F7 . #x2166)	;ROMAN NUMERAL SEVEN
                                (#xA2F8 . #x2167)	;ROMAN NUMERAL EIGHT
                                (#xA2F9 . #x2168)	;ROMAN NUMERAL NINE
                                (#xA2FA . #x2169)	;ROMAN NUMERAL TEN
                                (#xA2FB . #x216A)	;ROMAN NUMERAL ELEVEN
                                (#xA2FC . #x216B)	;ROMAN NUMERAL TWELVE
                                (#xA3A1 . #xFF01)	;FULLWIDTH EXCLAMATION MARK
                                (#xA3A2 . #xFF02)	;FULLWIDTH QUOTATION MARK
                                (#xA3A3 . #xFF03)	;FULLWIDTH NUMBER SIGN
                                (#xA3A4 . #xFFE5)	;FULLWIDTH YEN SIGN
                                (#xA3A5 . #xFF05)	;FULLWIDTH PERCENT SIGN
                                (#xA3A6 . #xFF06)	;FULLWIDTH AMPERSAND
                                (#xA3A7 . #xFF07)	;FULLWIDTH APOSTROPHE
                                (#xA3A8 . #xFF08)	;FULLWIDTH LEFT PARENTHESIS
                                (#xA3A9 . #xFF09)	;FULLWIDTH RIGHT PARENTHESIS
                                (#xA3AA . #xFF0A)	;FULLWIDTH ASTERISK
                                (#xA3AB . #xFF0B)	;FULLWIDTH PLUS SIGN
                                (#xA3AC . #xFF0C)	;FULLWIDTH COMMA
                                (#xA3AD . #xFF0D)	;FULLWIDTH HYPHEN-MINUS
                                (#xA3AE . #xFF0E)	;FULLWIDTH FULL STOP
                                (#xA3AF . #xFF0F)	;FULLWIDTH SOLIDUS
                                (#xA3B0 . #xFF10)	;FULLWIDTH DIGIT ZERO
                                (#xA3B1 . #xFF11)	;FULLWIDTH DIGIT ONE
                                (#xA3B2 . #xFF12)	;FULLWIDTH DIGIT TWO
                                (#xA3B3 . #xFF13)	;FULLWIDTH DIGIT THREE
                                (#xA3B4 . #xFF14)	;FULLWIDTH DIGIT FOUR
                                (#xA3B5 . #xFF15)	;FULLWIDTH DIGIT FIVE
                                (#xA3B6 . #xFF16)	;FULLWIDTH DIGIT SIX
                                (#xA3B7 . #xFF17)	;FULLWIDTH DIGIT SEVEN
                                (#xA3B8 . #xFF18)	;FULLWIDTH DIGIT EIGHT
                                (#xA3B9 . #xFF19)	;FULLWIDTH DIGIT NINE
                                (#xA3BA . #xFF1A)	;FULLWIDTH COLON
                                (#xA3BB . #xFF1B)	;FULLWIDTH SEMICOLON
                                (#xA3BC . #xFF1C)	;FULLWIDTH LESS-THAN SIGN
                                (#xA3BD . #xFF1D)	;FULLWIDTH EQUALS SIGN
                                (#xA3BE . #xFF1E)	;FULLWIDTH GREATER-THAN SIGN
                                (#xA3BF . #xFF1F)	;FULLWIDTH QUESTION MARK
                                (#xA3C0 . #xFF20)	;FULLWIDTH COMMERCIAL AT
                                (#xA3C1 . #xFF21)	;FULLWIDTH LATIN CAPITAL LETTER A
                                (#xA3C2 . #xFF22)	;FULLWIDTH LATIN CAPITAL LETTER B
                                (#xA3C3 . #xFF23)	;FULLWIDTH LATIN CAPITAL LETTER C
                                (#xA3C4 . #xFF24)	;FULLWIDTH LATIN CAPITAL LETTER D
                                (#xA3C5 . #xFF25)	;FULLWIDTH LATIN CAPITAL LETTER E
                                (#xA3C6 . #xFF26)	;FULLWIDTH LATIN CAPITAL LETTER F
                                (#xA3C7 . #xFF27)	;FULLWIDTH LATIN CAPITAL LETTER G
                                (#xA3C8 . #xFF28)	;FULLWIDTH LATIN CAPITAL LETTER H
                                (#xA3C9 . #xFF29)	;FULLWIDTH LATIN CAPITAL LETTER I
                                (#xA3CA . #xFF2A)	;FULLWIDTH LATIN CAPITAL LETTER J
                                (#xA3CB . #xFF2B)	;FULLWIDTH LATIN CAPITAL LETTER K
                                (#xA3CC . #xFF2C)	;FULLWIDTH LATIN CAPITAL LETTER L
                                (#xA3CD . #xFF2D)	;FULLWIDTH LATIN CAPITAL LETTER M
                                (#xA3CE . #xFF2E)	;FULLWIDTH LATIN CAPITAL LETTER N
                                (#xA3CF . #xFF2F)	;FULLWIDTH LATIN CAPITAL LETTER O
                                (#xA3D0 . #xFF30)	;FULLWIDTH LATIN CAPITAL LETTER P
                                (#xA3D1 . #xFF31)	;FULLWIDTH LATIN CAPITAL LETTER Q
                                (#xA3D2 . #xFF32)	;FULLWIDTH LATIN CAPITAL LETTER R
                                (#xA3D3 . #xFF33)	;FULLWIDTH LATIN CAPITAL LETTER S
                                (#xA3D4 . #xFF34)	;FULLWIDTH LATIN CAPITAL LETTER T
                                (#xA3D5 . #xFF35)	;FULLWIDTH LATIN CAPITAL LETTER U
                                (#xA3D6 . #xFF36)	;FULLWIDTH LATIN CAPITAL LETTER V
                                (#xA3D7 . #xFF37)	;FULLWIDTH LATIN CAPITAL LETTER W
                                (#xA3D8 . #xFF38)	;FULLWIDTH LATIN CAPITAL LETTER X
                                (#xA3D9 . #xFF39)	;FULLWIDTH LATIN CAPITAL LETTER Y
                                (#xA3DA . #xFF3A)	;FULLWIDTH LATIN CAPITAL LETTER Z
                                (#xA3DB . #xFF3B)	;FULLWIDTH LEFT SQUARE BRACKET
                                (#xA3DC . #xFF3C)	;FULLWIDTH REVERSE SOLIDUS
                                (#xA3DD . #xFF3D)	;FULLWIDTH RIGHT SQUARE BRACKET
                                (#xA3DE . #xFF3E)	;FULLWIDTH CIRCUMFLEX ACCENT
                                (#xA3DF . #xFF3F)	;FULLWIDTH LOW LINE
                                (#xA3E0 . #xFF40)	;FULLWIDTH GRAVE ACCENT
                                (#xA3E1 . #xFF41)	;FULLWIDTH LATIN SMALL LETTER A
                                (#xA3E2 . #xFF42)	;FULLWIDTH LATIN SMALL LETTER B
                                (#xA3E3 . #xFF43)	;FULLWIDTH LATIN SMALL LETTER C
                                (#xA3E4 . #xFF44)	;FULLWIDTH LATIN SMALL LETTER D
                                (#xA3E5 . #xFF45)	;FULLWIDTH LATIN SMALL LETTER E
                                (#xA3E6 . #xFF46)	;FULLWIDTH LATIN SMALL LETTER F
                                (#xA3E7 . #xFF47)	;FULLWIDTH LATIN SMALL LETTER G
                                (#xA3E8 . #xFF48)	;FULLWIDTH LATIN SMALL LETTER H
                                (#xA3E9 . #xFF49)	;FULLWIDTH LATIN SMALL LETTER I
                                (#xA3EA . #xFF4A)	;FULLWIDTH LATIN SMALL LETTER J
                                (#xA3EB . #xFF4B)	;FULLWIDTH LATIN SMALL LETTER K
                                (#xA3EC . #xFF4C)	;FULLWIDTH LATIN SMALL LETTER L
                                (#xA3ED . #xFF4D)	;FULLWIDTH LATIN SMALL LETTER M
                                (#xA3EE . #xFF4E)	;FULLWIDTH LATIN SMALL LETTER N
                                (#xA3EF . #xFF4F)	;FULLWIDTH LATIN SMALL LETTER O
                                (#xA3F0 . #xFF50)	;FULLWIDTH LATIN SMALL LETTER P
                                (#xA3F1 . #xFF51)	;FULLWIDTH LATIN SMALL LETTER Q
                                (#xA3F2 . #xFF52)	;FULLWIDTH LATIN SMALL LETTER R
                                (#xA3F3 . #xFF53)	;FULLWIDTH LATIN SMALL LETTER S
                                (#xA3F4 . #xFF54)	;FULLWIDTH LATIN SMALL LETTER T
                                (#xA3F5 . #xFF55)	;FULLWIDTH LATIN SMALL LETTER U
                                (#xA3F6 . #xFF56)	;FULLWIDTH LATIN SMALL LETTER V
                                (#xA3F7 . #xFF57)	;FULLWIDTH LATIN SMALL LETTER W
                                (#xA3F8 . #xFF58)	;FULLWIDTH LATIN SMALL LETTER X
                                (#xA3F9 . #xFF59)	;FULLWIDTH LATIN SMALL LETTER Y
                                (#xA3FA . #xFF5A)	;FULLWIDTH LATIN SMALL LETTER Z
                                (#xA3FB . #xFF5B)	;FULLWIDTH LEFT CURLY BRACKET
                                (#xA3FC . #xFF5C)	;FULLWIDTH VERTICAL LINE
                                (#xA3FD . #xFF5D)	;FULLWIDTH RIGHT CURLY BRACKET
                                (#xA3FE . #xFFE3)	;FULLWIDTH MACRON
                                (#xA4A1 . #x3041)	;HIRAGANA LETTER SMALL A
                                (#xA4A2 . #x3042)	;HIRAGANA LETTER A
                                (#xA4A3 . #x3043)	;HIRAGANA LETTER SMALL I
                                (#xA4A4 . #x3044)	;HIRAGANA LETTER I
                                (#xA4A5 . #x3045)	;HIRAGANA LETTER SMALL U
                                (#xA4A6 . #x3046)	;HIRAGANA LETTER U
                                (#xA4A7 . #x3047)	;HIRAGANA LETTER SMALL E
                                (#xA4A8 . #x3048)	;HIRAGANA LETTER E
                                (#xA4A9 . #x3049)	;HIRAGANA LETTER SMALL O
                                (#xA4AA . #x304A)	;HIRAGANA LETTER O
                                (#xA4AB . #x304B)	;HIRAGANA LETTER KA
                                (#xA4AC . #x304C)	;HIRAGANA LETTER GA
                                (#xA4AD . #x304D)	;HIRAGANA LETTER KI
                                (#xA4AE . #x304E)	;HIRAGANA LETTER GI
                                (#xA4AF . #x304F)	;HIRAGANA LETTER KU
                                (#xA4B0 . #x3050)	;HIRAGANA LETTER GU
                                (#xA4B1 . #x3051)	;HIRAGANA LETTER KE
                                (#xA4B2 . #x3052)	;HIRAGANA LETTER GE
                                (#xA4B3 . #x3053)	;HIRAGANA LETTER KO
                                (#xA4B4 . #x3054)	;HIRAGANA LETTER GO
                                (#xA4B5 . #x3055)	;HIRAGANA LETTER SA
                                (#xA4B6 . #x3056)	;HIRAGANA LETTER ZA
                                (#xA4B7 . #x3057)	;HIRAGANA LETTER SI
                                (#xA4B8 . #x3058)	;HIRAGANA LETTER ZI
                                (#xA4B9 . #x3059)	;HIRAGANA LETTER SU
                                (#xA4BA . #x305A)	;HIRAGANA LETTER ZU
                                (#xA4BB . #x305B)	;HIRAGANA LETTER SE
                                (#xA4BC . #x305C)	;HIRAGANA LETTER ZE
                                (#xA4BD . #x305D)	;HIRAGANA LETTER SO
                                (#xA4BE . #x305E)	;HIRAGANA LETTER ZO
                                (#xA4BF . #x305F)	;HIRAGANA LETTER TA
                                (#xA4C0 . #x3060)	;HIRAGANA LETTER DA
                                (#xA4C1 . #x3061)	;HIRAGANA LETTER TI
                                (#xA4C2 . #x3062)	;HIRAGANA LETTER DI
                                (#xA4C3 . #x3063)	;HIRAGANA LETTER SMALL TU
                                (#xA4C4 . #x3064)	;HIRAGANA LETTER TU
                                (#xA4C5 . #x3065)	;HIRAGANA LETTER DU
                                (#xA4C6 . #x3066)	;HIRAGANA LETTER TE
                                (#xA4C7 . #x3067)	;HIRAGANA LETTER DE
                                (#xA4C8 . #x3068)	;HIRAGANA LETTER TO
                                (#xA4C9 . #x3069)	;HIRAGANA LETTER DO
                                (#xA4CA . #x306A)	;HIRAGANA LETTER NA
                                (#xA4CB . #x306B)	;HIRAGANA LETTER NI
                                (#xA4CC . #x306C)	;HIRAGANA LETTER NU
                                (#xA4CD . #x306D)	;HIRAGANA LETTER NE
                                (#xA4CE . #x306E)	;HIRAGANA LETTER NO
                                (#xA4CF . #x306F)	;HIRAGANA LETTER HA
                                (#xA4D0 . #x3070)	;HIRAGANA LETTER BA
                                (#xA4D1 . #x3071)	;HIRAGANA LETTER PA
                                (#xA4D2 . #x3072)	;HIRAGANA LETTER HI
                                (#xA4D3 . #x3073)	;HIRAGANA LETTER BI
                                (#xA4D4 . #x3074)	;HIRAGANA LETTER PI
                                (#xA4D5 . #x3075)	;HIRAGANA LETTER HU
                                (#xA4D6 . #x3076)	;HIRAGANA LETTER BU
                                (#xA4D7 . #x3077)	;HIRAGANA LETTER PU
                                (#xA4D8 . #x3078)	;HIRAGANA LETTER HE
                                (#xA4D9 . #x3079)	;HIRAGANA LETTER BE
                                (#xA4DA . #x307A)	;HIRAGANA LETTER PE
                                (#xA4DB . #x307B)	;HIRAGANA LETTER HO
                                (#xA4DC . #x307C)	;HIRAGANA LETTER BO
                                (#xA4DD . #x307D)	;HIRAGANA LETTER PO
                                (#xA4DE . #x307E)	;HIRAGANA LETTER MA
                                (#xA4DF . #x307F)	;HIRAGANA LETTER MI
                                (#xA4E0 . #x3080)	;HIRAGANA LETTER MU
                                (#xA4E1 . #x3081)	;HIRAGANA LETTER ME
                                (#xA4E2 . #x3082)	;HIRAGANA LETTER MO
                                (#xA4E3 . #x3083)	;HIRAGANA LETTER SMALL YA
                                (#xA4E4 . #x3084)	;HIRAGANA LETTER YA
                                (#xA4E5 . #x3085)	;HIRAGANA LETTER SMALL YU
                                (#xA4E6 . #x3086)	;HIRAGANA LETTER YU
                                (#xA4E7 . #x3087)	;HIRAGANA LETTER SMALL YO
                                (#xA4E8 . #x3088)	;HIRAGANA LETTER YO
                                (#xA4E9 . #x3089)	;HIRAGANA LETTER RA
                                (#xA4EA . #x308A)	;HIRAGANA LETTER RI
                                (#xA4EB . #x308B)	;HIRAGANA LETTER RU
                                (#xA4EC . #x308C)	;HIRAGANA LETTER RE
                                (#xA4ED . #x308D)	;HIRAGANA LETTER RO
                                (#xA4EE . #x308E)	;HIRAGANA LETTER SMALL WA
                                (#xA4EF . #x308F)	;HIRAGANA LETTER WA
                                (#xA4F0 . #x3090)	;HIRAGANA LETTER WI
                                (#xA4F1 . #x3091)	;HIRAGANA LETTER WE
                                (#xA4F2 . #x3092)	;HIRAGANA LETTER WO
                                (#xA4F3 . #x3093)	;HIRAGANA LETTER N
                                (#xA5A1 . #x30A1)	;KATAKANA LETTER SMALL A
                                (#xA5A2 . #x30A2)	;KATAKANA LETTER A
                                (#xA5A3 . #x30A3)	;KATAKANA LETTER SMALL I
                                (#xA5A4 . #x30A4)	;KATAKANA LETTER I
                                (#xA5A5 . #x30A5)	;KATAKANA LETTER SMALL U
                                (#xA5A6 . #x30A6)	;KATAKANA LETTER U
                                (#xA5A7 . #x30A7)	;KATAKANA LETTER SMALL E
                                (#xA5A8 . #x30A8)	;KATAKANA LETTER E
                                (#xA5A9 . #x30A9)	;KATAKANA LETTER SMALL O
                                (#xA5AA . #x30AA)	;KATAKANA LETTER O
                                (#xA5AB . #x30AB)	;KATAKANA LETTER KA
                                (#xA5AC . #x30AC)	;KATAKANA LETTER GA
                                (#xA5AD . #x30AD)	;KATAKANA LETTER KI
                                (#xA5AE . #x30AE)	;KATAKANA LETTER GI
                                (#xA5AF . #x30AF)	;KATAKANA LETTER KU
                                (#xA5B0 . #x30B0)	;KATAKANA LETTER GU
                                (#xA5B1 . #x30B1)	;KATAKANA LETTER KE
                                (#xA5B2 . #x30B2)	;KATAKANA LETTER GE
                                (#xA5B3 . #x30B3)	;KATAKANA LETTER KO
                                (#xA5B4 . #x30B4)	;KATAKANA LETTER GO
                                (#xA5B5 . #x30B5)	;KATAKANA LETTER SA
                                (#xA5B6 . #x30B6)	;KATAKANA LETTER ZA
                                (#xA5B7 . #x30B7)	;KATAKANA LETTER SI
                                (#xA5B8 . #x30B8)	;KATAKANA LETTER ZI
                                (#xA5B9 . #x30B9)	;KATAKANA LETTER SU
                                (#xA5BA . #x30BA)	;KATAKANA LETTER ZU
                                (#xA5BB . #x30BB)	;KATAKANA LETTER SE
                                (#xA5BC . #x30BC)	;KATAKANA LETTER ZE
                                (#xA5BD . #x30BD)	;KATAKANA LETTER SO
                                (#xA5BE . #x30BE)	;KATAKANA LETTER ZO
                                (#xA5BF . #x30BF)	;KATAKANA LETTER TA
                                (#xA5C0 . #x30C0)	;KATAKANA LETTER DA
                                (#xA5C1 . #x30C1)	;KATAKANA LETTER TI
                                (#xA5C2 . #x30C2)	;KATAKANA LETTER DI
                                (#xA5C3 . #x30C3)	;KATAKANA LETTER SMALL TU
                                (#xA5C4 . #x30C4)	;KATAKANA LETTER TU
                                (#xA5C5 . #x30C5)	;KATAKANA LETTER DU
                                (#xA5C6 . #x30C6)	;KATAKANA LETTER TE
                                (#xA5C7 . #x30C7)	;KATAKANA LETTER DE
                                (#xA5C8 . #x30C8)	;KATAKANA LETTER TO
                                (#xA5C9 . #x30C9)	;KATAKANA LETTER DO
                                (#xA5CA . #x30CA)	;KATAKANA LETTER NA
                                (#xA5CB . #x30CB)	;KATAKANA LETTER NI
                                (#xA5CC . #x30CC)	;KATAKANA LETTER NU
                                (#xA5CD . #x30CD)	;KATAKANA LETTER NE
                                (#xA5CE . #x30CE)	;KATAKANA LETTER NO
                                (#xA5CF . #x30CF)	;KATAKANA LETTER HA
                                (#xA5D0 . #x30D0)	;KATAKANA LETTER BA
                                (#xA5D1 . #x30D1)	;KATAKANA LETTER PA
                                (#xA5D2 . #x30D2)	;KATAKANA LETTER HI
                                (#xA5D3 . #x30D3)	;KATAKANA LETTER BI
                                (#xA5D4 . #x30D4)	;KATAKANA LETTER PI
                                (#xA5D5 . #x30D5)	;KATAKANA LETTER HU
                                (#xA5D6 . #x30D6)	;KATAKANA LETTER BU
                                (#xA5D7 . #x30D7)	;KATAKANA LETTER PU
                                (#xA5D8 . #x30D8)	;KATAKANA LETTER HE
                                (#xA5D9 . #x30D9)	;KATAKANA LETTER BE
                                (#xA5DA . #x30DA)	;KATAKANA LETTER PE
                                (#xA5DB . #x30DB)	;KATAKANA LETTER HO
                                (#xA5DC . #x30DC)	;KATAKANA LETTER BO
                                (#xA5DD . #x30DD)	;KATAKANA LETTER PO
                                (#xA5DE . #x30DE)	;KATAKANA LETTER MA
                                (#xA5DF . #x30DF)	;KATAKANA LETTER MI
                                (#xA5E0 . #x30E0)	;KATAKANA LETTER MU
                                (#xA5E1 . #x30E1)	;KATAKANA LETTER ME
                                (#xA5E2 . #x30E2)	;KATAKANA LETTER MO
                                (#xA5E3 . #x30E3)	;KATAKANA LETTER SMALL YA
                                (#xA5E4 . #x30E4)	;KATAKANA LETTER YA
                                (#xA5E5 . #x30E5)	;KATAKANA LETTER SMALL YU
                                (#xA5E6 . #x30E6)	;KATAKANA LETTER YU
                                (#xA5E7 . #x30E7)	;KATAKANA LETTER SMALL YO
                                (#xA5E8 . #x30E8)	;KATAKANA LETTER YO
                                (#xA5E9 . #x30E9)	;KATAKANA LETTER RA
                                (#xA5EA . #x30EA)	;KATAKANA LETTER RI
                                (#xA5EB . #x30EB)	;KATAKANA LETTER RU
                                (#xA5EC . #x30EC)	;KATAKANA LETTER RE
                                (#xA5ED . #x30ED)	;KATAKANA LETTER RO
                                (#xA5EE . #x30EE)	;KATAKANA LETTER SMALL WA
                                (#xA5EF . #x30EF)	;KATAKANA LETTER WA
                                (#xA5F0 . #x30F0)	;KATAKANA LETTER WI
                                (#xA5F1 . #x30F1)	;KATAKANA LETTER WE
                                (#xA5F2 . #x30F2)	;KATAKANA LETTER WO
                                (#xA5F3 . #x30F3)	;KATAKANA LETTER N
                                (#xA5F4 . #x30F4)	;KATAKANA LETTER VU
                                (#xA5F5 . #x30F5)	;KATAKANA LETTER SMALL KA
                                (#xA5F6 . #x30F6)	;KATAKANA LETTER SMALL KE
                                (#xA6A1 . #x0391)	;GREEK CAPITAL LETTER ALPHA
                                (#xA6A2 . #x0392)	;GREEK CAPITAL LETTER BETA
                                (#xA6A3 . #x0393)	;GREEK CAPITAL LETTER GAMMA
                                (#xA6A4 . #x0394)	;GREEK CAPITAL LETTER DELTA
                                (#xA6A5 . #x0395)	;GREEK CAPITAL LETTER EPSILON
                                (#xA6A6 . #x0396)	;GREEK CAPITAL LETTER ZETA
                                (#xA6A7 . #x0397)	;GREEK CAPITAL LETTER ETA
                                (#xA6A8 . #x0398)	;GREEK CAPITAL LETTER THETA
                                (#xA6A9 . #x0399)	;GREEK CAPITAL LETTER IOTA
                                (#xA6AA . #x039A)	;GREEK CAPITAL LETTER KAPPA
                                (#xA6AB . #x039B)	;GREEK CAPITAL LETTER LAMDA
                                (#xA6AC . #x039C)	;GREEK CAPITAL LETTER MU
                                (#xA6AD . #x039D)	;GREEK CAPITAL LETTER NU
                                (#xA6AE . #x039E)	;GREEK CAPITAL LETTER XI
                                (#xA6AF . #x039F)	;GREEK CAPITAL LETTER OMICRON
                                (#xA6B0 . #x03A0)	;GREEK CAPITAL LETTER PI
                                (#xA6B1 . #x03A1)	;GREEK CAPITAL LETTER RHO
                                (#xA6B2 . #x03A3)	;GREEK CAPITAL LETTER SIGMA
                                (#xA6B3 . #x03A4)	;GREEK CAPITAL LETTER TAU
                                (#xA6B4 . #x03A5)	;GREEK CAPITAL LETTER UPSILON
                                (#xA6B5 . #x03A6)	;GREEK CAPITAL LETTER PHI
                                (#xA6B6 . #x03A7)	;GREEK CAPITAL LETTER CHI
                                (#xA6B7 . #x03A8)	;GREEK CAPITAL LETTER PSI
                                (#xA6B8 . #x03A9)	;GREEK CAPITAL LETTER OMEGA
                                (#xA6C1 . #x03B1)	;GREEK SMALL LETTER ALPHA
                                (#xA6C2 . #x03B2)	;GREEK SMALL LETTER BETA
                                (#xA6C3 . #x03B3)	;GREEK SMALL LETTER GAMMA
                                (#xA6C4 . #x03B4)	;GREEK SMALL LETTER DELTA
                                (#xA6C5 . #x03B5)	;GREEK SMALL LETTER EPSILON
                                (#xA6C6 . #x03B6)	;GREEK SMALL LETTER ZETA
                                (#xA6C7 . #x03B7)	;GREEK SMALL LETTER ETA
                                (#xA6C8 . #x03B8)	;GREEK SMALL LETTER THETA
                                (#xA6C9 . #x03B9)	;GREEK SMALL LETTER IOTA
                                (#xA6CA . #x03BA)	;GREEK SMALL LETTER KAPPA
                                (#xA6CB . #x03BB)	;GREEK SMALL LETTER LAMDA
                                (#xA6CC . #x03BC)	;GREEK SMALL LETTER MU
                                (#xA6CD . #x03BD)	;GREEK SMALL LETTER NU
                                (#xA6CE . #x03BE)	;GREEK SMALL LETTER XI
                                (#xA6CF . #x03BF)	;GREEK SMALL LETTER OMICRON
                                (#xA6D0 . #x03C0)	;GREEK SMALL LETTER PI
                                (#xA6D1 . #x03C1)	;GREEK SMALL LETTER RHO
                                (#xA6D2 . #x03C3)	;GREEK SMALL LETTER SIGMA
                                (#xA6D3 . #x03C4)	;GREEK SMALL LETTER TAU
                                (#xA6D4 . #x03C5)	;GREEK SMALL LETTER UPSILON
                                (#xA6D5 . #x03C6)	;GREEK SMALL LETTER PHI
                                (#xA6D6 . #x03C7)	;GREEK SMALL LETTER CHI
                                (#xA6D7 . #x03C8)	;GREEK SMALL LETTER PSI
                                (#xA6D8 . #x03C9)	;GREEK SMALL LETTER OMEGA
                                (#xA6E0 . #xFE35)	;PRESENTATION FORM FOR VERTICAL LEFT PARENTHESIS
                                (#xA6E1 . #xFE36)	;PRESENTATION FORM FOR VERTICAL RIGHT PARENTHESIS
                                (#xA6E2 . #xFE39)	;PRESENTATION FORM FOR VERTICAL LEFT TORTOISE SHELL BRACKET
                                (#xA6E3 . #xFE3A)	;PRESENTATION FORM FOR VERTICAL RIGHT TORTOISE SHELL BRACKET
                                (#xA6E4 . #xFE3F)	;PRESENTATION FORM FOR VERTICAL LEFT ANGLE BRACKET
                                (#xA6E5 . #xFE40)	;PRESENTATION FORM FOR VERTICAL RIGHT ANGLE BRACKET
                                (#xA6E6 . #xFE3D)	;PRESENTATION FORM FOR VERTICAL LEFT DOUBLE ANGLE BRACKET
                                (#xA6E7 . #xFE3E)	;PRESENTATION FORM FOR VERTICAL RIGHT DOUBLE ANGLE BRACKET
                                (#xA6E8 . #xFE41)	;PRESENTATION FORM FOR VERTICAL LEFT CORNER BRACKET
                                (#xA6E9 . #xFE42)	;PRESENTATION FORM FOR VERTICAL RIGHT CORNER BRACKET
                                (#xA6EA . #xFE43)	;PRESENTATION FORM FOR VERTICAL LEFT WHITE CORNER BRACKET
                                (#xA6EB . #xFE44)	;PRESENTATION FORM FOR VERTICAL RIGHT WHITE CORNER BRACKET
                                (#xA6EE . #xFE3B)	;PRESENTATION FORM FOR VERTICAL LEFT BLACK LENTICULAR BRACKET
                                (#xA6EF . #xFE3C)	;PRESENTATION FORM FOR VERTICAL RIGHT BLACK LENTICULAR BRACKET
                                (#xA6F0 . #xFE37)	;PRESENTATION FORM FOR VERTICAL LEFT CURLY BRACKET
                                (#xA6F1 . #xFE38)	;PRESENTATION FORM FOR VERTICAL RIGHT CURLY BRACKET
                                (#xA6F2 . #xFE31)	;PRESENTATION FORM FOR VERTICAL EM DASH
                                (#xA6F4 . #xFE33)	;PRESENTATION FORM FOR VERTICAL LOW LINE
                                (#xA6F5 . #xFE34)	;PRESENTATION FORM FOR VERTICAL WAVY LOW LINE
                                (#xA7A1 . #x0410)	;CYRILLIC CAPITAL LETTER A
                                (#xA7A2 . #x0411)	;CYRILLIC CAPITAL LETTER BE
                                (#xA7A3 . #x0412)	;CYRILLIC CAPITAL LETTER VE
                                (#xA7A4 . #x0413)	;CYRILLIC CAPITAL LETTER GHE
                                (#xA7A5 . #x0414)	;CYRILLIC CAPITAL LETTER DE
                                (#xA7A6 . #x0415)	;CYRILLIC CAPITAL LETTER IE
                                (#xA7A7 . #x0401)	;CYRILLIC CAPITAL LETTER IO
                                (#xA7A8 . #x0416)	;CYRILLIC CAPITAL LETTER ZHE
                                (#xA7A9 . #x0417)	;CYRILLIC CAPITAL LETTER ZE
                                (#xA7AA . #x0418)	;CYRILLIC CAPITAL LETTER I
                                (#xA7AB . #x0419)	;CYRILLIC CAPITAL LETTER SHORT I
                                (#xA7AC . #x041A)	;CYRILLIC CAPITAL LETTER KA
                                (#xA7AD . #x041B)	;CYRILLIC CAPITAL LETTER EL
                                (#xA7AE . #x041C)	;CYRILLIC CAPITAL LETTER EM
                                (#xA7AF . #x041D)	;CYRILLIC CAPITAL LETTER EN
                                (#xA7B0 . #x041E)	;CYRILLIC CAPITAL LETTER O
                                (#xA7B1 . #x041F)	;CYRILLIC CAPITAL LETTER PE
                                (#xA7B2 . #x0420)	;CYRILLIC CAPITAL LETTER ER
                                (#xA7B3 . #x0421)	;CYRILLIC CAPITAL LETTER ES
                                (#xA7B4 . #x0422)	;CYRILLIC CAPITAL LETTER TE
                                (#xA7B5 . #x0423)	;CYRILLIC CAPITAL LETTER U
                                (#xA7B6 . #x0424)	;CYRILLIC CAPITAL LETTER EF
                                (#xA7B7 . #x0425)	;CYRILLIC CAPITAL LETTER HA
                                (#xA7B8 . #x0426)	;CYRILLIC CAPITAL LETTER TSE
                                (#xA7B9 . #x0427)	;CYRILLIC CAPITAL LETTER CHE
                                (#xA7BA . #x0428)	;CYRILLIC CAPITAL LETTER SHA
                                (#xA7BB . #x0429)	;CYRILLIC CAPITAL LETTER SHCHA
                                (#xA7BC . #x042A)	;CYRILLIC CAPITAL LETTER HARD SIGN
                                (#xA7BD . #x042B)	;CYRILLIC CAPITAL LETTER YERU
                                (#xA7BE . #x042C)	;CYRILLIC CAPITAL LETTER SOFT SIGN
                                (#xA7BF . #x042D)	;CYRILLIC CAPITAL LETTER E
                                (#xA7C0 . #x042E)	;CYRILLIC CAPITAL LETTER YU
                                (#xA7C1 . #x042F)	;CYRILLIC CAPITAL LETTER YA
                                (#xA7D1 . #x0430)	;CYRILLIC SMALL LETTER A
                                (#xA7D2 . #x0431)	;CYRILLIC SMALL LETTER BE
                                (#xA7D3 . #x0432)	;CYRILLIC SMALL LETTER VE
                                (#xA7D4 . #x0433)	;CYRILLIC SMALL LETTER GHE
                                (#xA7D5 . #x0434)	;CYRILLIC SMALL LETTER DE
                                (#xA7D6 . #x0435)	;CYRILLIC SMALL LETTER IE
                                (#xA7D7 . #x0451)	;CYRILLIC SMALL LETTER IO
                                (#xA7D8 . #x0436)	;CYRILLIC SMALL LETTER ZHE
                                (#xA7D9 . #x0437)	;CYRILLIC SMALL LETTER ZE
                                (#xA7DA . #x0438)	;CYRILLIC SMALL LETTER I
                                (#xA7DB . #x0439)	;CYRILLIC SMALL LETTER SHORT I
                                (#xA7DC . #x043A)	;CYRILLIC SMALL LETTER KA
                                (#xA7DD . #x043B)	;CYRILLIC SMALL LETTER EL
                                (#xA7DE . #x043C)	;CYRILLIC SMALL LETTER EM
                                (#xA7DF . #x043D)	;CYRILLIC SMALL LETTER EN
                                (#xA7E0 . #x043E)	;CYRILLIC SMALL LETTER O
                                (#xA7E1 . #x043F)	;CYRILLIC SMALL LETTER PE
                                (#xA7E2 . #x0440)	;CYRILLIC SMALL LETTER ER
                                (#xA7E3 . #x0441)	;CYRILLIC SMALL LETTER ES
                                (#xA7E4 . #x0442)	;CYRILLIC SMALL LETTER TE
                                (#xA7E5 . #x0443)	;CYRILLIC SMALL LETTER U
                                (#xA7E6 . #x0444)	;CYRILLIC SMALL LETTER EF
                                (#xA7E7 . #x0445)	;CYRILLIC SMALL LETTER HA
                                (#xA7E8 . #x0446)	;CYRILLIC SMALL LETTER TSE
                                (#xA7E9 . #x0447)	;CYRILLIC SMALL LETTER CHE
                                (#xA7EA . #x0448)	;CYRILLIC SMALL LETTER SHA
                                (#xA7EB . #x0449)	;CYRILLIC SMALL LETTER SHCHA
                                (#xA7EC . #x044A)	;CYRILLIC SMALL LETTER HARD SIGN
                                (#xA7ED . #x044B)	;CYRILLIC SMALL LETTER YERU
                                (#xA7EE . #x044C)	;CYRILLIC SMALL LETTER SOFT SIGN
                                (#xA7EF . #x044D)	;CYRILLIC SMALL LETTER E
                                (#xA7F0 . #x044E)	;CYRILLIC SMALL LETTER YU
                                (#xA7F1 . #x044F)	;CYRILLIC SMALL LETTER YA
                                (#xA840 . #x02CA)	;MODIFIER LETTER ACUTE ACCENT
                                (#xA841 . #x02CB)	;MODIFIER LETTER GRAVE ACCENT
                                (#xA842 . #x02D9)	;DOT ABOVE
                                (#xA843 . #x2013)	;EN DASH
                                (#xA844 . #x2015)	;HORIZONTAL BAR
                                (#xA845 . #x2025)	;TWO DOT LEADER
                                (#xA846 . #x2035)	;REVERSED PRIME
                                (#xA847 . #x2105)	;CARE OF
                                (#xA848 . #x2109)	;DEGREE FAHRENHEIT
                                (#xA849 . #x2196)	;NORTH WEST ARROW
                                (#xA84A . #x2197)	;NORTH EAST ARROW
                                (#xA84B . #x2198)	;SOUTH EAST ARROW
                                (#xA84C . #x2199)	;SOUTH WEST ARROW
                                (#xA84D . #x2215)	;DIVISION SLASH
                                (#xA84E . #x221F)	;RIGHT ANGLE
                                (#xA84F . #x2223)	;DIVIDES
                                (#xA850 . #x2252)	;APPROXIMATELY EQUAL TO OR THE IMAGE OF
                                (#xA851 . #x2266)	;LESS-THAN OVER EQUAL TO
                                (#xA852 . #x2267)	;GREATER-THAN OVER EQUAL TO
                                (#xA853 . #x22BF)	;RIGHT TRIANGLE
                                (#xA854 . #x2550)	;BOX DRAWINGS DOUBLE HORIZONTAL
                                (#xA855 . #x2551)	;BOX DRAWINGS DOUBLE VERTICAL
                                (#xA856 . #x2552)	;BOX DRAWINGS DOWN SINGLE AND RIGHT DOUBLE
                                (#xA857 . #x2553)	;BOX DRAWINGS DOWN DOUBLE AND RIGHT SINGLE
                                (#xA858 . #x2554)	;BOX DRAWINGS DOUBLE DOWN AND RIGHT
                                (#xA859 . #x2555)	;BOX DRAWINGS DOWN SINGLE AND LEFT DOUBLE
                                (#xA85A . #x2556)	;BOX DRAWINGS DOWN DOUBLE AND LEFT SINGLE
                                (#xA85B . #x2557)	;BOX DRAWINGS DOUBLE DOWN AND LEFT
                                (#xA85C . #x2558)	;BOX DRAWINGS UP SINGLE AND RIGHT DOUBLE
                                (#xA85D . #x2559)	;BOX DRAWINGS UP DOUBLE AND RIGHT SINGLE
                                (#xA85E . #x255A)	;BOX DRAWINGS DOUBLE UP AND RIGHT
                                (#xA85F . #x255B)	;BOX DRAWINGS UP SINGLE AND LEFT DOUBLE
                                (#xA860 . #x255C)	;BOX DRAWINGS UP DOUBLE AND LEFT SINGLE
                                (#xA861 . #x255D)	;BOX DRAWINGS DOUBLE UP AND LEFT
                                (#xA862 . #x255E)	;BOX DRAWINGS VERTICAL SINGLE AND RIGHT DOUBLE
                                (#xA863 . #x255F)	;BOX DRAWINGS VERTICAL DOUBLE AND RIGHT SINGLE
                                (#xA864 . #x2560)	;BOX DRAWINGS DOUBLE VERTICAL AND RIGHT
                                (#xA865 . #x2561)	;BOX DRAWINGS VERTICAL SINGLE AND LEFT DOUBLE
                                (#xA866 . #x2562)	;BOX DRAWINGS VERTICAL DOUBLE AND LEFT SINGLE
                                (#xA867 . #x2563)	;BOX DRAWINGS DOUBLE VERTICAL AND LEFT
                                (#xA868 . #x2564)	;BOX DRAWINGS DOWN SINGLE AND HORIZONTAL DOUBLE
                                (#xA869 . #x2565)	;BOX DRAWINGS DOWN DOUBLE AND HORIZONTAL SINGLE
                                (#xA86A . #x2566)	;BOX DRAWINGS DOUBLE DOWN AND HORIZONTAL
                                (#xA86B . #x2567)	;BOX DRAWINGS UP SINGLE AND HORIZONTAL DOUBLE
                                (#xA86C . #x2568)	;BOX DRAWINGS UP DOUBLE AND HORIZONTAL SINGLE
                                (#xA86D . #x2569)	;BOX DRAWINGS DOUBLE UP AND HORIZONTAL
                                (#xA86E . #x256A)	;BOX DRAWINGS VERTICAL SINGLE AND HORIZONTAL DOUBLE
                                (#xA86F . #x256B)	;BOX DRAWINGS VERTICAL DOUBLE AND HORIZONTAL SINGLE
                                (#xA870 . #x256C)	;BOX DRAWINGS DOUBLE VERTICAL AND HORIZONTAL
                                (#xA871 . #x256D)	;BOX DRAWINGS LIGHT ARC DOWN AND RIGHT
                                (#xA872 . #x256E)	;BOX DRAWINGS LIGHT ARC DOWN AND LEFT
                                (#xA873 . #x256F)	;BOX DRAWINGS LIGHT ARC UP AND LEFT
                                (#xA874 . #x2570)	;BOX DRAWINGS LIGHT ARC UP AND RIGHT
                                (#xA875 . #x2571)	;BOX DRAWINGS LIGHT DIAGONAL UPPER RIGHT TO LOWER LEFT
                                (#xA876 . #x2572)	;BOX DRAWINGS LIGHT DIAGONAL UPPER LEFT TO LOWER RIGHT
                                (#xA877 . #x2573)	;BOX DRAWINGS LIGHT DIAGONAL CROSS
                                (#xA878 . #x2581)	;LOWER ONE EIGHTH BLOCK
                                (#xA879 . #x2582)	;LOWER ONE QUARTER BLOCK
                                (#xA87A . #x2583)	;LOWER THREE EIGHTHS BLOCK
                                (#xA87B . #x2584)	;LOWER HALF BLOCK
                                (#xA87C . #x2585)	;LOWER FIVE EIGHTHS BLOCK
                                (#xA87D . #x2586)	;LOWER THREE QUARTERS BLOCK
                                (#xA87E . #x2587)	;LOWER SEVEN EIGHTHS BLOCK
                                (#xA880 . #x2588)	;FULL BLOCK
                                (#xA881 . #x2589)	;LEFT SEVEN EIGHTHS BLOCK
                                (#xA882 . #x258A)	;LEFT THREE QUARTERS BLOCK
                                (#xA883 . #x258B)	;LEFT FIVE EIGHTHS BLOCK
                                (#xA884 . #x258C)	;LEFT HALF BLOCK
                                (#xA885 . #x258D)	;LEFT THREE EIGHTHS BLOCK
                                (#xA886 . #x258E)	;LEFT ONE QUARTER BLOCK
                                (#xA887 . #x258F)	;LEFT ONE EIGHTH BLOCK
                                (#xA888 . #x2593)	;DARK SHADE
                                (#xA889 . #x2594)	;UPPER ONE EIGHTH BLOCK
                                (#xA88A . #x2595)	;RIGHT ONE EIGHTH BLOCK
                                (#xA88B . #x25BC)	;BLACK DOWN-POINTING TRIANGLE
                                (#xA88C . #x25BD)	;WHITE DOWN-POINTING TRIANGLE
                                (#xA88D . #x25E2)	;BLACK LOWER RIGHT TRIANGLE
                                (#xA88E . #x25E3)	;BLACK LOWER LEFT TRIANGLE
                                (#xA88F . #x25E4)	;BLACK UPPER LEFT TRIANGLE
                                (#xA890 . #x25E5)	;BLACK UPPER RIGHT TRIANGLE
                                (#xA891 . #x2609)	;SUN
                                (#xA892 . #x2295)	;CIRCLED PLUS
                                (#xA893 . #x3012)	;POSTAL MARK
                                (#xA894 . #x301D)	;REVERSED DOUBLE PRIME QUOTATION MARK
                                (#xA895 . #x301E)	;DOUBLE PRIME QUOTATION MARK
                                (#xA8A1 . #x0101)	;LATIN SMALL LETTER A WITH MACRON
                                (#xA8A2 . #x00E1)	;LATIN SMALL LETTER A WITH ACUTE
                                (#xA8A3 . #x01CE)	;LATIN SMALL LETTER A WITH CARON
                                (#xA8A4 . #x00E0)	;LATIN SMALL LETTER A WITH GRAVE
                                (#xA8A5 . #x0113)	;LATIN SMALL LETTER E WITH MACRON
                                (#xA8A6 . #x00E9)	;LATIN SMALL LETTER E WITH ACUTE
                                (#xA8A7 . #x011B)	;LATIN SMALL LETTER E WITH CARON
                                (#xA8A8 . #x00E8)	;LATIN SMALL LETTER E WITH GRAVE
                                (#xA8A9 . #x012B)	;LATIN SMALL LETTER I WITH MACRON
                                (#xA8AA . #x00ED)	;LATIN SMALL LETTER I WITH ACUTE
                                (#xA8AB . #x01D0)	;LATIN SMALL LETTER I WITH CARON
                                (#xA8AC . #x00EC)	;LATIN SMALL LETTER I WITH GRAVE
                                (#xA8AD . #x014D)	;LATIN SMALL LETTER O WITH MACRON
                                (#xA8AE . #x00F3)	;LATIN SMALL LETTER O WITH ACUTE
                                (#xA8AF . #x01D2)	;LATIN SMALL LETTER O WITH CARON
                                (#xA8B0 . #x00F2)	;LATIN SMALL LETTER O WITH GRAVE
                                (#xA8B1 . #x016B)	;LATIN SMALL LETTER U WITH MACRON
                                (#xA8B2 . #x00FA)	;LATIN SMALL LETTER U WITH ACUTE
                                (#xA8B3 . #x01D4)	;LATIN SMALL LETTER U WITH CARON
                                (#xA8B4 . #x00F9)	;LATIN SMALL LETTER U WITH GRAVE
                                (#xA8B5 . #x01D6)	;LATIN SMALL LETTER U WITH DIAERESIS AND MACRON
                                (#xA8B6 . #x01D8)	;LATIN SMALL LETTER U WITH DIAERESIS AND ACUTE
                                (#xA8B7 . #x01DA)	;LATIN SMALL LETTER U WITH DIAERESIS AND CARON
                                (#xA8B8 . #x01DC)	;LATIN SMALL LETTER U WITH DIAERESIS AND GRAVE
                                (#xA8B9 . #x00FC)	;LATIN SMALL LETTER U WITH DIAERESIS
                                (#xA8BA . #x00EA)	;LATIN SMALL LETTER E WITH CIRCUMFLEX
                                (#xA8BB . #x0251)	;LATIN SMALL LETTER ALPHA
                                (#xA8BD . #x0144)	;LATIN SMALL LETTER N WITH ACUTE
                                (#xA8BE . #x0148)	;LATIN SMALL LETTER N WITH CARON
                                (#xA8C0 . #x0261)	;LATIN SMALL LETTER SCRIPT G
                                (#xA8C5 . #x3105)	;BOPOMOFO LETTER B
                                (#xA8C6 . #x3106)	;BOPOMOFO LETTER P
                                (#xA8C7 . #x3107)	;BOPOMOFO LETTER M
                                (#xA8C8 . #x3108)	;BOPOMOFO LETTER F
                                (#xA8C9 . #x3109)	;BOPOMOFO LETTER D
                                (#xA8CA . #x310A)	;BOPOMOFO LETTER T
                                (#xA8CB . #x310B)	;BOPOMOFO LETTER N
                                (#xA8CC . #x310C)	;BOPOMOFO LETTER L
                                (#xA8CD . #x310D)	;BOPOMOFO LETTER G
                                (#xA8CE . #x310E)	;BOPOMOFO LETTER K
                                (#xA8CF . #x310F)	;BOPOMOFO LETTER H
                                (#xA8D0 . #x3110)	;BOPOMOFO LETTER J
                                (#xA8D1 . #x3111)	;BOPOMOFO LETTER Q
                                (#xA8D2 . #x3112)	;BOPOMOFO LETTER X
                                (#xA8D3 . #x3113)	;BOPOMOFO LETTER ZH
                                (#xA8D4 . #x3114)	;BOPOMOFO LETTER CH
                                (#xA8D5 . #x3115)	;BOPOMOFO LETTER SH
                                (#xA8D6 . #x3116)	;BOPOMOFO LETTER R
                                (#xA8D7 . #x3117)	;BOPOMOFO LETTER Z
                                (#xA8D8 . #x3118)	;BOPOMOFO LETTER C
                                (#xA8D9 . #x3119)	;BOPOMOFO LETTER S
                                (#xA8DA . #x311A)	;BOPOMOFO LETTER A
                                (#xA8DB . #x311B)	;BOPOMOFO LETTER O
                                (#xA8DC . #x311C)	;BOPOMOFO LETTER E
                                (#xA8DD . #x311D)	;BOPOMOFO LETTER EH
                                (#xA8DE . #x311E)	;BOPOMOFO LETTER AI
                                (#xA8DF . #x311F)	;BOPOMOFO LETTER EI
                                (#xA8E0 . #x3120)	;BOPOMOFO LETTER AU
                                (#xA8E1 . #x3121)	;BOPOMOFO LETTER OU
                                (#xA8E2 . #x3122)	;BOPOMOFO LETTER AN
                                (#xA8E3 . #x3123)	;BOPOMOFO LETTER EN
                                (#xA8E4 . #x3124)	;BOPOMOFO LETTER ANG
                                (#xA8E5 . #x3125)	;BOPOMOFO LETTER ENG
                                (#xA8E6 . #x3126)	;BOPOMOFO LETTER ER
                                (#xA8E7 . #x3127)	;BOPOMOFO LETTER I
                                (#xA8E8 . #x3128)	;BOPOMOFO LETTER U
                                (#xA8E9 . #x3129)	;BOPOMOFO LETTER IU
                                (#xA940 . #x3021)	;HANGZHOU NUMERAL ONE
                                (#xA941 . #x3022)	;HANGZHOU NUMERAL TWO
                                (#xA942 . #x3023)	;HANGZHOU NUMERAL THREE
                                (#xA943 . #x3024)	;HANGZHOU NUMERAL FOUR
                                (#xA944 . #x3025)	;HANGZHOU NUMERAL FIVE
                                (#xA945 . #x3026)	;HANGZHOU NUMERAL SIX
                                (#xA946 . #x3027)	;HANGZHOU NUMERAL SEVEN
                                (#xA947 . #x3028)	;HANGZHOU NUMERAL EIGHT
                                (#xA948 . #x3029)	;HANGZHOU NUMERAL NINE
                                (#xA949 . #x32A3)	;CIRCLED IDEOGRAPH CORRECT
                                (#xA94A . #x338E)	;SQUARE MG
                                (#xA94B . #x338F)	;SQUARE KG
                                (#xA94C . #x339C)	;SQUARE MM
                                (#xA94D . #x339D)	;SQUARE CM
                                (#xA94E . #x339E)	;SQUARE KM
                                (#xA94F . #x33A1)	;SQUARE M SQUARED
                                (#xA950 . #x33C4)	;SQUARE CC
                                (#xA951 . #x33CE)	;SQUARE KM CAPITAL
                                (#xA952 . #x33D1)	;SQUARE LN
                                (#xA953 . #x33D2)	;SQUARE LOG
                                (#xA954 . #x33D5)	;SQUARE MIL
                                (#xA955 . #xFE30)	;PRESENTATION FORM FOR VERTICAL TWO DOT LEADER
                                (#xA956 . #xFFE2)	;FULLWIDTH NOT SIGN
                                (#xA957 . #xFFE4)	;FULLWIDTH BROKEN BAR
                                (#xA959 . #x2121)	;TELEPHONE SIGN
                                (#xA95A . #x3231)	;PARENTHESIZED IDEOGRAPH STOCK
                                (#xA95C . #x2010)	;HYPHEN
                                (#xA960 . #x30FC)	;KATAKANA-HIRAGANA PROLONGED SOUND MARK
                                (#xA961 . #x309B)	;KATAKANA-HIRAGANA VOICED SOUND MARK
                                (#xA962 . #x309C)	;KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK
                                (#xA963 . #x30FD)	;KATAKANA ITERATION MARK
                                (#xA964 . #x30FE)	;KATAKANA VOICED ITERATION MARK
                                (#xA965 . #x3006)	;IDEOGRAPHIC CLOSING MARK
                                (#xA966 . #x309D)	;HIRAGANA ITERATION MARK
                                (#xA967 . #x309E)	;HIRAGANA VOICED ITERATION MARK
                                (#xA968 . #xFE49)	;DASHED OVERLINE
                                (#xA969 . #xFE4A)	;CENTRELINE OVERLINE
                                (#xA96A . #xFE4B)	;WAVY OVERLINE
                                (#xA96B . #xFE4C)	;DOUBLE WAVY OVERLINE
                                (#xA96C . #xFE4D)	;DASHED LOW LINE
                                (#xA96D . #xFE4E)	;CENTRELINE LOW LINE
                                (#xA96E . #xFE4F)	;WAVY LOW LINE
                                (#xA96F . #xFE50)	;SMALL COMMA
                                (#xA970 . #xFE51)	;SMALL IDEOGRAPHIC COMMA
                                (#xA971 . #xFE52)	;SMALL FULL STOP
                                (#xA972 . #xFE54)	;SMALL SEMICOLON
                                (#xA973 . #xFE55)	;SMALL COLON
                                (#xA974 . #xFE56)	;SMALL QUESTION MARK
                                (#xA975 . #xFE57)	;SMALL EXCLAMATION MARK
                                (#xA976 . #xFE59)	;SMALL LEFT PARENTHESIS
                                (#xA977 . #xFE5A)	;SMALL RIGHT PARENTHESIS
                                (#xA978 . #xFE5B)	;SMALL LEFT CURLY BRACKET
                                (#xA979 . #xFE5C)	;SMALL RIGHT CURLY BRACKET
                                (#xA97A . #xFE5D)	;SMALL LEFT TORTOISE SHELL BRACKET
                                (#xA97B . #xFE5E)	;SMALL RIGHT TORTOISE SHELL BRACKET
                                (#xA97C . #xFE5F)	;SMALL NUMBER SIGN
                                (#xA97D . #xFE60)	;SMALL AMPERSAND
                                (#xA97E . #xFE61)	;SMALL ASTERISK
                                (#xA980 . #xFE62)	;SMALL PLUS SIGN
                                (#xA981 . #xFE63)	;SMALL HYPHEN-MINUS
                                (#xA982 . #xFE64)	;SMALL LESS-THAN SIGN
                                (#xA983 . #xFE65)	;SMALL GREATER-THAN SIGN
                                (#xA984 . #xFE66)	;SMALL EQUALS SIGN
                                (#xA985 . #xFE68)	;SMALL REVERSE SOLIDUS
                                (#xA986 . #xFE69)	;SMALL DOLLAR SIGN
                                (#xA987 . #xFE6A)	;SMALL PERCENT SIGN
                                (#xA988 . #xFE6B)	;SMALL COMMERCIAL AT
                                (#xA996 . #x3007)	;IDEOGRAPHIC NUMBER ZERO
                                (#xA9A4 . #x2500)	;BOX DRAWINGS LIGHT HORIZONTAL
                                (#xA9A5 . #x2501)	;BOX DRAWINGS HEAVY HORIZONTAL
                                (#xA9A6 . #x2502)	;BOX DRAWINGS LIGHT VERTICAL
                                (#xA9A7 . #x2503)	;BOX DRAWINGS HEAVY VERTICAL
                                (#xA9A8 . #x2504)	;BOX DRAWINGS LIGHT TRIPLE DASH HORIZONTAL
                                (#xA9A9 . #x2505)	;BOX DRAWINGS HEAVY TRIPLE DASH HORIZONTAL
                                (#xA9AA . #x2506)	;BOX DRAWINGS LIGHT TRIPLE DASH VERTICAL
                                (#xA9AB . #x2507)	;BOX DRAWINGS HEAVY TRIPLE DASH VERTICAL
                                (#xA9AC . #x2508)	;BOX DRAWINGS LIGHT QUADRUPLE DASH HORIZONTAL
                                (#xA9AD . #x2509)	;BOX DRAWINGS HEAVY QUADRUPLE DASH HORIZONTAL
                                (#xA9AE . #x250A)	;BOX DRAWINGS LIGHT QUADRUPLE DASH VERTICAL
                                (#xA9AF . #x250B)	;BOX DRAWINGS HEAVY QUADRUPLE DASH VERTICAL
                                (#xA9B0 . #x250C)	;BOX DRAWINGS LIGHT DOWN AND RIGHT
                                (#xA9B1 . #x250D)	;BOX DRAWINGS DOWN LIGHT AND RIGHT HEAVY
                                (#xA9B2 . #x250E)	;BOX DRAWINGS DOWN HEAVY AND RIGHT LIGHT
                                (#xA9B3 . #x250F)	;BOX DRAWINGS HEAVY DOWN AND RIGHT
                                (#xA9B4 . #x2510)	;BOX DRAWINGS LIGHT DOWN AND LEFT
                                (#xA9B5 . #x2511)	;BOX DRAWINGS DOWN LIGHT AND LEFT HEAVY
                                (#xA9B6 . #x2512)	;BOX DRAWINGS DOWN HEAVY AND LEFT LIGHT
                                (#xA9B7 . #x2513)	;BOX DRAWINGS HEAVY DOWN AND LEFT
                                (#xA9B8 . #x2514)	;BOX DRAWINGS LIGHT UP AND RIGHT
                                (#xA9B9 . #x2515)	;BOX DRAWINGS UP LIGHT AND RIGHT HEAVY
                                (#xA9BA . #x2516)	;BOX DRAWINGS UP HEAVY AND RIGHT LIGHT
                                (#xA9BB . #x2517)	;BOX DRAWINGS HEAVY UP AND RIGHT
                                (#xA9BC . #x2518)	;BOX DRAWINGS LIGHT UP AND LEFT
                                (#xA9BD . #x2519)	;BOX DRAWINGS UP LIGHT AND LEFT HEAVY
                                (#xA9BE . #x251A)	;BOX DRAWINGS UP HEAVY AND LEFT LIGHT
                                (#xA9BF . #x251B)	;BOX DRAWINGS HEAVY UP AND LEFT
                                (#xA9C0 . #x251C)	;BOX DRAWINGS LIGHT VERTICAL AND RIGHT
                                (#xA9C1 . #x251D)	;BOX DRAWINGS VERTICAL LIGHT AND RIGHT HEAVY
                                (#xA9C2 . #x251E)	;BOX DRAWINGS UP HEAVY AND RIGHT DOWN LIGHT
                                (#xA9C3 . #x251F)	;BOX DRAWINGS DOWN HEAVY AND RIGHT UP LIGHT
                                (#xA9C4 . #x2520)	;BOX DRAWINGS VERTICAL HEAVY AND RIGHT LIGHT
                                (#xA9C5 . #x2521)	;BOX DRAWINGS DOWN LIGHT AND RIGHT UP HEAVY
                                (#xA9C6 . #x2522)	;BOX DRAWINGS UP LIGHT AND RIGHT DOWN HEAVY
                                (#xA9C7 . #x2523)	;BOX DRAWINGS HEAVY VERTICAL AND RIGHT
                                (#xA9C8 . #x2524)	;BOX DRAWINGS LIGHT VERTICAL AND LEFT
                                (#xA9C9 . #x2525)	;BOX DRAWINGS VERTICAL LIGHT AND LEFT HEAVY
                                (#xA9CA . #x2526)	;BOX DRAWINGS UP HEAVY AND LEFT DOWN LIGHT
                                (#xA9CB . #x2527)	;BOX DRAWINGS DOWN HEAVY AND LEFT UP LIGHT
                                (#xA9CC . #x2528)	;BOX DRAWINGS VERTICAL HEAVY AND LEFT LIGHT
                                (#xA9CD . #x2529)	;BOX DRAWINGS DOWN LIGHT AND LEFT UP HEAVY
                                (#xA9CE . #x252A)	;BOX DRAWINGS UP LIGHT AND LEFT DOWN HEAVY
                                (#xA9CF . #x252B)	;BOX DRAWINGS HEAVY VERTICAL AND LEFT
                                (#xA9D0 . #x252C)	;BOX DRAWINGS LIGHT DOWN AND HORIZONTAL
                                (#xA9D1 . #x252D)	;BOX DRAWINGS LEFT HEAVY AND RIGHT DOWN LIGHT
                                (#xA9D2 . #x252E)	;BOX DRAWINGS RIGHT HEAVY AND LEFT DOWN LIGHT
                                (#xA9D3 . #x252F)	;BOX DRAWINGS DOWN LIGHT AND HORIZONTAL HEAVY
                                (#xA9D4 . #x2530)	;BOX DRAWINGS DOWN HEAVY AND HORIZONTAL LIGHT
                                (#xA9D5 . #x2531)	;BOX DRAWINGS RIGHT LIGHT AND LEFT DOWN HEAVY
                                (#xA9D6 . #x2532)	;BOX DRAWINGS LEFT LIGHT AND RIGHT DOWN HEAVY
                                (#xA9D7 . #x2533)	;BOX DRAWINGS HEAVY DOWN AND HORIZONTAL
                                (#xA9D8 . #x2534)	;BOX DRAWINGS LIGHT UP AND HORIZONTAL
                                (#xA9D9 . #x2535)	;BOX DRAWINGS LEFT HEAVY AND RIGHT UP LIGHT
                                (#xA9DA . #x2536)	;BOX DRAWINGS RIGHT HEAVY AND LEFT UP LIGHT
                                (#xA9DB . #x2537)	;BOX DRAWINGS UP LIGHT AND HORIZONTAL HEAVY
                                (#xA9DC . #x2538)	;BOX DRAWINGS UP HEAVY AND HORIZONTAL LIGHT
                                (#xA9DD . #x2539)	;BOX DRAWINGS RIGHT LIGHT AND LEFT UP HEAVY
                                (#xA9DE . #x253A)	;BOX DRAWINGS LEFT LIGHT AND RIGHT UP HEAVY
                                (#xA9DF . #x253B)	;BOX DRAWINGS HEAVY UP AND HORIZONTAL
                                (#xA9E0 . #x253C)	;BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL
                                (#xA9E1 . #x253D)	;BOX DRAWINGS LEFT HEAVY AND RIGHT VERTICAL LIGHT
                                (#xA9E2 . #x253E)	;BOX DRAWINGS RIGHT HEAVY AND LEFT VERTICAL LIGHT
                                (#xA9E3 . #x253F)	;BOX DRAWINGS VERTICAL LIGHT AND HORIZONTAL HEAVY
                                (#xA9E4 . #x2540)	;BOX DRAWINGS UP HEAVY AND DOWN HORIZONTAL LIGHT
                                (#xA9E5 . #x2541)	;BOX DRAWINGS DOWN HEAVY AND UP HORIZONTAL LIGHT
                                (#xA9E6 . #x2542)	;BOX DRAWINGS VERTICAL HEAVY AND HORIZONTAL LIGHT
                                (#xA9E7 . #x2543)	;BOX DRAWINGS LEFT UP HEAVY AND RIGHT DOWN LIGHT
                                (#xA9E8 . #x2544)	;BOX DRAWINGS RIGHT UP HEAVY AND LEFT DOWN LIGHT
                                (#xA9E9 . #x2545)	;BOX DRAWINGS LEFT DOWN HEAVY AND RIGHT UP LIGHT
                                (#xA9EA . #x2546)	;BOX DRAWINGS RIGHT DOWN HEAVY AND LEFT UP LIGHT
                                (#xA9EB . #x2547)	;BOX DRAWINGS DOWN LIGHT AND UP HORIZONTAL HEAVY
                                (#xA9EC . #x2548)	;BOX DRAWINGS UP LIGHT AND DOWN HORIZONTAL HEAVY
                                (#xA9ED . #x2549)	;BOX DRAWINGS RIGHT LIGHT AND LEFT VERTICAL HEAVY
                                (#xA9EE . #x254A)	;BOX DRAWINGS LEFT LIGHT AND RIGHT VERTICAL HEAVY
                                (#xA9EF . #x254B)	;BOX DRAWINGS HEAVY VERTICAL AND HORIZONTAL
                                (#xAA40 . #x72DC)	;CJK UNIFIED IDEOGRAPH
                                (#xAA41 . #x72DD)	;CJK UNIFIED IDEOGRAPH
                                (#xAA42 . #x72DF)	;CJK UNIFIED IDEOGRAPH
                                (#xAA43 . #x72E2)	;CJK UNIFIED IDEOGRAPH
                                (#xAA44 . #x72E3)	;CJK UNIFIED IDEOGRAPH
                                (#xAA45 . #x72E4)	;CJK UNIFIED IDEOGRAPH
                                (#xAA46 . #x72E5)	;CJK UNIFIED IDEOGRAPH
                                (#xAA47 . #x72E6)	;CJK UNIFIED IDEOGRAPH
                                (#xAA48 . #x72E7)	;CJK UNIFIED IDEOGRAPH
                                (#xAA49 . #x72EA)	;CJK UNIFIED IDEOGRAPH
                                (#xAA4A . #x72EB)	;CJK UNIFIED IDEOGRAPH
                                (#xAA4B . #x72F5)	;CJK UNIFIED IDEOGRAPH
                                (#xAA4C . #x72F6)	;CJK UNIFIED IDEOGRAPH
                                (#xAA4D . #x72F9)	;CJK UNIFIED IDEOGRAPH
                                (#xAA4E . #x72FD)	;CJK UNIFIED IDEOGRAPH
                                (#xAA4F . #x72FE)	;CJK UNIFIED IDEOGRAPH
                                (#xAA50 . #x72FF)	;CJK UNIFIED IDEOGRAPH
                                (#xAA51 . #x7300)	;CJK UNIFIED IDEOGRAPH
                                (#xAA52 . #x7302)	;CJK UNIFIED IDEOGRAPH
                                (#xAA53 . #x7304)	;CJK UNIFIED IDEOGRAPH
                                (#xAA54 . #x7305)	;CJK UNIFIED IDEOGRAPH
                                (#xAA55 . #x7306)	;CJK UNIFIED IDEOGRAPH
                                (#xAA56 . #x7307)	;CJK UNIFIED IDEOGRAPH
                                (#xAA57 . #x7308)	;CJK UNIFIED IDEOGRAPH
                                (#xAA58 . #x7309)	;CJK UNIFIED IDEOGRAPH
                                (#xAA59 . #x730B)	;CJK UNIFIED IDEOGRAPH
                                (#xAA5A . #x730C)	;CJK UNIFIED IDEOGRAPH
                                (#xAA5B . #x730D)	;CJK UNIFIED IDEOGRAPH
                                (#xAA5C . #x730F)	;CJK UNIFIED IDEOGRAPH
                                (#xAA5D . #x7310)	;CJK UNIFIED IDEOGRAPH
                                (#xAA5E . #x7311)	;CJK UNIFIED IDEOGRAPH
                                (#xAA5F . #x7312)	;CJK UNIFIED IDEOGRAPH
                                (#xAA60 . #x7314)	;CJK UNIFIED IDEOGRAPH
                                (#xAA61 . #x7318)	;CJK UNIFIED IDEOGRAPH
                                (#xAA62 . #x7319)	;CJK UNIFIED IDEOGRAPH
                                (#xAA63 . #x731A)	;CJK UNIFIED IDEOGRAPH
                                (#xAA64 . #x731F)	;CJK UNIFIED IDEOGRAPH
                                (#xAA65 . #x7320)	;CJK UNIFIED IDEOGRAPH
                                (#xAA66 . #x7323)	;CJK UNIFIED IDEOGRAPH
                                (#xAA67 . #x7324)	;CJK UNIFIED IDEOGRAPH
                                (#xAA68 . #x7326)	;CJK UNIFIED IDEOGRAPH
                                (#xAA69 . #x7327)	;CJK UNIFIED IDEOGRAPH
                                (#xAA6A . #x7328)	;CJK UNIFIED IDEOGRAPH
                                (#xAA6B . #x732D)	;CJK UNIFIED IDEOGRAPH
                                (#xAA6C . #x732F)	;CJK UNIFIED IDEOGRAPH
                                (#xAA6D . #x7330)	;CJK UNIFIED IDEOGRAPH
                                (#xAA6E . #x7332)	;CJK UNIFIED IDEOGRAPH
                                (#xAA6F . #x7333)	;CJK UNIFIED IDEOGRAPH
                                (#xAA70 . #x7335)	;CJK UNIFIED IDEOGRAPH
                                (#xAA71 . #x7336)	;CJK UNIFIED IDEOGRAPH
                                (#xAA72 . #x733A)	;CJK UNIFIED IDEOGRAPH
                                (#xAA73 . #x733B)	;CJK UNIFIED IDEOGRAPH
                                (#xAA74 . #x733C)	;CJK UNIFIED IDEOGRAPH
                                (#xAA75 . #x733D)	;CJK UNIFIED IDEOGRAPH
                                (#xAA76 . #x7340)	;CJK UNIFIED IDEOGRAPH
                                (#xAA77 . #x7341)	;CJK UNIFIED IDEOGRAPH
                                (#xAA78 . #x7342)	;CJK UNIFIED IDEOGRAPH
                                (#xAA79 . #x7343)	;CJK UNIFIED IDEOGRAPH
                                (#xAA7A . #x7344)	;CJK UNIFIED IDEOGRAPH
                                (#xAA7B . #x7345)	;CJK UNIFIED IDEOGRAPH
                                (#xAA7C . #x7346)	;CJK UNIFIED IDEOGRAPH
                                (#xAA7D . #x7347)	;CJK UNIFIED IDEOGRAPH
                                (#xAA7E . #x7348)	;CJK UNIFIED IDEOGRAPH
                                (#xAA80 . #x7349)	;CJK UNIFIED IDEOGRAPH
                                (#xAA81 . #x734A)	;CJK UNIFIED IDEOGRAPH
                                (#xAA82 . #x734B)	;CJK UNIFIED IDEOGRAPH
                                (#xAA83 . #x734C)	;CJK UNIFIED IDEOGRAPH
                                (#xAA84 . #x734E)	;CJK UNIFIED IDEOGRAPH
                                (#xAA85 . #x734F)	;CJK UNIFIED IDEOGRAPH
                                (#xAA86 . #x7351)	;CJK UNIFIED IDEOGRAPH
                                (#xAA87 . #x7353)	;CJK UNIFIED IDEOGRAPH
                                (#xAA88 . #x7354)	;CJK UNIFIED IDEOGRAPH
                                (#xAA89 . #x7355)	;CJK UNIFIED IDEOGRAPH
                                (#xAA8A . #x7356)	;CJK UNIFIED IDEOGRAPH
                                (#xAA8B . #x7358)	;CJK UNIFIED IDEOGRAPH
                                (#xAA8C . #x7359)	;CJK UNIFIED IDEOGRAPH
                                (#xAA8D . #x735A)	;CJK UNIFIED IDEOGRAPH
                                (#xAA8E . #x735B)	;CJK UNIFIED IDEOGRAPH
                                (#xAA8F . #x735C)	;CJK UNIFIED IDEOGRAPH
                                (#xAA90 . #x735D)	;CJK UNIFIED IDEOGRAPH
                                (#xAA91 . #x735E)	;CJK UNIFIED IDEOGRAPH
                                (#xAA92 . #x735F)	;CJK UNIFIED IDEOGRAPH
                                (#xAA93 . #x7361)	;CJK UNIFIED IDEOGRAPH
                                (#xAA94 . #x7362)	;CJK UNIFIED IDEOGRAPH
                                (#xAA95 . #x7363)	;CJK UNIFIED IDEOGRAPH
                                (#xAA96 . #x7364)	;CJK UNIFIED IDEOGRAPH
                                (#xAA97 . #x7365)	;CJK UNIFIED IDEOGRAPH
                                (#xAA98 . #x7366)	;CJK UNIFIED IDEOGRAPH
                                (#xAA99 . #x7367)	;CJK UNIFIED IDEOGRAPH
                                (#xAA9A . #x7368)	;CJK UNIFIED IDEOGRAPH
                                (#xAA9B . #x7369)	;CJK UNIFIED IDEOGRAPH
                                (#xAA9C . #x736A)	;CJK UNIFIED IDEOGRAPH
                                (#xAA9D . #x736B)	;CJK UNIFIED IDEOGRAPH
                                (#xAA9E . #x736E)	;CJK UNIFIED IDEOGRAPH
                                (#xAA9F . #x7370)	;CJK UNIFIED IDEOGRAPH
                                (#xAAA0 . #x7371)	;CJK UNIFIED IDEOGRAPH
                                (#xAB40 . #x7372)	;CJK UNIFIED IDEOGRAPH
                                (#xAB41 . #x7373)	;CJK UNIFIED IDEOGRAPH
                                (#xAB42 . #x7374)	;CJK UNIFIED IDEOGRAPH
                                (#xAB43 . #x7375)	;CJK UNIFIED IDEOGRAPH
                                (#xAB44 . #x7376)	;CJK UNIFIED IDEOGRAPH
                                (#xAB45 . #x7377)	;CJK UNIFIED IDEOGRAPH
                                (#xAB46 . #x7378)	;CJK UNIFIED IDEOGRAPH
                                (#xAB47 . #x7379)	;CJK UNIFIED IDEOGRAPH
                                (#xAB48 . #x737A)	;CJK UNIFIED IDEOGRAPH
                                (#xAB49 . #x737B)	;CJK UNIFIED IDEOGRAPH
                                (#xAB4A . #x737C)	;CJK UNIFIED IDEOGRAPH
                                (#xAB4B . #x737D)	;CJK UNIFIED IDEOGRAPH
                                (#xAB4C . #x737F)	;CJK UNIFIED IDEOGRAPH
                                (#xAB4D . #x7380)	;CJK UNIFIED IDEOGRAPH
                                (#xAB4E . #x7381)	;CJK UNIFIED IDEOGRAPH
                                (#xAB4F . #x7382)	;CJK UNIFIED IDEOGRAPH
                                (#xAB50 . #x7383)	;CJK UNIFIED IDEOGRAPH
                                (#xAB51 . #x7385)	;CJK UNIFIED IDEOGRAPH
                                (#xAB52 . #x7386)	;CJK UNIFIED IDEOGRAPH
                                (#xAB53 . #x7388)	;CJK UNIFIED IDEOGRAPH
                                (#xAB54 . #x738A)	;CJK UNIFIED IDEOGRAPH
                                (#xAB55 . #x738C)	;CJK UNIFIED IDEOGRAPH
                                (#xAB56 . #x738D)	;CJK UNIFIED IDEOGRAPH
                                (#xAB57 . #x738F)	;CJK UNIFIED IDEOGRAPH
                                (#xAB58 . #x7390)	;CJK UNIFIED IDEOGRAPH
                                (#xAB59 . #x7392)	;CJK UNIFIED IDEOGRAPH
                                (#xAB5A . #x7393)	;CJK UNIFIED IDEOGRAPH
                                (#xAB5B . #x7394)	;CJK UNIFIED IDEOGRAPH
                                (#xAB5C . #x7395)	;CJK UNIFIED IDEOGRAPH
                                (#xAB5D . #x7397)	;CJK UNIFIED IDEOGRAPH
                                (#xAB5E . #x7398)	;CJK UNIFIED IDEOGRAPH
                                (#xAB5F . #x7399)	;CJK UNIFIED IDEOGRAPH
                                (#xAB60 . #x739A)	;CJK UNIFIED IDEOGRAPH
                                (#xAB61 . #x739C)	;CJK UNIFIED IDEOGRAPH
                                (#xAB62 . #x739D)	;CJK UNIFIED IDEOGRAPH
                                (#xAB63 . #x739E)	;CJK UNIFIED IDEOGRAPH
                                (#xAB64 . #x73A0)	;CJK UNIFIED IDEOGRAPH
                                (#xAB65 . #x73A1)	;CJK UNIFIED IDEOGRAPH
                                (#xAB66 . #x73A3)	;CJK UNIFIED IDEOGRAPH
                                (#xAB67 . #x73A4)	;CJK UNIFIED IDEOGRAPH
                                (#xAB68 . #x73A5)	;CJK UNIFIED IDEOGRAPH
                                (#xAB69 . #x73A6)	;CJK UNIFIED IDEOGRAPH
                                (#xAB6A . #x73A7)	;CJK UNIFIED IDEOGRAPH
                                (#xAB6B . #x73A8)	;CJK UNIFIED IDEOGRAPH
                                (#xAB6C . #x73AA)	;CJK UNIFIED IDEOGRAPH
                                (#xAB6D . #x73AC)	;CJK UNIFIED IDEOGRAPH
                                (#xAB6E . #x73AD)	;CJK UNIFIED IDEOGRAPH
                                (#xAB6F . #x73B1)	;CJK UNIFIED IDEOGRAPH
                                (#xAB70 . #x73B4)	;CJK UNIFIED IDEOGRAPH
                                (#xAB71 . #x73B5)	;CJK UNIFIED IDEOGRAPH
                                (#xAB72 . #x73B6)	;CJK UNIFIED IDEOGRAPH
                                (#xAB73 . #x73B8)	;CJK UNIFIED IDEOGRAPH
                                (#xAB74 . #x73B9)	;CJK UNIFIED IDEOGRAPH
                                (#xAB75 . #x73BC)	;CJK UNIFIED IDEOGRAPH
                                (#xAB76 . #x73BD)	;CJK UNIFIED IDEOGRAPH
                                (#xAB77 . #x73BE)	;CJK UNIFIED IDEOGRAPH
                                (#xAB78 . #x73BF)	;CJK UNIFIED IDEOGRAPH
                                (#xAB79 . #x73C1)	;CJK UNIFIED IDEOGRAPH
                                (#xAB7A . #x73C3)	;CJK UNIFIED IDEOGRAPH
                                (#xAB7B . #x73C4)	;CJK UNIFIED IDEOGRAPH
                                (#xAB7C . #x73C5)	;CJK UNIFIED IDEOGRAPH
                                (#xAB7D . #x73C6)	;CJK UNIFIED IDEOGRAPH
                                (#xAB7E . #x73C7)	;CJK UNIFIED IDEOGRAPH
                                (#xAB80 . #x73CB)	;CJK UNIFIED IDEOGRAPH
                                (#xAB81 . #x73CC)	;CJK UNIFIED IDEOGRAPH
                                (#xAB82 . #x73CE)	;CJK UNIFIED IDEOGRAPH
                                (#xAB83 . #x73D2)	;CJK UNIFIED IDEOGRAPH
                                (#xAB84 . #x73D3)	;CJK UNIFIED IDEOGRAPH
                                (#xAB85 . #x73D4)	;CJK UNIFIED IDEOGRAPH
                                (#xAB86 . #x73D5)	;CJK UNIFIED IDEOGRAPH
                                (#xAB87 . #x73D6)	;CJK UNIFIED IDEOGRAPH
                                (#xAB88 . #x73D7)	;CJK UNIFIED IDEOGRAPH
                                (#xAB89 . #x73D8)	;CJK UNIFIED IDEOGRAPH
                                (#xAB8A . #x73DA)	;CJK UNIFIED IDEOGRAPH
                                (#xAB8B . #x73DB)	;CJK UNIFIED IDEOGRAPH
                                (#xAB8C . #x73DC)	;CJK UNIFIED IDEOGRAPH
                                (#xAB8D . #x73DD)	;CJK UNIFIED IDEOGRAPH
                                (#xAB8E . #x73DF)	;CJK UNIFIED IDEOGRAPH
                                (#xAB8F . #x73E1)	;CJK UNIFIED IDEOGRAPH
                                (#xAB90 . #x73E2)	;CJK UNIFIED IDEOGRAPH
                                (#xAB91 . #x73E3)	;CJK UNIFIED IDEOGRAPH
                                (#xAB92 . #x73E4)	;CJK UNIFIED IDEOGRAPH
                                (#xAB93 . #x73E6)	;CJK UNIFIED IDEOGRAPH
                                (#xAB94 . #x73E8)	;CJK UNIFIED IDEOGRAPH
                                (#xAB95 . #x73EA)	;CJK UNIFIED IDEOGRAPH
                                (#xAB96 . #x73EB)	;CJK UNIFIED IDEOGRAPH
                                (#xAB97 . #x73EC)	;CJK UNIFIED IDEOGRAPH
                                (#xAB98 . #x73EE)	;CJK UNIFIED IDEOGRAPH
                                (#xAB99 . #x73EF)	;CJK UNIFIED IDEOGRAPH
                                (#xAB9A . #x73F0)	;CJK UNIFIED IDEOGRAPH
                                (#xAB9B . #x73F1)	;CJK UNIFIED IDEOGRAPH
                                (#xAB9C . #x73F3)	;CJK UNIFIED IDEOGRAPH
                                (#xAB9D . #x73F4)	;CJK UNIFIED IDEOGRAPH
                                (#xAB9E . #x73F5)	;CJK UNIFIED IDEOGRAPH
                                (#xAB9F . #x73F6)	;CJK UNIFIED IDEOGRAPH
                                (#xABA0 . #x73F7)	;CJK UNIFIED IDEOGRAPH
                                (#xAC40 . #x73F8)	;CJK UNIFIED IDEOGRAPH
                                (#xAC41 . #x73F9)	;CJK UNIFIED IDEOGRAPH
                                (#xAC42 . #x73FA)	;CJK UNIFIED IDEOGRAPH
                                (#xAC43 . #x73FB)	;CJK UNIFIED IDEOGRAPH
                                (#xAC44 . #x73FC)	;CJK UNIFIED IDEOGRAPH
                                (#xAC45 . #x73FD)	;CJK UNIFIED IDEOGRAPH
                                (#xAC46 . #x73FE)	;CJK UNIFIED IDEOGRAPH
                                (#xAC47 . #x73FF)	;CJK UNIFIED IDEOGRAPH
                                (#xAC48 . #x7400)	;CJK UNIFIED IDEOGRAPH
                                (#xAC49 . #x7401)	;CJK UNIFIED IDEOGRAPH
                                (#xAC4A . #x7402)	;CJK UNIFIED IDEOGRAPH
                                (#xAC4B . #x7404)	;CJK UNIFIED IDEOGRAPH
                                (#xAC4C . #x7407)	;CJK UNIFIED IDEOGRAPH
                                (#xAC4D . #x7408)	;CJK UNIFIED IDEOGRAPH
                                (#xAC4E . #x740B)	;CJK UNIFIED IDEOGRAPH
                                (#xAC4F . #x740C)	;CJK UNIFIED IDEOGRAPH
                                (#xAC50 . #x740D)	;CJK UNIFIED IDEOGRAPH
                                (#xAC51 . #x740E)	;CJK UNIFIED IDEOGRAPH
                                (#xAC52 . #x7411)	;CJK UNIFIED IDEOGRAPH
                                (#xAC53 . #x7412)	;CJK UNIFIED IDEOGRAPH
                                (#xAC54 . #x7413)	;CJK UNIFIED IDEOGRAPH
                                (#xAC55 . #x7414)	;CJK UNIFIED IDEOGRAPH
                                (#xAC56 . #x7415)	;CJK UNIFIED IDEOGRAPH
                                (#xAC57 . #x7416)	;CJK UNIFIED IDEOGRAPH
                                (#xAC58 . #x7417)	;CJK UNIFIED IDEOGRAPH
                                (#xAC59 . #x7418)	;CJK UNIFIED IDEOGRAPH
                                (#xAC5A . #x7419)	;CJK UNIFIED IDEOGRAPH
                                (#xAC5B . #x741C)	;CJK UNIFIED IDEOGRAPH
                                (#xAC5C . #x741D)	;CJK UNIFIED IDEOGRAPH
                                (#xAC5D . #x741E)	;CJK UNIFIED IDEOGRAPH
                                (#xAC5E . #x741F)	;CJK UNIFIED IDEOGRAPH
                                (#xAC5F . #x7420)	;CJK UNIFIED IDEOGRAPH
                                (#xAC60 . #x7421)	;CJK UNIFIED IDEOGRAPH
                                (#xAC61 . #x7423)	;CJK UNIFIED IDEOGRAPH
                                (#xAC62 . #x7424)	;CJK UNIFIED IDEOGRAPH
                                (#xAC63 . #x7427)	;CJK UNIFIED IDEOGRAPH
                                (#xAC64 . #x7429)	;CJK UNIFIED IDEOGRAPH
                                (#xAC65 . #x742B)	;CJK UNIFIED IDEOGRAPH
                                (#xAC66 . #x742D)	;CJK UNIFIED IDEOGRAPH
                                (#xAC67 . #x742F)	;CJK UNIFIED IDEOGRAPH
                                (#xAC68 . #x7431)	;CJK UNIFIED IDEOGRAPH
                                (#xAC69 . #x7432)	;CJK UNIFIED IDEOGRAPH
                                (#xAC6A . #x7437)	;CJK UNIFIED IDEOGRAPH
                                (#xAC6B . #x7438)	;CJK UNIFIED IDEOGRAPH
                                (#xAC6C . #x7439)	;CJK UNIFIED IDEOGRAPH
                                (#xAC6D . #x743A)	;CJK UNIFIED IDEOGRAPH
                                (#xAC6E . #x743B)	;CJK UNIFIED IDEOGRAPH
                                (#xAC6F . #x743D)	;CJK UNIFIED IDEOGRAPH
                                (#xAC70 . #x743E)	;CJK UNIFIED IDEOGRAPH
                                (#xAC71 . #x743F)	;CJK UNIFIED IDEOGRAPH
                                (#xAC72 . #x7440)	;CJK UNIFIED IDEOGRAPH
                                (#xAC73 . #x7442)	;CJK UNIFIED IDEOGRAPH
                                (#xAC74 . #x7443)	;CJK UNIFIED IDEOGRAPH
                                (#xAC75 . #x7444)	;CJK UNIFIED IDEOGRAPH
                                (#xAC76 . #x7445)	;CJK UNIFIED IDEOGRAPH
                                (#xAC77 . #x7446)	;CJK UNIFIED IDEOGRAPH
                                (#xAC78 . #x7447)	;CJK UNIFIED IDEOGRAPH
                                (#xAC79 . #x7448)	;CJK UNIFIED IDEOGRAPH
                                (#xAC7A . #x7449)	;CJK UNIFIED IDEOGRAPH
                                (#xAC7B . #x744A)	;CJK UNIFIED IDEOGRAPH
                                (#xAC7C . #x744B)	;CJK UNIFIED IDEOGRAPH
                                (#xAC7D . #x744C)	;CJK UNIFIED IDEOGRAPH
                                (#xAC7E . #x744D)	;CJK UNIFIED IDEOGRAPH
                                (#xAC80 . #x744E)	;CJK UNIFIED IDEOGRAPH
                                (#xAC81 . #x744F)	;CJK UNIFIED IDEOGRAPH
                                (#xAC82 . #x7450)	;CJK UNIFIED IDEOGRAPH
                                (#xAC83 . #x7451)	;CJK UNIFIED IDEOGRAPH
                                (#xAC84 . #x7452)	;CJK UNIFIED IDEOGRAPH
                                (#xAC85 . #x7453)	;CJK UNIFIED IDEOGRAPH
                                (#xAC86 . #x7454)	;CJK UNIFIED IDEOGRAPH
                                (#xAC87 . #x7456)	;CJK UNIFIED IDEOGRAPH
                                (#xAC88 . #x7458)	;CJK UNIFIED IDEOGRAPH
                                (#xAC89 . #x745D)	;CJK UNIFIED IDEOGRAPH
                                (#xAC8A . #x7460)	;CJK UNIFIED IDEOGRAPH
                                (#xAC8B . #x7461)	;CJK UNIFIED IDEOGRAPH
                                (#xAC8C . #x7462)	;CJK UNIFIED IDEOGRAPH
                                (#xAC8D . #x7463)	;CJK UNIFIED IDEOGRAPH
                                (#xAC8E . #x7464)	;CJK UNIFIED IDEOGRAPH
                                (#xAC8F . #x7465)	;CJK UNIFIED IDEOGRAPH
                                (#xAC90 . #x7466)	;CJK UNIFIED IDEOGRAPH
                                (#xAC91 . #x7467)	;CJK UNIFIED IDEOGRAPH
                                (#xAC92 . #x7468)	;CJK UNIFIED IDEOGRAPH
                                (#xAC93 . #x7469)	;CJK UNIFIED IDEOGRAPH
                                (#xAC94 . #x746A)	;CJK UNIFIED IDEOGRAPH
                                (#xAC95 . #x746B)	;CJK UNIFIED IDEOGRAPH
                                (#xAC96 . #x746C)	;CJK UNIFIED IDEOGRAPH
                                (#xAC97 . #x746E)	;CJK UNIFIED IDEOGRAPH
                                (#xAC98 . #x746F)	;CJK UNIFIED IDEOGRAPH
                                (#xAC99 . #x7471)	;CJK UNIFIED IDEOGRAPH
                                (#xAC9A . #x7472)	;CJK UNIFIED IDEOGRAPH
                                (#xAC9B . #x7473)	;CJK UNIFIED IDEOGRAPH
                                (#xAC9C . #x7474)	;CJK UNIFIED IDEOGRAPH
                                (#xAC9D . #x7475)	;CJK UNIFIED IDEOGRAPH
                                (#xAC9E . #x7478)	;CJK UNIFIED IDEOGRAPH
                                (#xAC9F . #x7479)	;CJK UNIFIED IDEOGRAPH
                                (#xACA0 . #x747A)	;CJK UNIFIED IDEOGRAPH
                                (#xAD40 . #x747B)	;CJK UNIFIED IDEOGRAPH
                                (#xAD41 . #x747C)	;CJK UNIFIED IDEOGRAPH
                                (#xAD42 . #x747D)	;CJK UNIFIED IDEOGRAPH
                                (#xAD43 . #x747F)	;CJK UNIFIED IDEOGRAPH
                                (#xAD44 . #x7482)	;CJK UNIFIED IDEOGRAPH
                                (#xAD45 . #x7484)	;CJK UNIFIED IDEOGRAPH
                                (#xAD46 . #x7485)	;CJK UNIFIED IDEOGRAPH
                                (#xAD47 . #x7486)	;CJK UNIFIED IDEOGRAPH
                                (#xAD48 . #x7488)	;CJK UNIFIED IDEOGRAPH
                                (#xAD49 . #x7489)	;CJK UNIFIED IDEOGRAPH
                                (#xAD4A . #x748A)	;CJK UNIFIED IDEOGRAPH
                                (#xAD4B . #x748C)	;CJK UNIFIED IDEOGRAPH
                                (#xAD4C . #x748D)	;CJK UNIFIED IDEOGRAPH
                                (#xAD4D . #x748F)	;CJK UNIFIED IDEOGRAPH
                                (#xAD4E . #x7491)	;CJK UNIFIED IDEOGRAPH
                                (#xAD4F . #x7492)	;CJK UNIFIED IDEOGRAPH
                                (#xAD50 . #x7493)	;CJK UNIFIED IDEOGRAPH
                                (#xAD51 . #x7494)	;CJK UNIFIED IDEOGRAPH
                                (#xAD52 . #x7495)	;CJK UNIFIED IDEOGRAPH
                                (#xAD53 . #x7496)	;CJK UNIFIED IDEOGRAPH
                                (#xAD54 . #x7497)	;CJK UNIFIED IDEOGRAPH
                                (#xAD55 . #x7498)	;CJK UNIFIED IDEOGRAPH
                                (#xAD56 . #x7499)	;CJK UNIFIED IDEOGRAPH
                                (#xAD57 . #x749A)	;CJK UNIFIED IDEOGRAPH
                                (#xAD58 . #x749B)	;CJK UNIFIED IDEOGRAPH
                                (#xAD59 . #x749D)	;CJK UNIFIED IDEOGRAPH
                                (#xAD5A . #x749F)	;CJK UNIFIED IDEOGRAPH
                                (#xAD5B . #x74A0)	;CJK UNIFIED IDEOGRAPH
                                (#xAD5C . #x74A1)	;CJK UNIFIED IDEOGRAPH
                                (#xAD5D . #x74A2)	;CJK UNIFIED IDEOGRAPH
                                (#xAD5E . #x74A3)	;CJK UNIFIED IDEOGRAPH
                                (#xAD5F . #x74A4)	;CJK UNIFIED IDEOGRAPH
                                (#xAD60 . #x74A5)	;CJK UNIFIED IDEOGRAPH
                                (#xAD61 . #x74A6)	;CJK UNIFIED IDEOGRAPH
                                (#xAD62 . #x74AA)	;CJK UNIFIED IDEOGRAPH
                                (#xAD63 . #x74AB)	;CJK UNIFIED IDEOGRAPH
                                (#xAD64 . #x74AC)	;CJK UNIFIED IDEOGRAPH
                                (#xAD65 . #x74AD)	;CJK UNIFIED IDEOGRAPH
                                (#xAD66 . #x74AE)	;CJK UNIFIED IDEOGRAPH
                                (#xAD67 . #x74AF)	;CJK UNIFIED IDEOGRAPH
                                (#xAD68 . #x74B0)	;CJK UNIFIED IDEOGRAPH
                                (#xAD69 . #x74B1)	;CJK UNIFIED IDEOGRAPH
                                (#xAD6A . #x74B2)	;CJK UNIFIED IDEOGRAPH
                                (#xAD6B . #x74B3)	;CJK UNIFIED IDEOGRAPH
                                (#xAD6C . #x74B4)	;CJK UNIFIED IDEOGRAPH
                                (#xAD6D . #x74B5)	;CJK UNIFIED IDEOGRAPH
                                (#xAD6E . #x74B6)	;CJK UNIFIED IDEOGRAPH
                                (#xAD6F . #x74B7)	;CJK UNIFIED IDEOGRAPH
                                (#xAD70 . #x74B8)	;CJK UNIFIED IDEOGRAPH
                                (#xAD71 . #x74B9)	;CJK UNIFIED IDEOGRAPH
                                (#xAD72 . #x74BB)	;CJK UNIFIED IDEOGRAPH
                                (#xAD73 . #x74BC)	;CJK UNIFIED IDEOGRAPH
                                (#xAD74 . #x74BD)	;CJK UNIFIED IDEOGRAPH
                                (#xAD75 . #x74BE)	;CJK UNIFIED IDEOGRAPH
                                (#xAD76 . #x74BF)	;CJK UNIFIED IDEOGRAPH
                                (#xAD77 . #x74C0)	;CJK UNIFIED IDEOGRAPH
                                (#xAD78 . #x74C1)	;CJK UNIFIED IDEOGRAPH
                                (#xAD79 . #x74C2)	;CJK UNIFIED IDEOGRAPH
                                (#xAD7A . #x74C3)	;CJK UNIFIED IDEOGRAPH
                                (#xAD7B . #x74C4)	;CJK UNIFIED IDEOGRAPH
                                (#xAD7C . #x74C5)	;CJK UNIFIED IDEOGRAPH
                                (#xAD7D . #x74C6)	;CJK UNIFIED IDEOGRAPH
                                (#xAD7E . #x74C7)	;CJK UNIFIED IDEOGRAPH
                                (#xAD80 . #x74C8)	;CJK UNIFIED IDEOGRAPH
                                (#xAD81 . #x74C9)	;CJK UNIFIED IDEOGRAPH
                                (#xAD82 . #x74CA)	;CJK UNIFIED IDEOGRAPH
                                (#xAD83 . #x74CB)	;CJK UNIFIED IDEOGRAPH
                                (#xAD84 . #x74CC)	;CJK UNIFIED IDEOGRAPH
                                (#xAD85 . #x74CD)	;CJK UNIFIED IDEOGRAPH
                                (#xAD86 . #x74CE)	;CJK UNIFIED IDEOGRAPH
                                (#xAD87 . #x74CF)	;CJK UNIFIED IDEOGRAPH
                                (#xAD88 . #x74D0)	;CJK UNIFIED IDEOGRAPH
                                (#xAD89 . #x74D1)	;CJK UNIFIED IDEOGRAPH
                                (#xAD8A . #x74D3)	;CJK UNIFIED IDEOGRAPH
                                (#xAD8B . #x74D4)	;CJK UNIFIED IDEOGRAPH
                                (#xAD8C . #x74D5)	;CJK UNIFIED IDEOGRAPH
                                (#xAD8D . #x74D6)	;CJK UNIFIED IDEOGRAPH
                                (#xAD8E . #x74D7)	;CJK UNIFIED IDEOGRAPH
                                (#xAD8F . #x74D8)	;CJK UNIFIED IDEOGRAPH
                                (#xAD90 . #x74D9)	;CJK UNIFIED IDEOGRAPH
                                (#xAD91 . #x74DA)	;CJK UNIFIED IDEOGRAPH
                                (#xAD92 . #x74DB)	;CJK UNIFIED IDEOGRAPH
                                (#xAD93 . #x74DD)	;CJK UNIFIED IDEOGRAPH
                                (#xAD94 . #x74DF)	;CJK UNIFIED IDEOGRAPH
                                (#xAD95 . #x74E1)	;CJK UNIFIED IDEOGRAPH
                                (#xAD96 . #x74E5)	;CJK UNIFIED IDEOGRAPH
                                (#xAD97 . #x74E7)	;CJK UNIFIED IDEOGRAPH
                                (#xAD98 . #x74E8)	;CJK UNIFIED IDEOGRAPH
                                (#xAD99 . #x74E9)	;CJK UNIFIED IDEOGRAPH
                                (#xAD9A . #x74EA)	;CJK UNIFIED IDEOGRAPH
                                (#xAD9B . #x74EB)	;CJK UNIFIED IDEOGRAPH
                                (#xAD9C . #x74EC)	;CJK UNIFIED IDEOGRAPH
                                (#xAD9D . #x74ED)	;CJK UNIFIED IDEOGRAPH
                                (#xAD9E . #x74F0)	;CJK UNIFIED IDEOGRAPH
                                (#xAD9F . #x74F1)	;CJK UNIFIED IDEOGRAPH
                                (#xADA0 . #x74F2)	;CJK UNIFIED IDEOGRAPH
                                (#xAE40 . #x74F3)	;CJK UNIFIED IDEOGRAPH
                                (#xAE41 . #x74F5)	;CJK UNIFIED IDEOGRAPH
                                (#xAE42 . #x74F8)	;CJK UNIFIED IDEOGRAPH
                                (#xAE43 . #x74F9)	;CJK UNIFIED IDEOGRAPH
                                (#xAE44 . #x74FA)	;CJK UNIFIED IDEOGRAPH
                                (#xAE45 . #x74FB)	;CJK UNIFIED IDEOGRAPH
                                (#xAE46 . #x74FC)	;CJK UNIFIED IDEOGRAPH
                                (#xAE47 . #x74FD)	;CJK UNIFIED IDEOGRAPH
                                (#xAE48 . #x74FE)	;CJK UNIFIED IDEOGRAPH
                                (#xAE49 . #x7500)	;CJK UNIFIED IDEOGRAPH
                                (#xAE4A . #x7501)	;CJK UNIFIED IDEOGRAPH
                                (#xAE4B . #x7502)	;CJK UNIFIED IDEOGRAPH
                                (#xAE4C . #x7503)	;CJK UNIFIED IDEOGRAPH
                                (#xAE4D . #x7505)	;CJK UNIFIED IDEOGRAPH
                                (#xAE4E . #x7506)	;CJK UNIFIED IDEOGRAPH
                                (#xAE4F . #x7507)	;CJK UNIFIED IDEOGRAPH
                                (#xAE50 . #x7508)	;CJK UNIFIED IDEOGRAPH
                                (#xAE51 . #x7509)	;CJK UNIFIED IDEOGRAPH
                                (#xAE52 . #x750A)	;CJK UNIFIED IDEOGRAPH
                                (#xAE53 . #x750B)	;CJK UNIFIED IDEOGRAPH
                                (#xAE54 . #x750C)	;CJK UNIFIED IDEOGRAPH
                                (#xAE55 . #x750E)	;CJK UNIFIED IDEOGRAPH
                                (#xAE56 . #x7510)	;CJK UNIFIED IDEOGRAPH
                                (#xAE57 . #x7512)	;CJK UNIFIED IDEOGRAPH
                                (#xAE58 . #x7514)	;CJK UNIFIED IDEOGRAPH
                                (#xAE59 . #x7515)	;CJK UNIFIED IDEOGRAPH
                                (#xAE5A . #x7516)	;CJK UNIFIED IDEOGRAPH
                                (#xAE5B . #x7517)	;CJK UNIFIED IDEOGRAPH
                                (#xAE5C . #x751B)	;CJK UNIFIED IDEOGRAPH
                                (#xAE5D . #x751D)	;CJK UNIFIED IDEOGRAPH
                                (#xAE5E . #x751E)	;CJK UNIFIED IDEOGRAPH
                                (#xAE5F . #x7520)	;CJK UNIFIED IDEOGRAPH
                                (#xAE60 . #x7521)	;CJK UNIFIED IDEOGRAPH
                                (#xAE61 . #x7522)	;CJK UNIFIED IDEOGRAPH
                                (#xAE62 . #x7523)	;CJK UNIFIED IDEOGRAPH
                                (#xAE63 . #x7524)	;CJK UNIFIED IDEOGRAPH
                                (#xAE64 . #x7526)	;CJK UNIFIED IDEOGRAPH
                                (#xAE65 . #x7527)	;CJK UNIFIED IDEOGRAPH
                                (#xAE66 . #x752A)	;CJK UNIFIED IDEOGRAPH
                                (#xAE67 . #x752E)	;CJK UNIFIED IDEOGRAPH
                                (#xAE68 . #x7534)	;CJK UNIFIED IDEOGRAPH
                                (#xAE69 . #x7536)	;CJK UNIFIED IDEOGRAPH
                                (#xAE6A . #x7539)	;CJK UNIFIED IDEOGRAPH
                                (#xAE6B . #x753C)	;CJK UNIFIED IDEOGRAPH
                                (#xAE6C . #x753D)	;CJK UNIFIED IDEOGRAPH
                                (#xAE6D . #x753F)	;CJK UNIFIED IDEOGRAPH
                                (#xAE6E . #x7541)	;CJK UNIFIED IDEOGRAPH
                                (#xAE6F . #x7542)	;CJK UNIFIED IDEOGRAPH
                                (#xAE70 . #x7543)	;CJK UNIFIED IDEOGRAPH
                                (#xAE71 . #x7544)	;CJK UNIFIED IDEOGRAPH
                                (#xAE72 . #x7546)	;CJK UNIFIED IDEOGRAPH
                                (#xAE73 . #x7547)	;CJK UNIFIED IDEOGRAPH
                                (#xAE74 . #x7549)	;CJK UNIFIED IDEOGRAPH
                                (#xAE75 . #x754A)	;CJK UNIFIED IDEOGRAPH
                                (#xAE76 . #x754D)	;CJK UNIFIED IDEOGRAPH
                                (#xAE77 . #x7550)	;CJK UNIFIED IDEOGRAPH
                                (#xAE78 . #x7551)	;CJK UNIFIED IDEOGRAPH
                                (#xAE79 . #x7552)	;CJK UNIFIED IDEOGRAPH
                                (#xAE7A . #x7553)	;CJK UNIFIED IDEOGRAPH
                                (#xAE7B . #x7555)	;CJK UNIFIED IDEOGRAPH
                                (#xAE7C . #x7556)	;CJK UNIFIED IDEOGRAPH
                                (#xAE7D . #x7557)	;CJK UNIFIED IDEOGRAPH
                                (#xAE7E . #x7558)	;CJK UNIFIED IDEOGRAPH
                                (#xAE80 . #x755D)	;CJK UNIFIED IDEOGRAPH
                                (#xAE81 . #x755E)	;CJK UNIFIED IDEOGRAPH
                                (#xAE82 . #x755F)	;CJK UNIFIED IDEOGRAPH
                                (#xAE83 . #x7560)	;CJK UNIFIED IDEOGRAPH
                                (#xAE84 . #x7561)	;CJK UNIFIED IDEOGRAPH
                                (#xAE85 . #x7562)	;CJK UNIFIED IDEOGRAPH
                                (#xAE86 . #x7563)	;CJK UNIFIED IDEOGRAPH
                                (#xAE87 . #x7564)	;CJK UNIFIED IDEOGRAPH
                                (#xAE88 . #x7567)	;CJK UNIFIED IDEOGRAPH
                                (#xAE89 . #x7568)	;CJK UNIFIED IDEOGRAPH
                                (#xAE8A . #x7569)	;CJK UNIFIED IDEOGRAPH
                                (#xAE8B . #x756B)	;CJK UNIFIED IDEOGRAPH
                                (#xAE8C . #x756C)	;CJK UNIFIED IDEOGRAPH
                                (#xAE8D . #x756D)	;CJK UNIFIED IDEOGRAPH
                                (#xAE8E . #x756E)	;CJK UNIFIED IDEOGRAPH
                                (#xAE8F . #x756F)	;CJK UNIFIED IDEOGRAPH
                                (#xAE90 . #x7570)	;CJK UNIFIED IDEOGRAPH
                                (#xAE91 . #x7571)	;CJK UNIFIED IDEOGRAPH
                                (#xAE92 . #x7573)	;CJK UNIFIED IDEOGRAPH
                                (#xAE93 . #x7575)	;CJK UNIFIED IDEOGRAPH
                                (#xAE94 . #x7576)	;CJK UNIFIED IDEOGRAPH
                                (#xAE95 . #x7577)	;CJK UNIFIED IDEOGRAPH
                                (#xAE96 . #x757A)	;CJK UNIFIED IDEOGRAPH
                                (#xAE97 . #x757B)	;CJK UNIFIED IDEOGRAPH
                                (#xAE98 . #x757C)	;CJK UNIFIED IDEOGRAPH
                                (#xAE99 . #x757D)	;CJK UNIFIED IDEOGRAPH
                                (#xAE9A . #x757E)	;CJK UNIFIED IDEOGRAPH
                                (#xAE9B . #x7580)	;CJK UNIFIED IDEOGRAPH
                                (#xAE9C . #x7581)	;CJK UNIFIED IDEOGRAPH
                                (#xAE9D . #x7582)	;CJK UNIFIED IDEOGRAPH
                                (#xAE9E . #x7584)	;CJK UNIFIED IDEOGRAPH
                                (#xAE9F . #x7585)	;CJK UNIFIED IDEOGRAPH
                                (#xAEA0 . #x7587)	;CJK UNIFIED IDEOGRAPH
                                (#xAF40 . #x7588)	;CJK UNIFIED IDEOGRAPH
                                (#xAF41 . #x7589)	;CJK UNIFIED IDEOGRAPH
                                (#xAF42 . #x758A)	;CJK UNIFIED IDEOGRAPH
                                (#xAF43 . #x758C)	;CJK UNIFIED IDEOGRAPH
                                (#xAF44 . #x758D)	;CJK UNIFIED IDEOGRAPH
                                (#xAF45 . #x758E)	;CJK UNIFIED IDEOGRAPH
                                (#xAF46 . #x7590)	;CJK UNIFIED IDEOGRAPH
                                (#xAF47 . #x7593)	;CJK UNIFIED IDEOGRAPH
                                (#xAF48 . #x7595)	;CJK UNIFIED IDEOGRAPH
                                (#xAF49 . #x7598)	;CJK UNIFIED IDEOGRAPH
                                (#xAF4A . #x759B)	;CJK UNIFIED IDEOGRAPH
                                (#xAF4B . #x759C)	;CJK UNIFIED IDEOGRAPH
                                (#xAF4C . #x759E)	;CJK UNIFIED IDEOGRAPH
                                (#xAF4D . #x75A2)	;CJK UNIFIED IDEOGRAPH
                                (#xAF4E . #x75A6)	;CJK UNIFIED IDEOGRAPH
                                (#xAF4F . #x75A7)	;CJK UNIFIED IDEOGRAPH
                                (#xAF50 . #x75A8)	;CJK UNIFIED IDEOGRAPH
                                (#xAF51 . #x75A9)	;CJK UNIFIED IDEOGRAPH
                                (#xAF52 . #x75AA)	;CJK UNIFIED IDEOGRAPH
                                (#xAF53 . #x75AD)	;CJK UNIFIED IDEOGRAPH
                                (#xAF54 . #x75B6)	;CJK UNIFIED IDEOGRAPH
                                (#xAF55 . #x75B7)	;CJK UNIFIED IDEOGRAPH
                                (#xAF56 . #x75BA)	;CJK UNIFIED IDEOGRAPH
                                (#xAF57 . #x75BB)	;CJK UNIFIED IDEOGRAPH
                                (#xAF58 . #x75BF)	;CJK UNIFIED IDEOGRAPH
                                (#xAF59 . #x75C0)	;CJK UNIFIED IDEOGRAPH
                                (#xAF5A . #x75C1)	;CJK UNIFIED IDEOGRAPH
                                (#xAF5B . #x75C6)	;CJK UNIFIED IDEOGRAPH
                                (#xAF5C . #x75CB)	;CJK UNIFIED IDEOGRAPH
                                (#xAF5D . #x75CC)	;CJK UNIFIED IDEOGRAPH
                                (#xAF5E . #x75CE)	;CJK UNIFIED IDEOGRAPH
                                (#xAF5F . #x75CF)	;CJK UNIFIED IDEOGRAPH
                                (#xAF60 . #x75D0)	;CJK UNIFIED IDEOGRAPH
                                (#xAF61 . #x75D1)	;CJK UNIFIED IDEOGRAPH
                                (#xAF62 . #x75D3)	;CJK UNIFIED IDEOGRAPH
                                (#xAF63 . #x75D7)	;CJK UNIFIED IDEOGRAPH
                                (#xAF64 . #x75D9)	;CJK UNIFIED IDEOGRAPH
                                (#xAF65 . #x75DA)	;CJK UNIFIED IDEOGRAPH
                                (#xAF66 . #x75DC)	;CJK UNIFIED IDEOGRAPH
                                (#xAF67 . #x75DD)	;CJK UNIFIED IDEOGRAPH
                                (#xAF68 . #x75DF)	;CJK UNIFIED IDEOGRAPH
                                (#xAF69 . #x75E0)	;CJK UNIFIED IDEOGRAPH
                                (#xAF6A . #x75E1)	;CJK UNIFIED IDEOGRAPH
                                (#xAF6B . #x75E5)	;CJK UNIFIED IDEOGRAPH
                                (#xAF6C . #x75E9)	;CJK UNIFIED IDEOGRAPH
                                (#xAF6D . #x75EC)	;CJK UNIFIED IDEOGRAPH
                                (#xAF6E . #x75ED)	;CJK UNIFIED IDEOGRAPH
                                (#xAF6F . #x75EE)	;CJK UNIFIED IDEOGRAPH
                                (#xAF70 . #x75EF)	;CJK UNIFIED IDEOGRAPH
                                (#xAF71 . #x75F2)	;CJK UNIFIED IDEOGRAPH
                                (#xAF72 . #x75F3)	;CJK UNIFIED IDEOGRAPH
                                (#xAF73 . #x75F5)	;CJK UNIFIED IDEOGRAPH
                                (#xAF74 . #x75F6)	;CJK UNIFIED IDEOGRAPH
                                (#xAF75 . #x75F7)	;CJK UNIFIED IDEOGRAPH
                                (#xAF76 . #x75F8)	;CJK UNIFIED IDEOGRAPH
                                (#xAF77 . #x75FA)	;CJK UNIFIED IDEOGRAPH
                                (#xAF78 . #x75FB)	;CJK UNIFIED IDEOGRAPH
                                (#xAF79 . #x75FD)	;CJK UNIFIED IDEOGRAPH
                                (#xAF7A . #x75FE)	;CJK UNIFIED IDEOGRAPH
                                (#xAF7B . #x7602)	;CJK UNIFIED IDEOGRAPH
                                (#xAF7C . #x7604)	;CJK UNIFIED IDEOGRAPH
                                (#xAF7D . #x7606)	;CJK UNIFIED IDEOGRAPH
                                (#xAF7E . #x7607)	;CJK UNIFIED IDEOGRAPH
                                (#xAF80 . #x7608)	;CJK UNIFIED IDEOGRAPH
                                (#xAF81 . #x7609)	;CJK UNIFIED IDEOGRAPH
                                (#xAF82 . #x760B)	;CJK UNIFIED IDEOGRAPH
                                (#xAF83 . #x760D)	;CJK UNIFIED IDEOGRAPH
                                (#xAF84 . #x760E)	;CJK UNIFIED IDEOGRAPH
                                (#xAF85 . #x760F)	;CJK UNIFIED IDEOGRAPH
                                (#xAF86 . #x7611)	;CJK UNIFIED IDEOGRAPH
                                (#xAF87 . #x7612)	;CJK UNIFIED IDEOGRAPH
                                (#xAF88 . #x7613)	;CJK UNIFIED IDEOGRAPH
                                (#xAF89 . #x7614)	;CJK UNIFIED IDEOGRAPH
                                (#xAF8A . #x7616)	;CJK UNIFIED IDEOGRAPH
                                (#xAF8B . #x761A)	;CJK UNIFIED IDEOGRAPH
                                (#xAF8C . #x761C)	;CJK UNIFIED IDEOGRAPH
                                (#xAF8D . #x761D)	;CJK UNIFIED IDEOGRAPH
                                (#xAF8E . #x761E)	;CJK UNIFIED IDEOGRAPH
                                (#xAF8F . #x7621)	;CJK UNIFIED IDEOGRAPH
                                (#xAF90 . #x7623)	;CJK UNIFIED IDEOGRAPH
                                (#xAF91 . #x7627)	;CJK UNIFIED IDEOGRAPH
                                (#xAF92 . #x7628)	;CJK UNIFIED IDEOGRAPH
                                (#xAF93 . #x762C)	;CJK UNIFIED IDEOGRAPH
                                (#xAF94 . #x762E)	;CJK UNIFIED IDEOGRAPH
                                (#xAF95 . #x762F)	;CJK UNIFIED IDEOGRAPH
                                (#xAF96 . #x7631)	;CJK UNIFIED IDEOGRAPH
                                (#xAF97 . #x7632)	;CJK UNIFIED IDEOGRAPH
                                (#xAF98 . #x7636)	;CJK UNIFIED IDEOGRAPH
                                (#xAF99 . #x7637)	;CJK UNIFIED IDEOGRAPH
                                (#xAF9A . #x7639)	;CJK UNIFIED IDEOGRAPH
                                (#xAF9B . #x763A)	;CJK UNIFIED IDEOGRAPH
                                (#xAF9C . #x763B)	;CJK UNIFIED IDEOGRAPH
                                (#xAF9D . #x763D)	;CJK UNIFIED IDEOGRAPH
                                (#xAF9E . #x7641)	;CJK UNIFIED IDEOGRAPH
                                (#xAF9F . #x7642)	;CJK UNIFIED IDEOGRAPH
                                (#xAFA0 . #x7644)	;CJK UNIFIED IDEOGRAPH
                                (#xB040 . #x7645)	;CJK UNIFIED IDEOGRAPH
                                (#xB041 . #x7646)	;CJK UNIFIED IDEOGRAPH
                                (#xB042 . #x7647)	;CJK UNIFIED IDEOGRAPH
                                (#xB043 . #x7648)	;CJK UNIFIED IDEOGRAPH
                                (#xB044 . #x7649)	;CJK UNIFIED IDEOGRAPH
                                (#xB045 . #x764A)	;CJK UNIFIED IDEOGRAPH
                                (#xB046 . #x764B)	;CJK UNIFIED IDEOGRAPH
                                (#xB047 . #x764E)	;CJK UNIFIED IDEOGRAPH
                                (#xB048 . #x764F)	;CJK UNIFIED IDEOGRAPH
                                (#xB049 . #x7650)	;CJK UNIFIED IDEOGRAPH
                                (#xB04A . #x7651)	;CJK UNIFIED IDEOGRAPH
                                (#xB04B . #x7652)	;CJK UNIFIED IDEOGRAPH
                                (#xB04C . #x7653)	;CJK UNIFIED IDEOGRAPH
                                (#xB04D . #x7655)	;CJK UNIFIED IDEOGRAPH
                                (#xB04E . #x7657)	;CJK UNIFIED IDEOGRAPH
                                (#xB04F . #x7658)	;CJK UNIFIED IDEOGRAPH
                                (#xB050 . #x7659)	;CJK UNIFIED IDEOGRAPH
                                (#xB051 . #x765A)	;CJK UNIFIED IDEOGRAPH
                                (#xB052 . #x765B)	;CJK UNIFIED IDEOGRAPH
                                (#xB053 . #x765D)	;CJK UNIFIED IDEOGRAPH
                                (#xB054 . #x765F)	;CJK UNIFIED IDEOGRAPH
                                (#xB055 . #x7660)	;CJK UNIFIED IDEOGRAPH
                                (#xB056 . #x7661)	;CJK UNIFIED IDEOGRAPH
                                (#xB057 . #x7662)	;CJK UNIFIED IDEOGRAPH
                                (#xB058 . #x7664)	;CJK UNIFIED IDEOGRAPH
                                (#xB059 . #x7665)	;CJK UNIFIED IDEOGRAPH
                                (#xB05A . #x7666)	;CJK UNIFIED IDEOGRAPH
                                (#xB05B . #x7667)	;CJK UNIFIED IDEOGRAPH
                                (#xB05C . #x7668)	;CJK UNIFIED IDEOGRAPH
                                (#xB05D . #x7669)	;CJK UNIFIED IDEOGRAPH
                                (#xB05E . #x766A)	;CJK UNIFIED IDEOGRAPH
                                (#xB05F . #x766C)	;CJK UNIFIED IDEOGRAPH
                                (#xB060 . #x766D)	;CJK UNIFIED IDEOGRAPH
                                (#xB061 . #x766E)	;CJK UNIFIED IDEOGRAPH
                                (#xB062 . #x7670)	;CJK UNIFIED IDEOGRAPH
                                (#xB063 . #x7671)	;CJK UNIFIED IDEOGRAPH
                                (#xB064 . #x7672)	;CJK UNIFIED IDEOGRAPH
                                (#xB065 . #x7673)	;CJK UNIFIED IDEOGRAPH
                                (#xB066 . #x7674)	;CJK UNIFIED IDEOGRAPH
                                (#xB067 . #x7675)	;CJK UNIFIED IDEOGRAPH
                                (#xB068 . #x7676)	;CJK UNIFIED IDEOGRAPH
                                (#xB069 . #x7677)	;CJK UNIFIED IDEOGRAPH
                                (#xB06A . #x7679)	;CJK UNIFIED IDEOGRAPH
                                (#xB06B . #x767A)	;CJK UNIFIED IDEOGRAPH
                                (#xB06C . #x767C)	;CJK UNIFIED IDEOGRAPH
                                (#xB06D . #x767F)	;CJK UNIFIED IDEOGRAPH
                                (#xB06E . #x7680)	;CJK UNIFIED IDEOGRAPH
                                (#xB06F . #x7681)	;CJK UNIFIED IDEOGRAPH
                                (#xB070 . #x7683)	;CJK UNIFIED IDEOGRAPH
                                (#xB071 . #x7685)	;CJK UNIFIED IDEOGRAPH
                                (#xB072 . #x7689)	;CJK UNIFIED IDEOGRAPH
                                (#xB073 . #x768A)	;CJK UNIFIED IDEOGRAPH
                                (#xB074 . #x768C)	;CJK UNIFIED IDEOGRAPH
                                (#xB075 . #x768D)	;CJK UNIFIED IDEOGRAPH
                                (#xB076 . #x768F)	;CJK UNIFIED IDEOGRAPH
                                (#xB077 . #x7690)	;CJK UNIFIED IDEOGRAPH
                                (#xB078 . #x7692)	;CJK UNIFIED IDEOGRAPH
                                (#xB079 . #x7694)	;CJK UNIFIED IDEOGRAPH
                                (#xB07A . #x7695)	;CJK UNIFIED IDEOGRAPH
                                (#xB07B . #x7697)	;CJK UNIFIED IDEOGRAPH
                                (#xB07C . #x7698)	;CJK UNIFIED IDEOGRAPH
                                (#xB07D . #x769A)	;CJK UNIFIED IDEOGRAPH
                                (#xB07E . #x769B)	;CJK UNIFIED IDEOGRAPH
                                (#xB080 . #x769C)	;CJK UNIFIED IDEOGRAPH
                                (#xB081 . #x769D)	;CJK UNIFIED IDEOGRAPH
                                (#xB082 . #x769E)	;CJK UNIFIED IDEOGRAPH
                                (#xB083 . #x769F)	;CJK UNIFIED IDEOGRAPH
                                (#xB084 . #x76A0)	;CJK UNIFIED IDEOGRAPH
                                (#xB085 . #x76A1)	;CJK UNIFIED IDEOGRAPH
                                (#xB086 . #x76A2)	;CJK UNIFIED IDEOGRAPH
                                (#xB087 . #x76A3)	;CJK UNIFIED IDEOGRAPH
                                (#xB088 . #x76A5)	;CJK UNIFIED IDEOGRAPH
                                (#xB089 . #x76A6)	;CJK UNIFIED IDEOGRAPH
                                (#xB08A . #x76A7)	;CJK UNIFIED IDEOGRAPH
                                (#xB08B . #x76A8)	;CJK UNIFIED IDEOGRAPH
                                (#xB08C . #x76A9)	;CJK UNIFIED IDEOGRAPH
                                (#xB08D . #x76AA)	;CJK UNIFIED IDEOGRAPH
                                (#xB08E . #x76AB)	;CJK UNIFIED IDEOGRAPH
                                (#xB08F . #x76AC)	;CJK UNIFIED IDEOGRAPH
                                (#xB090 . #x76AD)	;CJK UNIFIED IDEOGRAPH
                                (#xB091 . #x76AF)	;CJK UNIFIED IDEOGRAPH
                                (#xB092 . #x76B0)	;CJK UNIFIED IDEOGRAPH
                                (#xB093 . #x76B3)	;CJK UNIFIED IDEOGRAPH
                                (#xB094 . #x76B5)	;CJK UNIFIED IDEOGRAPH
                                (#xB095 . #x76B6)	;CJK UNIFIED IDEOGRAPH
                                (#xB096 . #x76B7)	;CJK UNIFIED IDEOGRAPH
                                (#xB097 . #x76B8)	;CJK UNIFIED IDEOGRAPH
                                (#xB098 . #x76B9)	;CJK UNIFIED IDEOGRAPH
                                (#xB099 . #x76BA)	;CJK UNIFIED IDEOGRAPH
                                (#xB09A . #x76BB)	;CJK UNIFIED IDEOGRAPH
                                (#xB09B . #x76BC)	;CJK UNIFIED IDEOGRAPH
                                (#xB09C . #x76BD)	;CJK UNIFIED IDEOGRAPH
                                (#xB09D . #x76BE)	;CJK UNIFIED IDEOGRAPH
                                (#xB09E . #x76C0)	;CJK UNIFIED IDEOGRAPH
                                (#xB09F . #x76C1)	;CJK UNIFIED IDEOGRAPH
                                (#xB0A0 . #x76C3)	;CJK UNIFIED IDEOGRAPH
                                (#xB0A1 . #x554A)	;CJK UNIFIED IDEOGRAPH
                                (#xB0A2 . #x963F)	;CJK UNIFIED IDEOGRAPH
                                (#xB0A3 . #x57C3)	;CJK UNIFIED IDEOGRAPH
                                (#xB0A4 . #x6328)	;CJK UNIFIED IDEOGRAPH
                                (#xB0A5 . #x54CE)	;CJK UNIFIED IDEOGRAPH
                                (#xB0A6 . #x5509)	;CJK UNIFIED IDEOGRAPH
                                (#xB0A7 . #x54C0)	;CJK UNIFIED IDEOGRAPH
                                (#xB0A8 . #x7691)	;CJK UNIFIED IDEOGRAPH
                                (#xB0A9 . #x764C)	;CJK UNIFIED IDEOGRAPH
                                (#xB0AA . #x853C)	;CJK UNIFIED IDEOGRAPH
                                (#xB0AB . #x77EE)	;CJK UNIFIED IDEOGRAPH
                                (#xB0AC . #x827E)	;CJK UNIFIED IDEOGRAPH
                                (#xB0AD . #x788D)	;CJK UNIFIED IDEOGRAPH
                                (#xB0AE . #x7231)	;CJK UNIFIED IDEOGRAPH
                                (#xB0AF . #x9698)	;CJK UNIFIED IDEOGRAPH
                                (#xB0B0 . #x978D)	;CJK UNIFIED IDEOGRAPH
                                (#xB0B1 . #x6C28)	;CJK UNIFIED IDEOGRAPH
                                (#xB0B2 . #x5B89)	;CJK UNIFIED IDEOGRAPH
                                (#xB0B3 . #x4FFA)	;CJK UNIFIED IDEOGRAPH
                                (#xB0B4 . #x6309)	;CJK UNIFIED IDEOGRAPH
                                (#xB0B5 . #x6697)	;CJK UNIFIED IDEOGRAPH
                                (#xB0B6 . #x5CB8)	;CJK UNIFIED IDEOGRAPH
                                (#xB0B7 . #x80FA)	;CJK UNIFIED IDEOGRAPH
                                (#xB0B8 . #x6848)	;CJK UNIFIED IDEOGRAPH
                                (#xB0B9 . #x80AE)	;CJK UNIFIED IDEOGRAPH
                                (#xB0BA . #x6602)	;CJK UNIFIED IDEOGRAPH
                                (#xB0BB . #x76CE)	;CJK UNIFIED IDEOGRAPH
                                (#xB0BC . #x51F9)	;CJK UNIFIED IDEOGRAPH
                                (#xB0BD . #x6556)	;CJK UNIFIED IDEOGRAPH
                                (#xB0BE . #x71AC)	;CJK UNIFIED IDEOGRAPH
                                (#xB0BF . #x7FF1)	;CJK UNIFIED IDEOGRAPH
                                (#xB0C0 . #x8884)	;CJK UNIFIED IDEOGRAPH
                                (#xB0C1 . #x50B2)	;CJK UNIFIED IDEOGRAPH
                                (#xB0C2 . #x5965)	;CJK UNIFIED IDEOGRAPH
                                (#xB0C3 . #x61CA)	;CJK UNIFIED IDEOGRAPH
                                (#xB0C4 . #x6FB3)	;CJK UNIFIED IDEOGRAPH
                                (#xB0C5 . #x82AD)	;CJK UNIFIED IDEOGRAPH
                                (#xB0C6 . #x634C)	;CJK UNIFIED IDEOGRAPH
                                (#xB0C7 . #x6252)	;CJK UNIFIED IDEOGRAPH
                                (#xB0C8 . #x53ED)	;CJK UNIFIED IDEOGRAPH
                                (#xB0C9 . #x5427)	;CJK UNIFIED IDEOGRAPH
                                (#xB0CA . #x7B06)	;CJK UNIFIED IDEOGRAPH
                                (#xB0CB . #x516B)	;CJK UNIFIED IDEOGRAPH
                                (#xB0CC . #x75A4)	;CJK UNIFIED IDEOGRAPH
                                (#xB0CD . #x5DF4)	;CJK UNIFIED IDEOGRAPH
                                (#xB0CE . #x62D4)	;CJK UNIFIED IDEOGRAPH
                                (#xB0CF . #x8DCB)	;CJK UNIFIED IDEOGRAPH
                                (#xB0D0 . #x9776)	;CJK UNIFIED IDEOGRAPH
                                (#xB0D1 . #x628A)	;CJK UNIFIED IDEOGRAPH
                                (#xB0D2 . #x8019)	;CJK UNIFIED IDEOGRAPH
                                (#xB0D3 . #x575D)	;CJK UNIFIED IDEOGRAPH
                                (#xB0D4 . #x9738)	;CJK UNIFIED IDEOGRAPH
                                (#xB0D5 . #x7F62)	;CJK UNIFIED IDEOGRAPH
                                (#xB0D6 . #x7238)	;CJK UNIFIED IDEOGRAPH
                                (#xB0D7 . #x767D)	;CJK UNIFIED IDEOGRAPH
                                (#xB0D8 . #x67CF)	;CJK UNIFIED IDEOGRAPH
                                (#xB0D9 . #x767E)	;CJK UNIFIED IDEOGRAPH
                                (#xB0DA . #x6446)	;CJK UNIFIED IDEOGRAPH
                                (#xB0DB . #x4F70)	;CJK UNIFIED IDEOGRAPH
                                (#xB0DC . #x8D25)	;CJK UNIFIED IDEOGRAPH
                                (#xB0DD . #x62DC)	;CJK UNIFIED IDEOGRAPH
                                (#xB0DE . #x7A17)	;CJK UNIFIED IDEOGRAPH
                                (#xB0DF . #x6591)	;CJK UNIFIED IDEOGRAPH
                                (#xB0E0 . #x73ED)	;CJK UNIFIED IDEOGRAPH
                                (#xB0E1 . #x642C)	;CJK UNIFIED IDEOGRAPH
                                (#xB0E2 . #x6273)	;CJK UNIFIED IDEOGRAPH
                                (#xB0E3 . #x822C)	;CJK UNIFIED IDEOGRAPH
                                (#xB0E4 . #x9881)	;CJK UNIFIED IDEOGRAPH
                                (#xB0E5 . #x677F)	;CJK UNIFIED IDEOGRAPH
                                (#xB0E6 . #x7248)	;CJK UNIFIED IDEOGRAPH
                                (#xB0E7 . #x626E)	;CJK UNIFIED IDEOGRAPH
                                (#xB0E8 . #x62CC)	;CJK UNIFIED IDEOGRAPH
                                (#xB0E9 . #x4F34)	;CJK UNIFIED IDEOGRAPH
                                (#xB0EA . #x74E3)	;CJK UNIFIED IDEOGRAPH
                                (#xB0EB . #x534A)	;CJK UNIFIED IDEOGRAPH
                                (#xB0EC . #x529E)	;CJK UNIFIED IDEOGRAPH
                                (#xB0ED . #x7ECA)	;CJK UNIFIED IDEOGRAPH
                                (#xB0EE . #x90A6)	;CJK UNIFIED IDEOGRAPH
                                (#xB0EF . #x5E2E)	;CJK UNIFIED IDEOGRAPH
                                (#xB0F0 . #x6886)	;CJK UNIFIED IDEOGRAPH
                                (#xB0F1 . #x699C)	;CJK UNIFIED IDEOGRAPH
                                (#xB0F2 . #x8180)	;CJK UNIFIED IDEOGRAPH
                                (#xB0F3 . #x7ED1)	;CJK UNIFIED IDEOGRAPH
                                (#xB0F4 . #x68D2)	;CJK UNIFIED IDEOGRAPH
                                (#xB0F5 . #x78C5)	;CJK UNIFIED IDEOGRAPH
                                (#xB0F6 . #x868C)	;CJK UNIFIED IDEOGRAPH
                                (#xB0F7 . #x9551)	;CJK UNIFIED IDEOGRAPH
                                (#xB0F8 . #x508D)	;CJK UNIFIED IDEOGRAPH
                                (#xB0F9 . #x8C24)	;CJK UNIFIED IDEOGRAPH
                                (#xB0FA . #x82DE)	;CJK UNIFIED IDEOGRAPH
                                (#xB0FB . #x80DE)	;CJK UNIFIED IDEOGRAPH
                                (#xB0FC . #x5305)	;CJK UNIFIED IDEOGRAPH
                                (#xB0FD . #x8912)	;CJK UNIFIED IDEOGRAPH
                                (#xB0FE . #x5265)	;CJK UNIFIED IDEOGRAPH
                                (#xB140 . #x76C4)	;CJK UNIFIED IDEOGRAPH
                                (#xB141 . #x76C7)	;CJK UNIFIED IDEOGRAPH
                                (#xB142 . #x76C9)	;CJK UNIFIED IDEOGRAPH
                                (#xB143 . #x76CB)	;CJK UNIFIED IDEOGRAPH
                                (#xB144 . #x76CC)	;CJK UNIFIED IDEOGRAPH
                                (#xB145 . #x76D3)	;CJK UNIFIED IDEOGRAPH
                                (#xB146 . #x76D5)	;CJK UNIFIED IDEOGRAPH
                                (#xB147 . #x76D9)	;CJK UNIFIED IDEOGRAPH
                                (#xB148 . #x76DA)	;CJK UNIFIED IDEOGRAPH
                                (#xB149 . #x76DC)	;CJK UNIFIED IDEOGRAPH
                                (#xB14A . #x76DD)	;CJK UNIFIED IDEOGRAPH
                                (#xB14B . #x76DE)	;CJK UNIFIED IDEOGRAPH
                                (#xB14C . #x76E0)	;CJK UNIFIED IDEOGRAPH
                                (#xB14D . #x76E1)	;CJK UNIFIED IDEOGRAPH
                                (#xB14E . #x76E2)	;CJK UNIFIED IDEOGRAPH
                                (#xB14F . #x76E3)	;CJK UNIFIED IDEOGRAPH
                                (#xB150 . #x76E4)	;CJK UNIFIED IDEOGRAPH
                                (#xB151 . #x76E6)	;CJK UNIFIED IDEOGRAPH
                                (#xB152 . #x76E7)	;CJK UNIFIED IDEOGRAPH
                                (#xB153 . #x76E8)	;CJK UNIFIED IDEOGRAPH
                                (#xB154 . #x76E9)	;CJK UNIFIED IDEOGRAPH
                                (#xB155 . #x76EA)	;CJK UNIFIED IDEOGRAPH
                                (#xB156 . #x76EB)	;CJK UNIFIED IDEOGRAPH
                                (#xB157 . #x76EC)	;CJK UNIFIED IDEOGRAPH
                                (#xB158 . #x76ED)	;CJK UNIFIED IDEOGRAPH
                                (#xB159 . #x76F0)	;CJK UNIFIED IDEOGRAPH
                                (#xB15A . #x76F3)	;CJK UNIFIED IDEOGRAPH
                                (#xB15B . #x76F5)	;CJK UNIFIED IDEOGRAPH
                                (#xB15C . #x76F6)	;CJK UNIFIED IDEOGRAPH
                                (#xB15D . #x76F7)	;CJK UNIFIED IDEOGRAPH
                                (#xB15E . #x76FA)	;CJK UNIFIED IDEOGRAPH
                                (#xB15F . #x76FB)	;CJK UNIFIED IDEOGRAPH
                                (#xB160 . #x76FD)	;CJK UNIFIED IDEOGRAPH
                                (#xB161 . #x76FF)	;CJK UNIFIED IDEOGRAPH
                                (#xB162 . #x7700)	;CJK UNIFIED IDEOGRAPH
                                (#xB163 . #x7702)	;CJK UNIFIED IDEOGRAPH
                                (#xB164 . #x7703)	;CJK UNIFIED IDEOGRAPH
                                (#xB165 . #x7705)	;CJK UNIFIED IDEOGRAPH
                                (#xB166 . #x7706)	;CJK UNIFIED IDEOGRAPH
                                (#xB167 . #x770A)	;CJK UNIFIED IDEOGRAPH
                                (#xB168 . #x770C)	;CJK UNIFIED IDEOGRAPH
                                (#xB169 . #x770E)	;CJK UNIFIED IDEOGRAPH
                                (#xB16A . #x770F)	;CJK UNIFIED IDEOGRAPH
                                (#xB16B . #x7710)	;CJK UNIFIED IDEOGRAPH
                                (#xB16C . #x7711)	;CJK UNIFIED IDEOGRAPH
                                (#xB16D . #x7712)	;CJK UNIFIED IDEOGRAPH
                                (#xB16E . #x7713)	;CJK UNIFIED IDEOGRAPH
                                (#xB16F . #x7714)	;CJK UNIFIED IDEOGRAPH
                                (#xB170 . #x7715)	;CJK UNIFIED IDEOGRAPH
                                (#xB171 . #x7716)	;CJK UNIFIED IDEOGRAPH
                                (#xB172 . #x7717)	;CJK UNIFIED IDEOGRAPH
                                (#xB173 . #x7718)	;CJK UNIFIED IDEOGRAPH
                                (#xB174 . #x771B)	;CJK UNIFIED IDEOGRAPH
                                (#xB175 . #x771C)	;CJK UNIFIED IDEOGRAPH
                                (#xB176 . #x771D)	;CJK UNIFIED IDEOGRAPH
                                (#xB177 . #x771E)	;CJK UNIFIED IDEOGRAPH
                                (#xB178 . #x7721)	;CJK UNIFIED IDEOGRAPH
                                (#xB179 . #x7723)	;CJK UNIFIED IDEOGRAPH
                                (#xB17A . #x7724)	;CJK UNIFIED IDEOGRAPH
                                (#xB17B . #x7725)	;CJK UNIFIED IDEOGRAPH
                                (#xB17C . #x7727)	;CJK UNIFIED IDEOGRAPH
                                (#xB17D . #x772A)	;CJK UNIFIED IDEOGRAPH
                                (#xB17E . #x772B)	;CJK UNIFIED IDEOGRAPH
                                (#xB180 . #x772C)	;CJK UNIFIED IDEOGRAPH
                                (#xB181 . #x772E)	;CJK UNIFIED IDEOGRAPH
                                (#xB182 . #x7730)	;CJK UNIFIED IDEOGRAPH
                                (#xB183 . #x7731)	;CJK UNIFIED IDEOGRAPH
                                (#xB184 . #x7732)	;CJK UNIFIED IDEOGRAPH
                                (#xB185 . #x7733)	;CJK UNIFIED IDEOGRAPH
                                (#xB186 . #x7734)	;CJK UNIFIED IDEOGRAPH
                                (#xB187 . #x7739)	;CJK UNIFIED IDEOGRAPH
                                (#xB188 . #x773B)	;CJK UNIFIED IDEOGRAPH
                                (#xB189 . #x773D)	;CJK UNIFIED IDEOGRAPH
                                (#xB18A . #x773E)	;CJK UNIFIED IDEOGRAPH
                                (#xB18B . #x773F)	;CJK UNIFIED IDEOGRAPH
                                (#xB18C . #x7742)	;CJK UNIFIED IDEOGRAPH
                                (#xB18D . #x7744)	;CJK UNIFIED IDEOGRAPH
                                (#xB18E . #x7745)	;CJK UNIFIED IDEOGRAPH
                                (#xB18F . #x7746)	;CJK UNIFIED IDEOGRAPH
                                (#xB190 . #x7748)	;CJK UNIFIED IDEOGRAPH
                                (#xB191 . #x7749)	;CJK UNIFIED IDEOGRAPH
                                (#xB192 . #x774A)	;CJK UNIFIED IDEOGRAPH
                                (#xB193 . #x774B)	;CJK UNIFIED IDEOGRAPH
                                (#xB194 . #x774C)	;CJK UNIFIED IDEOGRAPH
                                (#xB195 . #x774D)	;CJK UNIFIED IDEOGRAPH
                                (#xB196 . #x774E)	;CJK UNIFIED IDEOGRAPH
                                (#xB197 . #x774F)	;CJK UNIFIED IDEOGRAPH
                                (#xB198 . #x7752)	;CJK UNIFIED IDEOGRAPH
                                (#xB199 . #x7753)	;CJK UNIFIED IDEOGRAPH
                                (#xB19A . #x7754)	;CJK UNIFIED IDEOGRAPH
                                (#xB19B . #x7755)	;CJK UNIFIED IDEOGRAPH
                                (#xB19C . #x7756)	;CJK UNIFIED IDEOGRAPH
                                (#xB19D . #x7757)	;CJK UNIFIED IDEOGRAPH
                                (#xB19E . #x7758)	;CJK UNIFIED IDEOGRAPH
                                (#xB19F . #x7759)	;CJK UNIFIED IDEOGRAPH
                                (#xB1A0 . #x775C)	;CJK UNIFIED IDEOGRAPH
                                (#xB1A1 . #x8584)	;CJK UNIFIED IDEOGRAPH
                                (#xB1A2 . #x96F9)	;CJK UNIFIED IDEOGRAPH
                                (#xB1A3 . #x4FDD)	;CJK UNIFIED IDEOGRAPH
                                (#xB1A4 . #x5821)	;CJK UNIFIED IDEOGRAPH
                                (#xB1A5 . #x9971)	;CJK UNIFIED IDEOGRAPH
                                (#xB1A6 . #x5B9D)	;CJK UNIFIED IDEOGRAPH
                                (#xB1A7 . #x62B1)	;CJK UNIFIED IDEOGRAPH
                                (#xB1A8 . #x62A5)	;CJK UNIFIED IDEOGRAPH
                                (#xB1A9 . #x66B4)	;CJK UNIFIED IDEOGRAPH
                                (#xB1AA . #x8C79)	;CJK UNIFIED IDEOGRAPH
                                (#xB1AB . #x9C8D)	;CJK UNIFIED IDEOGRAPH
                                (#xB1AC . #x7206)	;CJK UNIFIED IDEOGRAPH
                                (#xB1AD . #x676F)	;CJK UNIFIED IDEOGRAPH
                                (#xB1AE . #x7891)	;CJK UNIFIED IDEOGRAPH
                                (#xB1AF . #x60B2)	;CJK UNIFIED IDEOGRAPH
                                (#xB1B0 . #x5351)	;CJK UNIFIED IDEOGRAPH
                                (#xB1B1 . #x5317)	;CJK UNIFIED IDEOGRAPH
                                (#xB1B2 . #x8F88)	;CJK UNIFIED IDEOGRAPH
                                (#xB1B3 . #x80CC)	;CJK UNIFIED IDEOGRAPH
                                (#xB1B4 . #x8D1D)	;CJK UNIFIED IDEOGRAPH
                                (#xB1B5 . #x94A1)	;CJK UNIFIED IDEOGRAPH
                                (#xB1B6 . #x500D)	;CJK UNIFIED IDEOGRAPH
                                (#xB1B7 . #x72C8)	;CJK UNIFIED IDEOGRAPH
                                (#xB1B8 . #x5907)	;CJK UNIFIED IDEOGRAPH
                                (#xB1B9 . #x60EB)	;CJK UNIFIED IDEOGRAPH
                                (#xB1BA . #x7119)	;CJK UNIFIED IDEOGRAPH
                                (#xB1BB . #x88AB)	;CJK UNIFIED IDEOGRAPH
                                (#xB1BC . #x5954)	;CJK UNIFIED IDEOGRAPH
                                (#xB1BD . #x82EF)	;CJK UNIFIED IDEOGRAPH
                                (#xB1BE . #x672C)	;CJK UNIFIED IDEOGRAPH
                                (#xB1BF . #x7B28)	;CJK UNIFIED IDEOGRAPH
                                (#xB1C0 . #x5D29)	;CJK UNIFIED IDEOGRAPH
                                (#xB1C1 . #x7EF7)	;CJK UNIFIED IDEOGRAPH
                                (#xB1C2 . #x752D)	;CJK UNIFIED IDEOGRAPH
                                (#xB1C3 . #x6CF5)	;CJK UNIFIED IDEOGRAPH
                                (#xB1C4 . #x8E66)	;CJK UNIFIED IDEOGRAPH
                                (#xB1C5 . #x8FF8)	;CJK UNIFIED IDEOGRAPH
                                (#xB1C6 . #x903C)	;CJK UNIFIED IDEOGRAPH
                                (#xB1C7 . #x9F3B)	;CJK UNIFIED IDEOGRAPH
                                (#xB1C8 . #x6BD4)	;CJK UNIFIED IDEOGRAPH
                                (#xB1C9 . #x9119)	;CJK UNIFIED IDEOGRAPH
                                (#xB1CA . #x7B14)	;CJK UNIFIED IDEOGRAPH
                                (#xB1CB . #x5F7C)	;CJK UNIFIED IDEOGRAPH
                                (#xB1CC . #x78A7)	;CJK UNIFIED IDEOGRAPH
                                (#xB1CD . #x84D6)	;CJK UNIFIED IDEOGRAPH
                                (#xB1CE . #x853D)	;CJK UNIFIED IDEOGRAPH
                                (#xB1CF . #x6BD5)	;CJK UNIFIED IDEOGRAPH
                                (#xB1D0 . #x6BD9)	;CJK UNIFIED IDEOGRAPH
                                (#xB1D1 . #x6BD6)	;CJK UNIFIED IDEOGRAPH
                                (#xB1D2 . #x5E01)	;CJK UNIFIED IDEOGRAPH
                                (#xB1D3 . #x5E87)	;CJK UNIFIED IDEOGRAPH
                                (#xB1D4 . #x75F9)	;CJK UNIFIED IDEOGRAPH
                                (#xB1D5 . #x95ED)	;CJK UNIFIED IDEOGRAPH
                                (#xB1D6 . #x655D)	;CJK UNIFIED IDEOGRAPH
                                (#xB1D7 . #x5F0A)	;CJK UNIFIED IDEOGRAPH
                                (#xB1D8 . #x5FC5)	;CJK UNIFIED IDEOGRAPH
                                (#xB1D9 . #x8F9F)	;CJK UNIFIED IDEOGRAPH
                                (#xB1DA . #x58C1)	;CJK UNIFIED IDEOGRAPH
                                (#xB1DB . #x81C2)	;CJK UNIFIED IDEOGRAPH
                                (#xB1DC . #x907F)	;CJK UNIFIED IDEOGRAPH
                                (#xB1DD . #x965B)	;CJK UNIFIED IDEOGRAPH
                                (#xB1DE . #x97AD)	;CJK UNIFIED IDEOGRAPH
                                (#xB1DF . #x8FB9)	;CJK UNIFIED IDEOGRAPH
                                (#xB1E0 . #x7F16)	;CJK UNIFIED IDEOGRAPH
                                (#xB1E1 . #x8D2C)	;CJK UNIFIED IDEOGRAPH
                                (#xB1E2 . #x6241)	;CJK UNIFIED IDEOGRAPH
                                (#xB1E3 . #x4FBF)	;CJK UNIFIED IDEOGRAPH
                                (#xB1E4 . #x53D8)	;CJK UNIFIED IDEOGRAPH
                                (#xB1E5 . #x535E)	;CJK UNIFIED IDEOGRAPH
                                (#xB1E6 . #x8FA8)	;CJK UNIFIED IDEOGRAPH
                                (#xB1E7 . #x8FA9)	;CJK UNIFIED IDEOGRAPH
                                (#xB1E8 . #x8FAB)	;CJK UNIFIED IDEOGRAPH
                                (#xB1E9 . #x904D)	;CJK UNIFIED IDEOGRAPH
                                (#xB1EA . #x6807)	;CJK UNIFIED IDEOGRAPH
                                (#xB1EB . #x5F6A)	;CJK UNIFIED IDEOGRAPH
                                (#xB1EC . #x8198)	;CJK UNIFIED IDEOGRAPH
                                (#xB1ED . #x8868)	;CJK UNIFIED IDEOGRAPH
                                (#xB1EE . #x9CD6)	;CJK UNIFIED IDEOGRAPH
                                (#xB1EF . #x618B)	;CJK UNIFIED IDEOGRAPH
                                (#xB1F0 . #x522B)	;CJK UNIFIED IDEOGRAPH
                                (#xB1F1 . #x762A)	;CJK UNIFIED IDEOGRAPH
                                (#xB1F2 . #x5F6C)	;CJK UNIFIED IDEOGRAPH
                                (#xB1F3 . #x658C)	;CJK UNIFIED IDEOGRAPH
                                (#xB1F4 . #x6FD2)	;CJK UNIFIED IDEOGRAPH
                                (#xB1F5 . #x6EE8)	;CJK UNIFIED IDEOGRAPH
                                (#xB1F6 . #x5BBE)	;CJK UNIFIED IDEOGRAPH
                                (#xB1F7 . #x6448)	;CJK UNIFIED IDEOGRAPH
                                (#xB1F8 . #x5175)	;CJK UNIFIED IDEOGRAPH
                                (#xB1F9 . #x51B0)	;CJK UNIFIED IDEOGRAPH
                                (#xB1FA . #x67C4)	;CJK UNIFIED IDEOGRAPH
                                (#xB1FB . #x4E19)	;CJK UNIFIED IDEOGRAPH
                                (#xB1FC . #x79C9)	;CJK UNIFIED IDEOGRAPH
                                (#xB1FD . #x997C)	;CJK UNIFIED IDEOGRAPH
                                (#xB1FE . #x70B3)	;CJK UNIFIED IDEOGRAPH
                                (#xB240 . #x775D)	;CJK UNIFIED IDEOGRAPH
                                (#xB241 . #x775E)	;CJK UNIFIED IDEOGRAPH
                                (#xB242 . #x775F)	;CJK UNIFIED IDEOGRAPH
                                (#xB243 . #x7760)	;CJK UNIFIED IDEOGRAPH
                                (#xB244 . #x7764)	;CJK UNIFIED IDEOGRAPH
                                (#xB245 . #x7767)	;CJK UNIFIED IDEOGRAPH
                                (#xB246 . #x7769)	;CJK UNIFIED IDEOGRAPH
                                (#xB247 . #x776A)	;CJK UNIFIED IDEOGRAPH
                                (#xB248 . #x776D)	;CJK UNIFIED IDEOGRAPH
                                (#xB249 . #x776E)	;CJK UNIFIED IDEOGRAPH
                                (#xB24A . #x776F)	;CJK UNIFIED IDEOGRAPH
                                (#xB24B . #x7770)	;CJK UNIFIED IDEOGRAPH
                                (#xB24C . #x7771)	;CJK UNIFIED IDEOGRAPH
                                (#xB24D . #x7772)	;CJK UNIFIED IDEOGRAPH
                                (#xB24E . #x7773)	;CJK UNIFIED IDEOGRAPH
                                (#xB24F . #x7774)	;CJK UNIFIED IDEOGRAPH
                                (#xB250 . #x7775)	;CJK UNIFIED IDEOGRAPH
                                (#xB251 . #x7776)	;CJK UNIFIED IDEOGRAPH
                                (#xB252 . #x7777)	;CJK UNIFIED IDEOGRAPH
                                (#xB253 . #x7778)	;CJK UNIFIED IDEOGRAPH
                                (#xB254 . #x777A)	;CJK UNIFIED IDEOGRAPH
                                (#xB255 . #x777B)	;CJK UNIFIED IDEOGRAPH
                                (#xB256 . #x777C)	;CJK UNIFIED IDEOGRAPH
                                (#xB257 . #x7781)	;CJK UNIFIED IDEOGRAPH
                                (#xB258 . #x7782)	;CJK UNIFIED IDEOGRAPH
                                (#xB259 . #x7783)	;CJK UNIFIED IDEOGRAPH
                                (#xB25A . #x7786)	;CJK UNIFIED IDEOGRAPH
                                (#xB25B . #x7787)	;CJK UNIFIED IDEOGRAPH
                                (#xB25C . #x7788)	;CJK UNIFIED IDEOGRAPH
                                (#xB25D . #x7789)	;CJK UNIFIED IDEOGRAPH
                                (#xB25E . #x778A)	;CJK UNIFIED IDEOGRAPH
                                (#xB25F . #x778B)	;CJK UNIFIED IDEOGRAPH
                                (#xB260 . #x778F)	;CJK UNIFIED IDEOGRAPH
                                (#xB261 . #x7790)	;CJK UNIFIED IDEOGRAPH
                                (#xB262 . #x7793)	;CJK UNIFIED IDEOGRAPH
                                (#xB263 . #x7794)	;CJK UNIFIED IDEOGRAPH
                                (#xB264 . #x7795)	;CJK UNIFIED IDEOGRAPH
                                (#xB265 . #x7796)	;CJK UNIFIED IDEOGRAPH
                                (#xB266 . #x7797)	;CJK UNIFIED IDEOGRAPH
                                (#xB267 . #x7798)	;CJK UNIFIED IDEOGRAPH
                                (#xB268 . #x7799)	;CJK UNIFIED IDEOGRAPH
                                (#xB269 . #x779A)	;CJK UNIFIED IDEOGRAPH
                                (#xB26A . #x779B)	;CJK UNIFIED IDEOGRAPH
                                (#xB26B . #x779C)	;CJK UNIFIED IDEOGRAPH
                                (#xB26C . #x779D)	;CJK UNIFIED IDEOGRAPH
                                (#xB26D . #x779E)	;CJK UNIFIED IDEOGRAPH
                                (#xB26E . #x77A1)	;CJK UNIFIED IDEOGRAPH
                                (#xB26F . #x77A3)	;CJK UNIFIED IDEOGRAPH
                                (#xB270 . #x77A4)	;CJK UNIFIED IDEOGRAPH
                                (#xB271 . #x77A6)	;CJK UNIFIED IDEOGRAPH
                                (#xB272 . #x77A8)	;CJK UNIFIED IDEOGRAPH
                                (#xB273 . #x77AB)	;CJK UNIFIED IDEOGRAPH
                                (#xB274 . #x77AD)	;CJK UNIFIED IDEOGRAPH
                                (#xB275 . #x77AE)	;CJK UNIFIED IDEOGRAPH
                                (#xB276 . #x77AF)	;CJK UNIFIED IDEOGRAPH
                                (#xB277 . #x77B1)	;CJK UNIFIED IDEOGRAPH
                                (#xB278 . #x77B2)	;CJK UNIFIED IDEOGRAPH
                                (#xB279 . #x77B4)	;CJK UNIFIED IDEOGRAPH
                                (#xB27A . #x77B6)	;CJK UNIFIED IDEOGRAPH
                                (#xB27B . #x77B7)	;CJK UNIFIED IDEOGRAPH
                                (#xB27C . #x77B8)	;CJK UNIFIED IDEOGRAPH
                                (#xB27D . #x77B9)	;CJK UNIFIED IDEOGRAPH
                                (#xB27E . #x77BA)	;CJK UNIFIED IDEOGRAPH
                                (#xB280 . #x77BC)	;CJK UNIFIED IDEOGRAPH
                                (#xB281 . #x77BE)	;CJK UNIFIED IDEOGRAPH
                                (#xB282 . #x77C0)	;CJK UNIFIED IDEOGRAPH
                                (#xB283 . #x77C1)	;CJK UNIFIED IDEOGRAPH
                                (#xB284 . #x77C2)	;CJK UNIFIED IDEOGRAPH
                                (#xB285 . #x77C3)	;CJK UNIFIED IDEOGRAPH
                                (#xB286 . #x77C4)	;CJK UNIFIED IDEOGRAPH
                                (#xB287 . #x77C5)	;CJK UNIFIED IDEOGRAPH
                                (#xB288 . #x77C6)	;CJK UNIFIED IDEOGRAPH
                                (#xB289 . #x77C7)	;CJK UNIFIED IDEOGRAPH
                                (#xB28A . #x77C8)	;CJK UNIFIED IDEOGRAPH
                                (#xB28B . #x77C9)	;CJK UNIFIED IDEOGRAPH
                                (#xB28C . #x77CA)	;CJK UNIFIED IDEOGRAPH
                                (#xB28D . #x77CB)	;CJK UNIFIED IDEOGRAPH
                                (#xB28E . #x77CC)	;CJK UNIFIED IDEOGRAPH
                                (#xB28F . #x77CE)	;CJK UNIFIED IDEOGRAPH
                                (#xB290 . #x77CF)	;CJK UNIFIED IDEOGRAPH
                                (#xB291 . #x77D0)	;CJK UNIFIED IDEOGRAPH
                                (#xB292 . #x77D1)	;CJK UNIFIED IDEOGRAPH
                                (#xB293 . #x77D2)	;CJK UNIFIED IDEOGRAPH
                                (#xB294 . #x77D3)	;CJK UNIFIED IDEOGRAPH
                                (#xB295 . #x77D4)	;CJK UNIFIED IDEOGRAPH
                                (#xB296 . #x77D5)	;CJK UNIFIED IDEOGRAPH
                                (#xB297 . #x77D6)	;CJK UNIFIED IDEOGRAPH
                                (#xB298 . #x77D8)	;CJK UNIFIED IDEOGRAPH
                                (#xB299 . #x77D9)	;CJK UNIFIED IDEOGRAPH
                                (#xB29A . #x77DA)	;CJK UNIFIED IDEOGRAPH
                                (#xB29B . #x77DD)	;CJK UNIFIED IDEOGRAPH
                                (#xB29C . #x77DE)	;CJK UNIFIED IDEOGRAPH
                                (#xB29D . #x77DF)	;CJK UNIFIED IDEOGRAPH
                                (#xB29E . #x77E0)	;CJK UNIFIED IDEOGRAPH
                                (#xB29F . #x77E1)	;CJK UNIFIED IDEOGRAPH
                                (#xB2A0 . #x77E4)	;CJK UNIFIED IDEOGRAPH
                                (#xB2A1 . #x75C5)	;CJK UNIFIED IDEOGRAPH
                                (#xB2A2 . #x5E76)	;CJK UNIFIED IDEOGRAPH
                                (#xB2A3 . #x73BB)	;CJK UNIFIED IDEOGRAPH
                                (#xB2A4 . #x83E0)	;CJK UNIFIED IDEOGRAPH
                                (#xB2A5 . #x64AD)	;CJK UNIFIED IDEOGRAPH
                                (#xB2A6 . #x62E8)	;CJK UNIFIED IDEOGRAPH
                                (#xB2A7 . #x94B5)	;CJK UNIFIED IDEOGRAPH
                                (#xB2A8 . #x6CE2)	;CJK UNIFIED IDEOGRAPH
                                (#xB2A9 . #x535A)	;CJK UNIFIED IDEOGRAPH
                                (#xB2AA . #x52C3)	;CJK UNIFIED IDEOGRAPH
                                (#xB2AB . #x640F)	;CJK UNIFIED IDEOGRAPH
                                (#xB2AC . #x94C2)	;CJK UNIFIED IDEOGRAPH
                                (#xB2AD . #x7B94)	;CJK UNIFIED IDEOGRAPH
                                (#xB2AE . #x4F2F)	;CJK UNIFIED IDEOGRAPH
                                (#xB2AF . #x5E1B)	;CJK UNIFIED IDEOGRAPH
                                (#xB2B0 . #x8236)	;CJK UNIFIED IDEOGRAPH
                                (#xB2B1 . #x8116)	;CJK UNIFIED IDEOGRAPH
                                (#xB2B2 . #x818A)	;CJK UNIFIED IDEOGRAPH
                                (#xB2B3 . #x6E24)	;CJK UNIFIED IDEOGRAPH
                                (#xB2B4 . #x6CCA)	;CJK UNIFIED IDEOGRAPH
                                (#xB2B5 . #x9A73)	;CJK UNIFIED IDEOGRAPH
                                (#xB2B6 . #x6355)	;CJK UNIFIED IDEOGRAPH
                                (#xB2B7 . #x535C)	;CJK UNIFIED IDEOGRAPH
                                (#xB2B8 . #x54FA)	;CJK UNIFIED IDEOGRAPH
                                (#xB2B9 . #x8865)	;CJK UNIFIED IDEOGRAPH
                                (#xB2BA . #x57E0)	;CJK UNIFIED IDEOGRAPH
                                (#xB2BB . #x4E0D)	;CJK UNIFIED IDEOGRAPH
                                (#xB2BC . #x5E03)	;CJK UNIFIED IDEOGRAPH
                                (#xB2BD . #x6B65)	;CJK UNIFIED IDEOGRAPH
                                (#xB2BE . #x7C3F)	;CJK UNIFIED IDEOGRAPH
                                (#xB2BF . #x90E8)	;CJK UNIFIED IDEOGRAPH
                                (#xB2C0 . #x6016)	;CJK UNIFIED IDEOGRAPH
                                (#xB2C1 . #x64E6)	;CJK UNIFIED IDEOGRAPH
                                (#xB2C2 . #x731C)	;CJK UNIFIED IDEOGRAPH
                                (#xB2C3 . #x88C1)	;CJK UNIFIED IDEOGRAPH
                                (#xB2C4 . #x6750)	;CJK UNIFIED IDEOGRAPH
                                (#xB2C5 . #x624D)	;CJK UNIFIED IDEOGRAPH
                                (#xB2C6 . #x8D22)	;CJK UNIFIED IDEOGRAPH
                                (#xB2C7 . #x776C)	;CJK UNIFIED IDEOGRAPH
                                (#xB2C8 . #x8E29)	;CJK UNIFIED IDEOGRAPH
                                (#xB2C9 . #x91C7)	;CJK UNIFIED IDEOGRAPH
                                (#xB2CA . #x5F69)	;CJK UNIFIED IDEOGRAPH
                                (#xB2CB . #x83DC)	;CJK UNIFIED IDEOGRAPH
                                (#xB2CC . #x8521)	;CJK UNIFIED IDEOGRAPH
                                (#xB2CD . #x9910)	;CJK UNIFIED IDEOGRAPH
                                (#xB2CE . #x53C2)	;CJK UNIFIED IDEOGRAPH
                                (#xB2CF . #x8695)	;CJK UNIFIED IDEOGRAPH
                                (#xB2D0 . #x6B8B)	;CJK UNIFIED IDEOGRAPH
                                (#xB2D1 . #x60ED)	;CJK UNIFIED IDEOGRAPH
                                (#xB2D2 . #x60E8)	;CJK UNIFIED IDEOGRAPH
                                (#xB2D3 . #x707F)	;CJK UNIFIED IDEOGRAPH
                                (#xB2D4 . #x82CD)	;CJK UNIFIED IDEOGRAPH
                                (#xB2D5 . #x8231)	;CJK UNIFIED IDEOGRAPH
                                (#xB2D6 . #x4ED3)	;CJK UNIFIED IDEOGRAPH
                                (#xB2D7 . #x6CA7)	;CJK UNIFIED IDEOGRAPH
                                (#xB2D8 . #x85CF)	;CJK UNIFIED IDEOGRAPH
                                (#xB2D9 . #x64CD)	;CJK UNIFIED IDEOGRAPH
                                (#xB2DA . #x7CD9)	;CJK UNIFIED IDEOGRAPH
                                (#xB2DB . #x69FD)	;CJK UNIFIED IDEOGRAPH
                                (#xB2DC . #x66F9)	;CJK UNIFIED IDEOGRAPH
                                (#xB2DD . #x8349)	;CJK UNIFIED IDEOGRAPH
                                (#xB2DE . #x5395)	;CJK UNIFIED IDEOGRAPH
                                (#xB2DF . #x7B56)	;CJK UNIFIED IDEOGRAPH
                                (#xB2E0 . #x4FA7)	;CJK UNIFIED IDEOGRAPH
                                (#xB2E1 . #x518C)	;CJK UNIFIED IDEOGRAPH
                                (#xB2E2 . #x6D4B)	;CJK UNIFIED IDEOGRAPH
                                (#xB2E3 . #x5C42)	;CJK UNIFIED IDEOGRAPH
                                (#xB2E4 . #x8E6D)	;CJK UNIFIED IDEOGRAPH
                                (#xB2E5 . #x63D2)	;CJK UNIFIED IDEOGRAPH
                                (#xB2E6 . #x53C9)	;CJK UNIFIED IDEOGRAPH
                                (#xB2E7 . #x832C)	;CJK UNIFIED IDEOGRAPH
                                (#xB2E8 . #x8336)	;CJK UNIFIED IDEOGRAPH
                                (#xB2E9 . #x67E5)	;CJK UNIFIED IDEOGRAPH
                                (#xB2EA . #x78B4)	;CJK UNIFIED IDEOGRAPH
                                (#xB2EB . #x643D)	;CJK UNIFIED IDEOGRAPH
                                (#xB2EC . #x5BDF)	;CJK UNIFIED IDEOGRAPH
                                (#xB2ED . #x5C94)	;CJK UNIFIED IDEOGRAPH
                                (#xB2EE . #x5DEE)	;CJK UNIFIED IDEOGRAPH
                                (#xB2EF . #x8BE7)	;CJK UNIFIED IDEOGRAPH
                                (#xB2F0 . #x62C6)	;CJK UNIFIED IDEOGRAPH
                                (#xB2F1 . #x67F4)	;CJK UNIFIED IDEOGRAPH
                                (#xB2F2 . #x8C7A)	;CJK UNIFIED IDEOGRAPH
                                (#xB2F3 . #x6400)	;CJK UNIFIED IDEOGRAPH
                                (#xB2F4 . #x63BA)	;CJK UNIFIED IDEOGRAPH
                                (#xB2F5 . #x8749)	;CJK UNIFIED IDEOGRAPH
                                (#xB2F6 . #x998B)	;CJK UNIFIED IDEOGRAPH
                                (#xB2F7 . #x8C17)	;CJK UNIFIED IDEOGRAPH
                                (#xB2F8 . #x7F20)	;CJK UNIFIED IDEOGRAPH
                                (#xB2F9 . #x94F2)	;CJK UNIFIED IDEOGRAPH
                                (#xB2FA . #x4EA7)	;CJK UNIFIED IDEOGRAPH
                                (#xB2FB . #x9610)	;CJK UNIFIED IDEOGRAPH
                                (#xB2FC . #x98A4)	;CJK UNIFIED IDEOGRAPH
                                (#xB2FD . #x660C)	;CJK UNIFIED IDEOGRAPH
                                (#xB2FE . #x7316)	;CJK UNIFIED IDEOGRAPH
                                (#xB340 . #x77E6)	;CJK UNIFIED IDEOGRAPH
                                (#xB341 . #x77E8)	;CJK UNIFIED IDEOGRAPH
                                (#xB342 . #x77EA)	;CJK UNIFIED IDEOGRAPH
                                (#xB343 . #x77EF)	;CJK UNIFIED IDEOGRAPH
                                (#xB344 . #x77F0)	;CJK UNIFIED IDEOGRAPH
                                (#xB345 . #x77F1)	;CJK UNIFIED IDEOGRAPH
                                (#xB346 . #x77F2)	;CJK UNIFIED IDEOGRAPH
                                (#xB347 . #x77F4)	;CJK UNIFIED IDEOGRAPH
                                (#xB348 . #x77F5)	;CJK UNIFIED IDEOGRAPH
                                (#xB349 . #x77F7)	;CJK UNIFIED IDEOGRAPH
                                (#xB34A . #x77F9)	;CJK UNIFIED IDEOGRAPH
                                (#xB34B . #x77FA)	;CJK UNIFIED IDEOGRAPH
                                (#xB34C . #x77FB)	;CJK UNIFIED IDEOGRAPH
                                (#xB34D . #x77FC)	;CJK UNIFIED IDEOGRAPH
                                (#xB34E . #x7803)	;CJK UNIFIED IDEOGRAPH
                                (#xB34F . #x7804)	;CJK UNIFIED IDEOGRAPH
                                (#xB350 . #x7805)	;CJK UNIFIED IDEOGRAPH
                                (#xB351 . #x7806)	;CJK UNIFIED IDEOGRAPH
                                (#xB352 . #x7807)	;CJK UNIFIED IDEOGRAPH
                                (#xB353 . #x7808)	;CJK UNIFIED IDEOGRAPH
                                (#xB354 . #x780A)	;CJK UNIFIED IDEOGRAPH
                                (#xB355 . #x780B)	;CJK UNIFIED IDEOGRAPH
                                (#xB356 . #x780E)	;CJK UNIFIED IDEOGRAPH
                                (#xB357 . #x780F)	;CJK UNIFIED IDEOGRAPH
                                (#xB358 . #x7810)	;CJK UNIFIED IDEOGRAPH
                                (#xB359 . #x7813)	;CJK UNIFIED IDEOGRAPH
                                (#xB35A . #x7815)	;CJK UNIFIED IDEOGRAPH
                                (#xB35B . #x7819)	;CJK UNIFIED IDEOGRAPH
                                (#xB35C . #x781B)	;CJK UNIFIED IDEOGRAPH
                                (#xB35D . #x781E)	;CJK UNIFIED IDEOGRAPH
                                (#xB35E . #x7820)	;CJK UNIFIED IDEOGRAPH
                                (#xB35F . #x7821)	;CJK UNIFIED IDEOGRAPH
                                (#xB360 . #x7822)	;CJK UNIFIED IDEOGRAPH
                                (#xB361 . #x7824)	;CJK UNIFIED IDEOGRAPH
                                (#xB362 . #x7828)	;CJK UNIFIED IDEOGRAPH
                                (#xB363 . #x782A)	;CJK UNIFIED IDEOGRAPH
                                (#xB364 . #x782B)	;CJK UNIFIED IDEOGRAPH
                                (#xB365 . #x782E)	;CJK UNIFIED IDEOGRAPH
                                (#xB366 . #x782F)	;CJK UNIFIED IDEOGRAPH
                                (#xB367 . #x7831)	;CJK UNIFIED IDEOGRAPH
                                (#xB368 . #x7832)	;CJK UNIFIED IDEOGRAPH
                                (#xB369 . #x7833)	;CJK UNIFIED IDEOGRAPH
                                (#xB36A . #x7835)	;CJK UNIFIED IDEOGRAPH
                                (#xB36B . #x7836)	;CJK UNIFIED IDEOGRAPH
                                (#xB36C . #x783D)	;CJK UNIFIED IDEOGRAPH
                                (#xB36D . #x783F)	;CJK UNIFIED IDEOGRAPH
                                (#xB36E . #x7841)	;CJK UNIFIED IDEOGRAPH
                                (#xB36F . #x7842)	;CJK UNIFIED IDEOGRAPH
                                (#xB370 . #x7843)	;CJK UNIFIED IDEOGRAPH
                                (#xB371 . #x7844)	;CJK UNIFIED IDEOGRAPH
                                (#xB372 . #x7846)	;CJK UNIFIED IDEOGRAPH
                                (#xB373 . #x7848)	;CJK UNIFIED IDEOGRAPH
                                (#xB374 . #x7849)	;CJK UNIFIED IDEOGRAPH
                                (#xB375 . #x784A)	;CJK UNIFIED IDEOGRAPH
                                (#xB376 . #x784B)	;CJK UNIFIED IDEOGRAPH
                                (#xB377 . #x784D)	;CJK UNIFIED IDEOGRAPH
                                (#xB378 . #x784F)	;CJK UNIFIED IDEOGRAPH
                                (#xB379 . #x7851)	;CJK UNIFIED IDEOGRAPH
                                (#xB37A . #x7853)	;CJK UNIFIED IDEOGRAPH
                                (#xB37B . #x7854)	;CJK UNIFIED IDEOGRAPH
                                (#xB37C . #x7858)	;CJK UNIFIED IDEOGRAPH
                                (#xB37D . #x7859)	;CJK UNIFIED IDEOGRAPH
                                (#xB37E . #x785A)	;CJK UNIFIED IDEOGRAPH
                                (#xB380 . #x785B)	;CJK UNIFIED IDEOGRAPH
                                (#xB381 . #x785C)	;CJK UNIFIED IDEOGRAPH
                                (#xB382 . #x785E)	;CJK UNIFIED IDEOGRAPH
                                (#xB383 . #x785F)	;CJK UNIFIED IDEOGRAPH
                                (#xB384 . #x7860)	;CJK UNIFIED IDEOGRAPH
                                (#xB385 . #x7861)	;CJK UNIFIED IDEOGRAPH
                                (#xB386 . #x7862)	;CJK UNIFIED IDEOGRAPH
                                (#xB387 . #x7863)	;CJK UNIFIED IDEOGRAPH
                                (#xB388 . #x7864)	;CJK UNIFIED IDEOGRAPH
                                (#xB389 . #x7865)	;CJK UNIFIED IDEOGRAPH
                                (#xB38A . #x7866)	;CJK UNIFIED IDEOGRAPH
                                (#xB38B . #x7867)	;CJK UNIFIED IDEOGRAPH
                                (#xB38C . #x7868)	;CJK UNIFIED IDEOGRAPH
                                (#xB38D . #x7869)	;CJK UNIFIED IDEOGRAPH
                                (#xB38E . #x786F)	;CJK UNIFIED IDEOGRAPH
                                (#xB38F . #x7870)	;CJK UNIFIED IDEOGRAPH
                                (#xB390 . #x7871)	;CJK UNIFIED IDEOGRAPH
                                (#xB391 . #x7872)	;CJK UNIFIED IDEOGRAPH
                                (#xB392 . #x7873)	;CJK UNIFIED IDEOGRAPH
                                (#xB393 . #x7874)	;CJK UNIFIED IDEOGRAPH
                                (#xB394 . #x7875)	;CJK UNIFIED IDEOGRAPH
                                (#xB395 . #x7876)	;CJK UNIFIED IDEOGRAPH
                                (#xB396 . #x7878)	;CJK UNIFIED IDEOGRAPH
                                (#xB397 . #x7879)	;CJK UNIFIED IDEOGRAPH
                                (#xB398 . #x787A)	;CJK UNIFIED IDEOGRAPH
                                (#xB399 . #x787B)	;CJK UNIFIED IDEOGRAPH
                                (#xB39A . #x787D)	;CJK UNIFIED IDEOGRAPH
                                (#xB39B . #x787E)	;CJK UNIFIED IDEOGRAPH
                                (#xB39C . #x787F)	;CJK UNIFIED IDEOGRAPH
                                (#xB39D . #x7880)	;CJK UNIFIED IDEOGRAPH
                                (#xB39E . #x7881)	;CJK UNIFIED IDEOGRAPH
                                (#xB39F . #x7882)	;CJK UNIFIED IDEOGRAPH
                                (#xB3A0 . #x7883)	;CJK UNIFIED IDEOGRAPH
                                (#xB3A1 . #x573A)	;CJK UNIFIED IDEOGRAPH
                                (#xB3A2 . #x5C1D)	;CJK UNIFIED IDEOGRAPH
                                (#xB3A3 . #x5E38)	;CJK UNIFIED IDEOGRAPH
                                (#xB3A4 . #x957F)	;CJK UNIFIED IDEOGRAPH
                                (#xB3A5 . #x507F)	;CJK UNIFIED IDEOGRAPH
                                (#xB3A6 . #x80A0)	;CJK UNIFIED IDEOGRAPH
                                (#xB3A7 . #x5382)	;CJK UNIFIED IDEOGRAPH
                                (#xB3A8 . #x655E)	;CJK UNIFIED IDEOGRAPH
                                (#xB3A9 . #x7545)	;CJK UNIFIED IDEOGRAPH
                                (#xB3AA . #x5531)	;CJK UNIFIED IDEOGRAPH
                                (#xB3AB . #x5021)	;CJK UNIFIED IDEOGRAPH
                                (#xB3AC . #x8D85)	;CJK UNIFIED IDEOGRAPH
                                (#xB3AD . #x6284)	;CJK UNIFIED IDEOGRAPH
                                (#xB3AE . #x949E)	;CJK UNIFIED IDEOGRAPH
                                (#xB3AF . #x671D)	;CJK UNIFIED IDEOGRAPH
                                (#xB3B0 . #x5632)	;CJK UNIFIED IDEOGRAPH
                                (#xB3B1 . #x6F6E)	;CJK UNIFIED IDEOGRAPH
                                (#xB3B2 . #x5DE2)	;CJK UNIFIED IDEOGRAPH
                                (#xB3B3 . #x5435)	;CJK UNIFIED IDEOGRAPH
                                (#xB3B4 . #x7092)	;CJK UNIFIED IDEOGRAPH
                                (#xB3B5 . #x8F66)	;CJK UNIFIED IDEOGRAPH
                                (#xB3B6 . #x626F)	;CJK UNIFIED IDEOGRAPH
                                (#xB3B7 . #x64A4)	;CJK UNIFIED IDEOGRAPH
                                (#xB3B8 . #x63A3)	;CJK UNIFIED IDEOGRAPH
                                (#xB3B9 . #x5F7B)	;CJK UNIFIED IDEOGRAPH
                                (#xB3BA . #x6F88)	;CJK UNIFIED IDEOGRAPH
                                (#xB3BB . #x90F4)	;CJK UNIFIED IDEOGRAPH
                                (#xB3BC . #x81E3)	;CJK UNIFIED IDEOGRAPH
                                (#xB3BD . #x8FB0)	;CJK UNIFIED IDEOGRAPH
                                (#xB3BE . #x5C18)	;CJK UNIFIED IDEOGRAPH
                                (#xB3BF . #x6668)	;CJK UNIFIED IDEOGRAPH
                                (#xB3C0 . #x5FF1)	;CJK UNIFIED IDEOGRAPH
                                (#xB3C1 . #x6C89)	;CJK UNIFIED IDEOGRAPH
                                (#xB3C2 . #x9648)	;CJK UNIFIED IDEOGRAPH
                                (#xB3C3 . #x8D81)	;CJK UNIFIED IDEOGRAPH
                                (#xB3C4 . #x886C)	;CJK UNIFIED IDEOGRAPH
                                (#xB3C5 . #x6491)	;CJK UNIFIED IDEOGRAPH
                                (#xB3C6 . #x79F0)	;CJK UNIFIED IDEOGRAPH
                                (#xB3C7 . #x57CE)	;CJK UNIFIED IDEOGRAPH
                                (#xB3C8 . #x6A59)	;CJK UNIFIED IDEOGRAPH
                                (#xB3C9 . #x6210)	;CJK UNIFIED IDEOGRAPH
                                (#xB3CA . #x5448)	;CJK UNIFIED IDEOGRAPH
                                (#xB3CB . #x4E58)	;CJK UNIFIED IDEOGRAPH
                                (#xB3CC . #x7A0B)	;CJK UNIFIED IDEOGRAPH
                                (#xB3CD . #x60E9)	;CJK UNIFIED IDEOGRAPH
                                (#xB3CE . #x6F84)	;CJK UNIFIED IDEOGRAPH
                                (#xB3CF . #x8BDA)	;CJK UNIFIED IDEOGRAPH
                                (#xB3D0 . #x627F)	;CJK UNIFIED IDEOGRAPH
                                (#xB3D1 . #x901E)	;CJK UNIFIED IDEOGRAPH
                                (#xB3D2 . #x9A8B)	;CJK UNIFIED IDEOGRAPH
                                (#xB3D3 . #x79E4)	;CJK UNIFIED IDEOGRAPH
                                (#xB3D4 . #x5403)	;CJK UNIFIED IDEOGRAPH
                                (#xB3D5 . #x75F4)	;CJK UNIFIED IDEOGRAPH
                                (#xB3D6 . #x6301)	;CJK UNIFIED IDEOGRAPH
                                (#xB3D7 . #x5319)	;CJK UNIFIED IDEOGRAPH
                                (#xB3D8 . #x6C60)	;CJK UNIFIED IDEOGRAPH
                                (#xB3D9 . #x8FDF)	;CJK UNIFIED IDEOGRAPH
                                (#xB3DA . #x5F1B)	;CJK UNIFIED IDEOGRAPH
                                (#xB3DB . #x9A70)	;CJK UNIFIED IDEOGRAPH
                                (#xB3DC . #x803B)	;CJK UNIFIED IDEOGRAPH
                                (#xB3DD . #x9F7F)	;CJK UNIFIED IDEOGRAPH
                                (#xB3DE . #x4F88)	;CJK UNIFIED IDEOGRAPH
                                (#xB3DF . #x5C3A)	;CJK UNIFIED IDEOGRAPH
                                (#xB3E0 . #x8D64)	;CJK UNIFIED IDEOGRAPH
                                (#xB3E1 . #x7FC5)	;CJK UNIFIED IDEOGRAPH
                                (#xB3E2 . #x65A5)	;CJK UNIFIED IDEOGRAPH
                                (#xB3E3 . #x70BD)	;CJK UNIFIED IDEOGRAPH
                                (#xB3E4 . #x5145)	;CJK UNIFIED IDEOGRAPH
                                (#xB3E5 . #x51B2)	;CJK UNIFIED IDEOGRAPH
                                (#xB3E6 . #x866B)	;CJK UNIFIED IDEOGRAPH
                                (#xB3E7 . #x5D07)	;CJK UNIFIED IDEOGRAPH
                                (#xB3E8 . #x5BA0)	;CJK UNIFIED IDEOGRAPH
                                (#xB3E9 . #x62BD)	;CJK UNIFIED IDEOGRAPH
                                (#xB3EA . #x916C)	;CJK UNIFIED IDEOGRAPH
                                (#xB3EB . #x7574)	;CJK UNIFIED IDEOGRAPH
                                (#xB3EC . #x8E0C)	;CJK UNIFIED IDEOGRAPH
                                (#xB3ED . #x7A20)	;CJK UNIFIED IDEOGRAPH
                                (#xB3EE . #x6101)	;CJK UNIFIED IDEOGRAPH
                                (#xB3EF . #x7B79)	;CJK UNIFIED IDEOGRAPH
                                (#xB3F0 . #x4EC7)	;CJK UNIFIED IDEOGRAPH
                                (#xB3F1 . #x7EF8)	;CJK UNIFIED IDEOGRAPH
                                (#xB3F2 . #x7785)	;CJK UNIFIED IDEOGRAPH
                                (#xB3F3 . #x4E11)	;CJK UNIFIED IDEOGRAPH
                                (#xB3F4 . #x81ED)	;CJK UNIFIED IDEOGRAPH
                                (#xB3F5 . #x521D)	;CJK UNIFIED IDEOGRAPH
                                (#xB3F6 . #x51FA)	;CJK UNIFIED IDEOGRAPH
                                (#xB3F7 . #x6A71)	;CJK UNIFIED IDEOGRAPH
                                (#xB3F8 . #x53A8)	;CJK UNIFIED IDEOGRAPH
                                (#xB3F9 . #x8E87)	;CJK UNIFIED IDEOGRAPH
                                (#xB3FA . #x9504)	;CJK UNIFIED IDEOGRAPH
                                (#xB3FB . #x96CF)	;CJK UNIFIED IDEOGRAPH
                                (#xB3FC . #x6EC1)	;CJK UNIFIED IDEOGRAPH
                                (#xB3FD . #x9664)	;CJK UNIFIED IDEOGRAPH
                                (#xB3FE . #x695A)	;CJK UNIFIED IDEOGRAPH
                                (#xB440 . #x7884)	;CJK UNIFIED IDEOGRAPH
                                (#xB441 . #x7885)	;CJK UNIFIED IDEOGRAPH
                                (#xB442 . #x7886)	;CJK UNIFIED IDEOGRAPH
                                (#xB443 . #x7888)	;CJK UNIFIED IDEOGRAPH
                                (#xB444 . #x788A)	;CJK UNIFIED IDEOGRAPH
                                (#xB445 . #x788B)	;CJK UNIFIED IDEOGRAPH
                                (#xB446 . #x788F)	;CJK UNIFIED IDEOGRAPH
                                (#xB447 . #x7890)	;CJK UNIFIED IDEOGRAPH
                                (#xB448 . #x7892)	;CJK UNIFIED IDEOGRAPH
                                (#xB449 . #x7894)	;CJK UNIFIED IDEOGRAPH
                                (#xB44A . #x7895)	;CJK UNIFIED IDEOGRAPH
                                (#xB44B . #x7896)	;CJK UNIFIED IDEOGRAPH
                                (#xB44C . #x7899)	;CJK UNIFIED IDEOGRAPH
                                (#xB44D . #x789D)	;CJK UNIFIED IDEOGRAPH
                                (#xB44E . #x789E)	;CJK UNIFIED IDEOGRAPH
                                (#xB44F . #x78A0)	;CJK UNIFIED IDEOGRAPH
                                (#xB450 . #x78A2)	;CJK UNIFIED IDEOGRAPH
                                (#xB451 . #x78A4)	;CJK UNIFIED IDEOGRAPH
                                (#xB452 . #x78A6)	;CJK UNIFIED IDEOGRAPH
                                (#xB453 . #x78A8)	;CJK UNIFIED IDEOGRAPH
                                (#xB454 . #x78A9)	;CJK UNIFIED IDEOGRAPH
                                (#xB455 . #x78AA)	;CJK UNIFIED IDEOGRAPH
                                (#xB456 . #x78AB)	;CJK UNIFIED IDEOGRAPH
                                (#xB457 . #x78AC)	;CJK UNIFIED IDEOGRAPH
                                (#xB458 . #x78AD)	;CJK UNIFIED IDEOGRAPH
                                (#xB459 . #x78AE)	;CJK UNIFIED IDEOGRAPH
                                (#xB45A . #x78AF)	;CJK UNIFIED IDEOGRAPH
                                (#xB45B . #x78B5)	;CJK UNIFIED IDEOGRAPH
                                (#xB45C . #x78B6)	;CJK UNIFIED IDEOGRAPH
                                (#xB45D . #x78B7)	;CJK UNIFIED IDEOGRAPH
                                (#xB45E . #x78B8)	;CJK UNIFIED IDEOGRAPH
                                (#xB45F . #x78BA)	;CJK UNIFIED IDEOGRAPH
                                (#xB460 . #x78BB)	;CJK UNIFIED IDEOGRAPH
                                (#xB461 . #x78BC)	;CJK UNIFIED IDEOGRAPH
                                (#xB462 . #x78BD)	;CJK UNIFIED IDEOGRAPH
                                (#xB463 . #x78BF)	;CJK UNIFIED IDEOGRAPH
                                (#xB464 . #x78C0)	;CJK UNIFIED IDEOGRAPH
                                (#xB465 . #x78C2)	;CJK UNIFIED IDEOGRAPH
                                (#xB466 . #x78C3)	;CJK UNIFIED IDEOGRAPH
                                (#xB467 . #x78C4)	;CJK UNIFIED IDEOGRAPH
                                (#xB468 . #x78C6)	;CJK UNIFIED IDEOGRAPH
                                (#xB469 . #x78C7)	;CJK UNIFIED IDEOGRAPH
                                (#xB46A . #x78C8)	;CJK UNIFIED IDEOGRAPH
                                (#xB46B . #x78CC)	;CJK UNIFIED IDEOGRAPH
                                (#xB46C . #x78CD)	;CJK UNIFIED IDEOGRAPH
                                (#xB46D . #x78CE)	;CJK UNIFIED IDEOGRAPH
                                (#xB46E . #x78CF)	;CJK UNIFIED IDEOGRAPH
                                (#xB46F . #x78D1)	;CJK UNIFIED IDEOGRAPH
                                (#xB470 . #x78D2)	;CJK UNIFIED IDEOGRAPH
                                (#xB471 . #x78D3)	;CJK UNIFIED IDEOGRAPH
                                (#xB472 . #x78D6)	;CJK UNIFIED IDEOGRAPH
                                (#xB473 . #x78D7)	;CJK UNIFIED IDEOGRAPH
                                (#xB474 . #x78D8)	;CJK UNIFIED IDEOGRAPH
                                (#xB475 . #x78DA)	;CJK UNIFIED IDEOGRAPH
                                (#xB476 . #x78DB)	;CJK UNIFIED IDEOGRAPH
                                (#xB477 . #x78DC)	;CJK UNIFIED IDEOGRAPH
                                (#xB478 . #x78DD)	;CJK UNIFIED IDEOGRAPH
                                (#xB479 . #x78DE)	;CJK UNIFIED IDEOGRAPH
                                (#xB47A . #x78DF)	;CJK UNIFIED IDEOGRAPH
                                (#xB47B . #x78E0)	;CJK UNIFIED IDEOGRAPH
                                (#xB47C . #x78E1)	;CJK UNIFIED IDEOGRAPH
                                (#xB47D . #x78E2)	;CJK UNIFIED IDEOGRAPH
                                (#xB47E . #x78E3)	;CJK UNIFIED IDEOGRAPH
                                (#xB480 . #x78E4)	;CJK UNIFIED IDEOGRAPH
                                (#xB481 . #x78E5)	;CJK UNIFIED IDEOGRAPH
                                (#xB482 . #x78E6)	;CJK UNIFIED IDEOGRAPH
                                (#xB483 . #x78E7)	;CJK UNIFIED IDEOGRAPH
                                (#xB484 . #x78E9)	;CJK UNIFIED IDEOGRAPH
                                (#xB485 . #x78EA)	;CJK UNIFIED IDEOGRAPH
                                (#xB486 . #x78EB)	;CJK UNIFIED IDEOGRAPH
                                (#xB487 . #x78ED)	;CJK UNIFIED IDEOGRAPH
                                (#xB488 . #x78EE)	;CJK UNIFIED IDEOGRAPH
                                (#xB489 . #x78EF)	;CJK UNIFIED IDEOGRAPH
                                (#xB48A . #x78F0)	;CJK UNIFIED IDEOGRAPH
                                (#xB48B . #x78F1)	;CJK UNIFIED IDEOGRAPH
                                (#xB48C . #x78F3)	;CJK UNIFIED IDEOGRAPH
                                (#xB48D . #x78F5)	;CJK UNIFIED IDEOGRAPH
                                (#xB48E . #x78F6)	;CJK UNIFIED IDEOGRAPH
                                (#xB48F . #x78F8)	;CJK UNIFIED IDEOGRAPH
                                (#xB490 . #x78F9)	;CJK UNIFIED IDEOGRAPH
                                (#xB491 . #x78FB)	;CJK UNIFIED IDEOGRAPH
                                (#xB492 . #x78FC)	;CJK UNIFIED IDEOGRAPH
                                (#xB493 . #x78FD)	;CJK UNIFIED IDEOGRAPH
                                (#xB494 . #x78FE)	;CJK UNIFIED IDEOGRAPH
                                (#xB495 . #x78FF)	;CJK UNIFIED IDEOGRAPH
                                (#xB496 . #x7900)	;CJK UNIFIED IDEOGRAPH
                                (#xB497 . #x7902)	;CJK UNIFIED IDEOGRAPH
                                (#xB498 . #x7903)	;CJK UNIFIED IDEOGRAPH
                                (#xB499 . #x7904)	;CJK UNIFIED IDEOGRAPH
                                (#xB49A . #x7906)	;CJK UNIFIED IDEOGRAPH
                                (#xB49B . #x7907)	;CJK UNIFIED IDEOGRAPH
                                (#xB49C . #x7908)	;CJK UNIFIED IDEOGRAPH
                                (#xB49D . #x7909)	;CJK UNIFIED IDEOGRAPH
                                (#xB49E . #x790A)	;CJK UNIFIED IDEOGRAPH
                                (#xB49F . #x790B)	;CJK UNIFIED IDEOGRAPH
                                (#xB4A0 . #x790C)	;CJK UNIFIED IDEOGRAPH
                                (#xB4A1 . #x7840)	;CJK UNIFIED IDEOGRAPH
                                (#xB4A2 . #x50A8)	;CJK UNIFIED IDEOGRAPH
                                (#xB4A3 . #x77D7)	;CJK UNIFIED IDEOGRAPH
                                (#xB4A4 . #x6410)	;CJK UNIFIED IDEOGRAPH
                                (#xB4A5 . #x89E6)	;CJK UNIFIED IDEOGRAPH
                                (#xB4A6 . #x5904)	;CJK UNIFIED IDEOGRAPH
                                (#xB4A7 . #x63E3)	;CJK UNIFIED IDEOGRAPH
                                (#xB4A8 . #x5DDD)	;CJK UNIFIED IDEOGRAPH
                                (#xB4A9 . #x7A7F)	;CJK UNIFIED IDEOGRAPH
                                (#xB4AA . #x693D)	;CJK UNIFIED IDEOGRAPH
                                (#xB4AB . #x4F20)	;CJK UNIFIED IDEOGRAPH
                                (#xB4AC . #x8239)	;CJK UNIFIED IDEOGRAPH
                                (#xB4AD . #x5598)	;CJK UNIFIED IDEOGRAPH
                                (#xB4AE . #x4E32)	;CJK UNIFIED IDEOGRAPH
                                (#xB4AF . #x75AE)	;CJK UNIFIED IDEOGRAPH
                                (#xB4B0 . #x7A97)	;CJK UNIFIED IDEOGRAPH
                                (#xB4B1 . #x5E62)	;CJK UNIFIED IDEOGRAPH
                                (#xB4B2 . #x5E8A)	;CJK UNIFIED IDEOGRAPH
                                (#xB4B3 . #x95EF)	;CJK UNIFIED IDEOGRAPH
                                (#xB4B4 . #x521B)	;CJK UNIFIED IDEOGRAPH
                                (#xB4B5 . #x5439)	;CJK UNIFIED IDEOGRAPH
                                (#xB4B6 . #x708A)	;CJK UNIFIED IDEOGRAPH
                                (#xB4B7 . #x6376)	;CJK UNIFIED IDEOGRAPH
                                (#xB4B8 . #x9524)	;CJK UNIFIED IDEOGRAPH
                                (#xB4B9 . #x5782)	;CJK UNIFIED IDEOGRAPH
                                (#xB4BA . #x6625)	;CJK UNIFIED IDEOGRAPH
                                (#xB4BB . #x693F)	;CJK UNIFIED IDEOGRAPH
                                (#xB4BC . #x9187)	;CJK UNIFIED IDEOGRAPH
                                (#xB4BD . #x5507)	;CJK UNIFIED IDEOGRAPH
                                (#xB4BE . #x6DF3)	;CJK UNIFIED IDEOGRAPH
                                (#xB4BF . #x7EAF)	;CJK UNIFIED IDEOGRAPH
                                (#xB4C0 . #x8822)	;CJK UNIFIED IDEOGRAPH
                                (#xB4C1 . #x6233)	;CJK UNIFIED IDEOGRAPH
                                (#xB4C2 . #x7EF0)	;CJK UNIFIED IDEOGRAPH
                                (#xB4C3 . #x75B5)	;CJK UNIFIED IDEOGRAPH
                                (#xB4C4 . #x8328)	;CJK UNIFIED IDEOGRAPH
                                (#xB4C5 . #x78C1)	;CJK UNIFIED IDEOGRAPH
                                (#xB4C6 . #x96CC)	;CJK UNIFIED IDEOGRAPH
                                (#xB4C7 . #x8F9E)	;CJK UNIFIED IDEOGRAPH
                                (#xB4C8 . #x6148)	;CJK UNIFIED IDEOGRAPH
                                (#xB4C9 . #x74F7)	;CJK UNIFIED IDEOGRAPH
                                (#xB4CA . #x8BCD)	;CJK UNIFIED IDEOGRAPH
                                (#xB4CB . #x6B64)	;CJK UNIFIED IDEOGRAPH
                                (#xB4CC . #x523A)	;CJK UNIFIED IDEOGRAPH
                                (#xB4CD . #x8D50)	;CJK UNIFIED IDEOGRAPH
                                (#xB4CE . #x6B21)	;CJK UNIFIED IDEOGRAPH
                                (#xB4CF . #x806A)	;CJK UNIFIED IDEOGRAPH
                                (#xB4D0 . #x8471)	;CJK UNIFIED IDEOGRAPH
                                (#xB4D1 . #x56F1)	;CJK UNIFIED IDEOGRAPH
                                (#xB4D2 . #x5306)	;CJK UNIFIED IDEOGRAPH
                                (#xB4D3 . #x4ECE)	;CJK UNIFIED IDEOGRAPH
                                (#xB4D4 . #x4E1B)	;CJK UNIFIED IDEOGRAPH
                                (#xB4D5 . #x51D1)	;CJK UNIFIED IDEOGRAPH
                                (#xB4D6 . #x7C97)	;CJK UNIFIED IDEOGRAPH
                                (#xB4D7 . #x918B)	;CJK UNIFIED IDEOGRAPH
                                (#xB4D8 . #x7C07)	;CJK UNIFIED IDEOGRAPH
                                (#xB4D9 . #x4FC3)	;CJK UNIFIED IDEOGRAPH
                                (#xB4DA . #x8E7F)	;CJK UNIFIED IDEOGRAPH
                                (#xB4DB . #x7BE1)	;CJK UNIFIED IDEOGRAPH
                                (#xB4DC . #x7A9C)	;CJK UNIFIED IDEOGRAPH
                                (#xB4DD . #x6467)	;CJK UNIFIED IDEOGRAPH
                                (#xB4DE . #x5D14)	;CJK UNIFIED IDEOGRAPH
                                (#xB4DF . #x50AC)	;CJK UNIFIED IDEOGRAPH
                                (#xB4E0 . #x8106)	;CJK UNIFIED IDEOGRAPH
                                (#xB4E1 . #x7601)	;CJK UNIFIED IDEOGRAPH
                                (#xB4E2 . #x7CB9)	;CJK UNIFIED IDEOGRAPH
                                (#xB4E3 . #x6DEC)	;CJK UNIFIED IDEOGRAPH
                                (#xB4E4 . #x7FE0)	;CJK UNIFIED IDEOGRAPH
                                (#xB4E5 . #x6751)	;CJK UNIFIED IDEOGRAPH
                                (#xB4E6 . #x5B58)	;CJK UNIFIED IDEOGRAPH
                                (#xB4E7 . #x5BF8)	;CJK UNIFIED IDEOGRAPH
                                (#xB4E8 . #x78CB)	;CJK UNIFIED IDEOGRAPH
                                (#xB4E9 . #x64AE)	;CJK UNIFIED IDEOGRAPH
                                (#xB4EA . #x6413)	;CJK UNIFIED IDEOGRAPH
                                (#xB4EB . #x63AA)	;CJK UNIFIED IDEOGRAPH
                                (#xB4EC . #x632B)	;CJK UNIFIED IDEOGRAPH
                                (#xB4ED . #x9519)	;CJK UNIFIED IDEOGRAPH
                                (#xB4EE . #x642D)	;CJK UNIFIED IDEOGRAPH
                                (#xB4EF . #x8FBE)	;CJK UNIFIED IDEOGRAPH
                                (#xB4F0 . #x7B54)	;CJK UNIFIED IDEOGRAPH
                                (#xB4F1 . #x7629)	;CJK UNIFIED IDEOGRAPH
                                (#xB4F2 . #x6253)	;CJK UNIFIED IDEOGRAPH
                                (#xB4F3 . #x5927)	;CJK UNIFIED IDEOGRAPH
                                (#xB4F4 . #x5446)	;CJK UNIFIED IDEOGRAPH
                                (#xB4F5 . #x6B79)	;CJK UNIFIED IDEOGRAPH
                                (#xB4F6 . #x50A3)	;CJK UNIFIED IDEOGRAPH
                                (#xB4F7 . #x6234)	;CJK UNIFIED IDEOGRAPH
                                (#xB4F8 . #x5E26)	;CJK UNIFIED IDEOGRAPH
                                (#xB4F9 . #x6B86)	;CJK UNIFIED IDEOGRAPH
                                (#xB4FA . #x4EE3)	;CJK UNIFIED IDEOGRAPH
                                (#xB4FB . #x8D37)	;CJK UNIFIED IDEOGRAPH
                                (#xB4FC . #x888B)	;CJK UNIFIED IDEOGRAPH
                                (#xB4FD . #x5F85)	;CJK UNIFIED IDEOGRAPH
                                (#xB4FE . #x902E)	;CJK UNIFIED IDEOGRAPH
                                (#xB540 . #x790D)	;CJK UNIFIED IDEOGRAPH
                                (#xB541 . #x790E)	;CJK UNIFIED IDEOGRAPH
                                (#xB542 . #x790F)	;CJK UNIFIED IDEOGRAPH
                                (#xB543 . #x7910)	;CJK UNIFIED IDEOGRAPH
                                (#xB544 . #x7911)	;CJK UNIFIED IDEOGRAPH
                                (#xB545 . #x7912)	;CJK UNIFIED IDEOGRAPH
                                (#xB546 . #x7914)	;CJK UNIFIED IDEOGRAPH
                                (#xB547 . #x7915)	;CJK UNIFIED IDEOGRAPH
                                (#xB548 . #x7916)	;CJK UNIFIED IDEOGRAPH
                                (#xB549 . #x7917)	;CJK UNIFIED IDEOGRAPH
                                (#xB54A . #x7918)	;CJK UNIFIED IDEOGRAPH
                                (#xB54B . #x7919)	;CJK UNIFIED IDEOGRAPH
                                (#xB54C . #x791A)	;CJK UNIFIED IDEOGRAPH
                                (#xB54D . #x791B)	;CJK UNIFIED IDEOGRAPH
                                (#xB54E . #x791C)	;CJK UNIFIED IDEOGRAPH
                                (#xB54F . #x791D)	;CJK UNIFIED IDEOGRAPH
                                (#xB550 . #x791F)	;CJK UNIFIED IDEOGRAPH
                                (#xB551 . #x7920)	;CJK UNIFIED IDEOGRAPH
                                (#xB552 . #x7921)	;CJK UNIFIED IDEOGRAPH
                                (#xB553 . #x7922)	;CJK UNIFIED IDEOGRAPH
                                (#xB554 . #x7923)	;CJK UNIFIED IDEOGRAPH
                                (#xB555 . #x7925)	;CJK UNIFIED IDEOGRAPH
                                (#xB556 . #x7926)	;CJK UNIFIED IDEOGRAPH
                                (#xB557 . #x7927)	;CJK UNIFIED IDEOGRAPH
                                (#xB558 . #x7928)	;CJK UNIFIED IDEOGRAPH
                                (#xB559 . #x7929)	;CJK UNIFIED IDEOGRAPH
                                (#xB55A . #x792A)	;CJK UNIFIED IDEOGRAPH
                                (#xB55B . #x792B)	;CJK UNIFIED IDEOGRAPH
                                (#xB55C . #x792C)	;CJK UNIFIED IDEOGRAPH
                                (#xB55D . #x792D)	;CJK UNIFIED IDEOGRAPH
                                (#xB55E . #x792E)	;CJK UNIFIED IDEOGRAPH
                                (#xB55F . #x792F)	;CJK UNIFIED IDEOGRAPH
                                (#xB560 . #x7930)	;CJK UNIFIED IDEOGRAPH
                                (#xB561 . #x7931)	;CJK UNIFIED IDEOGRAPH
                                (#xB562 . #x7932)	;CJK UNIFIED IDEOGRAPH
                                (#xB563 . #x7933)	;CJK UNIFIED IDEOGRAPH
                                (#xB564 . #x7935)	;CJK UNIFIED IDEOGRAPH
                                (#xB565 . #x7936)	;CJK UNIFIED IDEOGRAPH
                                (#xB566 . #x7937)	;CJK UNIFIED IDEOGRAPH
                                (#xB567 . #x7938)	;CJK UNIFIED IDEOGRAPH
                                (#xB568 . #x7939)	;CJK UNIFIED IDEOGRAPH
                                (#xB569 . #x793D)	;CJK UNIFIED IDEOGRAPH
                                (#xB56A . #x793F)	;CJK UNIFIED IDEOGRAPH
                                (#xB56B . #x7942)	;CJK UNIFIED IDEOGRAPH
                                (#xB56C . #x7943)	;CJK UNIFIED IDEOGRAPH
                                (#xB56D . #x7944)	;CJK UNIFIED IDEOGRAPH
                                (#xB56E . #x7945)	;CJK UNIFIED IDEOGRAPH
                                (#xB56F . #x7947)	;CJK UNIFIED IDEOGRAPH
                                (#xB570 . #x794A)	;CJK UNIFIED IDEOGRAPH
                                (#xB571 . #x794B)	;CJK UNIFIED IDEOGRAPH
                                (#xB572 . #x794C)	;CJK UNIFIED IDEOGRAPH
                                (#xB573 . #x794D)	;CJK UNIFIED IDEOGRAPH
                                (#xB574 . #x794E)	;CJK UNIFIED IDEOGRAPH
                                (#xB575 . #x794F)	;CJK UNIFIED IDEOGRAPH
                                (#xB576 . #x7950)	;CJK UNIFIED IDEOGRAPH
                                (#xB577 . #x7951)	;CJK UNIFIED IDEOGRAPH
                                (#xB578 . #x7952)	;CJK UNIFIED IDEOGRAPH
                                (#xB579 . #x7954)	;CJK UNIFIED IDEOGRAPH
                                (#xB57A . #x7955)	;CJK UNIFIED IDEOGRAPH
                                (#xB57B . #x7958)	;CJK UNIFIED IDEOGRAPH
                                (#xB57C . #x7959)	;CJK UNIFIED IDEOGRAPH
                                (#xB57D . #x7961)	;CJK UNIFIED IDEOGRAPH
                                (#xB57E . #x7963)	;CJK UNIFIED IDEOGRAPH
                                (#xB580 . #x7964)	;CJK UNIFIED IDEOGRAPH
                                (#xB581 . #x7966)	;CJK UNIFIED IDEOGRAPH
                                (#xB582 . #x7969)	;CJK UNIFIED IDEOGRAPH
                                (#xB583 . #x796A)	;CJK UNIFIED IDEOGRAPH
                                (#xB584 . #x796B)	;CJK UNIFIED IDEOGRAPH
                                (#xB585 . #x796C)	;CJK UNIFIED IDEOGRAPH
                                (#xB586 . #x796E)	;CJK UNIFIED IDEOGRAPH
                                (#xB587 . #x7970)	;CJK UNIFIED IDEOGRAPH
                                (#xB588 . #x7971)	;CJK UNIFIED IDEOGRAPH
                                (#xB589 . #x7972)	;CJK UNIFIED IDEOGRAPH
                                (#xB58A . #x7973)	;CJK UNIFIED IDEOGRAPH
                                (#xB58B . #x7974)	;CJK UNIFIED IDEOGRAPH
                                (#xB58C . #x7975)	;CJK UNIFIED IDEOGRAPH
                                (#xB58D . #x7976)	;CJK UNIFIED IDEOGRAPH
                                (#xB58E . #x7979)	;CJK UNIFIED IDEOGRAPH
                                (#xB58F . #x797B)	;CJK UNIFIED IDEOGRAPH
                                (#xB590 . #x797C)	;CJK UNIFIED IDEOGRAPH
                                (#xB591 . #x797D)	;CJK UNIFIED IDEOGRAPH
                                (#xB592 . #x797E)	;CJK UNIFIED IDEOGRAPH
                                (#xB593 . #x797F)	;CJK UNIFIED IDEOGRAPH
                                (#xB594 . #x7982)	;CJK UNIFIED IDEOGRAPH
                                (#xB595 . #x7983)	;CJK UNIFIED IDEOGRAPH
                                (#xB596 . #x7986)	;CJK UNIFIED IDEOGRAPH
                                (#xB597 . #x7987)	;CJK UNIFIED IDEOGRAPH
                                (#xB598 . #x7988)	;CJK UNIFIED IDEOGRAPH
                                (#xB599 . #x7989)	;CJK UNIFIED IDEOGRAPH
                                (#xB59A . #x798B)	;CJK UNIFIED IDEOGRAPH
                                (#xB59B . #x798C)	;CJK UNIFIED IDEOGRAPH
                                (#xB59C . #x798D)	;CJK UNIFIED IDEOGRAPH
                                (#xB59D . #x798E