/*
 * builtin-trace.c
 *
 * Builtin 'trace' command:
 *
 * Display a continuously updated trace of any workload, CPU, specific PID,
 * system wide, etc.  Default format is loosely strace like, but any other
 * event may be specified using --event.
 *
 * Copyright (C) 2012, 2013, 2014, 2015 Red Hat Inc, Arnaldo Carvalho de Melo <acme@redhat.com>
 *
 * Initially based on the 'trace' prototype by Thomas Gleixner:
 *
 * http://lwn.net/Articles/415728/ ("Announcing a new utility: 'trace'")
 *
 * Released under the GPL v2. (and only v2, not any later version)
 */

#include <traceevent/event-parse.h>
#include <api/fs/tracing_path.h>
#include "builtin.h"
#include "util/cgroup.h"
#include "util/color.h"
#include "util/debug.h"
#include "util/env.h"
#include "util/event.h"
#include "util/evlist.h"
#include <subcmd/exec-cmd.h>
#include "util/machine.h"
#include "util/path.h"
#include "util/session.h"
#include "util/thread.h"
#include <subcmd/parse-options.h>
#include "util/strlist.h"
#include "util/intlist.h"
#include "util/thread_map.h"
#include "util/stat.h"
#include "trace/beauty/beauty.h"
#include "trace-event.h"
#include "util/parse-events.h"
#include "util/bpf-loader.h"
#include "callchain.h"
#include "print_binary.h"
#include "string2.h"
#include "syscalltbl.h"
#include "rb_resort.h"

#include <errno.h>
#include <inttypes.h>
#include <poll.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <linux/err.h>
#include <linux/filter.h>
#include <linux/kernel.h>
#include <linux/random.h>
#include <linux/stringify.h>
#include <linux/time64.h>
#include <fcntl.h>

#include "sane_ctype.h"

#ifndef O_CLOEXEC
# define O_CLOEXEC		02000000
#endif

#ifndef F_LINUX_SPECIFIC_BASE
# define F_LINUX_SPECIFIC_BASE	1024
#endif

struct trace {
	struct perf_tool	tool;
	struct syscalltbl	*sctbl;
	struct {
		int		max;
		struct syscall  *table;
		struct {
			struct perf_evsel *sys_enter,
					  *sys_exit,
					  *augmented;
		}		events;
	} syscalls;
	struct record_opts	opts;
	struct perf_evlist	*evlist;
	struct machine		*host;
	struct thread		*current;
	struct cgroup		*cgroup;
	u64			base_time;
	FILE			*output;
	unsigned long		nr_events;
	unsigned long		nr_events_printed;
	unsigned long		max_events;
	struct strlist		*ev_qualifier;
	struct {
		size_t		nr;
		int		*entries;
	}			ev_qualifier_ids;
	struct {
		size_t		nr;
		pid_t		*entries;
	}			filter_pids;
	double			duration_filter;
	double			runtime_ms;
	struct {
		u64		vfs_getname,
				proc_getname;
	} stats;
	unsigned int		max_stack;
	unsigned int		min_stack;
	bool			raw_augmented_syscalls;
	bool			not_ev_qualifier;
	bool			live;
	bool			full_time;
	bool			sched;
	bool			multiple_threads;
	bool			summary;
	bool			summary_only;
	bool			failure_only;
	bool			show_comm;
	bool			print_sample;
	bool			show_tool_stats;
	bool			trace_syscalls;
	bool			kernel_syscallchains;
	bool			force;
	bool			vfs_getname;
	int			trace_pgfaults;
};

struct tp_field {
	int offset;
	union {
		u64 (*integer)(struct tp_field *field, struct perf_sample *sample);
		void *(*pointer)(struct tp_field *field, struct perf_sample *sample);
	};
};

#define TP_UINT_FIELD(bits) \
static u64 tp_field__u##bits(struct tp_field *field, struct perf_sample *sample) \
{ \
	u##bits value; \
	memcpy(&value, sample->raw_data + field->offset, sizeof(value)); \
	return value;  \
}

TP_UINT_FIELD(8);
TP_UINT_FIELD(16);
TP_UINT_FIELD(32);
TP_UINT_FIELD(64);

#define TP_UINT_FIELD__SWAPPED(bits) \
static u64 tp_field__swapped_u##bits(struct tp_field *field, struct perf_sample *sample) \
{ \
	u##bits value; \
	memcpy(&value, sample->raw_data + field->offset, sizeof(value)); \
	return bswap_##bits(value);\
}

TP_UINT_FIELD__SWAPPED(16);
TP_UINT_FIELD__SWAPPED(32);
TP_UINT_FIELD__SWAPPED(64);

static int __tp_field__init_uint(struct tp_field *field, int size, int offset, bool needs_swap)
{
	field->offset = offset;

	switch (size) {
	case 1:
		field->integer = tp_field__u8;
		break;
	case 2:
		field->integer = needs_swap ? tp_field__swapped_u16 : tp_field__u16;
		break;
	case 4:
		field->integer = needs_swap ? tp_field__swapped_u32 : tp_field__u32;
		break;
	case 8:
		field->integer = needs_swap ? tp_field__swapped_u64 : tp_field__u64;
		break;
	default:
		return -1;
	}

	return 0;
}

static int tp_field__init_uint(struct tp_field *field, struct tep_format_field *format_field, bool needs_swap)
{
	return __tp_field__init_uint(field, format_field->size, format_field->offset, needs_swap);
}

static void *tp_field__ptr(struct tp_field *field, struct perf_sample *sample)
{
	return sample->raw_data + field->offset;
}

static int __tp_field__init_ptr(struct tp_field *field, int offset)
{
	field->offset = offset;
	field->pointer = tp_field__ptr;
	return 0;
}

static int tp_field__init_ptr(struct tp_field *field, struct tep_format_field *format_field)
{
	return __tp_field__init_ptr(field, format_field->offset);
}

struct syscall_tp {
	struct tp_field id;
	union {
		struct tp_field args, ret;
	};
};

static int perf_evsel__init_tp_uint_field(struct perf_evsel *evsel,
					  struct tp_field *field,
					  const char *name)
{
	struct tep_format_field *format_field = perf_evsel__field(evsel, name);

	if (format_field == NULL)
		return -1;

	return tp_field__init_uint(field, format_field, evsel->needs_swap);
}

#define perf_evsel__init_sc_tp_uint_field(evsel, name) \
	({ struct syscall_tp *sc = evsel->priv;\
	   perf_evsel__init_tp_uint_field(evsel, &sc->name, #name); })

static int perf_evsel__init_tp_ptr_field(struct perf_evsel *evsel,
					 struct tp_field *field,
					 const char *name)
{
	struct tep_format_field *format_field = perf_evsel__field(evsel, name);

	if (format_field == NULL)
		return -1;

	return tp_field__init_ptr(field, format_field);
}

#define perf_evsel__init_sc_tp_ptr_field(evsel, name) \
	({ struct syscall_tp *sc = evsel->priv;\
	   perf_evsel__init_tp_ptr_field(evsel, &sc->name, #name); })

static void perf_evsel__delete_priv(struct perf_evsel *evsel)
{
	zfree(&evsel->priv);
	perf_evsel__delete(evsel);
}

static int perf_evsel__init_syscall_tp(struct perf_evsel *evsel)
{
	struct syscall_tp *sc = evsel->priv = malloc(sizeof(struct syscall_tp));

	if (evsel->priv != NULL) {
		if (perf_evsel__init_tp_uint_field(evsel, &sc->id, "__syscall_nr"))
			goto out_delete;
		return 0;
	}

	return -ENOMEM;
out_delete:
	zfree(&evsel->priv);
	return -ENOENT;
}

static int perf_evsel__init_augmented_syscall_tp(struct perf_evsel *evsel)
{
	struct syscall_tp *sc = evsel->priv = malloc(sizeof(struct syscall_tp));

	if (evsel->priv != NULL) {       /* field, sizeof_field, offsetof_field */
		if (__tp_field__init_uint(&sc->id, sizeof(long), sizeof(long long), evsel->needs_swap))
			goto out_delete;

		return 0;
	}

	return -ENOMEM;
out_delete:
	zfree(&evsel->priv);
	return -EINVAL;
}

static int perf_evsel__init_augmented_syscall_tp_args(struct perf_evsel *evsel)
{
	struct syscall_tp *sc = evsel->priv;

	return __tp_field__init_ptr(&sc->args, sc->id.offset + sizeof(u64));
}

static int perf_evsel__init_augmented_syscall_tp_ret(struct perf_evsel *evsel)
{
	struct syscall_tp *sc = evsel->priv;

	return __tp_field__init_uint(&sc->ret, sizeof(u64), sc->id.offset + sizeof(u64), evsel->needs_swap);
}

static int perf_evsel__init_raw_syscall_tp(struct perf_evsel *evsel, void *handler)
{
	evsel->priv = malloc(sizeof(struct syscall_tp));
	if (evsel->priv != NULL) {
		if (perf_evsel__init_sc_tp_uint_field(evsel, id))
			goto out_delete;

		evsel->handler = handler;
		return 0;
	}

	return -ENOMEM;

out_delete:
	zfree(&evsel->priv);
	return -ENOENT;
}

static struct perf_evsel *perf_evsel__raw_syscall_newtp(const char *direction, void *handler)
{
	struct perf_evsel *evsel = perf_evsel__newtp("raw_syscalls", direction);

	/* older kernel (e.g., RHEL6) use syscalls:{enter,exit} */
	if (IS_ERR(evsel))
		evsel = perf_evsel__newtp("syscalls", direction);

	if (IS_ERR(evsel))
		return NULL;

	if (perf_evsel__init_raw_syscall_tp(evsel, handler))
		goto out_delete;

	return evsel;

out_delete:
	perf_evsel__delete_priv(evsel);
	return NULL;
}

#define perf_evsel__sc_tp_uint(evsel, name, sample) \
	({ struct syscall_tp *fields = evsel->priv; \
	   fields->name.integer(&fields->name, sample); })

#define perf_evsel__sc_tp_ptr(evsel, name, sample) \
	({ struct syscall_tp *fields = evsel->priv; \
	   fields->name.pointer(&fields->name, sample); })

size_t strarray__scnprintf(struct strarray *sa, char *bf, size_t size, const char *intfmt, int val)
{
	int idx = val - sa->offset;

	if (idx < 0 || idx >= sa->nr_entries || sa->entries[idx] == NULL)
		return scnprintf(bf, size, intfmt, val);

	return scnprintf(bf, size, "%s", sa->entries[idx]);
}

static size_t __syscall_arg__scnprintf_strarray(char *bf, size_t size,
						const char *intfmt,
					        struct syscall_arg *arg)
{
	return strarray__scnprintf(arg->parm, bf, size, intfmt, arg->val);
}

static size_t syscall_arg__scnprintf_strarray(char *bf, size_t size,
					      struct syscall_arg *arg)
{
	return __syscall_arg__scnprintf_strarray(bf, size, "%d", arg);
}

#define SCA_STRARRAY syscall_arg__scnprintf_strarray

struct strarrays {
	int		nr_entries;
	struct strarray **entries;
};

#define DEFINE_STRARRAYS(array) struct strarrays strarrays__##array = { \
	.nr_entries = ARRAY_SIZE(array), \
	.entries = array, \
}

size_t syscall_arg__scnprintf_strarrays(char *bf, size_t size,
					struct syscall_arg *arg)
{
	struct strarrays *sas = arg->parm;
	int i;

	for (i = 0; i < sas->nr_entries; ++i) {
		struct strarray *sa = sas->entries[i];
		int idx = arg->val - sa->offset;

		if (idx >= 0 && idx < sa->nr_entries) {
			if (sa->entries[idx] == NULL)
				break;
			return scnprintf(bf, size, "%s", sa->entries[idx]);
		}
	}

	return scnprintf(bf, size, "%d", arg->val);
}

#ifndef AT_FDCWD
#define AT_FDCWD	-100
#endif

static size_t syscall_arg__scnprintf_fd_at(char *bf, size_t size,
					   struct syscall_arg *arg)
{
	int fd = arg->val;

	if (fd == AT_FDCWD)
		return scnprintf(bf, size, "CWD");

	return syscall_arg__scnprintf_fd(bf, size, arg);
}

#define SCA_FDAT syscall_arg__scnprintf_fd_at

static size_t syscall_arg__scnprintf_close_fd(char *bf, size_t size,
					      struct syscall_arg *arg);

#define SCA_CLOSE_FD syscall_arg__scnprintf_close_fd

size_t syscall_arg__scnprintf_hex(char *bf, size_t size, struct syscall_arg *arg)
{
	return scnprintf(bf, size, "%#lx", arg->val);
}

size_t syscall_arg__scnprintf_int(char *bf, size_t size, struct syscall_arg *arg)
{
	return scnprintf(bf, size, "%d", arg->val);
}

size_t syscall_arg__scnprintf_long(char *bf, size_t size, struct syscall_arg *arg)
{
	return scnprintf(bf, size, "%ld", arg->val);
}

static const char *bpf_cmd[] = {
	"MAP_CREATE", "MAP_LOOKUP_ELEM", "MAP_UPDATE_ELEM", "MAP_DELETE_ELEM",
	"MAP_GET_NEXT_KEY", "PROG_LOAD",
};
static DEFINE_STRARRAY(bpf_cmd);

static const char *epoll_ctl_ops[] = { "ADD", "DEL", "MOD", };
static DEFINE_STRARRAY_OFFSET(epoll_ctl_ops, 1);

static const char *itimers[] = { "REAL", "VIRTUAL", "PROF", };
static DEFINE_STRARRAY(itimers);

static const char *keyctl_options[] = {
	"GET_KEYRING_ID", "JOIN_SESSION_KEYRING", "UPDATE", "REVOKE", "CHOWN",
	"SETPERM", "DESCRIBE", "CLEAR", "LINK", "UNLINK", "SEARCH", "READ",
	"INSTANTIATE", "NEGATE", "SET_REQKEY_KEYRING", "SET_TIMEOUT",
	"ASSUME_AUTHORITY", "GET_SECURITY", "SESSION_TO_PARENT", "REJECT",
	"INSTANTIATE_IOV", "INVALIDATE", "GET_PERSISTENT",
};
static DEFINE_STRARRAY(keyctl_options);

static const char *whences[] = { "SET", "CUR", "END",
#ifdef SEEK_DATA
"DATA",
#endif
#ifdef SEEK_HOLE
"HOLE",
#endif
};
static DEFINE_STRARRAY(whences);

static const char *fcntl_cmds[] = {
	"DUPFD", "GETFD", "SETFD", "GETFL", "SETFL", "GETLK", "SETLK",
	"SETLKW", "SETOWN", "GETOWN", "SETSIG", "GETSIG", "GETLK64",
	"SETLK64", "SETLKW64", "SETOWN_EX", "GETOWN_EX",
	"GETOWNER_UIDS",
};
static DEFINE_STRARRAY(fcntl_cmds);

static const char *fcntl_linux_specific_cmds[] = {
	"SETLEASE", "GETLEASE", "NOTIFY", [5] =	"CANCELLK", "DUPFD_CLOEXEC",
	"SETPIPE_SZ", "GETPIPE_SZ", "ADD_SEALS", "GET_SEALS",
	"GET_RW_HINT", "SET_RW_HINT", "GET_FILE_RW_HINT", "SET_FILE_RW_HINT",
};

static DEFINE_STRARRAY_OFFSET(fcntl_linux_specific_cmds, F_LINUX_SPECIFIC_BASE);

static struct strarray *fcntl_cmds_arrays[] = {
	&strarray__fcntl_cmds,
	&strarray__fcntl_linux_specific_cmds,
};

static DEFINE_STRARRAYS(fcntl_cmds_arrays);

static const char *rlimit_resources[] = {
	"CPU", "FSIZE", "DATA", "STACK", "CORE", "RSS", "NPROC", "NOFILE",
	"MEMLOCK", "AS", "LOCKS", "SIGPENDING", "MSGQUEUE", "NICE", "RTPRIO",
	"RTTIME",
};
static DEFINE_STRARRAY(rlimit_resources);

static const char *sighow[] = { "BLOCK", "UNBLOCK", "SETMASK", };
static DEFINE_STRARRAY(sighow);

static const char *clockid[] = {
	"REALTIME", "MONOTONIC", "PROCESS_CPUTIME_ID", "THREAD_CPUTIME_ID",
	"MONOTONIC_RAW", "REALTIME_COARSE", "MONOTONIC_COARSE", "BOOTTIME",
	"REALTIME_ALARM", "BOOTTIME_ALARM", "SGI_CYCLE", "TAI"
};
static DEFINE_STRARRAY(clockid);

static size_t syscall_arg__scnprintf_access_mode(char *bf, size_t size,
						 struct syscall_arg *arg)
{
	size_t printed = 0;
	int mode = arg->val;

	if (mode == F_OK) /* 0 */
		return scnprintf(bf, size, "F");
#define	P_MODE(n) \
	if (mode & n##_OK) { \
		printed += scnprintf(bf + printed, size - printed, "%s", #n); \
		mode &= ~n##_OK; \
	}

	P_MODE(R);
	P_MODE(W);
	P_MODE(X);
#undef P_MODE

	if (mode)
		printed += scnprintf(bf + printed, size - printed, "|%#x", mode);

	return printed;
}

#define SCA_ACCMODE syscall_arg__scnprintf_access_mode

static size_t syscall_arg__scnprintf_filename(char *bf, size_t size,
					      struct syscall_arg *arg);

#define SCA_FILENAME syscall_arg__scnprintf_filename

static size_t syscall_arg__scnprintf_pipe_flags(char *bf, size_t size,
						struct syscall_arg *arg)
{
	int printed = 0, flags = arg->val;

#define	P_FLAG(n) \
	if (flags & O_##n) { \
		printed += scnprintf(bf + printed, size - printed, "%s%s", printed ? "|" : "", #n); \
		flags &= ~O_##n; \
	}

	P_FLAG(CLOEXEC);
	P_FLAG(NONBLOCK);
#undef P_FLAG

	if (flags)
		printed += scnprintf(bf + printed, size - printed, "%s%#x", printed ? "|" : "", flags);

	return printed;
}

#define SCA_PIPE_FLAGS syscall_arg__scnprintf_pipe_flags

#ifndef GRND_NONBLOCK
#define GRND_NONBLOCK	0x0001
#endif
#ifndef GRND_RANDOM
#define GRND_RANDOM	0x0002
#endif

static size_t syscall_arg__scnprintf_getrandom_flags(char *bf, size_t size,
						   struct syscall_arg *arg)
{
	int printed = 0, flags = arg->val;

#define	P_FLAG(n) \
	if (flags & GRND_##n) { \
		printed += scnprintf(bf + printed, size - printed, "%s%s", printed ? "|" : "", #n); \
		flags &= ~GRND_##n; \
	}

	P_FLAG(RANDOM);
	P_FLAG(NONBLOCK);
#undef P_FLAG

	if (flags)
		printed += scnprintf(bf + printed, size - printed, "%s%#x", printed ? "|" : "", flags);

	return printed;
}

#define SCA_GETRANDOM_FLAGS syscall_arg__scnprintf_getrandom_flags

#define STRARRAY(name, array) \
	  { .scnprintf	= SCA_STRARRAY, \
	    .parm	= &strarray__##array, }

#include "trace/beauty/arch_errno_names.c"
#include "trace/beauty/eventfd.c"
#include "trace/beauty/futex_op.c"
#include "trace/beauty/futex_val3.c"
#include "trace/beauty/mmap.c"
#include "trace/beauty/mode_t.c"
#include "trace/beauty/msg_flags.c"
#include "trace/beauty/open_flags.c"
#include "trace/beauty/perf_event_open.c"
#include "trace/beauty/pid.c"
#include "trace/beauty/sched_policy.c"
#include "trace/beauty/seccomp.c"
#include "trace/beauty/signum.c"
#include "trace/beauty/socket_type.c"
#include "trace/beauty/waitid_options.c"

struct syscall_arg_fmt {
	size_t	   (*scnprintf)(char *bf, size_t size, struct syscall_arg *arg);
	unsigned long (*mask_val)(struct syscall_arg *arg, unsigned long val);
	void	   *parm;
	const char *name;
	bool	   show_zero;
};

static struct syscall_fmt {
	const char *name;
	const char *alias;
	struct syscall_arg_fmt arg[6];
	u8	   nr_args;
	bool	   errpid;
	bool	   timeout;
	bool	   hexret;
} syscall_fmts[] = {
	{ .name	    = "access",
	  .arg = { [1] = { .scnprintf = SCA_ACCMODE,  /* mode */ }, }, },
	{ .name	    = "bind",
	  .arg = { [1] = { .scnprintf = SCA_SOCKADDR, /* umyaddr */ }, }, },
	{ .name	    = "bpf",
	  .arg = { [0] = STRARRAY(cmd, bpf_cmd), }, },
	{ .name	    = "brk",	    .hexret = true,
	  .arg = { [0] = { .scnprintf = SCA_HEX, /* brk */ }, }, },
	{ .name     = "clock_gettime",
	  .arg = { [0] = STRARRAY(clk_id, clockid), }, },
	{ .name	    = "clone",	    .errpid = true, .nr_args = 5,
	  .arg = { [0] = { .name = "flags",	    .scnprintf = SCA_CLONE_FLAGS, },
		   [1] = { .name = "child_stack",   .scnprintf = SCA_HEX, },
		   [2] = { .name = "parent_tidptr", .scnprintf = SCA_HEX, },
		   [3] = { .name = "child_tidptr",  .scnprintf = SCA_HEX, },
		   [4] = { .name = "tls",	    .scnprintf = SCA_HEX, }, }, },
	{ .name	    = "close",
	  .arg = { [0] = { .scnprintf = SCA_CLOSE_FD, /* fd */ }, }, },
	{ .name	    = "connect",
	  .arg = { [1] = { .scnprintf = SCA_SOCKADDR, /* servaddr */ }, }, },
	{ .name	    = "epoll_ctl",
	  .arg = { [1] = STRARRAY(op, epoll_ctl_ops), }, },
	{ .name	    = "eventfd2",
	  .arg = { [1] = { .scnprintf = SCA_EFD_FLAGS, /* flags */ }, }, },
	{ .name	    = "fchmodat",
	  .arg = { [0] = { .scnprintf = SCA_FDAT, /* fd */ }, }, },
	{ .name	    = "fchownat",
	  .arg = { [0] = { .scnprintf = SCA_FDAT, /* fd */ }, }, },
	{ .name	    = "fcntl",
	  .arg = { [1] = { .scnprintf = SCA_FCNTL_CMD, /* cmd */
			   .parm      = &strarrays__fcntl_cmds_arrays,
			   .show_zero = true, },
		   [2] = { .scnprintf =  SCA_FCNTL_ARG, /* arg */ }, }, },
	{ .name	    = "flock",
	  .arg = { [1] = { .scnprintf = SCA_FLOCK, /* cmd */ }, }, },
	{ .name	    = "fstat", .alias = "newfstat", },
	{ .name	    = "fstatat", .alias = "newfstatat", },
	{ .name	    = "futex",
	  .arg = { [1] = { .scnprintf = SCA_FUTEX_OP, /* op */ },
		   [5] = { .scnprintf = SCA_FUTEX_VAL3, /* val3 */ }, }, },
	{ .name	    = "futimesat",
	  .arg = { [0] = { .scnprintf = SCA_FDAT, /* fd */ }, }, },
	{ .name	    = "getitimer",
	  .arg = { [0] = STRARRAY(which, itimers), }, },
	{ .name	    = "getpid",	    .errpid = true, },
	{ .name	    = "getpgid",    .errpid = true, },
	{ .name	    = "getppid",    .errpid = true, },
	{ .name	    = "getrandom",
	  .arg = { [2] = { .scnprintf = SCA_GETRANDOM_FLAGS, /* flags */ }, }, },
	{ .name	    = "getrlimit",
	  .arg = { [0] = STRARRAY(resource, rlimit_resources), }, },
	{ .name	    = "gettid",	    .errpid = true, },
	{ .name	    = "ioctl",
	  .arg = {
#if defined(__i386__) || defined(__x86_64__)
/*
 * FIXME: Make this available to all arches.
 */
		   [1] = { .scnprintf = SCA_IOCTL_CMD, /* cmd */ },
		   [2] = { .scnprintf = SCA_HEX, /* arg */ }, }, },
#else
		   [2] = { .scnprintf = SCA_HEX, /* arg */ }, }, },
#endif
	{ .name	    = "kcmp",	    .nr_args = 5,
	  .arg = { [0] = { .name = "pid1",	.scnprintf = SCA_PID, },
		   [1] = { .name = "pid2",	.scnprintf = SCA_PID, },
		   [2] = { .name = "type",	.scnprintf = SCA_KCMP_TYPE, },
		   [3] = { .name = "idx1",	.scnprintf = SCA_KCMP_IDX, },
		   [4] = { .name = "idx2",	.scnprintf = SCA_KCMP_IDX, }, }, },
	{ .name	    = "keyctl",
	  .arg = { [0] = STRARRAY(option, keyctl_options), }, },
	{ .name	    = "kill",
	  .arg = { [1] = { .scnprintf = SCA_SIGNUM, /* sig */ }, }, },
	{ .name	    = "linkat",
	  .arg = { [0] = { .scnprintf = SCA_FDAT, /* fd */ }, }, },
	{ .name	    = "lseek",
	  .arg = { [2] = STRARRAY(whence, whences), }, },
	{ .name	    = "lstat", .alias = "newlstat", },
	{ .name     = "madvise",
	  .arg = { [0] = { .scnprintf = SCA_HEX,      /* start */ },
		   [2] = { .scnprintf = SCA_MADV_BHV, /* behavior */ }, }, },
	{ .name	    = "mkdirat",
	  .arg = { [0] = { .scnprintf = SCA_FDAT, /* fd */ }, }, },
	{ .name	    = "mknodat",
	  .arg = { [0] = { .scnprintf = SCA_FDAT, /* fd */ }, }, },
	{ .name	    = "mlock",
	  .arg = { [0] = { .scnprintf = SCA_HEX, /* addr */ }, }, },
	{ .name	    = "mlockall",
	  .arg = { [0] = { .scnprintf = SCA_HEX, /* addr */ }, }, },
	{ .name	    = "mmap",	    .hexret = true,
/* The standard mmap maps to old_mmap on s390x */
#if defined(__s390x__)
	.alias = "old_mmap",
#endif
	  .arg = { [0] = { .scnprintf = SCA_HEX,	/* addr */ },
		   [2] = { .scnprintf = SCA_MMAP_PROT,	/* prot */ },
		   [3] = { .scnprintf = SCA_MMAP_FLAGS,	/* flags */ }, }, },
	{ .name	    = "mount",
	  .arg = { [0] = { .scnprintf = SCA_FILENAME, /* dev_name */ },
		   [3] = { .scnprintf = SCA_MOUNT_FLAGS, /* flags */
			   .mask_val  = SCAMV_MOUNT_FLAGS, /* flags */ }, }, },
	{ .name	    = "mprotect",
	  .arg = { [0] = { .scnprintf = SCA_HEX,	/* start */ },
		   [2] = { .scnprintf = SCA_MMAP_PROT,	/* prot */ }, }, },
	{ .name	    = "mq_unlink",
	  .arg = { [0] = { .scnprintf = SCA_FILENAME, /* u_name */ }, }, },
	{ .name	    = "mremap",	    .hexret = true,
	  .arg = { [0] = { .scnprintf = SCA_HEX,	  /* addr */ },
		   [3] = { .scnprintf = SCA_MREMAP_FLAGS, /* flags */ },
		   [4] = { .scnprintf = SCA_HEX,	  /* new_addr */ }, }, },
	{ .name	    = "munlock",
	  .arg = { [0] = { .scnprintf = SCA_HEX, /* addr */ }, }, },
	{ .name	    = "munmap",
	  .arg = { [0] = { .scnprintf = SCA_HEX, /* addr */ }, }, },
	{ .name	    = "name_to_handle_at",
	  .arg = { [0] = { .scnprintf = SCA_FDAT, /* dfd */ }, }, },
	{ .name	    = "newfstatat",
	  .arg = { [0] = { .scnprintf = SCA_FDAT, /* dfd */ }, }, },
	{ .name	    = "open",
	  .arg = { [1] = { .scnprintf = SCA_OPEN_FLAGS, /* flags */ }, }, },
	{ .name	    = "open_by_handle_at",
	  .arg = { [0] = { .scnprintf = SCA_FDAT,	/* dfd */ },
		   [2] = { .scnprintf = SCA_OPEN_FLAGS, /* flags */ }, }, },
	{ .name	    = "openat",
	  .arg = { [0] = { .scnprintf = SCA_FDAT,	/* dfd */ },
		   [2] = { .scnprintf = SCA_OPEN_FLAGS, /* flags */ }, }, },
	{ .name	    = "perf_event_open",
	  .arg = { [2] = { .scnprintf = SCA_INT,	/* cpu */ },
		   [3] = { .scnprintf = SCA_FD,		/* group_fd */ },
		   [4] = { .scnprintf = SCA_PERF_FLAGS, /* flags */ }, }, },
	{ .name	    = "pipe2",
	  .arg = { [1] = { .scnprintf = SCA_PIPE_FLAGS, /* flags */ }, }, },
	{ .name	    = "pkey_alloc",
	  .arg = { [1] = { .scnprintf = SCA_PKEY_ALLOC_ACCESS_RIGHTS,	/* access_rights */ }, }, },
	{ .name	    = "pkey_free",
	  .arg = { [0] = { .scnprintf = SCA_INT,	/* key */ }, }, },
	{ .name	    = "pkey_mprotect",
	  .arg = { [0] = { .scnprintf = SCA_HEX,	/* start */ },
		   [2] = { .scnprintf = SCA_MMAP_PROT,	/* prot */ },
		   [3] = { .scnprintf = SCA_INT,	/* pkey */ }, }, },
	{ .name	    = "poll", .timeout = true, },
	{ .name	    = "ppoll", .timeout = true, },
	{ .name	    = "prctl", .alias = "arch_prctl",
	  .arg = { [0] = { .scnprintf = SCA_PRCTL_OPTION, /* option */ },
		   [1] = { .scnprintf = SCA_PRCTL_ARG2, /* arg2 */ },
		   [2] = { .scnprintf = SCA_PRCTL_ARG3, /* arg3 */ }, }, },
	{ .name	    = "pread", .alias = "pread64", },
	{ .name	    = "preadv", .alias = "pread", },
	{ .name	    = "prlimit64",
	  .arg = { [1] = STRARRAY(resource, rlimit_resources), }, },
	{ .name	    = "pwrite", .alias = "pwrite64", },
	{ .name	    = "readlinkat",
	  .arg = { [0] = { .scnprintf = SCA_FDAT, /* dfd */ }, }, },
	{ .name	    = "recvfrom",
	  .arg = { [3] = { .scnprintf = SCA_MSG_FLAGS, /* flags */ }, }, },
	{ .name	    = "recvmmsg",
	  .arg = { [3] = { .scnprintf = SCA_MSG_FLAGS, /* flags */ }, }, },
	{ .name	    = "recvmsg",
	  .arg = { [2] = { .scnprintf = SCA_MSG_FLAGS, /* flags */ }, }, },
	{ .name	    = "renameat",
	  .arg = { [0] = { .scnprintf = SCA_FDAT, /* dfd */ }, }, },
	{ .name	    = "rt_sigaction",
	  .arg = { [0] = { .scnprintf = SCA_SIGNUM, /* sig */ }, }, },
	{ .name	    = "rt_sigprocmask",
	  .arg = { [0] = STRARRAY(how, sighow), }, },
	{ .name	    = "rt_sigqueueinfo",
	  .arg = { [1] = { .scnprintf = SCA_SIGNUM, /* sig */ }, }, },
	{ .name	    = "rt_tgsigqueueinfo",
	  .arg = { [2] = { .scnprintf = SCA_SIGNUM, /* sig */ }, }, },
	{ .name	    = "sched_setscheduler",
	  .arg = { [1] = { .scnprintf = SCA_SCHED_POLICY, /* policy */ }, }, },
	{ .name	    = "seccomp",
	  .arg = { [0] = { .scnprintf = SCA_SECCOMP_OP,	   /* op */ },
		   [1] = { .scnprintf = SCA_SECCOMP_FLAGS, /* flags */ }, }, },
	{ .name	    = "select", .timeout = true, },
	{ .name	    = "sendmmsg",
	  .arg = { [3] = { .scnprintf = SCA_MSG_FLAGS, /* flags */ }, }, },
	{ .name	    = "sendmsg",
	  .arg = { [2] = { .scnprintf = SCA_MSG_FLAGS, /* flags */ }, }, },
	{ .name	    = "sendto",
	  .arg = { [3] = { .scnprintf = SCA_MSG_FLAGS, /* flags */ },
		   [4] = { .scnprintf = SCA_SOCKADDR, /* addr */ }, }, },
	{ .name	    = "set_tid_address", .errpid = true, },
	{ .name	    = "setitimer",
	  .arg = { [0] = STRARRAY(which, itimers), }, },
	{ .name	    = "setrlimit",
	  .arg = { [0] = STRARRAY(resource, rlimit_resources), }, },
	{ .name	    = "socket",
	  .arg = { [0] = STRARRAY(family, socket_families),
		   [1] = { .scnprintf = SCA_SK_TYPE, /* type */ },
		   [2] = { .scnprintf = SCA_SK_PROTO, /* protocol */ }, }, },
	{ .name	    = "socketpair",
	  .arg = { [0] = STRARRAY(family, socket_families),
		   [1] = { .scnprintf = SCA_SK_TYPE, /* type */ },
		   [2] = { .scnprintf = SCA_SK_PROTO, /* protocol */ }, }, },
	{ .name	    = "stat", .alias = "newstat", },
	{ .name	    = "statx",
	  .arg = { [0] = { .scnprintf = SCA_FDAT,	 /* fdat */ },
		   [2] = { .scnprintf = SCA_STATX_FLAGS, /* flags */ } ,
		   [3] = { .scnprintf = SCA_STATX_MASK,	 /* mask */ }, }, },
	{ .name	    = "swapoff",
	  .arg = { [0] = { .scnprintf = SCA_FILENAME, /* specialfile */ }, }, },
	{ .name	    = "swapon",
	  .arg = { [0] = { .scnprintf = SCA_FILENAME, /* specialfile */ }, }, },
	{ .name	    = "symlinkat",
	  .arg = { [0] = { .scnprintf = SCA_FDAT, /* dfd */ }, }, },
	{ .name	    = "tgkill",
	  .arg = { [2] = { .scnprintf = SCA_SIGNUM, /* sig */ }, }, },
	{ .name	    = "tkill",
	  .arg = { [1] = { .scnprintf = SCA_SIGNUM, /* sig */ }, }, },
	{ .name     = "umount2", .alias = "umount",
	  .arg = { [0] = { .scnprintf = SCA_FILENAME, /* name */ }, }, },
	{ .name	    = "uname", .alias = "newuname", },
	{ .name	    = "unlinkat",
	  .arg = { [0] = { .scnprintf = SCA_FDAT, /* dfd */ }, }, },
	{ .name	    = "utimensat",
	  .arg = { [0] = { .scnprintf = SCA_FDAT, /* dirfd */ }, }, },
	{ .name	    = "wait4",	    .errpid = true,
	  .arg = { [2] = { .scnprintf = SCA_WAITID_OPTIONS, /* options */ }, }, },
	{ .name	    = "waitid",	    .errpid = true,
	  .arg = { [3] = { .scnprintf = SCA_WAITID_OPTIONS, /* options */ }, }, },
};

static int syscall_fmt__cmp(const void *name, const void *fmtp)
{
	const struct syscall_fmt *fmt = fmtp;
	return strcmp(name, fmt->name);
}

static struct syscall_fmt *syscall_fmt__find(const char *name)
{
	const int nmemb = ARRAY_SIZE(syscall_fmts);
	return bsearch(name, syscall_fmts, nmemb, sizeof(struct syscall_fmt), syscall_fmt__cmp);
}

static struct syscall_fmt *syscall_fmt__find_by_alias(const char *alias)
{
	int i, nmemb = ARRAY_SIZE(syscall_fmts);

	for (i = 0; i < nmemb; ++i) {
		if (syscall_fmts[i].alias && strcmp(syscall_fmts[i].alias, alias) == 0)
			return &syscall_fmts[i];
	}

	return NULL;
}

/*
 * is_exit: is this "exit" or "exit_group"?
 * is_open: is this "open" or "openat"? To associate the fd returned in sys_exit with the pathname in sys_enter.
 * args_size: sum of the sizes of the syscall arguments, anything after that is augmented stuff: pathname for openat, etc.
 */
struct syscall {
	struct tep_event_format *tp_format;
	int		    nr_args;
	int		    args_size;
	bool		    is_exit;
	bool		    is_open;
	struct tep_format_field *args;
	const char	    *name;
	struct syscall_fmt  *fmt;
	struct syscall_arg_fmt *arg_fmt;
};

/*
 * We need to have this 'calculated' boolean because in some cases we really
 * don't know what is the duration of a syscall, for instance, when we start
 * a session and some threads are waiting for a syscall to finish, say 'poll',
 * in which case all we can do is to print "( ? ) for duration and for the
 * start timestamp.
 */
static size_t fprintf_duration(unsigned long t, bool calculated, FILE *fp)
{
	double duration = (double)t / NSEC_PER_MSEC;
	size_t printed = fprintf(fp, "(");

	if (!calculated)
		printed += fprintf(fp, "         ");
	else if (duration >= 1.0)
		printed += color_fprintf(fp, PERF_COLOR_RED, "%6.3f ms", duration);
	else if (duration >= 0.01)
		printed += color_fprintf(fp, PERF_COLOR_YELLOW, "%6.3f ms", duration);
	else
		printed += color_fprintf(fp, PERF_COLOR_NORMAL, "%6.3f ms", duration);
	return printed + fprintf(fp, "): ");
}

/**
 * filename.ptr: The filename char pointer that will be vfs_getname'd
 * filename.entry_str_pos: Where to insert the string translated from
 *                         filename.ptr by the vfs_getname tracepoint/kprobe.
 * ret_scnprintf: syscall args may set this to a different syscall return
 *                formatter, for instance, fcntl may return fds, file flags, etc.
 */
struct thread_trace {
	u64		  entry_time;
	bool		  entry_pending;
	unsigned long	  nr_events;
	unsigned long	  pfmaj, pfmin;
	char		  *entry_str;
	double		  runtime_ms;
	size_t		  (*ret_scnprintf)(char *bf, size_t size, struct syscall_arg *arg);
        struct {
		unsigned long ptr;
		short int     entry_str_pos;
		bool	      pending_open;
		unsigned int  namelen;
		char	      *name;
	} filename;
	struct {
		int	  max;
		char	  **table;
	} paths;

	struct intlist *syscall_stats;
};

static struct thread_trace *thread_trace__new(void)
{
	struct thread_trace *ttrace =  zalloc(sizeof(struct thread_trace));

	if (ttrace)
		ttrace->paths.max = -1;

	ttrace->syscall_stats = intlist__new(NULL);

	return ttrace;
}

static struct thread_trace *thread__trace(struct thread *thread, FILE *fp)
{
	struct thread_trace *ttrace;

	if (thread == NULL)
		goto fail;

	if (thread__priv(thread) == NULL)
		thread__set_priv(thread, thread_trace__new());

	if (thread__priv(thread) == NULL)
		goto fail;

	ttrace = thread__priv(thread);
	++ttrace->nr_events;

	return ttrace;
fail:
	color_fprintf(fp, PERF_COLOR_RED,
		      "WARNING: not enough memory, dropping samples!\n");
	return NULL;
}


void syscall_arg__set_ret_scnprintf(struct syscall_arg *arg,
				    size_t (*ret_scnprintf)(char *bf, size_t size, struct syscall_arg *arg))
{
	struct thread_trace *ttrace = thread__priv(arg->thread);

	ttrace->ret_scnprintf = ret_scnprintf;
}

#define TRACE_PFMAJ		(1 << 0)
#define TRACE_PFMIN		(1 << 1)

static const size_t trace__entry_str_size = 2048;

static int trace__set_fd_pathname(struct thread *thread, int fd, const char *pathname)
{
	struct thread_trace *ttrace = thread__priv(thread);

	if (fd > ttrace->paths.max) {
		char **npath = realloc(ttrace->paths.table, (fd + 1) * sizeof(char *));

		if (npath == NULL)
			return -1;

		if (ttrace->paths.max != -1) {
			memset(npath + ttrace->paths.max + 1, 0,
			       (fd - ttrace->paths.max) * sizeof(char *));
		} else {
			memset(npath, 0, (fd + 1) * sizeof(char *));
		}

		ttrace->paths.table = npath;
		ttrace->paths.max   = fd;
	}

	ttrace->paths.table[fd] = strdup(pathname);

	return ttrace->paths.table[fd] != NULL ? 0 : -1;
}

static int thread__read_fd_path(struct thread *thread, int fd)
{
	char linkname[PATH_MAX], pathname[PATH_MAX];
	struct stat st;
	int ret;

	if (thread->pid_ == thread->tid) {
		scnprintf(linkname, sizeof(linkname),
			  "/proc/%d/fd/%d", thread->pid_, fd);
	} else {
		scnprintf(linkname, sizeof(linkname),
			  "/proc/%d/task/%d/fd/%d", thread->pid_, thread->tid, fd);
	}

	if (lstat(linkname, &st) < 0 || st.st_size + 1 > (off_t)sizeof(pathname))
		return -1;

	ret = readlink(linkname, pathname, sizeof(pathname));

	if (ret < 0 || ret > st.st_size)
		return -1;

	pathname[ret] = '\0';
	return trace__set_fd_pathname(thread, fd, pathname);
}

static const char *thread__fd_path(struct thread *thread, int fd,
				   struct trace *trace)
{
	struct thread_trace *ttrace = thread__priv(thread);

	if (ttrace == NULL)
		return NULL;

	if (fd < 0)
		return NULL;

	if ((fd > ttrace->paths.max || ttrace->paths.table[fd] == NULL)) {
		if (!trace->live)
			return NULL;
		++trace->stats.proc_getname;
		if (thread__read_fd_path(thread, fd))
			return NULL;
	}

	return ttrace->paths.table[fd];
}

size_t syscall_arg__scnprintf_fd(char *bf, size_t size, struct syscall_arg *arg)
{
	int fd = arg->val;
	size_t printed = scnprintf(bf, size, "%d", fd);
	const char *path = thread__fd_path(arg->thread, fd, arg->trace);

	if (path)
		printed += scnprintf(bf + printed, size - printed, "<%s>", path);

	return printed;
}

size_t pid__scnprintf_fd(struct trace *trace, pid_t pid, int fd, char *bf, size_t size)
{
        size_t printed = scnprintf(bf, size, "%d", fd);
	struct thread *thread = machine__find_thread(trace->host, pid, pid);

	if (thread) {
		const char *path = thread__fd_path(thread, fd, trace);

		if (path)
			printed += scnprintf(bf + printed, size - printed, "<%s>", path);

		thread__put(thread);
	}

        return printed;
}

static size_t syscall_arg__scnprintf_close_fd(char *bf, size_t size,
					      struct syscall_arg *arg)
{
	int fd = arg->val;
	size_t printed = syscall_arg__scnprintf_fd(bf, size, arg);
	struct thread_trace *ttrace = thread__priv(arg->thread);

	if (ttrace && fd >= 0 && fd <= ttrace->paths.max)
		zfree(&ttrace->paths.table[fd]);

	return printed;
}

static void thread__set_filename_pos(struct thread *thread, const char *bf,
				     unsigned long ptr)
{
	struct thread_trace *ttrace = thread__priv(thread);

	ttrace->filename.ptr = ptr;
	ttrace->filename.entry_str_pos = bf - ttrace->entry_str;
}

static size_t syscall_arg__scnprintf_augmented_string(struct syscall_arg *arg, char *bf, size_t size)
{
	struct augmented_arg *augmented_arg = arg->augmented.args;

	return scnprintf(bf, size, "%.*s", augmented_arg->size, augmented_arg->value);
}

static size_t syscall_arg__scnprintf_filename(char *bf, size_t size,
					      struct syscall_arg *arg)
{
	unsigned long ptr = arg->val;

	if (arg->augmented.args)
		return syscall_arg__scnprintf_augmented_string(arg, bf, size);

	if (!arg->trace->vfs_getname)
		return scnprintf(bf, size, "%#x", ptr);

	thread__set_filename_pos(arg->thread, bf, ptr);
	return 0;
}

static bool trace__filter_duration(struct trace *trace, double t)
{
	return t < (trace->duration_filter * NSEC_PER_MSEC);
}

static size_t __trace__fprintf_tstamp(struct trace *trace, u64 tstamp, FILE *fp)
{
	double ts = (double)(tstamp - trace->base_time) / NSEC_PER_MSEC;

	return fprintf(fp, "%10.3f ", ts);
}

/*
 * We're handling tstamp=0 as an undefined tstamp, i.e. like when we are
 * using ttrace->entry_time for a thread that receives a sys_exit without
 * first having received a sys_enter ("poll" issued before tracing session
 * starts, lost sys_enter exit due to ring buffer overflow).
 */
static size_t trace__fprintf_tstamp(struct trace *trace, u64 tstamp, FILE *fp)
{
	if (tstamp > 0)
		return __trace__fprintf_tstamp(trace, tstamp, fp);

	return fprintf(fp, "         ? ");
}

static bool done = false;
static bool interrupted = false;

static void sig_handler(int sig)
{
	done = true;
	interrupted = sig == SIGINT;
}

static size_t trace__fprintf_comm_tid(struct trace *trace, struct thread *thread, FILE *fp)
{
	size_t printed = 0;

	if (trace->multiple_threads) {
		if (trace->show_comm)
			printed += fprintf(fp, "%.14s/", thread__comm_str(thread));
		printed += fprintf(fp, "%d ", thread->tid);
	}

	return printed;
}

static size_t trace__fprintf_entry_head(struct trace *trace, struct thread *thread,
					u64 duration, bool duration_calculated, u64 tstamp, FILE *fp)
{
	size_t printed = trace__fprintf_tstamp(trace, tstamp, fp);
	printed += fprintf_duration(duration, duration_calculated, fp);
	return printed + trace__fprintf_comm_tid(trace, thread, fp);
}

static int trace__process_event(struct trace *trace, struct machine *machine,
				union perf_event *event, struct perf_sample *sample)
{
	int ret = 0;

	switch (event->header.type) {
	case PERF_RECORD_LOST:
		color_fprintf(trace->output, PERF_COLOR_RED,
			      "LOST %" PRIu64 " events!\n", event->lost.lost);
		ret = machine__process_lost_event(machine, event, sample);
		break;
	default:
		ret = machine__process_event(machine, event, sample);
		break;
	}

	return ret;
}

static int trace__tool_process(struct perf_tool *tool,
			       union perf_event *event,
			       struct perf_sample *sample,
			       struct machine *machine)
{
	struct trace *trace = container_of(tool, struct trace, tool);
	return trace__process_event(trace, machine, event, sample);
}

static char *trace__machine__resolve_kernel_addr(void *vmachine, unsigned long long *addrp, char **modp)
{
	struct machine *machine = vmachine;

	if (machine->kptr_restrict_warned)
		return NULL;

	if (symbol_conf.kptr_restrict) {
		pr_warning("Kernel address maps (/proc/{kallsyms,modules}) are restricted.\n\n"
			   "Check /proc/sys/kernel/kptr_restrict.\n\n"
			   "Kernel samples will not be resolved.\n");
		machine->kptr_restrict_warned = true;
		return NULL;
	}

	return machine__resolve_kernel_addr(vmachine, addrp, modp);
}

static int trace__symbols_init(struct trace *trace, struct perf_evlist *evlist)
{
	int err = symbol__init(NULL);

	if (err)
		return err;

	trace->host = machine__new_host();
	if (trace->host == NULL)
		return -ENOMEM;

	err = trace_event__register_resolver(trace->host, trace__machine__resolve_kernel_addr);
	if (err < 0)
		goto out;

	err = __machine__synthesize_threads(trace->host, &trace->tool, &trace->opts.target,
					    evlist->threads, trace__tool_process, false,
					    trace->opts.proc_map_timeout, 1);
out:
	if (err)
		symbol__exit();

	return err;
}

static void trace__symbols__exit(struct trace *trace)
{
	machine__exit(trace->host);
	trace->host = NULL;

	symbol__exit();
}

static int syscall__alloc_arg_fmts(struct syscall *sc, int nr_args)
{
	int idx;

	if (nr_args == 6 && sc->fmt && sc->fmt->nr_args != 0)
		nr_args = sc->fmt->nr_args;

	sc->arg_fmt = calloc(nr_args, sizeof(*sc->arg_fmt));
	if (sc->arg_fmt == NULL)
		return -1;

	for (idx = 0; idx < nr_args; ++idx) {
		if (sc->fmt)
			sc->arg_fmt[idx] = sc->fmt->arg[idx];
	}

	sc->nr_args = nr_args;
	return 0;
}

static int syscall__set_arg_fmts(struct syscall *sc)
{
	struct tep_format_field *field, *last_field = NULL;
	int idx = 0, len;

	for (field = sc->args; field; field = field->next, ++idx) {
		last_field = field;

		if (sc->fmt && sc->fmt->arg[idx].scnprintf)
			continue;

		if (strcmp(field->type, "const char *") == 0 &&
			 (strcmp(field->name, "filename") == 0 ||
			  strcmp(field->name, "path") == 0 ||
			  strcmp(field->name, "pathname") == 0))
			sc->arg_fmt[idx].scnprintf = SCA_FILENAME;
		else if (field->flags & TEP_FIELD_IS_POINTER)
			sc->arg_fmt[idx].scnprintf = syscall_arg__scnprintf_hex;
		else if (strcmp(field->type, "pid_t") == 0)
			sc->arg_fmt[idx].scnprintf = SCA_PID;
		else if (strcmp(field->type, "umode_t") == 0)
			sc->arg_fmt[idx].scnprintf = SCA_MODE_T;
		else if ((strcmp(field->type, "int") == 0 ||
			  strcmp(field->type, "unsigned int") == 0 ||
			  strcmp(field->type, "long") == 0) &&
			 (len = strlen(field->name)) >= 2 &&
			 strcmp(field->name + len - 2, "fd") == 0) {
			/*
			 * /sys/kernel/tracing/events/syscalls/sys_enter*
			 * egrep 'field:.*fd;' .../format|sed -r 's/.*field:([a-z ]+) [a-z_]*fd.+/\1/g'|sort|uniq -c
			 * 65 int
			 * 23 unsigned int
			 * 7 unsigned long
			 */
			sc->arg_fmt[idx].scnprintf = SCA_FD;
		}
	}

	if (last_field)
		sc->args_size = last_field->offset + last_field->size;

	return 0;
}

static int trace__read_syscall_info(struct trace *trace, int id)
{
	char tp_name[128];
	struct syscall *sc;
	const char *name = syscalltbl__name(trace->sctbl, id);

	if (name == NULL)
		return -1;

	if (id > trace->syscalls.max) {
		struct syscall *nsyscalls = realloc(trace->syscalls.table, (id + 1) * sizeof(*sc));

		if (nsyscalls == NULL)
			return -1;

		if (trace->syscalls.max != -1) {
			memset(nsyscalls + trace->syscalls.max + 1, 0,
			       (id - trace->syscalls.max) * sizeof(*sc));
		} else {
			memset(nsyscalls, 0, (id + 1) * sizeof(*sc));
		}

		trace->syscalls.table = nsyscalls;
		trace->syscalls.max   = id;
	}

	sc = trace->syscalls.table + id;
	sc->name = name;

	sc->fmt  = syscall_fmt__find(sc->name);

	snprintf(tp_name, sizeof(tp_name), "sys_enter_%s", sc->name);
	sc->tp_format = trace_event__tp_format("syscalls", tp_name);

	if (IS_ERR(sc->tp_format) && sc->fmt && sc->fmt->alias) {
		snprintf(tp_name, sizeof(tp_name), "sys_enter_%s", sc->fmt->alias);
		sc->tp_format = trace_event__tp_format("syscalls", tp_name);
	}

	if (syscall__alloc_arg_fmts(sc, IS_ERR(sc->tp_format) ? 6 : sc->tp_format->format.nr_fields))
		return -1;

	if (IS_ERR(sc->tp_format))
		return -1;

	sc->args = sc->tp_format->format.fields;
	/*
	 * We need to check and discard the first variable '__syscall_nr'
	 * or 'nr' that mean the syscall number. It is needless here.
	 * So drop '__syscall_nr' or 'nr' field but does not exist on older kernels.
	 */
	if (sc->args && (!strcmp(sc->args->name, "__syscall_nr") || !strcmp(sc->args->name, "nr"))) {
		sc->args = sc->args->next;
		--sc->nr_args;
	}

	sc->is_exit = !strcmp(name, "exit_group") || !strcmp(name, "exit");
	sc->is_open = !strcmp(name, "open") || !strcmp(name, "openat");

	return syscall__set_arg_fmts(sc);
}

static int trace__validate_ev_qualifier(struct trace *trace)
{
	int err = 0, i;
	size_t nr_allocated;
	struct str_node *pos;

	trace->ev_qualifier_ids.nr = strlist__nr_entries(trace->ev_qualifier);
	trace->ev_qualifier_ids.entries = malloc(trace->ev_qualifier_ids.nr *
						 sizeof(trace->ev_qualifier_ids.entries[0]));

	if (trace->ev_qualifier_ids.entries == NULL) {
		fputs("Error:\tNot enough memory for allocating events qualifier ids\n",
		       trace->output);
		err = -EINVAL;
		goto out;
	}

	nr_allocated = trace->ev_qualifier_ids.nr;
	i = 0;

	strlist__for_each_entry(pos, trace->ev_qualifier) {
		const char *sc = pos->s;
		int id = syscalltbl__id(trace->sctbl, sc), match_next = -1;

		if (id < 0) {
			id = syscalltbl__strglobmatch_first(trace->sctbl, sc, &match_next);
			if (id >= 0)
				goto matches;

			if (err == 0) {
				fputs("Error:\tInvalid syscall ", trace->output);
				err = -EINVAL;
			} else {
				fputs(", ", trace->output);
			}

			fputs(sc, trace->output);
		}
matches:
		trace->ev_qualifier_ids.entries[i++] = id;
		if (match_next == -1)
			continue;

		while (1) {
			id = syscalltbl__strglobmatch_next(trace->sctbl, sc, &match_next);
			if (id < 0)
				break;
			if (nr_allocated == trace->ev_qualifier_ids.nr) {
				void *entries;

				nr_allocated += 8;
				entries = realloc(trace->ev_qualifier_ids.entries,
						  nr_allocated * sizeof(trace->ev_qualifier_ids.entries[0]));
				if (entries == NULL) {
					err = -ENOMEM;
					fputs("\nError:\t Not enough memory for parsing\n", trace->output);
					goto out_free;
				}
				trace->ev_qualifier_ids.entries = entries;
			}
			trace->ev_qualifier_ids.nr++;
			trace->ev_qualifier_ids.entries[i++] = id;
		}
	}

	if (err < 0) {
		fputs("\nHint:\ttry 'perf list syscalls:sys_enter_*'"
		      "\nHint:\tand: 'man syscalls'\n", trace->output);
out_free:
		zfree(&trace->ev_qualifier_ids.entries);
		trace->ev_qualifier_ids.nr = 0;
	}
out:
	return err;
}

/*
 * args is to be interpreted as a series of longs but we need to handle
 * 8-byte unaligned accesses. args points to raw_data within the event
 * and raw_data is guaranteed to be 8-byte unaligned because it is
 * preceded by raw_size which is a u32. So we need to copy args to a temp
 * variable to read it. Most notably this avoids extended load instructions
 * on unaligned addresses
 */
unsigned long syscall_arg__val(struct syscall_arg *arg, u8 idx)
{
	unsigned long val;
	unsigned char *p = arg->args + sizeof(unsigned long) * idx;

	memcpy(&val, p, sizeof(val));
	return val;
}

static size_t syscall__scnprintf_name(struct syscall *sc, char *bf, size_t size,
				      struct syscall_arg *arg)
{
	if (sc->arg_fmt && sc->arg_fmt[arg->idx].name)
		return scnprintf(bf, size, "%s: ", sc->arg_fmt[arg->idx].name);

	return scnprintf(bf, size, "arg%d: ", arg->idx);
}

/*
 * Check if the value is in fact zero, i.e. mask whatever needs masking, such
 * as mount 'flags' argument that needs ignoring some magic flag, see comment
 * in tools/perf/trace/beauty/mount_flags.c
 */
static unsigned long syscall__mask_val(struct syscall *sc, struct syscall_arg *arg, unsigned long val)
{
	if (sc->arg_fmt && sc->arg_fmt[arg->idx].mask_val)
		return sc->arg_fmt[arg->idx].mask_val(arg, val);

	return val;
}

static size_t syscall__scnprintf_val(struct syscall *sc, char *bf, size_t size,
				     struct syscall_arg *arg, unsigned long val)
{
	if (sc->arg_fmt && sc->arg_fmt[arg->idx].scnprintf) {
		arg->val = val;
		if (sc->arg_fmt[arg->idx].parm)
			arg->parm = sc->arg_fmt[arg->idx].parm;
		return sc->arg_fmt[arg->idx].scnprintf(bf, size, arg);
	}
	return scnprintf(bf, size, "%ld", val);
}

static size_t syscall__scnprintf_args(struct syscall *sc, char *bf, size_t size,
				      unsigned char *args, void *augmented_args, int augmented_args_size,
				      struct trace *trace, struct thread *thread)
{
	size_t printed = 0;
	unsigned long val;
	u8 bit = 1;
	struct syscall_arg arg = {
		.args	= args,
		.augmented = {
			.size = augmented_args_size,
			.args = augmented_args,
		},
		.idx	= 0,
		.mask	= 0,
		.trace  = trace,
		.thread = thread,
	};
	struct thread_trace *ttrace = thread__priv(thread);

	/*
	 * Things like fcntl will set this in its 'cmd' formatter to pick the
	 * right formatter for the return value (an fd? file flags?), which is
	 * not needed for syscalls that always return a given type, say an fd.
	 */
	ttrace->ret_scnprintf = NULL;

	if (sc->args != NULL) {
		struct tep_format_field *field;

		for (field = sc->args; field;
		     field = field->next, ++arg.idx, bit <<= 1) {
			if (arg.mask & bit)
				continue;

			val = syscall_arg__val(&arg, arg.idx);
			/*
			 * Some syscall args need some mask, most don't and
			 * return val untouched.
			 */
			val = syscall__mask_val(sc, &arg, val);

			/*
 			 * Suppress this argument if its value is zero and
 			 * and we don't have a string associated in an
 			 * strarray for it.
 			 */
			if (val == 0 &&
			    !(sc->arg_fmt &&
			      (sc->arg_fmt[arg.idx].show_zero ||
			       sc->arg_fmt[arg.idx].scnprintf == SCA_STRARRAY ||
			       sc->arg_fmt[arg.idx].scnprintf == SCA_STRARRAYS) &&
			      sc->arg_fmt[arg.idx].parm))
				continue;

			printed += scnprintf(bf + printed, size - printed,
					     "%s%s: ", printed ? ", " : "", field->name);
			printed += syscall__scnprintf_val(sc, bf + printed, size - printed, &arg, val);
		}
	} else if (IS_ERR(sc->tp_format)) {
		/*
		 * If we managed to read the tracepoint /format file, then we
		 * may end up not having any args, like with gettid(), so only
		 * print the raw args when we didn't manage to read it.
		 */
		while (arg.idx < sc->nr_args) {
			if (arg.mask & bit)
				goto next_arg;
			val = syscall_arg__val(&arg, arg.idx);
			if (printed)
				printed += scnprintf(bf + printed, size - printed, ", ");
			printed += syscall__scnprintf_name(sc, bf + printed, size - printed, &arg);
			printed += syscall__scnprintf_val(sc, bf + printed, size - printed, &arg, val);
next_arg:
			++arg.idx;
			bit <<= 1;
		}
	}

	return printed;
}

typedef int (*tracepoint_handler)(struct trace *trace, struct perf_evsel *evsel,
				  union perf_event *event,
				  struct perf_sample *sample);

static struct syscall *trace__syscall_info(struct trace *trace,
					   struct perf_evsel *evsel, int id)
{

	if (id < 0) {

		/*
		 * XXX: Noticed on x86_64, reproduced as far back as 3.0.36, haven't tried
		 * before that, leaving at a higher verbosity level till that is
		 * explained. Reproduced with plain ftrace with:
		 *
		 * echo 1 > /t/events/raw_syscalls/sys_exit/enable
		 * grep "NR -1 " /t/trace_pipe
		 *
		 * After generating some load on the machine.
 		 */
		if (verbose > 1) {
			static u64 n;
			fprintf(trace->output, "Invalid syscall %d id, skipping (%s, %" PRIu64 ") ...\n",
				id, perf_evsel__name(evsel), ++n);
		}
		return NULL;
	}

	if ((id > trace->syscalls.max || trace->syscalls.table[id].name == NULL) &&
	    trace__read_syscall_info(trace, id))
		goto out_cant_read;

	if ((id > trace->syscalls.max || trace->syscalls.table[id].name == NULL))
		goto out_cant_read;

	return &trace->syscalls.table[id];

out_cant_read:
	if (verbose > 0) {
		fprintf(trace->output, "Problems reading syscall %d", id);
		if (id <= trace->syscalls.max && trace->syscalls.table[id].name != NULL)
			fprintf(trace->output, "(%s)", trace->syscalls.table[id].name);
		fputs(" information\n", trace->output);
	}
	return NULL;
}

static void thread__update_stats(struct thread_trace *ttrace,
				 int id, struct perf_sample *sample)
{
	struct int_node *inode;
	struct stats *stats;
	u64 duration = 0;

	inode = intlist__findnew(ttrace->syscall_stats, id);
	if (inode == NULL)
		return;

	stats = inode->priv;
	if (stats == NULL) {
		stats = malloc(sizeof(struct stats));
		if (stats == NULL)
			return;
		init_stats(stats);
		inode->priv = stats;
	}

	if (ttrace->entry_time && sample->time > ttrace->entry_time)
		duration = sample->time - ttrace->entry_time;

	update_stats(stats, duration);
}

static int trace__printf_interrupted_entry(struct trace *trace)
{
	struct thread_trace *ttrace;
	size_t printed;

	if (trace->failure_only || trace->current == NULL)
		return 0;

	ttrace = thread__priv(trace->current);

	if (!ttrace->entry_pending)
		return 0;

	printed  = trace__fprintf_entry_head(trace, trace->current, 0, false, ttrace->entry_time, trace->output);
	printed += fprintf(trace->output, "%-70s) ...\n", ttrace->entry_str);
	ttrace->entry_pending = false;

	++trace->nr_events_printed;

	return printed;
}

static int trace__fprintf_sample(struct trace *trace, struct perf_evsel *evsel,
				 struct perf_sample *sample, struct thread *thread)
{
	int printed = 0;

	if (trace->print_sample) {
		double ts = (double)sample->time / NSEC_PER_MSEC;

		printed += fprintf(trace->output, "%22s %10.3f %s %d/%d [%d]\n",
				   perf_evsel__name(evsel), ts,
				   thread__comm_str(thread),
				   sample->pid, sample->tid, sample->cpu);
	}

	return printed;
}

static void *syscall__augmented_args(struct syscall *sc, struct perf_sample *sample, int *augmented_args_size, bool raw_augmented)
{
	void *augmented_args = NULL;
	/*
	 * For now with BPF raw_augmented we hook into raw_syscalls:sys_enter
	 * and there we get all 6 syscall args plus the tracepoint common
	 * fields (sizeof(long)) and the syscall_nr (another long). So we check
	 * if that is the case and if so don't look after the sc->args_size,
	 * but always after the full raw_syscalls:sys_enter payload, which is
	 * fixed.
	 *
	 * We'll revisit this later to pass s->args_size to the BPF augmenter
	 * (now tools/perf/examples/bpf/augmented_raw_syscalls.c, so that it
	 * copies only what we need for each syscall, like what happens when we
	 * use syscalls:sys_enter_NAME, so that we reduce the kernel/userspace
	 * traffic to just what is needed for each syscall.
	 */
	int args_size = raw_augmented ? (8 * (int)sizeof(long)) : sc->args_size;

	*augmented_args_size = sample->raw_size - args_size;
	if (*augmented_args_size > 0)
		augmented_args = sample->raw_data + args_size;

	return augmented_args;
}

static int trace__sys_enter(struct trace *trace, struct perf_evsel *evsel,
			    union perf_event *event __maybe_unused,
			    struct perf_sample *sample)
{
	char *msg;
	void *args;
	size_t printed = 0;
	struct thread *thread;
	int id = perf_evsel__sc_tp_uint(evsel, id, sample), err = -1;
	int augmented_args_size = 0;
	void *augmented_args = NULL;
	struct syscall *sc = trace__syscall_info(trace, evsel, id);
	struct thread_trace *ttrace;

	if (sc == NULL)
		return -1;

	thread = machine__findnew_thread(trace->host, sample->pid, sample->tid);
	ttrace = thread__trace(thread, trace->output);
	if (ttrace == NULL)
		goto out_put;

	trace__fprintf_sample(trace, evsel, sample, thread);

	args = perf_evsel__sc_tp_ptr(evsel, args, sample);

	if (ttrace->entry_str == NULL) {
		ttrace->entry_str = malloc(trace__entry_str_size);
		if (!ttrace->entry_str)
			goto out_put;
	}

	if (!(trace->duration_filter || trace->summary_only || trace->min_stack))
		trace__printf_interrupted_entry(trace);
	/*
	 * If this is raw_syscalls.sys_enter, then it always comes with the 6 possible
	 * arguments, even if the syscall being handled, say "openat", uses only 4 arguments
	 * this breaks syscall__augmented_args() check for augmented args, as we calculate
	 * syscall->args_size using each syscalls:sys_enter_NAME tracefs format file,
	 * so when handling, say the openat syscall, we end up getting 6 args for the
	 * raw_syscalls:sys_enter event, when we expected just 4, we end up mistakenly
	 * thinking that the extra 2 u64 args are the augmented filename, so just check
	 * here and avoid using augmented syscalls when the evsel is the raw_syscalls one.
	 */
	if (evsel != trace->syscalls.events.sys_enter)
		augmented_args = syscall__augmented_args(sc, sample, &augmented_args_size, trace->raw_augmented_syscalls);
	ttrace->entry_time = sample->time;
	msg = ttrace->entry_str;
	printed += scnprintf(msg + printed, trace__entry_str_size - printed, "%s(", sc->name);

	printed += syscall__scnprintf_args(sc, msg + printed, trace__entry_str_size - printed,
					   args, augmented_args, augmented_args_size, trace, thread);

	if (sc->is_exit) {
		if (!(trace->duration_filter || trace->summary_only || trace->failure_only || trace->min_stack)) {
			trace__fprintf_entry_head(trace, thread, 0, false, ttrace->entry_time, trace->output);
			fprintf(trace->output, "%-70s)\n", ttrace->entry_str);
		}
	} else {
		ttrace->entry_pending = true;
		/* See trace__vfs_getname & trace__sys_exit */
		ttrace->filename.pending_open = false;
	}

	if (trace->current != thread) {
		thread__put(trace->current);
		trace->current = thread__get(thread);
	}
	err = 0;
out_put:
	thread__put(thread);
	return err;
}

static int trace__fprintf_sys_enter(struct trace *trace, struct perf_evsel *evsel,
				    struct perf_sample *sample)
{
	struct thread_trace *ttrace;
	struct thread *thread;
	int id = perf_evsel__sc_tp_uint(evsel, id, sample), err = -1;
	struct syscall *sc = trace__syscall_info(trace, evsel, id);
	char msg[1024];
	void *args, *augmented_args = NULL;
	int augmented_args_size;

	if (sc == NULL)
		return -1;

	thread = machine__findnew_thread(trace->host, sample->pid, sample->tid);
	ttrace = thread__trace(thread, trace->output);
	/*
	 * We need to get ttrace just to make sure it is there when syscall__scnprintf_args()
	 * and the rest of the beautifiers accessing it via struct syscall_arg touches it.
	 */
	if (ttrace == NULL)
		goto out_put;

	args = perf_evsel__sc_tp_ptr(evsel, args, sample);
	augmented_args = syscall__augmented_args(sc, sample, &augmented_args_size, trace->raw_augmented_syscalls);
	syscall__scnprintf_args(sc, msg, sizeof(msg), args, augmented_args, augmented_args_size, trace, thread);
	fprintf(trace->output, "%s", msg);
	err = 0;
out_put:
	thread__put(thread);
	return err;
}

static int trace__resolve_callchain(struct trace *trace, struct perf_evsel *evsel,
				    struct perf_sample *sample,
				    struct callchain_cursor *cursor)
{
	struct addr_location al;
	int max_stack = evsel->attr.sample_max_stack ?
			evsel->attr.sample_max_stack :
			trace->max_stack;
	int err;

	if (machine__resolve(trace->host, &al, sample) < 0)
		return -1;

	err = thread__resolve_callchain(al.thread, cursor, evsel, sample, NULL, NULL, max_stack);
	addr_location__put(&al);
	return err;
}

static int trace__fprintf_callchain(struct trace *trace, struct perf_sample *sample)
{
	/* TODO: user-configurable print_opts */
	const unsigned int print_opts = EVSEL__PRINT_SYM |
				        EVSEL__PRINT_DSO |
				        EVSEL__PRINT_UNKNOWN_AS_ADDR;

	return sample__fprintf_callchain(sample, 38, print_opts, &callchain_cursor, trace->output);
}

static const char *errno_to_name(struct perf_evsel *evsel, int err)
{
	struct perf_env *env = perf_evsel__env(evsel);
	const char *arch_name = perf_env__arch(env);

	return arch_syscalls__strerrno(arch_name, err);
}

static int trace__sys_exit(struct trace *trace, struct perf_evsel *evsel,
			   union perf_event *event __maybe_unused,
			   struct perf_sample *sample)
{
	long ret;
	u64 duration = 0;
	bool duration_calculated = false;
	struct thread *thread;
	int id = perf_evsel__sc_tp_uint(evsel, id, sample), err = -1, callchain_ret = 0;
	struct syscall *sc = trace__syscall_info(trace, evsel, id);
	struct thread_trace *ttrace;

	if (sc == NULL)
		return -1;

	thread = machine__findnew_thread(trace->host, sample->pid, sample->tid);
	ttrace = thread__trace(thread, trace->output);
	if (ttrace == NULL)
		goto out_put;

	trace__fprintf_sample(trace, evsel, sample, thread);

	if (trace->summary)
		thread__update_stats(ttrace, id, sample);

	ret = perf_evsel__sc_tp_uint(evsel, ret, sample);

	if (sc->is_open && ret >= 0 && ttrace->filename.pending_open) {
		trace__set_fd_pathname(thread, ret, ttrace->filename.name);
		ttrace->filename.pending_open = false;
		++trace->stats.vfs_getname;
	}

	if (ttrace->entry_time) {
		duration = sample->time - ttrace->entry_time;
		if (trace__filter_duration(trace, duration))
			goto out;
		duration_calculated = true;
	} else if (trace->duration_filter)
		goto out;

	if (sample->callchain) {
		callchain_ret = trace__resolve_callchain(trace, evsel, sample, &callchain_cursor);
		if (callchain_ret == 0) {
			if (callchain_cursor.nr < trace->min_stack)
				goto out;
			callchain_ret = 1;
		}
	}

	if (trace->summary_only || (ret >= 0 && trace->failure_only))
		goto out;

	trace__fprintf_entry_head(trace, thread, duration, duration_calculated, ttrace->entry_time, trace->output);

	if (ttrace->entry_pending) {
		fprintf(trace->output, "%-70s", ttrace->entry_str);
	} else {
		fprintf(trace->output, " ... [");
		color_fprintf(trace->output, PERF_COLOR_YELLOW, "continued");
		fprintf(trace->output, "]: %s()", sc->name);
	}

	if (sc->fmt == NULL) {
		if (ret < 0)
			goto errno_print;
signed_print:
		fprintf(trace->output, ") = %ld", ret);
	} else if (ret < 0) {
errno_print: {
		char bf[STRERR_BUFSIZE];
		const char *emsg = str_error_r(-ret, bf, sizeof(bf)),
			   *e = errno_to_name(evsel, -ret);

		fprintf(trace->output, ") = -1 %s %s", e, emsg);
	}
	} else if (ret == 0 && sc->fmt->timeout)
		fprintf(trace->output, ") = 0 Timeout");
	else if (ttrace->ret_scnprintf) {
		char bf[1024];
		struct syscall_arg arg = {
			.val	= ret,
			.thread	= thread,
			.trace	= trace,
		};
		ttrace->ret_scnprintf(bf, sizeof(bf), &arg);
		ttrace->ret_scnprintf = NULL;
		fprintf(trace->output, ") = %s", bf);
	} else if (sc->fmt->hexret)
		fprintf(trace->output, ") = %#lx", ret);
	else if (sc->fmt->errpid) {
		struct thread *child = machine__find_thread(trace->host, ret, ret);

		if (child != NULL) {
			fprintf(trace->output, ") = %ld", ret);
			if (child->comm_set)
				fprintf(trace->output, " (%s)", thread__comm_str(child));
			thread__put(child);
		}
	} else
		goto signed_print;

	fputc('\n', trace->output);

	/*
	 * We only consider an 'event' for the sake of --max-events a non-filtered
	 * sys_enter + sys_exit and other tracepoint events.
	 */
	if (++trace->nr_events_printed == trace->max_events && trace->max_events != ULONG_MAX)
		interrupted = true;

	if (callchain_ret > 0)
		trace__fprintf_callchain(trace, sample);
	else if (callchain_ret < 0)
		pr_err("Problem processing %s callchain, skipping...\n", perf_evsel__name(evsel));
out:
	ttrace->entry_pending = false;
	err = 0;
out_put:
	thread__put(thread);
	return err;
}

static int trace__vfs_getname(struct trace *trace, struct perf_evsel *evsel,
			      union perf_event *event __maybe_unused,
			      struct perf_sample *sample)
{
	struct thread *thread = machine__findnew_thread(trace->host, sample->pid, sample->tid);
	struct thread_trace *ttrace;
	size_t filename_len, entry_str_len, to_move;
	ssize_t remaining_space;
	char *pos;
	const char *filename = perf_evsel__rawptr(evsel, sample, "pathname");

	if (!thread)
		goto out;

	ttrace = thread__priv(thread);
	if (!ttrace)
		goto out_put;

	filename_len = strlen(filename);
	if (filename_len == 0)
		goto out_put;

	if (ttrace->filename.namelen < filename_len) {
		char *f = realloc(ttrace->filename.name, filename_len + 1);

		if (f == NULL)
			goto out_put;

		ttrace->filename.namelen = filename_len;
		ttrace->filename.name = f;
	}

	strcpy(ttrace->filename.name, filename);
	ttrace->filename.pending_open = true;

	if (!ttrace->filename.ptr)
		goto out_put;

	entry_str_len = strlen(ttrace->entry_str);
	remaining_space = trace__entry_str_size - entry_str_len - 1; /* \0 */
	if (remaining_space <= 0)
		goto out_put;

	if (filename_len > (size_t)remaining_space) {
		filename += filename_len - remaining_space;
		filename_len = remaining_space;
	}

	to_move = entry_str_len - ttrace->filename.entry_str_pos + 1; /* \0 */
	pos = ttrace->entry_str + ttrace->filename.entry_str_pos;
	memmove(pos + filename_len, pos, to_move);
	memcpy(pos, filename, filename_len);

	ttrace->filename.ptr = 0;
	ttrace->filename.entry_str_pos = 0;
out_put:
	thread__put(thread);
out:
	return 0;
}

static int trace__sched_stat_runtime(struct trace *trace, struct perf_evsel *evsel,
				     union perf_event *event __maybe_unused,
				     struct perf_sample *sample)
{
        u64 runtime = perf_evsel__intval(evsel, sample, "runtime");
	double runtime_ms = (double)runtime / NSEC_PER_MSEC;
	struct thread *thread = machine__findnew_thread(trace->host,
							sample->pid,
							sample->tid);
	struct thread_trace *ttrace = thread__trace(thread, trace->output);

	if (ttrace == NULL)
		goto out_dump;

	ttrace->runtime_ms += runtime_ms;
	trace->runtime_ms += runtime_ms;
out_put:
	thread__put(thread);
	return 0;

out_dump:
	fprintf(trace->output, "%s: comm=%s,pid=%u,runtime=%" PRIu64 ",vruntime=%" PRIu64 ")\n",
	       evsel->name,
	       perf_evsel__strval(evsel, sample, "comm"),
	       (pid_t)perf_evsel__intval(evsel, sample, "pid"),
	       runtime,
	       perf_evsel__intval(evsel, sample, "vruntime"));
	goto out_put;
}

static int bpf_output__printer(enum binary_printer_ops op,
			       unsigned int val, void *extra __maybe_unused, FILE *fp)
{
	unsigned char ch = (unsigned char)val;

	switch (op) {
	case BINARY_PRINT_CHAR_DATA:
		return fprintf(fp, "%c", isprint(ch) ? ch : '.');
	case BINARY_PRINT_DATA_BEGIN:
	case BINARY_PRINT_LINE_BEGIN:
	case BINARY_PRINT_ADDR:
	case BINARY_PRINT_NUM_DATA:
	case BINARY_PRINT_NUM_PAD:
	case BINARY_PRINT_SEP:
	case BINARY_PRINT_CHAR_PAD:
	case BINARY_PRINT_LINE_END:
	case BINARY_PRINT_DATA_END:
	default:
		break;
	}

	return 0;
}

static void bpf_output__fprintf(struct trace *trace,
				struct perf_sample *sample)
{
	binary__fprintf(sample->raw_data, sample->raw_size, 8,
			bpf_output__printer, NULL, trace->output);
	++trace->nr_events_printed;
}

static int trace__event_handler(struct trace *trace, struct perf_evsel *evsel,
				union perf_event *event __maybe_unused,
				struct perf_sample *sample)
{
	struct thread *thread;
	int callchain_ret = 0;
	/*
	 * Check if we called perf_evsel__disable(evsel) due to, for instance,
	 * this event's max_events having been hit and this is an entry coming
	 * from the ring buffer that we should discard, since the max events
	 * have already been considered/printed.
	 */
	if (evsel->disabled)
		return 0;

	thread = machine__findnew_thread(trace->host, sample->pid, sample->tid);

	if (sample->callchain) {
		callchain_ret = trace__resolve_callchain(trace, evsel, sample, &callchain_cursor);
		if (callchain_ret == 0) {
			if (callchain_cursor.nr < trace->min_stack)
				goto out;
			callchain_ret = 1;
		}
	}

	trace__printf_interrupted_entry(trace);
	trace__fprintf_tstamp(trace, sample->time, trace->output);

	if (trace->trace_syscalls)
		fprintf(trace->output, "(         ): ");

	if (thread)
		trace__fprintf_comm_tid(trace, thread, trace->output);

	if (evsel == trace->syscalls.events.augmented) {
		int id = perf_evsel__sc_tp_uint(evsel, id, sample);
		struct syscall *sc = trace__syscall_info(trace, evsel, id);

		if (sc) {
			fprintf(trace->output, "%s(", sc->name);
			trace__fprintf_sys_enter(trace, evsel, sample);
			fputc(')', trace->output);
			goto newline;
		}

		/*
		 * XXX: Not having the associated syscall info or not finding/adding
		 * 	the thread should never happen, but if it does...
		 * 	fall thru and print it as a bpf_output event.
		 */
	}

	fprintf(trace->output, "%s:", evsel->name);

	if (perf_evsel__is_bpf_output(evsel)) {
		bpf_output__fprintf(trace, sample);
	} else if (evsel->tp_format) {
		if (strncmp(evsel->tp_format->name, "sys_enter_", 10) ||
		    trace__fprintf_sys_enter(trace, evsel, sample)) {
			event_format__fprintf(evsel->tp_format, sample->cpu,
					      sample->raw_data, sample->raw_size,
					      trace->output);
			++trace->nr_events_printed;

			if (evsel->max_events != ULONG_MAX && ++evsel->nr_events_printed == evsel->max_events) {
				perf_evsel__disable(evsel);
				perf_evsel__close(evsel);
			}
		}
	}

newline:
	fprintf(trace->output, "\n");

	if (callchain_ret > 0)
		trace__fprintf_callchain(trace, sample);
	else if (callchain_ret < 0)
		pr_err("Problem processing %s callchain, skipping...\n", perf_evsel__name(evsel));
out:
	thread__put(thread);
	return 0;
}

static void print_location(FILE *f, struct perf_sample *sample,
			   struct addr_location *al,
			   bool print_dso, bool print_sym)
{

	if ((verbose > 0 || print_dso) && al->map)
		fprintf(f, "%s@", al->map->dso->long_name);

	if ((verbose > 0 || print_sym) && al->sym)
		fprintf(f, "%s+0x%" PRIx64, al->sym->name,
			al->addr - al->sym->start);
	else if (al->map)
		fprintf(f, "0x%" PRIx64, al->addr);
	else
		fprintf(f, "0x%" PRIx64, sample->addr);
}

static int trace__pgfault(struct trace *trace,
			  struct perf_evsel *evsel,
			  union perf_event *event __maybe_unused,
			  struct perf_sample *sample)
{
	struct thread *thread;
	struct addr_location al;
	char map_type = 'd';
	struct thread_trace *ttrace;
	int err = -1;
	int callchain_ret = 0;

	thread = machine__findnew_thread(trace->host, sample->pid, sample->tid);

	if (sample->callchain) {
		callchain_ret = trace__resolve_callchain(trace, evsel, sample, &callchain_cursor);
		if (callchain_ret == 0) {
			if (callchain_cursor.nr < trace->min_stack)
				goto out_put;
			callchain_ret = 1;
		}
	}

	ttrace = thread__trace(thread, trace->output);
	if (ttrace == NULL)
		goto out_put;

	if (evsel->attr.config == PERF_COUNT_SW_PAGE_FAULTS_MAJ)
		ttrace->pfmaj++;
	else
		ttrace->pfmin++;

	if (trace->summary_only)
		goto out;

	thread__find_symbol(thread, sample->cpumode, sample->ip, &al);

	trace__fprintf_entry_head(trace, thread, 0, true, sample->time, trace->output);

	fprintf(trace->output, "%sfault [",
		evsel->attr.config == PERF_COUNT_SW_PAGE_FAULTS_MAJ ?
		"maj" : "min");

	print_location(trace->output, sample, &al, false, true);

	fprintf(trace->output, "] => ");

	thread__find_symbol(thread, sample->cpumode, sample->addr, &al);

	if (!al.map) {
		thread__find_symbol(thread, sample->cpumode, sample->addr, &al);

		if (al.map)
			map_type = 'x';
		else
			map_type = '?';
	}

	print_location(trace->output, sample, &al, true, false);

	fprintf(trace->output, " (%c%c)\n", map_type, al.level);

	if (callchain_ret > 0)
		trace__fprintf_callchain(trace, sample);
	else if (callchain_ret < 0)
		pr_err("Problem processing %s callchain, skipping...\n", perf_evsel__name(evsel));

	++trace->nr_events_printed;
out:
	err = 0;
out_put:
	thread__put(thread);
	return err;
}

static void trace__set_base_time(struct trace *trace,
				 struct perf_evsel *evsel,
				 struct perf_sample *sample)
{
	/*
	 * BPF events were not setting PERF_SAMPLE_TIME, so be more robust
	 * and don't use sample->time unconditionally, we may end up having
	 * some other event in the future without PERF_SAMPLE_TIME for good
	 * reason, i.e. we may not be interested in its timestamps, just in
	 * it taking place, picking some piece of information when it
	 * appears in our event stream (vfs_getname comes to mind).
	 */
	if (trace->base_time == 0 && !trace->full_time &&
	    (evsel->attr.sample_type & PERF_SAMPLE_TIME))
		trace->base_time = sample->time;
}

static int trace__process_sample(struct perf_tool *tool,
				 union perf_event *event,
				 struct perf_sample *sample,
				 struct perf_evsel *evsel,
				 struct machine *machine __maybe_unused)
{
	struct trace *trace = container_of(tool, struct trace, tool);
	struct thread *thread;
	int err = 0;

	tracepoint_handler handler = evsel->handler;

	thread = machine__findnew_thread(trace->host, sample->pid, sample->tid);
	if (thread && thread__is_filtered(thread))
		goto out;

	trace__set_base_time(trace, evsel, sample);

	if (handler) {
		++trace->nr_events;
		handler(trace, evsel, event, sample);
	}
out:
	thread__put(thread);
	return err;
}

static int trace__record(struct trace *trace, int argc, const char **argv)
{
	unsigned int rec_argc, i, j;
	const char **rec_argv;
	const char * const record_args[] = {
		"record",
		"-R",
		"-m", "1024",
		"-c", "1",
	};

	const char * const sc_args[] = { "-e", };
	unsigned int sc_args_nr = ARRAY_SIZE(sc_args);
	const char * const majpf_args[] = { "-e", "major-faults" };
	unsigned int majpf_args_nr = ARRAY_SIZE(majpf_args);
	const char * const minpf_args[] = { "-e", "minor-faults" };
	unsigned int minpf_args_nr = ARRAY_SIZE(minpf_args);

	/* +1 is for the event string below */
	rec_argc = ARRAY_SIZE(record_args) + sc_args_nr + 1 +
		majpf_args_nr + minpf_args_nr + argc;
	rec_argv = calloc(rec_argc + 1, sizeof(char *));

	if (rec_argv == NULL)
		return -ENOMEM;

	j = 0;
	for (i = 0; i < ARRAY_SIZE(record_args); i++)
		rec_argv[j++] = record_args[i];

	if (trace->trace_syscalls) {
		for (i = 0; i < sc_args_nr; i++)
			rec_argv[j++] = sc_args[i];

		/* event string may be different for older kernels - e.g., RHEL6 */
		if (is_valid_tracepoint("raw_syscalls:sys_enter"))
			rec_argv[j++] = "raw_syscalls:sys_enter,raw_syscalls:sys_exit";
		else if (is_valid_tracepoint("syscalls:sys_enter"))
			rec_argv[j++] = "syscalls:sys_enter,syscalls:sys_exit";
		else {
			pr_err("Neither raw_syscalls nor syscalls events exist.\n");
			free(rec_argv);
			return -1;
		}
	}

	if (trace->trace_pgfaults & TRACE_PFMAJ)
		for (i = 0; i < majpf_args_nr; i++)
			rec_argv[j++] = majpf_args[i];

	if (trace->trace_pgfaults & TRACE_PFMIN)
		for (i = 0; i < minpf_args_nr; i++)
			rec_argv[j++] = minpf_args[i];

	for (i = 0; i < (unsigned int)argc; i++)
		rec_argv[j++] = argv[i];

	return cmd_record(j, rec_argv);
}

static size_t trace__fprintf_thread_summary(struct trace *trace, FILE *fp);

static bool perf_evlist__add_vfs_getname(struct perf_evlist *evlist)
{
	struct perf_evsel *evsel = perf_evsel__newtp("probe", "vfs_getname");

	if (IS_ERR(evsel))
		return false;

	if (perf_evsel__field(evsel, "pathname") == NULL) {
		perf_evsel__delete(evsel);
		return false;
	}

	evsel->handler = trace__vfs_getname;
	perf_evlist__add(evlist, evsel);
	return true;
}

static struct perf_evsel *perf_evsel__new_pgfault(u64 config)
{
	struct perf_evsel *evsel;
	struct perf_event_attr attr = {
		.type = PERF_TYPE_SOFTWARE,
		.mmap_data = 1,
	};

	attr.config = config;
	attr.sample_period = 1;

	event_attr_init(&attr);

	evsel = perf_evsel__new(&attr);
	if (evsel)
		evsel->handler = trace__pgfault;

	return evsel;
}

static void trace__handle_event(struct trace *trace, union perf_event *event, struct perf_sample *sample)
{
	const u32 type = event->header.type;
	struct perf_evsel *evsel;

	if (type != PERF_RECORD_SAMPLE) {
		trace__process_event(trace, trace->host, event, sample);
		return;
	}

	evsel = perf_evlist__id2evsel(trace->evlist, sample->id);
	if (evsel == NULL) {
		fprintf(trace->output, "Unknown tp ID %" PRIu64 ", skipping...\n", sample->id);
		return;
	}

	trace__set_base_time(trace, evsel, sample);

	if (evsel->attr.type == PERF_TYPE_TRACEPOINT &&
	    sample->raw_data == NULL) {
		fprintf(trace->output, "%s sample with no payload for tid: %d, cpu %d, raw_size=%d, skipping...\n",
		       perf_evsel__name(evsel), sample->tid,
		       sample->cpu, sample->raw_size);
	} else {
		tracepoint_handler handler = evsel->handler;
		handler(trace, evsel, event, sample);
	}

	if (trace->nr_events_printed >= trace->max_events && trace->max_events != ULONG_MAX)
		interrupted = true;
}

static int trace__add_syscall_newtp(struct trace *trace)
{
	int ret = -1;
	struct perf_evlist *evlist = trace->evlist;
	struct perf_evsel *sys_enter, *sys_exit;

	sys_enter = perf_evsel__raw_syscall_newtp("sys_enter", trace__sys_enter);
	if (sys_enter == NULL)
		goto out;

	if (perf_evsel__init_sc_tp_ptr_field(sys_enter, args))
		goto out_delete_sys_enter;

	sys_exit = perf_evsel__raw_syscall_newtp("sys_exit", trace__sys_exit);
	if (sys_exit == NULL)
		goto out_delete_sys_enter;

	if (perf_evsel__init_sc_tp_uint_field(sys_exit, ret))
		goto out_delete_sys_exit;

	perf_evsel__config_callchain(sys_enter, &trace->opts, &callchain_param);
	perf_evsel__config_callchain(sys_exit, &trace->opts, &callchain_param);

	perf_evlist__add(evlist, sys_enter);
	perf_evlist__add(evlist, sys_exit);

	if (callchain_param.enabled && !trace->kernel_syscallchains) {
		/*
		 * We're interested only in the user space callchain
		 * leading to the syscall, allow overriding that for
		 * debugging reasons using --kernel_syscall_callchains
		 */
		sys_exit->attr.exclude_callchain_kernel = 1;
	}

	trace->syscalls.events.sys_enter = sys_enter;
	trace->syscalls.events.sys_exit  = sys_exit;

	ret = 0;
out:
	return ret;

out_delete_sys_exit:
	perf_evsel__delete_priv(sys_exit);
out_delete_sys_enter:
	perf_evsel__delete_priv(sys_enter);
	goto out;
}

static int trace__set_ev_qualifier_filter(struct trace *trace)
{
	int err = -1;
	struct perf_evsel *sys_exit;
	char *filter = asprintf_expr_inout_ints("id", !trace->not_ev_qualifier,
						trace->ev_qualifier_ids.nr,
						trace->ev_qualifier_ids.entries);

	if (filter == NULL)
		goto out_enomem;

	if (!perf_evsel__append_tp_filter(trace->syscalls.events.sys_enter,
					  filter)) {
		sys_exit = trace->syscalls.events.sys_exit;
		err = perf_evsel__append_tp_filter(sys_exit, filter);
	}

	free(filter);
out:
	return err;
out_enomem:
	errno = ENOMEM;
	goto out;
}

static int trace__set_filter_loop_pids(struct trace *trace)
{
	unsigned int nr = 1;
	pid_t pids[32] = {
		getpid(),
	};
	struct thread *thread = machine__find_thread(trace->host, pids[0], pids[0]);

	while (thread && nr < ARRAY_SIZE(pids)) {
		struct thread *parent = machine__find_thread(trace->host, thread->ppid, thread->ppid);

		if (parent == NULL)
			break;

		if (!strcmp(thread__comm_str(parent), "sshd")) {
			pids[nr++] = parent->tid;
			break;
		}
		thread = parent;
	}

	return perf_evlist__set_filter_pids(trace->evlist, nr, pids);
}

static int trace__run(struct trace *trace, int argc, const char **argv)
{
	struct perf_evlist *evlist = trace->evlist;
	struct perf_evsel *evsel, *pgfault_maj = NULL, *pgfault_min = NULL;
	int err = -1, i;
	unsigned long before;
	const bool forks = argc > 0;
	bool draining = false;

	trace->live = true;

	if (trace->trace_syscalls && trace__add_syscall_newtp(trace))
		goto out_error_raw_syscalls;

	if (trace->trace_syscalls)
		trace->vfs_getname = perf_evlist__add_vfs_getname(evlist);

	if ((trace->trace_pgfaults & TRACE_PFMAJ)) {
		pgfault_maj = perf_evsel__new_pgfault(PERF_COUNT_SW_PAGE_FAULTS_MAJ);
		if (pgfault_maj == NULL)
			goto out_error_mem;
		perf_evsel__config_callchain(pgfault_maj, &trace->opts, &callchain_param);
		perf_evlist__add(evlist, pgfault_maj);
	}

	if ((trace->trace_pgfaults & TRACE_PFMIN)) {
		pgfault_min = perf_evsel__new_pgfault(PERF_COUNT_SW_PAGE_FAULTS_MIN);
		if (pgfault_min == NULL)
			goto out_error_mem;
		perf_evsel__config_callchain(pgfault_min, &trace->opts, &callchain_param);
		perf_evlist__add(evlist, pgfault_min);
	}

	if (trace->sched &&
	    perf_evlist__add_newtp(evlist, "sched", "sched_stat_runtime",
				   trace__sched_stat_runtime))
		goto out_error_sched_stat_runtime;

	/*
	 * If a global cgroup was set, apply it to all the events without an
	 * explicit cgroup. I.e.:
	 *
	 * 	trace -G A -e sched:*switch
	 *
	 * Will set all raw_syscalls:sys_{enter,exit}, pgfault, vfs_getname, etc
	 * _and_ sched:sched_switch to the 'A' cgroup, while:
	 *
	 * trace -e sched:*switch -G A
	 *
	 * will only set the sched:sched_switch event to the 'A' cgroup, all the
	 * other events (raw_syscalls:sys_{enter,exit}, etc are left "without"
	 * a cgroup (on the root cgroup, sys wide, etc).
	 *
	 * Multiple cgroups:
	 *
	 * trace -G A -e sched:*switch -G B
	 *
	 * the syscall ones go to the 'A' cgroup, the sched:sched_switch goes
	 * to the 'B' cgroup.
	 *
	 * evlist__set_default_cgroup() grabs a reference of the passed cgroup
	 * only for the evsels still without a cgroup, i.e. evsel->cgroup == NULL.
	 */
	if (trace->cgroup)
		evlist__set_default_cgroup(trace->evlist, trace->cgroup);

	err = perf_evlist__create_maps(evlist, &trace->opts.target);
	if (err < 0) {
		fprintf(trace->output, "Problems parsing the target to trace, check your options!\n");
		goto out_delete_evlist;
	}

	err = trace__symbols_init(trace, evlist);
	if (err < 0) {
		fprintf(trace->output, "Problems initializing symbol libraries!\n");
		goto out_delete_evlist;
	}

	perf_evlist__config(evlist, &trace->opts, &callchain_param);

	signal(SIGCHLD, sig_handler);
	signal(SIGINT, sig_handler);

	if (forks) {
		err = perf_evlist__prepare_workload(evlist, &trace->opts.target,
						    argv, false, NULL);
		if (err < 0) {
			fprintf(trace->output, "Couldn't run the workload!\n");
			goto out_delete_evlist;
		}
	}

	err = perf_evlist__open(evlist);
	if (err < 0)
		goto out_error_open;

	err = bpf__apply_obj_config();
	if (err) {
		char errbuf[BUFSIZ];

		bpf__strerror_apply_obj_config(err, errbuf, sizeof(errbuf));
		pr_err("ERROR: Apply config to BPF failed: %s\n",
			 errbuf);
		goto out_error_open;
	}

	/*
	 * Better not use !target__has_task() here because we need to cover the
	 * case where no threads were specified in the command line, but a
	 * workload was, and in that case we will fill in the thread_map when
	 * we fork the workload in perf_evlist__prepare_workload.
	 */
	if (trace->filter_pids.nr > 0)
		err = perf_evlist__set_filter_pids(evlist, trace->filter_pids.nr, trace->filter_pids.entries);
	else if (thread_map__pid(evlist->threads, 0) == -1)
		err = trace__set_filter_loop_pids(trace);

	if (err < 0)
		goto out_error_mem;

	if (trace->ev_qualifier_ids.nr > 0) {
		err = trace__set_ev_qualifier_filter(trace);
		if (err < 0)
			goto out_errno;

		pr_debug("event qualifier tracepoint filter: %s\n",
			 trace->syscalls.events.sys_exit->filter);
	}

	err = perf_evlist__apply_filters(evlist, &evsel);
	if (err < 0)
		goto out_error_apply_filters;

	err = perf_evlist__mmap(evlist, trace->opts.mmap_pages);
	if (err < 0)
		goto out_error_mmap;

	if (!target__none(&trace->opts.target) && !trace->opts.initial_delay)
		perf_evlist__enable(evlist);

	if (forks)
		perf_evlist__start_workload(evlist);

	if (trace->opts.initial_delay) {
		usleep(trace->opts.initial_delay * 1000);
		perf_evlist__enable(evlist);
	}

	trace->multiple_threads = thread_map__pid(evlist->threads, 0) == -1 ||
				  evlist->threads->nr > 1 ||
				  perf_evlist__first(evlist)->attr.inherit;

	/*
	 * Now that we already used evsel->attr to ask the kernel to setup the
	 * events, lets reuse evsel->attr.sample_max_stack as the limit in
	 * trace__resolve_callchain(), allowing per-event max-stack settings
	 * to override an explicitely set --max-stack global setting.
	 */
	evlist__for_each_entry(evlist, evsel) {
		if (evsel__has_callchain(evsel) &&
		    evsel->attr.sample_max_stack == 0)
			evsel->attr.sample_max_stack = trace->max_stack;
	}
again:
	before = trace->nr_events;

	for (i = 0; i < evlist->nr_mmaps; i++) {
		union perf_event *event;
		struct perf_mmap *md;

		md = &evlist->mmap[i];
		if (perf_mmap__read_init(md) < 0)
			continue;

		while ((event = perf_mmap__read_event(md)) != NULL) {
			struct perf_sample sample;

			++trace->nr_events;

			err = perf_evlist__parse_sample(evlist, event, &sample);
			if (err) {
				fprintf(trace->output, "Can't parse sample, err = %d, skipping...\n", err);
				goto next_event;
			}

			trace__handle_event(trace, event, &sample);
next_event:
			perf_mmap__consume(md);

			if (interrupted)
				goto out_disable;

			if (done && !draining) {
				perf_evlist__disable(evlist);
				draining = true;
			}
		}
		perf_mmap__read_done(md);
	}

	if (trace->nr_events == before) {
		int timeout = done ? 100 : -1;

		if (!draining && perf_evlist__poll(evlist, timeout) > 0) {
			if (perf_evlist__filter_pollfd(evlist, POLLERR | POLLHUP | POLLNVAL) == 0)
				draining = true;

			goto again;
		}
	} else {
		goto again;
	}

out_disable:
	thread__zput(trace->current);

	perf_evlist__disable(evlist);

	if (!err) {
		if (trace->summary)
			trace__fprintf_thread_summary(trace, trace->output);

		if (trace->show_tool_stats) {
			fprintf(trace->output, "Stats:\n "
					       " vfs_getname : %" PRIu64 "\n"
					       " proc_getname: %" PRIu64 "\n",
				trace->stats.vfs_getname,
				trace->stats.proc_getname);
		}
	}

out_delete_evlist:
	trace__symbols__exit(trace);

	perf_evlist__delete(evlist);
	cgroup__put(trace->cgroup);
	trace->evlist = NULL;
	trace->live = false;
	return err;
{
	char errbuf[BUFSIZ];

out_error_sched_stat_runtime:
	tracing_path__strerror_open_tp(errno, errbuf, sizeof(errbuf), "sched", "sched_stat_runtime");
	goto out_error;

out_error_raw_syscalls:
	tracing_path__strerror_open_tp(errno, errbuf, sizeof(errbuf), "raw_syscalls", "sys_(enter|exit)");
	goto out_error;

out_error_mmap:
	perf_evlist__strerror_mmap(evlist, errno, errbuf, sizeof(errbuf));
	goto out_error;

out_error_open:
	perf_evlist__strerror_open(evlist, errno, errbuf, sizeof(errbuf));

out_error:
	fprintf(trace->output, "%s\n", errbuf);
	goto out_delete_evlist;

out_error_apply_filters:
	fprintf(trace->output,
		"Failed to set filter \"%s\" on event %s with %d (%s)\n",
		evsel->filter, perf_evsel__name(evsel), errno,
		str_error_r(errno, errbuf, sizeof(errbuf)));
	goto out_delete_evlist;
}
out_error_mem:
	fprintf(trace->output, "Not enough memory to run!\n");
	goto out_delete_evlist;

out_errno:
	fprintf(trace->output, "errno=%d,%s\n", errno, strerror(errno));
	goto out_delete_evlist;
}

static int trace__replay(struct trace *trace)
{
	const struct perf_evsel_str_handler handlers[] = {
		{ "probe:vfs_getname",	     trace__vfs_getname, },
	};
	struct perf_data data = {
		.file      = {
			.path = input_name,
		},
		.mode      = PERF_DATA_MODE_READ,
		.force     = trace->force,
	};
	struct perf_session *session;
	struct perf_evsel *evsel;
	int err = -1;

	trace->tool.sample	  = trace__process_sample;
	trace->tool.mmap	  = perf_event__process_mmap;
	trace->tool.mmap2	  = perf_event__process_mmap2;
	trace->tool.comm	  = perf_event__process_comm;
	trace->tool.exit	  = perf_event__process_exit;
	trace->tool.fork	  = perf_event__process_fork;
	trace->tool.attr	  = perf_event__process_attr;
	trace->tool.tracing_data  = perf_event__process_tracing_data;
	trace->tool.build_id	  = perf_event__process_build_id;
	trace->tool.namespaces	  = perf_event__process_namespaces;

	trace->tool.ordered_events = true;
	trace->tool.ordering_requires_timestamps = true;

	/* add tid to output */
	trace->multiple_threads = true;

	session = perf_session__new(&data, false, &trace->tool);
	if (session == NULL)
		return -1;

	if (trace->opts.target.pid)
		symbol_conf.pid_list_str = strdup(trace->opts.target.pid);

	if (trace->opts.target.tid)
		symbol_conf.tid_list_str = strdup(trace->opts.target.tid);

	if (symbol__init(&session->header.env) < 0)
		goto out;

	trace->host = &session->machines.host;

	err = perf_session__set_tracepoints_handlers(session, handlers);
	if (err)
		goto out;

	evsel = perf_evlist__find_tracepoint_by_name(session->evlist,
						     "raw_syscalls:sys_enter");
	/* older kernels have syscalls tp versus raw_syscalls */
	if (evsel == NULL)
		evsel = perf_evlist__find_tracepoint_by_name(session->evlist,
							     "syscalls:sys_enter");

	if (evsel &&
	    (perf_evsel__init_raw_syscall_tp(evsel, trace__sys_enter) < 0 ||
	    perf_evsel__init_sc_tp_ptr_field(evsel, args))) {
		pr_err("Error during initialize raw_syscalls:sys_enter event\n");
		goto out;
	}

	evsel = perf_evlist__find_tracepoint_by_name(session->evlist,
						     "raw_syscalls:sys_exit");
	if (evsel == NULL)
		evsel = perf_evlist__find_tracepoint_by_name(session->evlist,
							     "syscalls:sys_exit");
	if (evsel &&
	    (perf_evsel__init_raw_syscall_tp(evsel, trace__sys_exit) < 0 ||
	    perf_evsel__init_sc_tp_uint_field(evsel, ret))) {
		pr_err("Error during initialize raw_syscalls:sys_exit event\n");
		goto out;
	}

	evlist__for_each_entry(session->evlist, evsel) {
		if (evsel->attr.type == PERF_TYPE_SOFTWARE &&
		    (evsel->attr.config == PERF_COUNT_SW_PAGE_FAULTS_MAJ ||
		     evsel->attr.config == PERF_COUNT_SW_PAGE_FAULTS_MIN ||
		     evsel->attr.config == PERF_COUNT_SW_PAGE_FAULTS))
			evsel->handler = trace__pgfault;
	}

	setup_pager();

	err = perf_session__process_events(session);
	if (err)
		pr_err("Failed to process events, error %d", err);

	else if (trace->summary)
		trace__fprintf_thread_summary(trace, trace->output);

out:
	perf_session__delete(session);

	return err;
}

static size_t trace__fprintf_threads_header(FILE *fp)
{
	size_t printed;

	printed  = fprintf(fp, "\n Summary of events:\n\n");

	return printed;
}

DEFINE_RESORT_RB(syscall_stats, a->msecs > b->msecs,
	struct stats 	*stats;
	double		msecs;
	int		syscall;
)
{
	struct int_node *source = rb_entry(nd, struct int_node, rb_node);
	struct stats *stats = source->priv;

	entry->syscall = source->i;
	entry->stats   = stats;
	entry->msecs   = stats ? (u64)stats->n * (avg_stats(stats) / NSEC_PER_MSEC) : 0;
}

static size_t thread__dump_stats(struct thread_trace *ttrace,
				 struct trace *trace, FILE *fp)
{
	size_t printed = 0;
	struct syscall *sc;
	struct rb_node *nd;
	DECLARE_RESORT_RB_INTLIST(syscall_stats, ttrace->syscall_stats);

	if (syscall_stats == NULL)
		return 0;

	printed += fprintf(fp, "\n");

	printed += fprintf(fp, "   syscall            calls    total       min       avg       max      stddev\n");
	printed += fprintf(fp, "                               (msec)    (msec)    (msec)    (msec)        (%%)\n");
	printed += fprintf(fp, "   --------------- -------- --------- --------- --------- ---------     ------\n");

	resort_rb__for_each_entry(nd, syscall_stats) {
		struct stats *stats = syscall_stats_entry->stats;
		if (stats) {
			double min = (double)(stats->min) / NSEC_PER_MSEC;
			double max = (double)(stats->max) / NSEC_PER_MSEC;
			double avg = avg_stats(stats);
			double pct;
			u64 n = (u64) stats->n;

			pct = avg ? 100.0 * stddev_stats(stats)/avg : 0.0;
			avg /= NSEC_PER_MSEC;

			sc = &trace->syscalls.table[syscall_stats_entry->syscall];
			printed += fprintf(fp, "   %-15s", sc->name);
			printed += fprintf(fp, " %8" PRIu64 " %9.3f %9.3f %9.3f",
					   n, syscall_stats_entry->msecs, min, avg);
			printed += fprintf(fp, " %9.3f %9.2f%%\n", max, pct);
		}
	}

	resort_rb__delete(syscall_stats);
	printed += fprintf(fp, "\n\n");

	return printed;
}

static size_t trace__fprintf_thread(FILE *fp, struct thread *thread, struct trace *trace)
{
	size_t printed = 0;
	struct thread_trace *ttrace = thread__priv(thread);
	double ratio;

	if (ttrace == NULL)
		return 0;

	ratio = (double)ttrace->nr_events / trace->nr_events * 100.0;

	printed += fprintf(fp, " %s (%d), ", thread__comm_str(thread), thread->tid);
	printed += fprintf(fp, "%lu events, ", ttrace->nr_events);
	printed += fprintf(fp, "%.1f%%", ratio);
	if (ttrace->pfmaj)
		printed += fprintf(fp, ", %lu majfaults", ttrace->pfmaj);
	if (ttrace->pfmin)
		printed += fprintf(fp, ", %lu minfaults", ttrace->pfmin);
	if (trace->sched)
		printed += fprintf(fp, ", %.3f msec\n", ttrace->runtime_ms);
	else if (fputc('\n', fp) != EOF)
		++printed;

	printed += thread__dump_stats(ttrace, trace, fp);

	return printed;
}

static unsigned long thread__nr_events(struct thread_trace *ttrace)
{
	return ttrace ? ttrace->nr_events : 0;
}

DEFINE_RESORT_RB(threads, (thread__nr_events(a->thread->priv) < thread__nr_events(b->thread->priv)),
	struct thread *thread;
)
{
	entry->thread = rb_entry(nd, struct thread, rb_node);
}

static size_t trace__fprintf_thread_summary(struct trace *trace, FILE *fp)
{
	size_t printed = trace__fprintf_threads_header(fp);
	struct rb_node *nd;
	int i;

	for (i = 0; i < THREADS__TABLE_SIZE; i++) {
		DECLARE_RESORT_RB_MACHINE_THREADS(threads, trace->host, i);

		if (threads == NULL) {
			fprintf(fp, "%s", "Error sorting output by nr_events!\n");
			return 0;
		}

		resort_rb__for_each_entry(nd, threads)
			printed += trace__fprintf_thread(fp, threads_entry->thread, trace);

		resort_rb__delete(threads);
	}
	return printed;
}

static int trace__set_duration(const struct option *opt, const char *str,
			       int unset __maybe_unused)
{
	struct trace *trace = opt->value;

	trace->duration_filter = atof(str);
	return 0;
}

static int trace__set_filter_pids(const struct option *opt, const char *str,
				  int unset __maybe_unused)
{
	int ret = -1;
	size_t i;
	struct trace *trace = opt->value;
	/*
	 * FIXME: introduce a intarray class, plain parse csv and create a
	 * { int nr, int entries[] } struct...
	 */
	struct intlist *list = intlist__new(str);

	if (list == NULL)
		return -1;

	i = trace->filter_pids.nr = intlist__nr_entries(list) + 1;
	trace->filter_pids.entries = calloc(i, sizeof(pid_t));

	if (trace->filter_pids.entries == NULL)
		goto out;

	trace->filter_pids.entries[0] = getpid();

	for (i = 1; i < trace->filter_pids.nr; ++i)
		trace->filter_pids.entries[i] = intlist__entry(list, i - 1)->i;

	intlist__delete(list);
	ret = 0;
out:
	return ret;
}

static int trace__open_output(struct trace *trace, const char *filename)
{
	struct stat st;

	if (!stat(filename, &st) && st.st_size) {
		char oldname[PATH_MAX];

		scnprintf(oldname, sizeof(oldname), "%s.old", filename);
		unlink(oldname);
		rename(filename, oldname);
	}

	trace->output = fopen(filename, "w");

	return trace->output == NULL ? -errno : 0;
}

static int parse_pagefaults(const struct option *opt, const char *str,
			    int unset __maybe_unused)
{
	int *trace_pgfaults = opt->value;

	if (strcmp(str, "all") == 0)
		*trace_pgfaults |= TRACE_PFMAJ | TRACE_PFMIN;
	else if (strcmp(str, "maj") == 0)
		*trace_pgfaults |= TRACE_PFMAJ;
	else if (strcmp(str, "min") == 0)
		*trace_pgfaults |= TRACE_PFMIN;
	else
		return -1;

	return 0;
}

static void evlist__set_evsel_handler(struct perf_evlist *evlist, void *handler)
{
	struct perf_evsel *evsel;

	evlist__for_each_entry(evlist, evsel)
		evsel->handler = handler;
}

static int evlist__set_syscall_tp_fields(struct perf_evlist *evlist)
{
	struct perf_evsel *evsel;

	evlist__for_each_entry(evlist, evsel) {
		if (evsel->priv || !evsel->tp_format)
			continue;

		if (strcmp(evsel->tp_format->system, "syscalls"))
			continue;

		if (perf_evsel__init_syscall_tp(evsel))
			return -1;

		if (!strncmp(evsel->tp_format->name, "sys_enter_", 10)) {
			struct syscall_tp *sc = evsel->priv;

			if (__tp_field__init_ptr(&sc->args, sc->id.offset + sizeof(u64)))
				return -1;
		} else if (!strncmp(evsel->tp_format->name, "sys_exit_", 9)) {
			struct syscall_tp *sc = evsel->priv;

			if (__tp_field__init_uint(&sc->ret, sizeof(u64), sc->id.offset + sizeof(u64), evsel->needs_swap))
				return -1;
		}
	}

	return 0;
}

/*
 * XXX: Hackish, just splitting the combined -e+--event (syscalls
 * (raw_syscalls:{sys_{enter,exit}} + events (tracepoints, HW, SW, etc) to use
 * existing facilities unchanged (trace->ev_qualifier + parse_options()).
 *
 * It'd be better to introduce a parse_options() variant that would return a
 * list with the terms it didn't match to an event...
 */
static int trace__parse_events_option(const struct option *opt, const char *str,
				      int unset __maybe_unused)
{
	struct trace *trace = (struct trace *)opt->value;
	const char *s = str;
	char *sep = NULL, *lists[2] = { NULL, NULL, };
	int len = strlen(str) + 1, err = -1, list, idx;
	char *strace_groups_dir = system_path(STRACE_GROUPS_DIR);
	char group_name[PATH_MAX];
	struct syscall_fmt *fmt;

	if (strace_groups_dir == NULL)
		return -1;

	if (*s == '!') {
		++s;
		trace->not_ev_qualifier = true;
	}

	while (1) {
		if ((sep = strchr(s, ',')) != NULL)
			*sep = '\0';

		list = 0;
		if (syscalltbl__id(trace->sctbl, s) >= 0 ||
		    syscalltbl__strglobmatch_first(trace->sctbl, s, &idx) >= 0) {
			list = 1;
			goto do_concat;
		}

		fmt = syscall_fmt__find_by_alias(s);
		if (fmt != NULL) {
			list = 1;
			s = fmt->name;
		} else {
			path__join(group_name, sizeof(group_name), strace_groups_dir, s);
			if (access(group_name, R_OK) == 0)
				list = 1;
		}
do_concat:
		if (lists[list]) {
			sprintf(lists[list] + strlen(lists[list]), ",%s", s);
		} else {
			lists[list] = malloc(len);
			if (lists[list] == NULL)
				goto out;
			strcpy(lists[list], s);
		}

		if (!sep)
			break;

		*sep = ',';
		s = sep + 1;
	}

	if (lists[1] != NULL) {
		struct strlist_config slist_config = {
			.dirname = strace_groups_dir,
		};

		trace->ev_qualifier = strlist__new(lists[1], &slist_config);
		if (trace->ev_qualifier == NULL) {
			fputs("Not enough memory to parse event qualifier", trace->output);
			goto out;
		}

		if (trace__validate_ev_qualifier(trace))
			goto out;
		trace->trace_syscalls = true;
	}

	err = 0;

	if (lists[0]) {
		struct option o = OPT_CALLBACK('e', "event", &trace->evlist, "event",
					       "event selector. use 'perf list' to list available events",
					       parse_events_option);
		err = parse_events_option(&o, lists[0], 0);
	}
out:
	if (sep)
		*sep = ',';

	return err;
}

static int trace__parse_cgroups(const struct option *opt, const char *str, int unset)
{
	struct trace *trace = opt->value;

	if (!list_empty(&trace->evlist->entries))
		return parse_cgroups(opt, str, unset);

	trace->cgroup = evlist__findnew_cgroup(trace->evlist, str);

	return 0;
}

int cmd_trace(int argc, const char **argv)
{
	const char *trace_usage[] = {
		"perf trace [<options>] [<command>]",
		"perf trace [<options>] -- <command> [<options>]",
		"perf trace record [<options>] [<command>]",
		"perf trace record [<options>] -- <command> [<options>]",
		NULL
	};
	struct trace trace = {
		.syscalls = {
			. max = -1,
		},
		.opts = {
			.target = {
				.uid	   = UINT_MAX,
				.uses_mmap = true,
			},
			.user_freq     = UINT_MAX,
			.user_interval = ULLONG_MAX,
			.no_buffering  = true,
			.mmap_pages    = UINT_MAX,
			.proc_map_timeout  = 500,
		},
		.output = stderr,
		.show_comm = true,
		.trace_syscalls = false,
		.kernel_syscallchains = false,
		.max_stack = UINT_MAX,
		.max_events = ULONG_MAX,
	};
	const char *output_name = NULL;
	const struct option trace_options[] = {
	OPT_CALLBACK('e', "event", &trace, "event",
		     "event/syscall selector. use 'perf list' to list available events",
		     trace__parse_events_option),
	OPT_BOOLEAN(0, "comm", &trace.show_comm,
		    "show the thread COMM next to its id"),
	OPT_BOOLEAN(0, "tool_stats", &trace.show_tool_stats, "show tool stats"),
	OPT_CALLBACK(0, "expr", &trace, "expr", "list of syscalls/events to trace",
		     trace__parse_events_option),
	OPT_STRING('o', "output", &output_name, "file", "output file name"),
	OPT_STRING('i', "input", &input_name, "file", "Analyze events in file"),
	OPT_STRING('p', "pid", &trace.opts.target.pid, "pid",
		    "trace events on existing process id"),
	OPT_STRING('t', "tid", &trace.opts.target.tid, "tid",
		    "trace events on existing thread id"),
	OPT_CALLBACK(0, "filter-pids", &trace, "CSV list of pids",
		     "pids to filter (by the kernel)", trace__set_filter_pids),
	OPT_BOOLEAN('a', "all-cpus", &trace.opts.target.system_wide,
		    "system-wide collection from all CPUs"),
	OPT_STRING('C', "cpu", &trace.opts.target.cpu_list, "cpu",
		    "list of cpus to monitor"),
	OPT_BOOLEAN(0, "no-inherit", &trace.opts.no_inherit,
		    "child tasks do not inherit counters"),
	OPT_CALLBACK('m', "mmap-pages", &trace.opts.mmap_pages, "pages",
		     "number of mmap data pages",
		     perf_evlist__parse_mmap_pages),
	OPT_STRING('u', "uid", &trace.opts.target.uid_str, "user",
		   "user to profile"),
	OPT_CALLBACK(0, "duration", &trace, "float",
		     "show only events with duration > N.M ms",
		     trace__set_duration),
	OPT_BOOLEAN(0, "sched", &trace.sched, "show blocking scheduler events"),
	OPT_INCR('v', "verbose", &verbose, "be more verbose"),
	OPT_BOOLEAN('T', "time", &trace.full_time,
		    "Show full timestamp, not time relative to first start"),
	OPT_BOOLEAN(0, "failure", &trace.failure_only,
		    "Show only syscalls that failed"),
	OPT_BOOLEAN('s', "summary", &trace.summary_only,
		    "Show only syscall summary with statistics"),
	OPT_BOOLEAN('S', "with-summary", &trace.summary,
		    "Show all syscalls and summary with statistics"),
	OPT_CALLBACK_DEFAULT('F', "pf", &trace.trace_pgfaults, "all|maj|min",
		     "Trace pagefaults", parse_pagefaults, "maj"),
	OPT_BOOLEAN(0, "syscalls", &trace.trace_syscalls, "Trace syscalls"),
	OPT_BOOLEAN('f', "force", &trace.force, "don't complain, do it"),
	OPT_CALLBACK(0, "call-graph", &trace.opts,
		     "record_mode[,record_size]", record_callchain_help,
		     &record_parse_callchain_opt),
	OPT_BOOLEAN(0, "kernel-syscall-graph", &trace.kernel_syscallchains,
		    "Show the kernel callchains on the syscall exit path"),
	OPT_ULONG(0, "max-events", &trace.max_events,
		"Set the maximum number of events to print, exit after that is reached. "),
	OPT_UINTEGER(0, "min-stack", &trace.min_stack,
		     "Set the minimum stack depth when parsing the callchain, "
		     "anything below the specified depth will be ignored."),
	OPT_UINTEGER(0, "max-stack", &trace.max_stack,
		     "Set the maximum stack depth when parsing the callchain, "
		     "anything beyond the specified depth will be ignored. "
		     "Default: kernel.perf_event_max_stack or " __stringify(PERF_MAX_STACK_DEPTH)),
	OPT_BOOLEAN(0, "print-sample", &trace.print_sample,
			"print the PERF_RECORD_SAMPLE PERF_SAMPLE_ info, for debugging"),
	OPT_UINTEGER(0, "proc-map-timeout", &trace.opts.proc_map_timeout,
			"per thread proc mmap processing timeout in ms"),
	OPT_CALLBACK('G', "cgroup", &trace, "name", "monitor event in cgroup name only",
		     trace__parse_cgroups),
	OPT_UINTEGER('D', "delay", &trace.opts.initial_delay,
		     "ms to wait before starting measurement after program "
		     "start"),
	OPT_END()
	};
	bool __maybe_unused max_stack_user_set = true;
	bool mmap_pages_user_set = true;
	struct perf_evsel *evsel;
	const char * const trace_subcommands[] = { "record", NULL };
	int err = -1;
	char bf[BUFSIZ];

	signal(SIGSEGV, sighandler_dump_stack);
	signal(SIGFPE, sighandler_dump_stack);

	trace.evlist = perf_evlist__new();
	trace.sctbl = syscalltbl__new();

	if (trace.evlist == NULL || trace.sctbl == NULL) {
		pr_err("Not enough memory to run!\n");
		err = -ENOMEM;
		goto out;
	}

	argc = parse_options_subcommand(argc, argv, trace_options, trace_subcommands,
				 trace_usage, PARSE_OPT_STOP_AT_NON_OPTION);

	if ((nr_cgroups || trace.cgroup) && !trace.opts.target.system_wide) {
		usage_with_options_msg(trace_usage, trace_options,
				       "cgroup monitoring only available in system-wide mode");
	}

	evsel = bpf__setup_output_event(trace.evlist, "__augmented_syscalls__");
	if (IS_ERR(evsel)) {
		bpf__strerror_setup_output_event(trace.evlist, PTR_ERR(evsel), bf, sizeof(bf));
		pr_err("ERROR: Setup trace syscalls enter failed: %s\n", bf);
		goto out;
	}

	if (evsel)
		trace.syscalls.events.augmented = evsel;

	err = bpf__setup_stdout(trace.evlist);
	if (err) {
		bpf__strerror_setup_stdout(trace.evlist, err, bf, sizeof(bf));
		pr_err("ERROR: Setup BPF stdout failed: %s\n", bf);
		goto out;
	}

	err = -1;

	if (trace.trace_pgfaults) {
		trace.opts.sample_address = true;
		trace.opts.sample_time = true;
	}

	if (trace.opts.mmap_pages == UINT_MAX)
		mmap_pages_user_set = false;

	if (trace.max_stack == UINT_MAX) {
		trace.max_stack = input_name ? PERF_MAX_STACK_DEPTH : sysctl__max_stack();
		max_stack_user_set = false;
	}

#ifdef HAVE_DWARF_UNWIND_SUPPORT
	if ((trace.min_stack || max_stack_user_set) && !callchain_param.enabled) {
		record_opts__parse_callchain(&trace.opts, &callchain_param, "dwarf", false);
	}
#endif

	if (callchain_param.enabled) {
		if (!mmap_pages_user_set && geteuid() == 0)
			trace.opts.mmap_pages = perf_event_mlock_kb_in_pages() * 4;

		symbol_conf.use_callchain = true;
	}

	if (trace.evlist->nr_entries > 0) {
		evlist__set_evsel_handler(trace.evlist, trace__event_handler);
		if (evlist__set_syscall_tp_fields(trace.evlist)) {
			perror("failed to set syscalls:* tracepoint fields");
			goto out;
		}
	}

	/*
	 * If we are augmenting syscalls, then combine what we put in the
	 * __augmented_syscalls__ BPF map with what is in the
	 * syscalls:sys_exit_FOO tracepoints, i.e. just like we do without BPF,
	 * combining raw_syscalls:sys_enter with raw_syscalls:sys_exit.
	 *
	 * We'll switch to look at two BPF maps, one for sys_enter and the
	 * other for sys_exit when we start augmenting the sys_exit paths with
	 * buffers that are being copied from kernel to userspace, think 'read'
	 * syscall.
	 */
	if (trace.syscalls.events.augmented) {
		evsel = trace.syscalls.events.augmented;

		if (perf_evsel__init_augmented_syscall_tp(evsel) ||
		    perf_evsel__init_augmented_syscall_tp_args(evsel))
			goto out;
		evsel->handler = trace__sys_enter;

		evlist__for_each_entry(trace.evlist, evsel) {
			bool raw_syscalls_sys_exit = strcmp(perf_evsel__name(evsel), "raw_syscalls:sys_exit") == 0;

			if (raw_syscalls_sys_exit) {
				trace.raw_augmented_syscalls = true;
				goto init_augmented_syscall_tp;
			}

			if (strstarts(perf_evsel__name(evsel), "syscalls:sys_exit_")) {
init_augmented_syscall_tp:
				perf_evsel__init_augmented_syscall_tp(evsel);
				perf_evsel__init_augmented_syscall_tp_ret(evsel);
				evsel->handler = trace__sys_exit;
			}
		}
	}

	if ((argc >= 1) && (strcmp(argv[0], "record") == 0))
		return trace__record(&trace, argc-1, &argv[1]);

	/* summary_only implies summary option, but don't overwrite summary if set */
	if (trace.summary_only)
		trace.summary = trace.summary_only;

	if (!trace.trace_syscalls && !trace.trace_pgfaults &&
	    trace.evlist->nr_entries == 0 /* Was --events used? */) {
		trace.trace_syscalls = true;
	}

	if (output_name != NULL) {
		err = trace__open_output(&trace, output_name);
		if (err < 0) {
			perror("failed to create output file");
			goto out;
		}
	}

	err = target__validate(&trace.opts.target);
	if (err) {
		target__strerror(&trace.opts.target, err, bf, sizeof(bf));
		fprintf(trace.output, "%s", bf);
		goto out_close;
	}

	err = target__parse_uid(&trace.opts.target);
	if (err) {
		target__strerror(&trace.opts.target, err, bf, sizeof(bf));
		fprintf(trace.output, "%s", bf);
		goto out_close;
	}

	if (!argc && target__none(&trace.opts.target))
		trace.opts.target.system_wide = true;

	if (input_name)
		err = trace__replay(&trace);
	else
		err = trace__run(&trace, argc, argv);

out_close:
	if (output_name != NULL)
		fclose(trace.output);
out:
	return err;
}
