/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.security.support;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.collect.ImmutableSortedSet;
import com.google.common.hash.HashCode;
import com.google.common.hash.Hashing;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.security.AccessController;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.action.DocWriteRequest;
import org.opensearch.action.admin.indices.create.CreateIndexRequest;
import org.opensearch.action.bulk.BulkRequest;
import org.opensearch.action.get.GetResponse;
import org.opensearch.action.get.MultiGetItemResponse;
import org.opensearch.action.get.MultiGetRequest;
import org.opensearch.action.get.MultiGetResponse;
import org.opensearch.action.index.IndexRequest;
import org.opensearch.action.support.WriteRequest;
import org.opensearch.client.Client;
import org.opensearch.common.settings.Settings;
import org.opensearch.common.util.concurrent.ThreadContext;
import org.opensearch.common.xcontent.XContentType;
import org.opensearch.core.action.ActionListener;
import org.opensearch.core.common.bytes.BytesReference;
import org.opensearch.core.xcontent.DeprecationHandler;
import org.opensearch.core.xcontent.NamedXContentRegistry;
import org.opensearch.core.xcontent.XContent;
import org.opensearch.core.xcontent.XContentParser;
import org.opensearch.security.DefaultObjectMapper;
import org.opensearch.security.configuration.ConfigurationMap;
import org.opensearch.security.securityconf.impl.CType;
import org.opensearch.security.securityconf.impl.SecurityDynamicConfiguration;
import org.opensearch.security.state.SecurityConfig;
import org.opensearch.security.support.SecurityUtils;
import org.opensearch.security.support.YamlConfigReader;

public class SecurityIndexHandler {
    private static final int MINIMUM_HASH_BITS = 128;
    private static final Logger LOGGER = LogManager.getLogger(SecurityIndexHandler.class);
    private final Settings settings;
    private final Client client;
    private final String indexName;
    public static final Map<String, Object> INDEX_SETTINGS = Map.of("index.number_of_shards", 1, "index.auto_expand_replicas", "0-all");

    public SecurityIndexHandler(String indexName, Settings settings, Client client) {
        this.indexName = indexName;
        this.settings = settings;
        this.client = client;
    }

    public void createIndex(ActionListener<Boolean> listener) {
        try (ThreadContext.StoredContext threadContext = this.client.threadPool().getThreadContext().stashContext();){
            this.client.admin().indices().create(new CreateIndexRequest(this.indexName).settings(INDEX_SETTINGS).waitForActiveShards(1), ActionListener.runBefore((ActionListener)ActionListener.wrap(r -> {
                if (r.isAcknowledged()) {
                    listener.onResponse((Object)true);
                } else {
                    listener.onFailure((Exception)new SecurityException("Couldn't create security index " + this.indexName));
                }
            }, arg_0 -> listener.onFailure(arg_0)), () -> ((ThreadContext.StoredContext)threadContext).restore()));
        }
    }

    public void uploadDefaultConfiguration(Path configDir, ActionListener<Set<SecurityConfig>> listener) {
        try (ThreadContext.StoredContext threadContext = this.client.threadPool().getThreadContext().stashContext();){
            AccessController.doPrivileged(() -> {
                try {
                    LOGGER.info("Uploading default security configuration from {}", (Object)configDir.toAbsolutePath());
                    BulkRequest bulkRequest = new BulkRequest().setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE);
                    ImmutableSortedSet.Builder configuration = new ImmutableSortedSet.Builder(Comparator.comparing(SecurityConfig::type));
                    for (CType<?> cType : CType.values()) {
                        boolean fileExists = Files.exists(cType.configFile(configDir), new LinkOption[0]);
                        if ((cType == CType.AUDIT || cType == CType.WHITELIST) && !fileExists) continue;
                        if (cType == CType.WHITELIST) {
                            LOGGER.warn("WHITELIST configuration type is deprecated and will be replaced with ALLOWLIST in the next major version");
                        }
                        BytesReference yamlContent = YamlConfigReader.yamlContentFor(cType, configDir);
                        HashCode hash = Hashing.goodFastHash((int)128).hashBytes(yamlContent.toBytesRef().bytes);
                        configuration.add((Object)new SecurityConfig(cType, hash.toString(), null));
                        bulkRequest.add(new IndexRequest(this.indexName).id(cType.toLCString()).opType(DocWriteRequest.OpType.INDEX).source(new Object[]{cType.toLCString(), yamlContent}));
                    }
                    this.client.bulk(bulkRequest, ActionListener.runBefore((ActionListener)ActionListener.wrap(r -> {
                        if (r.hasFailures()) {
                            listener.onFailure((Exception)new SecurityException(r.buildFailureMessage()));
                            return;
                        }
                        listener.onResponse((Object)configuration.build());
                    }, arg_0 -> ((ActionListener)listener).onFailure(arg_0)), () -> ((ThreadContext.StoredContext)threadContext).restore()));
                }
                catch (IOException ioe) {
                    listener.onFailure((Exception)new SecurityException(ioe));
                }
                return null;
            });
        }
    }

    public void loadConfiguration(Set<SecurityConfig> configuration, ActionListener<ConfigurationMap> listener) {
        try (ThreadContext.StoredContext threadContext = this.client.threadPool().getThreadContext().stashContext();){
            this.client.threadPool().getThreadContext().putHeader("_opendistro_security_conf_request", "true");
            List<CType<?>> configurationTypes = configuration.stream().map(SecurityConfig::type).collect(Collectors.toUnmodifiableList());
            this.client.multiGet(this.newMultiGetRequest(configurationTypes), ActionListener.runBefore((ActionListener)ActionListener.wrap(r -> {
                ConfigurationMap.Builder cTypeConfigsBuilder = new ConfigurationMap.Builder();
                boolean hasFailures = false;
                for (MultiGetItemResponse item : r.getResponses()) {
                    if (item.isFailed()) {
                        listener.onFailure((Exception)new SecurityException(this.multiGetFailureMessage(item.getId(), item.getFailure())));
                        hasFailures = true;
                        break;
                    }
                    CType<?> cType = CType.fromString(item.getId());
                    GetResponse cTypeResponse = item.getResponse();
                    if (cTypeResponse.isExists() && !cTypeResponse.isSourceEmpty()) {
                        SecurityDynamicConfiguration<?> config = this.buildDynamicConfiguration(cType, cTypeResponse.getSourceAsBytesRef(), cTypeResponse.getSeqNo(), cTypeResponse.getPrimaryTerm());
                        if (config.getVersion() != 2) {
                            listener.onFailure((Exception)new SecurityException("Version " + config.getVersion() + " is not supported for " + cType.name()));
                            hasFailures = true;
                            break;
                        }
                        cTypeConfigsBuilder.with(config);
                        continue;
                    }
                    if (!cType.emptyIfMissing()) {
                        listener.onFailure((Exception)new SecurityException("Missing required configuration for type: " + String.valueOf(cType)));
                        hasFailures = true;
                        break;
                    }
                    cTypeConfigsBuilder.with(SecurityDynamicConfiguration.fromJson(YamlConfigReader.emptyJsonConfigFor(cType), cType, 2, cTypeResponse.getSeqNo(), cTypeResponse.getPrimaryTerm()));
                }
                if (!hasFailures) {
                    listener.onResponse((Object)cTypeConfigsBuilder.build());
                }
            }, arg_0 -> listener.onFailure(arg_0)), () -> ((ThreadContext.StoredContext)threadContext).restore()));
        }
    }

    private MultiGetRequest newMultiGetRequest(List<CType<?>> configurationTypes) {
        MultiGetRequest request = new MultiGetRequest().realtime(true).refresh(true);
        for (CType<?> cType : configurationTypes) {
            request.add(this.indexName, cType.toLCString());
        }
        return request;
    }

    private SecurityDynamicConfiguration<?> buildDynamicConfiguration(CType<?> cType, BytesReference bytesRef, long seqNo, long primaryTerm) {
        try {
            String source = SecurityUtils.replaceEnvVars(this.configTypeSource((InputStream)bytesRef.streamInput()), this.settings);
            JsonNode jsonNode = DefaultObjectMapper.readTree(source);
            int version = 1;
            if (jsonNode.has("_meta") && jsonNode.get("_meta").has("config_version")) {
                version = jsonNode.get("_meta").get("config_version").asInt();
            }
            return SecurityDynamicConfiguration.fromJson(source, cType, version, seqNo, primaryTerm);
        }
        catch (IOException e) {
            throw new SecurityException("Couldn't parse content for " + String.valueOf(cType), e);
        }
    }

    private String configTypeSource(InputStream inputStream) throws IOException {
        XContent jsonContent = XContentType.JSON.xContent();
        try (XContentParser parser = jsonContent.createParser(NamedXContentRegistry.EMPTY, DeprecationHandler.THROW_UNSUPPORTED_OPERATION, inputStream);){
            parser.nextToken();
            parser.nextToken();
            parser.nextToken();
            String string = new String(parser.binaryValue(), StandardCharsets.UTF_8);
            return string;
        }
    }

    private String multiGetFailureMessage(String cTypeId, MultiGetResponse.Failure failure) {
        return String.format("Failure %s retrieving configuration for %s (index=%s)", failure, cTypeId, this.indexName);
    }
}

