/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as tslib_1 from "tslib";
import { Compiler, Inject, InjectionToken, Optional, PACKAGE_ROOT_URL, TRANSLATIONS, isDevMode, ɵConsole as Console, ViewEncapsulation, Injector, TRANSLATIONS_FORMAT, MissingTranslationStrategy, } from '@angular/core';
import { StaticSymbolCache, JitCompiler, ProviderMeta, I18NHtmlParser, ViewCompiler, CompileMetadataResolver, UrlResolver, TemplateParser, NgModuleCompiler, JitEvaluator, JitSummaryResolver, SummaryResolver, StyleCompiler, PipeResolver, ElementSchemaRegistry, DomElementSchemaRegistry, ResourceLoader, NgModuleResolver, HtmlParser, CompileReflector, CompilerConfig, DirectiveNormalizer, DirectiveResolver, Lexer, Parser } from '@angular/compiler';
import { JitReflector } from './compiler_reflector';
export var ERROR_COLLECTOR_TOKEN = new InjectionToken('ErrorCollector');
/**
 * A default provider for {@link PACKAGE_ROOT_URL} that maps to '/'.
 */
export var DEFAULT_PACKAGE_URL_PROVIDER = {
    provide: PACKAGE_ROOT_URL,
    useValue: '/'
};
var _NO_RESOURCE_LOADER = {
    get: function (url) {
        throw new Error("No ResourceLoader implementation has been provided. Can't read the url \"" + url + "\"");
    }
};
var baseHtmlParser = new InjectionToken('HtmlParser');
var CompilerImpl = /** @class */ (function () {
    function CompilerImpl(injector, _metadataResolver, templateParser, styleCompiler, viewCompiler, ngModuleCompiler, summaryResolver, compileReflector, jitEvaluator, compilerConfig, console) {
        this._metadataResolver = _metadataResolver;
        this._delegate = new JitCompiler(_metadataResolver, templateParser, styleCompiler, viewCompiler, ngModuleCompiler, summaryResolver, compileReflector, jitEvaluator, compilerConfig, console, this.getExtraNgModuleProviders.bind(this));
        this.injector = injector;
    }
    CompilerImpl.prototype.getExtraNgModuleProviders = function () {
        return [this._metadataResolver.getProviderMetadata(new ProviderMeta(Compiler, { useValue: this }))];
    };
    CompilerImpl.prototype.compileModuleSync = function (moduleType) {
        return this._delegate.compileModuleSync(moduleType);
    };
    CompilerImpl.prototype.compileModuleAsync = function (moduleType) {
        return this._delegate.compileModuleAsync(moduleType);
    };
    CompilerImpl.prototype.compileModuleAndAllComponentsSync = function (moduleType) {
        var result = this._delegate.compileModuleAndAllComponentsSync(moduleType);
        return {
            ngModuleFactory: result.ngModuleFactory,
            componentFactories: result.componentFactories,
        };
    };
    CompilerImpl.prototype.compileModuleAndAllComponentsAsync = function (moduleType) {
        return this._delegate.compileModuleAndAllComponentsAsync(moduleType)
            .then(function (result) { return ({
            ngModuleFactory: result.ngModuleFactory,
            componentFactories: result.componentFactories,
        }); });
    };
    CompilerImpl.prototype.loadAotSummaries = function (summaries) { this._delegate.loadAotSummaries(summaries); };
    CompilerImpl.prototype.hasAotSummary = function (ref) { return this._delegate.hasAotSummary(ref); };
    CompilerImpl.prototype.getComponentFactory = function (component) {
        return this._delegate.getComponentFactory(component);
    };
    CompilerImpl.prototype.clearCache = function () { this._delegate.clearCache(); };
    CompilerImpl.prototype.clearCacheFor = function (type) { this._delegate.clearCacheFor(type); };
    CompilerImpl.prototype.getModuleId = function (moduleType) {
        var meta = this._metadataResolver.getNgModuleMetadata(moduleType);
        return meta && meta.id || undefined;
    };
    return CompilerImpl;
}());
export { CompilerImpl };
/**
 * A set of providers that provide `JitCompiler` and its dependencies to use for
 * template compilation.
 */
export var COMPILER_PROVIDERS = [
    { provide: CompileReflector, useValue: new JitReflector() },
    { provide: ResourceLoader, useValue: _NO_RESOURCE_LOADER },
    { provide: JitSummaryResolver, deps: [] },
    { provide: SummaryResolver, useExisting: JitSummaryResolver },
    { provide: Console, deps: [] },
    { provide: Lexer, deps: [] },
    { provide: Parser, deps: [Lexer] },
    {
        provide: baseHtmlParser,
        useClass: HtmlParser,
        deps: [],
    },
    {
        provide: I18NHtmlParser,
        useFactory: function (parser, translations, format, config, console) {
            translations = translations || '';
            var missingTranslation = translations ? config.missingTranslation : MissingTranslationStrategy.Ignore;
            return new I18NHtmlParser(parser, translations, format, missingTranslation, console);
        },
        deps: [
            baseHtmlParser,
            [new Optional(), new Inject(TRANSLATIONS)],
            [new Optional(), new Inject(TRANSLATIONS_FORMAT)],
            [CompilerConfig],
            [Console],
        ]
    },
    {
        provide: HtmlParser,
        useExisting: I18NHtmlParser,
    },
    {
        provide: TemplateParser, deps: [CompilerConfig, CompileReflector,
            Parser, ElementSchemaRegistry,
            I18NHtmlParser, Console]
    },
    { provide: JitEvaluator, useClass: JitEvaluator, deps: [] },
    { provide: DirectiveNormalizer, deps: [ResourceLoader, UrlResolver, HtmlParser, CompilerConfig] },
    { provide: CompileMetadataResolver, deps: [CompilerConfig, HtmlParser, NgModuleResolver,
            DirectiveResolver, PipeResolver,
            SummaryResolver,
            ElementSchemaRegistry,
            DirectiveNormalizer, Console,
            [Optional, StaticSymbolCache],
            CompileReflector,
            [Optional, ERROR_COLLECTOR_TOKEN]] },
    DEFAULT_PACKAGE_URL_PROVIDER,
    { provide: StyleCompiler, deps: [UrlResolver] },
    { provide: ViewCompiler, deps: [CompileReflector] },
    { provide: NgModuleCompiler, deps: [CompileReflector] },
    { provide: CompilerConfig, useValue: new CompilerConfig() },
    { provide: Compiler, useClass: CompilerImpl, deps: [Injector, CompileMetadataResolver,
            TemplateParser, StyleCompiler,
            ViewCompiler, NgModuleCompiler,
            SummaryResolver, CompileReflector, JitEvaluator, CompilerConfig,
            Console] },
    { provide: DomElementSchemaRegistry, deps: [] },
    { provide: ElementSchemaRegistry, useExisting: DomElementSchemaRegistry },
    { provide: UrlResolver, deps: [PACKAGE_ROOT_URL] },
    { provide: DirectiveResolver, deps: [CompileReflector] },
    { provide: PipeResolver, deps: [CompileReflector] },
    { provide: NgModuleResolver, deps: [CompileReflector] },
];
/**
 * @publicApi
 */
var JitCompilerFactory = /** @class */ (function () {
    /* @internal */
    function JitCompilerFactory(defaultOptions) {
        var compilerOptions = {
            useJit: true,
            defaultEncapsulation: ViewEncapsulation.Emulated,
            missingTranslation: MissingTranslationStrategy.Warning,
        };
        this._defaultOptions = tslib_1.__spread([compilerOptions], defaultOptions);
    }
    JitCompilerFactory.prototype.createCompiler = function (options) {
        if (options === void 0) { options = []; }
        var opts = _mergeOptions(this._defaultOptions.concat(options));
        var injector = Injector.create([
            COMPILER_PROVIDERS, {
                provide: CompilerConfig,
                useFactory: function () {
                    return new CompilerConfig({
                        // let explicit values from the compiler options overwrite options
                        // from the app providers
                        useJit: opts.useJit,
                        jitDevMode: isDevMode(),
                        // let explicit values from the compiler options overwrite options
                        // from the app providers
                        defaultEncapsulation: opts.defaultEncapsulation,
                        missingTranslation: opts.missingTranslation,
                        preserveWhitespaces: opts.preserveWhitespaces,
                    });
                },
                deps: []
            },
            opts.providers
        ]);
        return injector.get(Compiler);
    };
    return JitCompilerFactory;
}());
export { JitCompilerFactory };
function _mergeOptions(optionsArr) {
    return {
        useJit: _lastDefined(optionsArr.map(function (options) { return options.useJit; })),
        defaultEncapsulation: _lastDefined(optionsArr.map(function (options) { return options.defaultEncapsulation; })),
        providers: _mergeArrays(optionsArr.map(function (options) { return options.providers; })),
        missingTranslation: _lastDefined(optionsArr.map(function (options) { return options.missingTranslation; })),
        preserveWhitespaces: _lastDefined(optionsArr.map(function (options) { return options.preserveWhitespaces; })),
    };
}
function _lastDefined(args) {
    for (var i = args.length - 1; i >= 0; i--) {
        if (args[i] !== undefined) {
            return args[i];
        }
    }
    return undefined;
}
function _mergeArrays(parts) {
    var result = [];
    parts.forEach(function (part) { return part && result.push.apply(result, tslib_1.__spread(part)); });
    return result;
}
//# sourceMappingURL=data:application/json;base64,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