'use strict';

var middlewareHostHeader = require('@aws-sdk/middleware-host-header');
var middlewareLogger = require('@aws-sdk/middleware-logger');
var middlewareRecursionDetection = require('@aws-sdk/middleware-recursion-detection');
var middlewareUserAgent = require('@aws-sdk/middleware-user-agent');
var configResolver = require('@smithy/config-resolver');
var core = require('@smithy/core');
var schema = require('@smithy/core/schema');
var middlewareContentLength = require('@smithy/middleware-content-length');
var middlewareEndpoint = require('@smithy/middleware-endpoint');
var middlewareRetry = require('@smithy/middleware-retry');
var smithyClient = require('@smithy/smithy-client');
var httpAuthSchemeProvider = require('./auth/httpAuthSchemeProvider');
var runtimeConfig = require('./runtimeConfig');
var regionConfigResolver = require('@aws-sdk/region-config-resolver');
var protocolHttp = require('@smithy/protocol-http');
var utilWaiter = require('@smithy/util-waiter');

const resolveClientEndpointParameters = (options) => {
    return Object.assign(options, {
        useDualstackEndpoint: options.useDualstackEndpoint ?? false,
        useFipsEndpoint: options.useFipsEndpoint ?? false,
        defaultSigningName: "cloudfront",
    });
};
const commonParams = {
    UseFIPS: { type: "builtInParams", name: "useFipsEndpoint" },
    Endpoint: { type: "builtInParams", name: "endpoint" },
    Region: { type: "builtInParams", name: "region" },
    UseDualStack: { type: "builtInParams", name: "useDualstackEndpoint" },
};

const getHttpAuthExtensionConfiguration = (runtimeConfig) => {
    const _httpAuthSchemes = runtimeConfig.httpAuthSchemes;
    let _httpAuthSchemeProvider = runtimeConfig.httpAuthSchemeProvider;
    let _credentials = runtimeConfig.credentials;
    return {
        setHttpAuthScheme(httpAuthScheme) {
            const index = _httpAuthSchemes.findIndex((scheme) => scheme.schemeId === httpAuthScheme.schemeId);
            if (index === -1) {
                _httpAuthSchemes.push(httpAuthScheme);
            }
            else {
                _httpAuthSchemes.splice(index, 1, httpAuthScheme);
            }
        },
        httpAuthSchemes() {
            return _httpAuthSchemes;
        },
        setHttpAuthSchemeProvider(httpAuthSchemeProvider) {
            _httpAuthSchemeProvider = httpAuthSchemeProvider;
        },
        httpAuthSchemeProvider() {
            return _httpAuthSchemeProvider;
        },
        setCredentials(credentials) {
            _credentials = credentials;
        },
        credentials() {
            return _credentials;
        },
    };
};
const resolveHttpAuthRuntimeConfig = (config) => {
    return {
        httpAuthSchemes: config.httpAuthSchemes(),
        httpAuthSchemeProvider: config.httpAuthSchemeProvider(),
        credentials: config.credentials(),
    };
};

const resolveRuntimeExtensions = (runtimeConfig, extensions) => {
    const extensionConfiguration = Object.assign(regionConfigResolver.getAwsRegionExtensionConfiguration(runtimeConfig), smithyClient.getDefaultExtensionConfiguration(runtimeConfig), protocolHttp.getHttpHandlerExtensionConfiguration(runtimeConfig), getHttpAuthExtensionConfiguration(runtimeConfig));
    extensions.forEach((extension) => extension.configure(extensionConfiguration));
    return Object.assign(runtimeConfig, regionConfigResolver.resolveAwsRegionExtensionConfiguration(extensionConfiguration), smithyClient.resolveDefaultRuntimeConfig(extensionConfiguration), protocolHttp.resolveHttpHandlerRuntimeConfig(extensionConfiguration), resolveHttpAuthRuntimeConfig(extensionConfiguration));
};

class CloudFrontClient extends smithyClient.Client {
    config;
    constructor(...[configuration]) {
        const _config_0 = runtimeConfig.getRuntimeConfig(configuration || {});
        super(_config_0);
        this.initConfig = _config_0;
        const _config_1 = resolveClientEndpointParameters(_config_0);
        const _config_2 = middlewareUserAgent.resolveUserAgentConfig(_config_1);
        const _config_3 = middlewareRetry.resolveRetryConfig(_config_2);
        const _config_4 = configResolver.resolveRegionConfig(_config_3);
        const _config_5 = middlewareHostHeader.resolveHostHeaderConfig(_config_4);
        const _config_6 = middlewareEndpoint.resolveEndpointConfig(_config_5);
        const _config_7 = httpAuthSchemeProvider.resolveHttpAuthSchemeConfig(_config_6);
        const _config_8 = resolveRuntimeExtensions(_config_7, configuration?.extensions || []);
        this.config = _config_8;
        this.middlewareStack.use(schema.getSchemaSerdePlugin(this.config));
        this.middlewareStack.use(middlewareUserAgent.getUserAgentPlugin(this.config));
        this.middlewareStack.use(middlewareRetry.getRetryPlugin(this.config));
        this.middlewareStack.use(middlewareContentLength.getContentLengthPlugin(this.config));
        this.middlewareStack.use(middlewareHostHeader.getHostHeaderPlugin(this.config));
        this.middlewareStack.use(middlewareLogger.getLoggerPlugin(this.config));
        this.middlewareStack.use(middlewareRecursionDetection.getRecursionDetectionPlugin(this.config));
        this.middlewareStack.use(core.getHttpAuthSchemeEndpointRuleSetPlugin(this.config, {
            httpAuthSchemeParametersProvider: httpAuthSchemeProvider.defaultCloudFrontHttpAuthSchemeParametersProvider,
            identityProviderConfigProvider: async (config) => new core.DefaultIdentityProviderConfig({
                "aws.auth#sigv4": config.credentials,
            }),
        }));
        this.middlewareStack.use(core.getHttpSigningPlugin(this.config));
    }
    destroy() {
        super.destroy();
    }
}

class CloudFrontServiceException extends smithyClient.ServiceException {
    constructor(options) {
        super(options);
        Object.setPrototypeOf(this, CloudFrontServiceException.prototype);
    }
}

class AccessDenied extends CloudFrontServiceException {
    name = "AccessDenied";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "AccessDenied",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, AccessDenied.prototype);
        this.Message = opts.Message;
    }
}
class IllegalUpdate extends CloudFrontServiceException {
    name = "IllegalUpdate";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "IllegalUpdate",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, IllegalUpdate.prototype);
        this.Message = opts.Message;
    }
}
class InvalidArgument extends CloudFrontServiceException {
    name = "InvalidArgument";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidArgument",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidArgument.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchDistribution extends CloudFrontServiceException {
    name = "NoSuchDistribution";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchDistribution",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchDistribution.prototype);
        this.Message = opts.Message;
    }
}
class TooManyDistributionCNAMEs extends CloudFrontServiceException {
    name = "TooManyDistributionCNAMEs";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyDistributionCNAMEs",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyDistributionCNAMEs.prototype);
        this.Message = opts.Message;
    }
}
class EntityNotFound extends CloudFrontServiceException {
    name = "EntityNotFound";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "EntityNotFound",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, EntityNotFound.prototype);
        this.Message = opts.Message;
    }
}
class InvalidIfMatchVersion extends CloudFrontServiceException {
    name = "InvalidIfMatchVersion";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidIfMatchVersion",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidIfMatchVersion.prototype);
        this.Message = opts.Message;
    }
}
class PreconditionFailed extends CloudFrontServiceException {
    name = "PreconditionFailed";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "PreconditionFailed",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, PreconditionFailed.prototype);
        this.Message = opts.Message;
    }
}
class BatchTooLarge extends CloudFrontServiceException {
    name = "BatchTooLarge";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "BatchTooLarge",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, BatchTooLarge.prototype);
        this.Message = opts.Message;
    }
}
class CachePolicyAlreadyExists extends CloudFrontServiceException {
    name = "CachePolicyAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "CachePolicyAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, CachePolicyAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class CachePolicyInUse extends CloudFrontServiceException {
    name = "CachePolicyInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "CachePolicyInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, CachePolicyInUse.prototype);
        this.Message = opts.Message;
    }
}
class CannotChangeImmutablePublicKeyFields extends CloudFrontServiceException {
    name = "CannotChangeImmutablePublicKeyFields";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "CannotChangeImmutablePublicKeyFields",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, CannotChangeImmutablePublicKeyFields.prototype);
        this.Message = opts.Message;
    }
}
class CannotDeleteEntityWhileInUse extends CloudFrontServiceException {
    name = "CannotDeleteEntityWhileInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "CannotDeleteEntityWhileInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, CannotDeleteEntityWhileInUse.prototype);
        this.Message = opts.Message;
    }
}
class CannotUpdateEntityWhileInUse extends CloudFrontServiceException {
    name = "CannotUpdateEntityWhileInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "CannotUpdateEntityWhileInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, CannotUpdateEntityWhileInUse.prototype);
        this.Message = opts.Message;
    }
}
class CNAMEAlreadyExists extends CloudFrontServiceException {
    name = "CNAMEAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "CNAMEAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, CNAMEAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class DistributionAlreadyExists extends CloudFrontServiceException {
    name = "DistributionAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "DistributionAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, DistributionAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class IllegalFieldLevelEncryptionConfigAssociationWithCacheBehavior extends CloudFrontServiceException {
    name = "IllegalFieldLevelEncryptionConfigAssociationWithCacheBehavior";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "IllegalFieldLevelEncryptionConfigAssociationWithCacheBehavior",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, IllegalFieldLevelEncryptionConfigAssociationWithCacheBehavior.prototype);
        this.Message = opts.Message;
    }
}
class InconsistentQuantities extends CloudFrontServiceException {
    name = "InconsistentQuantities";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InconsistentQuantities",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InconsistentQuantities.prototype);
        this.Message = opts.Message;
    }
}
class InvalidDefaultRootObject extends CloudFrontServiceException {
    name = "InvalidDefaultRootObject";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidDefaultRootObject",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidDefaultRootObject.prototype);
        this.Message = opts.Message;
    }
}
class InvalidErrorCode extends CloudFrontServiceException {
    name = "InvalidErrorCode";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidErrorCode",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidErrorCode.prototype);
        this.Message = opts.Message;
    }
}
class InvalidForwardCookies extends CloudFrontServiceException {
    name = "InvalidForwardCookies";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidForwardCookies",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidForwardCookies.prototype);
        this.Message = opts.Message;
    }
}
class InvalidFunctionAssociation extends CloudFrontServiceException {
    name = "InvalidFunctionAssociation";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidFunctionAssociation",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidFunctionAssociation.prototype);
        this.Message = opts.Message;
    }
}
class InvalidGeoRestrictionParameter extends CloudFrontServiceException {
    name = "InvalidGeoRestrictionParameter";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidGeoRestrictionParameter",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidGeoRestrictionParameter.prototype);
        this.Message = opts.Message;
    }
}
class InvalidHeadersForS3Origin extends CloudFrontServiceException {
    name = "InvalidHeadersForS3Origin";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidHeadersForS3Origin",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidHeadersForS3Origin.prototype);
        this.Message = opts.Message;
    }
}
class InvalidLambdaFunctionAssociation extends CloudFrontServiceException {
    name = "InvalidLambdaFunctionAssociation";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidLambdaFunctionAssociation",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidLambdaFunctionAssociation.prototype);
        this.Message = opts.Message;
    }
}
class InvalidLocationCode extends CloudFrontServiceException {
    name = "InvalidLocationCode";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidLocationCode",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidLocationCode.prototype);
        this.Message = opts.Message;
    }
}
class InvalidMinimumProtocolVersion extends CloudFrontServiceException {
    name = "InvalidMinimumProtocolVersion";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidMinimumProtocolVersion",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidMinimumProtocolVersion.prototype);
        this.Message = opts.Message;
    }
}
class InvalidOrigin extends CloudFrontServiceException {
    name = "InvalidOrigin";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidOrigin",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidOrigin.prototype);
        this.Message = opts.Message;
    }
}
class InvalidOriginAccessControl extends CloudFrontServiceException {
    name = "InvalidOriginAccessControl";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidOriginAccessControl",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidOriginAccessControl.prototype);
        this.Message = opts.Message;
    }
}
class InvalidOriginAccessIdentity extends CloudFrontServiceException {
    name = "InvalidOriginAccessIdentity";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidOriginAccessIdentity",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidOriginAccessIdentity.prototype);
        this.Message = opts.Message;
    }
}
class InvalidOriginKeepaliveTimeout extends CloudFrontServiceException {
    name = "InvalidOriginKeepaliveTimeout";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidOriginKeepaliveTimeout",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidOriginKeepaliveTimeout.prototype);
        this.Message = opts.Message;
    }
}
class InvalidOriginReadTimeout extends CloudFrontServiceException {
    name = "InvalidOriginReadTimeout";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidOriginReadTimeout",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidOriginReadTimeout.prototype);
        this.Message = opts.Message;
    }
}
class InvalidProtocolSettings extends CloudFrontServiceException {
    name = "InvalidProtocolSettings";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidProtocolSettings",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidProtocolSettings.prototype);
        this.Message = opts.Message;
    }
}
class InvalidQueryStringParameters extends CloudFrontServiceException {
    name = "InvalidQueryStringParameters";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidQueryStringParameters",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidQueryStringParameters.prototype);
        this.Message = opts.Message;
    }
}
class InvalidRelativePath extends CloudFrontServiceException {
    name = "InvalidRelativePath";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidRelativePath",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidRelativePath.prototype);
        this.Message = opts.Message;
    }
}
class InvalidRequiredProtocol extends CloudFrontServiceException {
    name = "InvalidRequiredProtocol";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidRequiredProtocol",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidRequiredProtocol.prototype);
        this.Message = opts.Message;
    }
}
class InvalidResponseCode extends CloudFrontServiceException {
    name = "InvalidResponseCode";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidResponseCode",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidResponseCode.prototype);
        this.Message = opts.Message;
    }
}
class InvalidTTLOrder extends CloudFrontServiceException {
    name = "InvalidTTLOrder";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidTTLOrder",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidTTLOrder.prototype);
        this.Message = opts.Message;
    }
}
class InvalidViewerCertificate extends CloudFrontServiceException {
    name = "InvalidViewerCertificate";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidViewerCertificate",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidViewerCertificate.prototype);
        this.Message = opts.Message;
    }
}
class InvalidWebACLId extends CloudFrontServiceException {
    name = "InvalidWebACLId";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidWebACLId",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidWebACLId.prototype);
        this.Message = opts.Message;
    }
}
class MissingBody extends CloudFrontServiceException {
    name = "MissingBody";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "MissingBody",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, MissingBody.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchCachePolicy extends CloudFrontServiceException {
    name = "NoSuchCachePolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchCachePolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchCachePolicy.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchFieldLevelEncryptionConfig extends CloudFrontServiceException {
    name = "NoSuchFieldLevelEncryptionConfig";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchFieldLevelEncryptionConfig",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchFieldLevelEncryptionConfig.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchOrigin extends CloudFrontServiceException {
    name = "NoSuchOrigin";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchOrigin",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchOrigin.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchOriginRequestPolicy extends CloudFrontServiceException {
    name = "NoSuchOriginRequestPolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchOriginRequestPolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchOriginRequestPolicy.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchRealtimeLogConfig extends CloudFrontServiceException {
    name = "NoSuchRealtimeLogConfig";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchRealtimeLogConfig",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchRealtimeLogConfig.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchResponseHeadersPolicy extends CloudFrontServiceException {
    name = "NoSuchResponseHeadersPolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchResponseHeadersPolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchResponseHeadersPolicy.prototype);
        this.Message = opts.Message;
    }
}
class RealtimeLogConfigOwnerMismatch extends CloudFrontServiceException {
    name = "RealtimeLogConfigOwnerMismatch";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "RealtimeLogConfigOwnerMismatch",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, RealtimeLogConfigOwnerMismatch.prototype);
        this.Message = opts.Message;
    }
}
class TooManyCacheBehaviors extends CloudFrontServiceException {
    name = "TooManyCacheBehaviors";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyCacheBehaviors",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyCacheBehaviors.prototype);
        this.Message = opts.Message;
    }
}
class TooManyCertificates extends CloudFrontServiceException {
    name = "TooManyCertificates";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyCertificates",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyCertificates.prototype);
        this.Message = opts.Message;
    }
}
class TooManyCookieNamesInWhiteList extends CloudFrontServiceException {
    name = "TooManyCookieNamesInWhiteList";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyCookieNamesInWhiteList",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyCookieNamesInWhiteList.prototype);
        this.Message = opts.Message;
    }
}
class TooManyDistributions extends CloudFrontServiceException {
    name = "TooManyDistributions";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyDistributions",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyDistributions.prototype);
        this.Message = opts.Message;
    }
}
class TooManyDistributionsAssociatedToCachePolicy extends CloudFrontServiceException {
    name = "TooManyDistributionsAssociatedToCachePolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyDistributionsAssociatedToCachePolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyDistributionsAssociatedToCachePolicy.prototype);
        this.Message = opts.Message;
    }
}
class TooManyDistributionsAssociatedToFieldLevelEncryptionConfig extends CloudFrontServiceException {
    name = "TooManyDistributionsAssociatedToFieldLevelEncryptionConfig";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyDistributionsAssociatedToFieldLevelEncryptionConfig",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyDistributionsAssociatedToFieldLevelEncryptionConfig.prototype);
        this.Message = opts.Message;
    }
}
class TooManyDistributionsAssociatedToKeyGroup extends CloudFrontServiceException {
    name = "TooManyDistributionsAssociatedToKeyGroup";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyDistributionsAssociatedToKeyGroup",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyDistributionsAssociatedToKeyGroup.prototype);
        this.Message = opts.Message;
    }
}
class TooManyDistributionsAssociatedToOriginAccessControl extends CloudFrontServiceException {
    name = "TooManyDistributionsAssociatedToOriginAccessControl";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyDistributionsAssociatedToOriginAccessControl",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyDistributionsAssociatedToOriginAccessControl.prototype);
        this.Message = opts.Message;
    }
}
class TooManyDistributionsAssociatedToOriginRequestPolicy extends CloudFrontServiceException {
    name = "TooManyDistributionsAssociatedToOriginRequestPolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyDistributionsAssociatedToOriginRequestPolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyDistributionsAssociatedToOriginRequestPolicy.prototype);
        this.Message = opts.Message;
    }
}
class TooManyDistributionsAssociatedToResponseHeadersPolicy extends CloudFrontServiceException {
    name = "TooManyDistributionsAssociatedToResponseHeadersPolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyDistributionsAssociatedToResponseHeadersPolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyDistributionsAssociatedToResponseHeadersPolicy.prototype);
        this.Message = opts.Message;
    }
}
class TooManyDistributionsWithFunctionAssociations extends CloudFrontServiceException {
    name = "TooManyDistributionsWithFunctionAssociations";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyDistributionsWithFunctionAssociations",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyDistributionsWithFunctionAssociations.prototype);
        this.Message = opts.Message;
    }
}
class TooManyDistributionsWithLambdaAssociations extends CloudFrontServiceException {
    name = "TooManyDistributionsWithLambdaAssociations";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyDistributionsWithLambdaAssociations",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyDistributionsWithLambdaAssociations.prototype);
        this.Message = opts.Message;
    }
}
class TooManyDistributionsWithSingleFunctionARN extends CloudFrontServiceException {
    name = "TooManyDistributionsWithSingleFunctionARN";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyDistributionsWithSingleFunctionARN",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyDistributionsWithSingleFunctionARN.prototype);
        this.Message = opts.Message;
    }
}
class TooManyFunctionAssociations extends CloudFrontServiceException {
    name = "TooManyFunctionAssociations";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyFunctionAssociations",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyFunctionAssociations.prototype);
        this.Message = opts.Message;
    }
}
class TooManyHeadersInForwardedValues extends CloudFrontServiceException {
    name = "TooManyHeadersInForwardedValues";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyHeadersInForwardedValues",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyHeadersInForwardedValues.prototype);
        this.Message = opts.Message;
    }
}
class TooManyKeyGroupsAssociatedToDistribution extends CloudFrontServiceException {
    name = "TooManyKeyGroupsAssociatedToDistribution";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyKeyGroupsAssociatedToDistribution",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyKeyGroupsAssociatedToDistribution.prototype);
        this.Message = opts.Message;
    }
}
class TooManyLambdaFunctionAssociations extends CloudFrontServiceException {
    name = "TooManyLambdaFunctionAssociations";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyLambdaFunctionAssociations",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyLambdaFunctionAssociations.prototype);
        this.Message = opts.Message;
    }
}
class TooManyOriginCustomHeaders extends CloudFrontServiceException {
    name = "TooManyOriginCustomHeaders";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyOriginCustomHeaders",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyOriginCustomHeaders.prototype);
        this.Message = opts.Message;
    }
}
class TooManyOriginGroupsPerDistribution extends CloudFrontServiceException {
    name = "TooManyOriginGroupsPerDistribution";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyOriginGroupsPerDistribution",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyOriginGroupsPerDistribution.prototype);
        this.Message = opts.Message;
    }
}
class TooManyOrigins extends CloudFrontServiceException {
    name = "TooManyOrigins";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyOrigins",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyOrigins.prototype);
        this.Message = opts.Message;
    }
}
class TooManyQueryStringParameters extends CloudFrontServiceException {
    name = "TooManyQueryStringParameters";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyQueryStringParameters",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyQueryStringParameters.prototype);
        this.Message = opts.Message;
    }
}
class TooManyTrustedSigners extends CloudFrontServiceException {
    name = "TooManyTrustedSigners";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyTrustedSigners",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyTrustedSigners.prototype);
        this.Message = opts.Message;
    }
}
class TrustedKeyGroupDoesNotExist extends CloudFrontServiceException {
    name = "TrustedKeyGroupDoesNotExist";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TrustedKeyGroupDoesNotExist",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TrustedKeyGroupDoesNotExist.prototype);
        this.Message = opts.Message;
    }
}
class TrustedSignerDoesNotExist extends CloudFrontServiceException {
    name = "TrustedSignerDoesNotExist";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TrustedSignerDoesNotExist",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TrustedSignerDoesNotExist.prototype);
        this.Message = opts.Message;
    }
}
class EntityAlreadyExists extends CloudFrontServiceException {
    name = "EntityAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "EntityAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, EntityAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class EntityLimitExceeded extends CloudFrontServiceException {
    name = "EntityLimitExceeded";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "EntityLimitExceeded",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, EntityLimitExceeded.prototype);
        this.Message = opts.Message;
    }
}
class InvalidTagging extends CloudFrontServiceException {
    name = "InvalidTagging";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidTagging",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidTagging.prototype);
        this.Message = opts.Message;
    }
}
class UnsupportedOperation extends CloudFrontServiceException {
    name = "UnsupportedOperation";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "UnsupportedOperation",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, UnsupportedOperation.prototype);
        this.Message = opts.Message;
    }
}
class TooManyCachePolicies extends CloudFrontServiceException {
    name = "TooManyCachePolicies";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyCachePolicies",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyCachePolicies.prototype);
        this.Message = opts.Message;
    }
}
class TooManyCookiesInCachePolicy extends CloudFrontServiceException {
    name = "TooManyCookiesInCachePolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyCookiesInCachePolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyCookiesInCachePolicy.prototype);
        this.Message = opts.Message;
    }
}
class TooManyHeadersInCachePolicy extends CloudFrontServiceException {
    name = "TooManyHeadersInCachePolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyHeadersInCachePolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyHeadersInCachePolicy.prototype);
        this.Message = opts.Message;
    }
}
class TooManyQueryStringsInCachePolicy extends CloudFrontServiceException {
    name = "TooManyQueryStringsInCachePolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyQueryStringsInCachePolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyQueryStringsInCachePolicy.prototype);
        this.Message = opts.Message;
    }
}
class CloudFrontOriginAccessIdentityAlreadyExists extends CloudFrontServiceException {
    name = "CloudFrontOriginAccessIdentityAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "CloudFrontOriginAccessIdentityAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, CloudFrontOriginAccessIdentityAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class TooManyCloudFrontOriginAccessIdentities extends CloudFrontServiceException {
    name = "TooManyCloudFrontOriginAccessIdentities";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyCloudFrontOriginAccessIdentities",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyCloudFrontOriginAccessIdentities.prototype);
        this.Message = opts.Message;
    }
}
class EntitySizeLimitExceeded extends CloudFrontServiceException {
    name = "EntitySizeLimitExceeded";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "EntitySizeLimitExceeded",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, EntitySizeLimitExceeded.prototype);
        this.Message = opts.Message;
    }
}
class ContinuousDeploymentPolicyAlreadyExists extends CloudFrontServiceException {
    name = "ContinuousDeploymentPolicyAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "ContinuousDeploymentPolicyAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ContinuousDeploymentPolicyAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class StagingDistributionInUse extends CloudFrontServiceException {
    name = "StagingDistributionInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "StagingDistributionInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, StagingDistributionInUse.prototype);
        this.Message = opts.Message;
    }
}
class TooManyContinuousDeploymentPolicies extends CloudFrontServiceException {
    name = "TooManyContinuousDeploymentPolicies";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyContinuousDeploymentPolicies",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyContinuousDeploymentPolicies.prototype);
        this.Message = opts.Message;
    }
}
class ContinuousDeploymentPolicyInUse extends CloudFrontServiceException {
    name = "ContinuousDeploymentPolicyInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "ContinuousDeploymentPolicyInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ContinuousDeploymentPolicyInUse.prototype);
        this.Message = opts.Message;
    }
}
class IllegalOriginAccessConfiguration extends CloudFrontServiceException {
    name = "IllegalOriginAccessConfiguration";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "IllegalOriginAccessConfiguration",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, IllegalOriginAccessConfiguration.prototype);
        this.Message = opts.Message;
    }
}
class InvalidDomainNameForOriginAccessControl extends CloudFrontServiceException {
    name = "InvalidDomainNameForOriginAccessControl";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidDomainNameForOriginAccessControl",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidDomainNameForOriginAccessControl.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchContinuousDeploymentPolicy extends CloudFrontServiceException {
    name = "NoSuchContinuousDeploymentPolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchContinuousDeploymentPolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchContinuousDeploymentPolicy.prototype);
        this.Message = opts.Message;
    }
}
class InvalidAssociation extends CloudFrontServiceException {
    name = "InvalidAssociation";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidAssociation",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidAssociation.prototype);
        this.Message = opts.Message;
    }
}
class FieldLevelEncryptionConfigAlreadyExists extends CloudFrontServiceException {
    name = "FieldLevelEncryptionConfigAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "FieldLevelEncryptionConfigAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, FieldLevelEncryptionConfigAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchFieldLevelEncryptionProfile extends CloudFrontServiceException {
    name = "NoSuchFieldLevelEncryptionProfile";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchFieldLevelEncryptionProfile",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchFieldLevelEncryptionProfile.prototype);
        this.Message = opts.Message;
    }
}
class QueryArgProfileEmpty extends CloudFrontServiceException {
    name = "QueryArgProfileEmpty";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "QueryArgProfileEmpty",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, QueryArgProfileEmpty.prototype);
        this.Message = opts.Message;
    }
}
class TooManyFieldLevelEncryptionConfigs extends CloudFrontServiceException {
    name = "TooManyFieldLevelEncryptionConfigs";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyFieldLevelEncryptionConfigs",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyFieldLevelEncryptionConfigs.prototype);
        this.Message = opts.Message;
    }
}
class TooManyFieldLevelEncryptionContentTypeProfiles extends CloudFrontServiceException {
    name = "TooManyFieldLevelEncryptionContentTypeProfiles";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyFieldLevelEncryptionContentTypeProfiles",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyFieldLevelEncryptionContentTypeProfiles.prototype);
        this.Message = opts.Message;
    }
}
class TooManyFieldLevelEncryptionQueryArgProfiles extends CloudFrontServiceException {
    name = "TooManyFieldLevelEncryptionQueryArgProfiles";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyFieldLevelEncryptionQueryArgProfiles",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyFieldLevelEncryptionQueryArgProfiles.prototype);
        this.Message = opts.Message;
    }
}
class FieldLevelEncryptionProfileAlreadyExists extends CloudFrontServiceException {
    name = "FieldLevelEncryptionProfileAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "FieldLevelEncryptionProfileAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, FieldLevelEncryptionProfileAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class FieldLevelEncryptionProfileSizeExceeded extends CloudFrontServiceException {
    name = "FieldLevelEncryptionProfileSizeExceeded";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "FieldLevelEncryptionProfileSizeExceeded",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, FieldLevelEncryptionProfileSizeExceeded.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchPublicKey extends CloudFrontServiceException {
    name = "NoSuchPublicKey";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchPublicKey",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchPublicKey.prototype);
        this.Message = opts.Message;
    }
}
class TooManyFieldLevelEncryptionEncryptionEntities extends CloudFrontServiceException {
    name = "TooManyFieldLevelEncryptionEncryptionEntities";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyFieldLevelEncryptionEncryptionEntities",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyFieldLevelEncryptionEncryptionEntities.prototype);
        this.Message = opts.Message;
    }
}
class TooManyFieldLevelEncryptionFieldPatterns extends CloudFrontServiceException {
    name = "TooManyFieldLevelEncryptionFieldPatterns";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyFieldLevelEncryptionFieldPatterns",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyFieldLevelEncryptionFieldPatterns.prototype);
        this.Message = opts.Message;
    }
}
class TooManyFieldLevelEncryptionProfiles extends CloudFrontServiceException {
    name = "TooManyFieldLevelEncryptionProfiles";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyFieldLevelEncryptionProfiles",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyFieldLevelEncryptionProfiles.prototype);
        this.Message = opts.Message;
    }
}
class FunctionAlreadyExists extends CloudFrontServiceException {
    name = "FunctionAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "FunctionAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, FunctionAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class FunctionSizeLimitExceeded extends CloudFrontServiceException {
    name = "FunctionSizeLimitExceeded";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "FunctionSizeLimitExceeded",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, FunctionSizeLimitExceeded.prototype);
        this.Message = opts.Message;
    }
}
class TooManyFunctions extends CloudFrontServiceException {
    name = "TooManyFunctions";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyFunctions",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyFunctions.prototype);
        this.Message = opts.Message;
    }
}
class TooManyInvalidationsInProgress extends CloudFrontServiceException {
    name = "TooManyInvalidationsInProgress";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyInvalidationsInProgress",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyInvalidationsInProgress.prototype);
        this.Message = opts.Message;
    }
}
class KeyGroupAlreadyExists extends CloudFrontServiceException {
    name = "KeyGroupAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "KeyGroupAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, KeyGroupAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class TooManyKeyGroups extends CloudFrontServiceException {
    name = "TooManyKeyGroups";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyKeyGroups",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyKeyGroups.prototype);
        this.Message = opts.Message;
    }
}
class TooManyPublicKeysInKeyGroup extends CloudFrontServiceException {
    name = "TooManyPublicKeysInKeyGroup";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyPublicKeysInKeyGroup",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyPublicKeysInKeyGroup.prototype);
        this.Message = opts.Message;
    }
}
class MonitoringSubscriptionAlreadyExists extends CloudFrontServiceException {
    name = "MonitoringSubscriptionAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "MonitoringSubscriptionAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, MonitoringSubscriptionAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class OriginAccessControlAlreadyExists extends CloudFrontServiceException {
    name = "OriginAccessControlAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "OriginAccessControlAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, OriginAccessControlAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class TooManyOriginAccessControls extends CloudFrontServiceException {
    name = "TooManyOriginAccessControls";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyOriginAccessControls",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyOriginAccessControls.prototype);
        this.Message = opts.Message;
    }
}
class OriginRequestPolicyAlreadyExists extends CloudFrontServiceException {
    name = "OriginRequestPolicyAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "OriginRequestPolicyAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, OriginRequestPolicyAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class TooManyCookiesInOriginRequestPolicy extends CloudFrontServiceException {
    name = "TooManyCookiesInOriginRequestPolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyCookiesInOriginRequestPolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyCookiesInOriginRequestPolicy.prototype);
        this.Message = opts.Message;
    }
}
class TooManyHeadersInOriginRequestPolicy extends CloudFrontServiceException {
    name = "TooManyHeadersInOriginRequestPolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyHeadersInOriginRequestPolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyHeadersInOriginRequestPolicy.prototype);
        this.Message = opts.Message;
    }
}
class TooManyOriginRequestPolicies extends CloudFrontServiceException {
    name = "TooManyOriginRequestPolicies";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyOriginRequestPolicies",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyOriginRequestPolicies.prototype);
        this.Message = opts.Message;
    }
}
class TooManyQueryStringsInOriginRequestPolicy extends CloudFrontServiceException {
    name = "TooManyQueryStringsInOriginRequestPolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyQueryStringsInOriginRequestPolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyQueryStringsInOriginRequestPolicy.prototype);
        this.Message = opts.Message;
    }
}
class PublicKeyAlreadyExists extends CloudFrontServiceException {
    name = "PublicKeyAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "PublicKeyAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, PublicKeyAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class TooManyPublicKeys extends CloudFrontServiceException {
    name = "TooManyPublicKeys";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyPublicKeys",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyPublicKeys.prototype);
        this.Message = opts.Message;
    }
}
class RealtimeLogConfigAlreadyExists extends CloudFrontServiceException {
    name = "RealtimeLogConfigAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "RealtimeLogConfigAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, RealtimeLogConfigAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class TooManyRealtimeLogConfigs extends CloudFrontServiceException {
    name = "TooManyRealtimeLogConfigs";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyRealtimeLogConfigs",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyRealtimeLogConfigs.prototype);
        this.Message = opts.Message;
    }
}
class ResponseHeadersPolicyAlreadyExists extends CloudFrontServiceException {
    name = "ResponseHeadersPolicyAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "ResponseHeadersPolicyAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ResponseHeadersPolicyAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class TooLongCSPInResponseHeadersPolicy extends CloudFrontServiceException {
    name = "TooLongCSPInResponseHeadersPolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooLongCSPInResponseHeadersPolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooLongCSPInResponseHeadersPolicy.prototype);
        this.Message = opts.Message;
    }
}
class TooManyCustomHeadersInResponseHeadersPolicy extends CloudFrontServiceException {
    name = "TooManyCustomHeadersInResponseHeadersPolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyCustomHeadersInResponseHeadersPolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyCustomHeadersInResponseHeadersPolicy.prototype);
        this.Message = opts.Message;
    }
}
class TooManyRemoveHeadersInResponseHeadersPolicy extends CloudFrontServiceException {
    name = "TooManyRemoveHeadersInResponseHeadersPolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyRemoveHeadersInResponseHeadersPolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyRemoveHeadersInResponseHeadersPolicy.prototype);
        this.Message = opts.Message;
    }
}
class TooManyResponseHeadersPolicies extends CloudFrontServiceException {
    name = "TooManyResponseHeadersPolicies";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyResponseHeadersPolicies",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyResponseHeadersPolicies.prototype);
        this.Message = opts.Message;
    }
}
class StreamingDistributionAlreadyExists extends CloudFrontServiceException {
    name = "StreamingDistributionAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "StreamingDistributionAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, StreamingDistributionAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class TooManyStreamingDistributionCNAMEs extends CloudFrontServiceException {
    name = "TooManyStreamingDistributionCNAMEs";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyStreamingDistributionCNAMEs",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyStreamingDistributionCNAMEs.prototype);
        this.Message = opts.Message;
    }
}
class TooManyStreamingDistributions extends CloudFrontServiceException {
    name = "TooManyStreamingDistributions";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyStreamingDistributions",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyStreamingDistributions.prototype);
        this.Message = opts.Message;
    }
}
class IllegalDelete extends CloudFrontServiceException {
    name = "IllegalDelete";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "IllegalDelete",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, IllegalDelete.prototype);
        this.Message = opts.Message;
    }
}
class CloudFrontOriginAccessIdentityInUse extends CloudFrontServiceException {
    name = "CloudFrontOriginAccessIdentityInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "CloudFrontOriginAccessIdentityInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, CloudFrontOriginAccessIdentityInUse.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchCloudFrontOriginAccessIdentity extends CloudFrontServiceException {
    name = "NoSuchCloudFrontOriginAccessIdentity";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchCloudFrontOriginAccessIdentity",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchCloudFrontOriginAccessIdentity.prototype);
        this.Message = opts.Message;
    }
}
class ResourceNotDisabled extends CloudFrontServiceException {
    name = "ResourceNotDisabled";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "ResourceNotDisabled",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ResourceNotDisabled.prototype);
        this.Message = opts.Message;
    }
}
class DistributionNotDisabled extends CloudFrontServiceException {
    name = "DistributionNotDisabled";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "DistributionNotDisabled",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, DistributionNotDisabled.prototype);
        this.Message = opts.Message;
    }
}
class ResourceInUse extends CloudFrontServiceException {
    name = "ResourceInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "ResourceInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ResourceInUse.prototype);
        this.Message = opts.Message;
    }
}
class FieldLevelEncryptionConfigInUse extends CloudFrontServiceException {
    name = "FieldLevelEncryptionConfigInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "FieldLevelEncryptionConfigInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, FieldLevelEncryptionConfigInUse.prototype);
        this.Message = opts.Message;
    }
}
class FieldLevelEncryptionProfileInUse extends CloudFrontServiceException {
    name = "FieldLevelEncryptionProfileInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "FieldLevelEncryptionProfileInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, FieldLevelEncryptionProfileInUse.prototype);
        this.Message = opts.Message;
    }
}
class FunctionInUse extends CloudFrontServiceException {
    name = "FunctionInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "FunctionInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, FunctionInUse.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchFunctionExists extends CloudFrontServiceException {
    name = "NoSuchFunctionExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchFunctionExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchFunctionExists.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchResource extends CloudFrontServiceException {
    name = "NoSuchResource";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchResource",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchResource.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchMonitoringSubscription extends CloudFrontServiceException {
    name = "NoSuchMonitoringSubscription";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchMonitoringSubscription",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchMonitoringSubscription.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchOriginAccessControl extends CloudFrontServiceException {
    name = "NoSuchOriginAccessControl";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchOriginAccessControl",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchOriginAccessControl.prototype);
        this.Message = opts.Message;
    }
}
class OriginAccessControlInUse extends CloudFrontServiceException {
    name = "OriginAccessControlInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "OriginAccessControlInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, OriginAccessControlInUse.prototype);
        this.Message = opts.Message;
    }
}
class OriginRequestPolicyInUse extends CloudFrontServiceException {
    name = "OriginRequestPolicyInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "OriginRequestPolicyInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, OriginRequestPolicyInUse.prototype);
        this.Message = opts.Message;
    }
}
class PublicKeyInUse extends CloudFrontServiceException {
    name = "PublicKeyInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "PublicKeyInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, PublicKeyInUse.prototype);
        this.Message = opts.Message;
    }
}
class RealtimeLogConfigInUse extends CloudFrontServiceException {
    name = "RealtimeLogConfigInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "RealtimeLogConfigInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, RealtimeLogConfigInUse.prototype);
        this.Message = opts.Message;
    }
}
class ResponseHeadersPolicyInUse extends CloudFrontServiceException {
    name = "ResponseHeadersPolicyInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "ResponseHeadersPolicyInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ResponseHeadersPolicyInUse.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchStreamingDistribution extends CloudFrontServiceException {
    name = "NoSuchStreamingDistribution";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchStreamingDistribution",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchStreamingDistribution.prototype);
        this.Message = opts.Message;
    }
}
class StreamingDistributionNotDisabled extends CloudFrontServiceException {
    name = "StreamingDistributionNotDisabled";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "StreamingDistributionNotDisabled",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, StreamingDistributionNotDisabled.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchInvalidation extends CloudFrontServiceException {
    name = "NoSuchInvalidation";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchInvalidation",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchInvalidation.prototype);
        this.Message = opts.Message;
    }
}
class TestFunctionFailed extends CloudFrontServiceException {
    name = "TestFunctionFailed";
    $fault = "server";
    Message;
    constructor(opts) {
        super({
            name: "TestFunctionFailed",
            $fault: "server",
            ...opts,
        });
        Object.setPrototypeOf(this, TestFunctionFailed.prototype);
        this.Message = opts.Message;
    }
}

const _A = "Aliases";
const _AA = "AssociateAlias";
const _AAN = "AwsAccountNumber";
const _AANL = "AwsAccountNumberList";
const _AAR = "AssociateAliasRequest";
const _ACAC = "AccessControlAllowCredentials";
const _ACAH = "AccessControlAllowHeaders";
const _ACAHL = "AccessControlAllowHeadersList";
const _ACAM = "AccessControlAllowMethods";
const _ACAML = "AccessControlAllowMethodsList";
const _ACAO = "AccessControlAllowOrigins";
const _ACAOL = "AccessControlAllowOriginsList";
const _ACEH = "AccessControlExposeHeaders";
const _ACEHL = "AccessControlExposeHeadersList";
const _ACMAS = "AccessControlMaxAgeSec";
const _ACMCA = "ACMCertificateArn";
const _AD = "AccessDenied";
const _ADTWACL = "AssociateDistributionTenantWebACL";
const _ADTWACLR = "AssociateDistributionTenantWebACLRequest";
const _ADTWACLRs = "AssociateDistributionTenantWebACLResult";
const _ADWACL = "AssociateDistributionWebACL";
const _ADWACLR = "AssociateDistributionWebACLRequest";
const _ADWACLRs = "AssociateDistributionWebACLResult";
const _AF = "AssociationFilter";
const _AI = "AnycastIps";
const _AICPR = "AliasICPRecordal";
const _AICPRl = "AliasICPRecordals";
const _AIL = "AnycastIpList";
const _AILC = "AnycastIpListCollection";
const _AILI = "AnycastIpListId";
const _AILS = "AnycastIpListSummary";
const _AILSn = "AnycastIpListSummaries";
const _AILn = "AnycastIpLists";
const _AIc = "AccountId";
const _AIn = "AnycastIp";
const _AL = "AliasList";
const _AM = "AllowedMethods";
const _ARN = "ARN";
const _ATKG = "ActiveTrustedKeyGroups";
const _ATS = "ActiveTrustedSigners";
const _ATSCN = "AdvertiseTrustStoreCaNames";
const _Ac = "Action";
const _Al = "Alias";
const _Ar = "Arn";
const _B = "Bucket";
const _BTL = "BatchTooLarge";
const _C = "Compress";
const _CA = "ConflictingAlias";
const _CAIL = "CreateAnycastIpList";
const _CAILR = "CreateAnycastIpListRequest";
const _CAILRr = "CreateAnycastIpListResult";
const _CAL = "ConflictingAliasesList";
const _CAe = "CertificateArn";
const _CAo = "ConnectionAttempts";
const _CAon = "ConflictingAliases";
const _CB = "CacheBehavior";
const _CBL = "CacheBehaviorList";
const _CBa = "CacheBehaviors";
const _CBo = "CookieBehavior";
const _CC = "CookiesConfig";
const _CCBS = "CaCertificatesBundleSource";
const _CCBSL = "CaCertificatesBundleS3Location";
const _CCDP = "CreateContinuousDeploymentPolicy";
const _CCDPR = "CreateContinuousDeploymentPolicyRequest";
const _CCDPRr = "CreateContinuousDeploymentPolicyResult";
const _CCF = "CreateConnectionFunction";
const _CCFOAI = "CreateCloudFrontOriginAccessIdentity";
const _CCFOAIR = "CreateCloudFrontOriginAccessIdentityRequest";
const _CCFOAIRr = "CreateCloudFrontOriginAccessIdentityResult";
const _CCFR = "CreateConnectionFunctionRequest";
const _CCFRr = "CreateConnectionFunctionResult";
const _CCG = "CreateConnectionGroup";
const _CCGR = "CreateConnectionGroupRequest";
const _CCGRr = "CreateConnectionGroupResult";
const _CCIPKF = "CannotChangeImmutablePublicKeyFields";
const _CCP = "CreateCachePolicy";
const _CCPR = "CreateCachePolicyRequest";
const _CCPRr = "CreateCachePolicyResult";
const _CCo = "CorsConfig";
const _CD = "CopyDistribution";
const _CDEWIU = "CannotDeleteEntityWhileInUse";
const _CDP = "ContinuousDeploymentPolicy";
const _CDPAE = "ContinuousDeploymentPolicyAlreadyExists";
const _CDPC = "ContinuousDeploymentPolicyConfig";
const _CDPI = "ContinuousDeploymentPolicyId";
const _CDPIU = "ContinuousDeploymentPolicyInUse";
const _CDPL = "ContinuousDeploymentPolicyList";
const _CDPS = "ContinuousDeploymentPolicySummary";
const _CDPSL = "ContinuousDeploymentPolicySummaryList";
const _CDR = "CopyDistributionRequest";
const _CDRo = "CopyDistributionResult";
const _CDRr = "CreateDistributionRequest";
const _CDRre = "CreateDistributionResult";
const _CDSHC = "ContinuousDeploymentSingleHeaderConfig";
const _CDSWC = "ContinuousDeploymentSingleWeightConfig";
const _CDT = "CreateDistributionTenant";
const _CDTR = "CreateDistributionTenantRequest";
const _CDTRr = "CreateDistributionTenantResult";
const _CDWT = "CreateDistributionWithTags";
const _CDWTR = "CreateDistributionWithTagsRequest";
const _CDWTRr = "CreateDistributionWithTagsResult";
const _CDr = "CreateDistribution";
const _CER = "CustomErrorResponse";
const _CERL = "CustomErrorResponseList";
const _CERu = "CustomErrorResponses";
const _CF = "ConnectionFunctions";
const _CFA = "ConnectionFunctionAssociation";
const _CFAo = "ConnectionFunctionArn";
const _CFC = "ConnectionFunctionConfig";
const _CFCo = "ConnectionFunctionCode";
const _CFDC = "CloudFrontDefaultCertificate";
const _CFEL = "ConnectionFunctionExecutionLogs";
const _CFEM = "ConnectionFunctionErrorMessage";
const _CFI = "ConnectionFunctionIdentifier";
const _CFLEC = "CreateFieldLevelEncryptionConfig";
const _CFLECR = "CreateFieldLevelEncryptionConfigRequest";
const _CFLECRr = "CreateFieldLevelEncryptionConfigResult";
const _CFLEP = "CreateFieldLevelEncryptionProfile";
const _CFLEPR = "CreateFieldLevelEncryptionProfileRequest";
const _CFLEPRr = "CreateFieldLevelEncryptionProfileResult";
const _CFO = "ConnectionFunctionOutput";
const _CFOAI = "CloudFrontOriginAccessIdentity";
const _CFOAIAE = "CloudFrontOriginAccessIdentityAlreadyExists";
const _CFOAIC = "CloudFrontOriginAccessIdentityConfig";
const _CFOAIIU = "CloudFrontOriginAccessIdentityInUse";
const _CFOAIL = "CloudFrontOriginAccessIdentityList";
const _CFOAIS = "CloudFrontOriginAccessIdentitySummary";
const _CFOAISL = "CloudFrontOriginAccessIdentitySummaryList";
const _CFR = "CreateFunctionRequest";
const _CFRr = "CreateFunctionResult";
const _CFS = "ConnectionFunctionSummary";
const _CFSL = "ConnectionFunctionSummaryList";
const _CFTR = "ConnectionFunctionTestResult";
const _CFr = "CreateFunction";
const _CG = "ConnectionGroup";
const _CGAF = "ConnectionGroupAssociationFilter";
const _CGI = "ConnectionGroupId";
const _CGS = "ConnectionGroupSummary";
const _CGSL = "ConnectionGroupSummaryList";
const _CGo = "ConnectionGroups";
const _CH = "CustomHeaders";
const _CHC = "CustomHeadersConfig";
const _CI = "CreateInvalidation";
const _CIFDT = "CreateInvalidationForDistributionTenant";
const _CIFDTR = "CreateInvalidationForDistributionTenantRequest";
const _CIFDTRr = "CreateInvalidationForDistributionTenantResult";
const _CIR = "CreateInvalidationRequest";
const _CIRr = "CreateInvalidationResult";
const _CKG = "CreateKeyGroup";
const _CKGR = "CreateKeyGroupRequest";
const _CKGRr = "CreateKeyGroupResult";
const _CKVS = "CreateKeyValueStore";
const _CKVSR = "CreateKeyValueStoreRequest";
const _CKVSRr = "CreateKeyValueStoreResult";
const _CM = "CachedMethods";
const _CMS = "CreateMonitoringSubscription";
const _CMSR = "CreateMonitoringSubscriptionRequest";
const _CMSRr = "CreateMonitoringSubscriptionResult";
const _CMo = "ConnectionMode";
const _CN = "CookieNames";
const _CNAME = "CNAME";
const _CNAMEAE = "CNAMEAlreadyExists";
const _CNL = "CookieNameList";
const _CO = "ConnectionObject";
const _COAC = "CreateOriginAccessControl";
const _COACR = "CreateOriginAccessControlRequest";
const _COACRr = "CreateOriginAccessControlResult";
const _COC = "CustomOriginConfig";
const _CORP = "CreateOriginRequestPolicy";
const _CORPR = "CreateOriginRequestPolicyRequest";
const _CORPRr = "CreateOriginRequestPolicyResult";
const _CP = "CachePolicy";
const _CPAE = "CachePolicyAlreadyExists";
const _CPC = "CachePolicyConfig";
const _CPCC = "CachePolicyCookiesConfig";
const _CPHC = "CachePolicyHeadersConfig";
const _CPI = "CachePolicyId";
const _CPIU = "CachePolicyInUse";
const _CPK = "CreatePublicKey";
const _CPKR = "CreatePublicKeyRequest";
const _CPKRr = "CreatePublicKeyResult";
const _CPL = "CachePolicyList";
const _CPQSC = "CachePolicyQueryStringsConfig";
const _CPS = "CachePolicySummary";
const _CPSL = "CachePolicySummaryList";
const _CPo = "CookiePreference";
const _CR = "CallerReference";
const _CRHP = "CreateResponseHeadersPolicy";
const _CRHPR = "CreateResponseHeadersPolicyRequest";
const _CRHPRr = "CreateResponseHeadersPolicyResult";
const _CRLC = "CreateRealtimeLogConfig";
const _CRLCR = "CreateRealtimeLogConfigRequest";
const _CRLCRr = "CreateRealtimeLogConfigResult";
const _CS = "CertificateStatus";
const _CSD = "CreateStreamingDistribution";
const _CSDR = "CreateStreamingDistributionRequest";
const _CSDRr = "CreateStreamingDistributionResult";
const _CSDWT = "CreateStreamingDistributionWithTags";
const _CSDWTR = "CreateStreamingDistributionWithTagsRequest";
const _CSDWTRr = "CreateStreamingDistributionWithTagsResult";
const _CSP = "ContentSecurityPolicy";
const _CSe = "CertificateSource";
const _CT = "CommentType";
const _CTLP = "CertificateTransparencyLoggingPreference";
const _CTO = "ContentTypeOptions";
const _CTP = "ContentTypeProfile";
const _CTPC = "ContentTypeProfileConfig";
const _CTPL = "ContentTypeProfileList";
const _CTPo = "ContentTypeProfiles";
const _CTS = "CreateTrustStore";
const _CTSR = "CreateTrustStoreRequest";
const _CTSRr = "CreateTrustStoreResult";
const _CT_ = "Content-Type";
const _CTo = "ContentType";
const _CTon = "ConnectionTimeout";
const _CTr = "CreatedTime";
const _CTre = "CreateTime";
const _CU = "ComputeUtilization";
const _CUEWIU = "CannotUpdateEntityWhileInUse";
const _CVO = "CreateVpcOrigin";
const _CVOR = "CreateVpcOriginRequest";
const _CVORr = "CreateVpcOriginResult";
const _Ce = "Certificate";
const _Ci = "Cidr";
const _Co = "Comment";
const _Coo = "Cookies";
const _Cu = "Customizations";
const _D = "Distribution";
const _DAE = "DistributionAlreadyExists";
const _DAIL = "DeleteAnycastIpList";
const _DAILR = "DeleteAnycastIpListRequest";
const _DC = "DistributionConfig";
const _DCB = "DefaultCacheBehavior";
const _DCDP = "DeleteContinuousDeploymentPolicy";
const _DCDPR = "DeleteContinuousDeploymentPolicyRequest";
const _DCF = "DeleteConnectionFunction";
const _DCFOAI = "DeleteCloudFrontOriginAccessIdentity";
const _DCFOAIR = "DeleteCloudFrontOriginAccessIdentityRequest";
const _DCFR = "DeleteConnectionFunctionRequest";
const _DCFRe = "DescribeConnectionFunctionRequest";
const _DCFRes = "DescribeConnectionFunctionResult";
const _DCFe = "DescribeConnectionFunction";
const _DCG = "DeleteConnectionGroup";
const _DCGR = "DeleteConnectionGroupRequest";
const _DCL = "DnsConfigurationList";
const _DCLo = "DomainConflictsList";
const _DCP = "DeleteCachePolicy";
const _DCPR = "DeleteCachePolicyRequest";
const _DCVR = "DomainControlValidationResource";
const _DCWT = "DistributionConfigWithTags";
const _DCn = "DnsConfiguration";
const _DCo = "DomainConflict";
const _DCom = "DomainConflicts";
const _DD = "DeleteDistribution";
const _DDR = "DeleteDistributionRequest";
const _DDT = "DeleteDistributionTenant";
const _DDTR = "DeleteDistributionTenantRequest";
const _DDTWACL = "DisassociateDistributionTenantWebACL";
const _DDTWACLR = "DisassociateDistributionTenantWebACLRequest";
const _DDTWACLRi = "DisassociateDistributionTenantWebACLResult";
const _DDWACL = "DisassociateDistributionWebACL";
const _DDWACLR = "DisassociateDistributionWebACLRequest";
const _DDWACLRi = "DisassociateDistributionWebACLResult";
const _DF = "DeleteFunction";
const _DFLEC = "DeleteFieldLevelEncryptionConfig";
const _DFLECR = "DeleteFieldLevelEncryptionConfigRequest";
const _DFLEP = "DeleteFieldLevelEncryptionProfile";
const _DFLEPR = "DeleteFieldLevelEncryptionProfileRequest";
const _DFR = "DeleteFunctionRequest";
const _DFRe = "DescribeFunctionRequest";
const _DFRes = "DescribeFunctionResult";
const _DFe = "DescribeFunction";
const _DI = "DistributionId";
const _DIL = "DistributionIdList";
const _DILS = "DistributionIdListSummary";
const _DIO = "DistributionIdOwner";
const _DIOIL = "DistributionIdOwnerItemList";
const _DIOL = "DistributionIdOwnerList";
const _DIo = "DomainItem";
const _DKG = "DeleteKeyGroup";
const _DKGR = "DeleteKeyGroupRequest";
const _DKVS = "DeleteKeyValueStore";
const _DKVSR = "DeleteKeyValueStoreRequest";
const _DKVSRe = "DescribeKeyValueStoreRequest";
const _DKVSRes = "DescribeKeyValueStoreResult";
const _DKVSe = "DescribeKeyValueStore";
const _DL = "DistributionList";
const _DLo = "DomainList";
const _DMS = "DeleteMonitoringSubscription";
const _DMSR = "DeleteMonitoringSubscriptionRequest";
const _DMSRe = "DeleteMonitoringSubscriptionResult";
const _DN = "DomainName";
const _DND = "DistributionNotDisabled";
const _DNn = "DnsName";
const _DOAC = "DeleteOriginAccessControl";
const _DOACR = "DeleteOriginAccessControlRequest";
const _DORP = "DeleteOriginRequestPolicy";
const _DORPR = "DeleteOriginRequestPolicyRequest";
const _DPK = "DeletePublicKey";
const _DPKR = "DeletePublicKeyRequest";
const _DR = "DomainResult";
const _DRHP = "DeleteResponseHeadersPolicy";
const _DRHPR = "DeleteResponseHeadersPolicyRequest";
const _DRI = "DistributionResourceId";
const _DRL = "DomainResultList";
const _DRLC = "DeleteRealtimeLogConfig";
const _DRLCR = "DeleteRealtimeLogConfigRequest";
const _DRO = "DefaultRootObject";
const _DRP = "DeleteResourcePolicy";
const _DRPR = "DeleteResourcePolicyRequest";
const _DS = "DistributionSummary";
const _DSD = "DeleteStreamingDistribution";
const _DSDR = "DeleteStreamingDistributionRequest";
const _DSL = "DistributionSummaryList";
const _DT = "DistributionTenant";
const _DTAF = "DistributionTenantAssociationFilter";
const _DTI = "DistributionTenantId";
const _DTL = "DistributionTenantList";
const _DTS = "DistributionTenantSummary";
const _DTSR = "DeleteTrustStoreRequest";
const _DTSe = "DeleteTrustStore";
const _DTTL = "DefaultTTL";
const _DV = "DefaultValue";
const _DVO = "DeleteVpcOrigin";
const _DVOR = "DeleteVpcOriginRequest";
const _DVORe = "DeleteVpcOriginResult";
const _De = "Description";
const _Def = "Definition";
const _Do = "Domains";
const _Dom = "Domain";
const _E = "Enabled";
const _EAE = "EntityAlreadyExists";
const _EAEB = "EnableAcceptEncodingBrotli";
const _EAEG = "EnableAcceptEncodingGzip";
const _EC = "ErrorCode";
const _ECMTTL = "ErrorCachingMinTTL";
const _EE = "EncryptionEntities";
const _EEL = "EncryptionEntityList";
const _EEn = "EncryptionEntity";
const _EK = "EncodedKey";
const _ELE = "EntityLimitExceeded";
const _ENF = "EntityNotFound";
const _EO = "EventObject";
const _EP = "EndPoints";
const _EPL = "EndPointList";
const _EPn = "EndPoint";
const _ESLE = "EntitySizeLimitExceeded";
const _ET = "ETag";
const _ETt = "ETtag";
const _ETv = "EventType";
const _F = "Format";
const _FA = "FunctionAssociations";
const _FAE = "FunctionAlreadyExists";
const _FAL = "FunctionAssociationList";
const _FARN = "FunctionARN";
const _FAu = "FunctionAssociation";
const _FB = "FunctionBlob";
const _FC = "FunctionConfig";
const _FCa = "FailoverCriteria";
const _FCu = "FunctionCode";
const _FEL = "FunctionExecutionLogs";
const _FELL = "FunctionExecutionLogList";
const _FEM = "FunctionErrorMessage";
const _FEO = "FunctionEventObject";
const _FIU = "FunctionInUse";
const _FL = "FunctionList";
const _FLE = "FieldLevelEncryption";
const _FLEC = "FieldLevelEncryptionConfig";
const _FLECAE = "FieldLevelEncryptionConfigAlreadyExists";
const _FLECIU = "FieldLevelEncryptionConfigInUse";
const _FLEI = "FieldLevelEncryptionId";
const _FLEL = "FieldLevelEncryptionList";
const _FLEP = "FieldLevelEncryptionProfile";
const _FLEPAE = "FieldLevelEncryptionProfileAlreadyExists";
const _FLEPC = "FieldLevelEncryptionProfileConfig";
const _FLEPIU = "FieldLevelEncryptionProfileInUse";
const _FLEPL = "FieldLevelEncryptionProfileList";
const _FLEPS = "FieldLevelEncryptionProfileSummary";
const _FLEPSE = "FieldLevelEncryptionProfileSizeExceeded";
const _FLEPSL = "FieldLevelEncryptionProfileSummaryList";
const _FLES = "FieldLevelEncryptionSummary";
const _FLESL = "FieldLevelEncryptionSummaryList";
const _FLi = "FieldList";
const _FM = "FunctionMetadata";
const _FO = "FrameOption";
const _FOr = "FrameOptions";
const _FOu = "FunctionOutput";
const _FP = "FieldPatterns";
const _FPL = "FieldPatternList";
const _FPi = "FieldPattern";
const _FS = "FunctionSummary";
const _FSL = "FunctionSummaryList";
const _FSLE = "FunctionSizeLimitExceeded";
const _FV = "ForwardedValues";
const _FWCTIU = "ForwardWhenContentTypeIsUnknown";
const _FWQAPIU = "ForwardWhenQueryArgProfileIsUnknown";
const _Fi = "Fields";
const _Fie = "Field";
const _Fo = "Forward";
const _GAIL = "GetAnycastIpList";
const _GAILR = "GetAnycastIpListRequest";
const _GAILRe = "GetAnycastIpListResult";
const _GC = "GrpcConfig";
const _GCDP = "GetContinuousDeploymentPolicy";
const _GCDPC = "GetContinuousDeploymentPolicyConfig";
const _GCDPCR = "GetContinuousDeploymentPolicyConfigRequest";
const _GCDPCRe = "GetContinuousDeploymentPolicyConfigResult";
const _GCDPR = "GetContinuousDeploymentPolicyRequest";
const _GCDPRe = "GetContinuousDeploymentPolicyResult";
const _GCF = "GetConnectionFunction";
const _GCFOAI = "GetCloudFrontOriginAccessIdentity";
const _GCFOAIC = "GetCloudFrontOriginAccessIdentityConfig";
const _GCFOAICR = "GetCloudFrontOriginAccessIdentityConfigRequest";
const _GCFOAICRe = "GetCloudFrontOriginAccessIdentityConfigResult";
const _GCFOAIR = "GetCloudFrontOriginAccessIdentityRequest";
const _GCFOAIRe = "GetCloudFrontOriginAccessIdentityResult";
const _GCFR = "GetConnectionFunctionRequest";
const _GCFRe = "GetConnectionFunctionResult";
const _GCG = "GetConnectionGroup";
const _GCGBRE = "GetConnectionGroupByRoutingEndpoint";
const _GCGBRER = "GetConnectionGroupByRoutingEndpointRequest";
const _GCGBRERe = "GetConnectionGroupByRoutingEndpointResult";
const _GCGR = "GetConnectionGroupRequest";
const _GCGRe = "GetConnectionGroupResult";
const _GCP = "GetCachePolicy";
const _GCPC = "GetCachePolicyConfig";
const _GCPCR = "GetCachePolicyConfigRequest";
const _GCPCRe = "GetCachePolicyConfigResult";
const _GCPR = "GetCachePolicyRequest";
const _GCPRe = "GetCachePolicyResult";
const _GD = "GetDistribution";
const _GDC = "GetDistributionConfig";
const _GDCR = "GetDistributionConfigRequest";
const _GDCRe = "GetDistributionConfigResult";
const _GDR = "GetDistributionRequest";
const _GDRe = "GetDistributionResult";
const _GDT = "GetDistributionTenant";
const _GDTBD = "GetDistributionTenantByDomain";
const _GDTBDR = "GetDistributionTenantByDomainRequest";
const _GDTBDRe = "GetDistributionTenantByDomainResult";
const _GDTR = "GetDistributionTenantRequest";
const _GDTRe = "GetDistributionTenantResult";
const _GF = "GetFunction";
const _GFLE = "GetFieldLevelEncryption";
const _GFLEC = "GetFieldLevelEncryptionConfig";
const _GFLECR = "GetFieldLevelEncryptionConfigRequest";
const _GFLECRe = "GetFieldLevelEncryptionConfigResult";
const _GFLEP = "GetFieldLevelEncryptionProfile";
const _GFLEPC = "GetFieldLevelEncryptionProfileConfig";
const _GFLEPCR = "GetFieldLevelEncryptionProfileConfigRequest";
const _GFLEPCRe = "GetFieldLevelEncryptionProfileConfigResult";
const _GFLEPR = "GetFieldLevelEncryptionProfileRequest";
const _GFLEPRe = "GetFieldLevelEncryptionProfileResult";
const _GFLER = "GetFieldLevelEncryptionRequest";
const _GFLERe = "GetFieldLevelEncryptionResult";
const _GFR = "GetFunctionRequest";
const _GFRe = "GetFunctionResult";
const _GI = "GetInvalidation";
const _GIFDT = "GetInvalidationForDistributionTenant";
const _GIFDTR = "GetInvalidationForDistributionTenantRequest";
const _GIFDTRe = "GetInvalidationForDistributionTenantResult";
const _GIR = "GetInvalidationRequest";
const _GIRe = "GetInvalidationResult";
const _GKG = "GetKeyGroup";
const _GKGC = "GetKeyGroupConfig";
const _GKGCR = "GetKeyGroupConfigRequest";
const _GKGCRe = "GetKeyGroupConfigResult";
const _GKGR = "GetKeyGroupRequest";
const _GKGRe = "GetKeyGroupResult";
const _GMCD = "GetManagedCertificateDetails";
const _GMCDR = "GetManagedCertificateDetailsRequest";
const _GMCDRe = "GetManagedCertificateDetailsResult";
const _GMS = "GetMonitoringSubscription";
const _GMSR = "GetMonitoringSubscriptionRequest";
const _GMSRe = "GetMonitoringSubscriptionResult";
const _GOAC = "GetOriginAccessControl";
const _GOACC = "GetOriginAccessControlConfig";
const _GOACCR = "GetOriginAccessControlConfigRequest";
const _GOACCRe = "GetOriginAccessControlConfigResult";
const _GOACR = "GetOriginAccessControlRequest";
const _GOACRe = "GetOriginAccessControlResult";
const _GORP = "GetOriginRequestPolicy";
const _GORPC = "GetOriginRequestPolicyConfig";
const _GORPCR = "GetOriginRequestPolicyConfigRequest";
const _GORPCRe = "GetOriginRequestPolicyConfigResult";
const _GORPR = "GetOriginRequestPolicyRequest";
const _GORPRe = "GetOriginRequestPolicyResult";
const _GPK = "GetPublicKey";
const _GPKC = "GetPublicKeyConfig";
const _GPKCR = "GetPublicKeyConfigRequest";
const _GPKCRe = "GetPublicKeyConfigResult";
const _GPKR = "GetPublicKeyRequest";
const _GPKRe = "GetPublicKeyResult";
const _GR = "GeoRestrictions";
const _GRC = "GeoRestrictionCustomization";
const _GRHP = "GetResponseHeadersPolicy";
const _GRHPC = "GetResponseHeadersPolicyConfig";
const _GRHPCR = "GetResponseHeadersPolicyConfigRequest";
const _GRHPCRe = "GetResponseHeadersPolicyConfigResult";
const _GRHPR = "GetResponseHeadersPolicyRequest";
const _GRHPRe = "GetResponseHeadersPolicyResult";
const _GRLC = "GetRealtimeLogConfig";
const _GRLCR = "GetRealtimeLogConfigRequest";
const _GRLCRe = "GetRealtimeLogConfigResult";
const _GRP = "GetResourcePolicy";
const _GRPR = "GetResourcePolicyRequest";
const _GRPRe = "GetResourcePolicyResult";
const _GRe = "GeoRestriction";
const _GSD = "GetStreamingDistribution";
const _GSDC = "GetStreamingDistributionConfig";
const _GSDCR = "GetStreamingDistributionConfigRequest";
const _GSDCRe = "GetStreamingDistributionConfigResult";
const _GSDR = "GetStreamingDistributionRequest";
const _GSDRe = "GetStreamingDistributionResult";
const _GTS = "GetTrustStore";
const _GTSR = "GetTrustStoreRequest";
const _GTSRe = "GetTrustStoreResult";
const _GVO = "GetVpcOrigin";
const _GVOR = "GetVpcOriginRequest";
const _GVORe = "GetVpcOriginResult";
const _H = "Headers";
const _HB = "HeaderBehavior";
const _HC = "HeadersConfig";
const _HL = "HeaderList";
const _HN = "HeaderName";
const _HTTPP = "HTTPPort";
const _HTTPSP = "HTTPSPort";
const _HV = "HttpVersion";
const _HVe = "HeaderValue";
const _He = "Header";
const _I = "Items";
const _IA = "InvalidArgument";
const _IAMCI = "IAMCertificateId";
const _IAT = "IpAddressType";
const _IAn = "InvalidAssociation";
const _IB = "InvalidationBatch";
const _IBn = "IncludeBody";
const _IC = "IpamConfig";
const _ICC = "IpamCidrConfigs";
const _ICCL = "IpamCidrConfigList";
const _ICCp = "IpamCidrConfig";
const _ICE = "IgnoreCertificateExpiry";
const _ICPRS = "ICPRecordalStatus";
const _ICn = "IncludeCookies";
const _ICp = "IpCount";
const _ID = "IsDefault";
const _IDNFOAC = "InvalidDomainNameForOriginAccessControl";
const _IDRO = "InvalidDefaultRootObject";
const _IDl = "IllegalDelete";
const _IE = "Ipv6Enabled";
const _IEC = "InvalidErrorCode";
const _IFA = "InvalidFunctionAssociation";
const _IFC = "InvalidForwardCookies";
const _IFLECAWCB = "IllegalFieldLevelEncryptionConfigAssociationWithCacheBehavior";
const _IGRP = "InvalidGeoRestrictionParameter";
const _IHFSO = "InvalidHeadersForS3Origin";
const _IIMV = "InvalidIfMatchVersion";
const _IIPVE = "IsIPV6Enabled";
const _IL = "InvalidationList";
const _ILC = "InvalidLocationCode";
const _ILFA = "InvalidLambdaFunctionAssociation";
const _IM = "IfMatch";
const _IMPV = "InvalidMinimumProtocolVersion";
const _IM_ = "If-Match";
const _IO = "InvalidOrigin";
const _IOAC = "IllegalOriginAccessConfiguration";
const _IOACn = "InvalidOriginAccessControl";
const _IOAI = "InvalidOriginAccessIdentity";
const _IOKT = "InvalidOriginKeepaliveTimeout";
const _IORT = "InvalidOriginReadTimeout";
const _IPA = "IpamPoolArn";
const _IPIB = "InProgressInvalidationBatches";
const _IPS = "InvalidProtocolSettings";
const _IQ = "InconsistentQuantities";
const _IQSP = "InvalidQueryStringParameters";
const _IRC = "InvalidResponseCode";
const _IRP = "InvalidRelativePath";
const _IRPn = "InvalidRequiredProtocol";
const _IS = "ImportSource";
const _ISL = "InvalidationSummaryList";
const _ISn = "InvalidationSummary";
const _ISnc = "IncludeSubdomains";
const _IT = "IsTruncated";
const _ITTL = "IdleTTL";
const _ITTLO = "InvalidTTLOrder";
const _ITn = "InvalidTagging";
const _IU = "IllegalUpdate";
const _IVC = "InvalidViewerCertificate";
const _IWACLI = "InvalidWebACLId";
const _Id = "Id";
const _Ide = "Identifier";
const _In = "Invalidation";
const _K = "Key";
const _KG = "KeyGroup";
const _KGAE = "KeyGroupAlreadyExists";
const _KGC = "KeyGroupConfig";
const _KGI = "KeyGroupId";
const _KGKPI = "KGKeyPairIds";
const _KGKPIL = "KGKeyPairIdsList";
const _KGL = "KeyGroupList";
const _KGS = "KeyGroupSummary";
const _KGSL = "KeyGroupSummaryList";
const _KPI = "KeyPairIds";
const _KPIL = "KeyPairIdList";
const _KPIe = "KeyPairId";
const _KSC = "KinesisStreamConfig";
const _KVS = "KeyValueStore";
const _KVSA = "KeyValueStoreAssociations";
const _KVSAL = "KeyValueStoreAssociationList";
const _KVSARN = "KeyValueStoreARN";
const _KVSAe = "KeyValueStoreAssociation";
const _KVSL = "KeyValueStoreList";
const _KVSSL = "KeyValueStoreSummaryList";
const _L = "Location";
const _LAIL = "ListAnycastIpLists";
const _LAILR = "ListAnycastIpListsRequest";
const _LAILRi = "ListAnycastIpListsResult";
const _LC = "LoggingConfig";
const _LCA = "ListConflictingAliases";
const _LCAR = "ListConflictingAliasesRequest";
const _LCARi = "ListConflictingAliasesResult";
const _LCDP = "ListContinuousDeploymentPolicies";
const _LCDPR = "ListContinuousDeploymentPoliciesRequest";
const _LCDPRi = "ListContinuousDeploymentPoliciesResult";
const _LCF = "ListConnectionFunctions";
const _LCFOAI = "ListCloudFrontOriginAccessIdentities";
const _LCFOAIR = "ListCloudFrontOriginAccessIdentitiesRequest";
const _LCFOAIRi = "ListCloudFrontOriginAccessIdentitiesResult";
const _LCFR = "ListConnectionFunctionsRequest";
const _LCFRi = "ListConnectionFunctionsResult";
const _LCG = "ListConnectionGroups";
const _LCGR = "ListConnectionGroupsRequest";
const _LCGRi = "ListConnectionGroupsResult";
const _LCP = "ListCachePolicies";
const _LCPR = "ListCachePoliciesRequest";
const _LCPRi = "ListCachePoliciesResult";
const _LD = "ListDistributions";
const _LDBAILI = "ListDistributionsByAnycastIpListId";
const _LDBAILIR = "ListDistributionsByAnycastIpListIdRequest";
const _LDBAILIRi = "ListDistributionsByAnycastIpListIdResult";
const _LDBCF = "ListDistributionsByConnectionFunction";
const _LDBCFR = "ListDistributionsByConnectionFunctionRequest";
const _LDBCFRi = "ListDistributionsByConnectionFunctionResult";
const _LDBCM = "ListDistributionsByConnectionMode";
const _LDBCMR = "ListDistributionsByConnectionModeRequest";
const _LDBCMRi = "ListDistributionsByConnectionModeResult";
const _LDBCPI = "ListDistributionsByCachePolicyId";
const _LDBCPIR = "ListDistributionsByCachePolicyIdRequest";
const _LDBCPIRi = "ListDistributionsByCachePolicyIdResult";
const _LDBKG = "ListDistributionsByKeyGroup";
const _LDBKGR = "ListDistributionsByKeyGroupRequest";
const _LDBKGRi = "ListDistributionsByKeyGroupResult";
const _LDBOR = "ListDistributionsByOwnedResource";
const _LDBORPI = "ListDistributionsByOriginRequestPolicyId";
const _LDBORPIR = "ListDistributionsByOriginRequestPolicyIdRequest";
const _LDBORPIRi = "ListDistributionsByOriginRequestPolicyIdResult";
const _LDBORR = "ListDistributionsByOwnedResourceRequest";
const _LDBORRi = "ListDistributionsByOwnedResourceResult";
const _LDBRHPI = "ListDistributionsByResponseHeadersPolicyId";
const _LDBRHPIR = "ListDistributionsByResponseHeadersPolicyIdRequest";
const _LDBRHPIRi = "ListDistributionsByResponseHeadersPolicyIdResult";
const _LDBRLC = "ListDistributionsByRealtimeLogConfig";
const _LDBRLCR = "ListDistributionsByRealtimeLogConfigRequest";
const _LDBRLCRi = "ListDistributionsByRealtimeLogConfigResult";
const _LDBTS = "ListDistributionsByTrustStore";
const _LDBTSR = "ListDistributionsByTrustStoreRequest";
const _LDBTSRi = "ListDistributionsByTrustStoreResult";
const _LDBVOI = "ListDistributionsByVpcOriginId";
const _LDBVOIR = "ListDistributionsByVpcOriginIdRequest";
const _LDBVOIRi = "ListDistributionsByVpcOriginIdResult";
const _LDBWACLI = "ListDistributionsByWebACLId";
const _LDBWACLIR = "ListDistributionsByWebACLIdRequest";
const _LDBWACLIRi = "ListDistributionsByWebACLIdResult";
const _LDC = "ListDomainConflicts";
const _LDCR = "ListDomainConflictsRequest";
const _LDCRi = "ListDomainConflictsResult";
const _LDR = "ListDistributionsRequest";
const _LDRi = "ListDistributionsResult";
const _LDT = "ListDistributionTenants";
const _LDTBC = "ListDistributionTenantsByCustomization";
const _LDTBCR = "ListDistributionTenantsByCustomizationRequest";
const _LDTBCRi = "ListDistributionTenantsByCustomizationResult";
const _LDTR = "ListDistributionTenantsRequest";
const _LDTRi = "ListDistributionTenantsResult";
const _LF = "ListFunctions";
const _LFA = "LambdaFunctionAssociations";
const _LFAL = "LambdaFunctionAssociationList";
const _LFARN = "LambdaFunctionARN";
const _LFAa = "LambdaFunctionAssociation";
const _LFLEC = "ListFieldLevelEncryptionConfigs";
const _LFLECR = "ListFieldLevelEncryptionConfigsRequest";
const _LFLECRi = "ListFieldLevelEncryptionConfigsResult";
const _LFLEP = "ListFieldLevelEncryptionProfiles";
const _LFLEPR = "ListFieldLevelEncryptionProfilesRequest";
const _LFLEPRi = "ListFieldLevelEncryptionProfilesResult";
const _LFR = "ListFunctionsRequest";
const _LFRi = "ListFunctionsResult";
const _LI = "ListInvalidations";
const _LIFDT = "ListInvalidationsForDistributionTenant";
const _LIFDTR = "ListInvalidationsForDistributionTenantRequest";
const _LIFDTRi = "ListInvalidationsForDistributionTenantResult";
const _LIR = "ListInvalidationsRequest";
const _LIRi = "ListInvalidationsResult";
const _LKG = "ListKeyGroups";
const _LKGR = "ListKeyGroupsRequest";
const _LKGRi = "ListKeyGroupsResult";
const _LKVS = "ListKeyValueStores";
const _LKVSR = "ListKeyValueStoresRequest";
const _LKVSRi = "ListKeyValueStoresResult";
const _LL = "LocationList";
const _LMT = "LastModifiedTime";
const _LOAC = "ListOriginAccessControls";
const _LOACR = "ListOriginAccessControlsRequest";
const _LOACRi = "ListOriginAccessControlsResult";
const _LORP = "ListOriginRequestPolicies";
const _LORPR = "ListOriginRequestPoliciesRequest";
const _LORPRi = "ListOriginRequestPoliciesResult";
const _LPK = "ListPublicKeys";
const _LPKR = "ListPublicKeysRequest";
const _LPKRi = "ListPublicKeysResult";
const _LRHP = "ListResponseHeadersPolicies";
const _LRHPR = "ListResponseHeadersPoliciesRequest";
const _LRHPRi = "ListResponseHeadersPoliciesResult";
const _LRLC = "ListRealtimeLogConfigs";
const _LRLCR = "ListRealtimeLogConfigsRequest";
const _LRLCRi = "ListRealtimeLogConfigsResult";
const _LSD = "ListStreamingDistributions";
const _LSDR = "ListStreamingDistributionsRequest";
const _LSDRi = "ListStreamingDistributionsResult";
const _LTFR = "ListTagsForResource";
const _LTFRR = "ListTagsForResourceRequest";
const _LTFRRi = "ListTagsForResourceResult";
const _LTS = "ListTrustStores";
const _LTSR = "ListTrustStoresRequest";
const _LTSRi = "ListTrustStoresResult";
const _LVO = "ListVpcOrigins";
const _LVOR = "ListVpcOriginsRequest";
const _LVORi = "ListVpcOriginsResult";
const _Lo = "Logging";
const _Loc = "Locations";
const _M = "Message";
const _MB = "MissingBody";
const _MBo = "ModeBlock";
const _MCD = "ManagedCertificateDetails";
const _MCR = "ManagedCertificateRequest";
const _MI = "MaxItems";
const _ML = "MethodsList";
const _MPV = "MinimumProtocolVersion";
const _MS = "MonitoringSubscription";
const _MSAE = "MonitoringSubscriptionAlreadyExists";
const _MTTL = "MinTTL";
const _MTTLa = "MaxTTL";
const _MTTLax = "MaximumTTL";
const _Ma = "Marker";
const _Me = "Members";
const _Met = "Method";
const _Mo = "Mode";
const _N = "Name";
const _NM = "NextMarker";
const _NOCC = "NumberOfCaCertificates";
const _NSCDP = "NoSuchContinuousDeploymentPolicy";
const _NSCFOAI = "NoSuchCloudFrontOriginAccessIdentity";
const _NSCP = "NoSuchCachePolicy";
const _NSD = "NoSuchDistribution";
const _NSFE = "NoSuchFunctionExists";
const _NSFLEC = "NoSuchFieldLevelEncryptionConfig";
const _NSFLEP = "NoSuchFieldLevelEncryptionProfile";
const _NSI = "NoSuchInvalidation";
const _NSMS = "NoSuchMonitoringSubscription";
const _NSO = "NoSuchOrigin";
const _NSOAC = "NoSuchOriginAccessControl";
const _NSORP = "NoSuchOriginRequestPolicy";
const _NSPK = "NoSuchPublicKey";
const _NSR = "NoSuchResource";
const _NSRHP = "NoSuchResponseHeadersPolicy";
const _NSRLC = "NoSuchRealtimeLogConfig";
const _NSSD = "NoSuchStreamingDistribution";
const _O = "Origins";
const _OAC = "OriginAccessControl";
const _OACAE = "OriginAccessControlAlreadyExists";
const _OACC = "OriginAccessControlConfig";
const _OACI = "OriginAccessControlId";
const _OACIU = "OriginAccessControlInUse";
const _OACL = "OriginAccessControlList";
const _OACOT = "OriginAccessControlOriginType";
const _OACS = "OriginAccessControlSummary";
const _OACSL = "OriginAccessControlSummaryList";
const _OAI = "OwnerAccountId";
const _OAIr = "OriginAccessIdentity";
const _OCH = "OriginCustomHeader";
const _OCHL = "OriginCustomHeadersList";
const _OEA = "OriginEndpointArn";
const _OG = "OriginGroups";
const _OGFC = "OriginGroupFailoverCriteria";
const _OGL = "OriginGroupList";
const _OGM = "OriginGroupMember";
const _OGML = "OriginGroupMemberList";
const _OGMr = "OriginGroupMembers";
const _OGr = "OriginGroup";
const _OI = "OriginId";
const _OKT = "OriginKeepaliveTimeout";
const _OL = "OriginList";
const _OO = "OriginOverride";
const _OP = "OriginPath";
const _OPP = "OriginProtocolPolicy";
const _ORP = "OriginRequestPolicy";
const _ORPAE = "OriginRequestPolicyAlreadyExists";
const _ORPC = "OriginRequestPolicyConfig";
const _ORPCC = "OriginRequestPolicyCookiesConfig";
const _ORPHC = "OriginRequestPolicyHeadersConfig";
const _ORPI = "OriginRequestPolicyId";
const _ORPIU = "OriginRequestPolicyInUse";
const _ORPL = "OriginRequestPolicyList";
const _ORPQSC = "OriginRequestPolicyQueryStringsConfig";
const _ORPS = "OriginRequestPolicySummary";
const _ORPSL = "OriginRequestPolicySummaryList";
const _ORT = "OriginReadTimeout";
const _OS = "OriginShield";
const _OSP = "OriginSslProtocols";
const _OSR = "OriginShieldRegion";
const _Or = "Origin";
const _Ov = "Override";
const _P = "Parameters";
const _PC = "PriceClass";
const _PCF = "PublishConnectionFunction";
const _PCFR = "PublishConnectionFunctionRequest";
const _PCFRu = "PublishConnectionFunctionResult";
const _PD = "PolicyDocument";
const _PDI = "PrimaryDistributionId";
const _PDN = "PrimaryDomainName";
const _PDS = "ParameterDefinitionSchema";
const _PDa = "ParameterDefinition";
const _PDar = "ParameterDefinitions";
const _PF = "PreconditionFailed";
const _PFR = "PublishFunctionRequest";
const _PFRu = "PublishFunctionResult";
const _PFu = "PublishFunction";
const _PI = "ProfileId";
const _PICKAFTO = "ParametersInCacheKeyAndForwardedToOrigin";
const _PIr = "ProviderId";
const _PK = "PublicKey";
const _PKAE = "PublicKeyAlreadyExists";
const _PKC = "PublicKeyConfig";
const _PKI = "PublicKeyId";
const _PKIL = "PublicKeyIdList";
const _PKIU = "PublicKeyInUse";
const _PKL = "PublicKeyList";
const _PKS = "PublicKeySummary";
const _PKSL = "PublicKeySummaryList";
const _PL = "PathList";
const _PP = "PathPattern";
const _PRP = "PutResourcePolicy";
const _PRPR = "PutResourcePolicyRequest";
const _PRPRu = "PutResourcePolicyResult";
const _Pa = "Paths";
const _Par = "Parameter";
const _Pat = "Path";
const _Pr = "Prefix";
const _Pre = "Preload";
const _Pro = "Protection";
const _Q = "Quantity";
const _QA = "QueryArg";
const _QAP = "QueryArgProfile";
const _QAPC = "QueryArgProfileConfig";
const _QAPE = "QueryArgProfileEmpty";
const _QAPL = "QueryArgProfileList";
const _QAPu = "QueryArgProfiles";
const _QS = "QueryStrings";
const _QSB = "QueryStringBehavior";
const _QSC = "QueryStringsConfig";
const _QSCK = "QueryStringCacheKeys";
const _QSCKL = "QueryStringCacheKeysList";
const _QSN = "QueryStringNames";
const _QSNL = "QueryStringNamesList";
const _QSu = "QueryString";
const _R = "Region";
const _RA = "ResourceArn";
const _RARN = "RoleARN";
const _RC = "ResponseCode";
const _RCT = "ResponseCompletionTimeout";
const _RE = "RoutingEndpoint";
const _RF = "RedirectFrom";
const _RHC = "RemoveHeadersConfig";
const _RHP = "ResponseHeadersPolicy";
const _RHPACAH = "ResponseHeadersPolicyAccessControlAllowHeaders";
const _RHPACAM = "ResponseHeadersPolicyAccessControlAllowMethods";
const _RHPACAO = "ResponseHeadersPolicyAccessControlAllowOrigins";
const _RHPACEH = "ResponseHeadersPolicyAccessControlExposeHeaders";
const _RHPAE = "ResponseHeadersPolicyAlreadyExists";
const _RHPC = "ResponseHeadersPolicyConfig";
const _RHPCC = "ResponseHeadersPolicyCorsConfig";
const _RHPCH = "ResponseHeadersPolicyCustomHeader";
const _RHPCHC = "ResponseHeadersPolicyCustomHeadersConfig";
const _RHPCHL = "ResponseHeadersPolicyCustomHeaderList";
const _RHPCSP = "ResponseHeadersPolicyContentSecurityPolicy";
const _RHPCTO = "ResponseHeadersPolicyContentTypeOptions";
const _RHPFO = "ResponseHeadersPolicyFrameOptions";
const _RHPI = "ResponseHeadersPolicyId";
const _RHPIU = "ResponseHeadersPolicyInUse";
const _RHPL = "ResponseHeadersPolicyList";
const _RHPRH = "ResponseHeadersPolicyRemoveHeader";
const _RHPRHC = "ResponseHeadersPolicyRemoveHeadersConfig";
const _RHPRHL = "ResponseHeadersPolicyRemoveHeaderList";
const _RHPRP = "ResponseHeadersPolicyReferrerPolicy";
const _RHPS = "ResponseHeadersPolicySummary";
const _RHPSHC = "ResponseHeadersPolicySecurityHeadersConfig";
const _RHPSL = "ResponseHeadersPolicySummaryList";
const _RHPSTHC = "ResponseHeadersPolicyServerTimingHeadersConfig";
const _RHPSTS = "ResponseHeadersPolicyStrictTransportSecurity";
const _RHPXSSP = "ResponseHeadersPolicyXSSProtection";
const _RI = "ResourceId";
const _RIU = "ResourceInUse";
const _RLC = "RealtimeLogConfig";
const _RLCA = "RealtimeLogConfigArn";
const _RLCAE = "RealtimeLogConfigAlreadyExists";
const _RLCIU = "RealtimeLogConfigInUse";
const _RLCL = "RealtimeLogConfigList";
const _RLCN = "RealtimeLogConfigName";
const _RLCOM = "RealtimeLogConfigOwnerMismatch";
const _RLCe = "RealtimeLogConfigs";
const _RMSC = "RealtimeMetricsSubscriptionConfig";
const _RMSS = "RealtimeMetricsSubscriptionStatus";
const _RND = "ResourceNotDisabled";
const _RP = "ReferrerPolicy";
const _RPP = "ResponsePagePath";
const _RT = "ResourceType";
const _RTe = "RestrictionType";
const _RTed = "RedirectTo";
const _RU = "ReportUri";
const _Re = "Restrictions";
const _Rea = "Reason";
const _Req = "Required";
const _Res = "Resource";
const _Ru = "Runtime";
const _S = "Status";
const _SARN = "SourceARN";
const _SARNt = "StreamARN";
const _SB = "SigningBehavior";
const _SC = "SelectionCriteria";
const _SCL = "StatusCodeList";
const _SCUI = "S3CanonicalUserId";
const _SCt = "StatusCodes";
const _SCta = "StatusCode";
const _SD = "StreamingDistribution";
const _SDAE = "StreamingDistributionAlreadyExists";
const _SDC = "StreamingDistributionConfig";
const _SDCWT = "StreamingDistributionConfigWithTags";
const _SDDN = "StagingDistributionDnsNames";
const _SDDNL = "StagingDistributionDnsNameList";
const _SDI = "StagingDistributionId";
const _SDIU = "StagingDistributionInUse";
const _SDL = "StreamingDistributionList";
const _SDND = "StreamingDistributionNotDisabled";
const _SDS = "StreamingDistributionSummary";
const _SDSL = "StreamingDistributionSummaryList";
const _SHC = "SecurityHeadersConfig";
const _SHCi = "SingleHeaderConfig";
const _SL = "SignerList";
const _SLC = "StreamingLoggingConfig";
const _SO = "S3Origin";
const _SOC = "S3OriginConfig";
const _SP = "SigningProtocol";
const _SPL = "SslProtocolsList";
const _SPs = "SslProtocol";
const _SR = "SamplingRate";
const _SS = "SmoothStreaming";
const _SSC = "SessionStickinessConfig";
const _SSCt = "StringSchemaConfig";
const _SSLSM = "SSLSupportMethod";
const _SSt = "StringSchema";
const _ST = "StreamType";
const _STHC = "ServerTimingHeadersConfig";
const _STS = "StrictTransportSecurity";
const _STo = "SourceType";
const _SWC = "SingleWeightConfig";
const _Si = "Signer";
const _St = "Stage";
const _Sta = "Staging";
const _T = "Type";
const _TC = "TrafficConfig";
const _TCF = "TestConnectionFunction";
const _TCFR = "TestConnectionFunctionRequest";
const _TCFRe = "TestConnectionFunctionResult";
const _TCe = "TenantConfig";
const _TDI = "TargetDistributionId";
const _TF = "TestFunction";
const _TFF = "TestFunctionFailed";
const _TFR = "TestFunctionRequest";
const _TFRe = "TestFunctionResult";
const _TK = "TagKeys";
const _TKG = "TrustedKeyGroups";
const _TKGDNE = "TrustedKeyGroupDoesNotExist";
const _TKGIL = "TrustedKeyGroupIdList";
const _TKL = "TagKeyList";
const _TL = "TagList";
const _TLCSPIRHP = "TooLongCSPInResponseHeadersPolicy";
const _TMC = "TooManyCertificates";
const _TMCB = "TooManyCacheBehaviors";
const _TMCDP = "TooManyContinuousDeploymentPolicies";
const _TMCFOAI = "TooManyCloudFrontOriginAccessIdentities";
const _TMCHIRHP = "TooManyCustomHeadersInResponseHeadersPolicy";
const _TMCICP = "TooManyCookiesInCachePolicy";
const _TMCIORP = "TooManyCookiesInOriginRequestPolicy";
const _TMCNIWL = "TooManyCookieNamesInWhiteList";
const _TMCP = "TooManyCachePolicies";
const _TMD = "TooManyDistributions";
const _TMDATCP = "TooManyDistributionsAssociatedToCachePolicy";
const _TMDATFLEC = "TooManyDistributionsAssociatedToFieldLevelEncryptionConfig";
const _TMDATKG = "TooManyDistributionsAssociatedToKeyGroup";
const _TMDATOAC = "TooManyDistributionsAssociatedToOriginAccessControl";
const _TMDATORP = "TooManyDistributionsAssociatedToOriginRequestPolicy";
const _TMDATRHP = "TooManyDistributionsAssociatedToResponseHeadersPolicy";
const _TMDCNAME = "TooManyDistributionCNAMEs";
const _TMDWFA = "TooManyDistributionsWithFunctionAssociations";
const _TMDWLA = "TooManyDistributionsWithLambdaAssociations";
const _TMDWSFARN = "TooManyDistributionsWithSingleFunctionARN";
const _TMF = "TooManyFunctions";
const _TMFA = "TooManyFunctionAssociations";
const _TMFLEC = "TooManyFieldLevelEncryptionConfigs";
const _TMFLECTP = "TooManyFieldLevelEncryptionContentTypeProfiles";
const _TMFLEEE = "TooManyFieldLevelEncryptionEncryptionEntities";
const _TMFLEFP = "TooManyFieldLevelEncryptionFieldPatterns";
const _TMFLEP = "TooManyFieldLevelEncryptionProfiles";
const _TMFLEQAP = "TooManyFieldLevelEncryptionQueryArgProfiles";
const _TMHICP = "TooManyHeadersInCachePolicy";
const _TMHIFV = "TooManyHeadersInForwardedValues";
const _TMHIORP = "TooManyHeadersInOriginRequestPolicy";
const _TMIIP = "TooManyInvalidationsInProgress";
const _TMKG = "TooManyKeyGroups";
const _TMKGATD = "TooManyKeyGroupsAssociatedToDistribution";
const _TMLFA = "TooManyLambdaFunctionAssociations";
const _TMO = "TooManyOrigins";
const _TMOAC = "TooManyOriginAccessControls";
const _TMOCH = "TooManyOriginCustomHeaders";
const _TMOGPD = "TooManyOriginGroupsPerDistribution";
const _TMORP = "TooManyOriginRequestPolicies";
const _TMPK = "TooManyPublicKeys";
const _TMPKIKG = "TooManyPublicKeysInKeyGroup";
const _TMQSICP = "TooManyQueryStringsInCachePolicy";
const _TMQSIORP = "TooManyQueryStringsInOriginRequestPolicy";
const _TMQSP = "TooManyQueryStringParameters";
const _TMRHIRHP = "TooManyRemoveHeadersInResponseHeadersPolicy";
const _TMRHP = "TooManyResponseHeadersPolicies";
const _TMRLC = "TooManyRealtimeLogConfigs";
const _TMSD = "TooManyStreamingDistributions";
const _TMSDCNAME = "TooManyStreamingDistributionCNAMEs";
const _TMTS = "TooManyTrustedSigners";
const _TOI = "TargetOriginId";
const _TR = "TestResult";
const _TRR = "TagResourceRequest";
const _TRa = "TargetResource";
const _TRag = "TagResource";
const _TS = "TrustedSigners";
const _TSC = "TrustStoreConfig";
const _TSDNE = "TrustedSignerDoesNotExist";
const _TSI = "TrustStoreIdentifier";
const _TSIr = "TrustStoreId";
const _TSL = "TrustStoreList";
const _TSS = "TrustStoreSummary";
const _TSr = "TrustStore";
const _Ta = "Tags";
const _Tag = "Tag";
const _UAIL = "UpdateAnycastIpList";
const _UAILR = "UpdateAnycastIpListRequest";
const _UAILRp = "UpdateAnycastIpListResult";
const _UCDP = "UpdateContinuousDeploymentPolicy";
const _UCDPR = "UpdateContinuousDeploymentPolicyRequest";
const _UCDPRp = "UpdateContinuousDeploymentPolicyResult";
const _UCF = "UpdateConnectionFunction";
const _UCFOAI = "UpdateCloudFrontOriginAccessIdentity";
const _UCFOAIR = "UpdateCloudFrontOriginAccessIdentityRequest";
const _UCFOAIRp = "UpdateCloudFrontOriginAccessIdentityResult";
const _UCFR = "UpdateConnectionFunctionRequest";
const _UCFRp = "UpdateConnectionFunctionResult";
const _UCG = "UpdateConnectionGroup";
const _UCGR = "UpdateConnectionGroupRequest";
const _UCGRp = "UpdateConnectionGroupResult";
const _UCP = "UpdateCachePolicy";
const _UCPR = "UpdateCachePolicyRequest";
const _UCPRp = "UpdateCachePolicyResult";
const _UD = "UpdateDistribution";
const _UDA = "UpdateDomainAssociation";
const _UDAR = "UpdateDomainAssociationRequest";
const _UDARp = "UpdateDomainAssociationResult";
const _UDR = "UpdateDistributionRequest";
const _UDRp = "UpdateDistributionResult";
const _UDT = "UpdateDistributionTenant";
const _UDTR = "UpdateDistributionTenantRequest";
const _UDTRp = "UpdateDistributionTenantResult";
const _UDWSC = "UpdateDistributionWithStagingConfig";
const _UDWSCR = "UpdateDistributionWithStagingConfigRequest";
const _UDWSCRp = "UpdateDistributionWithStagingConfigResult";
const _UF = "UpdateFunction";
const _UFLEC = "UpdateFieldLevelEncryptionConfig";
const _UFLECR = "UpdateFieldLevelEncryptionConfigRequest";
const _UFLECRp = "UpdateFieldLevelEncryptionConfigResult";
const _UFLEP = "UpdateFieldLevelEncryptionProfile";
const _UFLEPR = "UpdateFieldLevelEncryptionProfileRequest";
const _UFLEPRp = "UpdateFieldLevelEncryptionProfileResult";
const _UFR = "UpdateFunctionRequest";
const _UFRp = "UpdateFunctionResult";
const _UKG = "UpdateKeyGroup";
const _UKGR = "UpdateKeyGroupRequest";
const _UKGRp = "UpdateKeyGroupResult";
const _UKVS = "UpdateKeyValueStore";
const _UKVSR = "UpdateKeyValueStoreRequest";
const _UKVSRp = "UpdateKeyValueStoreResult";
const _UO = "UnsupportedOperation";
const _UOAC = "UpdateOriginAccessControl";
const _UOACR = "UpdateOriginAccessControlRequest";
const _UOACRp = "UpdateOriginAccessControlResult";
const _UORP = "UpdateOriginRequestPolicy";
const _UORPR = "UpdateOriginRequestPolicyRequest";
const _UORPRp = "UpdateOriginRequestPolicyResult";
const _UPK = "UpdatePublicKey";
const _UPKR = "UpdatePublicKeyRequest";
const _UPKRp = "UpdatePublicKeyResult";
const _UR = "UntagResource";
const _URHP = "UpdateResponseHeadersPolicy";
const _URHPR = "UpdateResponseHeadersPolicyRequest";
const _URHPRp = "UpdateResponseHeadersPolicyResult";
const _URLC = "UpdateRealtimeLogConfig";
const _URLCR = "UpdateRealtimeLogConfigRequest";
const _URLCRp = "UpdateRealtimeLogConfigResult";
const _URR = "UntagResourceRequest";
const _USD = "UpdateStreamingDistribution";
const _USDR = "UpdateStreamingDistributionRequest";
const _USDRp = "UpdateStreamingDistributionResult";
const _UTS = "UpdateTrustStore";
const _UTSR = "UpdateTrustStoreRequest";
const _UTSRp = "UpdateTrustStoreResult";
const _UVO = "UpdateVpcOrigin";
const _UVOR = "UpdateVpcOriginRequest";
const _UVORp = "UpdateVpcOriginResult";
const _V = "Version";
const _VC = "ViewerCertificate";
const _VDC = "VerifyDnsConfiguration";
const _VDCR = "VerifyDnsConfigurationRequest";
const _VDCRe = "VerifyDnsConfigurationResult";
const _VMC = "ViewerMtlsConfig";
const _VO = "VpcOrigin";
const _VOC = "VpcOriginConfig";
const _VOEC = "VpcOriginEndpointConfig";
const _VOI = "VpcOriginId";
const _VOL = "VpcOriginList";
const _VOS = "VpcOriginSummary";
const _VOSL = "VpcOriginSummaryList";
const _VPP = "ViewerProtocolPolicy";
const _VTD = "ValidationTokenDetails";
const _VTDL = "ValidationTokenDetailList";
const _VTDa = "ValidationTokenDetail";
const _VTH = "ValidationTokenHost";
const _Va = "Value";
const _W = "Weight";
const _WA = "WebAcl";
const _WAC = "WebAclCustomization";
const _WACLA = "WebACLArn";
const _WACLI = "WebACLId";
const _WN = "WhitelistedNames";
const _XSSP = "XSSProtection";
const _c = "client";
const _d = "domain";
const _e = "error";
const _h = "http";
const _hE = "httpError";
const _hH = "httpHeader";
const _hP = "httpPayload";
const _hQ = "httpQuery";
const _s = "server";
const _sST = "sensitiveStringType";
const _sm = "smithy.ts.sdk.synthetic.com.amazonaws.cloudfront";
const _xN = "xmlName";
const n0 = "com.amazonaws.cloudfront";
var CommentType = [0, n0, _CT, 8, 0];
var FunctionBlob = [0, n0, _FB, 8, 21];
var FunctionEventObject = [0, n0, _FEO, 8, 21];
var sensitiveStringType = [0, n0, _sST, 8, 0];
var AccessDenied$ = [-3, n0, _AD,
    { [_e]: _c, [_hE]: 403 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(AccessDenied$, AccessDenied);
var ActiveTrustedKeyGroups$ = [3, n0, _ATKG,
    0,
    [_E, _Q, _I],
    [2, 1, [() => KGKeyPairIdsList, 0]]
];
var ActiveTrustedSigners$ = [3, n0, _ATS,
    0,
    [_E, _Q, _I],
    [2, 1, [() => SignerList, 0]]
];
var Aliases$ = [3, n0, _A,
    0,
    [_Q, _I],
    [1, [() => AliasList, 0]]
];
var AliasICPRecordal$ = [3, n0, _AICPR,
    0,
    [_CNAME, _ICPRS],
    [0, 0]
];
var AllowedMethods$ = [3, n0, _AM,
    0,
    [_Q, _I, _CM],
    [1, [() => MethodsList, 0], [() => CachedMethods$, 0]]
];
var AnycastIpList$ = [3, n0, _AIL,
    0,
    [_Id, _N, _S, _Ar, _IAT, _IC, _AI, _ICp, _LMT],
    [0, 0, 0, 0, 0, [() => IpamConfig$, 0], [() => AnycastIps, 0], 1, 4]
];
var AnycastIpListCollection$ = [3, n0, _AILC,
    0,
    [_I, _Ma, _NM, _MI, _IT, _Q],
    [[() => AnycastIpListSummaries, 0], 0, 0, 1, 2, 1]
];
var AnycastIpListSummary$ = [3, n0, _AILS,
    0,
    [_Id, _N, _S, _Ar, _ICp, _LMT, _IAT, _ET, _IC],
    [0, 0, 0, 0, 1, 4, 0, 0, [() => IpamConfig$, 0]]
];
var AssociateAliasRequest$ = [3, n0, _AAR,
    0,
    [_TDI, _Al],
    [[0, 1], [0, { [_hQ]: _Al }]]
];
var AssociateDistributionTenantWebACLRequest$ = [3, n0, _ADTWACLR,
    0,
    [_Id, _WACLA, _IM],
    [[0, 1], 0, [0, { [_hH]: _IM_ }]]
];
var AssociateDistributionTenantWebACLResult$ = [3, n0, _ADTWACLRs,
    0,
    [_Id, _WACLA, _ET],
    [0, 0, [0, { [_hH]: _ET }]]
];
var AssociateDistributionWebACLRequest$ = [3, n0, _ADWACLR,
    0,
    [_Id, _WACLA, _IM],
    [[0, 1], 0, [0, { [_hH]: _IM_ }]]
];
var AssociateDistributionWebACLResult$ = [3, n0, _ADWACLRs,
    0,
    [_Id, _WACLA, _ET],
    [0, 0, [0, { [_hH]: _ET }]]
];
var BatchTooLarge$ = [-3, n0, _BTL,
    { [_e]: _c, [_hE]: 413 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(BatchTooLarge$, BatchTooLarge);
var CaCertificatesBundleS3Location$ = [3, n0, _CCBSL,
    0,
    [_B, _K, _R, _V],
    [0, 0, 0, 0]
];
var CacheBehavior$ = [3, n0, _CB,
    0,
    [_PP, _TOI, _TS, _TKG, _VPP, _AM, _SS, _C, _LFA, _FA, _FLEI, _RLCA, _CPI, _ORPI, _RHPI, _GC, _FV, _MTTL, _DTTL, _MTTLa],
    [0, 0, [() => TrustedSigners$, 0], [() => TrustedKeyGroups$, 0], 0, [() => AllowedMethods$, 0], 2, 2, [() => LambdaFunctionAssociations$, 0], [() => FunctionAssociations$, 0], 0, 0, 0, 0, 0, () => GrpcConfig$, [() => ForwardedValues$, 0], 1, 1, 1]
];
var CacheBehaviors$ = [3, n0, _CBa,
    0,
    [_Q, _I],
    [1, [() => CacheBehaviorList, 0]]
];
var CachedMethods$ = [3, n0, _CM,
    0,
    [_Q, _I],
    [1, [() => MethodsList, 0]]
];
var CachePolicy$ = [3, n0, _CP,
    0,
    [_Id, _LMT, _CPC],
    [0, 4, [() => CachePolicyConfig$, 0]]
];
var CachePolicyAlreadyExists$ = [-3, n0, _CPAE,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(CachePolicyAlreadyExists$, CachePolicyAlreadyExists);
var CachePolicyConfig$ = [3, n0, _CPC,
    0,
    [_Co, _N, _DTTL, _MTTLa, _MTTL, _PICKAFTO],
    [0, 0, 1, 1, 1, [() => ParametersInCacheKeyAndForwardedToOrigin$, 0]]
];
var CachePolicyCookiesConfig$ = [3, n0, _CPCC,
    0,
    [_CBo, _Coo],
    [0, [() => CookieNames$, 0]]
];
var CachePolicyHeadersConfig$ = [3, n0, _CPHC,
    0,
    [_HB, _H],
    [0, [() => Headers$, 0]]
];
var CachePolicyInUse$ = [-3, n0, _CPIU,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(CachePolicyInUse$, CachePolicyInUse);
var CachePolicyList$ = [3, n0, _CPL,
    0,
    [_NM, _MI, _Q, _I],
    [0, 1, 1, [() => CachePolicySummaryList, 0]]
];
var CachePolicyQueryStringsConfig$ = [3, n0, _CPQSC,
    0,
    [_QSB, _QS],
    [0, [() => QueryStringNames$, 0]]
];
var CachePolicySummary$ = [3, n0, _CPS,
    0,
    [_T, _CP],
    [0, [() => CachePolicy$, 0]]
];
var CannotChangeImmutablePublicKeyFields$ = [-3, n0, _CCIPKF,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(CannotChangeImmutablePublicKeyFields$, CannotChangeImmutablePublicKeyFields);
var CannotDeleteEntityWhileInUse$ = [-3, n0, _CDEWIU,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(CannotDeleteEntityWhileInUse$, CannotDeleteEntityWhileInUse);
var CannotUpdateEntityWhileInUse$ = [-3, n0, _CUEWIU,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(CannotUpdateEntityWhileInUse$, CannotUpdateEntityWhileInUse);
var Certificate$ = [3, n0, _Ce,
    0,
    [_Ar],
    [0]
];
var CloudFrontOriginAccessIdentity$ = [3, n0, _CFOAI,
    0,
    [_Id, _SCUI, _CFOAIC],
    [0, 0, () => CloudFrontOriginAccessIdentityConfig$]
];
var CloudFrontOriginAccessIdentityAlreadyExists$ = [-3, n0, _CFOAIAE,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(CloudFrontOriginAccessIdentityAlreadyExists$, CloudFrontOriginAccessIdentityAlreadyExists);
var CloudFrontOriginAccessIdentityConfig$ = [3, n0, _CFOAIC,
    0,
    [_CR, _Co],
    [0, 0]
];
var CloudFrontOriginAccessIdentityInUse$ = [-3, n0, _CFOAIIU,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(CloudFrontOriginAccessIdentityInUse$, CloudFrontOriginAccessIdentityInUse);
var CloudFrontOriginAccessIdentityList$ = [3, n0, _CFOAIL,
    0,
    [_Ma, _NM, _MI, _IT, _Q, _I],
    [0, 0, 1, 2, 1, [() => CloudFrontOriginAccessIdentitySummaryList, 0]]
];
var CloudFrontOriginAccessIdentitySummary$ = [3, n0, _CFOAIS,
    0,
    [_Id, _SCUI, _Co],
    [0, 0, 0]
];
var CNAMEAlreadyExists$ = [-3, n0, _CNAMEAE,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(CNAMEAlreadyExists$, CNAMEAlreadyExists);
var ConflictingAlias$ = [3, n0, _CA,
    0,
    [_Al, _DI, _AIc],
    [0, 0, 0]
];
var ConflictingAliasesList$ = [3, n0, _CAL,
    0,
    [_NM, _MI, _Q, _I],
    [0, 1, 1, [() => ConflictingAliases, 0]]
];
var ConnectionFunctionAssociation$ = [3, n0, _CFA,
    0,
    [_Id],
    [0]
];
var ConnectionFunctionSummary$ = [3, n0, _CFS,
    0,
    [_N, _Id, _CFC, _CFAo, _S, _St, _CTr, _LMT],
    [0, 0, [() => FunctionConfig$, 0], 0, 0, 0, 4, 4]
];
var ConnectionFunctionTestResult$ = [3, n0, _CFTR,
    0,
    [_CFS, _CU, _CFEL, _CFEM, _CFO],
    [[() => ConnectionFunctionSummary$, 0], 0, [() => FunctionExecutionLogList, 0], [() => sensitiveStringType, 0], [() => sensitiveStringType, 0]]
];
var ConnectionGroup$ = [3, n0, _CG,
    0,
    [_Id, _N, _Ar, _CTr, _LMT, _Ta, _IE, _RE, _AILI, _S, _E, _ID],
    [0, 0, 0, 4, 4, [() => Tags$, 0], 2, 0, 0, 0, 2, 2]
];
var ConnectionGroupAssociationFilter$ = [3, n0, _CGAF,
    0,
    [_AILI],
    [0]
];
var ConnectionGroupSummary$ = [3, n0, _CGS,
    0,
    [_Id, _N, _Ar, _RE, _CTr, _LMT, _ET, _AILI, _E, _S, _ID],
    [0, 0, 0, 0, 4, 4, 0, 0, 2, 0, 2]
];
var ContentTypeProfile$ = [3, n0, _CTP,
    0,
    [_F, _PI, _CTo],
    [0, 0, 0]
];
var ContentTypeProfileConfig$ = [3, n0, _CTPC,
    0,
    [_FWCTIU, _CTPo],
    [2, [() => ContentTypeProfiles$, 0]]
];
var ContentTypeProfiles$ = [3, n0, _CTPo,
    0,
    [_Q, _I],
    [1, [() => ContentTypeProfileList, 0]]
];
var ContinuousDeploymentPolicy$ = [3, n0, _CDP,
    0,
    [_Id, _LMT, _CDPC],
    [0, 4, [() => ContinuousDeploymentPolicyConfig$, 0]]
];
var ContinuousDeploymentPolicyAlreadyExists$ = [-3, n0, _CDPAE,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(ContinuousDeploymentPolicyAlreadyExists$, ContinuousDeploymentPolicyAlreadyExists);
var ContinuousDeploymentPolicyConfig$ = [3, n0, _CDPC,
    0,
    [_SDDN, _E, _TC],
    [[() => StagingDistributionDnsNames$, 0], 2, () => TrafficConfig$]
];
var ContinuousDeploymentPolicyInUse$ = [-3, n0, _CDPIU,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(ContinuousDeploymentPolicyInUse$, ContinuousDeploymentPolicyInUse);
var ContinuousDeploymentPolicyList$ = [3, n0, _CDPL,
    0,
    [_NM, _MI, _Q, _I],
    [0, 1, 1, [() => ContinuousDeploymentPolicySummaryList, 0]]
];
var ContinuousDeploymentPolicySummary$ = [3, n0, _CDPS,
    0,
    [_CDP],
    [[() => ContinuousDeploymentPolicy$, 0]]
];
var ContinuousDeploymentSingleHeaderConfig$ = [3, n0, _CDSHC,
    0,
    [_He, _Va],
    [0, 0]
];
var ContinuousDeploymentSingleWeightConfig$ = [3, n0, _CDSWC,
    0,
    [_W, _SSC],
    [1, () => SessionStickinessConfig$]
];
var CookieNames$ = [3, n0, _CN,
    0,
    [_Q, _I],
    [1, [() => CookieNameList, 0]]
];
var CookiePreference$ = [3, n0, _CPo,
    0,
    [_Fo, _WN],
    [0, [() => CookieNames$, 0]]
];
var CopyDistributionRequest$ = [3, n0, _CDR,
    0,
    [_PDI, _Sta, _IM, _CR, _E],
    [[0, 1], [2, { [_hH]: _Sta }], [0, { [_hH]: _IM_ }], 0, 2]
];
var CopyDistributionResult$ = [3, n0, _CDRo,
    0,
    [_D, _L, _ET],
    [[() => Distribution$, 16], [0, { [_hH]: _L }], [0, { [_hH]: _ET }]]
];
var CreateAnycastIpListRequest$ = [3, n0, _CAILR,
    0,
    [_N, _ICp, _Ta, _IAT, _ICC],
    [0, 1, [() => Tags$, 0], 0, [() => IpamCidrConfigList, 0]]
];
var CreateAnycastIpListResult$ = [3, n0, _CAILRr,
    0,
    [_AIL, _ET],
    [[() => AnycastIpList$, 16], [0, { [_hH]: _ET }]]
];
var CreateCachePolicyRequest$ = [3, n0, _CCPR,
    0,
    [_CPC],
    [[() => CachePolicyConfig$, { [_hP]: 1, [_xN]: _CPC }]]
];
var CreateCachePolicyResult$ = [3, n0, _CCPRr,
    0,
    [_CP, _L, _ET],
    [[() => CachePolicy$, 16], [0, { [_hH]: _L }], [0, { [_hH]: _ET }]]
];
var CreateCloudFrontOriginAccessIdentityRequest$ = [3, n0, _CCFOAIR,
    0,
    [_CFOAIC],
    [[() => CloudFrontOriginAccessIdentityConfig$, { [_hP]: 1, [_xN]: _CFOAIC }]]
];
var CreateCloudFrontOriginAccessIdentityResult$ = [3, n0, _CCFOAIRr,
    0,
    [_CFOAI, _L, _ET],
    [[() => CloudFrontOriginAccessIdentity$, 16], [0, { [_hH]: _L }], [0, { [_hH]: _ET }]]
];
var CreateConnectionFunctionRequest$ = [3, n0, _CCFR,
    0,
    [_N, _CFC, _CFCo, _Ta],
    [0, [() => FunctionConfig$, 0], [() => FunctionBlob, 0], [() => Tags$, 0]]
];
var CreateConnectionFunctionResult$ = [3, n0, _CCFRr,
    0,
    [_CFS, _L, _ET],
    [[() => ConnectionFunctionSummary$, 16], [0, { [_hH]: _L }], [0, { [_hH]: _ET }]]
];
var CreateConnectionGroupRequest$ = [3, n0, _CCGR,
    0,
    [_N, _IE, _Ta, _AILI, _E],
    [0, 2, [() => Tags$, 0], 0, 2]
];
var CreateConnectionGroupResult$ = [3, n0, _CCGRr,
    0,
    [_CG, _ET],
    [[() => ConnectionGroup$, 16], [0, { [_hH]: _ET }]]
];
var CreateContinuousDeploymentPolicyRequest$ = [3, n0, _CCDPR,
    0,
    [_CDPC],
    [[() => ContinuousDeploymentPolicyConfig$, { [_hP]: 1, [_xN]: _CDPC }]]
];
var CreateContinuousDeploymentPolicyResult$ = [3, n0, _CCDPRr,
    0,
    [_CDP, _L, _ET],
    [[() => ContinuousDeploymentPolicy$, 16], [0, { [_hH]: _L }], [0, { [_hH]: _ET }]]
];
var CreateDistributionRequest$ = [3, n0, _CDRr,
    0,
    [_DC],
    [[() => DistributionConfig$, { [_hP]: 1, [_xN]: _DC }]]
];
var CreateDistributionResult$ = [3, n0, _CDRre,
    0,
    [_D, _L, _ET],
    [[() => Distribution$, 16], [0, { [_hH]: _L }], [0, { [_hH]: _ET }]]
];
var CreateDistributionTenantRequest$ = [3, n0, _CDTR,
    0,
    [_DI, _N, _Do, _Ta, _Cu, _P, _CGI, _MCR, _E],
    [0, 0, () => DomainList, [() => Tags$, 0], [() => Customizations$, 0], () => _Parameters, 0, () => ManagedCertificateRequest$, 2]
];
var CreateDistributionTenantResult$ = [3, n0, _CDTRr,
    0,
    [_DT, _ET],
    [[() => DistributionTenant$, 16], [0, { [_hH]: _ET }]]
];
var CreateDistributionWithTagsRequest$ = [3, n0, _CDWTR,
    0,
    [_DCWT],
    [[() => DistributionConfigWithTags$, { [_hP]: 1, [_xN]: _DCWT }]]
];
var CreateDistributionWithTagsResult$ = [3, n0, _CDWTRr,
    0,
    [_D, _L, _ET],
    [[() => Distribution$, 16], [0, { [_hH]: _L }], [0, { [_hH]: _ET }]]
];
var CreateFieldLevelEncryptionConfigRequest$ = [3, n0, _CFLECR,
    0,
    [_FLEC],
    [[() => FieldLevelEncryptionConfig$, { [_hP]: 1, [_xN]: _FLEC }]]
];
var CreateFieldLevelEncryptionConfigResult$ = [3, n0, _CFLECRr,
    0,
    [_FLE, _L, _ET],
    [[() => FieldLevelEncryption$, 16], [0, { [_hH]: _L }], [0, { [_hH]: _ET }]]
];
var CreateFieldLevelEncryptionProfileRequest$ = [3, n0, _CFLEPR,
    0,
    [_FLEPC],
    [[() => FieldLevelEncryptionProfileConfig$, { [_hP]: 1, [_xN]: _FLEPC }]]
];
var CreateFieldLevelEncryptionProfileResult$ = [3, n0, _CFLEPRr,
    0,
    [_FLEP, _L, _ET],
    [[() => FieldLevelEncryptionProfile$, 16], [0, { [_hH]: _L }], [0, { [_hH]: _ET }]]
];
var CreateFunctionRequest$ = [3, n0, _CFR,
    0,
    [_N, _FC, _FCu],
    [0, [() => FunctionConfig$, 0], [() => FunctionBlob, 0]]
];
var CreateFunctionResult$ = [3, n0, _CFRr,
    0,
    [_FS, _L, _ET],
    [[() => FunctionSummary$, 16], [0, { [_hH]: _L }], [0, { [_hH]: _ET }]]
];
var CreateInvalidationForDistributionTenantRequest$ = [3, n0, _CIFDTR,
    0,
    [_Id, _IB],
    [[0, 1], [() => InvalidationBatch$, { [_hP]: 1, [_xN]: _IB }]]
];
var CreateInvalidationForDistributionTenantResult$ = [3, n0, _CIFDTRr,
    0,
    [_L, _In],
    [[0, { [_hH]: _L }], [() => Invalidation$, 16]]
];
var CreateInvalidationRequest$ = [3, n0, _CIR,
    0,
    [_DI, _IB],
    [[0, 1], [() => InvalidationBatch$, { [_hP]: 1, [_xN]: _IB }]]
];
var CreateInvalidationResult$ = [3, n0, _CIRr,
    0,
    [_L, _In],
    [[0, { [_hH]: _L }], [() => Invalidation$, 16]]
];
var CreateKeyGroupRequest$ = [3, n0, _CKGR,
    0,
    [_KGC],
    [[() => KeyGroupConfig$, { [_hP]: 1, [_xN]: _KGC }]]
];
var CreateKeyGroupResult$ = [3, n0, _CKGRr,
    0,
    [_KG, _L, _ET],
    [[() => KeyGroup$, 16], [0, { [_hH]: _L }], [0, { [_hH]: _ET }]]
];
var CreateKeyValueStoreRequest$ = [3, n0, _CKVSR,
    0,
    [_N, _Co, _IS],
    [0, 0, () => ImportSource$]
];
var CreateKeyValueStoreResult$ = [3, n0, _CKVSRr,
    0,
    [_KVS, _ET, _L],
    [[() => KeyValueStore$, 16], [0, { [_hH]: _ET }], [0, { [_hH]: _L }]]
];
var CreateMonitoringSubscriptionRequest$ = [3, n0, _CMSR,
    0,
    [_DI, _MS],
    [[0, 1], [() => MonitoringSubscription$, { [_hP]: 1, [_xN]: _MS }]]
];
var CreateMonitoringSubscriptionResult$ = [3, n0, _CMSRr,
    0,
    [_MS],
    [[() => MonitoringSubscription$, 16]]
];
var CreateOriginAccessControlRequest$ = [3, n0, _COACR,
    0,
    [_OACC],
    [[() => OriginAccessControlConfig$, { [_hP]: 1, [_xN]: _OACC }]]
];
var CreateOriginAccessControlResult$ = [3, n0, _COACRr,
    0,
    [_OAC, _L, _ET],
    [[() => OriginAccessControl$, 16], [0, { [_hH]: _L }], [0, { [_hH]: _ET }]]
];
var CreateOriginRequestPolicyRequest$ = [3, n0, _CORPR,
    0,
    [_ORPC],
    [[() => OriginRequestPolicyConfig$, { [_hP]: 1, [_xN]: _ORPC }]]
];
var CreateOriginRequestPolicyResult$ = [3, n0, _CORPRr,
    0,
    [_ORP, _L, _ET],
    [[() => OriginRequestPolicy$, 16], [0, { [_hH]: _L }], [0, { [_hH]: _ET }]]
];
var CreatePublicKeyRequest$ = [3, n0, _CPKR,
    0,
    [_PKC],
    [[() => PublicKeyConfig$, { [_hP]: 1, [_xN]: _PKC }]]
];
var CreatePublicKeyResult$ = [3, n0, _CPKRr,
    0,
    [_PK, _L, _ET],
    [[() => PublicKey$, 16], [0, { [_hH]: _L }], [0, { [_hH]: _ET }]]
];
var CreateRealtimeLogConfigRequest$ = [3, n0, _CRLCR,
    0,
    [_EP, _Fi, _N, _SR],
    [() => EndPointList, [() => FieldList, 0], 0, 1]
];
var CreateRealtimeLogConfigResult$ = [3, n0, _CRLCRr,
    0,
    [_RLC],
    [[() => RealtimeLogConfig$, 0]]
];
var CreateResponseHeadersPolicyRequest$ = [3, n0, _CRHPR,
    0,
    [_RHPC],
    [[() => ResponseHeadersPolicyConfig$, { [_hP]: 1, [_xN]: _RHPC }]]
];
var CreateResponseHeadersPolicyResult$ = [3, n0, _CRHPRr,
    0,
    [_RHP, _L, _ET],
    [[() => ResponseHeadersPolicy$, 16], [0, { [_hH]: _L }], [0, { [_hH]: _ET }]]
];
var CreateStreamingDistributionRequest$ = [3, n0, _CSDR,
    0,
    [_SDC],
    [[() => StreamingDistributionConfig$, { [_hP]: 1, [_xN]: _SDC }]]
];
var CreateStreamingDistributionResult$ = [3, n0, _CSDRr,
    0,
    [_SD, _L, _ET],
    [[() => StreamingDistribution$, 16], [0, { [_hH]: _L }], [0, { [_hH]: _ET }]]
];
var CreateStreamingDistributionWithTagsRequest$ = [3, n0, _CSDWTR,
    0,
    [_SDCWT],
    [[() => StreamingDistributionConfigWithTags$, { [_hP]: 1, [_xN]: _SDCWT }]]
];
var CreateStreamingDistributionWithTagsResult$ = [3, n0, _CSDWTRr,
    0,
    [_SD, _L, _ET],
    [[() => StreamingDistribution$, 16], [0, { [_hH]: _L }], [0, { [_hH]: _ET }]]
];
var CreateTrustStoreRequest$ = [3, n0, _CTSR,
    0,
    [_N, _CCBS, _Ta],
    [0, () => CaCertificatesBundleSource$, [() => Tags$, 0]]
];
var CreateTrustStoreResult$ = [3, n0, _CTSRr,
    0,
    [_TSr, _ET],
    [[() => TrustStore$, 16], [0, { [_hH]: _ET }]]
];
var CreateVpcOriginRequest$ = [3, n0, _CVOR,
    0,
    [_VOEC, _Ta],
    [[() => VpcOriginEndpointConfig$, 0], [() => Tags$, 0]]
];
var CreateVpcOriginResult$ = [3, n0, _CVORr,
    0,
    [_VO, _L, _ET],
    [[() => VpcOrigin$, 16], [0, { [_hH]: _L }], [0, { [_hH]: _ET }]]
];
var CustomErrorResponse$ = [3, n0, _CER,
    0,
    [_EC, _RPP, _RC, _ECMTTL],
    [1, 0, 0, 1]
];
var CustomErrorResponses$ = [3, n0, _CERu,
    0,
    [_Q, _I],
    [1, [() => CustomErrorResponseList, 0]]
];
var CustomHeaders$ = [3, n0, _CH,
    0,
    [_Q, _I],
    [1, [() => OriginCustomHeadersList, 0]]
];
var Customizations$ = [3, n0, _Cu,
    0,
    [_WA, _Ce, _GR],
    [() => WebAclCustomization$, () => Certificate$, [() => GeoRestrictionCustomization$, 0]]
];
var CustomOriginConfig$ = [3, n0, _COC,
    0,
    [_HTTPP, _HTTPSP, _OPP, _OSP, _ORT, _OKT, _IAT],
    [1, 1, 0, [() => OriginSslProtocols$, 0], 1, 1, 0]
];
var DefaultCacheBehavior$ = [3, n0, _DCB,
    0,
    [_TOI, _TS, _TKG, _VPP, _AM, _SS, _C, _LFA, _FA, _FLEI, _RLCA, _CPI, _ORPI, _RHPI, _GC, _FV, _MTTL, _DTTL, _MTTLa],
    [0, [() => TrustedSigners$, 0], [() => TrustedKeyGroups$, 0], 0, [() => AllowedMethods$, 0], 2, 2, [() => LambdaFunctionAssociations$, 0], [() => FunctionAssociations$, 0], 0, 0, 0, 0, 0, () => GrpcConfig$, [() => ForwardedValues$, 0], 1, 1, 1]
];
var DeleteAnycastIpListRequest$ = [3, n0, _DAILR,
    0,
    [_Id, _IM],
    [[0, 1], [0, { [_hH]: _IM_ }]]
];
var DeleteCachePolicyRequest$ = [3, n0, _DCPR,
    0,
    [_Id, _IM],
    [[0, 1], [0, { [_hH]: _IM_ }]]
];
var DeleteCloudFrontOriginAccessIdentityRequest$ = [3, n0, _DCFOAIR,
    0,
    [_Id, _IM],
    [[0, 1], [0, { [_hH]: _IM_ }]]
];
var DeleteConnectionFunctionRequest$ = [3, n0, _DCFR,
    0,
    [_Id, _IM],
    [[0, 1], [0, { [_hH]: _IM_ }]]
];
var DeleteConnectionGroupRequest$ = [3, n0, _DCGR,
    0,
    [_Id, _IM],
    [[0, 1], [0, { [_hH]: _IM_ }]]
];
var DeleteContinuousDeploymentPolicyRequest$ = [3, n0, _DCDPR,
    0,
    [_Id, _IM],
    [[0, 1], [0, { [_hH]: _IM_ }]]
];
var DeleteDistributionRequest$ = [3, n0, _DDR,
    0,
    [_Id, _IM],
    [[0, 1], [0, { [_hH]: _IM_ }]]
];
var DeleteDistributionTenantRequest$ = [3, n0, _DDTR,
    0,
    [_Id, _IM],
    [[0, 1], [0, { [_hH]: _IM_ }]]
];
var DeleteFieldLevelEncryptionConfigRequest$ = [3, n0, _DFLECR,
    0,
    [_Id, _IM],
    [[0, 1], [0, { [_hH]: _IM_ }]]
];
var DeleteFieldLevelEncryptionProfileRequest$ = [3, n0, _DFLEPR,
    0,
    [_Id, _IM],
    [[0, 1], [0, { [_hH]: _IM_ }]]
];
var DeleteFunctionRequest$ = [3, n0, _DFR,
    0,
    [_N, _IM],
    [[0, 1], [0, { [_hH]: _IM_ }]]
];
var DeleteKeyGroupRequest$ = [3, n0, _DKGR,
    0,
    [_Id, _IM],
    [[0, 1], [0, { [_hH]: _IM_ }]]
];
var DeleteKeyValueStoreRequest$ = [3, n0, _DKVSR,
    0,
    [_N, _IM],
    [[0, 1], [0, { [_hH]: _IM_ }]]
];
var DeleteMonitoringSubscriptionRequest$ = [3, n0, _DMSR,
    0,
    [_DI],
    [[0, 1]]
];
var DeleteMonitoringSubscriptionResult$ = [3, n0, _DMSRe,
    0,
    [],
    []
];
var DeleteOriginAccessControlRequest$ = [3, n0, _DOACR,
    0,
    [_Id, _IM],
    [[0, 1], [0, { [_hH]: _IM_ }]]
];
var DeleteOriginRequestPolicyRequest$ = [3, n0, _DORPR,
    0,
    [_Id, _IM],
    [[0, 1], [0, { [_hH]: _IM_ }]]
];
var DeletePublicKeyRequest$ = [3, n0, _DPKR,
    0,
    [_Id, _IM],
    [[0, 1], [0, { [_hH]: _IM_ }]]
];
var DeleteRealtimeLogConfigRequest$ = [3, n0, _DRLCR,
    0,
    [_N, _ARN],
    [0, 0]
];
var DeleteResourcePolicyRequest$ = [3, n0, _DRPR,
    0,
    [_RA],
    [0]
];
var DeleteResponseHeadersPolicyRequest$ = [3, n0, _DRHPR,
    0,
    [_Id, _IM],
    [[0, 1], [0, { [_hH]: _IM_ }]]
];
var DeleteStreamingDistributionRequest$ = [3, n0, _DSDR,
    0,
    [_Id, _IM],
    [[0, 1], [0, { [_hH]: _IM_ }]]
];
var DeleteTrustStoreRequest$ = [3, n0, _DTSR,
    0,
    [_Id, _IM],
    [[0, 1], [0, { [_hH]: _IM_ }]]
];
var DeleteVpcOriginRequest$ = [3, n0, _DVOR,
    0,
    [_Id, _IM],
    [[0, 1], [0, { [_hH]: _IM_ }]]
];
var DeleteVpcOriginResult$ = [3, n0, _DVORe,
    0,
    [_VO, _ET],
    [[() => VpcOrigin$, 16], [0, { [_hH]: _ET }]]
];
var DescribeConnectionFunctionRequest$ = [3, n0, _DCFRe,
    0,
    [_Ide, _St],
    [[0, 1], [0, { [_hQ]: _St }]]
];
var DescribeConnectionFunctionResult$ = [3, n0, _DCFRes,
    0,
    [_CFS, _ET],
    [[() => ConnectionFunctionSummary$, 16], [0, { [_hH]: _ET }]]
];
var DescribeFunctionRequest$ = [3, n0, _DFRe,
    0,
    [_N, _St],
    [[0, 1], [0, { [_hQ]: _St }]]
];
var DescribeFunctionResult$ = [3, n0, _DFRes,
    0,
    [_FS, _ET],
    [[() => FunctionSummary$, 16], [0, { [_hH]: _ET }]]
];
var DescribeKeyValueStoreRequest$ = [3, n0, _DKVSRe,
    0,
    [_N],
    [[0, 1]]
];
var DescribeKeyValueStoreResult$ = [3, n0, _DKVSRes,
    0,
    [_KVS, _ET],
    [[() => KeyValueStore$, 16], [0, { [_hH]: _ET }]]
];
var DisassociateDistributionTenantWebACLRequest$ = [3, n0, _DDTWACLR,
    0,
    [_Id, _IM],
    [[0, 1], [0, { [_hH]: _IM_ }]]
];
var DisassociateDistributionTenantWebACLResult$ = [3, n0, _DDTWACLRi,
    0,
    [_Id, _ET],
    [0, [0, { [_hH]: _ET }]]
];
var DisassociateDistributionWebACLRequest$ = [3, n0, _DDWACLR,
    0,
    [_Id, _IM],
    [[0, 1], [0, { [_hH]: _IM_ }]]
];
var DisassociateDistributionWebACLResult$ = [3, n0, _DDWACLRi,
    0,
    [_Id, _ET],
    [0, [0, { [_hH]: _ET }]]
];
var Distribution$ = [3, n0, _D,
    0,
    [_Id, _ARN, _S, _LMT, _IPIB, _DN, _ATS, _ATKG, _DC, _AICPRl],
    [0, 0, 0, 4, 1, 0, [() => ActiveTrustedSigners$, 0], [() => ActiveTrustedKeyGroups$, 0], [() => DistributionConfig$, 0], [() => AliasICPRecordals, 0]]
];
var DistributionAlreadyExists$ = [-3, n0, _DAE,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(DistributionAlreadyExists$, DistributionAlreadyExists);
var DistributionConfig$ = [3, n0, _DC,
    0,
    [_CR, _A, _DRO, _O, _OG, _DCB, _CBa, _CERu, _Co, _Lo, _PC, _E, _VC, _Re, _WACLI, _HV, _IIPVE, _CDPI, _Sta, _AILI, _TCe, _CMo, _VMC, _CFA],
    [0, [() => Aliases$, 0], 0, [() => Origins$, 0], [() => OriginGroups$, 0], [() => DefaultCacheBehavior$, 0], [() => CacheBehaviors$, 0], [() => CustomErrorResponses$, 0], [() => CommentType, 0], () => LoggingConfig$, 0, 2, () => ViewerCertificate$, [() => Restrictions$, 0], 0, 0, 2, 0, 2, 0, [() => TenantConfig$, 0], 0, () => ViewerMtlsConfig$, () => ConnectionFunctionAssociation$]
];
var DistributionConfigWithTags$ = [3, n0, _DCWT,
    0,
    [_DC, _Ta],
    [[() => DistributionConfig$, 0], [() => Tags$, 0]]
];
var DistributionIdList$ = [3, n0, _DIL,
    0,
    [_Ma, _NM, _MI, _IT, _Q, _I],
    [0, 0, 1, 2, 1, [() => DistributionIdListSummary, 0]]
];
var DistributionIdOwner$ = [3, n0, _DIO,
    0,
    [_DI, _OAI],
    [0, 0]
];
var DistributionIdOwnerList$ = [3, n0, _DIOL,
    0,
    [_Ma, _NM, _MI, _IT, _Q, _I],
    [0, 0, 1, 2, 1, [() => DistributionIdOwnerItemList, 0]]
];
var DistributionList$ = [3, n0, _DL,
    0,
    [_Ma, _NM, _MI, _IT, _Q, _I],
    [0, 0, 1, 2, 1, [() => DistributionSummaryList, 0]]
];
var DistributionNotDisabled$ = [-3, n0, _DND,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(DistributionNotDisabled$, DistributionNotDisabled);
var DistributionResourceId$ = [3, n0, _DRI,
    0,
    [_DI, _DTI],
    [0, 0]
];
var DistributionSummary$ = [3, n0, _DS,
    0,
    [_Id, _ARN, _ET, _S, _LMT, _DN, _A, _O, _OG, _DCB, _CBa, _CERu, _Co, _PC, _E, _VC, _Re, _WACLI, _HV, _IIPVE, _AICPRl, _Sta, _CMo, _AILI, _VMC, _CFA],
    [0, 0, 0, 0, 4, 0, [() => Aliases$, 0], [() => Origins$, 0], [() => OriginGroups$, 0], [() => DefaultCacheBehavior$, 0], [() => CacheBehaviors$, 0], [() => CustomErrorResponses$, 0], [() => sensitiveStringType, 0], 0, 2, () => ViewerCertificate$, [() => Restrictions$, 0], 0, 0, 2, [() => AliasICPRecordals, 0], 2, 0, 0, () => ViewerMtlsConfig$, () => ConnectionFunctionAssociation$]
];
var DistributionTenant$ = [3, n0, _DT,
    0,
    [_Id, _DI, _N, _Ar, _Do, _Ta, _Cu, _P, _CGI, _CTr, _LMT, _E, _S],
    [0, 0, 0, 0, () => DomainResultList, [() => Tags$, 0], [() => Customizations$, 0], () => _Parameters, 0, 4, 4, 2, 0]
];
var DistributionTenantAssociationFilter$ = [3, n0, _DTAF,
    0,
    [_DI, _CGI],
    [0, 0]
];
var DistributionTenantSummary$ = [3, n0, _DTS,
    0,
    [_Id, _DI, _N, _Ar, _Do, _CGI, _Cu, _CTr, _LMT, _ET, _E, _S],
    [0, 0, 0, 0, () => DomainResultList, 0, [() => Customizations$, 0], 4, 4, 0, 2, 0]
];
var DnsConfiguration$ = [3, n0, _DCn,
    0,
    [_Dom, _S, _Rea],
    [0, 0, 0]
];
var DomainConflict$ = [3, n0, _DCo,
    0,
    [_Dom, _RT, _RI, _AIc],
    [0, 0, 0, 0]
];
var DomainItem$ = [3, n0, _DIo,
    0,
    [_Dom],
    [0]
];
var DomainResult$ = [3, n0, _DR,
    0,
    [_Dom, _S],
    [0, 0]
];
var EncryptionEntities$ = [3, n0, _EE,
    0,
    [_Q, _I],
    [1, [() => EncryptionEntityList, 0]]
];
var EncryptionEntity$ = [3, n0, _EEn,
    0,
    [_PKI, _PIr, _FP],
    [0, 0, [() => FieldPatterns$, 0]]
];
var EndPoint$ = [3, n0, _EPn,
    0,
    [_ST, _KSC],
    [0, () => KinesisStreamConfig$]
];
var EntityAlreadyExists$ = [-3, n0, _EAE,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(EntityAlreadyExists$, EntityAlreadyExists);
var EntityLimitExceeded$ = [-3, n0, _ELE,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(EntityLimitExceeded$, EntityLimitExceeded);
var EntityNotFound$ = [-3, n0, _ENF,
    { [_e]: _c, [_hE]: 404 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(EntityNotFound$, EntityNotFound);
var EntitySizeLimitExceeded$ = [-3, n0, _ESLE,
    { [_e]: _c, [_hE]: 413 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(EntitySizeLimitExceeded$, EntitySizeLimitExceeded);
var FieldLevelEncryption$ = [3, n0, _FLE,
    0,
    [_Id, _LMT, _FLEC],
    [0, 4, [() => FieldLevelEncryptionConfig$, 0]]
];
var FieldLevelEncryptionConfig$ = [3, n0, _FLEC,
    0,
    [_CR, _Co, _QAPC, _CTPC],
    [0, 0, [() => QueryArgProfileConfig$, 0], [() => ContentTypeProfileConfig$, 0]]
];
var FieldLevelEncryptionConfigAlreadyExists$ = [-3, n0, _FLECAE,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(FieldLevelEncryptionConfigAlreadyExists$, FieldLevelEncryptionConfigAlreadyExists);
var FieldLevelEncryptionConfigInUse$ = [-3, n0, _FLECIU,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(FieldLevelEncryptionConfigInUse$, FieldLevelEncryptionConfigInUse);
var FieldLevelEncryptionList$ = [3, n0, _FLEL,
    0,
    [_NM, _MI, _Q, _I],
    [0, 1, 1, [() => FieldLevelEncryptionSummaryList, 0]]
];
var FieldLevelEncryptionProfile$ = [3, n0, _FLEP,
    0,
    [_Id, _LMT, _FLEPC],
    [0, 4, [() => FieldLevelEncryptionProfileConfig$, 0]]
];
var FieldLevelEncryptionProfileAlreadyExists$ = [-3, n0, _FLEPAE,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(FieldLevelEncryptionProfileAlreadyExists$, FieldLevelEncryptionProfileAlreadyExists);
var FieldLevelEncryptionProfileConfig$ = [3, n0, _FLEPC,
    0,
    [_N, _CR, _Co, _EE],
    [0, 0, 0, [() => EncryptionEntities$, 0]]
];
var FieldLevelEncryptionProfileInUse$ = [-3, n0, _FLEPIU,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(FieldLevelEncryptionProfileInUse$, FieldLevelEncryptionProfileInUse);
var FieldLevelEncryptionProfileList$ = [3, n0, _FLEPL,
    0,
    [_NM, _MI, _Q, _I],
    [0, 1, 1, [() => FieldLevelEncryptionProfileSummaryList, 0]]
];
var FieldLevelEncryptionProfileSizeExceeded$ = [-3, n0, _FLEPSE,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(FieldLevelEncryptionProfileSizeExceeded$, FieldLevelEncryptionProfileSizeExceeded);
var FieldLevelEncryptionProfileSummary$ = [3, n0, _FLEPS,
    0,
    [_Id, _LMT, _N, _EE, _Co],
    [0, 4, 0, [() => EncryptionEntities$, 0], 0]
];
var FieldLevelEncryptionSummary$ = [3, n0, _FLES,
    0,
    [_Id, _LMT, _Co, _QAPC, _CTPC],
    [0, 4, 0, [() => QueryArgProfileConfig$, 0], [() => ContentTypeProfileConfig$, 0]]
];
var FieldPatterns$ = [3, n0, _FP,
    0,
    [_Q, _I],
    [1, [() => FieldPatternList, 0]]
];
var ForwardedValues$ = [3, n0, _FV,
    0,
    [_QSu, _Coo, _H, _QSCK],
    [2, [() => CookiePreference$, 0], [() => Headers$, 0], [() => QueryStringCacheKeys$, 0]]
];
var FunctionAlreadyExists$ = [-3, n0, _FAE,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(FunctionAlreadyExists$, FunctionAlreadyExists);
var FunctionAssociation$ = [3, n0, _FAu,
    0,
    [_FARN, _ETv],
    [0, 0]
];
var FunctionAssociations$ = [3, n0, _FA,
    0,
    [_Q, _I],
    [1, [() => FunctionAssociationList, 0]]
];
var FunctionConfig$ = [3, n0, _FC,
    0,
    [_Co, _Ru, _KVSA],
    [0, 0, [() => KeyValueStoreAssociations$, 0]]
];
var FunctionInUse$ = [-3, n0, _FIU,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(FunctionInUse$, FunctionInUse);
var FunctionList$ = [3, n0, _FL,
    0,
    [_NM, _MI, _Q, _I],
    [0, 1, 1, [() => FunctionSummaryList, 0]]
];
var FunctionMetadata$ = [3, n0, _FM,
    0,
    [_FARN, _St, _CTr, _LMT],
    [0, 0, 4, 4]
];
var FunctionSizeLimitExceeded$ = [-3, n0, _FSLE,
    { [_e]: _c, [_hE]: 413 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(FunctionSizeLimitExceeded$, FunctionSizeLimitExceeded);
var FunctionSummary$ = [3, n0, _FS,
    0,
    [_N, _S, _FC, _FM],
    [0, 0, [() => FunctionConfig$, 0], () => FunctionMetadata$]
];
var GeoRestriction$ = [3, n0, _GRe,
    0,
    [_RTe, _Q, _I],
    [0, 1, [() => LocationList, 0]]
];
var GeoRestrictionCustomization$ = [3, n0, _GRC,
    0,
    [_RTe, _Loc],
    [0, [() => LocationList, 0]]
];
var GetAnycastIpListRequest$ = [3, n0, _GAILR,
    0,
    [_Id],
    [[0, 1]]
];
var GetAnycastIpListResult$ = [3, n0, _GAILRe,
    0,
    [_AIL, _ET],
    [[() => AnycastIpList$, 16], [0, { [_hH]: _ET }]]
];
var GetCachePolicyConfigRequest$ = [3, n0, _GCPCR,
    0,
    [_Id],
    [[0, 1]]
];
var GetCachePolicyConfigResult$ = [3, n0, _GCPCRe,
    0,
    [_CPC, _ET],
    [[() => CachePolicyConfig$, 16], [0, { [_hH]: _ET }]]
];
var GetCachePolicyRequest$ = [3, n0, _GCPR,
    0,
    [_Id],
    [[0, 1]]
];
var GetCachePolicyResult$ = [3, n0, _GCPRe,
    0,
    [_CP, _ET],
    [[() => CachePolicy$, 16], [0, { [_hH]: _ET }]]
];
var GetCloudFrontOriginAccessIdentityConfigRequest$ = [3, n0, _GCFOAICR,
    0,
    [_Id],
    [[0, 1]]
];
var GetCloudFrontOriginAccessIdentityConfigResult$ = [3, n0, _GCFOAICRe,
    0,
    [_CFOAIC, _ET],
    [[() => CloudFrontOriginAccessIdentityConfig$, 16], [0, { [_hH]: _ET }]]
];
var GetCloudFrontOriginAccessIdentityRequest$ = [3, n0, _GCFOAIR,
    0,
    [_Id],
    [[0, 1]]
];
var GetCloudFrontOriginAccessIdentityResult$ = [3, n0, _GCFOAIRe,
    0,
    [_CFOAI, _ET],
    [[() => CloudFrontOriginAccessIdentity$, 16], [0, { [_hH]: _ET }]]
];
var GetConnectionFunctionRequest$ = [3, n0, _GCFR,
    0,
    [_Ide, _St],
    [[0, 1], [0, { [_hQ]: _St }]]
];
var GetConnectionFunctionResult$ = [3, n0, _GCFRe,
    0,
    [_CFCo, _ET, _CTo],
    [[() => FunctionBlob, 16], [0, { [_hH]: _ET }], [0, { [_hH]: _CT_ }]]
];
var GetConnectionGroupByRoutingEndpointRequest$ = [3, n0, _GCGBRER,
    0,
    [_RE],
    [[0, { [_hQ]: _RE }]]
];
var GetConnectionGroupByRoutingEndpointResult$ = [3, n0, _GCGBRERe,
    0,
    [_CG, _ET],
    [[() => ConnectionGroup$, 16], [0, { [_hH]: _ET }]]
];
var GetConnectionGroupRequest$ = [3, n0, _GCGR,
    0,
    [_Ide],
    [[0, 1]]
];
var GetConnectionGroupResult$ = [3, n0, _GCGRe,
    0,
    [_CG, _ET],
    [[() => ConnectionGroup$, 16], [0, { [_hH]: _ET }]]
];
var GetContinuousDeploymentPolicyConfigRequest$ = [3, n0, _GCDPCR,
    0,
    [_Id],
    [[0, 1]]
];
var GetContinuousDeploymentPolicyConfigResult$ = [3, n0, _GCDPCRe,
    0,
    [_CDPC, _ET],
    [[() => ContinuousDeploymentPolicyConfig$, 16], [0, { [_hH]: _ET }]]
];
var GetContinuousDeploymentPolicyRequest$ = [3, n0, _GCDPR,
    0,
    [_Id],
    [[0, 1]]
];
var GetContinuousDeploymentPolicyResult$ = [3, n0, _GCDPRe,
    0,
    [_CDP, _ET],
    [[() => ContinuousDeploymentPolicy$, 16], [0, { [_hH]: _ET }]]
];
var GetDistributionConfigRequest$ = [3, n0, _GDCR,
    0,
    [_Id],
    [[0, 1]]
];
var GetDistributionConfigResult$ = [3, n0, _GDCRe,
    0,
    [_DC, _ET],
    [[() => DistributionConfig$, 16], [0, { [_hH]: _ET }]]
];
var GetDistributionRequest$ = [3, n0, _GDR,
    0,
    [_Id],
    [[0, 1]]
];
var GetDistributionResult$ = [3, n0, _GDRe,
    0,
    [_D, _ET],
    [[() => Distribution$, 16], [0, { [_hH]: _ET }]]
];
var GetDistributionTenantByDomainRequest$ = [3, n0, _GDTBDR,
    0,
    [_Dom],
    [[0, { [_hQ]: _d }]]
];
var GetDistributionTenantByDomainResult$ = [3, n0, _GDTBDRe,
    0,
    [_DT, _ET],
    [[() => DistributionTenant$, 16], [0, { [_hH]: _ET }]]
];
var GetDistributionTenantRequest$ = [3, n0, _GDTR,
    0,
    [_Ide],
    [[0, 1]]
];
var GetDistributionTenantResult$ = [3, n0, _GDTRe,
    0,
    [_DT, _ET],
    [[() => DistributionTenant$, 16], [0, { [_hH]: _ET }]]
];
var GetFieldLevelEncryptionConfigRequest$ = [3, n0, _GFLECR,
    0,
    [_Id],
    [[0, 1]]
];
var GetFieldLevelEncryptionConfigResult$ = [3, n0, _GFLECRe,
    0,
    [_FLEC, _ET],
    [[() => FieldLevelEncryptionConfig$, 16], [0, { [_hH]: _ET }]]
];
var GetFieldLevelEncryptionProfileConfigRequest$ = [3, n0, _GFLEPCR,
    0,
    [_Id],
    [[0, 1]]
];
var GetFieldLevelEncryptionProfileConfigResult$ = [3, n0, _GFLEPCRe,
    0,
    [_FLEPC, _ET],
    [[() => FieldLevelEncryptionProfileConfig$, 16], [0, { [_hH]: _ET }]]
];
var GetFieldLevelEncryptionProfileRequest$ = [3, n0, _GFLEPR,
    0,
    [_Id],
    [[0, 1]]
];
var GetFieldLevelEncryptionProfileResult$ = [3, n0, _GFLEPRe,
    0,
    [_FLEP, _ET],
    [[() => FieldLevelEncryptionProfile$, 16], [0, { [_hH]: _ET }]]
];
var GetFieldLevelEncryptionRequest$ = [3, n0, _GFLER,
    0,
    [_Id],
    [[0, 1]]
];
var GetFieldLevelEncryptionResult$ = [3, n0, _GFLERe,
    0,
    [_FLE, _ET],
    [[() => FieldLevelEncryption$, 16], [0, { [_hH]: _ET }]]
];
var GetFunctionRequest$ = [3, n0, _GFR,
    0,
    [_N, _St],
    [[0, 1], [0, { [_hQ]: _St }]]
];
var GetFunctionResult$ = [3, n0, _GFRe,
    0,
    [_FCu, _ET, _CTo],
    [[() => FunctionBlob, 16], [0, { [_hH]: _ET }], [0, { [_hH]: _CT_ }]]
];
var GetInvalidationForDistributionTenantRequest$ = [3, n0, _GIFDTR,
    0,
    [_DTI, _Id],
    [[0, 1], [0, 1]]
];
var GetInvalidationForDistributionTenantResult$ = [3, n0, _GIFDTRe,
    0,
    [_In],
    [[() => Invalidation$, 16]]
];
var GetInvalidationRequest$ = [3, n0, _GIR,
    0,
    [_DI, _Id],
    [[0, 1], [0, 1]]
];
var GetInvalidationResult$ = [3, n0, _GIRe,
    0,
    [_In],
    [[() => Invalidation$, 16]]
];
var GetKeyGroupConfigRequest$ = [3, n0, _GKGCR,
    0,
    [_Id],
    [[0, 1]]
];
var GetKeyGroupConfigResult$ = [3, n0, _GKGCRe,
    0,
    [_KGC, _ET],
    [[() => KeyGroupConfig$, 16], [0, { [_hH]: _ET }]]
];
var GetKeyGroupRequest$ = [3, n0, _GKGR,
    0,
    [_Id],
    [[0, 1]]
];
var GetKeyGroupResult$ = [3, n0, _GKGRe,
    0,
    [_KG, _ET],
    [[() => KeyGroup$, 16], [0, { [_hH]: _ET }]]
];
var GetManagedCertificateDetailsRequest$ = [3, n0, _GMCDR,
    0,
    [_Ide],
    [[0, 1]]
];
var GetManagedCertificateDetailsResult$ = [3, n0, _GMCDRe,
    0,
    [_MCD],
    [[() => ManagedCertificateDetails$, 16]]
];
var GetMonitoringSubscriptionRequest$ = [3, n0, _GMSR,
    0,
    [_DI],
    [[0, 1]]
];
var GetMonitoringSubscriptionResult$ = [3, n0, _GMSRe,
    0,
    [_MS],
    [[() => MonitoringSubscription$, 16]]
];
var GetOriginAccessControlConfigRequest$ = [3, n0, _GOACCR,
    0,
    [_Id],
    [[0, 1]]
];
var GetOriginAccessControlConfigResult$ = [3, n0, _GOACCRe,
    0,
    [_OACC, _ET],
    [[() => OriginAccessControlConfig$, 16], [0, { [_hH]: _ET }]]
];
var GetOriginAccessControlRequest$ = [3, n0, _GOACR,
    0,
    [_Id],
    [[0, 1]]
];
var GetOriginAccessControlResult$ = [3, n0, _GOACRe,
    0,
    [_OAC, _ET],
    [[() => OriginAccessControl$, 16], [0, { [_hH]: _ET }]]
];
var GetOriginRequestPolicyConfigRequest$ = [3, n0, _GORPCR,
    0,
    [_Id],
    [[0, 1]]
];
var GetOriginRequestPolicyConfigResult$ = [3, n0, _GORPCRe,
    0,
    [_ORPC, _ET],
    [[() => OriginRequestPolicyConfig$, 16], [0, { [_hH]: _ET }]]
];
var GetOriginRequestPolicyRequest$ = [3, n0, _GORPR,
    0,
    [_Id],
    [[0, 1]]
];
var GetOriginRequestPolicyResult$ = [3, n0, _GORPRe,
    0,
    [_ORP, _ET],
    [[() => OriginRequestPolicy$, 16], [0, { [_hH]: _ET }]]
];
var GetPublicKeyConfigRequest$ = [3, n0, _GPKCR,
    0,
    [_Id],
    [[0, 1]]
];
var GetPublicKeyConfigResult$ = [3, n0, _GPKCRe,
    0,
    [_PKC, _ET],
    [[() => PublicKeyConfig$, 16], [0, { [_hH]: _ET }]]
];
var GetPublicKeyRequest$ = [3, n0, _GPKR,
    0,
    [_Id],
    [[0, 1]]
];
var GetPublicKeyResult$ = [3, n0, _GPKRe,
    0,
    [_PK, _ET],
    [[() => PublicKey$, 16], [0, { [_hH]: _ET }]]
];
var GetRealtimeLogConfigRequest$ = [3, n0, _GRLCR,
    0,
    [_N, _ARN],
    [0, 0]
];
var GetRealtimeLogConfigResult$ = [3, n0, _GRLCRe,
    0,
    [_RLC],
    [[() => RealtimeLogConfig$, 0]]
];
var GetResourcePolicyRequest$ = [3, n0, _GRPR,
    0,
    [_RA],
    [0]
];
var GetResourcePolicyResult$ = [3, n0, _GRPRe,
    0,
    [_RA, _PD],
    [0, 0]
];
var GetResponseHeadersPolicyConfigRequest$ = [3, n0, _GRHPCR,
    0,
    [_Id],
    [[0, 1]]
];
var GetResponseHeadersPolicyConfigResult$ = [3, n0, _GRHPCRe,
    0,
    [_RHPC, _ET],
    [[() => ResponseHeadersPolicyConfig$, 16], [0, { [_hH]: _ET }]]
];
var GetResponseHeadersPolicyRequest$ = [3, n0, _GRHPR,
    0,
    [_Id],
    [[0, 1]]
];
var GetResponseHeadersPolicyResult$ = [3, n0, _GRHPRe,
    0,
    [_RHP, _ET],
    [[() => ResponseHeadersPolicy$, 16], [0, { [_hH]: _ET }]]
];
var GetStreamingDistributionConfigRequest$ = [3, n0, _GSDCR,
    0,
    [_Id],
    [[0, 1]]
];
var GetStreamingDistributionConfigResult$ = [3, n0, _GSDCRe,
    0,
    [_SDC, _ET],
    [[() => StreamingDistributionConfig$, 16], [0, { [_hH]: _ET }]]
];
var GetStreamingDistributionRequest$ = [3, n0, _GSDR,
    0,
    [_Id],
    [[0, 1]]
];
var GetStreamingDistributionResult$ = [3, n0, _GSDRe,
    0,
    [_SD, _ET],
    [[() => StreamingDistribution$, 16], [0, { [_hH]: _ET }]]
];
var GetTrustStoreRequest$ = [3, n0, _GTSR,
    0,
    [_Ide],
    [[0, 1]]
];
var GetTrustStoreResult$ = [3, n0, _GTSRe,
    0,
    [_TSr, _ET],
    [[() => TrustStore$, 16], [0, { [_hH]: _ET }]]
];
var GetVpcOriginRequest$ = [3, n0, _GVOR,
    0,
    [_Id],
    [[0, 1]]
];
var GetVpcOriginResult$ = [3, n0, _GVORe,
    0,
    [_VO, _ET],
    [[() => VpcOrigin$, 16], [0, { [_hH]: _ET }]]
];
var GrpcConfig$ = [3, n0, _GC,
    0,
    [_E],
    [2]
];
var Headers$ = [3, n0, _H,
    0,
    [_Q, _I],
    [1, [() => HeaderList, 0]]
];
var IllegalDelete$ = [-3, n0, _IDl,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(IllegalDelete$, IllegalDelete);
var IllegalFieldLevelEncryptionConfigAssociationWithCacheBehavior$ = [-3, n0, _IFLECAWCB,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(IllegalFieldLevelEncryptionConfigAssociationWithCacheBehavior$, IllegalFieldLevelEncryptionConfigAssociationWithCacheBehavior);
var IllegalOriginAccessConfiguration$ = [-3, n0, _IOAC,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(IllegalOriginAccessConfiguration$, IllegalOriginAccessConfiguration);
var IllegalUpdate$ = [-3, n0, _IU,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(IllegalUpdate$, IllegalUpdate);
var ImportSource$ = [3, n0, _IS,
    0,
    [_STo, _SARN],
    [0, 0]
];
var InconsistentQuantities$ = [-3, n0, _IQ,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InconsistentQuantities$, InconsistentQuantities);
var InvalidArgument$ = [-3, n0, _IA,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidArgument$, InvalidArgument);
var InvalidAssociation$ = [-3, n0, _IAn,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidAssociation$, InvalidAssociation);
var Invalidation$ = [3, n0, _In,
    0,
    [_Id, _S, _CTre, _IB],
    [0, 0, 4, [() => InvalidationBatch$, 0]]
];
var InvalidationBatch$ = [3, n0, _IB,
    0,
    [_Pa, _CR],
    [[() => Paths$, 0], 0]
];
var InvalidationList$ = [3, n0, _IL,
    0,
    [_Ma, _NM, _MI, _IT, _Q, _I],
    [0, 0, 1, 2, 1, [() => InvalidationSummaryList, 0]]
];
var InvalidationSummary$ = [3, n0, _ISn,
    0,
    [_Id, _CTre, _S],
    [0, 4, 0]
];
var InvalidDefaultRootObject$ = [-3, n0, _IDRO,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidDefaultRootObject$, InvalidDefaultRootObject);
var InvalidDomainNameForOriginAccessControl$ = [-3, n0, _IDNFOAC,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidDomainNameForOriginAccessControl$, InvalidDomainNameForOriginAccessControl);
var InvalidErrorCode$ = [-3, n0, _IEC,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidErrorCode$, InvalidErrorCode);
var InvalidForwardCookies$ = [-3, n0, _IFC,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidForwardCookies$, InvalidForwardCookies);
var InvalidFunctionAssociation$ = [-3, n0, _IFA,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidFunctionAssociation$, InvalidFunctionAssociation);
var InvalidGeoRestrictionParameter$ = [-3, n0, _IGRP,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidGeoRestrictionParameter$, InvalidGeoRestrictionParameter);
var InvalidHeadersForS3Origin$ = [-3, n0, _IHFSO,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidHeadersForS3Origin$, InvalidHeadersForS3Origin);
var InvalidIfMatchVersion$ = [-3, n0, _IIMV,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidIfMatchVersion$, InvalidIfMatchVersion);
var InvalidLambdaFunctionAssociation$ = [-3, n0, _ILFA,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidLambdaFunctionAssociation$, InvalidLambdaFunctionAssociation);
var InvalidLocationCode$ = [-3, n0, _ILC,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidLocationCode$, InvalidLocationCode);
var InvalidMinimumProtocolVersion$ = [-3, n0, _IMPV,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidMinimumProtocolVersion$, InvalidMinimumProtocolVersion);
var InvalidOrigin$ = [-3, n0, _IO,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidOrigin$, InvalidOrigin);
var InvalidOriginAccessControl$ = [-3, n0, _IOACn,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidOriginAccessControl$, InvalidOriginAccessControl);
var InvalidOriginAccessIdentity$ = [-3, n0, _IOAI,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidOriginAccessIdentity$, InvalidOriginAccessIdentity);
var InvalidOriginKeepaliveTimeout$ = [-3, n0, _IOKT,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidOriginKeepaliveTimeout$, InvalidOriginKeepaliveTimeout);
var InvalidOriginReadTimeout$ = [-3, n0, _IORT,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidOriginReadTimeout$, InvalidOriginReadTimeout);
var InvalidProtocolSettings$ = [-3, n0, _IPS,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidProtocolSettings$, InvalidProtocolSettings);
var InvalidQueryStringParameters$ = [-3, n0, _IQSP,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidQueryStringParameters$, InvalidQueryStringParameters);
var InvalidRelativePath$ = [-3, n0, _IRP,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidRelativePath$, InvalidRelativePath);
var InvalidRequiredProtocol$ = [-3, n0, _IRPn,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidRequiredProtocol$, InvalidRequiredProtocol);
var InvalidResponseCode$ = [-3, n0, _IRC,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidResponseCode$, InvalidResponseCode);
var InvalidTagging$ = [-3, n0, _ITn,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidTagging$, InvalidTagging);
var InvalidTTLOrder$ = [-3, n0, _ITTLO,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidTTLOrder$, InvalidTTLOrder);
var InvalidViewerCertificate$ = [-3, n0, _IVC,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidViewerCertificate$, InvalidViewerCertificate);
var InvalidWebACLId$ = [-3, n0, _IWACLI,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(InvalidWebACLId$, InvalidWebACLId);
var IpamCidrConfig$ = [3, n0, _ICCp,
    0,
    [_Ci, _IPA, _AIn, _S],
    [0, 0, 0, 0]
];
var IpamConfig$ = [3, n0, _IC,
    0,
    [_Q, _ICC],
    [1, [() => IpamCidrConfigList, 0]]
];
var KeyGroup$ = [3, n0, _KG,
    0,
    [_Id, _LMT, _KGC],
    [0, 4, [() => KeyGroupConfig$, 0]]
];
var KeyGroupAlreadyExists$ = [-3, n0, _KGAE,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(KeyGroupAlreadyExists$, KeyGroupAlreadyExists);
var KeyGroupConfig$ = [3, n0, _KGC,
    0,
    [_N, _I, _Co],
    [0, [() => PublicKeyIdList, 0], 0]
];
var KeyGroupList$ = [3, n0, _KGL,
    0,
    [_NM, _MI, _Q, _I],
    [0, 1, 1, [() => KeyGroupSummaryList, 0]]
];
var KeyGroupSummary$ = [3, n0, _KGS,
    0,
    [_KG],
    [[() => KeyGroup$, 0]]
];
var KeyPairIds$ = [3, n0, _KPI,
    0,
    [_Q, _I],
    [1, [() => KeyPairIdList, 0]]
];
var KeyValueStore$ = [3, n0, _KVS,
    0,
    [_N, _Id, _Co, _ARN, _S, _LMT],
    [0, 0, 0, 0, 0, 4]
];
var KeyValueStoreAssociation$ = [3, n0, _KVSAe,
    0,
    [_KVSARN],
    [0]
];
var KeyValueStoreAssociations$ = [3, n0, _KVSA,
    0,
    [_Q, _I],
    [1, [() => KeyValueStoreAssociationList, 0]]
];
var KeyValueStoreList$ = [3, n0, _KVSL,
    0,
    [_NM, _MI, _Q, _I],
    [0, 1, 1, [() => KeyValueStoreSummaryList, 0]]
];
var KGKeyPairIds$ = [3, n0, _KGKPI,
    0,
    [_KGI, _KPI],
    [0, [() => KeyPairIds$, 0]]
];
var KinesisStreamConfig$ = [3, n0, _KSC,
    0,
    [_RARN, _SARNt],
    [0, 0]
];
var LambdaFunctionAssociation$ = [3, n0, _LFAa,
    0,
    [_LFARN, _ETv, _IBn],
    [0, 0, 2]
];
var LambdaFunctionAssociations$ = [3, n0, _LFA,
    0,
    [_Q, _I],
    [1, [() => LambdaFunctionAssociationList, 0]]
];
var ListAnycastIpListsRequest$ = [3, n0, _LAILR,
    0,
    [_Ma, _MI],
    [[0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }]]
];
var ListAnycastIpListsResult$ = [3, n0, _LAILRi,
    0,
    [_AILn],
    [[() => AnycastIpListCollection$, { [_hP]: 1, [_xN]: _AILC }]]
];
var ListCachePoliciesRequest$ = [3, n0, _LCPR,
    0,
    [_T, _Ma, _MI],
    [[0, { [_hQ]: _T }], [0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }]]
];
var ListCachePoliciesResult$ = [3, n0, _LCPRi,
    0,
    [_CPL],
    [[() => CachePolicyList$, 16]]
];
var ListCloudFrontOriginAccessIdentitiesRequest$ = [3, n0, _LCFOAIR,
    0,
    [_Ma, _MI],
    [[0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }]]
];
var ListCloudFrontOriginAccessIdentitiesResult$ = [3, n0, _LCFOAIRi,
    0,
    [_CFOAIL],
    [[() => CloudFrontOriginAccessIdentityList$, 16]]
];
var ListConflictingAliasesRequest$ = [3, n0, _LCAR,
    0,
    [_DI, _Al, _Ma, _MI],
    [[0, { [_hQ]: _DI }], [0, { [_hQ]: _Al }], [0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }]]
];
var ListConflictingAliasesResult$ = [3, n0, _LCARi,
    0,
    [_CAL],
    [[() => ConflictingAliasesList$, 16]]
];
var ListConnectionFunctionsRequest$ = [3, n0, _LCFR,
    0,
    [_Ma, _MI, _St],
    [0, 1, 0]
];
var ListConnectionFunctionsResult$ = [3, n0, _LCFRi,
    0,
    [_NM, _CF],
    [0, [() => ConnectionFunctionSummaryList, 0]]
];
var ListConnectionGroupsRequest$ = [3, n0, _LCGR,
    0,
    [_AF, _Ma, _MI],
    [() => ConnectionGroupAssociationFilter$, 0, 1]
];
var ListConnectionGroupsResult$ = [3, n0, _LCGRi,
    0,
    [_NM, _CGo],
    [0, [() => ConnectionGroupSummaryList, 0]]
];
var ListContinuousDeploymentPoliciesRequest$ = [3, n0, _LCDPR,
    0,
    [_Ma, _MI],
    [[0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }]]
];
var ListContinuousDeploymentPoliciesResult$ = [3, n0, _LCDPRi,
    0,
    [_CDPL],
    [[() => ContinuousDeploymentPolicyList$, 16]]
];
var ListDistributionsByAnycastIpListIdRequest$ = [3, n0, _LDBAILIR,
    0,
    [_Ma, _MI, _AILI],
    [[0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }], [0, 1]]
];
var ListDistributionsByAnycastIpListIdResult$ = [3, n0, _LDBAILIRi,
    0,
    [_DL],
    [[() => DistributionList$, 16]]
];
var ListDistributionsByCachePolicyIdRequest$ = [3, n0, _LDBCPIR,
    0,
    [_Ma, _MI, _CPI],
    [[0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }], [0, 1]]
];
var ListDistributionsByCachePolicyIdResult$ = [3, n0, _LDBCPIRi,
    0,
    [_DIL],
    [[() => DistributionIdList$, 16]]
];
var ListDistributionsByConnectionFunctionRequest$ = [3, n0, _LDBCFR,
    0,
    [_Ma, _MI, _CFI],
    [[0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }], [0, { [_hQ]: _CFI }]]
];
var ListDistributionsByConnectionFunctionResult$ = [3, n0, _LDBCFRi,
    0,
    [_DL],
    [[() => DistributionList$, 16]]
];
var ListDistributionsByConnectionModeRequest$ = [3, n0, _LDBCMR,
    0,
    [_Ma, _MI, _CMo],
    [[0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }], [0, 1]]
];
var ListDistributionsByConnectionModeResult$ = [3, n0, _LDBCMRi,
    0,
    [_DL],
    [[() => DistributionList$, 16]]
];
var ListDistributionsByKeyGroupRequest$ = [3, n0, _LDBKGR,
    0,
    [_Ma, _MI, _KGI],
    [[0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }], [0, 1]]
];
var ListDistributionsByKeyGroupResult$ = [3, n0, _LDBKGRi,
    0,
    [_DIL],
    [[() => DistributionIdList$, 16]]
];
var ListDistributionsByOriginRequestPolicyIdRequest$ = [3, n0, _LDBORPIR,
    0,
    [_Ma, _MI, _ORPI],
    [[0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }], [0, 1]]
];
var ListDistributionsByOriginRequestPolicyIdResult$ = [3, n0, _LDBORPIRi,
    0,
    [_DIL],
    [[() => DistributionIdList$, 16]]
];
var ListDistributionsByOwnedResourceRequest$ = [3, n0, _LDBORR,
    0,
    [_RA, _Ma, _MI],
    [[0, 1], [0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }]]
];
var ListDistributionsByOwnedResourceResult$ = [3, n0, _LDBORRi,
    0,
    [_DL],
    [[() => DistributionIdOwnerList$, 16]]
];
var ListDistributionsByRealtimeLogConfigRequest$ = [3, n0, _LDBRLCR,
    0,
    [_Ma, _MI, _RLCN, _RLCA],
    [0, 1, 0, 0]
];
var ListDistributionsByRealtimeLogConfigResult$ = [3, n0, _LDBRLCRi,
    0,
    [_DL],
    [[() => DistributionList$, 16]]
];
var ListDistributionsByResponseHeadersPolicyIdRequest$ = [3, n0, _LDBRHPIR,
    0,
    [_Ma, _MI, _RHPI],
    [[0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }], [0, 1]]
];
var ListDistributionsByResponseHeadersPolicyIdResult$ = [3, n0, _LDBRHPIRi,
    0,
    [_DIL],
    [[() => DistributionIdList$, 16]]
];
var ListDistributionsByTrustStoreRequest$ = [3, n0, _LDBTSR,
    0,
    [_TSI, _Ma, _MI],
    [[0, { [_hQ]: _TSI }], [0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }]]
];
var ListDistributionsByTrustStoreResult$ = [3, n0, _LDBTSRi,
    0,
    [_DL],
    [[() => DistributionList$, 16]]
];
var ListDistributionsByVpcOriginIdRequest$ = [3, n0, _LDBVOIR,
    0,
    [_Ma, _MI, _VOI],
    [[0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }], [0, 1]]
];
var ListDistributionsByVpcOriginIdResult$ = [3, n0, _LDBVOIRi,
    0,
    [_DIL],
    [[() => DistributionIdList$, 16]]
];
var ListDistributionsByWebACLIdRequest$ = [3, n0, _LDBWACLIR,
    0,
    [_Ma, _MI, _WACLI],
    [[0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }], [0, 1]]
];
var ListDistributionsByWebACLIdResult$ = [3, n0, _LDBWACLIRi,
    0,
    [_DL],
    [[() => DistributionList$, 16]]
];
var ListDistributionsRequest$ = [3, n0, _LDR,
    0,
    [_Ma, _MI],
    [[0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }]]
];
var ListDistributionsResult$ = [3, n0, _LDRi,
    0,
    [_DL],
    [[() => DistributionList$, 16]]
];
var ListDistributionTenantsByCustomizationRequest$ = [3, n0, _LDTBCR,
    0,
    [_WACLA, _CAe, _Ma, _MI],
    [0, 0, 0, 1]
];
var ListDistributionTenantsByCustomizationResult$ = [3, n0, _LDTBCRi,
    0,
    [_NM, _DTL],
    [0, [() => DistributionTenantList, 0]]
];
var ListDistributionTenantsRequest$ = [3, n0, _LDTR,
    0,
    [_AF, _Ma, _MI],
    [() => DistributionTenantAssociationFilter$, 0, 1]
];
var ListDistributionTenantsResult$ = [3, n0, _LDTRi,
    0,
    [_NM, _DTL],
    [0, [() => DistributionTenantList, 0]]
];
var ListDomainConflictsRequest$ = [3, n0, _LDCR,
    0,
    [_Dom, _DCVR, _MI, _Ma],
    [0, () => DistributionResourceId$, 1, 0]
];
var ListDomainConflictsResult$ = [3, n0, _LDCRi,
    0,
    [_DCom, _NM],
    [[() => DomainConflictsList, 0], 0]
];
var ListFieldLevelEncryptionConfigsRequest$ = [3, n0, _LFLECR,
    0,
    [_Ma, _MI],
    [[0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }]]
];
var ListFieldLevelEncryptionConfigsResult$ = [3, n0, _LFLECRi,
    0,
    [_FLEL],
    [[() => FieldLevelEncryptionList$, 16]]
];
var ListFieldLevelEncryptionProfilesRequest$ = [3, n0, _LFLEPR,
    0,
    [_Ma, _MI],
    [[0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }]]
];
var ListFieldLevelEncryptionProfilesResult$ = [3, n0, _LFLEPRi,
    0,
    [_FLEPL],
    [[() => FieldLevelEncryptionProfileList$, 16]]
];
var ListFunctionsRequest$ = [3, n0, _LFR,
    0,
    [_Ma, _MI, _St],
    [[0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }], [0, { [_hQ]: _St }]]
];
var ListFunctionsResult$ = [3, n0, _LFRi,
    0,
    [_FL],
    [[() => FunctionList$, 16]]
];
var ListInvalidationsForDistributionTenantRequest$ = [3, n0, _LIFDTR,
    0,
    [_Id, _Ma, _MI],
    [[0, 1], [0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }]]
];
var ListInvalidationsForDistributionTenantResult$ = [3, n0, _LIFDTRi,
    0,
    [_IL],
    [[() => InvalidationList$, 16]]
];
var ListInvalidationsRequest$ = [3, n0, _LIR,
    0,
    [_DI, _Ma, _MI],
    [[0, 1], [0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }]]
];
var ListInvalidationsResult$ = [3, n0, _LIRi,
    0,
    [_IL],
    [[() => InvalidationList$, 16]]
];
var ListKeyGroupsRequest$ = [3, n0, _LKGR,
    0,
    [_Ma, _MI],
    [[0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }]]
];
var ListKeyGroupsResult$ = [3, n0, _LKGRi,
    0,
    [_KGL],
    [[() => KeyGroupList$, 16]]
];
var ListKeyValueStoresRequest$ = [3, n0, _LKVSR,
    0,
    [_Ma, _MI, _S],
    [[0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }], [0, { [_hQ]: _S }]]
];
var ListKeyValueStoresResult$ = [3, n0, _LKVSRi,
    0,
    [_KVSL],
    [[() => KeyValueStoreList$, 16]]
];
var ListOriginAccessControlsRequest$ = [3, n0, _LOACR,
    0,
    [_Ma, _MI],
    [[0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }]]
];
var ListOriginAccessControlsResult$ = [3, n0, _LOACRi,
    0,
    [_OACL],
    [[() => OriginAccessControlList$, 16]]
];
var ListOriginRequestPoliciesRequest$ = [3, n0, _LORPR,
    0,
    [_T, _Ma, _MI],
    [[0, { [_hQ]: _T }], [0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }]]
];
var ListOriginRequestPoliciesResult$ = [3, n0, _LORPRi,
    0,
    [_ORPL],
    [[() => OriginRequestPolicyList$, 16]]
];
var ListPublicKeysRequest$ = [3, n0, _LPKR,
    0,
    [_Ma, _MI],
    [[0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }]]
];
var ListPublicKeysResult$ = [3, n0, _LPKRi,
    0,
    [_PKL],
    [[() => PublicKeyList$, 16]]
];
var ListRealtimeLogConfigsRequest$ = [3, n0, _LRLCR,
    0,
    [_MI, _Ma],
    [[1, { [_hQ]: _MI }], [0, { [_hQ]: _Ma }]]
];
var ListRealtimeLogConfigsResult$ = [3, n0, _LRLCRi,
    0,
    [_RLCe],
    [[() => RealtimeLogConfigs$, 16]]
];
var ListResponseHeadersPoliciesRequest$ = [3, n0, _LRHPR,
    0,
    [_T, _Ma, _MI],
    [[0, { [_hQ]: _T }], [0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }]]
];
var ListResponseHeadersPoliciesResult$ = [3, n0, _LRHPRi,
    0,
    [_RHPL],
    [[() => ResponseHeadersPolicyList$, 16]]
];
var ListStreamingDistributionsRequest$ = [3, n0, _LSDR,
    0,
    [_Ma, _MI],
    [[0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }]]
];
var ListStreamingDistributionsResult$ = [3, n0, _LSDRi,
    0,
    [_SDL],
    [[() => StreamingDistributionList$, 16]]
];
var ListTagsForResourceRequest$ = [3, n0, _LTFRR,
    0,
    [_Res],
    [[0, { [_hQ]: _Res }]]
];
var ListTagsForResourceResult$ = [3, n0, _LTFRRi,
    0,
    [_Ta],
    [[() => Tags$, 16]]
];
var ListTrustStoresRequest$ = [3, n0, _LTSR,
    0,
    [_Ma, _MI],
    [0, 1]
];
var ListTrustStoresResult$ = [3, n0, _LTSRi,
    0,
    [_NM, _TSL],
    [0, [() => TrustStoreList, 0]]
];
var ListVpcOriginsRequest$ = [3, n0, _LVOR,
    0,
    [_Ma, _MI],
    [[0, { [_hQ]: _Ma }], [1, { [_hQ]: _MI }]]
];
var ListVpcOriginsResult$ = [3, n0, _LVORi,
    0,
    [_VOL],
    [[() => VpcOriginList$, 16]]
];
var LoggingConfig$ = [3, n0, _LC,
    0,
    [_E, _ICn, _B, _Pr],
    [2, 2, 0, 0]
];
var ManagedCertificateDetails$ = [3, n0, _MCD,
    0,
    [_CAe, _CS, _VTH, _VTD],
    [0, 0, 0, () => ValidationTokenDetailList]
];
var ManagedCertificateRequest$ = [3, n0, _MCR,
    0,
    [_VTH, _PDN, _CTLP],
    [0, 0, 0]
];
var MissingBody$ = [-3, n0, _MB,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(MissingBody$, MissingBody);
var MonitoringSubscription$ = [3, n0, _MS,
    0,
    [_RMSC],
    [() => RealtimeMetricsSubscriptionConfig$]
];
var MonitoringSubscriptionAlreadyExists$ = [-3, n0, _MSAE,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(MonitoringSubscriptionAlreadyExists$, MonitoringSubscriptionAlreadyExists);
var NoSuchCachePolicy$ = [-3, n0, _NSCP,
    { [_e]: _c, [_hE]: 404 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(NoSuchCachePolicy$, NoSuchCachePolicy);
var NoSuchCloudFrontOriginAccessIdentity$ = [-3, n0, _NSCFOAI,
    { [_e]: _c, [_hE]: 404 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(NoSuchCloudFrontOriginAccessIdentity$, NoSuchCloudFrontOriginAccessIdentity);
var NoSuchContinuousDeploymentPolicy$ = [-3, n0, _NSCDP,
    { [_e]: _c, [_hE]: 404 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(NoSuchContinuousDeploymentPolicy$, NoSuchContinuousDeploymentPolicy);
var NoSuchDistribution$ = [-3, n0, _NSD,
    { [_e]: _c, [_hE]: 404 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(NoSuchDistribution$, NoSuchDistribution);
var NoSuchFieldLevelEncryptionConfig$ = [-3, n0, _NSFLEC,
    { [_e]: _c, [_hE]: 404 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(NoSuchFieldLevelEncryptionConfig$, NoSuchFieldLevelEncryptionConfig);
var NoSuchFieldLevelEncryptionProfile$ = [-3, n0, _NSFLEP,
    { [_e]: _c, [_hE]: 404 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(NoSuchFieldLevelEncryptionProfile$, NoSuchFieldLevelEncryptionProfile);
var NoSuchFunctionExists$ = [-3, n0, _NSFE,
    { [_e]: _c, [_hE]: 404 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(NoSuchFunctionExists$, NoSuchFunctionExists);
var NoSuchInvalidation$ = [-3, n0, _NSI,
    { [_e]: _c, [_hE]: 404 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(NoSuchInvalidation$, NoSuchInvalidation);
var NoSuchMonitoringSubscription$ = [-3, n0, _NSMS,
    { [_e]: _c, [_hE]: 404 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(NoSuchMonitoringSubscription$, NoSuchMonitoringSubscription);
var NoSuchOrigin$ = [-3, n0, _NSO,
    { [_e]: _c, [_hE]: 404 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(NoSuchOrigin$, NoSuchOrigin);
var NoSuchOriginAccessControl$ = [-3, n0, _NSOAC,
    { [_e]: _c, [_hE]: 404 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(NoSuchOriginAccessControl$, NoSuchOriginAccessControl);
var NoSuchOriginRequestPolicy$ = [-3, n0, _NSORP,
    { [_e]: _c, [_hE]: 404 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(NoSuchOriginRequestPolicy$, NoSuchOriginRequestPolicy);
var NoSuchPublicKey$ = [-3, n0, _NSPK,
    { [_e]: _c, [_hE]: 404 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(NoSuchPublicKey$, NoSuchPublicKey);
var NoSuchRealtimeLogConfig$ = [-3, n0, _NSRLC,
    { [_e]: _c, [_hE]: 404 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(NoSuchRealtimeLogConfig$, NoSuchRealtimeLogConfig);
var NoSuchResource$ = [-3, n0, _NSR,
    { [_e]: _c, [_hE]: 404 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(NoSuchResource$, NoSuchResource);
var NoSuchResponseHeadersPolicy$ = [-3, n0, _NSRHP,
    { [_e]: _c, [_hE]: 404 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(NoSuchResponseHeadersPolicy$, NoSuchResponseHeadersPolicy);
var NoSuchStreamingDistribution$ = [-3, n0, _NSSD,
    { [_e]: _c, [_hE]: 404 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(NoSuchStreamingDistribution$, NoSuchStreamingDistribution);
var Origin$ = [3, n0, _Or,
    0,
    [_Id, _DN, _OP, _CH, _SOC, _COC, _VOC, _CAo, _CTon, _RCT, _OS, _OACI],
    [0, 0, 0, [() => CustomHeaders$, 0], () => S3OriginConfig$, [() => CustomOriginConfig$, 0], () => VpcOriginConfig$, 1, 1, 1, () => OriginShield$, 0]
];
var OriginAccessControl$ = [3, n0, _OAC,
    0,
    [_Id, _OACC],
    [0, () => OriginAccessControlConfig$]
];
var OriginAccessControlAlreadyExists$ = [-3, n0, _OACAE,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(OriginAccessControlAlreadyExists$, OriginAccessControlAlreadyExists);
var OriginAccessControlConfig$ = [3, n0, _OACC,
    0,
    [_N, _De, _SP, _SB, _OACOT],
    [0, 0, 0, 0, 0]
];
var OriginAccessControlInUse$ = [-3, n0, _OACIU,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(OriginAccessControlInUse$, OriginAccessControlInUse);
var OriginAccessControlList$ = [3, n0, _OACL,
    0,
    [_Ma, _NM, _MI, _IT, _Q, _I],
    [0, 0, 1, 2, 1, [() => OriginAccessControlSummaryList, 0]]
];
var OriginAccessControlSummary$ = [3, n0, _OACS,
    0,
    [_Id, _De, _N, _SP, _SB, _OACOT],
    [0, 0, 0, 0, 0, 0]
];
var OriginCustomHeader$ = [3, n0, _OCH,
    0,
    [_HN, _HVe],
    [0, [() => sensitiveStringType, 0]]
];
var OriginGroup$ = [3, n0, _OGr,
    0,
    [_Id, _FCa, _Me, _SC],
    [0, [() => OriginGroupFailoverCriteria$, 0], [() => OriginGroupMembers$, 0], 0]
];
var OriginGroupFailoverCriteria$ = [3, n0, _OGFC,
    0,
    [_SCt],
    [[() => StatusCodes$, 0]]
];
var OriginGroupMember$ = [3, n0, _OGM,
    0,
    [_OI],
    [0]
];
var OriginGroupMembers$ = [3, n0, _OGMr,
    0,
    [_Q, _I],
    [1, [() => OriginGroupMemberList, 0]]
];
var OriginGroups$ = [3, n0, _OG,
    0,
    [_Q, _I],
    [1, [() => OriginGroupList, 0]]
];
var OriginRequestPolicy$ = [3, n0, _ORP,
    0,
    [_Id, _LMT, _ORPC],
    [0, 4, [() => OriginRequestPolicyConfig$, 0]]
];
var OriginRequestPolicyAlreadyExists$ = [-3, n0, _ORPAE,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(OriginRequestPolicyAlreadyExists$, OriginRequestPolicyAlreadyExists);
var OriginRequestPolicyConfig$ = [3, n0, _ORPC,
    0,
    [_Co, _N, _HC, _CC, _QSC],
    [0, 0, [() => OriginRequestPolicyHeadersConfig$, 0], [() => OriginRequestPolicyCookiesConfig$, 0], [() => OriginRequestPolicyQueryStringsConfig$, 0]]
];
var OriginRequestPolicyCookiesConfig$ = [3, n0, _ORPCC,
    0,
    [_CBo, _Coo],
    [0, [() => CookieNames$, 0]]
];
var OriginRequestPolicyHeadersConfig$ = [3, n0, _ORPHC,
    0,
    [_HB, _H],
    [0, [() => Headers$, 0]]
];
var OriginRequestPolicyInUse$ = [-3, n0, _ORPIU,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(OriginRequestPolicyInUse$, OriginRequestPolicyInUse);
var OriginRequestPolicyList$ = [3, n0, _ORPL,
    0,
    [_NM, _MI, _Q, _I],
    [0, 1, 1, [() => OriginRequestPolicySummaryList, 0]]
];
var OriginRequestPolicyQueryStringsConfig$ = [3, n0, _ORPQSC,
    0,
    [_QSB, _QS],
    [0, [() => QueryStringNames$, 0]]
];
var OriginRequestPolicySummary$ = [3, n0, _ORPS,
    0,
    [_T, _ORP],
    [0, [() => OriginRequestPolicy$, 0]]
];
var Origins$ = [3, n0, _O,
    0,
    [_Q, _I],
    [1, [() => OriginList, 0]]
];
var OriginShield$ = [3, n0, _OS,
    0,
    [_E, _OSR],
    [2, 0]
];
var OriginSslProtocols$ = [3, n0, _OSP,
    0,
    [_Q, _I],
    [1, [() => SslProtocolsList, 0]]
];
var Parameter$ = [3, n0, _Par,
    0,
    [_N, _Va],
    [0, 0]
];
var ParameterDefinition$ = [3, n0, _PDa,
    0,
    [_N, _Def],
    [0, [() => ParameterDefinitionSchema$, 0]]
];
var ParameterDefinitionSchema$ = [3, n0, _PDS,
    0,
    [_SSt],
    [[() => StringSchemaConfig$, 0]]
];
var ParametersInCacheKeyAndForwardedToOrigin$ = [3, n0, _PICKAFTO,
    0,
    [_EAEG, _EAEB, _HC, _CC, _QSC],
    [2, 2, [() => CachePolicyHeadersConfig$, 0], [() => CachePolicyCookiesConfig$, 0], [() => CachePolicyQueryStringsConfig$, 0]]
];
var Paths$ = [3, n0, _Pa,
    0,
    [_Q, _I],
    [1, [() => PathList, 0]]
];
var PreconditionFailed$ = [-3, n0, _PF,
    { [_e]: _c, [_hE]: 412 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(PreconditionFailed$, PreconditionFailed);
var PublicKey$ = [3, n0, _PK,
    0,
    [_Id, _CTr, _PKC],
    [0, 4, () => PublicKeyConfig$]
];
var PublicKeyAlreadyExists$ = [-3, n0, _PKAE,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(PublicKeyAlreadyExists$, PublicKeyAlreadyExists);
var PublicKeyConfig$ = [3, n0, _PKC,
    0,
    [_CR, _N, _EK, _Co],
    [0, 0, 0, 0]
];
var PublicKeyInUse$ = [-3, n0, _PKIU,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(PublicKeyInUse$, PublicKeyInUse);
var PublicKeyList$ = [3, n0, _PKL,
    0,
    [_NM, _MI, _Q, _I],
    [0, 1, 1, [() => PublicKeySummaryList, 0]]
];
var PublicKeySummary$ = [3, n0, _PKS,
    0,
    [_Id, _N, _CTr, _EK, _Co],
    [0, 0, 4, 0, 0]
];
var PublishConnectionFunctionRequest$ = [3, n0, _PCFR,
    0,
    [_Id, _IM],
    [[0, 1], [0, { [_hH]: _IM_ }]]
];
var PublishConnectionFunctionResult$ = [3, n0, _PCFRu,
    0,
    [_CFS],
    [[() => ConnectionFunctionSummary$, 16]]
];
var PublishFunctionRequest$ = [3, n0, _PFR,
    0,
    [_N, _IM],
    [[0, 1], [0, { [_hH]: _IM_ }]]
];
var PublishFunctionResult$ = [3, n0, _PFRu,
    0,
    [_FS],
    [[() => FunctionSummary$, 16]]
];
var PutResourcePolicyRequest$ = [3, n0, _PRPR,
    0,
    [_RA, _PD],
    [0, 0]
];
var PutResourcePolicyResult$ = [3, n0, _PRPRu,
    0,
    [_RA],
    [0]
];
var QueryArgProfile$ = [3, n0, _QAP,
    0,
    [_QA, _PI],
    [0, 0]
];
var QueryArgProfileConfig$ = [3, n0, _QAPC,
    0,
    [_FWQAPIU, _QAPu],
    [2, [() => QueryArgProfiles$, 0]]
];
var QueryArgProfileEmpty$ = [-3, n0, _QAPE,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(QueryArgProfileEmpty$, QueryArgProfileEmpty);
var QueryArgProfiles$ = [3, n0, _QAPu,
    0,
    [_Q, _I],
    [1, [() => QueryArgProfileList, 0]]
];
var QueryStringCacheKeys$ = [3, n0, _QSCK,
    0,
    [_Q, _I],
    [1, [() => QueryStringCacheKeysList, 0]]
];
var QueryStringNames$ = [3, n0, _QSN,
    0,
    [_Q, _I],
    [1, [() => QueryStringNamesList, 0]]
];
var RealtimeLogConfig$ = [3, n0, _RLC,
    0,
    [_ARN, _N, _SR, _EP, _Fi],
    [0, 0, 1, () => EndPointList, [() => FieldList, 0]]
];
var RealtimeLogConfigAlreadyExists$ = [-3, n0, _RLCAE,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(RealtimeLogConfigAlreadyExists$, RealtimeLogConfigAlreadyExists);
var RealtimeLogConfigInUse$ = [-3, n0, _RLCIU,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(RealtimeLogConfigInUse$, RealtimeLogConfigInUse);
var RealtimeLogConfigOwnerMismatch$ = [-3, n0, _RLCOM,
    { [_e]: _c, [_hE]: 401 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(RealtimeLogConfigOwnerMismatch$, RealtimeLogConfigOwnerMismatch);
var RealtimeLogConfigs$ = [3, n0, _RLCe,
    0,
    [_MI, _I, _IT, _Ma, _NM],
    [1, [() => RealtimeLogConfigList, 0], 2, 0, 0]
];
var RealtimeMetricsSubscriptionConfig$ = [3, n0, _RMSC,
    0,
    [_RMSS],
    [0]
];
var ResourceInUse$ = [-3, n0, _RIU,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(ResourceInUse$, ResourceInUse);
var ResourceNotDisabled$ = [-3, n0, _RND,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(ResourceNotDisabled$, ResourceNotDisabled);
var ResponseHeadersPolicy$ = [3, n0, _RHP,
    0,
    [_Id, _LMT, _RHPC],
    [0, 4, [() => ResponseHeadersPolicyConfig$, 0]]
];
var ResponseHeadersPolicyAccessControlAllowHeaders$ = [3, n0, _RHPACAH,
    0,
    [_Q, _I],
    [1, [() => AccessControlAllowHeadersList, 0]]
];
var ResponseHeadersPolicyAccessControlAllowMethods$ = [3, n0, _RHPACAM,
    0,
    [_Q, _I],
    [1, [() => AccessControlAllowMethodsList, 0]]
];
var ResponseHeadersPolicyAccessControlAllowOrigins$ = [3, n0, _RHPACAO,
    0,
    [_Q, _I],
    [1, [() => AccessControlAllowOriginsList, 0]]
];
var ResponseHeadersPolicyAccessControlExposeHeaders$ = [3, n0, _RHPACEH,
    0,
    [_Q, _I],
    [1, [() => AccessControlExposeHeadersList, 0]]
];
var ResponseHeadersPolicyAlreadyExists$ = [-3, n0, _RHPAE,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(ResponseHeadersPolicyAlreadyExists$, ResponseHeadersPolicyAlreadyExists);
var ResponseHeadersPolicyConfig$ = [3, n0, _RHPC,
    0,
    [_Co, _N, _CCo, _SHC, _STHC, _CHC, _RHC],
    [0, 0, [() => ResponseHeadersPolicyCorsConfig$, 0], () => ResponseHeadersPolicySecurityHeadersConfig$, () => ResponseHeadersPolicyServerTimingHeadersConfig$, [() => ResponseHeadersPolicyCustomHeadersConfig$, 0], [() => ResponseHeadersPolicyRemoveHeadersConfig$, 0]]
];
var ResponseHeadersPolicyContentSecurityPolicy$ = [3, n0, _RHPCSP,
    0,
    [_Ov, _CSP],
    [2, 0]
];
var ResponseHeadersPolicyContentTypeOptions$ = [3, n0, _RHPCTO,
    0,
    [_Ov],
    [2]
];
var ResponseHeadersPolicyCorsConfig$ = [3, n0, _RHPCC,
    0,
    [_ACAO, _ACAH, _ACAM, _ACAC, _ACEH, _ACMAS, _OO],
    [[() => ResponseHeadersPolicyAccessControlAllowOrigins$, 0], [() => ResponseHeadersPolicyAccessControlAllowHeaders$, 0], [() => ResponseHeadersPolicyAccessControlAllowMethods$, 0], 2, [() => ResponseHeadersPolicyAccessControlExposeHeaders$, 0], 1, 2]
];
var ResponseHeadersPolicyCustomHeader$ = [3, n0, _RHPCH,
    0,
    [_He, _Va, _Ov],
    [0, 0, 2]
];
var ResponseHeadersPolicyCustomHeadersConfig$ = [3, n0, _RHPCHC,
    0,
    [_Q, _I],
    [1, [() => ResponseHeadersPolicyCustomHeaderList, 0]]
];
var ResponseHeadersPolicyFrameOptions$ = [3, n0, _RHPFO,
    0,
    [_Ov, _FO],
    [2, 0]
];
var ResponseHeadersPolicyInUse$ = [-3, n0, _RHPIU,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(ResponseHeadersPolicyInUse$, ResponseHeadersPolicyInUse);
var ResponseHeadersPolicyList$ = [3, n0, _RHPL,
    0,
    [_NM, _MI, _Q, _I],
    [0, 1, 1, [() => ResponseHeadersPolicySummaryList, 0]]
];
var ResponseHeadersPolicyReferrerPolicy$ = [3, n0, _RHPRP,
    0,
    [_Ov, _RP],
    [2, 0]
];
var ResponseHeadersPolicyRemoveHeader$ = [3, n0, _RHPRH,
    0,
    [_He],
    [0]
];
var ResponseHeadersPolicyRemoveHeadersConfig$ = [3, n0, _RHPRHC,
    0,
    [_Q, _I],
    [1, [() => ResponseHeadersPolicyRemoveHeaderList, 0]]
];
var ResponseHeadersPolicySecurityHeadersConfig$ = [3, n0, _RHPSHC,
    0,
    [_XSSP, _FOr, _RP, _CSP, _CTO, _STS],
    [() => ResponseHeadersPolicyXSSProtection$, () => ResponseHeadersPolicyFrameOptions$, () => ResponseHeadersPolicyReferrerPolicy$, () => ResponseHeadersPolicyContentSecurityPolicy$, () => ResponseHeadersPolicyContentTypeOptions$, () => ResponseHeadersPolicyStrictTransportSecurity$]
];
var ResponseHeadersPolicyServerTimingHeadersConfig$ = [3, n0, _RHPSTHC,
    0,
    [_E, _SR],
    [2, 1]
];
var ResponseHeadersPolicyStrictTransportSecurity$ = [3, n0, _RHPSTS,
    0,
    [_Ov, _ISnc, _Pre, _ACMAS],
    [2, 2, 2, 1]
];
var ResponseHeadersPolicySummary$ = [3, n0, _RHPS,
    0,
    [_T, _RHP],
    [0, [() => ResponseHeadersPolicy$, 0]]
];
var ResponseHeadersPolicyXSSProtection$ = [3, n0, _RHPXSSP,
    0,
    [_Ov, _Pro, _MBo, _RU],
    [2, 2, 2, 0]
];
var Restrictions$ = [3, n0, _Re,
    0,
    [_GRe],
    [[() => GeoRestriction$, 0]]
];
var S3Origin$ = [3, n0, _SO,
    0,
    [_DN, _OAIr],
    [0, 0]
];
var S3OriginConfig$ = [3, n0, _SOC,
    0,
    [_OAIr, _ORT],
    [0, 1]
];
var SessionStickinessConfig$ = [3, n0, _SSC,
    0,
    [_ITTL, _MTTLax],
    [1, 1]
];
var Signer$ = [3, n0, _Si,
    0,
    [_AAN, _KPI],
    [0, [() => KeyPairIds$, 0]]
];
var StagingDistributionDnsNames$ = [3, n0, _SDDN,
    0,
    [_Q, _I],
    [1, [() => StagingDistributionDnsNameList, 0]]
];
var StagingDistributionInUse$ = [-3, n0, _SDIU,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(StagingDistributionInUse$, StagingDistributionInUse);
var StatusCodes$ = [3, n0, _SCt,
    0,
    [_Q, _I],
    [1, [() => StatusCodeList, 0]]
];
var StreamingDistribution$ = [3, n0, _SD,
    0,
    [_Id, _ARN, _S, _LMT, _DN, _ATS, _SDC],
    [0, 0, 0, 4, 0, [() => ActiveTrustedSigners$, 0], [() => StreamingDistributionConfig$, 0]]
];
var StreamingDistributionAlreadyExists$ = [-3, n0, _SDAE,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(StreamingDistributionAlreadyExists$, StreamingDistributionAlreadyExists);
var StreamingDistributionConfig$ = [3, n0, _SDC,
    0,
    [_CR, _SO, _A, _Co, _Lo, _TS, _PC, _E],
    [0, () => S3Origin$, [() => Aliases$, 0], 0, () => StreamingLoggingConfig$, [() => TrustedSigners$, 0], 0, 2]
];
var StreamingDistributionConfigWithTags$ = [3, n0, _SDCWT,
    0,
    [_SDC, _Ta],
    [[() => StreamingDistributionConfig$, 0], [() => Tags$, 0]]
];
var StreamingDistributionList$ = [3, n0, _SDL,
    0,
    [_Ma, _NM, _MI, _IT, _Q, _I],
    [0, 0, 1, 2, 1, [() => StreamingDistributionSummaryList, 0]]
];
var StreamingDistributionNotDisabled$ = [-3, n0, _SDND,
    { [_e]: _c, [_hE]: 409 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(StreamingDistributionNotDisabled$, StreamingDistributionNotDisabled);
var StreamingDistributionSummary$ = [3, n0, _SDS,
    0,
    [_Id, _ARN, _S, _LMT, _DN, _SO, _A, _TS, _Co, _PC, _E],
    [0, 0, 0, 4, 0, () => S3Origin$, [() => Aliases$, 0], [() => TrustedSigners$, 0], 0, 0, 2]
];
var StreamingLoggingConfig$ = [3, n0, _SLC,
    0,
    [_E, _B, _Pr],
    [2, 0, 0]
];
var StringSchemaConfig$ = [3, n0, _SSCt,
    0,
    [_Co, _DV, _Req],
    [[() => sensitiveStringType, 0], 0, 2]
];
var Tag$ = [3, n0, _Tag,
    0,
    [_K, _Va],
    [0, 0]
];
var TagKeys$ = [3, n0, _TK,
    0,
    [_I],
    [[() => TagKeyList, 0]]
];
var TagResourceRequest$ = [3, n0, _TRR,
    0,
    [_Res, _Ta],
    [[0, { [_hQ]: _Res }], [() => Tags$, { [_hP]: 1, [_xN]: _Ta }]]
];
var Tags$ = [3, n0, _Ta,
    0,
    [_I],
    [[() => TagList, 0]]
];
var TenantConfig$ = [3, n0, _TCe,
    0,
    [_PDar],
    [[() => ParameterDefinitions, 0]]
];
var TestConnectionFunctionRequest$ = [3, n0, _TCFR,
    0,
    [_Id, _IM, _St, _CO],
    [[0, 1], [0, { [_hH]: _IM_ }], 0, [() => FunctionEventObject, 0]]
];
var TestConnectionFunctionResult$ = [3, n0, _TCFRe,
    0,
    [_CFTR],
    [[() => ConnectionFunctionTestResult$, 16]]
];
var TestFunctionFailed$ = [-3, n0, _TFF,
    { [_e]: _s, [_hE]: 500 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TestFunctionFailed$, TestFunctionFailed);
var TestFunctionRequest$ = [3, n0, _TFR,
    0,
    [_N, _IM, _St, _EO],
    [[0, 1], [0, { [_hH]: _IM_ }], 0, [() => FunctionEventObject, 0]]
];
var TestFunctionResult$ = [3, n0, _TFRe,
    0,
    [_TR],
    [[() => TestResult$, 16]]
];
var TestResult$ = [3, n0, _TR,
    0,
    [_FS, _CU, _FEL, _FEM, _FOu],
    [[() => FunctionSummary$, 0], 0, [() => FunctionExecutionLogList, 0], [() => sensitiveStringType, 0], [() => sensitiveStringType, 0]]
];
var TooLongCSPInResponseHeadersPolicy$ = [-3, n0, _TLCSPIRHP,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooLongCSPInResponseHeadersPolicy$, TooLongCSPInResponseHeadersPolicy);
var TooManyCacheBehaviors$ = [-3, n0, _TMCB,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyCacheBehaviors$, TooManyCacheBehaviors);
var TooManyCachePolicies$ = [-3, n0, _TMCP,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyCachePolicies$, TooManyCachePolicies);
var TooManyCertificates$ = [-3, n0, _TMC,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyCertificates$, TooManyCertificates);
var TooManyCloudFrontOriginAccessIdentities$ = [-3, n0, _TMCFOAI,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyCloudFrontOriginAccessIdentities$, TooManyCloudFrontOriginAccessIdentities);
var TooManyContinuousDeploymentPolicies$ = [-3, n0, _TMCDP,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyContinuousDeploymentPolicies$, TooManyContinuousDeploymentPolicies);
var TooManyCookieNamesInWhiteList$ = [-3, n0, _TMCNIWL,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyCookieNamesInWhiteList$, TooManyCookieNamesInWhiteList);
var TooManyCookiesInCachePolicy$ = [-3, n0, _TMCICP,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyCookiesInCachePolicy$, TooManyCookiesInCachePolicy);
var TooManyCookiesInOriginRequestPolicy$ = [-3, n0, _TMCIORP,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyCookiesInOriginRequestPolicy$, TooManyCookiesInOriginRequestPolicy);
var TooManyCustomHeadersInResponseHeadersPolicy$ = [-3, n0, _TMCHIRHP,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyCustomHeadersInResponseHeadersPolicy$, TooManyCustomHeadersInResponseHeadersPolicy);
var TooManyDistributionCNAMEs$ = [-3, n0, _TMDCNAME,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyDistributionCNAMEs$, TooManyDistributionCNAMEs);
var TooManyDistributions$ = [-3, n0, _TMD,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyDistributions$, TooManyDistributions);
var TooManyDistributionsAssociatedToCachePolicy$ = [-3, n0, _TMDATCP,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyDistributionsAssociatedToCachePolicy$, TooManyDistributionsAssociatedToCachePolicy);
var TooManyDistributionsAssociatedToFieldLevelEncryptionConfig$ = [-3, n0, _TMDATFLEC,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyDistributionsAssociatedToFieldLevelEncryptionConfig$, TooManyDistributionsAssociatedToFieldLevelEncryptionConfig);
var TooManyDistributionsAssociatedToKeyGroup$ = [-3, n0, _TMDATKG,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyDistributionsAssociatedToKeyGroup$, TooManyDistributionsAssociatedToKeyGroup);
var TooManyDistributionsAssociatedToOriginAccessControl$ = [-3, n0, _TMDATOAC,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyDistributionsAssociatedToOriginAccessControl$, TooManyDistributionsAssociatedToOriginAccessControl);
var TooManyDistributionsAssociatedToOriginRequestPolicy$ = [-3, n0, _TMDATORP,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyDistributionsAssociatedToOriginRequestPolicy$, TooManyDistributionsAssociatedToOriginRequestPolicy);
var TooManyDistributionsAssociatedToResponseHeadersPolicy$ = [-3, n0, _TMDATRHP,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyDistributionsAssociatedToResponseHeadersPolicy$, TooManyDistributionsAssociatedToResponseHeadersPolicy);
var TooManyDistributionsWithFunctionAssociations$ = [-3, n0, _TMDWFA,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyDistributionsWithFunctionAssociations$, TooManyDistributionsWithFunctionAssociations);
var TooManyDistributionsWithLambdaAssociations$ = [-3, n0, _TMDWLA,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyDistributionsWithLambdaAssociations$, TooManyDistributionsWithLambdaAssociations);
var TooManyDistributionsWithSingleFunctionARN$ = [-3, n0, _TMDWSFARN,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyDistributionsWithSingleFunctionARN$, TooManyDistributionsWithSingleFunctionARN);
var TooManyFieldLevelEncryptionConfigs$ = [-3, n0, _TMFLEC,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyFieldLevelEncryptionConfigs$, TooManyFieldLevelEncryptionConfigs);
var TooManyFieldLevelEncryptionContentTypeProfiles$ = [-3, n0, _TMFLECTP,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyFieldLevelEncryptionContentTypeProfiles$, TooManyFieldLevelEncryptionContentTypeProfiles);
var TooManyFieldLevelEncryptionEncryptionEntities$ = [-3, n0, _TMFLEEE,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyFieldLevelEncryptionEncryptionEntities$, TooManyFieldLevelEncryptionEncryptionEntities);
var TooManyFieldLevelEncryptionFieldPatterns$ = [-3, n0, _TMFLEFP,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyFieldLevelEncryptionFieldPatterns$, TooManyFieldLevelEncryptionFieldPatterns);
var TooManyFieldLevelEncryptionProfiles$ = [-3, n0, _TMFLEP,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyFieldLevelEncryptionProfiles$, TooManyFieldLevelEncryptionProfiles);
var TooManyFieldLevelEncryptionQueryArgProfiles$ = [-3, n0, _TMFLEQAP,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyFieldLevelEncryptionQueryArgProfiles$, TooManyFieldLevelEncryptionQueryArgProfiles);
var TooManyFunctionAssociations$ = [-3, n0, _TMFA,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyFunctionAssociations$, TooManyFunctionAssociations);
var TooManyFunctions$ = [-3, n0, _TMF,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyFunctions$, TooManyFunctions);
var TooManyHeadersInCachePolicy$ = [-3, n0, _TMHICP,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyHeadersInCachePolicy$, TooManyHeadersInCachePolicy);
var TooManyHeadersInForwardedValues$ = [-3, n0, _TMHIFV,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyHeadersInForwardedValues$, TooManyHeadersInForwardedValues);
var TooManyHeadersInOriginRequestPolicy$ = [-3, n0, _TMHIORP,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyHeadersInOriginRequestPolicy$, TooManyHeadersInOriginRequestPolicy);
var TooManyInvalidationsInProgress$ = [-3, n0, _TMIIP,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyInvalidationsInProgress$, TooManyInvalidationsInProgress);
var TooManyKeyGroups$ = [-3, n0, _TMKG,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyKeyGroups$, TooManyKeyGroups);
var TooManyKeyGroupsAssociatedToDistribution$ = [-3, n0, _TMKGATD,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyKeyGroupsAssociatedToDistribution$, TooManyKeyGroupsAssociatedToDistribution);
var TooManyLambdaFunctionAssociations$ = [-3, n0, _TMLFA,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyLambdaFunctionAssociations$, TooManyLambdaFunctionAssociations);
var TooManyOriginAccessControls$ = [-3, n0, _TMOAC,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyOriginAccessControls$, TooManyOriginAccessControls);
var TooManyOriginCustomHeaders$ = [-3, n0, _TMOCH,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyOriginCustomHeaders$, TooManyOriginCustomHeaders);
var TooManyOriginGroupsPerDistribution$ = [-3, n0, _TMOGPD,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyOriginGroupsPerDistribution$, TooManyOriginGroupsPerDistribution);
var TooManyOriginRequestPolicies$ = [-3, n0, _TMORP,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyOriginRequestPolicies$, TooManyOriginRequestPolicies);
var TooManyOrigins$ = [-3, n0, _TMO,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyOrigins$, TooManyOrigins);
var TooManyPublicKeys$ = [-3, n0, _TMPK,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyPublicKeys$, TooManyPublicKeys);
var TooManyPublicKeysInKeyGroup$ = [-3, n0, _TMPKIKG,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyPublicKeysInKeyGroup$, TooManyPublicKeysInKeyGroup);
var TooManyQueryStringParameters$ = [-3, n0, _TMQSP,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyQueryStringParameters$, TooManyQueryStringParameters);
var TooManyQueryStringsInCachePolicy$ = [-3, n0, _TMQSICP,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyQueryStringsInCachePolicy$, TooManyQueryStringsInCachePolicy);
var TooManyQueryStringsInOriginRequestPolicy$ = [-3, n0, _TMQSIORP,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyQueryStringsInOriginRequestPolicy$, TooManyQueryStringsInOriginRequestPolicy);
var TooManyRealtimeLogConfigs$ = [-3, n0, _TMRLC,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyRealtimeLogConfigs$, TooManyRealtimeLogConfigs);
var TooManyRemoveHeadersInResponseHeadersPolicy$ = [-3, n0, _TMRHIRHP,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyRemoveHeadersInResponseHeadersPolicy$, TooManyRemoveHeadersInResponseHeadersPolicy);
var TooManyResponseHeadersPolicies$ = [-3, n0, _TMRHP,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyResponseHeadersPolicies$, TooManyResponseHeadersPolicies);
var TooManyStreamingDistributionCNAMEs$ = [-3, n0, _TMSDCNAME,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyStreamingDistributionCNAMEs$, TooManyStreamingDistributionCNAMEs);
var TooManyStreamingDistributions$ = [-3, n0, _TMSD,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyStreamingDistributions$, TooManyStreamingDistributions);
var TooManyTrustedSigners$ = [-3, n0, _TMTS,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TooManyTrustedSigners$, TooManyTrustedSigners);
var TrafficConfig$ = [3, n0, _TC,
    0,
    [_SWC, _SHCi, _T],
    [() => ContinuousDeploymentSingleWeightConfig$, () => ContinuousDeploymentSingleHeaderConfig$, 0]
];
var TrustedKeyGroupDoesNotExist$ = [-3, n0, _TKGDNE,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TrustedKeyGroupDoesNotExist$, TrustedKeyGroupDoesNotExist);
var TrustedKeyGroups$ = [3, n0, _TKG,
    0,
    [_E, _Q, _I],
    [2, 1, [() => TrustedKeyGroupIdList, 0]]
];
var TrustedSignerDoesNotExist$ = [-3, n0, _TSDNE,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(TrustedSignerDoesNotExist$, TrustedSignerDoesNotExist);
var TrustedSigners$ = [3, n0, _TS,
    0,
    [_E, _Q, _I],
    [2, 1, [() => AwsAccountNumberList, 0]]
];
var TrustStore$ = [3, n0, _TSr,
    0,
    [_Id, _Ar, _N, _S, _NOCC, _LMT, _Rea],
    [0, 0, 0, 0, 1, 4, 0]
];
var TrustStoreConfig$ = [3, n0, _TSC,
    0,
    [_TSIr, _ATSCN, _ICE],
    [0, 2, 2]
];
var TrustStoreSummary$ = [3, n0, _TSS,
    0,
    [_Id, _Ar, _N, _S, _NOCC, _LMT, _Rea, _ET],
    [0, 0, 0, 0, 1, 4, 0, 0]
];
var UnsupportedOperation$ = [-3, n0, _UO,
    { [_e]: _c, [_hE]: 400 },
    [_M],
    [0]
];
schema.TypeRegistry.for(n0).registerError(UnsupportedOperation$, UnsupportedOperation);
var UntagResourceRequest$ = [3, n0, _URR,
    0,
    [_Res, _TK],
    [[0, { [_hQ]: _Res }], [() => TagKeys$, { [_hP]: 1, [_xN]: _TK }]]
];
var UpdateAnycastIpListRequest$ = [3, n0, _UAILR,
    0,
    [_Id, _IAT, _IM],
    [[0, 1], 0, [0, { [_hH]: _IM_ }]]
];
var UpdateAnycastIpListResult$ = [3, n0, _UAILRp,
    0,
    [_AIL, _ET],
    [[() => AnycastIpList$, 16], [0, { [_hH]: _ET }]]
];
var UpdateCachePolicyRequest$ = [3, n0, _UCPR,
    0,
    [_CPC, _Id, _IM],
    [[() => CachePolicyConfig$, { [_hP]: 1, [_xN]: _CPC }], [0, 1], [0, { [_hH]: _IM_ }]]
];
var UpdateCachePolicyResult$ = [3, n0, _UCPRp,
    0,
    [_CP, _ET],
    [[() => CachePolicy$, 16], [0, { [_hH]: _ET }]]
];
var UpdateCloudFrontOriginAccessIdentityRequest$ = [3, n0, _UCFOAIR,
    0,
    [_CFOAIC, _Id, _IM],
    [[() => CloudFrontOriginAccessIdentityConfig$, { [_hP]: 1, [_xN]: _CFOAIC }], [0, 1], [0, { [_hH]: _IM_ }]]
];
var UpdateCloudFrontOriginAccessIdentityResult$ = [3, n0, _UCFOAIRp,
    0,
    [_CFOAI, _ET],
    [[() => CloudFrontOriginAccessIdentity$, 16], [0, { [_hH]: _ET }]]
];
var UpdateConnectionFunctionRequest$ = [3, n0, _UCFR,
    0,
    [_Id, _IM, _CFC, _CFCo],
    [[0, 1], [0, { [_hH]: _IM_ }], [() => FunctionConfig$, 0], [() => FunctionBlob, 0]]
];
var UpdateConnectionFunctionResult$ = [3, n0, _UCFRp,
    0,
    [_CFS, _ET],
    [[() => ConnectionFunctionSummary$, 16], [0, { [_hH]: _ET }]]
];
var UpdateConnectionGroupRequest$ = [3, n0, _UCGR,
    0,
    [_Id, _IE, _IM, _AILI, _E],
    [[0, 1], 2, [0, { [_hH]: _IM_ }], 0, 2]
];
var UpdateConnectionGroupResult$ = [3, n0, _UCGRp,
    0,
    [_CG, _ET],
    [[() => ConnectionGroup$, 16], [0, { [_hH]: _ET }]]
];
var UpdateContinuousDeploymentPolicyRequest$ = [3, n0, _UCDPR,
    0,
    [_CDPC, _Id, _IM],
    [[() => ContinuousDeploymentPolicyConfig$, { [_hP]: 1, [_xN]: _CDPC }], [0, 1], [0, { [_hH]: _IM_ }]]
];
var UpdateContinuousDeploymentPolicyResult$ = [3, n0, _UCDPRp,
    0,
    [_CDP, _ET],
    [[() => ContinuousDeploymentPolicy$, 16], [0, { [_hH]: _ET }]]
];
var UpdateDistributionRequest$ = [3, n0, _UDR,
    0,
    [_DC, _Id, _IM],
    [[() => DistributionConfig$, { [_hP]: 1, [_xN]: _DC }], [0, 1], [0, { [_hH]: _IM_ }]]
];
var UpdateDistributionResult$ = [3, n0, _UDRp,
    0,
    [_D, _ET],
    [[() => Distribution$, 16], [0, { [_hH]: _ET }]]
];
var UpdateDistributionTenantRequest$ = [3, n0, _UDTR,
    0,
    [_Id, _DI, _Do, _Cu, _P, _CGI, _IM, _MCR, _E],
    [[0, 1], 0, () => DomainList, [() => Customizations$, 0], () => _Parameters, 0, [0, { [_hH]: _IM_ }], () => ManagedCertificateRequest$, 2]
];
var UpdateDistributionTenantResult$ = [3, n0, _UDTRp,
    0,
    [_DT, _ET],
    [[() => DistributionTenant$, 16], [0, { [_hH]: _ET }]]
];
var UpdateDistributionWithStagingConfigRequest$ = [3, n0, _UDWSCR,
    0,
    [_Id, _SDI, _IM],
    [[0, 1], [0, { [_hQ]: _SDI }], [0, { [_hH]: _IM_ }]]
];
var UpdateDistributionWithStagingConfigResult$ = [3, n0, _UDWSCRp,
    0,
    [_D, _ET],
    [[() => Distribution$, 16], [0, { [_hH]: _ET }]]
];
var UpdateDomainAssociationRequest$ = [3, n0, _UDAR,
    0,
    [_Dom, _TRa, _IM],
    [0, () => DistributionResourceId$, [0, { [_hH]: _IM_ }]]
];
var UpdateDomainAssociationResult$ = [3, n0, _UDARp,
    0,
    [_Dom, _RI, _ET],
    [0, 0, [0, { [_hH]: _ET }]]
];
var UpdateFieldLevelEncryptionConfigRequest$ = [3, n0, _UFLECR,
    0,
    [_FLEC, _Id, _IM],
    [[() => FieldLevelEncryptionConfig$, { [_hP]: 1, [_xN]: _FLEC }], [0, 1], [0, { [_hH]: _IM_ }]]
];
var UpdateFieldLevelEncryptionConfigResult$ = [3, n0, _UFLECRp,
    0,
    [_FLE, _ET],
    [[() => FieldLevelEncryption$, 16], [0, { [_hH]: _ET }]]
];
var UpdateFieldLevelEncryptionProfileRequest$ = [3, n0, _UFLEPR,
    0,
    [_FLEPC, _Id, _IM],
    [[() => FieldLevelEncryptionProfileConfig$, { [_hP]: 1, [_xN]: _FLEPC }], [0, 1], [0, { [_hH]: _IM_ }]]
];
var UpdateFieldLevelEncryptionProfileResult$ = [3, n0, _UFLEPRp,
    0,
    [_FLEP, _ET],
    [[() => FieldLevelEncryptionProfile$, 16], [0, { [_hH]: _ET }]]
];
var UpdateFunctionRequest$ = [3, n0, _UFR,
    0,
    [_N, _IM, _FC, _FCu],
    [[0, 1], [0, { [_hH]: _IM_ }], [() => FunctionConfig$, 0], [() => FunctionBlob, 0]]
];
var UpdateFunctionResult$ = [3, n0, _UFRp,
    0,
    [_FS, _ET],
    [[() => FunctionSummary$, 16], [0, { [_hH]: _ETt }]]
];
var UpdateKeyGroupRequest$ = [3, n0, _UKGR,
    0,
    [_KGC, _Id, _IM],
    [[() => KeyGroupConfig$, { [_hP]: 1, [_xN]: _KGC }], [0, 1], [0, { [_hH]: _IM_ }]]
];
var UpdateKeyGroupResult$ = [3, n0, _UKGRp,
    0,
    [_KG, _ET],
    [[() => KeyGroup$, 16], [0, { [_hH]: _ET }]]
];
var UpdateKeyValueStoreRequest$ = [3, n0, _UKVSR,
    0,
    [_N, _Co, _IM],
    [[0, 1], 0, [0, { [_hH]: _IM_ }]]
];
var UpdateKeyValueStoreResult$ = [3, n0, _UKVSRp,
    0,
    [_KVS, _ET],
    [[() => KeyValueStore$, 16], [0, { [_hH]: _ET }]]
];
var UpdateOriginAccessControlRequest$ = [3, n0, _UOACR,
    0,
    [_OACC, _Id, _IM],
    [[() => OriginAccessControlConfig$, { [_hP]: 1, [_xN]: _OACC }], [0, 1], [0, { [_hH]: _IM_ }]]
];
var UpdateOriginAccessControlResult$ = [3, n0, _UOACRp,
    0,
    [_OAC, _ET],
    [[() => OriginAccessControl$, 16], [0, { [_hH]: _ET }]]
];
var UpdateOriginRequestPolicyRequest$ = [3, n0, _UORPR,
    0,
    [_ORPC, _Id, _IM],
    [[() => OriginRequestPolicyConfig$, { [_hP]: 1, [_xN]: _ORPC }], [0, 1], [0, { [_hH]: _IM_ }]]
];
var UpdateOriginRequestPolicyResult$ = [3, n0, _UORPRp,
    0,
    [_ORP, _ET],
    [[() => OriginRequestPolicy$, 16], [0, { [_hH]: _ET }]]
];
var UpdatePublicKeyRequest$ = [3, n0, _UPKR,
    0,
    [_PKC, _Id, _IM],
    [[() => PublicKeyConfig$, { [_hP]: 1, [_xN]: _PKC }], [0, 1], [0, { [_hH]: _IM_ }]]
];
var UpdatePublicKeyResult$ = [3, n0, _UPKRp,
    0,
    [_PK, _ET],
    [[() => PublicKey$, 16], [0, { [_hH]: _ET }]]
];
var UpdateRealtimeLogConfigRequest$ = [3, n0, _URLCR,
    0,
    [_EP, _Fi, _N, _ARN, _SR],
    [() => EndPointList, [() => FieldList, 0], 0, 0, 1]
];
var UpdateRealtimeLogConfigResult$ = [3, n0, _URLCRp,
    0,
    [_RLC],
    [[() => RealtimeLogConfig$, 0]]
];
var UpdateResponseHeadersPolicyRequest$ = [3, n0, _URHPR,
    0,
    [_RHPC, _Id, _IM],
    [[() => ResponseHeadersPolicyConfig$, { [_hP]: 1, [_xN]: _RHPC }], [0, 1], [0, { [_hH]: _IM_ }]]
];
var UpdateResponseHeadersPolicyResult$ = [3, n0, _URHPRp,
    0,
    [_RHP, _ET],
    [[() => ResponseHeadersPolicy$, 16], [0, { [_hH]: _ET }]]
];
var UpdateStreamingDistributionRequest$ = [3, n0, _USDR,
    0,
    [_SDC, _Id, _IM],
    [[() => StreamingDistributionConfig$, { [_hP]: 1, [_xN]: _SDC }], [0, 1], [0, { [_hH]: _IM_ }]]
];
var UpdateStreamingDistributionResult$ = [3, n0, _USDRp,
    0,
    [_SD, _ET],
    [[() => StreamingDistribution$, 16], [0, { [_hH]: _ET }]]
];
var UpdateTrustStoreRequest$ = [3, n0, _UTSR,
    0,
    [_Id, _CCBS, _IM],
    [[0, 1], [() => CaCertificatesBundleSource$, 16], [0, { [_hH]: _IM_ }]]
];
var UpdateTrustStoreResult$ = [3, n0, _UTSRp,
    0,
    [_TSr, _ET],
    [[() => TrustStore$, 16], [0, { [_hH]: _ET }]]
];
var UpdateVpcOriginRequest$ = [3, n0, _UVOR,
    0,
    [_VOEC, _Id, _IM],
    [[() => VpcOriginEndpointConfig$, { [_hP]: 1, [_xN]: _VOEC }], [0, 1], [0, { [_hH]: _IM_ }]]
];
var UpdateVpcOriginResult$ = [3, n0, _UVORp,
    0,
    [_VO, _ET],
    [[() => VpcOrigin$, 16], [0, { [_hH]: _ET }]]
];
var ValidationTokenDetail$ = [3, n0, _VTDa,
    0,
    [_Dom, _RTed, _RF],
    [0, 0, 0]
];
var VerifyDnsConfigurationRequest$ = [3, n0, _VDCR,
    0,
    [_Dom, _Ide],
    [0, 0]
];
var VerifyDnsConfigurationResult$ = [3, n0, _VDCRe,
    0,
    [_DCL],
    [[() => DnsConfigurationList, 0]]
];
var ViewerCertificate$ = [3, n0, _VC,
    0,
    [_CFDC, _IAMCI, _ACMCA, _SSLSM, _MPV, _Ce, _CSe],
    [2, 0, 0, 0, 0, 0, 0]
];
var ViewerMtlsConfig$ = [3, n0, _VMC,
    0,
    [_Mo, _TSC],
    [0, () => TrustStoreConfig$]
];
var VpcOrigin$ = [3, n0, _VO,
    0,
    [_Id, _Ar, _AIc, _S, _CTr, _LMT, _VOEC],
    [0, 0, 0, 0, 4, 4, [() => VpcOriginEndpointConfig$, 0]]
];
var VpcOriginConfig$ = [3, n0, _VOC,
    0,
    [_VOI, _OAI, _ORT, _OKT],
    [0, 0, 1, 1]
];
var VpcOriginEndpointConfig$ = [3, n0, _VOEC,
    0,
    [_N, _Ar, _HTTPP, _HTTPSP, _OPP, _OSP],
    [0, 0, 1, 1, 0, [() => OriginSslProtocols$, 0]]
];
var VpcOriginList$ = [3, n0, _VOL,
    0,
    [_Ma, _NM, _MI, _IT, _Q, _I],
    [0, 0, 1, 2, 1, [() => VpcOriginSummaryList, 0]]
];
var VpcOriginSummary$ = [3, n0, _VOS,
    0,
    [_Id, _N, _S, _CTr, _LMT, _Ar, _AIc, _OEA],
    [0, 0, 0, 4, 4, 0, 0, 0]
];
var WebAclCustomization$ = [3, n0, _WAC,
    0,
    [_Ac, _Ar],
    [0, 0]
];
var __Unit = "unit";
var CloudFrontServiceException$ = [-3, _sm, "CloudFrontServiceException", 0, [], []];
schema.TypeRegistry.for(_sm).registerError(CloudFrontServiceException$, CloudFrontServiceException);
var AccessControlAllowHeadersList = [1, n0, _ACAHL,
    0, [0,
        { [_xN]: _He }]
];
var AccessControlAllowMethodsList = [1, n0, _ACAML,
    0, [0,
        { [_xN]: _Met }]
];
var AccessControlAllowOriginsList = [1, n0, _ACAOL,
    0, [0,
        { [_xN]: _Or }]
];
var AccessControlExposeHeadersList = [1, n0, _ACEHL,
    0, [0,
        { [_xN]: _He }]
];
var AliasICPRecordals = [1, n0, _AICPRl,
    0, [() => AliasICPRecordal$,
        { [_xN]: _AICPR }]
];
var AliasList = [1, n0, _AL,
    0, [0,
        { [_xN]: _CNAME }]
];
var AnycastIpListSummaries = [1, n0, _AILSn,
    0, [() => AnycastIpListSummary$,
        { [_xN]: _AILS }]
];
var AnycastIps = [1, n0, _AI,
    0, [0,
        { [_xN]: _AIn }]
];
var AwsAccountNumberList = [1, n0, _AANL,
    0, [0,
        { [_xN]: _AAN }]
];
var CacheBehaviorList = [1, n0, _CBL,
    0, [() => CacheBehavior$,
        { [_xN]: _CB }]
];
var CachePolicySummaryList = [1, n0, _CPSL,
    0, [() => CachePolicySummary$,
        { [_xN]: _CPS }]
];
var CloudFrontOriginAccessIdentitySummaryList = [1, n0, _CFOAISL,
    0, [() => CloudFrontOriginAccessIdentitySummary$,
        { [_xN]: _CFOAIS }]
];
var ConflictingAliases = [1, n0, _CAon,
    0, [() => ConflictingAlias$,
        { [_xN]: _CA }]
];
var ConnectionFunctionSummaryList = [1, n0, _CFSL,
    0, [() => ConnectionFunctionSummary$,
        { [_xN]: _CFS }]
];
var ConnectionGroupSummaryList = [1, n0, _CGSL,
    0, [() => ConnectionGroupSummary$,
        { [_xN]: _CGS }]
];
var ContentTypeProfileList = [1, n0, _CTPL,
    0, [() => ContentTypeProfile$,
        { [_xN]: _CTP }]
];
var ContinuousDeploymentPolicySummaryList = [1, n0, _CDPSL,
    0, [() => ContinuousDeploymentPolicySummary$,
        { [_xN]: _CDPS }]
];
var CookieNameList = [1, n0, _CNL,
    0, [0,
        { [_xN]: _N }]
];
var CustomErrorResponseList = [1, n0, _CERL,
    0, [() => CustomErrorResponse$,
        { [_xN]: _CER }]
];
var DistributionIdListSummary = [1, n0, _DILS,
    0, [0,
        { [_xN]: _DI }]
];
var DistributionIdOwnerItemList = [1, n0, _DIOIL,
    0, [() => DistributionIdOwner$,
        { [_xN]: _DIO }]
];
var DistributionSummaryList = [1, n0, _DSL,
    0, [() => DistributionSummary$,
        { [_xN]: _DS }]
];
var DistributionTenantList = [1, n0, _DTL,
    0, [() => DistributionTenantSummary$,
        { [_xN]: _DTS }]
];
var DnsConfigurationList = [1, n0, _DCL,
    0, [() => DnsConfiguration$,
        { [_xN]: _DCn }]
];
var DomainConflictsList = [1, n0, _DCLo,
    0, [() => DomainConflict$,
        { [_xN]: _DCom }]
];
var DomainList = [1, n0, _DLo,
    0, () => DomainItem$
];
var DomainResultList = [1, n0, _DRL,
    0, () => DomainResult$
];
var EncryptionEntityList = [1, n0, _EEL,
    0, [() => EncryptionEntity$,
        { [_xN]: _EEn }]
];
var EndPointList = [1, n0, _EPL,
    0, () => EndPoint$
];
var FieldLevelEncryptionProfileSummaryList = [1, n0, _FLEPSL,
    0, [() => FieldLevelEncryptionProfileSummary$,
        { [_xN]: _FLEPS }]
];
var FieldLevelEncryptionSummaryList = [1, n0, _FLESL,
    0, [() => FieldLevelEncryptionSummary$,
        { [_xN]: _FLES }]
];
var FieldList = [1, n0, _FLi,
    0, [0,
        { [_xN]: _Fie }]
];
var FieldPatternList = [1, n0, _FPL,
    0, [0,
        { [_xN]: _FPi }]
];
var FunctionAssociationList = [1, n0, _FAL,
    0, [() => FunctionAssociation$,
        { [_xN]: _FAu }]
];
var FunctionExecutionLogList = [1, n0, _FELL,
    8, 0
];
var FunctionSummaryList = [1, n0, _FSL,
    0, [() => FunctionSummary$,
        { [_xN]: _FS }]
];
var HeaderList = [1, n0, _HL,
    0, [0,
        { [_xN]: _N }]
];
var InvalidationSummaryList = [1, n0, _ISL,
    0, [() => InvalidationSummary$,
        { [_xN]: _ISn }]
];
var IpamCidrConfigList = [1, n0, _ICCL,
    0, [() => IpamCidrConfig$,
        { [_xN]: _ICCp }]
];
var KeyGroupSummaryList = [1, n0, _KGSL,
    0, [() => KeyGroupSummary$,
        { [_xN]: _KGS }]
];
var KeyPairIdList = [1, n0, _KPIL,
    0, [0,
        { [_xN]: _KPIe }]
];
var KeyValueStoreAssociationList = [1, n0, _KVSAL,
    0, [() => KeyValueStoreAssociation$,
        { [_xN]: _KVSAe }]
];
var KeyValueStoreSummaryList = [1, n0, _KVSSL,
    0, [() => KeyValueStore$,
        { [_xN]: _KVS }]
];
var KGKeyPairIdsList = [1, n0, _KGKPIL,
    0, [() => KGKeyPairIds$,
        { [_xN]: _KG }]
];
var LambdaFunctionAssociationList = [1, n0, _LFAL,
    0, [() => LambdaFunctionAssociation$,
        { [_xN]: _LFAa }]
];
var LocationList = [1, n0, _LL,
    0, [0,
        { [_xN]: _L }]
];
var MethodsList = [1, n0, _ML,
    0, [0,
        { [_xN]: _Met }]
];
var OriginAccessControlSummaryList = [1, n0, _OACSL,
    0, [() => OriginAccessControlSummary$,
        { [_xN]: _OACS }]
];
var OriginCustomHeadersList = [1, n0, _OCHL,
    0, [() => OriginCustomHeader$,
        { [_xN]: _OCH }]
];
var OriginGroupList = [1, n0, _OGL,
    0, [() => OriginGroup$,
        { [_xN]: _OGr }]
];
var OriginGroupMemberList = [1, n0, _OGML,
    0, [() => OriginGroupMember$,
        { [_xN]: _OGM }]
];
var OriginList = [1, n0, _OL,
    0, [() => Origin$,
        { [_xN]: _Or }]
];
var OriginRequestPolicySummaryList = [1, n0, _ORPSL,
    0, [() => OriginRequestPolicySummary$,
        { [_xN]: _ORPS }]
];
var ParameterDefinitions = [1, n0, _PDar,
    0, [() => ParameterDefinition$,
        0]
];
var _Parameters = [1, n0, _P,
    0, () => Parameter$
];
var PathList = [1, n0, _PL,
    0, [0,
        { [_xN]: _Pat }]
];
var PublicKeyIdList = [1, n0, _PKIL,
    0, [0,
        { [_xN]: _PK }]
];
var PublicKeySummaryList = [1, n0, _PKSL,
    0, [() => PublicKeySummary$,
        { [_xN]: _PKS }]
];
var QueryArgProfileList = [1, n0, _QAPL,
    0, [() => QueryArgProfile$,
        { [_xN]: _QAP }]
];
var QueryStringCacheKeysList = [1, n0, _QSCKL,
    0, [0,
        { [_xN]: _N }]
];
var QueryStringNamesList = [1, n0, _QSNL,
    0, [0,
        { [_xN]: _N }]
];
var RealtimeLogConfigList = [1, n0, _RLCL,
    0, [() => RealtimeLogConfig$,
        0]
];
var ResponseHeadersPolicyCustomHeaderList = [1, n0, _RHPCHL,
    0, [() => ResponseHeadersPolicyCustomHeader$,
        { [_xN]: _RHPCH }]
];
var ResponseHeadersPolicyRemoveHeaderList = [1, n0, _RHPRHL,
    0, [() => ResponseHeadersPolicyRemoveHeader$,
        { [_xN]: _RHPRH }]
];
var ResponseHeadersPolicySummaryList = [1, n0, _RHPSL,
    0, [() => ResponseHeadersPolicySummary$,
        { [_xN]: _RHPS }]
];
var SignerList = [1, n0, _SL,
    0, [() => Signer$,
        { [_xN]: _Si }]
];
var SslProtocolsList = [1, n0, _SPL,
    0, [0,
        { [_xN]: _SPs }]
];
var StagingDistributionDnsNameList = [1, n0, _SDDNL,
    0, [0,
        { [_xN]: _DNn }]
];
var StatusCodeList = [1, n0, _SCL,
    0, [1,
        { [_xN]: _SCta }]
];
var StreamingDistributionSummaryList = [1, n0, _SDSL,
    0, [() => StreamingDistributionSummary$,
        { [_xN]: _SDS }]
];
var TagKeyList = [1, n0, _TKL,
    0, [0,
        { [_xN]: _K }]
];
var TagList = [1, n0, _TL,
    0, [() => Tag$,
        { [_xN]: _Tag }]
];
var TrustedKeyGroupIdList = [1, n0, _TKGIL,
    0, [0,
        { [_xN]: _KG }]
];
var TrustStoreList = [1, n0, _TSL,
    0, [() => TrustStoreSummary$,
        { [_xN]: _TSS }]
];
var ValidationTokenDetailList = [1, n0, _VTDL,
    0, () => ValidationTokenDetail$
];
var VpcOriginSummaryList = [1, n0, _VOSL,
    0, [() => VpcOriginSummary$,
        { [_xN]: _VOS }]
];
var CaCertificatesBundleSource$ = [4, n0, _CCBS,
    0,
    [_CCBSL],
    [() => CaCertificatesBundleS3Location$]
];
var AssociateAlias$ = [9, n0, _AA,
    { [_h]: ["PUT", "/2020-05-31/distribution/{TargetDistributionId}/associate-alias", 200] }, () => AssociateAliasRequest$, () => __Unit
];
var AssociateDistributionTenantWebACL$ = [9, n0, _ADTWACL,
    { [_h]: ["PUT", "/2020-05-31/distribution-tenant/{Id}/associate-web-acl", 200] }, () => AssociateDistributionTenantWebACLRequest$, () => AssociateDistributionTenantWebACLResult$
];
var AssociateDistributionWebACL$ = [9, n0, _ADWACL,
    { [_h]: ["PUT", "/2020-05-31/distribution/{Id}/associate-web-acl", 200] }, () => AssociateDistributionWebACLRequest$, () => AssociateDistributionWebACLResult$
];
var CopyDistribution$ = [9, n0, _CD,
    { [_h]: ["POST", "/2020-05-31/distribution/{PrimaryDistributionId}/copy", 201] }, () => CopyDistributionRequest$, () => CopyDistributionResult$
];
var CreateAnycastIpList$ = [9, n0, _CAIL,
    { [_h]: ["POST", "/2020-05-31/anycast-ip-list", 202] }, () => CreateAnycastIpListRequest$, () => CreateAnycastIpListResult$
];
var CreateCachePolicy$ = [9, n0, _CCP,
    { [_h]: ["POST", "/2020-05-31/cache-policy", 201] }, () => CreateCachePolicyRequest$, () => CreateCachePolicyResult$
];
var CreateCloudFrontOriginAccessIdentity$ = [9, n0, _CCFOAI,
    { [_h]: ["POST", "/2020-05-31/origin-access-identity/cloudfront", 201] }, () => CreateCloudFrontOriginAccessIdentityRequest$, () => CreateCloudFrontOriginAccessIdentityResult$
];
var CreateConnectionFunction$ = [9, n0, _CCF,
    { [_h]: ["POST", "/2020-05-31/connection-function", 201] }, () => CreateConnectionFunctionRequest$, () => CreateConnectionFunctionResult$
];
var CreateConnectionGroup$ = [9, n0, _CCG,
    { [_h]: ["POST", "/2020-05-31/connection-group", 201] }, () => CreateConnectionGroupRequest$, () => CreateConnectionGroupResult$
];
var CreateContinuousDeploymentPolicy$ = [9, n0, _CCDP,
    { [_h]: ["POST", "/2020-05-31/continuous-deployment-policy", 201] }, () => CreateContinuousDeploymentPolicyRequest$, () => CreateContinuousDeploymentPolicyResult$
];
var CreateDistribution$ = [9, n0, _CDr,
    { [_h]: ["POST", "/2020-05-31/distribution", 201] }, () => CreateDistributionRequest$, () => CreateDistributionResult$
];
var CreateDistributionTenant$ = [9, n0, _CDT,
    { [_h]: ["POST", "/2020-05-31/distribution-tenant", 201] }, () => CreateDistributionTenantRequest$, () => CreateDistributionTenantResult$
];
var CreateDistributionWithTags$ = [9, n0, _CDWT,
    { [_h]: ["POST", "/2020-05-31/distribution?WithTags", 201] }, () => CreateDistributionWithTagsRequest$, () => CreateDistributionWithTagsResult$
];
var CreateFieldLevelEncryptionConfig$ = [9, n0, _CFLEC,
    { [_h]: ["POST", "/2020-05-31/field-level-encryption", 201] }, () => CreateFieldLevelEncryptionConfigRequest$, () => CreateFieldLevelEncryptionConfigResult$
];
var CreateFieldLevelEncryptionProfile$ = [9, n0, _CFLEP,
    { [_h]: ["POST", "/2020-05-31/field-level-encryption-profile", 201] }, () => CreateFieldLevelEncryptionProfileRequest$, () => CreateFieldLevelEncryptionProfileResult$
];
var CreateFunction$ = [9, n0, _CFr,
    { [_h]: ["POST", "/2020-05-31/function", 201] }, () => CreateFunctionRequest$, () => CreateFunctionResult$
];
var CreateInvalidation$ = [9, n0, _CI,
    { [_h]: ["POST", "/2020-05-31/distribution/{DistributionId}/invalidation", 201] }, () => CreateInvalidationRequest$, () => CreateInvalidationResult$
];
var CreateInvalidationForDistributionTenant$ = [9, n0, _CIFDT,
    { [_h]: ["POST", "/2020-05-31/distribution-tenant/{Id}/invalidation", 201] }, () => CreateInvalidationForDistributionTenantRequest$, () => CreateInvalidationForDistributionTenantResult$
];
var CreateKeyGroup$ = [9, n0, _CKG,
    { [_h]: ["POST", "/2020-05-31/key-group", 201] }, () => CreateKeyGroupRequest$, () => CreateKeyGroupResult$
];
var CreateKeyValueStore$ = [9, n0, _CKVS,
    { [_h]: ["POST", "/2020-05-31/key-value-store", 201] }, () => CreateKeyValueStoreRequest$, () => CreateKeyValueStoreResult$
];
var CreateMonitoringSubscription$ = [9, n0, _CMS,
    { [_h]: ["POST", "/2020-05-31/distributions/{DistributionId}/monitoring-subscription", 200] }, () => CreateMonitoringSubscriptionRequest$, () => CreateMonitoringSubscriptionResult$
];
var CreateOriginAccessControl$ = [9, n0, _COAC,
    { [_h]: ["POST", "/2020-05-31/origin-access-control", 201] }, () => CreateOriginAccessControlRequest$, () => CreateOriginAccessControlResult$
];
var CreateOriginRequestPolicy$ = [9, n0, _CORP,
    { [_h]: ["POST", "/2020-05-31/origin-request-policy", 201] }, () => CreateOriginRequestPolicyRequest$, () => CreateOriginRequestPolicyResult$
];
var CreatePublicKey$ = [9, n0, _CPK,
    { [_h]: ["POST", "/2020-05-31/public-key", 201] }, () => CreatePublicKeyRequest$, () => CreatePublicKeyResult$
];
var CreateRealtimeLogConfig$ = [9, n0, _CRLC,
    { [_h]: ["POST", "/2020-05-31/realtime-log-config", 201] }, () => CreateRealtimeLogConfigRequest$, () => CreateRealtimeLogConfigResult$
];
var CreateResponseHeadersPolicy$ = [9, n0, _CRHP,
    { [_h]: ["POST", "/2020-05-31/response-headers-policy", 201] }, () => CreateResponseHeadersPolicyRequest$, () => CreateResponseHeadersPolicyResult$
];
var CreateStreamingDistribution$ = [9, n0, _CSD,
    { [_h]: ["POST", "/2020-05-31/streaming-distribution", 201] }, () => CreateStreamingDistributionRequest$, () => CreateStreamingDistributionResult$
];
var CreateStreamingDistributionWithTags$ = [9, n0, _CSDWT,
    { [_h]: ["POST", "/2020-05-31/streaming-distribution?WithTags", 201] }, () => CreateStreamingDistributionWithTagsRequest$, () => CreateStreamingDistributionWithTagsResult$
];
var CreateTrustStore$ = [9, n0, _CTS,
    { [_h]: ["POST", "/2020-05-31/trust-store", 201] }, () => CreateTrustStoreRequest$, () => CreateTrustStoreResult$
];
var CreateVpcOrigin$ = [9, n0, _CVO,
    { [_h]: ["POST", "/2020-05-31/vpc-origin", 202] }, () => CreateVpcOriginRequest$, () => CreateVpcOriginResult$
];
var DeleteAnycastIpList$ = [9, n0, _DAIL,
    { [_h]: ["DELETE", "/2020-05-31/anycast-ip-list/{Id}", 204] }, () => DeleteAnycastIpListRequest$, () => __Unit
];
var DeleteCachePolicy$ = [9, n0, _DCP,
    { [_h]: ["DELETE", "/2020-05-31/cache-policy/{Id}", 204] }, () => DeleteCachePolicyRequest$, () => __Unit
];
var DeleteCloudFrontOriginAccessIdentity$ = [9, n0, _DCFOAI,
    { [_h]: ["DELETE", "/2020-05-31/origin-access-identity/cloudfront/{Id}", 204] }, () => DeleteCloudFrontOriginAccessIdentityRequest$, () => __Unit
];
var DeleteConnectionFunction$ = [9, n0, _DCF,
    { [_h]: ["DELETE", "/2020-05-31/connection-function/{Id}", 204] }, () => DeleteConnectionFunctionRequest$, () => __Unit
];
var DeleteConnectionGroup$ = [9, n0, _DCG,
    { [_h]: ["DELETE", "/2020-05-31/connection-group/{Id}", 204] }, () => DeleteConnectionGroupRequest$, () => __Unit
];
var DeleteContinuousDeploymentPolicy$ = [9, n0, _DCDP,
    { [_h]: ["DELETE", "/2020-05-31/continuous-deployment-policy/{Id}", 204] }, () => DeleteContinuousDeploymentPolicyRequest$, () => __Unit
];
var DeleteDistribution$ = [9, n0, _DD,
    { [_h]: ["DELETE", "/2020-05-31/distribution/{Id}", 204] }, () => DeleteDistributionRequest$, () => __Unit
];
var DeleteDistributionTenant$ = [9, n0, _DDT,
    { [_h]: ["DELETE", "/2020-05-31/distribution-tenant/{Id}", 204] }, () => DeleteDistributionTenantRequest$, () => __Unit
];
var DeleteFieldLevelEncryptionConfig$ = [9, n0, _DFLEC,
    { [_h]: ["DELETE", "/2020-05-31/field-level-encryption/{Id}", 204] }, () => DeleteFieldLevelEncryptionConfigRequest$, () => __Unit
];
var DeleteFieldLevelEncryptionProfile$ = [9, n0, _DFLEP,
    { [_h]: ["DELETE", "/2020-05-31/field-level-encryption-profile/{Id}", 204] }, () => DeleteFieldLevelEncryptionProfileRequest$, () => __Unit
];
var DeleteFunction$ = [9, n0, _DF,
    { [_h]: ["DELETE", "/2020-05-31/function/{Name}", 204] }, () => DeleteFunctionRequest$, () => __Unit
];
var DeleteKeyGroup$ = [9, n0, _DKG,
    { [_h]: ["DELETE", "/2020-05-31/key-group/{Id}", 204] }, () => DeleteKeyGroupRequest$, () => __Unit
];
var DeleteKeyValueStore$ = [9, n0, _DKVS,
    { [_h]: ["DELETE", "/2020-05-31/key-value-store/{Name}", 204] }, () => DeleteKeyValueStoreRequest$, () => __Unit
];
var DeleteMonitoringSubscription$ = [9, n0, _DMS,
    { [_h]: ["DELETE", "/2020-05-31/distributions/{DistributionId}/monitoring-subscription", 200] }, () => DeleteMonitoringSubscriptionRequest$, () => DeleteMonitoringSubscriptionResult$
];
var DeleteOriginAccessControl$ = [9, n0, _DOAC,
    { [_h]: ["DELETE", "/2020-05-31/origin-access-control/{Id}", 204] }, () => DeleteOriginAccessControlRequest$, () => __Unit
];
var DeleteOriginRequestPolicy$ = [9, n0, _DORP,
    { [_h]: ["DELETE", "/2020-05-31/origin-request-policy/{Id}", 204] }, () => DeleteOriginRequestPolicyRequest$, () => __Unit
];
var DeletePublicKey$ = [9, n0, _DPK,
    { [_h]: ["DELETE", "/2020-05-31/public-key/{Id}", 204] }, () => DeletePublicKeyRequest$, () => __Unit
];
var DeleteRealtimeLogConfig$ = [9, n0, _DRLC,
    { [_h]: ["POST", "/2020-05-31/delete-realtime-log-config", 204] }, () => DeleteRealtimeLogConfigRequest$, () => __Unit
];
var DeleteResourcePolicy$ = [9, n0, _DRP,
    { [_h]: ["POST", "/2020-05-31/delete-resource-policy", 200] }, () => DeleteResourcePolicyRequest$, () => __Unit
];
var DeleteResponseHeadersPolicy$ = [9, n0, _DRHP,
    { [_h]: ["DELETE", "/2020-05-31/response-headers-policy/{Id}", 204] }, () => DeleteResponseHeadersPolicyRequest$, () => __Unit
];
var DeleteStreamingDistribution$ = [9, n0, _DSD,
    { [_h]: ["DELETE", "/2020-05-31/streaming-distribution/{Id}", 204] }, () => DeleteStreamingDistributionRequest$, () => __Unit
];
var DeleteTrustStore$ = [9, n0, _DTSe,
    { [_h]: ["DELETE", "/2020-05-31/trust-store/{Id}", 204] }, () => DeleteTrustStoreRequest$, () => __Unit
];
var DeleteVpcOrigin$ = [9, n0, _DVO,
    { [_h]: ["DELETE", "/2020-05-31/vpc-origin/{Id}", 202] }, () => DeleteVpcOriginRequest$, () => DeleteVpcOriginResult$
];
var DescribeConnectionFunction$ = [9, n0, _DCFe,
    { [_h]: ["GET", "/2020-05-31/connection-function/{Identifier}/describe", 200] }, () => DescribeConnectionFunctionRequest$, () => DescribeConnectionFunctionResult$
];
var DescribeFunction$ = [9, n0, _DFe,
    { [_h]: ["GET", "/2020-05-31/function/{Name}/describe", 200] }, () => DescribeFunctionRequest$, () => DescribeFunctionResult$
];
var DescribeKeyValueStore$ = [9, n0, _DKVSe,
    { [_h]: ["GET", "/2020-05-31/key-value-store/{Name}", 200] }, () => DescribeKeyValueStoreRequest$, () => DescribeKeyValueStoreResult$
];
var DisassociateDistributionTenantWebACL$ = [9, n0, _DDTWACL,
    { [_h]: ["PUT", "/2020-05-31/distribution-tenant/{Id}/disassociate-web-acl", 200] }, () => DisassociateDistributionTenantWebACLRequest$, () => DisassociateDistributionTenantWebACLResult$
];
var DisassociateDistributionWebACL$ = [9, n0, _DDWACL,
    { [_h]: ["PUT", "/2020-05-31/distribution/{Id}/disassociate-web-acl", 200] }, () => DisassociateDistributionWebACLRequest$, () => DisassociateDistributionWebACLResult$
];
var GetAnycastIpList$ = [9, n0, _GAIL,
    { [_h]: ["GET", "/2020-05-31/anycast-ip-list/{Id}", 200] }, () => GetAnycastIpListRequest$, () => GetAnycastIpListResult$
];
var GetCachePolicy$ = [9, n0, _GCP,
    { [_h]: ["GET", "/2020-05-31/cache-policy/{Id}", 200] }, () => GetCachePolicyRequest$, () => GetCachePolicyResult$
];
var GetCachePolicyConfig$ = [9, n0, _GCPC,
    { [_h]: ["GET", "/2020-05-31/cache-policy/{Id}/config", 200] }, () => GetCachePolicyConfigRequest$, () => GetCachePolicyConfigResult$
];
var GetCloudFrontOriginAccessIdentity$ = [9, n0, _GCFOAI,
    { [_h]: ["GET", "/2020-05-31/origin-access-identity/cloudfront/{Id}", 200] }, () => GetCloudFrontOriginAccessIdentityRequest$, () => GetCloudFrontOriginAccessIdentityResult$
];
var GetCloudFrontOriginAccessIdentityConfig$ = [9, n0, _GCFOAIC,
    { [_h]: ["GET", "/2020-05-31/origin-access-identity/cloudfront/{Id}/config", 200] }, () => GetCloudFrontOriginAccessIdentityConfigRequest$, () => GetCloudFrontOriginAccessIdentityConfigResult$
];
var GetConnectionFunction$ = [9, n0, _GCF,
    { [_h]: ["GET", "/2020-05-31/connection-function/{Identifier}", 200] }, () => GetConnectionFunctionRequest$, () => GetConnectionFunctionResult$
];
var GetConnectionGroup$ = [9, n0, _GCG,
    { [_h]: ["GET", "/2020-05-31/connection-group/{Identifier}", 200] }, () => GetConnectionGroupRequest$, () => GetConnectionGroupResult$
];
var GetConnectionGroupByRoutingEndpoint$ = [9, n0, _GCGBRE,
    { [_h]: ["GET", "/2020-05-31/connection-group", 200] }, () => GetConnectionGroupByRoutingEndpointRequest$, () => GetConnectionGroupByRoutingEndpointResult$
];
var GetContinuousDeploymentPolicy$ = [9, n0, _GCDP,
    { [_h]: ["GET", "/2020-05-31/continuous-deployment-policy/{Id}", 200] }, () => GetContinuousDeploymentPolicyRequest$, () => GetContinuousDeploymentPolicyResult$
];
var GetContinuousDeploymentPolicyConfig$ = [9, n0, _GCDPC,
    { [_h]: ["GET", "/2020-05-31/continuous-deployment-policy/{Id}/config", 200] }, () => GetContinuousDeploymentPolicyConfigRequest$, () => GetContinuousDeploymentPolicyConfigResult$
];
var GetDistribution$ = [9, n0, _GD,
    { [_h]: ["GET", "/2020-05-31/distribution/{Id}", 200] }, () => GetDistributionRequest$, () => GetDistributionResult$
];
var GetDistributionConfig$ = [9, n0, _GDC,
    { [_h]: ["GET", "/2020-05-31/distribution/{Id}/config", 200] }, () => GetDistributionConfigRequest$, () => GetDistributionConfigResult$
];
var GetDistributionTenant$ = [9, n0, _GDT,
    { [_h]: ["GET", "/2020-05-31/distribution-tenant/{Identifier}", 200] }, () => GetDistributionTenantRequest$, () => GetDistributionTenantResult$
];
var GetDistributionTenantByDomain$ = [9, n0, _GDTBD,
    { [_h]: ["GET", "/2020-05-31/distribution-tenant", 200] }, () => GetDistributionTenantByDomainRequest$, () => GetDistributionTenantByDomainResult$
];
var GetFieldLevelEncryption$ = [9, n0, _GFLE,
    { [_h]: ["GET", "/2020-05-31/field-level-encryption/{Id}", 200] }, () => GetFieldLevelEncryptionRequest$, () => GetFieldLevelEncryptionResult$
];
var GetFieldLevelEncryptionConfig$ = [9, n0, _GFLEC,
    { [_h]: ["GET", "/2020-05-31/field-level-encryption/{Id}/config", 200] }, () => GetFieldLevelEncryptionConfigRequest$, () => GetFieldLevelEncryptionConfigResult$
];
var GetFieldLevelEncryptionProfile$ = [9, n0, _GFLEP,
    { [_h]: ["GET", "/2020-05-31/field-level-encryption-profile/{Id}", 200] }, () => GetFieldLevelEncryptionProfileRequest$, () => GetFieldLevelEncryptionProfileResult$
];
var GetFieldLevelEncryptionProfileConfig$ = [9, n0, _GFLEPC,
    { [_h]: ["GET", "/2020-05-31/field-level-encryption-profile/{Id}/config", 200] }, () => GetFieldLevelEncryptionProfileConfigRequest$, () => GetFieldLevelEncryptionProfileConfigResult$
];
var GetFunction$ = [9, n0, _GF,
    { [_h]: ["GET", "/2020-05-31/function/{Name}", 200] }, () => GetFunctionRequest$, () => GetFunctionResult$
];
var GetInvalidation$ = [9, n0, _GI,
    { [_h]: ["GET", "/2020-05-31/distribution/{DistributionId}/invalidation/{Id}", 200] }, () => GetInvalidationRequest$, () => GetInvalidationResult$
];
var GetInvalidationForDistributionTenant$ = [9, n0, _GIFDT,
    { [_h]: ["GET", "/2020-05-31/distribution-tenant/{DistributionTenantId}/invalidation/{Id}", 200] }, () => GetInvalidationForDistributionTenantRequest$, () => GetInvalidationForDistributionTenantResult$
];
var GetKeyGroup$ = [9, n0, _GKG,
    { [_h]: ["GET", "/2020-05-31/key-group/{Id}", 200] }, () => GetKeyGroupRequest$, () => GetKeyGroupResult$
];
var GetKeyGroupConfig$ = [9, n0, _GKGC,
    { [_h]: ["GET", "/2020-05-31/key-group/{Id}/config", 200] }, () => GetKeyGroupConfigRequest$, () => GetKeyGroupConfigResult$
];
var GetManagedCertificateDetails$ = [9, n0, _GMCD,
    { [_h]: ["GET", "/2020-05-31/managed-certificate/{Identifier}", 200] }, () => GetManagedCertificateDetailsRequest$, () => GetManagedCertificateDetailsResult$
];
var GetMonitoringSubscription$ = [9, n0, _GMS,
    { [_h]: ["GET", "/2020-05-31/distributions/{DistributionId}/monitoring-subscription", 200] }, () => GetMonitoringSubscriptionRequest$, () => GetMonitoringSubscriptionResult$
];
var GetOriginAccessControl$ = [9, n0, _GOAC,
    { [_h]: ["GET", "/2020-05-31/origin-access-control/{Id}", 200] }, () => GetOriginAccessControlRequest$, () => GetOriginAccessControlResult$
];
var GetOriginAccessControlConfig$ = [9, n0, _GOACC,
    { [_h]: ["GET", "/2020-05-31/origin-access-control/{Id}/config", 200] }, () => GetOriginAccessControlConfigRequest$, () => GetOriginAccessControlConfigResult$
];
var GetOriginRequestPolicy$ = [9, n0, _GORP,
    { [_h]: ["GET", "/2020-05-31/origin-request-policy/{Id}", 200] }, () => GetOriginRequestPolicyRequest$, () => GetOriginRequestPolicyResult$
];
var GetOriginRequestPolicyConfig$ = [9, n0, _GORPC,
    { [_h]: ["GET", "/2020-05-31/origin-request-policy/{Id}/config", 200] }, () => GetOriginRequestPolicyConfigRequest$, () => GetOriginRequestPolicyConfigResult$
];
var GetPublicKey$ = [9, n0, _GPK,
    { [_h]: ["GET", "/2020-05-31/public-key/{Id}", 200] }, () => GetPublicKeyRequest$, () => GetPublicKeyResult$
];
var GetPublicKeyConfig$ = [9, n0, _GPKC,
    { [_h]: ["GET", "/2020-05-31/public-key/{Id}/config", 200] }, () => GetPublicKeyConfigRequest$, () => GetPublicKeyConfigResult$
];
var GetRealtimeLogConfig$ = [9, n0, _GRLC,
    { [_h]: ["POST", "/2020-05-31/get-realtime-log-config", 200] }, () => GetRealtimeLogConfigRequest$, () => GetRealtimeLogConfigResult$
];
var GetResourcePolicy$ = [9, n0, _GRP,
    { [_h]: ["POST", "/2020-05-31/get-resource-policy", 200] }, () => GetResourcePolicyRequest$, () => GetResourcePolicyResult$
];
var GetResponseHeadersPolicy$ = [9, n0, _GRHP,
    { [_h]: ["GET", "/2020-05-31/response-headers-policy/{Id}", 200] }, () => GetResponseHeadersPolicyRequest$, () => GetResponseHeadersPolicyResult$
];
var GetResponseHeadersPolicyConfig$ = [9, n0, _GRHPC,
    { [_h]: ["GET", "/2020-05-31/response-headers-policy/{Id}/config", 200] }, () => GetResponseHeadersPolicyConfigRequest$, () => GetResponseHeadersPolicyConfigResult$
];
var GetStreamingDistribution$ = [9, n0, _GSD,
    { [_h]: ["GET", "/2020-05-31/streaming-distribution/{Id}", 200] }, () => GetStreamingDistributionRequest$, () => GetStreamingDistributionResult$
];
var GetStreamingDistributionConfig$ = [9, n0, _GSDC,
    { [_h]: ["GET", "/2020-05-31/streaming-distribution/{Id}/config", 200] }, () => GetStreamingDistributionConfigRequest$, () => GetStreamingDistributionConfigResult$
];
var GetTrustStore$ = [9, n0, _GTS,
    { [_h]: ["GET", "/2020-05-31/trust-store/{Identifier}", 200] }, () => GetTrustStoreRequest$, () => GetTrustStoreResult$
];
var GetVpcOrigin$ = [9, n0, _GVO,
    { [_h]: ["GET", "/2020-05-31/vpc-origin/{Id}", 200] }, () => GetVpcOriginRequest$, () => GetVpcOriginResult$
];
var ListAnycastIpLists$ = [9, n0, _LAIL,
    { [_h]: ["GET", "/2020-05-31/anycast-ip-list", 200] }, () => ListAnycastIpListsRequest$, () => ListAnycastIpListsResult$
];
var ListCachePolicies$ = [9, n0, _LCP,
    { [_h]: ["GET", "/2020-05-31/cache-policy", 200] }, () => ListCachePoliciesRequest$, () => ListCachePoliciesResult$
];
var ListCloudFrontOriginAccessIdentities$ = [9, n0, _LCFOAI,
    { [_h]: ["GET", "/2020-05-31/origin-access-identity/cloudfront", 200] }, () => ListCloudFrontOriginAccessIdentitiesRequest$, () => ListCloudFrontOriginAccessIdentitiesResult$
];
var ListConflictingAliases$ = [9, n0, _LCA,
    { [_h]: ["GET", "/2020-05-31/conflicting-alias", 200] }, () => ListConflictingAliasesRequest$, () => ListConflictingAliasesResult$
];
var ListConnectionFunctions$ = [9, n0, _LCF,
    { [_h]: ["POST", "/2020-05-31/connection-functions", 200] }, () => ListConnectionFunctionsRequest$, () => ListConnectionFunctionsResult$
];
var ListConnectionGroups$ = [9, n0, _LCG,
    { [_h]: ["POST", "/2020-05-31/connection-groups", 200] }, () => ListConnectionGroupsRequest$, () => ListConnectionGroupsResult$
];
var ListContinuousDeploymentPolicies$ = [9, n0, _LCDP,
    { [_h]: ["GET", "/2020-05-31/continuous-deployment-policy", 200] }, () => ListContinuousDeploymentPoliciesRequest$, () => ListContinuousDeploymentPoliciesResult$
];
var ListDistributions$ = [9, n0, _LD,
    { [_h]: ["GET", "/2020-05-31/distribution", 200] }, () => ListDistributionsRequest$, () => ListDistributionsResult$
];
var ListDistributionsByAnycastIpListId$ = [9, n0, _LDBAILI,
    { [_h]: ["GET", "/2020-05-31/distributionsByAnycastIpListId/{AnycastIpListId}", 200] }, () => ListDistributionsByAnycastIpListIdRequest$, () => ListDistributionsByAnycastIpListIdResult$
];
var ListDistributionsByCachePolicyId$ = [9, n0, _LDBCPI,
    { [_h]: ["GET", "/2020-05-31/distributionsByCachePolicyId/{CachePolicyId}", 200] }, () => ListDistributionsByCachePolicyIdRequest$, () => ListDistributionsByCachePolicyIdResult$
];
var ListDistributionsByConnectionFunction$ = [9, n0, _LDBCF,
    { [_h]: ["GET", "/2020-05-31/distributionsByConnectionFunction", 200] }, () => ListDistributionsByConnectionFunctionRequest$, () => ListDistributionsByConnectionFunctionResult$
];
var ListDistributionsByConnectionMode$ = [9, n0, _LDBCM,
    { [_h]: ["GET", "/2020-05-31/distributionsByConnectionMode/{ConnectionMode}", 200] }, () => ListDistributionsByConnectionModeRequest$, () => ListDistributionsByConnectionModeResult$
];
var ListDistributionsByKeyGroup$ = [9, n0, _LDBKG,
    { [_h]: ["GET", "/2020-05-31/distributionsByKeyGroupId/{KeyGroupId}", 200] }, () => ListDistributionsByKeyGroupRequest$, () => ListDistributionsByKeyGroupResult$
];
var ListDistributionsByOriginRequestPolicyId$ = [9, n0, _LDBORPI,
    { [_h]: ["GET", "/2020-05-31/distributionsByOriginRequestPolicyId/{OriginRequestPolicyId}", 200] }, () => ListDistributionsByOriginRequestPolicyIdRequest$, () => ListDistributionsByOriginRequestPolicyIdResult$
];
var ListDistributionsByOwnedResource$ = [9, n0, _LDBOR,
    { [_h]: ["GET", "/2020-05-31/distributionsByOwnedResource/{ResourceArn}", 200] }, () => ListDistributionsByOwnedResourceRequest$, () => ListDistributionsByOwnedResourceResult$
];
var ListDistributionsByRealtimeLogConfig$ = [9, n0, _LDBRLC,
    { [_h]: ["POST", "/2020-05-31/distributionsByRealtimeLogConfig", 200] }, () => ListDistributionsByRealtimeLogConfigRequest$, () => ListDistributionsByRealtimeLogConfigResult$
];
var ListDistributionsByResponseHeadersPolicyId$ = [9, n0, _LDBRHPI,
    { [_h]: ["GET", "/2020-05-31/distributionsByResponseHeadersPolicyId/{ResponseHeadersPolicyId}", 200] }, () => ListDistributionsByResponseHeadersPolicyIdRequest$, () => ListDistributionsByResponseHeadersPolicyIdResult$
];
var ListDistributionsByTrustStore$ = [9, n0, _LDBTS,
    { [_h]: ["GET", "/2020-05-31/distributionsByTrustStore", 200] }, () => ListDistributionsByTrustStoreRequest$, () => ListDistributionsByTrustStoreResult$
];
var ListDistributionsByVpcOriginId$ = [9, n0, _LDBVOI,
    { [_h]: ["GET", "/2020-05-31/distributionsByVpcOriginId/{VpcOriginId}", 200] }, () => ListDistributionsByVpcOriginIdRequest$, () => ListDistributionsByVpcOriginIdResult$
];
var ListDistributionsByWebACLId$ = [9, n0, _LDBWACLI,
    { [_h]: ["GET", "/2020-05-31/distributionsByWebACLId/{WebACLId}", 200] }, () => ListDistributionsByWebACLIdRequest$, () => ListDistributionsByWebACLIdResult$
];
var ListDistributionTenants$ = [9, n0, _LDT,
    { [_h]: ["POST", "/2020-05-31/distribution-tenants", 200] }, () => ListDistributionTenantsRequest$, () => ListDistributionTenantsResult$
];
var ListDistributionTenantsByCustomization$ = [9, n0, _LDTBC,
    { [_h]: ["POST", "/2020-05-31/distribution-tenants-by-customization", 200] }, () => ListDistributionTenantsByCustomizationRequest$, () => ListDistributionTenantsByCustomizationResult$
];
var ListDomainConflicts$ = [9, n0, _LDC,
    { [_h]: ["POST", "/2020-05-31/domain-conflicts", 200] }, () => ListDomainConflictsRequest$, () => ListDomainConflictsResult$
];
var ListFieldLevelEncryptionConfigs$ = [9, n0, _LFLEC,
    { [_h]: ["GET", "/2020-05-31/field-level-encryption", 200] }, () => ListFieldLevelEncryptionConfigsRequest$, () => ListFieldLevelEncryptionConfigsResult$
];
var ListFieldLevelEncryptionProfiles$ = [9, n0, _LFLEP,
    { [_h]: ["GET", "/2020-05-31/field-level-encryption-profile", 200] }, () => ListFieldLevelEncryptionProfilesRequest$, () => ListFieldLevelEncryptionProfilesResult$
];
var ListFunctions$ = [9, n0, _LF,
    { [_h]: ["GET", "/2020-05-31/function", 200] }, () => ListFunctionsRequest$, () => ListFunctionsResult$
];
var ListInvalidations$ = [9, n0, _LI,
    { [_h]: ["GET", "/2020-05-31/distribution/{DistributionId}/invalidation", 200] }, () => ListInvalidationsRequest$, () => ListInvalidationsResult$
];
var ListInvalidationsForDistributionTenant$ = [9, n0, _LIFDT,
    { [_h]: ["GET", "/2020-05-31/distribution-tenant/{Id}/invalidation", 200] }, () => ListInvalidationsForDistributionTenantRequest$, () => ListInvalidationsForDistributionTenantResult$
];
var ListKeyGroups$ = [9, n0, _LKG,
    { [_h]: ["GET", "/2020-05-31/key-group", 200] }, () => ListKeyGroupsRequest$, () => ListKeyGroupsResult$
];
var ListKeyValueStores$ = [9, n0, _LKVS,
    { [_h]: ["GET", "/2020-05-31/key-value-store", 200] }, () => ListKeyValueStoresRequest$, () => ListKeyValueStoresResult$
];
var ListOriginAccessControls$ = [9, n0, _LOAC,
    { [_h]: ["GET", "/2020-05-31/origin-access-control", 200] }, () => ListOriginAccessControlsRequest$, () => ListOriginAccessControlsResult$
];
var ListOriginRequestPolicies$ = [9, n0, _LORP,
    { [_h]: ["GET", "/2020-05-31/origin-request-policy", 200] }, () => ListOriginRequestPoliciesRequest$, () => ListOriginRequestPoliciesResult$
];
var ListPublicKeys$ = [9, n0, _LPK,
    { [_h]: ["GET", "/2020-05-31/public-key", 200] }, () => ListPublicKeysRequest$, () => ListPublicKeysResult$
];
var ListRealtimeLogConfigs$ = [9, n0, _LRLC,
    { [_h]: ["GET", "/2020-05-31/realtime-log-config", 200] }, () => ListRealtimeLogConfigsRequest$, () => ListRealtimeLogConfigsResult$
];
var ListResponseHeadersPolicies$ = [9, n0, _LRHP,
    { [_h]: ["GET", "/2020-05-31/response-headers-policy", 200] }, () => ListResponseHeadersPoliciesRequest$, () => ListResponseHeadersPoliciesResult$
];
var ListStreamingDistributions$ = [9, n0, _LSD,
    { [_h]: ["GET", "/2020-05-31/streaming-distribution", 200] }, () => ListStreamingDistributionsRequest$, () => ListStreamingDistributionsResult$
];
var ListTagsForResource$ = [9, n0, _LTFR,
    { [_h]: ["GET", "/2020-05-31/tagging", 200] }, () => ListTagsForResourceRequest$, () => ListTagsForResourceResult$
];
var ListTrustStores$ = [9, n0, _LTS,
    { [_h]: ["POST", "/2020-05-31/trust-stores", 200] }, () => ListTrustStoresRequest$, () => ListTrustStoresResult$
];
var ListVpcOrigins$ = [9, n0, _LVO,
    { [_h]: ["GET", "/2020-05-31/vpc-origin", 200] }, () => ListVpcOriginsRequest$, () => ListVpcOriginsResult$
];
var PublishConnectionFunction$ = [9, n0, _PCF,
    { [_h]: ["POST", "/2020-05-31/connection-function/{Id}/publish", 200] }, () => PublishConnectionFunctionRequest$, () => PublishConnectionFunctionResult$
];
var PublishFunction$ = [9, n0, _PFu,
    { [_h]: ["POST", "/2020-05-31/function/{Name}/publish", 200] }, () => PublishFunctionRequest$, () => PublishFunctionResult$
];
var PutResourcePolicy$ = [9, n0, _PRP,
    { [_h]: ["POST", "/2020-05-31/put-resource-policy", 200] }, () => PutResourcePolicyRequest$, () => PutResourcePolicyResult$
];
var TagResource$ = [9, n0, _TRag,
    { [_h]: ["POST", "/2020-05-31/tagging?Operation=Tag", 204] }, () => TagResourceRequest$, () => __Unit
];
var TestConnectionFunction$ = [9, n0, _TCF,
    { [_h]: ["POST", "/2020-05-31/connection-function/{Id}/test", 200] }, () => TestConnectionFunctionRequest$, () => TestConnectionFunctionResult$
];
var TestFunction$ = [9, n0, _TF,
    { [_h]: ["POST", "/2020-05-31/function/{Name}/test", 200] }, () => TestFunctionRequest$, () => TestFunctionResult$
];
var UntagResource$ = [9, n0, _UR,
    { [_h]: ["POST", "/2020-05-31/tagging?Operation=Untag", 204] }, () => UntagResourceRequest$, () => __Unit
];
var UpdateAnycastIpList$ = [9, n0, _UAIL,
    { [_h]: ["PUT", "/2020-05-31/anycast-ip-list/{Id}", 200] }, () => UpdateAnycastIpListRequest$, () => UpdateAnycastIpListResult$
];
var UpdateCachePolicy$ = [9, n0, _UCP,
    { [_h]: ["PUT", "/2020-05-31/cache-policy/{Id}", 200] }, () => UpdateCachePolicyRequest$, () => UpdateCachePolicyResult$
];
var UpdateCloudFrontOriginAccessIdentity$ = [9, n0, _UCFOAI,
    { [_h]: ["PUT", "/2020-05-31/origin-access-identity/cloudfront/{Id}/config", 200] }, () => UpdateCloudFrontOriginAccessIdentityRequest$, () => UpdateCloudFrontOriginAccessIdentityResult$
];
var UpdateConnectionFunction$ = [9, n0, _UCF,
    { [_h]: ["PUT", "/2020-05-31/connection-function/{Id}", 200] }, () => UpdateConnectionFunctionRequest$, () => UpdateConnectionFunctionResult$
];
var UpdateConnectionGroup$ = [9, n0, _UCG,
    { [_h]: ["PUT", "/2020-05-31/connection-group/{Id}", 200] }, () => UpdateConnectionGroupRequest$, () => UpdateConnectionGroupResult$
];
var UpdateContinuousDeploymentPolicy$ = [9, n0, _UCDP,
    { [_h]: ["PUT", "/2020-05-31/continuous-deployment-policy/{Id}", 200] }, () => UpdateContinuousDeploymentPolicyRequest$, () => UpdateContinuousDeploymentPolicyResult$
];
var UpdateDistribution$ = [9, n0, _UD,
    { [_h]: ["PUT", "/2020-05-31/distribution/{Id}/config", 200] }, () => UpdateDistributionRequest$, () => UpdateDistributionResult$
];
var UpdateDistributionTenant$ = [9, n0, _UDT,
    { [_h]: ["PUT", "/2020-05-31/distribution-tenant/{Id}", 200] }, () => UpdateDistributionTenantRequest$, () => UpdateDistributionTenantResult$
];
var UpdateDistributionWithStagingConfig$ = [9, n0, _UDWSC,
    { [_h]: ["PUT", "/2020-05-31/distribution/{Id}/promote-staging-config", 200] }, () => UpdateDistributionWithStagingConfigRequest$, () => UpdateDistributionWithStagingConfigResult$
];
var UpdateDomainAssociation$ = [9, n0, _UDA,
    { [_h]: ["POST", "/2020-05-31/domain-association", 200] }, () => UpdateDomainAssociationRequest$, () => UpdateDomainAssociationResult$
];
var UpdateFieldLevelEncryptionConfig$ = [9, n0, _UFLEC,
    { [_h]: ["PUT", "/2020-05-31/field-level-encryption/{Id}/config", 200] }, () => UpdateFieldLevelEncryptionConfigRequest$, () => UpdateFieldLevelEncryptionConfigResult$
];
var UpdateFieldLevelEncryptionProfile$ = [9, n0, _UFLEP,
    { [_h]: ["PUT", "/2020-05-31/field-level-encryption-profile/{Id}/config", 200] }, () => UpdateFieldLevelEncryptionProfileRequest$, () => UpdateFieldLevelEncryptionProfileResult$
];
var UpdateFunction$ = [9, n0, _UF,
    { [_h]: ["PUT", "/2020-05-31/function/{Name}", 200] }, () => UpdateFunctionRequest$, () => UpdateFunctionResult$
];
var UpdateKeyGroup$ = [9, n0, _UKG,
    { [_h]: ["PUT", "/2020-05-31/key-group/{Id}", 200] }, () => UpdateKeyGroupRequest$, () => UpdateKeyGroupResult$
];
var UpdateKeyValueStore$ = [9, n0, _UKVS,
    { [_h]: ["PUT", "/2020-05-31/key-value-store/{Name}", 200] }, () => UpdateKeyValueStoreRequest$, () => UpdateKeyValueStoreResult$
];
var UpdateOriginAccessControl$ = [9, n0, _UOAC,
    { [_h]: ["PUT", "/2020-05-31/origin-access-control/{Id}/config", 200] }, () => UpdateOriginAccessControlRequest$, () => UpdateOriginAccessControlResult$
];
var UpdateOriginRequestPolicy$ = [9, n0, _UORP,
    { [_h]: ["PUT", "/2020-05-31/origin-request-policy/{Id}", 200] }, () => UpdateOriginRequestPolicyRequest$, () => UpdateOriginRequestPolicyResult$
];
var UpdatePublicKey$ = [9, n0, _UPK,
    { [_h]: ["PUT", "/2020-05-31/public-key/{Id}/config", 200] }, () => UpdatePublicKeyRequest$, () => UpdatePublicKeyResult$
];
var UpdateRealtimeLogConfig$ = [9, n0, _URLC,
    { [_h]: ["PUT", "/2020-05-31/realtime-log-config", 200] }, () => UpdateRealtimeLogConfigRequest$, () => UpdateRealtimeLogConfigResult$
];
var UpdateResponseHeadersPolicy$ = [9, n0, _URHP,
    { [_h]: ["PUT", "/2020-05-31/response-headers-policy/{Id}", 200] }, () => UpdateResponseHeadersPolicyRequest$, () => UpdateResponseHeadersPolicyResult$
];
var UpdateStreamingDistribution$ = [9, n0, _USD,
    { [_h]: ["PUT", "/2020-05-31/streaming-distribution/{Id}/config", 200] }, () => UpdateStreamingDistributionRequest$, () => UpdateStreamingDistributionResult$
];
var UpdateTrustStore$ = [9, n0, _UTS,
    { [_h]: ["PUT", "/2020-05-31/trust-store/{Id}", 200] }, () => UpdateTrustStoreRequest$, () => UpdateTrustStoreResult$
];
var UpdateVpcOrigin$ = [9, n0, _UVO,
    { [_h]: ["PUT", "/2020-05-31/vpc-origin/{Id}", 202] }, () => UpdateVpcOriginRequest$, () => UpdateVpcOriginResult$
];
var VerifyDnsConfiguration$ = [9, n0, _VDC,
    { [_h]: ["POST", "/2020-05-31/verify-dns-configuration", 200] }, () => VerifyDnsConfigurationRequest$, () => VerifyDnsConfigurationResult$
];

class AssociateAliasCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "AssociateAlias", {})
    .n("CloudFrontClient", "AssociateAliasCommand")
    .sc(AssociateAlias$)
    .build() {
}

class AssociateDistributionTenantWebACLCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "AssociateDistributionTenantWebACL", {})
    .n("CloudFrontClient", "AssociateDistributionTenantWebACLCommand")
    .sc(AssociateDistributionTenantWebACL$)
    .build() {
}

class AssociateDistributionWebACLCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "AssociateDistributionWebACL", {})
    .n("CloudFrontClient", "AssociateDistributionWebACLCommand")
    .sc(AssociateDistributionWebACL$)
    .build() {
}

class CopyDistributionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CopyDistribution", {})
    .n("CloudFrontClient", "CopyDistributionCommand")
    .sc(CopyDistribution$)
    .build() {
}

class CreateAnycastIpListCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CreateAnycastIpList", {})
    .n("CloudFrontClient", "CreateAnycastIpListCommand")
    .sc(CreateAnycastIpList$)
    .build() {
}

class CreateCachePolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CreateCachePolicy", {})
    .n("CloudFrontClient", "CreateCachePolicyCommand")
    .sc(CreateCachePolicy$)
    .build() {
}

class CreateCloudFrontOriginAccessIdentityCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CreateCloudFrontOriginAccessIdentity", {})
    .n("CloudFrontClient", "CreateCloudFrontOriginAccessIdentityCommand")
    .sc(CreateCloudFrontOriginAccessIdentity$)
    .build() {
}

class CreateConnectionFunctionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CreateConnectionFunction", {})
    .n("CloudFrontClient", "CreateConnectionFunctionCommand")
    .sc(CreateConnectionFunction$)
    .build() {
}

class CreateConnectionGroupCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CreateConnectionGroup", {})
    .n("CloudFrontClient", "CreateConnectionGroupCommand")
    .sc(CreateConnectionGroup$)
    .build() {
}

class CreateContinuousDeploymentPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CreateContinuousDeploymentPolicy", {})
    .n("CloudFrontClient", "CreateContinuousDeploymentPolicyCommand")
    .sc(CreateContinuousDeploymentPolicy$)
    .build() {
}

class CreateDistributionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CreateDistribution", {})
    .n("CloudFrontClient", "CreateDistributionCommand")
    .sc(CreateDistribution$)
    .build() {
}

class CreateDistributionTenantCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CreateDistributionTenant", {})
    .n("CloudFrontClient", "CreateDistributionTenantCommand")
    .sc(CreateDistributionTenant$)
    .build() {
}

class CreateDistributionWithTagsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CreateDistributionWithTags", {})
    .n("CloudFrontClient", "CreateDistributionWithTagsCommand")
    .sc(CreateDistributionWithTags$)
    .build() {
}

class CreateFieldLevelEncryptionConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CreateFieldLevelEncryptionConfig", {})
    .n("CloudFrontClient", "CreateFieldLevelEncryptionConfigCommand")
    .sc(CreateFieldLevelEncryptionConfig$)
    .build() {
}

class CreateFieldLevelEncryptionProfileCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CreateFieldLevelEncryptionProfile", {})
    .n("CloudFrontClient", "CreateFieldLevelEncryptionProfileCommand")
    .sc(CreateFieldLevelEncryptionProfile$)
    .build() {
}

class CreateFunctionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CreateFunction", {})
    .n("CloudFrontClient", "CreateFunctionCommand")
    .sc(CreateFunction$)
    .build() {
}

class CreateInvalidationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CreateInvalidation", {})
    .n("CloudFrontClient", "CreateInvalidationCommand")
    .sc(CreateInvalidation$)
    .build() {
}

class CreateInvalidationForDistributionTenantCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CreateInvalidationForDistributionTenant", {})
    .n("CloudFrontClient", "CreateInvalidationForDistributionTenantCommand")
    .sc(CreateInvalidationForDistributionTenant$)
    .build() {
}

class CreateKeyGroupCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CreateKeyGroup", {})
    .n("CloudFrontClient", "CreateKeyGroupCommand")
    .sc(CreateKeyGroup$)
    .build() {
}

class CreateKeyValueStoreCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CreateKeyValueStore", {})
    .n("CloudFrontClient", "CreateKeyValueStoreCommand")
    .sc(CreateKeyValueStore$)
    .build() {
}

class CreateMonitoringSubscriptionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CreateMonitoringSubscription", {})
    .n("CloudFrontClient", "CreateMonitoringSubscriptionCommand")
    .sc(CreateMonitoringSubscription$)
    .build() {
}

class CreateOriginAccessControlCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CreateOriginAccessControl", {})
    .n("CloudFrontClient", "CreateOriginAccessControlCommand")
    .sc(CreateOriginAccessControl$)
    .build() {
}

class CreateOriginRequestPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CreateOriginRequestPolicy", {})
    .n("CloudFrontClient", "CreateOriginRequestPolicyCommand")
    .sc(CreateOriginRequestPolicy$)
    .build() {
}

class CreatePublicKeyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CreatePublicKey", {})
    .n("CloudFrontClient", "CreatePublicKeyCommand")
    .sc(CreatePublicKey$)
    .build() {
}

class CreateRealtimeLogConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CreateRealtimeLogConfig", {})
    .n("CloudFrontClient", "CreateRealtimeLogConfigCommand")
    .sc(CreateRealtimeLogConfig$)
    .build() {
}

class CreateResponseHeadersPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CreateResponseHeadersPolicy", {})
    .n("CloudFrontClient", "CreateResponseHeadersPolicyCommand")
    .sc(CreateResponseHeadersPolicy$)
    .build() {
}

class CreateStreamingDistributionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CreateStreamingDistribution", {})
    .n("CloudFrontClient", "CreateStreamingDistributionCommand")
    .sc(CreateStreamingDistribution$)
    .build() {
}

class CreateStreamingDistributionWithTagsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CreateStreamingDistributionWithTags", {})
    .n("CloudFrontClient", "CreateStreamingDistributionWithTagsCommand")
    .sc(CreateStreamingDistributionWithTags$)
    .build() {
}

class CreateTrustStoreCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CreateTrustStore", {})
    .n("CloudFrontClient", "CreateTrustStoreCommand")
    .sc(CreateTrustStore$)
    .build() {
}

class CreateVpcOriginCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "CreateVpcOrigin", {})
    .n("CloudFrontClient", "CreateVpcOriginCommand")
    .sc(CreateVpcOrigin$)
    .build() {
}

class DeleteAnycastIpListCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DeleteAnycastIpList", {})
    .n("CloudFrontClient", "DeleteAnycastIpListCommand")
    .sc(DeleteAnycastIpList$)
    .build() {
}

class DeleteCachePolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DeleteCachePolicy", {})
    .n("CloudFrontClient", "DeleteCachePolicyCommand")
    .sc(DeleteCachePolicy$)
    .build() {
}

class DeleteCloudFrontOriginAccessIdentityCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DeleteCloudFrontOriginAccessIdentity", {})
    .n("CloudFrontClient", "DeleteCloudFrontOriginAccessIdentityCommand")
    .sc(DeleteCloudFrontOriginAccessIdentity$)
    .build() {
}

class DeleteConnectionFunctionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DeleteConnectionFunction", {})
    .n("CloudFrontClient", "DeleteConnectionFunctionCommand")
    .sc(DeleteConnectionFunction$)
    .build() {
}

class DeleteConnectionGroupCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DeleteConnectionGroup", {})
    .n("CloudFrontClient", "DeleteConnectionGroupCommand")
    .sc(DeleteConnectionGroup$)
    .build() {
}

class DeleteContinuousDeploymentPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DeleteContinuousDeploymentPolicy", {})
    .n("CloudFrontClient", "DeleteContinuousDeploymentPolicyCommand")
    .sc(DeleteContinuousDeploymentPolicy$)
    .build() {
}

class DeleteDistributionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DeleteDistribution", {})
    .n("CloudFrontClient", "DeleteDistributionCommand")
    .sc(DeleteDistribution$)
    .build() {
}

class DeleteDistributionTenantCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DeleteDistributionTenant", {})
    .n("CloudFrontClient", "DeleteDistributionTenantCommand")
    .sc(DeleteDistributionTenant$)
    .build() {
}

class DeleteFieldLevelEncryptionConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DeleteFieldLevelEncryptionConfig", {})
    .n("CloudFrontClient", "DeleteFieldLevelEncryptionConfigCommand")
    .sc(DeleteFieldLevelEncryptionConfig$)
    .build() {
}

class DeleteFieldLevelEncryptionProfileCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DeleteFieldLevelEncryptionProfile", {})
    .n("CloudFrontClient", "DeleteFieldLevelEncryptionProfileCommand")
    .sc(DeleteFieldLevelEncryptionProfile$)
    .build() {
}

class DeleteFunctionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DeleteFunction", {})
    .n("CloudFrontClient", "DeleteFunctionCommand")
    .sc(DeleteFunction$)
    .build() {
}

class DeleteKeyGroupCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DeleteKeyGroup", {})
    .n("CloudFrontClient", "DeleteKeyGroupCommand")
    .sc(DeleteKeyGroup$)
    .build() {
}

class DeleteKeyValueStoreCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DeleteKeyValueStore", {})
    .n("CloudFrontClient", "DeleteKeyValueStoreCommand")
    .sc(DeleteKeyValueStore$)
    .build() {
}

class DeleteMonitoringSubscriptionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DeleteMonitoringSubscription", {})
    .n("CloudFrontClient", "DeleteMonitoringSubscriptionCommand")
    .sc(DeleteMonitoringSubscription$)
    .build() {
}

class DeleteOriginAccessControlCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DeleteOriginAccessControl", {})
    .n("CloudFrontClient", "DeleteOriginAccessControlCommand")
    .sc(DeleteOriginAccessControl$)
    .build() {
}

class DeleteOriginRequestPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DeleteOriginRequestPolicy", {})
    .n("CloudFrontClient", "DeleteOriginRequestPolicyCommand")
    .sc(DeleteOriginRequestPolicy$)
    .build() {
}

class DeletePublicKeyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DeletePublicKey", {})
    .n("CloudFrontClient", "DeletePublicKeyCommand")
    .sc(DeletePublicKey$)
    .build() {
}

class DeleteRealtimeLogConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DeleteRealtimeLogConfig", {})
    .n("CloudFrontClient", "DeleteRealtimeLogConfigCommand")
    .sc(DeleteRealtimeLogConfig$)
    .build() {
}

class DeleteResourcePolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DeleteResourcePolicy", {})
    .n("CloudFrontClient", "DeleteResourcePolicyCommand")
    .sc(DeleteResourcePolicy$)
    .build() {
}

class DeleteResponseHeadersPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DeleteResponseHeadersPolicy", {})
    .n("CloudFrontClient", "DeleteResponseHeadersPolicyCommand")
    .sc(DeleteResponseHeadersPolicy$)
    .build() {
}

class DeleteStreamingDistributionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DeleteStreamingDistribution", {})
    .n("CloudFrontClient", "DeleteStreamingDistributionCommand")
    .sc(DeleteStreamingDistribution$)
    .build() {
}

class DeleteTrustStoreCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DeleteTrustStore", {})
    .n("CloudFrontClient", "DeleteTrustStoreCommand")
    .sc(DeleteTrustStore$)
    .build() {
}

class DeleteVpcOriginCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DeleteVpcOrigin", {})
    .n("CloudFrontClient", "DeleteVpcOriginCommand")
    .sc(DeleteVpcOrigin$)
    .build() {
}

class DescribeConnectionFunctionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DescribeConnectionFunction", {})
    .n("CloudFrontClient", "DescribeConnectionFunctionCommand")
    .sc(DescribeConnectionFunction$)
    .build() {
}

class DescribeFunctionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DescribeFunction", {})
    .n("CloudFrontClient", "DescribeFunctionCommand")
    .sc(DescribeFunction$)
    .build() {
}

class DescribeKeyValueStoreCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DescribeKeyValueStore", {})
    .n("CloudFrontClient", "DescribeKeyValueStoreCommand")
    .sc(DescribeKeyValueStore$)
    .build() {
}

class DisassociateDistributionTenantWebACLCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DisassociateDistributionTenantWebACL", {})
    .n("CloudFrontClient", "DisassociateDistributionTenantWebACLCommand")
    .sc(DisassociateDistributionTenantWebACL$)
    .build() {
}

class DisassociateDistributionWebACLCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "DisassociateDistributionWebACL", {})
    .n("CloudFrontClient", "DisassociateDistributionWebACLCommand")
    .sc(DisassociateDistributionWebACL$)
    .build() {
}

class GetAnycastIpListCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetAnycastIpList", {})
    .n("CloudFrontClient", "GetAnycastIpListCommand")
    .sc(GetAnycastIpList$)
    .build() {
}

class GetCachePolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetCachePolicy", {})
    .n("CloudFrontClient", "GetCachePolicyCommand")
    .sc(GetCachePolicy$)
    .build() {
}

class GetCachePolicyConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetCachePolicyConfig", {})
    .n("CloudFrontClient", "GetCachePolicyConfigCommand")
    .sc(GetCachePolicyConfig$)
    .build() {
}

class GetCloudFrontOriginAccessIdentityCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetCloudFrontOriginAccessIdentity", {})
    .n("CloudFrontClient", "GetCloudFrontOriginAccessIdentityCommand")
    .sc(GetCloudFrontOriginAccessIdentity$)
    .build() {
}

class GetCloudFrontOriginAccessIdentityConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetCloudFrontOriginAccessIdentityConfig", {})
    .n("CloudFrontClient", "GetCloudFrontOriginAccessIdentityConfigCommand")
    .sc(GetCloudFrontOriginAccessIdentityConfig$)
    .build() {
}

class GetConnectionFunctionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetConnectionFunction", {})
    .n("CloudFrontClient", "GetConnectionFunctionCommand")
    .sc(GetConnectionFunction$)
    .build() {
}

class GetConnectionGroupByRoutingEndpointCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetConnectionGroupByRoutingEndpoint", {})
    .n("CloudFrontClient", "GetConnectionGroupByRoutingEndpointCommand")
    .sc(GetConnectionGroupByRoutingEndpoint$)
    .build() {
}

class GetConnectionGroupCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetConnectionGroup", {})
    .n("CloudFrontClient", "GetConnectionGroupCommand")
    .sc(GetConnectionGroup$)
    .build() {
}

class GetContinuousDeploymentPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetContinuousDeploymentPolicy", {})
    .n("CloudFrontClient", "GetContinuousDeploymentPolicyCommand")
    .sc(GetContinuousDeploymentPolicy$)
    .build() {
}

class GetContinuousDeploymentPolicyConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetContinuousDeploymentPolicyConfig", {})
    .n("CloudFrontClient", "GetContinuousDeploymentPolicyConfigCommand")
    .sc(GetContinuousDeploymentPolicyConfig$)
    .build() {
}

class GetDistributionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetDistribution", {})
    .n("CloudFrontClient", "GetDistributionCommand")
    .sc(GetDistribution$)
    .build() {
}

class GetDistributionConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetDistributionConfig", {})
    .n("CloudFrontClient", "GetDistributionConfigCommand")
    .sc(GetDistributionConfig$)
    .build() {
}

class GetDistributionTenantByDomainCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetDistributionTenantByDomain", {})
    .n("CloudFrontClient", "GetDistributionTenantByDomainCommand")
    .sc(GetDistributionTenantByDomain$)
    .build() {
}

class GetDistributionTenantCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetDistributionTenant", {})
    .n("CloudFrontClient", "GetDistributionTenantCommand")
    .sc(GetDistributionTenant$)
    .build() {
}

class GetFieldLevelEncryptionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetFieldLevelEncryption", {})
    .n("CloudFrontClient", "GetFieldLevelEncryptionCommand")
    .sc(GetFieldLevelEncryption$)
    .build() {
}

class GetFieldLevelEncryptionConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetFieldLevelEncryptionConfig", {})
    .n("CloudFrontClient", "GetFieldLevelEncryptionConfigCommand")
    .sc(GetFieldLevelEncryptionConfig$)
    .build() {
}

class GetFieldLevelEncryptionProfileCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetFieldLevelEncryptionProfile", {})
    .n("CloudFrontClient", "GetFieldLevelEncryptionProfileCommand")
    .sc(GetFieldLevelEncryptionProfile$)
    .build() {
}

class GetFieldLevelEncryptionProfileConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetFieldLevelEncryptionProfileConfig", {})
    .n("CloudFrontClient", "GetFieldLevelEncryptionProfileConfigCommand")
    .sc(GetFieldLevelEncryptionProfileConfig$)
    .build() {
}

class GetFunctionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetFunction", {})
    .n("CloudFrontClient", "GetFunctionCommand")
    .sc(GetFunction$)
    .build() {
}

class GetInvalidationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetInvalidation", {})
    .n("CloudFrontClient", "GetInvalidationCommand")
    .sc(GetInvalidation$)
    .build() {
}

class GetInvalidationForDistributionTenantCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetInvalidationForDistributionTenant", {})
    .n("CloudFrontClient", "GetInvalidationForDistributionTenantCommand")
    .sc(GetInvalidationForDistributionTenant$)
    .build() {
}

class GetKeyGroupCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetKeyGroup", {})
    .n("CloudFrontClient", "GetKeyGroupCommand")
    .sc(GetKeyGroup$)
    .build() {
}

class GetKeyGroupConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetKeyGroupConfig", {})
    .n("CloudFrontClient", "GetKeyGroupConfigCommand")
    .sc(GetKeyGroupConfig$)
    .build() {
}

class GetManagedCertificateDetailsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetManagedCertificateDetails", {})
    .n("CloudFrontClient", "GetManagedCertificateDetailsCommand")
    .sc(GetManagedCertificateDetails$)
    .build() {
}

class GetMonitoringSubscriptionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetMonitoringSubscription", {})
    .n("CloudFrontClient", "GetMonitoringSubscriptionCommand")
    .sc(GetMonitoringSubscription$)
    .build() {
}

class GetOriginAccessControlCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetOriginAccessControl", {})
    .n("CloudFrontClient", "GetOriginAccessControlCommand")
    .sc(GetOriginAccessControl$)
    .build() {
}

class GetOriginAccessControlConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetOriginAccessControlConfig", {})
    .n("CloudFrontClient", "GetOriginAccessControlConfigCommand")
    .sc(GetOriginAccessControlConfig$)
    .build() {
}

class GetOriginRequestPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetOriginRequestPolicy", {})
    .n("CloudFrontClient", "GetOriginRequestPolicyCommand")
    .sc(GetOriginRequestPolicy$)
    .build() {
}

class GetOriginRequestPolicyConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetOriginRequestPolicyConfig", {})
    .n("CloudFrontClient", "GetOriginRequestPolicyConfigCommand")
    .sc(GetOriginRequestPolicyConfig$)
    .build() {
}

class GetPublicKeyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetPublicKey", {})
    .n("CloudFrontClient", "GetPublicKeyCommand")
    .sc(GetPublicKey$)
    .build() {
}

class GetPublicKeyConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetPublicKeyConfig", {})
    .n("CloudFrontClient", "GetPublicKeyConfigCommand")
    .sc(GetPublicKeyConfig$)
    .build() {
}

class GetRealtimeLogConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetRealtimeLogConfig", {})
    .n("CloudFrontClient", "GetRealtimeLogConfigCommand")
    .sc(GetRealtimeLogConfig$)
    .build() {
}

class GetResourcePolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetResourcePolicy", {})
    .n("CloudFrontClient", "GetResourcePolicyCommand")
    .sc(GetResourcePolicy$)
    .build() {
}

class GetResponseHeadersPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetResponseHeadersPolicy", {})
    .n("CloudFrontClient", "GetResponseHeadersPolicyCommand")
    .sc(GetResponseHeadersPolicy$)
    .build() {
}

class GetResponseHeadersPolicyConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetResponseHeadersPolicyConfig", {})
    .n("CloudFrontClient", "GetResponseHeadersPolicyConfigCommand")
    .sc(GetResponseHeadersPolicyConfig$)
    .build() {
}

class GetStreamingDistributionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetStreamingDistribution", {})
    .n("CloudFrontClient", "GetStreamingDistributionCommand")
    .sc(GetStreamingDistribution$)
    .build() {
}

class GetStreamingDistributionConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetStreamingDistributionConfig", {})
    .n("CloudFrontClient", "GetStreamingDistributionConfigCommand")
    .sc(GetStreamingDistributionConfig$)
    .build() {
}

class GetTrustStoreCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetTrustStore", {})
    .n("CloudFrontClient", "GetTrustStoreCommand")
    .sc(GetTrustStore$)
    .build() {
}

class GetVpcOriginCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "GetVpcOrigin", {})
    .n("CloudFrontClient", "GetVpcOriginCommand")
    .sc(GetVpcOrigin$)
    .build() {
}

class ListAnycastIpListsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListAnycastIpLists", {})
    .n("CloudFrontClient", "ListAnycastIpListsCommand")
    .sc(ListAnycastIpLists$)
    .build() {
}

class ListCachePoliciesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListCachePolicies", {})
    .n("CloudFrontClient", "ListCachePoliciesCommand")
    .sc(ListCachePolicies$)
    .build() {
}

class ListCloudFrontOriginAccessIdentitiesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListCloudFrontOriginAccessIdentities", {})
    .n("CloudFrontClient", "ListCloudFrontOriginAccessIdentitiesCommand")
    .sc(ListCloudFrontOriginAccessIdentities$)
    .build() {
}

class ListConflictingAliasesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListConflictingAliases", {})
    .n("CloudFrontClient", "ListConflictingAliasesCommand")
    .sc(ListConflictingAliases$)
    .build() {
}

class ListConnectionFunctionsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListConnectionFunctions", {})
    .n("CloudFrontClient", "ListConnectionFunctionsCommand")
    .sc(ListConnectionFunctions$)
    .build() {
}

class ListConnectionGroupsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListConnectionGroups", {})
    .n("CloudFrontClient", "ListConnectionGroupsCommand")
    .sc(ListConnectionGroups$)
    .build() {
}

class ListContinuousDeploymentPoliciesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListContinuousDeploymentPolicies", {})
    .n("CloudFrontClient", "ListContinuousDeploymentPoliciesCommand")
    .sc(ListContinuousDeploymentPolicies$)
    .build() {
}

class ListDistributionsByAnycastIpListIdCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListDistributionsByAnycastIpListId", {})
    .n("CloudFrontClient", "ListDistributionsByAnycastIpListIdCommand")
    .sc(ListDistributionsByAnycastIpListId$)
    .build() {
}

class ListDistributionsByCachePolicyIdCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListDistributionsByCachePolicyId", {})
    .n("CloudFrontClient", "ListDistributionsByCachePolicyIdCommand")
    .sc(ListDistributionsByCachePolicyId$)
    .build() {
}

class ListDistributionsByConnectionFunctionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListDistributionsByConnectionFunction", {})
    .n("CloudFrontClient", "ListDistributionsByConnectionFunctionCommand")
    .sc(ListDistributionsByConnectionFunction$)
    .build() {
}

class ListDistributionsByConnectionModeCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListDistributionsByConnectionMode", {})
    .n("CloudFrontClient", "ListDistributionsByConnectionModeCommand")
    .sc(ListDistributionsByConnectionMode$)
    .build() {
}

class ListDistributionsByKeyGroupCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListDistributionsByKeyGroup", {})
    .n("CloudFrontClient", "ListDistributionsByKeyGroupCommand")
    .sc(ListDistributionsByKeyGroup$)
    .build() {
}

class ListDistributionsByOriginRequestPolicyIdCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListDistributionsByOriginRequestPolicyId", {})
    .n("CloudFrontClient", "ListDistributionsByOriginRequestPolicyIdCommand")
    .sc(ListDistributionsByOriginRequestPolicyId$)
    .build() {
}

class ListDistributionsByOwnedResourceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListDistributionsByOwnedResource", {})
    .n("CloudFrontClient", "ListDistributionsByOwnedResourceCommand")
    .sc(ListDistributionsByOwnedResource$)
    .build() {
}

class ListDistributionsByRealtimeLogConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListDistributionsByRealtimeLogConfig", {})
    .n("CloudFrontClient", "ListDistributionsByRealtimeLogConfigCommand")
    .sc(ListDistributionsByRealtimeLogConfig$)
    .build() {
}

class ListDistributionsByResponseHeadersPolicyIdCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListDistributionsByResponseHeadersPolicyId", {})
    .n("CloudFrontClient", "ListDistributionsByResponseHeadersPolicyIdCommand")
    .sc(ListDistributionsByResponseHeadersPolicyId$)
    .build() {
}

class ListDistributionsByTrustStoreCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListDistributionsByTrustStore", {})
    .n("CloudFrontClient", "ListDistributionsByTrustStoreCommand")
    .sc(ListDistributionsByTrustStore$)
    .build() {
}

class ListDistributionsByVpcOriginIdCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListDistributionsByVpcOriginId", {})
    .n("CloudFrontClient", "ListDistributionsByVpcOriginIdCommand")
    .sc(ListDistributionsByVpcOriginId$)
    .build() {
}

class ListDistributionsByWebACLIdCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListDistributionsByWebACLId", {})
    .n("CloudFrontClient", "ListDistributionsByWebACLIdCommand")
    .sc(ListDistributionsByWebACLId$)
    .build() {
}

class ListDistributionsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListDistributions", {})
    .n("CloudFrontClient", "ListDistributionsCommand")
    .sc(ListDistributions$)
    .build() {
}

class ListDistributionTenantsByCustomizationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListDistributionTenantsByCustomization", {})
    .n("CloudFrontClient", "ListDistributionTenantsByCustomizationCommand")
    .sc(ListDistributionTenantsByCustomization$)
    .build() {
}

class ListDistributionTenantsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListDistributionTenants", {})
    .n("CloudFrontClient", "ListDistributionTenantsCommand")
    .sc(ListDistributionTenants$)
    .build() {
}

class ListDomainConflictsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListDomainConflicts", {})
    .n("CloudFrontClient", "ListDomainConflictsCommand")
    .sc(ListDomainConflicts$)
    .build() {
}

class ListFieldLevelEncryptionConfigsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListFieldLevelEncryptionConfigs", {})
    .n("CloudFrontClient", "ListFieldLevelEncryptionConfigsCommand")
    .sc(ListFieldLevelEncryptionConfigs$)
    .build() {
}

class ListFieldLevelEncryptionProfilesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListFieldLevelEncryptionProfiles", {})
    .n("CloudFrontClient", "ListFieldLevelEncryptionProfilesCommand")
    .sc(ListFieldLevelEncryptionProfiles$)
    .build() {
}

class ListFunctionsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListFunctions", {})
    .n("CloudFrontClient", "ListFunctionsCommand")
    .sc(ListFunctions$)
    .build() {
}

class ListInvalidationsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListInvalidations", {})
    .n("CloudFrontClient", "ListInvalidationsCommand")
    .sc(ListInvalidations$)
    .build() {
}

class ListInvalidationsForDistributionTenantCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListInvalidationsForDistributionTenant", {})
    .n("CloudFrontClient", "ListInvalidationsForDistributionTenantCommand")
    .sc(ListInvalidationsForDistributionTenant$)
    .build() {
}

class ListKeyGroupsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListKeyGroups", {})
    .n("CloudFrontClient", "ListKeyGroupsCommand")
    .sc(ListKeyGroups$)
    .build() {
}

class ListKeyValueStoresCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListKeyValueStores", {})
    .n("CloudFrontClient", "ListKeyValueStoresCommand")
    .sc(ListKeyValueStores$)
    .build() {
}

class ListOriginAccessControlsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListOriginAccessControls", {})
    .n("CloudFrontClient", "ListOriginAccessControlsCommand")
    .sc(ListOriginAccessControls$)
    .build() {
}

class ListOriginRequestPoliciesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListOriginRequestPolicies", {})
    .n("CloudFrontClient", "ListOriginRequestPoliciesCommand")
    .sc(ListOriginRequestPolicies$)
    .build() {
}

class ListPublicKeysCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListPublicKeys", {})
    .n("CloudFrontClient", "ListPublicKeysCommand")
    .sc(ListPublicKeys$)
    .build() {
}

class ListRealtimeLogConfigsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListRealtimeLogConfigs", {})
    .n("CloudFrontClient", "ListRealtimeLogConfigsCommand")
    .sc(ListRealtimeLogConfigs$)
    .build() {
}

class ListResponseHeadersPoliciesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListResponseHeadersPolicies", {})
    .n("CloudFrontClient", "ListResponseHeadersPoliciesCommand")
    .sc(ListResponseHeadersPolicies$)
    .build() {
}

class ListStreamingDistributionsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListStreamingDistributions", {})
    .n("CloudFrontClient", "ListStreamingDistributionsCommand")
    .sc(ListStreamingDistributions$)
    .build() {
}

class ListTagsForResourceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListTagsForResource", {})
    .n("CloudFrontClient", "ListTagsForResourceCommand")
    .sc(ListTagsForResource$)
    .build() {
}

class ListTrustStoresCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListTrustStores", {})
    .n("CloudFrontClient", "ListTrustStoresCommand")
    .sc(ListTrustStores$)
    .build() {
}

class ListVpcOriginsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "ListVpcOrigins", {})
    .n("CloudFrontClient", "ListVpcOriginsCommand")
    .sc(ListVpcOrigins$)
    .build() {
}

class PublishConnectionFunctionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "PublishConnectionFunction", {})
    .n("CloudFrontClient", "PublishConnectionFunctionCommand")
    .sc(PublishConnectionFunction$)
    .build() {
}

class PublishFunctionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "PublishFunction", {})
    .n("CloudFrontClient", "PublishFunctionCommand")
    .sc(PublishFunction$)
    .build() {
}

class PutResourcePolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "PutResourcePolicy", {})
    .n("CloudFrontClient", "PutResourcePolicyCommand")
    .sc(PutResourcePolicy$)
    .build() {
}

class TagResourceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "TagResource", {})
    .n("CloudFrontClient", "TagResourceCommand")
    .sc(TagResource$)
    .build() {
}

class TestConnectionFunctionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "TestConnectionFunction", {})
    .n("CloudFrontClient", "TestConnectionFunctionCommand")
    .sc(TestConnectionFunction$)
    .build() {
}

class TestFunctionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "TestFunction", {})
    .n("CloudFrontClient", "TestFunctionCommand")
    .sc(TestFunction$)
    .build() {
}

class UntagResourceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "UntagResource", {})
    .n("CloudFrontClient", "UntagResourceCommand")
    .sc(UntagResource$)
    .build() {
}

class UpdateAnycastIpListCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "UpdateAnycastIpList", {})
    .n("CloudFrontClient", "UpdateAnycastIpListCommand")
    .sc(UpdateAnycastIpList$)
    .build() {
}

class UpdateCachePolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "UpdateCachePolicy", {})
    .n("CloudFrontClient", "UpdateCachePolicyCommand")
    .sc(UpdateCachePolicy$)
    .build() {
}

class UpdateCloudFrontOriginAccessIdentityCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "UpdateCloudFrontOriginAccessIdentity", {})
    .n("CloudFrontClient", "UpdateCloudFrontOriginAccessIdentityCommand")
    .sc(UpdateCloudFrontOriginAccessIdentity$)
    .build() {
}

class UpdateConnectionFunctionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "UpdateConnectionFunction", {})
    .n("CloudFrontClient", "UpdateConnectionFunctionCommand")
    .sc(UpdateConnectionFunction$)
    .build() {
}

class UpdateConnectionGroupCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "UpdateConnectionGroup", {})
    .n("CloudFrontClient", "UpdateConnectionGroupCommand")
    .sc(UpdateConnectionGroup$)
    .build() {
}

class UpdateContinuousDeploymentPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "UpdateContinuousDeploymentPolicy", {})
    .n("CloudFrontClient", "UpdateContinuousDeploymentPolicyCommand")
    .sc(UpdateContinuousDeploymentPolicy$)
    .build() {
}

class UpdateDistributionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "UpdateDistribution", {})
    .n("CloudFrontClient", "UpdateDistributionCommand")
    .sc(UpdateDistribution$)
    .build() {
}

class UpdateDistributionTenantCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "UpdateDistributionTenant", {})
    .n("CloudFrontClient", "UpdateDistributionTenantCommand")
    .sc(UpdateDistributionTenant$)
    .build() {
}

class UpdateDistributionWithStagingConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "UpdateDistributionWithStagingConfig", {})
    .n("CloudFrontClient", "UpdateDistributionWithStagingConfigCommand")
    .sc(UpdateDistributionWithStagingConfig$)
    .build() {
}

class UpdateDomainAssociationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "UpdateDomainAssociation", {})
    .n("CloudFrontClient", "UpdateDomainAssociationCommand")
    .sc(UpdateDomainAssociation$)
    .build() {
}

class UpdateFieldLevelEncryptionConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "UpdateFieldLevelEncryptionConfig", {})
    .n("CloudFrontClient", "UpdateFieldLevelEncryptionConfigCommand")
    .sc(UpdateFieldLevelEncryptionConfig$)
    .build() {
}

class UpdateFieldLevelEncryptionProfileCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "UpdateFieldLevelEncryptionProfile", {})
    .n("CloudFrontClient", "UpdateFieldLevelEncryptionProfileCommand")
    .sc(UpdateFieldLevelEncryptionProfile$)
    .build() {
}

class UpdateFunctionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "UpdateFunction", {})
    .n("CloudFrontClient", "UpdateFunctionCommand")
    .sc(UpdateFunction$)
    .build() {
}

class UpdateKeyGroupCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "UpdateKeyGroup", {})
    .n("CloudFrontClient", "UpdateKeyGroupCommand")
    .sc(UpdateKeyGroup$)
    .build() {
}

class UpdateKeyValueStoreCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "UpdateKeyValueStore", {})
    .n("CloudFrontClient", "UpdateKeyValueStoreCommand")
    .sc(UpdateKeyValueStore$)
    .build() {
}

class UpdateOriginAccessControlCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "UpdateOriginAccessControl", {})
    .n("CloudFrontClient", "UpdateOriginAccessControlCommand")
    .sc(UpdateOriginAccessControl$)
    .build() {
}

class UpdateOriginRequestPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "UpdateOriginRequestPolicy", {})
    .n("CloudFrontClient", "UpdateOriginRequestPolicyCommand")
    .sc(UpdateOriginRequestPolicy$)
    .build() {
}

class UpdatePublicKeyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "UpdatePublicKey", {})
    .n("CloudFrontClient", "UpdatePublicKeyCommand")
    .sc(UpdatePublicKey$)
    .build() {
}

class UpdateRealtimeLogConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "UpdateRealtimeLogConfig", {})
    .n("CloudFrontClient", "UpdateRealtimeLogConfigCommand")
    .sc(UpdateRealtimeLogConfig$)
    .build() {
}

class UpdateResponseHeadersPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "UpdateResponseHeadersPolicy", {})
    .n("CloudFrontClient", "UpdateResponseHeadersPolicyCommand")
    .sc(UpdateResponseHeadersPolicy$)
    .build() {
}

class UpdateStreamingDistributionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "UpdateStreamingDistribution", {})
    .n("CloudFrontClient", "UpdateStreamingDistributionCommand")
    .sc(UpdateStreamingDistribution$)
    .build() {
}

class UpdateTrustStoreCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "UpdateTrustStore", {})
    .n("CloudFrontClient", "UpdateTrustStoreCommand")
    .sc(UpdateTrustStore$)
    .build() {
}

class UpdateVpcOriginCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "UpdateVpcOrigin", {})
    .n("CloudFrontClient", "UpdateVpcOriginCommand")
    .sc(UpdateVpcOrigin$)
    .build() {
}

class VerifyDnsConfigurationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("Cloudfront2020_05_31", "VerifyDnsConfiguration", {})
    .n("CloudFrontClient", "VerifyDnsConfigurationCommand")
    .sc(VerifyDnsConfiguration$)
    .build() {
}

const commands = {
    AssociateAliasCommand,
    AssociateDistributionTenantWebACLCommand,
    AssociateDistributionWebACLCommand,
    CopyDistributionCommand,
    CreateAnycastIpListCommand,
    CreateCachePolicyCommand,
    CreateCloudFrontOriginAccessIdentityCommand,
    CreateConnectionFunctionCommand,
    CreateConnectionGroupCommand,
    CreateContinuousDeploymentPolicyCommand,
    CreateDistributionCommand,
    CreateDistributionTenantCommand,
    CreateDistributionWithTagsCommand,
    CreateFieldLevelEncryptionConfigCommand,
    CreateFieldLevelEncryptionProfileCommand,
    CreateFunctionCommand,
    CreateInvalidationCommand,
    CreateInvalidationForDistributionTenantCommand,
    CreateKeyGroupCommand,
    CreateKeyValueStoreCommand,
    CreateMonitoringSubscriptionCommand,
    CreateOriginAccessControlCommand,
    CreateOriginRequestPolicyCommand,
    CreatePublicKeyCommand,
    CreateRealtimeLogConfigCommand,
    CreateResponseHeadersPolicyCommand,
    CreateStreamingDistributionCommand,
    CreateStreamingDistributionWithTagsCommand,
    CreateTrustStoreCommand,
    CreateVpcOriginCommand,
    DeleteAnycastIpListCommand,
    DeleteCachePolicyCommand,
    DeleteCloudFrontOriginAccessIdentityCommand,
    DeleteConnectionFunctionCommand,
    DeleteConnectionGroupCommand,
    DeleteContinuousDeploymentPolicyCommand,
    DeleteDistributionCommand,
    DeleteDistributionTenantCommand,
    DeleteFieldLevelEncryptionConfigCommand,
    DeleteFieldLevelEncryptionProfileCommand,
    DeleteFunctionCommand,
    DeleteKeyGroupCommand,
    DeleteKeyValueStoreCommand,
    DeleteMonitoringSubscriptionCommand,
    DeleteOriginAccessControlCommand,
    DeleteOriginRequestPolicyCommand,
    DeletePublicKeyCommand,
    DeleteRealtimeLogConfigCommand,
    DeleteResourcePolicyCommand,
    DeleteResponseHeadersPolicyCommand,
    DeleteStreamingDistributionCommand,
    DeleteTrustStoreCommand,
    DeleteVpcOriginCommand,
    DescribeConnectionFunctionCommand,
    DescribeFunctionCommand,
    DescribeKeyValueStoreCommand,
    DisassociateDistributionTenantWebACLCommand,
    DisassociateDistributionWebACLCommand,
    GetAnycastIpListCommand,
    GetCachePolicyCommand,
    GetCachePolicyConfigCommand,
    GetCloudFrontOriginAccessIdentityCommand,
    GetCloudFrontOriginAccessIdentityConfigCommand,
    GetConnectionFunctionCommand,
    GetConnectionGroupCommand,
    GetConnectionGroupByRoutingEndpointCommand,
    GetContinuousDeploymentPolicyCommand,
    GetContinuousDeploymentPolicyConfigCommand,
    GetDistributionCommand,
    GetDistributionConfigCommand,
    GetDistributionTenantCommand,
    GetDistributionTenantByDomainCommand,
    GetFieldLevelEncryptionCommand,
    GetFieldLevelEncryptionConfigCommand,
    GetFieldLevelEncryptionProfileCommand,
    GetFieldLevelEncryptionProfileConfigCommand,
    GetFunctionCommand,
    GetInvalidationCommand,
    GetInvalidationForDistributionTenantCommand,
    GetKeyGroupCommand,
    GetKeyGroupConfigCommand,
    GetManagedCertificateDetailsCommand,
    GetMonitoringSubscriptionCommand,
    GetOriginAccessControlCommand,
    GetOriginAccessControlConfigCommand,
    GetOriginRequestPolicyCommand,
    GetOriginRequestPolicyConfigCommand,
    GetPublicKeyCommand,
    GetPublicKeyConfigCommand,
    GetRealtimeLogConfigCommand,
    GetResourcePolicyCommand,
    GetResponseHeadersPolicyCommand,
    GetResponseHeadersPolicyConfigCommand,
    GetStreamingDistributionCommand,
    GetStreamingDistributionConfigCommand,
    GetTrustStoreCommand,
    GetVpcOriginCommand,
    ListAnycastIpListsCommand,
    ListCachePoliciesCommand,
    ListCloudFrontOriginAccessIdentitiesCommand,
    ListConflictingAliasesCommand,
    ListConnectionFunctionsCommand,
    ListConnectionGroupsCommand,
    ListContinuousDeploymentPoliciesCommand,
    ListDistributionsCommand,
    ListDistributionsByAnycastIpListIdCommand,
    ListDistributionsByCachePolicyIdCommand,
    ListDistributionsByConnectionFunctionCommand,
    ListDistributionsByConnectionModeCommand,
    ListDistributionsByKeyGroupCommand,
    ListDistributionsByOriginRequestPolicyIdCommand,
    ListDistributionsByOwnedResourceCommand,
    ListDistributionsByRealtimeLogConfigCommand,
    ListDistributionsByResponseHeadersPolicyIdCommand,
    ListDistributionsByTrustStoreCommand,
    ListDistributionsByVpcOriginIdCommand,
    ListDistributionsByWebACLIdCommand,
    ListDistributionTenantsCommand,
    ListDistributionTenantsByCustomizationCommand,
    ListDomainConflictsCommand,
    ListFieldLevelEncryptionConfigsCommand,
    ListFieldLevelEncryptionProfilesCommand,
    ListFunctionsCommand,
    ListInvalidationsCommand,
    ListInvalidationsForDistributionTenantCommand,
    ListKeyGroupsCommand,
    ListKeyValueStoresCommand,
    ListOriginAccessControlsCommand,
    ListOriginRequestPoliciesCommand,
    ListPublicKeysCommand,
    ListRealtimeLogConfigsCommand,
    ListResponseHeadersPoliciesCommand,
    ListStreamingDistributionsCommand,
    ListTagsForResourceCommand,
    ListTrustStoresCommand,
    ListVpcOriginsCommand,
    PublishConnectionFunctionCommand,
    PublishFunctionCommand,
    PutResourcePolicyCommand,
    TagResourceCommand,
    TestConnectionFunctionCommand,
    TestFunctionCommand,
    UntagResourceCommand,
    UpdateAnycastIpListCommand,
    UpdateCachePolicyCommand,
    UpdateCloudFrontOriginAccessIdentityCommand,
    UpdateConnectionFunctionCommand,
    UpdateConnectionGroupCommand,
    UpdateContinuousDeploymentPolicyCommand,
    UpdateDistributionCommand,
    UpdateDistributionTenantCommand,
    UpdateDistributionWithStagingConfigCommand,
    UpdateDomainAssociationCommand,
    UpdateFieldLevelEncryptionConfigCommand,
    UpdateFieldLevelEncryptionProfileCommand,
    UpdateFunctionCommand,
    UpdateKeyGroupCommand,
    UpdateKeyValueStoreCommand,
    UpdateOriginAccessControlCommand,
    UpdateOriginRequestPolicyCommand,
    UpdatePublicKeyCommand,
    UpdateRealtimeLogConfigCommand,
    UpdateResponseHeadersPolicyCommand,
    UpdateStreamingDistributionCommand,
    UpdateTrustStoreCommand,
    UpdateVpcOriginCommand,
    VerifyDnsConfigurationCommand,
};
class CloudFront extends CloudFrontClient {
}
smithyClient.createAggregatedClient(commands, CloudFront);

const paginateListCloudFrontOriginAccessIdentities = core.createPaginator(CloudFrontClient, ListCloudFrontOriginAccessIdentitiesCommand, "Marker", "CloudFrontOriginAccessIdentityList.NextMarker", "MaxItems");

const paginateListConnectionFunctions = core.createPaginator(CloudFrontClient, ListConnectionFunctionsCommand, "Marker", "NextMarker", "MaxItems");

const paginateListConnectionGroups = core.createPaginator(CloudFrontClient, ListConnectionGroupsCommand, "Marker", "NextMarker", "MaxItems");

const paginateListDistributions = core.createPaginator(CloudFrontClient, ListDistributionsCommand, "Marker", "DistributionList.NextMarker", "MaxItems");

const paginateListDistributionsByConnectionFunction = core.createPaginator(CloudFrontClient, ListDistributionsByConnectionFunctionCommand, "Marker", "DistributionList.NextMarker", "MaxItems");

const paginateListDistributionsByConnectionMode = core.createPaginator(CloudFrontClient, ListDistributionsByConnectionModeCommand, "Marker", "DistributionList.NextMarker", "MaxItems");

const paginateListDistributionsByTrustStore = core.createPaginator(CloudFrontClient, ListDistributionsByTrustStoreCommand, "Marker", "DistributionList.NextMarker", "MaxItems");

const paginateListDistributionTenants = core.createPaginator(CloudFrontClient, ListDistributionTenantsCommand, "Marker", "NextMarker", "MaxItems");

const paginateListDistributionTenantsByCustomization = core.createPaginator(CloudFrontClient, ListDistributionTenantsByCustomizationCommand, "Marker", "NextMarker", "MaxItems");

const paginateListDomainConflicts = core.createPaginator(CloudFrontClient, ListDomainConflictsCommand, "Marker", "NextMarker", "MaxItems");

const paginateListInvalidations = core.createPaginator(CloudFrontClient, ListInvalidationsCommand, "Marker", "InvalidationList.NextMarker", "MaxItems");

const paginateListInvalidationsForDistributionTenant = core.createPaginator(CloudFrontClient, ListInvalidationsForDistributionTenantCommand, "Marker", "InvalidationList.NextMarker", "MaxItems");

const paginateListKeyValueStores = core.createPaginator(CloudFrontClient, ListKeyValueStoresCommand, "Marker", "KeyValueStoreList.NextMarker", "MaxItems");

const paginateListOriginAccessControls = core.createPaginator(CloudFrontClient, ListOriginAccessControlsCommand, "Marker", "OriginAccessControlList.NextMarker", "MaxItems");

const paginateListPublicKeys = core.createPaginator(CloudFrontClient, ListPublicKeysCommand, "Marker", "PublicKeyList.NextMarker", "MaxItems");

const paginateListStreamingDistributions = core.createPaginator(CloudFrontClient, ListStreamingDistributionsCommand, "Marker", "StreamingDistributionList.NextMarker", "MaxItems");

const paginateListTrustStores = core.createPaginator(CloudFrontClient, ListTrustStoresCommand, "Marker", "NextMarker", "MaxItems");

const checkState$3 = async (client, input) => {
    let reason;
    try {
        let result = await client.send(new GetDistributionCommand(input));
        reason = result;
        try {
            const returnComparator = () => {
                return result.Distribution.Status;
            };
            if (returnComparator() === "Deployed") {
                return { state: utilWaiter.WaiterState.SUCCESS, reason };
            }
        }
        catch (e) { }
    }
    catch (exception) {
        reason = exception;
    }
    return { state: utilWaiter.WaiterState.RETRY, reason };
};
const waitForDistributionDeployed = async (params, input) => {
    const serviceDefaults = { minDelay: 60, maxDelay: 2100 };
    return utilWaiter.createWaiter({ ...serviceDefaults, ...params }, input, checkState$3);
};
const waitUntilDistributionDeployed = async (params, input) => {
    const serviceDefaults = { minDelay: 60, maxDelay: 2100 };
    const result = await utilWaiter.createWaiter({ ...serviceDefaults, ...params }, input, checkState$3);
    return utilWaiter.checkExceptions(result);
};

const checkState$2 = async (client, input) => {
    let reason;
    try {
        let result = await client.send(new GetInvalidationCommand(input));
        reason = result;
        try {
            const returnComparator = () => {
                return result.Invalidation.Status;
            };
            if (returnComparator() === "Completed") {
                return { state: utilWaiter.WaiterState.SUCCESS, reason };
            }
        }
        catch (e) { }
    }
    catch (exception) {
        reason = exception;
    }
    return { state: utilWaiter.WaiterState.RETRY, reason };
};
const waitForInvalidationCompleted = async (params, input) => {
    const serviceDefaults = { minDelay: 20, maxDelay: 600 };
    return utilWaiter.createWaiter({ ...serviceDefaults, ...params }, input, checkState$2);
};
const waitUntilInvalidationCompleted = async (params, input) => {
    const serviceDefaults = { minDelay: 20, maxDelay: 600 };
    const result = await utilWaiter.createWaiter({ ...serviceDefaults, ...params }, input, checkState$2);
    return utilWaiter.checkExceptions(result);
};

const checkState$1 = async (client, input) => {
    let reason;
    try {
        let result = await client.send(new GetInvalidationForDistributionTenantCommand(input));
        reason = result;
        try {
            const returnComparator = () => {
                return result.Invalidation.Status;
            };
            if (returnComparator() === "Completed") {
                return { state: utilWaiter.WaiterState.SUCCESS, reason };
            }
        }
        catch (e) { }
    }
    catch (exception) {
        reason = exception;
    }
    return { state: utilWaiter.WaiterState.RETRY, reason };
};
const waitForInvalidationForDistributionTenantCompleted = async (params, input) => {
    const serviceDefaults = { minDelay: 20, maxDelay: 600 };
    return utilWaiter.createWaiter({ ...serviceDefaults, ...params }, input, checkState$1);
};
const waitUntilInvalidationForDistributionTenantCompleted = async (params, input) => {
    const serviceDefaults = { minDelay: 20, maxDelay: 600 };
    const result = await utilWaiter.createWaiter({ ...serviceDefaults, ...params }, input, checkState$1);
    return utilWaiter.checkExceptions(result);
};

const checkState = async (client, input) => {
    let reason;
    try {
        let result = await client.send(new GetStreamingDistributionCommand(input));
        reason = result;
        try {
            const returnComparator = () => {
                return result.StreamingDistribution.Status;
            };
            if (returnComparator() === "Deployed") {
                return { state: utilWaiter.WaiterState.SUCCESS, reason };
            }
        }
        catch (e) { }
    }
    catch (exception) {
        reason = exception;
    }
    return { state: utilWaiter.WaiterState.RETRY, reason };
};
const waitForStreamingDistributionDeployed = async (params, input) => {
    const serviceDefaults = { minDelay: 60, maxDelay: 1500 };
    return utilWaiter.createWaiter({ ...serviceDefaults, ...params }, input, checkState);
};
const waitUntilStreamingDistributionDeployed = async (params, input) => {
    const serviceDefaults = { minDelay: 60, maxDelay: 1500 };
    const result = await utilWaiter.createWaiter({ ...serviceDefaults, ...params }, input, checkState);
    return utilWaiter.checkExceptions(result);
};

const ResponseHeadersPolicyAccessControlAllowMethodsValues = {
    ALL: "ALL",
    DELETE: "DELETE",
    GET: "GET",
    HEAD: "HEAD",
    OPTIONS: "OPTIONS",
    PATCH: "PATCH",
    POST: "POST",
    PUT: "PUT",
};
const ICPRecordalStatus = {
    APPROVED: "APPROVED",
    PENDING: "PENDING",
    SUSPENDED: "SUSPENDED",
};
const Method = {
    DELETE: "DELETE",
    GET: "GET",
    HEAD: "HEAD",
    OPTIONS: "OPTIONS",
    PATCH: "PATCH",
    POST: "POST",
    PUT: "PUT",
};
const IpAddressType = {
    DualStack: "dualstack",
    Ipv4: "ipv4",
    Ipv6: "ipv6",
};
const IpamCidrStatus = {
    Advertised: "advertised",
    Advertising: "advertising",
    Deprovisioned: "deprovisioned",
    Deprovisioning: "deprovisioning",
    FailedAdvertise: "failed-advertise",
    FailedDeprovision: "failed-deprovision",
    FailedProvision: "failed-provision",
    FailedWithdraw: "failed-withdraw",
    Provisioned: "provisioned",
    Provisioning: "provisioning",
    Withdrawing: "withdrawing",
    Withdrawn: "withdrawn",
};
const ItemSelection = {
    all: "all",
    none: "none",
    whitelist: "whitelist",
};
const EventType = {
    origin_request: "origin-request",
    origin_response: "origin-response",
    viewer_request: "viewer-request",
    viewer_response: "viewer-response",
};
const ViewerProtocolPolicy = {
    allow_all: "allow-all",
    https_only: "https-only",
    redirect_to_https: "redirect-to-https",
};
const CachePolicyCookieBehavior = {
    all: "all",
    allExcept: "allExcept",
    none: "none",
    whitelist: "whitelist",
};
const CachePolicyHeaderBehavior = {
    none: "none",
    whitelist: "whitelist",
};
const CachePolicyQueryStringBehavior = {
    all: "all",
    allExcept: "allExcept",
    none: "none",
    whitelist: "whitelist",
};
const CachePolicyType = {
    custom: "custom",
    managed: "managed",
};
const CertificateSource = {
    acm: "acm",
    cloudfront: "cloudfront",
    iam: "iam",
};
const CertificateTransparencyLoggingPreference = {
    Disabled: "disabled",
    Enabled: "enabled",
};
const ConnectionMode = {
    Direct: "direct",
    TenantOnly: "tenant-only",
};
const HttpVersion = {
    http1_1: "http1.1",
    http2: "http2",
    http2and3: "http2and3",
    http3: "http3",
};
const OriginGroupSelectionCriteria = {
    Default: "default",
    MediaQualityBased: "media-quality-based",
};
const OriginProtocolPolicy = {
    http_only: "http-only",
    https_only: "https-only",
    match_viewer: "match-viewer",
};
const SslProtocol = {
    SSLv3: "SSLv3",
    TLSv1: "TLSv1",
    TLSv1_1: "TLSv1.1",
    TLSv1_2: "TLSv1.2",
};
const PriceClass = {
    None: "None",
    PriceClass_100: "PriceClass_100",
    PriceClass_200: "PriceClass_200",
    PriceClass_All: "PriceClass_All",
};
const GeoRestrictionType = {
    blacklist: "blacklist",
    none: "none",
    whitelist: "whitelist",
};
const MinimumProtocolVersion = {
    SSLv3: "SSLv3",
    TLSv1: "TLSv1",
    TLSv1_1_2016: "TLSv1.1_2016",
    TLSv1_2016: "TLSv1_2016",
    TLSv1_2_2018: "TLSv1.2_2018",
    TLSv1_2_2019: "TLSv1.2_2019",
    TLSv1_2_2021: "TLSv1.2_2021",
    TLSv1_2_2025: "TLSv1.2_2025",
    TLSv1_3_2025: "TLSv1.3_2025",
};
const SSLSupportMethod = {
    sni_only: "sni-only",
    static_ip: "static-ip",
    vip: "vip",
};
const ViewerMtlsMode = {
    Optional: "optional",
    Required: "required",
};
const FunctionRuntime = {
    cloudfront_js_1_0: "cloudfront-js-1.0",
    cloudfront_js_2_0: "cloudfront-js-2.0",
};
const FunctionStage = {
    DEVELOPMENT: "DEVELOPMENT",
    LIVE: "LIVE",
};
const ContinuousDeploymentPolicyType = {
    SingleHeader: "SingleHeader",
    SingleWeight: "SingleWeight",
};
const CustomizationActionType = {
    disable: "disable",
    override: "override",
};
const ValidationTokenHost = {
    CloudFront: "cloudfront",
    SelfHosted: "self-hosted",
};
const DomainStatus = {
    Active: "active",
    Inactive: "inactive",
};
const Format = {
    URLEncoded: "URLEncoded",
};
const ImportSourceType = {
    S3: "S3",
};
const RealtimeMetricsSubscriptionStatus = {
    Disabled: "Disabled",
    Enabled: "Enabled",
};
const OriginAccessControlOriginTypes = {
    lambda: "lambda",
    mediapackagev2: "mediapackagev2",
    mediastore: "mediastore",
    s3: "s3",
};
const OriginAccessControlSigningBehaviors = {
    always: "always",
    never: "never",
    no_override: "no-override",
};
const OriginAccessControlSigningProtocols = {
    sigv4: "sigv4",
};
const OriginRequestPolicyCookieBehavior = {
    all: "all",
    allExcept: "allExcept",
    none: "none",
    whitelist: "whitelist",
};
const OriginRequestPolicyHeaderBehavior = {
    allExcept: "allExcept",
    allViewer: "allViewer",
    allViewerAndWhitelistCloudFront: "allViewerAndWhitelistCloudFront",
    none: "none",
    whitelist: "whitelist",
};
const OriginRequestPolicyQueryStringBehavior = {
    all: "all",
    allExcept: "allExcept",
    none: "none",
    whitelist: "whitelist",
};
const FrameOptionsList = {
    DENY: "DENY",
    SAMEORIGIN: "SAMEORIGIN",
};
const ReferrerPolicyList = {
    no_referrer: "no-referrer",
    no_referrer_when_downgrade: "no-referrer-when-downgrade",
    origin: "origin",
    origin_when_cross_origin: "origin-when-cross-origin",
    same_origin: "same-origin",
    strict_origin: "strict-origin",
    strict_origin_when_cross_origin: "strict-origin-when-cross-origin",
    unsafe_url: "unsafe-url",
};
const TrustStoreStatus = {
    Active: "active",
    Failed: "failed",
    Pending: "pending",
};
const ManagedCertificateStatus = {
    Expired: "expired",
    Failed: "failed",
    Inactive: "inactive",
    Issued: "issued",
    PendingValidation: "pending-validation",
    Revoked: "revoked",
    ValidationTimedOut: "validation-timed-out",
};
const DistributionResourceType = {
    Distribution: "distribution",
    DistributionTenant: "distribution-tenant",
};
const OriginRequestPolicyType = {
    custom: "custom",
    managed: "managed",
};
const ResponseHeadersPolicyType = {
    custom: "custom",
    managed: "managed",
};
const DnsConfigurationStatus = {
    Invalid: "invalid-configuration",
    Unknown: "unknown-configuration",
    Valid: "valid-configuration",
};

Object.defineProperty(exports, "$Command", {
    enumerable: true,
    get: function () { return smithyClient.Command; }
});
Object.defineProperty(exports, "__Client", {
    enumerable: true,
    get: function () { return smithyClient.Client; }
});
exports.AccessDenied = AccessDenied;
exports.AccessDenied$ = AccessDenied$;
exports.ActiveTrustedKeyGroups$ = ActiveTrustedKeyGroups$;
exports.ActiveTrustedSigners$ = ActiveTrustedSigners$;
exports.AliasICPRecordal$ = AliasICPRecordal$;
exports.Aliases$ = Aliases$;
exports.AllowedMethods$ = AllowedMethods$;
exports.AnycastIpList$ = AnycastIpList$;
exports.AnycastIpListCollection$ = AnycastIpListCollection$;
exports.AnycastIpListSummary$ = AnycastIpListSummary$;
exports.AssociateAlias$ = AssociateAlias$;
exports.AssociateAliasCommand = AssociateAliasCommand;
exports.AssociateAliasRequest$ = AssociateAliasRequest$;
exports.AssociateDistributionTenantWebACL$ = AssociateDistributionTenantWebACL$;
exports.AssociateDistributionTenantWebACLCommand = AssociateDistributionTenantWebACLCommand;
exports.AssociateDistributionTenantWebACLRequest$ = AssociateDistributionTenantWebACLRequest$;
exports.AssociateDistributionTenantWebACLResult$ = AssociateDistributionTenantWebACLResult$;
exports.AssociateDistributionWebACL$ = AssociateDistributionWebACL$;
exports.AssociateDistributionWebACLCommand = AssociateDistributionWebACLCommand;
exports.AssociateDistributionWebACLRequest$ = AssociateDistributionWebACLRequest$;
exports.AssociateDistributionWebACLResult$ = AssociateDistributionWebACLResult$;
exports.BatchTooLarge = BatchTooLarge;
exports.BatchTooLarge$ = BatchTooLarge$;
exports.CNAMEAlreadyExists = CNAMEAlreadyExists;
exports.CNAMEAlreadyExists$ = CNAMEAlreadyExists$;
exports.CaCertificatesBundleS3Location$ = CaCertificatesBundleS3Location$;
exports.CaCertificatesBundleSource$ = CaCertificatesBundleSource$;
exports.CacheBehavior$ = CacheBehavior$;
exports.CacheBehaviors$ = CacheBehaviors$;
exports.CachePolicy$ = CachePolicy$;
exports.CachePolicyAlreadyExists = CachePolicyAlreadyExists;
exports.CachePolicyAlreadyExists$ = CachePolicyAlreadyExists$;
exports.CachePolicyConfig$ = CachePolicyConfig$;
exports.CachePolicyCookieBehavior = CachePolicyCookieBehavior;
exports.CachePolicyCookiesConfig$ = CachePolicyCookiesConfig$;
exports.CachePolicyHeaderBehavior = CachePolicyHeaderBehavior;
exports.CachePolicyHeadersConfig$ = CachePolicyHeadersConfig$;
exports.CachePolicyInUse = CachePolicyInUse;
exports.CachePolicyInUse$ = CachePolicyInUse$;
exports.CachePolicyList$ = CachePolicyList$;
exports.CachePolicyQueryStringBehavior = CachePolicyQueryStringBehavior;
exports.CachePolicyQueryStringsConfig$ = CachePolicyQueryStringsConfig$;
exports.CachePolicySummary$ = CachePolicySummary$;
exports.CachePolicyType = CachePolicyType;
exports.CachedMethods$ = CachedMethods$;
exports.CannotChangeImmutablePublicKeyFields = CannotChangeImmutablePublicKeyFields;
exports.CannotChangeImmutablePublicKeyFields$ = CannotChangeImmutablePublicKeyFields$;
exports.CannotDeleteEntityWhileInUse = CannotDeleteEntityWhileInUse;
exports.CannotDeleteEntityWhileInUse$ = CannotDeleteEntityWhileInUse$;
exports.CannotUpdateEntityWhileInUse = CannotUpdateEntityWhileInUse;
exports.CannotUpdateEntityWhileInUse$ = CannotUpdateEntityWhileInUse$;
exports.Certificate$ = Certificate$;
exports.CertificateSource = CertificateSource;
exports.CertificateTransparencyLoggingPreference = CertificateTransparencyLoggingPreference;
exports.CloudFront = CloudFront;
exports.CloudFrontClient = CloudFrontClient;
exports.CloudFrontOriginAccessIdentity$ = CloudFrontOriginAccessIdentity$;
exports.CloudFrontOriginAccessIdentityAlreadyExists = CloudFrontOriginAccessIdentityAlreadyExists;
exports.CloudFrontOriginAccessIdentityAlreadyExists$ = CloudFrontOriginAccessIdentityAlreadyExists$;
exports.CloudFrontOriginAccessIdentityConfig$ = CloudFrontOriginAccessIdentityConfig$;
exports.CloudFrontOriginAccessIdentityInUse = CloudFrontOriginAccessIdentityInUse;
exports.CloudFrontOriginAccessIdentityInUse$ = CloudFrontOriginAccessIdentityInUse$;
exports.CloudFrontOriginAccessIdentityList$ = CloudFrontOriginAccessIdentityList$;
exports.CloudFrontOriginAccessIdentitySummary$ = CloudFrontOriginAccessIdentitySummary$;
exports.CloudFrontServiceException = CloudFrontServiceException;
exports.CloudFrontServiceException$ = CloudFrontServiceException$;
exports.ConflictingAlias$ = ConflictingAlias$;
exports.ConflictingAliasesList$ = ConflictingAliasesList$;
exports.ConnectionFunctionAssociation$ = ConnectionFunctionAssociation$;
exports.ConnectionFunctionSummary$ = ConnectionFunctionSummary$;
exports.ConnectionFunctionTestResult$ = ConnectionFunctionTestResult$;
exports.ConnectionGroup$ = ConnectionGroup$;
exports.ConnectionGroupAssociationFilter$ = ConnectionGroupAssociationFilter$;
exports.ConnectionGroupSummary$ = ConnectionGroupSummary$;
exports.ConnectionMode = ConnectionMode;
exports.ContentTypeProfile$ = ContentTypeProfile$;
exports.ContentTypeProfileConfig$ = ContentTypeProfileConfig$;
exports.ContentTypeProfiles$ = ContentTypeProfiles$;
exports.ContinuousDeploymentPolicy$ = ContinuousDeploymentPolicy$;
exports.ContinuousDeploymentPolicyAlreadyExists = ContinuousDeploymentPolicyAlreadyExists;
exports.ContinuousDeploymentPolicyAlreadyExists$ = ContinuousDeploymentPolicyAlreadyExists$;
exports.ContinuousDeploymentPolicyConfig$ = ContinuousDeploymentPolicyConfig$;
exports.ContinuousDeploymentPolicyInUse = ContinuousDeploymentPolicyInUse;
exports.ContinuousDeploymentPolicyInUse$ = ContinuousDeploymentPolicyInUse$;
exports.ContinuousDeploymentPolicyList$ = ContinuousDeploymentPolicyList$;
exports.ContinuousDeploymentPolicySummary$ = ContinuousDeploymentPolicySummary$;
exports.ContinuousDeploymentPolicyType = ContinuousDeploymentPolicyType;
exports.ContinuousDeploymentSingleHeaderConfig$ = ContinuousDeploymentSingleHeaderConfig$;
exports.ContinuousDeploymentSingleWeightConfig$ = ContinuousDeploymentSingleWeightConfig$;
exports.CookieNames$ = CookieNames$;
exports.CookiePreference$ = CookiePreference$;
exports.CopyDistribution$ = CopyDistribution$;
exports.CopyDistributionCommand = CopyDistributionCommand;
exports.CopyDistributionRequest$ = CopyDistributionRequest$;
exports.CopyDistributionResult$ = CopyDistributionResult$;
exports.CreateAnycastIpList$ = CreateAnycastIpList$;
exports.CreateAnycastIpListCommand = CreateAnycastIpListCommand;
exports.CreateAnycastIpListRequest$ = CreateAnycastIpListRequest$;
exports.CreateAnycastIpListResult$ = CreateAnycastIpListResult$;
exports.CreateCachePolicy$ = CreateCachePolicy$;
exports.CreateCachePolicyCommand = CreateCachePolicyCommand;
exports.CreateCachePolicyRequest$ = CreateCachePolicyRequest$;
exports.CreateCachePolicyResult$ = CreateCachePolicyResult$;
exports.CreateCloudFrontOriginAccessIdentity$ = CreateCloudFrontOriginAccessIdentity$;
exports.CreateCloudFrontOriginAccessIdentityCommand = CreateCloudFrontOriginAccessIdentityCommand;
exports.CreateCloudFrontOriginAccessIdentityRequest$ = CreateCloudFrontOriginAccessIdentityRequest$;
exports.CreateCloudFrontOriginAccessIdentityResult$ = CreateCloudFrontOriginAccessIdentityResult$;
exports.CreateConnectionFunction$ = CreateConnectionFunction$;
exports.CreateConnectionFunctionCommand = CreateConnectionFunctionCommand;
exports.CreateConnectionFunctionRequest$ = CreateConnectionFunctionRequest$;
exports.CreateConnectionFunctionResult$ = CreateConnectionFunctionResult$;
exports.CreateConnectionGroup$ = CreateConnectionGroup$;
exports.CreateConnectionGroupCommand = CreateConnectionGroupCommand;
exports.CreateConnectionGroupRequest$ = CreateConnectionGroupRequest$;
exports.CreateConnectionGroupResult$ = CreateConnectionGroupResult$;
exports.CreateContinuousDeploymentPolicy$ = CreateContinuousDeploymentPolicy$;
exports.CreateContinuousDeploymentPolicyCommand = CreateContinuousDeploymentPolicyCommand;
exports.CreateContinuousDeploymentPolicyRequest$ = CreateContinuousDeploymentPolicyRequest$;
exports.CreateContinuousDeploymentPolicyResult$ = CreateContinuousDeploymentPolicyResult$;
exports.CreateDistribution$ = CreateDistribution$;
exports.CreateDistributionCommand = CreateDistributionCommand;
exports.CreateDistributionRequest$ = CreateDistributionRequest$;
exports.CreateDistributionResult$ = CreateDistributionResult$;
exports.CreateDistributionTenant$ = CreateDistributionTenant$;
exports.CreateDistributionTenantCommand = CreateDistributionTenantCommand;
exports.CreateDistributionTenantRequest$ = CreateDistributionTenantRequest$;
exports.CreateDistributionTenantResult$ = CreateDistributionTenantResult$;
exports.CreateDistributionWithTags$ = CreateDistributionWithTags$;
exports.CreateDistributionWithTagsCommand = CreateDistributionWithTagsCommand;
exports.CreateDistributionWithTagsRequest$ = CreateDistributionWithTagsRequest$;
exports.CreateDistributionWithTagsResult$ = CreateDistributionWithTagsResult$;
exports.CreateFieldLevelEncryptionConfig$ = CreateFieldLevelEncryptionConfig$;
exports.CreateFieldLevelEncryptionConfigCommand = CreateFieldLevelEncryptionConfigCommand;
exports.CreateFieldLevelEncryptionConfigRequest$ = CreateFieldLevelEncryptionConfigRequest$;
exports.CreateFieldLevelEncryptionConfigResult$ = CreateFieldLevelEncryptionConfigResult$;
exports.CreateFieldLevelEncryptionProfile$ = CreateFieldLevelEncryptionProfile$;
exports.CreateFieldLevelEncryptionProfileCommand = CreateFieldLevelEncryptionProfileCommand;
exports.CreateFieldLevelEncryptionProfileRequest$ = CreateFieldLevelEncryptionProfileRequest$;
exports.CreateFieldLevelEncryptionProfileResult$ = CreateFieldLevelEncryptionProfileResult$;
exports.CreateFunction$ = CreateFunction$;
exports.CreateFunctionCommand = CreateFunctionCommand;
exports.CreateFunctionRequest$ = CreateFunctionRequest$;
exports.CreateFunctionResult$ = CreateFunctionResult$;
exports.CreateInvalidation$ = CreateInvalidation$;
exports.CreateInvalidationCommand = CreateInvalidationCommand;
exports.CreateInvalidationForDistributionTenant$ = CreateInvalidationForDistributionTenant$;
exports.CreateInvalidationForDistributionTenantCommand = CreateInvalidationForDistributionTenantCommand;
exports.CreateInvalidationForDistributionTenantRequest$ = CreateInvalidationForDistributionTenantRequest$;
exports.CreateInvalidationForDistributionTenantResult$ = CreateInvalidationForDistributionTenantResult$;
exports.CreateInvalidationRequest$ = CreateInvalidationRequest$;
exports.CreateInvalidationResult$ = CreateInvalidationResult$;
exports.CreateKeyGroup$ = CreateKeyGroup$;
exports.CreateKeyGroupCommand = CreateKeyGroupCommand;
exports.CreateKeyGroupRequest$ = CreateKeyGroupRequest$;
exports.CreateKeyGroupResult$ = CreateKeyGroupResult$;
exports.CreateKeyValueStore$ = CreateKeyValueStore$;
exports.CreateKeyValueStoreCommand = CreateKeyValueStoreCommand;
exports.CreateKeyValueStoreRequest$ = CreateKeyValueStoreRequest$;
exports.CreateKeyValueStoreResult$ = CreateKeyValueStoreResult$;
exports.CreateMonitoringSubscription$ = CreateMonitoringSubscription$;
exports.CreateMonitoringSubscriptionCommand = CreateMonitoringSubscriptionCommand;
exports.CreateMonitoringSubscriptionRequest$ = CreateMonitoringSubscriptionRequest$;
exports.CreateMonitoringSubscriptionResult$ = CreateMonitoringSubscriptionResult$;
exports.CreateOriginAccessControl$ = CreateOriginAccessControl$;
exports.CreateOriginAccessControlCommand = CreateOriginAccessControlCommand;
exports.CreateOriginAccessControlRequest$ = CreateOriginAccessControlRequest$;
exports.CreateOriginAccessControlResult$ = CreateOriginAccessControlResult$;
exports.CreateOriginRequestPolicy$ = CreateOriginRequestPolicy$;
exports.CreateOriginRequestPolicyCommand = CreateOriginRequestPolicyCommand;
exports.CreateOriginRequestPolicyRequest$ = CreateOriginRequestPolicyRequest$;
exports.CreateOriginRequestPolicyResult$ = CreateOriginRequestPolicyResult$;
exports.CreatePublicKey$ = CreatePublicKey$;
exports.CreatePublicKeyCommand = CreatePublicKeyCommand;
exports.CreatePublicKeyRequest$ = CreatePublicKeyRequest$;
exports.CreatePublicKeyResult$ = CreatePublicKeyResult$;
exports.CreateRealtimeLogConfig$ = CreateRealtimeLogConfig$;
exports.CreateRealtimeLogConfigCommand = CreateRealtimeLogConfigCommand;
exports.CreateRealtimeLogConfigRequest$ = CreateRealtimeLogConfigRequest$;
exports.CreateRealtimeLogConfigResult$ = CreateRealtimeLogConfigResult$;
exports.CreateResponseHeadersPolicy$ = CreateResponseHeadersPolicy$;
exports.CreateResponseHeadersPolicyCommand = CreateResponseHeadersPolicyCommand;
exports.CreateResponseHeadersPolicyRequest$ = CreateResponseHeadersPolicyRequest$;
exports.CreateResponseHeadersPolicyResult$ = CreateResponseHeadersPolicyResult$;
exports.CreateStreamingDistribution$ = CreateStreamingDistribution$;
exports.CreateStreamingDistributionCommand = CreateStreamingDistributionCommand;
exports.CreateStreamingDistributionRequest$ = CreateStreamingDistributionRequest$;
exports.CreateStreamingDistributionResult$ = CreateStreamingDistributionResult$;
exports.CreateStreamingDistributionWithTags$ = CreateStreamingDistributionWithTags$;
exports.CreateStreamingDistributionWithTagsCommand = CreateStreamingDistributionWithTagsCommand;
exports.CreateStreamingDistributionWithTagsRequest$ = CreateStreamingDistributionWithTagsRequest$;
exports.CreateStreamingDistributionWithTagsResult$ = CreateStreamingDistributionWithTagsResult$;
exports.CreateTrustStore$ = CreateTrustStore$;
exports.CreateTrustStoreCommand = CreateTrustStoreCommand;
exports.CreateTrustStoreRequest$ = CreateTrustStoreRequest$;
exports.CreateTrustStoreResult$ = CreateTrustStoreResult$;
exports.CreateVpcOrigin$ = CreateVpcOrigin$;
exports.CreateVpcOriginCommand = CreateVpcOriginCommand;
exports.CreateVpcOriginRequest$ = CreateVpcOriginRequest$;
exports.CreateVpcOriginResult$ = CreateVpcOriginResult$;
exports.CustomErrorResponse$ = CustomErrorResponse$;
exports.CustomErrorResponses$ = CustomErrorResponses$;
exports.CustomHeaders$ = CustomHeaders$;
exports.CustomOriginConfig$ = CustomOriginConfig$;
exports.CustomizationActionType = CustomizationActionType;
exports.Customizations$ = Customizations$;
exports.DefaultCacheBehavior$ = DefaultCacheBehavior$;
exports.DeleteAnycastIpList$ = DeleteAnycastIpList$;
exports.DeleteAnycastIpListCommand = DeleteAnycastIpListCommand;
exports.DeleteAnycastIpListRequest$ = DeleteAnycastIpListRequest$;
exports.DeleteCachePolicy$ = DeleteCachePolicy$;
exports.DeleteCachePolicyCommand = DeleteCachePolicyCommand;
exports.DeleteCachePolicyRequest$ = DeleteCachePolicyRequest$;
exports.DeleteCloudFrontOriginAccessIdentity$ = DeleteCloudFrontOriginAccessIdentity$;
exports.DeleteCloudFrontOriginAccessIdentityCommand = DeleteCloudFrontOriginAccessIdentityCommand;
exports.DeleteCloudFrontOriginAccessIdentityRequest$ = DeleteCloudFrontOriginAccessIdentityRequest$;
exports.DeleteConnectionFunction$ = DeleteConnectionFunction$;
exports.DeleteConnectionFunctionCommand = DeleteConnectionFunctionCommand;
exports.DeleteConnectionFunctionRequest$ = DeleteConnectionFunctionRequest$;
exports.DeleteConnectionGroup$ = DeleteConnectionGroup$;
exports.DeleteConnectionGroupCommand = DeleteConnectionGroupCommand;
exports.DeleteConnectionGroupRequest$ = DeleteConnectionGroupRequest$;
exports.DeleteContinuousDeploymentPolicy$ = DeleteContinuousDeploymentPolicy$;
exports.DeleteContinuousDeploymentPolicyCommand = DeleteContinuousDeploymentPolicyCommand;
exports.DeleteContinuousDeploymentPolicyRequest$ = DeleteContinuousDeploymentPolicyRequest$;
exports.DeleteDistribution$ = DeleteDistribution$;
exports.DeleteDistributionCommand = DeleteDistributionCommand;
exports.DeleteDistributionRequest$ = DeleteDistributionRequest$;
exports.DeleteDistributionTenant$ = DeleteDistributionTenant$;
exports.DeleteDistributionTenantCommand = DeleteDistributionTenantCommand;
exports.DeleteDistributionTenantRequest$ = DeleteDistributionTenantRequest$;
exports.DeleteFieldLevelEncryptionConfig$ = DeleteFieldLevelEncryptionConfig$;
exports.DeleteFieldLevelEncryptionConfigCommand = DeleteFieldLevelEncryptionConfigCommand;
exports.DeleteFieldLevelEncryptionConfigRequest$ = DeleteFieldLevelEncryptionConfigRequest$;
exports.DeleteFieldLevelEncryptionProfile$ = DeleteFieldLevelEncryptionProfile$;
exports.DeleteFieldLevelEncryptionProfileCommand = DeleteFieldLevelEncryptionProfileCommand;
exports.DeleteFieldLevelEncryptionProfileRequest$ = DeleteFieldLevelEncryptionProfileRequest$;
exports.DeleteFunction$ = DeleteFunction$;
exports.DeleteFunctionCommand = DeleteFunctionCommand;
exports.DeleteFunctionRequest$ = DeleteFunctionRequest$;
exports.DeleteKeyGroup$ = DeleteKeyGroup$;
exports.DeleteKeyGroupCommand = DeleteKeyGroupCommand;
exports.DeleteKeyGroupRequest$ = DeleteKeyGroupRequest$;
exports.DeleteKeyValueStore$ = DeleteKeyValueStore$;
exports.DeleteKeyValueStoreCommand = DeleteKeyValueStoreCommand;
exports.DeleteKeyValueStoreRequest$ = DeleteKeyValueStoreRequest$;
exports.DeleteMonitoringSubscription$ = DeleteMonitoringSubscription$;
exports.DeleteMonitoringSubscriptionCommand = DeleteMonitoringSubscriptionCommand;
exports.DeleteMonitoringSubscriptionRequest$ = DeleteMonitoringSubscriptionRequest$;
exports.DeleteMonitoringSubscriptionResult$ = DeleteMonitoringSubscriptionResult$;
exports.DeleteOriginAccessControl$ = DeleteOriginAccessControl$;
exports.DeleteOriginAccessControlCommand = DeleteOriginAccessControlCommand;
exports.DeleteOriginAccessControlRequest$ = DeleteOriginAccessControlRequest$;
exports.DeleteOriginRequestPolicy$ = DeleteOriginRequestPolicy$;
exports.DeleteOriginRequestPolicyCommand = DeleteOriginRequestPolicyCommand;
exports.DeleteOriginRequestPolicyRequest$ = DeleteOriginRequestPolicyRequest$;
exports.DeletePublicKey$ = DeletePublicKey$;
exports.DeletePublicKeyCommand = DeletePublicKeyCommand;
exports.DeletePublicKeyRequest$ = DeletePublicKeyRequest$;
exports.DeleteRealtimeLogConfig$ = DeleteRealtimeLogConfig$;
exports.DeleteRealtimeLogConfigCommand = DeleteRealtimeLogConfigCommand;
exports.DeleteRealtimeLogConfigRequest$ = DeleteRealtimeLogConfigRequest$;
exports.DeleteResourcePolicy$ = DeleteResourcePolicy$;
exports.DeleteResourcePolicyCommand = DeleteResourcePolicyCommand;
exports.DeleteResourcePolicyRequest$ = DeleteResourcePolicyRequest$;
exports.DeleteResponseHeadersPolicy$ = DeleteResponseHeadersPolicy$;
exports.DeleteResponseHeadersPolicyCommand = DeleteResponseHeadersPolicyCommand;
exports.DeleteResponseHeadersPolicyRequest$ = DeleteResponseHeadersPolicyRequest$;
exports.DeleteStreamingDistribution$ = DeleteStreamingDistribution$;
exports.DeleteStreamingDistributionCommand = DeleteStreamingDistributionCommand;
exports.DeleteStreamingDistributionRequest$ = DeleteStreamingDistributionRequest$;
exports.DeleteTrustStore$ = DeleteTrustStore$;
exports.DeleteTrustStoreCommand = DeleteTrustStoreCommand;
exports.DeleteTrustStoreRequest$ = DeleteTrustStoreRequest$;
exports.DeleteVpcOrigin$ = DeleteVpcOrigin$;
exports.DeleteVpcOriginCommand = DeleteVpcOriginCommand;
exports.DeleteVpcOriginRequest$ = DeleteVpcOriginRequest$;
exports.DeleteVpcOriginResult$ = DeleteVpcOriginResult$;
exports.DescribeConnectionFunction$ = DescribeConnectionFunction$;
exports.DescribeConnectionFunctionCommand = DescribeConnectionFunctionCommand;
exports.DescribeConnectionFunctionRequest$ = DescribeConnectionFunctionRequest$;
exports.DescribeConnectionFunctionResult$ = DescribeConnectionFunctionResult$;
exports.DescribeFunction$ = DescribeFunction$;
exports.DescribeFunctionCommand = DescribeFunctionCommand;
exports.DescribeFunctionRequest$ = DescribeFunctionRequest$;
exports.DescribeFunctionResult$ = DescribeFunctionResult$;
exports.DescribeKeyValueStore$ = DescribeKeyValueStore$;
exports.DescribeKeyValueStoreCommand = DescribeKeyValueStoreCommand;
exports.DescribeKeyValueStoreRequest$ = DescribeKeyValueStoreRequest$;
exports.DescribeKeyValueStoreResult$ = DescribeKeyValueStoreResult$;
exports.DisassociateDistributionTenantWebACL$ = DisassociateDistributionTenantWebACL$;
exports.DisassociateDistributionTenantWebACLCommand = DisassociateDistributionTenantWebACLCommand;
exports.DisassociateDistributionTenantWebACLRequest$ = DisassociateDistributionTenantWebACLRequest$;
exports.DisassociateDistributionTenantWebACLResult$ = DisassociateDistributionTenantWebACLResult$;
exports.DisassociateDistributionWebACL$ = DisassociateDistributionWebACL$;
exports.DisassociateDistributionWebACLCommand = DisassociateDistributionWebACLCommand;
exports.DisassociateDistributionWebACLRequest$ = DisassociateDistributionWebACLRequest$;
exports.DisassociateDistributionWebACLResult$ = DisassociateDistributionWebACLResult$;
exports.Distribution$ = Distribution$;
exports.DistributionAlreadyExists = DistributionAlreadyExists;
exports.DistributionAlreadyExists$ = DistributionAlreadyExists$;
exports.DistributionConfig$ = DistributionConfig$;
exports.DistributionConfigWithTags$ = DistributionConfigWithTags$;
exports.DistributionIdList$ = DistributionIdList$;
exports.DistributionIdOwner$ = DistributionIdOwner$;
exports.DistributionIdOwnerList$ = DistributionIdOwnerList$;
exports.DistributionList$ = DistributionList$;
exports.DistributionNotDisabled = DistributionNotDisabled;
exports.DistributionNotDisabled$ = DistributionNotDisabled$;
exports.DistributionResourceId$ = DistributionResourceId$;
exports.DistributionResourceType = DistributionResourceType;
exports.DistributionSummary$ = DistributionSummary$;
exports.DistributionTenant$ = DistributionTenant$;
exports.DistributionTenantAssociationFilter$ = DistributionTenantAssociationFilter$;
exports.DistributionTenantSummary$ = DistributionTenantSummary$;
exports.DnsConfiguration$ = DnsConfiguration$;
exports.DnsConfigurationStatus = DnsConfigurationStatus;
exports.DomainConflict$ = DomainConflict$;
exports.DomainItem$ = DomainItem$;
exports.DomainResult$ = DomainResult$;
exports.DomainStatus = DomainStatus;
exports.EncryptionEntities$ = EncryptionEntities$;
exports.EncryptionEntity$ = EncryptionEntity$;
exports.EndPoint$ = EndPoint$;
exports.EntityAlreadyExists = EntityAlreadyExists;
exports.EntityAlreadyExists$ = EntityAlreadyExists$;
exports.EntityLimitExceeded = EntityLimitExceeded;
exports.EntityLimitExceeded$ = EntityLimitExceeded$;
exports.EntityNotFound = EntityNotFound;
exports.EntityNotFound$ = EntityNotFound$;
exports.EntitySizeLimitExceeded = EntitySizeLimitExceeded;
exports.EntitySizeLimitExceeded$ = EntitySizeLimitExceeded$;
exports.EventType = EventType;
exports.FieldLevelEncryption$ = FieldLevelEncryption$;
exports.FieldLevelEncryptionConfig$ = FieldLevelEncryptionConfig$;
exports.FieldLevelEncryptionConfigAlreadyExists = FieldLevelEncryptionConfigAlreadyExists;
exports.FieldLevelEncryptionConfigAlreadyExists$ = FieldLevelEncryptionConfigAlreadyExists$;
exports.FieldLevelEncryptionConfigInUse = FieldLevelEncryptionConfigInUse;
exports.FieldLevelEncryptionConfigInUse$ = FieldLevelEncryptionConfigInUse$;
exports.FieldLevelEncryptionList$ = FieldLevelEncryptionList$;
exports.FieldLevelEncryptionProfile$ = FieldLevelEncryptionProfile$;
exports.FieldLevelEncryptionProfileAlreadyExists = FieldLevelEncryptionProfileAlreadyExists;
exports.FieldLevelEncryptionProfileAlreadyExists$ = FieldLevelEncryptionProfileAlreadyExists$;
exports.FieldLevelEncryptionProfileConfig$ = FieldLevelEncryptionProfileConfig$;
exports.FieldLevelEncryptionProfileInUse = FieldLevelEncryptionProfileInUse;
exports.FieldLevelEncryptionProfileInUse$ = FieldLevelEncryptionProfileInUse$;
exports.FieldLevelEncryptionProfileList$ = FieldLevelEncryptionProfileList$;
exports.FieldLevelEncryptionProfileSizeExceeded = FieldLevelEncryptionProfileSizeExceeded;
exports.FieldLevelEncryptionProfileSizeExceeded$ = FieldLevelEncryptionProfileSizeExceeded$;
exports.FieldLevelEncryptionProfileSummary$ = FieldLevelEncryptionProfileSummary$;
exports.FieldLevelEncryptionSummary$ = FieldLevelEncryptionSummary$;
exports.FieldPatterns$ = FieldPatterns$;
exports.Format = Format;
exports.ForwardedValues$ = ForwardedValues$;
exports.FrameOptionsList = FrameOptionsList;
exports.FunctionAlreadyExists = FunctionAlreadyExists;
exports.FunctionAlreadyExists$ = FunctionAlreadyExists$;
exports.FunctionAssociation$ = FunctionAssociation$;
exports.FunctionAssociations$ = FunctionAssociations$;
exports.FunctionConfig$ = FunctionConfig$;
exports.FunctionInUse = FunctionInUse;
exports.FunctionInUse$ = FunctionInUse$;
exports.FunctionList$ = FunctionList$;
exports.FunctionMetadata$ = FunctionMetadata$;
exports.FunctionRuntime = FunctionRuntime;
exports.FunctionSizeLimitExceeded = FunctionSizeLimitExceeded;
exports.FunctionSizeLimitExceeded$ = FunctionSizeLimitExceeded$;
exports.FunctionStage = FunctionStage;
exports.FunctionSummary$ = FunctionSummary$;
exports.GeoRestriction$ = GeoRestriction$;
exports.GeoRestrictionCustomization$ = GeoRestrictionCustomization$;
exports.GeoRestrictionType = GeoRestrictionType;
exports.GetAnycastIpList$ = GetAnycastIpList$;
exports.GetAnycastIpListCommand = GetAnycastIpListCommand;
exports.GetAnycastIpListRequest$ = GetAnycastIpListRequest$;
exports.GetAnycastIpListResult$ = GetAnycastIpListResult$;
exports.GetCachePolicy$ = GetCachePolicy$;
exports.GetCachePolicyCommand = GetCachePolicyCommand;
exports.GetCachePolicyConfig$ = GetCachePolicyConfig$;
exports.GetCachePolicyConfigCommand = GetCachePolicyConfigCommand;
exports.GetCachePolicyConfigRequest$ = GetCachePolicyConfigRequest$;
exports.GetCachePolicyConfigResult$ = GetCachePolicyConfigResult$;
exports.GetCachePolicyRequest$ = GetCachePolicyRequest$;
exports.GetCachePolicyResult$ = GetCachePolicyResult$;
exports.GetCloudFrontOriginAccessIdentity$ = GetCloudFrontOriginAccessIdentity$;
exports.GetCloudFrontOriginAccessIdentityCommand = GetCloudFrontOriginAccessIdentityCommand;
exports.GetCloudFrontOriginAccessIdentityConfig$ = GetCloudFrontOriginAccessIdentityConfig$;
exports.GetCloudFrontOriginAccessIdentityConfigCommand = GetCloudFrontOriginAccessIdentityConfigCommand;
exports.GetCloudFrontOriginAccessIdentityConfigRequest$ = GetCloudFrontOriginAccessIdentityConfigRequest$;
exports.GetCloudFrontOriginAccessIdentityConfigResult$ = GetCloudFrontOriginAccessIdentityConfigResult$;
exports.GetCloudFrontOriginAccessIdentityRequest$ = GetCloudFrontOriginAccessIdentityRequest$;
exports.GetCloudFrontOriginAccessIdentityResult$ = GetCloudFrontOriginAccessIdentityResult$;
exports.GetConnectionFunction$ = GetConnectionFunction$;
exports.GetConnectionFunctionCommand = GetConnectionFunctionCommand;
exports.GetConnectionFunctionRequest$ = GetConnectionFunctionRequest$;
exports.GetConnectionFunctionResult$ = GetConnectionFunctionResult$;
exports.GetConnectionGroup$ = GetConnectionGroup$;
exports.GetConnectionGroupByRoutingEndpoint$ = GetConnectionGroupByRoutingEndpoint$;
exports.GetConnectionGroupByRoutingEndpointCommand = GetConnectionGroupByRoutingEndpointCommand;
exports.GetConnectionGroupByRoutingEndpointRequest$ = GetConnectionGroupByRoutingEndpointRequest$;
exports.GetConnectionGroupByRoutingEndpointResult$ = GetConnectionGroupByRoutingEndpointResult$;
exports.GetConnectionGroupCommand = GetConnectionGroupCommand;
exports.GetConnectionGroupRequest$ = GetConnectionGroupRequest$;
exports.GetConnectionGroupResult$ = GetConnectionGroupResult$;
exports.GetContinuousDeploymentPolicy$ = GetContinuousDeploymentPolicy$;
exports.GetContinuousDeploymentPolicyCommand = GetContinuousDeploymentPolicyCommand;
exports.GetContinuousDeploymentPolicyConfig$ = GetContinuousDeploymentPolicyConfig$;
exports.GetContinuousDeploymentPolicyConfigCommand = GetContinuousDeploymentPolicyConfigCommand;
exports.GetContinuousDeploymentPolicyConfigRequest$ = GetContinuousDeploymentPolicyConfigRequest$;
exports.GetContinuousDeploymentPolicyConfigResult$ = GetContinuousDeploymentPolicyConfigResult$;
exports.GetContinuousDeploymentPolicyRequest$ = GetContinuousDeploymentPolicyRequest$;
exports.GetContinuousDeploymentPolicyResult$ = GetContinuousDeploymentPolicyResult$;
exports.GetDistribution$ = GetDistribution$;
exports.GetDistributionCommand = GetDistributionCommand;
exports.GetDistributionConfig$ = GetDistributionConfig$;
exports.GetDistributionConfigCommand = GetDistributionConfigCommand;
exports.GetDistributionConfigRequest$ = GetDistributionConfigRequest$;
exports.GetDistributionConfigResult$ = GetDistributionConfigResult$;
exports.GetDistributionRequest$ = GetDistributionRequest$;
exports.GetDistributionResult$ = GetDistributionResult$;
exports.GetDistributionTenant$ = GetDistributionTenant$;
exports.GetDistributionTenantByDomain$ = GetDistributionTenantByDomain$;
exports.GetDistributionTenantByDomainCommand = GetDistributionTenantByDomainCommand;
exports.GetDistributionTenantByDomainRequest$ = GetDistributionTenantByDomainRequest$;
exports.GetDistributionTenantByDomainResult$ = GetDistributionTenantByDomainResult$;
exports.GetDistributionTenantCommand = GetDistributionTenantCommand;
exports.GetDistributionTenantRequest$ = GetDistributionTenantRequest$;
exports.GetDistributionTenantResult$ = GetDistributionTenantResult$;
exports.GetFieldLevelEncryption$ = GetFieldLevelEncryption$;
exports.GetFieldLevelEncryptionCommand = GetFieldLevelEncryptionCommand;
exports.GetFieldLevelEncryptionConfig$ = GetFieldLevelEncryptionConfig$;
exports.GetFieldLevelEncryptionConfigCommand = GetFieldLevelEncryptionConfigCommand;
exports.GetFieldLevelEncryptionConfigRequest$ = GetFieldLevelEncryptionConfigRequest$;
exports.GetFieldLevelEncryptionConfigResult$ = GetFieldLevelEncryptionConfigResult$;
exports.GetFieldLevelEncryptionProfile$ = GetFieldLevelEncryptionProfile$;
exports.GetFieldLevelEncryptionProfileCommand = GetFieldLevelEncryptionProfileCommand;
exports.GetFieldLevelEncryptionProfileConfig$ = GetFieldLevelEncryptionProfileConfig$;
exports.GetFieldLevelEncryptionProfileConfigCommand = GetFieldLevelEncryptionProfileConfigCommand;
exports.GetFieldLevelEncryptionProfileConfigRequest$ = GetFieldLevelEncryptionProfileConfigRequest$;
exports.GetFieldLevelEncryptionProfileConfigResult$ = GetFieldLevelEncryptionProfileConfigResult$;
exports.GetFieldLevelEncryptionProfileRequest$ = GetFieldLevelEncryptionProfileRequest$;
exports.GetFieldLevelEncryptionProfileResult$ = GetFieldLevelEncryptionProfileResult$;
exports.GetFieldLevelEncryptionRequest$ = GetFieldLevelEncryptionRequest$;
exports.GetFieldLevelEncryptionResult$ = GetFieldLevelEncryptionResult$;
exports.GetFunction$ = GetFunction$;
exports.GetFunctionCommand = GetFunctionCommand;
exports.GetFunctionRequest$ = GetFunctionRequest$;
exports.GetFunctionResult$ = GetFunctionResult$;
exports.GetInvalidation$ = GetInvalidation$;
exports.GetInvalidationCommand = GetInvalidationCommand;
exports.GetInvalidationForDistributionTenant$ = GetInvalidationForDistributionTenant$;
exports.GetInvalidationForDistributionTenantCommand = GetInvalidationForDistributionTenantCommand;
exports.GetInvalidationForDistributionTenantRequest$ = GetInvalidationForDistributionTenantRequest$;
exports.GetInvalidationForDistributionTenantResult$ = GetInvalidationForDistributionTenantResult$;
exports.GetInvalidationRequest$ = GetInvalidationRequest$;
exports.GetInvalidationResult$ = GetInvalidationResult$;
exports.GetKeyGroup$ = GetKeyGroup$;
exports.GetKeyGroupCommand = GetKeyGroupCommand;
exports.GetKeyGroupConfig$ = GetKeyGroupConfig$;
exports.GetKeyGroupConfigCommand = GetKeyGroupConfigCommand;
exports.GetKeyGroupConfigRequest$ = GetKeyGroupConfigRequest$;
exports.GetKeyGroupConfigResult$ = GetKeyGroupConfigResult$;
exports.GetKeyGroupRequest$ = GetKeyGroupRequest$;
exports.GetKeyGroupResult$ = GetKeyGroupResult$;
exports.GetManagedCertificateDetails$ = GetManagedCertificateDetails$;
exports.GetManagedCertificateDetailsCommand = GetManagedCertificateDetailsCommand;
exports.GetManagedCertificateDetailsRequest$ = GetManagedCertificateDetailsRequest$;
exports.GetManagedCertificateDetailsResult$ = GetManagedCertificateDetailsResult$;
exports.GetMonitoringSubscription$ = GetMonitoringSubscription$;
exports.GetMonitoringSubscriptionCommand = GetMonitoringSubscriptionCommand;
exports.GetMonitoringSubscriptionRequest$ = GetMonitoringSubscriptionRequest$;
exports.GetMonitoringSubscriptionResult$ = GetMonitoringSubscriptionResult$;
exports.GetOriginAccessControl$ = GetOriginAccessControl$;
exports.GetOriginAccessControlCommand = GetOriginAccessControlCommand;
exports.GetOriginAccessControlConfig$ = GetOriginAccessControlConfig$;
exports.GetOriginAccessControlConfigCommand = GetOriginAccessControlConfigCommand;
exports.GetOriginAccessControlConfigRequest$ = GetOriginAccessControlConfigRequest$;
exports.GetOriginAccessControlConfigResult$ = GetOriginAccessControlConfigResult$;
exports.GetOriginAccessControlRequest$ = GetOriginAccessControlRequest$;
exports.GetOriginAccessControlResult$ = GetOriginAccessControlResult$;
exports.GetOriginRequestPolicy$ = GetOriginRequestPolicy$;
exports.GetOriginRequestPolicyCommand = GetOriginRequestPolicyCommand;
exports.GetOriginRequestPolicyConfig$ = GetOriginRequestPolicyConfig$;
exports.GetOriginRequestPolicyConfigCommand = GetOriginRequestPolicyConfigCommand;
exports.GetOriginRequestPolicyConfigRequest$ = GetOriginRequestPolicyConfigRequest$;
exports.GetOriginRequestPolicyConfigResult$ = GetOriginRequestPolicyConfigResult$;
exports.GetOriginRequestPolicyRequest$ = GetOriginRequestPolicyRequest$;
exports.GetOriginRequestPolicyResult$ = GetOriginRequestPolicyResult$;
exports.GetPublicKey$ = GetPublicKey$;
exports.GetPublicKeyCommand = GetPublicKeyCommand;
exports.GetPublicKeyConfig$ = GetPublicKeyConfig$;
exports.GetPublicKeyConfigCommand = GetPublicKeyConfigCommand;
exports.GetPublicKeyConfigRequest$ = GetPublicKeyConfigRequest$;
exports.GetPublicKeyConfigResult$ = GetPublicKeyConfigResult$;
exports.GetPublicKeyRequest$ = GetPublicKeyRequest$;
exports.GetPublicKeyResult$ = GetPublicKeyResult$;
exports.GetRealtimeLogConfig$ = GetRealtimeLogConfig$;
exports.GetRealtimeLogConfigCommand = GetRealtimeLogConfigCommand;
exports.GetRealtimeLogConfigRequest$ = GetRealtimeLogConfigRequest$;
exports.GetRealtimeLogConfigResult$ = GetRealtimeLogConfigResult$;
exports.GetResourcePolicy$ = GetResourcePolicy$;
exports.GetResourcePolicyCommand = GetResourcePolicyCommand;
exports.GetResourcePolicyRequest$ = GetResourcePolicyRequest$;
exports.GetResourcePolicyResult$ = GetResourcePolicyResult$;
exports.GetResponseHeadersPolicy$ = GetResponseHeadersPolicy$;
exports.GetResponseHeadersPolicyCommand = GetResponseHeadersPolicyCommand;
exports.GetResponseHeadersPolicyConfig$ = GetResponseHeadersPolicyConfig$;
exports.GetResponseHeadersPolicyConfigCommand = GetResponseHeadersPolicyConfigCommand;
exports.GetResponseHeadersPolicyConfigRequest$ = GetResponseHeadersPolicyConfigRequest$;
exports.GetResponseHeadersPolicyConfigResult$ = GetResponseHeadersPolicyConfigResult$;
exports.GetResponseHeadersPolicyRequest$ = GetResponseHeadersPolicyRequest$;
exports.GetResponseHeadersPolicyResult$ = GetResponseHeadersPolicyResult$;
exports.GetStreamingDistribution$ = GetStreamingDistribution$;
exports.GetStreamingDistributionCommand = GetStreamingDistributionCommand;
exports.GetStreamingDistributionConfig$ = GetStreamingDistributionConfig$;
exports.GetStreamingDistributionConfigCommand = GetStreamingDistributionConfigCommand;
exports.GetStreamingDistributionConfigRequest$ = GetStreamingDistributionConfigRequest$;
exports.GetStreamingDistributionConfigResult$ = GetStreamingDistributionConfigResult$;
exports.GetStreamingDistributionRequest$ = GetStreamingDistributionRequest$;
exports.GetStreamingDistributionResult$ = GetStreamingDistributionResult$;
exports.GetTrustStore$ = GetTrustStore$;
exports.GetTrustStoreCommand = GetTrustStoreCommand;
exports.GetTrustStoreRequest$ = GetTrustStoreRequest$;
exports.GetTrustStoreResult$ = GetTrustStoreResult$;
exports.GetVpcOrigin$ = GetVpcOrigin$;
exports.GetVpcOriginCommand = GetVpcOriginCommand;
exports.GetVpcOriginRequest$ = GetVpcOriginRequest$;
exports.GetVpcOriginResult$ = GetVpcOriginResult$;
exports.GrpcConfig$ = GrpcConfig$;
exports.Headers$ = Headers$;
exports.HttpVersion = HttpVersion;
exports.ICPRecordalStatus = ICPRecordalStatus;
exports.IllegalDelete = IllegalDelete;
exports.IllegalDelete$ = IllegalDelete$;
exports.IllegalFieldLevelEncryptionConfigAssociationWithCacheBehavior = IllegalFieldLevelEncryptionConfigAssociationWithCacheBehavior;
exports.IllegalFieldLevelEncryptionConfigAssociationWithCacheBehavior$ = IllegalFieldLevelEncryptionConfigAssociationWithCacheBehavior$;
exports.IllegalOriginAccessConfiguration = IllegalOriginAccessConfiguration;
exports.IllegalOriginAccessConfiguration$ = IllegalOriginAccessConfiguration$;
exports.IllegalUpdate = IllegalUpdate;
exports.IllegalUpdate$ = IllegalUpdate$;
exports.ImportSource$ = ImportSource$;
exports.ImportSourceType = ImportSourceType;
exports.InconsistentQuantities = InconsistentQuantities;
exports.InconsistentQuantities$ = InconsistentQuantities$;
exports.InvalidArgument = InvalidArgument;
exports.InvalidArgument$ = InvalidArgument$;
exports.InvalidAssociation = InvalidAssociation;
exports.InvalidAssociation$ = InvalidAssociation$;
exports.InvalidDefaultRootObject = InvalidDefaultRootObject;
exports.InvalidDefaultRootObject$ = InvalidDefaultRootObject$;
exports.InvalidDomainNameForOriginAccessControl = InvalidDomainNameForOriginAccessControl;
exports.InvalidDomainNameForOriginAccessControl$ = InvalidDomainNameForOriginAccessControl$;
exports.InvalidErrorCode = InvalidErrorCode;
exports.InvalidErrorCode$ = InvalidErrorCode$;
exports.InvalidForwardCookies = InvalidForwardCookies;
exports.InvalidForwardCookies$ = InvalidForwardCookies$;
exports.InvalidFunctionAssociation = InvalidFunctionAssociation;
exports.InvalidFunctionAssociation$ = InvalidFunctionAssociation$;
exports.InvalidGeoRestrictionParameter = InvalidGeoRestrictionParameter;
exports.InvalidGeoRestrictionParameter$ = InvalidGeoRestrictionParameter$;
exports.InvalidHeadersForS3Origin = InvalidHeadersForS3Origin;
exports.InvalidHeadersForS3Origin$ = InvalidHeadersForS3Origin$;
exports.InvalidIfMatchVersion = InvalidIfMatchVersion;
exports.InvalidIfMatchVersion$ = InvalidIfMatchVersion$;
exports.InvalidLambdaFunctionAssociation = InvalidLambdaFunctionAssociation;
exports.InvalidLambdaFunctionAssociation$ = InvalidLambdaFunctionAssociation$;
exports.InvalidLocationCode = InvalidLocationCode;
exports.InvalidLocationCode$ = InvalidLocationCode$;
exports.InvalidMinimumProtocolVersion = InvalidMinimumProtocolVersion;
exports.InvalidMinimumProtocolVersion$ = InvalidMinimumProtocolVersion$;
exports.InvalidOrigin = InvalidOrigin;
exports.InvalidOrigin$ = InvalidOrigin$;
exports.InvalidOriginAccessControl = InvalidOriginAccessControl;
exports.InvalidOriginAccessControl$ = InvalidOriginAccessControl$;
exports.InvalidOriginAccessIdentity = InvalidOriginAccessIdentity;
exports.InvalidOriginAccessIdentity$ = InvalidOriginAccessIdentity$;
exports.InvalidOriginKeepaliveTimeout = InvalidOriginKeepaliveTimeout;
exports.InvalidOriginKeepaliveTimeout$ = InvalidOriginKeepaliveTimeout$;
exports.InvalidOriginReadTimeout = InvalidOriginReadTimeout;
exports.InvalidOriginReadTimeout$ = InvalidOriginReadTimeout$;
exports.InvalidProtocolSettings = InvalidProtocolSettings;
exports.InvalidProtocolSettings$ = InvalidProtocolSettings$;
exports.InvalidQueryStringParameters = InvalidQueryStringParameters;
exports.InvalidQueryStringParameters$ = InvalidQueryStringParameters$;
exports.InvalidRelativePath = InvalidRelativePath;
exports.InvalidRelativePath$ = InvalidRelativePath$;
exports.InvalidRequiredProtocol = InvalidRequiredProtocol;
exports.InvalidRequiredProtocol$ = InvalidRequiredProtocol$;
exports.InvalidResponseCode = InvalidResponseCode;
exports.InvalidResponseCode$ = InvalidResponseCode$;
exports.InvalidTTLOrder = InvalidTTLOrder;
exports.InvalidTTLOrder$ = InvalidTTLOrder$;
exports.InvalidTagging = InvalidTagging;
exports.InvalidTagging$ = InvalidTagging$;
exports.InvalidViewerCertificate = InvalidViewerCertificate;
exports.InvalidViewerCertificate$ = InvalidViewerCertificate$;
exports.InvalidWebACLId = InvalidWebACLId;
exports.InvalidWebACLId$ = InvalidWebACLId$;
exports.Invalidation$ = Invalidation$;
exports.InvalidationBatch$ = InvalidationBatch$;
exports.InvalidationList$ = InvalidationList$;
exports.InvalidationSummary$ = InvalidationSummary$;
exports.IpAddressType = IpAddressType;
exports.IpamCidrConfig$ = IpamCidrConfig$;
exports.IpamCidrStatus = IpamCidrStatus;
exports.IpamConfig$ = IpamConfig$;
exports.ItemSelection = ItemSelection;
exports.KGKeyPairIds$ = KGKeyPairIds$;
exports.KeyGroup$ = KeyGroup$;
exports.KeyGroupAlreadyExists = KeyGroupAlreadyExists;
exports.KeyGroupAlreadyExists$ = KeyGroupAlreadyExists$;
exports.KeyGroupConfig$ = KeyGroupConfig$;
exports.KeyGroupList$ = KeyGroupList$;
exports.KeyGroupSummary$ = KeyGroupSummary$;
exports.KeyPairIds$ = KeyPairIds$;
exports.KeyValueStore$ = KeyValueStore$;
exports.KeyValueStoreAssociation$ = KeyValueStoreAssociation$;
exports.KeyValueStoreAssociations$ = KeyValueStoreAssociations$;
exports.KeyValueStoreList$ = KeyValueStoreList$;
exports.KinesisStreamConfig$ = KinesisStreamConfig$;
exports.LambdaFunctionAssociation$ = LambdaFunctionAssociation$;
exports.LambdaFunctionAssociations$ = LambdaFunctionAssociations$;
exports.ListAnycastIpLists$ = ListAnycastIpLists$;
exports.ListAnycastIpListsCommand = ListAnycastIpListsCommand;
exports.ListAnycastIpListsRequest$ = ListAnycastIpListsRequest$;
exports.ListAnycastIpListsResult$ = ListAnycastIpListsResult$;
exports.ListCachePolicies$ = ListCachePolicies$;
exports.ListCachePoliciesCommand = ListCachePoliciesCommand;
exports.ListCachePoliciesRequest$ = ListCachePoliciesRequest$;
exports.ListCachePoliciesResult$ = ListCachePoliciesResult$;
exports.ListCloudFrontOriginAccessIdentities$ = ListCloudFrontOriginAccessIdentities$;
exports.ListCloudFrontOriginAccessIdentitiesCommand = ListCloudFrontOriginAccessIdentitiesCommand;
exports.ListCloudFrontOriginAccessIdentitiesRequest$ = ListCloudFrontOriginAccessIdentitiesRequest$;
exports.ListCloudFrontOriginAccessIdentitiesResult$ = ListCloudFrontOriginAccessIdentitiesResult$;
exports.ListConflictingAliases$ = ListConflictingAliases$;
exports.ListConflictingAliasesCommand = ListConflictingAliasesCommand;
exports.ListConflictingAliasesRequest$ = ListConflictingAliasesRequest$;
exports.ListConflictingAliasesResult$ = ListConflictingAliasesResult$;
exports.ListConnectionFunctions$ = ListConnectionFunctions$;
exports.ListConnectionFunctionsCommand = ListConnectionFunctionsCommand;
exports.ListConnectionFunctionsRequest$ = ListConnectionFunctionsRequest$;
exports.ListConnectionFunctionsResult$ = ListConnectionFunctionsResult$;
exports.ListConnectionGroups$ = ListConnectionGroups$;
exports.ListConnectionGroupsCommand = ListConnectionGroupsCommand;
exports.ListConnectionGroupsRequest$ = ListConnectionGroupsRequest$;
exports.ListConnectionGroupsResult$ = ListConnectionGroupsResult$;
exports.ListContinuousDeploymentPolicies$ = ListContinuousDeploymentPolicies$;
exports.ListContinuousDeploymentPoliciesCommand = ListContinuousDeploymentPoliciesCommand;
exports.ListContinuousDeploymentPoliciesRequest$ = ListContinuousDeploymentPoliciesRequest$;
exports.ListContinuousDeploymentPoliciesResult$ = ListContinuousDeploymentPoliciesResult$;
exports.ListDistributionTenants$ = ListDistributionTenants$;
exports.ListDistributionTenantsByCustomization$ = ListDistributionTenantsByCustomization$;
exports.ListDistributionTenantsByCustomizationCommand = ListDistributionTenantsByCustomizationCommand;
exports.ListDistributionTenantsByCustomizationRequest$ = ListDistributionTenantsByCustomizationRequest$;
exports.ListDistributionTenantsByCustomizationResult$ = ListDistributionTenantsByCustomizationResult$;
exports.ListDistributionTenantsCommand = ListDistributionTenantsCommand;
exports.ListDistributionTenantsRequest$ = ListDistributionTenantsRequest$;
exports.ListDistributionTenantsResult$ = ListDistributionTenantsResult$;
exports.ListDistributions$ = ListDistributions$;
exports.ListDistributionsByAnycastIpListId$ = ListDistributionsByAnycastIpListId$;
exports.ListDistributionsByAnycastIpListIdCommand = ListDistributionsByAnycastIpListIdCommand;
exports.ListDistributionsByAnycastIpListIdRequest$ = ListDistributionsByAnycastIpListIdRequest$;
exports.ListDistributionsByAnycastIpListIdResult$ = ListDistributionsByAnycastIpListIdResult$;
exports.ListDistributionsByCachePolicyId$ = ListDistributionsByCachePolicyId$;
exports.ListDistributionsByCachePolicyIdCommand = ListDistributionsByCachePolicyIdCommand;
exports.ListDistributionsByCachePolicyIdRequest$ = ListDistributionsByCachePolicyIdRequest$;
exports.ListDistributionsByCachePolicyIdResult$ = ListDistributionsByCachePolicyIdResult$;
exports.ListDistributionsByConnectionFunction$ = ListDistributionsByConnectionFunction$;
exports.ListDistributionsByConnectionFunctionCommand = ListDistributionsByConnectionFunctionCommand;
exports.ListDistributionsByConnectionFunctionRequest$ = ListDistributionsByConnectionFunctionRequest$;
exports.ListDistributionsByConnectionFunctionResult$ = ListDistributionsByConnectionFunctionResult$;
exports.ListDistributionsByConnectionMode$ = ListDistributionsByConnectionMode$;
exports.ListDistributionsByConnectionModeCommand = ListDistributionsByConnectionModeCommand;
exports.ListDistributionsByConnectionModeRequest$ = ListDistributionsByConnectionModeRequest$;
exports.ListDistributionsByConnectionModeResult$ = ListDistributionsByConnectionModeResult$;
exports.ListDistributionsByKeyGroup$ = ListDistributionsByKeyGroup$;
exports.ListDistributionsByKeyGroupCommand = ListDistributionsByKeyGroupCommand;
exports.ListDistributionsByKeyGroupRequest$ = ListDistributionsByKeyGroupRequest$;
exports.ListDistributionsByKeyGroupResult$ = ListDistributionsByKeyGroupResult$;
exports.ListDistributionsByOriginRequestPolicyId$ = ListDistributionsByOriginRequestPolicyId$;
exports.ListDistributionsByOriginRequestPolicyIdCommand = ListDistributionsByOriginRequestPolicyIdCommand;
exports.ListDistributionsByOriginRequestPolicyIdRequest$ = ListDistributionsByOriginRequestPolicyIdRequest$;
exports.ListDistributionsByOriginRequestPolicyIdResult$ = ListDistributionsByOriginRequestPolicyIdResult$;
exports.ListDistributionsByOwnedResource$ = ListDistributionsByOwnedResource$;
exports.ListDistributionsByOwnedResourceCommand = ListDistributionsByOwnedResourceCommand;
exports.ListDistributionsByOwnedResourceRequest$ = ListDistributionsByOwnedResourceRequest$;
exports.ListDistributionsByOwnedResourceResult$ = ListDistributionsByOwnedResourceResult$;
exports.ListDistributionsByRealtimeLogConfig$ = ListDistributionsByRealtimeLogConfig$;
exports.ListDistributionsByRealtimeLogConfigCommand = ListDistributionsByRealtimeLogConfigCommand;
exports.ListDistributionsByRealtimeLogConfigRequest$ = ListDistributionsByRealtimeLogConfigRequest$;
exports.ListDistributionsByRealtimeLogConfigResult$ = ListDistributionsByRealtimeLogConfigResult$;
exports.ListDistributionsByResponseHeadersPolicyId$ = ListDistributionsByResponseHeadersPolicyId$;
exports.ListDistributionsByResponseHeadersPolicyIdCommand = ListDistributionsByResponseHeadersPolicyIdCommand;
exports.ListDistributionsByResponseHeadersPolicyIdRequest$ = ListDistributionsByResponseHeadersPolicyIdRequest$;
exports.ListDistributionsByResponseHeadersPolicyIdResult$ = ListDistributionsByResponseHeadersPolicyIdResult$;
exports.ListDistributionsByTrustStore$ = ListDistributionsByTrustStore$;
exports.ListDistributionsByTrustStoreCommand = ListDistributionsByTrustStoreCommand;
exports.ListDistributionsByTrustStoreRequest$ = ListDistributionsByTrustStoreRequest$;
exports.ListDistributionsByTrustStoreResult$ = ListDistributionsByTrustStoreResult$;
exports.ListDistributionsByVpcOriginId$ = ListDistributionsByVpcOriginId$;
exports.ListDistributionsByVpcOriginIdCommand = ListDistributionsByVpcOriginIdCommand;
exports.ListDistributionsByVpcOriginIdRequest$ = ListDistributionsByVpcOriginIdRequest$;
exports.ListDistributionsByVpcOriginIdResult$ = ListDistributionsByVpcOriginIdResult$;
exports.ListDistributionsByWebACLId$ = ListDistributionsByWebACLId$;
exports.ListDistributionsByWebACLIdCommand = ListDistributionsByWebACLIdCommand;
exports.ListDistributionsByWebACLIdRequest$ = ListDistributionsByWebACLIdRequest$;
exports.ListDistributionsByWebACLIdResult$ = ListDistributionsByWebACLIdResult$;
exports.ListDistributionsCommand = ListDistributionsCommand;
exports.ListDistributionsRequest$ = ListDistributionsRequest$;
exports.ListDistributionsResult$ = ListDistributionsResult$;
exports.ListDomainConflicts$ = ListDomainConflicts$;
exports.ListDomainConflictsCommand = ListDomainConflictsCommand;
exports.ListDomainConflictsRequest$ = ListDomainConflictsRequest$;
exports.ListDomainConflictsResult$ = ListDomainConflictsResult$;
exports.ListFieldLevelEncryptionConfigs$ = ListFieldLevelEncryptionConfigs$;
exports.ListFieldLevelEncryptionConfigsCommand = ListFieldLevelEncryptionConfigsCommand;
exports.ListFieldLevelEncryptionConfigsRequest$ = ListFieldLevelEncryptionConfigsRequest$;
exports.ListFieldLevelEncryptionConfigsResult$ = ListFieldLevelEncryptionConfigsResult$;
exports.ListFieldLevelEncryptionProfiles$ = ListFieldLevelEncryptionProfiles$;
exports.ListFieldLevelEncryptionProfilesCommand = ListFieldLevelEncryptionProfilesCommand;
exports.ListFieldLevelEncryptionProfilesRequest$ = ListFieldLevelEncryptionProfilesRequest$;
exports.ListFieldLevelEncryptionProfilesResult$ = ListFieldLevelEncryptionProfilesResult$;
exports.ListFunctions$ = ListFunctions$;
exports.ListFunctionsCommand = ListFunctionsCommand;
exports.ListFunctionsRequest$ = ListFunctionsRequest$;
exports.ListFunctionsResult$ = ListFunctionsResult$;
exports.ListInvalidations$ = ListInvalidations$;
exports.ListInvalidationsCommand = ListInvalidationsCommand;
exports.ListInvalidationsForDistributionTenant$ = ListInvalidationsForDistributionTenant$;
exports.ListInvalidationsForDistributionTenantCommand = ListInvalidationsForDistributionTenantCommand;
exports.ListInvalidationsForDistributionTenantRequest$ = ListInvalidationsForDistributionTenantRequest$;
exports.ListInvalidationsForDistributionTenantResult$ = ListInvalidationsForDistributionTenantResult$;
exports.ListInvalidationsRequest$ = ListInvalidationsRequest$;
exports.ListInvalidationsResult$ = ListInvalidationsResult$;
exports.ListKeyGroups$ = ListKeyGroups$;
exports.ListKeyGroupsCommand = ListKeyGroupsCommand;
exports.ListKeyGroupsRequest$ = ListKeyGroupsRequest$;
exports.ListKeyGroupsResult$ = ListKeyGroupsResult$;
exports.ListKeyValueStores$ = ListKeyValueStores$;
exports.ListKeyValueStoresCommand = ListKeyValueStoresCommand;
exports.ListKeyValueStoresRequest$ = ListKeyValueStoresRequest$;
exports.ListKeyValueStoresResult$ = ListKeyValueStoresResult$;
exports.ListOriginAccessControls$ = ListOriginAccessControls$;
exports.ListOriginAccessControlsCommand = ListOriginAccessControlsCommand;
exports.ListOriginAccessControlsRequest$ = ListOriginAccessControlsRequest$;
exports.ListOriginAccessControlsResult$ = ListOriginAccessControlsResult$;
exports.ListOriginRequestPolicies$ = ListOriginRequestPolicies$;
exports.ListOriginRequestPoliciesCommand = ListOriginRequestPoliciesCommand;
exports.ListOriginRequestPoliciesRequest$ = ListOriginRequestPoliciesRequest$;
exports.ListOriginRequestPoliciesResult$ = ListOriginRequestPoliciesResult$;
exports.ListPublicKeys$ = ListPublicKeys$;
exports.ListPublicKeysCommand = ListPublicKeysCommand;
exports.ListPublicKeysRequest$ = ListPublicKeysRequest$;
exports.ListPublicKeysResult$ = ListPublicKeysResult$;
exports.ListRealtimeLogConfigs$ = ListRealtimeLogConfigs$;
exports.ListRealtimeLogConfigsCommand = ListRealtimeLogConfigsCommand;
exports.ListRealtimeLogConfigsRequest$ = ListRealtimeLogConfigsRequest$;
exports.ListRealtimeLogConfigsResult$ = ListRealtimeLogConfigsResult$;
exports.ListResponseHeadersPolicies$ = ListResponseHeadersPolicies$;
exports.ListResponseHeadersPoliciesCommand = ListResponseHeadersPoliciesCommand;
exports.ListResponseHeadersPoliciesRequest$ = ListResponseHeadersPoliciesRequest$;
exports.ListResponseHeadersPoliciesResult$ = ListResponseHeadersPoliciesResult$;
exports.ListStreamingDistributions$ = ListStreamingDistributions$;
exports.ListStreamingDistributionsCommand = ListStreamingDistributionsCommand;
exports.ListStreamingDistributionsRequest$ = ListStreamingDistributionsRequest$;
exports.ListStreamingDistributionsResult$ = ListStreamingDistributionsResult$;
exports.ListTagsForResource$ = ListTagsForResource$;
exports.ListTagsForResourceCommand = ListTagsForResourceCommand;
exports.ListTagsForResourceRequest$ = ListTagsForResourceRequest$;
exports.ListTagsForResourceResult$ = ListTagsForResourceResult$;
exports.ListTrustStores$ = ListTrustStores$;
exports.ListTrustStoresCommand = ListTrustStoresCommand;
exports.ListTrustStoresRequest$ = ListTrustStoresRequest$;
exports.ListTrustStoresResult$ = ListTrustStoresResult$;
exports.ListVpcOrigins$ = ListVpcOrigins$;
exports.ListVpcOriginsCommand = ListVpcOriginsCommand;
exports.ListVpcOriginsRequest$ = ListVpcOriginsRequest$;
exports.ListVpcOriginsResult$ = ListVpcOriginsResult$;
exports.LoggingConfig$ = LoggingConfig$;
exports.ManagedCertificateDetails$ = ManagedCertificateDetails$;
exports.ManagedCertificateRequest$ = ManagedCertificateRequest$;
exports.ManagedCertificateStatus = ManagedCertificateStatus;
exports.Method = Method;
exports.MinimumProtocolVersion = MinimumProtocolVersion;
exports.MissingBody = MissingBody;
exports.MissingBody$ = MissingBody$;
exports.MonitoringSubscription$ = MonitoringSubscription$;
exports.MonitoringSubscriptionAlreadyExists = MonitoringSubscriptionAlreadyExists;
exports.MonitoringSubscriptionAlreadyExists$ = MonitoringSubscriptionAlreadyExists$;
exports.NoSuchCachePolicy = NoSuchCachePolicy;
exports.NoSuchCachePolicy$ = NoSuchCachePolicy$;
exports.NoSuchCloudFrontOriginAccessIdentity = NoSuchCloudFrontOriginAccessIdentity;
exports.NoSuchCloudFrontOriginAccessIdentity$ = NoSuchCloudFrontOriginAccessIdentity$;
exports.NoSuchContinuousDeploymentPolicy = NoSuchContinuousDeploymentPolicy;
exports.NoSuchContinuousDeploymentPolicy$ = NoSuchContinuousDeploymentPolicy$;
exports.NoSuchDistribution = NoSuchDistribution;
exports.NoSuchDistribution$ = NoSuchDistribution$;
exports.NoSuchFieldLevelEncryptionConfig = NoSuchFieldLevelEncryptionConfig;
exports.NoSuchFieldLevelEncryptionConfig$ = NoSuchFieldLevelEncryptionConfig$;
exports.NoSuchFieldLevelEncryptionProfile = NoSuchFieldLevelEncryptionProfile;
exports.NoSuchFieldLevelEncryptionProfile$ = NoSuchFieldLevelEncryptionProfile$;
exports.NoSuchFunctionExists = NoSuchFunctionExists;
exports.NoSuchFunctionExists$ = NoSuchFunctionExists$;
exports.NoSuchInvalidation = NoSuchInvalidation;
exports.NoSuchInvalidation$ = NoSuchInvalidation$;
exports.NoSuchMonitoringSubscription = NoSuchMonitoringSubscription;
exports.NoSuchMonitoringSubscription$ = NoSuchMonitoringSubscription$;
exports.NoSuchOrigin = NoSuchOrigin;
exports.NoSuchOrigin$ = NoSuchOrigin$;
exports.NoSuchOriginAccessControl = NoSuchOriginAccessControl;
exports.NoSuchOriginAccessControl$ = NoSuchOriginAccessControl$;
exports.NoSuchOriginRequestPolicy = NoSuchOriginRequestPolicy;
exports.NoSuchOriginRequestPolicy$ = NoSuchOriginRequestPolicy$;
exports.NoSuchPublicKey = NoSuchPublicKey;
exports.NoSuchPublicKey$ = NoSuchPublicKey$;
exports.NoSuchRealtimeLogConfig = NoSuchRealtimeLogConfig;
exports.NoSuchRealtimeLogConfig$ = NoSuchRealtimeLogConfig$;
exports.NoSuchResource = NoSuchResource;
exports.NoSuchResource$ = NoSuchResource$;
exports.NoSuchResponseHeadersPolicy = NoSuchResponseHeadersPolicy;
exports.NoSuchResponseHeadersPolicy$ = NoSuchResponseHeadersPolicy$;
exports.NoSuchStreamingDistribution = NoSuchStreamingDistribution;
exports.NoSuchStreamingDistribution$ = NoSuchStreamingDistribution$;
exports.Origin$ = Origin$;
exports.OriginAccessControl$ = OriginAccessControl$;
exports.OriginAccessControlAlreadyExists = OriginAccessControlAlreadyExists;
exports.OriginAccessControlAlreadyExists$ = OriginAccessControlAlreadyExists$;
exports.OriginAccessControlConfig$ = OriginAccessControlConfig$;
exports.OriginAccessControlInUse = OriginAccessControlInUse;
exports.OriginAccessControlInUse$ = OriginAccessControlInUse$;
exports.OriginAccessControlList$ = OriginAccessControlList$;
exports.OriginAccessControlOriginTypes = OriginAccessControlOriginTypes;
exports.OriginAccessControlSigningBehaviors = OriginAccessControlSigningBehaviors;
exports.OriginAccessControlSigningProtocols = OriginAccessControlSigningProtocols;
exports.OriginAccessControlSummary$ = OriginAccessControlSummary$;
exports.OriginCustomHeader$ = OriginCustomHeader$;
exports.OriginGroup$ = OriginGroup$;
exports.OriginGroupFailoverCriteria$ = OriginGroupFailoverCriteria$;
exports.OriginGroupMember$ = OriginGroupMember$;
exports.OriginGroupMembers$ = OriginGroupMembers$;
exports.OriginGroupSelectionCriteria = OriginGroupSelectionCriteria;
exports.OriginGroups$ = OriginGroups$;
exports.OriginProtocolPolicy = OriginProtocolPolicy;
exports.OriginRequestPolicy$ = OriginRequestPolicy$;
exports.OriginRequestPolicyAlreadyExists = OriginRequestPolicyAlreadyExists;
exports.OriginRequestPolicyAlreadyExists$ = OriginRequestPolicyAlreadyExists$;
exports.OriginRequestPolicyConfig$ = OriginRequestPolicyConfig$;
exports.OriginRequestPolicyCookieBehavior = OriginRequestPolicyCookieBehavior;
exports.OriginRequestPolicyCookiesConfig$ = OriginRequestPolicyCookiesConfig$;
exports.OriginRequestPolicyHeaderBehavior = OriginRequestPolicyHeaderBehavior;
exports.OriginRequestPolicyHeadersConfig$ = OriginRequestPolicyHeadersConfig$;
exports.OriginRequestPolicyInUse = OriginRequestPolicyInUse;
exports.OriginRequestPolicyInUse$ = OriginRequestPolicyInUse$;
exports.OriginRequestPolicyList$ = OriginRequestPolicyList$;
exports.OriginRequestPolicyQueryStringBehavior = OriginRequestPolicyQueryStringBehavior;
exports.OriginRequestPolicyQueryStringsConfig$ = OriginRequestPolicyQueryStringsConfig$;
exports.OriginRequestPolicySummary$ = OriginRequestPolicySummary$;
exports.OriginRequestPolicyType = OriginRequestPolicyType;
exports.OriginShield$ = OriginShield$;
exports.OriginSslProtocols$ = OriginSslProtocols$;
exports.Origins$ = Origins$;
exports.Parameter$ = Parameter$;
exports.ParameterDefinition$ = ParameterDefinition$;
exports.ParameterDefinitionSchema$ = ParameterDefinitionSchema$;
exports.ParametersInCacheKeyAndForwardedToOrigin$ = ParametersInCacheKeyAndForwardedToOrigin$;
exports.Paths$ = Paths$;
exports.PreconditionFailed = PreconditionFailed;
exports.PreconditionFailed$ = PreconditionFailed$;
exports.PriceClass = PriceClass;
exports.PublicKey$ = PublicKey$;
exports.PublicKeyAlreadyExists = PublicKeyAlreadyExists;
exports.PublicKeyAlreadyExists$ = PublicKeyAlreadyExists$;
exports.PublicKeyConfig$ = PublicKeyConfig$;
exports.PublicKeyInUse = PublicKeyInUse;
exports.PublicKeyInUse$ = PublicKeyInUse$;
exports.PublicKeyList$ = PublicKeyList$;
exports.PublicKeySummary$ = PublicKeySummary$;
exports.PublishConnectionFunction$ = PublishConnectionFunction$;
exports.PublishConnectionFunctionCommand = PublishConnectionFunctionCommand;
exports.PublishConnectionFunctionRequest$ = PublishConnectionFunctionRequest$;
exports.PublishConnectionFunctionResult$ = PublishConnectionFunctionResult$;
exports.PublishFunction$ = PublishFunction$;
exports.PublishFunctionCommand = PublishFunctionCommand;
exports.PublishFunctionRequest$ = PublishFunctionRequest$;
exports.PublishFunctionResult$ = PublishFunctionResult$;
exports.PutResourcePolicy$ = PutResourcePolicy$;
exports.PutResourcePolicyCommand = PutResourcePolicyCommand;
exports.PutResourcePolicyRequest$ = PutResourcePolicyRequest$;
exports.PutResourcePolicyResult$ = PutResourcePolicyResult$;
exports.QueryArgProfile$ = QueryArgProfile$;
exports.QueryArgProfileConfig$ = QueryArgProfileConfig$;
exports.QueryArgProfileEmpty = QueryArgProfileEmpty;
exports.QueryArgProfileEmpty$ = QueryArgProfileEmpty$;
exports.QueryArgProfiles$ = QueryArgProfiles$;
exports.QueryStringCacheKeys$ = QueryStringCacheKeys$;
exports.QueryStringNames$ = QueryStringNames$;
exports.RealtimeLogConfig$ = RealtimeLogConfig$;
exports.RealtimeLogConfigAlreadyExists = RealtimeLogConfigAlreadyExists;
exports.RealtimeLogConfigAlreadyExists$ = RealtimeLogConfigAlreadyExists$;
exports.RealtimeLogConfigInUse = RealtimeLogConfigInUse;
exports.RealtimeLogConfigInUse$ = RealtimeLogConfigInUse$;
exports.RealtimeLogConfigOwnerMismatch = RealtimeLogConfigOwnerMismatch;
exports.RealtimeLogConfigOwnerMismatch$ = RealtimeLogConfigOwnerMismatch$;
exports.RealtimeLogConfigs$ = RealtimeLogConfigs$;
exports.RealtimeMetricsSubscriptionConfig$ = RealtimeMetricsSubscriptionConfig$;
exports.RealtimeMetricsSubscriptionStatus = RealtimeMetricsSubscriptionStatus;
exports.ReferrerPolicyList = ReferrerPolicyList;
exports.ResourceInUse = ResourceInUse;
exports.ResourceInUse$ = ResourceInUse$;
exports.ResourceNotDisabled = ResourceNotDisabled;
exports.ResourceNotDisabled$ = ResourceNotDisabled$;
exports.ResponseHeadersPolicy$ = ResponseHeadersPolicy$;
exports.ResponseHeadersPolicyAccessControlAllowHeaders$ = ResponseHeadersPolicyAccessControlAllowHeaders$;
exports.ResponseHeadersPolicyAccessControlAllowMethods$ = ResponseHeadersPolicyAccessControlAllowMethods$;
exports.ResponseHeadersPolicyAccessControlAllowMethodsValues = ResponseHeadersPolicyAccessControlAllowMethodsValues;
exports.ResponseHeadersPolicyAccessControlAllowOrigins$ = ResponseHeadersPolicyAccessControlAllowOrigins$;
exports.ResponseHeadersPolicyAccessControlExposeHeaders$ = ResponseHeadersPolicyAccessControlExposeHeaders$;
exports.ResponseHeadersPolicyAlreadyExists = ResponseHeadersPolicyAlreadyExists;
exports.ResponseHeadersPolicyAlreadyExists$ = ResponseHeadersPolicyAlreadyExists$;
exports.ResponseHeadersPolicyConfig$ = ResponseHeadersPolicyConfig$;
exports.ResponseHeadersPolicyContentSecurityPolicy$ = ResponseHeadersPolicyContentSecurityPolicy$;
exports.ResponseHeadersPolicyContentTypeOptions$ = ResponseHeadersPolicyContentTypeOptions$;
exports.ResponseHeadersPolicyCorsConfig$ = ResponseHeadersPolicyCorsConfig$;
exports.ResponseHeadersPolicyCustomHeader$ = ResponseHeadersPolicyCustomHeader$;
exports.ResponseHeadersPolicyCustomHeadersConfig$ = ResponseHeadersPolicyCustomHeadersConfig$;
exports.ResponseHeadersPolicyFrameOptions$ = ResponseHeadersPolicyFrameOptions$;
exports.ResponseHeadersPolicyInUse = ResponseHeadersPolicyInUse;
exports.ResponseHeadersPolicyInUse$ = ResponseHeadersPolicyInUse$;
exports.ResponseHeadersPolicyList$ = ResponseHeadersPolicyList$;
exports.ResponseHeadersPolicyReferrerPolicy$ = ResponseHeadersPolicyReferrerPolicy$;
exports.ResponseHeadersPolicyRemoveHeader$ = ResponseHeadersPolicyRemoveHeader$;
exports.ResponseHeadersPolicyRemoveHeadersConfig$ = ResponseHeadersPolicyRemoveHeadersConfig$;
exports.ResponseHeadersPolicySecurityHeadersConfig$ = ResponseHeadersPolicySecurityHeadersConfig$;
exports.ResponseHeadersPolicyServerTimingHeadersConfig$ = ResponseHeadersPolicyServerTimingHeadersConfig$;
exports.ResponseHeadersPolicyStrictTransportSecurity$ = ResponseHeadersPolicyStrictTransportSecurity$;
exports.ResponseHeadersPolicySummary$ = ResponseHeadersPolicySummary$;
exports.ResponseHeadersPolicyType = ResponseHeadersPolicyType;
exports.ResponseHeadersPolicyXSSProtection$ = ResponseHeadersPolicyXSSProtection$;
exports.Restrictions$ = Restrictions$;
exports.S3Origin$ = S3Origin$;
exports.S3OriginConfig$ = S3OriginConfig$;
exports.SSLSupportMethod = SSLSupportMethod;
exports.SessionStickinessConfig$ = SessionStickinessConfig$;
exports.Signer$ = Signer$;
exports.SslProtocol = SslProtocol;
exports.StagingDistributionDnsNames$ = StagingDistributionDnsNames$;
exports.StagingDistributionInUse = StagingDistributionInUse;
exports.StagingDistributionInUse$ = StagingDistributionInUse$;
exports.StatusCodes$ = StatusCodes$;
exports.StreamingDistribution$ = StreamingDistribution$;
exports.StreamingDistributionAlreadyExists = StreamingDistributionAlreadyExists;
exports.StreamingDistributionAlreadyExists$ = StreamingDistributionAlreadyExists$;
exports.StreamingDistributionConfig$ = StreamingDistributionConfig$;
exports.StreamingDistributionConfigWithTags$ = StreamingDistributionConfigWithTags$;
exports.StreamingDistributionList$ = StreamingDistributionList$;
exports.StreamingDistributionNotDisabled = StreamingDistributionNotDisabled;
exports.StreamingDistributionNotDisabled$ = StreamingDistributionNotDisabled$;
exports.StreamingDistributionSummary$ = StreamingDistributionSummary$;
exports.StreamingLoggingConfig$ = StreamingLoggingConfig$;
exports.StringSchemaConfig$ = StringSchemaConfig$;
exports.Tag$ = Tag$;
exports.TagKeys$ = TagKeys$;
exports.TagResource$ = TagResource$;
exports.TagResourceCommand = TagResourceCommand;
exports.TagResourceRequest$ = TagResourceRequest$;
exports.Tags$ = Tags$;
exports.TenantConfig$ = TenantConfig$;
exports.TestConnectionFunction$ = TestConnectionFunction$;
exports.TestConnectionFunctionCommand = TestConnectionFunctionCommand;
exports.TestConnectionFunctionRequest$ = TestConnectionFunctionRequest$;
exports.TestConnectionFunctionResult$ = TestConnectionFunctionResult$;
exports.TestFunction$ = TestFunction$;
exports.TestFunctionCommand = TestFunctionCommand;
exports.TestFunctionFailed = TestFunctionFailed;
exports.TestFunctionFailed$ = TestFunctionFailed$;
exports.TestFunctionRequest$ = TestFunctionRequest$;
exports.TestFunctionResult$ = TestFunctionResult$;
exports.TestResult$ = TestResult$;
exports.TooLongCSPInResponseHeadersPolicy = TooLongCSPInResponseHeadersPolicy;
exports.TooLongCSPInResponseHeadersPolicy$ = TooLongCSPInResponseHeadersPolicy$;
exports.TooManyCacheBehaviors = TooManyCacheBehaviors;
exports.TooManyCacheBehaviors$ = TooManyCacheBehaviors$;
exports.TooManyCachePolicies = TooManyCachePolicies;
exports.TooManyCachePolicies$ = TooManyCachePolicies$;
exports.TooManyCertificates = TooManyCertificates;
exports.TooManyCertificates$ = TooManyCertificates$;
exports.TooManyCloudFrontOriginAccessIdentities = TooManyCloudFrontOriginAccessIdentities;
exports.TooManyCloudFrontOriginAccessIdentities$ = TooManyCloudFrontOriginAccessIdentities$;
exports.TooManyContinuousDeploymentPolicies = TooManyContinuousDeploymentPolicies;
exports.TooManyContinuousDeploymentPolicies$ = TooManyContinuousDeploymentPolicies$;
exports.TooManyCookieNamesInWhiteList = TooManyCookieNamesInWhiteList;
exports.TooManyCookieNamesInWhiteList$ = TooManyCookieNamesInWhiteList$;
exports.TooManyCookiesInCachePolicy = TooManyCookiesInCachePolicy;
exports.TooManyCookiesInCachePolicy$ = TooManyCookiesInCachePolicy$;
exports.TooManyCookiesInOriginRequestPolicy = TooManyCookiesInOriginRequestPolicy;
exports.TooManyCookiesInOriginRequestPolicy$ = TooManyCookiesInOriginRequestPolicy$;
exports.TooManyCustomHeadersInResponseHeadersPolicy = TooManyCustomHeadersInResponseHeadersPolicy;
exports.TooManyCustomHeadersInResponseHeadersPolicy$ = TooManyCustomHeadersInResponseHeadersPolicy$;
exports.TooManyDistributionCNAMEs = TooManyDistributionCNAMEs;
exports.TooManyDistributionCNAMEs$ = TooManyDistributionCNAMEs$;
exports.TooManyDistributions = TooManyDistributions;
exports.TooManyDistributions$ = TooManyDistributions$;
exports.TooManyDistributionsAssociatedToCachePolicy = TooManyDistributionsAssociatedToCachePolicy;
exports.TooManyDistributionsAssociatedToCachePolicy$ = TooManyDistributionsAssociatedToCachePolicy$;
exports.TooManyDistributionsAssociatedToFieldLevelEncryptionConfig = TooManyDistributionsAssociatedToFieldLevelEncryptionConfig;
exports.TooManyDistributionsAssociatedToFieldLevelEncryptionConfig$ = TooManyDistributionsAssociatedToFieldLevelEncryptionConfig$;
exports.TooManyDistributionsAssociatedToKeyGroup = TooManyDistributionsAssociatedToKeyGroup;
exports.TooManyDistributionsAssociatedToKeyGroup$ = TooManyDistributionsAssociatedToKeyGroup$;
exports.TooManyDistributionsAssociatedToOriginAccessControl = TooManyDistributionsAssociatedToOriginAccessControl;
exports.TooManyDistributionsAssociatedToOriginAccessControl$ = TooManyDistributionsAssociatedToOriginAccessControl$;
exports.TooManyDistributionsAssociatedToOriginRequestPolicy = TooManyDistributionsAssociatedToOriginRequestPolicy;
exports.TooManyDistributionsAssociatedToOriginRequestPolicy$ = TooManyDistributionsAssociatedToOriginRequestPolicy$;
exports.TooManyDistributionsAssociatedToResponseHeadersPolicy = TooManyDistributionsAssociatedToResponseHeadersPolicy;
exports.TooManyDistributionsAssociatedToResponseHeadersPolicy$ = TooManyDistributionsAssociatedToResponseHeadersPolicy$;
exports.TooManyDistributionsWithFunctionAssociations = TooManyDistributionsWithFunctionAssociations;
exports.TooManyDistributionsWithFunctionAssociations$ = TooManyDistributionsWithFunctionAssociations$;
exports.TooManyDistributionsWithLambdaAssociations = TooManyDistributionsWithLambdaAssociations;
exports.TooManyDistributionsWithLambdaAssociations$ = TooManyDistributionsWithLambdaAssociations$;
exports.TooManyDistributionsWithSingleFunctionARN = TooManyDistributionsWithSingleFunctionARN;
exports.TooManyDistributionsWithSingleFunctionARN$ = TooManyDistributionsWithSingleFunctionARN$;
exports.TooManyFieldLevelEncryptionConfigs = TooManyFieldLevelEncryptionConfigs;
exports.TooManyFieldLevelEncryptionConfigs$ = TooManyFieldLevelEncryptionConfigs$;
exports.TooManyFieldLevelEncryptionContentTypeProfiles = TooManyFieldLevelEncryptionContentTypeProfiles;
exports.TooManyFieldLevelEncryptionContentTypeProfiles$ = TooManyFieldLevelEncryptionContentTypeProfiles$;
exports.TooManyFieldLevelEncryptionEncryptionEntities = TooManyFieldLevelEncryptionEncryptionEntities;
exports.TooManyFieldLevelEncryptionEncryptionEntities$ = TooManyFieldLevelEncryptionEncryptionEntities$;
exports.TooManyFieldLevelEncryptionFieldPatterns = TooManyFieldLevelEncryptionFieldPatterns;
exports.TooManyFieldLevelEncryptionFieldPatterns$ = TooManyFieldLevelEncryptionFieldPatterns$;
exports.TooManyFieldLevelEncryptionProfiles = TooManyFieldLevelEncryptionProfiles;
exports.TooManyFieldLevelEncryptionProfiles$ = TooManyFieldLevelEncryptionProfiles$;
exports.TooManyFieldLevelEncryptionQueryArgProfiles = TooManyFieldLevelEncryptionQueryArgProfiles;
exports.TooManyFieldLevelEncryptionQueryArgProfiles$ = TooManyFieldLevelEncryptionQueryArgProfiles$;
exports.TooManyFunctionAssociations = TooManyFunctionAssociations;
exports.TooManyFunctionAssociations$ = TooManyFunctionAssociations$;
exports.TooManyFunctions = TooManyFunctions;
exports.TooManyFunctions$ = TooManyFunctions$;
exports.TooManyHeadersInCachePolicy = TooManyHeadersInCachePolicy;
exports.TooManyHeadersInCachePolicy$ = TooManyHeadersInCachePolicy$;
exports.TooManyHeadersInForwardedValues = TooManyHeadersInForwardedValues;
exports.TooManyHeadersInForwardedValues$ = TooManyHeadersInForwardedValues$;
exports.TooManyHeadersInOriginRequestPolicy = TooManyHeadersInOriginRequestPolicy;
exports.TooManyHeadersInOriginRequestPolicy$ = TooManyHeadersInOriginRequestPolicy$;
exports.TooManyInvalidationsInProgress = TooManyInvalidationsInProgress;
exports.TooManyInvalidationsInProgress$ = TooManyInvalidationsInProgress$;
exports.TooManyKeyGroups = TooManyKeyGroups;
exports.TooManyKeyGroups$ = TooManyKeyGroups$;
exports.TooManyKeyGroupsAssociatedToDistribution = TooManyKeyGroupsAssociatedToDistribution;
exports.TooManyKeyGroupsAssociatedToDistribution$ = TooManyKeyGroupsAssociatedToDistribution$;
exports.TooManyLambdaFunctionAssociations = TooManyLambdaFunctionAssociations;
exports.TooManyLambdaFunctionAssociations$ = TooManyLambdaFunctionAssociations$;
exports.TooManyOriginAccessControls = TooManyOriginAccessControls;
exports.TooManyOriginAccessControls$ = TooManyOriginAccessControls$;
exports.TooManyOriginCustomHeaders = TooManyOriginCustomHeaders;
exports.TooManyOriginCustomHeaders$ = TooManyOriginCustomHeaders$;
exports.TooManyOriginGroupsPerDistribution = TooManyOriginGroupsPerDistribution;
exports.TooManyOriginGroupsPerDistribution$ = TooManyOriginGroupsPerDistribution$;
exports.TooManyOriginRequestPolicies = TooManyOriginRequestPolicies;
exports.TooManyOriginRequestPolicies$ = TooManyOriginRequestPolicies$;
exports.TooManyOrigins = TooManyOrigins;
exports.TooManyOrigins$ = TooManyOrigins$;
exports.TooManyPublicKeys = TooManyPublicKeys;
exports.TooManyPublicKeys$ = TooManyPublicKeys$;
exports.TooManyPublicKeysInKeyGroup = TooManyPublicKeysInKeyGroup;
exports.TooManyPublicKeysInKeyGroup$ = TooManyPublicKeysInKeyGroup$;
exports.TooManyQueryStringParameters = TooManyQueryStringParameters;
exports.TooManyQueryStringParameters$ = TooManyQueryStringParameters$;
exports.TooManyQueryStringsInCachePolicy = TooManyQueryStringsInCachePolicy;
exports.TooManyQueryStringsInCachePolicy$ = TooManyQueryStringsInCachePolicy$;
exports.TooManyQueryStringsInOriginRequestPolicy = TooManyQueryStringsInOriginRequestPolicy;
exports.TooManyQueryStringsInOriginRequestPolicy$ = TooManyQueryStringsInOriginRequestPolicy$;
exports.TooManyRealtimeLogConfigs = TooManyRealtimeLogConfigs;
exports.TooManyRealtimeLogConfigs$ = TooManyRealtimeLogConfigs$;
exports.TooManyRemoveHeadersInResponseHeadersPolicy = TooManyRemoveHeadersInResponseHeadersPolicy;
exports.TooManyRemoveHeadersInResponseHeadersPolicy$ = TooManyRemoveHeadersInResponseHeadersPolicy$;
exports.TooManyResponseHeadersPolicies = TooManyResponseHeadersPolicies;
exports.TooManyResponseHeadersPolicies$ = TooManyResponseHeadersPolicies$;
exports.TooManyStreamingDistributionCNAMEs = TooManyStreamingDistributionCNAMEs;
exports.TooManyStreamingDistributionCNAMEs$ = TooManyStreamingDistributionCNAMEs$;
exports.TooManyStreamingDistributions = TooManyStreamingDistributions;
exports.TooManyStreamingDistributions$ = TooManyStreamingDistributions$;
exports.TooManyTrustedSigners = TooManyTrustedSigners;
exports.TooManyTrustedSigners$ = TooManyTrustedSigners$;
exports.TrafficConfig$ = TrafficConfig$;
exports.TrustStore$ = TrustStore$;
exports.TrustStoreConfig$ = TrustStoreConfig$;
exports.TrustStoreStatus = TrustStoreStatus;
exports.TrustStoreSummary$ = TrustStoreSummary$;
exports.TrustedKeyGroupDoesNotExist = TrustedKeyGroupDoesNotExist;
exports.TrustedKeyGroupDoesNotExist$ = TrustedKeyGroupDoesNotExist$;
exports.TrustedKeyGroups$ = TrustedKeyGroups$;
exports.TrustedSignerDoesNotExist = TrustedSignerDoesNotExist;
exports.TrustedSignerDoesNotExist$ = TrustedSignerDoesNotExist$;
exports.TrustedSigners$ = TrustedSigners$;
exports.UnsupportedOperation = UnsupportedOperation;
exports.UnsupportedOperation$ = UnsupportedOperation$;
exports.UntagResource$ = UntagResource$;
exports.UntagResourceCommand = UntagResourceCommand;
exports.UntagResourceRequest$ = UntagResourceRequest$;
exports.UpdateAnycastIpList$ = UpdateAnycastIpList$;
exports.UpdateAnycastIpListCommand = UpdateAnycastIpListCommand;
exports.UpdateAnycastIpListRequest$ = UpdateAnycastIpListRequest$;
exports.UpdateAnycastIpListResult$ = UpdateAnycastIpListResult$;
exports.UpdateCachePolicy$ = UpdateCachePolicy$;
exports.UpdateCachePolicyCommand = UpdateCachePolicyCommand;
exports.UpdateCachePolicyRequest$ = UpdateCachePolicyRequest$;
exports.UpdateCachePolicyResult$ = UpdateCachePolicyResult$;
exports.UpdateCloudFrontOriginAccessIdentity$ = UpdateCloudFrontOriginAccessIdentity$;
exports.UpdateCloudFrontOriginAccessIdentityCommand = UpdateCloudFrontOriginAccessIdentityCommand;
exports.UpdateCloudFrontOriginAccessIdentityRequest$ = UpdateCloudFrontOriginAccessIdentityRequest$;
exports.UpdateCloudFrontOriginAccessIdentityResult$ = UpdateCloudFrontOriginAccessIdentityResult$;
exports.UpdateConnectionFunction$ = UpdateConnectionFunction$;
exports.UpdateConnectionFunctionCommand = UpdateConnectionFunctionCommand;
exports.UpdateConnectionFunctionRequest$ = UpdateConnectionFunctionRequest$;
exports.UpdateConnectionFunctionResult$ = UpdateConnectionFunctionResult$;
exports.UpdateConnectionGroup$ = UpdateConnectionGroup$;
exports.UpdateConnectionGroupCommand = UpdateConnectionGroupCommand;
exports.UpdateConnectionGroupRequest$ = UpdateConnectionGroupRequest$;
exports.UpdateConnectionGroupResult$ = UpdateConnectionGroupResult$;
exports.UpdateContinuousDeploymentPolicy$ = UpdateContinuousDeploymentPolicy$;
exports.UpdateContinuousDeploymentPolicyCommand = UpdateContinuousDeploymentPolicyCommand;
exports.UpdateContinuousDeploymentPolicyRequest$ = UpdateContinuousDeploymentPolicyRequest$;
exports.UpdateContinuousDeploymentPolicyResult$ = UpdateContinuousDeploymentPolicyResult$;
exports.UpdateDistribution$ = UpdateDistribution$;
exports.UpdateDistributionCommand = UpdateDistributionCommand;
exports.UpdateDistributionRequest$ = UpdateDistributionRequest$;
exports.UpdateDistributionResult$ = UpdateDistributionResult$;
exports.UpdateDistributionTenant$ = UpdateDistributionTenant$;
exports.UpdateDistributionTenantCommand = UpdateDistributionTenantCommand;
exports.UpdateDistributionTenantRequest$ = UpdateDistributionTenantRequest$;
exports.UpdateDistributionTenantResult$ = UpdateDistributionTenantResult$;
exports.UpdateDistributionWithStagingConfig$ = UpdateDistributionWithStagingConfig$;
exports.UpdateDistributionWithStagingConfigCommand = UpdateDistributionWithStagingConfigCommand;
exports.UpdateDistributionWithStagingConfigRequest$ = UpdateDistributionWithStagingConfigRequest$;
exports.UpdateDistributionWithStagingConfigResult$ = UpdateDistributionWithStagingConfigResult$;
exports.UpdateDomainAssociation$ = UpdateDomainAssociation$;
exports.UpdateDomainAssociationCommand = UpdateDomainAssociationCommand;
exports.UpdateDomainAssociationRequest$ = UpdateDomainAssociationRequest$;
exports.UpdateDomainAssociationResult$ = UpdateDomainAssociationResult$;
exports.UpdateFieldLevelEncryptionConfig$ = UpdateFieldLevelEncryptionConfig$;
exports.UpdateFieldLevelEncryptionConfigCommand = UpdateFieldLevelEncryptionConfigCommand;
exports.UpdateFieldLevelEncryptionConfigRequest$ = UpdateFieldLevelEncryptionConfigRequest$;
exports.UpdateFieldLevelEncryptionConfigResult$ = UpdateFieldLevelEncryptionConfigResult$;
exports.UpdateFieldLevelEncryptionProfile$ = UpdateFieldLevelEncryptionProfile$;
exports.UpdateFieldLevelEncryptionProfileCommand = UpdateFieldLevelEncryptionProfileCommand;
exports.UpdateFieldLevelEncryptionProfileRequest$ = UpdateFieldLevelEncryptionProfileRequest$;
exports.UpdateFieldLevelEncryptionProfileResult$ = UpdateFieldLevelEncryptionProfileResult$;
exports.UpdateFunction$ = UpdateFunction$;
exports.UpdateFunctionCommand = UpdateFunctionCommand;
exports.UpdateFunctionRequest$ = UpdateFunctionRequest$;
exports.UpdateFunctionResult$ = UpdateFunctionResult$;
exports.UpdateKeyGroup$ = UpdateKeyGroup$;
exports.UpdateKeyGroupCommand = UpdateKeyGroupCommand;
exports.UpdateKeyGroupRequest$ = UpdateKeyGroupRequest$;
exports.UpdateKeyGroupResult$ = UpdateKeyGroupResult$;
exports.UpdateKeyValueStore$ = UpdateKeyValueStore$;
exports.UpdateKeyValueStoreCommand = UpdateKeyValueStoreCommand;
exports.UpdateKeyValueStoreRequest$ = UpdateKeyValueStoreRequest$;
exports.UpdateKeyValueStoreResult$ = UpdateKeyValueStoreResult$;
exports.UpdateOriginAccessControl$ = UpdateOriginAccessControl$;
exports.UpdateOriginAccessControlCommand = UpdateOriginAccessControlCommand;
exports.UpdateOriginAccessControlRequest$ = UpdateOriginAccessControlRequest$;
exports.UpdateOriginAccessControlResult$ = UpdateOriginAccessControlResult$;
exports.UpdateOriginRequestPolicy$ = UpdateOriginRequestPolicy$;
exports.UpdateOriginRequestPolicyCommand = UpdateOriginRequestPolicyCommand;
exports.UpdateOriginRequestPolicyRequest$ = UpdateOriginRequestPolicyRequest$;
exports.UpdateOriginRequestPolicyResult$ = UpdateOriginRequestPolicyResult$;
exports.UpdatePublicKey$ = UpdatePublicKey$;
exports.UpdatePublicKeyCommand = UpdatePublicKeyCommand;
exports.UpdatePublicKeyRequest$ = UpdatePublicKeyRequest$;
exports.UpdatePublicKeyResult$ = UpdatePublicKeyResult$;
exports.UpdateRealtimeLogConfig$ = UpdateRealtimeLogConfig$;
exports.UpdateRealtimeLogConfigCommand = UpdateRealtimeLogConfigCommand;
exports.UpdateRealtimeLogConfigRequest$ = UpdateRealtimeLogConfigRequest$;
exports.UpdateRealtimeLogConfigResult$ = UpdateRealtimeLogConfigResult$;
exports.UpdateResponseHeadersPolicy$ = UpdateResponseHeadersPolicy$;
exports.UpdateResponseHeadersPolicyCommand = UpdateResponseHeadersPolicyCommand;
exports.UpdateResponseHeadersPolicyRequest$ = UpdateResponseHeadersPolicyRequest$;
exports.UpdateResponseHeadersPolicyResult$ = UpdateResponseHeadersPolicyResult$;
exports.UpdateStreamingDistribution$ = UpdateStreamingDistribution$;
exports.UpdateStreamingDistributionCommand = UpdateStreamingDistributionCommand;
exports.UpdateStreamingDistributionRequest$ = UpdateStreamingDistributionRequest$;
exports.UpdateStreamingDistributionResult$ = UpdateStreamingDistributionResult$;
exports.UpdateTrustStore$ = UpdateTrustStore$;
exports.UpdateTrustStoreCommand = UpdateTrustStoreCommand;
exports.UpdateTrustStoreRequest$ = UpdateTrustStoreRequest$;
exports.UpdateTrustStoreResult$ = UpdateTrustStoreResult$;
exports.UpdateVpcOrigin$ = UpdateVpcOrigin$;
exports.UpdateVpcOriginCommand = UpdateVpcOriginCommand;
exports.UpdateVpcOriginRequest$ = UpdateVpcOriginRequest$;
exports.UpdateVpcOriginResult$ = UpdateVpcOriginResult$;
exports.ValidationTokenDetail$ = ValidationTokenDetail$;
exports.ValidationTokenHost = ValidationTokenHost;
exports.VerifyDnsConfiguration$ = VerifyDnsConfiguration$;
exports.VerifyDnsConfigurationCommand = VerifyDnsConfigurationCommand;
exports.VerifyDnsConfigurationRequest$ = VerifyDnsConfigurationRequest$;
exports.VerifyDnsConfigurationResult$ = VerifyDnsConfigurationResult$;
exports.ViewerCertificate$ = ViewerCertificate$;
exports.ViewerMtlsConfig$ = ViewerMtlsConfig$;
exports.ViewerMtlsMode = ViewerMtlsMode;
exports.ViewerProtocolPolicy = ViewerProtocolPolicy;
exports.VpcOrigin$ = VpcOrigin$;
exports.VpcOriginConfig$ = VpcOriginConfig$;
exports.VpcOriginEndpointConfig$ = VpcOriginEndpointConfig$;
exports.VpcOriginList$ = VpcOriginList$;
exports.VpcOriginSummary$ = VpcOriginSummary$;
exports.WebAclCustomization$ = WebAclCustomization$;
exports.paginateListCloudFrontOriginAccessIdentities = paginateListCloudFrontOriginAccessIdentities;
exports.paginateListConnectionFunctions = paginateListConnectionFunctions;
exports.paginateListConnectionGroups = paginateListConnectionGroups;
exports.paginateListDistributionTenants = paginateListDistributionTenants;
exports.paginateListDistributionTenantsByCustomization = paginateListDistributionTenantsByCustomization;
exports.paginateListDistributions = paginateListDistributions;
exports.paginateListDistributionsByConnectionFunction = paginateListDistributionsByConnectionFunction;
exports.paginateListDistributionsByConnectionMode = paginateListDistributionsByConnectionMode;
exports.paginateListDistributionsByTrustStore = paginateListDistributionsByTrustStore;
exports.paginateListDomainConflicts = paginateListDomainConflicts;
exports.paginateListInvalidations = paginateListInvalidations;
exports.paginateListInvalidationsForDistributionTenant = paginateListInvalidationsForDistributionTenant;
exports.paginateListKeyValueStores = paginateListKeyValueStores;
exports.paginateListOriginAccessControls = paginateListOriginAccessControls;
exports.paginateListPublicKeys = paginateListPublicKeys;
exports.paginateListStreamingDistributions = paginateListStreamingDistributions;
exports.paginateListTrustStores = paginateListTrustStores;
exports.waitForDistributionDeployed = waitForDistributionDeployed;
exports.waitForInvalidationCompleted = waitForInvalidationCompleted;
exports.waitForInvalidationForDistributionTenantCompleted = waitForInvalidationForDistributionTenantCompleted;
exports.waitForStreamingDistributionDeployed = waitForStreamingDistributionDeployed;
exports.waitUntilDistributionDeployed = waitUntilDistributionDeployed;
exports.waitUntilInvalidationCompleted = waitUntilInvalidationCompleted;
exports.waitUntilInvalidationForDistributionTenantCompleted = waitUntilInvalidationForDistributionTenantCompleted;
exports.waitUntilStreamingDistributionDeployed = waitUntilStreamingDistributionDeployed;
