﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/glacier/GlacierRequest.h>
#include <aws/glacier/Glacier_EXPORTS.h>
#include <aws/glacier/model/JobParameters.h>

#include <utility>

namespace Aws {
namespace Glacier {
namespace Model {

/**
 * <p>Provides options for initiating an Amazon S3 Glacier job.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/glacier-2012-06-01/InitiateJobInput">AWS
 * API Reference</a></p>
 */
class InitiateJobRequest : public GlacierRequest {
 public:
  AWS_GLACIER_API InitiateJobRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "InitiateJob"; }

  AWS_GLACIER_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The <code>AccountId</code> value is the AWS account ID of the account that
   * owns the vault. You can either specify an AWS account ID or optionally a single
   * '<code>-</code>' (hyphen), in which case Amazon S3 Glacier uses the AWS account
   * ID associated with the credentials used to sign the request. If you use an
   * account ID, do not include any hyphens ('-') in the ID.</p>
   */
  inline const Aws::String& GetAccountId() const { return m_accountId; }
  inline bool AccountIdHasBeenSet() const { return m_accountIdHasBeenSet; }
  template <typename AccountIdT = Aws::String>
  void SetAccountId(AccountIdT&& value) {
    m_accountIdHasBeenSet = true;
    m_accountId = std::forward<AccountIdT>(value);
  }
  template <typename AccountIdT = Aws::String>
  InitiateJobRequest& WithAccountId(AccountIdT&& value) {
    SetAccountId(std::forward<AccountIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the vault.</p>
   */
  inline const Aws::String& GetVaultName() const { return m_vaultName; }
  inline bool VaultNameHasBeenSet() const { return m_vaultNameHasBeenSet; }
  template <typename VaultNameT = Aws::String>
  void SetVaultName(VaultNameT&& value) {
    m_vaultNameHasBeenSet = true;
    m_vaultName = std::forward<VaultNameT>(value);
  }
  template <typename VaultNameT = Aws::String>
  InitiateJobRequest& WithVaultName(VaultNameT&& value) {
    SetVaultName(std::forward<VaultNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Provides options for specifying job information.</p>
   */
  inline const JobParameters& GetJobParameters() const { return m_jobParameters; }
  inline bool JobParametersHasBeenSet() const { return m_jobParametersHasBeenSet; }
  template <typename JobParametersT = JobParameters>
  void SetJobParameters(JobParametersT&& value) {
    m_jobParametersHasBeenSet = true;
    m_jobParameters = std::forward<JobParametersT>(value);
  }
  template <typename JobParametersT = JobParameters>
  InitiateJobRequest& WithJobParameters(JobParametersT&& value) {
    SetJobParameters(std::forward<JobParametersT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_accountId;

  Aws::String m_vaultName;

  JobParameters m_jobParameters;
  bool m_accountIdHasBeenSet = false;
  bool m_vaultNameHasBeenSet = false;
  bool m_jobParametersHasBeenSet = false;
};

}  // namespace Model
}  // namespace Glacier
}  // namespace Aws
