"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.RawRsaKeyringNode = void 0;
const material_management_node_1 = require("@aws-crypto/material-management-node");
const crypto_1 = require("crypto");
const raw_keyring_1 = require("@aws-crypto/raw-keyring");
const oaep_hash_supported_1 = require("./oaep_hash_supported");
const supportedOaepHash = [
    'sha1',
    'sha256',
    'sha384',
    'sha512',
    undefined,
];
class RawRsaKeyringNode extends material_management_node_1.KeyringNode {
    constructor(input) {
        super();
        const { rsaKey, keyName, keyNamespace, padding = crypto_1.constants.RSA_PKCS1_OAEP_PADDING, oaepHash, } = input;
        const { publicKey, privateKey } = rsaKey;
        /* Precondition: RsaKeyringNode needs either a public or a private key to operate. */
        (0, material_management_node_1.needs)(publicKey || privateKey, 'No Key provided.');
        /* Precondition: RsaKeyringNode needs identifying information for encrypt and decrypt. */
        (0, material_management_node_1.needs)(keyName && keyNamespace, 'Identifying information must be defined.');
        /* Precondition: The AWS ESDK only supports specific hash values for OAEP padding. */
        (0, material_management_node_1.needs)(padding === crypto_1.constants.RSA_PKCS1_OAEP_PADDING
            ? oaep_hash_supported_1.oaepHashSupported
                ? supportedOaepHash.includes(oaepHash)
                : !oaepHash || oaepHash === 'sha1'
            : !oaepHash, 'Unsupported oaepHash');
        const _wrapKey = async (material) => {
            /* Precondition: Public key must be defined to support encrypt. */
            if (!publicKey)
                throw new Error('No public key defined in constructor.  Encrypt disabled.');
            const { buffer, byteOffset, byteLength } = (0, material_management_node_1.unwrapDataKey)(material.getUnencryptedDataKey());
            const encryptedDataKey = (0, crypto_1.publicEncrypt)({ key: publicKey, padding, oaepHash }, Buffer.from(buffer, byteOffset, byteLength));
            const providerInfo = this.keyName;
            const providerId = this.keyNamespace;
            const flag = material_management_node_1.KeyringTraceFlag.WRAPPING_KEY_ENCRYPTED_DATA_KEY;
            const edk = new material_management_node_1.EncryptedDataKey({
                encryptedDataKey,
                providerInfo,
                providerId,
            });
            return material.addEncryptedDataKey(edk, flag);
        };
        const _unwrapKey = async (material, edk) => {
            /* Precondition: Private key must be defined to support decrypt. */
            if (!privateKey)
                throw new Error('No private key defined in constructor.  Decrypt disabled.');
            const trace = {
                keyName: this.keyName,
                keyNamespace: this.keyNamespace,
                flags: material_management_node_1.KeyringTraceFlag.WRAPPING_KEY_DECRYPTED_DATA_KEY,
            };
            const { buffer, byteOffset, byteLength } = edk.encryptedDataKey;
            const encryptedDataKey = Buffer.from(buffer, byteOffset, byteLength);
            const unencryptedDataKey = (0, crypto_1.privateDecrypt)({ key: privateKey, padding, oaepHash }, encryptedDataKey);
            return material.setUnencryptedDataKey(unencryptedDataKey, trace);
        };
        (0, material_management_node_1.readOnlyProperty)(this, 'keyName', keyName);
        (0, material_management_node_1.readOnlyProperty)(this, 'keyNamespace', keyNamespace);
        (0, material_management_node_1.readOnlyProperty)(this, '_wrapKey', _wrapKey);
        (0, material_management_node_1.readOnlyProperty)(this, '_unwrapKey', _unwrapKey);
    }
    _filter({ providerId, providerInfo }) {
        const { keyNamespace, keyName } = this;
        return providerId === keyNamespace && providerInfo === keyName;
    }
    _onEncrypt = (0, raw_keyring_1._onEncrypt)(randomBytesAsync);
    _onDecrypt = (0, raw_keyring_1._onDecrypt)();
}
exports.RawRsaKeyringNode = RawRsaKeyringNode;
(0, material_management_node_1.immutableClass)(RawRsaKeyringNode);
async function randomBytesAsync(size) {
    return new Promise((resolve, reject) => {
        (0, crypto_1.randomBytes)(size, (err, buffer) => {
            if (err)
                return reject(err);
            resolve(buffer);
        });
    });
}
//# sourceMappingURL=data:application/json;base64,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