"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DataViewsService = void 0;
var _i18n = require("@osd/i18n");
var _ = require("../..");
var _data_view = require("./data_view");
var _ensure_default_data_view = require("./ensure_default_data_view");
var _common = require("../../../../opensearch_dashboards_utils/common");
var _lib = require("../lib");
var _utils = require("../utils");
var _errors = require("../errors");
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          * Copyright OpenSearch Contributors
                                                                                                                                                                                                                                                                                                                          * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                                                                                                                          */
const MAX_ATTEMPTS_TO_RESOLVE_CONFLICTS = 3;
const savedObjectType = 'index-pattern';
/**
 * @experimental This class is experimental and may change in future versions
 */
class DataViewsService {
  constructor({
    patterns,
    uiSettings,
    savedObjectsClient,
    apiClient,
    fieldFormats,
    onNotification,
    onError,
    onUnsupportedTimePattern,
    onRedirectNoDataView = () => {},
    canUpdateUiSetting
  }) {
    _defineProperty(this, "config", void 0);
    _defineProperty(this, "savedObjectsClient", void 0);
    _defineProperty(this, "savedObjectsCache", void 0);
    _defineProperty(this, "apiClient", void 0);
    _defineProperty(this, "fieldFormats", void 0);
    _defineProperty(this, "onNotification", void 0);
    _defineProperty(this, "onError", void 0);
    _defineProperty(this, "onUnsupportedTimePattern", void 0);
    _defineProperty(this, "patterns", void 0);
    _defineProperty(this, "ensureDefaultDataView", void 0);
    _defineProperty(this, "getDataSource", async id => {
      return await this.savedObjectsClient.get('data-source', id);
    });
    /**
     * Finds a data source by its title.
     *
     * @param title - The title of the data source to find.
     * @param size - The number of results to return. Defaults to 10.
     * @returns The first matching data source or undefined if not found.
     */
    _defineProperty(this, "findDataSourceByTitle", async (title, size = 10) => {
      const savedObjectsResponse = await this.savedObjectsClient.find({
        type: 'data-source',
        fields: ['title', 'dataSourceEngineType'],
        search: title,
        searchFields: ['title'],
        perPage: size
      });
      return savedObjectsResponse[0] || undefined;
    });
    /**
     * Get list of data view ids
     * @param refresh Force refresh of data view list
     */
    _defineProperty(this, "getIds", async (refresh = false) => {
      if (!this.savedObjectsCache || refresh) {
        await this.refreshSavedObjectsCache();
      }
      if (!this.savedObjectsCache) {
        return [];
      }
      return this.savedObjectsCache.map(obj => obj === null || obj === void 0 ? void 0 : obj.id);
    });
    /**
     * Get list of data view titles
     * @param refresh Force refresh of data view list
     */
    _defineProperty(this, "getTitles", async (refresh = false) => {
      if (!this.savedObjectsCache || refresh) {
        await this.refreshSavedObjectsCache();
      }
      if (!this.savedObjectsCache) {
        return [];
      }
      return this.savedObjectsCache.map(obj => {
        var _obj$attributes;
        return obj === null || obj === void 0 || (_obj$attributes = obj.attributes) === null || _obj$attributes === void 0 ? void 0 : _obj$attributes.title;
      });
    });
    /**
     * Get list of data view ids with titles
     * @param refresh Force refresh of data view list
     */
    _defineProperty(this, "getIdsWithTitle", async (refresh = false) => {
      if (!this.savedObjectsCache || refresh) {
        await this.refreshSavedObjectsCache();
      }
      if (!this.savedObjectsCache) {
        return [];
      }
      return this.savedObjectsCache.map(obj => {
        var _obj$attributes2;
        return {
          id: obj === null || obj === void 0 ? void 0 : obj.id,
          title: obj === null || obj === void 0 || (_obj$attributes2 = obj.attributes) === null || _obj$attributes2 === void 0 ? void 0 : _obj$attributes2.title
        };
      });
    });
    /**
     * Clear data view list cache
     * @param id optionally clear a single id
     */
    _defineProperty(this, "clearCache", (id, clearSavedObjectsCache = true) => {
      if (clearSavedObjectsCache) {
        this.savedObjectsCache = null;
      }
      this.patterns.clearCache(id, clearSavedObjectsCache);
    });
    _defineProperty(this, "getCache", async () => {
      if (!this.savedObjectsCache) {
        await this.refreshSavedObjectsCache();
      }
      return this.savedObjectsCache;
    });
    _defineProperty(this, "saveToCache", (id, dataView) => {
      this.patterns.saveToCache(id, dataView);
    });
    /**
     * Get default data view
     */
    _defineProperty(this, "getDefault", async () => {
      const defaultDataViewId = await this.config.get('defaultIndex');
      if (defaultDataViewId) {
        return await this.get(defaultDataViewId);
      }
      return null;
    });
    /**
     * Optionally set default data view, unless force = true
     * @param id
     * @param force
     */
    _defineProperty(this, "setDefault", async (id, force = false) => {
      if (force || !this.config.get('defaultIndex')) {
        await this.config.set('defaultIndex', id);
      }
    });
    /**
     * Get field list by providing { pattern }
     * @param options
     */
    _defineProperty(this, "getFieldsForWildcard", async (options = {}) => {
      const metaFields = await this.config.get(_.UI_SETTINGS.META_FIELDS);
      return this.apiClient.getFieldsForWildcard({
        pattern: options.pattern,
        metaFields,
        type: options.type,
        params: options.params || {},
        dataSourceId: options.dataSourceId
      });
    });
    /**
     * Get field list by providing an index patttern (or spec)
     * @param options
     */
    _defineProperty(this, "getFieldsForDataView", async (dataView, options = {}) => {
      var _dataView$dataSourceR;
      return this.getFieldsForWildcard({
        pattern: dataView.title,
        ...options,
        type: dataView.type,
        params: dataView.typeMeta && dataView.typeMeta.params,
        dataSourceId: (_dataView$dataSourceR = dataView.dataSourceRef) === null || _dataView$dataSourceR === void 0 ? void 0 : _dataView$dataSourceR.id
      });
    });
    /**
     * Refresh field list for a given data view
     * @param dataView
     */
    _defineProperty(this, "refreshFields", async (dataView, skipType = false) => {
      try {
        const dataViewCopy = skipType ? {
          ...dataView,
          type: undefined
        } : dataView;
        const fields = await this.getFieldsForDataView(dataViewCopy);
        const scripted = dataView.getScriptedFields().map(field => field.spec);
        dataView.fields.replaceAll([...fields, ...scripted]);
      } catch (err) {
        if (err instanceof _lib.DataViewMissingIndices) {
          this.onNotification({
            title: err.message,
            color: 'danger',
            iconType: 'alert'
          });
        }
        this.onError(err, {
          title: _i18n.i18n.translate('data.dataViews.fetchFieldErrorTitle', {
            defaultMessage: 'Error fetching fields for data view {title} (ID: {id})',
            values: {
              id: dataView.id,
              title: dataView.title
            }
          })
        });
      }
    });
    /**
     * Refreshes a field list from a spec before an data view instance is created
     * @param fields
     * @param id
     * @param title
     * @param options
     */
    _defineProperty(this, "refreshFieldSpecMap", async (fields, id, title, options) => {
      const scriptdFields = Object.values(fields).filter(field => field.scripted);
      try {
        const newFields = await this.getFieldsForWildcard(options);
        return this.fieldArrayToMap([...newFields, ...scriptdFields]);
      } catch (err) {
        if (err instanceof _lib.DataViewMissingIndices) {
          this.onNotification({
            title: err.message,
            color: 'danger',
            iconType: 'alert'
          });
          return {};
        }
        this.onError(err, {
          title: _i18n.i18n.translate('data.dataViews.fetchFieldErrorTitle', {
            defaultMessage: 'Error fetching fields for data view {title} (ID: {id})',
            values: {
              id,
              title
            }
          })
        });
      }
      return fields;
    });
    /**
     * Applies a set of formats to a set of fields
     * @param fieldSpecs
     * @param fieldFormatMap
     */
    _defineProperty(this, "addFormatsToFields", (fieldSpecs, fieldFormatMap) => {
      Object.entries(fieldFormatMap).forEach(([fieldName, value]) => {
        const field = fieldSpecs.find(fld => fld.name === fieldName);
        if (field) {
          field.format = value;
        }
      });
    });
    /**
     * Converts field array to map
     * @param fields
     */
    _defineProperty(this, "fieldArrayToMap", fields => fields.reduce((collector, field) => {
      collector[field.name] = field;
      return collector;
    }, {}));
    /**
     * Converts data view saved object to data view spec
     * @param savedObject
     */
    _defineProperty(this, "savedObjectToSpec", savedObject => {
      const {
        id,
        version,
        attributes: {
          title,
          displayName,
          description,
          signalType,
          timeFieldName,
          intervalName,
          fields,
          sourceFilters,
          fieldFormatMap,
          typeMeta,
          type
        },
        references
      } = savedObject;
      const parsedSourceFilters = sourceFilters ? JSON.parse(sourceFilters) : undefined;
      const parsedTypeMeta = typeMeta ? JSON.parse(typeMeta) : undefined;
      const parsedFieldFormatMap = fieldFormatMap ? JSON.parse(fieldFormatMap) : {};
      const parsedFields = fields ? JSON.parse(fields) : [];
      const dataSourceRef = Array.isArray(references) ? references[0] : undefined;
      this.addFormatsToFields(parsedFields, parsedFieldFormatMap);
      return {
        id,
        version,
        title,
        displayName,
        description,
        signalType,
        intervalName,
        timeFieldName,
        sourceFilters: parsedSourceFilters,
        fields: this.fieldArrayToMap(parsedFields),
        typeMeta: parsedTypeMeta,
        type,
        dataSourceRef
      };
    });
    /**
     * Get an data view by id. Cache optimized
     * @param id
     * @param onlyCheckCache - Only check cache for data view if it doesn't exist it will not error out
     */
    _defineProperty(this, "get", async (id, onlyCheckCache = false) => {
      const cache = await this.patterns.get(id, true);
      if (cache || onlyCheckCache) {
        return cache;
      }
      const savedObject = await this.savedObjectsClient.get(savedObjectType, id);
      if (!savedObject.version) {
        throw new _common.SavedObjectNotFound(savedObjectType, id, 'management/opensearch-dashboards/indexPatterns');
      }
      const spec = this.savedObjectToSpec(savedObject);
      const {
        title,
        type,
        typeMeta,
        dataSourceRef
      } = spec;
      const parsedFieldFormats = savedObject.attributes.fieldFormatMap ? JSON.parse(savedObject.attributes.fieldFormatMap) : {};
      const isFieldRefreshRequired = this.isFieldRefreshRequired(spec.fields);
      let isSaveRequired = isFieldRefreshRequired;
      try {
        spec.fields = isFieldRefreshRequired ? await this.refreshFieldSpecMap(spec.fields || {}, id, spec.title, {
          pattern: title,
          metaFields: await this.config.get(_.UI_SETTINGS.META_FIELDS),
          type,
          params: typeMeta && typeMeta.params,
          dataSourceId: dataSourceRef === null || dataSourceRef === void 0 ? void 0 : dataSourceRef.id
        }) : spec.fields;
      } catch (err) {
        isSaveRequired = false;
        if (err instanceof _lib.DataViewMissingIndices) {
          this.onNotification({
            title: err.message,
            color: 'danger',
            iconType: 'alert'
          });
        } else {
          this.onError(err, {
            title: _i18n.i18n.translate('data.dataViews.fetchFieldErrorTitle', {
              defaultMessage: 'Error fetching fields for data view {title} (ID: {id})',
              values: {
                id,
                title
              }
            })
          });
        }
      }
      Object.entries(parsedFieldFormats).forEach(([fieldName, value]) => {
        var _spec$fields;
        const field = (_spec$fields = spec.fields) === null || _spec$fields === void 0 ? void 0 : _spec$fields[fieldName];
        if (field) {
          field.format = value;
        }
      });
      const dataView = await this.create(spec, true);
      this.patterns.saveToCache(id, dataView);
      if (isSaveRequired) {
        try {
          this.updateSavedObject(dataView);
        } catch (err) {
          this.onError(err, {
            title: _i18n.i18n.translate('data.dataViews.fetchFieldSaveErrorTitle', {
              defaultMessage: 'Error saving after fetching fields for data view {title} (ID: {id})',
              values: {
                id: dataView.id,
                title: dataView.title
              }
            })
          });
        }
      }
      if (dataView.isUnsupportedTimePattern()) {
        this.onUnsupportedTimePattern({
          id: dataView.id,
          title: dataView.title,
          index: dataView.getIndex()
        });
      }
      dataView.resetOriginalSavedObjectBody();
      return dataView;
    });
    /**
     * Get an data view by title if cached
     * @param id
     */
    _defineProperty(this, "getByTitle", (title, ignoreErrors = false) => {
      const dataView = this.patterns.getByTitle(title);
      if (!dataView && !ignoreErrors) {
        throw new _errors.MissingDataViewError(`Missing data view: ${title}`);
      }
      return dataView;
    });
    _defineProperty(this, "find", async (search, size = 10) => {
      const savedObjects = await this.savedObjectsClient.find({
        type: 'index-pattern',
        fields: ['title'],
        search,
        searchFields: ['title'],
        perPage: size
      });
      const getDataViewPromises = savedObjects.map(async savedObject => {
        return await this.get(savedObject.id);
      });
      return await Promise.all(getDataViewPromises);
    });
    this.apiClient = apiClient;
    this.config = uiSettings;
    this.savedObjectsClient = savedObjectsClient;
    this.fieldFormats = fieldFormats;
    this.onNotification = onNotification;
    this.onError = onError;
    this.onUnsupportedTimePattern = onUnsupportedTimePattern;
    this.patterns = patterns;
    this.ensureDefaultDataView = (0, _ensure_default_data_view.createEnsureDefaultDataView)(uiSettings, onRedirectNoDataView, canUpdateUiSetting, savedObjectsClient);
  }

  /**
   * Refresh cache of data view ids and titles
   */
  async refreshSavedObjectsCache() {
    this.savedObjectsCache = await this.savedObjectsClient.find({
      type: 'index-pattern',
      fields: ['title'],
      perPage: 10000
    });
    this.savedObjectsCache = await Promise.all(this.savedObjectsCache.map(async obj => {
      // TODO: This behaviour will cause the data view title to be resolved differently depending on how its fetched since the get method in this service will not append the datasource title
      if (obj.type === 'index-pattern') {
        const result = {
          ...obj
        };
        result.attributes.title = await (0, _utils.getDataViewTitle)(obj.attributes.title, obj.references, this.getDataSource);
        return result;
      } else {
        return obj;
      }
    }));
  }
  isFieldRefreshRequired(specs) {
    if (!specs) {
      return true;
    }
    return Object.values(specs).every(spec => {
      // See https://github.com/elastic/kibana/pull/8421
      const hasFieldCaps = 'aggregatable' in spec && 'searchable' in spec;

      // See https://github.com/elastic/kibana/pull/11969
      const hasDocValuesFlag = ('readFromDocValues' in spec);
      return !hasFieldCaps || !hasDocValuesFlag;
    });
  }
  migrate(dataView, newTitle) {
    return this.savedObjectsClient.update(savedObjectType, dataView.id, {
      title: newTitle,
      intervalName: null
    }, {
      version: dataView.version
    }).then(({
      attributes: {
        title,
        intervalName
      }
    }) => {
      dataView.title = title;
      dataView.intervalName = intervalName;
    }).then(() => this);
  }

  /**
   * Create a new data view instance
   * @param spec
   * @param skipFetchFields
   */
  async create(spec, skipFetchFields = false) {
    const shortDotsEnable = await this.config.get(_.UI_SETTINGS.SHORT_DOTS_ENABLE);
    const metaFields = await this.config.get(_.UI_SETTINGS.META_FIELDS);
    const dataView = new _data_view.DataView({
      spec,
      savedObjectsClient: this.savedObjectsClient,
      fieldFormats: this.fieldFormats,
      shortDotsEnable,
      metaFields
    });
    if (!skipFetchFields) {
      await this.refreshFields(dataView);
    }
    return dataView;
  }
  /**
   * Create a new data view and save it right away
   * @param spec
   * @param override Overwrite if existing data view exists
   * @param skipFetchFields
   */
  async createAndSave(spec, override = false, skipFetchFields = false) {
    const dataView = await this.create(spec, skipFetchFields);
    await this.createSavedObject(dataView, override);
    await this.setDefault(dataView.id);
    return dataView;
  }

  /**
   * Save a new data view
   * @param dataView
   * @param override Overwrite if existing data view exists
   */

  async createSavedObject(dataView, override = false) {
    var _dataView$dataSourceR2;
    const dupe = await (0, _utils.findByTitle)(this.savedObjectsClient, dataView.title, (_dataView$dataSourceR2 = dataView.dataSourceRef) === null || _dataView$dataSourceR2 === void 0 ? void 0 : _dataView$dataSourceR2.id);
    if (dupe) {
      if (override) {
        await this.delete(dupe.id);
      } else {
        throw new _errors.DuplicateDataViewError(`Duplicate data view: ${dataView.title}`);
      }
    }
    const body = dataView.getAsSavedObjectBody();
    const references = dataView.getSaveObjectReference();
    const response = await this.savedObjectsClient.create(savedObjectType, body, {
      id: dataView.id,
      references
    });
    dataView.id = response.id;
    this.patterns.saveToCache(dataView.id, dataView);
    return dataView;
  }

  /**
   * Save existing data view. Will attempt to merge differences if there are conflicts
   * @param dataView
   * @param saveAttempts
   */
  async updateSavedObject(dataView, saveAttempts = 0, ignoreErrors = false) {
    if (!dataView.id) return;
    const body = dataView.getAsSavedObjectBody();
    const originalBody = dataView.getOriginalSavedObjectBody();
    const originalChangedKeys = [];
    Object.entries(body).forEach(([key, value]) => {
      if (value !== originalBody[key]) {
        originalChangedKeys.push(key);
      }
    });
    return this.savedObjectsClient.update(savedObjectType, dataView.id, body, {
      version: dataView.version
    }).then(resp => {
      dataView.id = resp.id;
      dataView.version = resp.version;
    }).catch(async err => {
      var _err$res;
      if ((err === null || err === void 0 || (_err$res = err.res) === null || _err$res === void 0 ? void 0 : _err$res.status) === 409 && saveAttempts++ < MAX_ATTEMPTS_TO_RESOLVE_CONFLICTS) {
        const samePattern = await this.get(dataView.id);
        // What keys changed from now and what the server returned
        const updatedBody = samePattern.getAsSavedObjectBody();

        // Build a list of changed keys from the server response
        // and ensure we ignore the key if the server response
        // is the same as the original response (since that is expected
        // if we made a change in that key)

        const serverChangedKeys = [];
        Object.entries(updatedBody).forEach(([key, value]) => {
          if (value !== body[key] && value !== originalBody[key]) {
            serverChangedKeys.push(key);
          }
        });
        let unresolvedCollision = false;
        for (const originalKey of originalChangedKeys) {
          for (const serverKey of serverChangedKeys) {
            if (originalKey === serverKey) {
              unresolvedCollision = true;
              break;
            }
          }
        }
        if (unresolvedCollision) {
          if (ignoreErrors) {
            return;
          }
          const title = _i18n.i18n.translate('data.dataViews.unableWriteLabel', {
            defaultMessage: 'Unable to write data view! Refresh the page to get the most up to date changes for this data view.'
          });
          this.onNotification({
            title,
            color: 'danger'
          });
          throw err;
        }
        serverChangedKeys.forEach(key => {
          dataView[key] = samePattern[key];
        });
        dataView.version = samePattern.version;
        this.patterns.clearCache(dataView.id);
        return this.updateSavedObject(dataView, saveAttempts, ignoreErrors);
      }
      throw err;
    });
  }

  /**
   * Deletes an data view from .kibana index
   * @param dataViewId: Id of OpenSearch Dashboards Index Pattern to delete
   */
  async delete(dataViewId) {
    this.patterns.clearCache(dataViewId);
    return this.savedObjectsClient.delete('index-pattern', dataViewId);
  }
  isLongNumeralsSupported() {
    return this.config.get(_.UI_SETTINGS.DATA_WITH_LONG_NUMERALS);
  }

  /**
   * Convert a DataView to a Dataset object
   * @experimental This method is experimental and may change in future versions
   * @param dataView DataView object to convert to Dataset
   */
  async convertToDataset(dataView) {
    var _dataView$dataSourceR3;
    if (dataView.toDataset) {
      return await dataView.toDataset();
    }
    return {
      id: dataView.id || '',
      title: dataView.title,
      type: dataView.type || _.DEFAULT_DATA.SET_TYPES.INDEX_PATTERN,
      timeFieldName: dataView.timeFieldName,
      ...(((_dataView$dataSourceR3 = dataView.dataSourceRef) === null || _dataView$dataSourceR3 === void 0 ? void 0 : _dataView$dataSourceR3.id) && {
        dataSource: {
          id: dataView.dataSourceRef.id,
          title: dataView.dataSourceRef.name || dataView.dataSourceRef.id,
          type: dataView.dataSourceRef.type || _.DEFAULT_DATA.SOURCE_TYPES.OPENSEARCH
        }
      })
    };
  }
}
exports.DataViewsService = DataViewsService;