"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defineResourceAccessManagementRoutes = defineResourceAccessManagementRoutes;
var _configSchema = require("@osd/config-schema");
/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */

/* ******************* Schemas ******************* */

const stringArray = _configSchema.schema.arrayOf(_configSchema.schema.string());

// Inner object allowed on each access level
const recipientsSchema = _configSchema.schema.object({
  users: _configSchema.schema.maybe(stringArray),
  roles: _configSchema.schema.maybe(stringArray),
  backend_roles: _configSchema.schema.maybe(stringArray)
}, {
  unknowns: 'forbid',
  validate: v => {
    if (!v.users && !v.roles && !v.backend_roles) {
      return 'Each share_with entry must include at least one of "users", "roles", or "backend_roles".';
    }
  }
});

// share_with shape used by both PUT and PATCH (add/revoke)
const shareWithSchema = _configSchema.schema.recordOf(_configSchema.schema.string(), recipientsSchema);

// PUT body must include non-empty share_with
const putBodySchema = _configSchema.schema.object({
  resource_id: _configSchema.schema.string({
    minLength: 1
  }),
  resource_type: _configSchema.schema.string({
    minLength: 1
  }),
  share_with: shareWithSchema
}, {
  unknowns: 'allow',
  validate: val => {
    if (Object.keys(val.share_with).length === 0) {
      return '"share_with" must not be empty.';
    }
  }
});

// PATCH schema: add/revoke must be shareWithSchema
const postBodySchema = _configSchema.schema.object({
  resource_id: _configSchema.schema.string({
    minLength: 1
  }),
  resource_type: _configSchema.schema.string({
    minLength: 1
  }),
  add: _configSchema.schema.maybe(shareWithSchema),
  revoke: _configSchema.schema.maybe(shareWithSchema)
}, {
  validate: value => {
    if (!value.add && !value.revoke) {
      return 'Request body must include at least one of "add" or "revoke".';
    }
  }
});

/* *******************Route definitions******************* */

const SECURITY_RESOURCE_API_PREFIX = '/_plugins/_security/api/resource';
const LIST_TYPES_API = `${SECURITY_RESOURCE_API_PREFIX}/types`;
const LIST_SHARING_INFO_API = `${SECURITY_RESOURCE_API_PREFIX}/list`;
const SHARE_API = `${SECURITY_RESOURCE_API_PREFIX}/share`;
function defineResourceAccessManagementRoutes(router) {
  // GET registered resource types
  router.get({
    path: '/api/resource/types',
    validate: false
  }, async (context, request, response) => {
    try {
      const client = context.core.opensearch.client.asCurrentUser;
      const result = await client.transport.request({
        method: 'GET',
        path: LIST_TYPES_API
      });
      return response.ok({
        body: result.body
      });
    } catch (error) {
      return response.customError({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // GET accessible shared resources filtered by resourceType
  router.get({
    path: '/api/resource/list',
    validate: {
      query: _configSchema.schema.object({
        resourceType: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const client = context.core.opensearch.client.asCurrentUser;
      const {
        resourceType
      } = request.query;
      const result = await client.transport.request({
        method: 'GET',
        path: LIST_SHARING_INFO_API,
        querystring: {
          resource_type: resourceType
        } // backend expects snake_case
      });

      return response.ok({
        body: result.body
      });
    } catch (error) {
      return response.customError({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // GET sharing info for a specific resource
  router.get({
    path: '/api/resource/view',
    validate: {
      query: _configSchema.schema.object({
        resourceId: _configSchema.schema.string(),
        resourceType: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const client = context.core.opensearch.client.asCurrentUser;
      const {
        resourceId,
        resourceType
      } = request.query;
      const result = await client.transport.request({
        method: 'GET',
        path: SHARE_API,
        querystring: {
          resource_id: resourceId,
          resource_type: resourceType
        } // backend expects snake_case
      });

      return response.ok({
        body: result.body
      });
    } catch (error) {
      return response.customError({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // PUT share a resource — requires `share_with`
  router.put({
    path: '/api/resource/share',
    validate: {
      body: putBodySchema
    }
  }, async (context, request, response) => {
    try {
      const client = context.core.opensearch.client.asCurrentUser;
      const result = await client.transport.request({
        method: 'PUT',
        path: SHARE_API,
        body: request.body
      });
      return response.ok({
        body: result.body
      });
    } catch (error) {
      return response.customError({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // POST update sharing — `add`/`revoke` adhere to share_with schema
  router.post({
    path: '/api/resource/update_sharing',
    validate: {
      body: postBodySchema
    }
  }, async (context, request, response) => {
    try {
      const client = context.core.opensearch.client.asCurrentUser;
      const result = await client.transport.request({
        method: 'POST',
        path: SHARE_API,
        body: request.body
      });
      return response.ok({
        body: result.body
      });
    } catch (error) {
      return response.customError({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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