# (c) Copyright 2009-2015. CodeWeavers, Inc.

import os
import sys
import traceback

from gi.repository import GLib
from gi.repository import GdkPixbuf
from gi.repository import Gdk
from gi.repository import Gtk

import cxlog
import cxutils
import distversion

# for localization
from cxutils import cxgettext as _
cxutils.setup_textdomain()


def toplevel_quit():
    """If there are no visible toplevel windows left, call Gtk.main_quit().

    This should be called when a toplevel is closed on a process that might have
    multiple toplevels and needs to live as long as at least one is still open."""
    for toplevel in Gtk.Window.list_toplevels():
        if toplevel.props.visible and toplevel.props.window and \
            toplevel.props.window.get_window_type() != Gdk.WindowType.TEMP:
            break
    else:
        Gtk.main_quit()

def get_ui_path(basename):
    try:
        ui_dir = os.environ["CX_GLADEPATH"]
    except KeyError:
        ui_dir = os.path.join(cxutils.CX_ROOT, "lib", "python", "glade")
    return os.path.join(ui_dir, basename + ".ui")


ICONS = {}
def get_std_icon(basename, sizes=cxutils.S_MEDIUM):
    # Assume scanning the cache is faster than disk accesses
    for size in sizes:
        key = basename + "." + size
        if key in ICONS:
            return ICONS[key]

    root = os.path.join(cxutils.CX_ROOT, 'share', 'icons')
    filename = cxutils.get_icon_path(root, '', basename, sizes)
    if filename:
        try:
            ICONS[key] = GdkPixbuf.Pixbuf.new_from_file(filename)
            return ICONS[key]
        except GLib.Error: # pylint: disable=E0712
            cxlog.warn("could not load icon file %s:\n%s" % (cxlog.debug_str(filename), traceback.format_exc()))
    return None


def get_std_icon_list(basename='crossover'):
    icons = []
    root = os.path.join(cxutils.CX_ROOT, 'share', 'icons')
    for filename in cxutils.get_icon_paths(root, '', basename):
        try:
            icons.append(GdkPixbuf.Pixbuf.new_from_file(filename))
        except GLib.Error: # pylint: disable=E0712
            cxlog.warn("could not load icon file %s:\n%s" % (cxlog.debug_str(filename), traceback.format_exc()))
    icons.sort(key=lambda x: x.get_property('width'))
    return icons


def set_default_icon(basename='crossover'):
    Gtk.Window.set_default_icon_list(get_std_icon_list(basename))


def get_icon_name(names, size, symbolic=False):
    """Return the name of the first icon available, None if none are available"""

    icon_theme = Gtk.IconTheme.get_default()
    if isinstance(size, Gtk.IconSize):
        size = Gtk.IconSize.lookup(size)[1]

    name_extensions = ['', '-symbolic']
    if symbolic:
        name_extensions = ['-symbolic', '']

    for name in names:
        for name_extension in name_extensions:
            result = icon_theme.lookup_icon(name + name_extension, size, Gtk.IconLookupFlags.USE_BUILTIN)
            if result:
                return name + name_extension

    return None


def load_icon(names, size, symbolic=False):
    "load the first available icon, or return None if none are available"

    icon_theme = Gtk.IconTheme.get_default()
    if isinstance(size, Gtk.IconSize):
        size = Gtk.IconSize.lookup(size)[1]

    name = get_icon_name(names, size, symbolic)
    if name:
        return icon_theme.load_icon(name, size, Gtk.IconLookupFlags.USE_BUILTIN)

    return None


# Menu functions
# These were added in GTK 3.22, but we want to support 3.18
if hasattr(Gtk.Menu, 'popup_at_rect'):
    def popup_at_rect(menu, window, rect, rect_anchor, menu_anchor, event):
        menu.popup_at_rect(window, rect, rect_anchor, menu_anchor, event)

    def popup_at_widget(menu, widget, widget_anchor, menu_anchor, event):
        menu.popup_at_widget(widget, widget_anchor, menu_anchor, event)

    def popup_at_pointer(menu, _widget, event):
        menu.popup_at_pointer(event)
else:
    def _position_at_rect_edge(rect, anchor):
        if anchor == Gdk.Gravity.NORTH_WEST:
            return rect.x, rect.y
        if anchor == Gdk.Gravity.SOUTH_WEST:
            return rect.x, rect.y+rect.height
        cxlog.err("Unsupported gravity in _position_at_rect_edge: %s" % anchor)
        return rect.x, rect.y

    def _position_by_rect(_menu, _menu_x, _menu_y, userdata):
        (rect, rect_anchor, menu_anchor) = userdata
        rect_edge_x, rect_edge_y = _position_at_rect_edge(rect, rect_anchor)
        if menu_anchor != Gdk.Gravity.NORTH_WEST:
            cxlog.err("Unsupported gravity in _position_by_rect: %s" % menu_anchor)
        return rect_edge_x, rect_edge_y, True

    def _popup_at_screen_rect(menu, rect, rect_anchor, menu_anchor, event):
        try:
            timestamp = event.time
        except AttributeError:
            timestamp = Gtk.get_current_event_time()
        try:
            button = event.button
        except AttributeError:
            button = 0
        try:
            device = event.device
        except AttributeError:
            device = None
        menu.popup_for_device(device, None, None, _position_by_rect, (rect, rect_anchor, menu_anchor), button, timestamp)

    def _get_root_position(widget, x, y): # pylint: disable=C0103
        try:
            return widget.get_root_coords(x, y)
        except AttributeError:
            if widget.props.parent:
                parent_x, parent_y = widget.translate_coordinates(widget.props.parent, x, y)
                return _get_root_position(widget.props.parent, parent_x, parent_y)

            if widget.props.window:
                return widget.props.window.get_root_coords(x, y)

        raise ValueError("_get_root_position: widget not realized")

    def _detach_menu(menu):
        menu.detach()

    def popup_at_rect(menu, window, rect, rect_anchor, menu_anchor, event):
        screen_rect = Gdk.Rectangle()
        screen_rect.x, screen_rect.y = _get_root_position(window, 0, 0)
        screen_rect.x += rect.x
        screen_rect.y += rect.y
        screen_rect.width = rect.width
        screen_rect.height = rect.height
        if not menu.get_attach_widget():
            menu.attach_to_widget(window)
            menu.connect('hide', _detach_menu)
        _popup_at_screen_rect(menu, screen_rect, rect_anchor, menu_anchor, event)

    def popup_at_widget(menu, widget, widget_anchor, menu_anchor, event):
        rect = Gdk.Rectangle()
        rect.x, rect.y = _get_root_position(widget, 0, 0)
        rect.width = widget.get_allocated_width()
        rect.height = widget.get_allocated_height()
        if not menu.get_attach_widget():
            menu.attach_to_widget(widget)
            menu.connect('hide', _detach_menu)
        _popup_at_screen_rect(menu, rect, widget_anchor, menu_anchor, event)

    def popup_at_pointer(menu, widget, event):
        rect = Gdk.Rectangle()
        rect.x = event.x_root
        rect.y = event.y_root
        rect.width = 0
        rect.height = 0
        if not menu.get_attach_widget():
            menu.attach_to_widget(widget)
            menu.connect('hide', _detach_menu)
        _popup_at_screen_rect(menu, rect, Gdk.Gravity.NORTH_WEST, Gdk.Gravity.NORTH_WEST, event)


def rgba_parse(spec):
    result = Gdk.RGBA()
    if not result.parse(spec):
        raise ValueError("Failed to parse color '%s'" % spec)
    return result


#####
#
# CXMessageDlg
#
#####

def _messagedlg_handle_response(dialog, response, response_function, close_on_response, user_data):
    if response_function:
        args = []
        if not close_on_response:
            args.append(dialog)
        args.append(response)
        if user_data is not None:
            args.append(user_data)

        response_function(*args)

    if close_on_response:
        dialog.destroy()


def CXMessageDlg(message=None, primary=None, secondary=None, markup=None, buttons=Gtk.ButtonsType.OK, response_function=None, close_on_response=True, parent=None, user_data=None, button_array=None, message_type=None):
    if not markup:
        if message is None:
            text = []
            if primary is not None:
                text.append('<span weight="bold" size="larger">%s</span>' % cxutils.html_escape(primary))
            if secondary is not None:
                text.append(cxutils.html_escape(secondary))
            markup = '\n\n'.join(text)
        else:
            markup = cxutils.html_escape(message)
    else:
        # &nbsp; is an important HTML typographic entity (e.g. in French) but
        # it is not supported by GTK+. So manually convert it to the
        # corresponding Unicode character
        markup = markup.replace("&nbsp;", "\u00a0")
    if message_type is None:
        cxlog.warn("Callers to cxguitools.CXMessageDlg should always set an alert type.")
        message_type = Gtk.MessageType.INFO
    if button_array:
        dialog = Gtk.MessageDialog(buttons=Gtk.ButtonsType.NONE, modal=True, transient_for=parent, message_type=message_type)
        for button in button_array:
            new_button = dialog.add_button(button[0], button[1])
            if len(button) > 2:
                new_button.get_style_context().add_class(button[2])
    else:
        dialog = Gtk.MessageDialog(buttons=buttons, modal=True, transient_for=parent, message_type=message_type)
    dialog.set_markup(markup)
    dialog.connect('response', _messagedlg_handle_response, response_function, close_on_response, user_data)
    dialog.show()


#####
#
# Standard warning dialog if running as root
#
#####

def return_from_root_check(response):
    if response:
        sys.exit()
    # dismiss this dialog
    Gtk.main_quit()

# This should be called before loading the main dialog to make sure the code
# this protects is not run at all
def warn_if_root():
    if os.getuid() == 0:
        # We probably shouldn't be running as root.
        warning = _("Running this tool as root is very dangerous, and will only allow you to configure CrossOver for the root user.")
        if distversion.HAS_MULTI_USER:
            warning = _("%s\n\nIf you wish to create a bottle that all the users of this computer can access, log in as a user and use the 'Publish Bottle' feature.") % warning
        CXMessageDlg(warning, buttons=None, button_array=[
            [_("Exit"), 1, 'suggested-action'],
            # Where Root is the administrator account
            [_("Configure CrossOver for Root"), 0]],
                     response_function=return_from_root_check, message_type=Gtk.MessageType.WARNING)
        Gtk.main()


#####
#
# Standard File Picker selectors
#
#####

FILTERS_RUNNABLE = set(('exe', 'lnk'))
FILTERS_INSTALLABLE = set(('install',))
FILTERS_CXARCHIVES = set(('archives', 'cxarchives'))
FILTERS_ALLFILES = set(('all',))


def add_ipattern(file_picker, pattern):
    """Adds a case-insensitive pattern based on the given case-sensitive one.
    """
    insensitive = []
    for char in pattern:
        if char.isalpha():
            insensitive.append("[%s%s]" % (char.lower(), char.upper()))
        else:
            insensitive.append(char)
    file_picker.add_pattern("".join(insensitive))


def get_filter(name):
    if name == 'archives':
        file_filter = Gtk.FileFilter()
        # These are Unix files so keep them case-sensitive
        file_filter.add_pattern("*.cpio.7z")
        file_filter.add_pattern("*.cpio.gz")
        file_filter.add_pattern("*.cpio.bz2")
        file_filter.add_pattern("*.cpio.Z")
        file_filter.add_pattern("*.tar.7z")
        file_filter.add_pattern("*.tar.gz")
        file_filter.add_pattern("*.tar.bz2")
        file_filter.add_pattern("*.tar.Z")
        file_filter.add_pattern("*.tgz")
        return file_filter

    if name == 'cxarchives':
        file_filter = Gtk.FileFilter()
        # These are Unix files so keep them case-sensitive
        file_filter.add_pattern("*.cxarchive")
        return file_filter

    if name == 'exe':
        file_filter = Gtk.FileFilter()
        add_ipattern(file_filter, "*.bat")
        add_ipattern(file_filter, "*.cmd")
        add_ipattern(file_filter, "*.com")
        add_ipattern(file_filter, "*.exe")
        add_ipattern(file_filter, "autorun.inf")
        return file_filter

    if name == 'install':
        file_filter = Gtk.FileFilter()
        # executables
        add_ipattern(file_filter, "*.bat")
        add_ipattern(file_filter, "*.cmd")
        add_ipattern(file_filter, "*.com")
        add_ipattern(file_filter, "*.dll")
        add_ipattern(file_filter, "*.exe")
        add_ipattern(file_filter, "autorun.inf")
        # fonts
        add_ipattern(file_filter, "*.otf")
        add_ipattern(file_filter, "*.ttc")
        add_ipattern(file_filter, "*.ttf")
        # msi
        add_ipattern(file_filter, "*.msi")
        add_ipattern(file_filter, "*.msp")
        # msu
        add_ipattern(file_filter, "*.msu")
        # archives
        add_ipattern(file_filter, "*.zip")
        add_ipattern(file_filter, "*.cab")
        add_ipattern(file_filter, "*.tgz")
        add_ipattern(file_filter, "*.tar.gz")
        add_ipattern(file_filter, "*.tar.bz2")
        add_ipattern(file_filter, "*.tar")
        add_ipattern(file_filter, "*.tbz")
        add_ipattern(file_filter, "*.tb2")
        add_ipattern(file_filter, "*.rar")
        add_ipattern(file_filter, "*.7z")
        return file_filter

    if name == 'lnk':
        file_filter = Gtk.FileFilter()
        add_ipattern(file_filter, "*.lnk")
        return file_filter

    if name == 'tie':
        file_filter = Gtk.FileFilter()
        file_filter.add_pattern("*.tie")
        file_filter.add_pattern("*.c4p")
        return file_filter

    if name == 'all':
        file_filter = Gtk.FileFilter()
        add_ipattern(file_filter, "*")
        return file_filter

    return None


def add_filters(file_picker, filters):
    """Adds the specified set of standard file filters to the file picker."""
    file_filters = {}
    default = ''

    if 'archives' in filters:
        file_filters[_("Unix Archives")] = get_filter('archives')

    if 'cxarchives' in filters:
        default = _("CrossOver Bottle Archives")
        file_filters[default] = get_filter('cxarchives')

    if 'exe' in filters:
        default = _("Windows Programs")
        file_filters[default] = get_filter('exe')

    if 'install' in filters:
        default = _("Installer Files")
        file_filters[default] = get_filter('install')

    if 'lnk' in filters:
        file_filters[_("Windows Shortcuts")] = get_filter('lnk')

    if 'all' in filters:
        file_filters[_("All Files")] = get_filter('all')

    names = sorted(file_filters)
    for name in names:
        file_filter = file_filters[name]
        file_filter.set_name(name)
        file_picker.add_filter(file_filter)
        if name == default:
            # Make this filter the default
            file_picker.set_filter(file_filter)


def filter_path(filter_name, path):
    info = Gtk.FileFilterInfo()
    info.contains = Gtk.FileFilterFlags.FILENAME | Gtk.FileFilterFlags.DISPLAY_NAME
    info.display_name = path
    info.filename = path

    return get_filter(filter_name).filter(info)


def _skip_confirm_extensionless(file_picker):
    """If we're going to append an extension to a filename,
    skip the overwrite confirm dialog.
    """
    # pylint: disable=R0201
    filename = os.path.basename(file_picker.get_filename())
    extension = os.path.splitext(filename)[1]
    if extension == "":
        return Gtk.FileChooserConfirmation.ACCEPT_FILENAME
    return Gtk.FileChooserConfirmation.CONFIRM

def _append_extension(file_picker, response, default_extension):
    """Append extension to a filename if it lacks one."""
    # pylint: disable=R0201
    if response == Gtk.ResponseType.OK:
        filename = os.path.basename(file_picker.get_filename())
        basename = os.path.splitext(filename)[0]
        extension = os.path.splitext(filename)[1]

        if extension == "":
            # we have to restart the 'response' event because the overwrite check already happened
            file_picker.stop_emission_by_name('response')

            filename = basename + "." + default_extension

            file_picker.set_current_name(filename)
            file_picker.response(Gtk.ResponseType.OK)
    return True

def set_default_extension(file_picker, extension):
    file_picker.connect("response", _append_extension, extension)
    file_picker.connect("confirm-overwrite", _skip_confirm_extensionless)

def show_help():
    if distversion.IS_CJBUILD:
        help_url = "https://www.crossoverchina.com/redirect/?product=Crossover&version=%s&referer=Help_Quickhelp" % distversion.CX_VERSION
    else:
        help_url = "https://support.codeweavers.com/user-guides/crossover-linux-user-guide"

    cxutils.launch_url(help_url)
