# (c) Copyright 2010-2015. CodeWeavers, Inc.

import os

import cxproduct
import cxlog
import cxutils


TIE_VERSION = 2


#
# Association installation / uninstallation
#

def install():
    cxassoc = os.path.join(cxutils.CX_ROOT, 'bin', 'cxassoc')
    if cxutils.system((cxassoc, '--tie', '--removeall', '--install')):
        return 1
    return 0


def uninstall():
    cxassoc = os.path.join(cxutils.CX_ROOT, 'bin', 'cxassoc')
    if cxutils.system((cxassoc, '--tie', '--removeall')):
        return 1
    return 0


#
# Product registration / unregistration
#

def _get_product_key(config, cxroot):
    """Figures out which entry contains the current product's information."""
    for key, section in config.items():
        if section.get('Root', None) == cxroot:
            return key
    return None


def _get_best_tie(config, product_key=None):
    """Picks (one of) the product(s) with the highest TieVersion."""
    best_version = 0
    best_key = None
    best_is_current = False
    for key, section in config.items():
        try:
            version = int(section.get('TieVersion', '0'), 10)
            # Pick the currently installed tie handler in case of a version tie!
            # And pick product_key as the second best
            installed = section.get('TieInstalled', 'false') == 'true'
            if version > best_version or \
                    (version == best_version and installed) or \
                    (version == best_version and key.lower() == product_key and not best_is_current):
                best_version = version
                best_key = key
                best_is_current = installed
        except TypeError:
            pass
    return best_key


def _get_current_tie(config):
    """Returns the key of the currently installed tie handler."""
    for key, section in config.items():
        if section.get('TieInstalled', 'false') == 'true':
            return key
    return None


def _run_action(section, action):
    environ = os.environ.copy()
    environ['CX_ROOT'] = section['Root']

    # Split the command and only then expand the individual args to avoid
    # trouble with quoting.
    argv = []
    for arg in cxutils.cmdlinetoargv(section[action]):
        argv.append(cxutils.expand_unix_string(environ, arg))
    if cxutils.system(argv):
        return 1
    return 0


def register():
    """Adds a product to the list of registered products.

    Also installs its .tie file handler, uninstalling that of another product
    if appropriate."""
    product = cxproduct.this_product()
    product_tie_version = TIE_VERSION

    config = cxproduct.get_cross_product_config()
    wconfig = config.get_save_config()
    wconfig.lock_file()

    # Pick a unique key for the current product
    product_key = _get_product_key(config, cxutils.CX_ROOT)
    if product_key is None:
        i = 1
        while True:
            product_key = 'Product-%d-%d' % (os.geteuid(), i)
            if product_key not in config:
                break
            i += 1
    product_key = product_key.lower()

    # Register the product in the shared configuration file
    section = wconfig[product_key]
    section['Root'] = product['root']
    section['Name'] = product['name']
    section['PublicVersion'] = product['publicversion']
    old_version = section.get('ProductVersion', '')
    section['ProductVersion'] = product['productversion']
    section['ProductID'] = product['productid']
    section['TieVersion'] = "%d" % product_tie_version
    section['TieInstall'] = '"${CX_ROOT}/bin/%s" --install' % cxlog.name0()
    section['TieUninstall'] = '"${CX_ROOT}/bin/%s" --uninstall' % cxlog.name0()

    # Then check if we should install our trampoline
    if section.get('TieInstalled', 'false') != 'true' or \
       cxutils.cmp_versions(old_version, "12.5.0") <= 0:
        best_tie = _get_best_tie(config, product_key)
        # Don't try to set ourselves as the default handler if we cannot
        # uninstall the current one (typically because it belongs to root)
        current_tie = _get_current_tie(config)
        if best_tie == product_key and \
                (current_tie is None or current_tie in wconfig):
            if current_tie is not None and current_tie != product_key and \
                    'TieUninstall' in wconfig[current_tie]:
                _run_action(config[current_tie], 'TieUninstall')
                del wconfig[current_tie]['TieInstalled']
            install()
            section['TieInstalled'] = 'true'

    # Unlock the file last to avoid races in the install/uninstall steps
    wconfig.save_and_unlock_file()
    return 0


def unregister():
    """Removes a product from the list of registered products.

    Also uninstalls its .tie file handler and installs that of another product
    if appropriate."""

    config = cxproduct.get_cross_product_config()

    wconfig = config.get_save_config()
    wconfig.lock_file()

    product_key = _get_product_key(config, cxutils.CX_ROOT)
    if product_key is None:
        cxlog.warn('this product has already been unregistered')
        return 0
    installed = config[product_key].get('TieInstalled', 'false') == 'true'
    del wconfig[product_key]

    if installed:
        uninstall()
        best_tie = _get_best_tie(config)
        if best_tie and best_tie in wconfig:
            if _run_action(config[best_tie], 'TieInstall') == 0:
                wconfig[best_tie]['TieInstalled'] = 'true'

    # Unlock the file last to avoid races in the install/uninstall steps
    wconfig.save_and_unlock_file(True)
    return 0


#
# Tie file handling
#

def open_file(filename):
    cxinstaller = os.path.join(cxutils.CX_ROOT, 'bin', 'cxinstaller')
    os.execl(cxinstaller, cxinstaller, '--tiefile', filename)
