/*******************************************************************************
* parsers.cpp: Command line and configuration files parsers
*-------------------------------------------------------------------------------
* (c)1999-2001 VideoLAN
* $Id: parsers.cpp,v 1.11.2.1.2.3 2003/06/04 17:49:33 alexis Exp $
*
* Authors: Benoit Steiner <benny@via.ecp.fr>
*          Arnaud de Bossoreille de Ribou <bozo@via.ecp.fr>
*          Cyril Deguet <asmax@via.ecp.fr>
*          Damien Lucas <nitrox@videolan.org>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
*-------------------------------------------------------------------------------
*
*******************************************************************************/


//------------------------------------------------------------------------------
// Preamble
//------------------------------------------------------------------------------
#include "defs.h"


#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#ifdef HAVE_OPENDIR
#include <dirent.h>
#endif
#include <sys/types.h>
#include <string.h>

#include "common.h"
#include "debug.h"
#include "reflect.h"
#include "serialization.h"
#include "string.h"
#include "vector.h"
#include "hashtable.h"
#include "stack.h"
#include "buffers.h"
#include "exception.h"
#include "file.h"
#include "log.h"
#include "stream.h"

#include "parsers.h"
#include "lexer.h"

#include "stack.cpp"
#include "stream.cpp"


//******************************************************************************
// class C_ParserException
//******************************************************************************
//
//******************************************************************************

//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
E_Parser::E_Parser(int iCode, const C_String& strMsg) :
                        E_Exception(iCode, strMsg)
{
  // Nothing to do
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
E_Parser::E_Parser(int iCode, const C_String& strMsg, E_Exception e) :
                        E_Exception(iCode, strMsg, e)
{
  // Nothing to do
}





//******************************************************************************
// class C_CfgFileParser
//******************************************************************************
//
//******************************************************************************

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_CfgFileParser::C_CfgFileParser(C_ParserHandler* pCallBack) : 
    m_cContextStack(10)
{
  ASSERT(pCallBack);
  m_pCallBack = pCallBack;

  // Init the file position indicators
  m_iLineNumber = 0;

}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
C_Vector<C_String> C_CfgFileParser::Parse(const C_String& strFileName, bool bCompletePath,
                                                        const C_String& strRoot)
{
  C_Vector<C_String> vInclude;
  
  ASSERT(m_pCallBack);

  m_pCallBack->SetRoot(strRoot);
#ifdef DEBUG
  if(strRoot!="")
    fprintf(stderr, "Reading file %s for module '%s'\n", strFileName.GetString(),
                                                            strRoot.GetString());
  else
    fprintf(stderr, "Reading file %s for base configuration\n", 
		                                        strFileName.GetString());
#endif
  try
  {
    // Open the cfg file
    C_File* pCfgFile = new C_File(strFileName);
    pCfgFile->Open("r");
    yyin = pCfgFile->GetHandle();

    // Parse the file
    int iTokenType = 0;
    while ((iTokenType = yylex()) != 0)
    {
      C_String strFirstToken(yytext);
      if (iTokenType == TOK_END)
      {
        // Unstack the section
        if(m_cContextStack.Size() == 0)
        {
          throw E_Parser(GEN_ERR, strFirstToken + "tag at line " + 
                         m_iLineNumber + "in excess");
        }
        C_String* pstrSection = m_cContextStack.Pop();
        m_pCallBack->OnEndSection(*pstrSection);
        delete pstrSection;
        if (yylex() == TOK_NEWLINE)
        {
          m_iLineNumber++;
        }
        else {
          throw E_Parser(GEN_ERR,
                         C_String("Error in the configuration file at line ") +
                         m_iLineNumber);
        }
      }
      else if (iTokenType == TOK_BEGIN)
      { 
        if (yylex() != TOK_VALUE)
        {
          throw E_Parser(GEN_ERR,
                         C_String("Error in the configuration file at line ") +
                         m_iLineNumber);
        }
        else if(m_cContextStack.Size() == m_cContextStack.Capacity())
        {
          throw E_Parser(GEN_ERR, C_String("Error line ") + m_iLineNumber +
                         ": only " + m_cContextStack.Capacity() +
                         " section levels allowed");
        }
        C_String strValue(yytext);
        // Remove the quotes. FIXME: should be done by the lexer
        C_String strSection(strValue.SubString(1, strValue.Length()-1));
        m_cContextStack.Push(new C_String(strSection));
        m_pCallBack->OnStartSection(strSection);
        if (yylex() == TOK_NEWLINE)
        {
          m_iLineNumber++;
        }
        else {
          throw E_Parser(GEN_ERR,
                         C_String("Error in the configuration file at line ") +
                         m_iLineNumber);
        }
      }
      else if (iTokenType == TOK_VAR)
      {
        if (yylex() != TOK_EQUAL) 
        {
          throw E_Parser(GEN_ERR,
                         C_String("Error in the configuration file at line ") +
                         m_iLineNumber);
        }
        else if (yylex() != TOK_VALUE)
        {
          throw E_Parser(GEN_ERR,
                         C_String("Error in the configuration file at line ") +
                         m_iLineNumber);
        }
        C_String strValue(yytext);
        // Remove the quotes. FIXME: should be done by the lexer
        C_String strStrippedValue(strValue.SubString(1, strValue.Length()-1));
        if(strFirstToken=="ConfigPath")
        {
	  C_String* strInc=new C_String(strStrippedValue+"/input.cfg");
	  C_String* strIncRoot=new C_String(m_pCallBack->GetSection());
	  vInclude.Add(strInc);
	  vInclude.Add(strIncRoot);
	}
	else
        {
	  m_pCallBack->OnProperty(strFirstToken, strStrippedValue);
	}
	if (yylex() == TOK_NEWLINE)
        {
          m_iLineNumber++;
        }
        else
       	{
          throw E_Parser(GEN_ERR,
                         C_String("Error in the configuration file at line ") +
                         m_iLineNumber);
        }
      } 
      else if (iTokenType == TOK_NEWLINE)
      {
        m_iLineNumber++;
      }
      else
      {
        throw E_Parser(GEN_ERR,
                       C_String("Error in the configuration file at line ") +
                       m_iLineNumber);
      }
    }
  }
  catch(E_Exception e)
  {
    throw E_Parser(GEN_ERR, "Parsing of file '" + strFileName + "' failed", e);
  }
 
  if(strRoot=="")
  {
    m_pCallBack->OnStartSection("local1");
    m_pCallBack->OnProperty("ProgramCount", m_pCallBack->GetLocalCount());
    m_pCallBack->OnEndSection("local1");
  }
  else
  {
    m_pCallBack->OnProperty("ProgramCount", m_pCallBack->GetLocalCount());
  }
  return vInclude;
}

// Needed for flex functions
extern "C" int yywrap(void)
{
  return 1;
}


void C_CfgFileParser::AddCmdLineSetting(unsigned char sOption, C_String sValue)
{
  C_String sMeth;
  C_String sDest1;
  C_String sDest2;

  switch(sOption)
  {
    case 'd':  /* Destination spec */
      /* Format string should be methode:/dest:port */
     if(sValue=="") throw E_Parser(GEN_ERR, "-d option needs argument"); 
      /* On rcupre le port de destination s'ilest prcis */
      if(sValue.Find(':', 1)!=GEN_ERR)
      {
        sMeth=sValue.SubString(0,sValue.Find(':', 1));
        sValue=sValue.SubString(sValue.Find(':', 1)+1,
                                                  strlen(sValue.GetString()));
        if(sValue.Find(':', 1)!=GEN_ERR)
        {
          sDest1=sValue.SubString(0,sValue.Find(':', 1));
          sDest2=sValue.SubString(sValue.Find(':', 1)+1,
                                                  strlen(sValue.GetString()));
        }
        else
        {
          sDest1=sValue;
          sDest2="";
        }
      }
      else
      {
        throw E_Parser(GEN_ERR, "No methode specified for output");
      }
      if(sMeth=="udp" || sMeth=="rtp")
      {
        if(sDest1=="") throw E_Parser(GEN_ERR, "No IP destination specified");
        if(sDest2.ToInt()<=0 && sDest2!="")
          throw E_Parser(GEN_ERR, "Invalid port specification");
        m_pCallBack->OnStartSection("Channels");
        m_pCallBack->OnProperty("output1", "network");
        m_pCallBack->OnEndSection("Channels");
        m_pCallBack->OnStartSection("output1");
        m_pCallBack->OnProperty("DstHost", sDest1);
        if(sDest2!="") m_pCallBack->OnProperty("DstPort", sDest2);
        if(sDest1.ToInt()<=239 && sDest1.ToInt()>=224)
          m_pCallBack->OnProperty("Type", "multicast");
        if(sMeth=="udp") m_pCallBack->OnProperty("Protocol", "udp");
	else m_pCallBack->OnProperty("Protocol", "rtp");
        m_pCallBack->OnEndSection("output1");
      }
      else if (sMeth=="file")
      {
        if(sDest1=="") throw E_Parser(GEN_ERR, "No Filname specified");
        m_pCallBack->OnStartSection("Channels");
          m_pCallBack->OnProperty("output1", "file");
        m_pCallBack->OnEndSection("Channels");
        m_pCallBack->OnStartSection("output1");
          m_pCallBack->OnProperty("FileName", sDest1);
          m_pCallBack->OnProperty("Append", "no");
        m_pCallBack->OnEndSection("output1");
      }
      break;

    case 't':
      if(sValue.ToInt()>0)
      {
        m_pCallBack->OnStartSection("output1");
        m_pCallBack->OnProperty("TTL", sValue.ToInt());
        m_pCallBack->OnEndSection("output1");
      }
      else
      {
        printf("Invalid value for TTL option\n");
        exit(0);
      }
      break;
      
    case 'l':
      /* Store looping property in the program specified */
      m_pCallBack->OnStartSection("program1");
      m_pCallBack->OnProperty("loop", "true");
      m_pCallBack->OnEndSection("program1");
      break;

    case 130:
      /* Log to file */
      m_pCallBack->OnStartSection("global");
      m_pCallBack->OnProperty("logfile", sValue);
      m_pCallBack->OnEndSection("global");
      break;
      
  }
}

void C_CfgFileParser::AddCmdLineTarget(C_String sTarget, C_String sTags)
{
  C_String sMeth;            /* First part of the tager definition */
  C_String sFile;            /* Second part of the target definition */
  C_String sOptions;         /* Third part of the target definition */
  C_String sProgramLaunch;   /* Name of the program to be launch at startup */

  /* Target Parsing */
  if(sTarget.Find(':', 1)!=GEN_ERR)
  {
    sMeth=sTarget.SubString(0,sTarget.Find(':', 1));
    sFile=sTarget.SubString(sTarget.Find(':', 1)+1, strlen(sTarget.GetString()));
    if(sFile.Find(':', 1)!=GEN_ERR)
    {
      sOptions=sFile.SubString(sFile.Find(':', 1)+1, strlen(sFile.GetString()));
      sFile=sFile.SubString(0,sFile.Find(':', 1));
    }
    else
    {
      sOptions="";
    }
  }
  else
    throw E_Parser(GEN_ERR, "Syntax error in target definition");
  if(sMeth=="") throw E_Parser(GEN_ERR, "No method specified");
  if(sFile=="") throw E_Parser(GEN_ERR, "No file/device specified");
  
 
  /* Input section */
  m_pCallBack->OnStartSection("input1");
  m_pCallBack->OnProperty("ProgramCount", "1");
  m_pCallBack->OnEndSection("input1");
  
  /* Inputs section */
  m_pCallBack->OnStartSection("Inputs");
  if(sMeth == "dvd" || sMeth == "file")
  {
    m_pCallBack->OnProperty("input1", "local");
  }
  else if(sMeth=="dvb")
  {
    m_pCallBack->OnProperty("input1", "dvb");
  }
  m_pCallBack->OnEndSection("Inputs");
  
  /* Input Section */
  if(sMeth == "dvd")
  {
    m_pCallBack->OnStartSection("input1.1");
    m_pCallBack->OnProperty("Name", "program1");
    m_pCallBack->OnProperty("Device", sFile);
    m_pCallBack->OnProperty("Type", "Dvd");
    sProgramLaunch = "program1";
    m_pCallBack->OnEndSection("input1.1"); 
  }
  else if(sMeth == "dvb")
  {
    m_pCallBack->OnStartSection("input1");
    m_pCallBack->OnProperty("DeviceNumber", sOptions.ToInt());
    m_pCallBack->OnProperty("SendMethod", 0);
    sProgramLaunch = sFile;
    sProgramLaunch.Replace(' ', '_');
    m_pCallBack->OnEndSection("input1"); 
  }
  else if(sMeth == "file")
  {
    m_pCallBack->OnStartSection("input1.1");
    m_pCallBack->OnProperty("Name", "program1");
    m_pCallBack->OnProperty("FileName", sFile);
    C_File fFile = C_File(sFile);
    fFile.Open("r");
      byte bFormat;
      fFile.Read(&bFormat, 1);
      if(bFormat==0x47) m_pCallBack->OnProperty("Type", "MPEG2-TS");
      fFile.Seek(4+1, SEEK_CUR);
      fFile.Read(&bFormat, 1);
      if(bFormat==0x01) m_pCallBack->OnProperty("Type", "MPEG1-PS");
      else if(bFormat==0x04) m_pCallBack->OnProperty("Type", "MPEG2-PS");
    fFile.Close();
    sProgramLaunch = "program1";
    m_pCallBack->OnEndSection("input1.1"); 
  }

  /* Automatic start section */
  m_pCallBack->OnStartSection("LaunchOnStartUp");
  m_pCallBack->OnProperty("command1", "start "+sProgramLaunch
                                                   +" output1 input1 "+sTags);
  m_pCallBack->OnEndSection("LaunchOnStartUp");
}
